% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pathway_heatmap.R
\name{pathway_heatmap}
\alias{pathway_heatmap}
\title{Create pathway heatmap}
\arguments{
\item{abundance}{A matrix or data frame of pathway abundance data, where columns
correspond to samples and rows correspond to pathways. Must contain at least
two samples.}

\item{metadata}{A data frame of metadata, where each row corresponds to a sample
and each column corresponds to a metadata variable.}

\item{group}{A character string specifying the column name in the metadata data frame
that contains the group variable. Must contain at least two groups.}

\item{colors}{A vector of colors used for the background of the facet labels in the
heatmap. If NULL or not provided, a default color set is used for the facet strips.}

\item{font_size}{A numeric value specifying the font size for the heatmap.}

\item{show_row_names}{A logical value indicating whether to show row names in the heatmap.}

\item{show_legend}{A logical value indicating whether to show the legend in the heatmap.}

\item{custom_theme}{A custom theme for the heatmap.}
}
\value{
A ggplot heatmap object representing the heatmap of the predicted functional
  pathway abundance data.
}
\description{
This function creates a heatmap of the predicted functional pathway abundance data.
The function first makes the abundance data relative, then converts the abundance data
to a long format and orders the samples based on the environment information.
The heatmap is then created using the `ggplot2` library.
}
\examples{
\donttest{
library(ggpicrust2)
library(ggh4x)
library(dplyr)
library(tidyr)
library(tibble)
library(magrittr)

# Create example functional pathway abundance data
kegg_abundance_example <- matrix(rnorm(30), nrow = 3, ncol = 10)
colnames(kegg_abundance_example) <- paste0("Sample", 1:10)
rownames(kegg_abundance_example) <- c("PathwayA", "PathwayB", "PathwayC")

# Create example metadata
metadata_example <- data.frame(
  sample_name = colnames(kegg_abundance_example),
  group = factor(rep(c("Control", "Treatment"), each = 5))
)

# Custom colors for facet strips
custom_colors <- c("skyblue", "salmon")

# Create a heatmap using custom colors for facet strips
pathway_heatmap(kegg_abundance_example, metadata_example, "group", colors = custom_colors)

# Use real dataset
data("metacyc_abundance")
data("metadata")
metacyc_daa_results_df <- pathway_daa(
  abundance = metacyc_abundance \%>\% column_to_rownames("pathway"),
  metadata = metadata,
  group = "Environment",
  daa_method = "LinDA"
)
annotated_metacyc_daa_results_df <- pathway_annotation(
  pathway = "MetaCyc",
  daa_results_df = metacyc_daa_results_df,
  ko_to_kegg = FALSE
)
feature_with_p_0.05 <- metacyc_daa_results_df \%>\% filter(p_adjust < 0.05)
pathway_heatmap(
  abundance = metacyc_abundance \%>\%
    right_join(
      annotated_metacyc_daa_results_df \%>\%
      select(all_of(c("feature","description"))),
      by = c("pathway" = "feature")
    ) \%>\%
    filter(pathway \%in\% feature_with_p_0.05$feature) \%>\%
    select(-"pathway") \%>\%
    column_to_rownames("description"),
  metadata = metadata,
  group = "Environment",
  colors = custom_colors
)
}
}
