% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sratoolkit_functions.R
\name{fastqDump}
\alias{fastqDump}
\title{Download or convert fastq data from NCBI Sequence Read Archive in a single thread (Windows compatible)}
\usage{
fastqDump(queryMetadata_object, sratoolkitPath = "",
  outputDirectory = ".", arguments = "-v --split-3",
  filenames = "accessions", source = "sra", cleanup = FALSE,
  fastqDumpHelp = FALSE)
}
\arguments{
\item{queryMetadata_object}{A list object returned from `queryMetadata` where one of the 
entities queried was `fastqMetadata`.}

\item{sratoolkitPath}{String. A path to a local copy of sratoolkit. Only necessary if sratoolkit
is not on your $PATH. Assumes executables are inside `bin`.}

\item{outputDirectory}{String. A path to the directory where you would like the files to be stored.}

\item{arguments}{A string variable of arguments to be passed directly to `fastq-dump`.
Defaults to "-v --split 3" to show progress and split paired-end data. 
Use fastqDumpHelp = TRUE to see a list of arguments.}

\item{filenames}{String. How would you like the downloaded fastq files to be named?
"accessions" names files with SRA accession numbers
"IDs" names files with their materialSampleID
"locality_IDs" names files with their locality and materialSampleID.}

\item{source}{String. `fastq-dump` can retrieve files directly from SRA, or it can convert .sra files
previously downloaded with `prefetch` that are in the current working directory. 
"sra" downloads from SRA
"local" converts .sra files in the current working directory.}

\item{cleanup}{Logical. cleanup = T will delete any intermediate .sra files.}

\item{fastqDumpHelp}{Logical. fastqDumpHelp = T will show the help page for `fastq-dump` and then quit.}
}
\value{
This function will not return anything within r. It simply downloads fastq files. It will print command line
stdout to the console, and also provide a start and end time and amount of time elapsed during the download.
}
\description{
`fastqDump()` uses the SRAtoolkit command-line function `fastq-dump` to download fastq
files from all samples returned by a \code{\link{queryMetadata}} query of GEOME, when one of the
entities queried was `fastqMetadata`
}
\details{
This function works best with sratoolkit functions of version 2.9.6 or greater. \href{https://www.ncbi.nlm.nih.gov/sra/docs/toolkitsoft/}{SRAtoolkit} functions can (ideally)
be in your $PATH, or you can supply a path to them using the sratoolkitPath argument.

`fastqDump()` downloads files to the current working directory unless a different one is assigned through outputDirectory.

`fastq-dump` will automatically split paired-end data into three files with:
\itemize{
 \item file_1.fastq having read 1
 \item file_2.fastq having read 2
 \item file.fastq having unmatched reads
 }
}
\examples{
\donttest{
# Run a query of GEOME first
acaoli <- queryMetadata(
    entity = "fastqMetadata", 
    query = "genus = Acanthurus AND specificEpithet = olivaceus AND _exists_:bioSample",
    select=c("Event"))

#trim to 3 entries for expediency
acaoli$fastqMetadata<-acaoli$fastqMetadata[1:3,]
acaoli$Event<-acaoli$Event[1:3,]

# Download straight from SRA, naming files with their locality and materialSampleID
fastqDump(queryMetadata_object = acaoli, filenames = "locality_IDs", source = "sra")

# A generally faster option is to run prefetch first, followed by fastqDump, with cleanup = T to 
# remove the prefetched .sra files.
prefetch(queryMetadata_object = acaoli)
fastqDump(queryMetadata_object = acaoli, filenames = "IDs", source = "local", cleanup = T)
}
}
\seealso{
\url{https://www.ncbi.nlm.nih.gov/sra/docs/toolkitsoft/} to download pre-compiled executables for sratoolkit or
\url{https://github.com/ncbi/sra-tools/wiki/Building-and-Installing-from-Source>} to install from source

See \code{\link{prefetch}} to download .sra files prior to converting them locally. This two step process works faster than
just using `fastqDump()`.
See \code{\link{fasterqDump}} for a faster, multithreaded version of `fastqDump()` that does not work on Windows.
}
