#' @title Occluded Surface Packing (OSP)
#' @name osp
#'
#' @description {Implements the 'occluded surface' packing density metric (OSP) averaged by residue, as described in (Fleming and Richards 2000).}
#'
#' @param file a SRF File (.srf) generated by 'occluded_surface' in fibos_files folder.
#'
#' @return A table containing:
#'  \describe{
#' 	  \item{\code{Resnum}}{residue id.}
#' 	  \item{\code{Resname}}{residue name.}
#' 	  \item{\code{OS}}{the summed areas of dots in residue.}
#' 	  \item{\code{`os*[1-raylen]`}}{'OS' areas weighted by (1-raylen). Raylen is the average lengths of normals normalized by 2.8 \eqn{\text{\AA}} (water diameter). So, raylen is a value between 0 and 1. A raylen close to 1 indicates worse packaging, and the 'OS' will be reduced.}
#'    \item{\code{OSP}}{average occluded surface packing value (OSP) by residue.}
#'  }
#' @seealso [occluded_surface()]
#'
#' @author Herson Soares
#' @author Joao Romanelli
#' @author Patrick Fleming
#' @author Carlos Silveira.
#'
#' @references
#' Fleming PJ, Richards FM (2000). "Protein packing: Dependence on protein size, secondary structure and amino acid composition." <doi:10.1006/jmbi.2000.3750>
#'
#' Pattabiraman N, Ward KB, Fleming PJ (1995). "Occluded molecular surface: Analysis of protein packing." <doi:10.1002/jmr.300080603>
#'
#' Soares HHM, Romanelli JPR, Fleming PJ, da Silveira CH (2024). "bioRxiv, 2024.11.01.621530." <doi:10.1101/2024.11.01.621530>
#'
#' @examples
#' \donttest{
#' library(fibos)
#' 
#' #Configure the Environment
#' fibos_config() 
#'
#' # Calculate FIBOS per atom and create .srf files in fibos_files folder
#' pdb_fibos <- occluded_surface("1ptx", method = "FIBOS")

#' # Calculate OSP metric per residue from .srf file in fibos_files folder
#' pdb_osp <- osp(fs::path("fibos_files","prot_1ptx.srf"))
#' }
#' @export
osp = function(file){
  if(reticulate::virtualenv_exists("fibos_venv")){
    reticulate::use_virtualenv("fibos_venv")
    if(!reticulate::py_module_available("fibos")){
      warning("Module 'python' 'fibos' not available. Use 'fibos_config()' to install the module.")
    }
    else{
      return(osp_internal(file))
    }
  } else {
    warning("The virtual environment fibos_venv was not found. Please run 'fibos_config()' to configure it.")
  }
}

#' @title Read OSP Values
#' @name read_osp
#'
#' @description Internal function to read OSP values from a PAK file generated by the OSP calculation.
#'
#' @param prot_file OSP File (.pak).
#' @return A data frame containing the OSP data.
#'
#' @importFrom readr read_table
#' @keywords internal
#' 
read_osp = function(prot_file){
  if (endsWith(prot_file, ".pak") == FALSE){
    prot_file = paste(prot_file,".pak",sep = "")
  }
  if(file.exists(prot_file) ==  FALSE){
    stop("File not Found: ", prot_file)
  }
  osp_data = readr::read_table(prot_file, show_col_types = FALSE)
  return(osp_data)
}
