% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eim-class.R
\name{welchtest}
\alias{welchtest}
\title{Performs a matrix-wise Welch's t-test for two eim objects}
\usage{
welchtest(
  object1 = NULL,
  object2 = NULL,
  X1 = NULL,
  W1 = NULL,
  X2 = NULL,
  W2 = NULL,
  nboot = 50,
  seed = NULL,
  alternative = "two.sided",
  ...
)
}
\arguments{
\item{object1}{An \code{eim} object, as returned by \link{eim}.}

\item{object2}{A second \code{eim} object to compare with \code{object}.}

\item{X1}{A \verb{(b x c)} matrix representing candidate votes per ballot box.}

\item{W1}{A \verb{(b x g)} matrix representing group votes per ballot box.}

\item{X2}{A second \verb{(b x c)} matrix to compare with \code{X}.}

\item{W2}{A second \verb{(b x g)} matrix to compare with \code{W}.}

\item{nboot}{Integer specifying how many times to run the EM algorithm per object.}

\item{seed}{An optional integer indicating the random seed for the randomized algorithms. This argument is only applicable if \code{initial_prob = "random"} or \code{method} is either \code{"mcmc"} or \code{"mvn_cdf"}. Aditionally, it sets the random draws of the ballot boxes.}

\item{alternative}{Character string specifying the type of alternative hypothesis to test. Must be one of \code{"two.sided"} (default), \code{"greater"}, or \code{"less"}. If \code{"two.sided"}, the test checks for any difference in estimated probabilities. If \code{"greater"}, it tests whether the first object has systematically higher probabilities than the second. If \code{"less"}, it tests whether the first has systematically lower probabilities.}

\item{...}{Additional arguments passed to \link{bootstrap} and \link{run_em}.}
}
\value{
A list with components:
\itemize{
\item \code{pvals}: a numeric matrix of p-values with the same dimensions as the estimated probability matrices (\code{pvals}) from the input objects.
\item \code{statistic}: a numeric matrix of t-statistics with the same dimensions as the estimated probability matrices (\code{pvals}).
\item \code{eim1} and \code{eim2}: the original \code{eim} objects used for comparison.
}

Each entry in the pvals matrix is the p-value from Welch's t-test between the corresponding
entries of the two estimated probability matrices.
}
\description{
This function compares two \code{eim} objects (or sets of matrices that can be converted to such objects) by computing a Welch's t-test on each component
of their estimated probability matrices (\code{p}). The Welch test is applied using bootstrap-derived standard deviations, and the result is a matrix
of p-values corresponding to each group-candidate combination.
}
\details{
It uses Welch's t-test to analyze if there is a significant difference between the estimated probabilities between a treatment and a control set. The test is performed independently for each component of the probability matrix.

The user must provide either of the following (but not both):
\itemize{
\item Two \code{eim} objects via \code{object1} and \code{object2}, or
\item Four matrices: \code{X1}, \code{W1}, \code{X2}, and \code{W2}, which will be converted into \code{eim} objects internally.
}

The Welch test is computed using the formula:

\deqn{
t_{ij} = \frac{p_{1,ij} - p_{2,ij}}{\sqrt{(s_{1,ij}^2 + s_{2,ij}^2) / n}},
}
In this expression, \eqn{s_{1,ij}^2} and \eqn{s_{2,ij}^2} represent the bootstrap sample variances for the treatment and control sets, respectively, while \eqn{p_{1,ij}} and \eqn{p_{2,ij}} are the corresponding estimated probability matrices obtained via the EM algorithm. The number of bootstrap samples is denoted by \eqn{n}, and the degrees of freedom for each component are calculated using the Welch-Satterthwaite equation
}
\examples{
sim1 <- simulate_election(num_ballots = 100, num_candidates = 3, num_groups = 5, seed = 123)
sim2 <- simulate_election(num_ballots = 100, num_candidates = 3, num_groups = 5, seed = 124)

result <- welchtest(sim1, sim2, nboot = 100)

# Check which entries are significantly different
which(result$pvals < 0.05, arr.ind = TRUE)

}
