#' @title Predict Method for fastNaiveBayes fits
#' @description Uses a fastNaiveBayes model and a new data set to create the classifications.
#'     This can either be the raw probabilities generated by the fastNaiveBayes model or the classes themselves.
#'
#' @param object A fitted object of class "fastNaiveBayes".
#' @param newdata A numeric matrix with 1's and 0's to indicate the presence or absence of features. A Sparse dgcMatrix is also accepted.
#'     Note that if newdata contains features that were not encountered in the training data, these are omitted from the prediction.
#'     Furthermore, newdata can contain fewer features than encountered in the training data. In this case, newdata will be padded with
#'     extra columns all filled with 0's.
#' @param type If "raw", the conditional a-posterior probabilities for each class are returned, and the class with maximal probability else.
#' @param sparse Use a sparse Matrix? If true a sparse matrix will be constructed from x, which can give up to a 40\% speed up.
#'     It's possible to directly feed a sparse dgcMatrix as x, which will set this parameter to TRUE
#' @param threshold A threshold for the minimum probability. For Bernoulli and Multinomial event models Laplace smoothing solves this,
#' but in the case of Gaussian event models, this ensures numerical probabilities
#' @param ... Not used.
#' @return If type = 'class', a factor with classified class levels. If type = 'raw', a matrix with the predicted probabilities of
#'     each class, where each column in the matrix corresponds to a class level.
#'
#' @export
#' @import Matrix
#'
#' @details In the extremely unlikely case that two classes have the exact same estimated probability, the first encountered class
#'     is used as the classification and a warning is issued.
#'
#'     Using a sparse matrix directly can be especially useful if it's necessary to use predict multiple times on the same matrix or
#'     on different subselections of the same initial matrix, see examples for further details.
#' @examples
#' rm(list = ls())
#' library(fastNaiveBayes)
#'
#' cars <- mtcars
#' y <- as.factor(ifelse(cars$mpg > 25, "High", "Low"))
#' x <- cars[, 2:ncol(cars)]
#'
#' # Mixed event models
#' dist <- fastNaiveBayes::fastNaiveBayes.detect_distribution(x, nrows = nrow(x))
#' print(dist)
#' mod <- fastNaiveBayes.mixed(x, y, laplace = 1)
#' pred <- predict(mod, newdata = x)
#' mean(pred != y)
predict.fastNaiveBayes.mixed <- function(object, newdata, type = c("class", "raw"),
                                         sparse = FALSE, threshold = .Machine$double.eps, ...) {
  type <- match.arg(type)
  if (class(newdata)[1] != "dgCMatrix") {
    if (!is.matrix(newdata)) {
      newdata <- as.matrix(newdata)
    }
    if (sparse) {
      newdata <- Matrix(newdata, sparse = TRUE)
    }
  } else {
    sparse <- TRUE
  }

  names <- object$names
  distribution <- object$distribution

  other_names <- setdiff(names, colnames(newdata))
  if (length(other_names) > 0) {
    if (sparse) {
      other_mat <- Matrix(0L, nrow = nrow(newdata), ncol = length(other_names), sparse = TRUE)
    } else {
      other_mat <- matrix(0L, nrow = nrow(newdata), ncol = length(other_names))
    }
    colnames(other_mat) <- other_names

    newdata <- cbind(newdata, other_mat)
  }
  newdata <- newdata[, names]

  probs <- NULL
  for (i in 1:length(object$models)) {
    model <- object$models[[i]]
    newnames <- model$names
    newx <- newdata[, model$names]
    if (length(newnames) == 1) {
      newx <- as.matrix(newx)
      colnames(newx) <- newnames
    }
    if (is.null(probs)) {
      probs <- stats::predict(model, newx, type = "rawprob", sparse, threshold)
    } else {
      probs <- probs + stats::predict(model, newx, type = "rawprob", sparse, threshold)
    }
  }
  probs <- exp(probs)

  priors <- as.vector(object$priors)
  for (i in 1:length(priors)) {
    probs[, i] <- probs[, i] * priors[i]
  }

  denom <- rowSums(probs)
  denom[denom==0] <- 1
  probs <- probs / denom

  if (type == "class") {
    class <- names(object$priors)[max.col(probs, ties.method = "first")]
    return(as.factor(class))
  }
  return(probs)
}
