% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_annual_stats.R
\name{calc_annual_stats}
\alias{calc_annual_stats}
\title{Calculate annual summary statistics}
\usage{
calc_annual_stats(
  data,
  dates = Date,
  values = Value,
  groups = STATION_NUMBER,
  station_number,
  roll_days = 1,
  roll_align = "right",
  percentiles = c(10, 90),
  water_year_start = 1,
  start_year,
  end_year,
  exclude_years,
  months = 1:12,
  transpose = FALSE,
  ignore_missing = FALSE,
  allowed_missing = ifelse(ignore_missing, 100, 0)
)
}
\arguments{
\item{data}{Data frame of daily data that contains columns of dates, flow values, and (optional) groups (e.g. station numbers).
Leave blank if using \code{station_number} argument.}

\item{dates}{Name of column in \code{data} that contains dates formatted YYYY-MM-DD. Only required if dates column name is not 
'Date' (default). Leave blank if using \code{station_number} argument.}

\item{values}{Name of column in \code{data} that contains numeric flow values, in units of cubic metres per second.
Only required if values column name is not 'Value' (default). Leave blank if using \code{station_number} argument.}

\item{groups}{Name of column in \code{data} that contains unique identifiers for different data sets, if applicable. Only required 
if groups column name is not 'STATION_NUMBER'. Function will automatically group by a column named 'STATION_NUMBER' if 
present. Remove the 'STATION_NUMBER' column beforehand to remove this grouping. Leave blank if using \code{station_number} 
argument.}

\item{station_number}{Character string vector of seven digit Water Survey of Canada station numbers (e.g. \code{"08NM116"}) of
which to extract daily streamflow data from a HYDAT database. Requires \code{tidyhydat} package and a HYDAT database.
Leave blank if using \code{data} argument.}

\item{roll_days}{Numeric value of the number of days to apply a rolling mean. Default \code{1}.}

\item{roll_align}{Character string identifying the direction of the rolling mean from the specified date, either by the first 
(\code{'left'}), last (\code{'right'}), or middle (\code{'center'}) day of the rolling n-day group of observations. 
Default \code{'right'}.}

\item{percentiles}{Numeric vector of percentiles to calculate. Set to \code{NA} if none required. Default \code{c(10,90)}.}

\item{water_year_start}{Numeric value indicating the month (\code{1} through \code{12}) of the start of water year for
analysis. Default \code{1}.}

\item{start_year}{Numeric value of the first year to consider for analysis. Leave blank to use the first year of the source data.}

\item{end_year}{Numeric value of the last year to consider for analysis. Leave blank to use the last year of the source data.}

\item{exclude_years}{Numeric vector of years to exclude from analysis. Leave blank to include all years.}

\item{months}{Numeric vector of months to include in analysis (e.g. \code{6:8} for Jun-Aug). Leave blank to summarize 
all months (default \code{1:12}).}

\item{transpose}{Logical value indicating whether to transpose rows and columns of results. Default \code{FALSE}.}

\item{ignore_missing}{Logical value indicating whether dates with missing values should be included in the calculation. If
\code{TRUE} then a statistic will be calculated regardless of missing dates. If \code{FALSE} then only those statistics from
time periods with no missing dates will be returned. Default \code{FALSE}.}

\item{allowed_missing}{Numeric value between 0 and 100 indicating the \strong{percentage} of missing dates allowed to be
included to calculate a statistic (0 to 100 percent). If \code{'ignore_missing = FALSE'} then it defaults to \code{0} (zero missing dates allowed),
if \code{'ignore_missing = TRUE'} then it defaults to \code{100} (any missing dates allowed); consistent with 
\code{ignore_missing} usage. Supersedes \code{ignore_missing} when used.}
}
\value{
A tibble data frame with the following columns:
  \item{Year}{calendar or water year selected}
  \item{Mean}{annual mean of all daily flows for a given year}
  \item{Median}{annual median of all daily flows for a given year}
  \item{Maximum}{annual maximum of all daily flows for a given year}
  \item{Minimum}{annual minimum of all daily flows for a given year}
  \item{P'n'}{each annual n-th percentile selected of all daily flows}
  Default percentile columns:
  \item{P10}{annual 10th percentile of all daily flows for a given year}
  \item{P90}{annual 90th percentile of all daily flows for a given year}
  Transposing data creates a column of "Statistics" and subsequent columns for each year selected.
}
\description{
Calculates means, medians, maximums, minimums, and percentiles for each year from all years of a daily streamflow 
   data set. Calculates statistics from all values, unless specified. Returns a tibble with statistics.
}
\examples{
# Run if HYDAT database has been downloaded (using tidyhydat::download_hydat())
if (file.exists(tidyhydat::hy_downloaded_db())) {

# Calculate annual statistics from a data frame using the data argument
flow_data <- tidyhydat::hy_daily_flows(station_number = "08NM116")
calc_annual_stats(data = flow_data)

# Calculate annual statistics using station_number argument
calc_annual_stats(station_number = "08NM116")

# Calculate annual statistics regardless if there 
# is missing data for a given year
calc_annual_stats(station_number = "08NM116",
                  ignore_missing = TRUE)
                  
# Calculate annual statistics for water years starting in October
calc_annual_stats(station_number = "08NM116",
                  water_year_start = 10)
                  
# Calculate annual statistics filtered for custom years
calc_annual_stats(station_number = "08NM116",
                  start_year = 1981,
                  end_year = 2010,
                  exclude_years = c(1991,1993:1995))
                  
# Calculate annual statistics for 7-day flows for July-September 
# months only, with 25 and 75th percentiles
calc_annual_stats(station_number = "08NM116",
                  roll_days = 7,
                  months = 7:9,
                  percentiles = c(25,75))
                  
}
}
