
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful, 
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR Description. See the 
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU Library General 
# Public License along with this library; if not, write to the 
# Free Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
# MA 02111-1307 USA

# Copyrights (C)
# for this R-port: 
#   1999 - 2006, Diethelm Wuertz, GPL
#   Diethelm Wuertz <wuertz@itp.phys.ethz.ch>
#   info@rmetrics.org
#   www.rmetrics.org
# for the code accessed (or partly included) from other R-ports:
#   see R's copyright and license files
# for the code accessed (or partly included) from contributed R-ports
# and other sources
#   see Rmetrics's copyright file


################################################################################
# FUNCTION:          DICKEY FULLER TEST:
#  pdftest            Returns probabilities for the ADF Test
#  qdftest            Returns quantiles for the ADF Test
#  .dfTable           Augmented Dickey-Fuller finite sample test table
# FUNCTION:          PROBABILIY AND QUANTILES:
#  punitroot          Returns cumulative probability for unit root distributions
#  qunitroot          Returns quantiles for unit root distributions
# INTERNAL:          UTILITY FUNCTIONS:
#  .strsplitUrcval    String split function for S-Plus compatibility
#  .urcval            Implements unit root statists
#  .probsUrcval       Probability values
#  .urc1 ... .urc12   Statistical values
################################################################################


################################################################################
# Functions required from fBasics::Hypothesis Testing
# FUNCTION:          NORMALITY TESTS:
#  fHTEST             Class Representation
#  show.fHTEST        S4 Print Method
# FUNCTION:          PVALUE AND STATISTICS TABLES:
#  pPlot              General finite sample probability plot
#  pTable             Interpolated probabilities from finite sample table
#  .pTable            Utility function called by the function 'pTable'
#  qTable             Interpolated quantiles from finite sample table
#  .qTable            Utility function called by the function 'qTable'
# FUNCTION:          INTERNAL FUNCTIONS:
#  .interpTable.old   'akima' interpolation utility function
#  .interpTable.new   'akima' interpolation utility function
################################################################################


################################################################################
# AUGMENTED DICKEY-FULLER TEST:


pdftest = 
function(q, n.sample, trend = c("nc", "c", "ct"), statistic = c("t", "n")) 
{   # A function implemented by Diethelm Wuertz

    # Description:
    #   Probabilities for the ADF Test
    
    # Arguments:
    
    # FUNCTION:
    
    # Compute Probabilities:
    X = .dfTable(trend = trend[1], statistic = statistic[1])
    ans = pTable(t(X), q, n.sample)
    
    # Return Value:
    ans
} 


# ------------------------------------------------------------------------------


qdftest = 
function(p, n.sample, trend = c("nc", "c", "ct"), statistic = c("t", "n"))
{   # A function implemented by Diethelm Wuertz

    # Description:
    #   Quantiles for the ADF Test
    
    # Arguments:
    
    # FUNCTION:
    
    # Compute Quantiles:
    X = .dfTable(trend = trend[1], statistic = statistic[1])
    ans = qTable(X = t(X), p, n.sample)
    
    # Return Value:
    ans
} 


# ------------------------------------------------------------------------------


.dfTable =
function(trend = c("nc", "c", "ct"), statistic = c("t", "n"))
{   # A function implemented by Diethelm Wuertz

    # Description:     
    #   Critical Values for the ADF test.
    
    # FUNCTION:
      
    # Select Type:
    type = trend[1]
    statistic = statistic[1]
    
    # Tables:
    if (statistic == "t") {
        # Hamilton Table B.6 - OLS t-Statistic
        if (type == "nc") {
            table = cbind(
                c(-2.66, -2.26, -1.95, -1.60, +0.92, +1.33, +1.70, +2.16),
                c(-2.62, -2.25, -1.95, -1.61, +0.91, +1.31, +1.66, +2.08),
                c(-2.60, -2.24, -1.95, -1.61, +0.90, +1.29, +1.64, +2.03),
                c(-2.58, -2.23, -1.95, -1.62, +0.89, +1.29, +1.63, +2.01),
                c(-2.58, -2.23, -1.95, -1.62, +0.89, +1.28, +1.62, +2.00),
                c(-2.58, -2.23, -1.95, -1.62, +0.89, +1.28, +1.62, +2.00))
        } else if (type == "c") {
            table = cbind(
                c(-3.75, -3.33, -3.00, -2.63, -0.37, +0.00, +0.34, +0.72),
                c(-3.58, -3.22, -2.93, -2.60, -0.40, -0.03, +0.29, +0.66),
                c(-3.51, -3.17, -2.89, -2.58, -0.42, -0.05, +0.26, +0.63),
                c(-3.46, -3.14, -2.88, -2.57, -0.42, -0.06, +0.24, +0.62),
                c(-3.44, -3.13, -2.87, -2.57, -0.43, -0.07, +0.24, +0.61),
                c(-3.43, -3.12, -2.86, -2.57, -0.44, -0.07, +0.23, +0.60))
        } else if (type == "ct") {
            table = cbind(
                c(-4.38, -3.95, -3.60, -3.24, -1.14, -0.80, -0.50, -0.15),
                c(-4.15, -3.80, -3.50, -3.18, -1.19, -0.87, -0.58, -0.24),
                c(-4.04, -3.73, -3.45, -3.15, -1.22, -0.90, -0.62, -0.28),
                c(-3.99, -3.69, -3.43, -3.13, -1.23, -0.92, -0.64, -0.31),
                c(-3.98, -3.68, -3.42, -3.13, -1.24, -0.93, -0.65, -0.32),
                c(-3.96, -3.66, -3.41, -3.12, -1.25, -0.94, -0.66, -0.33))      
        } else {
            stop("Invalid type specified")
        }
    } else if (statistic == "z" || statistic == "n") {
        # Hamilton Table B.5 - Based on OLS Autoregressive Coefficient
        if (type == "nc") {
            table = cbind(
                c(-11.9,  -9.3,  -7.3,  -5.3, +1.01, +1.40, +1.79, +2.28),
                c(-12.9,  -9.9,  -7.7,  -5.5, +0.97, +1.35, +1.70, +2.16),
                c(-13.3, -10.2,  -7.9,  -5.6, +0.95, +1.31, +1.65, +2.09),
                c(-13.6, -10.3,  -8.0,  -5.7, +0.93, +1.28, +1.62, +2.04),
                c(-13.7, -10.4,  -8.0,  -5.7, +0.93, +1.28, +1.61, +2.04),
                c(-13.8, -10.5,  -8.1,  -5.7, +0.93, +1.28, +1.60, +2.03))
        } else if (type == "c") {
            table = cbind(
                c(-17.2, -14.6, -12.5, -10.2, -0.76, +0.01, +0.65, +1.40),
                c(-18.9, -15.7, -13.3, -10.7, -0.81, -0.07, +0.53, +1.22),
                c(-19.8, -16.3, -13.7, -11.0, -0.83, -0.10, +0.47, +1.14),
                c(-20.3, -16.6, -14.0, -11.2, -0.84, -0.12, +0.43, +1.09),
                c(-20.5, -16.8, -14.0, -11.2, -0.84, -0.13, +0.42, +1.06),
                c(-20.7, -16.9, -14.1, -11.3, -0.85, -0.13, +0.41, +1.04))
        } else if (type == "ct") {
            table = cbind(
                c(-22.5, -19.9, -17.9, -15.6, -3.66, -2.51, -1.53, -0.43),
                c(-25.7, -22.4, -19.8, -16.8, -3.71, -2.60, -1.66, -0.65),
                c(-27.4, -23.6, -20.7, -17.5, -3.74, -2.62, -1.73, -0.75),
                c(-28.4, -24.4, -21.3, -18.0, -3.75, -2.64, -1.78, -0.82),
                c(-28.9, -24.8, -21.5, -18.1, -3.76, -2.65, -1.78, -0.84),
                c(-29.5, -25.1, -21.8, -18.3, -3.77, -2.66, -1.79, -0.87))      
        } else {
            stop("Invalid type specified")
        }
    } else {
        stop("Invalid statistic specified")
    }
            
    # Transpose:
    table = t(table)
    colnames(table) = c("0.010", "0.025", "0.050", "0.100", "0.900", 
        "0.950", "0.975", "0.990")
    rownames(table) = c(" 25", " 50", "100", "250", "500", "Inf")
    
    # Return Value:
    as.data.frame(table)
} 


################################################################################
# PROBABILIY AND QUANTILES:


punitroot =
function(q, n.sample = 0, trend = c("c", "nc", "ct", "ctt"), 
statistic = c("t", "n"), na.rm = FALSE) 
{   # A function implemented by Diethelm Wuertz

    # Description:
    #   Returns the cumulative probability for unit root distributions
    
    # Arguments:
    #   q - vector of quantiles 
    #   n.sample - a numerical values giving the number of observations 
    #       from which the test statistics are computed. The default 
    #       value is 0 which means that asymptotic cumulative probabilities  
    #       will be returned.
    #   trend - a character string describing the regression type underlying
    #       the computation of the statistics. Selections are 
    #       "nc" for a constant regression no intercept nor time trend,
    #       "c" for a regression with an intercept but no trend component,
    #       "ct" for a regression including intercept and trend. 
    #       The default value is "c". 
    #   statistic - a character string denoting the type of the test 
    #       statistic. The default choice is "t" for t-statistic, and 
    #       and the alternative choice is "n" for normalized statistic 
    #       also named as rho-statistic.
    #   na.rm - a logical value. If set to TRUE, missing values will 
    #       be removed, the default is FALSE. 
    
    # Value:
    #   A vector of probabilities for the unit root distribution.
    
    # FUNCTION:

    # Settings:
    trend = trend[1]
    statistic = statistic[1]
    
    # Remove any NA's in q:
    if(any(is.na(q))) {
        if(!na.rm) stop("NAs are not allowed when na.rm=FALSE")
        else q = na.omit(q) }

    # Settings - Will be Checked in Routine urcval:
    if (statistic == "t") itt = 1
    if (statistic == "n") itt = 2
    if (trend == "nc")    itv = 1
    if (trend == "c")     itv = 2
    if (trend == "ct")    itv = 3
    if (trend == "ctt")   itv = 4
    
    # Calculate - Call "urcval":
    result = 0 * q
    for (i in 1:length(q)) {
        result[i] = .urcval(arg = q[i], nobs = n.sample, niv = 1, 
            itt = itt, itv = itv, nc = 2) }
            
    # Return Value:
    result  
}


# ------------------------------------------------------------------------------


qunitroot =
function(p, n.sample = 0, trend = c("c", "nc", "ct", "ctt"), 
statistic = c("t", "n"), na.rm = FALSE)
{   # A function implemented by Diethelm Wuertz

    # Description:
    #   Returns the quantile for unit root distributions
    
    # Arguments:
    #   q - vector of quantiles 
    #   n.sample - a numerical values giving the number of observations 
    #       from which the test statistics are computed. The default 
    #       value is 0 which means that asymptotic cumulative probabilities  
    #       will be returned.
    #   trend - a character string describing the regression type underlying
    #       the computation of the statistics. Selections are 
    #       "nc" for a constant regression no intercept nor time trend,
    #       "c" for a regression with an intercept but no trend component,
    #       "ct" for a regression including intercept and trend. 
    #       The default value is "c". 
    #   statistic - a character string denoting the type of the test 
    #       statistic. The default choice is "t" for t-statistic, and 
    #       and the alternative choice is "n" for normalized statistic 
    #       also named as rho-statistic.
    #   na.rm - a logical value. If set to TRUE, missing values will 
    #       be removed, the default is FALSE. 

    # Value:
    #   A vector of quantiles for the unit root distribution.
    
    # FUNCTION:
    
    # Settings:
    trend = trend[1]
    statistic = statistic[1]
    
    # Remove any NA's in p:
    if (any(is.na(p))) {
        if(!na.rm) 
            stop("NAs are not allowed when na.rm = FALSE")
        else 
            p = na.omit(p) 
    }

    # Settings - Will be Checked in Routine urcval:
    if (statistic == "t") itt = 1
    if (statistic == "n") itt = 2
    if (trend == "nc")    itv = 1
    if (trend == "c")     itv = 2
    if (trend == "ct")    itv = 3
    if (trend == "ctt")   itv = 4
    
    # Calculate:
    result = 0*p
    for (i in 1:length(p)) {
        result[i] = .urcval(arg = p[i], nobs = n.sample, niv = 1, 
            itt = itt, itv = itv, nc = 1) }
            
    # Return Value:
    result  
}


################################################################################
# UTILITY FUNCTIONS:

.strsplitUrcval = 
function(x, sep = " ") 
{   # A function implemented by Diethelm Wuertz

    # Description:
    #   strsplit function for SPlus compatibility
    
    # FUNCTION:
    
    # Split:
    if (exists("strsplit")) {
        # R:
        ans = strsplit(x = x , split = sep)
    } else {
        # Splus:
        ans = lapply(lapply(X = x, FUN = unpaste, sep = sep), unlist)
    }
    
    # Return Value:
    ans 
}
    
    
# ------------------------------------------------------------------------------


.urcval = 
function (arg, nobs = 0, niv = 1, itt = 1, itv = 1, nc = 1)
{   # A function implemented by Diethelm Wuertz

    # Description:
    #   Implements the unit root statists distribution 
    
    # Arguments: 
    #   arg - level of test (between .0001 and .9999) if nc = 1, 
    #       else test statistic if nc = 2
    #   nobs - sample size (0 for asymptotic)
    #   niv - number of integrated variables
    #   itt - 1 or 2 for tau or z test
    #   itv - 1, 2, 3, 4 for nc, c, ct, ctt
    #   nc - 1 or 2 for critical value or P value
    
    # Value:
    #   val - critical value if nc = 1 (returned by routine), 
    #       else P value if nc = 2 (returned by routine)
    
    # FUNCTION:
    
    # Read Probs Table:
    # probs = read.table("library/fSeries/libs/.urcProbs.tab")
    probs = .probsUrcval
    cnorm = probs[, 2]
    probs = probs[, 1]
    
    # Read from appropriate urc Table:
    # fnames = 'urc-1.tab' .... 'urc-12.tab'  (niv) 
    # Skip copyright line and groups of 222 lines as necessary
    skip = 886*(itt-1) + 2*itt + (itv-1)*222
   
    # Filename:
    # root = paste(.tableDir(), "/.urc", sep = "")
    # fname = paste(root, as.character(niv), ".tab", sep = "")
    
    # Get Data:
    TABLE = get(paste(".urc", as.character(niv), sep = ""))
    
    # Read Parameters:
    # parms = scan(fname, what = "", skip = skip - 1, nlines = 1, quiet = TRUE)   
    parms = .strsplitUrcval(TABLE[skip], sep = " ")[[1]]
    parms = parms[parms != ""]
    nz = as.numeric(parms[2])
    nreg = as.numeric(parms[3])
    model = as.numeric(parms[4])
    minsize = as.numeric(parms[5])
    
    # Read Data Group:
    nvar = 4
    if (model == 2 || model == 4) nvar = 3
    nlines = 221 
    # scan(fname, skip = skip, nlines = 221, quiet = TRUE)
    # beta = matrix(SCAN, byrow = TRUE, ncol = nvar+1) 
    beta = matrix(TABLE[(skip+1):(skip+nlines)], ncol = 1)
    beta = unlist(apply(beta, 2, .strsplitUrcval))
    beta = beta[beta != ""]
    beta = matrix(as.numeric(beta), byrow = TRUE, nrow = nlines) 
    wght = t(beta[, nvar+1])
    beta = t(beta[, 1:4])
    
    # Check for Number of Observations:
    if (nobs != 0 && nobs < minsize)
        print("Warning: Specified sample size may be too small")
        
    # Urcvals:
    ans = NULL
    for (i in 1:length(arg)) {
        # Set size and stat and check for level 
        size = stat = arg[i]
        # if (nc == 1) 
        #    stop("Level must be between .0001 and .9999")   
        
        # Calculate: 
        np = 9
        cval = pval = 0.0
        precrt = 2.0
        
        # Call J. G. McKinnon's Fortran Subroutines:
        if (nc == 1) {
            if (min(size) < 0.0001 || max(size) > 0.9999) {
                ans = c(ans, NA) 
            } else {
                result = .Fortran("fcrit", 
                    as.double(probs), as.double(cnorm), as.double(beta), 
                    as.double(wght), as.double(cval), as.double(size), 
                    as.double(precrt), as.integer(nobs), as.integer(model), 
                    as.integer(nreg), as.integer(np), as.integer(0),
                    PACKAGE = "fSeries") 
                ans = c(ans, result[[5]]) 
            } 
        }
        if (nc == 2) {
            result = .Fortran("fpval", 
                as.double(beta), as.double(cnorm), as.double(wght), 
                as.double(probs), as.double(pval), as.double(stat), 
                as.double(precrt), as.integer(nobs), as.integer(model), 
                as.integer(nreg), as.integer(np), as.integer(0),
                PACKAGE = "fSeries") 
            ans = c(ans, result[[5]]) 
        } 
    }
    
    # Return Value:
    ans 
}


# ------------------------------------------------------------------------------


.probsUrcval = 
structure(list(V1 = c(1.0e-04, 2.0e-04, 5.0e-04, 0.001, 0.002, 0.003, 
    0.004, 0.005, 0.006, 0.007, 0.008, 0.009, 0.01, 0.015, 0.02, 
    0.025, 0.03, 0.035, 0.04, 0.045, 0.05, 0.055, 0.06, 0.065, 0.07, 
    0.075, 0.08, 0.085, 0.09, 0.095, 0.1, 0.105, 0.11, 0.115, 0.12, 
    0.125, 0.13, 0.135, 0.14, 0.145, 0.15, 0.155, 0.16, 0.165, 0.17, 
    0.175, 0.18, 0.185, 0.19, 0.195, 0.2, 0.205, 0.21, 0.215, 0.22, 
    0.225, 0.23, 0.235, 0.24, 0.245, 0.25, 0.255, 0.26, 0.265, 0.27, 
    0.275, 0.28, 0.285, 0.29, 0.295, 0.3, 0.305, 0.31, 0.315, 0.32, 
    0.325, 0.33, 0.335, 0.34, 0.345, 0.35, 0.355, 0.36, 0.365, 0.37, 
    0.375, 0.38, 0.385, 0.39, 0.395, 0.4, 0.405, 0.41, 0.415, 0.42, 
    0.425, 0.43, 0.435, 0.44, 0.445, 0.45, 0.455, 0.46, 0.465, 0.47, 
    0.475, 0.48, 0.485, 0.49, 0.495, 0.5, 0.505, 0.51, 0.515, 0.52, 
    0.525, 0.53, 0.535, 0.54, 0.545, 0.55, 0.555, 0.56, 0.565, 0.57, 
    0.575, 0.58, 0.585, 0.59, 0.595, 0.6, 0.605, 0.61, 0.615, 0.62, 
    0.625, 0.63, 0.635, 0.64, 0.645, 0.65, 0.655, 0.66, 0.665, 0.67, 
    0.675, 0.68, 0.685, 0.69, 0.695, 0.7, 0.705, 0.71, 0.715, 0.72, 
    0.725, 0.73, 0.735, 0.74, 0.745, 0.75, 0.755, 0.76, 0.765, 0.77, 
    0.775, 0.78, 0.785, 0.79, 0.795, 0.8, 0.805, 0.81, 0.815, 0.82, 
    0.825, 0.83, 0.835, 0.84, 0.845, 0.85, 0.855, 0.86, 0.865, 0.87, 
    0.875, 0.88, 0.885, 0.89, 0.895, 0.9, 0.905, 0.91, 0.915, 0.92, 
    0.925, 0.93, 0.935, 0.94, 0.945, 0.95, 0.955, 0.96, 0.965, 0.97, 
    0.975, 0.98, 0.985, 0.99, 0.991, 0.992, 0.993, 0.994, 0.995, 
    0.996, 0.997, 0.998, 0.999, 0.9995, 0.9998, 0.9999), V2 = c(-3.71901649, 
    -3.5400838, -3.29052673, -3.09023231, -2.87816174, -2.74778139, 
    -2.65206981, -2.5758293, -2.51214433, -2.45726339, -2.40891555, 
    -2.36561813, -2.32634787, -2.17009038, -2.05374891, -1.95996398, 
    -1.88079361, -1.81191067, -1.75068607, -1.69539771, -1.64485363, 
    -1.59819314, -1.55477359, -1.51410189, -1.47579103, -1.43953147, 
    -1.40507156, -1.37220381, -1.34075503, -1.31057911, -1.28155157, 
    -1.25356544, -1.22652812, -1.20035886, -1.17498679, -1.15034938, 
    -1.12639113, -1.10306256, -1.08031934, -1.05812162, -1.03643339, 
    -1.01522203, -0.99445788, -0.97411388, -0.95416525, -0.93458929, 
    -0.91536509, -0.89647336, -0.8778963, -0.85961736, -0.84162123, 
    -0.82389363, -0.80642125, -0.78919165, -0.77219321, -0.75541503, 
    -0.73884685, -0.72247905, -0.70630256, -0.69030882, -0.67448975, 
    -0.65883769, -0.64334541, -0.62800601, -0.61281299, -0.59776013, 
    -0.58284151, -0.5680515, -0.55338472, -0.53883603, -0.52440051, 
    -0.51007346, -0.49585035, -0.48172685, -0.4676988, -0.45376219, 
    -0.43991317, -0.42614801, -0.41246313, -0.39885507, -0.38532047, 
    -0.37185609, -0.35845879, -0.34512553, -0.33185335, -0.31863936, 
    -0.30548079, -0.2923749, -0.27931903, -0.26631061, -0.2533471, 
    -0.24042603, -0.22754498, -0.21470157, -0.20189348, -0.18911843, 
    -0.17637416, -0.16365849, -0.15096922, -0.13830421, -0.12566135, 
    -0.11303854, -0.10043372, -0.08784484, -0.07526986, -0.06270678, 
    -0.05015358, -0.03760829, -0.02506891, -0.01253347, 0, 0.01253347, 
    0.02506891, 0.03760829, 0.05015358, 0.06270678, 0.07526986, 0.08784484, 
    0.10043372, 0.11303854, 0.12566135, 0.13830421, 0.15096922, 0.16365849, 
    0.17637416, 0.18911843, 0.20189348, 0.21470157, 0.22754498, 0.24042603, 
    0.2533471, 0.26631061, 0.27931903, 0.2923749, 0.30548079, 0.31863936, 
    0.33185335, 0.34512553, 0.35845879, 0.37185609, 0.38532047, 0.39885507, 
    0.41246313, 0.42614801, 0.43991317, 0.45376219, 0.4676988, 0.48172685, 
    0.49585035, 0.51007346, 0.52440051, 0.53883603, 0.55338472, 0.5680515, 
    0.58284151, 0.59776013, 0.61281299, 0.62800601, 0.64334541, 0.65883769, 
    0.67448975, 0.69030882, 0.70630256, 0.72247905, 0.73884685, 0.75541503, 
    0.77219321, 0.78919165, 0.80642125, 0.82389363, 0.84162123, 0.85961736, 
    0.8778963, 0.89647336, 0.91536509, 0.93458929, 0.95416525, 0.97411388, 
    0.99445788, 1.01522203, 1.03643339, 1.05812162, 1.08031934, 1.10306256, 
    1.12639113, 1.15034938, 1.17498679, 1.20035886, 1.22652812, 1.25356544, 
    1.28155157, 1.31057911, 1.34075503, 1.37220381, 1.40507156, 1.43953147, 
    1.47579103, 1.51410189, 1.55477359, 1.59819314, 1.64485363, 1.69539771, 
    1.75068607, 1.81191067, 1.88079361, 1.95996398, 2.05374891, 2.17009038, 
    2.32634787, 2.36561813, 2.40891555, 2.45726339, 2.51214433, 2.5758293, 
    2.65206981, 2.74778139, 2.87816174, 3.09023231, 3.29052673, 3.5400838, 
    3.71901649)), .Names = c("V1", "V2"), class = "data.frame", 
    row.names = c("1", 
    "2", "3", "4", "5", "6", "7", "8", "9", "10", "11", "12", "13", 
    "14", "15", "16", "17", "18", "19", "20", "21", "22", "23", "24", 
    "25", "26", "27", "28", "29", "30", "31", "32", "33", "34", "35", 
    "36", "37", "38", "39", "40", "41", "42", "43", "44", "45", "46", 
    "47", "48", "49", "50", "51", "52", "53", "54", "55", "56", "57", 
    "58", "59", "60", "61", "62", "63", "64", "65", "66", "67", "68", 
    "69", "70", "71", "72", "73", "74", "75", "76", "77", "78", "79", 
    "80", "81", "82", "83", "84", "85", "86", "87", "88", "89", "90", 
    "91", "92", "93", "94", "95", "96", "97", "98", "99", "100", 
    "101", "102", "103", "104", "105", "106", "107", "108", "109", 
    "110", "111", "112", "113", "114", "115", "116", "117", "118", 
    "119", "120", "121", "122", "123", "124", "125", "126", "127", 
    "128", "129", "130", "131", "132", "133", "134", "135", "136", 
    "137", "138", "139", "140", "141", "142", "143", "144", "145", 
    "146", "147", "148", "149", "150", "151", "152", "153", "154", 
    "155", "156", "157", "158", "159", "160", "161", "162", "163", 
    "164", "165", "166", "167", "168", "169", "170", "171", "172", 
    "173", "174", "175", "176", "177", "178", "179", "180", "181", 
    "182", "183", "184", "185", "186", "187", "188", "189", "190", 
    "191", "192", "193", "194", "195", "196", "197", "198", "199", 
    "200", "201", "202", "203", "204", "205", "206", "207", "208", 
    "209", "210", "211", "212", "213", "214", "215", "216", "217", 
    "218", "219", "220", "221")
)


# ------------------------------------------------------------------------------


.urc1 = 
c("Copyright (C) James G. MacKinnon, 1995", "dfnc      0  1  2   20", 
" -.38929681e+01 -.10711437e+02 -.55619955e+02  .33188192e-02", 
" -.37185872e+01 -.95896921e+01 -.34166532e+02  .24951489e-02", 
" -.34759164e+01 -.78746051e+01 -.17490657e+02  .15559156e-02", 
" -.32842708e+01 -.64108912e+01 -.11983755e+02  .11445964e-02", 
" -.30830457e+01 -.50567633e+01 -.68847143e+01  .84100765e-03", 
" -.29598602e+01 -.42584716e+01 -.58142463e+01  .71049934e-03", 
" -.28698825e+01 -.37289550e+01 -.58714971e+01  .62391151e-03", 
" -.27977077e+01 -.34146427e+01 -.37615300e+01  .58693244e-03", 
" -.27380050e+01 -.31208498e+01 -.30462883e+01  .52576407e-03", 
" -.26868436e+01 -.28722476e+01 -.25438563e+01  .49052976e-03", 
" -.26416852e+01 -.26793402e+01 -.18769783e+01  .46804316e-03", 
" -.26012597e+01 -.25000004e+01 -.16307933e+01  .44346134e-03", 
" -.25649446e+01 -.23127399e+01 -.20306605e+01  .43039003e-03", 
" -.24206178e+01 -.17291886e+01 -.13466493e+01  .36467475e-03", 
" -.23133304e+01 -.13423615e+01 -.13432257e+01  .32107164e-03", 
" -.22273143e+01 -.10746264e+01 -.87624270e+00  .29789755e-03", 
" -.21549047e+01 -.84657628e+00 -.10780188e+01  .27830583e-03", 
" -.20921655e+01 -.67744316e+00 -.10430697e+01  .26027602e-03", 
" -.20364584e+01 -.54192487e+00 -.96965194e+00  .25249134e-03", 
" -.19864146e+01 -.41616890e+00 -.11334045e+01  .24079995e-03", 
" -.19407684e+01 -.33025231e+00 -.67821426e+00  .23323022e-03", 
" -.18988585e+01 -.22970773e+00 -.85640849e+00  .22075141e-03", 
" -.18598488e+01 -.14639728e+00 -.10428806e+01  .21334955e-03", 
" -.18233573e+01 -.89663803e-01 -.80855945e+00  .20963010e-03", 
" -.17891398e+01 -.36466865e-01 -.63243896e+00  .20834232e-03", 
" -.17567993e+01  .11402654e-01 -.51066497e+00  .20539933e-03", 
" -.17262105e+01  .66220956e-01 -.63708003e+00  .19964762e-03", 
" -.16969838e+01  .10236824e+00 -.46672787e+00  .19574132e-03", 
" -.16690859e+01  .14265332e+00 -.45061418e+00  .18888062e-03", 
" -.16424271e+01  .18701823e+00 -.66084335e+00  .18426555e-03", 
" -.16167279e+01  .21626789e+00 -.64616991e+00  .18128089e-03", 
" -.15920520e+01  .24778988e+00 -.63597502e+00  .17954902e-03", 
" -.15682114e+01  .27323562e+00 -.61618157e+00  .17773495e-03", 
" -.15452740e+01  .29991006e+00 -.60544565e+00  .17142168e-03", 
" -.15230643e+01  .33148333e+00 -.76580637e+00  .16769202e-03", 
" -.15014865e+01  .35136387e+00 -.69024691e+00  .16887147e-03", 
" -.14805700e+01  .37478667e+00 -.72930388e+00  .16872218e-03", 
" -.14602522e+01  .39420657e+00 -.70443601e+00  .16401731e-03", 
" -.14404242e+01  .41028681e+00 -.64338518e+00  .16224607e-03", 
" -.14210785e+01  .42119200e+00 -.55548400e+00  .16193196e-03", 
" -.14022406e+01  .43747746e+00 -.57670820e+00  .16097849e-03", 
" -.13837876e+01  .44575298e+00 -.43236279e+00  .15803832e-03", 
" -.13657717e+01  .45528883e+00 -.35165569e+00  .15694908e-03", 
" -.13481032e+01  .46306181e+00 -.31093184e+00  .15603528e-03", 
" -.13309144e+01  .47972473e+00 -.39121260e+00  .15557033e-03", 
" -.13139547e+01  .48814175e+00 -.38006723e+00  .15236727e-03", 
" -.12973931e+01  .49859437e+00 -.35456698e+00  .14909593e-03", 
" -.12811030e+01  .50420631e+00 -.25583170e+00  .14793470e-03", 
" -.12651262e+01  .51429788e+00 -.28893557e+00  .14655593e-03", 
" -.12494004e+01  .52135123e+00 -.27882309e+00  .14412602e-03", 
" -.12339522e+01  .52756595e+00 -.23217506e+00  .14437838e-03", 
" -.12187168e+01  .53480958e+00 -.26406268e+00  .14432408e-03", 
" -.12037702e+01  .54338564e+00 -.29729024e+00  .14262955e-03", 
" -.11890155e+01  .54817266e+00 -.24900712e+00  .14111421e-03", 
" -.11744677e+01  .55263933e+00 -.22129035e+00  .14067648e-03", 
" -.11601444e+01  .55919565e+00 -.23623151e+00  .13973685e-03", 
" -.11460177e+01  .56517698e+00 -.24565447e+00  .13926384e-03", 
" -.11320096e+01  .56603215e+00 -.17524226e+00  .13969780e-03", 
" -.11182292e+01  .57011095e+00 -.19439876e+00  .13916343e-03", 
" -.11045866e+01  .57209736e+00 -.16074907e+00  .13818167e-03", 
" -.10911074e+01  .57234850e+00 -.10524258e+00  .13722018e-03", 
" -.10777556e+01  .57229676e+00 -.32550732e-01  .13619288e-03", 
" -.10645684e+01  .57302396e+00  .14379749e-01  .13553410e-03", 
" -.10515244e+01  .57362590e+00  .84639180e-01  .13526527e-03", 
" -.10386317e+01  .57725998e+00  .73590640e-01  .13519196e-03", 
" -.10258492e+01  .58044340e+00  .51060038e-01  .13379316e-03", 
" -.10131700e+01  .58385933e+00  .61245025e-02  .13428360e-03", 
" -.10006270e+01  .58959535e+00 -.64975446e-01  .13403253e-03", 
" -.98816340e+00  .58962904e+00 -.38969170e-01  .13388645e-03", 
" -.97580892e+00  .59228543e+00 -.92028608e-01  .13256283e-03", 
" -.96353988e+00  .59023118e+00 -.25245537e-01  .13350108e-03", 
" -.95135437e+00  .59235081e+00 -.66927698e-01  .13166673e-03", 
" -.93927127e+00  .59495523e+00 -.11772564e+00  .13109874e-03", 
" -.92725721e+00  .59465251e+00 -.98842151e-01  .13075847e-03", 
" -.91538294e+00  .59862948e+00 -.13373870e+00  .12991997e-03", 
" -.90351591e+00  .59736071e+00 -.11575297e+00  .13004651e-03", 
" -.89171740e+00  .59827558e+00 -.11376391e+00  .12981779e-03", 
" -.87999358e+00  .59854644e+00 -.11262976e+00  .12937930e-03", 
" -.86836393e+00  .60125923e+00 -.15700540e+00  .12809482e-03", 
" -.85676630e+00  .60235243e+00 -.15341154e+00  .12727490e-03", 
" -.84521821e+00  .60724747e+00 -.28142705e+00  .12854899e-03", 
" -.83368446e+00  .60414764e+00 -.22759417e+00  .12886527e-03", 
" -.82217754e+00  .59882470e+00 -.10369380e+00  .12729110e-03", 
" -.81077477e+00  .60266754e+00 -.19236920e+00  .12744791e-03", 
" -.79938013e+00  .60203622e+00 -.15455660e+00  .12851757e-03", 
" -.78799780e+00  .60310963e+00 -.21236507e+00  .12847280e-03", 
" -.77666556e+00  .60512141e+00 -.24195949e+00  .12867783e-03", 
" -.76531152e+00  .60161040e+00 -.18435951e+00  .12764627e-03", 
" -.75400509e+00  .60005644e+00 -.15758711e+00  .12719162e-03", 
" -.74272550e+00  .60263001e+00 -.19073741e+00  .12737733e-03", 
" -.73140950e+00  .60169755e+00 -.16809940e+00  .12901165e-03", 
" -.72010821e+00  .60073899e+00 -.17118517e+00  .12989901e-03", 
" -.70881438e+00  .60281341e+00 -.19734629e+00  .13096756e-03", 
" -.69752254e+00  .60574100e+00 -.23900570e+00  .13200027e-03", 
" -.68621242e+00  .60709003e+00 -.25007855e+00  .13205742e-03", 
" -.67486899e+00  .60595186e+00 -.21780355e+00  .13308173e-03", 
" -.66355310e+00  .60879991e+00 -.23907614e+00  .13334734e-03", 
" -.65217906e+00  .60934490e+00 -.22402739e+00  .13313113e-03", 
" -.64078470e+00  .61045036e+00 -.22525697e+00  .13423481e-03", 
" -.62935668e+00  .61165455e+00 -.18555733e+00  .13561164e-03", 
" -.61784992e+00  .60906224e+00 -.10217160e+00  .13524395e-03", 
" -.60634066e+00  .61133645e+00 -.93563312e-01  .13574028e-03", 
" -.59479908e+00  .61243094e+00 -.18009253e-01  .13677974e-03", 
" -.58317131e+00  .61383424e+00  .33976678e-01  .13851627e-03", 
" -.57147764e+00  .61383949e+00  .12970947e+00  .13919280e-03", 
" -.55974664e+00  .61815202e+00  .13567448e+00  .13865735e-03", 
" -.54796415e+00  .62425007e+00  .12681084e+00  .14027742e-03", 
" -.53605781e+00  .62788652e+00  .13922660e+00  .14160362e-03", 
" -.52407823e+00  .63003047e+00  .20012740e+00  .14263868e-03", 
" -.51203077e+00  .63456809e+00  .22849852e+00  .14476471e-03", 
" -.49993482e+00  .64102940e+00  .23651226e+00  .14745515e-03", 
" -.48774646e+00  .64854569e+00  .19183910e+00  .14882984e-03", 
" -.47546258e+00  .65391229e+00  .20689462e+00  .15063090e-03", 
" -.46312562e+00  .66190450e+00  .15084854e+00  .15052565e-03", 
" -.45068236e+00  .67032467e+00  .10052172e+00  .15261383e-03", 
" -.43816571e+00  .67906278e+00  .18043158e-01  .15193656e-03", 
" -.42553012e+00  .68596166e+00 -.34373075e-01  .15324255e-03", 
" -.41278915e+00  .68751627e+00  .18499262e-02  .15211374e-03", 
" -.39996901e+00  .69262508e+00 -.44106098e-01  .15262241e-03", 
" -.38705587e+00  .69470103e+00 -.37223540e-01  .15233648e-03", 
" -.37407697e+00  .69915156e+00 -.68104682e-01  .15310849e-03", 
" -.36103600e+00  .70418582e+00 -.98476385e-01  .15417730e-03", 
" -.34792319e+00  .70730664e+00 -.10085193e+00  .15510072e-03", 
" -.33471267e+00  .71104064e+00 -.12671226e+00  .15539396e-03", 
" -.32142107e+00  .71478513e+00 -.15318544e+00  .15678198e-03", 
" -.30803148e+00  .71372891e+00 -.10938638e+00  .15886770e-03", 
" -.29463058e+00  .72056282e+00 -.22811198e+00  .15869168e-03", 
" -.28109052e+00  .72106303e+00 -.20822006e+00  .15986958e-03", 
" -.26754399e+00  .73003235e+00 -.37500115e+00  .16005206e-03", 
" -.25384762e+00  .73088340e+00 -.36414154e+00  .15982446e-03", 
" -.24009915e+00  .73329574e+00 -.39708747e+00  .16064215e-03", 
" -.22627489e+00  .73550500e+00 -.38861738e+00  .16028259e-03", 
" -.21236427e+00  .73562685e+00 -.37745695e+00  .15906987e-03", 
" -.19835027e+00  .73617467e+00 -.37412904e+00  .15881342e-03", 
" -.18428951e+00  .73644506e+00 -.34303983e+00  .16010471e-03", 
" -.17011954e+00  .73618375e+00 -.29890353e+00  .16091489e-03", 
" -.15586820e+00  .73373246e+00 -.22708090e+00  .16122387e-03", 
" -.14152198e+00  .73271165e+00 -.21753843e+00  .15985951e-03", 
" -.12710086e+00  .73048568e+00 -.16997032e+00  .15998493e-03", 
" -.11258798e+00  .72754793e+00 -.10077334e+00  .15998672e-03", 
" -.97976070e-01  .72884778e+00 -.10897580e+00  .16140512e-03", 
" -.83287262e-01  .73024638e+00 -.10172413e+00  .16110431e-03", 
" -.68499480e-01  .73431662e+00 -.18566360e+00  .16282783e-03", 
" -.53596308e-01  .73398093e+00 -.15607347e+00  .16572782e-03", 
" -.38591970e-01  .73389218e+00 -.17541939e+00  .16622391e-03", 
" -.23470382e-01  .73281912e+00 -.18774382e+00  .16697962e-03", 
" -.82414051e-02  .73243552e+00 -.16681816e+00  .16735938e-03", 
"  .70938188e-02  .73311643e+00 -.16926117e+00  .17001641e-03", 
"  .22558531e-01  .73323536e+00 -.16912639e+00  .17134410e-03", 
"  .38124040e-01  .73423643e+00 -.21631637e+00  .17166919e-03", 
"  .53822876e-01  .73695639e+00 -.24374285e+00  .17135216e-03", 
"  .69723974e-01  .73497901e+00 -.21283984e+00  .17178850e-03", 
"  .85748648e-01  .73353054e+00 -.22887618e+00  .17378309e-03", 
"  .10194776e+00  .72751724e+00 -.14073615e+00  .17367948e-03", 
"  .11822374e+00  .72919921e+00 -.19181418e+00  .17409813e-03", 
"  .13466118e+00  .72841873e+00 -.20592264e+00  .17404024e-03", 
"  .15123202e+00  .72748214e+00 -.14282500e+00  .17445653e-03", 
"  .16799709e+00  .72206466e+00 -.15965775e-01  .17698924e-03", 
"  .18492770e+00  .71874017e+00  .66004976e-01  .17778151e-03", 
"  .20202274e+00  .71934368e+00  .25624359e-01  .17955598e-03", 
"  .21927304e+00  .72039275e+00  .25580380e-01  .18184571e-03", 
"  .23678102e+00  .71748717e+00  .94060252e-01  .18364653e-03", 
"  .25451636e+00  .70885897e+00  .25088485e+00  .18473165e-03", 
"  .27238855e+00  .70503684e+00  .37460085e+00  .18585039e-03", 
"  .29045530e+00  .70469131e+00  .43622979e+00  .18669006e-03", 
"  .30876289e+00  .69991385e+00  .57078447e+00  .18731820e-03", 
"  .32730986e+00  .69817689e+00  .65322255e+00  .18881434e-03", 
"  .34608201e+00  .69954440e+00  .65207222e+00  .18932827e-03", 
"  .36509160e+00  .70118350e+00  .66744563e+00  .19036477e-03", 
"  .38438450e+00  .70479523e+00  .63848692e+00  .19104889e-03", 
"  .40399054e+00  .70127591e+00  .77381319e+00  .19254653e-03", 
"  .42381859e+00  .70578585e+00  .77661271e+00  .19357949e-03", 
"  .44400981e+00  .70845188e+00  .76564843e+00  .19270800e-03", 
"  .46452123e+00  .71504692e+00  .66984153e+00  .19399621e-03", 
"  .48536196e+00  .71727201e+00  .72934990e+00  .19557474e-03", 
"  .50650098e+00  .73110911e+00  .54567534e+00  .19837528e-03", 
"  .52811782e+00  .72881888e+00  .73890340e+00  .19966728e-03", 
"  .55006427e+00  .73523577e+00  .74836455e+00  .19991704e-03", 
"  .57244866e+00  .74134438e+00  .80091013e+00  .19902737e-03", 
"  .59531037e+00  .74367625e+00  .91978665e+00  .20239438e-03", 
"  .61869583e+00  .74658014e+00  .99643135e+00  .20585053e-03", 
"  .64249730e+00  .76305579e+00  .87771089e+00  .20526129e-03", 
"  .66691707e+00  .76996444e+00  .97362052e+00  .20866944e-03", 
"  .69183128e+00  .78557766e+00  .88754062e+00  .20955134e-03", 
"  .71744906e+00  .80040433e+00  .80308001e+00  .21295825e-03", 
"  .74377276e+00  .81214280e+00  .83723176e+00  .21328053e-03", 
"  .77080898e+00  .82540821e+00  .85895798e+00  .21879241e-03", 
"  .79865774e+00  .83593059e+00  .91965289e+00  .22425873e-03", 
"  .82745140e+00  .84399837e+00  .10515597e+01  .22793382e-03", 
"  .85706010e+00  .86064730e+00  .11167888e+01  .23196873e-03", 
"  .88773648e+00  .87641750e+00  .11796358e+01  .23909700e-03", 
"  .91952791e+00  .89954267e+00  .11230715e+01  .23965676e-03", 
"  .95259765e+00  .91572736e+00  .13446001e+01  .24244577e-03", 
"  .98685842e+00  .94668499e+00  .13083248e+01  .24366251e-03", 
"  .10226723e+01  .97271007e+00  .15254280e+01  .24879761e-03", 
"  .10602764e+01  .10038143e+01  .15791882e+01  .25394977e-03", 
"  .10998111e+01  .10352814e+01  .18038552e+01  .25746326e-03", 
"  .11415718e+01  .10682334e+01  .20945465e+01  .25962231e-03", 
"  .11857427e+01  .11068815e+01  .24142752e+01  .26296780e-03", 
"  .12329803e+01  .11470414e+01  .28497297e+01  .26686870e-03", 
"  .12836101e+01  .12111123e+01  .29446612e+01  .27474800e-03", 
"  .13383168e+01  .12755673e+01  .33808455e+01  .28272951e-03", 
"  .13981251e+01  .13492646e+01  .38000620e+01  .29185651e-03", 
"  .14640642e+01  .14544896e+01  .41103714e+01  .31385459e-03", 
"  .15383455e+01  .15713643e+01  .46105100e+01  .32810887e-03", 
"  .16233569e+01  .17314438e+01  .51765144e+01  .35016457e-03", 
"  .17241804e+01  .19269880e+01  .63508030e+01  .38437873e-03", 
"  .18485834e+01  .22458904e+01  .72317207e+01  .41561758e-03", 
"  .20150248e+01  .27264767e+01  .88446031e+01  .48601732e-03", 
"  .20567136e+01  .28527315e+01  .96184847e+01  .51796899e-03", 
"  .21027347e+01  .30008151e+01  .10382866e+02  .53898121e-03", 
"  .21540126e+01  .31822666e+01  .11071890e+02  .55772110e-03", 
"  .22124108e+01  .33454490e+01  .12865521e+02  .58784437e-03", 
"  .22797420e+01  .36015037e+01  .14019740e+02  .63354539e-03", 
"  .23603355e+01  .39707581e+01  .14523069e+02  .71569175e-03", 
"  .24616852e+01  .44258179e+01  .15562243e+02  .77596696e-03", 
"  .26009217e+01  .49197115e+01  .22640137e+02  .90787028e-03", 
"  .28233497e+01  .63045355e+01  .26417722e+02  .11716504e-02", 
"  .30328032e+01  .75631954e+01  .35087491e+02  .16240091e-02", 
"  .32991405e+01  .94748927e+01  .53382413e+02  .23643422e-02", 
"  .34781086e+01  .10960181e+02  .65777105e+02  .31382626e-02", 
"dfc       0  2  2   20", " -.46498737e+01 -.19585128e+02 -.13404859e+03  .28987599e-02", 
" -.44931648e+01 -.16807926e+02 -.12204710e+03  .22083000e-02", 
" -.42675648e+01 -.14425831e+02 -.87644297e+02  .14140247e-02", 
" -.40912176e+01 -.12441367e+02 -.68492333e+02  .10882695e-02", 
" -.39058347e+01 -.10513102e+02 -.51918428e+02  .77958830e-03", 
" -.37927154e+01 -.94090272e+01 -.44208442e+02  .66318864e-03", 
" -.37100105e+01 -.86780936e+01 -.37830866e+02  .59497541e-03", 
" -.36435750e+01 -.81822049e+01 -.32355767e+02  .55007457e-03", 
" -.35890810e+01 -.76729277e+01 -.30421975e+02  .50967313e-03", 
" -.35421556e+01 -.73119578e+01 -.27319567e+02  .48484389e-03", 
" -.35007351e+01 -.69999449e+01 -.25038104e+02  .45285603e-03", 
" -.34637847e+01 -.67024239e+01 -.23833155e+02  .43212199e-03", 
" -.34301855e+01 -.64601835e+01 -.22150622e+02  .40625530e-03", 
" -.32980044e+01 -.55041761e+01 -.17079587e+02  .35316546e-03", 
" -.32003475e+01 -.48184935e+01 -.14518699e+02  .31144263e-03", 
" -.31220215e+01 -.43156408e+01 -.12363387e+02  .28433742e-03", 
" -.30560484e+01 -.39360126e+01 -.10355988e+02  .27365317e-03", 
" -.29989010e+01 -.36004495e+01 -.92929232e+01  .25607319e-03", 
" -.29484715e+01 -.33101354e+01 -.82865912e+01  .24419727e-03", 
" -.29028807e+01 -.30689608e+01 -.73753301e+01  .23798936e-03", 
" -.28613794e+01 -.28583890e+01 -.65356937e+01  .22568290e-03", 
" -.28231707e+01 -.26704867e+01 -.58684031e+01  .22093751e-03", 
" -.27876810e+01 -.25009584e+01 -.52732527e+01  .21380944e-03", 
" -.27545889e+01 -.23414925e+01 -.48579683e+01  .20908348e-03", 
" -.27234940e+01 -.21950703e+01 -.45079621e+01  .20412734e-03", 
" -.26940314e+01 -.20720578e+01 -.40170328e+01  .19962789e-03", 
" -.26661673e+01 -.19520849e+01 -.36714342e+01  .19401044e-03", 
" -.26395787e+01 -.18455989e+01 -.32400536e+01  .19334773e-03", 
" -.26142084e+01 -.17412463e+01 -.29756301e+01  .18854058e-03", 
" -.25899690e+01 -.16382011e+01 -.28189050e+01  .18342614e-03", 
" -.25666852e+01 -.15407058e+01 -.26824738e+01  .17867718e-03", 
" -.25442992e+01 -.14455909e+01 -.26141252e+01  .17529619e-03", 
" -.25226529e+01 -.13636472e+01 -.24331178e+01  .17198577e-03", 
" -.25017734e+01 -.12860456e+01 -.22376568e+01  .16710155e-03", 
" -.24815082e+01 -.12146560e+01 -.20028670e+01  .16209084e-03", 
" -.24618127e+01 -.11438367e+01 -.19410583e+01  .15818320e-03", 
" -.24427398e+01 -.10795815e+01 -.17379897e+01  .15707067e-03", 
" -.24242373e+01 -.10129871e+01 -.16157265e+01  .15540008e-03", 
" -.24061235e+01 -.95863004e+00 -.13584182e+01  .15496957e-03", 
" -.23884705e+01 -.90317855e+00 -.11645286e+01  .15380726e-03", 
" -.23712834e+01 -.84897090e+00 -.10361953e+01  .15197444e-03", 
" -.23544588e+01 -.79871409e+00 -.89103868e+00  .15155235e-03", 
" -.23380393e+01 -.74708850e+00 -.82182554e+00  .15056603e-03", 
" -.23219515e+01 -.70155967e+00 -.63796928e+00  .14928147e-03", 
" -.23061761e+01 -.65942293e+00 -.43314474e+00  .14830706e-03", 
" -.22907540e+01 -.61285910e+00 -.36048529e+00  .14721140e-03", 
" -.22756297e+01 -.56779943e+00 -.29121491e+00  .14719167e-03", 
" -.22607880e+01 -.52409236e+00 -.25606839e+00  .14580297e-03", 
" -.22462575e+01 -.47609723e+00 -.30994831e+00  .14528649e-03", 
" -.22318566e+01 -.43929292e+00 -.22440805e+00  .14505919e-03", 
" -.22177730e+01 -.39701536e+00 -.24856318e+00  .14188639e-03", 
" -.22038352e+01 -.36444851e+00 -.11354554e+00  .14171640e-03", 
" -.21901914e+01 -.32532352e+00 -.93216784e-01  .14218242e-03", 
" -.21767391e+01 -.28747114e+00 -.10310017e+00  .14155611e-03", 
" -.21634883e+01 -.25016662e+00 -.10675131e+00  .14050226e-03", 
" -.21503874e+01 -.21601205e+00 -.10476651e+00  .14015033e-03", 
" -.21374432e+01 -.18583360e+00 -.51211317e-01  .14011318e-03", 
" -.21246420e+01 -.15683431e+00 -.29204874e-01  .13883378e-03", 
" -.21120088e+01 -.13213346e+00  .12852807e+00  .13857794e-03", 
" -.20995474e+01 -.99815891e-01  .99058341e-01  .13742797e-03", 
" -.20872295e+01 -.70378462e-01  .10800415e+00  .13793884e-03", 
" -.20750609e+01 -.42044889e-01  .14101216e+00  .13621927e-03", 
" -.20630352e+01 -.15141684e-01  .18865304e+00  .13639356e-03", 
" -.20511360e+01  .15952055e-01  .10012626e+00  .13490015e-03", 
" -.20393918e+01  .44522529e-01  .86565146e-01  .13472776e-03", 
" -.20277077e+01  .67059234e-01  .16819054e+00  .13407648e-03", 
" -.20161490e+01  .90904806e-01  .19550829e+00  .13409339e-03", 
" -.20046920e+01  .11665323e+00  .15704451e+00  .13291655e-03", 
" -.19933276e+01  .13968718e+00  .17247884e+00  .13151197e-03", 
" -.19820619e+01  .16173157e+00  .19898819e+00  .13162402e-03", 
" -.19708860e+01  .18237481e+00  .25236052e+00  .13061907e-03", 
" -.19598292e+01  .20391525e+00  .28727209e+00  .13018764e-03", 
" -.19488223e+01  .22372134e+00  .32812856e+00  .12990573e-03", 
" -.19379266e+01  .24774843e+00  .26125313e+00  .12962020e-03", 
" -.19270853e+01  .26745749e+00  .26979370e+00  .12787068e-03", 
" -.19163151e+01  .28673153e+00  .28767581e+00  .12819650e-03", 
" -.19056274e+01  .30480271e+00  .31440368e+00  .12750112e-03", 
" -.18950092e+01  .32487740e+00  .29289280e+00  .12633421e-03", 
" -.18844373e+01  .34183301e+00  .33094959e+00  .12617140e-03", 
" -.18739396e+01  .36093334e+00  .29415769e+00  .12542124e-03", 
" -.18635243e+01  .38142067e+00  .22614552e+00  .12490566e-03", 
" -.18531302e+01  .39716105e+00  .24303066e+00  .12487050e-03", 
" -.18427987e+01  .40960430e+00  .36111823e+00  .12450876e-03", 
" -.18325490e+01  .42827858e+00  .35044668e+00  .12429566e-03", 
" -.18223120e+01  .44312533e+00  .38373089e+00  .12339286e-03", 
" -.18121476e+01  .45825395e+00  .41736017e+00  .12354584e-03", 
" -.18020189e+01  .47456176e+00  .39780568e+00  .12299378e-03", 
" -.17919441e+01  .49156607e+00  .37160352e+00  .12115398e-03", 
" -.17818927e+01  .50669755e+00  .35931258e+00  .12090362e-03", 
" -.17718547e+01  .52020078e+00  .36852790e+00  .11990979e-03", 
" -.17618784e+01  .53552131e+00  .37399927e+00  .12096096e-03", 
" -.17519035e+01  .54669720e+00  .42995618e+00  .12096099e-03", 
" -.17419868e+01  .56180886e+00  .40902997e+00  .12127621e-03", 
" -.17321082e+01  .57467469e+00  .42868053e+00  .12035010e-03", 
" -.17222375e+01  .58708826e+00  .45175084e+00  .12043651e-03", 
" -.17123798e+01  .59917697e+00  .48860303e+00  .12051206e-03", 
" -.17025509e+01  .61170671e+00  .49474275e+00  .12071179e-03", 
" -.16927408e+01  .62506026e+00  .46000167e+00  .12050594e-03", 
" -.16829693e+01  .63659023e+00  .49564593e+00  .12111163e-03", 
" -.16731939e+01  .64805654e+00  .50804098e+00  .12038583e-03", 
" -.16634091e+01  .65672036e+00  .55607608e+00  .12102541e-03", 
" -.16536497e+01  .66769315e+00  .55911498e+00  .12149389e-03", 
" -.16439062e+01  .67742799e+00  .59076277e+00  .12261731e-03", 
" -.16341886e+01  .68994994e+00  .56825647e+00  .12213119e-03", 
" -.16244687e+01  .70100834e+00  .57240705e+00  .12293942e-03", 
" -.16147364e+01  .71044565e+00  .60084676e+00  .12159089e-03", 
" -.16050132e+01  .72047181e+00  .61499829e+00  .12113944e-03", 
" -.15952755e+01  .72923289e+00  .65089030e+00  .12120266e-03", 
" -.15855407e+01  .73946890e+00  .64516260e+00  .12082145e-03", 
" -.15758340e+01  .74917410e+00  .64206303e+00  .12086709e-03", 
" -.15661226e+01  .76123129e+00  .60141833e+00  .12178366e-03", 
" -.15563725e+01  .76883728e+00  .64611202e+00  .12207516e-03", 
" -.15466262e+01  .77749522e+00  .67164232e+00  .12318791e-03", 
" -.15368887e+01  .79044802e+00  .59971309e+00  .12346950e-03", 
" -.15271193e+01  .80044531e+00  .60613339e+00  .12451886e-03", 
" -.15173571e+01  .80941987e+00  .61721320e+00  .12384741e-03", 
" -.15075829e+01  .82011412e+00  .59323485e+00  .12382238e-03", 
" -.14977672e+01  .83044274e+00  .58712998e+00  .12380260e-03", 
" -.14879122e+01  .83904973e+00  .59087687e+00  .12307983e-03", 
" -.14780499e+01  .84732674e+00  .61608062e+00  .12266640e-03", 
" -.14681619e+01  .85677150e+00  .60979842e+00  .12252223e-03", 
" -.14582463e+01  .86435803e+00  .64988385e+00  .12330911e-03", 
" -.14482883e+01  .87451774e+00  .61543666e+00  .12347216e-03", 
" -.14382813e+01  .88263113e+00  .62973992e+00  .12380557e-03", 
" -.14282536e+01  .88956398e+00  .67468172e+00  .12417842e-03", 
" -.14182008e+01  .89929049e+00  .67428793e+00  .12422941e-03", 
" -.14080885e+01  .90829942e+00  .67159819e+00  .12506386e-03", 
" -.13979342e+01  .91703848e+00  .66877596e+00  .12625226e-03", 
" -.13877694e+01  .92859017e+00  .61995067e+00  .12699355e-03", 
" -.13775230e+01  .93604289e+00  .63846649e+00  .12666064e-03", 
" -.13672057e+01  .94426847e+00  .62511510e+00  .12737116e-03", 
" -.13568339e+01  .95188127e+00  .63766879e+00  .12721756e-03", 
" -.13464079e+01  .95959007e+00  .66127287e+00  .12944453e-03", 
" -.13359058e+01  .96366545e+00  .74578117e+00  .13000200e-03", 
" -.13253685e+01  .97110447e+00  .79643193e+00  .13007992e-03", 
" -.13147624e+01  .97855405e+00  .83213055e+00  .13042507e-03", 
" -.13041106e+01  .98835692e+00  .81911505e+00  .13139037e-03", 
" -.12933783e+01  .99697287e+00  .86808312e+00  .13181101e-03", 
" -.12825497e+01  .10047337e+01  .89757564e+00  .13204824e-03", 
" -.12716606e+01  .10146399e+01  .90814494e+00  .13257959e-03", 
" -.12606975e+01  .10258937e+01  .87659985e+00  .13332493e-03", 
" -.12496250e+01  .10349109e+01  .88896456e+00  .13332758e-03", 
" -.12384725e+01  .10438386e+01  .91800985e+00  .13398711e-03", 
" -.12271853e+01  .10534207e+01  .90595380e+00  .13466330e-03", 
" -.12158071e+01  .10597995e+01  .10029103e+01  .13481324e-03", 
" -.12043342e+01  .10694094e+01  .10105287e+01  .13585221e-03", 
" -.11927374e+01  .10817023e+01  .96515325e+00  .13634858e-03", 
" -.11810471e+01  .10916938e+01  .10085506e+01  .13694656e-03", 
" -.11692236e+01  .11026291e+01  .10268510e+01  .13788806e-03", 
" -.11572565e+01  .11113878e+01  .10630020e+01  .13883535e-03", 
" -.11451871e+01  .11272707e+01  .97342183e+00  .13931866e-03", 
" -.11329554e+01  .11390215e+01  .97571997e+00  .14041549e-03", 
" -.11205568e+01  .11488008e+01  .10272798e+01  .14092650e-03", 
" -.11080251e+01  .11621514e+01  .99811480e+00  .14220521e-03", 
" -.10953689e+01  .11751472e+01  .99909208e+00  .14229817e-03", 
" -.10825027e+01  .11842760e+01  .11029791e+01  .14424766e-03", 
" -.10694471e+01  .11955497e+01  .11515205e+01  .14635427e-03", 
" -.10562396e+01  .12073469e+01  .11921759e+01  .14761855e-03", 
" -.10428562e+01  .12232390e+01  .11346513e+01  .14852001e-03", 
" -.10292620e+01  .12348075e+01  .11945083e+01  .14902807e-03", 
" -.10154635e+01  .12433879e+01  .13252913e+01  .14947863e-03", 
" -.10014375e+01  .12567831e+01  .13560022e+01  .15208154e-03", 
" -.98722361e+00  .12729485e+01  .13623659e+01  .15296326e-03", 
" -.97276525e+00  .12849843e+01  .14498376e+01  .15374600e-03", 
" -.95804968e+00  .12990845e+01  .14649926e+01  .15448352e-03", 
" -.94307789e+00  .13116712e+01  .15199257e+01  .15402427e-03", 
" -.92788147e+00  .13271247e+01  .15335271e+01  .15443462e-03", 
" -.91240062e+00  .13458079e+01  .14685748e+01  .15632675e-03", 
" -.89662363e+00  .13603465e+01  .15140667e+01  .15685230e-03", 
" -.88051439e+00  .13707204e+01  .16170701e+01  .15674102e-03", 
" -.86415213e+00  .13905530e+01  .15259264e+01  .15760527e-03", 
" -.84741271e+00  .14046344e+01  .15529140e+01  .15989949e-03", 
" -.83034617e+00  .14173885e+01  .16209250e+01  .16350912e-03", 
" -.81294478e+00  .14362241e+01  .15551920e+01  .16586488e-03", 
" -.79514864e+00  .14511181e+01  .15562824e+01  .16590867e-03", 
" -.77696661e+00  .14677019e+01  .15249573e+01  .16638401e-03", 
" -.75843335e+00  .14815932e+01  .15680324e+01  .16662967e-03", 
" -.73948936e+00  .15027783e+01  .14016770e+01  .17062759e-03", 
" -.72001765e+00  .15155065e+01  .14202473e+01  .17421195e-03", 
" -.70007960e+00  .15308187e+01  .13741596e+01  .17593826e-03", 
" -.67965634e+00  .15432939e+01  .14005642e+01  .17941820e-03", 
" -.65874834e+00  .15628266e+01  .12723723e+01  .18215045e-03", 
" -.63728288e+00  .15767863e+01  .12949684e+01  .18458866e-03", 
" -.61520090e+00  .15943728e+01  .12109636e+01  .18836483e-03", 
" -.59249185e+00  .16124836e+01  .11000707e+01  .19057883e-03", 
" -.56903837e+00  .16211059e+01  .11553000e+01  .19433841e-03", 
" -.54503602e+00  .16400816e+01  .11198099e+01  .19489117e-03", 
" -.52014209e+00  .16479358e+01  .12904679e+01  .19787026e-03", 
" -.49448261e+00  .16663249e+01  .12330400e+01  .20123336e-03", 
" -.46782751e+00  .16726921e+01  .14163846e+01  .20200546e-03", 
" -.44022652e+00  .16912618e+01  .12920816e+01  .20616949e-03", 
" -.41150914e+00  .17030679e+01  .13956081e+01  .20791146e-03", 
" -.38163960e+00  .17218323e+01  .12886276e+01  .21143949e-03", 
" -.35052328e+00  .17449335e+01  .12444879e+01  .21608096e-03", 
" -.31784963e+00  .17583988e+01  .13309349e+01  .22137616e-03", 
" -.28358669e+00  .17819671e+01  .11939449e+01  .22824610e-03", 
" -.24752936e+00  .18043432e+01  .11492554e+01  .23344280e-03", 
" -.20921434e+00  .18128613e+01  .14681372e+01  .23912457e-03", 
" -.16854843e+00  .18291524e+01  .16811082e+01  .24296201e-03", 
" -.12506770e+00  .18484229e+01  .18448104e+01  .24956133e-03", 
" -.78452312e-01  .18935948e+01  .16486030e+01  .25539564e-03", 
" -.27798950e-01  .19267952e+01  .17041255e+01  .25829514e-03", 
"  .27706392e-01  .19619369e+01  .18949738e+01  .26897317e-03", 
"  .89286662e-01  .19957209e+01  .22039115e+01  .28266973e-03", 
"  .15856116e+00  .20391692e+01  .25297069e+01  .30069415e-03", 
"  .23824409e+00  .20909362e+01  .29792799e+01  .32333274e-03", 
"  .33270796e+00  .21330179e+01  .43119730e+01  .35243873e-03", 
"  .44958524e+00  .22800521e+01  .42114918e+01  .39183260e-03", 
"  .60714202e+00  .24578350e+01  .54652585e+01  .44240428e-03", 
"  .64662561e+00  .25244955e+01  .55903544e+01  .46150771e-03", 
"  .69039936e+00  .25833949e+01  .59261945e+01  .48237641e-03", 
"  .73897923e+00  .26541195e+01  .63608976e+01  .50414902e-03", 
"  .79395621e+00  .27338103e+01  .71352379e+01  .54670677e-03", 
"  .85790385e+00  .28512874e+01  .80839936e+01  .59870411e-03", 
"  .93404481e+00  .30496355e+01  .82089804e+01  .66194575e-03", 
"  .10303788e+01  .32107748e+01  .10398269e+02  .75922721e-03", 
"  .11623571e+01  .34537569e+01  .14242427e+02  .90555619e-03", 
"  .13751885e+01  .40430952e+01  .19400154e+02  .11609585e-02", 
"  .15748041e+01  .47176520e+01  .24795177e+02  .15491660e-02", 
"  .18295984e+01  .59233078e+01  .29935654e+02  .23156824e-02", 
"  .20018937e+01  .65929082e+01  .40856258e+02  .30589721e-02", 
"dfct      0  3  3   20", " -.51292249e+01 -.26719405e+02 -.71505058e+02 -.18132204e+04  .33885902e-02", 
" -.49772843e+01 -.23023496e+02 -.10625910e+03 -.82440218e+03  .26636323e-02", 
" -.47675056e+01 -.18937358e+02 -.10617991e+03 -.28437677e+03  .17715683e-02", 
" -.45953135e+01 -.16764209e+02 -.68301228e+02 -.46216138e+03  .12267776e-02", 
" -.44175149e+01 -.14124948e+02 -.60038464e+02 -.32338440e+03  .93777691e-03", 
" -.43074125e+01 -.12945902e+02 -.44683263e+02 -.33647743e+03  .79468331e-03", 
" -.42275728e+01 -.12030693e+02 -.36566098e+02 -.35238619e+03  .69113580e-03", 
" -.41638843e+01 -.11350053e+02 -.32774943e+02 -.29277850e+03  .63012787e-03", 
" -.41111804e+01 -.10738118e+02 -.32793566e+02 -.22242918e+03  .58445069e-03", 
" -.40655516e+01 -.10355485e+02 -.24941082e+02 -.26916719e+03  .55462829e-03", 
" -.40257160e+01 -.99051615e+01 -.24342994e+02 -.23823165e+03  .53346611e-03", 
" -.39903382e+01 -.94980973e+01 -.24469611e+02 -.20211084e+03  .51061829e-03", 
" -.39579703e+01 -.91969574e+01 -.22702701e+02 -.18839198e+03  .48941112e-03", 
" -.38304297e+01 -.79968631e+01 -.15371282e+02 -.18622806e+03  .41578111e-03", 
" -.37359746e+01 -.71683122e+01 -.11342941e+02 -.16467362e+03  .37278025e-03", 
" -.36604348e+01 -.64961092e+01 -.98084411e+01 -.13865435e+03  .33483802e-03", 
" -.35971452e+01 -.59584730e+01 -.76698007e+01 -.13980717e+03  .31748485e-03", 
" -.35420886e+01 -.55368953e+01 -.49209822e+01 -.15127142e+03  .29998883e-03", 
" -.34933873e+01 -.51618070e+01 -.38734324e+01 -.14161612e+03  .28230115e-03", 
" -.34498312e+01 -.47945374e+01 -.45103958e+01 -.11819984e+03  .26593309e-03", 
" -.34098271e+01 -.45119686e+01 -.36630374e+01 -.10996942e+03  .26006414e-03", 
" -.33731039e+01 -.42530540e+01 -.26810539e+01 -.11147716e+03  .25015961e-03", 
" -.33389269e+01 -.40161454e+01 -.22793812e+01 -.10599232e+03  .24376895e-03", 
" -.33071958e+01 -.37837582e+01 -.26840288e+01 -.88568817e+02  .23631403e-03", 
" -.32772698e+01 -.35989629e+01 -.14096929e+01 -.97886557e+02  .23105111e-03", 
" -.32490638e+01 -.34004849e+01 -.18647137e+01 -.82041248e+02  .22374466e-03", 
" -.32222496e+01 -.32298164e+01 -.16497471e+01 -.76770146e+02  .21599403e-03", 
" -.31966701e+01 -.30914518e+01 -.46648607e+00 -.84832160e+02  .21510158e-03", 
" -.31723541e+01 -.29442737e+01 -.25122362e+00 -.79841118e+02  .21196397e-03", 
" -.31490499e+01 -.27984615e+01 -.60841107e+00 -.67248271e+02  .20392752e-03", 
" -.31266008e+01 -.26801993e+01  .18064468e+00 -.70566800e+02  .19867722e-03", 
" -.31050614e+01 -.25556146e+01  .30483030e+00 -.66331356e+02  .19657521e-03", 
" -.30842631e+01 -.24325536e+01  .55237751e-01 -.57441312e+02  .19363302e-03", 
" -.30641240e+01 -.23516058e+01  .19369931e+01 -.78414410e+02  .18938648e-03", 
" -.30447284e+01 -.22387749e+01  .19473749e+01 -.76524563e+02  .18911144e-03", 
" -.30258531e+01 -.21380320e+01  .20895666e+01 -.75536284e+02  .18775206e-03", 
" -.30075743e+01 -.20346668e+01  .19087771e+01 -.69841588e+02  .18362816e-03", 
" -.29897721e+01 -.19406836e+01  .16726132e+01 -.61191433e+02  .18188085e-03", 
" -.29724186e+01 -.18630055e+01  .21982439e+01 -.62991422e+02  .17953184e-03", 
" -.29556152e+01 -.17764191e+01  .23943049e+01 -.63323257e+02  .17739996e-03", 
" -.29391270e+01 -.17019814e+01  .29011393e+01 -.67401446e+02  .17631621e-03", 
" -.29230915e+01 -.16194464e+01  .29100092e+01 -.65734967e+02  .17383308e-03", 
" -.29073542e+01 -.15544466e+01  .33714972e+01 -.68447922e+02  .17132397e-03", 
" -.28920421e+01 -.14746117e+01  .32872920e+01 -.65347757e+02  .17065386e-03", 
" -.28770072e+01 -.13999656e+01  .31762313e+01 -.62357562e+02  .16854506e-03", 
" -.28622767e+01 -.13248669e+01  .29248375e+01 -.57117555e+02  .16775877e-03", 
" -.28478144e+01 -.12576021e+01  .30275682e+01 -.57490293e+02  .16810700e-03", 
" -.28336368e+01 -.11832725e+01  .25774957e+01 -.50417268e+02  .16717041e-03", 
" -.28196693e+01 -.11351672e+01  .32842425e+01 -.56866936e+02  .16507566e-03", 
" -.28059229e+01 -.10897831e+01  .40987862e+01 -.65845043e+02  .16409938e-03", 
" -.27924244e+01 -.10357843e+01  .43139195e+01 -.66451953e+02  .16122474e-03", 
" -.27791594e+01 -.98116276e+00  .44046139e+01 -.65621184e+02  .16021626e-03", 
" -.27660918e+01 -.93565330e+00  .49513456e+01 -.71803048e+02  .16028087e-03", 
" -.27532950e+01 -.87603023e+00  .47595728e+01 -.68431777e+02  .16026079e-03", 
" -.27405770e+01 -.83901838e+00  .54578623e+01 -.74961638e+02  .15959372e-03", 
" -.27281403e+01 -.77841620e+00  .49174947e+01 -.65332097e+02  .15821915e-03", 
" -.27158486e+01 -.72739169e+00  .49094846e+01 -.64184597e+02  .15675119e-03", 
" -.27037032e+01 -.67268668e+00  .44944629e+01 -.56574353e+02  .15596401e-03", 
" -.26916730e+01 -.62973528e+00  .45915487e+01 -.55634219e+02  .15596307e-03", 
" -.26798653e+01 -.57694690e+00  .42675244e+01 -.50038814e+02  .15574682e-03", 
" -.26681478e+01 -.53657585e+00  .44389738e+01 -.50652347e+02  .15428007e-03", 
" -.26566314e+01 -.48471837e+00  .40575016e+01 -.44261354e+02  .15527884e-03", 
" -.26452172e+01 -.44006839e+00  .40839919e+01 -.44101683e+02  .15518621e-03", 
" -.26339288e+01 -.39080840e+00  .35922950e+01 -.35487137e+02  .15303640e-03", 
" -.26226920e+01 -.35680800e+00  .39609952e+01 -.39628037e+02  .15324745e-03", 
" -.26116268e+01 -.31585896e+00  .38918079e+01 -.37337359e+02  .15411009e-03", 
" -.26007008e+01 -.27508044e+00  .38825003e+01 -.37238795e+02  .15260244e-03", 
" -.25898604e+01 -.23595815e+00  .38374761e+01 -.35703014e+02  .15206038e-03", 
" -.25791024e+01 -.19275104e+00  .36180677e+01 -.32854471e+02  .15233346e-03", 
" -.25684780e+01 -.14422428e+00  .29497493e+01 -.23265165e+02  .15187371e-03", 
" -.25578907e+01 -.11708115e+00  .34865837e+01 -.30227654e+02  .15146681e-03", 
" -.25474479e+01 -.78928335e-01  .33787342e+01 -.28511927e+02  .15084094e-03", 
" -.25370510e+01 -.46002003e-01  .34911293e+01 -.29808341e+02  .15135005e-03", 
" -.25267346e+01 -.88433368e-02  .32582071e+01 -.25573831e+02  .15146718e-03", 
" -.25165101e+01  .30239666e-01  .29400120e+01 -.21369329e+02  .15071137e-03", 
" -.25063486e+01  .57189111e-01  .31815506e+01 -.23456797e+02  .15069188e-03", 
" -.24963173e+01  .96684467e-01  .28368008e+01 -.18810400e+02  .14896826e-03", 
" -.24863552e+01  .13442678e+00  .25591894e+01 -.15152288e+02  .14803579e-03", 
" -.24764134e+01  .16707005e+00  .23911364e+01 -.12213498e+02  .14642068e-03", 
" -.24665145e+01  .19830071e+00  .22971835e+01 -.99591831e+01  .14631912e-03", 
" -.24567076e+01  .22601828e+00  .24265859e+01 -.11559708e+02  .14551435e-03", 
" -.24469298e+01  .25058604e+00  .26670623e+01 -.14423830e+02  .14500881e-03", 
" -.24372232e+01  .27801061e+00  .26743891e+01 -.13439708e+02  .14467920e-03", 
" -.24276224e+01  .31395462e+00  .22621507e+01 -.75122092e+01  .14434048e-03", 
" -.24180230e+01  .33846950e+00  .23723621e+01 -.81512060e+01  .14468125e-03", 
" -.24085010e+01  .37196042e+00  .21008909e+01 -.50165956e+01  .14364886e-03", 
" -.23990018e+01  .40127084e+00  .18503210e+01  .94670959e-01  .14471542e-03", 
" -.23895519e+01  .42357106e+00  .20564043e+01 -.20926551e+01  .14384185e-03", 
" -.23801178e+01  .44483141e+00  .23172042e+01 -.57502842e+01  .14349328e-03", 
" -.23707386e+01  .46799263e+00  .24510974e+01 -.78690664e+01  .14289131e-03", 
" -.23613945e+01  .48453194e+00  .28496225e+01 -.12430669e+02  .14246123e-03", 
" -.23521315e+01  .51224819e+00  .27172324e+01 -.10805897e+02  .14311951e-03", 
" -.23428615e+01  .53499021e+00  .27393295e+01 -.10477170e+02  .14373853e-03", 
" -.23336594e+01  .56658689e+00  .23561888e+01 -.56697405e+01  .14372550e-03", 
" -.23244550e+01  .58537293e+00  .25731450e+01 -.83767262e+01  .14275271e-03", 
" -.23152939e+01  .61050006e+00  .23544778e+01 -.45072384e+01  .14221504e-03", 
" -.23061545e+01  .63321402e+00  .22490796e+01 -.23168224e+01  .14150262e-03", 
" -.22970767e+01  .65712680e+00  .21628346e+01 -.11015245e+01  .14192615e-03", 
" -.22879866e+01  .67872982e+00  .20570538e+01  .11234809e+01  .14112228e-03", 
" -.22789051e+01  .68916911e+00  .26927433e+01 -.74046893e+01  .14024701e-03", 
" -.22698488e+01  .70828441e+00  .27444158e+01 -.74737131e+01  .13996383e-03", 
" -.22608368e+01  .73335832e+00  .24663581e+01 -.32373265e+01  .14066022e-03", 
" -.22518533e+01  .75221949e+00  .25753942e+01 -.46643976e+01  .14096756e-03", 
" -.22428768e+01  .77080224e+00  .27153251e+01 -.66507102e+01  .13965135e-03", 
" -.22339450e+01  .79382399e+00  .26031705e+01 -.52426895e+01  .13841971e-03", 
" -.22250064e+01  .81629951e+00  .24828676e+01 -.38413947e+01  .13786648e-03", 
" -.22160811e+01  .84061753e+00  .22139888e+01 -.16662817e-01  .13701353e-03", 
" -.22071578e+01  .85684905e+00  .24074056e+01 -.24624360e+01  .13804943e-03", 
" -.21982600e+01  .87649237e+00  .24263314e+01 -.30569545e+01  .13797874e-03", 
" -.21893693e+01  .89717147e+00  .23049354e+01 -.91761864e+00  .13818453e-03", 
" -.21804723e+01  .91407390e+00  .23926728e+01 -.21744109e+01  .13840677e-03", 
" -.21715633e+01  .93349657e+00  .23061015e+01 -.90584152e+00  .13761597e-03", 
" -.21626276e+01  .93971963e+00  .28345674e+01 -.71659328e+01  .13832364e-03", 
" -.21537543e+01  .95788791e+00  .29207276e+01 -.89025672e+01  .13807326e-03", 
" -.21448725e+01  .97685127e+00  .28226055e+01 -.73310995e+01  .13747915e-03", 
" -.21360234e+01  .10018044e+01  .25081261e+01 -.37834633e+01  .13608001e-03", 
" -.21271410e+01  .10221381e+01  .23591607e+01 -.17552750e+01  .13647218e-03", 
" -.21182572e+01  .10376907e+01  .24550352e+01 -.28045144e+01  .13606637e-03", 
" -.21093539e+01  .10545503e+01  .24832597e+01 -.36485319e+01  .13663996e-03", 
" -.21004392e+01  .10707521e+01  .25169620e+01 -.43821238e+01  .13792304e-03", 
" -.20915177e+01  .10766210e+01  .31197524e+01 -.12823056e+02  .13844332e-03", 
" -.20825577e+01  .10893451e+01  .31880812e+01 -.13158546e+02  .13863815e-03", 
" -.20736342e+01  .11094116e+01  .29682100e+01 -.10469190e+02  .13787602e-03", 
" -.20646655e+01  .11188882e+01  .33881070e+01 -.16443775e+02  .13929772e-03", 
" -.20556978e+01  .11354394e+01  .33064686e+01 -.15043946e+02  .13896461e-03", 
" -.20467186e+01  .11540450e+01  .31419242e+01 -.12675850e+02  .14019431e-03", 
" -.20377119e+01  .11701541e+01  .31345858e+01 -.12882730e+02  .14036960e-03", 
" -.20286865e+01  .11840489e+01  .32405907e+01 -.14277327e+02  .14100853e-03", 
" -.20196066e+01  .11908254e+01  .36997872e+01 -.20796341e+02  .13997280e-03", 
" -.20105035e+01  .11986852e+01  .40948873e+01 -.26414970e+02  .14001360e-03", 
" -.20014101e+01  .12174824e+01  .38387160e+01 -.22856574e+02  .13914600e-03", 
" -.19922608e+01  .12258490e+01  .41415715e+01 -.26562092e+02  .13826548e-03", 
" -.19831055e+01  .12421260e+01  .41281558e+01 -.26661964e+02  .13867907e-03", 
" -.19739381e+01  .12638568e+01  .37981407e+01 -.22613934e+02  .13827443e-03", 
" -.19646817e+01  .12662124e+01  .44468385e+01 -.31496535e+02  .13817797e-03", 
" -.19554013e+01  .12818893e+01  .42857828e+01 -.28948690e+02  .13691489e-03", 
" -.19460905e+01  .12949651e+01  .43324966e+01 -.29752916e+02  .13695602e-03", 
" -.19367325e+01  .13059400e+01  .45148127e+01 -.32156138e+02  .13653650e-03", 
" -.19273449e+01  .13196473e+01  .44970527e+01 -.32103802e+02  .13685557e-03", 
" -.19179420e+01  .13390194e+01  .42389864e+01 -.28474253e+02  .13683870e-03", 
" -.19084974e+01  .13616852e+01  .38076995e+01 -.23024484e+02  .13672194e-03", 
" -.18989633e+01  .13792449e+01  .35856095e+01 -.19966922e+02  .13753257e-03", 
" -.18893909e+01  .14027174e+01  .30218952e+01 -.12727231e+02  .13828068e-03", 
" -.18797154e+01  .14114457e+01  .32282461e+01 -.14933104e+02  .13922397e-03", 
" -.18700179e+01  .14268665e+01  .31472097e+01 -.14232556e+02  .14006429e-03", 
" -.18602368e+01  .14356002e+01  .34593436e+01 -.18790799e+02  .14016004e-03", 
" -.18504258e+01  .14500848e+01  .33863764e+01 -.17434797e+02  .14059273e-03", 
" -.18405255e+01  .14662367e+01  .32540039e+01 -.16131579e+02  .14128370e-03", 
" -.18305438e+01  .14807586e+01  .31937663e+01 -.15461995e+02  .14214132e-03", 
" -.18204991e+01  .14948178e+01  .32497226e+01 -.16927232e+02  .14075805e-03", 
" -.18103510e+01  .15097792e+01  .31988594e+01 -.16425651e+02  .14139416e-03", 
" -.18001244e+01  .15250088e+01  .30654010e+01 -.14728335e+02  .14027316e-03", 
" -.17897880e+01  .15308619e+01  .34085340e+01 -.18667621e+02  .14037122e-03", 
" -.17793754e+01  .15397714e+01  .36001242e+01 -.20586580e+02  .14081775e-03", 
" -.17688669e+01  .15515074e+01  .36250524e+01 -.20123236e+02  .14178541e-03", 
" -.17582836e+01  .15657976e+01  .37108570e+01 -.21530377e+02  .14184305e-03", 
" -.17476215e+01  .15888913e+01  .33709576e+01 -.18128033e+02  .14275169e-03", 
" -.17368102e+01  .16075322e+01  .30628266e+01 -.13698175e+02  .14441364e-03", 
" -.17258802e+01  .16217158e+01  .29465403e+01 -.10831516e+02  .14414746e-03", 
" -.17147789e+01  .16263007e+01  .33347902e+01 -.14567624e+02  .14487648e-03", 
" -.17036055e+01  .16385091e+01  .35964522e+01 -.19478659e+02  .14580910e-03", 
" -.16923035e+01  .16575011e+01  .33875450e+01 -.16471074e+02  .14675254e-03", 
" -.16808178e+01  .16735087e+01  .33470112e+01 -.15979085e+02  .14779663e-03", 
" -.16692208e+01  .16976318e+01  .29191207e+01 -.11105685e+02  .14845919e-03", 
" -.16574499e+01  .17126720e+01  .30190391e+01 -.12544351e+02  .14911957e-03", 
" -.16455203e+01  .17321196e+01  .27843557e+01 -.88951804e+01  .15133137e-03", 
" -.16334202e+01  .17505771e+01  .27140815e+01 -.84134683e+01  .15114220e-03", 
" -.16211175e+01  .17647263e+01  .28301550e+01 -.93961723e+01  .15302209e-03", 
" -.16086311e+01  .17808217e+01  .28613978e+01 -.93538199e+01  .15394567e-03", 
" -.15959047e+01  .17966812e+01  .29838201e+01 -.11103841e+02  .15518715e-03", 
" -.15830072e+01  .18192877e+01  .26503963e+01 -.55630700e+01  .15500475e-03", 
" -.15698372e+01  .18341156e+01  .27395950e+01 -.51385562e+01  .15687623e-03", 
" -.15564282e+01  .18562127e+01  .25012140e+01 -.78716052e+00  .15882836e-03", 
" -.15427607e+01  .18773714e+01  .23435865e+01  .28472436e+01  .16039372e-03", 
" -.15288002e+01  .18956652e+01  .22980392e+01  .56595326e+01  .16232841e-03", 
" -.15145361e+01  .19093733e+01  .26414426e+01  .23517865e+01  .16365345e-03", 
" -.14999657e+01  .19260694e+01  .28047512e+01  .16894430e+01  .16566139e-03", 
" -.14850839e+01  .19527455e+01  .25169865e+01  .76460826e+01  .16641569e-03", 
" -.14698449e+01  .19750883e+01  .25025928e+01  .98837033e+01  .17034911e-03", 
" -.14541516e+01  .19847568e+01  .31504806e+01  .42753172e+01  .17141782e-03", 
" -.14381040e+01  .20099117e+01  .32157148e+01  .48328179e+01  .17448019e-03", 
" -.14215628e+01  .20239976e+01  .38039014e+01 -.28153304e+00  .17656234e-03", 
" -.14046128e+01  .20543491e+01  .35563272e+01  .57751371e+01  .17699420e-03", 
" -.13871400e+01  .20916555e+01  .30732965e+01  .14767552e+02  .17892036e-03", 
" -.13690755e+01  .21113911e+01  .34866191e+01  .12462613e+02  .18268588e-03", 
" -.13503417e+01  .21236448e+01  .43946426e+01  .24200196e+01  .18601054e-03", 
" -.13310719e+01  .21629672e+01  .40026267e+01  .99316285e+01  .18721400e-03", 
" -.13111432e+01  .21994885e+01  .40790645e+01  .10240756e+02  .19061032e-03", 
" -.12904021e+01  .22420436e+01  .37791777e+01  .15485402e+02  .19388647e-03", 
" -.12689099e+01  .23008585e+01  .30005740e+01  .25415100e+02  .19911109e-03", 
" -.12464024e+01  .23286200e+01  .37709529e+01  .17058953e+02  .20395617e-03", 
" -.12228862e+01  .23634295e+01  .42775097e+01  .11250807e+02  .20944630e-03", 
" -.11982425e+01  .24090033e+01  .43932276e+01  .89032503e+01  .21405604e-03", 
" -.11723118e+01  .24458887e+01  .47568950e+01  .68177565e+01  .22029962e-03", 
" -.11450172e+01  .24912967e+01  .50152698e+01  .29440321e+01  .22563475e-03", 
" -.11161953e+01  .25419524e+01  .50310148e+01  .26089919e+01  .23322639e-03", 
" -.10856992e+01  .26004777e+01  .46951057e+01  .78180145e+01  .24328614e-03", 
" -.10532375e+01  .26844960e+01  .31743490e+01  .24984198e+02  .25189864e-03", 
" -.10185068e+01  .27604940e+01  .22321465e+01  .34995579e+02  .26108219e-03", 
" -.98105983e+00  .28205207e+01  .18424027e+01  .42493438e+02  .27192017e-03", 
" -.94030112e+00  .28636585e+01  .23652930e+01  .34268050e+02  .28174168e-03", 
" -.89611941e+00  .29410406e+01  .15984543e+01  .44446099e+02  .29244718e-03", 
" -.84734516e+00  .30284974e+01  .36655117e+00  .60866280e+02  .30364579e-03", 
" -.79301687e+00  .31032474e+01 -.70383002e-01  .67919785e+02  .32259590e-03", 
" -.73134643e+00  .31699769e+01  .57238705e+00  .56455304e+02  .34359911e-03", 
" -.65988222e+00  .32397533e+01  .10038603e+01  .55590683e+02  .36281585e-03", 
" -.57498908e+00  .33704549e+01 -.86853477e+00  .90162869e+02  .40371043e-03", 
" -.46882616e+00  .34565692e+01  .10856959e+01  .73944809e+02  .45839375e-03", 
" -.32542503e+00  .36467386e+01  .97278076e+00  .80096055e+02  .53559421e-03", 
" -.28927049e+00  .36667954e+01  .26798444e+01  .58498617e+02  .54654176e-03", 
" -.24936161e+00  .37425553e+01  .19014721e+01  .69517502e+02  .56565084e-03", 
" -.20449854e+00  .38183521e+01  .12114951e+01  .81493244e+02  .59004685e-03", 
" -.15389825e+00  .38686285e+01  .37704385e+01  .48132614e+02  .63012289e-03", 
" -.94616438e-01  .39007553e+01  .82410287e+01 -.17459360e+02  .66143145e-03", 
" -.23862549e-01  .40207808e+01  .71436152e+01  .20570070e+02  .75075038e-03", 
"  .65520718e-01  .41594945e+01  .67479375e+01  .46321645e+02  .82719858e-03", 
"  .18834040e+00  .43459788e+01  .97447054e+01  .80854981e+01  .96268852e-03", 
"  .38863151e+00  .46531364e+01  .14968841e+02 -.83553453e+01  .13263463e-02", 
"  .57587025e+00  .50705940e+01  .21732595e+02 -.63042215e+02  .17499956e-02", 
"  .81904547e+00  .52559469e+01  .56007846e+02 -.41178944e+03  .26299291e-02", 
"  .97720956e+00  .66104917e+01  .32425153e+02 -.15735229e+03  .34296861e-02", 
"dfctt     0  4  3   20", " -.55203738e+01 -.31573920e+02 -.86741237e+02 -.31486773e+04  .34049457e-02", 
" -.53722293e+01 -.27530853e+02 -.11680029e+03 -.20178905e+04  .25197708e-02", 
" -.51641203e+01 -.23421907e+02 -.10029535e+03 -.13862244e+04  .16233541e-02", 
" -.49963159e+01 -.20402271e+02 -.95399500e+02 -.85566047e+03  .12391789e-02", 
" -.48215068e+01 -.17515821e+02 -.81400323e+02 -.57967322e+03  .93628963e-03", 
" -.47148498e+01 -.15789796e+02 -.78398177e+02 -.37021503e+03  .77917825e-03", 
" -.46362048e+01 -.14712738e+02 -.71388050e+02 -.28384204e+03  .68536117e-03", 
" -.45742073e+01 -.13889386e+02 -.63911470e+02 -.27575959e+03  .62922240e-03", 
" -.45219077e+01 -.13342048e+02 -.52147100e+02 -.34975817e+03  .56989764e-03", 
" -.44774493e+01 -.12745941e+02 -.50774612e+02 -.28897110e+03  .52986152e-03", 
" -.44382835e+01 -.12320050e+02 -.43928991e+02 -.32435830e+03  .50425576e-03", 
" -.44032402e+01 -.11898527e+02 -.41987082e+02 -.28352878e+03  .48382649e-03", 
" -.43716532e+01 -.11475579e+02 -.41169259e+02 -.25738818e+03  .46762394e-03", 
" -.42459347e+01 -.10151347e+02 -.27072037e+02 -.28116026e+03  .40673028e-03", 
" -.41531562e+01 -.91494991e+01 -.21151819e+02 -.25955611e+03  .36291386e-03", 
" -.40785885e+01 -.84477624e+01 -.13785806e+02 -.27643404e+03  .33322714e-03", 
" -.40161871e+01 -.78118983e+01 -.11233922e+02 -.25983048e+03  .30681824e-03", 
" -.39623485e+01 -.72413686e+01 -.10490893e+02 -.23112439e+03  .29952181e-03", 
" -.39145346e+01 -.67549057e+01 -.10458095e+02 -.19465528e+03  .27996546e-03", 
" -.38713511e+01 -.63791855e+01 -.72745926e+01 -.21188348e+03  .26202213e-03", 
" -.38320005e+01 -.60292882e+01 -.58425661e+01 -.20351709e+03  .25717123e-03", 
" -.37958308e+01 -.56823982e+01 -.66714427e+01 -.16778336e+03  .25115599e-03", 
" -.37621519e+01 -.54185274e+01 -.50807443e+01 -.16284050e+03  .24021316e-03", 
" -.37308097e+01 -.51519106e+01 -.43962253e+01 -.15575632e+03  .23315571e-03", 
" -.37013329e+01 -.48909299e+01 -.46127977e+01 -.13805115e+03  .22651153e-03", 
" -.36734450e+01 -.46584419e+01 -.42812254e+01 -.12930097e+03  .22282202e-03", 
" -.36470421e+01 -.44469832e+01 -.39055199e+01 -.11963747e+03  .21688503e-03", 
" -.36219713e+01 -.42393773e+01 -.39901920e+01 -.10655243e+03  .21395845e-03", 
" -.35980125e+01 -.40385933e+01 -.44825701e+01 -.87421581e+02  .21015093e-03", 
" -.35749409e+01 -.38863815e+01 -.31705470e+01 -.91575452e+02  .20594688e-03", 
" -.35528814e+01 -.37231175e+01 -.26504986e+01 -.89802647e+02  .20374610e-03", 
" -.35316186e+01 -.35771730e+01 -.17006456e+01 -.95137411e+02  .20074557e-03", 
" -.35111690e+01 -.34281206e+01 -.14572020e+01 -.88823919e+02  .19771071e-03", 
" -.34912798e+01 -.33246402e+01  .54450104e+00 -.10529503e+03  .19505981e-03", 
" -.34721127e+01 -.31982298e+01  .12299956e+01 -.10782721e+03  .19171396e-03", 
" -.34535619e+01 -.30682392e+01  .14401814e+01 -.10450007e+03  .18750699e-03", 
" -.34355772e+01 -.29221308e+01  .27244532e+00 -.81879399e+02  .18505756e-03", 
" -.34180890e+01 -.27875713e+01 -.39651967e+00 -.67477718e+02  .18442578e-03", 
" -.34010644e+01 -.26650701e+01 -.53517431e+00 -.62083235e+02  .18281898e-03", 
" -.33844341e+01 -.25473884e+01 -.89601352e+00 -.52323166e+02  .17990421e-03", 
" -.33682391e+01 -.24309855e+01 -.14250261e+01 -.39953713e+02  .17902237e-03", 
" -.33524526e+01 -.23209365e+01 -.16511685e+01 -.32633842e+02  .17772438e-03", 
" -.33369486e+01 -.22324878e+01 -.10952830e+01 -.33898330e+02  .17716768e-03", 
" -.33218069e+01 -.21292829e+01 -.12427849e+01 -.29262976e+02  .17590048e-03", 
" -.33069404e+01 -.20356615e+01 -.13349973e+01 -.22870554e+02  .17262011e-03", 
" -.32924254e+01 -.19338076e+01 -.18074708e+01 -.13723386e+02  .16977878e-03", 
" -.32781736e+01 -.18455878e+01 -.17466936e+01 -.11593312e+02  .16925105e-03", 
" -.32642035e+01 -.17592453e+01 -.16237844e+01 -.10573581e+02  .16758185e-03", 
" -.32504354e+01 -.16712519e+01 -.18846659e+01 -.47565895e+01  .16492147e-03", 
" -.32369973e+01 -.15812146e+01 -.23639859e+01  .61780252e+01  .16366049e-03", 
" -.32237084e+01 -.15063052e+01 -.22156737e+01  .78469977e+01  .16150195e-03", 
" -.32106151e+01 -.14459980e+01 -.13763164e+01 -.64321664e+00  .16109084e-03", 
" -.31977579e+01 -.13762699e+01 -.11731892e+01 -.38029328e+00  .16124874e-03", 
" -.31850141e+01 -.13138409e+01 -.86386888e+00 -.11739891e+01  .15984506e-03", 
" -.31725322e+01 -.12438335e+01 -.87600314e+00  .17499708e+01  .15732242e-03", 
" -.31602221e+01 -.11833252e+01 -.43385937e+00 -.19839342e+01  .15685009e-03", 
" -.31480574e+01 -.11170898e+01 -.34750234e+00 -.15281434e+01  .15578257e-03", 
" -.31360660e+01 -.10593153e+01  .83674002e-01 -.54249904e+01  .15551014e-03", 
" -.31242534e+01 -.99586903e+00  .10734494e+00 -.35124729e+01  .15400228e-03", 
" -.31125396e+01 -.93856493e+00  .24747601e+00 -.22558241e+01  .15408296e-03", 
" -.31009640e+01 -.89191004e+00  .96081172e+00 -.97136526e+01  .15277893e-03", 
" -.30895208e+01 -.83614483e+00  .10567576e+01 -.81413813e+01  .15137273e-03", 
" -.30782334e+01 -.77801056e+00  .94745183e+00 -.42958647e+01  .15278063e-03", 
" -.30670494e+01 -.72824670e+00  .12474319e+01 -.56319656e+01  .15368596e-03", 
" -.30559960e+01 -.66947814e+00  .10563020e+01 -.13970883e+01  .15262705e-03", 
" -.30450447e+01 -.62280023e+00  .14797765e+01 -.55848957e+01  .15178904e-03", 
" -.30342373e+01 -.56403209e+00  .10956690e+01  .21101094e+01  .15300430e-03", 
" -.30235317e+01 -.50631537e+00  .80870165e+00  .70648849e+01  .15134739e-03", 
" -.30129274e+01 -.45473444e+00  .67107222e+00  .11664105e+02  .15166811e-03", 
" -.30024133e+01 -.40993226e+00  .10102149e+01  .85167669e+01  .15018663e-03", 
" -.29919861e+01 -.36746228e+00  .14128057e+01  .44020250e+01  .14951125e-03", 
" -.29816472e+01 -.31997749e+00  .14872697e+01  .43472616e+01  .14889984e-03", 
" -.29713730e+01 -.27945683e+00  .18597379e+01  .50093278e+00  .14778137e-03", 
" -.29611831e+01 -.23416745e+00  .19443381e+01  .15416241e+00  .14675083e-03", 
" -.29510741e+01 -.19037071e+00  .19983934e+01  .62220349e+00  .14630601e-03", 
" -.29410662e+01 -.14940732e+00  .22478383e+01 -.15116161e+01  .14650878e-03", 
" -.29311307e+01 -.10645892e+00  .22717074e+01 -.44205960e+00  .14643521e-03", 
" -.29212710e+01 -.61322333e-01  .21878050e+01  .17066475e+01  .14501740e-03", 
" -.29114312e+01 -.26848273e-01  .26100777e+01 -.29671059e+01  .14247026e-03", 
" -.29016838e+01  .69531748e-02  .31409260e+01 -.96823689e+01  .14191505e-03", 
" -.28920156e+01  .51998313e-01  .29330771e+01 -.58839442e+01  .14053969e-03", 
" -.28824204e+01  .94186024e-01  .29445238e+01 -.60798914e+01  .14089539e-03", 
" -.28728364e+01  .12498501e+00  .33881110e+01 -.10305961e+02  .13998408e-03", 
" -.28633186e+01  .15953039e+00  .36584273e+01 -.12924943e+02  .13954293e-03", 
" -.28538507e+01  .19923683e+00  .35891476e+01 -.11132116e+02  .13923149e-03", 
" -.28444570e+01  .23802963e+00  .35366529e+01 -.96401788e+01  .13902039e-03", 
" -.28350566e+01  .26916161e+00  .38062311e+01 -.12263364e+02  .13949433e-03", 
" -.28257294e+01  .30688081e+00  .36505338e+01 -.89769548e+01  .13857126e-03", 
" -.28164724e+01  .34555323e+00  .35090531e+01 -.61922188e+01  .13820944e-03", 
" -.28072644e+01  .38384142e+00  .34060621e+01 -.44347318e+01  .13772453e-03", 
" -.27980767e+01  .41759214e+00  .34494627e+01 -.38985644e+01  .13672544e-03", 
" -.27889268e+01  .44984649e+00  .36296227e+01 -.60358182e+01  .13670101e-03", 
" -.27798276e+01  .48757594e+00  .34934103e+01 -.41021864e+01  .13628427e-03", 
" -.27707648e+01  .52159721e+00  .35398281e+01 -.45605252e+01  .13512913e-03", 
" -.27617021e+01  .55301586e+00  .35881348e+01 -.42351549e+01  .13549241e-03", 
" -.27526437e+01  .57962235e+00  .38462828e+01 -.64921823e+01  .13516682e-03", 
" -.27436346e+01  .60751390e+00  .40057678e+01 -.74172386e+01  .13464588e-03", 
" -.27346785e+01  .64383313e+00  .37385060e+01 -.29726485e+01  .13403984e-03", 
" -.27257071e+01  .66992552e+00  .39206190e+01 -.42748396e+01  .13291806e-03", 
" -.27167890e+01  .70008728e+00  .39668157e+01 -.43163366e+01  .13299711e-03", 
" -.27079142e+01  .72902693e+00  .40739285e+01 -.50629311e+01  .13228693e-03", 
" -.26990582e+01  .76431474e+00  .37612055e+01 -.38268577e+00  .13223585e-03", 
" -.26902269e+01  .79938976e+00  .34929019e+01  .35895460e+01  .13139026e-03", 
" -.26814040e+01  .83439184e+00  .31172510e+01  .95975908e+01  .13070049e-03", 
" -.26725774e+01  .85686116e+00  .34588154e+01  .55350637e+01  .13090218e-03", 
" -.26637953e+01  .88667748e+00  .34627501e+01  .58235234e+01  .13112851e-03", 
" -.26550043e+01  .91586181e+00  .34355009e+01  .60206994e+01  .13080156e-03", 
" -.26462183e+01  .93793000e+00  .37097668e+01  .30930574e+01  .13089189e-03", 
" -.26374871e+01  .97049747e+00  .34856186e+01  .63463460e+01  .13142878e-03", 
" -.26287357e+01  .99647732e+00  .35941073e+01  .48489590e+01  .13100264e-03", 
" -.26199777e+01  .10248607e+01  .34798506e+01  .66769665e+01  .13179127e-03", 
" -.26112402e+01  .10508839e+01  .34803250e+01  .74857305e+01  .13200999e-03", 
" -.26025230e+01  .10736314e+01  .37468467e+01  .43092557e+01  .13163365e-03", 
" -.25938098e+01  .11078444e+01  .32611716e+01  .11651811e+02  .13235688e-03", 
" -.25850593e+01  .11303697e+01  .32744802e+01  .13037988e+02  .13205331e-03", 
" -.25763384e+01  .11599138e+01  .30556114e+01  .16160666e+02  .13247128e-03", 
" -.25676076e+01  .11811214e+01  .32968247e+01  .12998009e+02  .13258120e-03", 
" -.25588559e+01  .12010643e+01  .35127609e+01  .10866765e+02  .13276735e-03", 
" -.25501319e+01  .12266915e+01  .35899672e+01  .87596046e+01  .13270153e-03", 
" -.25414050e+01  .12556654e+01  .33376115e+01  .12585183e+02  .13323724e-03", 
" -.25326478e+01  .12785389e+01  .33983722e+01  .12039681e+02  .13164318e-03", 
" -.25239291e+01  .13078148e+01  .31496011e+01  .15937781e+02  .13201258e-03", 
" -.25151889e+01  .13323355e+01  .31757808e+01  .15151296e+02  .13137173e-03", 
" -.25064270e+01  .13526384e+01  .33637259e+01  .12927669e+02  .13093992e-03", 
" -.24976546e+01  .13802834e+01  .31513363e+01  .15795348e+02  .13100512e-03", 
" -.24888519e+01  .14028441e+01  .31387380e+01  .16124422e+02  .13102061e-03", 
" -.24800414e+01  .14263838e+01  .31028818e+01  .16765086e+02  .13180184e-03", 
" -.24711734e+01  .14390018e+01  .35755151e+01  .11024262e+02  .13127714e-03", 
" -.24623455e+01  .14628684e+01  .35668241e+01  .10935081e+02  .13119950e-03", 
" -.24534434e+01  .14797997e+01  .37916296e+01  .84060459e+01  .13078106e-03", 
" -.24445633e+01  .15023903e+01  .37728195e+01  .88111989e+01  .13137264e-03", 
" -.24356615e+01  .15263446e+01  .36910253e+01  .95641760e+01  .13049248e-03", 
" -.24267158e+01  .15467140e+01  .37723029e+01  .81984873e+01  .13116154e-03", 
" -.24177444e+01  .15675386e+01  .38060014e+01  .78804552e+01  .13150059e-03", 
" -.24087379e+01  .15904423e+01  .37112613e+01  .89399740e+01  .13116520e-03", 
" -.23997452e+01  .16222562e+01  .31066908e+01  .17458203e+02  .13107824e-03", 
" -.23906670e+01  .16386439e+01  .33097187e+01  .15337494e+02  .13179976e-03", 
" -.23815663e+01  .16599831e+01  .32549092e+01  .16747380e+02  .13072510e-03", 
" -.23724294e+01  .16814130e+01  .32505626e+01  .16884104e+02  .13182602e-03", 
" -.23632627e+01  .17003962e+01  .33202691e+01  .16347350e+02  .13235871e-03", 
" -.23540572e+01  .17238553e+01  .31561371e+01  .18540142e+02  .13231800e-03", 
" -.23447843e+01  .17382090e+01  .34688937e+01  .14379872e+02  .13180604e-03", 
" -.23354721e+01  .17594646e+01  .33508001e+01  .16310661e+02  .13136359e-03", 
" -.23261293e+01  .17819282e+01  .32367631e+01  .17507752e+02  .13161247e-03", 
" -.23167176e+01  .18031463e+01  .31637423e+01  .18634417e+02  .13353676e-03", 
" -.23072520e+01  .18213280e+01  .32796147e+01  .16842024e+02  .13355636e-03", 
" -.22977520e+01  .18419501e+01  .31949857e+01  .18674052e+02  .13387368e-03", 
" -.22881805e+01  .18609653e+01  .32093095e+01  .19134496e+02  .13480548e-03", 
" -.22785652e+01  .18817109e+01  .31640128e+01  .19940459e+02  .13428978e-03", 
" -.22688624e+01  .18998458e+01  .32852577e+01  .18302961e+02  .13372997e-03", 
" -.22591030e+01  .19179886e+01  .34100081e+01  .16360873e+02  .13481943e-03", 
" -.22492744e+01  .19382420e+01  .33711012e+01  .17047880e+02  .13414931e-03", 
" -.22393758e+01  .19591830e+01  .33421954e+01  .17152269e+02  .13514559e-03", 
" -.22294187e+01  .19881747e+01  .29137054e+01  .22113459e+02  .13600443e-03", 
" -.22193763e+01  .20141595e+01  .25589946e+01  .27111044e+02  .13708074e-03", 
" -.22091943e+01  .20314510e+01  .25645179e+01  .28186579e+02  .13763991e-03", 
" -.21989697e+01  .20573865e+01  .22082206e+01  .33450771e+02  .13727297e-03", 
" -.21886067e+01  .20741051e+01  .23162356e+01  .32322670e+02  .13687510e-03", 
" -.21781408e+01  .20870115e+01  .26738720e+01  .27051530e+02  .13759582e-03", 
" -.21675808e+01  .21005220e+01  .29926252e+01  .23116203e+02  .13780736e-03", 
" -.21568945e+01  .21139547e+01  .32554684e+01  .19876912e+02  .13874200e-03", 
" -.21461384e+01  .21349391e+01  .32610740e+01  .19383746e+02  .13905446e-03", 
" -.21352429e+01  .21496359e+01  .35258910e+01  .16353157e+02  .13893732e-03", 
" -.21242273e+01  .21713166e+01  .34309430e+01  .17682249e+02  .13911113e-03", 
" -.21130781e+01  .21918650e+01  .33755286e+01  .19135651e+02  .14140293e-03", 
" -.21017523e+01  .22067536e+01  .36177981e+01  .16746221e+02  .14270959e-03", 
" -.20903076e+01  .22226857e+01  .38335154e+01  .14464843e+02  .14316583e-03", 
" -.20787297e+01  .22499405e+01  .35144872e+01  .19025735e+02  .14410490e-03", 
" -.20670032e+01  .22783621e+01  .30795905e+01  .25692019e+02  .14601143e-03", 
" -.20551277e+01  .23081806e+01  .26598000e+01  .31526226e+02  .14615639e-03", 
" -.20430164e+01  .23295682e+01  .26140453e+01  .33154164e+02  .14756840e-03", 
" -.20306891e+01  .23489358e+01  .25696460e+01  .35098516e+02  .14846882e-03", 
" -.20181361e+01  .23615490e+01  .31178624e+01  .27750503e+02  .15057640e-03", 
" -.20054078e+01  .23801828e+01  .34228140e+01  .23509703e+02  .15212916e-03", 
" -.19924100e+01  .23998986e+01  .34692517e+01  .24483183e+02  .15477440e-03", 
" -.19792394e+01  .24314925e+01  .28968075e+01  .34106512e+02  .15480145e-03", 
" -.19657642e+01  .24586493e+01  .25805293e+01  .39601828e+02  .15623226e-03", 
" -.19520135e+01  .24819522e+01  .25282649e+01  .42291781e+02  .15775375e-03", 
" -.19379529e+01  .25027560e+01  .25566819e+01  .44515653e+02  .16019627e-03", 
" -.19235948e+01  .25236447e+01  .27330009e+01  .44098043e+02  .16060809e-03", 
" -.19089105e+01  .25498169e+01  .28288877e+01  .43026193e+02  .16232851e-03", 
" -.18938540e+01  .25776982e+01  .27729576e+01  .45258279e+02  .16479591e-03", 
" -.18784454e+01  .26084647e+01  .26715097e+01  .47996132e+02  .16636624e-03", 
" -.18625863e+01  .26370594e+01  .26811371e+01  .49733949e+02  .17009157e-03", 
" -.18463152e+01  .26721051e+01  .24566838e+01  .55427228e+02  .17294376e-03", 
" -.18295708e+01  .27205470e+01  .16866785e+01  .66804388e+02  .17506706e-03", 
" -.18122199e+01  .27439770e+01  .20515556e+01  .66273795e+02  .17686553e-03", 
" -.17943115e+01  .27805295e+01  .19473579e+01  .70372975e+02  .18128510e-03", 
" -.17758428e+01  .28398643e+01  .68873741e+00  .92365140e+02  .18408898e-03", 
" -.17565700e+01  .28704310e+01  .98081257e+00  .93692999e+02  .18877392e-03", 
" -.17366050e+01  .29135509e+01  .88495022e+00  .10098762e+03  .19437387e-03", 
" -.17156821e+01  .29386076e+01  .17494411e+01  .96999792e+02  .19759627e-03", 
" -.16938872e+01  .29801824e+01  .21649352e+01  .96125509e+02  .19974501e-03", 
" -.16711003e+01  .30452178e+01  .15102825e+01  .10994460e+03  .20424990e-03", 
" -.16471135e+01  .31177082e+01  .60025974e+00  .12722026e+03  .20854144e-03", 
" -.16218335e+01  .31826893e+01  .37599283e+00  .13496849e+03  .21494827e-03", 
" -.15949174e+01  .32292670e+01  .16556271e+01  .12193739e+03  .22043625e-03", 
" -.15662420e+01  .33078187e+01  .14219401e+01  .12883213e+03  .22516000e-03", 
" -.15356109e+01  .33813182e+01  .19564618e+01  .12422492e+03  .23506692e-03", 
" -.15024081e+01  .34557191e+01  .25414522e+01  .11918045e+03  .24527318e-03", 
" -.14664446e+01  .35359697e+01  .33756709e+01  .11043185e+03  .25789596e-03", 
" -.14269787e+01  .36120358e+01  .48799517e+01  .91927397e+02  .26912501e-03", 
" -.13836136e+01  .37603318e+01  .33086784e+01  .11072580e+03  .28672505e-03", 
" -.13345828e+01  .38600388e+01  .39049643e+01  .10573340e+03  .30071572e-03", 
" -.12788072e+01  .39610064e+01  .54152225e+01  .86825133e+02  .31718890e-03", 
" -.12139419e+01  .41249298e+01  .39068051e+01  .11068911e+03  .33089657e-03", 
" -.11361957e+01  .43161514e+01  .25346930e+01  .12834096e+03  .35525424e-03", 
" -.10384252e+01  .45927838e+01 -.21582739e+01  .19162197e+03  .40305353e-03", 
" -.90463204e+00  .47408919e+01  .24284878e+01  .14631760e+03  .49450271e-03", 
" -.87065859e+00  .47886968e+01  .21243504e+01  .16118888e+03  .51462395e-03", 
" -.83308456e+00  .48064054e+01  .44846437e+01  .13454917e+03  .54331783e-03", 
" -.79078908e+00  .47743822e+01  .95058765e+01  .76352915e+02  .57445109e-03", 
" -.74319970e+00  .48890094e+01  .87511714e+01  .82910369e+02  .60155465e-03", 
" -.68752188e+00  .49515373e+01  .10354235e+02  .72666017e+02  .65298072e-03", 
" -.62078807e+00  .50917432e+01  .91434751e+01  .10439170e+03  .71437668e-03", 
" -.53610228e+00  .51699802e+01  .12907562e+02  .66622295e+02  .79413604e-03", 
" -.42062161e+00  .54388890e+01  .12536632e+02  .78689571e+02  .92711852e-03", 
" -.23158398e+00  .57743419e+01  .14872920e+02  .12928733e+03  .12600497e-02", 
" -.53439338e-01  .63370922e+01  .76830910e+01  .27469126e+03  .16928855e-02", 
"  .17710437e+00  .73707772e+01 -.14125371e+02  .68085398e+03  .25975406e-02", 
"  .33663426e+00  .73939067e+01  .87448337e+01  .52107645e+03  .35576610e-02", 
"dfanc     0  1  2   20", " -.30801328e+02  .26888417e+03 -.13957430e+04  .44569537e-01", 
" -.28162007e+02  .22507681e+03 -.10625566e+04  .32002582e-01", 
" -.24692068e+02  .17536107e+03 -.74586770e+03  .19366415e-01", 
" -.22103908e+02  .14364084e+03 -.57279244e+03  .13353128e-01", 
" -.19537123e+02  .11497298e+03 -.42328353e+03  .94706675e-02", 
" -.18051909e+02  .10042749e+03 -.36143667e+03  .77633639e-02", 
" -.16999593e+02  .90115774e+02 -.30938065e+03  .65945424e-02", 
" -.16183148e+02  .82456256e+02 -.27388162e+03  .59713081e-02", 
" -.15520522e+02  .76468159e+02 -.24590960e+03  .54017208e-02", 
" -.14963058e+02  .71690598e+02 -.22430900e+03  .48795638e-02", 
" -.14484982e+02  .68041985e+02 -.21384487e+03  .45789671e-02", 
" -.14061211e+02  .64588007e+02 -.19829896e+03  .43217225e-02", 
" -.13683454e+02  .61641536e+02 -.18680104e+03  .40786416e-02", 
" -.12240139e+02  .51160472e+02 -.14980986e+03  .33528690e-02", 
" -.11221677e+02  .44198632e+02 -.12392563e+03  .28541574e-02", 
" -.10436253e+02  .39021718e+02 -.10248145e+03  .25735961e-02", 
" -.98007132e+01  .35398563e+02 -.94681395e+02  .23279303e-02", 
" -.92651493e+01  .32337901e+02 -.84754462e+02  .21072081e-02", 
" -.88035581e+01  .29766346e+02 -.76521362e+02  .19554062e-02", 
" -.83983012e+01  .27581924e+02 -.69410497e+02  .18219696e-02", 
" -.80376980e+01  .25687576e+02 -.62832295e+02  .17097844e-02", 
" -.77138541e+01  .24137493e+02 -.58928808e+02  .16360994e-02", 
" -.74188359e+01  .22702232e+02 -.54355437e+02  .15642777e-02", 
" -.71484006e+01  .21419986e+02 -.50666496e+02  .14836636e-02", 
" -.68990821e+01  .20273979e+02 -.47248734e+02  .14489951e-02", 
" -.66681049e+01  .19248659e+02 -.44152829e+02  .13832026e-02", 
" -.64522787e+01  .18264077e+02 -.40844729e+02  .13449843e-02", 
" -.62502978e+01  .17347534e+02 -.37338359e+02  .12824613e-02", 
" -.60612705e+01  .16573567e+02 -.35506116e+02  .12442842e-02", 
" -.58824357e+01  .15817174e+02 -.33026164e+02  .12011203e-02", 
" -.57135126e+01  .15148343e+02 -.31326324e+02  .11528078e-02", 
" -.55535840e+01  .14518300e+02 -.29746002e+02  .11072519e-02", 
" -.54012556e+01  .13929493e+02 -.28175879e+02  .10769480e-02", 
" -.52565360e+01  .13397198e+02 -.27105171e+02  .10458183e-02", 
" -.51184302e+01  .12903803e+02 -.26182053e+02  .10235272e-02", 
" -.49866300e+01  .12432267e+02 -.25047859e+02  .10023983e-02", 
" -.48602759e+01  .11996184e+02 -.24032794e+02  .95612357e-03", 
" -.47387887e+01  .11553232e+02 -.22662545e+02  .92078471e-03", 
" -.46224038e+01  .11173011e+02 -.22090304e+02  .90476899e-03", 
" -.45100801e+01  .10779282e+02 -.21016942e+02  .88776044e-03", 
" -.44019547e+01  .10406615e+02 -.20102271e+02  .87827067e-03", 
" -.42976540e+01  .10049102e+02 -.19018064e+02  .85574241e-03", 
" -.41971907e+01  .96973194e+01 -.17531834e+02  .84756748e-03", 
" -.40999651e+01  .93578814e+01 -.16287431e+02  .82569467e-03", 
" -.40061182e+01  .90634698e+01 -.15514335e+02  .80402887e-03", 
" -.39149920e+01  .87695008e+01 -.14577527e+02  .79066685e-03", 
" -.38266302e+01  .84735404e+01 -.13486943e+02  .77427883e-03", 
" -.37411133e+01  .82177329e+01 -.12892142e+02  .76224153e-03", 
" -.36581740e+01  .79704842e+01 -.12299536e+02  .75710118e-03", 
" -.35777268e+01  .77505568e+01 -.12198441e+02  .74252891e-03", 
" -.34994222e+01  .75145898e+01 -.11451271e+02  .73017467e-03", 
" -.34235594e+01  .73120374e+01 -.11187618e+02  .72059892e-03", 
" -.33494836e+01  .70926124e+01 -.10386035e+02  .70670752e-03", 
" -.32774503e+01  .69014222e+01 -.10142180e+02  .69101641e-03", 
" -.32071654e+01  .66959683e+01 -.94578447e+01  .67337329e-03", 
" -.31388368e+01  .65146508e+01 -.90507146e+01  .65512660e-03", 
" -.30718505e+01  .63139710e+01 -.82293261e+01  .64345476e-03", 
" -.30066242e+01  .61449830e+01 -.79142933e+01  .63328369e-03", 
" -.29430419e+01  .59828396e+01 -.76568982e+01  .62915825e-03", 
" -.28807899e+01  .57990730e+01 -.68588520e+01  .61949652e-03", 
" -.28201381e+01  .56392507e+01 -.64224006e+01  .60727875e-03", 
" -.27607685e+01  .54903331e+01 -.61307904e+01  .60514304e-03", 
" -.27027986e+01  .53515365e+01 -.59380549e+01  .59030993e-03", 
" -.26460310e+01  .52114713e+01 -.56466044e+01  .58251875e-03", 
" -.25904949e+01  .50815694e+01 -.56162556e+01  .57630400e-03", 
" -.25359407e+01  .49384203e+01 -.51624372e+01  .56408727e-03", 
" -.24826696e+01  .48100456e+01 -.49392107e+01  .55335545e-03", 
" -.24306041e+01  .46966131e+01 -.48897920e+01  .54244830e-03", 
" -.23794852e+01  .45741191e+01 -.46097198e+01  .53602803e-03", 
" -.23294021e+01  .44717609e+01 -.47452434e+01  .52396209e-03", 
" -.22803463e+01  .43667535e+01 -.47182688e+01  .51746149e-03", 
" -.22322975e+01  .42715801e+01 -.47279876e+01  .51065377e-03", 
" -.21851242e+01  .41784637e+01 -.47778215e+01  .50529098e-03", 
" -.21387880e+01  .40790566e+01 -.47546664e+01  .50184130e-03", 
" -.20932860e+01  .39833225e+01 -.45705919e+01  .49246328e-03", 
" -.20486430e+01  .38964769e+01 -.46209620e+01  .48104023e-03", 
" -.20047016e+01  .37992605e+01 -.44285644e+01  .47528257e-03", 
" -.19615398e+01  .37052221e+01 -.42346632e+01  .46771974e-03", 
" -.19192265e+01  .36262702e+01 -.42442973e+01  .46051147e-03", 
" -.18774803e+01  .35390572e+01 -.41685084e+01  .44760964e-03", 
" -.18366344e+01  .34698231e+01 -.42837493e+01  .44105781e-03", 
" -.17962658e+01  .33804210e+01 -.39620914e+01  .43888763e-03", 
" -.17566147e+01  .32872945e+01 -.35346092e+01  .43265647e-03", 
" -.17176677e+01  .32129003e+01 -.33687261e+01  .42415290e-03", 
" -.16792990e+01  .31348338e+01 -.31749184e+01  .41707558e-03", 
" -.16415848e+01  .30663236e+01 -.31555113e+01  .41404273e-03", 
" -.16043436e+01  .29787248e+01 -.26721131e+01  .40587147e-03", 
" -.15676891e+01  .29053631e+01 -.25460262e+01  .40217872e-03", 
" -.15317561e+01  .28482786e+01 -.26748305e+01  .39738248e-03", 
" -.14962307e+01  .27800775e+01 -.25876453e+01  .39253735e-03", 
" -.14612660e+01  .27141253e+01 -.24026689e+01  .38789448e-03", 
" -.14269027e+01  .26610214e+01 -.25266099e+01  .38469286e-03", 
" -.13929851e+01  .26068131e+01 -.25970414e+01  .38211957e-03", 
" -.13595908e+01  .25577415e+01 -.26491125e+01  .37934680e-03", 
" -.13266295e+01  .25071892e+01 -.27388524e+01  .37742541e-03", 
" -.12939891e+01  .24436747e+01 -.25452575e+01  .37465899e-03", 
" -.12618718e+01  .23849735e+01 -.23487554e+01  .37007241e-03", 
" -.12301039e+01  .23271417e+01 -.22239398e+01  .36558438e-03", 
" -.11988702e+01  .22712517e+01 -.20535778e+01  .36205732e-03", 
" -.11680111e+01  .22194598e+01 -.19350591e+01  .35699448e-03", 
" -.11375143e+01  .21655563e+01 -.17841743e+01  .35152742e-03", 
" -.11074317e+01  .21185959e+01 -.17702819e+01  .35060929e-03", 
" -.10778075e+01  .20797973e+01 -.18534106e+01  .34429282e-03", 
" -.10485372e+01  .20378514e+01 -.18165767e+01  .34338724e-03", 
" -.10195175e+01  .19882071e+01 -.16632235e+01  .34029370e-03", 
" -.99090915e+00  .19509723e+01 -.16953729e+01  .33646095e-03", 
" -.96261282e+00  .19100686e+01 -.16316560e+01  .33083357e-03", 
" -.93455685e+00  .18657166e+01 -.15038651e+01  .32644580e-03", 
" -.90679388e+00  .18155247e+01 -.12434422e+01  .32358142e-03", 
" -.87953968e+00  .17920704e+01 -.14558756e+01  .31888769e-03", 
" -.85246854e+00  .17581464e+01 -.14321282e+01  .31443708e-03", 
" -.82567890e+00  .17264775e+01 -.14529704e+01  .31062630e-03", 
" -.79914040e+00  .16929458e+01 -.14493836e+01  .30951730e-03", 
" -.77294594e+00  .16646868e+01 -.14833415e+01  .30582875e-03", 
" -.74691842e+00  .16271214e+01 -.13053820e+01  .30266863e-03", 
" -.72112176e+00  .15971871e+01 -.12774697e+01  .29859758e-03", 
" -.69567532e+00  .15764273e+01 -.13702397e+01  .29532598e-03", 
" -.67037828e+00  .15453111e+01 -.11978334e+01  .29667395e-03", 
" -.64531709e+00  .15250657e+01 -.13051476e+01  .29372894e-03", 
" -.62047989e+00  .15010599e+01 -.12599769e+01  .29139892e-03", 
" -.59577287e+00  .14712701e+01 -.10967715e+01  .28808432e-03", 
" -.57124064e+00  .14351364e+01 -.76046502e+00  .28360663e-03", 
" -.54696872e+00  .14139227e+01 -.75205466e+00  .28216394e-03", 
" -.52287012e+00  .13939154e+01 -.74296789e+00  .28012067e-03", 
" -.49883507e+00  .13650715e+01 -.62253485e+00  .28025828e-03", 
" -.47520154e+00  .13573808e+01 -.84081433e+00  .27698803e-03", 
" -.45164274e+00  .13350360e+01 -.72132339e+00  .27383243e-03", 
" -.42820728e+00  .13168117e+01 -.72711598e+00  .26947264e-03", 
" -.40500780e+00  .13058736e+01 -.88052639e+00  .26679048e-03", 
" -.38198015e+00  .12945352e+01 -.96769554e+00  .26432684e-03", 
" -.35896092e+00  .12755502e+01 -.98594081e+00  .26285344e-03", 
" -.33614779e+00  .12609189e+01 -.10511345e+01  .26099217e-03", 
" -.31353392e+00  .12477575e+01 -.11241527e+01  .25793450e-03", 
" -.29101509e+00  .12328623e+01 -.12081441e+01  .25442320e-03", 
" -.26872329e+00  .12159038e+01 -.11770838e+01  .25213396e-03", 
" -.24639727e+00  .11904211e+01 -.10752311e+01  .24867242e-03", 
" -.22433907e+00  .11665634e+01 -.91894821e+00  .24548956e-03", 
" -.20239802e+00  .11474189e+01 -.87760493e+00  .24122921e-03", 
" -.18058435e+00  .11261264e+01 -.79741689e+00  .24032027e-03", 
" -.15899835e+00  .11094119e+01 -.77471861e+00  .23743970e-03", 
" -.13743784e+00  .10852324e+01 -.61231971e+00  .23443876e-03", 
" -.11603475e+00  .10654631e+01 -.52109629e+00  .23583027e-03", 
" -.94796402e-01  .10499103e+01 -.53569810e+00  .23178695e-03", 
" -.73753607e-01  .10411414e+01 -.65738532e+00  .23153618e-03", 
" -.52713489e-01  .10192498e+01 -.54638397e+00  .22987437e-03", 
" -.31901198e-01  .10105092e+01 -.70361631e+00  .22780985e-03", 
" -.11143305e-01  .99541006e+00 -.66366067e+00  .22671456e-03", 
"  .95026329e-02  .98464381e+00 -.73950828e+00  .22683673e-03", 
"  .30047214e-01  .97160920e+00 -.74439363e+00  .22512765e-03", 
"  .50511670e-01  .95229483e+00 -.58650513e+00  .22355043e-03", 
"  .70863736e-01  .94337234e+00 -.66194969e+00  .22092014e-03", 
"  .91193378e-01  .92970952e+00 -.63866569e+00  .21829709e-03", 
"  .11142007e+00  .91610654e+00 -.62120264e+00  .21674809e-03", 
"  .13164437e+00  .89800934e+00 -.49400766e+00  .21414049e-03", 
"  .15173832e+00  .88573625e+00 -.48131969e+00  .21219328e-03", 
"  .17174280e+00  .87525141e+00 -.47894818e+00  .21280851e-03", 
"  .19179122e+00  .85435751e+00 -.27878403e+00  .20940193e-03", 
"  .21177843e+00  .84140868e+00 -.22234935e+00  .20779296e-03", 
"  .23166172e+00  .83060713e+00 -.12905140e+00  .20840385e-03", 
"  .25153988e+00  .82248229e+00 -.14795276e+00  .20642987e-03", 
"  .27149783e+00  .80643623e+00  .12492295e-01  .20626128e-03", 
"  .29131050e+00  .79859684e+00  .28415235e-01  .20556710e-03", 
"  .31113388e+00  .79376700e+00  .19485946e-01  .20337691e-03", 
"  .33096381e+00  .78542128e+00  .11840985e+00  .20299467e-03", 
"  .35082557e+00  .77784177e+00  .19781718e+00  .20229637e-03", 
"  .37070176e+00  .77196436e+00  .24749366e+00  .20241733e-03", 
"  .39059468e+00  .76922007e+00  .24542048e+00  .20269936e-03", 
"  .41054751e+00  .76501991e+00  .34323763e+00  .20159899e-03", 
"  .43061619e+00  .75584590e+00  .47228460e+00  .19916944e-03", 
"  .45071835e+00  .75279534e+00  .51710692e+00  .19786668e-03", 
"  .47101118e+00  .74728711e+00  .58919249e+00  .19593445e-03", 
"  .49130699e+00  .74640975e+00  .64120298e+00  .19522635e-03", 
"  .51173098e+00  .75085292e+00  .59995706e+00  .19388984e-03", 
"  .53225129e+00  .75264849e+00  .65361815e+00  .19077765e-03", 
"  .55296741e+00  .75309746e+00  .75869396e+00  .19073120e-03", 
"  .57389787e+00  .75426916e+00  .81645130e+00  .19044704e-03", 
"  .59498464e+00  .76382681e+00  .72037050e+00  .19091304e-03", 
"  .61628216e+00  .76953690e+00  .75149265e+00  .19305829e-03", 
"  .63787723e+00  .77072191e+00  .89752078e+00  .19547346e-03", 
"  .65959392e+00  .78537866e+00  .85604731e+00  .19756513e-03", 
"  .68174226e+00  .79204438e+00  .95790354e+00  .19851707e-03", 
"  .70418719e+00  .80191481e+00  .98704929e+00  .19701387e-03", 
"  .72700790e+00  .81434247e+00  .10213894e+01  .19765422e-03", 
"  .75017394e+00  .83337167e+00  .96340334e+00  .19926249e-03", 
"  .77389653e+00  .83989381e+00  .11303563e+01  .19936098e-03", 
"  .79798680e+00  .86108724e+00  .11221565e+01  .20176401e-03", 
"  .82262847e+00  .88754859e+00  .10047691e+01  .20451458e-03", 
"  .84793939e+00  .90956921e+00  .10138958e+01  .20213787e-03", 
"  .87392773e+00  .93419443e+00  .10590266e+01  .20393423e-03", 
"  .90064970e+00  .96049217e+00  .11238001e+01  .20668117e-03", 
"  .92816750e+00  .99611255e+00  .10569969e+01  .20682900e-03", 
"  .95663848e+00  .10311538e+01  .10574843e+01  .20768287e-03", 
"  .98625398e+00  .10623600e+01  .12856663e+01  .20945471e-03", 
"  .10169919e+01  .11081806e+01  .12574817e+01  .21314028e-03", 
"  .10491569e+01  .11544556e+01  .13470263e+01  .21947752e-03", 
"  .10829547e+01  .11932199e+01  .16194694e+01  .22480499e-03", 
"  .11184268e+01  .12584179e+01  .15680963e+01  .23104879e-03", 
"  .11560394e+01  .13119807e+01  .18928992e+01  .23876297e-03", 
"  .11960884e+01  .13819793e+01  .21209744e+01  .24287685e-03", 
"  .12391317e+01  .14596832e+01  .23122612e+01  .24708106e-03", 
"  .12856195e+01  .15541232e+01  .24210556e+01  .26075966e-03", 
"  .13364559e+01  .16668911e+01  .24562485e+01  .27557465e-03", 
"  .13925894e+01  .18049310e+01  .24417948e+01  .28899120e-03", 
"  .14555587e+01  .19673393e+01  .24662744e+01  .30650377e-03", 
"  .15274917e+01  .21768035e+01  .22289625e+01  .33212043e-03", 
"  .16119138e+01  .24118241e+01  .28066836e+01  .36189111e-03", 
"  .17147977e+01  .27333884e+01  .30933679e+01  .39806627e-03", 
"  .18471010e+01  .31368017e+01  .45266749e+01  .45160792e-03", 
"  .20324163e+01  .38355853e+01  .51578425e+01  .53705018e-03", 
"  .20805488e+01  .40114196e+01  .57702783e+01  .57243563e-03", 
"  .21344760e+01  .42165752e+01  .66406844e+01  .61081464e-03", 
"  .21955679e+01  .44644923e+01  .77879919e+01  .65601379e-03", 
"  .22663040e+01  .47709784e+01  .82269169e+01  .71019241e-03", 
"  .23497562e+01  .51728739e+01  .85901091e+01  .79164133e-03", 
"  .24529727e+01  .55973697e+01  .10903176e+02  .88649852e-03", 
"  .25853898e+01  .62559473e+01  .12372269e+02  .10312938e-02", 
"  .27744414e+01  .71949803e+01  .16589875e+02  .12818612e-02", 
"  .31021508e+01  .86327433e+01  .30635959e+02  .17854089e-02", 
"  .34256768e+01  .10322716e+02  .46572748e+02  .25773256e-02", 
"  .38697969e+01  .13590952e+02  .57281532e+02  .41918701e-02", 
"  .41878713e+01  .15564476e+02  .81850153e+02  .58185791e-02", 
"dfac      0  2  2   20", " -.39107193e+02  .39170998e+03 -.21364919e+04  .43006598e-01", 
" -.36335908e+02  .34062222e+03 -.17719625e+04  .31598018e-01", 
" -.32672377e+02  .27784106e+03 -.13325189e+04  .19968368e-01", 
" -.29900653e+02  .23533795e+03 -.10628664e+04  .14173190e-01", 
" -.27092913e+02  .19456279e+03 -.82210571e+03  .10210394e-01", 
" -.25468763e+02  .17330128e+03 -.70612310e+03  .83817336e-02", 
" -.24311748e+02  .15886865e+03 -.63172814e+03  .72069327e-02", 
" -.23413213e+02  .14811744e+03 -.57864372e+03  .64581540e-02", 
" -.22679023e+02  .13950761e+03 -.53516500e+03  .59819270e-02", 
" -.22059845e+02  .13275360e+03 -.50459067e+03  .55926835e-02", 
" -.21518133e+02  .12644245e+03 -.46848855e+03  .51608231e-02", 
" -.21045861e+02  .12136826e+03 -.44383391e+03  .49244270e-02", 
" -.20617121e+02  .11663524e+03 -.41975921e+03  .47586282e-02", 
" -.18975991e+02  .99933809e+02 -.34625653e+03  .37303822e-02", 
" -.17811289e+02  .88617706e+02 -.29139262e+03  .32841892e-02", 
" -.16907547e+02  .80412087e+02 -.25544014e+03  .29692466e-02", 
" -.16168489e+02  .74000785e+02 -.22866052e+03  .26211199e-02", 
" -.15541605e+02  .68706425e+02 -.20654322e+03  .23980753e-02", 
" -.14999412e+02  .64363237e+02 -.19005221e+03  .22996793e-02", 
" -.14519493e+02  .60588211e+02 -.17591453e+03  .22116865e-02", 
" -.14089391e+02  .57237432e+02 -.16216044e+03  .21095266e-02", 
" -.13700418e+02  .54421909e+02 -.15388938e+03  .20138254e-02", 
" -.13345394e+02  .51896834e+02 -.14521630e+03  .19479386e-02", 
" -.13018246e+02  .49628017e+02 -.13787560e+03  .18772027e-02", 
" -.12715515e+02  .47550937e+02 -.13071178e+03  .17929125e-02", 
" -.12433471e+02  .45694993e+02 -.12472929e+03  .17550962e-02", 
" -.12168224e+02  .43870362e+02 -.11760467e+03  .16744268e-02", 
" -.11919474e+02  .42234637e+02 -.11139461e+03  .16272829e-02", 
" -.11684504e+02  .40756159e+02 -.10698605e+03  .15850951e-02", 
" -.11461621e+02  .39295335e+02 -.10115600e+03  .15274398e-02", 
" -.11249995e+02  .37973447e+02 -.96431026e+02  .14909991e-02", 
" -.11048079e+02  .36704835e+02 -.91751816e+02  .14520745e-02", 
" -.10855527e+02  .35527767e+02 -.87961368e+02  .14296997e-02", 
" -.10671721e+02  .34421956e+02 -.84054064e+02  .13955818e-02", 
" -.10495399e+02  .33370295e+02 -.80225982e+02  .13618934e-02", 
" -.10326334e+02  .32403405e+02 -.77508028e+02  .13375844e-02", 
" -.10164101e+02  .31523082e+02 -.75268107e+02  .13123936e-02", 
" -.10007817e+02  .30668069e+02 -.72716879e+02  .12896882e-02", 
" -.98568200e+01  .29836817e+02 -.70047930e+02  .12625033e-02", 
" -.97108154e+01  .29023901e+02 -.67173447e+02  .12274122e-02", 
" -.95697384e+01  .28295508e+02 -.65649319e+02  .12145222e-02", 
" -.94331608e+01  .27563940e+02 -.63222211e+02  .11865721e-02", 
" -.93005449e+01  .26847981e+02 -.60732267e+02  .11659478e-02", 
" -.91719456e+01  .26181862e+02 -.58681213e+02  .11418782e-02", 
" -.90470585e+01  .25527400e+02 -.56517257e+02  .11199237e-02", 
" -.89255540e+01  .24906971e+02 -.54745829e+02  .11004706e-02", 
" -.88077359e+01  .24342480e+02 -.53503354e+02  .10872347e-02", 
" -.86928836e+01  .23758041e+02 -.51410618e+02  .10788578e-02", 
" -.85805960e+01  .23176822e+02 -.49264453e+02  .10653335e-02", 
" -.84716253e+01  .22658729e+02 -.47807550e+02  .10435828e-02", 
" -.83651852e+01  .22165611e+02 -.46795295e+02  .10375167e-02", 
" -.82611023e+01  .21670876e+02 -.45465909e+02  .10115753e-02", 
" -.81594346e+01  .21198779e+02 -.44131879e+02  .99364695e-03", 
" -.80599552e+01  .20710922e+02 -.42135198e+02  .98767145e-03", 
" -.79626496e+01  .20241220e+02 -.40374198e+02  .97193414e-03", 
" -.78674265e+01  .19806010e+02 -.39149052e+02  .96614265e-03", 
" -.77744173e+01  .19423651e+02 -.38872955e+02  .96437335e-03", 
" -.76829850e+01  .19003281e+02 -.37518346e+02  .95748673e-03", 
" -.75934485e+01  .18605199e+02 -.36480508e+02  .94693976e-03", 
" -.75058094e+01  .18225388e+02 -.35390091e+02  .93999880e-03", 
" -.74194723e+01  .17856273e+02 -.34619312e+02  .92521864e-03", 
" -.73349651e+01  .17492890e+02 -.33711634e+02  .91456099e-03", 
" -.72524375e+01  .17188371e+02 -.33716344e+02  .91070611e-03", 
" -.71707805e+01  .16821439e+02 -.32377471e+02  .89988382e-03", 
" -.70905136e+01  .16477116e+02 -.31315347e+02  .88086840e-03", 
" -.70118240e+01  .16157467e+02 -.30603720e+02  .87300743e-03", 
" -.69343954e+01  .15827457e+02 -.29452028e+02  .86098722e-03", 
" -.68583402e+01  .15520992e+02 -.28578022e+02  .84824495e-03", 
" -.67835275e+01  .15213495e+02 -.27619127e+02  .84118731e-03", 
" -.67097357e+01  .14906655e+02 -.26488152e+02  .83036839e-03", 
" -.66372681e+01  .14632401e+02 -.25964496e+02  .81052308e-03", 
" -.65658355e+01  .14360147e+02 -.25436146e+02  .80508564e-03", 
" -.64956666e+01  .14102529e+02 -.24913056e+02  .80224416e-03", 
" -.64261980e+01  .13830011e+02 -.24064664e+02  .79608442e-03", 
" -.63578281e+01  .13576489e+02 -.23496164e+02  .79751575e-03", 
" -.62902595e+01  .13317959e+02 -.22833302e+02  .78855810e-03", 
" -.62235941e+01  .13064984e+02 -.22197798e+02  .77543475e-03", 
" -.61580390e+01  .12833749e+02 -.21817709e+02  .76741719e-03", 
" -.60932373e+01  .12588324e+02 -.21061307e+02  .75247531e-03", 
" -.60292683e+01  .12357573e+02 -.20480261e+02  .74826531e-03", 
" -.59661469e+01  .12129990e+02 -.19933239e+02  .74160712e-03", 
" -.59039024e+01  .11901800e+02 -.19140548e+02  .73709060e-03", 
" -.58425399e+01  .11702642e+02 -.18866404e+02  .72874377e-03", 
" -.57815131e+01  .11475271e+02 -.18057734e+02  .72093138e-03", 
" -.57216190e+01  .11282549e+02 -.17723349e+02  .71151798e-03", 
" -.56624357e+01  .11097087e+02 -.17585866e+02  .70646144e-03", 
" -.56036450e+01  .10893427e+02 -.17033505e+02  .70141082e-03", 
" -.55456635e+01  .10713359e+02 -.16816514e+02  .69326670e-03", 
" -.54882576e+01  .10524922e+02 -.16434864e+02  .68802806e-03", 
" -.54314404e+01  .10341816e+02 -.16162157e+02  .68220385e-03", 
" -.53749876e+01  .10134561e+02 -.15305977e+02  .67094126e-03", 
" -.53194259e+01  .99622561e+01 -.15090158e+02  .66147618e-03", 
" -.52644916e+01  .97960081e+01 -.14868194e+02  .65543204e-03", 
" -.52100107e+01  .96302003e+01 -.14623497e+02  .64711219e-03", 
" -.51560880e+01  .94593500e+01 -.14178054e+02  .63942647e-03", 
" -.51025892e+01  .92887543e+01 -.13675813e+02  .62972255e-03", 
" -.50496797e+01  .91263184e+01 -.13373032e+02  .62910506e-03", 
" -.49973868e+01  .89882921e+01 -.13460373e+02  .62346641e-03", 
" -.49453274e+01  .88210848e+01 -.12948750e+02  .61923818e-03", 
" -.48937814e+01  .86710259e+01 -.12801842e+02  .61463668e-03", 
" -.48427903e+01  .85179203e+01 -.12433109e+02  .61164259e-03", 
" -.47920941e+01  .83533615e+01 -.11808111e+02  .60555026e-03", 
" -.47419387e+01  .82157737e+01 -.11725535e+02  .59628620e-03", 
" -.46922515e+01  .80776828e+01 -.11566782e+02  .59357816e-03", 
" -.46428893e+01  .79282224e+01 -.11011531e+02  .58734713e-03", 
" -.45938536e+01  .77731046e+01 -.10372655e+02  .58596703e-03", 
" -.45452583e+01  .76300520e+01 -.99566638e+01  .58229115e-03", 
" -.44971056e+01  .74946398e+01 -.96212635e+01  .57573892e-03", 
" -.44493543e+01  .73722903e+01 -.95764596e+01  .56362596e-03", 
" -.44019680e+01  .72501612e+01 -.93380154e+01  .56040190e-03", 
" -.43548129e+01  .71222318e+01 -.90579610e+01  .55518797e-03", 
" -.43080619e+01  .70095045e+01 -.90217941e+01  .55131653e-03", 
" -.42616714e+01  .68920897e+01 -.88385522e+01  .54981133e-03", 
" -.42155254e+01  .67693495e+01 -.84607953e+01  .54490409e-03", 
" -.41697029e+01  .66555996e+01 -.82874789e+01  .54825100e-03", 
" -.41241908e+01  .65408620e+01 -.81102728e+01  .54286450e-03", 
" -.40790469e+01  .64317580e+01 -.79790945e+01  .53840930e-03", 
" -.40341884e+01  .63188883e+01 -.76296126e+01  .53509372e-03", 
" -.39894487e+01  .62050780e+01 -.73178272e+01  .52803565e-03", 
" -.39450530e+01  .60964150e+01 -.70378328e+01  .52691854e-03", 
" -.39009838e+01  .59989623e+01 -.69817081e+01  .52354552e-03", 
" -.38571334e+01  .58995963e+01 -.67891786e+01  .52526731e-03", 
" -.38135661e+01  .58076859e+01 -.67907661e+01  .52656304e-03", 
" -.37702150e+01  .57183360e+01 -.67948276e+01  .52175594e-03", 
" -.37269135e+01  .56176188e+01 -.64739186e+01  .51640682e-03", 
" -.36838459e+01  .55265429e+01 -.64001287e+01  .51541354e-03", 
" -.36411175e+01  .54317174e+01 -.61310328e+01  .51209890e-03", 
" -.35986172e+01  .53523510e+01 -.61807753e+01  .51018336e-03", 
" -.35562134e+01  .52622788e+01 -.60501368e+01  .50660094e-03", 
" -.35138609e+01  .51645704e+01 -.56573572e+01  .50380388e-03", 
" -.34715929e+01  .50638007e+01 -.52299970e+01  .50046206e-03", 
" -.34296771e+01  .49781046e+01 -.49955290e+01  .49911995e-03", 
" -.33877331e+01  .48838484e+01 -.47283582e+01  .49164570e-03", 
" -.33461122e+01  .47984584e+01 -.44938570e+01  .48784340e-03", 
" -.33046614e+01  .47171897e+01 -.42686156e+01  .48167569e-03", 
" -.32633054e+01  .46424415e+01 -.41442629e+01  .48343278e-03", 
" -.32219719e+01  .45623467e+01 -.39764142e+01  .48291449e-03", 
" -.31807762e+01  .44853089e+01 -.37671228e+01  .47954244e-03", 
" -.31396674e+01  .44081587e+01 -.36061400e+01  .47827438e-03", 
" -.30987919e+01  .43403269e+01 -.35844055e+01  .47367523e-03", 
" -.30580634e+01  .42845662e+01 -.36606287e+01  .47144269e-03", 
" -.30174228e+01  .42237397e+01 -.36143013e+01  .47195044e-03", 
" -.29767080e+01  .41556408e+01 -.34388748e+01  .46887391e-03", 
" -.29360820e+01  .40902323e+01 -.33075440e+01  .46885075e-03", 
" -.28955076e+01  .40317966e+01 -.32802810e+01  .47006224e-03", 
" -.28548932e+01  .39548242e+01 -.28607928e+01  .47283701e-03", 
" -.28144743e+01  .38995600e+01 -.27240062e+01  .47148425e-03", 
" -.27740962e+01  .38483843e+01 -.26301211e+01  .47286533e-03", 
" -.27336974e+01  .38036384e+01 -.27005181e+01  .46784109e-03", 
" -.26932230e+01  .37456700e+01 -.24732766e+01  .46152112e-03", 
" -.26528850e+01  .37038768e+01 -.25631593e+01  .45959718e-03", 
" -.26124522e+01  .36520953e+01 -.24229277e+01  .45821087e-03", 
" -.25718730e+01  .35847792e+01 -.19414159e+01  .45571339e-03", 
" -.25313234e+01  .35334534e+01 -.17771591e+01  .45551543e-03", 
" -.24907685e+01  .34876623e+01 -.16912604e+01  .45307128e-03", 
" -.24501762e+01  .34379098e+01 -.13798441e+01  .44944605e-03", 
" -.24094794e+01  .33942558e+01 -.12727392e+01  .44599316e-03", 
" -.23687814e+01  .33551715e+01 -.11107786e+01  .44110526e-03", 
" -.23280148e+01  .33188963e+01 -.96151454e+00  .43912840e-03", 
" -.22869300e+01  .32607510e+01 -.38764150e+00  .44016402e-03", 
" -.22460441e+01  .32368824e+01 -.44986766e+00  .43802452e-03", 
" -.22048285e+01  .32059971e+01 -.46122866e+00  .43679490e-03", 
" -.21633411e+01  .31554297e+01  .21651755e-01  .43331840e-03", 
" -.21217153e+01  .31113989e+01  .37605609e+00  .42881521e-03", 
" -.20799770e+01  .30714394e+01  .74629351e+00  .43143076e-03", 
" -.20380333e+01  .30417626e+01  .82930062e+00  .42910234e-03", 
" -.19958502e+01  .30103026e+01  .10069457e+01  .42507314e-03", 
" -.19534232e+01  .29794126e+01  .11912795e+01  .42014482e-03", 
" -.19109452e+01  .29619437e+01  .12134192e+01  .41571057e-03", 
" -.18680233e+01  .29310967e+01  .15043376e+01  .41625825e-03", 
" -.18249158e+01  .29156287e+01  .15496727e+01  .41775965e-03", 
" -.17814948e+01  .28953821e+01  .17727063e+01  .41973480e-03", 
" -.17375472e+01  .28670196e+01  .21268179e+01  .41778991e-03", 
" -.16932643e+01  .28505456e+01  .22196569e+01  .41829321e-03", 
" -.16485812e+01  .28315248e+01  .23620180e+01  .41730675e-03", 
" -.16034481e+01  .28143120e+01  .24984089e+01  .41444105e-03", 
" -.15578382e+01  .27959335e+01  .27854735e+01  .41190502e-03", 
" -.15117640e+01  .27873622e+01  .27340798e+01  .41322354e-03", 
" -.14653001e+01  .27957592e+01  .24797290e+01  .41134245e-03", 
" -.14181527e+01  .27936902e+01  .24740497e+01  .41058310e-03", 
" -.13704280e+01  .28005875e+01  .23016183e+01  .41539086e-03", 
" -.13219796e+01  .28018470e+01  .22008476e+01  .41551085e-03", 
" -.12728207e+01  .27952900e+01  .23826045e+01  .41470828e-03", 
" -.12229499e+01  .28016908e+01  .23109925e+01  .41362413e-03", 
" -.11722532e+01  .28092465e+01  .22393482e+01  .41363463e-03", 
" -.11203995e+01  .27973968e+01  .25737174e+01  .41718471e-03", 
" -.10677798e+01  .28138054e+01  .23842780e+01  .42136818e-03", 
" -.10139103e+01  .28193447e+01  .24077907e+01  .42586073e-03", 
" -.95888085e+00  .28046035e+01  .30837467e+01  .42530488e-03", 
" -.90282630e+00  .28113650e+01  .33746518e+01  .42345187e-03", 
" -.84518196e+00  .28290863e+01  .33066140e+01  .42924427e-03", 
" -.78603051e+00  .28440110e+01  .35626691e+01  .43259621e-03", 
" -.72522581e+00  .28662711e+01  .38103911e+01  .43921408e-03", 
" -.66250508e+00  .28880912e+01  .41576121e+01  .43901704e-03", 
" -.59763187e+00  .29265370e+01  .41449250e+01  .44373274e-03", 
" -.53043493e+00  .29712740e+01  .43448217e+01  .44529751e-03", 
" -.46031832e+00  .30087632e+01  .48199128e+01  .45402326e-03", 
" -.38717480e+00  .30687230e+01  .50399396e+01  .45503417e-03", 
" -.31010331e+00  .31086550e+01  .58359828e+01  .46291490e-03", 
" -.22871576e+00  .31698455e+01  .67177882e+01  .46859058e-03", 
" -.14269751e+00  .33048838e+01  .65008784e+01  .47001072e-03", 
" -.50180858e-01  .34136705e+01  .72313939e+01  .47117301e-03", 
"  .49917270e-01  .35707697e+01  .74569360e+01  .48274971e-03", 
"  .15962386e+00  .37151312e+01  .89501494e+01  .49988554e-03", 
"  .28166903e+00  .39445417e+01  .93914542e+01  .53751701e-03", 
"  .42079875e+00  .42059742e+01  .10870013e+02  .57098563e-03", 
"  .58435187e+00  .45429553e+01  .13030297e+02  .59930862e-03", 
"  .78583086e+00  .50190184e+01  .16524175e+02  .67827329e-03", 
"  .10551304e+01  .58729060e+01  .19753353e+02  .78977237e-03", 
"  .11230627e+01  .61189090e+01  .20403204e+02  .82167146e-03", 
"  .11977617e+01  .63701182e+01  .22351999e+02  .86910453e-03", 
"  .12805992e+01  .67727079e+01  .22254590e+02  .91600649e-03", 
"  .13758181e+01  .71042687e+01  .24908797e+02  .99648880e-03", 
"  .14860366e+01  .76425219e+01  .26900247e+02  .10573031e-02", 
"  .16191086e+01  .82609105e+01  .29557806e+02  .11784478e-02", 
"  .17888170e+01  .89824283e+01  .35354122e+02  .13358317e-02", 
"  .20246137e+01  .10007299e+02  .46398712e+02  .15323270e-02", 
"  .24156881e+01  .12048512e+02  .63933182e+02  .21243330e-02", 
"  .27943447e+01  .14324685e+02  .83358871e+02  .28779051e-02", 
"  .32989561e+01  .17864414e+02  .11876785e+03  .45460051e-02", 
"  .36473367e+01  .21973535e+02  .11766075e+03  .62428196e-02", 
"dfact     0  3  3   20", " -.49745391e+02  .62484775e+03 -.47792041e+04  .14174175e+05  .56898721e-01", 
" -.46852449e+02  .56980597e+03 -.46357722e+04  .18077112e+05  .39258864e-01", 
" -.42871914e+02  .48691365e+03 -.39464488e+04  .16832354e+05  .26311165e-01", 
" -.39804794e+02  .42148160e+03 -.32240352e+04  .13312782e+05  .18569457e-01", 
" -.36697187e+02  .35948742e+03 -.25899754e+04  .10370376e+05  .13605455e-01", 
" -.34871855e+02  .32474924e+03 -.22146638e+04  .83438073e+04  .10827219e-01", 
" -.33568135e+02  .30127080e+03 -.19676090e+04  .69320417e+04  .93278826e-02", 
" -.32550348e+02  .28384968e+03 -.18073460e+04  .62134219e+04  .86466759e-02", 
" -.31711782e+02  .26910272e+03 -.16582920e+04  .54688708e+04  .81324923e-02", 
" -.30997797e+02  .25651692e+03 -.15144127e+04  .45323141e+04  .76328854e-02", 
" -.30381388e+02  .24645749e+03 -.14268781e+04  .42025272e+04  .72209681e-02", 
" -.29836559e+02  .23808183e+03 -.13659542e+04  .40497895e+04  .67933079e-02", 
" -.29346684e+02  .22983392e+03 -.12694832e+04  .33883765e+04  .64789560e-02", 
" -.27454282e+02  .20164790e+03 -.10496164e+04  .26517942e+04  .54590803e-02", 
" -.26103265e+02  .18350964e+03 -.93970872e+03  .24958047e+04  .46570929e-02", 
" -.25046566e+02  .16930723e+03 -.82634097e+03  .19599223e+04  .42234429e-02", 
" -.24177380e+02  .15828157e+03 -.75106423e+03  .16999864e+04  .38838252e-02", 
" -.23435768e+02  .14901349e+03 -.68899382e+03  .15255796e+04  .36175687e-02", 
" -.22790648e+02  .14137646e+03 -.64289365e+03  .13953052e+04  .33997445e-02", 
" -.22216596e+02  .13429489e+03 -.58389751e+03  .10639884e+04  .31794193e-02", 
" -.21700977e+02  .12828930e+03 -.54359796e+03  .91840034e+03  .30506743e-02", 
" -.21233273e+02  .12316909e+03 -.51980047e+03  .93840209e+03  .29378012e-02", 
" -.20803333e+02  .11839239e+03 -.49178996e+03  .88159848e+03  .27673879e-02", 
" -.20407967e+02  .11425894e+03 -.47296153e+03  .89263986e+03  .26590460e-02", 
" -.20038230e+02  .11012773e+03 -.44105489e+03  .72436987e+03  .25631426e-02", 
" -.19694308e+02  .10658615e+03 -.42213481e+03  .68905453e+03  .24726615e-02", 
" -.19371036e+02  .10342348e+03 -.41082289e+03  .72388135e+03  .23972581e-02", 
" -.19065778e+02  .10041985e+03 -.39771964e+03  .73462360e+03  .23634846e-02", 
" -.18776350e+02  .97458382e+02 -.37871070e+03  .66175337e+03  .22795395e-02", 
" -.18501329e+02  .94654936e+02 -.35922302e+03  .55955816e+03  .22174426e-02", 
" -.18239738e+02  .92128515e+02 -.34679036e+03  .54636337e+03  .21912156e-02", 
" -.17989931e+02  .89737594e+02 -.33432468e+03  .52304943e+03  .21299032e-02", 
" -.17750478e+02  .87438689e+02 -.32223904e+03  .49920952e+03  .20951565e-02", 
" -.17521109e+02  .85238070e+02 -.30863406e+03  .44921394e+03  .20560488e-02", 
" -.17300573e+02  .83245886e+02 -.30165383e+03  .47519735e+03  .20051227e-02", 
" -.17089422e+02  .81523581e+02 -.30127690e+03  .56274976e+03  .19561314e-02", 
" -.16885347e+02  .79723884e+02 -.29274925e+03  .54070936e+03  .19252086e-02", 
" -.16688591e+02  .78017903e+02 -.28420500e+03  .50134698e+03  .18663444e-02", 
" -.16497531e+02  .76318500e+02 -.27553682e+03  .48542843e+03  .18365538e-02", 
" -.16312142e+02  .74560580e+02 -.26090003e+03  .38309531e+03  .18061048e-02", 
" -.16133042e+02  .73035214e+02 -.25530863e+03  .39805074e+03  .17531281e-02", 
" -.15959114e+02  .71545176e+02 -.24701231e+03  .36246513e+03  .17399200e-02", 
" -.15789854e+02  .70033309e+02 -.23584969e+03  .28792779e+03  .17065747e-02", 
" -.15625835e+02  .68666234e+02 -.22865316e+03  .25672534e+03  .16902497e-02", 
" -.15465728e+02  .67298970e+02 -.22059523e+03  .21839320e+03  .16792815e-02", 
" -.15309738e+02  .66038228e+02 -.21559126e+03  .21597013e+03  .16579203e-02", 
" -.15157822e+02  .64735041e+02 -.20546398e+03  .13943460e+03  .16044939e-02", 
" -.15009520e+02  .63500942e+02 -.19806537e+03  .10181458e+03  .15889219e-02", 
" -.14864767e+02  .62332623e+02 -.19196206e+03  .77332519e+02  .15739217e-02", 
" -.14723381e+02  .61202637e+02 -.18615184e+03  .57323546e+02  .15604070e-02", 
" -.14585486e+02  .60181948e+02 -.18433124e+03  .85342301e+02  .15435140e-02", 
" -.14450059e+02  .59099115e+02 -.17824451e+03  .58100049e+02  .15332725e-02", 
" -.14317118e+02  .58057969e+02 -.17419332e+03  .59581285e+02  .15271905e-02", 
" -.14187385e+02  .57126261e+02 -.17330457e+03  .97535630e+02  .15158509e-02", 
" -.14059835e+02  .56159723e+02 -.16932463e+03  .88643782e+02  .14916725e-02", 
" -.13935047e+02  .55283084e+02 -.16798799e+03  .10823486e+03  .14735142e-02", 
" -.13812635e+02  .54373377e+02 -.16479642e+03  .11742020e+03  .14519448e-02", 
" -.13692658e+02  .53578266e+02 -.16624857e+03  .18058224e+03  .14273013e-02", 
" -.13574774e+02  .52775424e+02 -.16616977e+03  .22009901e+03  .14101249e-02", 
" -.13458498e+02  .51877287e+02 -.16135020e+03  .20732105e+03  .14136654e-02", 
" -.13344756e+02  .51115163e+02 -.16085051e+03  .23448604e+03  .13988234e-02", 
" -.13232965e+02  .50372133e+02 -.16017787e+03  .26163096e+03  .13809287e-02", 
" -.13122497e+02  .49572742e+02 -.15677393e+03  .26007353e+03  .13609137e-02", 
" -.13013871e+02  .48805639e+02 -.15361320e+03  .25294135e+03  .13617212e-02", 
" -.12906549e+02  .48029315e+02 -.14993092e+03  .24025116e+03  .13416557e-02", 
" -.12801433e+02  .47334892e+02 -.14863372e+03  .25412844e+03  .13338146e-02", 
" -.12697463e+02  .46593813e+02 -.14510637e+03  .24605234e+03  .13204026e-02", 
" -.12595467e+02  .45937744e+02 -.14411453e+03  .26259009e+03  .13000512e-02", 
" -.12494209e+02  .45161432e+02 -.13762049e+03  .21397040e+03  .12986967e-02", 
" -.12395166e+02  .44597416e+02 -.14014402e+03  .27186093e+03  .12896081e-02", 
" -.12296905e+02  .43979890e+02 -.14076227e+03  .31512738e+03  .12764310e-02", 
" -.12200118e+02  .43341599e+02 -.13894174e+03  .32135185e+03  .12673879e-02", 
" -.12104397e+02  .42637713e+02 -.13295081e+03  .27370574e+03  .12555322e-02", 
" -.12010210e+02  .42025036e+02 -.13087086e+03  .27198160e+03  .12481652e-02", 
" -.11917589e+02  .41507370e+02 -.13249279e+03  .31635486e+03  .12481849e-02", 
" -.11825284e+02  .40843168e+02 -.12692241e+03  .27255118e+03  .12407673e-02", 
" -.11734403e+02  .40271129e+02 -.12543128e+03  .27956047e+03  .12284184e-02", 
" -.11644248e+02  .39658094e+02 -.12168389e+03  .25783990e+03  .12117269e-02", 
" -.11555451e+02  .39096567e+02 -.11933480e+03  .24918128e+03  .12063274e-02", 
" -.11467390e+02  .38540621e+02 -.11813344e+03  .26385804e+03  .12093627e-02", 
" -.11380558e+02  .38001542e+02 -.11692834e+03  .27743576e+03  .11919744e-02", 
" -.11294506e+02  .37470452e+02 -.11554046e+03  .28521516e+03  .11944330e-02", 
" -.11209426e+02  .36916386e+02 -.11232357e+03  .26619252e+03  .11845186e-02", 
" -.11125103e+02  .36374547e+02 -.10908613e+03  .24347514e+03  .11848266e-02", 
" -.11041545e+02  .35820641e+02 -.10477104e+03  .20913295e+03  .11858361e-02", 
" -.10958997e+02  .35282733e+02 -.10065428e+03  .17416001e+03  .11853457e-02", 
" -.10877091e+02  .34788251e+02 -.99104317e+02  .17445276e+03  .11863386e-02", 
" -.10796229e+02  .34308220e+02 -.97382346e+02  .17299620e+03  .11838803e-02", 
" -.10715675e+02  .33775904e+02 -.93603177e+02  .15005087e+03  .11825981e-02", 
" -.10636320e+02  .33354100e+02 -.94727064e+02  .18670813e+03  .11698236e-02", 
" -.10557555e+02  .32885309e+02 -.92628618e+02  .17821168e+03  .11596546e-02", 
" -.10479757e+02  .32462793e+02 -.92184800e+02  .19013856e+03  .11592561e-02", 
" -.10402305e+02  .32029597e+02 -.91412824e+02  .19782272e+03  .11489038e-02", 
" -.10325830e+02  .31647700e+02 -.92528345e+02  .22951257e+03  .11424856e-02", 
" -.10249326e+02  .31191967e+02 -.90805686e+02  .23178023e+03  .11337877e-02", 
" -.10173871e+02  .30792220e+02 -.90621517e+02  .24403755e+03  .11206051e-02", 
" -.10098814e+02  .30308374e+02 -.85527122e+02  .19067531e+03  .11172465e-02", 
" -.10024694e+02  .29948881e+02 -.86543074e+02  .21723823e+03  .11053499e-02", 
" -.99508272e+01  .29545051e+02 -.85489942e+02  .21989614e+03  .10997863e-02", 
" -.98775525e+01  .29144301e+02 -.83842212e+02  .21093407e+03  .10871300e-02", 
" -.98046191e+01  .28705003e+02 -.80227310e+02  .17760924e+03  .10801871e-02", 
" -.97322215e+01  .28281585e+02 -.77393242e+02  .15670492e+03  .10625920e-02", 
" -.96603922e+01  .27862297e+02 -.74421793e+02  .13385683e+03  .10533338e-02", 
" -.95892797e+01  .27518872e+02 -.75024295e+02  .15656816e+03  .10463383e-02", 
" -.95187011e+01  .27144055e+02 -.73030635e+02  .14093233e+03  .10339064e-02", 
" -.94484364e+01  .26770405e+02 -.71661889e+02  .14027211e+03  .10316206e-02", 
" -.93783216e+01  .26372895e+02 -.69324926e+02  .12465777e+03  .10273819e-02", 
" -.93087310e+01  .26013546e+02 -.68438496e+02  .12877678e+03  .10136187e-02", 
" -.92397933e+01  .25723720e+02 -.70604815e+02  .16816502e+03  .10117422e-02", 
" -.91709773e+01  .25394840e+02 -.71315834e+02  .19593361e+03  .10028094e-02", 
" -.91026538e+01  .25072299e+02 -.71695370e+02  .21375477e+03  .99627683e-03", 
" -.90347818e+01  .24726573e+02 -.70082063e+02  .20129931e+03  .99706900e-03", 
" -.89670592e+01  .24388480e+02 -.69711461e+02  .21178546e+03  .99163710e-03", 
" -.88999370e+01  .24075518e+02 -.69827439e+02  .22728476e+03  .99080215e-03", 
" -.88328706e+01  .23720764e+02 -.67995238e+02  .21848490e+03  .99579606e-03", 
" -.87661342e+01  .23351137e+02 -.65071559e+02  .19358394e+03  .99959569e-03", 
" -.86998109e+01  .23013511e+02 -.63534504e+02  .18789618e+03  .99614171e-03", 
" -.86338343e+01  .22672130e+02 -.60956478e+02  .16249387e+03  .98539254e-03", 
" -.85682781e+01  .22360182e+02 -.59826746e+02  .15656961e+03  .98345437e-03", 
" -.85027841e+01  .22025836e+02 -.58262962e+02  .15034924e+03  .98090688e-03", 
" -.84378409e+01  .21722250e+02 -.57678274e+02  .15514990e+03  .96852272e-03", 
" -.83728688e+01  .21392132e+02 -.55616688e+02  .13886404e+03  .96409117e-03", 
" -.83082615e+01  .21094549e+02 -.55340016e+02  .14790586e+03  .95842550e-03", 
" -.82438985e+01  .20783226e+02 -.54422810e+02  .15263244e+03  .95475362e-03", 
" -.81795910e+01  .20415707e+02 -.50340709e+02  .11301773e+03  .94834802e-03", 
" -.81155846e+01  .20087090e+02 -.47664342e+02  .87901160e+02  .93770155e-03", 
" -.80518555e+01  .19800482e+02 -.47057400e+02  .88642442e+02  .94156985e-03", 
" -.79881509e+01  .19495317e+02 -.45739130e+02  .84426282e+02  .93617796e-03", 
" -.79249527e+01  .19228426e+02 -.45599035e+02  .91089754e+02  .93062294e-03", 
" -.78617559e+01  .18957674e+02 -.46008230e+02  .11032701e+03  .92400335e-03", 
" -.77986625e+01  .18672762e+02 -.44826747e+02  .10203887e+03  .91957326e-03", 
" -.77356543e+01  .18373425e+02 -.42858657e+02  .84747417e+02  .90859723e-03", 
" -.76727022e+01  .18079290e+02 -.41285273e+02  .74338109e+02  .90305248e-03", 
" -.76100563e+01  .17771571e+02 -.38606788e+02  .47659849e+02  .89551908e-03", 
" -.75475022e+01  .17480219e+02 -.36835950e+02  .31378134e+02  .89257454e-03", 
" -.74851629e+01  .17188262e+02 -.34493082e+02  .90952520e+01  .89047932e-03", 
" -.74230845e+01  .16945900e+02 -.34800580e+02  .22551574e+02  .89401582e-03", 
" -.73609710e+01  .16684834e+02 -.33558579e+02  .12830645e+02  .89497910e-03", 
" -.72988388e+01  .16411621e+02 -.32063766e+02 -.17369158e+01  .88181288e-03", 
" -.72368601e+01  .16157802e+02 -.31186794e+02 -.75084224e+01  .87599471e-03", 
" -.71749585e+01  .15946816e+02 -.33422103e+02  .34855093e+02  .87329229e-03", 
" -.71130660e+01  .15676125e+02 -.32046599e+02  .28277456e+02  .87550965e-03", 
" -.70511450e+01  .15419881e+02 -.31115654e+02  .25014337e+02  .86693492e-03", 
" -.69895498e+01  .15185321e+02 -.30405417e+02  .21799386e+02  .86868118e-03", 
" -.69277742e+01  .14934204e+02 -.29012197e+02  .93492698e+01  .85603992e-03", 
" -.68661926e+01  .14719649e+02 -.29603367e+02  .24989209e+02  .85428381e-03", 
" -.68047013e+01  .14513455e+02 -.30138886e+02  .36033040e+02  .85590067e-03", 
" -.67429593e+01  .14268584e+02 -.29268442e+02  .34502837e+02  .85131675e-03", 
" -.66810505e+01  .14014554e+02 -.27544857e+02  .19587316e+02  .84994199e-03", 
" -.66191239e+01  .13748677e+02 -.25364619e+02  .10873953e+01  .84173869e-03", 
" -.65572290e+01  .13494946e+02 -.23542816e+02 -.12594736e+02  .84174637e-03", 
" -.64955573e+01  .13327743e+02 -.25144241e+02  .73130211e+01  .83905619e-03", 
" -.64335489e+01  .13092883e+02 -.24359404e+02  .93597441e+01  .83870773e-03", 
" -.63713155e+01  .12889780e+02 -.25163670e+02  .28554890e+02  .83403835e-03", 
" -.63089619e+01  .12641826e+02 -.23418438e+02  .14841956e+02  .82731618e-03", 
" -.62464304e+01  .12403165e+02 -.22230633e+02  .95793094e+01  .82686601e-03", 
" -.61837940e+01  .12142646e+02 -.19142560e+02 -.21858803e+02  .81748241e-03", 
" -.61210817e+01  .11943161e+02 -.19232959e+02 -.17383386e+02  .81195732e-03", 
" -.60582552e+01  .11727662e+02 -.18628919e+02 -.17403447e+02  .80780771e-03", 
" -.59952303e+01  .11519163e+02 -.17986092e+02 -.20756247e+02  .80849876e-03", 
" -.59317499e+01  .11279519e+02 -.15813808e+02 -.42921599e+02  .80622151e-03", 
" -.58680701e+01  .11026841e+02 -.12668628e+02 -.79650301e+02  .80867904e-03", 
" -.58041545e+01  .10831166e+02 -.12917425e+02 -.68628884e+02  .80385974e-03", 
" -.57401231e+01  .10611880e+02 -.11536243e+02 -.79469226e+02  .80019545e-03", 
" -.56757110e+01  .10415833e+02 -.11419347e+02 -.72898009e+02  .79834516e-03", 
" -.56110485e+01  .10206469e+02 -.10350643e+02 -.77569577e+02  .79858140e-03", 
" -.55461040e+01  .10019462e+02 -.10202889e+02 -.73070916e+02  .79553550e-03", 
" -.54808911e+01  .98648300e+01 -.11804530e+02 -.39774258e+02  .79315460e-03", 
" -.54153207e+01  .97059533e+01 -.12362463e+02 -.30799483e+02  .78953557e-03", 
" -.53493712e+01  .95944456e+01 -.15638440e+02  .19226266e+02  .79127777e-03", 
" -.52826455e+01  .93995623e+01 -.14611594e+02  .11889862e+02  .78438217e-03", 
" -.52154700e+01  .92159597e+01 -.14023517e+02  .11175642e+02  .77643330e-03", 
" -.51476869e+01  .90501736e+01 -.14371828e+02  .23504083e+02  .77595659e-03", 
" -.50793809e+01  .88964227e+01 -.15303123e+02  .43424386e+02  .77419031e-03", 
" -.50103423e+01  .86968164e+01 -.13383614e+02  .24351428e+02  .76840308e-03", 
" -.49407245e+01  .85218388e+01 -.13168542e+02  .33559718e+02  .76625793e-03", 
" -.48703305e+01  .83487355e+01 -.13312094e+02  .45711428e+02  .76170790e-03", 
" -.47990002e+01  .81443512e+01 -.11301796e+02  .27734803e+02  .75453419e-03", 
" -.47272419e+01  .80013159e+01 -.11821915e+02  .42790150e+02  .75953324e-03", 
" -.46543734e+01  .78510414e+01 -.12426663e+02  .67750469e+02  .75814433e-03", 
" -.45804342e+01  .76965643e+01 -.13090266e+02  .92007363e+02  .76104521e-03", 
" -.45054538e+01  .75167585e+01 -.12513014e+02  .10481371e+03  .75918127e-03", 
" -.44295684e+01  .73478358e+01 -.11022627e+02  .98426398e+02  .77570979e-03", 
" -.43523317e+01  .72066828e+01 -.10683464e+02  .10140482e+03  .78043084e-03", 
" -.42739619e+01  .70966660e+01 -.11509988e+02  .12325081e+03  .78719567e-03", 
" -.41941539e+01  .69858665e+01 -.12647508e+02  .15186940e+03  .79509623e-03", 
" -.41126287e+01  .68326811e+01 -.11277606e+02  .14387828e+03  .79136021e-03", 
" -.40293404e+01  .67184023e+01 -.11821851e+02  .16560583e+03  .79855429e-03", 
" -.39442096e+01  .65598693e+01 -.96208091e+01  .15193238e+03  .80128594e-03", 
" -.38572913e+01  .64837411e+01 -.10284985e+02  .16677288e+03  .81260755e-03", 
" -.37680464e+01  .63796161e+01 -.99278524e+01  .17208506e+03  .80898046e-03", 
" -.36758611e+01  .61883999e+01 -.52980300e+01  .13170653e+03  .81316854e-03", 
" -.35813589e+01  .60758789e+01 -.38835489e+01  .12760789e+03  .80072960e-03", 
" -.34840335e+01  .59977195e+01 -.31784236e+01  .13069718e+03  .81335352e-03", 
" -.33831313e+01  .59418776e+01 -.28546146e+01  .13901746e+03  .81843953e-03", 
" -.32783019e+01  .58419401e+01  .74010586e+00  .10274517e+03  .82276909e-03", 
" -.31693521e+01  .58066778e+01  .14139182e+01  .10663369e+03  .84477390e-03", 
" -.30555861e+01  .58119492e+01  .10982816e+01  .12575601e+03  .87476218e-03", 
" -.29354316e+01  .57746528e+01  .22954927e+01  .13524964e+03  .88556389e-03", 
" -.28083802e+01  .57310449e+01  .50548282e+01  .11726045e+03  .90395828e-03", 
" -.26741656e+01  .58476125e+01  .11714260e+01  .18816972e+03  .92826867e-03", 
" -.25296619e+01  .57713321e+01  .93076926e+01  .95417257e+02  .94320266e-03", 
" -.23735748e+01  .58880108e+01  .80328104e+01  .13160600e+03  .97893335e-03", 
" -.22030892e+01  .60360174e+01  .56664909e+01  .18742090e+03  .99433512e-03", 
" -.20144883e+01  .63487653e+01 -.12159582e+01  .29271252e+03  .10335997e-02", 
" -.18003437e+01  .65247324e+01  .24672695e+01  .26560844e+03  .10890486e-02", 
" -.15521712e+01  .68551723e+01  .24373394e+01  .32171627e+03  .11662082e-02", 
" -.12495092e+01  .70408117e+01  .20092477e+02  .15354797e+03  .13001773e-02", 
" -.85501019e+00  .79291559e+01  .19838405e+02  .23553183e+03  .14387103e-02", 
" -.75698847e+00  .80330221e+01  .26168403e+02  .18471695e+03  .14886784e-02", 
" -.65081405e+00  .84051739e+01  .24528165e+02  .22181803e+03  .15416245e-02", 
" -.53228423e+00  .89130645e+01  .17257287e+02  .33965751e+03  .15706539e-02", 
" -.39867798e+00  .92249077e+01  .25834904e+02  .25742182e+03  .16473833e-02", 
" -.24398546e+00  .95335598e+01  .40289965e+02  .10238374e+03  .17593747e-02", 
" -.61119149e-01  .10155465e+02  .43345670e+02  .14274358e+03  .19356772e-02", 
"  .16717228e+00  .11218432e+02  .32740482e+02  .43001447e+03  .21261538e-02", 
"  .47688168e+00  .12569004e+02  .46078154e+02  .34525798e+03  .24912949e-02", 
"  .98289768e+00  .14231060e+02  .97336301e+02  .76888722e+01  .34001636e-02", 
"  .14541467e+01  .16457602e+02  .14191239e+03 -.16269734e+03  .44686226e-02", 
"  .20574784e+01  .21835568e+02  .16043957e+03 -.10842820e+03  .68029909e-02", 
"  .24709755e+01  .23897226e+02  .27375159e+03 -.10770053e+04  .92707575e-02", 
"dfactt    0  4  3   20", " -.58613829e+02  .87543112e+03 -.84188255e+04  .36881692e+05  .55482898e-01", 
" -.55469366e+02  .78766361e+03 -.73069726e+04  .31622641e+05  .40698219e-01", 
" -.51176248e+02  .67378227e+03 -.58965609e+04  .24541497e+05  .26755344e-01", 
" -.47916373e+02  .59590579e+03 -.50642753e+04  .21282784e+05  .19347284e-01", 
" -.44580394e+02  .51874277e+03 -.42147585e+04  .17592452e+05  .13854666e-01", 
" -.42601565e+02  .47613416e+03 -.38046074e+04  .16258496e+05  .11524802e-01", 
" -.41179197e+02  .44466642e+03 -.34400731e+04  .14420725e+05  .10233745e-01", 
" -.40077600e+02  .42218930e+03 -.32147807e+04  .13435796e+05  .90986605e-02", 
" -.39173606e+02  .40405134e+03 -.30279148e+04  .12529680e+05  .84665441e-02", 
" -.38403521e+02  .38846927e+03 -.28500733e+04  .11467423e+05  .79333290e-02", 
" -.37731977e+02  .37543530e+03 -.27154864e+04  .10766353e+05  .74194641e-02", 
" -.37134184e+02  .36373242e+03 -.26010225e+04  .10318264e+05  .70535358e-02", 
" -.36597816e+02  .35331210e+03 -.24847452e+04  .96060516e+04  .68131019e-02", 
" -.34518176e+02  .31466413e+03 -.20833542e+04  .74784730e+04  .55557699e-02", 
" -.33020499e+02  .28855906e+03 -.18453236e+04  .64618786e+04  .48625751e-02", 
" -.31845752e+02  .26873845e+03 -.16641521e+04  .56232653e+04  .44755282e-02", 
" -.30876780e+02  .25315181e+03 -.15322495e+04  .50650104e+04  .42320106e-02", 
" -.30051319e+02  .24054945e+03 -.14415423e+04  .48468778e+04  .38785880e-02", 
" -.29326750e+02  .22922408e+03 -.13458656e+04  .44796985e+04  .36772314e-02", 
" -.28683697e+02  .21944003e+03 -.12643371e+04  .41385697e+04  .34496607e-02", 
" -.28104862e+02  .21109793e+03 -.12098680e+04  .40581124e+04  .33350395e-02", 
" -.27578642e+02  .20356812e+03 -.11510777e+04  .37953896e+04  .32139308e-02", 
" -.27094189e+02  .19656993e+03 -.10913086e+04  .35018519e+04  .30683123e-02", 
" -.26645566e+02  .19014626e+03 -.10357959e+04  .32201431e+04  .30209265e-02", 
" -.26227612e+02  .18426881e+03 -.98632306e+03  .29815836e+04  .28983199e-02", 
" -.25836836e+02  .17907263e+03 -.95219955e+03  .29038509e+04  .27966268e-02", 
" -.25468893e+02  .17421074e+03 -.91869693e+03  .27955264e+04  .27061079e-02", 
" -.25122284e+02  .16966712e+03 -.88652659e+03  .26875349e+04  .26069853e-02", 
" -.24792029e+02  .16521121e+03 -.84791234e+03  .24782539e+04  .25618687e-02", 
" -.24478926e+02  .16109476e+03 -.81270628e+03  .22733299e+04  .25056238e-02", 
" -.24180163e+02  .15721032e+03 -.78093588e+03  .21038056e+04  .24307885e-02", 
" -.23894175e+02  .15366777e+03 -.75956129e+03  .20667066e+04  .24060875e-02", 
" -.23621019e+02  .15035343e+03 -.73959874e+03  .20279098e+04  .23745848e-02", 
" -.23358664e+02  .14717835e+03 -.71743147e+03  .19242608e+04  .23125316e-02", 
" -.23105815e+02  .14404146e+03 -.69355695e+03  .18166411e+04  .22568013e-02", 
" -.22861986e+02  .14107898e+03 -.67230881e+03  .17243252e+04  .22226740e-02", 
" -.22626954e+02  .13839325e+03 -.66091029e+03  .17588910e+04  .21986271e-02", 
" -.22399726e+02  .13574173e+03 -.64357447e+03  .16934416e+04  .21821342e-02", 
" -.22180127e+02  .13329891e+03 -.63206334e+03  .16950966e+04  .21276564e-02", 
" -.21966579e+02  .13084104e+03 -.61733009e+03  .16579132e+04  .20880654e-02", 
" -.21759625e+02  .12857910e+03 -.60788322e+03  .16759215e+04  .20165295e-02", 
" -.21558513e+02  .12629667e+03 -.59376471e+03  .16361224e+04  .19737546e-02", 
" -.21362984e+02  .12415494e+03 -.58365953e+03  .16474688e+04  .19769558e-02", 
" -.21172689e+02  .12207388e+03 -.57234308e+03  .16326107e+04  .19377545e-02", 
" -.20986813e+02  .11991638e+03 -.55422757e+03  .15176745e+04  .18895279e-02", 
" -.20806533e+02  .11814276e+03 -.55266759e+03  .16111036e+04  .18722415e-02", 
" -.20630230e+02  .11623097e+03 -.54091605e+03  .15717811e+04  .18507203e-02", 
" -.20457832e+02  .11429677e+03 -.52522648e+03  .14746016e+04  .18387036e-02", 
" -.20288861e+02  .11246239e+03 -.51430711e+03  .14510342e+04  .18317322e-02", 
" -.20123573e+02  .11066432e+03 -.50300603e+03  .14193181e+04  .18163353e-02", 
" -.19961757e+02  .10888962e+03 -.49014006e+03  .13596373e+04  .18312552e-02", 
" -.19803594e+02  .10717998e+03 -.47787943e+03  .12974785e+04  .18061377e-02", 
" -.19648829e+02  .10564710e+03 -.47255842e+03  .13245572e+04  .17789181e-02", 
" -.19496556e+02  .10407071e+03 -.46377661e+03  .13068069e+04  .17716031e-02", 
" -.19347474e+02  .10260553e+03 -.45793300e+03  .13140374e+04  .17599942e-02", 
" -.19201237e+02  .10118133e+03 -.45315717e+03  .13393066e+04  .17422438e-02", 
" -.19057728e+02  .99764954e+02 -.44622831e+03  .13280772e+04  .17276363e-02", 
" -.18916479e+02  .98306762e+02 -.43646830e+03  .12884182e+04  .17077685e-02", 
" -.18777288e+02  .96864583e+02 -.42600186e+03  .12351075e+04  .16795953e-02", 
" -.18640215e+02  .95461998e+02 -.41769762e+03  .12161600e+04  .16736642e-02", 
" -.18505611e+02  .94090827e+02 -.40819407e+03  .11688280e+04  .16611932e-02", 
" -.18373322e+02  .92825339e+02 -.40223627e+03  .11591135e+04  .16457151e-02", 
" -.18243025e+02  .91593229e+02 -.39660254e+03  .11517230e+04  .16286590e-02", 
" -.18114816e+02  .90380224e+02 -.39136382e+03  .11573884e+04  .16259440e-02", 
" -.17988494e+02  .89200323e+02 -.38685123e+03  .11743135e+04  .16201146e-02", 
" -.17863951e+02  .88009796e+02 -.38012514e+03  .11531284e+04  .16030499e-02", 
" -.17740602e+02  .86761261e+02 -.36924435e+03  .10731989e+04  .15931392e-02", 
" -.17619142e+02  .85556923e+02 -.36040954e+03  .10258216e+04  .15892192e-02", 
" -.17499759e+02  .84461094e+02 -.35450505e+03  .10065268e+04  .15809790e-02", 
" -.17381923e+02  .83420480e+02 -.35115089e+03  .10178545e+04  .15782186e-02", 
" -.17265083e+02  .82279600e+02 -.34278973e+03  .97345087e+03  .15691547e-02", 
" -.17149573e+02  .81149558e+02 -.33415304e+03  .92626790e+03  .15576821e-02", 
" -.17035783e+02  .80099970e+02 -.32806381e+03  .90280501e+03  .15323734e-02", 
" -.16923555e+02  .79107386e+02 -.32371112e+03  .90044204e+03  .15299156e-02", 
" -.16812372e+02  .78030489e+02 -.31369518e+03  .81984832e+03  .15266255e-02", 
" -.16702617e+02  .77069606e+02 -.30884729e+03  .80338203e+03  .15111742e-02", 
" -.16593345e+02  .75981004e+02 -.29807655e+03  .71491073e+03  .14834357e-02", 
" -.16485630e+02  .74968208e+02 -.29037738e+03  .66884032e+03  .14487885e-02", 
" -.16379078e+02  .74008004e+02 -.28470844e+03  .64422629e+03  .14367259e-02", 
" -.16273586e+02  .73096942e+02 -.28037801e+03  .63232224e+03  .14230402e-02", 
" -.16169794e+02  .72253727e+02 -.27793081e+03  .64227398e+03  .14098299e-02", 
" -.16066738e+02  .71384528e+02 -.27453993e+03  .64597738e+03  .13984095e-02", 
" -.15964664e+02  .70578752e+02 -.27430393e+03  .69336933e+03  .13839919e-02", 
" -.15863686e+02  .69761627e+02 -.27177531e+03  .70085116e+03  .13788382e-02", 
" -.15763305e+02  .68875331e+02 -.26627410e+03  .68040947e+03  .13631276e-02", 
" -.15663878e+02  .68052308e+02 -.26264427e+03  .67283614e+03  .13546658e-02", 
" -.15565398e+02  .67196807e+02 -.25708337e+03  .64118441e+03  .13460287e-02", 
" -.15468054e+02  .66440257e+02 -.25508136e+03  .65016376e+03  .13445365e-02", 
" -.15371100e+02  .65610013e+02 -.25002753e+03  .62942295e+03  .13370621e-02", 
" -.15275498e+02  .64918688e+02 -.25049742e+03  .67094979e+03  .13180327e-02", 
" -.15179855e+02  .64074872e+02 -.24417148e+03  .62943855e+03  .13254175e-02", 
" -.15084847e+02  .63213275e+02 -.23668835e+03  .57618748e+03  .13099492e-02", 
" -.14991059e+02  .62422175e+02 -.23164770e+03  .55133616e+03  .13094719e-02", 
" -.14897954e+02  .61661222e+02 -.22744636e+03  .53518430e+03  .13056490e-02", 
" -.14805274e+02  .60833489e+02 -.21905642e+03  .45727429e+03  .13079218e-02", 
" -.14714042e+02  .60195305e+02 -.21917737e+03  .48699669e+03  .12961112e-02", 
" -.14622749e+02  .59466863e+02 -.21554941e+03  .47623585e+03  .12808850e-02", 
" -.14531964e+02  .58719462e+02 -.21096532e+03  .45208619e+03  .12770224e-02", 
" -.14441953e+02  .58000477e+02 -.20739501e+03  .44243523e+03  .12644584e-02", 
" -.14352632e+02  .57296235e+02 -.20379921e+03  .43031694e+03  .12695510e-02", 
" -.14263871e+02  .56631405e+02 -.20199838e+03  .44343111e+03  .12673665e-02", 
" -.14175313e+02  .55912042e+02 -.19783553e+03  .43142554e+03  .12580647e-02", 
" -.14087685e+02  .55198764e+02 -.19206250e+03  .38543722e+03  .12418002e-02", 
" -.14000854e+02  .54594451e+02 -.19204064e+03  .41763609e+03  .12383416e-02", 
" -.13914039e+02  .53925797e+02 -.18920270e+03  .41852163e+03  .12354003e-02", 
" -.13827653e+02  .53257882e+02 -.18611055e+03  .41687182e+03  .12299221e-02", 
" -.13741693e+02  .52570526e+02 -.18116251e+03  .38593841e+03  .12209628e-02", 
" -.13656665e+02  .51965131e+02 -.17960641e+03  .39611177e+03  .12166647e-02", 
" -.13571756e+02  .51341044e+02 -.17716823e+03  .39551042e+03  .12041981e-02", 
" -.13487533e+02  .50730455e+02 -.17473685e+03  .39278730e+03  .11855292e-02", 
" -.13403552e+02  .50115431e+02 -.17208492e+03  .38955234e+03  .11776618e-02", 
" -.13320190e+02  .49551405e+02 -.17079355e+03  .39524839e+03  .11788487e-02", 
" -.13237161e+02  .48932197e+02 -.16686753e+03  .37204521e+03  .11716642e-02", 
" -.13153844e+02  .48275604e+02 -.16165282e+03  .33227627e+03  .11730481e-02", 
" -.13071403e+02  .47666785e+02 -.15795731e+03  .31100433e+03  .11684375e-02", 
" -.12989483e+02  .47117341e+02 -.15663226e+03  .31870411e+03  .11742357e-02", 
" -.12907770e+02  .46552426e+02 -.15471861e+03  .32150041e+03  .11736879e-02", 
" -.12826409e+02  .46037756e+02 -.15489654e+03  .34539399e+03  .11722270e-02", 
" -.12745082e+02  .45468654e+02 -.15246504e+03  .33999145e+03  .11728636e-02", 
" -.12663792e+02  .44837673e+02 -.14737897e+03  .30611180e+03  .11755926e-02", 
" -.12583098e+02  .44246154e+02 -.14242349e+03  .26377835e+03  .11643735e-02", 
" -.12502757e+02  .43712186e+02 -.14087666e+03  .26634182e+03  .11567369e-02", 
" -.12422535e+02  .43143400e+02 -.13747812e+03  .24704792e+03  .11577793e-02", 
" -.12342354e+02  .42595991e+02 -.13575651e+03  .25340577e+03  .11582665e-02", 
" -.12262502e+02  .42022793e+02 -.13152371e+03  .22006315e+03  .11550992e-02", 
" -.12183262e+02  .41540429e+02 -.13179575e+03  .24935951e+03  .11512899e-02", 
" -.12103893e+02  .41008150e+02 -.12969106e+03  .24728260e+03  .11479342e-02", 
" -.12024821e+02  .40510400e+02 -.12837899e+03  .24787797e+03  .11351464e-02", 
" -.11946129e+02  .40036334e+02 -.12813560e+03  .26580041e+03  .11355418e-02", 
" -.11867090e+02  .39509666e+02 -.12659592e+03  .27829422e+03  .11336968e-02", 
" -.11789001e+02  .39085989e+02 -.12795519e+03  .31289574e+03  .11307872e-02", 
" -.11710321e+02  .38535526e+02 -.12365178e+03  .27853281e+03  .11287131e-02", 
" -.11631768e+02  .38006132e+02 -.12048514e+03  .26236653e+03  .11198925e-02", 
" -.11553473e+02  .37512471e+02 -.11888884e+03  .26281521e+03  .11081041e-02", 
" -.11475385e+02  .37030856e+02 -.11760480e+03  .26819403e+03  .11140888e-02", 
" -.11397053e+02  .36497185e+02 -.11444110e+03  .25794052e+03  .11119716e-02", 
" -.11318877e+02  .36006822e+02 -.11261718e+03  .25810211e+03  .11042443e-02", 
" -.11241086e+02  .35584934e+02 -.11372394e+03  .29204223e+03  .10999132e-02", 
" -.11163131e+02  .35089804e+02 -.11084098e+03  .27554888e+03  .10968515e-02", 
" -.11085314e+02  .34643539e+02 -.11037954e+03  .28927787e+03  .10832823e-02", 
" -.11006928e+02  .34119791e+02 -.10681737e+03  .26791301e+03  .10826516e-02", 
" -.10928910e+02  .33611032e+02 -.10317192e+03  .23987439e+03  .10768282e-02", 
" -.10850865e+02  .33123982e+02 -.10085076e+03  .23549041e+03  .10764361e-02", 
" -.10772883e+02  .32609654e+02 -.95652390e+02  .17983128e+03  .10699844e-02", 
" -.10695142e+02  .32165679e+02 -.93943896e+02  .17254500e+03  .10624323e-02", 
" -.10616941e+02  .31704503e+02 -.92803944e+02  .17980532e+03  .10530053e-02", 
" -.10539024e+02  .31268756e+02 -.92214473e+02  .19446656e+03  .10540498e-02", 
" -.10461069e+02  .30875065e+02 -.93639251e+02  .23343541e+03  .10497963e-02", 
" -.10382651e+02  .30410802e+02 -.91379748e+02  .22021092e+03  .10483759e-02", 
" -.10304225e+02  .29956968e+02 -.89429514e+02  .21398850e+03  .10483228e-02", 
" -.10225859e+02  .29569454e+02 -.90779735e+02  .25051196e+03  .10500188e-02", 
" -.10147674e+02  .29216817e+02 -.93485985e+02  .30477778e+03  .10452419e-02", 
" -.10068839e+02  .28776842e+02 -.91839025e+02  .30181850e+03  .10400988e-02", 
" -.99897169e+01  .28312948e+02 -.88909643e+02  .28313309e+03  .10377712e-02", 
" -.99106122e+01  .27898280e+02 -.88386076e+02  .29651018e+03  .10345245e-02", 
" -.98309864e+01  .27441924e+02 -.85873419e+02  .28408555e+03  .10350594e-02", 
" -.97514833e+01  .27032446e+02 -.84832244e+02  .28273219e+03  .10393702e-02", 
" -.96714094e+01  .26559893e+02 -.80829054e+02  .24525010e+03  .10358191e-02", 
" -.95911282e+01  .26153758e+02 -.80185003e+02  .25040355e+03  .10300491e-02", 
" -.95106532e+01  .25776385e+02 -.80708722e+02  .27012309e+03  .10393546e-02", 
" -.94295294e+01  .25320029e+02 -.77093781e+02  .23445787e+03  .10349842e-02", 
" -.93482087e+01  .24873465e+02 -.74298651e+02  .21367648e+03  .10403996e-02", 
" -.92665018e+01  .24455230e+02 -.73059010e+02  .21711717e+03  .10350471e-02", 
" -.91843784e+01  .24041335e+02 -.72702066e+02  .23494273e+03  .10399341e-02", 
" -.91017112e+01  .23587969e+02 -.69560539e+02  .21228079e+03  .10420081e-02", 
" -.90191365e+01  .23249433e+02 -.71561059e+02  .25402545e+03  .10391987e-02", 
" -.89355202e+01  .22807185e+02 -.68770524e+02  .23755969e+03  .10322580e-02", 
" -.88515639e+01  .22384125e+02 -.65838405e+02  .21125400e+03  .10328625e-02", 
" -.87671504e+01  .22075082e+02 -.70079697e+02  .28906088e+03  .10229018e-02", 
" -.86816827e+01  .21612788e+02 -.66223843e+02  .25813777e+03  .10176412e-02", 
" -.85960280e+01  .21196746e+02 -.63289102e+02  .22974523e+03  .10229570e-02", 
" -.85098282e+01  .20845052e+02 -.63799780e+02  .24851029e+03  .10206462e-02", 
" -.84227550e+01  .20467179e+02 -.63490549e+02  .26195451e+03  .10247613e-02", 
" -.83343335e+01  .20033395e+02 -.62169996e+02  .27718593e+03  .10323117e-02", 
" -.82455787e+01  .19687562e+02 -.63769973e+02  .31636279e+03  .10271361e-02", 
" -.81559385e+01  .19337918e+02 -.64764103e+02  .34657528e+03  .10156850e-02", 
" -.80651224e+01  .18924796e+02 -.63409727e+02  .35132755e+03  .10238644e-02", 
" -.79733400e+01  .18525322e+02 -.62350603e+02  .35786983e+03  .10312605e-02", 
" -.78802020e+01  .18056944e+02 -.57343768e+02  .31293144e+03  .10284943e-02", 
" -.77862890e+01  .17679735e+02 -.57186620e+02  .32757788e+03  .10192030e-02", 
" -.76911199e+01  .17320398e+02 -.57314819e+02  .34720541e+03  .10219616e-02", 
" -.75942853e+01  .16910351e+02 -.55790682e+02  .35050423e+03  .10285045e-02", 
" -.74963352e+01  .16553046e+02 -.55901838e+02  .36971284e+03  .10341203e-02", 
" -.73959269e+01  .16010690e+02 -.48015693e+02  .29615390e+03  .10398063e-02", 
" -.72948130e+01  .15650979e+02 -.47104544e+02  .29647771e+03  .10248670e-02", 
" -.71918864e+01  .15327167e+02 -.48499821e+02  .33020118e+03  .10255245e-02", 
" -.70869962e+01  .14969304e+02 -.48140106e+02  .34446574e+03  .10241036e-02", 
" -.69797625e+01  .14548078e+02 -.44393621e+02  .31584334e+03  .10118158e-02", 
" -.68704660e+01  .14201596e+02 -.44909456e+02  .34965735e+03  .10138557e-02", 
" -.67584840e+01  .13859232e+02 -.44929322e+02  .36745923e+03  .10148678e-02", 
" -.66436855e+01  .13510705e+02 -.45359613e+02  .40305432e+03  .10442452e-02", 
" -.65258682e+01  .13146015e+02 -.43919887e+02  .40831639e+03  .10419956e-02", 
" -.64050487e+01  .12825684e+02 -.43735430e+02  .42595094e+03  .10513819e-02", 
" -.62801134e+01  .12468876e+02 -.43153187e+02  .45461414e+03  .10594123e-02", 
" -.61508816e+01  .12184017e+02 -.46080350e+02  .52877520e+03  .10474186e-02", 
" -.60166969e+01  .11736229e+02 -.40989823e+02  .51286447e+03  .10609494e-02", 
" -.58770614e+01  .11344771e+02 -.37615256e+02  .50861241e+03  .10760784e-02", 
" -.57319269e+01  .11094022e+02 -.41211926e+02  .60532576e+03  .10966010e-02", 
" -.55792488e+01  .10746312e+02 -.39075508e+02  .62900022e+03  .11036566e-02", 
" -.54191266e+01  .10430167e+02 -.36499413e+02  .64252241e+03  .11280209e-02", 
" -.52493468e+01  .10178107e+02 -.36316119e+02  .70063620e+03  .11408049e-02", 
" -.50666080e+01  .97319329e+01 -.26934284e+02  .64960955e+03  .11913864e-02", 
" -.48711482e+01  .96121738e+01 -.31695600e+02  .79114677e+03  .12201665e-02", 
" -.46572547e+01  .92870369e+01 -.21801258e+02  .73157534e+03  .12755192e-02", 
" -.44210879e+01  .91628864e+01 -.18412755e+02  .76717617e+03  .13158766e-02", 
" -.41544766e+01  .91169927e+01 -.14045288e+02  .78553058e+03  .13819188e-02", 
" -.38430170e+01  .88249416e+01  .90055128e+01  .56644482e+03  .14605365e-02", 
" -.34664859e+01  .90966894e+01  .17122141e+02  .51604544e+03  .15325146e-02", 
" -.29725480e+01  .95722599e+01  .29586437e+02  .48289145e+03  .17758586e-02", 
" -.28502994e+01  .98645713e+01  .23866879e+02  .59997716e+03  .18114086e-02", 
" -.27158203e+01  .10012942e+02  .28558482e+02  .57473263e+03  .19075669e-02", 
" -.25676298e+01  .10222769e+02  .35575525e+02  .52147178e+03  .19919288e-02", 
" -.23997921e+01  .10372445e+02  .45188133e+02  .46638196e+03  .21386999e-02", 
" -.22068066e+01  .10514092e+02  .56109684e+02  .45549459e+03  .22166158e-02", 
" -.19807679e+01  .11062544e+02  .60353639e+02  .50816723e+03  .23892392e-02", 
" -.17003639e+01  .11950235e+02  .62265649e+02  .60920833e+03  .25717344e-02", 
" -.13206934e+01  .12857035e+02  .91786882e+02  .42928345e+03  .30120168e-02", 
" -.71849222e+00  .15693018e+02  .84833823e+02  .10581405e+04  .39233445e-02", 
" -.16604153e+00  .19618222e+02  .45292100e+02  .20713620e+04  .52665258e-02", 
"  .54126900e+00  .24777524e+02  .43732736e+02  .30548718e+04  .80645073e-02", 
"  .10254609e+01  .25579300e+02  .20240633e+03  .18475344e+04  .10903284e-01"
)


.urc2 = 
c("Copyright (C) James G. MacKinnon, 1995", "conc      1  2  2   20", 
" -.45836519e+01 -.18786631e+02 -.98042252e+02  .42871581e-02", 
" -.44176480e+01 -.17225943e+02 -.66798922e+02  .30522929e-02", 
" -.41950669e+01 -.14169697e+02 -.52346097e+02  .20868807e-02", 
" -.40171434e+01 -.12117426e+02 -.40508567e+02  .15549521e-02", 
" -.38288088e+01 -.10236756e+02 -.29978980e+02  .10638945e-02", 
" -.37129389e+01 -.92915475e+01 -.21744769e+02  .87472918e-03", 
" -.36293963e+01 -.85335349e+01 -.18645551e+02  .78626616e-03", 
" -.35625179e+01 -.79408984e+01 -.17018258e+02  .73836489e-03", 
" -.35065833e+01 -.74829078e+01 -.15526038e+02  .68551198e-03", 
" -.34587033e+01 -.71144521e+01 -.13990855e+02  .64539768e-03", 
" -.34167863e+01 -.67743298e+01 -.13179039e+02  .62198310e-03", 
" -.33793598e+01 -.64918299e+01 -.12052671e+02  .59540430e-03", 
" -.33449776e+01 -.62713641e+01 -.10990966e+02  .57981730e-03", 
" -.32101182e+01 -.53341671e+01 -.83783448e+01  .50326238e-03", 
" -.31105226e+01 -.46597293e+01 -.73266982e+01  .45618970e-03", 
" -.30298761e+01 -.41754340e+01 -.66073614e+01  .41589133e-03", 
" -.29626426e+01 -.37711418e+01 -.61078674e+01  .39594211e-03", 
" -.29037471e+01 -.34884062e+01 -.49278025e+01  .38216085e-03", 
" -.28519970e+01 -.32105464e+01 -.47021377e+01  .36049284e-03", 
" -.28054079e+01 -.29681294e+01 -.44034489e+01  .33519987e-03", 
" -.27628870e+01 -.27696792e+01 -.39481145e+01  .32129984e-03", 
" -.27235239e+01 -.25917374e+01 -.37515799e+01  .31278595e-03", 
" -.26871603e+01 -.24322786e+01 -.34896099e+01  .30003273e-03", 
" -.26532382e+01 -.22752876e+01 -.34839301e+01  .29085110e-03", 
" -.26212439e+01 -.21359862e+01 -.34394673e+01  .28322435e-03", 
" -.25910169e+01 -.20161215e+01 -.32144852e+01  .27581863e-03", 
" -.25623317e+01 -.19057376e+01 -.30353208e+01  .26813208e-03", 
" -.25350829e+01 -.17916430e+01 -.31748737e+01  .25749965e-03", 
" -.25089246e+01 -.17003377e+01 -.29633199e+01  .24926984e-03", 
" -.24838501e+01 -.16120926e+01 -.28678686e+01  .24731206e-03", 
" -.24598103e+01 -.15277692e+01 -.27997368e+01  .24806596e-03", 
" -.24367900e+01 -.14450335e+01 -.27590482e+01  .24745640e-03", 
" -.24145089e+01 -.13775692e+01 -.24669332e+01  .24099576e-03", 
" -.23929164e+01 -.13124539e+01 -.23111829e+01  .23524224e-03", 
" -.23721407e+01 -.12364960e+01 -.24433048e+01  .23364111e-03", 
" -.23518477e+01 -.11758493e+01 -.23963631e+01  .22913011e-03", 
" -.23321733e+01 -.11265189e+01 -.20873085e+01  .22167499e-03", 
" -.23130696e+01 -.10707198e+01 -.20122853e+01  .21980550e-03", 
" -.22944587e+01 -.10127774e+01 -.20220442e+01  .21746175e-03", 
" -.22763431e+01 -.95743104e+00 -.20802249e+01  .21218906e-03", 
" -.22586352e+01 -.90317521e+00 -.21657682e+01  .20845335e-03", 
" -.22413886e+01 -.85246252e+00 -.22141815e+01  .20734451e-03", 
" -.22244965e+01 -.80355178e+00 -.22828491e+01  .20610112e-03", 
" -.22079520e+01 -.76421404e+00 -.21359219e+01  .20571295e-03", 
" -.21917604e+01 -.72252253e+00 -.20695079e+01  .20306704e-03", 
" -.21758155e+01 -.69131845e+00 -.18615691e+01  .19981207e-03", 
" -.21601877e+01 -.66413952e+00 -.15806871e+01  .20312616e-03", 
" -.21448664e+01 -.61988061e+00 -.17260972e+01  .20024892e-03", 
" -.21298079e+01 -.58605162e+00 -.16529660e+01  .19952918e-03", 
" -.21149793e+01 -.56226837e+00 -.13642022e+01  .19801495e-03", 
" -.21004858e+01 -.52980807e+00 -.12745895e+01  .19951799e-03", 
" -.20861391e+01 -.50361163e+00 -.11525372e+01  .20021571e-03", 
" -.20720412e+01 -.47383887e+00 -.11332463e+01  .20101170e-03", 
" -.20581222e+01 -.44760739e+00 -.10689268e+01  .19745282e-03", 
" -.20444313e+01 -.41587804e+00 -.11483721e+01  .19867105e-03", 
" -.20308802e+01 -.39212910e+00 -.10546622e+01  .20081947e-03", 
" -.20175809e+01 -.36265707e+00 -.10844851e+01  .19915494e-03", 
" -.20044467e+01 -.33444262e+00 -.11087996e+01  .20081411e-03", 
" -.19914462e+01 -.31443778e+00 -.98386088e+00  .19974974e-03", 
" -.19785819e+01 -.29377576e+00 -.91297418e+00  .19655050e-03", 
" -.19659145e+01 -.26762968e+00 -.96515373e+00  .19520370e-03", 
" -.19533942e+01 -.24626280e+00 -.92112400e+00  .19646798e-03", 
" -.19409728e+01 -.22714035e+00 -.85315949e+00  .19297388e-03", 
" -.19287400e+01 -.20377381e+00 -.88337732e+00  .19545379e-03", 
" -.19165754e+01 -.18502947e+00 -.86727524e+00  .19527557e-03", 
" -.19045658e+01 -.16477942e+00 -.86326945e+00  .19566380e-03", 
" -.18926804e+01 -.14285389e+00 -.89143369e+00  .19661760e-03", 
" -.18808919e+01 -.12295412e+00 -.89863215e+00  .19511864e-03", 
" -.18691948e+01 -.10737733e+00 -.83267780e+00  .19237347e-03", 
" -.18576276e+01 -.87655512e-01 -.86809130e+00  .19306644e-03", 
" -.18461080e+01 -.75188840e-01 -.70774051e+00  .18886992e-03", 
" -.18346926e+01 -.60336642e-01 -.67768811e+00  .18894779e-03", 
" -.18233913e+01 -.39205177e-01 -.78063492e+00  .18742969e-03", 
" -.18121595e+01 -.24985630e-01 -.73647473e+00  .18871729e-03", 
" -.18010748e+01 -.62875752e-03 -.88746826e+00  .18583934e-03", 
" -.17899998e+01  .15059867e-01 -.90280343e+00  .18648997e-03", 
" -.17790099e+01  .33270727e-01 -.98041327e+00  .18378424e-03", 
" -.17681439e+01  .52989190e-01 -.10981856e+01  .18090205e-03", 
" -.17572980e+01  .67626579e-01 -.11316028e+01  .18038668e-03", 
" -.17464864e+01  .78569728e-01 -.10858774e+01  .17972896e-03", 
" -.17357592e+01  .91624031e-01 -.10996467e+01  .17765311e-03", 
" -.17250834e+01  .10038802e+00 -.10269242e+01  .17793723e-03", 
" -.17144549e+01  .10904268e+00 -.95747111e+00  .17580255e-03", 
" -.17039073e+01  .12219995e+00 -.99586067e+00  .17370706e-03", 
" -.16934262e+01  .13243873e+00 -.97589505e+00  .17197697e-03", 
" -.16829589e+01  .14383958e+00 -.99047626e+00  .17219931e-03", 
" -.16725443e+01  .15173191e+00 -.91504125e+00  .17061485e-03", 
" -.16621483e+01  .15622396e+00 -.80409720e+00  .16704158e-03", 
" -.16518271e+01  .16701012e+00 -.80727036e+00  .16416760e-03", 
" -.16415916e+01  .17901510e+00 -.82916889e+00  .16352164e-03", 
" -.16313425e+01  .18758243e+00 -.81027202e+00  .16173530e-03", 
" -.16211335e+01  .19404380e+00 -.76550562e+00  .16171732e-03", 
" -.16109800e+01  .20374386e+00 -.77430061e+00  .16307486e-03", 
" -.16008359e+01  .21201881e+00 -.77560122e+00  .16181563e-03", 
" -.15907772e+01  .22459946e+00 -.86052651e+00  .16145577e-03", 
" -.15806943e+01  .23278845e+00 -.84880486e+00  .16063169e-03", 
" -.15706109e+01  .23899470e+00 -.83331088e+00  .16244730e-03", 
" -.15606265e+01  .25086101e+00 -.89041047e+00  .16436121e-03", 
" -.15506159e+01  .25714031e+00 -.86690820e+00  .16316870e-03", 
" -.15406039e+01  .25862364e+00 -.72604016e+00  .16330874e-03", 
" -.15306679e+01  .26721931e+00 -.75121930e+00  .16179546e-03", 
" -.15207914e+01  .28310956e+00 -.95613145e+00  .16003631e-03", 
" -.15108772e+01  .28790099e+00 -.88822422e+00  .16025314e-03", 
" -.15009704e+01  .29646853e+00 -.93671452e+00  .16189133e-03", 
" -.14910531e+01  .30101415e+00 -.88909654e+00  .16251347e-03", 
" -.14811563e+01  .30627485e+00 -.87380265e+00  .16188332e-03", 
" -.14712513e+01  .30993725e+00 -.83429989e+00  .15965619e-03", 
" -.14613588e+01  .31604337e+00 -.87084619e+00  .16027394e-03", 
" -.14514224e+01  .31659659e+00 -.80538020e+00  .15910275e-03", 
" -.14415037e+01  .31694038e+00 -.70527328e+00  .15939498e-03", 
" -.14316466e+01  .32321303e+00 -.72317376e+00  .16115292e-03", 
" -.14217701e+01  .32902126e+00 -.74474183e+00  .16166593e-03", 
" -.14118750e+01  .33419775e+00 -.72022671e+00  .16187624e-03", 
" -.14019622e+01  .33697917e+00 -.69017036e+00  .16120112e-03", 
" -.13921236e+01  .34617129e+00 -.79327940e+00  .16418093e-03", 
" -.13822497e+01  .35440133e+00 -.89671806e+00  .16311580e-03", 
" -.13723049e+01  .35724168e+00 -.89584461e+00  .16363646e-03", 
" -.13623832e+01  .36299692e+00 -.91749526e+00  .16378647e-03", 
" -.13524246e+01  .36442252e+00 -.90952161e+00  .16518792e-03", 
" -.13424428e+01  .36581315e+00 -.89543837e+00  .16432983e-03", 
" -.13324950e+01  .37111401e+00 -.92103799e+00  .16656568e-03", 
" -.13224521e+01  .37124002e+00 -.87884473e+00  .16887610e-03", 
" -.13124081e+01  .37148348e+00 -.81410147e+00  .16756200e-03", 
" -.13023540e+01  .37356263e+00 -.79982733e+00  .16793501e-03", 
" -.12922486e+01  .37258898e+00 -.72196972e+00  .16884081e-03", 
" -.12821378e+01  .37447478e+00 -.72576488e+00  .16950034e-03", 
" -.12719716e+01  .37396831e+00 -.69644018e+00  .16850628e-03", 
" -.12617959e+01  .37791164e+00 -.74533633e+00  .16930216e-03", 
" -.12515544e+01  .37856999e+00 -.73787060e+00  .17087832e-03", 
" -.12412907e+01  .37842095e+00 -.69415371e+00  .17236136e-03", 
" -.12309812e+01  .37940438e+00 -.67411035e+00  .17191024e-03", 
" -.12205965e+01  .37628850e+00 -.65065534e+00  .17250463e-03", 
" -.12101979e+01  .37857165e+00 -.68026395e+00  .17405141e-03", 
" -.11997738e+01  .37664023e+00 -.61637925e+00  .17405392e-03", 
" -.11892750e+01  .37756750e+00 -.61143032e+00  .17349760e-03", 
" -.11787167e+01  .37554154e+00 -.56969487e+00  .17457235e-03", 
" -.11680931e+01  .37626291e+00 -.59057816e+00  .17477775e-03", 
" -.11574468e+01  .38079989e+00 -.66866865e+00  .17582189e-03", 
" -.11466718e+01  .37726245e+00 -.60965717e+00  .17618490e-03", 
" -.11358714e+01  .37808679e+00 -.62703758e+00  .17747324e-03", 
" -.11249773e+01  .38048448e+00 -.68972889e+00  .18006468e-03", 
" -.11140139e+01  .37927032e+00 -.63828819e+00  .18413593e-03", 
" -.11029733e+01  .38207040e+00 -.71770813e+00  .18696867e-03", 
" -.10918717e+01  .38585605e+00 -.82646198e+00  .18696184e-03", 
" -.10805737e+01  .37887188e+00 -.73840892e+00  .18879341e-03", 
" -.10692627e+01  .38004923e+00 -.79635241e+00  .19150664e-03", 
" -.10577965e+01  .37840751e+00 -.82475820e+00  .19167822e-03", 
" -.10462415e+01  .37522736e+00 -.79241927e+00  .19024662e-03", 
" -.10346064e+01  .36971404e+00 -.69322225e+00  .18910428e-03", 
" -.10228215e+01  .36584958e+00 -.65854700e+00  .18761196e-03", 
" -.10109486e+01  .36178961e+00 -.61525877e+00  .18946771e-03", 
" -.99892467e+00  .35712776e+00 -.58164616e+00  .19139645e-03", 
" -.98679645e+00  .35812190e+00 -.69017649e+00  .19200696e-03", 
" -.97458308e+00  .35625185e+00 -.67411458e+00  .19126236e-03", 
" -.96215216e+00  .34901648e+00 -.56471040e+00  .19128064e-03", 
" -.94957729e+00  .34653098e+00 -.56384651e+00  .19149381e-03", 
" -.93685900e+00  .34088388e+00 -.47467247e+00  .19529760e-03", 
" -.92396602e+00  .34026067e+00 -.55192957e+00  .19545575e-03", 
" -.91084347e+00  .33334363e+00 -.48707625e+00  .19728140e-03", 
" -.89750621e+00  .32458531e+00 -.42959732e+00  .19738252e-03", 
" -.88405013e+00  .32212606e+00 -.45514481e+00  .19604450e-03", 
" -.87034894e+00  .32048496e+00 -.53366025e+00  .20038725e-03", 
" -.85651532e+00  .32406715e+00 -.67510137e+00  .20127717e-03", 
" -.84238902e+00  .31978453e+00 -.62842208e+00  .20491886e-03", 
" -.82799682e+00  .31663599e+00 -.69202182e+00  .20889884e-03", 
" -.81326210e+00  .30673156e+00 -.61416018e+00  .21109307e-03", 
" -.79831414e+00  .30129834e+00 -.58708119e+00  .21096927e-03", 
" -.78296292e+00  .29007264e+00 -.49145536e+00  .21317613e-03", 
" -.76739269e+00  .29096670e+00 -.64130366e+00  .21569887e-03", 
" -.75147192e+00  .28576481e+00 -.62375885e+00  .21832835e-03", 
" -.73512240e+00  .27242753e+00 -.50677665e+00  .22166904e-03", 
" -.71843566e+00  .26541947e+00 -.51105917e+00  .22329256e-03", 
" -.70127447e+00  .25542147e+00 -.46973168e+00  .22562223e-03", 
" -.68369226e+00  .24746858e+00 -.44511794e+00  .23422196e-03", 
" -.66551222e+00  .23067164e+00 -.22521675e+00  .23813487e-03", 
" -.64691969e+00  .22137632e+00 -.14884289e+00  .24136098e-03", 
" -.62776314e+00  .21243686e+00 -.12295539e+00  .24460789e-03", 
" -.60803120e+00  .20391747e+00 -.45042197e-01  .24639859e-03", 
" -.58776534e+00  .20785281e+00 -.25340464e+00  .24902360e-03", 
" -.56659219e+00  .19242274e+00 -.95872269e-01  .25504682e-03", 
" -.54481681e+00  .18585259e+00  .67592600e-02  .26126298e-03", 
" -.52227691e+00  .17939219e+00  .12805968e+00  .26779072e-03", 
" -.49902402e+00  .18629510e+00 -.25909939e-02  .27751017e-03", 
" -.47455597e+00  .16634178e+00  .29408282e+00  .28023730e-03", 
" -.44942893e+00  .17086447e+00  .19787528e+00  .28969207e-03", 
" -.42325207e+00  .16852489e+00  .26021021e+00  .29895524e-03", 
" -.39615260e+00  .18004527e+00  .96294101e-01  .30580018e-03", 
" -.36785654e+00  .18188125e+00  .14757189e+00  .30826841e-03", 
" -.33829034e+00  .18925293e+00  .29704680e-01  .31344492e-03", 
" -.30737709e+00  .18591648e+00  .15930691e+00  .32347276e-03", 
" -.27520404e+00  .19224474e+00  .56279420e-01  .33901556e-03", 
" -.24153106e+00  .20738951e+00 -.19314340e+00  .35458201e-03", 
" -.20623329e+00  .21925393e+00 -.29555825e+00  .36179101e-03", 
" -.16891557e+00  .21524323e+00 -.58033195e-01  .36667913e-03", 
" -.12982055e+00  .22764905e+00 -.11091925e+00  .37743688e-03", 
" -.88416563e-01  .23244162e+00  .41516632e-01  .37875060e-03", 
" -.44438320e-01  .23101699e+00  .25251890e+00  .39202092e-03", 
"  .20407836e-02  .25688208e+00  .12594515e+00  .40426609e-03", 
"  .51955913e-01  .27193770e+00  .17387645e+00  .41698965e-03", 
"  .10550876e+00  .29007497e+00  .29808872e+00  .42932288e-03", 
"  .16342412e+00  .32060604e+00  .16091836e+00  .43606894e-03", 
"  .22672716e+00  .34367384e+00  .22515148e+00  .45091243e-03", 
"  .29592592e+00  .39560344e+00 -.99697843e-01  .47597510e-03", 
"  .37299278e+00  .42204440e+00  .29981665e+00  .50956229e-03", 
"  .45974143e+00  .53328913e+00 -.75770223e+00  .53178950e-03", 
"  .55949351e+00  .64560262e+00 -.11221509e+01  .57812830e-03", 
"  .67800023e+00  .82113990e+00 -.25159632e+01  .63083375e-03", 
"  .82605690e+00  .92166124e+00 -.17594389e+01  .70270734e-03", 
"  .10230413e+01  .11755749e+01 -.65377972e+00  .82986163e-03", 
"  .10721119e+01  .12785427e+01 -.79467956e+00  .86739436e-03", 
"  .11264329e+01  .13673086e+01 -.50169868e+00  .89968081e-03", 
"  .11874489e+01  .14431379e+01  .11366734e+00  .94786362e-03", 
"  .12554238e+01  .16468018e+01 -.17543823e+01  .10032816e-02", 
"  .13346118e+01  .18229906e+01 -.16891448e+01  .10457654e-02", 
"  .14300124e+01  .19909269e+01 -.48283608e+00  .11340558e-02", 
"  .15479886e+01  .22589383e+01  .15607755e+01  .12954805e-02", 
"  .17068162e+01  .27790800e+01  .13713224e+01  .15047178e-02", 
"  .19634130e+01  .38463706e+01 -.16940449e+01  .19776850e-02", 
"  .22005907e+01  .52551209e+01 -.71521486e+01  .25192819e-02", 
"  .24965608e+01  .67223425e+01 -.46960766e+00  .38447868e-02", 
"  .27115749e+01  .73022269e+01  .18818133e+02  .50910498e-02", 
"coc       1  3  2   20", " -.50866463e+01 -.27171376e+02 -.14761129e+03  .38614785e-02", 
" -.49307063e+01 -.24801972e+02 -.11415806e+03  .29335441e-02", 
" -.47130312e+01 -.21555457e+02 -.78169391e+02  .19261066e-02", 
" -.45421104e+01 -.19014152e+02 -.58893474e+02  .14409780e-02", 
" -.43614287e+01 -.16374677e+02 -.47855563e+02  .10976621e-02", 
" -.42507081e+01 -.15001344e+02 -.39036942e+02  .94277082e-03", 
" -.41699629e+01 -.13950608e+02 -.35603628e+02  .83717391e-03", 
" -.41058287e+01 -.13186737e+02 -.32439516e+02  .74868341e-03", 
" -.40518989e+01 -.12622038e+02 -.29132524e+02  .71231558e-03", 
" -.40059611e+01 -.12100812e+02 -.27349548e+02  .67089199e-03", 
" -.39653682e+01 -.11681013e+02 -.25182715e+02  .62991411e-03", 
" -.39292453e+01 -.11320054e+02 -.23024149e+02  .61558861e-03", 
" -.38966965e+01 -.10951663e+02 -.22268897e+02  .58303704e-03", 
" -.37668925e+01 -.96654604e+01 -.17578750e+02  .49142428e-03", 
" -.36703041e+01 -.88062638e+01 -.14187010e+02  .43583598e-03", 
" -.35933614e+01 -.80978985e+01 -.12645969e+02  .37505477e-03", 
" -.35283003e+01 -.75721330e+01 -.10833692e+02  .36579070e-03", 
" -.34721919e+01 -.71005556e+01 -.10053706e+02  .34443616e-03", 
" -.34221752e+01 -.67344427e+01 -.85536373e+01  .32959392e-03", 
" -.33770670e+01 -.64203593e+01 -.71489682e+01  .30627586e-03", 
" -.33361543e+01 -.61199057e+01 -.64217721e+01  .29742883e-03", 
" -.32983984e+01 -.58495534e+01 -.60135748e+01  .30026006e-03", 
" -.32634078e+01 -.56108986e+01 -.53604802e+01  .28949819e-03", 
" -.32306678e+01 -.53881158e+01 -.49829151e+01  .27671715e-03", 
" -.31999176e+01 -.51912225e+01 -.44480190e+01  .26094218e-03", 
" -.31709582e+01 -.49923103e+01 -.43269203e+01  .25290191e-03", 
" -.31432933e+01 -.48277165e+01 -.38045987e+01  .24982706e-03", 
" -.31168974e+01 -.46859762e+01 -.30725840e+01  .23673843e-03", 
" -.30918977e+01 -.45156702e+01 -.32064056e+01  .23451583e-03", 
" -.30678958e+01 -.43642385e+01 -.31840191e+01  .22884343e-03", 
" -.30447357e+01 -.42284483e+01 -.30126115e+01  .22496028e-03", 
" -.30225757e+01 -.40923692e+01 -.29099485e+01  .22499798e-03", 
" -.30011011e+01 -.39699923e+01 -.27842453e+01  .22089407e-03", 
" -.29803475e+01 -.38528396e+01 -.26113551e+01  .22105202e-03", 
" -.29602810e+01 -.37422429e+01 -.24511556e+01  .22169628e-03", 
" -.29408636e+01 -.36355601e+01 -.23406614e+01  .21959099e-03", 
" -.29219264e+01 -.35428934e+01 -.19780216e+01  .21579401e-03", 
" -.29035152e+01 -.34527856e+01 -.17582644e+01  .21230209e-03", 
" -.28855536e+01 -.33648278e+01 -.16580446e+01  .20903132e-03", 
" -.28681476e+01 -.32704811e+01 -.16591743e+01  .20621831e-03", 
" -.28511416e+01 -.31805916e+01 -.16509023e+01  .20420116e-03", 
" -.28344757e+01 -.30967130e+01 -.16052662e+01  .20287179e-03", 
" -.28181859e+01 -.30152029e+01 -.16440510e+01  .19915462e-03", 
" -.28021970e+01 -.29472725e+01 -.14196223e+01  .19375175e-03", 
" -.27865707e+01 -.28735521e+01 -.13448543e+01  .19076409e-03", 
" -.27712281e+01 -.28063523e+01 -.12314353e+01  .19048114e-03", 
" -.27562024e+01 -.27399919e+01 -.11311190e+01  .18852408e-03", 
" -.27414168e+01 -.26742731e+01 -.10544648e+01  .18428514e-03", 
" -.27269614e+01 -.26087348e+01 -.99688283e+00  .18134083e-03", 
" -.27127751e+01 -.25411396e+01 -.10577602e+01  .17986086e-03", 
" -.26987926e+01 -.24776436e+01 -.10782297e+01  .17889435e-03", 
" -.26850356e+01 -.24185074e+01 -.10019691e+01  .17920122e-03", 
" -.26713804e+01 -.23657555e+01 -.90158775e+00  .17679202e-03", 
" -.26579540e+01 -.23168784e+01 -.74693272e+00  .17764910e-03", 
" -.26446982e+01 -.22679534e+01 -.64016260e+00  .17723920e-03", 
" -.26317135e+01 -.22126108e+01 -.65058659e+00  .17681592e-03", 
" -.26188723e+01 -.21580217e+01 -.75596393e+00  .17722207e-03", 
" -.26061923e+01 -.21116850e+01 -.60773709e+00  .17712702e-03", 
" -.25936509e+01 -.20606015e+01 -.66041465e+00  .17471253e-03", 
" -.25812400e+01 -.20173082e+01 -.56759378e+00  .17381891e-03", 
" -.25690321e+01 -.19678539e+01 -.62081652e+00  .17228251e-03", 
" -.25569319e+01 -.19247170e+01 -.55860353e+00  .17339947e-03", 
" -.25449809e+01 -.18827412e+01 -.48724014e+00  .16939067e-03", 
" -.25331449e+01 -.18415581e+01 -.41875830e+00  .17101826e-03", 
" -.25214357e+01 -.18014803e+01 -.34361343e+00  .16814535e-03", 
" -.25098227e+01 -.17645545e+01 -.25458061e+00  .16704860e-03", 
" -.24983196e+01 -.17265876e+01 -.19008146e+00  .16894031e-03", 
" -.24869121e+01 -.16884587e+01 -.18350886e+00  .16768075e-03", 
" -.24756391e+01 -.16490759e+01 -.20343040e+00  .16662597e-03", 
" -.24643908e+01 -.16171261e+01 -.10833637e+00  .16416228e-03", 
" -.24533346e+01 -.15769907e+01 -.16844538e+00  .16195946e-03", 
" -.24423417e+01 -.15379171e+01 -.25918853e+00  .16092060e-03", 
" -.24313788e+01 -.15072992e+01 -.17953529e+00  .15919770e-03", 
" -.24205646e+01 -.14712678e+01 -.19253879e+00  .15996238e-03", 
" -.24097775e+01 -.14404840e+01 -.15170427e+00  .15792001e-03", 
" -.23991104e+01 -.14068423e+01 -.16381743e+00  .15485111e-03", 
" -.23885202e+01 -.13730114e+01 -.17820226e+00  .15523093e-03", 
" -.23779497e+01 -.13454821e+01 -.68078995e-01  .15687070e-03", 
" -.23674974e+01 -.13095371e+01 -.18280883e+00  .15839287e-03", 
" -.23570824e+01 -.12802605e+01 -.18735917e+00  .15743742e-03", 
" -.23467372e+01 -.12480254e+01 -.21527738e+00  .15993554e-03", 
" -.23364426e+01 -.12192975e+01 -.20078907e+00  .16034259e-03", 
" -.23262282e+01 -.11930409e+01 -.13858179e+00  .16088990e-03", 
" -.23161048e+01 -.11590616e+01 -.25861687e+00  .16237530e-03", 
" -.23059912e+01 -.11282360e+01 -.35601191e+00  .16165744e-03", 
" -.22959369e+01 -.10974262e+01 -.42904531e+00  .16112429e-03", 
" -.22858814e+01 -.10753359e+01 -.29573000e+00  .15909974e-03", 
" -.22758940e+01 -.10444420e+01 -.41621686e+00  .15983648e-03", 
" -.22659693e+01 -.10209742e+01 -.34855757e+00  .15972801e-03", 
" -.22560236e+01 -.10007510e+01 -.24136985e+00  .15675387e-03", 
" -.22461119e+01 -.97977767e+00 -.18416490e+00  .15715539e-03", 
" -.22362772e+01 -.95737680e+00 -.12421993e+00  .15834169e-03", 
" -.22264291e+01 -.93869508e+00 -.17693586e-01  .15962870e-03", 
" -.22166542e+01 -.91553727e+00 -.38308453e-01  .16145231e-03", 
" -.22069260e+01 -.88943246e+00 -.11240201e+00  .16272782e-03", 
" -.21972194e+01 -.86651750e+00 -.10921318e+00  .16400697e-03", 
" -.21875278e+01 -.84492186e+00 -.97221320e-01  .16423375e-03", 
" -.21778275e+01 -.82391904e+00 -.93121805e-01  .16462924e-03", 
" -.21681805e+01 -.80212173e+00 -.10761511e+00  .16438593e-03", 
" -.21585237e+01 -.78278819e+00 -.86432088e-01  .16477076e-03", 
" -.21489105e+01 -.76229689e+00 -.11207078e+00  .16531811e-03", 
" -.21392712e+01 -.74425433e+00 -.83934037e-01  .16696711e-03", 
" -.21297136e+01 -.72213233e+00 -.14531665e+00  .16904667e-03", 
" -.21201656e+01 -.70107202e+00 -.16504947e+00  .16822430e-03", 
" -.21105976e+01 -.68368369e+00 -.12210709e+00  .16797747e-03", 
" -.21010894e+01 -.66243247e+00 -.15497402e+00  .16788962e-03", 
" -.20915632e+01 -.64521505e+00 -.93475420e-01  .16775845e-03", 
" -.20820265e+01 -.62977496e+00 -.40699842e-01  .16777542e-03", 
" -.20725456e+01 -.60860809e+00 -.10434091e+00  .16666690e-03", 
" -.20629845e+01 -.59648078e+00 -.86793643e-02  .16804050e-03", 
" -.20534363e+01 -.58040741e+00 -.58362865e-02  .16995581e-03", 
" -.20438943e+01 -.56690961e+00  .68472138e-01  .17028447e-03", 
" -.20343642e+01 -.55285815e+00  .13575108e+00  .16991586e-03", 
" -.20248052e+01 -.53784656e+00  .16173694e+00  .16887756e-03", 
" -.20152487e+01 -.52530183e+00  .21861827e+00  .16809718e-03", 
" -.20057134e+01 -.50919669e+00  .18842447e+00  .16744440e-03", 
" -.19961648e+01 -.49352978e+00  .18106180e+00  .16631696e-03", 
" -.19866057e+01 -.48073460e+00  .21706664e+00  .16648992e-03", 
" -.19770358e+01 -.46607409e+00  .20566610e+00  .16602831e-03", 
" -.19674753e+01 -.45049422e+00  .22308362e+00  .16593731e-03", 
" -.19578285e+01 -.44104035e+00  .31685930e+00  .16719789e-03", 
" -.19481795e+01 -.42958226e+00  .35328421e+00  .16777186e-03", 
" -.19385488e+01 -.41719726e+00  .39389661e+00  .16732645e-03", 
" -.19288968e+01 -.40374499e+00  .39789584e+00  .16786476e-03", 
" -.19192331e+01 -.38691270e+00  .29923865e+00  .16697780e-03", 
" -.19095988e+01 -.36818480e+00  .20609520e+00  .16693499e-03", 
" -.18998226e+01 -.35941163e+00  .24818681e+00  .16792572e-03", 
" -.18900844e+01 -.34610037e+00  .21069250e+00  .16880687e-03", 
" -.18803259e+01 -.33117165e+00  .17220116e+00  .16951465e-03", 
" -.18705252e+01 -.31704217e+00  .13346230e+00  .17108321e-03", 
" -.18606565e+01 -.30335443e+00  .86365004e-01  .17104038e-03", 
" -.18507677e+01 -.29026431e+00  .68196177e-01  .17197624e-03", 
" -.18408099e+01 -.28269528e+00  .14909313e+00  .17331053e-03", 
" -.18308420e+01 -.27216156e+00  .16157147e+00  .17304125e-03", 
" -.18208193e+01 -.26067107e+00  .14125723e+00  .17254443e-03", 
" -.18107940e+01 -.24791219e+00  .10446500e+00  .17271232e-03", 
" -.18007108e+01 -.23259369e+00  .13912750e-01  .17184582e-03", 
" -.17905538e+01 -.21856613e+00 -.88065069e-01  .17123345e-03", 
" -.17803150e+01 -.21285480e+00 -.14280975e-01  .17139190e-03", 
" -.17700772e+01 -.19999409e+00 -.77571827e-01  .17264465e-03", 
" -.17597432e+01 -.19106031e+00 -.69209972e-01  .17147848e-03", 
" -.17493338e+01 -.18218852e+00 -.68365250e-01  .17216753e-03", 
" -.17388269e+01 -.17656640e+00 -.41921739e-01  .17050152e-03", 
" -.17282812e+01 -.16629670e+00 -.80921669e-01  .17269965e-03", 
" -.17176402e+01 -.15521516e+00 -.17772815e+00  .17295341e-03", 
" -.17069604e+01 -.14384253e+00 -.23041876e+00  .17491604e-03", 
" -.16961661e+01 -.13852901e+00 -.19330982e+00  .17352501e-03", 
" -.16852335e+01 -.13598033e+00 -.11263467e+00  .17345320e-03", 
" -.16742779e+01 -.12955831e+00 -.98353916e-01  .17574562e-03", 
" -.16632206e+01 -.12255368e+00 -.10619176e+00  .17461920e-03", 
" -.16520216e+01 -.11777111e+00 -.87873714e-01  .17653690e-03", 
" -.16407330e+01 -.11397623e+00 -.70865997e-01  .17721153e-03", 
" -.16293302e+01 -.10834439e+00 -.96736560e-01  .17976937e-03", 
" -.16179319e+01 -.10039137e+00 -.11281880e+00  .17970468e-03", 
" -.16063298e+01 -.95531999e-01 -.81659441e-01  .17952474e-03", 
" -.15946275e+01 -.90776183e-01 -.65572257e-01  .18223288e-03", 
" -.15827998e+01 -.86440353e-01 -.29837640e-01  .18434542e-03", 
" -.15707662e+01 -.87327954e-01  .76993295e-01  .18441336e-03", 
" -.15586028e+01 -.84667544e-01  .88750789e-01  .18449705e-03", 
" -.15463148e+01 -.81459098e-01  .88916433e-01  .18365568e-03", 
" -.15338722e+01 -.76866146e-01  .69743013e-01  .18430784e-03", 
" -.15212332e+01 -.75955305e-01  .93235230e-01  .18527875e-03", 
" -.15084336e+01 -.74046449e-01  .10186645e+00  .18609646e-03", 
" -.14954464e+01 -.72169419e-01  .11137764e+00  .18601075e-03", 
" -.14823124e+01 -.71244762e-01  .17491495e+00  .18940656e-03", 
" -.14689415e+01 -.69440273e-01  .20026571e+00  .18914923e-03", 
" -.14553270e+01 -.66927212e-01  .19143224e+00  .18828426e-03", 
" -.14414994e+01 -.66397917e-01  .19414544e+00  .19133206e-03", 
" -.14273862e+01 -.68752976e-01  .25306359e+00  .19443049e-03", 
" -.14130658e+01 -.71351635e-01  .31370384e+00  .19520166e-03", 
" -.13984480e+01 -.70801177e-01  .27511205e+00  .19524131e-03", 
" -.13835481e+01 -.74082870e-01  .34605100e+00  .19842665e-03", 
" -.13683000e+01 -.78736067e-01  .40626997e+00  .20030130e-03", 
" -.13528179e+01 -.72926371e-01  .25541491e+00  .20598824e-03", 
" -.13369773e+01 -.78305469e-01  .38798529e+00  .20471476e-03", 
" -.13207254e+01 -.81755600e-01  .40272444e+00  .20649070e-03", 
" -.13041296e+01 -.81808932e-01  .35465334e+00  .21179587e-03", 
" -.12870818e+01 -.83997180e-01  .32932730e+00  .21360835e-03", 
" -.12695577e+01 -.87399973e-01  .33136355e+00  .21557503e-03", 
" -.12515976e+01 -.89300774e-01  .28868508e+00  .22114285e-03", 
" -.12331170e+01 -.90520076e-01  .23235665e+00  .22327525e-03", 
" -.12140195e+01 -.97256535e-01  .26895446e+00  .22867590e-03", 
" -.11942738e+01 -.11604779e+00  .58589018e+00  .22962818e-03", 
" -.11739648e+01 -.11989725e+00  .56167500e+00  .23567394e-03", 
" -.11529590e+01 -.12927051e+00  .67672710e+00  .23871124e-03", 
" -.11311342e+01 -.14309047e+00  .82917625e+00  .24815476e-03", 
" -.11085851e+01 -.14495549e+00  .64986348e+00  .25301378e-03", 
" -.10851229e+01 -.15936919e+00  .80192461e+00  .26241768e-03", 
" -.10607377e+01 -.17115951e+00  .89260591e+00  .26523946e-03", 
" -.10352347e+01 -.18743307e+00  .10148311e+01  .26853954e-03", 
" -.10086674e+01 -.19568819e+00  .10277693e+01  .27512818e-03", 
" -.98090034e+00 -.20067625e+00  .89100453e+00  .28614835e-03", 
" -.95150742e+00 -.22048327e+00  .99723131e+00  .28908464e-03", 
" -.92064017e+00 -.23138252e+00  .90202743e+00  .29290616e-03", 
" -.88799810e+00 -.24341708e+00  .74914984e+00  .30577840e-03", 
" -.85363217e+00 -.25792716e+00  .70513086e+00  .31408214e-03", 
" -.81696455e+00 -.28243750e+00  .77189605e+00  .32808512e-03", 
" -.77798628e+00 -.29650323e+00  .53262793e+00  .34136418e-03", 
" -.73607595e+00 -.32834707e+00  .54438830e+00  .35304585e-03", 
" -.69086661e+00 -.37182629e+00  .65046054e+00  .37117034e-03", 
" -.64210015e+00 -.41179772e+00  .77303872e+00  .37494645e-03", 
" -.58889422e+00 -.44506664e+00  .41440204e+00  .39055142e-03", 
" -.53035582e+00 -.49906992e+00  .40322223e+00  .40995518e-03", 
" -.46495843e+00 -.58152660e+00  .79369878e+00  .41907497e-03", 
" -.39130238e+00 -.63534984e+00  .21257781e+00  .44691869e-03", 
" -.30590176e+00 -.71821816e+00 -.28701513e+00  .47890868e-03", 
" -.20410974e+00 -.86486597e+00 -.19698067e+00  .52994961e-03", 
" -.77650216e-01 -.99029787e+00 -.15642023e+01  .58221940e-03", 
"  .93472251e-01 -.12128946e+01 -.27392580e+01  .67985149e-03", 
"  .13605266e+00 -.12252363e+01 -.40157723e+01  .69385264e-03", 
"  .18341606e+00 -.12852719e+01 -.45988640e+01  .72530981e-03", 
"  .23659158e+00 -.13900712e+01 -.44566459e+01  .77908731e-03", 
"  .29695632e+00 -.15170405e+01 -.42546521e+01  .82923165e-03", 
"  .36686334e+00 -.16407138e+01 -.44374745e+01  .93221285e-03", 
"  .45044813e+00 -.17182513e+01 -.64749229e+01  .10613713e-02", 
"  .55466028e+00 -.18207546e+01 -.83556863e+01  .11245921e-02", 
"  .69583903e+00 -.19015623e+01 -.12959783e+02  .13299568e-02", 
"  .92649582e+00 -.22975707e+01 -.14712574e+02  .17391438e-02", 
"  .11425518e+01 -.24169428e+01 -.23198312e+02  .22102168e-02", 
"  .14130998e+01 -.26466914e+01 -.31112200e+02  .33916129e-02", 
"  .16016616e+01 -.25423405e+01 -.41336472e+02  .45180154e-02", 
"coct      1  4  3   25", " -.54923655e+01 -.33669050e+02 -.27697199e+03  .13597675e+04  .49451223e-02", 
" -.53397448e+01 -.31203752e+02 -.19942978e+03  .73730772e+03  .36813644e-02", 
" -.51271486e+01 -.27852822e+02 -.10693891e+03 -.76064416e+02  .26065663e-02", 
" -.49569640e+01 -.25046036e+02 -.90882387e+02  .25498544e+03  .19718847e-02", 
" -.47814471e+01 -.21663740e+02 -.97911122e+02  .63219092e+03  .14855913e-02", 
" -.46733344e+01 -.19921610e+02 -.91078566e+02  .64503763e+03  .12473947e-02", 
" -.45946792e+01 -.18801506e+02 -.77554513e+02  .48919279e+03  .10954207e-02", 
" -.45312520e+01 -.18147922e+02 -.57410819e+02  .24488838e+03  .10227357e-02", 
" -.44780914e+01 -.17629830e+02 -.42535857e+02  .75259220e+02  .93470296e-03", 
" -.44328883e+01 -.17156942e+02 -.31576850e+02 -.45714988e+02  .83679656e-03", 
" -.43934981e+01 -.16583884e+02 -.32634671e+02  .79852984e+01  .77134941e-03", 
" -.43582573e+01 -.16170975e+02 -.24818916e+02 -.11743735e+03  .71984676e-03", 
" -.43260829e+01 -.15795455e+02 -.21334662e+02 -.14073422e+03  .70259207e-03", 
" -.41998163e+01 -.13963010e+02 -.31202311e+02  .11250139e+03  .59417473e-03", 
" -.41059156e+01 -.12893073e+02 -.26878334e+02  .12533192e+03  .55898524e-03", 
" -.40309824e+01 -.12017899e+02 -.23995258e+02  .99090507e+02  .51944990e-03", 
" -.39679015e+01 -.11347469e+02 -.20060736e+02  .69079351e+02  .48719042e-03", 
" -.39130801e+01 -.10716116e+02 -.21805166e+02  .11175983e+03  .47063547e-03", 
" -.38644514e+01 -.10253900e+02 -.19010902e+02  .96856290e+02  .45692870e-03", 
" -.38205495e+01 -.99021014e+01 -.12852786e+02  .15810022e+02  .44382122e-03", 
" -.37804754e+01 -.95289443e+01 -.12378985e+02  .36953736e+02  .40357619e-03", 
" -.37438265e+01 -.91595109e+01 -.13410327e+02  .68766555e+02  .39177096e-03", 
" -.37095415e+01 -.88820217e+01 -.10813311e+02  .42145143e+02  .38346617e-03", 
" -.36778299e+01 -.85427741e+01 -.13698772e+02  .98409495e+02  .36882645e-03", 
" -.36479660e+01 -.82246759e+01 -.16715302e+02  .15480445e+03  .35454330e-03", 
" -.36196277e+01 -.79842850e+01 -.16045123e+02  .15150596e+03  .34159782e-03", 
" -.35927834e+01 -.77357750e+01 -.16855148e+02  .16857345e+03  .34338116e-03", 
" -.35672055e+01 -.75140727e+01 -.17212056e+02  .18194799e+03  .33756331e-03", 
" -.35427494e+01 -.73229568e+01 -.15934717e+02  .16429128e+03  .32366082e-03", 
" -.35193990e+01 -.71425433e+01 -.15213921e+02  .16067736e+03  .31983375e-03", 
" -.34969574e+01 -.69415520e+01 -.16743043e+02  .19495753e+03  .31481462e-03", 
" -.34752406e+01 -.67965869e+01 -.15376013e+02  .18060991e+03  .30688512e-03", 
" -.34542087e+01 -.66723707e+01 -.12919059e+02  .14267095e+03  .30035716e-03", 
" -.34339849e+01 -.65180641e+01 -.13324317e+02  .16003312e+03  .29446252e-03", 
" -.34143422e+01 -.63907112e+01 -.12215046e+02  .14626068e+03  .28588388e-03", 
" -.33952858e+01 -.62740190e+01 -.10583684e+02  .12252653e+03  .28494358e-03", 
" -.33769115e+01 -.61390463e+01 -.10312594e+02  .11922188e+03  .27876978e-03", 
" -.33588933e+01 -.60271372e+01 -.97424796e+01  .11692414e+03  .27649916e-03", 
" -.33415366e+01 -.58932409e+01 -.98483739e+01  .11649039e+03  .27435888e-03", 
" -.33244530e+01 -.57887984e+01 -.89736967e+01  .10430040e+03  .27456726e-03", 
" -.33078072e+01 -.56783890e+01 -.87503675e+01  .10242439e+03  .26867909e-03", 
" -.32915553e+01 -.55700390e+01 -.88798962e+01  .10813302e+03  .26652984e-03", 
" -.32756434e+01 -.54739576e+01 -.84361999e+01  .10459212e+03  .26796271e-03", 
" -.32601038e+01 -.53805652e+01 -.75460075e+01  .89825834e+02  .26527270e-03", 
" -.32448472e+01 -.52928006e+01 -.71368563e+01  .87715396e+02  .26592327e-03", 
" -.32300151e+01 -.51770332e+01 -.83139160e+01  .10893455e+03  .25994234e-03", 
" -.32153256e+01 -.50989032e+01 -.75112719e+01  .98623773e+02  .25539150e-03", 
" -.32009881e+01 -.50057009e+01 -.78003673e+01  .10566903e+03  .25477506e-03", 
" -.31868618e+01 -.49217554e+01 -.77541938e+01  .10707597e+03  .25025260e-03", 
" -.31729416e+01 -.48533522e+01 -.69212241e+01  .97913414e+02  .24913430e-03", 
" -.31592963e+01 -.47742491e+01 -.66305886e+01  .94384937e+02  .24742455e-03", 
" -.31458976e+01 -.46930462e+01 -.63791867e+01  .88354936e+02  .24333581e-03", 
" -.31326748e+01 -.46081544e+01 -.70069731e+01  .99479378e+02  .23830344e-03", 
" -.31196665e+01 -.45296226e+01 -.70673746e+01  .10017825e+03  .23392448e-03", 
" -.31068636e+01 -.44494395e+01 -.74374832e+01  .10509281e+03  .22937316e-03", 
" -.30942356e+01 -.43776815e+01 -.70583016e+01  .96979392e+02  .23009026e-03", 
" -.30818225e+01 -.43001573e+01 -.71736910e+01  .97796170e+02  .23184875e-03", 
" -.30694602e+01 -.42520972e+01 -.56615862e+01  .72699102e+02  .23099266e-03", 
" -.30572771e+01 -.41836956e+01 -.58500807e+01  .77536249e+02  .22798954e-03", 
" -.30452107e+01 -.41203751e+01 -.60144891e+01  .83052358e+02  .22538591e-03", 
" -.30332979e+01 -.40765875e+01 -.46138785e+01  .60316546e+02  .22808711e-03", 
" -.30215409e+01 -.40300460e+01 -.35290836e+01  .42611460e+02  .22989186e-03", 
" -.30099295e+01 -.39626937e+01 -.39341013e+01  .49530557e+02  .22910646e-03", 
" -.29984325e+01 -.39192998e+01 -.28420112e+01  .32114267e+02  .22739744e-03", 
" -.29870956e+01 -.38537248e+01 -.33758369e+01  .43346790e+02  .22853277e-03", 
" -.29758939e+01 -.37755825e+01 -.46535144e+01  .62431490e+02  .23115634e-03", 
" -.29647331e+01 -.37212363e+01 -.48000127e+01  .67693156e+02  .22966485e-03", 
" -.29537361e+01 -.36550964e+01 -.55021699e+01  .79867423e+02  .23045810e-03", 
" -.29428204e+01 -.35932007e+01 -.61365029e+01  .90619700e+02  .22795912e-03", 
" -.29319887e+01 -.35363160e+01 -.62118392e+01  .90399406e+02  .22562782e-03", 
" -.29211807e+01 -.34967602e+01 -.54779023e+01  .79492798e+02  .22344208e-03", 
" -.29105441e+01 -.34435561e+01 -.54390180e+01  .77629120e+02  .22172681e-03", 
" -.28999470e+01 -.33924878e+01 -.56997306e+01  .84321169e+02  .22101865e-03", 
" -.28894537e+01 -.33458801e+01 -.55932150e+01  .83159622e+02  .22192772e-03", 
" -.28790487e+01 -.32994961e+01 -.56412999e+01  .85707049e+02  .22372894e-03", 
" -.28687964e+01 -.32348064e+01 -.65896387e+01  .10034103e+03  .22272277e-03", 
" -.28584861e+01 -.32055679e+01 -.54041757e+01  .80871966e+02  .22138442e-03", 
" -.28483270e+01 -.31461299e+01 -.62375735e+01  .93828774e+02  .22194980e-03", 
" -.28381911e+01 -.30941235e+01 -.70920497e+01  .11084856e+03  .22223863e-03", 
" -.28281465e+01 -.30525906e+01 -.68120078e+01  .10523616e+03  .22210030e-03", 
" -.28181180e+01 -.30047644e+01 -.73791834e+01  .11652349e+03  .22336708e-03", 
" -.28081928e+01 -.29596874e+01 -.75948113e+01  .12109862e+03  .22262876e-03", 
" -.27983255e+01 -.29162425e+01 -.77840127e+01  .12531663e+03  .22280781e-03", 
" -.27885525e+01 -.28671295e+01 -.80502987e+01  .12698840e+03  .22006601e-03", 
" -.27787513e+01 -.28434279e+01 -.70198265e+01  .11141277e+03  .21900877e-03", 
" -.27690588e+01 -.27901279e+01 -.79008588e+01  .12493352e+03  .21653341e-03", 
" -.27593990e+01 -.27463976e+01 -.80661619e+01  .12619653e+03  .21257387e-03", 
" -.27497888e+01 -.26982839e+01 -.89752332e+01  .14435520e+03  .21445862e-03", 
" -.27402371e+01 -.26531534e+01 -.95388598e+01  .15527455e+03  .21462826e-03", 
" -.27307095e+01 -.26112346e+01 -.99977862e+01  .16547886e+03  .21451303e-03", 
" -.27211470e+01 -.25916004e+01 -.90833937e+01  .15241695e+03  .21811446e-03", 
" -.27116584e+01 -.25587399e+01 -.89490622e+01  .15154349e+03  .21637534e-03", 
" -.27022347e+01 -.25190671e+01 -.92514520e+01  .15707502e+03  .21551170e-03", 
" -.26928778e+01 -.24686349e+01 -.10225611e+02  .17345548e+03  .21356398e-03", 
" -.26835773e+01 -.24258646e+01 -.10594901e+02  .17900280e+03  .21431563e-03", 
" -.26742422e+01 -.23897062e+01 -.10592142e+02  .17687420e+03  .21125078e-03", 
" -.26649366e+01 -.23533056e+01 -.10846546e+02  .18212835e+03  .21265899e-03", 
" -.26556189e+01 -.23205216e+01 -.11073652e+02  .18704184e+03  .21289476e-03", 
" -.26463097e+01 -.23026553e+01 -.10221722e+02  .17341778e+03  .21528664e-03", 
" -.26370373e+01 -.22807621e+01 -.96773797e+01  .16564217e+03  .21286343e-03", 
" -.26277917e+01 -.22597067e+01 -.90765205e+01  .15624824e+03  .21346639e-03", 
" -.26186065e+01 -.22312020e+01 -.87329098e+01  .14796435e+03  .21397618e-03", 
" -.26093955e+01 -.22014090e+01 -.87450891e+01  .14851806e+03  .21389753e-03", 
" -.26001874e+01 -.21773979e+01 -.86389851e+01  .14849109e+03  .21247236e-03", 
" -.25910659e+01 -.21388648e+01 -.91521664e+01  .15646202e+03  .21571206e-03", 
" -.25819557e+01 -.21078826e+01 -.93197141e+01  .16079565e+03  .21523866e-03", 
" -.25728432e+01 -.20816921e+01 -.89477917e+01  .15199116e+03  .21559549e-03", 
" -.25636682e+01 -.20651776e+01 -.83637965e+01  .14315616e+03  .21595270e-03", 
" -.25545336e+01 -.20490089e+01 -.76203809e+01  .13052424e+03  .21553030e-03", 
" -.25453903e+01 -.20305413e+01 -.71676384e+01  .12476319e+03  .21485860e-03", 
" -.25362959e+01 -.20089805e+01 -.70168567e+01  .12370799e+03  .21664819e-03", 
" -.25271825e+01 -.19867569e+01 -.68654101e+01  .12216439e+03  .21656134e-03", 
" -.25180837e+01 -.19710727e+01 -.61585372e+01  .11065293e+03  .21430155e-03", 
" -.25090265e+01 -.19419305e+01 -.62846049e+01  .11126866e+03  .21564177e-03", 
" -.24999593e+01 -.19115611e+01 -.65485957e+01  .11386289e+03  .21464971e-03", 
" -.24907860e+01 -.18952413e+01 -.63057315e+01  .11259319e+03  .21264103e-03", 
" -.24816974e+01 -.18679265e+01 -.64847986e+01  .11577107e+03  .21594137e-03", 
" -.24725739e+01 -.18528034e+01 -.59249855e+01  .10641723e+03  .21837521e-03", 
" -.24634946e+01 -.18235027e+01 -.63524335e+01  .11455626e+03  .22002169e-03", 
" -.24543994e+01 -.18010639e+01 -.62661084e+01  .11358451e+03  .22147148e-03", 
" -.24452469e+01 -.17897050e+01 -.56953931e+01  .10536442e+03  .21904913e-03", 
" -.24361219e+01 -.17652603e+01 -.59871486e+01  .11129696e+03  .21928419e-03", 
" -.24270042e+01 -.17401633e+01 -.61246934e+01  .11203615e+03  .22009512e-03", 
" -.24178573e+01 -.17115971e+01 -.65800325e+01  .12001536e+03  .21795748e-03", 
" -.24086557e+01 -.17001090e+01 -.58515744e+01  .10701127e+03  .21912695e-03", 
" -.23994645e+01 -.16745111e+01 -.63165212e+01  .11615971e+03  .21640233e-03", 
" -.23902521e+01 -.16545794e+01 -.63257738e+01  .11714056e+03  .21834538e-03", 
" -.23810059e+01 -.16403114e+01 -.61661752e+01  .11668037e+03  .21759916e-03", 
" -.23717373e+01 -.16200936e+01 -.61216295e+01  .11440864e+03  .21849458e-03", 
" -.23624871e+01 -.15977218e+01 -.63483947e+01  .11886326e+03  .21949016e-03", 
" -.23531425e+01 -.15988997e+01 -.52532089e+01  .10349218e+03  .21800936e-03", 
" -.23438679e+01 -.15723491e+01 -.55937457e+01  .10745716e+03  .21881845e-03", 
" -.23345434e+01 -.15474016e+01 -.59960242e+01  .11406923e+03  .21922317e-03", 
" -.23251959e+01 -.15200334e+01 -.65343803e+01  .12268176e+03  .21985504e-03", 
" -.23157838e+01 -.14985448e+01 -.68135857e+01  .12721315e+03  .22008344e-03", 
" -.23062627e+01 -.14955302e+01 -.60855338e+01  .11676039e+03  .21874358e-03", 
" -.22967838e+01 -.14849325e+01 -.55906882e+01  .10899120e+03  .21884658e-03", 
" -.22872493e+01 -.14741480e+01 -.54332224e+01  .10975264e+03  .21781664e-03", 
" -.22776945e+01 -.14509666e+01 -.60471195e+01  .12207705e+03  .22059129e-03", 
" -.22680361e+01 -.14445724e+01 -.55217028e+01  .11546117e+03  .21768251e-03", 
" -.22583675e+01 -.14245962e+01 -.58083763e+01  .12017550e+03  .21599119e-03", 
" -.22486490e+01 -.14006780e+01 -.64928075e+01  .13284687e+03  .21698800e-03", 
" -.22389132e+01 -.13760561e+01 -.71488347e+01  .14392384e+03  .21790370e-03", 
" -.22290718e+01 -.13638183e+01 -.71365953e+01  .14542942e+03  .21883889e-03", 
" -.22191568e+01 -.13494893e+01 -.72558224e+01  .14772392e+03  .21824334e-03", 
" -.22092550e+01 -.13329701e+01 -.72321915e+01  .14533590e+03  .21708221e-03", 
" -.21992155e+01 -.13294203e+01 -.67062864e+01  .13879586e+03  .21702293e-03", 
" -.21891521e+01 -.13160683e+01 -.67344004e+01  .13796025e+03  .21831080e-03", 
" -.21789951e+01 -.13106972e+01 -.65211889e+01  .13851385e+03  .21793065e-03", 
" -.21687924e+01 -.12902570e+01 -.68686232e+01  .14173790e+03  .21509913e-03", 
" -.21584910e+01 -.12882435e+01 -.63541106e+01  .13388864e+03  .21817031e-03", 
" -.21481302e+01 -.12699844e+01 -.68244791e+01  .14169485e+03  .21828861e-03", 
" -.21376570e+01 -.12658717e+01 -.65833982e+01  .13976962e+03  .21947719e-03", 
" -.21271445e+01 -.12439252e+01 -.72804600e+01  .15038533e+03  .22086136e-03", 
" -.21165089e+01 -.12337727e+01 -.73465977e+01  .15325460e+03  .22029381e-03", 
" -.21058313e+01 -.12132668e+01 -.80385149e+01  .16402236e+03  .22018228e-03", 
" -.20950445e+01 -.11956062e+01 -.86183686e+01  .17482292e+03  .22165372e-03", 
" -.20840926e+01 -.11982533e+01 -.80392268e+01  .16773704e+03  .21901344e-03", 
" -.20730806e+01 -.11807371e+01 -.87701616e+01  .18154334e+03  .21952056e-03", 
" -.20618842e+01 -.11882982e+01 -.77903904e+01  .16515707e+03  .22224134e-03", 
" -.20506127e+01 -.11839989e+01 -.76246776e+01  .16278393e+03  .22342549e-03", 
" -.20392268e+01 -.11778747e+01 -.75674342e+01  .16244339e+03  .22604741e-03", 
" -.20276220e+01 -.12027346e+01 -.55161158e+01  .12891884e+03  .22521803e-03", 
" -.20159300e+01 -.12141741e+01 -.43236814e+01  .10826005e+03  .22403152e-03", 
" -.20041312e+01 -.12164260e+01 -.37637404e+01  .99064965e+02  .22475271e-03", 
" -.19921179e+01 -.12272499e+01 -.27558490e+01  .82174986e+02  .22363987e-03", 
" -.19799477e+01 -.12512220e+01 -.89460753e+00  .49984558e+02  .22394485e-03", 
" -.19676089e+01 -.12498518e+01 -.98212938e+00  .53513660e+02  .22390338e-03", 
" -.19551581e+01 -.12352845e+01 -.17846514e+01  .68008904e+02  .22849549e-03", 
" -.19425850e+01 -.12265361e+01 -.20949264e+01  .73999409e+02  .23124778e-03", 
" -.19298006e+01 -.12042562e+01 -.32192074e+01  .91443850e+02  .23589232e-03", 
" -.19166937e+01 -.11995236e+01 -.37184438e+01  .10193589e+03  .23639916e-03", 
" -.19034871e+01 -.11797106e+01 -.47051213e+01  .11585968e+03  .23706022e-03", 
" -.18899269e+01 -.11849281e+01 -.45367846e+01  .11471025e+03  .23671362e-03", 
" -.18760827e+01 -.11962615e+01 -.40595107e+01  .10791875e+03  .24000529e-03", 
" -.18619955e+01 -.12168110e+01 -.29303678e+01  .90727137e+02  .24466902e-03", 
" -.18476533e+01 -.12125969e+01 -.35742961e+01  .10233471e+03  .25127283e-03", 
" -.18330507e+01 -.12179765e+01 -.34861425e+01  .10087405e+03  .25356544e-03", 
" -.18180683e+01 -.12291222e+01 -.32749582e+01  .10079825e+03  .25911046e-03", 
" -.18027008e+01 -.12537339e+01 -.24204026e+01  .91527496e+02  .26203682e-03", 
" -.17869522e+01 -.12878785e+01 -.95662824e+00  .70475020e+02  .26468861e-03", 
" -.17709229e+01 -.12984028e+01 -.59347848e+00  .62042954e+02  .27052651e-03", 
" -.17542966e+01 -.13310205e+01  .19690113e+00  .52738101e+02  .27207257e-03", 
" -.17373673e+01 -.13476237e+01  .49553349e+00  .46638992e+02  .27579139e-03", 
" -.17199698e+01 -.13665030e+01  .54483639e+00  .48964769e+02  .27462772e-03", 
" -.17019794e+01 -.13859700e+01  .63865817e+00  .49443920e+02  .27497208e-03", 
" -.16834155e+01 -.13940046e+01 -.37174639e+00  .68761357e+02  .28143042e-03", 
" -.16640590e+01 -.14556099e+01  .16166048e+01  .42091509e+02  .28402836e-03", 
" -.16441930e+01 -.14722599e+01  .12344750e+01  .49689841e+02  .28674783e-03", 
" -.16235195e+01 -.14941501e+01  .84924392e+00  .57692349e+02  .28870285e-03", 
" -.16018932e+01 -.15580995e+01  .29023670e+01  .23352848e+02  .29712093e-03", 
" -.15794699e+01 -.16032614e+01  .38344022e+01  .75848322e+01  .30293231e-03", 
" -.15560115e+01 -.16363236e+01  .38152553e+01  .59426232e+01  .30719306e-03", 
" -.15312409e+01 -.17158371e+01  .59768495e+01 -.24998542e+02  .31937812e-03", 
" -.15052476e+01 -.17472288e+01  .43792531e+01  .91845089e+01  .32759876e-03", 
" -.14777640e+01 -.17887419e+01  .30865782e+01  .39959908e+02  .34157030e-03", 
" -.14484935e+01 -.18733445e+01  .45088365e+01  .19704351e+02  .35393756e-03", 
" -.14172603e+01 -.19532388e+01  .48261727e+01  .20694857e+02  .36047410e-03", 
" -.13836835e+01 -.20880347e+01  .80557433e+01 -.23238044e+02  .37062606e-03", 
" -.13474488e+01 -.22032008e+01  .88197753e+01 -.22225484e+02  .38798469e-03", 
" -.13081868e+01 -.23062293e+01  .93422216e+01 -.31574899e+02  .40862354e-03", 
" -.12648227e+01 -.24955926e+01  .13336526e+02 -.86684358e+02  .42772348e-03", 
" -.12170852e+01 -.26098679e+01  .12030038e+02 -.63738768e+02  .46787224e-03", 
" -.11631819e+01 -.27872008e+01  .13011772e+02 -.82809324e+02  .49959118e-03", 
" -.11017133e+01 -.29939014e+01  .12917279e+02 -.67594142e+02  .52994402e-03", 
" -.10302368e+01 -.32615262e+01  .14691499e+02 -.85703339e+02  .59393132e-03", 
" -.94449039e+00 -.34161352e+01  .49397322e+01  .77805729e+02  .62930444e-03", 
" -.83623964e+00 -.39356407e+01  .12617741e+02 -.65578151e+02  .67932800e-03", 
" -.68830312e+00 -.46475175e+01  .16054855e+02 -.10599547e+03  .83781960e-03", 
" -.65048642e+00 -.49452054e+01  .23054693e+02 -.20396347e+03  .88030906e-03", 
" -.60869514e+00 -.51605801e+01  .21779856e+02 -.16474954e+03  .90240607e-03", 
" -.56223280e+00 -.54656634e+01  .28190978e+02 -.27959818e+03  .94075847e-03", 
" -.50913892e+00 -.58695358e+01  .37372984e+02 -.42471697e+03  .98071081e-03", 
" -.44780803e+00 -.61685878e+01  .36348948e+02 -.40650097e+03  .10451765e-02", 
" -.37244070e+00 -.68441670e+01  .49752217e+02 -.58734195e+03  .11342935e-02", 
" -.27933405e+00 -.72952911e+01  .43486500e+02 -.47108931e+03  .12467850e-02", 
" -.15254639e+00 -.79501060e+01  .31374850e+02 -.19922583e+03  .14952388e-02", 
"  .54330368e-01 -.92674199e+01  .35395823e+02 -.30288484e+03  .20045891e-02", 
"  .25203289e+00 -.10710927e+02  .35283377e+02 -.23239752e+03  .27508185e-02", 
"  .50093323e+00 -.12187949e+02  .41729847e+01  .17536016e+03  .40502268e-02", 
"  .68041949e+00 -.14624606e+02  .60290511e+02 -.70276040e+03  .51218805e-02", 
"coctt     1  5  3   20", " -.58354994e+01 -.43164144e+02 -.20726044e+03 -.68680039e+03  .47910489e-02", 
" -.56890784e+01 -.38017951e+02 -.26316538e+03  .70485019e+03  .36640354e-02", 
" -.54828539e+01 -.33198620e+02 -.22222684e+03  .71456892e+03  .23410532e-02", 
" -.53168387e+01 -.30374320e+02 -.16765041e+03  .42395050e+03  .17770476e-02", 
" -.51408110e+01 -.27452920e+02 -.12664718e+03  .28840691e+03  .13762685e-02", 
" -.50336397e+01 -.25686192e+02 -.10416191e+03  .16988680e+03  .11033416e-02", 
" -.49553036e+01 -.24561695e+02 -.81781860e+02 -.38140483e+02  .97569998e-03", 
" -.48933521e+01 -.23661147e+02 -.68859425e+02 -.12115059e+03  .87871607e-03", 
" -.48417065e+01 -.22846020e+02 -.62691561e+02 -.13233108e+03  .81960488e-03", 
" -.47980034e+01 -.22033851e+02 -.64930542e+02 -.26442930e+02  .76915988e-03", 
" -.47589474e+01 -.21433481e+02 -.61851423e+02 -.11375690e+02  .69026926e-03", 
" -.47241503e+01 -.20906862e+02 -.58147577e+02 -.23119164e+02  .67598816e-03", 
" -.46926344e+01 -.20451402e+02 -.53924429e+02 -.39067007e+02  .64521638e-03", 
" -.45669958e+01 -.18653039e+02 -.45179789e+02 -.25024209e+02  .56907687e-03", 
" -.44743092e+01 -.17421410e+02 -.37516698e+02 -.43727276e+02  .52210587e-03", 
" -.44001450e+01 -.16484638e+02 -.31665146e+02 -.47163352e+02  .48511006e-03", 
" -.43381014e+01 -.15672225e+02 -.28021606e+02 -.55400928e+02  .44623138e-03", 
" -.42839686e+01 -.15007661e+02 -.25671944e+02 -.41333006e+02  .43543829e-03", 
" -.42361171e+01 -.14404166e+02 -.25196702e+02 -.13665287e+02  .40564303e-03", 
" -.41932874e+01 -.13847823e+02 -.24978689e+02  .27674691e+01  .39558732e-03", 
" -.41538632e+01 -.13391016e+02 -.23971839e+02  .98798763e+01  .37492498e-03", 
" -.41174719e+01 -.12988565e+02 -.23030595e+02  .21880114e+02  .35202206e-03", 
" -.40836912e+01 -.12655367e+02 -.19752042e+02 -.56354040e+01  .34528754e-03", 
" -.40523086e+01 -.12287525e+02 -.20323875e+02  .19962505e+02  .34200866e-03", 
" -.40229557e+01 -.11923227e+02 -.21991864e+02  .55105433e+02  .33192888e-03", 
" -.39950281e+01 -.11640275e+02 -.20432952e+02  .44855085e+02  .32160091e-03", 
" -.39684473e+01 -.11379538e+02 -.19485917e+02  .47068863e+02  .31003531e-03", 
" -.39431291e+01 -.11143942e+02 -.17752707e+02  .34934048e+02  .30530802e-03", 
" -.39190271e+01 -.10897502e+02 -.17703980e+02  .46078212e+02  .29672727e-03", 
" -.38959208e+01 -.10664815e+02 -.17887188e+02  .58328932e+02  .29100234e-03", 
" -.38737542e+01 -.10436746e+02 -.18457389e+02  .75350672e+02  .28770882e-03", 
" -.38523910e+01 -.10241907e+02 -.17676688e+02  .75495732e+02  .27672731e-03", 
" -.38318156e+01 -.10045908e+02 -.17104424e+02  .71603388e+02  .27311494e-03", 
" -.38120039e+01 -.98466583e+01 -.17057648e+02  .74113968e+02  .26849388e-03", 
" -.37926273e+01 -.96948300e+01 -.15408686e+02  .59078697e+02  .26937738e-03", 
" -.37738826e+01 -.95239098e+01 -.15545075e+02  .69742074e+02  .26593361e-03", 
" -.37556824e+01 -.93740952e+01 -.14686074e+02  .64157330e+02  .25902096e-03", 
" -.37380097e+01 -.92237353e+01 -.14172498e+02  .62211048e+02  .25953413e-03", 
" -.37208829e+01 -.90702531e+01 -.14049622e+02  .66605746e+02  .25736897e-03", 
" -.37040917e+01 -.89350533e+01 -.13349515e+02  .62420976e+02  .25264694e-03", 
" -.36877165e+01 -.88007073e+01 -.12776601e+02  .58725495e+02  .25202340e-03", 
" -.36716933e+01 -.86802841e+01 -.11706724e+02  .48846117e+02  .24754482e-03", 
" -.36560883e+01 -.85450904e+01 -.12019005e+02  .58684698e+02  .24568463e-03", 
" -.36408583e+01 -.83982546e+01 -.13105026e+02  .78542053e+02  .24718027e-03", 
" -.36258068e+01 -.82911698e+01 -.12234430e+02  .70828905e+02  .24359445e-03", 
" -.36110296e+01 -.81929826e+01 -.11219311e+02  .62117251e+02  .24367966e-03", 
" -.35966585e+01 -.80688224e+01 -.11529375e+02  .68486071e+02  .24398530e-03", 
" -.35824563e+01 -.79661865e+01 -.10963149e+02  .64051280e+02  .23797254e-03", 
" -.35685180e+01 -.78740306e+01 -.10208140e+02  .58839859e+02  .23510189e-03", 
" -.35549096e+01 -.77571304e+01 -.10613334e+02  .67519059e+02  .23226838e-03", 
" -.35414347e+01 -.76569685e+01 -.10621539e+02  .72367474e+02  .23112300e-03", 
" -.35281835e+01 -.75694220e+01 -.99606008e+01  .67675259e+02  .23144795e-03", 
" -.35152003e+01 -.74712831e+01 -.98570628e+01  .69435154e+02  .23010973e-03", 
" -.35023313e+01 -.73847886e+01 -.95909404e+01  .70622230e+02  .22909865e-03", 
" -.34896568e+01 -.72989595e+01 -.92899094e+01  .70339628e+02  .22679125e-03", 
" -.34771922e+01 -.72138751e+01 -.88973649e+01  .67787793e+02  .22462976e-03", 
" -.34648963e+01 -.71293969e+01 -.87299845e+01  .69739058e+02  .22393901e-03", 
" -.34526906e+01 -.70589560e+01 -.80342054e+01  .64466515e+02  .22172816e-03", 
" -.34406432e+01 -.69901708e+01 -.72563230e+01  .56683155e+02  .22647322e-03", 
" -.34288143e+01 -.69047205e+01 -.73769292e+01  .60600706e+02  .22741780e-03", 
" -.34171488e+01 -.68188363e+01 -.75662154e+01  .64592773e+02  .22841733e-03", 
" -.34055873e+01 -.67425238e+01 -.72547035e+01  .60967008e+02  .22452924e-03", 
" -.33942092e+01 -.66501923e+01 -.80483718e+01  .73918724e+02  .22101262e-03", 
" -.33827798e+01 -.65991569e+01 -.67407979e+01  .57441180e+02  .22035067e-03", 
" -.33715699e+01 -.65282715e+01 -.67420744e+01  .61839003e+02  .21975210e-03", 
" -.33604777e+01 -.64518901e+01 -.69454277e+01  .65468905e+02  .21682001e-03", 
" -.33494643e+01 -.63959260e+01 -.62612594e+01  .59387742e+02  .21690647e-03", 
" -.33385665e+01 -.63379883e+01 -.56638633e+01  .53368951e+02  .21801260e-03", 
" -.33277523e+01 -.62677963e+01 -.57603144e+01  .56393611e+02  .21490603e-03", 
" -.33170537e+01 -.61982380e+01 -.59931702e+01  .61354466e+02  .21524841e-03", 
" -.33064601e+01 -.61348908e+01 -.59124584e+01  .62019624e+02  .21273633e-03", 
" -.32959349e+01 -.60763506e+01 -.57212264e+01  .62769480e+02  .21310490e-03", 
" -.32854685e+01 -.60319368e+01 -.48159221e+01  .53778816e+02  .21390867e-03", 
" -.32751382e+01 -.59717007e+01 -.46514358e+01  .52849475e+02  .21235398e-03", 
" -.32648541e+01 -.59115334e+01 -.46415972e+01  .54385085e+02  .21292596e-03", 
" -.32545844e+01 -.58727710e+01 -.35807412e+01  .42349579e+02  .21446785e-03", 
" -.32444416e+01 -.58195862e+01 -.33081099e+01  .40331330e+02  .21713798e-03", 
" -.32343826e+01 -.57621663e+01 -.33957598e+01  .43908692e+02  .21412610e-03", 
" -.32243869e+01 -.57110010e+01 -.31041561e+01  .42067112e+02  .21532906e-03", 
" -.32144107e+01 -.56714398e+01 -.21897988e+01  .30472390e+02  .21352256e-03", 
" -.32045693e+01 -.56127474e+01 -.23288661e+01  .33378799e+02  .21338489e-03", 
" -.31947813e+01 -.55567152e+01 -.23139600e+01  .33210243e+02  .21032696e-03", 
" -.31850720e+01 -.54936113e+01 -.28437109e+01  .41733240e+02  .20790115e-03", 
" -.31753887e+01 -.54415578e+01 -.28448684e+01  .43583342e+02  .20657095e-03", 
" -.31657608e+01 -.53884439e+01 -.29176608e+01  .45803900e+02  .20774168e-03", 
" -.31561983e+01 -.53309216e+01 -.32504984e+01  .51266788e+02  .20482236e-03", 
" -.31466504e+01 -.52764554e+01 -.35109380e+01  .55591487e+02  .20312001e-03", 
" -.31371249e+01 -.52406556e+01 -.29240960e+01  .50180727e+02  .20231095e-03", 
" -.31276882e+01 -.51900302e+01 -.29700801e+01  .50826132e+02  .20049868e-03", 
" -.31183240e+01 -.51301312e+01 -.36130537e+01  .60336434e+02  .19996082e-03", 
" -.31089702e+01 -.50843908e+01 -.35514084e+01  .60830639e+02  .19988769e-03", 
" -.30996230e+01 -.50430383e+01 -.32712473e+01  .57894518e+02  .19971536e-03", 
" -.30903527e+01 -.49922497e+01 -.35370307e+01  .62664119e+02  .19795790e-03", 
" -.30811050e+01 -.49521599e+01 -.31458337e+01  .57287183e+02  .19783458e-03", 
" -.30718455e+01 -.49112025e+01 -.30728831e+01  .57753663e+02  .19756662e-03", 
" -.30626481e+01 -.48695345e+01 -.29042088e+01  .55503634e+02  .19607186e-03", 
" -.30534315e+01 -.48365203e+01 -.25282230e+01  .52813054e+02  .19787361e-03", 
" -.30442932e+01 -.47944168e+01 -.25138065e+01  .53801304e+02  .19690342e-03", 
" -.30351897e+01 -.47490239e+01 -.26555093e+01  .56257998e+02  .19768465e-03", 
" -.30260994e+01 -.47092077e+01 -.25196532e+01  .54511869e+02  .19782766e-03", 
" -.30170376e+01 -.46657411e+01 -.26553577e+01  .57114778e+02  .19848421e-03", 
" -.30080159e+01 -.46282820e+01 -.24453394e+01  .55111944e+02  .19745918e-03", 
" -.29989842e+01 -.45878225e+01 -.25116708e+01  .56553271e+02  .19759422e-03", 
" -.29899483e+01 -.45536613e+01 -.23179527e+01  .55560342e+02  .19752179e-03", 
" -.29809054e+01 -.45219566e+01 -.19928198e+01  .51445312e+02  .19736788e-03", 
" -.29719497e+01 -.44760176e+01 -.22979640e+01  .55822452e+02  .19625286e-03", 
" -.29629985e+01 -.44368520e+01 -.23849640e+01  .57701118e+02  .19444712e-03", 
" -.29540267e+01 -.44060040e+01 -.20786222e+01  .54789154e+02  .19491362e-03", 
" -.29451073e+01 -.43618829e+01 -.25061653e+01  .61832701e+02  .19338092e-03", 
" -.29361146e+01 -.43391485e+01 -.18738982e+01  .54498063e+02  .19290052e-03", 
" -.29272040e+01 -.43050246e+01 -.17292538e+01  .52350763e+02  .19347974e-03", 
" -.29182172e+01 -.42819348e+01 -.12915405e+01  .49057094e+02  .19300454e-03", 
" -.29092956e+01 -.42432827e+01 -.15423645e+01  .53042436e+02  .19358219e-03", 
" -.29003561e+01 -.42158381e+01 -.13089719e+01  .51262708e+02  .19489368e-03", 
" -.28914091e+01 -.41855789e+01 -.12352591e+01  .51714119e+02  .19331956e-03", 
" -.28824765e+01 -.41624783e+01 -.61758714e+00  .43856703e+02  .19396808e-03", 
" -.28735873e+01 -.41306926e+01 -.52227622e+00  .42533448e+02  .19206388e-03", 
" -.28646864e+01 -.41012876e+01 -.41708650e+00  .42244900e+02  .18992815e-03", 
" -.28557024e+01 -.40777729e+01 -.10037415e+00  .39819561e+02  .19040061e-03", 
" -.28467985e+01 -.40468789e+01  .81890472e-01  .36514708e+02  .18865573e-03", 
" -.28378646e+01 -.40278389e+01  .82322257e+00  .26763624e+02  .18929181e-03", 
" -.28288977e+01 -.39985638e+01  .88369539e+00  .25864657e+02  .18974740e-03", 
" -.28199456e+01 -.39642546e+01  .54818844e+00  .31814218e+02  .18979936e-03", 
" -.28109501e+01 -.39350973e+01  .45778135e+00  .34631236e+02  .19004317e-03", 
" -.28019803e+01 -.39018315e+01  .30305195e+00  .36919951e+02  .18887231e-03", 
" -.27929887e+01 -.38724026e+01  .27520701e+00  .37651346e+02  .18776342e-03", 
" -.27839786e+01 -.38455898e+01  .27522068e+00  .39011561e+02  .18821754e-03", 
" -.27749417e+01 -.38226003e+01  .52406381e+00  .36080901e+02  .18887378e-03", 
" -.27658670e+01 -.38019073e+01  .95786198e+00  .30037697e+02  .18797504e-03", 
" -.27568321e+01 -.37648244e+01  .44448851e+00  .37162512e+02  .18766969e-03", 
" -.27476867e+01 -.37516491e+01  .11359541e+01  .28898715e+02  .18544270e-03", 
" -.27385715e+01 -.37362853e+01  .17058844e+01  .22254483e+02  .18511675e-03", 
" -.27293965e+01 -.37170347e+01  .20173271e+01  .18483184e+02  .18561480e-03", 
" -.27202064e+01 -.36934203e+01  .21149017e+01  .17758804e+02  .18601134e-03", 
" -.27109656e+01 -.36780403e+01  .26178972e+01  .11483409e+02  .18274152e-03", 
" -.27016552e+01 -.36612823e+01  .27341992e+01  .12888528e+02  .18345626e-03", 
" -.26923583e+01 -.36389266e+01  .26886851e+01  .14961973e+02  .18297532e-03", 
" -.26830914e+01 -.35968272e+01  .15639210e+01  .31383462e+02  .18258227e-03", 
" -.26737338e+01 -.35778082e+01  .16793421e+01  .31521797e+02  .18481079e-03", 
" -.26643524e+01 -.35555078e+01  .18078343e+01  .29312382e+02  .19092480e-03", 
" -.26549330e+01 -.35350698e+01  .18736248e+01  .30140565e+02  .18992112e-03", 
" -.26454562e+01 -.35103245e+01  .15769794e+01  .35665281e+02  .18958276e-03", 
" -.26359327e+01 -.34952930e+01  .18572816e+01  .33042720e+02  .19105131e-03", 
" -.26263460e+01 -.34796015e+01  .20446431e+01  .31608591e+02  .18972556e-03", 
" -.26166893e+01 -.34699859e+01  .24625906e+01  .27630466e+02  .18823439e-03", 
" -.26069798e+01 -.34598395e+01  .29224678e+01  .22870528e+02  .18719160e-03", 
" -.25972415e+01 -.34361735e+01  .26396119e+01  .27728177e+02  .18924989e-03", 
" -.25873659e+01 -.34306568e+01  .33131709e+01  .19411960e+02  .18940837e-03", 
" -.25775269e+01 -.34185915e+01  .36504632e+01  .16605182e+02  .18951606e-03", 
" -.25675777e+01 -.34072352e+01  .39299602e+01  .14072551e+02  .19011424e-03", 
" -.25575804e+01 -.33852651e+01  .35607819e+01  .20305807e+02  .18895329e-03", 
" -.25475278e+01 -.33720080e+01  .37288907e+01  .19747780e+02  .19018592e-03", 
" -.25373925e+01 -.33545745e+01  .36332137e+01  .21822527e+02  .19330870e-03", 
" -.25270946e+01 -.33592786e+01  .45338319e+01  .12657501e+02  .19246736e-03", 
" -.25167948e+01 -.33419289e+01  .44335073e+01  .14538361e+02  .19273005e-03", 
" -.25063842e+01 -.33367914e+01  .49538933e+01  .89133059e+01  .19379827e-03", 
" -.24959057e+01 -.33281441e+01  .53011833e+01  .48736570e+01  .19592679e-03", 
" -.24852689e+01 -.33243830e+01  .55181963e+01  .49202604e+01  .19734207e-03", 
" -.24745718e+01 -.33177067e+01  .58020541e+01  .28827651e+01  .19714660e-03", 
" -.24637882e+01 -.33076133e+01  .59077624e+01  .22777301e+01  .19710862e-03", 
" -.24529228e+01 -.32943770e+01  .57926796e+01  .55252190e+01  .19427542e-03", 
" -.24418907e+01 -.32864967e+01  .57948091e+01  .70955015e+01  .19716381e-03", 
" -.24307914e+01 -.32813898e+01  .61500709e+01  .28525236e+01  .19786672e-03", 
" -.24195564e+01 -.32662769e+01  .57856634e+01  .98243839e+01  .20295672e-03", 
" -.24081286e+01 -.32716534e+01  .64856093e+01  .14601146e+01  .20280869e-03", 
" -.23965934e+01 -.32729437e+01  .70375300e+01 -.56740888e+01  .20110020e-03", 
" -.23850079e+01 -.32549631e+01  .65551408e+01  .17581541e+01  .20032130e-03", 
" -.23731985e+01 -.32429907e+01  .61395868e+01  .90710683e+01  .20136699e-03", 
" -.23612602e+01 -.32259857e+01  .54335700e+01  .20542969e+02  .20130270e-03", 
" -.23490635e+01 -.32423174e+01  .64453701e+01  .90384467e+01  .20243229e-03", 
" -.23367251e+01 -.32414422e+01  .65548880e+01  .94393844e+01  .20455809e-03", 
" -.23241615e+01 -.32411766e+01  .65226642e+01  .11905391e+02  .20796077e-03", 
" -.23113501e+01 -.32572175e+01  .72004783e+01  .60441277e+01  .21127992e-03", 
" -.22983875e+01 -.32654096e+01  .74718621e+01  .50748277e+01  .21306836e-03", 
" -.22851489e+01 -.32919348e+01  .86583836e+01 -.73296080e+01  .21632351e-03", 
" -.22717760e+01 -.32910806e+01  .85177772e+01 -.37528117e+01  .21802983e-03", 
" -.22580597e+01 -.33091623e+01  .89842938e+01 -.63509383e+01  .21746174e-03", 
" -.22440816e+01 -.33251896e+01  .93188296e+01 -.72825892e+01  .21973650e-03", 
" -.22298935e+01 -.33461299e+01  .99675266e+01 -.12688255e+02  .22266670e-03", 
" -.22154055e+01 -.33577406e+01  .10083982e+02 -.11399067e+02  .22546857e-03", 
" -.22006730e+01 -.33671721e+01  .10182197e+02 -.10820382e+02  .22785684e-03", 
" -.21855253e+01 -.33749700e+01  .98536107e+01 -.33783985e+01  .22812777e-03", 
" -.21699347e+01 -.33965405e+01  .99780057e+01 -.55348628e+00  .23061216e-03", 
" -.21539129e+01 -.34281388e+01  .10426718e+02 -.15094558e+01  .23030792e-03", 
" -.21375489e+01 -.34425851e+01  .10033661e+02  .77347867e+01  .23420451e-03", 
" -.21207093e+01 -.34725927e+01  .10417490e+02  .54942039e+01  .23654236e-03", 
" -.21032822e+01 -.35143400e+01  .11020523e+02  .18177647e+01  .24037494e-03", 
" -.20853755e+01 -.35639021e+01  .11742445e+02 -.48567205e+00  .24392063e-03", 
" -.20669503e+01 -.35944747e+01  .11638313e+02  .50120141e+01  .24584528e-03", 
" -.20478469e+01 -.36104585e+01  .10370080e+02  .27359479e+02  .24682067e-03", 
" -.20278353e+01 -.36920929e+01  .12014868e+02  .13728020e+02  .25048255e-03", 
" -.20070685e+01 -.37721334e+01  .13619118e+02 -.10484197e+01  .25392626e-03", 
" -.19854791e+01 -.38394363e+01  .14486786e+02 -.82893044e+01  .26119813e-03", 
" -.19629680e+01 -.38915835e+01  .14226188e+02  .14782530e+01  .26345781e-03", 
" -.19390950e+01 -.39981247e+01  .15857538e+02 -.11011174e+02  .26507147e-03", 
" -.19141928e+01 -.41022069e+01  .17428100e+02 -.23759965e+02  .27658220e-03", 
" -.18876870e+01 -.42183510e+01  .18937724e+02 -.35060674e+02  .27708828e-03", 
" -.18595175e+01 -.43648739e+01  .21148249e+02 -.53372109e+02  .28226718e-03", 
" -.18294407e+01 -.45156752e+01  .22778580e+02 -.62409183e+02  .28414196e-03", 
" -.17969386e+01 -.47260198e+01  .26602738e+02 -.98271259e+02  .29799603e-03", 
" -.17616368e+01 -.49870510e+01  .31926638e+02 -.15284366e+03  .31622713e-03", 
" -.17232076e+01 -.52154109e+01  .33482867e+02 -.14898767e+03  .33189283e-03", 
" -.16808058e+01 -.55018807e+01  .37413385e+02 -.18084221e+03  .35509704e-03", 
" -.16334714e+01 -.57869395e+01  .39262908e+02 -.18233532e+03  .36593280e-03", 
" -.15792783e+01 -.61562362e+01  .42181286e+02 -.19442404e+03  .38657470e-03", 
" -.15155150e+01 -.66611835e+01  .46766633e+02 -.21440968e+03  .42736280e-03", 
" -.14385435e+01 -.73194841e+01  .53376131e+02 -.25404405e+03  .46752176e-03", 
" -.13403867e+01 -.83210620e+01  .65892263e+02 -.33860157e+03  .51956369e-03", 
" -.12057063e+01 -.96631077e+01  .78757443e+02 -.41252165e+03  .63934974e-03", 
" -.11713081e+01 -.10058804e+02  .84426299e+02 -.46243733e+03  .66208963e-03", 
" -.11334760e+01 -.10408660e+02  .84945976e+02 -.43861322e+03  .67436558e-03", 
" -.10907036e+01 -.10869516e+02  .88428409e+02 -.44689940e+03  .73385927e-03", 
" -.10420678e+01 -.11367103e+02  .92142114e+02 -.46743140e+03  .74942188e-03", 
" -.98517502e+00 -.12003129e+02  .96940691e+02 -.48323875e+03  .80105982e-03", 
" -.91656730e+00 -.12789582e+02  .10261438e+03 -.50042588e+03  .89248209e-03", 
" -.83048675e+00 -.13706816e+02  .10472028e+03 -.45489297e+03  .10120099e-02", 
" -.71268135e+00 -.15003901e+02  .10715750e+03 -.38901933e+03  .12244657e-02", 
" -.51720138e+00 -.17682839e+02  .12868096e+03 -.48949011e+03  .16148630e-02", 
" -.33213624e+00 -.20022064e+02  .13153342e+03 -.36070737e+03  .22125595e-02", 
" -.98314576e-01 -.23662816e+02  .15371969e+03 -.38017178e+03  .33011494e-02", 
"  .71795704e-01 -.26556379e+02  .17143742e+03 -.38078365e+03  .44043305e-02", 
"coanc     1  2  3   20", " -.42572121e+02  .52200582e+03 -.47094919e+04  .22867930e+05  .81241944e-01", 
" -.39570204e+02  .43943160e+03 -.31476846e+04  .94457758e+04  .56359009e-01", 
" -.35752964e+02  .37228074e+03 -.28083186e+04  .11072799e+05  .36705875e-01", 
" -.32833406e+02  .31851791e+03 -.22715356e+04  .86200975e+04  .25937875e-01", 
" -.29888381e+02  .26655175e+03 -.17072743e+04  .53335779e+04  .17801188e-01", 
" -.28156585e+02  .24068657e+03 -.15682620e+04  .58345786e+04  .14475305e-01", 
" -.26920444e+02  .22162761e+03 -.13869382e+04  .47722727e+04  .12326142e-01", 
" -.25958147e+02  .20679382e+03 -.12476392e+04  .41857575e+04  .11333537e-01", 
" -.25167761e+02  .19499362e+03 -.11397411e+04  .36970624e+04  .10534432e-01", 
" -.24506996e+02  .18668159e+03 -.10996601e+04  .37653152e+04  .98599891e-02", 
" -.23927119e+02  .17821421e+03 -.10073261e+04  .31047601e+04  .93213876e-02", 
" -.23415280e+02  .17108128e+03 -.94010873e+03  .27116824e+04  .89270264e-02", 
" -.22954712e+02  .16458222e+03 -.87999203e+03  .24399306e+04  .82565294e-02", 
" -.21194375e+02  .14429487e+03 -.81168159e+03  .30554401e+04  .71214793e-02", 
" -.19932963e+02  .12955593e+03 -.71570015e+03  .27948227e+04  .62652872e-02", 
" -.18949066e+02  .11915565e+03 -.66907086e+03  .28375884e+04  .54839163e-02", 
" -.18138252e+02  .11029807e+03 -.60792747e+03  .26021298e+04  .52372847e-02", 
" -.17449758e+02  .10273405e+03 -.54174974e+03  .21791823e+04  .49021107e-02", 
" -.16853996e+02  .96704480e+02 -.50109618e+03  .19840560e+04  .45582928e-02", 
" -.16324836e+02  .91177144e+02 -.45468499e+03  .17204175e+04  .42340178e-02", 
" -.15850471e+02  .86180496e+02 -.40400720e+03  .13426707e+04  .39681957e-02", 
" -.15420906e+02  .82112633e+02 -.37454441e+03  .11732425e+04  .38159876e-02", 
" -.15027367e+02  .78646171e+02 -.35923690e+03  .11502299e+04  .36916303e-02", 
" -.14663440e+02  .75274920e+02 -.33542237e+03  .10269090e+04  .35567606e-02", 
" -.14328060e+02  .72691420e+02 -.33371098e+03  .11368977e+04  .33912958e-02", 
" -.14014633e+02  .70400167e+02 -.33989793e+03  .13605804e+04  .32617977e-02", 
" -.13719793e+02  .68051777e+02 -.33473089e+03  .14389371e+04  .31989876e-02", 
" -.13441173e+02  .65607940e+02 -.31661815e+03  .13345329e+04  .31200113e-02", 
" -.13178300e+02  .63513674e+02 -.30930631e+03  .13616746e+04  .30142235e-02", 
" -.12928969e+02  .61458408e+02 -.29664176e+03  .13074767e+04  .29334500e-02", 
" -.12691653e+02  .59590195e+02 -.28749882e+03  .12915862e+04  .28077417e-02", 
" -.12465151e+02  .57799987e+02 -.27835638e+03  .12634476e+04  .27316264e-02", 
" -.12249203e+02  .56147028e+02 -.27065340e+03  .12509346e+04  .26257177e-02", 
" -.12041922e+02  .54569399e+02 -.26538381e+03  .12773719e+04  .26069681e-02", 
" -.11844190e+02  .53215953e+02 -.26410464e+03  .13315925e+04  .25099181e-02", 
" -.11653389e+02  .51689511e+02 -.24832128e+03  .11743418e+04  .24872758e-02", 
" -.11468582e+02  .50049996e+02 -.22708155e+03  .96788331e+03  .24206035e-02", 
" -.11291589e+02  .48859850e+02 -.22680214e+03  .10427041e+04  .23260004e-02", 
" -.11119870e+02  .47602696e+02 -.22198103e+03  .10634019e+04  .22877845e-02", 
" -.10953741e+02  .46313058e+02 -.21073469e+03  .97805074e+03  .22174638e-02", 
" -.10794242e+02  .45376520e+02 -.21316274e+03  .10469818e+04  .21839901e-02", 
" -.10638993e+02  .44340276e+02 -.21038978e+03  .10674634e+04  .21789329e-02", 
" -.10487992e+02  .43187920e+02 -.19966573e+03  .98842852e+03  .21752808e-02", 
" -.10341719e+02  .42215968e+02 -.19554506e+03  .98241577e+03  .21578069e-02", 
" -.10198728e+02  .41144780e+02 -.18597979e+03  .90694854e+03  .21409286e-02", 
" -.10059600e+02  .40079036e+02 -.17420683e+03  .79321936e+03  .21183578e-02", 
" -.99244368e+01  .39111143e+02 -.16635540e+03  .73995029e+03  .20828602e-02", 
" -.97926287e+01  .38249549e+02 -.16254034e+03  .73166004e+03  .20553245e-02", 
" -.96643644e+01  .37408551e+02 -.15848099e+03  .72420153e+03  .20438248e-02", 
" -.95387693e+01  .36589110e+02 -.15502917e+03  .73092182e+03  .20087994e-02", 
" -.94165088e+01  .35839992e+02 -.15258552e+03  .73728886e+03  .19898010e-02", 
" -.92965496e+01  .35090673e+02 -.14901044e+03  .71734539e+03  .19848808e-02", 
" -.91797983e+01  .34325623e+02 -.14228054e+03  .65819342e+03  .19421719e-02", 
" -.90649932e+01  .33594970e+02 -.13847880e+03  .64646504e+03  .19207882e-02", 
" -.89528296e+01  .32869720e+02 -.13316582e+03  .60683251e+03  .19154059e-02", 
" -.88430666e+01  .32231512e+02 -.13189779e+03  .62123182e+03  .18914989e-02", 
" -.87349313e+01  .31536211e+02 -.12764963e+03  .60431675e+03  .19063530e-02", 
" -.86296113e+01  .31001645e+02 -.12904601e+03  .64389525e+03  .18624525e-02", 
" -.85255538e+01  .30361992e+02 -.12589003e+03  .63265705e+03  .18423940e-02", 
" -.84241122e+01  .29825928e+02 -.12495015e+03  .63892526e+03  .18187729e-02", 
" -.83238028e+01  .29103698e+02 -.11510983e+03  .54114544e+03  .17900302e-02", 
" -.82260123e+01  .28592235e+02 -.11447042e+03  .55248653e+03  .17793827e-02", 
" -.81298729e+01  .28078690e+02 -.11344547e+03  .56175401e+03  .17542813e-02", 
" -.80352611e+01  .27596366e+02 -.11395397e+03  .58935606e+03  .17176689e-02", 
" -.79418420e+01  .27002874e+02 -.10745078e+03  .52467756e+03  .16986464e-02", 
" -.78498314e+01  .26440522e+02 -.10290753e+03  .49312199e+03  .16691916e-02", 
" -.77590053e+01  .25865064e+02 -.97439169e+02  .44539976e+03  .16479023e-02", 
" -.76699411e+01  .25301508e+02 -.91615687e+02  .38869202e+03  .16223311e-02", 
" -.75826483e+01  .24898575e+02 -.93060957e+02  .42742590e+03  .16061303e-02", 
" -.74965147e+01  .24419903e+02 -.90463888e+02  .41802324e+03  .15991254e-02", 
" -.74109763e+01  .23862815e+02 -.83136771e+02  .33999660e+03  .15898202e-02", 
" -.73273860e+01  .23427238e+02 -.80087790e+02  .30899363e+03  .15846719e-02", 
" -.72448905e+01  .22974714e+02 -.75846606e+02  .26177461e+03  .15628168e-02", 
" -.71629667e+01  .22490004e+02 -.71993948e+02  .23422557e+03  .15473474e-02", 
" -.70824674e+01  .22035302e+02 -.68447015e+02  .20559709e+03  .15374862e-02", 
" -.70030361e+01  .21591258e+02 -.65007250e+02  .17727423e+03  .15328289e-02", 
" -.69246966e+01  .21240700e+02 -.66269774e+02  .21306886e+03  .15210941e-02", 
" -.68474732e+01  .20886088e+02 -.66228317e+02  .22815097e+03  .15131080e-02", 
" -.67706717e+01  .20427271e+02 -.60683729e+02  .17018797e+03  .14931748e-02", 
" -.66951560e+01  .20004214e+02 -.56832994e+02  .13581949e+03  .14786733e-02", 
" -.66210955e+01  .19678108e+02 -.56300258e+02  .13751422e+03  .14759429e-02", 
" -.65474323e+01  .19273135e+02 -.52191305e+02  .96250722e+02  .14536926e-02", 
" -.64745220e+01  .18907051e+02 -.50990871e+02  .99559210e+02  .14357005e-02", 
" -.64023137e+01  .18496202e+02 -.46221216e+02  .48248491e+02  .14068207e-02", 
" -.63310712e+01  .18107184e+02 -.42380772e+02  .11288669e+02  .14014068e-02", 
" -.62607656e+01  .17776540e+02 -.41616422e+02  .17420923e+02  .13797913e-02", 
" -.61910643e+01  .17462362e+02 -.41386962e+02  .27248505e+02  .13696026e-02", 
" -.61221835e+01  .17154320e+02 -.40653734e+02  .28672111e+02  .13581290e-02", 
" -.60539765e+01  .16832303e+02 -.38787775e+02  .14142674e+02  .13333320e-02", 
" -.59864424e+01  .16521070e+02 -.37193078e+02 -.96342003e+00  .13295992e-02", 
" -.59193848e+01  .16155095e+02 -.33580918e+02 -.30144608e+02  .12907991e-02", 
" -.58532440e+01  .15863525e+02 -.32633406e+02 -.33290054e+02  .12715600e-02", 
" -.57879272e+01  .15618915e+02 -.33603250e+02 -.10019125e+02  .12384977e-02", 
" -.57230836e+01  .15330669e+02 -.32060719e+02 -.21327130e+02  .12384923e-02", 
" -.56589592e+01  .15050417e+02 -.30446068e+02 -.37505508e+02  .12286416e-02", 
" -.55952526e+01  .14784250e+02 -.30249443e+02 -.27248123e+02  .12433171e-02", 
" -.55320325e+01  .14557386e+02 -.31999420e+02  .47140904e+01  .12313630e-02", 
" -.54693641e+01  .14298865e+02 -.31388836e+02  .35508702e+01  .12306981e-02", 
" -.54073735e+01  .14057568e+02 -.31472749e+02  .13265085e+02  .12177949e-02", 
" -.53459963e+01  .13833675e+02 -.32901002e+02  .47509570e+02  .12072011e-02", 
" -.52849481e+01  .13573089e+02 -.32159945e+02  .50425913e+02  .11952227e-02", 
" -.52243687e+01  .13313712e+02 -.31121032e+02  .47752979e+02  .11753540e-02", 
" -.51642820e+01  .13069182e+02 -.30980896e+02  .55836838e+02  .11527104e-02", 
" -.51048139e+01  .12836263e+02 -.30661461e+02  .60168829e+02  .11370395e-02", 
" -.50460455e+01  .12635870e+02 -.31581151e+02  .78470095e+02  .11286709e-02", 
" -.49871102e+01  .12327631e+02 -.27512774e+02  .36313053e+02  .11222139e-02", 
" -.49289242e+01  .12086822e+02 -.25975267e+02  .22130674e+02  .11242868e-02", 
" -.48712596e+01  .11877392e+02 -.26398394e+02  .39202648e+02  .11197284e-02", 
" -.48138415e+01  .11670737e+02 -.26583468e+02  .46522169e+02  .11166903e-02", 
" -.47565526e+01  .11390620e+02 -.22831799e+02  .40196222e+01  .11177196e-02", 
" -.47000491e+01  .11180777e+02 -.22266816e+02  .80400774e+00  .11142210e-02", 
" -.46438392e+01  .10979986e+02 -.22182788e+02  .30408592e+01  .11003670e-02", 
" -.45882468e+01  .10803675e+02 -.23035640e+02  .20118624e+02  .10861089e-02", 
" -.45327567e+01  .10591218e+02 -.22660796e+02  .26183713e+02  .10876658e-02", 
" -.44778390e+01  .10413393e+02 -.22756473e+02  .27337819e+02  .10950759e-02", 
" -.44230277e+01  .10228235e+02 -.22617564e+02  .24642992e+02  .10918808e-02", 
" -.43685611e+01  .10067492e+02 -.24601070e+02  .58752282e+02  .10827143e-02", 
" -.43143676e+01  .98544023e+01 -.23158752e+02  .46338545e+02  .10784410e-02", 
" -.42606565e+01  .97317896e+01 -.26332042e+02  .93185834e+02  .10693817e-02", 
" -.42072346e+01  .95687925e+01 -.27494454e+02  .11494891e+03  .10589889e-02", 
" -.41538590e+01  .93856462e+01 -.27966357e+02  .13183841e+03  .10545978e-02", 
" -.41009795e+01  .91847350e+01 -.26162784e+02  .11038459e+03  .10456634e-02", 
" -.40481310e+01  .89506400e+01 -.22893257e+02  .71625995e+02  .10420861e-02", 
" -.39958456e+01  .87977971e+01 -.23476626e+02  .81636650e+02  .10447547e-02", 
" -.39437175e+01  .86365882e+01 -.23809109e+02  .89414854e+02  .10499576e-02", 
" -.38916043e+01  .84198250e+01 -.21741715e+02  .74443174e+02  .10454331e-02", 
" -.38400563e+01  .83016956e+01 -.24088067e+02  .10823638e+03  .10334905e-02", 
" -.37886848e+01  .81118267e+01 -.22476581e+02  .91556305e+02  .10261731e-02", 
" -.37376439e+01  .79673610e+01 -.22886639e+02  .97631922e+02  .10194587e-02", 
" -.36864394e+01  .77663276e+01 -.20908350e+02  .77751847e+02  .10207886e-02", 
" -.36357735e+01  .76410485e+01 -.22632509e+02  .10512561e+03  .10228904e-02", 
" -.35850161e+01  .74540603e+01 -.21216452e+02  .92126665e+02  .10198128e-02", 
" -.35346768e+01  .73211937e+01 -.22851125e+02  .12292628e+03  .10148139e-02", 
" -.34843079e+01  .71706133e+01 -.23571069e+02  .13988570e+03  .10173615e-02", 
" -.34340227e+01  .70008111e+01 -.23218238e+02  .14236020e+03  .10158516e-02", 
" -.33841736e+01  .68764669e+01 -.24909349e+02  .17406190e+03  .10179164e-02", 
" -.33344990e+01  .67483869e+01 -.25894967e+02  .18970487e+03  .10166414e-02", 
" -.32850205e+01  .66375247e+01 -.27708336e+02  .21765644e+03  .99316650e-03", 
" -.32356565e+01  .64745196e+01 -.25653469e+02  .18548923e+03  .99055765e-03", 
" -.31863367e+01  .62968970e+01 -.23584889e+02  .16321241e+03  .98986907e-03", 
" -.31372859e+01  .61747202e+01 -.24288795e+02  .17455173e+03  .97849059e-03", 
" -.30882394e+01  .60909418e+01 -.26727833e+02  .20461349e+03  .97592809e-03", 
" -.30392915e+01  .59772561e+01 -.27928130e+02  .22256437e+03  .95992770e-03", 
" -.29903709e+01  .58355659e+01 -.27675489e+02  .22202939e+03  .94436284e-03", 
" -.29414071e+01  .56445629e+01 -.24799085e+02  .18707476e+03  .93517240e-03", 
" -.28928215e+01  .55300584e+01 -.25233258e+02  .19411017e+03  .92834224e-03", 
" -.28441073e+01  .54037725e+01 -.25127575e+02  .19202836e+03  .91505989e-03", 
" -.27954655e+01  .52561903e+01 -.24284437e+02  .18516864e+03  .90350246e-03", 
" -.27466777e+01  .50637454e+01 -.21269047e+02  .15307210e+03  .89482093e-03", 
" -.26980103e+01  .48851035e+01 -.18537371e+02  .12008791e+03  .90197924e-03", 
" -.26494595e+01  .47595303e+01 -.19029889e+02  .13255600e+03  .90215907e-03", 
" -.26010917e+01  .46191193e+01 -.18117510e+02  .12299173e+03  .90039938e-03", 
" -.25526586e+01  .44841247e+01 -.17797701e+02  .12694191e+03  .90045268e-03", 
" -.25044016e+01  .43549059e+01 -.16910713e+02  .11500044e+03  .88961771e-03", 
" -.24558981e+01  .42084075e+01 -.15900696e+02  .10744849e+03  .88616670e-03", 
" -.24074656e+01  .40491713e+01 -.13441784e+02  .76463526e+02  .87710683e-03", 
" -.23587572e+01  .39206412e+01 -.13238940e+02  .75910632e+02  .89095574e-03", 
" -.23100093e+01  .37258392e+01 -.95061848e+01  .33161959e+02  .87923159e-03", 
" -.22613331e+01  .35983475e+01 -.90800538e+01  .30758400e+02  .89232444e-03", 
" -.22124654e+01  .34364939e+01 -.72147225e+01  .15148088e+02  .87556842e-03", 
" -.21634497e+01  .32940534e+01 -.63390780e+01  .10879717e+02  .88196214e-03", 
" -.21147191e+01  .32195139e+01 -.82696874e+01  .37753394e+02  .88031717e-03", 
" -.20655880e+01  .30854701e+01 -.74004858e+01  .34335323e+02  .88525799e-03", 
" -.20163616e+01  .29606281e+01 -.68266922e+01  .30680272e+02  .86761052e-03", 
" -.19671069e+01  .28414943e+01 -.60238354e+01  .22018303e+02  .85605862e-03", 
" -.19175727e+01  .27163452e+01 -.52881842e+01  .16156856e+02  .85062099e-03", 
" -.18676978e+01  .25680495e+01 -.38739063e+01  .44921246e+01  .84556491e-03", 
" -.18176742e+01  .24264096e+01 -.25196631e+01 -.57133736e+01  .83842214e-03", 
" -.17674476e+01  .22888190e+01 -.91881774e+00 -.24434937e+02  .82533384e-03", 
" -.17173183e+01  .22114143e+01 -.22401400e+01 -.57832735e+01  .82256340e-03", 
" -.16666594e+01  .20867569e+01 -.14266462e+00 -.37376546e+02  .81473069e-03", 
" -.16158833e+01  .20017267e+01 -.30832985e+00 -.37511722e+02  .82159299e-03", 
" -.15644140e+01  .18319601e+01  .38873648e+01 -.95652403e+02  .82329867e-03", 
" -.15129435e+01  .17517353e+01  .29183821e+01 -.77716926e+02  .82200586e-03", 
" -.14608868e+01  .16678155e+01  .22379030e+01 -.68576052e+02  .81850420e-03", 
" -.14084193e+01  .15372515e+01  .42772392e+01 -.95663190e+02  .83458787e-03", 
" -.13555694e+01  .14847561e+01  .21998780e+01 -.67993925e+02  .83251053e-03", 
" -.13022403e+01  .14223532e+01  .33489240e+00 -.38725014e+02  .83875988e-03", 
" -.12482685e+01  .13443413e+01 -.38357760e+00 -.26433497e+02  .82942073e-03", 
" -.11934953e+01  .12612495e+01 -.13658838e+01 -.78868300e+01  .82733355e-03", 
" -.11381567e+01  .11850681e+01 -.24203140e+01  .96299662e+01  .82701530e-03", 
" -.10821637e+01  .11078368e+01 -.30678927e+01  .20863906e+02  .82050159e-03", 
" -.10256181e+01  .10605573e+01 -.38149031e+01  .25025496e+02  .82091885e-03", 
" -.96796232e+00  .10081805e+01 -.44068754e+01  .27201694e+02  .81842619e-03", 
" -.90921621e+00  .90760856e+00 -.25069244e+01  .29820618e+01  .81444955e-03", 
" -.84959167e+00  .84991253e+00 -.22153342e+01 -.38678536e+01  .80959105e-03", 
" -.78858394e+00  .79407497e+00 -.24751903e+01  .20343179e+01  .81869479e-03", 
" -.72608771e+00  .67927643e+00  .29360358e+00 -.28621128e+02  .82540764e-03", 
" -.66213379e+00  .61810807e+00  .51908704e+00 -.29312007e+02  .83623897e-03", 
" -.59640343e+00  .47606609e+00  .52672933e+01 -.85835452e+02  .84672174e-03", 
" -.52918439e+00  .45184872e+00  .42249693e+01 -.65117954e+02  .85614198e-03", 
" -.46015031e+00  .44771069e+00  .30921493e+01 -.49128427e+02  .88895260e-03", 
" -.38841499e+00  .42436948e+00  .23931388e+01 -.36743402e+02  .90684907e-03", 
" -.31470051e+00  .38410228e+00  .37256323e+01 -.50635313e+02  .89276556e-03", 
" -.23883700e+00  .33700744e+00  .67052735e+01 -.94705779e+02  .87374406e-03", 
" -.16030589e+00  .29472317e+00  .75995030e+01 -.97732366e+02  .86579976e-03", 
" -.79483971e-01  .29285653e+00  .76726513e+01 -.99895914e+02  .87585528e-03", 
"  .39122123e-02  .37180347e+00  .42256370e+01 -.53100704e+02  .88182114e-03", 
"  .90936883e-01  .35149137e+00  .54460044e+01 -.59761691e+02  .87088242e-03", 
"  .18163366e+00  .36285646e+00  .65733775e+01 -.76979031e+02  .88946311e-03", 
"  .27675980e+00  .33586127e+00  .10412193e+02 -.13189270e+03  .91408012e-03", 
"  .37631567e+00  .49157690e+00  .55014001e+01 -.74661651e+02  .90527872e-03", 
"  .48231733e+00  .62713516e+00  .17141851e+01 -.18657207e+02  .91303083e-03", 
"  .59654695e+00  .77441219e+00 -.14032605e+01  .24476001e+02  .91984503e-03", 
"  .72159163e+00  .94073152e+00 -.17955456e+01  .19450591e+02  .95528911e-03", 
"  .86116460e+00  .12923712e+01 -.94459450e+01  .11072220e+03  .97591708e-03", 
"  .10237592e+01  .16162148e+01 -.14105972e+02  .19900594e+03  .10605176e-02", 
"  .12231860e+01  .19831477e+01 -.10509574e+02  .11738147e+03  .11653777e-02", 
"  .14887532e+01  .28919576e+01 -.24030339e+02  .30654174e+03  .13732354e-02", 
"  .15561116e+01  .29568070e+01 -.16931767e+02  .21217080e+03  .14425217e-02", 
"  .16302245e+01  .32952808e+01 -.21084271e+02  .25422457e+03  .15155264e-02", 
"  .17137595e+01  .36132629e+01 -.23198095e+02  .27251863e+03  .15883181e-02", 
"  .18090987e+01  .38730538e+01 -.19126575e+02  .21312901e+03  .17300653e-02", 
"  .19205965e+01  .41113626e+01 -.10649384e+02  .99270117e+02  .18244810e-02", 
"  .20559205e+01  .46591296e+01 -.15540034e+02  .20412444e+03  .19419829e-02", 
"  .22261701e+01  .58356915e+01 -.42571975e+02  .59599796e+03  .23100490e-02", 
"  .24604671e+01  .75383839e+01 -.69666936e+02  .98270545e+03  .28716656e-02", 
"  .28575344e+01  .99755783e+01 -.94266935e+02  .15187725e+04  .38923310e-02", 
"  .32471529e+01  .13182064e+02 -.14197312e+03  .22696567e+04  .53240409e-02", 
"  .37484764e+01  .16093214e+02 -.93584459e+02  .17123950e+04  .79128041e-02", 
"  .41292948e+01  .15861308e+02  .10489217e+03 -.94452976e+03  .11234018e-01", 
"coac      1  3  3   20", " -.48828045e+02  .65218020e+03 -.61906471e+04  .31011072e+05  .76061275e-01", 
" -.45800815e+02  .56783106e+03 -.47277432e+04  .19086251e+05  .56434607e-01", 
" -.41771360e+02  .47677410e+03 -.37421406e+04  .15119079e+05  .35803944e-01", 
" -.38675789e+02  .41192265e+03 -.31185414e+04  .13257509e+05  .25911606e-01", 
" -.35595442e+02  .35352010e+03 -.25475265e+04  .10387205e+05  .19311946e-01", 
" -.33769342e+02  .31852594e+03 -.21342995e+04  .76626099e+04  .15661806e-01", 
" -.32473654e+02  .29681610e+03 -.19765657e+04  .74202911e+04  .13551894e-01", 
" -.31457256e+02  .27956516e+03 -.18255173e+04  .68481775e+04  .12170429e-01", 
" -.30628180e+02  .26587198e+03 -.16996825e+04  .61891121e+04  .11052153e-01", 
" -.29920114e+02  .25468564e+03 -.16094315e+04  .58553130e+04  .10438345e-01", 
" -.29303144e+02  .24467782e+03 -.15222553e+04  .55170178e+04  .96050195e-02", 
" -.28755863e+02  .23610767e+03 -.14599096e+04  .54249090e+04  .89117074e-02", 
" -.28272027e+02  .22897428e+03 -.14081576e+04  .52794018e+04  .84150624e-02", 
" -.26381891e+02  .20098016e+03 -.11668879e+04  .40951926e+04  .70392698e-02", 
" -.25027216e+02  .18248966e+03 -.10536927e+04  .40520837e+04  .61864418e-02", 
" -.23963847e+02  .16767232e+03 -.91658466e+03  .32616294e+04  .57608709e-02", 
" -.23087021e+02  .15566678e+03 -.80012334e+03  .25260161e+04  .54412378e-02", 
" -.22343235e+02  .14638868e+03 -.74144634e+03  .24191395e+04  .50128214e-02", 
" -.21698395e+02  .13882627e+03 -.70446383e+03  .24504373e+04  .47712525e-02", 
" -.21122734e+02  .13199707e+03 -.66217531e+03  .23472338e+04  .43500342e-02", 
" -.20607962e+02  .12605186e+03 -.62107630e+03  .21717553e+04  .40311499e-02", 
" -.20138799e+02  .12083475e+03 -.58864189e+03  .20414286e+04  .38449541e-02", 
" -.19707073e+02  .11581236e+03 -.55009295e+03  .18729234e+04  .37796892e-02", 
" -.19307958e+02  .11086168e+03 -.49253805e+03  .13979376e+04  .36800057e-02", 
" -.18938700e+02  .10703274e+03 -.47149329e+03  .13276276e+04  .35278336e-02", 
" -.18593138e+02  .10332656e+03 -.44216830e+03  .11307649e+04  .34108671e-02", 
" -.18268982e+02  .10019305e+03 -.43181413e+03  .11707805e+04  .33222324e-02", 
" -.17962216e+02  .96990847e+02 -.40992559e+03  .10786989e+04  .32745652e-02", 
" -.17672394e+02  .94004734e+02 -.38762902e+03  .94020014e+03  .31157708e-02", 
" -.17396866e+02  .91280864e+02 -.37049815e+03  .85409696e+03  .30789065e-02", 
" -.17134589e+02  .88650212e+02 -.35122942e+03  .74328528e+03  .29843574e-02", 
" -.16884331e+02  .86234860e+02 -.33792245e+03  .71274312e+03  .29668808e-02", 
" -.16644821e+02  .84020982e+02 -.32821037e+03  .70926819e+03  .28711820e-02", 
" -.16415951e+02  .81995712e+02 -.32200502e+03  .74588256e+03  .28402663e-02", 
" -.16195949e+02  .80062570e+02 -.31701269e+03  .80155518e+03  .27753813e-02", 
" -.15982949e+02  .78089057e+02 -.30493843e+03  .74699144e+03  .27712094e-02", 
" -.15777072e+02  .76127055e+02 -.29159025e+03  .68088342e+03  .26903687e-02", 
" -.15577994e+02  .74155670e+02 -.27419687e+03  .56362193e+03  .26433863e-02", 
" -.15388096e+02  .72725332e+02 -.27626896e+03  .65549680e+03  .25932022e-02", 
" -.15202011e+02  .71041504e+02 -.26754210e+03  .64595595e+03  .25584521e-02", 
" -.15021593e+02  .69379876e+02 -.25536305e+03  .57080481e+03  .25194713e-02", 
" -.14846237e+02  .67699001e+02 -.24022538e+03  .46446627e+03  .24620481e-02", 
" -.14676669e+02  .66154714e+02 -.22823313e+03  .38907357e+03  .24197754e-02", 
" -.14512983e+02  .65060006e+02 -.23627432e+03  .55848687e+03  .23533544e-02", 
" -.14351822e+02  .63738996e+02 -.23223224e+03  .58218616e+03  .23320420e-02", 
" -.14195213e+02  .62373069e+02 -.22291039e+03  .53121777e+03  .22824394e-02", 
" -.14042728e+02  .61161981e+02 -.21917148e+03  .54711770e+03  .22445924e-02", 
" -.13894253e+02  .60010001e+02 -.21543666e+03  .54821012e+03  .22308595e-02", 
" -.13749320e+02  .58876412e+02 -.21052936e+03  .53354608e+03  .22046134e-02", 
" -.13606776e+02  .57638001e+02 -.20000127e+03  .45173747e+03  .21834836e-02", 
" -.13467464e+02  .56357564e+02 -.18538681e+03  .31976289e+03  .21590268e-02", 
" -.13331499e+02  .55356091e+02 -.18562324e+03  .38715513e+03  .21196658e-02", 
" -.13198435e+02  .54281974e+02 -.17898545e+03  .35175361e+03  .20804653e-02", 
" -.13068680e+02  .53330171e+02 -.17579612e+03  .35377310e+03  .20499410e-02", 
" -.12941790e+02  .52527644e+02 -.17947013e+03  .44499406e+03  .20231376e-02", 
" -.12816812e+02  .51646213e+02 -.17798548e+03  .46564860e+03  .20138965e-02", 
" -.12693639e+02  .50672417e+02 -.17182905e+03  .43341497e+03  .19964653e-02", 
" -.12572743e+02  .49791940e+02 -.16994460e+03  .45712156e+03  .19821602e-02", 
" -.12453995e+02  .48935357e+02 -.16796571e+03  .47495292e+03  .19435000e-02", 
" -.12337843e+02  .48086392e+02 -.16373899e+03  .45841985e+03  .19519606e-02", 
" -.12223097e+02  .47244513e+02 -.16076614e+03  .46350590e+03  .19197209e-02", 
" -.12110621e+02  .46460734e+02 -.15875620e+03  .47502225e+03  .19101412e-02", 
" -.11999782e+02  .45630395e+02 -.15234175e+03  .41685866e+03  .18581231e-02", 
" -.11890269e+02  .44738044e+02 -.14523192e+03  .37470931e+03  .18489651e-02", 
" -.11783209e+02  .44058064e+02 -.14569684e+03  .41382327e+03  .18236221e-02", 
" -.11677822e+02  .43366375e+02 -.14512979e+03  .44046015e+03  .18093368e-02", 
" -.11573546e+02  .42616971e+02 -.14126247e+03  .42596339e+03  .17917807e-02", 
" -.11470751e+02  .41878606e+02 -.13588521e+03  .37359964e+03  .18141337e-02", 
" -.11369149e+02  .41102510e+02 -.12897721e+03  .31319598e+03  .18211395e-02", 
" -.11269344e+02  .40408511e+02 -.12450319e+03  .27797169e+03  .17968264e-02", 
" -.11170879e+02  .39711018e+02 -.12036872e+03  .25748713e+03  .17987903e-02", 
" -.11074380e+02  .39175756e+02 -.12224706e+03  .30069076e+03  .17990360e-02", 
" -.10978312e+02  .38507982e+02 -.11954910e+03  .30564531e+03  .17961640e-02", 
" -.10883511e+02  .37820854e+02 -.11374776e+03  .25524330e+03  .17968638e-02", 
" -.10789744e+02  .37098162e+02 -.10627895e+03  .19242278e+03  .17685676e-02", 
" -.10697031e+02  .36491453e+02 -.10394719e+03  .18689884e+03  .17585070e-02", 
" -.10606173e+02  .35930121e+02 -.10231661e+03  .18633339e+03  .17431579e-02", 
" -.10515730e+02  .35333482e+02 -.98650354e+02  .15919256e+03  .17080668e-02", 
" -.10426321e+02  .34681361e+02 -.91800169e+02  .91737953e+02  .16885525e-02", 
" -.10338168e+02  .34151867e+02 -.91098892e+02  .10590763e+03  .16856586e-02", 
" -.10251000e+02  .33570080e+02 -.88109475e+02  .98077520e+02  .16685955e-02", 
" -.10164394e+02  .32992682e+02 -.84450374e+02  .73020658e+02  .16530557e-02", 
" -.10078932e+02  .32479648e+02 -.83813480e+02  .87042354e+02  .16510410e-02", 
" -.99939401e+01  .31876456e+02 -.79038087e+02  .58819748e+02  .16204574e-02", 
" -.99100349e+01  .31376663e+02 -.78419190e+02  .70516713e+02  .15930698e-02", 
" -.98271177e+01  .30839743e+02 -.74940570e+02  .46347017e+02  .15816360e-02", 
" -.97449575e+01  .30310936e+02 -.71510859e+02  .23394515e+02  .15650267e-02", 
" -.96637703e+01  .29828891e+02 -.70326493e+02  .29665625e+02  .15415496e-02", 
" -.95832577e+01  .29343688e+02 -.67762719e+02  .83526602e+01  .15521892e-02", 
" -.95034613e+01  .28909135e+02 -.68108960e+02  .32021577e+02  .15332799e-02", 
" -.94244815e+01  .28437552e+02 -.66291440e+02  .29904521e+02  .15135768e-02", 
" -.93457411e+01  .27885278e+02 -.60224819e+02 -.27656933e+02  .15057762e-02", 
" -.92678735e+01  .27445112e+02 -.59002286e+02 -.29428703e+02  .14850165e-02", 
" -.91902878e+01  .26947651e+02 -.55610160e+02 -.53612827e+02  .14774875e-02", 
" -.91138495e+01  .26553482e+02 -.56394211e+02 -.25175291e+02  .14785964e-02", 
" -.90380287e+01  .26154637e+02 -.56614800e+02 -.40086056e+01  .14778984e-02", 
" -.89630262e+01  .25778838e+02 -.57669248e+02  .27244324e+02  .14724333e-02", 
" -.88882136e+01  .25366407e+02 -.56476224e+02  .24248826e+02  .14524293e-02", 
" -.88142353e+01  .24998647e+02 -.56880809e+02  .37255800e+02  .14501130e-02", 
" -.87406219e+01  .24580435e+02 -.54008038e+02  .57490337e+01  .14441592e-02", 
" -.86673721e+01  .24108077e+02 -.49137052e+02 -.43257291e+02  .14374566e-02", 
" -.85945811e+01  .23693237e+02 -.47315001e+02 -.50339741e+02  .14160259e-02", 
" -.85224150e+01  .23270075e+02 -.44745582e+02 -.65337250e+02  .14194608e-02", 
" -.84507470e+01  .22902395e+02 -.44714997e+02 -.50192649e+02  .13953840e-02", 
" -.83794640e+01  .22513461e+02 -.43667177e+02 -.46434193e+02  .13920838e-02", 
" -.83088554e+01  .22148072e+02 -.42830753e+02 -.43360460e+02  .13909971e-02", 
" -.82385094e+01  .21736894e+02 -.39403156e+02 -.75385287e+02  .13787855e-02", 
" -.81684119e+01  .21336475e+02 -.37402124e+02 -.85764565e+02  .13753567e-02", 
" -.80990780e+01  .21029662e+02 -.38958346e+02 -.57706456e+02  .13575685e-02", 
" -.80299652e+01  .20654685e+02 -.36854534e+02 -.71609847e+02  .13475601e-02", 
" -.79613091e+01  .20274238e+02 -.34475730e+02 -.88073602e+02  .13558354e-02", 
" -.78932861e+01  .19970070e+02 -.35462720e+02 -.66111255e+02  .13440287e-02", 
" -.78255589e+01  .19658545e+02 -.36918777e+02 -.27775059e+02  .13519517e-02", 
" -.77579890e+01  .19352459e+02 -.37959768e+02 -.22142589e+01  .13523244e-02", 
" -.76911324e+01  .19066232e+02 -.39437589e+02  .30976577e+02  .13408525e-02", 
" -.76241044e+01  .18695176e+02 -.36625581e+02  .65441089e+01  .13321963e-02", 
" -.75575158e+01  .18384330e+02 -.37264593e+02  .30138582e+02  .13219735e-02", 
" -.74912314e+01  .18039257e+02 -.34731104e+02  .34098909e+00  .13223912e-02", 
" -.74254425e+01  .17781426e+02 -.37238116e+02  .44180316e+02  .13099628e-02", 
" -.73600129e+01  .17500644e+02 -.37050885e+02  .44001587e+02  .12847993e-02", 
" -.72943231e+01  .17106871e+02 -.32769640e+02  .45636536e+01  .12795274e-02", 
" -.72293217e+01  .16814460e+02 -.33242507e+02  .26938577e+02  .12786847e-02", 
" -.71640626e+01  .16423672e+02 -.28194926e+02 -.32864755e+02  .12690434e-02", 
" -.70992143e+01  .16106283e+02 -.26849111e+02 -.38363323e+02  .12642245e-02", 
" -.70347354e+01  .15746377e+02 -.22319324e+02 -.90866037e+02  .12573087e-02", 
" -.69703308e+01  .15398180e+02 -.19727140e+02 -.10985146e+03  .12621134e-02", 
" -.69062657e+01  .15097061e+02 -.19778678e+02 -.92985038e+02  .12452339e-02", 
" -.68426369e+01  .14776309e+02 -.16508075e+02 -.13354093e+03  .12410609e-02", 
" -.67790246e+01  .14473176e+02 -.14486078e+02 -.15538123e+03  .12605509e-02", 
" -.67155846e+01  .14178545e+02 -.13127752e+02 -.16526242e+03  .12445089e-02", 
" -.66526313e+01  .13937593e+02 -.14413891e+02 -.13753947e+03  .12341555e-02", 
" -.65896649e+01  .13675901e+02 -.14755867e+02 -.12330097e+03  .12365690e-02", 
" -.65268444e+01  .13401630e+02 -.14218100e+02 -.12345896e+03  .12329629e-02", 
" -.64641435e+01  .13168246e+02 -.15962536e+02 -.93789276e+02  .12260434e-02", 
" -.64015069e+01  .12893913e+02 -.14850635e+02 -.10108105e+03  .12198674e-02", 
" -.63388561e+01  .12596278e+02 -.13025186e+02 -.11138763e+03  .12288228e-02", 
" -.62762747e+01  .12261740e+02 -.89641411e+01 -.15542260e+03  .12243383e-02", 
" -.62142383e+01  .12011021e+02 -.87777961e+01 -.15124895e+03  .12343945e-02", 
" -.61518419e+01  .11679301e+02 -.46211826e+01 -.19764302e+03  .12238281e-02", 
" -.60895225e+01  .11421039e+02 -.53488928e+01 -.17110793e+03  .12140679e-02", 
" -.60275509e+01  .11166804e+02 -.47382091e+01 -.17029465e+03  .12137060e-02", 
" -.59655349e+01  .10913958e+02 -.43700780e+01 -.16872891e+03  .12148122e-02", 
" -.59036775e+01  .10661764e+02 -.31996875e+01 -.18294998e+03  .12082475e-02", 
" -.58418934e+01  .10441478e+02 -.46620412e+01 -.14919186e+03  .11915666e-02", 
" -.57801947e+01  .10226419e+02 -.56527658e+01 -.12787638e+03  .12004408e-02", 
" -.57182090e+01  .99689416e+01 -.40289900e+01 -.14728338e+03  .12123238e-02", 
" -.56565969e+01  .97580984e+01 -.52924112e+01 -.12278629e+03  .11908644e-02", 
" -.55945719e+01  .95133502e+01 -.52615330e+01 -.11463618e+03  .12131002e-02", 
" -.55325977e+01  .92403638e+01 -.35735856e+01 -.12647484e+03  .12000687e-02", 
" -.54704256e+01  .89641361e+01 -.15436716e+01 -.14503017e+03  .11973558e-02", 
" -.54081705e+01  .86937262e+01 -.30501387e-01 -.15681919e+03  .11774521e-02", 
" -.53462802e+01  .85071819e+01 -.19565178e+01 -.12828627e+03  .11534324e-02", 
" -.52841082e+01  .82682485e+01 -.17501323e+01 -.11834821e+03  .11413119e-02", 
" -.52217047e+01  .80012229e+01 -.29184267e+00 -.12427988e+03  .11346622e-02", 
" -.51591946e+01  .77188311e+01  .27855036e+01 -.15717552e+03  .11493584e-02", 
" -.50968970e+01  .74939546e+01  .35978596e+01 -.16856207e+03  .11447003e-02", 
" -.50342564e+01  .72433642e+01  .53376278e+01 -.18671387e+03  .11546066e-02", 
" -.49714279e+01  .70195669e+01  .50942392e+01 -.17795181e+03  .11464845e-02", 
" -.49087559e+01  .68711153e+01  .16336881e+01 -.12938133e+03  .11514573e-02", 
" -.48457572e+01  .66930923e+01 -.64121267e+00 -.91451288e+02  .11480392e-02", 
" -.47821283e+01  .63809247e+01  .43086049e+01 -.15050907e+03  .11292625e-02", 
" -.47185529e+01  .61818362e+01  .33353520e+01 -.12864427e+03  .11250097e-02", 
" -.46548989e+01  .60120226e+01  .12326835e+01 -.96423057e+02  .11147958e-02", 
" -.45909748e+01  .57920670e+01  .18677367e+01 -.10151468e+03  .11168946e-02", 
" -.45265100e+01  .55864791e+01  .17017630e+01 -.95424597e+02  .11012447e-02", 
" -.44617572e+01  .53937127e+01  .98974211e+00 -.84849370e+02  .10956031e-02", 
" -.43968957e+01  .51917731e+01  .12009423e+01 -.85824300e+02  .10869110e-02", 
" -.43315280e+01  .49938422e+01  .95243733e+00 -.78741206e+02  .10619519e-02", 
" -.42657368e+01  .47993071e+01 -.16250133e+00 -.57088372e+02  .10414027e-02", 
" -.41995243e+01  .45846460e+01  .66559029e+00 -.64025848e+02  .10294939e-02", 
" -.41328831e+01  .43829618e+01  .68491727e+00 -.59167639e+02  .10476633e-02", 
" -.40656760e+01  .41853881e+01  .16929377e+00 -.47103571e+02  .10567184e-02", 
" -.39978028e+01  .39321899e+01  .17890398e+01 -.57427663e+02  .10596376e-02", 
" -.39291765e+01  .36687355e+01  .38474635e+01 -.72432736e+02  .10602802e-02", 
" -.38605866e+01  .35474903e+01 -.49897131e+00 -.84761566e+01  .10685793e-02", 
" -.37908886e+01  .33484465e+01 -.13005136e+01  .91576113e+01  .10714990e-02", 
" -.37205043e+01  .31206209e+01 -.72853895e-01 -.23005868e+01  .10726126e-02", 
" -.36497333e+01  .29700790e+01 -.24958188e+01  .30881686e+02  .10609416e-02", 
" -.35781993e+01  .27410106e+01  .82604373e+00 -.22414398e+02  .10763120e-02", 
" -.35052063e+01  .24638002e+01  .46110170e+01 -.67070406e+02  .10673328e-02", 
" -.34316304e+01  .22552333e+01  .50471571e+01 -.67803320e+02  .10702915e-02", 
" -.33567514e+01  .20430906e+01  .46857052e+01 -.48891999e+02  .10802971e-02", 
" -.32812008e+01  .18743100e+01  .34302617e+01 -.27413349e+02  .10649462e-02", 
" -.32039023e+01  .16130026e+01  .66692776e+01 -.70857761e+02  .10904733e-02", 
" -.31259686e+01  .14792877e+01  .37871905e+01 -.30061979e+02  .11162362e-02", 
" -.30465682e+01  .13757997e+01 -.72381931e+00  .26885672e+02  .11215559e-02", 
" -.29655020e+01  .11877255e+01 -.14971676e+01  .46443095e+02  .11192236e-02", 
" -.28826395e+01  .97077677e+00  .35861856e+00  .15565010e+02  .11316957e-02", 
" -.27977204e+01  .78030007e+00 -.16040570e+00  .25413103e+02  .11330898e-02", 
" -.27107398e+01  .52628576e+00  .34216490e+01 -.25994560e+02  .11501861e-02", 
" -.26220285e+01  .38537952e+00  .12977132e+01 -.71346908e+00  .11476904e-02", 
" -.25307856e+01  .21135483e+00  .62845759e+00  .80653423e+01  .11576454e-02", 
" -.24369014e+01  .39879607e-01 -.39597601e+00  .27567139e+02  .11708093e-02", 
" -.23390608e+01 -.32584077e+00  .84973428e+01 -.87416750e+02  .11713590e-02", 
" -.22384472e+01 -.58298146e+00  .12659529e+02 -.14313077e+03  .11822478e-02", 
" -.21341873e+01 -.71445024e+00  .96972485e+01 -.10851529e+03  .11915683e-02", 
" -.20253914e+01 -.89834301e+00  .78634339e+01 -.71686129e+02  .11796609e-02", 
" -.19118482e+01 -.11587845e+01  .10251035e+02 -.84321249e+02  .12055131e-02", 
" -.17922833e+01 -.14655021e+01  .16961031e+02 -.17691475e+03  .12224587e-02", 
" -.16666290e+01 -.16656971e+01  .18370145e+02 -.20304569e+03  .12418443e-02", 
" -.15341761e+01 -.16640513e+01  .99116397e+01 -.10045938e+03  .12256358e-02", 
" -.13920089e+01 -.18279279e+01  .91477076e+01 -.96791945e+02  .12658186e-02", 
" -.12397921e+01 -.19823528e+01  .93044959e+01 -.10371214e+03  .12943855e-02", 
" -.10742225e+01 -.20103356e+01  .82228695e+00  .16249796e+02  .12974809e-02", 
" -.89209734e+00 -.22486648e+01  .80397037e+01 -.98136987e+02  .13038466e-02", 
" -.68858668e+00 -.21634327e+01 -.32465669e+01  .24751653e+02  .13959469e-02", 
" -.45226772e+00 -.22967662e+01 -.38630262e+01  .16160286e+02  .14915172e-02", 
" -.16837047e+00 -.24461563e+01 -.16266925e+01 -.53510210e+02  .15655110e-02", 
"  .19844150e+00 -.20479385e+01 -.23269812e+02  .17497538e+03  .17815485e-02", 
"  .28903701e+00 -.21111527e+01 -.16359375e+02  .45331052e+02  .18407554e-02", 
"  .38860599e+00 -.21541673e+01 -.12690828e+02 -.35897908e+02  .18836621e-02", 
"  .49971461e+00 -.23214384e+01 -.51819631e+01 -.12606497e+03  .19696714e-02", 
"  .62317599e+00 -.21258110e+01 -.12386079e+02 -.51693658e+02  .21147934e-02", 
"  .76606191e+00 -.20914021e+01 -.12017363e+02 -.60082261e+02  .23391606e-02", 
"  .93306757e+00 -.16161248e+01 -.21818585e+02 -.13174950e+02  .24623955e-02", 
"  .11446356e+01 -.13971063e+01 -.18979605e+02 -.10206599e+03  .27102256e-02", 
"  .14261438e+01 -.41838964e+00 -.34666108e+02 -.41089714e+02  .32800761e-02", 
"  .18880338e+01  .14397282e+01 -.78793004e+02  .39718169e+03  .42293392e-02", 
"  .23352118e+01  .30456976e+01 -.11230096e+03  .73579872e+03  .57731525e-02", 
"  .29037842e+01  .42908102e+01 -.34943177e+02 -.86109384e+03  .87682574e-02", 
"  .33185508e+01  .56969142e+01  .15153347e+02 -.19690332e+04  .12433482e-01", 
"coact     1  4  3   25", " -.58053367e+02  .92180470e+03 -.10835069e+05  .74734337e+05  .83002246e-01", 
" -.54751980e+02  .81177712e+03 -.85114867e+04  .49902835e+05  .64817624e-01", 
" -.50416079e+02  .67814455e+03 -.58701692e+04  .23644852e+05  .41657794e-01", 
" -.47111167e+02  .59921915e+03 -.51511324e+04  .23516341e+05  .31479061e-01", 
" -.43785232e+02  .52359695e+03 -.44097652e+04  .21500670e+05  .22190772e-01", 
" -.41804187e+02  .48000150e+03 -.39832804e+04  .20451396e+05  .19232484e-01", 
" -.40380543e+02  .44674264e+03 -.34744338e+04  .15857811e+05  .16681441e-01", 
" -.39278426e+02  .42437824e+03 -.32477033e+04  .14619314e+05  .14902299e-01", 
" -.38362556e+02  .40453796e+03 -.29816810e+04  .12405273e+05  .13250329e-01", 
" -.37586411e+02  .38879088e+03 -.28192791e+04  .11714930e+05  .12207488e-01", 
" -.36918754e+02  .37599284e+03 -.26916275e+04  .11027605e+05  .11343188e-01", 
" -.36326793e+02  .36601786e+03 -.26699947e+04  .11960393e+05  .10908059e-01", 
" -.35787556e+02  .35513170e+03 -.25302705e+04  .10890632e+05  .99765752e-02", 
" -.33706112e+02  .31706955e+03 -.21902371e+04  .97810497e+04  .81077124e-02", 
" -.32214457e+02  .29155404e+03 -.19722451e+04  .88869199e+04  .72082591e-02", 
" -.31041981e+02  .27157310e+03 -.17691251e+04  .75244608e+04  .68142581e-02", 
" -.30070881e+02  .25548169e+03 -.16183738e+04  .67269564e+04  .62428987e-02", 
" -.29239173e+02  .24130271e+03 -.14481069e+04  .53047086e+04  .59196794e-02", 
" -.28516334e+02  .23028763e+03 -.13661900e+04  .51052797e+04  .56630731e-02", 
" -.27874405e+02  .22070170e+03 -.12968017e+04  .49159783e+04  .52080034e-02", 
" -.27294245e+02  .21188210e+03 -.12167391e+04  .44653512e+04  .49793847e-02", 
" -.26765993e+02  .20442281e+03 -.11793166e+04  .46419686e+04  .47398819e-02", 
" -.26281042e+02  .19738875e+03 -.11234527e+04  .44096853e+04  .45025810e-02", 
" -.25831576e+02  .19093280e+03 -.10703574e+04  .41412229e+04  .44473009e-02", 
" -.25413805e+02  .18531333e+03 -.10413480e+04  .42339880e+04  .42797779e-02", 
" -.25023504e+02  .18038996e+03 -.10342106e+04  .46802596e+04  .41372246e-02", 
" -.24654483e+02  .17538372e+03 -.10009132e+04  .46237912e+04  .40647632e-02", 
" -.24306526e+02  .17080348e+03 -.97221540e+03  .45811096e+04  .39016798e-02", 
" -.23977097e+02  .16646173e+03 -.93837127e+03  .43881940e+04  .38176588e-02", 
" -.23662581e+02  .16224735e+03 -.90081943e+03  .41161785e+04  .37964111e-02", 
" -.23362381e+02  .15828377e+03 -.86729600e+03  .39102482e+04  .36988976e-02", 
" -.23074578e+02  .15445335e+03 -.83397249e+03  .36966694e+04  .35757008e-02", 
" -.22800232e+02  .15091549e+03 -.80543767e+03  .35304959e+04  .35039894e-02", 
" -.22536223e+02  .14746189e+03 -.77513211e+03  .33754721e+04  .34393468e-02", 
" -.22282088e+02  .14417930e+03 -.74422869e+03  .31189176e+04  .33998788e-02", 
" -.22037142e+02  .14118512e+03 -.72746211e+03  .31171559e+04  .33286499e-02", 
" -.21800599e+02  .13839472e+03 -.71450582e+03  .31315573e+04  .31953436e-02", 
" -.21571741e+02  .13557001e+03 -.69205178e+03  .29875869e+04  .31696496e-02", 
" -.21350541e+02  .13294814e+03 -.67888926e+03  .30113158e+04  .30986892e-02", 
" -.21136504e+02  .13059047e+03 -.67602279e+03  .31985796e+04  .30285637e-02", 
" -.20928405e+02  .12818708e+03 -.66360175e+03  .32033577e+04  .29899290e-02", 
" -.20725762e+02  .12581694e+03 -.64652807e+03  .30796245e+04  .29301033e-02", 
" -.20528881e+02  .12336665e+03 -.62087476e+03  .28509125e+04  .28558761e-02", 
" -.20336746e+02  .12102403e+03 -.59909645e+03  .26779391e+04  .28263609e-02", 
" -.20150586e+02  .11894236e+03 -.58743536e+03  .26504157e+04  .28275094e-02", 
" -.19968632e+02  .11692987e+03 -.57752264e+03  .26468110e+04  .28018946e-02", 
" -.19790821e+02  .11496262e+03 -.57170986e+03  .27683057e+04  .27692939e-02", 
" -.19617354e+02  .11301583e+03 -.55901444e+03  .27132370e+04  .28047695e-02", 
" -.19446819e+02  .11094206e+03 -.53818026e+03  .25554719e+04  .28065816e-02", 
" -.19280094e+02  .10888555e+03 -.51425978e+03  .23337620e+04  .27519418e-02", 
" -.19117891e+02  .10711187e+03 -.50231049e+03  .22797044e+04  .27349429e-02", 
" -.18958150e+02  .10525702e+03 -.48692552e+03  .21982982e+04  .26780989e-02", 
" -.18802302e+02  .10346766e+03 -.46958012e+03  .20571924e+04  .26178602e-02", 
" -.18649840e+02  .10183919e+03 -.45693049e+03  .19451719e+04  .26050569e-02", 
" -.18499927e+02  .10021835e+03 -.44424341e+03  .18537022e+04  .25775197e-02", 
" -.18352348e+02  .98595115e+02 -.43123950e+03  .17577584e+04  .25365076e-02", 
" -.18207982e+02  .97121083e+02 -.42583062e+03  .18124904e+04  .25102197e-02", 
" -.18065618e+02  .95698036e+02 -.42174704e+03  .18651780e+04  .25075947e-02", 
" -.17926065e+02  .94223922e+02 -.41032738e+03  .17764931e+04  .25177059e-02", 
" -.17787903e+02  .92745502e+02 -.40064235e+03  .17416459e+04  .24691326e-02", 
" -.17652779e+02  .91472715e+02 -.39977130e+03  .18232325e+04  .24719150e-02", 
" -.17519712e+02  .90032920e+02 -.38404344e+03  .16454908e+04  .24542483e-02", 
" -.17389118e+02  .88809233e+02 -.37982558e+03  .16477002e+04  .24060488e-02", 
" -.17260061e+02  .87448171e+02 -.36732499e+03  .15365118e+04  .23454420e-02", 
" -.17132521e+02  .86099632e+02 -.35397204e+03  .14067060e+04  .23432884e-02", 
" -.17007356e+02  .84912045e+02 -.35109391e+03  .14567109e+04  .23248461e-02", 
" -.16883780e+02  .83722965e+02 -.34505962e+03  .14224164e+04  .23129345e-02", 
" -.16761880e+02  .82580909e+02 -.34087898e+03  .14364862e+04  .22787055e-02", 
" -.16641684e+02  .81393125e+02 -.33410490e+03  .14243993e+04  .22614227e-02", 
" -.16523007e+02  .80302368e+02 -.33226634e+03  .14920562e+04  .22370453e-02", 
" -.16405997e+02  .79244168e+02 -.32953870e+03  .15271304e+04  .22146903e-02", 
" -.16289772e+02  .78133324e+02 -.32472829e+03  .15399284e+04  .21661956e-02", 
" -.16175707e+02  .77136859e+02 -.32347206e+03  .15843075e+04  .21712694e-02", 
" -.16062353e+02  .75936231e+02 -.30956531e+03  .14457762e+04  .21749035e-02", 
" -.15950501e+02  .74914186e+02 -.30554576e+03  .14521732e+04  .21836690e-02", 
" -.15840308e+02  .73954579e+02 -.30513121e+03  .15283757e+04  .21583248e-02", 
" -.15731585e+02  .73044572e+02 -.30420159e+03  .15709515e+04  .21318423e-02", 
" -.15623365e+02  .72041641e+02 -.29940062e+03  .15731367e+04  .21125696e-02", 
" -.15515825e+02  .70972983e+02 -.29080746e+03  .15170586e+04  .21159114e-02", 
" -.15409866e+02  .70092892e+02 -.29032461e+03  .15663795e+04  .20979683e-02", 
" -.15304371e+02  .69002635e+02 -.27764909e+03  .14369640e+04  .20722210e-02", 
" -.15200614e+02  .68079123e+02 -.27157772e+03  .13844751e+04  .20798696e-02", 
" -.15098154e+02  .67239703e+02 -.27095355e+03  .14271623e+04  .20558251e-02", 
" -.14996385e+02  .66327896e+02 -.26406961e+03  .13565725e+04  .20437174e-02", 
" -.14895376e+02  .65444016e+02 -.25996658e+03  .13477139e+04  .20189890e-02", 
" -.14794955e+02  .64469095e+02 -.25107748e+03  .12723673e+04  .20232488e-02", 
" -.14695572e+02  .63563770e+02 -.24558968e+03  .12560919e+04  .19961864e-02", 
" -.14596865e+02  .62660968e+02 -.24003175e+03  .12382414e+04  .19764366e-02", 
" -.14499838e+02  .61935056e+02 -.24256727e+03  .13280041e+04  .19634808e-02", 
" -.14402863e+02  .61044221e+02 -.23539675e+03  .12673839e+04  .19542931e-02", 
" -.14306828e+02  .60162383e+02 -.22684982e+03  .11673150e+04  .19619581e-02", 
" -.14211445e+02  .59284521e+02 -.21930200e+03  .10982357e+04  .19177623e-02", 
" -.14116603e+02  .58403530e+02 -.21323230e+03  .10739786e+04  .19076870e-02", 
" -.14022982e+02  .57668142e+02 -.21297112e+03  .11266186e+04  .19107118e-02", 
" -.13929500e+02  .56819275e+02 -.20601052e+03  .10653412e+04  .19241922e-02", 
" -.13836864e+02  .55890178e+02 -.19124021e+03  .86610888e+03  .19205632e-02", 
" -.13745247e+02  .55155579e+02 -.18931992e+03  .88586220e+03  .19025988e-02", 
" -.13653989e+02  .54408715e+02 -.18830864e+03  .93683781e+03  .19007604e-02", 
" -.13563996e+02  .53799613e+02 -.19250158e+03  .10413461e+04  .18850994e-02", 
" -.13474084e+02  .53152156e+02 -.19366038e+03  .10926074e+04  .18875815e-02", 
" -.13384486e+02  .52389560e+02 -.18946204e+03  .10804116e+04  .18945316e-02", 
" -.13295605e+02  .51641059e+02 -.18450483e+03  .10476276e+04  .19092166e-02", 
" -.13207030e+02  .50875642e+02 -.17892328e+03  .10075126e+04  .19326402e-02", 
" -.13119154e+02  .50135398e+02 -.17430661e+03  .98633688e+03  .19103825e-02", 
" -.13031849e+02  .49403419e+02 -.16718592e+03  .89991827e+03  .18964594e-02", 
" -.12944940e+02  .48612022e+02 -.15680161e+03  .77372361e+03  .18971993e-02", 
" -.12858353e+02  .47886290e+02 -.15258200e+03  .76094851e+03  .18966858e-02", 
" -.12772368e+02  .47161382e+02 -.14624905e+03  .70684690e+03  .19126299e-02", 
" -.12686646e+02  .46424818e+02 -.14034682e+03  .66105791e+03  .19025461e-02", 
" -.12601597e+02  .45744634e+02 -.13546052e+03  .60892819e+03  .19093437e-02", 
" -.12516712e+02  .44979464e+02 -.12422303e+03  .44469247e+03  .18865528e-02", 
" -.12432865e+02  .44395463e+02 -.12388765e+03  .46690359e+03  .18834184e-02", 
" -.12349179e+02  .43745424e+02 -.12067291e+03  .46188982e+03  .18920740e-02", 
" -.12265809e+02  .43195267e+02 -.12378208e+03  .55445928e+03  .18882913e-02", 
" -.12182825e+02  .42590958e+02 -.12234968e+03  .56979418e+03  .18677767e-02", 
" -.12100439e+02  .42112703e+02 -.12880300e+03  .71056884e+03  .18599820e-02", 
" -.12017848e+02  .41478081e+02 -.12610133e+03  .71059151e+03  .18488992e-02", 
" -.11935720e+02  .40838224e+02 -.12066576e+03  .64433273e+03  .18496666e-02", 
" -.11853953e+02  .40256192e+02 -.11967977e+03  .66581255e+03  .18256757e-02", 
" -.11772815e+02  .39722261e+02 -.12066470e+03  .71216042e+03  .18082058e-02", 
" -.11691708e+02  .39245579e+02 -.12527820e+03  .80516521e+03  .17929467e-02", 
" -.11610553e+02  .38627131e+02 -.12063411e+03  .75104985e+03  .17774784e-02", 
" -.11529782e+02  .37992954e+02 -.11425971e+03  .67256756e+03  .17641991e-02", 
" -.11449176e+02  .37472152e+02 -.11671721e+03  .74864535e+03  .17682360e-02", 
" -.11368854e+02  .36852789e+02 -.11091606e+03  .68023049e+03  .17664040e-02", 
" -.11289053e+02  .36351202e+02 -.11252372e+03  .73321345e+03  .17657259e-02", 
" -.11208665e+02  .35672045e+02 -.10433153e+03  .63791421e+03  .17672020e-02", 
" -.11128922e+02  .35170708e+02 -.10597504e+03  .69795431e+03  .17721654e-02", 
" -.11049581e+02  .34657263e+02 -.10696884e+03  .75270845e+03  .17661377e-02", 
" -.10969813e+02  .34046476e+02 -.10235406e+03  .71809470e+03  .17651557e-02", 
" -.10890788e+02  .33634824e+02 -.10906690e+03  .85493189e+03  .17459457e-02", 
" -.10811637e+02  .33119364e+02 -.10951574e+03  .88932337e+03  .17484726e-02", 
" -.10732389e+02  .32519059e+02 -.10531375e+03  .86801426e+03  .17313770e-02", 
" -.10653479e+02  .31951978e+02 -.10204595e+03  .85794905e+03  .17424758e-02", 
" -.10574659e+02  .31385432e+02 -.99224665e+02  .85578544e+03  .17326057e-02", 
" -.10495935e+02  .30868334e+02 -.99523073e+02  .89941758e+03  .17354601e-02", 
" -.10417193e+02  .30297336e+02 -.94125553e+02  .83353829e+03  .17194039e-02", 
" -.10338772e+02  .29870262e+02 -.98576469e+02  .93677729e+03  .17012198e-02", 
" -.10260422e+02  .29408864e+02 -.10048822e+03  .99952908e+03  .16801564e-02", 
" -.10181477e+02  .28822781e+02 -.95570947e+02  .95619120e+03  .16748051e-02", 
" -.10103189e+02  .28324611e+02 -.94480081e+02  .96359648e+03  .16880126e-02", 
" -.10024693e+02  .27719114e+02 -.86245343e+02  .85124172e+03  .16967624e-02", 
" -.99462331e+01  .27199492e+02 -.83083615e+02  .82330915e+03  .17068139e-02", 
" -.98676442e+01  .26732661e+02 -.85471541e+02  .90226253e+03  .16700362e-02", 
" -.97892696e+01  .26282340e+02 -.87781410e+02  .98173132e+03  .16698911e-02", 
" -.97104940e+01  .25748097e+02 -.84788063e+02  .96596115e+03  .16516039e-02", 
" -.96315934e+01  .25157165e+02 -.77437927e+02  .87062489e+03  .16610315e-02", 
" -.95521167e+01  .24474251e+02 -.65645014e+02  .71400960e+03  .16334564e-02", 
" -.94730709e+01  .23979070e+02 -.65590781e+02  .75001120e+03  .16176154e-02", 
" -.93942341e+01  .23519307e+02 -.64284200e+02  .72914098e+03  .15969686e-02", 
" -.93150644e+01  .23024304e+02 -.63734951e+02  .75906071e+03  .15912953e-02", 
" -.92356937e+01  .22546207e+02 -.63330015e+02  .78770311e+03  .16079330e-02", 
" -.91562361e+01  .22065536e+02 -.61648644e+02  .77200235e+03  .16134535e-02", 
" -.90762814e+01  .21474684e+02 -.54218110e+02  .67490809e+03  .16162329e-02", 
" -.89963049e+01  .21097774e+02 -.61674416e+02  .82654047e+03  .16227970e-02", 
" -.89164942e+01  .20654651e+02 -.61901995e+02  .84408173e+03  .16065619e-02", 
" -.88359315e+01  .20116480e+02 -.57915342e+02  .80795327e+03  .16042117e-02", 
" -.87555054e+01  .19690565e+02 -.60244957e+02  .87430403e+03  .15983244e-02", 
" -.86742759e+01  .19042666e+02 -.49274143e+02  .73162467e+03  .16105832e-02", 
" -.85933810e+01  .18663172e+02 -.54515417e+02  .83894345e+03  .16102587e-02", 
" -.85115455e+01  .18037330e+02 -.43675051e+02  .68279902e+03  .16161733e-02", 
" -.84293860e+01  .17430191e+02 -.33676888e+02  .53038111e+03  .16326883e-02", 
" -.83471464e+01  .16997156e+02 -.34913245e+02  .56292024e+03  .16388877e-02", 
" -.82649032e+01  .16550954e+02 -.34031383e+02  .55859289e+03  .16265752e-02", 
" -.81821567e+01  .16069983e+02 -.31841738e+02  .54888779e+03  .16244544e-02", 
" -.80984927e+01  .15528898e+02 -.26261192e+02  .48265011e+03  .16060051e-02", 
" -.80140057e+01  .14963359e+02 -.19217722e+02  .38567701e+03  .15721827e-02", 
" -.79291979e+01  .14418070e+02 -.13509504e+02  .30736383e+03  .15721990e-02", 
" -.78439535e+01  .13918051e+02 -.11667839e+02  .29827013e+03  .15580600e-02", 
" -.77581704e+01  .13451679e+02 -.12653460e+02  .35613575e+03  .15464488e-02", 
" -.76718172e+01  .13018468e+02 -.14703578e+02  .41083926e+03  .15597119e-02", 
" -.75852294e+01  .12613513e+02 -.16273839e+02  .43558097e+03  .15474674e-02", 
" -.74976847e+01  .12174699e+02 -.16850052e+02  .45914413e+03  .15286926e-02", 
" -.74090904e+01  .11749390e+02 -.22908666e+02  .60831590e+03  .15379796e-02", 
" -.73194243e+01  .11248867e+02 -.23108223e+02  .65254390e+03  .15466010e-02", 
" -.72291064e+01  .10713713e+02 -.17501281e+02  .57314460e+03  .15346674e-02", 
" -.71378787e+01  .10180864e+02 -.13382002e+02  .52868082e+03  .15420851e-02", 
" -.70455038e+01  .96531413e+01 -.10640140e+02  .51380988e+03  .15283170e-02", 
" -.69518689e+01  .91229405e+01 -.92618989e+01  .52807014e+03  .15354333e-02", 
" -.68575389e+01  .85972254e+01 -.61933693e+01  .49982665e+03  .15288312e-02", 
" -.67619016e+01  .81813906e+01 -.10420470e+02  .57655306e+03  .15149993e-02", 
" -.66652290e+01  .77428964e+01 -.12205498e+02  .61717482e+03  .15141001e-02", 
" -.65670612e+01  .72370745e+01 -.92816761e+01  .56193651e+03  .15124632e-02", 
" -.64662077e+01  .65710505e+01 -.14552029e+01  .47384990e+03  .15115136e-02", 
" -.63644104e+01  .59638671e+01  .38582255e+01  .41682611e+03  .14949056e-02", 
" -.62614521e+01  .54872418e+01  .31271526e+01  .43558543e+03  .14988163e-02", 
" -.61550931e+01  .47470669e+01  .15737626e+02  .25976067e+03  .15206906e-02", 
" -.60472432e+01  .40568906e+01  .25200614e+02  .14655798e+03  .15370147e-02", 
" -.59382747e+01  .36530977e+01  .19438823e+02  .24705276e+03  .14882645e-02", 
" -.58267416e+01  .32512923e+01  .11660065e+02  .38836842e+03  .15414216e-02", 
" -.57110487e+01  .25607240e+01  .18075475e+02  .32192911e+03  .15689180e-02", 
" -.55932498e+01  .20389610e+01  .17093397e+02  .34278522e+03  .15982211e-02", 
" -.54717199e+01  .13241436e+01  .27121503e+02  .18158872e+03  .16247220e-02", 
" -.53462488e+01  .55906074e+00  .36816130e+02  .67095285e+02  .16269247e-02", 
" -.52168897e+01 -.56990886e-01  .39648305e+02  .16792561e+02  .16435357e-02", 
" -.50832262e+01 -.52168848e+00  .29982653e+02  .17762122e+03  .16384505e-02", 
" -.49432836e+01 -.13386095e+01  .38426400e+02  .81792325e+02  .16636559e-02", 
" -.47977327e+01 -.20914052e+01  .43742161e+02  .28549477e+01  .16776368e-02", 
" -.46459501e+01 -.27999740e+01  .44115921e+02  .20169987e+02  .16959438e-02", 
" -.44854783e+01 -.36000782e+01  .45193870e+02  .53229161e+02  .17317616e-02", 
" -.43161380e+01 -.43960422e+01  .47678100e+02  .41292306e+02  .17379582e-02", 
" -.41356262e+01 -.51421543e+01  .44568387e+02  .94446372e+02  .17531302e-02", 
" -.39388839e+01 -.64938576e+01  .70011021e+02 -.25773199e+03  .18408859e-02", 
" -.37265130e+01 -.73656384e+01  .64030805e+02 -.15163010e+03  .18517729e-02", 
" -.34916390e+01 -.83119124e+01  .60002829e+02 -.12250031e+03  .19531202e-02", 
" -.32263468e+01 -.94453154e+01  .54956270e+02  .14567725e+01  .20865022e-02", 
" -.29178864e+01 -.10658984e+02  .44172240e+02  .17014710e+03  .21387789e-02", 
" -.25413726e+01 -.12998456e+02  .75877203e+02 -.27568701e+03  .23663325e-02", 
" -.20513642e+01 -.15368228e+02  .85197990e+02 -.47652557e+03  .26491308e-02", 
" -.19300414e+01 -.16248778e+02  .10598177e+03 -.81881488e+03  .27324200e-02", 
" -.17980019e+01 -.16761762e+02  .99841767e+02 -.74589031e+03  .27805364e-02", 
" -.16504618e+01 -.17908205e+02  .13157408e+03 -.13382200e+04  .29305155e-02", 
" -.14860615e+01 -.18620707e+02  .12197240e+03 -.11166808e+04  .29858222e-02", 
" -.12962089e+01 -.19836944e+02  .13095995e+03 -.11933088e+04  .32148275e-02", 
" -.10723053e+01 -.21109391e+02  .14174084e+03 -.13922336e+04  .34549068e-02", 
" -.79828867e+00 -.22118829e+02  .12543314e+03 -.12749119e+04  .37040436e-02", 
" -.43058005e+00 -.23673413e+02  .10427028e+03 -.94304621e+03  .43622854e-02", 
"  .15298849e+00 -.25968305e+02  .65984582e+02 -.58846126e+03  .56578432e-02", 
"  .69787301e+00 -.28078526e+02  .13440315e+02  .14384094e+03  .75586288e-02", 
"  .13664531e+01 -.28981485e+02 -.10644613e+03  .79508672e+03  .11040341e-01", 
"  .18490678e+01 -.29320455e+02 -.21650818e+03  .16255014e+04  .15344035e-01", 
"coactt    1  5  3   20", " -.66013848e+02  .11278008e+04 -.12014900e+05  .62795634e+05  .84750822e-01", 
" -.62752673e+02  .10316559e+04 -.10854221e+05  .57772820e+05  .61939204e-01", 
" -.58205986e+02  .89491726e+03 -.89863957e+04  .47030313e+05  .39113041e-01", 
" -.54711449e+02  .79628058e+03 -.76898141e+04  .39358320e+05  .28328126e-01", 
" -.51150881e+02  .69750753e+03 -.63211875e+04  .30687106e+05  .20221799e-01", 
" -.49054152e+02  .64466783e+03 -.56762440e+04  .26971804e+05  .16285411e-01", 
" -.47524326e+02  .60319812e+03 -.50452638e+04  .22344780e+05  .14816845e-01", 
" -.46350784e+02  .57540173e+03 -.47435836e+04  .20999903e+05  .13498208e-01", 
" -.45377765e+02  .55267381e+03 -.45039579e+04  .20018854e+05  .12130490e-01", 
" -.44551784e+02  .53323269e+03 -.42698754e+04  .18659206e+05  .11418405e-01", 
" -.43825571e+02  .51565234e+03 -.40385462e+04  .17165078e+05  .10913814e-01", 
" -.43189557e+02  .50184036e+03 -.39032234e+04  .16626615e+05  .10226305e-01", 
" -.42610483e+02  .48854138e+03 -.37362442e+04  .15540405e+05  .97325499e-02", 
" -.40363135e+02  .43971427e+03 -.32163708e+04  .12985897e+05  .81990272e-02", 
" -.38742399e+02  .40563054e+03 -.28470828e+04  .11041595e+05  .74319689e-02", 
" -.37471516e+02  .38007137e+03 -.25874139e+04  .98056084e+04  .66370609e-02", 
" -.36420201e+02  .36011264e+03 -.24122519e+04  .91704121e+04  .62227059e-02", 
" -.35520994e+02  .34312694e+03 -.22504425e+04  .84276198e+04  .58962588e-02", 
" -.34735565e+02  .32887851e+03 -.21284171e+04  .79810966e+04  .55692033e-02", 
" -.34029172e+02  .31548447e+03 -.19905237e+04  .72533919e+04  .53573591e-02", 
" -.33397023e+02  .30425414e+03 -.18911310e+04  .68357597e+04  .51249278e-02", 
" -.32822413e+02  .29456096e+03 -.18189417e+04  .66483827e+04  .48577505e-02", 
" -.32290408e+02  .28498401e+03 -.17182715e+04  .60481142e+04  .47891211e-02", 
" -.31799102e+02  .27666688e+03 -.16486112e+04  .57935231e+04  .45649859e-02", 
" -.31339269e+02  .26883093e+03 -.15791077e+04  .54831702e+04  .43721279e-02", 
" -.30908076e+02  .26168676e+03 -.15212602e+04  .52850861e+04  .42390321e-02", 
" -.30502673e+02  .25494687e+03 -.14626228e+04  .50315025e+04  .40990182e-02", 
" -.30115941e+02  .24837002e+03 -.14004356e+04  .47224359e+04  .39936819e-02", 
" -.29752518e+02  .24261798e+03 -.13565467e+04  .45683516e+04  .39124242e-02", 
" -.29405976e+02  .23697213e+03 -.13042317e+04  .42862738e+04  .37202265e-02", 
" -.29075284e+02  .23186429e+03 -.12687381e+04  .42015292e+04  .35441983e-02", 
" -.28758518e+02  .22685951e+03 -.12267035e+04  .40149807e+04  .35428160e-02", 
" -.28454918e+02  .22212379e+03 -.11890820e+04  .38804601e+04  .34447560e-02", 
" -.28164305e+02  .21783214e+03 -.11615679e+04  .38381475e+04  .33418229e-02", 
" -.27882606e+02  .21337548e+03 -.11197979e+04  .36123711e+04  .32939509e-02", 
" -.27611027e+02  .20915093e+03 -.10820985e+04  .34242508e+04  .32932834e-02", 
" -.27348556e+02  .20503887e+03 -.10415361e+04  .31742873e+04  .32751600e-02", 
" -.27095606e+02  .20144015e+03 -.10204400e+04  .31593161e+04  .32359517e-02", 
" -.26850730e+02  .19785825e+03 -.99342580e+03  .30551595e+04  .31638069e-02", 
" -.26613206e+02  .19447673e+03 -.97356703e+03  .30567760e+04  .31325759e-02", 
" -.26381670e+02  .19113020e+03 -.94952786e+03  .29814634e+04  .30994610e-02", 
" -.26156571e+02  .18784065e+03 -.92353494e+03  .28690299e+04  .30247682e-02", 
" -.25937677e+02  .18469240e+03 -.89961706e+03  .27742367e+04  .30107167e-02", 
" -.25722966e+02  .18137087e+03 -.86456379e+03  .25443245e+04  .30146709e-02", 
" -.25514558e+02  .17823513e+03 -.83333827e+03  .23389191e+04  .30165887e-02", 
" -.25312082e+02  .17552089e+03 -.81950647e+03  .23498950e+04  .30203209e-02", 
" -.25113428e+02  .17259660e+03 -.79109717e+03  .21656124e+04  .29239781e-02", 
" -.24919348e+02  .16989453e+03 -.77162890e+03  .20995760e+04  .28789934e-02", 
" -.24729094e+02  .16725493e+03 -.75297299e+03  .20461632e+04  .28120355e-02", 
" -.24543369e+02  .16477440e+03 -.73851987e+03  .20283852e+04  .27750080e-02", 
" -.24361952e+02  .16236203e+03 -.72401261e+03  .20059111e+04  .27718739e-02", 
" -.24182667e+02  .15974686e+03 -.69876345e+03  .18548913e+04  .27434648e-02", 
" -.24007761e+02  .15739162e+03 -.68281251e+03  .18112779e+04  .27361765e-02", 
" -.23835683e+02  .15496276e+03 -.65991712e+03  .16691058e+04  .27243846e-02", 
" -.23666748e+02  .15263014e+03 -.64193342e+03  .16003756e+04  .27221638e-02", 
" -.23501720e+02  .15049098e+03 -.62842268e+03  .15658853e+04  .26462099e-02", 
" -.23339069e+02  .14834179e+03 -.61278134e+03  .14983571e+04  .25953640e-02", 
" -.23179217e+02  .14633283e+03 -.60380328e+03  .15317154e+04  .25691667e-02", 
" -.23021797e+02  .14421918e+03 -.58576186e+03  .14263298e+04  .25733147e-02", 
" -.22867254e+02  .14220909e+03 -.56988589e+03  .13378008e+04  .25570886e-02", 
" -.22715218e+02  .14038315e+03 -.56403726e+03  .13964711e+04  .25331608e-02", 
" -.22565755e+02  .13850860e+03 -.55324788e+03  .13847332e+04  .25328007e-02", 
" -.22417842e+02  .13656214e+03 -.53755756e+03  .13067308e+04  .25194699e-02", 
" -.22271906e+02  .13478179e+03 -.52856133e+03  .13050635e+04  .24782201e-02", 
" -.22128344e+02  .13298035e+03 -.51669545e+03  .12714592e+04  .24591287e-02", 
" -.21985881e+02  .13104954e+03 -.49762055e+03  .11412415e+04  .24505447e-02", 
" -.21845984e+02  .12931279e+03 -.48610383e+03  .10980834e+04  .24828847e-02", 
" -.21707897e+02  .12762414e+03 -.47733746e+03  .11020863e+04  .24604091e-02", 
" -.21571532e+02  .12590529e+03 -.46391106e+03  .10317565e+04  .24538485e-02", 
" -.21437028e+02  .12423660e+03 -.45177277e+03  .97598034e+03  .24469628e-02", 
" -.21303935e+02  .12258821e+03 -.43894600e+03  .89833066e+03  .24247866e-02", 
" -.21173305e+02  .12113203e+03 -.43390850e+03  .91848912e+03  .23921998e-02", 
" -.21043228e+02  .11950327e+03 -.42157030e+03  .86276425e+03  .23772861e-02", 
" -.20914950e+02  .11794987e+03 -.40994625e+03  .80026166e+03  .23380913e-02", 
" -.20788243e+02  .11647975e+03 -.40152963e+03  .77874346e+03  .23373412e-02", 
" -.20662233e+02  .11496079e+03 -.39139344e+03  .74058772e+03  .23327938e-02", 
" -.20537716e+02  .11342685e+03 -.37904292e+03  .67116431e+03  .23237922e-02", 
" -.20414081e+02  .11183571e+03 -.36341718e+03  .56786695e+03  .23379487e-02", 
" -.20292656e+02  .11047979e+03 -.35817742e+03  .58628368e+03  .23130474e-02", 
" -.20172438e+02  .10916421e+03 -.35404874e+03  .61861363e+03  .22966225e-02", 
" -.20053036e+02  .10783030e+03 -.34819902e+03  .63035462e+03  .22717797e-02", 
" -.19934699e+02  .10651137e+03 -.34300963e+03  .65030231e+03  .22681404e-02", 
" -.19817580e+02  .10515664e+03 -.33448562e+03  .62523936e+03  .22629239e-02", 
" -.19700964e+02  .10373390e+03 -.32009515e+03  .50293555e+03  .22414782e-02", 
" -.19585645e+02  .10240126e+03 -.31117625e+03  .47215598e+03  .22377188e-02", 
" -.19471420e+02  .10114194e+03 -.30450637e+03  .45563799e+03  .22433139e-02", 
" -.19358437e+02  .99929188e+02 -.29980459e+03  .46850317e+03  .22420792e-02", 
" -.19246299e+02  .98714995e+02 -.29443584e+03  .47352486e+03  .22160190e-02", 
" -.19134753e+02  .97439680e+02 -.28588774e+03  .44137510e+03  .21754901e-02", 
" -.19023951e+02  .96160250e+02 -.27627087e+03  .39182805e+03  .21653948e-02", 
" -.18914513e+02  .94953560e+02 -.26801129e+03  .34938223e+03  .21576466e-02", 
" -.18805317e+02  .93680530e+02 -.25677213e+03  .26984302e+03  .21459238e-02", 
" -.18697227e+02  .92538655e+02 -.25304816e+03  .29931818e+03  .21609395e-02", 
" -.18589717e+02  .91366807e+02 -.24530006e+03  .25930489e+03  .21445554e-02", 
" -.18482419e+02  .90064718e+02 -.23186217e+03  .15736945e+03  .21472553e-02", 
" -.18375629e+02  .88796637e+02 -.22064173e+03  .87258292e+02  .21580946e-02", 
" -.18269679e+02  .87560926e+02 -.20982808e+03  .16074476e+02  .21614158e-02", 
" -.18164916e+02  .86466853e+02 -.20478624e+03  .10933940e+02  .21485092e-02", 
" -.18061406e+02  .85427456e+02 -.20107935e+03  .20304247e+02  .21334592e-02", 
" -.17958594e+02  .84423406e+02 -.19819356e+03  .33702550e+02  .21146261e-02", 
" -.17855917e+02  .83364711e+02 -.19336116e+03  .31774571e+02  .20815056e-02", 
" -.17753641e+02  .82251572e+02 -.18511306e+03 -.18866621e+02  .20850550e-02", 
" -.17651773e+02  .81142107e+02 -.17713186e+03 -.61703107e+02  .20951503e-02", 
" -.17551003e+02  .80171496e+02 -.17551943e+03 -.24652552e+02  .20771914e-02", 
" -.17450370e+02  .79100169e+02 -.16863580e+03 -.53668441e+02  .20471723e-02", 
" -.17350097e+02  .78090624e+02 -.16496745e+03 -.39985632e+02  .20248270e-02", 
" -.17250346e+02  .77054902e+02 -.15907174e+03 -.59606951e+02  .20100977e-02", 
" -.17151074e+02  .76052503e+02 -.15424435e+03 -.62434571e+02  .19960760e-02", 
" -.17052218e+02  .75066688e+02 -.14990732e+03 -.68407600e+02  .19960427e-02", 
" -.16953370e+02  .73986172e+02 -.14071990e+03 -.13089713e+03  .20016849e-02", 
" -.16855415e+02  .72970128e+02 -.13452274e+03 -.15402459e+03  .19995360e-02", 
" -.16757888e+02  .72042248e+02 -.13254320e+03 -.12212265e+03  .19913943e-02", 
" -.16661068e+02  .71116541e+02 -.12879567e+03 -.12374746e+03  .20136880e-02", 
" -.16564525e+02  .70161026e+02 -.12334933e+03 -.15215930e+03  .19975772e-02", 
" -.16468117e+02  .69254584e+02 -.12153177e+03 -.12208270e+03  .19655758e-02", 
" -.16372048e+02  .68277488e+02 -.11401005e+03 -.18255539e+03  .19524403e-02", 
" -.16276271e+02  .67285558e+02 -.10706798e+03 -.21841992e+03  .19497238e-02", 
" -.16181094e+02  .66371379e+02 -.10290718e+03 -.22463860e+03  .19388714e-02", 
" -.16085879e+02  .65411988e+02 -.96939108e+02 -.24869281e+03  .19306659e-02", 
" -.15990945e+02  .64458377e+02 -.89742221e+02 -.29962033e+03  .19155481e-02", 
" -.15897057e+02  .63636640e+02 -.89221359e+02 -.25981054e+03  .19009565e-02", 
" -.15802582e+02  .62775883e+02 -.87873908e+02 -.22640253e+03  .18604416e-02", 
" -.15708248e+02  .61876172e+02 -.83928576e+02 -.22881725e+03  .18823792e-02", 
" -.15614551e+02  .61036094e+02 -.81642643e+02 -.21933475e+03  .18559556e-02", 
" -.15520558e+02  .60104102e+02 -.75567900e+02 -.25131529e+03  .18813395e-02", 
" -.15426808e+02  .59145291e+02 -.67290080e+02 -.31677373e+03  .18480550e-02", 
" -.15333886e+02  .58331109e+02 -.65806728e+02 -.29154214e+03  .18413013e-02", 
" -.15240409e+02  .57397671e+02 -.59224736e+02 -.32622040e+03  .18307350e-02", 
" -.15147264e+02  .56499401e+02 -.52875730e+02 -.37118260e+03  .18103168e-02", 
" -.15054287e+02  .55578690e+02 -.45979189e+02 -.41231849e+03  .18018279e-02", 
" -.14961387e+02  .54724761e+02 -.42398765e+02 -.41350696e+03  .17977275e-02", 
" -.14868634e+02  .53862121e+02 -.37295450e+02 -.43996324e+03  .17970372e-02", 
" -.14775957e+02  .53059180e+02 -.35929683e+02 -.41704033e+03  .17867933e-02", 
" -.14683547e+02  .52235064e+02 -.32237066e+02 -.42782834e+03  .17621306e-02", 
" -.14590264e+02  .51252375e+02 -.23225694e+02 -.48419507e+03  .17221424e-02", 
" -.14497736e+02  .50391455e+02 -.18047433e+02 -.51141358e+03  .17161055e-02", 
" -.14404765e+02  .49423185e+02 -.81259996e+01 -.59132745e+03  .17031488e-02", 
" -.14312352e+02  .48569163e+02 -.27321081e+01 -.62263155e+03  .16903715e-02", 
" -.14219868e+02  .47708189e+02  .32146303e+01 -.66156429e+03  .16984340e-02", 
" -.14127714e+02  .46909510e+02  .61897416e+01 -.66172478e+03  .16869366e-02", 
" -.14035438e+02  .46104534e+02  .98904787e+01 -.67403959e+03  .16823815e-02", 
" -.13942945e+02  .45300308e+02  .12880251e+02 -.67150037e+03  .16751195e-02", 
" -.13850392e+02  .44500293e+02  .15579753e+02 -.66084409e+03  .16632844e-02", 
" -.13757404e+02  .43655840e+02  .19675786e+02 -.66622278e+03  .16532764e-02", 
" -.13664511e+02  .42806942e+02  .25933368e+02 -.71498290e+03  .16528998e-02", 
" -.13571763e+02  .42015760e+02  .28594118e+02 -.70900059e+03  .16492355e-02", 
" -.13478606e+02  .41194095e+02  .32707777e+02 -.72389357e+03  .16639281e-02", 
" -.13385734e+02  .40487160e+02  .31666829e+02 -.67172064e+03  .16553652e-02", 
" -.13292643e+02  .39714691e+02  .33480004e+02 -.65407990e+03  .16639123e-02", 
" -.13198868e+02  .38885463e+02  .38031269e+02 -.67234926e+03  .16742469e-02", 
" -.13105537e+02  .38133579e+02  .40426681e+02 -.67261707e+03  .16785017e-02", 
" -.13011520e+02  .37311957e+02  .44495933e+02 -.68183002e+03  .16705743e-02", 
" -.12917001e+02  .36450898e+02  .49994941e+02 -.70257958e+03  .16738827e-02", 
" -.12822369e+02  .35653948e+02  .52803499e+02 -.69416156e+03  .16899750e-02", 
" -.12727204e+02  .34813675e+02  .58303212e+02 -.72434085e+03  .16743813e-02", 
" -.12631666e+02  .33946937e+02  .65541633e+02 -.78556298e+03  .16735420e-02", 
" -.12536178e+02  .33080974e+02  .73120287e+02 -.84770827e+03  .16860748e-02", 
" -.12440411e+02  .32326464e+02  .74485617e+02 -.82601238e+03  .16743062e-02", 
" -.12343975e+02  .31488947e+02  .80581274e+02 -.87371755e+03  .16822809e-02", 
" -.12246439e+02  .30644895e+02  .85302054e+02 -.89514557e+03  .16719119e-02", 
" -.12149227e+02  .29884893e+02  .87133252e+02 -.88638154e+03  .16524339e-02", 
" -.12051698e+02  .29105884e+02  .90156992e+02 -.89307149e+03  .16489935e-02", 
" -.11953221e+02  .28280297e+02  .94168620e+02 -.89982114e+03  .16441861e-02", 
" -.11855143e+02  .27586016e+02  .93675938e+02 -.86400224e+03  .16566135e-02", 
" -.11755630e+02  .26764556e+02  .98068975e+02 -.88536986e+03  .16672002e-02", 
" -.11655080e+02  .25895573e+02  .10374005e+03 -.91427588e+03  .16829245e-02", 
" -.11554344e+02  .25034618e+02  .10938439e+03 -.94356747e+03  .16779183e-02", 
" -.11453481e+02  .24359313e+02  .10653759e+03 -.87465032e+03  .16581517e-02", 
" -.11351559e+02  .23582085e+02  .10931899e+03 -.88025795e+03  .16707829e-02", 
" -.11249263e+02  .22865384e+02  .10885182e+03 -.84533093e+03  .16707691e-02", 
" -.11145826e+02  .22083447e+02  .11098325e+03 -.83782747e+03  .16573841e-02", 
" -.11041293e+02  .21331905e+02  .10969053e+03 -.77658406e+03  .16638999e-02", 
" -.10935930e+02  .20571140e+02  .11047382e+03 -.75451285e+03  .16463962e-02", 
" -.10829606e+02  .19723288e+02  .11575801e+03 -.78979943e+03  .16420999e-02", 
" -.10722214e+02  .18882905e+02  .11962768e+03 -.79954503e+03  .16616323e-02", 
" -.10613250e+02  .18081165e+02  .12072294e+03 -.77569846e+03  .16579615e-02", 
" -.10503571e+02  .17264938e+02  .12355666e+03 -.78009709e+03  .16770251e-02", 
" -.10392071e+02  .16369333e+02  .12894725e+03 -.81323740e+03  .16975848e-02", 
" -.10279461e+02  .15520249e+02  .13150979e+03 -.80860866e+03  .16876643e-02", 
" -.10165151e+02  .14604816e+02  .13706128e+03 -.84170748e+03  .16679085e-02", 
" -.10050049e+02  .13807494e+02  .13684021e+03 -.80203515e+03  .16574860e-02", 
" -.99323045e+01  .12783179e+02  .14874857e+03 -.92737626e+03  .16710022e-02", 
" -.98128585e+01  .11862617e+02  .15334001e+03 -.94374636e+03  .16658600e-02", 
" -.96914402e+01  .10971870e+02  .15356081e+03 -.88885205e+03  .16846982e-02", 
" -.95680456e+01  .10087215e+02  .15535674e+03 -.87482890e+03  .16869925e-02", 
" -.94429710e+01  .91278926e+01  .16063050e+03 -.90165084e+03  .16301807e-02", 
" -.93150535e+01  .83023650e+01  .15734601e+03 -.81493012e+03  .16498456e-02", 
" -.91848205e+01  .73914796e+01  .15917699e+03 -.80659186e+03  .17025397e-02", 
" -.90519297e+01  .65112949e+01  .15715165e+03 -.72810401e+03  .17032305e-02", 
" -.89163340e+01  .56905034e+01  .15283810e+03 -.63951233e+03  .16943506e-02", 
" -.87762595e+01  .45792576e+01  .16237645e+03 -.73099878e+03  .16978082e-02", 
" -.86318690e+01  .33460173e+01  .17342233e+03 -.81110350e+03  .16767154e-02", 
" -.84836240e+01  .21881455e+01  .18070105e+03 -.85409022e+03  .16794078e-02", 
" -.83310067e+01  .11048113e+01  .18253313e+03 -.82886297e+03  .16923103e-02", 
" -.81734413e+01 -.48077237e-01  .18749578e+03 -.85119868e+03  .17161301e-02", 
" -.80116198e+01 -.11854528e+01  .19192275e+03 -.86656485e+03  .17532625e-02", 
" -.78421739e+01 -.24581211e+01  .19928186e+03 -.91682664e+03  .17751693e-02", 
" -.76659273e+01 -.37281646e+01  .20506798e+03 -.94850616e+03  .17858053e-02", 
" -.74807431e+01 -.51992387e+01  .21817212e+03 -.10731098e+04  .17841124e-02", 
" -.72865748e+01 -.65843666e+01  .22325627e+03 -.10842149e+04  .18026200e-02", 
" -.70800609e+01 -.82854157e+01  .24128713e+03 -.12723762e+04  .18213505e-02", 
" -.68587420e+01 -.10163059e+02  .25708176e+03 -.13803480e+04  .18731164e-02", 
" -.66225704e+01 -.12060512e+02  .27253757e+03 -.15121109e+04  .19278828e-02", 
" -.63637076e+01 -.14303453e+02  .29539290e+03 -.17322503e+04  .19661789e-02", 
" -.60795467e+01 -.16694692e+02  .31416368e+03 -.18605527e+04  .19960389e-02", 
" -.57602927e+01 -.19273253e+02  .33037733e+03 -.19631403e+04  .20662522e-02", 
" -.53919014e+01 -.22051401e+02  .32897399e+03 -.17529934e+04  .21907818e-02", 
" -.49449418e+01 -.26187459e+02  .36441899e+03 -.19995928e+04  .23487540e-02", 
" -.43621164e+01 -.32000049e+02  .41336318e+03 -.22815206e+04  .25953019e-02", 
" -.42196106e+01 -.33416209e+02  .42595615e+03 -.23589685e+04  .26518320e-02", 
" -.40628316e+01 -.34957189e+02  .43231089e+03 -.23104468e+04  .28022710e-02", 
" -.38889456e+01 -.36967604e+02  .45403028e+03 -.24507394e+04  .29798233e-02", 
" -.36939014e+01 -.39000906e+02  .46594872e+03 -.24602466e+04  .31613260e-02", 
" -.34680044e+01 -.41504810e+02  .48637313e+03 -.25716036e+04  .33986206e-02", 
" -.32039434e+01 -.44393586e+02  .50630119e+03 -.26315401e+04  .36734524e-02", 
" -.28764535e+01 -.48471575e+02  .55440626e+03 -.30532260e+04  .41026478e-02", 
" -.24359399e+01 -.53526105e+02  .56787659e+03 -.28021055e+04  .45404686e-02", 
" -.17451021e+01 -.60273305e+02  .50738634e+03 -.13804713e+04  .56338847e-02", 
" -.11010872e+01 -.69970572e+02  .60068217e+03 -.19436138e+04  .76032594e-02", 
" -.31913253e+00 -.79566325e+02  .54132931e+03 -.29831120e+03  .10899028e-01", 
"  .24227323e+00 -.90293597e+02  .65946463e+03 -.10975730e+04  .14334562e-01"
)


.urc3 = 
c("Copyright (C) James G. MacKinnon, 1995", "conc      2  3  2   25", 
" -.50589919e+01 -.26495143e+02 -.11718178e+03  .41322146e-02", 
" -.48983402e+01 -.24472782e+02 -.76892431e+02  .30877319e-02", 
" -.46833721e+01 -.20522627e+02 -.66905582e+02  .19643816e-02", 
" -.45109831e+01 -.18166007e+02 -.42686073e+02  .15997312e-02", 
" -.43280670e+01 -.15623926e+02 -.34778932e+02  .10899398e-02", 
" -.42168677e+01 -.14227922e+02 -.29106676e+02  .93724275e-03", 
" -.41353560e+01 -.13271504e+02 -.24354277e+02  .79263483e-03", 
" -.40705486e+01 -.12531111e+02 -.21699029e+02  .70178299e-03", 
" -.40168475e+01 -.11953183e+02 -.18901694e+02  .67631117e-03", 
" -.39701528e+01 -.11474087e+02 -.16397720e+02  .64882502e-03", 
" -.39298543e+01 -.11018887e+02 -.15342296e+02  .63773583e-03", 
" -.38933987e+01 -.10653465e+02 -.13635102e+02  .61441333e-03", 
" -.38604074e+01 -.10319575e+02 -.12640087e+02  .58339492e-03", 
" -.37297555e+01 -.89979620e+01 -.10866059e+02  .49717449e-03", 
" -.36330179e+01 -.81112405e+01 -.89907379e+01  .45966094e-03", 
" -.35551062e+01 -.74628021e+01 -.72237589e+01  .42175650e-03", 
" -.34897301e+01 -.69351339e+01 -.59536832e+01  .39587885e-03", 
" -.34331918e+01 -.64505032e+01 -.61551181e+01  .36758661e-03", 
" -.33826721e+01 -.61053516e+01 -.46677185e+01  .35428899e-03", 
" -.33374780e+01 -.57610836e+01 -.47018035e+01  .33360522e-03", 
" -.32963625e+01 -.54392586e+01 -.51177220e+01  .32005376e-03", 
" -.32582501e+01 -.51846940e+01 -.46002676e+01  .30951285e-03", 
" -.32230169e+01 -.49511702e+01 -.41102984e+01  .30939833e-03", 
" -.31899222e+01 -.47457038e+01 -.36697348e+01  .30580928e-03", 
" -.31588672e+01 -.45456627e+01 -.35300239e+01  .28991402e-03", 
" -.31294407e+01 -.43766670e+01 -.29897156e+01  .28748876e-03", 
" -.31015986e+01 -.42082497e+01 -.26793509e+01  .28150931e-03", 
" -.30750734e+01 -.40452402e+01 -.26374575e+01  .28377116e-03", 
" -.30497231e+01 -.38952425e+01 -.25833849e+01  .28064699e-03", 
" -.30254542e+01 -.37665270e+01 -.20256043e+01  .28211802e-03", 
" -.30020039e+01 -.36527313e+01 -.15133675e+01  .27424085e-03", 
" -.29795476e+01 -.35406814e+01 -.10252164e+01  .27487597e-03", 
" -.29579782e+01 -.34172907e+01 -.10929306e+01  .27058360e-03", 
" -.29370056e+01 -.33157541e+01 -.69100736e+00  .26390126e-03", 
" -.29167493e+01 -.32069513e+01 -.70135707e+00  .26111141e-03", 
" -.28970966e+01 -.31032314e+01 -.73108446e+00  .25733751e-03", 
" -.28780623e+01 -.30064519e+01 -.65449904e+00  .25293626e-03", 
" -.28594861e+01 -.29155834e+01 -.60394229e+00  .24709183e-03", 
" -.28413814e+01 -.28237005e+01 -.64818249e+00  .24544084e-03", 
" -.28237802e+01 -.27395499e+01 -.55739255e+00  .24138346e-03", 
" -.28065915e+01 -.26556113e+01 -.60881777e+00  .24001433e-03", 
" -.27897314e+01 -.25846114e+01 -.43265264e+00  .23792807e-03", 
" -.27733106e+01 -.25119582e+01 -.34262830e+00  .23819709e-03", 
" -.27573038e+01 -.24273704e+01 -.65903031e+00  .23508323e-03", 
" -.27414742e+01 -.23653041e+01 -.50257284e+00  .23151197e-03", 
" -.27260116e+01 -.22988196e+01 -.41945659e+00  .22627581e-03", 
" -.27109185e+01 -.22299753e+01 -.45242805e+00  .22239552e-03", 
" -.26960273e+01 -.21682593e+01 -.37374069e+00  .21861604e-03", 
" -.26814044e+01 -.21007948e+01 -.61085260e+00  .21833838e-03", 
" -.26670310e+01 -.20383624e+01 -.67740276e+00  .21588938e-03", 
" -.26529043e+01 -.19778685e+01 -.71793211e+00  .21371098e-03", 
" -.26389304e+01 -.19306368e+01 -.53869388e+00  .21387224e-03", 
" -.26252420e+01 -.18696395e+01 -.74889571e+00  .21346321e-03", 
" -.26117796e+01 -.18163383e+01 -.73611325e+00  .21028343e-03", 
" -.25984674e+01 -.17639318e+01 -.75797521e+00  .20831122e-03", 
" -.25853656e+01 -.17188487e+01 -.60687197e+00  .20608156e-03", 
" -.25723812e+01 -.16764070e+01 -.41711383e+00  .20316923e-03", 
" -.25595429e+01 -.16339159e+01 -.30497678e+00  .19940289e-03", 
" -.25469200e+01 -.15869848e+01 -.34251836e+00  .19884387e-03", 
" -.25344344e+01 -.15500223e+01 -.11572811e+00  .19811379e-03", 
" -.25221091e+01 -.15023011e+01 -.20202530e+00  .19973273e-03", 
" -.25099242e+01 -.14615134e+01 -.12724100e+00  .20003709e-03", 
" -.24978456e+01 -.14174174e+01 -.25605406e+00  .19838425e-03", 
" -.24858350e+01 -.13858186e+01 -.74485031e-01  .19711009e-03", 
" -.24740394e+01 -.13468554e+01 -.32816798e-01  .19431920e-03", 
" -.24623203e+01 -.13145137e+01  .91260476e-01  .19123628e-03", 
" -.24507709e+01 -.12693623e+01 -.13541066e+00  .18882445e-03", 
" -.24392751e+01 -.12367547e+01 -.25458933e-01  .18926359e-03", 
" -.24278772e+01 -.12060345e+01  .11340285e+00  .18939670e-03", 
" -.24165691e+01 -.11793926e+01  .38933507e+00  .18544057e-03", 
" -.24053560e+01 -.11524036e+01  .57989747e+00  .18351956e-03", 
" -.23942626e+01 -.11175048e+01  .50334405e+00  .18594818e-03", 
" -.23833178e+01 -.10742961e+01  .22530176e+00  .18714239e-03", 
" -.23724199e+01 -.10427443e+01  .25256223e+00  .18519342e-03", 
" -.23615728e+01 -.10160809e+01  .43607468e+00  .18312162e-03", 
" -.23508249e+01 -.98210443e+00  .34723230e+00  .18364041e-03", 
" -.23401711e+01 -.94581884e+00  .23265943e+00  .18235699e-03", 
" -.23295453e+01 -.92042680e+00  .33066301e+00  .17934991e-03", 
" -.23190393e+01 -.88999307e+00  .31090237e+00  .17655341e-03", 
" -.23085866e+01 -.86001687e+00  .27118522e+00  .17655094e-03", 
" -.22981222e+01 -.83680793e+00  .35321371e+00  .17697185e-03", 
" -.22878148e+01 -.80448964e+00  .24329476e+00  .17578483e-03", 
" -.22774979e+01 -.77947526e+00  .29699447e+00  .17598146e-03", 
" -.22672270e+01 -.75511573e+00  .29432827e+00  .17596160e-03", 
" -.22569919e+01 -.73574299e+00  .42498002e+00  .17499637e-03", 
" -.22468727e+01 -.70694557e+00  .30580474e+00  .17505339e-03", 
" -.22368144e+01 -.67648418e+00  .15516535e+00  .17544956e-03", 
" -.22267405e+01 -.65957725e+00  .36577522e+00  .17756915e-03", 
" -.22167105e+01 -.63908856e+00  .44890754e+00  .17619715e-03", 
" -.22067604e+01 -.61326741e+00  .39871979e+00  .17527404e-03", 
" -.21968252e+01 -.59361683e+00  .46814964e+00  .17569700e-03", 
" -.21869656e+01 -.57096566e+00  .47996473e+00  .17412230e-03", 
" -.21770619e+01 -.54926234e+00  .46760667e+00  .17429376e-03", 
" -.21672554e+01 -.52468493e+00  .37555299e+00  .17387836e-03", 
" -.21574722e+01 -.49608292e+00  .12786561e+00  .17425225e-03", 
" -.21477321e+01 -.47263151e+00  .86174213e-01  .17208429e-03", 
" -.21379254e+01 -.45550765e+00  .13710655e+00  .17185154e-03", 
" -.21282574e+01 -.43023813e+00 -.12778493e-02  .17208859e-03", 
" -.21185359e+01 -.41125835e+00 -.32227863e-01  .17270344e-03", 
" -.21088583e+01 -.39690392e+00  .79789332e-01  .17228443e-03", 
" -.20992171e+01 -.38002975e+00  .14396828e+00  .17350807e-03", 
" -.20895873e+01 -.36575835e+00  .27080749e+00  .17290418e-03", 
" -.20799847e+01 -.34442546e+00  .16316195e+00  .17412434e-03", 
" -.20703801e+01 -.32556087e+00  .52209173e-01  .17574715e-03", 
" -.20608277e+01 -.30143163e+00 -.10186259e+00  .17516654e-03", 
" -.20512311e+01 -.29069864e+00  .50213832e-01  .17562626e-03", 
" -.20416476e+01 -.27761458e+00  .13239530e+00  .17802874e-03", 
" -.20320714e+01 -.26557744e+00  .24041487e+00  .17730539e-03", 
" -.20225021e+01 -.25414977e+00  .34809926e+00  .17734815e-03", 
" -.20130081e+01 -.23740467e+00  .34274609e+00  .17569893e-03", 
" -.20035124e+01 -.22000151e+00  .30184979e+00  .17859270e-03", 
" -.19939722e+01 -.20571299e+00  .29765337e+00  .17907963e-03", 
" -.19844415e+01 -.19034277e+00  .27097579e+00  .17543431e-03", 
" -.19749212e+01 -.17954054e+00  .38000126e+00  .17650905e-03", 
" -.19653507e+01 -.17000692e+00  .48953598e+00  .17424260e-03", 
" -.19557503e+01 -.16039915e+00  .55527892e+00  .17275635e-03", 
" -.19461686e+01 -.15047086e+00  .62435477e+00  .17423190e-03", 
" -.19365876e+01 -.14281661e+00  .77006985e+00  .17392508e-03", 
" -.19270181e+01 -.12705166e+00  .66067068e+00  .17583437e-03", 
" -.19174414e+01 -.11607424e+00  .72541261e+00  .17545591e-03", 
" -.19078341e+01 -.11074634e+00  .95726275e+00  .17413692e-03", 
" -.18982317e+01 -.98157625e-01  .95166129e+00  .17054295e-03", 
" -.18886490e+01 -.80602735e-01  .79621469e+00  .17071222e-03", 
" -.18790406e+01 -.69292347e-01  .82417751e+00  .16923140e-03", 
" -.18693857e+01 -.59377113e-01  .85285955e+00  .16992688e-03", 
" -.18597103e+01 -.48604931e-01  .83824502e+00  .17220310e-03", 
" -.18500305e+01 -.31947426e-01  .67308832e+00  .16916068e-03", 
" -.18403123e+01 -.23029557e-01  .72158810e+00  .16932368e-03", 
" -.18305620e+01 -.14862577e-01  .74174683e+00  .17017988e-03", 
" -.18207460e+01 -.91192534e-02  .83777766e+00  .17193614e-03", 
" -.18109023e+01  .22499218e-03  .79953205e+00  .17297863e-03", 
" -.18011313e+01  .15549133e-01  .70147226e+00  .17388689e-03", 
" -.17912967e+01  .26926881e-01  .67776505e+00  .17265549e-03", 
" -.17813944e+01  .36056795e-01  .68165265e+00  .17179512e-03", 
" -.17714162e+01  .38147683e-01  .83161695e+00  .17258126e-03", 
" -.17614051e+01  .46344403e-01  .76282582e+00  .17343627e-03", 
" -.17513616e+01  .51669109e-01  .83910919e+00  .17473298e-03", 
" -.17412304e+01  .58270420e-01  .82501936e+00  .17252319e-03", 
" -.17310972e+01  .69015826e-01  .68866623e+00  .17426792e-03", 
" -.17209480e+01  .81163893e-01  .56735633e+00  .17440228e-03", 
" -.17107059e+01  .91098774e-01  .51263416e+00  .17789651e-03", 
" -.17004139e+01  .93814142e-01  .65616962e+00  .17889096e-03", 
" -.16900686e+01  .10258444e+00  .59754233e+00  .17603117e-03", 
" -.16796598e+01  .10789470e+00  .63836423e+00  .17686853e-03", 
" -.16691798e+01  .11440218e+00  .60952817e+00  .17658667e-03", 
" -.16586909e+01  .12255963e+00  .62254479e+00  .17665863e-03", 
" -.16480654e+01  .12458135e+00  .71986893e+00  .17672643e-03", 
" -.16373457e+01  .12401780e+00  .90157311e+00  .17668928e-03", 
" -.16265691e+01  .12513642e+00  .10137439e+01  .17946810e-03", 
" -.16157357e+01  .12815064e+00  .11266332e+01  .17839525e-03", 
" -.16047963e+01  .13318831e+00  .10763351e+01  .17672685e-03", 
" -.15937631e+01  .13746270e+00  .10655485e+01  .17620038e-03", 
" -.15826055e+01  .13884775e+00  .11339411e+01  .17536052e-03", 
" -.15713984e+01  .15159078e+00  .90543881e+00  .17492168e-03", 
" -.15601039e+01  .15758164e+00  .89074680e+00  .17513535e-03", 
" -.15486508e+01  .15701942e+00  .10239751e+01  .17681741e-03", 
" -.15371181e+01  .16269153e+00  .97372319e+00  .17776686e-03", 
" -.15254580e+01  .16281109e+00  .10942340e+01  .17606358e-03", 
" -.15136099e+01  .16036498e+00  .12503478e+01  .17312400e-03", 
" -.15016383e+01  .15922374e+00  .13118617e+01  .17631351e-03", 
" -.14895600e+01  .16062197e+00  .13276638e+01  .17961059e-03", 
" -.14773170e+01  .16086813e+00  .13227325e+01  .18032054e-03", 
" -.14649937e+01  .16881249e+00  .11650155e+01  .18262649e-03", 
" -.14524586e+01  .16969258e+00  .11516623e+01  .18723954e-03", 
" -.14397879e+01  .17751201e+00  .92220347e+00  .18913927e-03", 
" -.14268814e+01  .17207032e+00  .10803855e+01  .19447832e-03", 
" -.14137610e+01  .17276675e+00  .10370419e+01  .19554253e-03", 
" -.14005131e+01  .17888379e+00  .85839122e+00  .19805294e-03", 
" -.13870330e+01  .17288113e+00  .10219129e+01  .20066444e-03", 
" -.13733411e+01  .17575020e+00  .91057357e+00  .19930717e-03", 
" -.13593861e+01  .17271608e+00  .94356484e+00  .20166049e-03", 
" -.13452234e+01  .16983166e+00  .91697347e+00  .20038689e-03", 
" -.13307769e+01  .16391094e+00  .10439296e+01  .19996472e-03", 
" -.13160517e+01  .16368460e+00  .97399595e+00  .20183982e-03", 
" -.13010076e+01  .15597935e+00  .10530575e+01  .20457770e-03", 
" -.12856200e+01  .14640182e+00  .11749712e+01  .20801648e-03", 
" -.12699826e+01  .14373799e+00  .10926516e+01  .21469358e-03", 
" -.12539749e+01  .13471535e+00  .12458250e+01  .21584839e-03", 
" -.12375651e+01  .12600906e+00  .12998496e+01  .21772364e-03", 
" -.12206981e+01  .11794981e+00  .12598573e+01  .21857703e-03", 
" -.12033894e+01  .10127559e+00  .15515860e+01  .22076118e-03", 
" -.11855923e+01  .82578498e-01  .18471555e+01  .22571484e-03", 
" -.11673875e+01  .79489650e-01  .17060229e+01  .23075064e-03", 
" -.11485448e+01  .66601896e-01  .17334724e+01  .23006528e-03", 
" -.11291711e+01  .57089138e-01  .17237661e+01  .23262433e-03", 
" -.11091226e+01  .41386784e-01  .19071818e+01  .23712959e-03", 
" -.10884295e+01  .41307751e-01  .15455152e+01  .24899091e-03", 
" -.10667689e+01  .12904492e-01  .19127570e+01  .25477320e-03", 
" -.10444366e+01  .14657031e-03  .18270853e+01  .26384438e-03", 
" -.10211062e+01 -.23769076e-01  .20257108e+01  .27048689e-03", 
" -.99673350e+00 -.38681445e-01  .19785817e+01  .28062147e-03", 
" -.97124849e+00 -.67849026e-01  .23707301e+01  .28410071e-03", 
" -.94454749e+00 -.74163838e-01  .20927073e+01  .29341474e-03", 
" -.91636780e+00 -.89095536e-01  .20130603e+01  .30345754e-03", 
" -.88633966e+00 -.11787075e+00  .20480833e+01  .30321796e-03", 
" -.85458007e+00 -.14064398e+00  .20038434e+01  .31367336e-03", 
" -.82068206e+00 -.15974993e+00  .18377845e+01  .32970922e-03", 
" -.78439931e+00 -.17009440e+00  .14404634e+01  .34829568e-03", 
" -.74504974e+00 -.20505955e+00  .15463041e+01  .36629016e-03", 
" -.70237436e+00 -.21527341e+00  .86341810e+00  .38105712e-03", 
" -.65549774e+00 -.27346751e+00  .15110922e+01  .39882705e-03", 
" -.60333263e+00 -.30277583e+00  .10689268e+01  .42262413e-03", 
" -.54506130e+00 -.35587788e+00  .15732044e+01  .44899047e-03", 
" -.47895350e+00 -.39634403e+00  .14791054e+01  .47950609e-03", 
" -.40307350e+00 -.42688967e+00  .15965011e+01  .51523489e-03", 
" -.31363431e+00 -.45407941e+00  .17202712e+01  .55868376e-03", 
" -.20463888e+00 -.49181143e+00  .18121282e+01  .62206931e-03", 
" -.66864145e-01 -.46544980e+00  .15068807e+01  .69567997e-03", 
"  .12345578e+00 -.39826695e+00  .61954973e+00  .79335134e-03", 
"  .17216331e+00 -.39782307e+00  .84833701e+00  .83351083e-03", 
"  .22632781e+00 -.38217847e+00  .88131082e+00  .89245615e-03", 
"  .28609669e+00 -.26312152e+00 -.11081794e+01  .94410645e-03", 
"  .35547304e+00 -.22553542e+00 -.85345535e+00  .98345192e-03", 
"  .43611139e+00 -.19683257e+00  .43656226e+00  .10321455e-02", 
"  .53348882e+00 -.17638551e+00  .24824970e+01  .12113065e-02", 
"  .65696913e+00 -.62694636e-01  .25923192e+01  .13620352e-02", 
"  .82445799e+00  .34698775e+00 -.21038658e+01  .15961270e-02", 
"  .10986458e+01  .87894644e+00 -.15747739e+01  .22166694e-02", 
"  .13626365e+01  .91170359e+00  .14246860e+02  .30280825e-02", 
"  .16841971e+01  .21256478e+01  .15030068e+02  .43042000e-02", 
"  .19127723e+01  .31583542e+01  .11216357e+02  .58342854e-02", 
"coc       2  4  2   20", " -.54704599e+01 -.32862413e+02 -.17688385e+03  .39964604e-02", 
" -.53182081e+01 -.30115909e+02 -.14437756e+03  .29945273e-02", 
" -.51008846e+01 -.26675629e+02 -.10200120e+03  .19401226e-02", 
" -.49274093e+01 -.24045351e+02 -.76641680e+02  .14699784e-02", 
" -.47505818e+01 -.20986138e+02 -.60551661e+02  .11046519e-02", 
" -.46420754e+01 -.19180484e+02 -.54079433e+02  .91806117e-03", 
" -.45628808e+01 -.17969501e+02 -.48939409e+02  .78277425e-03", 
" -.44992703e+01 -.17129179e+02 -.43031737e+02  .69537784e-03", 
" -.44465595e+01 -.16449572e+02 -.38908288e+02  .67703156e-03", 
" -.44013124e+01 -.15824408e+02 -.36823472e+02  .64210526e-03", 
" -.43609862e+01 -.15355916e+02 -.33706775e+02  .60059943e-03", 
" -.43250566e+01 -.14935744e+02 -.31018147e+02  .58676570e-03", 
" -.42923884e+01 -.14589908e+02 -.28225383e+02  .56050569e-03", 
" -.41646094e+01 -.12994897e+02 -.22721292e+02  .46313088e-03", 
" -.40700963e+01 -.11892410e+02 -.19216734e+02  .42442424e-03", 
" -.39937926e+01 -.11079956e+02 -.16203331e+02  .37907019e-03", 
" -.39296393e+01 -.10426564e+02 -.14055271e+02  .34584721e-03", 
" -.38740201e+01 -.98923908e+01 -.11744936e+02  .33116809e-03", 
" -.38246884e+01 -.94261434e+01 -.10220512e+02  .31021992e-03", 
" -.37804713e+01 -.89913343e+01 -.94194434e+01  .29654561e-03", 
" -.37401661e+01 -.85983045e+01 -.87599040e+01  .28687402e-03", 
" -.37027521e+01 -.82714549e+01 -.79850786e+01  .28002413e-03", 
" -.36681191e+01 -.79700820e+01 -.72513419e+01  .26354354e-03", 
" -.36357797e+01 -.76906712e+01 -.65917114e+01  .25770460e-03", 
" -.36052686e+01 -.74314594e+01 -.61659898e+01  .25665053e-03", 
" -.35765699e+01 -.71899388e+01 -.56620361e+01  .25345830e-03", 
" -.35491642e+01 -.69847806e+01 -.49036829e+01  .25425428e-03", 
" -.35231695e+01 -.67796056e+01 -.44768504e+01  .24705035e-03", 
" -.34983776e+01 -.65919712e+01 -.39318382e+01  .24306966e-03", 
" -.34745291e+01 -.64156250e+01 -.34403647e+01  .23480924e-03", 
" -.34516387e+01 -.62395597e+01 -.32362775e+01  .23096116e-03", 
" -.34296512e+01 -.60686285e+01 -.31153086e+01  .22775256e-03", 
" -.34083177e+01 -.59136362e+01 -.29013849e+01  .22181780e-03", 
" -.33877831e+01 -.57602630e+01 -.28423049e+01  .22053483e-03", 
" -.33678676e+01 -.56264517e+01 -.24375724e+01  .21877592e-03", 
" -.33485667e+01 -.54897297e+01 -.22933585e+01  .21651849e-03", 
" -.33297443e+01 -.53625601e+01 -.21071826e+01  .21201054e-03", 
" -.33114737e+01 -.52421104e+01 -.18688313e+01  .20990723e-03", 
" -.32937260e+01 -.51157432e+01 -.18793190e+01  .20791307e-03", 
" -.32764274e+01 -.49980928e+01 -.18247728e+01  .20469503e-03", 
" -.32595241e+01 -.48906802e+01 -.16053035e+01  .19819770e-03", 
" -.32430770e+01 -.47849436e+01 -.14065715e+01  .20026518e-03", 
" -.32269082e+01 -.46822093e+01 -.13046868e+01  .20021115e-03", 
" -.32111263e+01 -.45864018e+01 -.11126070e+01  .19635886e-03", 
" -.31955977e+01 -.44985352e+01 -.83869445e+00  .19522325e-03", 
" -.31804237e+01 -.44048165e+01 -.74149363e+00  .19017712e-03", 
" -.31655203e+01 -.43117337e+01 -.69242586e+00  .18931667e-03", 
" -.31508662e+01 -.42274261e+01 -.56815619e+00  .18926043e-03", 
" -.31365257e+01 -.41415273e+01 -.49249921e+00  .18500137e-03", 
" -.31223075e+01 -.40689250e+01 -.23859107e+00  .18331660e-03", 
" -.31083263e+01 -.39979250e+01  .52411103e-01  .17962659e-03", 
" -.30946792e+01 -.39201755e+01  .15519523e+00  .18124981e-03", 
" -.30812124e+01 -.38424826e+01  .19323871e+00  .17902877e-03", 
" -.30679281e+01 -.37711368e+01  .35719783e+00  .17537526e-03", 
" -.30548384e+01 -.36974526e+01  .34887147e+00  .17643314e-03", 
" -.30418895e+01 -.36314243e+01  .48492607e+00  .17552873e-03", 
" -.30291488e+01 -.35620068e+01  .54367664e+00  .17782011e-03", 
" -.30165598e+01 -.35023149e+01  .77178589e+00  .17395896e-03", 
" -.30041501e+01 -.34372502e+01  .85994031e+00  .17450092e-03", 
" -.29918972e+01 -.33706356e+01  .92364057e+00  .17307082e-03", 
" -.29797156e+01 -.33123811e+01  .10726521e+01  .17486316e-03", 
" -.29676658e+01 -.32544544e+01  .12127460e+01  .17509141e-03", 
" -.29557758e+01 -.31971591e+01  .12946795e+01  .17560610e-03", 
" -.29440558e+01 -.31340247e+01  .12232142e+01  .17427103e-03", 
" -.29324474e+01 -.30736903e+01  .11990538e+01  .17363523e-03", 
" -.29209405e+01 -.30159607e+01  .11984932e+01  .17371622e-03", 
" -.29095486e+01 -.29562259e+01  .11011795e+01  .17326119e-03", 
" -.28982240e+01 -.29069607e+01  .12380486e+01  .17082282e-03", 
" -.28870261e+01 -.28572262e+01  .13314702e+01  .16878306e-03", 
" -.28759014e+01 -.28088582e+01  .14224668e+01  .16707571e-03", 
" -.28648605e+01 -.27641303e+01  .15951407e+01  .16731985e-03", 
" -.28539402e+01 -.27146237e+01  .16036766e+01  .16785315e-03", 
" -.28431219e+01 -.26593421e+01  .14929438e+01  .16862801e-03", 
" -.28323563e+01 -.26123469e+01  .15713000e+01  .16825522e-03", 
" -.28216470e+01 -.25667898e+01  .15989666e+01  .16533440e-03", 
" -.28110741e+01 -.25169103e+01  .15257686e+01  .16577789e-03", 
" -.28005032e+01 -.24767866e+01  .16545736e+01  .16583516e-03", 
" -.27900961e+01 -.24272682e+01  .16025020e+01  .16346734e-03", 
" -.27796991e+01 -.23830999e+01  .16307535e+01  .16458059e-03", 
" -.27693764e+01 -.23401822e+01  .16604430e+01  .16588304e-03", 
" -.27591266e+01 -.22960532e+01  .16714157e+01  .16536224e-03", 
" -.27489480e+01 -.22536023e+01  .16673521e+01  .16492388e-03", 
" -.27387556e+01 -.22123172e+01  .16340473e+01  .16177352e-03", 
" -.27286857e+01 -.21699093e+01  .15943718e+01  .16033041e-03", 
" -.27186315e+01 -.21303487e+01  .16060826e+01  .16037973e-03", 
" -.27086665e+01 -.20869260e+01  .15555380e+01  .15962272e-03", 
" -.26986905e+01 -.20488421e+01  .15590626e+01  .15704779e-03", 
" -.26887870e+01 -.20124031e+01  .16033421e+01  .15677154e-03", 
" -.26789405e+01 -.19726430e+01  .15422066e+01  .15598294e-03", 
" -.26691023e+01 -.19356880e+01  .15740259e+01  .15667903e-03", 
" -.26593046e+01 -.18986737e+01  .15668616e+01  .15578822e-03", 
" -.26495414e+01 -.18636679e+01  .15790347e+01  .15605720e-03", 
" -.26398452e+01 -.18276443e+01  .15945290e+01  .15649393e-03", 
" -.26301456e+01 -.17939847e+01  .16034525e+01  .15640532e-03", 
" -.26204800e+01 -.17583903e+01  .15772136e+01  .15853575e-03", 
" -.26108501e+01 -.17256297e+01  .15950633e+01  .15714137e-03", 
" -.26012378e+01 -.16938667e+01  .16110992e+01  .15721443e-03", 
" -.25916556e+01 -.16617076e+01  .16152078e+01  .15659309e-03", 
" -.25820999e+01 -.16308831e+01  .16474136e+01  .15697709e-03", 
" -.25725224e+01 -.16063481e+01  .17980618e+01  .15791967e-03", 
" -.25630523e+01 -.15715979e+01  .17323945e+01  .15880936e-03", 
" -.25535330e+01 -.15448717e+01  .18195319e+01  .16004472e-03", 
" -.25440619e+01 -.15155908e+01  .18673732e+01  .15931092e-03", 
" -.25346230e+01 -.14791982e+01  .17127003e+01  .15939883e-03", 
" -.25251517e+01 -.14517366e+01  .17390801e+01  .15875284e-03", 
" -.25157127e+01 -.14253812e+01  .18387539e+01  .15799703e-03", 
" -.25062760e+01 -.13945569e+01  .17727328e+01  .15806610e-03", 
" -.24968874e+01 -.13628557e+01  .17326824e+01  .15837445e-03", 
" -.24874707e+01 -.13361545e+01  .17539390e+01  .15728232e-03", 
" -.24780509e+01 -.13093063e+01  .17513406e+01  .15703970e-03", 
" -.24686272e+01 -.12836109e+01  .17772567e+01  .15651847e-03", 
" -.24592077e+01 -.12599297e+01  .18605653e+01  .15526347e-03", 
" -.24498172e+01 -.12339174e+01  .18730225e+01  .15675072e-03", 
" -.24404044e+01 -.12092059e+01  .18874886e+01  .16115713e-03", 
" -.24309926e+01 -.11828629e+01  .18830812e+01  .16221167e-03", 
" -.24215796e+01 -.11600936e+01  .19441851e+01  .16324188e-03", 
" -.24121327e+01 -.11392780e+01  .20260126e+01  .16228345e-03", 
" -.24027411e+01 -.11121826e+01  .19981810e+01  .16265712e-03", 
" -.23933023e+01 -.10895727e+01  .20305906e+01  .16256800e-03", 
" -.23838824e+01 -.10647844e+01  .20313126e+01  .16176113e-03", 
" -.23744277e+01 -.10396204e+01  .20075090e+01  .16013721e-03", 
" -.23649211e+01 -.10210562e+01  .21246872e+01  .16195813e-03", 
" -.23554505e+01 -.99672812e+00  .21023771e+01  .16031584e-03", 
" -.23459841e+01 -.97035543e+00  .20420118e+01  .16023743e-03", 
" -.23364493e+01 -.95172903e+00  .21308781e+01  .16116145e-03", 
" -.23269041e+01 -.93001181e+00  .21265216e+01  .16323198e-03", 
" -.23173645e+01 -.90676952e+00  .21335086e+01  .16416472e-03", 
" -.23077732e+01 -.88498513e+00  .21098679e+01  .16461722e-03", 
" -.22981343e+01 -.86648339e+00  .21642498e+01  .16268628e-03", 
" -.22884688e+01 -.84666872e+00  .21931071e+01  .16228834e-03", 
" -.22787720e+01 -.82840146e+00  .22476389e+01  .16192188e-03", 
" -.22690403e+01 -.80644401e+00  .21753180e+01  .16223710e-03", 
" -.22593599e+01 -.77955294e+00  .20363817e+01  .16142439e-03", 
" -.22496241e+01 -.75710596e+00  .19948210e+01  .15989182e-03", 
" -.22398377e+01 -.72949860e+00  .18092696e+01  .15940748e-03", 
" -.22299707e+01 -.71114529e+00  .18021188e+01  .16033966e-03", 
" -.22200448e+01 -.69583216e+00  .18646854e+01  .15989474e-03", 
" -.22101043e+01 -.67619236e+00  .18153598e+01  .15950852e-03", 
" -.22001579e+01 -.65120265e+00  .16616184e+01  .15885966e-03", 
" -.21900929e+01 -.63602201e+00  .17159691e+01  .15843243e-03", 
" -.21799898e+01 -.62202708e+00  .18190783e+01  .15836093e-03", 
" -.21698548e+01 -.60187661e+00  .17614557e+01  .15948717e-03", 
" -.21596795e+01 -.57926492e+00  .16248778e+01  .16006739e-03", 
" -.21494335e+01 -.56207056e+00  .16435300e+01  .15983964e-03", 
" -.21391252e+01 -.54224503e+00  .15587454e+01  .16084080e-03", 
" -.21287298e+01 -.52679723e+00  .15752048e+01  .16098069e-03", 
" -.21183290e+01 -.50539435e+00  .14742567e+01  .16169137e-03", 
" -.21077631e+01 -.49309534e+00  .15197030e+01  .16422908e-03", 
" -.20971301e+01 -.47909926e+00  .15604207e+01  .16252172e-03", 
" -.20864209e+01 -.46289662e+00  .15262217e+01  .16296947e-03", 
" -.20756422e+01 -.44548554e+00  .14617986e+01  .16247248e-03", 
" -.20647523e+01 -.42955263e+00  .14208668e+01  .16523163e-03", 
" -.20537879e+01 -.41792534e+00  .15021526e+01  .16464470e-03", 
" -.20427387e+01 -.40312389e+00  .14746318e+01  .16631333e-03", 
" -.20315758e+01 -.38898080e+00  .14365173e+01  .16598944e-03", 
" -.20203155e+01 -.37888610e+00  .15128601e+01  .16852823e-03", 
" -.20089079e+01 -.37007218e+00  .15995820e+01  .16857788e-03", 
" -.19974076e+01 -.35970409e+00  .16457369e+01  .16958560e-03", 
" -.19857894e+01 -.34792599e+00  .16444414e+01  .17143504e-03", 
" -.19740337e+01 -.33735943e+00  .16579899e+01  .17224891e-03", 
" -.19621568e+01 -.32448000e+00  .16280564e+01  .17438828e-03", 
" -.19502020e+01 -.30817574e+00  .15659549e+01  .17334358e-03", 
" -.19380220e+01 -.29796292e+00  .15787455e+01  .17621592e-03", 
" -.19257209e+01 -.28668558e+00  .15376680e+01  .17785778e-03", 
" -.19132685e+01 -.27496232e+00  .15148250e+01  .17957186e-03", 
" -.19006764e+01 -.26482481e+00  .15224095e+01  .17991154e-03", 
" -.18879206e+01 -.25109163e+00  .14426376e+01  .18322462e-03", 
" -.18748834e+01 -.24499342e+00  .15030597e+01  .18535328e-03", 
" -.18616657e+01 -.23615012e+00  .15053160e+01  .18740815e-03", 
" -.18483309e+01 -.22136741e+00  .14038468e+01  .18881507e-03", 
" -.18347034e+01 -.21416754e+00  .14214237e+01  .18924273e-03", 
" -.18208185e+01 -.21438561e+00  .16127419e+01  .18805335e-03", 
" -.18066915e+01 -.20909570e+00  .16438774e+01  .19101799e-03", 
" -.17922876e+01 -.20486326e+00  .16781444e+01  .19136271e-03", 
" -.17776094e+01 -.20203792e+00  .17252261e+01  .19014597e-03", 
" -.17626191e+01 -.20076264e+00  .18132046e+01  .19049049e-03", 
" -.17473627e+01 -.19558531e+00  .18276450e+01  .19072627e-03", 
" -.17317667e+01 -.18919548e+00  .17949955e+01  .19431778e-03", 
" -.17158223e+01 -.18240384e+00  .17198568e+01  .19408854e-03", 
" -.16995410e+01 -.17263673e+00  .15834623e+01  .19791974e-03", 
" -.16828064e+01 -.16854489e+00  .15545310e+01  .20185414e-03", 
" -.16655928e+01 -.16590921e+00  .15075094e+01  .20185791e-03", 
" -.16478631e+01 -.17544752e+00  .17431513e+01  .20403928e-03", 
" -.16297001e+01 -.17228025e+00  .16782930e+01  .20748942e-03", 
" -.16109139e+01 -.17392768e+00  .16519460e+01  .21548033e-03", 
" -.15914920e+01 -.18187237e+00  .17145369e+01  .22063263e-03", 
" -.15714715e+01 -.18876613e+00  .17897954e+01  .22459717e-03", 
" -.15506846e+01 -.20250169e+00  .20021050e+01  .22575342e-03", 
" -.15291891e+01 -.21140994e+00  .20951721e+01  .23401765e-03", 
" -.15069069e+01 -.20347094e+00  .17885926e+01  .23678410e-03", 
" -.14836237e+01 -.20884188e+00  .17360618e+01  .24736060e-03", 
" -.14591939e+01 -.22227091e+00  .18240756e+01  .25002630e-03", 
" -.14337193e+01 -.22581095e+00  .17081065e+01  .25300226e-03", 
" -.14066769e+01 -.25589709e+00  .20329647e+01  .25949008e-03", 
" -.13782461e+01 -.27422833e+00  .21148029e+01  .26945071e-03", 
" -.13481660e+01 -.28966593e+00  .20550904e+01  .28237276e-03", 
" -.13160856e+01 -.30318862e+00  .19026636e+01  .29046061e-03", 
" -.12817945e+01 -.33260415e+00  .20474595e+01  .30182528e-03", 
" -.12446816e+01 -.37036866e+00  .22679830e+01  .31902307e-03", 
" -.12045610e+01 -.41658439e+00  .26374859e+01  .33748971e-03", 
" -.11604944e+01 -.48269269e+00  .33054080e+01  .35051345e-03", 
" -.11121741e+01 -.52272105e+00  .34092967e+01  .36648990e-03", 
" -.10581864e+01 -.57863169e+00  .36034214e+01  .38969950e-03", 
" -.99726795e+00 -.60966400e+00  .33080201e+01  .41602700e-03", 
" -.92714913e+00 -.67079329e+00  .33617278e+01  .45342572e-03", 
" -.84508196e+00 -.73314768e+00  .33466681e+01  .48831201e-03", 
" -.74579266e+00 -.81391713e+00  .34525636e+01  .52157314e-03", 
" -.61968126e+00 -.97422543e+00  .44524198e+01  .59810868e-03", 
" -.44808388e+00 -.10681568e+01  .31722674e+01  .71175051e-03", 
" -.40413168e+00 -.11552443e+01  .40607023e+01  .73689227e-03", 
" -.35561507e+00 -.11868635e+01  .33948977e+01  .75994660e-03", 
" -.30131794e+00 -.12455868e+01  .31303488e+01  .77809734e-03", 
" -.23895569e+00 -.13223042e+01  .27911458e+01  .84145044e-03", 
" -.16689697e+00 -.13677278e+01  .20977868e+01  .93707840e-03", 
" -.80063233e-01 -.14859108e+01  .25144350e+01  .10303028e-02", 
"  .28689661e-01 -.15189407e+01  .34735729e+00  .11615465e-02", 
"  .17919574e+00 -.18443240e+01  .26522849e+01  .13267043e-02", 
"  .42176811e+00 -.19156741e+01 -.33735838e+01  .18028865e-02", 
"  .65138066e+00 -.22359475e+01 -.47324830e+01  .25433598e-02", 
"  .93750763e+00 -.22266242e+01 -.18785914e+02  .38021351e-02", 
"  .11331603e+01 -.14524030e+01 -.42062904e+02  .47793585e-02", 
"coct      2  5  2   20", " -.58179096e+01 -.40269001e+02 -.22541791e+03  .43370186e-02", 
" -.56637089e+01 -.37137903e+02 -.18360268e+03  .29291199e-02", 
" -.54547624e+01 -.32973644e+02 -.13278804e+03  .18296435e-02", 
" -.52893696e+01 -.29487564e+02 -.10846481e+03  .13309824e-02", 
" -.51127065e+01 -.26380767e+02 -.83563958e+02  .10603354e-02", 
" -.50053471e+01 -.24453026e+02 -.71685706e+02  .91909444e-03", 
" -.49265719e+01 -.23189957e+02 -.62521721e+02  .80944464e-03", 
" -.48641741e+01 -.22130326e+02 -.57272039e+02  .73985664e-03", 
" -.48122632e+01 -.21275829e+02 -.53441870e+02  .70154324e-03", 
" -.47677187e+01 -.20568888e+02 -.49660536e+02  .65646523e-03", 
" -.47282070e+01 -.19973302e+02 -.46381356e+02  .61234835e-03", 
" -.46930654e+01 -.19454372e+02 -.43482404e+02  .57303877e-03", 
" -.46614455e+01 -.18978097e+02 -.41235061e+02  .54835634e-03", 
" -.45356410e+01 -.17167969e+02 -.32972206e+02  .45881880e-03", 
" -.44423880e+01 -.15924410e+02 -.27265441e+02  .41087765e-03", 
" -.43673274e+01 -.14989883e+02 -.22904492e+02  .38350996e-03", 
" -.43041603e+01 -.14230103e+02 -.19392431e+02  .37399516e-03", 
" -.42497881e+01 -.13563264e+02 -.17022690e+02  .34751050e-03", 
" -.42013140e+01 -.13002140e+02 -.15102831e+02  .31584702e-03", 
" -.41578394e+01 -.12494105e+02 -.13729555e+02  .29590345e-03", 
" -.41179194e+01 -.12050145e+02 -.12476266e+02  .28448712e-03", 
" -.40813247e+01 -.11641122e+02 -.11478560e+02  .27757134e-03", 
" -.40471720e+01 -.11291589e+02 -.10268775e+02  .27634068e-03", 
" -.40153356e+01 -.10962699e+02 -.92013945e+01  .27415537e-03", 
" -.39854300e+01 -.10646923e+02 -.85225445e+01  .26863501e-03", 
" -.39572600e+01 -.10349962e+02 -.79430059e+01  .26245393e-03", 
" -.39302601e+01 -.10091059e+02 -.72282159e+01  .25165714e-03", 
" -.39047696e+01 -.98401360e+01 -.65158262e+01  .24125607e-03", 
" -.38803334e+01 -.96029364e+01 -.59252153e+01  .23430552e-03", 
" -.38568769e+01 -.93827876e+01 -.53747025e+01  .22261509e-03", 
" -.38345043e+01 -.91604793e+01 -.51892752e+01  .22007032e-03", 
" -.38127522e+01 -.89658453e+01 -.47408829e+01  .22016120e-03", 
" -.37919204e+01 -.87663580e+01 -.45575846e+01  .21327157e-03", 
" -.37717563e+01 -.85940067e+01 -.39237279e+01  .20953463e-03", 
" -.37521793e+01 -.84301067e+01 -.33657630e+01  .21118866e-03", 
" -.37332070e+01 -.82628552e+01 -.30183878e+01  .20720301e-03", 
" -.37147706e+01 -.81074243e+01 -.26122280e+01  .20789058e-03", 
" -.36968835e+01 -.79501817e+01 -.24180982e+01  .20637688e-03", 
" -.36794077e+01 -.77997141e+01 -.23127052e+01  .20471883e-03", 
" -.36623244e+01 -.76660063e+01 -.18982729e+01  .20369388e-03", 
" -.36457173e+01 -.75220312e+01 -.18619354e+01  .20186485e-03", 
" -.36294821e+01 -.73903939e+01 -.16306627e+01  .20409858e-03", 
" -.36136206e+01 -.72608596e+01 -.14892503e+01  .19940733e-03", 
" -.35980257e+01 -.71455063e+01 -.11663249e+01  .19933614e-03", 
" -.35828201e+01 -.70267038e+01 -.97349411e+00  .20011982e-03", 
" -.35679151e+01 -.69070763e+01 -.85881963e+00  .19869061e-03", 
" -.35532524e+01 -.67955930e+01 -.67381514e+00  .19809155e-03", 
" -.35388321e+01 -.66901894e+01 -.46285415e+00  .19568530e-03", 
" -.35247686e+01 -.65776655e+01 -.39519831e+00  .19059459e-03", 
" -.35108995e+01 -.64718594e+01 -.31174672e+00  .18701018e-03", 
" -.34972232e+01 -.63708437e+01 -.16363046e+00  .18673568e-03", 
" -.34837753e+01 -.62731919e+01  .31364314e-01  .18636543e-03", 
" -.34705142e+01 -.61775825e+01  .14176764e+00  .18449930e-03", 
" -.34574918e+01 -.60851383e+01  .29219527e+00  .18494311e-03", 
" -.34446099e+01 -.59970430e+01  .43651205e+00  .18554297e-03", 
" -.34319652e+01 -.59039855e+01  .42918408e+00  .18384179e-03", 
" -.34195083e+01 -.58115963e+01  .40076276e+00  .18160702e-03", 
" -.34071380e+01 -.57230815e+01  .40426134e+00  .18263384e-03", 
" -.33949224e+01 -.56438354e+01  .56806028e+00  .18263547e-03", 
" -.33828635e+01 -.55637629e+01  .71507507e+00  .18190138e-03", 
" -.33709264e+01 -.54843665e+01  .79917976e+00  .18039313e-03", 
" -.33591615e+01 -.54050841e+01  .85110451e+00  .17862448e-03", 
" -.33475214e+01 -.53281263e+01  .91859019e+00  .17771397e-03", 
" -.33360050e+01 -.52497713e+01  .93662764e+00  .17637639e-03", 
" -.33246417e+01 -.51714926e+01  .91666183e+00  .17687007e-03", 
" -.33133430e+01 -.50968831e+01  .87492171e+00  .17643941e-03", 
" -.33020835e+01 -.50336849e+01  .10389577e+01  .17532738e-03", 
" -.32910436e+01 -.49621651e+01  .10470721e+01  .17471711e-03", 
" -.32800537e+01 -.48940122e+01  .10882749e+01  .17585910e-03", 
" -.32691480e+01 -.48269038e+01  .11168771e+01  .17372703e-03", 
" -.32583245e+01 -.47655621e+01  .12936152e+01  .17269130e-03", 
" -.32476279e+01 -.47015708e+01  .13711744e+01  .17469489e-03", 
" -.32370370e+01 -.46354854e+01  .13837334e+01  .17420580e-03", 
" -.32264932e+01 -.45742722e+01  .14240464e+01  .17507480e-03", 
" -.32160467e+01 -.45117027e+01  .14533482e+01  .17326625e-03", 
" -.32056352e+01 -.44541412e+01  .15419254e+01  .17403130e-03", 
" -.31953872e+01 -.43880749e+01  .14722872e+01  .17271781e-03", 
" -.31851196e+01 -.43315156e+01  .15348030e+01  .17290464e-03", 
" -.31749019e+01 -.42755525e+01  .15985107e+01  .17379875e-03", 
" -.31647989e+01 -.42185765e+01  .16028688e+01  .17281982e-03", 
" -.31547501e+01 -.41634000e+01  .16365477e+01  .17237940e-03", 
" -.31447583e+01 -.41103798e+01  .17114083e+01  .17265234e-03", 
" -.31348100e+01 -.40583213e+01  .17857719e+01  .17171438e-03", 
" -.31249208e+01 -.40030262e+01  .17823257e+01  .17050457e-03", 
" -.31150956e+01 -.39503305e+01  .18155350e+01  .16969511e-03", 
" -.31052976e+01 -.39009527e+01  .19208357e+01  .16808173e-03", 
" -.30955699e+01 -.38477543e+01  .19017691e+01  .16594579e-03", 
" -.30858559e+01 -.37993636e+01  .19543925e+01  .16677064e-03", 
" -.30761691e+01 -.37520141e+01  .20267205e+01  .16625367e-03", 
" -.30665390e+01 -.37070346e+01  .21170040e+01  .16605119e-03", 
" -.30569609e+01 -.36598029e+01  .21598255e+01  .16614517e-03", 
" -.30473612e+01 -.36172958e+01  .22398658e+01  .16698072e-03", 
" -.30378597e+01 -.35660939e+01  .21535424e+01  .16703248e-03", 
" -.30284020e+01 -.35172829e+01  .21349151e+01  .16818869e-03", 
" -.30189269e+01 -.34798053e+01  .23487914e+01  .16940917e-03", 
" -.30095284e+01 -.34343787e+01  .23716202e+01  .16784281e-03", 
" -.30001584e+01 -.33898941e+01  .23667450e+01  .16909684e-03", 
" -.29908168e+01 -.33418797e+01  .22889954e+01  .16598223e-03", 
" -.29814948e+01 -.32965100e+01  .22722398e+01  .16424187e-03", 
" -.29721428e+01 -.32553124e+01  .23177549e+01  .16432144e-03", 
" -.29628649e+01 -.32111871e+01  .23119076e+01  .16340277e-03", 
" -.29535776e+01 -.31710508e+01  .23342128e+01  .16227060e-03", 
" -.29442933e+01 -.31338836e+01  .24258202e+01  .16342349e-03", 
" -.29350677e+01 -.30927593e+01  .24138445e+01  .16331263e-03", 
" -.29258583e+01 -.30493627e+01  .23321586e+01  .15951994e-03", 
" -.29165998e+01 -.30150446e+01  .24469445e+01  .15952990e-03", 
" -.29073705e+01 -.29812051e+01  .25950484e+01  .15852094e-03", 
" -.28981920e+01 -.29427985e+01  .26323324e+01  .15818419e-03", 
" -.28890419e+01 -.29023886e+01  .26227169e+01  .15790223e-03", 
" -.28798807e+01 -.28652712e+01  .26402686e+01  .15870235e-03", 
" -.28706822e+01 -.28326966e+01  .27716261e+01  .15747252e-03", 
" -.28615165e+01 -.27949491e+01  .27339257e+01  .15659127e-03", 
" -.28523157e+01 -.27625500e+01  .28410923e+01  .15646634e-03", 
" -.28431531e+01 -.27248248e+01  .27956362e+01  .15596625e-03", 
" -.28339587e+01 -.26913386e+01  .28450647e+01  .15673033e-03", 
" -.28248163e+01 -.26552604e+01  .28015176e+01  .15664457e-03", 
" -.28156563e+01 -.26182797e+01  .27686134e+01  .15848626e-03", 
" -.28065185e+01 -.25798571e+01  .27099746e+01  .15708599e-03", 
" -.27973814e+01 -.25439590e+01  .26848880e+01  .15944754e-03", 
" -.27881849e+01 -.25117797e+01  .27252212e+01  .15872009e-03", 
" -.27790199e+01 -.24768638e+01  .27087974e+01  .15727369e-03", 
" -.27698459e+01 -.24408252e+01  .26385257e+01  .15594754e-03", 
" -.27605963e+01 -.24120354e+01  .27103539e+01  .15573114e-03", 
" -.27513627e+01 -.23803767e+01  .27108678e+01  .15554085e-03", 
" -.27420881e+01 -.23517480e+01  .27575843e+01  .15681192e-03", 
" -.27328428e+01 -.23195373e+01  .27592963e+01  .15717727e-03", 
" -.27235679e+01 -.22884773e+01  .27831327e+01  .15741321e-03", 
" -.27142739e+01 -.22559350e+01  .27560921e+01  .15729244e-03", 
" -.27049899e+01 -.22227005e+01  .27072503e+01  .15699287e-03", 
" -.26956761e+01 -.21896110e+01  .26553338e+01  .15635388e-03", 
" -.26863141e+01 -.21605596e+01  .26864815e+01  .15540338e-03", 
" -.26769235e+01 -.21310602e+01  .26890244e+01  .15670723e-03", 
" -.26675197e+01 -.20995001e+01  .26698276e+01  .15741101e-03", 
" -.26580478e+01 -.20702891e+01  .26578622e+01  .15706756e-03", 
" -.26485157e+01 -.20436345e+01  .26661631e+01  .15513997e-03", 
" -.26389437e+01 -.20230280e+01  .28220070e+01  .15491370e-03", 
" -.26293248e+01 -.19983345e+01  .28342371e+01  .15665656e-03", 
" -.26196770e+01 -.19751888e+01  .29048542e+01  .15841772e-03", 
" -.26100513e+01 -.19478620e+01  .29100124e+01  .15728298e-03", 
" -.26003443e+01 -.19218228e+01  .28952091e+01  .15695864e-03", 
" -.25905650e+01 -.18985909e+01  .29206145e+01  .15319119e-03", 
" -.25808035e+01 -.18690909e+01  .28685050e+01  .15293972e-03", 
" -.25709604e+01 -.18433992e+01  .28880723e+01  .15381890e-03", 
" -.25610558e+01 -.18200805e+01  .29364984e+01  .15512101e-03", 
" -.25511034e+01 -.17954054e+01  .29215493e+01  .15409697e-03", 
" -.25410988e+01 -.17728713e+01  .29791155e+01  .15286174e-03", 
" -.25310328e+01 -.17494735e+01  .29886062e+01  .15488159e-03", 
" -.25208783e+01 -.17271295e+01  .30234626e+01  .15452178e-03", 
" -.25107012e+01 -.17018444e+01  .29819434e+01  .15520169e-03", 
" -.25004528e+01 -.16775823e+01  .29641043e+01  .15543221e-03", 
" -.24901017e+01 -.16550672e+01  .29537127e+01  .15575898e-03", 
" -.24796046e+01 -.16409900e+01  .31063253e+01  .15622843e-03", 
" -.24690659e+01 -.16207560e+01  .31404325e+01  .15783968e-03", 
" -.24585163e+01 -.15940771e+01  .30427006e+01  .15835883e-03", 
" -.24478543e+01 -.15727181e+01  .30614694e+01  .16016225e-03", 
" -.24370578e+01 -.15518474e+01  .30545294e+01  .16418315e-03", 
" -.24261896e+01 -.15313508e+01  .30539984e+01  .16341039e-03", 
" -.24152288e+01 -.15100248e+01  .30284739e+01  .16355089e-03", 
" -.24041802e+01 -.14849045e+01  .29091217e+01  .16352236e-03", 
" -.23930061e+01 -.14646841e+01  .29136222e+01  .16296836e-03", 
" -.23817185e+01 -.14461691e+01  .29477960e+01  .16282817e-03", 
" -.23702390e+01 -.14352308e+01  .31113258e+01  .16481482e-03", 
" -.23586512e+01 -.14234644e+01  .32497210e+01  .16718117e-03", 
" -.23469647e+01 -.14083499e+01  .33528296e+01  .16719038e-03", 
" -.23351235e+01 -.13918745e+01  .33912918e+01  .16696446e-03", 
" -.23231406e+01 -.13753345e+01  .33790812e+01  .16626113e-03", 
" -.23109852e+01 -.13608350e+01  .34288438e+01  .16664727e-03", 
" -.22986746e+01 -.13457763e+01  .34474998e+01  .16995736e-03", 
" -.22861829e+01 -.13350326e+01  .35361777e+01  .17218856e-03", 
" -.22735317e+01 -.13191177e+01  .35497809e+01  .17426363e-03", 
" -.22606684e+01 -.13077154e+01  .36264945e+01  .17421961e-03", 
" -.22476386e+01 -.12928026e+01  .35982320e+01  .17510343e-03", 
" -.22343428e+01 -.12875404e+01  .37593611e+01  .17613195e-03", 
" -.22209024e+01 -.12698471e+01  .36272028e+01  .17768837e-03", 
" -.22071587e+01 -.12578925e+01  .36090203e+01  .18160013e-03", 
" -.21931861e+01 -.12528119e+01  .37645187e+01  .18329992e-03", 
" -.21789716e+01 -.12411868e+01  .37497002e+01  .18401166e-03", 
" -.21644552e+01 -.12282631e+01  .36548613e+01  .18391096e-03", 
" -.21496000e+01 -.12226008e+01  .37272220e+01  .18718384e-03", 
" -.21344190e+01 -.12202299e+01  .38764199e+01  .18760348e-03", 
" -.21189261e+01 -.12191629e+01  .40402966e+01  .18896388e-03", 
" -.21029804e+01 -.12262520e+01  .43478398e+01  .18884308e-03", 
" -.20866744e+01 -.12251882e+01  .43995994e+01  .19356846e-03", 
" -.20699696e+01 -.12194140e+01  .43834021e+01  .19504521e-03", 
" -.20527057e+01 -.12268132e+01  .45236103e+01  .20113902e-03", 
" -.20349947e+01 -.12356844e+01  .47275673e+01  .20299239e-03", 
" -.20167182e+01 -.12446329e+01  .48974592e+01  .20356937e-03", 
" -.19979078e+01 -.12529174e+01  .50527430e+01  .20704767e-03", 
" -.19784182e+01 -.12618948e+01  .51417664e+01  .20453821e-03", 
" -.19583014e+01 -.12640479e+01  .50930678e+01  .20988301e-03", 
" -.19373167e+01 -.12763133e+01  .51407283e+01  .21254530e-03", 
" -.19154734e+01 -.12952013e+01  .52977615e+01  .21830440e-03", 
" -.18926747e+01 -.13199523e+01  .55365604e+01  .21883665e-03", 
" -.18687132e+01 -.13469703e+01  .56811977e+01  .22800006e-03", 
" -.18436213e+01 -.13817552e+01  .59441241e+01  .23451625e-03", 
" -.18171280e+01 -.14193071e+01  .61672844e+01  .23874340e-03", 
" -.17891644e+01 -.14619984e+01  .64888478e+01  .24222179e-03", 
" -.17594522e+01 -.15028315e+01  .66598111e+01  .25502799e-03", 
" -.17275747e+01 -.15672767e+01  .71366131e+01  .25954374e-03", 
" -.16932421e+01 -.16275901e+01  .73229033e+01  .27207099e-03", 
" -.16559329e+01 -.17149502e+01  .78684304e+01  .28385120e-03", 
" -.16152830e+01 -.17925342e+01  .80990207e+01  .29586091e-03", 
" -.15700153e+01 -.19136053e+01  .89717429e+01  .30913874e-03", 
" -.15190675e+01 -.20362637e+01  .91038221e+01  .34184973e-03", 
" -.14605378e+01 -.22111810e+01  .98374553e+01  .36320797e-03", 
" -.13921255e+01 -.24225920e+01  .10631166e+02  .39450785e-03", 
" -.13091407e+01 -.26993569e+01  .11683956e+02  .41785316e-03", 
" -.12030458e+01 -.30729766e+01  .12870207e+02  .47497055e-03", 
" -.10561731e+01 -.36258242e+01  .14152264e+02  .55935859e-03", 
" -.10188267e+01 -.37715412e+01  .14553365e+02  .57841732e-03", 
" -.97701898e+00 -.39486827e+01  .15224015e+02  .59987200e-03", 
" -.93017029e+00 -.41859474e+01  .16447170e+02  .65345475e-03", 
" -.87655877e+00 -.44087983e+01  .16928446e+02  .70560938e-03", 
" -.81398264e+00 -.47160157e+01  .18093474e+02  .77483254e-03", 
" -.73828460e+00 -.51397297e+01  .20528207e+02  .85196487e-03", 
" -.64283315e+00 -.56378614e+01  .22318781e+02  .93088236e-03", 
" -.51261712e+00 -.62672519e+01  .22091259e+02  .11369078e-02", 
" -.29760049e+00 -.74817774e+01  .22673490e+02  .15200170e-02", 
" -.93854076e-01 -.87040579e+01  .23509276e+02  .21254358e-02", 
"  .16446139e+00 -.10373195e+02  .22421533e+02  .30995254e-02", 
"  .34809116e+00 -.11981711e+02  .29681333e+02  .40771566e-02", 
"coctt     2  6  3   20", " -.61308400e+01 -.47142293e+02 -.31437041e+03  .40465228e+03  .50727464e-02", 
" -.59817352e+01 -.44050249e+02 -.21977883e+03 -.49997399e+03  .37724852e-02", 
" -.57737514e+01 -.38719048e+02 -.20595321e+03  .13473211e+03  .25327182e-02", 
" -.56088741e+01 -.35109412e+02 -.17546729e+03  .21553665e+03  .17703513e-02", 
" -.54350699e+01 -.31748409e+02 -.13800045e+03  .21255697e+03  .13400542e-02", 
" -.53284855e+01 -.29951350e+02 -.10559074e+03  .18275433e+02  .11491370e-02", 
" -.52514061e+01 -.28430785e+02 -.98033700e+02  .97015148e+02  .10036584e-02", 
" -.51902033e+01 -.27267830e+02 -.88951390e+02  .46940864e+02  .93053988e-03", 
" -.51393535e+01 -.26193588e+02 -.92848377e+02  .20659973e+03  .88576310e-03", 
" -.50947766e+01 -.25587194e+02 -.77682374e+02  .64468747e+02  .82922411e-03", 
" -.50559177e+01 -.24917553e+02 -.73801843e+02  .66590084e+02  .79703698e-03", 
" -.50215120e+01 -.24283770e+02 -.72070647e+02  .91823732e+02  .75253053e-03", 
" -.49900113e+01 -.23734979e+02 -.71001590e+02  .12896846e+03  .71606702e-03", 
" -.48659285e+01 -.21722465e+02 -.56424548e+02  .66437182e+02  .62540492e-03", 
" -.47735119e+01 -.20308540e+02 -.48836696e+02  .72532744e+02  .58081619e-03", 
" -.46989134e+01 -.19263941e+02 -.41842755e+02  .60373438e+02  .51229846e-03", 
" -.46369311e+01 -.18304029e+02 -.40492296e+02  .88953917e+02  .47252810e-03", 
" -.45829538e+01 -.17535199e+02 -.39308725e+02  .13143262e+03  .45526363e-03", 
" -.45349906e+01 -.16902403e+02 -.35442825e+02  .11877034e+03  .43225731e-03", 
" -.44918475e+01 -.16360959e+02 -.31530761e+02  .10229732e+03  .40838430e-03", 
" -.44525911e+01 -.15836408e+02 -.29997659e+02  .10289131e+03  .37884813e-03", 
" -.44163512e+01 -.15374098e+02 -.28538949e+02  .10756454e+03  .36361761e-03", 
" -.43828918e+01 -.14898628e+02 -.30037430e+02  .14797060e+03  .35446778e-03", 
" -.43514827e+01 -.14533435e+02 -.27072895e+02  .12425654e+03  .35230062e-03", 
" -.43219545e+01 -.14175108e+02 -.25799471e+02  .12409852e+03  .33976268e-03", 
" -.42938789e+01 -.13877280e+02 -.23276043e+02  .11023248e+03  .33241184e-03", 
" -.42673496e+01 -.13588561e+02 -.20949647e+02  .90327047e+02  .32784940e-03", 
" -.42419868e+01 -.13297940e+02 -.20764243e+02  .10528480e+03  .32046567e-03", 
" -.42178627e+01 -.13002498e+02 -.21131996e+02  .12354087e+03  .31115934e-03", 
" -.41947365e+01 -.12731887e+02 -.20902252e+02  .12864890e+03  .30227195e-03", 
" -.41724123e+01 -.12523899e+02 -.18016996e+02  .98205795e+02  .29758136e-03", 
" -.41510753e+01 -.12289660e+02 -.17613950e+02  .10660962e+03  .29323439e-03", 
" -.41304794e+01 -.12056475e+02 -.17388501e+02  .10956100e+03  .28621266e-03", 
" -.41104434e+01 -.11847825e+02 -.16761827e+02  .10886429e+03  .28429739e-03", 
" -.40911537e+01 -.11640044e+02 -.16162791e+02  .10462823e+03  .28021979e-03", 
" -.40723822e+01 -.11443860e+02 -.15989879e+02  .11120323e+03  .27399291e-03", 
" -.40541757e+01 -.11263102e+02 -.15168695e+02  .10680764e+03  .27134543e-03", 
" -.40364462e+01 -.11089997e+02 -.14248799e+02  .99844992e+02  .26754331e-03", 
" -.40191835e+01 -.10928967e+02 -.12814116e+02  .83703923e+02  .26155198e-03", 
" -.40023233e+01 -.10766508e+02 -.12323880e+02  .83607162e+02  .26017187e-03", 
" -.39859048e+01 -.10604614e+02 -.11921445e+02  .81995709e+02  .25271957e-03", 
" -.39698393e+01 -.10457743e+02 -.11063421e+02  .75782379e+02  .24894785e-03", 
" -.39541209e+01 -.10305062e+02 -.10701650e+02  .75261288e+02  .24876158e-03", 
" -.39388005e+01 -.10164679e+02 -.97714627e+01  .66917403e+02  .24410634e-03", 
" -.39237890e+01 -.10013588e+02 -.98966394e+01  .71856722e+02  .24351923e-03", 
" -.39090919e+01 -.98725258e+01 -.97897583e+01  .75610104e+02  .23981128e-03", 
" -.38946112e+01 -.97433613e+01 -.92366472e+01  .72409952e+02  .24000204e-03", 
" -.38804280e+01 -.96181031e+01 -.85744704e+01  .67873683e+02  .23488479e-03", 
" -.38665401e+01 -.94814620e+01 -.88251250e+01  .75811064e+02  .23537559e-03", 
" -.38528357e+01 -.93639094e+01 -.82237929e+01  .71033228e+02  .23488020e-03", 
" -.38393483e+01 -.92428089e+01 -.81274518e+01  .74152421e+02  .22947621e-03", 
" -.38260700e+01 -.91280611e+01 -.80146986e+01  .78526248e+02  .22652665e-03", 
" -.38129674e+01 -.90142861e+01 -.78109162e+01  .78106209e+02  .22734980e-03", 
" -.38000732e+01 -.89006730e+01 -.77124450e+01  .79901855e+02  .22387154e-03", 
" -.37874151e+01 -.87815767e+01 -.81581263e+01  .88838315e+02  .22359889e-03", 
" -.37748780e+01 -.86809805e+01 -.76295599e+01  .84610846e+02  .22230330e-03", 
" -.37624773e+01 -.85896891e+01 -.67217168e+01  .74463794e+02  .22200418e-03", 
" -.37501945e+01 -.85113941e+01 -.54357008e+01  .61400605e+02  .21758988e-03", 
" -.37381462e+01 -.84148572e+01 -.51343257e+01  .60441935e+02  .22137583e-03", 
" -.37262031e+01 -.83325123e+01 -.41537713e+01  .49946600e+02  .21976968e-03", 
" -.37143693e+01 -.82554706e+01 -.30605495e+01  .38229490e+02  .21885652e-03", 
" -.37027181e+01 -.81641507e+01 -.28242508e+01  .37006276e+02  .21320978e-03", 
" -.36912342e+01 -.80715791e+01 -.25904692e+01  .35079775e+02  .21550525e-03", 
" -.36797636e+01 -.80020334e+01 -.13543619e+01  .20455915e+02  .21670303e-03", 
" -.36685438e+01 -.78996280e+01 -.19829683e+01  .30275814e+02  .21843617e-03", 
" -.36574072e+01 -.78144215e+01 -.18220401e+01  .30896563e+02  .21890683e-03", 
" -.36463500e+01 -.77273975e+01 -.18948580e+01  .34006792e+02  .21859917e-03", 
" -.36354451e+01 -.76319511e+01 -.23477656e+01  .42396740e+02  .21361459e-03", 
" -.36245956e+01 -.75470897e+01 -.24181376e+01  .44740040e+02  .20980795e-03", 
" -.36138741e+01 -.74631077e+01 -.24712951e+01  .48042877e+02  .20996001e-03", 
" -.36031722e+01 -.73869766e+01 -.23428828e+01  .49437100e+02  .21002296e-03", 
" -.35926053e+01 -.73116479e+01 -.21405039e+01  .49701257e+02  .21010698e-03", 
" -.35821024e+01 -.72462064e+01 -.14350795e+01  .42685828e+02  .21068367e-03", 
" -.35716720e+01 -.71693998e+01 -.15278969e+01  .45614486e+02  .21376788e-03", 
" -.35613095e+01 -.70956284e+01 -.14994476e+01  .46549681e+02  .21341803e-03", 
" -.35510356e+01 -.70256298e+01 -.13986393e+01  .47709076e+02  .21381660e-03", 
" -.35408350e+01 -.69514665e+01 -.15366981e+01  .52143795e+02  .21369872e-03", 
" -.35307260e+01 -.68807116e+01 -.14609943e+01  .52615535e+02  .21148285e-03", 
" -.35206634e+01 -.68199641e+01 -.94546625e+00  .47342085e+02  .21037885e-03", 
" -.35106718e+01 -.67527339e+01 -.79004821e+00  .45958207e+02  .20846214e-03", 
" -.35007632e+01 -.66847969e+01 -.71165835e+00  .47326904e+02  .20707120e-03", 
" -.34908273e+01 -.66272333e+01 -.29916236e+00  .44213375e+02  .20685187e-03", 
" -.34810152e+01 -.65617301e+01 -.20410096e+00  .43789562e+02  .20508572e-03", 
" -.34712116e+01 -.65134300e+01  .71877216e+00  .33109392e+02  .20472360e-03", 
" -.34615399e+01 -.64409443e+01  .34218285e+00  .37883618e+02  .20501118e-03", 
" -.34518610e+01 -.63799689e+01  .41659050e+00  .38970808e+02  .20136590e-03", 
" -.34421969e+01 -.63268672e+01  .72387337e+00  .37344230e+02  .20147853e-03", 
" -.34325953e+01 -.62787252e+01  .15062589e+01  .27640038e+02  .19906113e-03", 
" -.34231099e+01 -.62059257e+01  .99959557e+00  .34657495e+02  .19846742e-03", 
" -.34136311e+01 -.61319861e+01  .25032917e+00  .45604850e+02  .19744031e-03", 
" -.34041758e+01 -.60754176e+01  .43873854e+00  .44335609e+02  .19780416e-03", 
" -.33947865e+01 -.60113985e+01  .27378684e+00  .46833375e+02  .19754659e-03", 
" -.33854096e+01 -.59498395e+01  .65511263e-01  .50472155e+02  .19648388e-03", 
" -.33760310e+01 -.58967895e+01  .31969817e+00  .47280372e+02  .19646130e-03", 
" -.33667050e+01 -.58524555e+01  .10548057e+01  .38203305e+02  .19573653e-03", 
" -.33574920e+01 -.57779182e+01  .20214486e+00  .49550200e+02  .19545134e-03", 
" -.33482355e+01 -.57233536e+01  .29046297e+00  .48862115e+02  .19564717e-03", 
" -.33389413e+01 -.56828356e+01  .96139231e+00  .41306454e+02  .19612969e-03", 
" -.33297683e+01 -.56199149e+01  .45125180e+00  .49419789e+02  .19646513e-03", 
" -.33205965e+01 -.55646501e+01  .51322036e+00  .48155216e+02  .19615796e-03", 
" -.33114412e+01 -.55127515e+01  .51379398e+00  .49506003e+02  .19714403e-03", 
" -.33022656e+01 -.54702647e+01  .10428044e+01  .42336775e+02  .19554543e-03", 
" -.32931069e+01 -.54284411e+01  .16404473e+01  .34346131e+02  .19234264e-03", 
" -.32839837e+01 -.53762897e+01  .15619208e+01  .36122390e+02  .19245765e-03", 
" -.32749023e+01 -.53221192e+01  .13870650e+01  .38510276e+02  .19199446e-03", 
" -.32658276e+01 -.52710263e+01  .12494843e+01  .41346451e+02  .18871585e-03", 
" -.32567303e+01 -.52337672e+01  .18471248e+01  .34056077e+02  .18833553e-03", 
" -.32476789e+01 -.51943434e+01  .23747608e+01  .27911756e+02  .18798368e-03", 
" -.32386990e+01 -.51333137e+01  .17788215e+01  .35616980e+02  .18707154e-03", 
" -.32296460e+01 -.50931866e+01  .21562258e+01  .31408105e+02  .18591851e-03", 
" -.32206030e+01 -.50610051e+01  .30376841e+01  .19710906e+02  .18568208e-03", 
" -.32115578e+01 -.50146741e+01  .29721847e+01  .21456867e+02  .18436449e-03", 
" -.32024976e+01 -.49792091e+01  .34889433e+01  .15738461e+02  .18218753e-03", 
" -.31934789e+01 -.49374676e+01  .36820981e+01  .13572155e+02  .18257992e-03", 
" -.31844585e+01 -.48955592e+01  .39337408e+01  .97469781e+01  .18116793e-03", 
" -.31754156e+01 -.48574000e+01  .42756825e+01  .59951431e+01  .18189319e-03", 
" -.31663321e+01 -.48304033e+01  .51101394e+01 -.38386050e+01  .18263452e-03", 
" -.31572709e+01 -.47993695e+01  .57635555e+01 -.11710654e+02  .18318000e-03", 
" -.31482558e+01 -.47619819e+01  .61475467e+01 -.16266889e+02  .18313614e-03", 
" -.31392373e+01 -.47141243e+01  .57824145e+01 -.91965470e+01  .18306993e-03", 
" -.31302008e+01 -.46763967e+01  .61439967e+01 -.14464695e+02  .18210136e-03", 
" -.31211060e+01 -.46431038e+01  .65478134e+01 -.19331416e+02  .18049828e-03", 
" -.31119808e+01 -.46174408e+01  .73832869e+01 -.29453647e+02  .18163913e-03", 
" -.31028667e+01 -.45858701e+01  .79114235e+01 -.36399641e+02  .18387658e-03", 
" -.30937651e+01 -.45487009e+01  .80243543e+01 -.37345973e+02  .18449606e-03", 
" -.30846610e+01 -.45097715e+01  .80341625e+01 -.36259386e+02  .18784624e-03", 
" -.30755343e+01 -.44757014e+01  .83193288e+01 -.39497651e+02  .18565410e-03", 
" -.30664105e+01 -.44276711e+01  .77557323e+01 -.31157984e+02  .18542817e-03", 
" -.30571772e+01 -.44012091e+01  .83226585e+01 -.37940113e+02  .18656099e-03", 
" -.30479796e+01 -.43683091e+01  .86131712e+01 -.41657371e+02  .18752303e-03", 
" -.30387915e+01 -.43286483e+01  .84587980e+01 -.38310971e+02  .18570989e-03", 
" -.30295514e+01 -.42935796e+01  .86663187e+01 -.40953933e+02  .18615659e-03", 
" -.30202379e+01 -.42716164e+01  .93639686e+01 -.48755378e+02  .18830348e-03", 
" -.30109668e+01 -.42302076e+01  .90576787e+01 -.43828437e+02  .18840919e-03", 
" -.30016020e+01 -.41952669e+01  .88619425e+01 -.38933065e+02  .18866464e-03", 
" -.29922353e+01 -.41585586e+01  .87813610e+01 -.37526094e+02  .18924266e-03", 
" -.29828426e+01 -.41259897e+01  .89811104e+01 -.40339836e+02  .19078346e-03", 
" -.29733737e+01 -.40907497e+01  .88238973e+01 -.36842392e+02  .19182181e-03", 
" -.29639397e+01 -.40471436e+01  .82824727e+01 -.28782956e+02  .19261564e-03", 
" -.29544404e+01 -.40106472e+01  .81509831e+01 -.26886270e+02  .19196862e-03", 
" -.29448764e+01 -.39825756e+01  .84923809e+01 -.31061389e+02  .19223613e-03", 
" -.29352322e+01 -.39473378e+01  .82429727e+01 -.26654793e+02  .19198215e-03", 
" -.29255100e+01 -.39300907e+01  .89393533e+01 -.34522742e+02  .18917401e-03", 
" -.29157643e+01 -.39079099e+01  .93303082e+01 -.37981874e+02  .18920000e-03", 
" -.29060374e+01 -.38745208e+01  .92878996e+01 -.36512064e+02  .19240481e-03", 
" -.28962359e+01 -.38428386e+01  .92886010e+01 -.35544058e+02  .19412878e-03", 
" -.28864244e+01 -.38009240e+01  .87454422e+01 -.27815740e+02  .19682601e-03", 
" -.28765072e+01 -.37699724e+01  .87672216e+01 -.28067658e+02  .19588345e-03", 
" -.28664788e+01 -.37514713e+01  .93251249e+01 -.34299438e+02  .19572187e-03", 
" -.28564431e+01 -.37203552e+01  .93508307e+01 -.34537583e+02  .19756383e-03", 
" -.28462680e+01 -.37017441e+01  .98179868e+01 -.39609880e+02  .19959215e-03", 
" -.28360719e+01 -.36760329e+01  .99938731e+01 -.41161829e+02  .19903650e-03", 
" -.28257466e+01 -.36573237e+01  .10403284e+02 -.45107347e+02  .19732910e-03", 
" -.28153720e+01 -.36351517e+01  .10668648e+02 -.47982953e+02  .19519916e-03", 
" -.28049060e+01 -.36156666e+01  .10933075e+02 -.49194426e+02  .19582999e-03", 
" -.27943666e+01 -.35993740e+01  .11465017e+02 -.55067186e+02  .19703401e-03", 
" -.27837377e+01 -.35848748e+01  .12123078e+02 -.63484153e+02  .19751505e-03", 
" -.27730162e+01 -.35615530e+01  .12260309e+02 -.63535192e+02  .19815262e-03", 
" -.27621295e+01 -.35499543e+01  .12894283e+02 -.70655573e+02  .19652726e-03", 
" -.27511946e+01 -.35307085e+01  .13217618e+02 -.74044430e+02  .19701197e-03", 
" -.27401519e+01 -.35162778e+01  .13698800e+02 -.78816427e+02  .19784606e-03", 
" -.27290099e+01 -.34939675e+01  .13825426e+02 -.79889309e+02  .19663007e-03", 
" -.27177525e+01 -.34730866e+01  .13867069e+02 -.79022634e+02  .19724720e-03", 
" -.27063381e+01 -.34512811e+01  .13867836e+02 -.77645544e+02  .19750266e-03", 
" -.26947837e+01 -.34431306e+01  .14614913e+02 -.86861887e+02  .20127746e-03", 
" -.26831352e+01 -.34255980e+01  .14818054e+02 -.88414628e+02  .20113597e-03", 
" -.26712800e+01 -.34131954e+01  .15246947e+02 -.93291328e+02  .20074119e-03", 
" -.26593130e+01 -.33998898e+01  .15622454e+02 -.96935801e+02  .19896236e-03", 
" -.26471634e+01 -.33948358e+01  .16441033e+02 -.10752158e+03  .20278910e-03", 
" -.26348385e+01 -.33793351e+01  .16572693e+02 -.10751441e+03  .20391427e-03", 
" -.26223455e+01 -.33729507e+01  .17140904e+02 -.11338420e+03  .20827116e-03", 
" -.26096803e+01 -.33587352e+01  .17336139e+02 -.11439673e+03  .20920259e-03", 
" -.25968527e+01 -.33465902e+01  .17613885e+02 -.11642652e+03  .21224169e-03", 
" -.25837905e+01 -.33265878e+01  .17394510e+02 -.11294681e+03  .21274306e-03", 
" -.25705139e+01 -.33097914e+01  .17342010e+02 -.11130819e+03  .21111437e-03", 
" -.25569459e+01 -.33062686e+01  .17663096e+02 -.11229110e+03  .21387224e-03", 
" -.25431036e+01 -.33071371e+01  .18283513e+02 -.11814618e+03  .21521445e-03", 
" -.25290713e+01 -.33074943e+01  .19033605e+02 -.12731603e+03  .21069739e-03", 
" -.25146851e+01 -.33114775e+01  .19567774e+02 -.12957497e+03  .21374603e-03", 
" -.25000113e+01 -.33159947e+01  .20119302e+02 -.13266188e+03  .21702410e-03", 
" -.24850144e+01 -.33199530e+01  .20618999e+02 -.13650295e+03  .21958016e-03", 
" -.24696067e+01 -.33355513e+01  .21683240e+02 -.14883102e+03  .22072034e-03", 
" -.24539640e+01 -.33316951e+01  .21682654e+02 -.14574148e+03  .22274067e-03", 
" -.24379848e+01 -.33166506e+01  .21167969e+02 -.13734874e+03  .22553814e-03", 
" -.24215363e+01 -.33202118e+01  .21258396e+02 -.13417366e+03  .22772304e-03", 
" -.24045244e+01 -.33587486e+01  .23066667e+02 -.15299187e+03  .22658761e-03", 
" -.23870822e+01 -.33679150e+01  .23309755e+02 -.15241538e+03  .22916492e-03", 
" -.23690575e+01 -.33799721e+01  .23542720e+02 -.15107984e+03  .23890114e-03", 
" -.23505585e+01 -.33779587e+01  .23150924e+02 -.14364187e+03  .23991360e-03", 
" -.23312772e+01 -.34012742e+01  .23664854e+02 -.14682935e+03  .24203889e-03", 
" -.23113549e+01 -.34268193e+01  .24065369e+02 -.14669327e+03  .24495312e-03", 
" -.22904849e+01 -.34971434e+01  .26358617e+02 -.16861184e+03  .24636476e-03", 
" -.22689271e+01 -.35463278e+01  .27588422e+02 -.17748580e+03  .24737249e-03", 
" -.22462922e+01 -.35964182e+01  .28512370e+02 -.18394816e+03  .24925256e-03", 
" -.22227013e+01 -.36439788e+01  .29043627e+02 -.18288212e+03  .25765147e-03", 
" -.21978093e+01 -.37253968e+01  .30979221e+02 -.19994233e+03  .26600816e-03", 
" -.21717095e+01 -.37813067e+01  .31426529e+02 -.19892608e+03  .27099585e-03", 
" -.21438665e+01 -.38674775e+01  .33077565e+02 -.21279420e+03  .27973897e-03", 
" -.21142242e+01 -.39523115e+01  .33958382e+02 -.21408211e+03  .28223095e-03", 
" -.20823608e+01 -.40797245e+01  .35981682e+02 -.22698862e+03  .29187854e-03", 
" -.20478734e+01 -.42024336e+01  .36909681e+02 -.22374872e+03  .30590642e-03", 
" -.20103779e+01 -.43711408e+01  .39666851e+02 -.24532636e+03  .32101062e-03", 
" -.19691812e+01 -.45140194e+01  .39675742e+02 -.22830748e+03  .34225325e-03", 
" -.19229864e+01 -.46998186e+01  .39374462e+02 -.20020232e+03  .35722350e-03", 
" -.18699143e+01 -.50450643e+01  .44595505e+02 -.23793286e+03  .39962790e-03", 
" -.18081904e+01 -.54420862e+01  .49021628e+02 -.25881774e+03  .42481835e-03", 
" -.17333716e+01 -.59913513e+01  .55456958e+02 -.29446412e+03  .46233227e-03", 
" -.16387132e+01 -.66989043e+01  .62123267e+02 -.31964259e+03  .53227753e-03", 
" -.15076075e+01 -.77214219e+01  .67660475e+02 -.29901333e+03  .65950394e-03", 
" -.14736606e+01 -.80571179e+01  .71209510e+02 -.31073977e+03  .70241953e-03", 
" -.14357680e+01 -.84620949e+01  .76019063e+02 -.33606927e+03  .72701115e-03", 
" -.13933378e+01 -.88994328e+01  .81707801e+02 -.37986642e+03  .75135188e-03", 
" -.13445470e+01 -.93756826e+01  .84704117e+02 -.37344651e+03  .79759994e-03", 
" -.12878084e+01 -.98546079e+01  .84721812e+02 -.33032917e+03  .86662685e-03", 
" -.12187905e+01 -.10575051e+02  .91500677e+02 -.36690765e+03  .93639466e-03", 
" -.11322288e+01 -.11413370e+02  .96968943e+02 -.38648132e+03  .10603822e-02", 
" -.10122539e+01 -.12708364e+02  .10741035e+03 -.43755476e+03  .12266351e-02", 
" -.81479099e+00 -.14869377e+02  .11621187e+03 -.38457387e+03  .17423446e-02", 
" -.62625604e+00 -.17436704e+02  .15065516e+03 -.72224933e+03  .22775510e-02", 
" -.38303025e+00 -.20928657e+02  .17445768e+03 -.79957009e+03  .33806264e-02", 
" -.21256259e+00 -.23271915e+02  .18993865e+03 -.89111787e+03  .45438278e-02", 
"coanc     2  3  3   20", " -.51792942e+02  .75767123e+03 -.73960095e+04  .35100529e+05  .76511523e-01", 
" -.48587017e+02  .66253180e+03 -.58294783e+04  .23534338e+05  .56623257e-01", 
" -.44462143e+02  .56898640e+03 -.49799853e+04  .22106165e+05  .33703446e-01", 
" -.41290741e+02  .49496979e+03 -.40171202e+04  .15899413e+05  .26503944e-01", 
" -.38059145e+02  .42456035e+03 -.31879241e+04  .11277280e+05  .19047322e-01", 
" -.36154732e+02  .38777944e+03 -.28928667e+04  .11006076e+05  .15388360e-01", 
" -.34792984e+02  .36034515e+03 -.25621674e+04  .88742361e+04  .13105273e-01", 
" -.33735309e+02  .34060071e+03 -.23611991e+04  .79105724e+04  .11496624e-01", 
" -.32865646e+02  .32404224e+03 -.21777503e+04  .69275929e+04  .11015108e-01", 
" -.32128226e+02  .31186770e+03 -.21000361e+04  .69581899e+04  .10397579e-01", 
" -.31488798e+02  .30153300e+03 -.20418883e+04  .71086022e+04  .97835751e-02", 
" -.30918518e+02  .29134666e+03 -.19315162e+04  .64766488e+04  .94567447e-02", 
" -.30408266e+02  .28266545e+03 -.18522259e+04  .61356135e+04  .91773094e-02", 
" -.28436958e+02  .25213606e+03 -.16371163e+04  .57605720e+04  .74902625e-02", 
" -.27014630e+02  .23052878e+03 -.14717859e+04  .52692565e+04  .65291948e-02", 
" -.25896081e+02  .21364431e+03 -.13304025e+04  .47351310e+04  .59718066e-02", 
" -.24974426e+02  .19962030e+03 -.11794310e+04  .37262017e+04  .55149270e-02", 
" -.24192664e+02  .18903609e+03 -.11111585e+04  .36084621e+04  .51025550e-02", 
" -.23505741e+02  .17926514e+03 -.10245700e+04  .31908815e+04  .48961938e-02", 
" -.22900361e+02  .17153354e+03 -.98235730e+03  .32200786e+04  .46649280e-02", 
" -.22353236e+02  .16452073e+03 -.94163418e+03  .32119109e+04  .43886106e-02", 
" -.21855432e+02  .15804234e+03 -.88852291e+03  .29564286e+04  .41893743e-02", 
" -.21400258e+02  .15258958e+03 -.86068807e+03  .29888014e+04  .40054702e-02", 
" -.20977675e+02  .14714669e+03 -.81135412e+03  .26977269e+04  .39778052e-02", 
" -.20584851e+02  .14235059e+03 -.77572375e+03  .25483575e+04  .39193847e-02", 
" -.20217380e+02  .13793943e+03 -.74470325e+03  .24370643e+04  .37422876e-02", 
" -.19871944e+02  .13393770e+03 -.71675810e+03  .23121645e+04  .36636392e-02", 
" -.19544540e+02  .12981742e+03 -.67524349e+03  .20300183e+04  .36215887e-02", 
" -.19234709e+02  .12625959e+03 -.65187144e+03  .19646231e+04  .35725250e-02", 
" -.18940137e+02  .12280384e+03 -.62413688e+03  .18253593e+04  .35476979e-02", 
" -.18658293e+02  .11933839e+03 -.58923916e+03  .15913302e+04  .34342310e-02", 
" -.18390589e+02  .11635745e+03 -.56903008e+03  .15284941e+04  .33617525e-02", 
" -.18134354e+02  .11360918e+03 -.55519551e+03  .15423241e+04  .32888224e-02", 
" -.17886886e+02  .11072429e+03 -.52837660e+03  .13645843e+04  .31859203e-02", 
" -.17649360e+02  .10793291e+03 -.49958995e+03  .11567177e+04  .31676154e-02", 
" -.17421836e+02  .10555619e+03 -.48561457e+03  .11185542e+04  .31104856e-02", 
" -.17201561e+02  .10311741e+03 -.46530104e+03  .10047978e+04  .30649616e-02", 
" -.16990414e+02  .10100818e+03 -.45406926e+03  .98179546e+03  .29910166e-02", 
" -.16785547e+02  .99069132e+02 -.44936383e+03  .10421000e+04  .29212364e-02", 
" -.16586503e+02  .96921419e+02 -.43046909e+03  .92453570e+03  .28710867e-02", 
" -.16393635e+02  .94931605e+02 -.41661431e+03  .86714261e+03  .27946331e-02", 
" -.16206105e+02  .93081926e+02 -.40745897e+03  .86664079e+03  .27760919e-02", 
" -.16024198e+02  .91279406e+02 -.39636053e+03  .83002162e+03  .27391945e-02", 
" -.15847283e+02  .89616484e+02 -.38900445e+03  .82729532e+03  .27149634e-02", 
" -.15674617e+02  .87828147e+02 -.37254261e+03  .70586019e+03  .26784082e-02", 
" -.15506691e+02  .86210914e+02 -.36303414e+03  .67883018e+03  .26093637e-02", 
" -.15342766e+02  .84704852e+02 -.35711476e+03  .69268065e+03  .25884966e-02", 
" -.15182582e+02  .83173960e+02 -.34965707e+03  .70753996e+03  .24878195e-02", 
" -.15025910e+02  .81765470e+02 -.34559177e+03  .74757402e+03  .24582513e-02", 
" -.14872028e+02  .80221742e+02 -.33308115e+03  .67555129e+03  .24144987e-02", 
" -.14722514e+02  .78891627e+02 -.32722118e+03  .67077645e+03  .23834094e-02", 
" -.14576008e+02  .77622024e+02 -.32361978e+03  .70067992e+03  .23493446e-02", 
" -.14432331e+02  .76285759e+02 -.31595060e+03  .68434612e+03  .23238499e-02", 
" -.14291786e+02  .75060383e+02 -.31144976e+03  .70167161e+03  .23054852e-02", 
" -.14153410e+02  .73731775e+02 -.30014962e+03  .62737434e+03  .22612661e-02", 
" -.14017942e+02  .72526708e+02 -.29366142e+03  .60858987e+03  .22216023e-02", 
" -.13885708e+02  .71461188e+02 -.29171943e+03  .64544310e+03  .21913502e-02", 
" -.13755308e+02  .70296340e+02 -.28388063e+03  .60627503e+03  .21571388e-02", 
" -.13627062e+02  .69148576e+02 -.27659912e+03  .57821702e+03  .21463936e-02", 
" -.13500826e+02  .68027924e+02 -.26897369e+03  .53460546e+03  .21360321e-02", 
" -.13376533e+02  .66893210e+02 -.25941678e+03  .46716319e+03  .20909070e-02", 
" -.13255005e+02  .65864305e+02 -.25364741e+03  .45122819e+03  .20687893e-02", 
" -.13135369e+02  .64903629e+02 -.25057083e+03  .46648462e+03  .20541964e-02", 
" -.13016691e+02  .63853829e+02 -.24349982e+03  .43675974e+03  .20466011e-02", 
" -.12900079e+02  .62852502e+02 -.23751056e+03  .41671005e+03  .20163833e-02", 
" -.12785234e+02  .61875333e+02 -.23135288e+03  .38875097e+03  .20103004e-02", 
" -.12671711e+02  .60827320e+02 -.22054124e+03  .29698480e+03  .19857969e-02", 
" -.12560708e+02  .59980549e+02 -.21737200e+03  .29026946e+03  .19622441e-02", 
" -.12450487e+02  .59007573e+02 -.20927834e+03  .23979752e+03  .19244236e-02", 
" -.12342203e+02  .58166571e+02 -.20582035e+03  .23294095e+03  .19059021e-02", 
" -.12234606e+02  .57189599e+02 -.19548618e+03  .13911153e+03  .18908975e-02", 
" -.12129089e+02  .56378304e+02 -.19385542e+03  .17016837e+03  .19004444e-02", 
" -.12024648e+02  .55553504e+02 -.19074249e+03  .18331718e+03  .18840347e-02", 
" -.11921481e+02  .54709100e+02 -.18469630e+03  .14633192e+03  .18759682e-02", 
" -.11819603e+02  .53960017e+02 -.18339080e+03  .16988864e+03  .18687380e-02", 
" -.11718929e+02  .53252764e+02 -.18335507e+03  .20554633e+03  .18673599e-02", 
" -.11619613e+02  .52535472e+02 -.18213178e+03  .23093369e+03  .18480284e-02", 
" -.11520825e+02  .51704408e+02 -.17369204e+03  .14802515e+03  .18528393e-02", 
" -.11423951e+02  .51036504e+02 -.17226888e+03  .15451153e+03  .18451035e-02", 
" -.11328043e+02  .50412048e+02 -.17364401e+03  .20856095e+03  .18166055e-02", 
" -.11233000e+02  .49727206e+02 -.17159964e+03  .21952981e+03  .17988357e-02", 
" -.11138718e+02  .48979571e+02 -.16501545e+03  .15956677e+03  .17735400e-02", 
" -.11045448e+02  .48324322e+02 -.16406222e+03  .18653413e+03  .17763105e-02", 
" -.10953105e+02  .47557412e+02 -.15630113e+03  .12354948e+03  .17665119e-02", 
" -.10861714e+02  .46924911e+02 -.15527480e+03  .14432167e+03  .17557653e-02", 
" -.10770797e+02  .46236199e+02 -.15221449e+03  .14841622e+03  .17168142e-02", 
" -.10681614e+02  .45661105e+02 -.15215059e+03  .17463723e+03  .17061069e-02", 
" -.10592417e+02  .44908669e+02 -.14343288e+03  .92704181e+02  .17055368e-02", 
" -.10504138e+02  .44284172e+02 -.14117354e+03  .94166580e+02  .16994614e-02", 
" -.10416424e+02  .43608852e+02 -.13605741e+03  .59035972e+02  .16948501e-02", 
" -.10329527e+02  .42939170e+02 -.13063318e+03  .18531246e+02  .16759878e-02", 
" -.10244016e+02  .42447236e+02 -.13361660e+03  .86310280e+02  .16700274e-02", 
" -.10159167e+02  .41908232e+02 -.13273019e+03  .92136989e+02  .16670271e-02", 
" -.10075141e+02  .41429999e+02 -.13540641e+03  .15277295e+03  .16490575e-02", 
" -.99911981e+01  .40869315e+02 -.13434562e+03  .16955628e+03  .16460727e-02", 
" -.99072460e+01  .40214676e+02 -.12859524e+03  .12660181e+03  .16272994e-02", 
" -.98243818e+01  .39641762e+02 -.12697508e+03  .14431126e+03  .16377780e-02", 
" -.97426799e+01  .39106317e+02 -.12456174e+03  .13101663e+03  .16255124e-02", 
" -.96611844e+01  .38582819e+02 -.12354593e+03  .14122856e+03  .16094563e-02", 
" -.95802217e+01  .38036095e+02 -.12146398e+03  .14385731e+03  .15790121e-02", 
" -.95000324e+01  .37550759e+02 -.12169993e+03  .17118354e+03  .15671423e-02", 
" -.94199846e+01  .37001359e+02 -.11908384e+03  .16634152e+03  .15620862e-02", 
" -.93409059e+01  .36518572e+02 -.11856314e+03  .18020530e+03  .15624563e-02", 
" -.92621297e+01  .35962593e+02 -.11427146e+03  .15249234e+03  .15544216e-02", 
" -.91835774e+01  .35386052e+02 -.10922657e+03  .11557969e+03  .15312055e-02", 
" -.91058383e+01  .34906113e+02 -.10844062e+03  .12991452e+03  .15403966e-02", 
" -.90283283e+01  .34405247e+02 -.10710758e+03  .14356075e+03  .15394481e-02", 
" -.89517188e+01  .33928030e+02 -.10537889e+03  .14596640e+03  .15183681e-02", 
" -.88750005e+01  .33360610e+02 -.10039458e+03  .11758500e+03  .15150587e-02", 
" -.87988673e+01  .32924458e+02 -.10141069e+03  .15591457e+03  .15020187e-02", 
" -.87233313e+01  .32478802e+02 -.10162346e+03  .18346184e+03  .15098477e-02", 
" -.86480869e+01  .32044037e+02 -.10158802e+03  .20322969e+03  .14975654e-02", 
" -.85732343e+01  .31530608e+02 -.97132971e+02  .16998126e+03  .14953801e-02", 
" -.84987061e+01  .31035456e+02 -.94397523e+02  .16566460e+03  .14989524e-02", 
" -.84244746e+01  .30490885e+02 -.87409635e+02  .90595954e+02  .14901678e-02", 
" -.83504811e+01  .29986351e+02 -.83492026e+02  .64451799e+02  .14958731e-02", 
" -.82770276e+01  .29570571e+02 -.82698395e+02  .69792840e+02  .14930349e-02", 
" -.82038508e+01  .29122302e+02 -.81202257e+02  .70603872e+02  .14784619e-02", 
" -.81309286e+01  .28659086e+02 -.78067303e+02  .47969347e+02  .14565854e-02", 
" -.80581411e+01  .28205565e+02 -.76051853e+02  .46003275e+02  .14502239e-02", 
" -.79857672e+01  .27729396e+02 -.71693806e+02  .50559558e+01  .14508345e-02", 
" -.79133737e+01  .27237404e+02 -.67305429e+02 -.28515002e+02  .14449133e-02", 
" -.78416967e+01  .26799137e+02 -.64373519e+02 -.52089597e+02  .14549835e-02", 
" -.77702843e+01  .26417389e+02 -.64741539e+02 -.27993536e+02  .14260764e-02", 
" -.76989770e+01  .25980259e+02 -.61420946e+02 -.57953989e+02  .14014582e-02", 
" -.76277367e+01  .25534952e+02 -.58820078e+02 -.65404049e+02  .14021230e-02", 
" -.75569383e+01  .25125116e+02 -.57243096e+02 -.65210639e+02  .13901412e-02", 
" -.74863097e+01  .24766011e+02 -.58418955e+02 -.28995414e+02  .13826828e-02", 
" -.74155077e+01  .24287902e+02 -.53312749e+02 -.74850433e+02  .13748083e-02", 
" -.73452639e+01  .23889633e+02 -.51780457e+02 -.80841112e+02  .13562041e-02", 
" -.72753726e+01  .23492695e+02 -.49029399e+02 -.10799430e+03  .13595447e-02", 
" -.72054574e+01  .23093279e+02 -.46990483e+02 -.11783366e+03  .13513654e-02", 
" -.71358972e+01  .22717486e+02 -.45559804e+02 -.12051894e+03  .13493612e-02", 
" -.70660052e+01  .22327377e+02 -.44440394e+02 -.11573721e+03  .13410023e-02", 
" -.69970316e+01  .21995201e+02 -.44624993e+02 -.10008892e+03  .13397500e-02", 
" -.69277087e+01  .21578431e+02 -.40858814e+02 -.13189680e+03  .13342522e-02", 
" -.68587958e+01  .21315339e+02 -.45369645e+02 -.56008537e+02  .13205692e-02", 
" -.67898281e+01  .20970294e+02 -.45075735e+02 -.46288036e+02  .13285241e-02", 
" -.67209243e+01  .20622967e+02 -.44620403e+02 -.39473626e+02  .13206692e-02", 
" -.66515644e+01  .20226008e+02 -.42313991e+02 -.51407141e+02  .13259582e-02", 
" -.65825738e+01  .19920964e+02 -.44452031e+02 -.64016364e+01  .13164623e-02", 
" -.65139125e+01  .19589792e+02 -.44972334e+02  .15955380e+02  .13181137e-02", 
" -.64446590e+01  .19142988e+02 -.39115806e+02 -.47382988e+02  .13256446e-02", 
" -.63756049e+01  .18775662e+02 -.37916450e+02 -.44577427e+02  .13282646e-02", 
" -.63070720e+01  .18420369e+02 -.36144412e+02 -.54627813e+02  .13163411e-02", 
" -.62381183e+01  .18053131e+02 -.34315255e+02 -.64564956e+02  .13153400e-02", 
" -.61691900e+01  .17719866e+02 -.34382061e+02 -.49043357e+02  .13018476e-02", 
" -.61005600e+01  .17409281e+02 -.34431590e+02 -.36401110e+02  .13085315e-02", 
" -.60314833e+01  .17049198e+02 -.33215436e+02 -.33710473e+02  .13080198e-02", 
" -.59627876e+01  .16763140e+02 -.34354492e+02 -.11184968e+02  .12884155e-02", 
" -.58940467e+01  .16449814e+02 -.33462533e+02 -.19561080e+02  .12927949e-02", 
" -.58250538e+01  .16140100e+02 -.33739965e+02 -.38016528e+01  .12894125e-02", 
" -.57559104e+01  .15781487e+02 -.31330133e+02 -.22819447e+02  .12837651e-02", 
" -.56870993e+01  .15491218e+02 -.31242190e+02 -.19229367e+02  .12655289e-02", 
" -.56175938e+01  .15126498e+02 -.28515282e+02 -.41496366e+02  .12699728e-02", 
" -.55479746e+01  .14760977e+02 -.25820201e+02 -.65029845e+02  .12540744e-02", 
" -.54782267e+01  .14410412e+02 -.23736933e+02 -.77744527e+02  .12535521e-02", 
" -.54081571e+01  .14036726e+02 -.20373658e+02 -.10788721e+03  .12312008e-02", 
" -.53382370e+01  .13729931e+02 -.20642969e+02 -.86955698e+02  .12285904e-02", 
" -.52679698e+01  .13339501e+02 -.15193225e+02 -.15118751e+03  .12129523e-02", 
" -.51973956e+01  .13015364e+02 -.14365603e+02 -.14893500e+03  .11969986e-02", 
" -.51267832e+01  .12701312e+02 -.13478723e+02 -.15055131e+03  .11947800e-02", 
" -.50556975e+01  .12361058e+02 -.11131057e+02 -.17261071e+03  .11936177e-02", 
" -.49845524e+01  .12074757e+02 -.11521756e+02 -.15578581e+03  .11790948e-02", 
" -.49124857e+01  .11718656e+02 -.97431742e+01 -.15814475e+03  .11722475e-02", 
" -.48406256e+01  .11429465e+02 -.10401263e+02 -.13703704e+03  .11931633e-02", 
" -.47685268e+01  .11167014e+02 -.12153259e+02 -.10240641e+03  .12012864e-02", 
" -.46957244e+01  .10810256e+02 -.86813797e+01 -.13897884e+03  .11899109e-02", 
" -.46228816e+01  .10514173e+02 -.77119874e+01 -.14945535e+03  .11992126e-02", 
" -.45495525e+01  .10242025e+02 -.84596460e+01 -.12850695e+03  .11879191e-02", 
" -.44757837e+01  .10003992e+02 -.10735800e+02 -.91455349e+02  .11958939e-02", 
" -.44013255e+01  .96623576e+01 -.82647252e+01 -.10828304e+03  .12022992e-02", 
" -.43264415e+01  .93677819e+01 -.74508853e+01 -.11113969e+03  .12036411e-02", 
" -.42505489e+01  .90375897e+01 -.55277006e+01 -.12757389e+03  .11837570e-02", 
" -.41741778e+01  .87410678e+01 -.51835921e+01 -.12225215e+03  .11895704e-02", 
" -.40972558e+01  .84738862e+01 -.66757053e+01 -.88640077e+02  .11946903e-02", 
" -.40191842e+01  .81631161e+01 -.62907107e+01 -.77238970e+02  .11957211e-02", 
" -.39405758e+01  .78176002e+01 -.35567376e+01 -.10144135e+03  .11869428e-02", 
" -.38611092e+01  .75735491e+01 -.69687628e+01 -.41193148e+02  .11898651e-02", 
" -.37810007e+01  .72753847e+01 -.59559540e+01 -.45564589e+02  .11844384e-02", 
" -.36997720e+01  .69947289e+01 -.68262035e+01 -.23340674e+02  .11943893e-02", 
" -.36176377e+01  .66949411e+01 -.58006179e+01 -.29776034e+02  .12148879e-02", 
" -.35341296e+01  .64137086e+01 -.70676053e+01 -.87320149e+00  .12118566e-02", 
" -.34494358e+01  .60763271e+01 -.48060161e+01 -.21727100e+02  .12270148e-02", 
" -.33636741e+01  .57701212e+01 -.40856119e+01 -.15285233e+02  .12099798e-02", 
" -.32765579e+01  .54372563e+01 -.11531998e+01 -.48191130e+02  .12166714e-02", 
" -.31879776e+01  .52120120e+01 -.43299813e+01 -.29047493e+01  .12283793e-02", 
" -.30975274e+01  .50055165e+01 -.99298648e+01  .89342512e+02  .12293203e-02", 
" -.30056487e+01  .47168036e+01 -.93782093e+01  .88337626e+02  .12222616e-02", 
" -.29111216e+01  .42973863e+01 -.28667793e+01  .10038020e+02  .12206486e-02", 
" -.28151448e+01  .40395554e+01 -.37269843e+01  .24613111e+02  .12048709e-02", 
" -.27160508e+01  .36936255e+01 -.13198744e+01 -.16399782e+01  .12299952e-02", 
" -.26146765e+01  .34621751e+01 -.52683524e+01  .61856861e+02  .12454017e-02", 
" -.25100984e+01  .30414357e+01  .23289402e+01 -.43095960e+02  .12474872e-02", 
" -.24021019e+01  .26714595e+01  .60008921e+01 -.87047696e+02  .12476096e-02", 
" -.22903065e+01  .23121150e+01  .79165815e+01 -.10412718e+03  .12668573e-02", 
" -.21751374e+01  .21740488e+01  .61649776e+00 -.10942105e+02  .12752392e-02", 
" -.20543421e+01  .18221167e+01  .20542382e+01 -.19407899e+02  .12689617e-02", 
" -.19272681e+01  .14153031e+01  .57063431e+01 -.58231862e+02  .13042905e-02", 
" -.17941930e+01  .10572690e+01  .85284667e+01 -.86506229e+02  .13119816e-02", 
" -.16533094e+01  .77092980e+00  .69124445e+01 -.54604586e+02  .13525772e-02", 
" -.15013698e+01  .32770424e+00  .10813057e+02 -.95562824e+02  .13838453e-02", 
" -.13393612e+01  .15107276e+00  .56114943e+01 -.51463296e+02  .14224230e-02", 
" -.11623542e+01  .67027485e-02 -.32602108e+01  .61720298e+02  .14380918e-02", 
" -.96385054e+00 -.51270409e+00  .71216731e+01 -.75750404e+02  .15035462e-02", 
" -.73824937e+00 -.78098611e+00  .55702899e+01 -.60371123e+02  .15873100e-02", 
" -.47246992e+00 -.10560420e+01  .86583399e+01 -.10048844e+03  .17384832e-02", 
" -.14992597e+00 -.11463059e+01  .13261859e+02 -.19076920e+03  .18228549e-02", 
"  .26697004e+00 -.91248082e+00  .81758421e+01 -.13334740e+03  .19397881e-02", 
"  .36770059e+00 -.72047355e+00  .10173144e+01 -.28508902e+02  .20193023e-02", 
"  .47828800e+00 -.64895951e+00  .52303815e+01 -.10533897e+03  .20742350e-02", 
"  .60102088e+00 -.59623765e+00  .41533624e+01 -.39914024e+02  .21373819e-02", 
"  .73750213e+00 -.36542383e+00  .10372514e+01  .65462804e+01  .22257217e-02", 
"  .89257552e+00 -.14498497e+00  .53389776e+01 -.49084715e+02  .22769912e-02", 
"  .10764052e+01  .18958057e+00 -.32797988e+00  .72406934e+02  .24658891e-02", 
"  .13041964e+01  .42201437e+00  .16847815e+02 -.16490192e+03  .29397185e-02", 
"  .16116339e+01  .98792262e+00  .25638564e+02 -.23919067e+03  .36052725e-02", 
"  .21085155e+01  .30208595e+01  .13067908e+02 -.33909851e+02  .47425851e-02", 
"  .25774180e+01  .47639700e+01  .46898457e+02 -.51006252e+03  .65020090e-02", 
"  .31603250e+01  .10175372e+02 -.41217643e+02  .74307701e+03  .98501634e-02", 
"  .36100914e+01  .11511237e+02 -.20408728e+01  .52392126e+03  .13457807e-01", 
"coac      2  4  3   20", " -.57336119e+02  .88456435e+03 -.85307733e+04  .37571813e+05  .81549359e-01", 
" -.54115101e+02  .79356154e+03 -.73838677e+04  .31931435e+05  .58308959e-01", 
" -.49751534e+02  .67746691e+03 -.60165668e+04  .25922185e+05  .38299748e-01", 
" -.46454658e+02  .59537469e+03 -.50524056e+04  .21389442e+05  .28283042e-01", 
" -.43136089e+02  .52156199e+03 -.43090537e+04  .18569369e+05  .19540772e-01", 
" -.41170054e+02  .47875374e+03 -.38264907e+04  .15867648e+05  .15987736e-01", 
" -.39750644e+02  .44915149e+03 -.35206918e+04  .14381848e+05  .13372225e-01", 
" -.38641270e+02  .42542524e+03 -.32618085e+04  .13145221e+05  .12518080e-01", 
" -.37729298e+02  .40691274e+03 -.30676544e+04  .12201236e+05  .11244492e-01", 
" -.36951789e+02  .39135142e+03 -.29209728e+04  .11691308e+05  .10746284e-01", 
" -.36278847e+02  .37813736e+03 -.27698722e+04  .10708596e+05  .10277202e-01", 
" -.35681701e+02  .36655816e+03 -.26519843e+04  .10160888e+05  .10133955e-01", 
" -.35147560e+02  .35689511e+03 -.25742020e+04  .99988847e+04  .92750025e-02", 
" -.33063553e+02  .31874885e+03 -.22186238e+04  .85718374e+04  .78116667e-02", 
" -.31560226e+02  .29106582e+03 -.19090478e+04  .67217486e+04  .67543339e-02", 
" -.30389761e+02  .27201774e+03 -.17660014e+04  .64033170e+04  .60596100e-02", 
" -.29418634e+02  .25577786e+03 -.16046436e+04  .55044626e+04  .55647720e-02", 
" -.28591026e+02  .24273395e+03 -.14928761e+04  .50089140e+04  .51900444e-02", 
" -.27867120e+02  .23173569e+03 -.14153304e+04  .48761948e+04  .48586395e-02", 
" -.27220906e+02  .22179376e+03 -.13332675e+04  .45705006e+04  .46398567e-02", 
" -.26641060e+02  .21327852e+03 -.12666051e+04  .43004365e+04  .44669966e-02", 
" -.26112435e+02  .20546378e+03 -.12022699e+04  .40542205e+04  .42150811e-02", 
" -.25625296e+02  .19837613e+03 -.11420520e+04  .37651402e+04  .40632278e-02", 
" -.25174266e+02  .19183168e+03 -.10832075e+04  .34505755e+04  .40068342e-02", 
" -.24755444e+02  .18608884e+03 -.10427325e+04  .33275415e+04  .38819644e-02", 
" -.24361212e+02  .18030486e+03 -.98643297e+03  .30095097e+04  .37554429e-02", 
" -.23990291e+02  .17509759e+03 -.94359593e+03  .28436356e+04  .36102448e-02", 
" -.23641422e+02  .17053851e+03 -.91369328e+03  .27611857e+04  .34965071e-02", 
" -.23310659e+02  .16633459e+03 -.89118299e+03  .27685527e+04  .34557985e-02", 
" -.22995572e+02  .16231631e+03 -.86901676e+03  .27782271e+04  .33748458e-02", 
" -.22694592e+02  .15836196e+03 -.83826750e+03  .26472343e+04  .32577354e-02", 
" -.22407735e+02  .15472113e+03 -.81281754e+03  .25569532e+04  .31823941e-02", 
" -.22131822e+02  .15123516e+03 -.78917046e+03  .24842033e+04  .31396578e-02", 
" -.21868118e+02  .14797424e+03 -.76813835e+03  .24371159e+04  .30413280e-02", 
" -.21614197e+02  .14484541e+03 -.74565890e+03  .23457732e+04  .29433890e-02", 
" -.21369031e+02  .14190863e+03 -.72980830e+03  .23425381e+04  .28719971e-02", 
" -.21133296e+02  .13906321e+03 -.70732114e+03  .22074866e+04  .28387498e-02", 
" -.20905065e+02  .13627214e+03 -.68488357e+03  .20839978e+04  .27256541e-02", 
" -.20683776e+02  .13371514e+03 -.67060919e+03  .20606501e+04  .27326236e-02", 
" -.20468452e+02  .13104674e+03 -.64672774e+03  .19114280e+04  .27422924e-02", 
" -.20260148e+02  .12857587e+03 -.63006273e+03  .18578913e+04  .27005881e-02", 
" -.20057933e+02  .12626062e+03 -.61711742e+03  .18550071e+04  .26943683e-02", 
" -.19861267e+02  .12397033e+03 -.60020055e+03  .17707181e+04  .26879674e-02", 
" -.19668695e+02  .12164682e+03 -.58192914e+03  .16956388e+04  .26819537e-02", 
" -.19480848e+02  .11938198e+03 -.56078752e+03  .15529532e+04  .26678818e-02", 
" -.19298428e+02  .11727793e+03 -.54601680e+03  .14971496e+04  .26319869e-02", 
" -.19120039e+02  .11525964e+03 -.53388243e+03  .14732353e+04  .25792741e-02", 
" -.18946015e+02  .11330108e+03 -.51920623e+03  .13981833e+04  .25254904e-02", 
" -.18775482e+02  .11144142e+03 -.50932094e+03  .13941494e+04  .24932116e-02", 
" -.18608841e+02  .10949421e+03 -.49131373e+03  .12736691e+04  .25120916e-02", 
" -.18446561e+02  .10778163e+03 -.48278690e+03  .12745554e+04  .24850031e-02", 
" -.18287552e+02  .10606760e+03 -.47189080e+03  .12388088e+04  .24592025e-02", 
" -.18131342e+02  .10443009e+03 -.46372122e+03  .12390876e+04  .24209339e-02", 
" -.17978070e+02  .10277814e+03 -.45238743e+03  .11937782e+04  .24498271e-02", 
" -.17827730e+02  .10117182e+03 -.44189526e+03  .11589223e+04  .24107817e-02", 
" -.17679843e+02  .99620252e+02 -.43287951e+03  .11361390e+04  .24009727e-02", 
" -.17534712e+02  .98100477e+02 -.42403576e+03  .11202901e+04  .23979014e-02", 
" -.17392496e+02  .96603243e+02 -.41186824e+03  .10354665e+04  .23525510e-02", 
" -.17252220e+02  .95071766e+02 -.39875558e+03  .95761467e+03  .23192575e-02", 
" -.17113935e+02  .93661065e+02 -.38981283e+03  .91916937e+03  .23061912e-02", 
" -.16978266e+02  .92235662e+02 -.37816559e+03  .84074815e+03  .23177920e-02", 
" -.16844247e+02  .90765708e+02 -.36471312e+03  .75809835e+03  .22616472e-02", 
" -.16712573e+02  .89363782e+02 -.35370130e+03  .70422219e+03  .22031022e-02", 
" -.16582778e+02  .88140044e+02 -.35031123e+03  .73688426e+03  .21877836e-02", 
" -.16455968e+02  .86998277e+02 -.34835043e+03  .78627098e+03  .21668960e-02", 
" -.16330378e+02  .85797859e+02 -.34127190e+03  .75228365e+03  .21624494e-02", 
" -.16206184e+02  .84635397e+02 -.33805526e+03  .79071585e+03  .21731739e-02", 
" -.16083615e+02  .83501339e+02 -.33355681e+03  .79603437e+03  .21658004e-02", 
" -.15962279e+02  .82259080e+02 -.32366851e+03  .73696879e+03  .21487168e-02", 
" -.15843096e+02  .81134728e+02 -.31800205e+03  .72916103e+03  .21017693e-02", 
" -.15725555e+02  .80033533e+02 -.31195732e+03  .70938186e+03  .20928458e-02", 
" -.15609549e+02  .78903280e+02 -.30406513e+03  .67262916e+03  .20690509e-02", 
" -.15494002e+02  .77705840e+02 -.29256554e+03  .58753181e+03  .20723628e-02", 
" -.15380804e+02  .76694519e+02 -.28890519e+03  .59914270e+03  .20699038e-02", 
" -.15268448e+02  .75633255e+02 -.28123539e+03  .54875697e+03  .20271305e-02", 
" -.15157657e+02  .74673199e+02 -.27805128e+03  .55367370e+03  .20118386e-02", 
" -.15047854e+02  .73681756e+02 -.27385886e+03  .55721372e+03  .19917338e-02", 
" -.14939023e+02  .72653468e+02 -.26638222e+03  .51023717e+03  .19812296e-02", 
" -.14831119e+02  .71583825e+02 -.25715631e+03  .45441012e+03  .19543369e-02", 
" -.14724669e+02  .70634472e+02 -.25285639e+03  .45725326e+03  .19300452e-02", 
" -.14619376e+02  .69699588e+02 -.24848007e+03  .45062934e+03  .19106777e-02", 
" -.14515256e+02  .68841467e+02 -.24733900e+03  .48918068e+03  .19079008e-02", 
" -.14412326e+02  .68037331e+02 -.24811897e+03  .54909274e+03  .18981390e-02", 
" -.14310209e+02  .67061206e+02 -.23857245e+03  .47363953e+03  .18919241e-02", 
" -.14208914e+02  .66165164e+02 -.23371557e+03  .46102643e+03  .18746286e-02", 
" -.14108206e+02  .65238837e+02 -.22597499e+03  .40163643e+03  .18487599e-02", 
" -.14008905e+02  .64379875e+02 -.22085315e+03  .37763063e+03  .18344624e-02", 
" -.13910278e+02  .63545485e+02 -.21666565e+03  .36322702e+03  .18240850e-02", 
" -.13812728e+02  .62781226e+02 -.21540622e+03  .38147457e+03  .17953748e-02", 
" -.13715888e+02  .62007969e+02 -.21301859e+03  .38813628e+03  .17988120e-02", 
" -.13619210e+02  .61080434e+02 -.20333116e+03  .30894208e+03  .17558588e-02", 
" -.13523264e+02  .60255266e+02 -.19900430e+03  .29859216e+03  .17520392e-02", 
" -.13427962e+02  .59438028e+02 -.19544553e+03  .30126728e+03  .17412122e-02", 
" -.13333722e+02  .58643018e+02 -.19095862e+03  .28313645e+03  .17402316e-02", 
" -.13240588e+02  .57973846e+02 -.19165424e+03  .32779550e+03  .17198038e-02", 
" -.13147806e+02  .57167828e+02 -.18514173e+03  .28257196e+03  .17413843e-02", 
" -.13055776e+02  .56476515e+02 -.18369194e+03  .29231192e+03  .17292314e-02", 
" -.12963872e+02  .55647017e+02 -.17559542e+03  .22785173e+03  .17268912e-02", 
" -.12873155e+02  .54989409e+02 -.17546549e+03  .26104646e+03  .17188532e-02", 
" -.12782423e+02  .54194293e+02 -.16914034e+03  .22001016e+03  .16946393e-02", 
" -.12692783e+02  .53477924e+02 -.16460510e+03  .19017309e+03  .16958211e-02", 
" -.12603387e+02  .52724574e+02 -.15899009e+03  .15432759e+03  .17087283e-02", 
" -.12514271e+02  .51973813e+02 -.15340522e+03  .11811130e+03  .17012871e-02", 
" -.12425953e+02  .51308473e+02 -.15259336e+03  .14719101e+03  .16849000e-02", 
" -.12338517e+02  .50697433e+02 -.15297820e+03  .18340430e+03  .16749655e-02", 
" -.12251099e+02  .50012194e+02 -.15035908e+03  .18671802e+03  .16780624e-02", 
" -.12164050e+02  .49311781e+02 -.14623603e+03  .16989208e+03  .16674956e-02", 
" -.12077083e+02  .48548799e+02 -.13942468e+03  .12663863e+03  .16571050e-02", 
" -.11991006e+02  .47839068e+02 -.13345012e+03  .76402240e+02  .16530583e-02", 
" -.11905240e+02  .47171588e+02 -.13071677e+03  .81090858e+02  .16513667e-02", 
" -.11820125e+02  .46549009e+02 -.12886639e+03  .92588203e+02  .16511149e-02", 
" -.11735591e+02  .45898406e+02 -.12490049e+03  .72917353e+02  .16415843e-02", 
" -.11651166e+02  .45250664e+02 -.12100710e+03  .53719449e+02  .16530294e-02", 
" -.11567321e+02  .44676562e+02 -.12064162e+03  .74184063e+02  .16549132e-02", 
" -.11483294e+02  .44051488e+02 -.11809263e+03  .75237796e+02  .16344530e-02", 
" -.11399971e+02  .43435337e+02 -.11510698e+03  .65598533e+02  .16301405e-02", 
" -.11317162e+02  .42863092e+02 -.11401012e+03  .80645794e+02  .16111122e-02", 
" -.11234491e+02  .42248928e+02 -.11040587e+03  .60607299e+02  .16058471e-02", 
" -.11152159e+02  .41668871e+02 -.10778415e+03  .49439396e+02  .16036128e-02", 
" -.11070025e+02  .41131374e+02 -.10845377e+03  .86311523e+02  .16052749e-02", 
" -.10987811e+02  .40501816e+02 -.10475162e+03  .76388761e+02  .16163039e-02", 
" -.10906001e+02  .39931397e+02 -.10287666e+03  .76662175e+02  .16176005e-02", 
" -.10824564e+02  .39320569e+02 -.97988001e+02  .35708283e+02  .16178390e-02", 
" -.10743409e+02  .38775646e+02 -.96962385e+02  .50124524e+02  .16239472e-02", 
" -.10662104e+02  .38161423e+02 -.93072589e+02  .31860450e+02  .16286545e-02", 
" -.10581472e+02  .37687854e+02 -.95570360e+02  .92917564e+02  .16071254e-02", 
" -.10501081e+02  .37191560e+02 -.96311932e+02  .12871362e+03  .16068036e-02", 
" -.10420617e+02  .36624906e+02 -.93671833e+02  .12299024e+03  .16126482e-02", 
" -.10340407e+02  .36062131e+02 -.90832265e+02  .11149140e+03  .15956767e-02", 
" -.10260538e+02  .35523374e+02 -.88205490e+02  .93369575e+02  .15709057e-02", 
" -.10180857e+02  .35021912e+02 -.86861237e+02  .92696877e+02  .15679993e-02", 
" -.10101196e+02  .34499345e+02 -.84406462e+02  .74160528e+02  .15596902e-02", 
" -.10021544e+02  .34018710e+02 -.85221895e+02  .10919051e+03  .15543153e-02", 
" -.99418589e+01  .33482812e+02 -.82873435e+02  .10125674e+03  .15540766e-02", 
" -.98623302e+01  .32947371e+02 -.79772621e+02  .79229406e+02  .15656891e-02", 
" -.97830674e+01  .32486739e+02 -.80934843e+02  .11649297e+03  .15594386e-02", 
" -.97039368e+01  .31982275e+02 -.79148040e+02  .11580477e+03  .15560584e-02", 
" -.96245552e+01  .31444074e+02 -.75608389e+02  .89594579e+02  .15391631e-02", 
" -.95455146e+01  .30983022e+02 -.74902746e+02  .91639497e+02  .15354786e-02", 
" -.94664253e+01  .30512089e+02 -.74124544e+02  .95430152e+02  .15321517e-02", 
" -.93872189e+01  .29990070e+02 -.71185727e+02  .80567365e+02  .15216508e-02", 
" -.93079705e+01  .29507443e+02 -.70115650e+02  .88672545e+02  .15374967e-02", 
" -.92286940e+01  .28970405e+02 -.66833328e+02  .70706339e+02  .15476859e-02", 
" -.91488581e+01  .28405904e+02 -.62413489e+02  .41077879e+02  .15592280e-02", 
" -.90696729e+01  .27904562e+02 -.59405309e+02  .22421191e+02  .15630441e-02", 
" -.89903285e+01  .27441589e+02 -.58908766e+02  .38364252e+02  .15672268e-02", 
" -.89108963e+01  .26958675e+02 -.57782086e+02  .51711150e+02  .15736361e-02", 
" -.88310817e+01  .26447919e+02 -.55098161e+02  .41337024e+02  .15755932e-02", 
" -.87509254e+01  .25904105e+02 -.50528012e+02  .26692936e+00  .15611391e-02", 
" -.86708908e+01  .25452906e+02 -.51498252e+02  .37259856e+02  .15685045e-02", 
" -.85909668e+01  .25020776e+02 -.52639643e+02  .76406520e+02  .15653314e-02", 
" -.85109340e+01  .24538791e+02 -.50307340e+02  .62810323e+02  .15618709e-02", 
" -.84307562e+01  .24035615e+02 -.46466176e+02  .28805644e+02  .15404477e-02", 
" -.83505285e+01  .23611572e+02 -.47116519e+02  .59053704e+02  .15186980e-02", 
" -.82698740e+01  .23137684e+02 -.45556211e+02  .60360242e+02  .14972633e-02", 
" -.81886496e+01  .22595347e+02 -.40316792e+02  .10665318e+02  .14921541e-02", 
" -.81075272e+01  .22092356e+02 -.35395217e+02 -.44578197e+02  .14822598e-02", 
" -.80261654e+01  .21604068e+02 -.32896159e+02 -.54021060e+02  .14956285e-02", 
" -.79447370e+01  .21242698e+02 -.36646293e+02  .14384724e+02  .15068984e-02", 
" -.78628358e+01  .20803321e+02 -.35952517e+02  .21980894e+02  .14927436e-02", 
" -.77802835e+01  .20356829e+02 -.35391858e+02  .33232439e+02  .14695965e-02", 
" -.76980086e+01  .19988851e+02 -.37355181e+02  .67255094e+02  .14416566e-02", 
" -.76152588e+01  .19593886e+02 -.38312073e+02  .91624094e+02  .14266162e-02", 
" -.75316359e+01  .19107382e+02 -.35044409e+02  .66849597e+02  .14291586e-02", 
" -.74475997e+01  .18655485e+02 -.33777966e+02  .70601992e+02  .14331940e-02", 
" -.73628808e+01  .18164014e+02 -.30881900e+02  .49660000e+02  .14292454e-02", 
" -.72779015e+01  .17741608e+02 -.31217295e+02  .69904590e+02  .14150794e-02", 
" -.71923138e+01  .17268760e+02 -.28204973e+02  .41607035e+02  .14093090e-02", 
" -.71063767e+01  .16852483e+02 -.28522096e+02  .67001427e+02  .13888338e-02", 
" -.70196001e+01  .16333287e+02 -.21902456e+02 -.15967365e+02  .13946461e-02", 
" -.69321709e+01  .15875258e+02 -.20395042e+02 -.18313950e+02  .13956235e-02", 
" -.68444215e+01  .15495263e+02 -.21746887e+02  .94343413e+01  .14037505e-02", 
" -.67554516e+01  .15041797e+02 -.19753327e+02 -.60876788e+01  .14096654e-02", 
" -.66656749e+01  .14590845e+02 -.17588976e+02 -.25805167e+02  .14077929e-02", 
" -.65747469e+01  .14093350e+02 -.13806314e+02 -.60310118e+02  .14034368e-02", 
" -.64834852e+01  .13695091e+02 -.15008260e+02 -.28476335e+02  .14047488e-02", 
" -.63912676e+01  .13290387e+02 -.15306555e+02 -.12118633e+02  .14013303e-02", 
" -.62982317e+01  .12951667e+02 -.19545336e+02  .60131069e+02  .13906720e-02", 
" -.62033785e+01  .12450989e+02 -.15692238e+02  .26214264e+02  .13808698e-02", 
" -.61080009e+01  .12087534e+02 -.18327771e+02  .73098845e+02  .13654796e-02", 
" -.60110645e+01  .11594830e+02 -.14070812e+02  .28930563e+02  .13589926e-02", 
" -.59123715e+01  .11151775e+02 -.14071392e+02  .45571724e+02  .13669779e-02", 
" -.58122679e+01  .10745707e+02 -.17538568e+02  .12180314e+03  .13781876e-02", 
" -.57109416e+01  .10369036e+02 -.20435449e+02  .17280054e+03  .13848881e-02", 
" -.56079556e+01  .99480665e+01 -.19944653e+02  .17478735e+03  .13710679e-02", 
" -.55025962e+01  .94890333e+01 -.19752635e+02  .18612680e+03  .13964041e-02", 
" -.53954822e+01  .90062585e+01 -.16835460e+02  .15530470e+03  .14181709e-02", 
" -.52863485e+01  .84971508e+01 -.12542661e+02  .10809463e+03  .14207504e-02", 
" -.51748794e+01  .80786007e+01 -.14778461e+02  .15623489e+03  .14254260e-02", 
" -.50604336e+01  .74882169e+01 -.78313757e+01  .80763172e+02  .14012677e-02", 
" -.49432735e+01  .70924083e+01 -.11867434e+02  .15324233e+03  .14270126e-02", 
" -.48229462e+01  .65417166e+01 -.76443290e+01  .11384227e+03  .14649891e-02", 
" -.46988771e+01  .60255042e+01 -.67161479e+01  .11926202e+03  .14821770e-02", 
" -.45714239e+01  .54960854e+01 -.35258970e+01  .91017908e+02  .14697638e-02", 
" -.44404057e+01  .51163914e+01 -.70588571e+01  .13523126e+03  .15033105e-02", 
" -.43033490e+01  .44493589e+01  .59277473e+00  .59712313e+02  .15300154e-02", 
" -.41612270e+01  .38842081e+01  .46374999e+01  .65643581e+01  .15590240e-02", 
" -.40133897e+01  .33531409e+01  .63154517e+01  .98416752e+00  .15668150e-02", 
" -.38575970e+01  .26626835e+01  .15307105e+02 -.10667826e+03  .15635396e-02", 
" -.36935147e+01  .20149911e+01  .21261373e+02 -.16810570e+03  .16274200e-02", 
" -.35194043e+01  .14229146e+01  .23164301e+02 -.18361811e+03  .16475161e-02", 
" -.33320921e+01  .42996676e+00  .40978447e+02 -.37645703e+03  .17018288e-02", 
" -.31325929e+01 -.24231990e-01  .33185278e+02 -.25050417e+03  .17698980e-02", 
" -.29145975e+01 -.50004721e+00  .25561558e+02 -.13516368e+03  .17811496e-02", 
" -.26722933e+01 -.12716098e+01  .34596379e+02 -.26945931e+03  .18606660e-02", 
" -.23984800e+01 -.18417329e+01  .30036347e+02 -.21503229e+03  .18717557e-02", 
" -.20781606e+01 -.27487060e+01  .37673155e+02 -.28290312e+03  .20010054e-02", 
" -.16917318e+01 -.33937287e+01  .37668967e+02 -.29147052e+03  .22255373e-02", 
" -.11888709e+01 -.33884637e+01  .58696259e-01  .19328379e+03  .24090520e-02", 
" -.10670250e+01 -.32639792e+01 -.10620308e+02  .32069534e+03  .24670380e-02", 
" -.93228448e+00 -.34318274e+01 -.54395213e+01  .22401572e+03  .25908677e-02", 
" -.78207605e+00 -.38711342e+01  .99822049e+01  .37558466e+01  .27326453e-02", 
" -.61398196e+00 -.43381999e+01  .28033140e+02 -.25783810e+03  .27910040e-02", 
" -.42429082e+00 -.43413423e+01  .27844391e+02 -.30483892e+03  .30569708e-02", 
" -.20082300e+00 -.44861154e+01  .33924902e+02 -.40758696e+03  .32329311e-02", 
"  .74671196e-01 -.48093632e+01  .56022957e+02 -.78874585e+03  .35842736e-02", 
"  .44246337e+00 -.49963708e+01  .60058678e+02 -.81334365e+03  .39093127e-02", 
"  .10225651e+01 -.54281490e+01  .11306102e+03 -.16593941e+04  .51081645e-02", 
"  .15481636e+01 -.37096172e+01  .82254789e+02 -.15177408e+04  .73887900e-02", 
"  .21975481e+01  .51385945e+00 -.46192269e+02 -.14281164e+03  .10547025e-01", 
"  .26698158e+01  .26874617e+01 -.31605093e+02 -.11492941e+04  .14239262e-01", 
"coact     2  5  3   20", " -.65626356e+02  .11859188e+04 -.14416961e+05  .86978743e+05  .83333794e-01", 
" -.62164860e+02  .10559096e+04 -.11834103e+05  .65555238e+05  .57666994e-01", 
" -.57608324e+02  .90939071e+03 -.94522033e+04  .48838325e+05  .35977492e-01", 
" -.54113563e+02  .80552062e+03 -.79226495e+04  .39160205e+05  .26797155e-01", 
" -.50567078e+02  .70934738e+03 -.66261005e+04  .31319136e+05  .20086540e-01", 
" -.48444844e+02  .65364075e+03 -.58761369e+04  .26599669e+05  .17505894e-01", 
" -.46937026e+02  .61635529e+03 -.54375159e+04  .24431819e+05  .15881764e-01", 
" -.45761793e+02  .58939709e+03 -.51621246e+04  .23372614e+05  .14396927e-01", 
" -.44770404e+02  .56453581e+03 -.48249837e+04  .21212705e+05  .12838685e-01", 
" -.43937115e+02  .54379859e+03 -.45360461e+04  .19293357e+05  .11866572e-01", 
" -.43217811e+02  .52839317e+03 -.44099432e+04  .19176068e+05  .11202050e-01", 
" -.42574585e+02  .51429837e+03 -.42783496e+04  .18811762e+05  .10728576e-01", 
" -.41994817e+02  .50049576e+03 -.40992944e+04  .17746285e+05  .10307480e-01", 
" -.39757946e+02  .45255008e+03 -.35928404e+04  .15421609e+05  .79233249e-02", 
" -.38142250e+02  .41995004e+03 -.32850493e+04  .14261467e+05  .71495502e-02", 
" -.36866277e+02  .39397462e+03 -.30116700e+04  .12948206e+05  .66334492e-02", 
" -.35817890e+02  .37401959e+03 -.28257229e+04  .12216918e+05  .62913918e-02", 
" -.34912779e+02  .35562658e+03 -.26038116e+04  .10850401e+05  .56960762e-02", 
" -.34123935e+02  .34104911e+03 -.24696102e+04  .10319580e+05  .53775826e-02", 
" -.33421242e+02  .32806531e+03 -.23423126e+04  .97440356e+04  .50203596e-02", 
" -.32787500e+02  .31677220e+03 -.22417030e+04  .93470445e+04  .47673515e-02", 
" -.32208029e+02  .30627823e+03 -.21381812e+04  .88502161e+04  .46093415e-02", 
" -.31675932e+02  .29673793e+03 -.20321992e+04  .81534724e+04  .44728533e-02", 
" -.31184268e+02  .28834650e+03 -.19560317e+04  .78188154e+04  .43567542e-02", 
" -.30724382e+02  .28053893e+03 -.18877293e+04  .75360827e+04  .42882634e-02", 
" -.30293558e+02  .27330730e+03 -.18225621e+04  .72490672e+04  .41225410e-02", 
" -.29886530e+02  .26622024e+03 -.17434371e+04  .67311105e+04  .39902187e-02", 
" -.29501809e+02  .26007095e+03 -.16997506e+04  .66745421e+04  .38150298e-02", 
" -.29136377e+02  .25367442e+03 -.16230273e+04  .61401831e+04  .37214920e-02", 
" -.28788349e+02  .24807206e+03 -.15776914e+04  .60019245e+04  .36743218e-02", 
" -.28456337e+02  .24275503e+03 -.15311556e+04  .57906016e+04  .35260213e-02", 
" -.28138669e+02  .23755338e+03 -.14786778e+04  .54894318e+04  .35349380e-02", 
" -.27833334e+02  .23260237e+03 -.14297404e+04  .52043220e+04  .34395420e-02", 
" -.27540593e+02  .22813943e+03 -.13970269e+04  .51396169e+04  .33157626e-02", 
" -.27258570e+02  .22376473e+03 -.13573540e+04  .49333177e+04  .32974076e-02", 
" -.26987186e+02  .21960630e+03 -.13192979e+04  .47272116e+04  .32794209e-02", 
" -.26724032e+02  .21567422e+03 -.12908118e+04  .46582272e+04  .32162198e-02", 
" -.26470634e+02  .21195962e+03 -.12647118e+04  .46121463e+04  .31814280e-02", 
" -.26223741e+02  .20809663e+03 -.12262328e+04  .43900851e+04  .31555678e-02", 
" -.25985334e+02  .20446446e+03 -.11897331e+04  .41587846e+04  .30794325e-02", 
" -.25752790e+02  .20099550e+03 -.11577497e+04  .39758736e+04  .30097941e-02", 
" -.25527350e+02  .19779417e+03 -.11358729e+04  .39299751e+04  .29900842e-02", 
" -.25307183e+02  .19466175e+03 -.11138980e+04  .38715076e+04  .29863147e-02", 
" -.25091959e+02  .19139469e+03 -.10808548e+04  .36644930e+04  .29500514e-02", 
" -.24883794e+02  .18858345e+03 -.10654083e+04  .36766647e+04  .29043173e-02", 
" -.24680007e+02  .18558836e+03 -.10365199e+04  .35037096e+04  .28575625e-02", 
" -.24481298e+02  .18283748e+03 -.10161463e+04  .34312830e+04  .28508068e-02", 
" -.24286843e+02  .18011174e+03 -.99274113e+03  .33022985e+04  .28265834e-02", 
" -.24096499e+02  .17736916e+03 -.96540988e+03  .31203623e+04  .28111179e-02", 
" -.23909987e+02  .17474674e+03 -.94299088e+03  .30168066e+04  .27789000e-02", 
" -.23727332e+02  .17219897e+03 -.92149424e+03  .29071070e+04  .27430032e-02", 
" -.23548749e+02  .16968405e+03 -.89920399e+03  .27986000e+04  .26875848e-02", 
" -.23373491e+02  .16729119e+03 -.87973652e+03  .27055084e+04  .26399755e-02", 
" -.23201122e+02  .16487543e+03 -.85823809e+03  .26030861e+04  .26094397e-02", 
" -.23032419e+02  .16257874e+03 -.83830894e+03  .24956348e+04  .26022638e-02", 
" -.22867287e+02  .16050293e+03 -.82728090e+03  .25000292e+04  .25749649e-02", 
" -.22704017e+02  .15826495e+03 -.80632581e+03  .23684327e+04  .25713183e-02", 
" -.22544740e+02  .15636159e+03 -.79923562e+03  .24074660e+04  .25367600e-02", 
" -.22387237e+02  .15436044e+03 -.78591717e+03  .23693644e+04  .25136792e-02", 
" -.22231476e+02  .15222853e+03 -.76579687e+03  .22471100e+04  .25016792e-02", 
" -.22079069e+02  .15032526e+03 -.75349387e+03  .22123777e+04  .24960367e-02", 
" -.21928682e+02  .14847438e+03 -.74342346e+03  .22107886e+04  .24743608e-02", 
" -.21780602e+02  .14667103e+03 -.73452613e+03  .22210787e+04  .24401856e-02", 
" -.21633687e+02  .14476484e+03 -.71988438e+03  .21618851e+04  .24184839e-02", 
" -.21490062e+02  .14309897e+03 -.71438403e+03  .22089670e+04  .24021249e-02", 
" -.21347045e+02  .14112540e+03 -.69340669e+03  .20648284e+04  .24004982e-02", 
" -.21207476e+02  .13945508e+03 -.68301909e+03  .20316060e+04  .24222780e-02", 
" -.21069175e+02  .13769087e+03 -.66748241e+03  .19349777e+04  .23879408e-02", 
" -.20932467e+02  .13606765e+03 -.65876021e+03  .19283130e+04  .23873663e-02", 
" -.20797222e+02  .13435203e+03 -.64402184e+03  .18435675e+04  .23555146e-02", 
" -.20663702e+02  .13265029e+03 -.62857986e+03  .17483880e+04  .23411704e-02", 
" -.20531408e+02  .13093927e+03 -.61272742e+03  .16563035e+04  .23739581e-02", 
" -.20401226e+02  .12931445e+03 -.59915690e+03  .15804200e+04  .23510040e-02", 
" -.20271753e+02  .12775462e+03 -.59098911e+03  .15982022e+04  .23421581e-02", 
" -.20143888e+02  .12617071e+03 -.57799512e+03  .15251650e+04  .23564574e-02", 
" -.20017963e+02  .12472972e+03 -.57016510e+03  .15153813e+04  .23367449e-02", 
" -.19892743e+02  .12322168e+03 -.55940204e+03  .14731747e+04  .23090762e-02", 
" -.19769170e+02  .12172496e+03 -.54739016e+03  .14073478e+04  .23049061e-02", 
" -.19647011e+02  .12036195e+03 -.54143451e+03  .14196962e+04  .22817128e-02", 
" -.19525786e+02  .11896395e+03 -.53332062e+03  .14114945e+04  .22757243e-02", 
" -.19406290e+02  .11763632e+03 -.52631418e+03  .14043637e+04  .22544305e-02", 
" -.19287217e+02  .11622100e+03 -.51556889e+03  .13579947e+04  .22636090e-02", 
" -.19169700e+02  .11500192e+03 -.51328638e+03  .14078138e+04  .22546825e-02", 
" -.19053289e+02  .11372794e+03 -.50721259e+03  .14125758e+04  .22567151e-02", 
" -.18937792e+02  .11244955e+03 -.50029836e+03  .14051715e+04  .22391411e-02", 
" -.18823143e+02  .11114994e+03 -.49082510e+03  .13577892e+04  .22166874e-02", 
" -.18709575e+02  .10995311e+03 -.48696817e+03  .13896110e+04  .21996585e-02", 
" -.18596847e+02  .10867951e+03 -.47785222e+03  .13521692e+04  .21782290e-02", 
" -.18484629e+02  .10737973e+03 -.46700047e+03  .12843894e+04  .21723161e-02", 
" -.18373290e+02  .10615728e+03 -.45992357e+03  .12681084e+04  .21744963e-02", 
" -.18263376e+02  .10513363e+03 -.46212302e+03  .13689268e+04  .21711979e-02", 
" -.18153893e+02  .10390062e+03 -.45222026e+03  .13104306e+04  .21848994e-02", 
" -.18044990e+02  .10276884e+03 -.44799106e+03  .13242064e+04  .21691738e-02", 
" -.17937013e+02  .10158901e+03 -.44001518e+03  .12915580e+04  .21927825e-02", 
" -.17829824e+02  .10044634e+03 -.43329173e+03  .12737278e+04  .21776623e-02", 
" -.17723035e+02  .99265485e+02 -.42507616e+03  .12442373e+04  .21477574e-02", 
" -.17616568e+02  .98080182e+02 -.41645128e+03  .12071076e+04  .21403827e-02", 
" -.17511279e+02  .96871977e+02 -.40425369e+03  .11124329e+04  .21172627e-02", 
" -.17406723e+02  .95713937e+02 -.39435813e+03  .10461277e+04  .20736882e-02", 
" -.17302700e+02  .94615410e+02 -.38733507e+03  .10201560e+04  .20873577e-02", 
" -.17198984e+02  .93481173e+02 -.37874442e+03  .97369087e+03  .20721867e-02", 
" -.17096134e+02  .92372860e+02 -.37057781e+03  .93247166e+03  .20344197e-02", 
" -.16993399e+02  .91239996e+02 -.36121333e+03  .87478017e+03  .20371173e-02", 
" -.16891962e+02  .90215000e+02 -.35555327e+03  .86034200e+03  .20116281e-02", 
" -.16791057e+02  .89178884e+02 -.34859309e+03  .82387944e+03  .19944654e-02", 
" -.16690628e+02  .88205035e+02 -.34589248e+03  .85901646e+03  .19814308e-02", 
" -.16589927e+02  .87130337e+02 -.33831538e+03  .82600191e+03  .19581887e-02", 
" -.16489894e+02  .86093822e+02 -.33221754e+03  .81212746e+03  .19515879e-02", 
" -.16390456e+02  .85084491e+02 -.32686913e+03  .80462849e+03  .19534959e-02", 
" -.16291877e+02  .84153719e+02 -.32267490e+03  .79706821e+03  .19192908e-02", 
" -.16193497e+02  .83191566e+02 -.31792793e+03  .78983059e+03  .18938400e-02", 
" -.16095376e+02  .82150011e+02 -.30803638e+03  .70948776e+03  .19043541e-02", 
" -.15997053e+02  .81057367e+02 -.29729979e+03  .63348371e+03  .18812030e-02", 
" -.15899763e+02  .80106999e+02 -.29244421e+03  .62642587e+03  .18777268e-02", 
" -.15802731e+02  .79218832e+02 -.29034368e+03  .64897783e+03  .18785800e-02", 
" -.15705579e+02  .78240940e+02 -.28421189e+03  .62589573e+03  .18927237e-02", 
" -.15608891e+02  .77285054e+02 -.27955956e+03  .63068295e+03  .18792962e-02", 
" -.15512905e+02  .76349048e+02 -.27387990e+03  .61057404e+03  .19003471e-02", 
" -.15417346e+02  .75495342e+02 -.27222270e+03  .63724770e+03  .19081294e-02", 
" -.15321578e+02  .74491805e+02 -.26329525e+03  .57433858e+03  .18958785e-02", 
" -.15226534e+02  .73639930e+02 -.26102627e+03  .59404979e+03  .18722572e-02", 
" -.15131699e+02  .72775223e+02 -.25751855e+03  .59471390e+03  .18591609e-02", 
" -.15037098e+02  .71898716e+02 -.25323487e+03  .58921425e+03  .18547430e-02", 
" -.14942000e+02  .70882752e+02 -.24324950e+03  .51822806e+03  .18492248e-02", 
" -.14847518e+02  .69893830e+02 -.23317528e+03  .44069034e+03  .18431545e-02", 
" -.14753173e+02  .68997120e+02 -.22774248e+03  .41821465e+03  .18491901e-02", 
" -.14658853e+02  .68089269e+02 -.22184675e+03  .38956898e+03  .18399843e-02", 
" -.14565100e+02  .67228044e+02 -.21685222e+03  .36499480e+03  .18346370e-02", 
" -.14471185e+02  .66354426e+02 -.21202564e+03  .35174095e+03  .18487085e-02", 
" -.14377882e+02  .65528409e+02 -.20854837e+03  .35101145e+03  .18447727e-02", 
" -.14284414e+02  .64687606e+02 -.20443724e+03  .33979382e+03  .18305836e-02", 
" -.14191219e+02  .63912019e+02 -.20437353e+03  .38937604e+03  .18109046e-02", 
" -.14098087e+02  .63047839e+02 -.19812138e+03  .35072197e+03  .18016410e-02", 
" -.14004976e+02  .62300619e+02 -.19867319e+03  .40018474e+03  .17952626e-02", 
" -.13911802e+02  .61439361e+02 -.19287184e+03  .36912582e+03  .17745861e-02", 
" -.13818887e+02  .60634167e+02 -.18914726e+03  .35875892e+03  .17798097e-02", 
" -.13725568e+02  .59802032e+02 -.18540750e+03  .36079501e+03  .17750423e-02", 
" -.13632363e+02  .58953871e+02 -.17994941e+03  .33225663e+03  .17717053e-02", 
" -.13538723e+02  .58043545e+02 -.17202457e+03  .27919080e+03  .17497825e-02", 
" -.13445545e+02  .57195047e+02 -.16572532e+03  .23724289e+03  .17362087e-02", 
" -.13352174e+02  .56389443e+02 -.16262937e+03  .24640153e+03  .17152676e-02", 
" -.13259168e+02  .55667739e+02 -.16346461e+03  .30414082e+03  .17390818e-02", 
" -.13165530e+02  .54791766e+02 -.15614226e+03  .25649892e+03  .17046191e-02", 
" -.13072323e+02  .54072407e+02 -.15595030e+03  .29033381e+03  .16803287e-02", 
" -.12978791e+02  .53264398e+02 -.15173013e+03  .28073012e+03  .16625133e-02", 
" -.12885244e+02  .52435457e+02 -.14532942e+03  .23569529e+03  .16664932e-02", 
" -.12791475e+02  .51609538e+02 -.13859529e+03  .18055715e+03  .16284253e-02", 
" -.12697407e+02  .50799415e+02 -.13322423e+03  .14874399e+03  .16304556e-02", 
" -.12603102e+02  .49989361e+02 -.12852111e+03  .12802990e+03  .16034665e-02", 
" -.12508937e+02  .49250872e+02 -.12693086e+03  .14415897e+03  .15902759e-02", 
" -.12414554e+02  .48436961e+02 -.12057968e+03  .97724830e+02  .15915064e-02", 
" -.12319485e+02  .47641195e+02 -.11711994e+03  .99531465e+02  .15847115e-02", 
" -.12223816e+02  .46775856e+02 -.10996033e+03  .52189080e+02  .16107672e-02", 
" -.12128231e+02  .45938419e+02 -.10345526e+03  .94023923e+01  .16029291e-02", 
" -.12032804e+02  .45173986e+02 -.99795378e+02 -.45634372e+01  .16022663e-02", 
" -.11936942e+02  .44412541e+02 -.96140831e+02 -.21324670e+02  .16006320e-02", 
" -.11840089e+02  .43585197e+02 -.90623366e+02 -.48467234e+02  .16111259e-02", 
" -.11743131e+02  .42722192e+02 -.82553253e+02 -.11205201e+03  .16313906e-02", 
" -.11645926e+02  .41992653e+02 -.82040606e+02 -.77111821e+02  .16006726e-02", 
" -.11547814e+02  .41107862e+02 -.72901345e+02 -.15541765e+03  .16053431e-02", 
" -.11449989e+02  .40467850e+02 -.76247917e+02 -.69080093e+02  .16211611e-02", 
" -.11351830e+02  .39752634e+02 -.74700861e+02 -.55124153e+02  .16054372e-02", 
" -.11252569e+02  .38942707e+02 -.67851690e+02 -.11829942e+03  .15778991e-02", 
" -.11152659e+02  .38131075e+02 -.63900944e+02 -.11855905e+03  .15567695e-02", 
" -.11052490e+02  .37431892e+02 -.64156747e+02 -.78664084e+02  .15734674e-02", 
" -.10951651e+02  .36647635e+02 -.60042555e+02 -.88122313e+02  .15827422e-02", 
" -.10849955e+02  .35835620e+02 -.53899945e+02 -.13296476e+03  .15881845e-02", 
" -.10747856e+02  .35064359e+02 -.49576078e+02 -.15562401e+03  .15836983e-02", 
" -.10644778e+02  .34295631e+02 -.46446813e+02 -.15646117e+03  .16049194e-02", 
" -.10541096e+02  .33604139e+02 -.47311409e+02 -.10397524e+03  .16203514e-02", 
" -.10436106e+02  .32816919e+02 -.43580187e+02 -.11589792e+03  .16512328e-02", 
" -.10330487e+02  .32094715e+02 -.43169834e+02 -.87345167e+02  .16283197e-02", 
" -.10224084e+02  .31386270e+02 -.44222435e+02 -.32317075e+02  .16315701e-02", 
" -.10116907e+02  .30684675e+02 -.43948931e+02  .70830148e+00  .16397019e-02", 
" -.10007901e+02  .29868642e+02 -.38716912e+02 -.37709634e+02  .16307478e-02", 
" -.98980176e+01  .29014428e+02 -.30419211e+02 -.11954473e+03  .16526752e-02", 
" -.97864700e+01  .28149492e+02 -.24093435e+02 -.15838926e+03  .16755630e-02", 
" -.96732876e+01  .27220117e+02 -.15044715e+02 -.23136008e+03  .16617337e-02", 
" -.95591262e+01  .26325965e+02 -.74935555e+01 -.28659084e+03  .16835500e-02", 
" -.94434117e+01  .25402982e+02  .13856090e+01 -.36232731e+03  .16898606e-02", 
" -.93260836e+01  .24574161e+02  .46625114e+01 -.36385781e+03  .17005605e-02", 
" -.92077011e+01  .23783865e+02  .67683253e+01 -.35731171e+03  .16779330e-02", 
" -.90869616e+01  .22931675e+02  .11927661e+02 -.39788821e+03  .16919341e-02", 
" -.89634893e+01  .21896162e+02  .24501165e+02 -.51481658e+03  .17049507e-02", 
" -.88382653e+01  .20927588e+02  .32831563e+02 -.57166927e+03  .16484820e-02", 
" -.87116445e+01  .20158575e+02  .31779001e+02 -.51779291e+03  .16387214e-02", 
" -.85827292e+01  .19438961e+02  .29076790e+02 -.45774323e+03  .16374845e-02", 
" -.84511776e+01  .18561605e+02  .34048566e+02 -.49686354e+03  .16404493e-02", 
" -.83160897e+01  .17656401e+02  .38077275e+02 -.51216406e+03  .16558003e-02", 
" -.81773966e+01  .16600470e+02  .49299439e+02 -.61412380e+03  .16775649e-02", 
" -.80359058e+01  .15705285e+02  .53381934e+02 -.64755546e+03  .16865147e-02", 
" -.78907186e+01  .14839930e+02  .52137758e+02 -.58230473e+03  .17099267e-02", 
" -.77416301e+01  .14018933e+02  .48331379e+02 -.48570239e+03  .16990325e-02", 
" -.75874226e+01  .12977401e+02  .56328066e+02 -.55829434e+03  .17101761e-02", 
" -.74284558e+01  .12147301e+02  .52605675e+02 -.48248667e+03  .17383115e-02", 
" -.72633184e+01  .11085984e+02  .58580315e+02 -.51863481e+03  .17545041e-02", 
" -.70913878e+01  .10004197e+02  .62799954e+02 -.52857968e+03  .17519388e-02", 
" -.69131079e+01  .90630268e+01  .58060052e+02 -.41880809e+03  .17804576e-02", 
" -.67250648e+01  .78628716e+01  .62030512e+02 -.40502670e+03  .17881125e-02", 
" -.65280417e+01  .67621332e+01  .62362519e+02 -.37426914e+03  .18196688e-02", 
" -.63184437e+01  .55168202e+01  .65696902e+02 -.37331880e+03  .18122569e-02", 
" -.60973344e+01  .43921738e+01  .62222133e+02 -.28837745e+03  .18437544e-02", 
" -.58583956e+01  .32377016e+01  .52451062e+02 -.10137994e+03  .18901706e-02", 
" -.55982501e+01  .17260192e+01  .56735855e+02 -.81969682e+02  .20316265e-02", 
" -.53118560e+01 -.58120379e-01  .73837753e+02 -.28053851e+03  .20916419e-02", 
" -.49896373e+01 -.18449460e+01  .80662771e+02 -.34166381e+03  .21525524e-02", 
" -.46162343e+01 -.37196148e+01  .73355308e+02 -.16831387e+03  .23280901e-02", 
" -.41635349e+01 -.65134132e+01  .92676496e+02 -.32184121e+03  .24285497e-02", 
" -.35737056e+01 -.98984663e+01  .10053689e+03 -.28632472e+03  .27130308e-02", 
" -.34282087e+01 -.10840568e+02  .10854314e+03 -.35190214e+03  .28604111e-02", 
" -.32697310e+01 -.11615135e+02  .10420750e+03 -.25521885e+03  .30004562e-02", 
" -.30941490e+01 -.12492909e+02  .98267285e+02 -.13150913e+03  .29964124e-02", 
" -.28941395e+01 -.13816988e+02  .10650352e+03 -.20736763e+03  .31614285e-02", 
" -.26665407e+01 -.15308756e+02  .11964711e+03 -.32975543e+03  .32992053e-02", 
" -.23986492e+01 -.16487494e+02  .10608762e+03 -.12822938e+03  .35913962e-02", 
" -.20663730e+01 -.17878367e+02  .81297286e+02  .26435446e+03  .39535258e-02", 
" -.16238847e+01 -.19786870e+02  .53992101e+02  .64579278e+03  .46178109e-02", 
" -.92656342e+00 -.23325067e+02  .51495774e+02  .49486825e+03  .62344458e-02", 
" -.29177519e+00 -.24945177e+02 -.58654155e+02  .18795049e+04  .84193946e-02", 
"  .48935463e+00 -.28937181e+02 -.69518189e+02  .16783504e+04  .11535157e-01", 
"  .10494188e+01 -.35622090e+02  .91138611e+02 -.37779217e+03  .15220418e-01", 
"coactt    2  6  3   25", " -.73305648e+02  .14663794e+04 -.19709485e+05  .13610413e+06  .93618417e-01", 
" -.69767688e+02  .13381174e+04 -.17687062e+05  .12457985e+06  .70204818e-01", 
" -.65024446e+02  .11704563e+04 -.14638555e+05  .98260175e+05  .44683275e-01", 
" -.61303756e+02  .10348547e+04 -.11967693e+05  .74626005e+05  .32030196e-01", 
" -.57562360e+02  .91813347e+03 -.10139756e+05  .61687916e+05  .23811511e-01", 
" -.55332990e+02  .85287853e+03 -.92153146e+04  .55868515e+05  .20135177e-01", 
" -.53742094e+02  .80785325e+03 -.86001146e+04  .52380426e+05  .17653523e-01", 
" -.52487079e+02  .77038221e+03 -.79425784e+04  .46439088e+05  .16298141e-01", 
" -.51448759e+02  .74064080e+03 -.74682956e+04  .42745009e+05  .15199768e-01", 
" -.50570406e+02  .71687537e+03 -.71466342e+04  .40755861e+05  .13885480e-01", 
" -.49804925e+02  .69673250e+03 -.68715827e+04  .38827562e+05  .12944591e-01", 
" -.49127690e+02  .67900407e+03 -.66330193e+04  .37315479e+05  .12307262e-01", 
" -.48514533e+02  .66305464e+03 -.64160978e+04  .35809313e+05  .11617809e-01", 
" -.46124223e+02  .60075125e+03 -.54873597e+04  .28450962e+05  .10646185e-01", 
" -.44386195e+02  .55733891e+03 -.48989760e+04  .24510695e+05  .93262942e-02", 
" -.43023681e+02  .52497441e+03 -.44837728e+04  .21791155e+05  .83690984e-02", 
" -.41895110e+02  .49907425e+03 -.41756546e+04  .20023769e+05  .76775683e-02", 
" -.40929414e+02  .47747044e+03 -.39244075e+04  .18598254e+05  .70634389e-02", 
" -.40083456e+02  .45865157e+03 -.36976528e+04  .17273857e+05  .65525997e-02", 
" -.39329011e+02  .44214814e+03 -.34964932e+04  .15947774e+05  .60002172e-02", 
" -.38648873e+02  .42813031e+03 -.33617467e+04  .15496863e+05  .57036513e-02", 
" -.38027067e+02  .41549055e+03 -.32425928e+04  .15079179e+05  .55791969e-02", 
" -.37453748e+02  .40353942e+03 -.31029433e+04  .14159474e+05  .53150496e-02", 
" -.36920578e+02  .39211919e+03 -.29486030e+04  .12937437e+05  .51763760e-02", 
" -.36424922e+02  .38218402e+03 -.28425863e+04  .12356790e+05  .51051027e-02", 
" -.35959571e+02  .37302543e+03 -.27564351e+04  .12118151e+05  .49319385e-02", 
" -.35519895e+02  .36441068e+03 -.26725267e+04  .11793162e+05  .48133788e-02", 
" -.35103600e+02  .35591600e+03 -.25596279e+04  .10859632e+05  .46950328e-02", 
" -.34709043e+02  .34845261e+03 -.24953856e+04  .10734200e+05  .45836285e-02", 
" -.34332949e+02  .34138730e+03 -.24291648e+04  .10449508e+05  .45293733e-02", 
" -.33972415e+02  .33436027e+03 -.23508505e+04  .99813326e+04  .43573638e-02", 
" -.33629525e+02  .32786629e+03 -.22778394e+04  .94847657e+04  .42015791e-02", 
" -.33301326e+02  .32212494e+03 -.22382729e+04  .95161673e+04  .41288816e-02", 
" -.32986484e+02  .31676469e+03 -.22026055e+04  .95171219e+04  .39759543e-02", 
" -.32679199e+02  .31102082e+03 -.21426664e+04  .91983850e+04  .39213781e-02", 
" -.32384104e+02  .30571268e+03 -.20909742e+04  .89395020e+04  .38412108e-02", 
" -.32099370e+02  .30057594e+03 -.20399327e+04  .86853955e+04  .38159447e-02", 
" -.31822123e+02  .29559680e+03 -.19935163e+04  .85005316e+04  .37163883e-02", 
" -.31553254e+02  .29070508e+03 -.19432835e+04  .82270704e+04  .37393205e-02", 
" -.31293346e+02  .28598482e+03 -.18874364e+04  .77901093e+04  .36432194e-02", 
" -.31040506e+02  .28147279e+03 -.18410960e+04  .75384328e+04  .35551434e-02", 
" -.30795876e+02  .27743483e+03 -.18126982e+04  .75226868e+04  .35747384e-02", 
" -.30556757e+02  .27344237e+03 -.17811431e+04  .74392960e+04  .35126614e-02", 
" -.30323330e+02  .26938275e+03 -.17434205e+04  .72906926e+04  .35319498e-02", 
" -.30094856e+02  .26529450e+03 -.16937107e+04  .68830182e+04  .35211153e-02", 
" -.29872853e+02  .26150334e+03 -.16580644e+04  .67299867e+04  .34470295e-02", 
" -.29655664e+02  .25777784e+03 -.16209460e+04  .65294391e+04  .34379825e-02", 
" -.29443866e+02  .25419454e+03 -.15842151e+04  .63101165e+04  .33915241e-02", 
" -.29236289e+02  .25062826e+03 -.15463869e+04  .60679238e+04  .33267645e-02", 
" -.29032328e+02  .24705863e+03 -.15057524e+04  .57961869e+04  .32832346e-02", 
" -.28832806e+02  .24367863e+03 -.14688314e+04  .55335798e+04  .31952205e-02", 
" -.28637118e+02  .24039238e+03 -.14350561e+04  .53229553e+04  .31400617e-02", 
" -.28445636e+02  .23731511e+03 -.14071160e+04  .51829394e+04  .31324250e-02", 
" -.28257907e+02  .23433563e+03 -.13830621e+04  .51195681e+04  .31382880e-02", 
" -.28072706e+02  .23130936e+03 -.13547420e+04  .49755097e+04  .30710458e-02", 
" -.27889907e+02  .22809034e+03 -.13110779e+04  .45905756e+04  .30186016e-02", 
" -.27712032e+02  .22538166e+03 -.12904514e+04  .45221699e+04  .30361698e-02", 
" -.27536557e+02  .22264680e+03 -.12674001e+04  .44199159e+04  .29534983e-02", 
" -.27363545e+02  .22000036e+03 -.12506312e+04  .44508032e+04  .29273971e-02", 
" -.27191828e+02  .21708256e+03 -.12136372e+04  .41216792e+04  .28803838e-02", 
" -.27024923e+02  .21467615e+03 -.12001742e+04  .41315414e+04  .29184320e-02", 
" -.26859690e+02  .21212490e+03 -.11776176e+04  .40193708e+04  .28665092e-02", 
" -.26697862e+02  .20983412e+03 -.11661977e+04  .40538853e+04  .28547784e-02", 
" -.26538051e+02  .20749795e+03 -.11476984e+04  .39438207e+04  .28509159e-02", 
" -.26379090e+02  .20504389e+03 -.11265885e+04  .38549535e+04  .28382703e-02", 
" -.26222656e+02  .20270685e+03 -.11076464e+04  .37733770e+04  .28221918e-02", 
" -.26067774e+02  .20029470e+03 -.10834045e+04  .36025259e+04  .28606106e-02", 
" -.25915923e+02  .19803235e+03 -.10639952e+04  .34993026e+04  .28504497e-02", 
" -.25764715e+02  .19570661e+03 -.10434249e+04  .34136937e+04  .28194965e-02", 
" -.25614562e+02  .19318770e+03 -.10105490e+04  .31357778e+04  .27906901e-02", 
" -.25466955e+02  .19086214e+03 -.98331562e+03  .29084168e+04  .27635852e-02", 
" -.25321165e+02  .18873305e+03 -.96826647e+03  .28844195e+04  .27668354e-02", 
" -.25177670e+02  .18673008e+03 -.95811984e+03  .29253593e+04  .27345935e-02", 
" -.25035167e+02  .18467834e+03 -.94340394e+03  .28877530e+04  .26951106e-02", 
" -.24893931e+02  .18266678e+03 -.93036850e+03  .28870145e+04  .27016879e-02", 
" -.24754505e+02  .18066869e+03 -.91511543e+03  .28359876e+04  .26945394e-02", 
" -.24615757e+02  .17851308e+03 -.88975928e+03  .26164483e+04  .26838422e-02", 
" -.24479464e+02  .17657830e+03 -.87470339e+03  .25606103e+04  .26555652e-02", 
" -.24344398e+02  .17474496e+03 -.86641375e+03  .26074047e+04  .26428877e-02", 
" -.24209910e+02  .17273862e+03 -.84691956e+03  .24933074e+04  .26089797e-02", 
" -.24077015e+02  .17089764e+03 -.83506396e+03  .24809140e+04  .25894117e-02", 
" -.23945098e+02  .16905764e+03 -.82335477e+03  .24815665e+04  .25906326e-02", 
" -.23813346e+02  .16690619e+03 -.79167436e+03  .21602648e+04  .26006971e-02", 
" -.23684142e+02  .16510824e+03 -.77810977e+03  .21215944e+04  .25784620e-02", 
" -.23555911e+02  .16324288e+03 -.75778198e+03  .19547572e+04  .26106839e-02", 
" -.23429137e+02  .16160675e+03 -.74973270e+03  .19717808e+04  .26148822e-02", 
" -.23303187e+02  .16000838e+03 -.74371207e+03  .20104663e+04  .25746208e-02", 
" -.23178261e+02  .15829285e+03 -.72991445e+03  .19548823e+04  .25359068e-02", 
" -.23053244e+02  .15655037e+03 -.71620120e+03  .19162554e+04  .25301910e-02", 
" -.22929332e+02  .15486850e+03 -.70500595e+03  .19004732e+04  .25175105e-02", 
" -.22805769e+02  .15291815e+03 -.67604675e+03  .16035597e+04  .25303783e-02", 
" -.22683828e+02  .15127391e+03 -.66514209e+03  .15887553e+04  .25364664e-02", 
" -.22562959e+02  .14965577e+03 -.65498516e+03  .15913207e+04  .25283617e-02", 
" -.22442469e+02  .14799688e+03 -.63908915e+03  .14693656e+04  .25149704e-02", 
" -.22323296e+02  .14649045e+03 -.63260310e+03  .15173868e+04  .25034610e-02", 
" -.22204622e+02  .14482120e+03 -.61369781e+03  .13493862e+04  .24789815e-02", 
" -.22086670e+02  .14325928e+03 -.60039300e+03  .12653142e+04  .24637507e-02", 
" -.21969229e+02  .14163317e+03 -.58223926e+03  .10968978e+04  .24445798e-02", 
" -.21852528e+02  .14009073e+03 -.57041621e+03  .10434195e+04  .24722155e-02", 
" -.21736925e+02  .13863969e+03 -.56237344e+03  .10485549e+04  .24550149e-02", 
" -.21621710e+02  .13719487e+03 -.55520857e+03  .10695590e+04  .24292551e-02", 
" -.21506669e+02  .13572975e+03 -.54690716e+03  .10762990e+04  .24294763e-02", 
" -.21392534e+02  .13420920e+03 -.53085763e+03  .92373775e+03  .24123552e-02", 
" -.21278439e+02  .13262852e+03 -.51340789e+03  .76924032e+03  .23721471e-02", 
" -.21165767e+02  .13124596e+03 -.50737288e+03  .80795351e+03  .23949886e-02", 
" -.21052912e+02  .12970718e+03 -.49020209e+03  .65452303e+03  .24010430e-02", 
" -.20940902e+02  .12834186e+03 -.48322889e+03  .65819964e+03  .23853711e-02", 
" -.20829338e+02  .12689266e+03 -.47091837e+03  .59534736e+03  .23682513e-02", 
" -.20718996e+02  .12568861e+03 -.47076661e+03  .70052521e+03  .23410502e-02", 
" -.20608905e+02  .12439781e+03 -.46635912e+03  .74458947e+03  .23255460e-02", 
" -.20498871e+02  .12304715e+03 -.45723987e+03  .71735159e+03  .22914095e-02", 
" -.20389155e+02  .12175189e+03 -.45143581e+03  .72907912e+03  .22698374e-02", 
" -.20279432e+02  .12036999e+03 -.44051162e+03  .67506349e+03  .22458960e-02", 
" -.20170398e+02  .11896306e+03 -.42755268e+03  .59049563e+03  .22387978e-02", 
" -.20062100e+02  .11763030e+03 -.41570125e+03  .48974454e+03  .22224022e-02", 
" -.19953763e+02  .11626814e+03 -.40437965e+03  .43238364e+03  .22357892e-02", 
" -.19844300e+02  .11472859e+03 -.38541584e+03  .25898408e+03  .22364332e-02", 
" -.19736531e+02  .11351431e+03 -.38341660e+03  .35140138e+03  .22082546e-02", 
" -.19628788e+02  .11219830e+03 -.37471726e+03  .33345826e+03  .22143448e-02", 
" -.19521501e+02  .11081391e+03 -.35947208e+03  .19795328e+03  .22030310e-02", 
" -.19414674e+02  .10952106e+03 -.34933543e+03  .13737510e+03  .22156745e-02", 
" -.19307752e+02  .10815621e+03 -.33527035e+03  .22588746e+02  .22027732e-02", 
" -.19201073e+02  .10690360e+03 -.32824387e+03  .29196585e+02  .21885170e-02", 
" -.19094917e+02  .10560688e+03 -.31523979e+03 -.96200027e+02  .22235914e-02", 
" -.18988684e+02  .10429961e+03 -.30441623e+03 -.14685157e+03  .22075817e-02", 
" -.18882783e+02  .10315035e+03 -.30244418e+03 -.66174993e+02  .21753679e-02", 
" -.18777040e+02  .10192275e+03 -.29278607e+03 -.14074319e+03  .21543727e-02", 
" -.18671519e+02  .10068871e+03 -.28534331e+03 -.14029603e+03  .21371832e-02", 
" -.18565807e+02  .99478782e+02 -.27853091e+03 -.15060085e+03  .21653834e-02", 
" -.18460458e+02  .98358252e+02 -.27648951e+03 -.83139866e+02  .21558817e-02", 
" -.18354954e+02  .97129668e+02 -.26788802e+03 -.11928571e+03  .21559511e-02", 
" -.18250045e+02  .96072343e+02 -.26935109e+03  .40993062e+01  .21572579e-02", 
" -.18144846e+02  .94869720e+02 -.26084623e+03 -.42242088e+02  .21595752e-02", 
" -.18039422e+02  .93715748e+02 -.25572864e+03 -.34014560e+02  .21643258e-02", 
" -.17934117e+02  .92489155e+02 -.24500938e+03 -.12066907e+03  .21690700e-02", 
" -.17828619e+02  .91239193e+02 -.23341604e+03 -.21427858e+03  .21643060e-02", 
" -.17722988e+02  .90045747e+02 -.22596107e+03 -.23368092e+03  .21585266e-02", 
" -.17618082e+02  .88939752e+02 -.22146146e+03 -.21611824e+03  .21210321e-02", 
" -.17513030e+02  .87895128e+02 -.22194292e+03 -.12231164e+03  .21253598e-02", 
" -.17407527e+02  .86795838e+02 -.21830198e+03 -.99417442e+02  .21061550e-02", 
" -.17301195e+02  .85458387e+02 -.19930139e+03 -.33042370e+03  .21096715e-02", 
" -.17194991e+02  .84169848e+02 -.18532597e+03 -.44924613e+03  .20941912e-02", 
" -.17089526e+02  .83074524e+02 -.18056980e+03 -.44721170e+03  .20822803e-02", 
" -.16983402e+02  .81860593e+02 -.17074226e+03 -.49727650e+03  .20733642e-02", 
" -.16877666e+02  .80805732e+02 -.16843680e+03 -.45397866e+03  .20788764e-02", 
" -.16771601e+02  .79715018e+02 -.16455218e+03 -.42456074e+03  .21188507e-02", 
" -.16665653e+02  .78685192e+02 -.16356908e+03 -.36259365e+03  .21127574e-02", 
" -.16558974e+02  .77581120e+02 -.15955690e+03 -.33065931e+03  .20884610e-02", 
" -.16451825e+02  .76339368e+02 -.14617468e+03 -.44422299e+03  .20878033e-02", 
" -.16344934e+02  .75251370e+02 -.14214949e+03 -.42257455e+03  .20869057e-02", 
" -.16237181e+02  .73987006e+02 -.12677383e+03 -.57422218e+03  .20815620e-02", 
" -.16128723e+02  .72692735e+02 -.11185475e+03 -.70655315e+03  .20716638e-02", 
" -.16020307e+02  .71492440e+02 -.99180096e+02 -.84281419e+03  .20567366e-02", 
" -.15912095e+02  .70494230e+02 -.10047371e+03 -.73374040e+03  .20429210e-02", 
" -.15802776e+02  .69344278e+02 -.94091065e+02 -.73805579e+03  .20360356e-02", 
" -.15692984e+02  .68154646e+02 -.85707795e+02 -.76382531e+03  .20409940e-02", 
" -.15582824e+02  .66913169e+02 -.73631447e+02 -.85641254e+03  .20400479e-02", 
" -.15471940e+02  .65657634e+02 -.59815855e+02 -.98649539e+03  .20356738e-02", 
" -.15360673e+02  .64460801e+02 -.49639941e+02 -.10603104e+04  .20496491e-02", 
" -.15249304e+02  .63250370e+02 -.38544153e+02 -.11389073e+04  .20026022e-02", 
" -.15137335e+02  .62136679e+02 -.33672409e+02 -.11204012e+04  .19870264e-02", 
" -.15024769e+02  .60935884e+02 -.19663643e+02 -.12942217e+04  .19960508e-02", 
" -.14911601e+02  .59803924e+02 -.12379596e+02 -.13365099e+04  .20039598e-02", 
" -.14797556e+02  .58635975e+02 -.16318939e+01 -.14542696e+04  .19792129e-02", 
" -.14682675e+02  .57550257e+02  .52798245e+00 -.14029174e+04  .20137176e-02", 
" -.14566507e+02  .56267584e+02  .14397879e+02 -.15195671e+04  .19995098e-02", 
" -.14450628e+02  .55155627e+02  .23284361e+02 -.16061788e+04  .20100085e-02", 
" -.14332834e+02  .53862939e+02  .35800536e+02 -.16925968e+04  .20120386e-02", 
" -.14215048e+02  .52793230e+02  .39894495e+02 -.16932010e+04  .19996924e-02", 
" -.14095521e+02  .51512759e+02  .53359027e+02 -.18166824e+04  .20111932e-02", 
" -.13975264e+02  .50364658e+02  .58398347e+02 -.18034820e+04  .20434344e-02", 
" -.13853908e+02  .49212804e+02  .65585792e+02 -.18509960e+04  .20229578e-02", 
" -.13731194e+02  .48114742e+02  .66485396e+02 -.17727066e+04  .20542958e-02", 
" -.13607070e+02  .46997883e+02  .68356675e+02 -.17196413e+04  .20795519e-02", 
" -.13481798e+02  .45910766e+02  .67367469e+02 -.16095217e+04  .20792105e-02", 
" -.13355605e+02  .44748257e+02  .73368185e+02 -.16331293e+04  .20858062e-02", 
" -.13227285e+02  .43494364e+02  .81890376e+02 -.16646237e+04  .20727783e-02", 
" -.13098196e+02  .42423361e+02  .81485577e+02 -.15838217e+04  .20322971e-02", 
" -.12967432e+02  .41289923e+02  .83585056e+02 -.15403741e+04  .20439064e-02", 
" -.12833726e+02  .39936838e+02  .97420623e+02 -.16666052e+04  .20496747e-02", 
" -.12697543e+02  .38402126e+02  .11942054e+03 -.19045975e+04  .20651777e-02", 
" -.12559982e+02  .37013062e+02  .13316661e+03 -.20148475e+04  .20417364e-02", 
" -.12420602e+02  .35623973e+02  .14719761e+03 -.21382964e+04  .20657665e-02", 
" -.12278897e+02  .34577816e+02  .13449404e+03 -.18038372e+04  .20619772e-02", 
" -.12134346e+02  .33262530e+02  .14132243e+03 -.18178555e+04  .20907625e-02", 
" -.11988038e+02  .32101333e+02  .14150263e+03 -.17538523e+04  .20893444e-02", 
" -.11838386e+02  .30920712e+02  .13778677e+03 -.15993049e+04  .20893097e-02", 
" -.11684803e+02  .29418028e+02  .15448682e+03 -.17795650e+04  .21246896e-02", 
" -.11527450e+02  .27923301e+02  .16674268e+03 -.18678809e+04  .21428505e-02", 
" -.11366965e+02  .26484949e+02  .17720073e+03 -.19594909e+04  .21547716e-02", 
" -.11201457e+02  .24854831e+02  .19221225e+03 -.20580749e+04  .21559413e-02", 
" -.11031351e+02  .23290713e+02  .20142494e+03 -.20707493e+04  .21846583e-02", 
" -.10858045e+02  .21969058e+02  .20112244e+03 -.19982976e+04  .21697468e-02", 
" -.10678902e+02  .20270296e+02  .22001025e+03 -.21794662e+04  .21416263e-02", 
" -.10493323e+02  .18712096e+02  .22567457e+03 -.21610377e+04  .21600856e-02", 
" -.10301471e+02  .17188579e+02  .22941723e+03 -.21359560e+04  .22048042e-02", 
" -.10100847e+02  .15333835e+02  .24299656e+03 -.21736478e+04  .22311428e-02", 
" -.98920401e+01  .13571291e+02  .25367886e+03 -.22385303e+04  .22873481e-02", 
" -.96736950e+01  .11720092e+02  .26457315e+03 -.22906581e+04  .22692912e-02", 
" -.94437088e+01  .97654350e+01  .27737842e+03 -.23653927e+04  .22732759e-02", 
" -.92001077e+01  .80062640e+01  .26864763e+03 -.20815348e+04  .23059334e-02", 
" -.89392482e+01  .58214724e+01  .27758061e+03 -.20542436e+04  .23647561e-02", 
" -.86590720e+01  .35911344e+01  .27970268e+03 -.19138105e+04  .23818736e-02", 
" -.83538261e+01  .70462285e+00  .31494979e+03 -.22836456e+04  .23600492e-02", 
" -.80190637e+01 -.23265044e+01  .34463306e+03 -.25619784e+04  .25505094e-02", 
" -.76430450e+01 -.55965843e+01  .36583304e+03 -.26549626e+04  .26591501e-02", 
" -.72072410e+01 -.95654537e+01  .39902686e+03 -.28912651e+04  .29150824e-02", 
" -.66846693e+01 -.14188368e+02  .42908988e+03 -.30610420e+04  .32119037e-02", 
" -.60067428e+01 -.20527042e+02  .48376826e+03 -.35660853e+04  .35333050e-02", 
" -.58411380e+01 -.22070444e+02  .49565952e+03 -.36626360e+04  .35888165e-02", 
" -.56605269e+01 -.23812952e+02  .50764892e+03 -.37151045e+04  .37393385e-02", 
" -.54593993e+01 -.25932397e+02  .52877591e+03 -.38937617e+04  .39441822e-02", 
" -.52316392e+01 -.28202721e+02  .54088035e+03 -.39186401e+04  .41947891e-02", 
" -.49732375e+01 -.30872259e+02  .55522767e+03 -.38497172e+04  .43510520e-02", 
" -.46651666e+01 -.34643979e+02  .60532933e+03 -.43040685e+04  .46318046e-02", 
" -.42860554e+01 -.38677618e+02  .61221995e+03 -.38475128e+04  .48863020e-02", 
" -.37823007e+01 -.44489906e+02  .66021687e+03 -.41139323e+04  .57888124e-02", 
" -.29858206e+01 -.52698148e+02  .61197794e+03 -.21184201e+04  .77802523e-02", 
" -.22656145e+01 -.61029987e+02  .63891460e+03 -.19144194e+04  .99297897e-02", 
" -.13551650e+01 -.76440275e+02  .86473718e+03 -.42594601e+04  .14032729e-01", 
" -.73725316e+00 -.83726331e+02  .78595994e+03 -.17460082e+04  .17773512e-01"
)


.urc4 = 
c("Copyright (C) James G. MacKinnon, 1995", "conc      3  4  3   20", 
" -.54580088e+01 -.32103735e+02 -.16441921e+03 -.17069815e+03  .50497378e-02", 
" -.53022567e+01 -.29776828e+02 -.10167860e+03 -.69397283e+03  .36013713e-02", 
" -.50872065e+01 -.25668280e+02 -.97898205e+02 -.11738090e+03  .24600651e-02", 
" -.49141598e+01 -.23219577e+02 -.62553182e+02 -.27431152e+03  .18317209e-02", 
" -.47349746e+01 -.20370384e+02 -.45443105e+02 -.25427387e+03  .14099363e-02", 
" -.46268946e+01 -.18408982e+02 -.52502899e+02 -.48374554e+02  .11650449e-02", 
" -.45473994e+01 -.17104335e+02 -.56823269e+02  .10605503e+03  .10269516e-02", 
" -.44829692e+01 -.16532630e+02 -.37583647e+02 -.89741681e+02  .92288046e-03", 
" -.44306628e+01 -.15739881e+02 -.37089206e+02 -.55841182e+02  .88105185e-03", 
" -.43846312e+01 -.15296505e+02 -.27113514e+02 -.13750084e+03  .78716786e-03", 
" -.43447562e+01 -.14760005e+02 -.25023053e+02 -.14380575e+03  .72423841e-03", 
" -.43091854e+01 -.14225082e+02 -.26678867e+02 -.10307478e+03  .68921900e-03", 
" -.42767684e+01 -.13771329e+02 -.30172355e+02 -.11794035e+02  .66409358e-03", 
" -.41479533e+01 -.12304040e+02 -.23649053e+02  .29213905e+01  .57092385e-03", 
" -.40527055e+01 -.11273913e+02 -.18733414e+02 -.10763345e+02  .48900744e-03", 
" -.39760499e+01 -.10546880e+02 -.12391850e+02 -.61740670e+02  .46289460e-03", 
" -.39119310e+01 -.98178689e+01 -.13830434e+02 -.26777082e+02  .43542692e-03", 
" -.38559671e+01 -.93374415e+01 -.98087208e+01 -.48593874e+02  .40027254e-03", 
" -.38066560e+01 -.88553117e+01 -.97155504e+01 -.22415977e+02  .37129470e-03", 
" -.37623294e+01 -.84111099e+01 -.10219720e+02 -.20800031e+01  .35761177e-03", 
" -.37218002e+01 -.80475066e+01 -.89768044e+01 -.80201953e+01  .35497687e-03", 
" -.36844235e+01 -.77283403e+01 -.79356371e+01 -.84013162e+01  .34735784e-03", 
" -.36498199e+01 -.74229117e+01 -.75975526e+01 -.65531070e+01  .33659349e-03", 
" -.36171044e+01 -.71982257e+01 -.49709159e+01 -.29914436e+02  .33519853e-03", 
" -.35864111e+01 -.69681469e+01 -.35131722e+01 -.42897342e+02  .32552580e-03", 
" -.35575669e+01 -.67356990e+01 -.26374831e+01 -.53566067e+02  .31725032e-03", 
" -.35301064e+01 -.65291339e+01 -.17963594e+01 -.56496168e+02  .30620129e-03", 
" -.35040012e+01 -.63307164e+01 -.16138964e+01 -.51597515e+02  .29532540e-03", 
" -.34791023e+01 -.61400096e+01 -.13342868e+01 -.49998142e+02  .28753319e-03", 
" -.34552094e+01 -.59702477e+01 -.46135813e+00 -.57423093e+02  .27380145e-03", 
" -.34321968e+01 -.58119301e+01  .27071584e+00 -.64386218e+02  .27477700e-03", 
" -.34100549e+01 -.56513219e+01  .28505888e+00 -.59387144e+02  .26447610e-03", 
" -.33887376e+01 -.55048922e+01  .88106951e+00 -.65988263e+02  .26237834e-03", 
" -.33681016e+01 -.53723162e+01  .17011769e+01 -.74523053e+02  .26332981e-03", 
" -.33481853e+01 -.52108970e+01  .50869553e+00 -.55726515e+02  .26250891e-03", 
" -.33287839e+01 -.50842651e+01  .77926904e+00 -.56000162e+02  .26248097e-03", 
" -.33099300e+01 -.49735513e+01  .17207607e+01 -.65197656e+02  .26169893e-03", 
" -.32916925e+01 -.48277873e+01  .41043851e+00 -.44499693e+02  .26082791e-03", 
" -.32738700e+01 -.47152902e+01  .74416357e+00 -.46428860e+02  .26143598e-03", 
" -.32565477e+01 -.45905665e+01  .73302997e-01 -.34360255e+02  .26428091e-03", 
" -.32396991e+01 -.44776158e+01  .17652667e+00 -.34949034e+02  .26023467e-03", 
" -.32230917e+01 -.43871815e+01  .71281716e+00 -.37101100e+02  .25911896e-03", 
" -.32069150e+01 -.42826907e+01  .41014070e+00 -.29246285e+02  .25213969e-03", 
" -.31911601e+01 -.41664778e+01 -.38267395e+00 -.17243272e+02  .25114733e-03", 
" -.31756320e+01 -.40745781e+01 -.19318311e+00 -.19485493e+02  .25006171e-03", 
" -.31604495e+01 -.39709527e+01 -.71507985e+00 -.11996762e+02  .24854246e-03", 
" -.31454661e+01 -.38884724e+01 -.21834399e+00 -.19275643e+02  .24939067e-03", 
" -.31307864e+01 -.37990407e+01 -.40463852e+00 -.15284424e+02  .25141049e-03", 
" -.31163854e+01 -.37079884e+01 -.10052828e+01 -.47554326e+01  .25235961e-03", 
" -.31021923e+01 -.36445183e+01 -.39450983e-01 -.15852385e+02  .24835491e-03", 
" -.30881671e+01 -.35836214e+01  .93849216e+00 -.29484516e+02  .24381727e-03", 
" -.30744183e+01 -.35011964e+01  .33111966e+00 -.17351727e+02  .24013820e-03", 
" -.30608870e+01 -.34334572e+01  .60575914e+00 -.18415584e+02  .23615012e-03", 
" -.30475644e+01 -.33613101e+01  .73747933e+00 -.21058902e+02  .23242445e-03", 
" -.30344877e+01 -.32823876e+01  .41923414e+00 -.16568088e+02  .22966983e-03", 
" -.30215701e+01 -.32071139e+01  .11145785e+00 -.11694050e+02  .22939465e-03", 
" -.30088228e+01 -.31359982e+01 -.21249241e-01 -.10404333e+02  .22676393e-03", 
" -.29961840e+01 -.30715513e+01 -.96084950e-02 -.90143675e+01  .22549295e-03", 
" -.29837313e+01 -.30153108e+01  .46867962e+00 -.15086350e+02  .22556978e-03", 
" -.29713724e+01 -.29548434e+01  .41284420e+00 -.12417393e+02  .22411312e-03", 
" -.29591438e+01 -.29100221e+01  .12928484e+01 -.23896206e+02  .21901647e-03", 
" -.29470875e+01 -.28573614e+01  .15050879e+01 -.24634232e+02  .22057614e-03", 
" -.29351687e+01 -.28058193e+01  .18364696e+01 -.28134931e+02  .21852859e-03", 
" -.29233639e+01 -.27591872e+01  .23447825e+01 -.34167452e+02  .22115360e-03", 
" -.29117214e+01 -.27046746e+01  .24318458e+01 -.35173941e+02  .21945933e-03", 
" -.29001852e+01 -.26511793e+01  .23544963e+01 -.31896577e+02  .21901084e-03", 
" -.28887882e+01 -.25886563e+01  .17903269e+01 -.23858153e+02  .22131593e-03", 
" -.28774697e+01 -.25344669e+01  .16515223e+01 -.21173171e+02  .22048017e-03", 
" -.28661932e+01 -.24857857e+01  .16177480e+01 -.19105578e+02  .22079654e-03", 
" -.28550725e+01 -.24437973e+01  .21599074e+01 -.26884814e+02  .22112676e-03", 
" -.28440729e+01 -.23903235e+01  .19610082e+01 -.23866714e+02  .21819384e-03", 
" -.28331418e+01 -.23421723e+01  .19077318e+01 -.21497785e+02  .21960751e-03", 
" -.28222767e+01 -.23052732e+01  .23634660e+01 -.26275302e+02  .21916469e-03", 
" -.28114860e+01 -.22532856e+01  .19609609e+01 -.19855492e+02  .21906062e-03", 
" -.28008514e+01 -.22098988e+01  .23741442e+01 -.27085662e+02  .21966999e-03", 
" -.27901917e+01 -.21830722e+01  .34507027e+01 -.42770543e+02  .21598964e-03", 
" -.27796441e+01 -.21382898e+01  .33479168e+01 -.40879732e+02  .21772487e-03", 
" -.27691699e+01 -.20923557e+01  .31906681e+01 -.39024459e+02  .21735236e-03", 
" -.27587535e+01 -.20460065e+01  .29546615e+01 -.36237175e+02  .21825629e-03", 
" -.27483909e+01 -.20004291e+01  .26255723e+01 -.30914968e+02  .22083976e-03", 
" -.27380951e+01 -.19700340e+01  .32594562e+01 -.39430756e+02  .21917541e-03", 
" -.27278810e+01 -.19250050e+01  .29631105e+01 -.35167440e+02  .21681590e-03", 
" -.27177338e+01 -.18794298e+01  .26310221e+01 -.30602546e+02  .21851951e-03", 
" -.27076219e+01 -.18422619e+01  .27284500e+01 -.31365978e+02  .21823541e-03", 
" -.26975218e+01 -.18172092e+01  .34038817e+01 -.40569357e+02  .21698257e-03", 
" -.26875504e+01 -.17727117e+01  .29975091e+01 -.34579611e+02  .21626340e-03", 
" -.26775656e+01 -.17446084e+01  .35186366e+01 -.41395774e+02  .21566203e-03", 
" -.26676694e+01 -.17040373e+01  .35286474e+01 -.43095906e+02  .21780000e-03", 
" -.26577743e+01 -.16702029e+01  .34058969e+01 -.39756501e+02  .21927686e-03", 
" -.26479535e+01 -.16347587e+01  .33881562e+01 -.39372882e+02  .21825844e-03", 
" -.26381485e+01 -.16030087e+01  .35276457e+01 -.40961782e+02  .21671184e-03", 
" -.26284341e+01 -.15625519e+01  .31968507e+01 -.36177495e+02  .21807911e-03", 
" -.26187160e+01 -.15234353e+01  .27804303e+01 -.29448365e+02  .21699240e-03", 
" -.26089794e+01 -.15022281e+01  .33601553e+01 -.36294067e+02  .21880957e-03", 
" -.25993129e+01 -.14719394e+01  .35659769e+01 -.39853995e+02  .22014994e-03", 
" -.25897154e+01 -.14400944e+01  .37584409e+01 -.43280488e+02  .22019070e-03", 
" -.25800600e+01 -.14189008e+01  .42291322e+01 -.49149937e+02  .22225734e-03", 
" -.25704407e+01 -.13974633e+01  .46146768e+01 -.53697336e+02  .22277254e-03", 
" -.25608793e+01 -.13694929e+01  .47910486e+01 -.56490826e+02  .22320324e-03", 
" -.25513397e+01 -.13380411e+01  .47102670e+01 -.55465017e+02  .22455936e-03", 
" -.25417755e+01 -.13165595e+01  .50127883e+01 -.58211136e+02  .22410048e-03", 
" -.25322605e+01 -.12924661e+01  .52509703e+01 -.60921359e+02  .22473469e-03", 
" -.25227991e+01 -.12660809e+01  .54717843e+01 -.64835166e+02  .22138770e-03", 
" -.25133609e+01 -.12374460e+01  .54952510e+01 -.64447740e+02  .21950156e-03", 
" -.25039605e+01 -.11987871e+01  .50496562e+01 -.59588084e+02  .21843650e-03", 
" -.24945544e+01 -.11663652e+01  .47879033e+01 -.56068090e+02  .21800117e-03", 
" -.24851972e+01 -.11212978e+01  .38818145e+01 -.44631412e+02  .21578025e-03", 
" -.24757698e+01 -.10893268e+01  .33609686e+01 -.35915087e+02  .21403676e-03", 
" -.24663924e+01 -.10586272e+01  .32147699e+01 -.33667932e+02  .21457589e-03", 
" -.24569777e+01 -.10396783e+01  .35213417e+01 -.36316827e+02  .21628358e-03", 
" -.24475547e+01 -.10213733e+01  .38289155e+01 -.39890130e+02  .21547119e-03", 
" -.24381680e+01 -.99949026e+00  .40395996e+01 -.42986875e+02  .21373137e-03", 
" -.24287760e+01 -.97310812e+00  .38756559e+01 -.40316570e+02  .21341976e-03", 
" -.24193671e+01 -.95471781e+00  .42130248e+01 -.44770937e+02  .21179458e-03", 
" -.24099746e+01 -.92973938e+00  .39850740e+01 -.40152335e+02  .21416679e-03", 
" -.24005626e+01 -.91771580e+00  .46274805e+01 -.48433337e+02  .21314212e-03", 
" -.23911831e+01 -.89589267e+00  .48176479e+01 -.51778309e+02  .21413069e-03", 
" -.23817875e+01 -.86883956e+00  .45906736e+01 -.48128525e+02  .21173895e-03", 
" -.23723487e+01 -.85453529e+00  .49756410e+01 -.52241042e+02  .20990657e-03", 
" -.23629664e+01 -.83054672e+00  .49517643e+01 -.51746418e+02  .20874724e-03", 
" -.23535426e+01 -.80663530e+00  .49395140e+01 -.51848613e+02  .21087959e-03", 
" -.23441266e+01 -.78076041e+00  .47565424e+01 -.49804599e+02  .21029588e-03", 
" -.23346906e+01 -.75485328e+00  .45346011e+01 -.46662788e+02  .21014010e-03", 
" -.23252085e+01 -.73969658e+00  .48916836e+01 -.51672104e+02  .21052469e-03", 
" -.23157247e+01 -.71990172e+00  .49227011e+01 -.52099965e+02  .21173910e-03", 
" -.23062378e+01 -.70175328e+00  .51166778e+01 -.54503914e+02  .20885715e-03", 
" -.22967744e+01 -.66845061e+00  .44071827e+01 -.44599395e+02  .21064527e-03", 
" -.22872577e+01 -.63873904e+00  .40417310e+01 -.41270376e+02  .21346658e-03", 
" -.22777068e+01 -.62220894e+00  .42705545e+01 -.44618599e+02  .21018702e-03", 
" -.22681013e+01 -.60797461e+00  .44798092e+01 -.46599437e+02  .20965859e-03", 
" -.22584660e+01 -.58753581e+00  .43510795e+01 -.44127169e+02  .20922693e-03", 
" -.22488250e+01 -.56454255e+00  .41207303e+01 -.40733332e+02  .20923811e-03", 
" -.22391871e+01 -.53871012e+00  .38313703e+01 -.37216429e+02  .21151626e-03", 
" -.22294982e+01 -.51762787e+00  .37378802e+01 -.36506753e+02  .21152441e-03", 
" -.22197519e+01 -.49626164e+00  .36051694e+01 -.34702975e+02  .20942119e-03", 
" -.22099755e+01 -.47489408e+00  .35065936e+01 -.34124020e+02  .21115105e-03", 
" -.22002123e+01 -.44662369e+00  .30209729e+01 -.28042814e+02  .21432446e-03", 
" -.21903125e+01 -.43514604e+00  .33733708e+01 -.33510145e+02  .21635423e-03", 
" -.21803972e+01 -.41927819e+00  .34717484e+01 -.35130859e+02  .21586597e-03", 
" -.21704057e+01 -.40219468e+00  .34199730e+01 -.34545192e+02  .21311637e-03", 
" -.21604184e+01 -.38130879e+00  .32696601e+01 -.32995073e+02  .21511347e-03", 
" -.21503384e+01 -.37049764e+00  .34809327e+01 -.35251887e+02  .21684977e-03", 
" -.21401668e+01 -.36928946e+00  .41515024e+01 -.43303859e+02  .21690222e-03", 
" -.21300273e+01 -.35600236e+00  .43264458e+01 -.45502893e+02  .21575628e-03", 
" -.21198112e+01 -.34104295e+00  .44653226e+01 -.48322183e+02  .21585898e-03", 
" -.21095294e+01 -.32471293e+00  .45026204e+01 -.50076824e+02  .21553400e-03", 
" -.20991329e+01 -.31760366e+00  .49023888e+01 -.55897717e+02  .21516851e-03", 
" -.20887140e+01 -.30827954e+00  .51556525e+01 -.58703853e+02  .21524446e-03", 
" -.20781892e+01 -.30477973e+00  .56615849e+01 -.65133923e+02  .21441468e-03", 
" -.20675935e+01 -.29710540e+00  .59174052e+01 -.67795836e+02  .21409288e-03", 
" -.20570066e+01 -.26495729e+00  .48892719e+01 -.53386849e+02  .21503115e-03", 
" -.20463351e+01 -.24237648e+00  .45476680e+01 -.49642134e+02  .21368959e-03", 
" -.20355376e+01 -.22314095e+00  .42598897e+01 -.46230068e+02  .21866758e-03", 
" -.20246584e+01 -.20157252e+00  .37143426e+01 -.38354314e+02  .21829387e-03", 
" -.20136579e+01 -.19986132e+00  .42072666e+01 -.43675307e+02  .22095980e-03", 
" -.20025553e+01 -.18209692e+00  .38271206e+01 -.38045719e+02  .21828267e-03", 
" -.19913577e+01 -.17144992e+00  .38765409e+01 -.38403529e+02  .21863611e-03", 
" -.19800076e+01 -.17902136e+00  .49575151e+01 -.53121316e+02  .22110444e-03", 
" -.19686199e+01 -.16633146e+00  .49628676e+01 -.53683782e+02  .22158454e-03", 
" -.19570772e+01 -.15666089e+00  .49957344e+01 -.53483413e+02  .22297842e-03", 
" -.19454122e+01 -.14641615e+00  .50291942e+01 -.54203561e+02  .22093615e-03", 
" -.19336390e+01 -.13810430e+00  .52010584e+01 -.56591151e+02  .21698324e-03", 
" -.19216861e+01 -.13007566e+00  .51621136e+01 -.55139806e+02  .21833843e-03", 
" -.19096056e+01 -.12135487e+00  .51539721e+01 -.54456323e+02  .22004338e-03", 
" -.18973785e+01 -.11599945e+00  .53125123e+01 -.55673815e+02  .21812464e-03", 
" -.18850336e+01 -.10644453e+00  .54517822e+01 -.58896304e+02  .21859953e-03", 
" -.18724923e+01 -.96284657e-01  .54400751e+01 -.60009917e+02  .21907417e-03", 
" -.18597100e+01 -.94771143e-01  .55146193e+01 -.57743242e+02  .21812413e-03", 
" -.18467518e+01 -.10974466e+00  .64780692e+01 -.68256190e+02  .22002752e-03", 
" -.18337158e+01 -.93411479e-01  .59705360e+01 -.61066446e+02  .22089174e-03", 
" -.18204297e+01 -.89430541e-01  .60595255e+01 -.61091512e+02  .22598459e-03", 
" -.18068008e+01 -.10477047e+00  .70104591e+01 -.72955533e+02  .22776325e-03", 
" -.17930113e+01 -.10237477e+00  .70990731e+01 -.73579874e+02  .23064219e-03", 
" -.17790019e+01 -.97517042e-01  .70863114e+01 -.72992686e+02  .23122399e-03", 
" -.17648195e+01 -.89958326e-01  .71457551e+01 -.75622493e+02  .23239108e-03", 
" -.17502844e+01 -.85038101e-01  .70011723e+01 -.74073997e+02  .23497329e-03", 
" -.17355186e+01 -.74045139e-01  .65711579e+01 -.67423042e+02  .23818897e-03", 
" -.17204721e+01 -.42560057e-01  .48066370e+01 -.42350588e+02  .23984491e-03", 
" -.17049480e+01 -.56735958e-01  .55720923e+01 -.52013565e+02  .23951239e-03", 
" -.16892076e+01 -.42906396e-01  .48373942e+01 -.41371558e+02  .24515505e-03", 
" -.16731049e+01 -.35532229e-01  .46631337e+01 -.40413688e+02  .24722601e-03", 
" -.16565664e+01 -.41196200e-01  .51025480e+01 -.46323181e+02  .24452379e-03", 
" -.16395311e+01 -.46793458e-01  .52410843e+01 -.47684068e+02  .24726513e-03", 
" -.16220873e+01 -.54340051e-01  .54939001e+01 -.49982272e+02  .24960633e-03", 
" -.16040653e+01 -.68776034e-01  .58280942e+01 -.51594748e+02  .25013602e-03", 
" -.15854842e+01 -.77642238e-01  .57028847e+01 -.45885106e+02  .25439447e-03", 
" -.15663353e+01 -.10614562e+00  .69078180e+01 -.60307525e+02  .25718524e-03", 
" -.15465170e+01 -.12628142e+00  .72050372e+01 -.60767929e+02  .26447509e-03", 
" -.15261262e+01 -.13924918e+00  .74625549e+01 -.63493758e+02  .26882499e-03", 
" -.15050022e+01 -.13436462e+00  .68089601e+01 -.54210652e+02  .27107228e-03", 
" -.14829917e+01 -.13579475e+00  .60728439e+01 -.41216759e+02  .27636310e-03", 
" -.14601364e+01 -.12512447e+00  .49210559e+01 -.25088335e+02  .27460352e-03", 
" -.14360961e+01 -.15993594e+00  .62668613e+01 -.44908110e+02  .27980835e-03", 
" -.14110220e+01 -.15055247e+00  .51819577e+01 -.32054171e+02  .28201218e-03", 
" -.13844055e+01 -.18236196e+00  .57685437e+01 -.39000933e+02  .29818838e-03", 
" -.13562702e+01 -.23127868e+00  .69815973e+01 -.52507925e+02  .30647633e-03", 
" -.13265760e+01 -.26965676e+00  .76872415e+01 -.59686541e+02  .31764918e-03", 
" -.12948116e+01 -.31349127e+00  .79343243e+01 -.56252015e+02  .32785249e-03", 
" -.12608965e+01 -.35097103e+00  .82820475e+01 -.57933310e+02  .33688696e-03", 
" -.12242491e+01 -.36218446e+00  .69401769e+01 -.38221378e+02  .35236835e-03", 
" -.11842953e+01 -.44063850e+00  .88898077e+01 -.60672975e+02  .36687438e-03", 
" -.11402685e+01 -.51073658e+00  .10122396e+02 -.75290498e+02  .40128913e-03", 
" -.10912481e+01 -.60541476e+00  .12479175e+02 -.10900904e+03  .43082084e-03", 
" -.10360940e+01 -.66857236e+00  .12206538e+02 -.10342605e+03  .46189742e-03", 
" -.97250842e+00 -.71018070e+00  .10952345e+02 -.88921438e+02  .50188042e-03", 
" -.89696432e+00 -.77059521e+00  .84633642e+01 -.53864040e+02  .56040671e-03", 
" -.80474102e+00 -.94069529e+00  .10223300e+02 -.67450325e+02  .61922434e-03", 
" -.68654630e+00 -.10371132e+01  .77720869e+01 -.27716405e+02  .72328066e-03", 
" -.52067185e+00 -.11647777e+01  .50114137e+01  .14922769e+02  .89572133e-03", 
" -.47763569e+00 -.12553473e+01  .86691218e+01 -.43658854e+02  .91978451e-03", 
" -.42979771e+00 -.13211068e+01  .11023378e+02 -.87657719e+02  .97180432e-03", 
" -.37601155e+00 -.13757374e+01  .10734238e+02 -.71127027e+02  .99626685e-03", 
" -.31366184e+00 -.14196048e+01  .95688259e+01 -.50500641e+02  .10863957e-02", 
" -.24083729e+00 -.15996239e+01  .16741068e+02 -.14439017e+03  .11431285e-02", 
" -.15309872e+00 -.15563184e+01  .12495120e+02 -.85117211e+02  .12909014e-02", 
" -.40787103e-01 -.14394735e+01  .50823538e+01 -.42342499e+01  .14787206e-02", 
"  .11465518e+00 -.11108780e+01 -.10804391e+02  .18266170e+03  .18554038e-02", 
"  .37559026e+00 -.11566402e+01  .12210543e+01 -.65914172e+02  .25227651e-02", 
"  .62332101e+00 -.28212606e+00 -.21890148e+02  .17168423e+03  .31970724e-02", 
"  .93939915e+00  .28015633e-02 -.14936398e+02  .10005762e+03  .50399347e-02", 
"  .11718545e+01  .69655321e-01  .63323502e+01 -.15548859e+03  .70151609e-02", 
"coc       3  5  2   25", " -.58042792e+01 -.38526936e+02 -.23668577e+03  .43553242e-02", 
" -.56548150e+01 -.35689507e+02 -.17833451e+03  .32527505e-02", 
" -.54425884e+01 -.31334754e+02 -.14470810e+03  .22998961e-02", 
" -.52728533e+01 -.28175660e+02 -.11683782e+03  .16107052e-02", 
" -.50952538e+01 -.25185185e+02 -.88961557e+02  .10892776e-02", 
" -.49879597e+01 -.23337601e+02 -.76552241e+02  .92124114e-03", 
" -.49087833e+01 -.22129330e+02 -.65344309e+02  .82366857e-03", 
" -.48465367e+01 -.21078967e+02 -.59632548e+02  .76325139e-03", 
" -.47946358e+01 -.20225372e+02 -.56089570e+02  .70411891e-03", 
" -.47496796e+01 -.19572804e+02 -.51537785e+02  .66985821e-03", 
" -.47107460e+01 -.18897622e+02 -.49769938e+02  .64548295e-03", 
" -.46754380e+01 -.18412080e+02 -.46117641e+02  .61956857e-03", 
" -.46437418e+01 -.17910237e+02 -.44803851e+02  .60826282e-03", 
" -.45171004e+01 -.16191067e+02 -.34433241e+02  .49760551e-03", 
" -.44233907e+01 -.14950880e+02 -.29352442e+02  .45654850e-03", 
" -.43477985e+01 -.14012959e+02 -.25996269e+02  .41763825e-03", 
" -.42844345e+01 -.13255914e+02 -.22054205e+02  .38066783e-03", 
" -.42296898e+01 -.12595362e+02 -.19641471e+02  .34676300e-03", 
" -.41805852e+01 -.12076358e+02 -.16906533e+02  .33360183e-03", 
" -.41365395e+01 -.11579405e+02 -.15485022e+02  .32840690e-03", 
" -.40963584e+01 -.11170421e+02 -.13357447e+02  .30741960e-03", 
" -.40593243e+01 -.10798883e+02 -.11547138e+02  .29979225e-03", 
" -.40250807e+01 -.10446880e+02 -.10140777e+02  .29168707e-03", 
" -.39930431e+01 -.10097759e+02 -.98263165e+01  .28486105e-03", 
" -.39628347e+01 -.97984364e+01 -.88396793e+01  .27105634e-03", 
" -.39343908e+01 -.95104079e+01 -.82613580e+01  .26390197e-03", 
" -.39073335e+01 -.92539907e+01 -.74778464e+01  .25425829e-03", 
" -.38815460e+01 -.90033543e+01 -.70565442e+01  .25356848e-03", 
" -.38569451e+01 -.87700513e+01 -.64535788e+01  .24405549e-03", 
" -.38332970e+01 -.85485467e+01 -.60936914e+01  .24517728e-03", 
" -.38105416e+01 -.83442361e+01 -.55720055e+01  .23480327e-03", 
" -.37888193e+01 -.81242371e+01 -.57697336e+01  .23184203e-03", 
" -.37677891e+01 -.79251459e+01 -.56411466e+01  .23150117e-03", 
" -.37473214e+01 -.77630270e+01 -.48648709e+01  .22651566e-03", 
" -.37275697e+01 -.75954443e+01 -.43952930e+01  .22268357e-03", 
" -.37083659e+01 -.74343292e+01 -.41106789e+01  .22218165e-03", 
" -.36898172e+01 -.72688600e+01 -.39702365e+01  .22014476e-03", 
" -.36717175e+01 -.71190511e+01 -.37416283e+01  .21755668e-03", 
" -.36540200e+01 -.69861013e+01 -.31709192e+01  .21539439e-03", 
" -.36368115e+01 -.68442245e+01 -.30287684e+01  .21482925e-03", 
" -.36199852e+01 -.67219275e+01 -.24600610e+01  .21529918e-03", 
" -.36036214e+01 -.65897454e+01 -.22911573e+01  .21103807e-03", 
" -.35875601e+01 -.64738069e+01 -.17882694e+01  .20380967e-03", 
" -.35718787e+01 -.63490616e+01 -.17221610e+01  .20169623e-03", 
" -.35565144e+01 -.62298675e+01 -.15378545e+01  .19878483e-03", 
" -.35414630e+01 -.61134955e+01 -.13881617e+01  .19655269e-03", 
" -.35266567e+01 -.60020096e+01 -.12333411e+01  .19723801e-03", 
" -.35121744e+01 -.58892128e+01 -.12498356e+01  .19754307e-03", 
" -.34979478e+01 -.57781731e+01 -.12171051e+01  .19866825e-03", 
" -.34839056e+01 -.56764288e+01 -.11091447e+01  .19826604e-03", 
" -.34700726e+01 -.55838266e+01 -.75111112e+00  .19689895e-03", 
" -.34564369e+01 -.54882410e+01 -.62021559e+00  .19658951e-03", 
" -.34429742e+01 -.54098585e+01 -.48881623e-01  .19797679e-03", 
" -.34297521e+01 -.53198479e+01  .43516774e-01  .20022505e-03", 
" -.34167256e+01 -.52303093e+01  .10865923e+00  .20116575e-03", 
" -.34039326e+01 -.51383842e+01  .84024447e-01  .19945399e-03", 
" -.33912816e+01 -.50534988e+01  .24289358e+00  .19627109e-03", 
" -.33788019e+01 -.49685757e+01  .30218867e+00  .19467206e-03", 
" -.33664223e+01 -.48927682e+01  .53925775e+00  .19339349e-03", 
" -.33542343e+01 -.48100555e+01  .56613825e+00  .19235870e-03", 
" -.33422086e+01 -.47287730e+01  .56941282e+00  .19167708e-03", 
" -.33302783e+01 -.46521180e+01  .62003606e+00  .18827942e-03", 
" -.33184950e+01 -.45763852e+01  .68202326e+00  .18509796e-03", 
" -.33068134e+01 -.45040641e+01  .79637373e+00  .18840468e-03", 
" -.32952347e+01 -.44330613e+01  .82981386e+00  .18981394e-03", 
" -.32837358e+01 -.43694429e+01  .10571654e+01  .18909777e-03", 
" -.32724039e+01 -.43033465e+01  .11938057e+01  .19001938e-03", 
" -.32611830e+01 -.42362410e+01  .12826237e+01  .18793259e-03", 
" -.32501111e+01 -.41645930e+01  .12325767e+01  .19024943e-03", 
" -.32390690e+01 -.41015810e+01  .13945717e+01  .18858072e-03", 
" -.32281038e+01 -.40431101e+01  .16080297e+01  .18902873e-03", 
" -.32172074e+01 -.39890834e+01  .19228732e+01  .18704359e-03", 
" -.32064587e+01 -.39238110e+01  .18828465e+01  .18381442e-03", 
" -.31958233e+01 -.38580226e+01  .18310949e+01  .18313830e-03", 
" -.31852425e+01 -.37993163e+01  .19806262e+01  .18333970e-03", 
" -.31747437e+01 -.37398761e+01  .20595072e+01  .18306895e-03", 
" -.31642963e+01 -.36795046e+01  .21061487e+01  .18350899e-03", 
" -.31539019e+01 -.36219231e+01  .21102299e+01  .18173145e-03", 
" -.31436282e+01 -.35596113e+01  .19901664e+01  .18179767e-03", 
" -.31334109e+01 -.34961622e+01  .18178413e+01  .18035816e-03", 
" -.31232497e+01 -.34347383e+01  .16933500e+01  .17892080e-03", 
" -.31131824e+01 -.33768144e+01  .17164939e+01  .17628176e-03", 
" -.31031080e+01 -.33214130e+01  .16798484e+01  .17799841e-03", 
" -.30931080e+01 -.32662361e+01  .16453674e+01  .17953625e-03", 
" -.30831444e+01 -.32125242e+01  .16397912e+01  .17744231e-03", 
" -.30732235e+01 -.31652508e+01  .18076502e+01  .17692774e-03", 
" -.30633086e+01 -.31170838e+01  .18847443e+01  .17713742e-03", 
" -.30534892e+01 -.30643703e+01  .18365826e+01  .17503817e-03", 
" -.30436982e+01 -.30129687e+01  .17753310e+01  .17456336e-03", 
" -.30339392e+01 -.29660615e+01  .18078799e+01  .17647676e-03", 
" -.30241857e+01 -.29252562e+01  .20262294e+01  .17492578e-03", 
" -.30144655e+01 -.28831942e+01  .21483300e+01  .17731887e-03", 
" -.30048006e+01 -.28372003e+01  .21617075e+01  .17589048e-03", 
" -.29951271e+01 -.27972668e+01  .22970734e+01  .17561935e-03", 
" -.29854940e+01 -.27566641e+01  .24359322e+01  .17270517e-03", 
" -.29759106e+01 -.27123908e+01  .24105968e+01  .17404578e-03", 
" -.29664082e+01 -.26643198e+01  .23181967e+01  .17571575e-03", 
" -.29569152e+01 -.26209945e+01  .23213909e+01  .17503277e-03", 
" -.29474115e+01 -.25793612e+01  .23508694e+01  .17215243e-03", 
" -.29379811e+01 -.25353693e+01  .23045175e+01  .17172779e-03", 
" -.29285678e+01 -.24881405e+01  .22008812e+01  .17020986e-03", 
" -.29191643e+01 -.24446719e+01  .21738716e+01  .17281161e-03", 
" -.29097495e+01 -.24056055e+01  .22206978e+01  .17196626e-03", 
" -.29003178e+01 -.23696371e+01  .23226824e+01  .17285173e-03", 
" -.28908954e+01 -.23316731e+01  .23467556e+01  .17374668e-03", 
" -.28815546e+01 -.22932684e+01  .23681963e+01  .17484542e-03", 
" -.28721823e+01 -.22570942e+01  .24771008e+01  .17710394e-03", 
" -.28628331e+01 -.22184219e+01  .24730790e+01  .17462976e-03", 
" -.28534882e+01 -.21822807e+01  .25044769e+01  .17488078e-03", 
" -.28441807e+01 -.21395380e+01  .23986409e+01  .17514804e-03", 
" -.28348390e+01 -.21016305e+01  .23937146e+01  .17613215e-03", 
" -.28254954e+01 -.20647740e+01  .23750331e+01  .17673135e-03", 
" -.28161884e+01 -.20243338e+01  .23048531e+01  .17661483e-03", 
" -.28068338e+01 -.19908230e+01  .23190936e+01  .17622108e-03", 
" -.27975088e+01 -.19518948e+01  .22270459e+01  .17564485e-03", 
" -.27881513e+01 -.19216691e+01  .23489391e+01  .17501222e-03", 
" -.27788603e+01 -.18825404e+01  .22677769e+01  .17464830e-03", 
" -.27695438e+01 -.18484513e+01  .23217789e+01  .17274252e-03", 
" -.27601819e+01 -.18134389e+01  .22577192e+01  .17343932e-03", 
" -.27508405e+01 -.17758606e+01  .21301251e+01  .17530367e-03", 
" -.27414952e+01 -.17433069e+01  .21783074e+01  .17431136e-03", 
" -.27320910e+01 -.17169366e+01  .23673850e+01  .17574546e-03", 
" -.27227374e+01 -.16793114e+01  .22490776e+01  .17595038e-03", 
" -.27133803e+01 -.16424662e+01  .21317768e+01  .17542767e-03", 
" -.27039745e+01 -.16078874e+01  .20972787e+01  .17610492e-03", 
" -.26945238e+01 -.15753490e+01  .20886237e+01  .17581907e-03", 
" -.26850617e+01 -.15437525e+01  .21036609e+01  .17344572e-03", 
" -.26755453e+01 -.15164552e+01  .22227840e+01  .17466865e-03", 
" -.26660103e+01 -.14903499e+01  .23518798e+01  .17368296e-03", 
" -.26564590e+01 -.14611726e+01  .23744358e+01  .17145311e-03", 
" -.26468611e+01 -.14329751e+01  .23990077e+01  .17319307e-03", 
" -.26372016e+01 -.14097229e+01  .25441247e+01  .17486513e-03", 
" -.26275758e+01 -.13781572e+01  .25058872e+01  .17622793e-03", 
" -.26178886e+01 -.13522858e+01  .25831148e+01  .17576009e-03", 
" -.26081351e+01 -.13287834e+01  .27142917e+01  .17676156e-03", 
" -.25984000e+01 -.12989423e+01  .26765593e+01  .17864240e-03", 
" -.25886149e+01 -.12709606e+01  .26900999e+01  .17801235e-03", 
" -.25787862e+01 -.12468646e+01  .27685469e+01  .17779006e-03", 
" -.25688848e+01 -.12231692e+01  .28708587e+01  .17954013e-03", 
" -.25589600e+01 -.11971154e+01  .28845250e+01  .18007981e-03", 
" -.25489991e+01 -.11699670e+01  .28695864e+01  .17954873e-03", 
" -.25389934e+01 -.11480008e+01  .30029800e+01  .18259599e-03", 
" -.25289503e+01 -.11191854e+01  .29704095e+01  .18378111e-03", 
" -.25188154e+01 -.10946883e+01  .30137797e+01  .18454038e-03", 
" -.25086307e+01 -.10667496e+01  .29234660e+01  .18439255e-03", 
" -.24983504e+01 -.10449275e+01  .29860026e+01  .18252278e-03", 
" -.24880625e+01 -.10196212e+01  .29877701e+01  .18114058e-03", 
" -.24777441e+01 -.98847157e+00  .28306195e+01  .18242211e-03", 
" -.24672698e+01 -.97047631e+00  .30136423e+01  .18124618e-03", 
" -.24567271e+01 -.95011752e+00  .31108904e+01  .18146421e-03", 
" -.24460867e+01 -.93398140e+00  .33267360e+01  .18164882e-03", 
" -.24353941e+01 -.90848363e+00  .32820945e+01  .18380373e-03", 
" -.24246144e+01 -.88779410e+00  .33438351e+01  .18415584e-03", 
" -.24137376e+01 -.86587965e+00  .33778121e+01  .18429196e-03", 
" -.24027678e+01 -.84502035e+00  .34300505e+01  .18568315e-03", 
" -.23916903e+01 -.82404273e+00  .34381817e+01  .18413900e-03", 
" -.23805257e+01 -.80593429e+00  .35432733e+01  .18588222e-03", 
" -.23693377e+01 -.77813469e+00  .34218085e+01  .18613625e-03", 
" -.23580016e+01 -.75727705e+00  .34647724e+01  .18771446e-03", 
" -.23464618e+01 -.73986358e+00  .35607545e+01  .18873969e-03", 
" -.23348212e+01 -.72224692e+00  .36268660e+01  .18730041e-03", 
" -.23230772e+01 -.70154532e+00  .35997817e+01  .18920159e-03", 
" -.23112097e+01 -.67998780e+00  .35512912e+01  .18821962e-03", 
" -.22991944e+01 -.65690423e+00  .35062967e+01  .19003545e-03", 
" -.22870018e+01 -.64236221e+00  .36402547e+01  .18953081e-03", 
" -.22746771e+01 -.62243535e+00  .36110606e+01  .19015410e-03", 
" -.22621720e+01 -.60765440e+00  .37201122e+01  .19193159e-03", 
" -.22495402e+01 -.58943881e+00  .37573412e+01  .19069529e-03", 
" -.22367469e+01 -.56625823e+00  .36319805e+01  .19310530e-03", 
" -.22237204e+01 -.54951618e+00  .36653488e+01  .19387453e-03", 
" -.22105083e+01 -.52888676e+00  .35670395e+01  .19391503e-03", 
" -.21970792e+01 -.50975450e+00  .35103688e+01  .19492274e-03", 
" -.21834014e+01 -.49592265e+00  .35754046e+01  .19783330e-03", 
" -.21694863e+01 -.48165567e+00  .36369982e+01  .19641019e-03", 
" -.21552928e+01 -.46581758e+00  .35831701e+01  .20047653e-03", 
" -.21408651e+01 -.45161922e+00  .36481286e+01  .20231613e-03", 
" -.21261699e+01 -.43597519e+00  .35946490e+01  .20762849e-03", 
" -.21110999e+01 -.43084587e+00  .37744240e+01  .20872812e-03", 
" -.20958061e+01 -.41954116e+00  .38552960e+01  .21316841e-03", 
" -.20801754e+01 -.40789375e+00  .39305752e+01  .21075702e-03", 
" -.20641242e+01 -.39326822e+00  .38825650e+01  .21048241e-03", 
" -.20476924e+01 -.38306443e+00  .39358483e+01  .21337462e-03", 
" -.20307870e+01 -.37775477e+00  .40075678e+01  .21326676e-03", 
" -.20135036e+01 -.36732721e+00  .40648746e+01  .21264326e-03", 
" -.19957455e+01 -.36292681e+00  .42289706e+01  .21255193e-03", 
" -.19774519e+01 -.35405436e+00  .42439124e+01  .21770893e-03", 
" -.19585993e+01 -.34010375e+00  .39946308e+01  .22290675e-03", 
" -.19391251e+01 -.32534685e+00  .37729251e+01  .22403235e-03", 
" -.19189568e+01 -.31699174e+00  .36810720e+01  .22893509e-03", 
" -.18979783e+01 -.32619331e+00  .39822967e+01  .22601746e-03", 
" -.18761507e+01 -.32368968e+00  .38972777e+01  .22874882e-03", 
" -.18533596e+01 -.33224963e+00  .41633322e+01  .22682227e-03", 
" -.18296174e+01 -.33356532e+00  .40954854e+01  .23372414e-03", 
" -.18047059e+01 -.34107722e+00  .41111672e+01  .24086663e-03", 
" -.17784834e+01 -.35680107e+00  .43620014e+01  .24852427e-03", 
" -.17508231e+01 -.37119369e+00  .44507653e+01  .25655466e-03", 
" -.17216037e+01 -.38281644e+00  .43751062e+01  .25472510e-03", 
" -.16904852e+01 -.39988070e+00  .44549784e+01  .26445120e-03", 
" -.16570432e+01 -.42536685e+00  .45370711e+01  .26986942e-03", 
" -.16211326e+01 -.44392432e+00  .45488178e+01  .28275405e-03", 
" -.15820078e+01 -.46701158e+00  .43056356e+01  .29453958e-03", 
" -.15389543e+01 -.51845849e+00  .46625816e+01  .31603153e-03", 
" -.14911274e+01 -.56649727e+00  .46136427e+01  .33487822e-03", 
" -.14372127e+01 -.63295122e+00  .47663430e+01  .36492529e-03", 
" -.13750058e+01 -.70341583e+00  .46073131e+01  .41388706e-03", 
" -.13017829e+01 -.81397614e+00  .51761856e+01  .45543833e-03", 
" -.12127866e+01 -.89035871e+00  .44431536e+01  .50642669e-03", 
" -.10982634e+01 -.10243999e+01  .44402071e+01  .59039614e-03", 
" -.93818641e+00 -.11396691e+01  .28648977e+01  .73933826e-03", 
" -.89686485e+00 -.11931299e+01  .30828692e+01  .76913350e-03", 
" -.85084109e+00 -.12501287e+01  .37731671e+01  .78831595e-03", 
" -.79823979e+00 -.13641828e+01  .44400447e+01  .84804109e-03", 
" -.73874452e+00 -.14214859e+01  .46260863e+01  .91397269e-03", 
" -.66907981e+00 -.14900403e+01  .49492125e+01  .10031643e-02", 
" -.58443495e+00 -.15275175e+01  .30233626e+01  .10879598e-02", 
" -.47740967e+00 -.15201223e+01  .61584706e-02  .12444509e-02", 
" -.32891274e+00 -.16845991e+01  .48973252e+00  .15194760e-02", 
" -.80836631e-01 -.20897161e+01  .18780225e+01  .19752457e-02", 
"  .15253552e+00 -.24681379e+01  .55492438e+01  .26142219e-02", 
"  .44637577e+00 -.28152717e+01  .50660522e+01  .41831603e-02", 
"  .65076893e+00 -.18302433e+01 -.20008620e+02  .57245183e-02", 
"coct      3  6  3   20", " -.61091781e+01 -.47031409e+02 -.20576162e+03 -.91887076e+03  .49379429e-02", 
" -.59610397e+01 -.42854410e+02 -.20297359e+03 -.22026096e+03  .36534049e-02", 
" -.57573345e+01 -.37665165e+02 -.17156640e+03  .19819377e+03  .24106878e-02", 
" -.55896725e+01 -.34541577e+02 -.11241633e+03 -.22791185e+03  .17491856e-02", 
" -.54157049e+01 -.30744280e+02 -.10015418e+03  .19983562e+02  .12618393e-02", 
" -.53104794e+01 -.28462916e+02 -.94933122e+02  .13173593e+03  .11373168e-02", 
" -.52327607e+01 -.26988142e+02 -.85381920e+02  .12485140e+03  .10202756e-02", 
" -.51708761e+01 -.25889400e+02 -.76429908e+02  .10312359e+03  .93279501e-03", 
" -.51191278e+01 -.24787662e+02 -.82173516e+02  .27311895e+03  .85145055e-03", 
" -.50751254e+01 -.23962254e+02 -.79785991e+02  .30478256e+03  .79697219e-03", 
" -.50361187e+01 -.23350030e+02 -.72071231e+02  .23903337e+03  .75775035e-03", 
" -.50013529e+01 -.22796743e+02 -.65763493e+02  .19286892e+03  .73619466e-03", 
" -.49699708e+01 -.22206600e+02 -.66484787e+02  .24098168e+03  .73566471e-03", 
" -.48449525e+01 -.20155783e+02 -.59115368e+02  .27794590e+03  .59658406e-03", 
" -.47521391e+01 -.18721591e+02 -.55681343e+02  .33846310e+03  .53661459e-03", 
" -.46781359e+01 -.17598116e+02 -.50072194e+02  .30576338e+03  .51037444e-03", 
" -.46152267e+01 -.16792036e+02 -.41697164e+02  .23294373e+03  .44704061e-03", 
" -.45609699e+01 -.16043375e+02 -.38393689e+02  .22273875e+03  .41591931e-03", 
" -.45126791e+01 -.15457879e+02 -.32673155e+02  .17973802e+03  .39564343e-03", 
" -.44692083e+01 -.14949849e+02 -.27028438e+02  .12850928e+03  .37716694e-03", 
" -.44295026e+01 -.14433579e+02 -.26778852e+02  .15682280e+03  .36213264e-03", 
" -.43928088e+01 -.14031279e+02 -.22737503e+02  .12496153e+03  .35546885e-03", 
" -.43586900e+01 -.13671612e+02 -.18109389e+02  .77974578e+02  .33567566e-03", 
" -.43269122e+01 -.13292620e+02 -.16574061e+02  .74036040e+02  .32599197e-03", 
" -.42968665e+01 -.12980744e+02 -.13658890e+02  .50897049e+02  .32233833e-03", 
" -.42688391e+01 -.12609802e+02 -.14920866e+02  .80601619e+02  .31718370e-03", 
" -.42420648e+01 -.12299075e+02 -.14553034e+02  .89041299e+02  .31206997e-03", 
" -.42165653e+01 -.12002065e+02 -.14330831e+02  .96944401e+02  .30772346e-03", 
" -.41920112e+01 -.11756155e+02 -.12371443e+02  .78349472e+02  .30100301e-03", 
" -.41686404e+01 -.11516194e+02 -.10757308e+02  .65411610e+02  .29978060e-03", 
" -.41461655e+01 -.11300728e+02 -.82931554e+01  .38365431e+02  .29978845e-03", 
" -.41247177e+01 -.11044274e+02 -.85918747e+01  .47904527e+02  .30024705e-03", 
" -.41039764e+01 -.10813907e+02 -.82017273e+01  .48518960e+02  .29532338e-03", 
" -.40839040e+01 -.10580232e+02 -.88232513e+01  .62722981e+02  .28655447e-03", 
" -.40643349e+01 -.10363729e+02 -.93332498e+01  .76395384e+02  .28785207e-03", 
" -.40455373e+01 -.10148326e+02 -.97321739e+01  .85648304e+02  .28300380e-03", 
" -.40271655e+01 -.99556697e+01 -.96569017e+01  .89146640e+02  .27726314e-03", 
" -.40092418e+01 -.97829035e+01 -.89263119e+01  .84017069e+02  .27155955e-03", 
" -.39918122e+01 -.96115683e+01 -.84947940e+01  .82138902e+02  .27197008e-03", 
" -.39747746e+01 -.94530222e+01 -.77583722e+01  .75823952e+02  .27098467e-03", 
" -.39582332e+01 -.92889672e+01 -.76922939e+01  .79365715e+02  .26748013e-03", 
" -.39420776e+01 -.91486249e+01 -.62334998e+01  .60834083e+02  .26177464e-03", 
" -.39262695e+01 -.89874530e+01 -.67094359e+01  .72654422e+02  .25785736e-03", 
" -.39107814e+01 -.88454356e+01 -.62365969e+01  .69594414e+02  .25578247e-03", 
" -.38955791e+01 -.87201776e+01 -.51433388e+01  .58932459e+02  .25441572e-03", 
" -.38807279e+01 -.85761871e+01 -.51893221e+01  .62129860e+02  .25256279e-03", 
" -.38661948e+01 -.84285431e+01 -.55055221e+01  .68028384e+02  .24864043e-03", 
" -.38518030e+01 -.83016788e+01 -.54042404e+01  .72339183e+02  .24740562e-03", 
" -.38376892e+01 -.81648159e+01 -.57765431e+01  .80233848e+02  .24969057e-03", 
" -.38238350e+01 -.80394674e+01 -.56342756e+01  .80227220e+02  .24896273e-03", 
" -.38102451e+01 -.79130196e+01 -.56238947e+01  .81144733e+02  .24519360e-03", 
" -.37967808e+01 -.78076395e+01 -.48353072e+01  .73033382e+02  .24929893e-03", 
" -.37835779e+01 -.76937455e+01 -.46645778e+01  .73213146e+02  .24691854e-03", 
" -.37704934e+01 -.75941858e+01 -.38323566e+01  .63387452e+02  .24191430e-03", 
" -.37576501e+01 -.74916626e+01 -.30543616e+01  .52837324e+02  .24060540e-03", 
" -.37449708e+01 -.73778476e+01 -.34272353e+01  .59941276e+02  .24117040e-03", 
" -.37324331e+01 -.72800836e+01 -.30125128e+01  .57032798e+02  .23852961e-03", 
" -.37200539e+01 -.71832353e+01 -.26184821e+01  .53891501e+02  .23777812e-03", 
" -.37078637e+01 -.70830778e+01 -.25937531e+01  .56009200e+02  .23706915e-03", 
" -.36958685e+01 -.69714261e+01 -.31369161e+01  .64316243e+02  .23532389e-03", 
" -.36838825e+01 -.68869572e+01 -.24850905e+01  .57475786e+02  .23624037e-03", 
" -.36721554e+01 -.67803734e+01 -.30512371e+01  .66123362e+02  .23336116e-03", 
" -.36605194e+01 -.66990065e+01 -.22196310e+01  .55703973e+02  .23250250e-03", 
" -.36490207e+01 -.65994836e+01 -.24762588e+01  .59107435e+02  .23318185e-03", 
" -.36375538e+01 -.65289803e+01 -.15195662e+01  .48709283e+02  .23213752e-03", 
" -.36262735e+01 -.64447142e+01 -.12485517e+01  .46282027e+02  .22805828e-03", 
" -.36150384e+01 -.63636406e+01 -.10198078e+01  .43771957e+02  .22604991e-03", 
" -.36039508e+01 -.62801187e+01 -.89634437e+00  .43172097e+02  .22659937e-03", 
" -.35929229e+01 -.62135810e+01 -.32954378e-01  .33677123e+02  .22553837e-03", 
" -.35820349e+01 -.61384277e+01  .41187609e+00  .27911207e+02  .22330945e-03", 
" -.35712447e+01 -.60549385e+01  .25713011e+00  .30792011e+02  .22389305e-03", 
" -.35605493e+01 -.59740160e+01  .13715876e+00  .33146281e+02  .22324498e-03", 
" -.35499338e+01 -.58915441e+01 -.88160984e-01  .37395573e+02  .22315592e-03", 
" -.35393941e+01 -.58184067e+01  .16661014e+00  .35417119e+02  .22260967e-03", 
" -.35289161e+01 -.57527135e+01  .68489174e+00  .28869767e+02  .22130230e-03", 
" -.35185214e+01 -.56795314e+01  .82265023e+00  .27039437e+02  .22179304e-03", 
" -.35081874e+01 -.56121066e+01  .12495540e+01  .21784421e+02  .22003947e-03", 
" -.34979467e+01 -.55304951e+01  .64168199e+00  .31516288e+02  .22145882e-03", 
" -.34878011e+01 -.54507112e+01  .32394319e+00  .34900053e+02  .22350281e-03", 
" -.34776632e+01 -.53875471e+01  .68482646e+00  .31466409e+02  .22489426e-03", 
" -.34675773e+01 -.53284599e+01  .11921904e+01  .25718743e+02  .22432208e-03", 
" -.34575780e+01 -.52615302e+01  .12216168e+01  .26349352e+02  .22416632e-03", 
" -.34476270e+01 -.52003323e+01  .15871940e+01  .21504889e+02  .22553225e-03", 
" -.34377381e+01 -.51229000e+01  .78808894e+00  .34401203e+02  .22244743e-03", 
" -.34278937e+01 -.50570314e+01  .87380576e+00  .32300283e+02  .22165766e-03", 
" -.34181293e+01 -.49855954e+01  .58393112e+00  .35587283e+02  .22339583e-03", 
" -.34084034e+01 -.49172396e+01  .29950019e+00  .40596827e+02  .22296261e-03", 
" -.33986566e+01 -.48597054e+01  .57505202e+00  .36866770e+02  .21991243e-03", 
" -.33889716e+01 -.47987595e+01  .63345438e+00  .36004350e+02  .21985966e-03", 
" -.33793428e+01 -.47438846e+01  .92111371e+00  .33858185e+02  .22000669e-03", 
" -.33697823e+01 -.46787366e+01  .68176224e+00  .38146199e+02  .21805887e-03", 
" -.33602347e+01 -.46160074e+01  .42014536e+00  .43007482e+02  .21467870e-03", 
" -.33507617e+01 -.45604397e+01  .78897748e+00  .37071257e+02  .21501202e-03", 
" -.33412469e+01 -.45073548e+01  .94465968e+00  .36713665e+02  .21638065e-03", 
" -.33318134e+01 -.44518664e+01  .11996747e+01  .32334692e+02  .21537587e-03", 
" -.33224145e+01 -.43913356e+01  .90317036e+00  .37674617e+02  .21477609e-03", 
" -.33130174e+01 -.43363373e+01  .93036405e+00  .38542538e+02  .21457852e-03", 
" -.33036306e+01 -.42868198e+01  .13993599e+01  .31578570e+02  .21102202e-03", 
" -.32943013e+01 -.42283744e+01  .11172421e+01  .36073310e+02  .21035114e-03", 
" -.32849749e+01 -.41768433e+01  .12015984e+01  .36639940e+02  .21025227e-03", 
" -.32756209e+01 -.41342482e+01  .15759396e+01  .33182084e+02  .21045860e-03", 
" -.32663559e+01 -.40794720e+01  .14740724e+01  .34979482e+02  .20750781e-03", 
" -.32570990e+01 -.40234518e+01  .13656183e+01  .36195703e+02  .20932149e-03", 
" -.32478661e+01 -.39730437e+01  .14580803e+01  .35207984e+02  .20636333e-03", 
" -.32386622e+01 -.39228528e+01  .16354822e+01  .32333121e+02  .20335528e-03", 
" -.32294532e+01 -.38661055e+01  .13191028e+01  .36402476e+02  .20098982e-03", 
" -.32202398e+01 -.38200787e+01  .14837350e+01  .35308796e+02  .20091942e-03", 
" -.32110471e+01 -.37687930e+01  .14740613e+01  .34609680e+02  .20186362e-03", 
" -.32018426e+01 -.37238196e+01  .18361427e+01  .28628746e+02  .20039657e-03", 
" -.31926845e+01 -.36731445e+01  .17845653e+01  .28642834e+02  .19906808e-03", 
" -.31835017e+01 -.36266225e+01  .17648358e+01  .30069900e+02  .19941572e-03", 
" -.31743495e+01 -.35766742e+01  .16833974e+01  .31315544e+02  .20218888e-03", 
" -.31651689e+01 -.35254516e+01  .14876113e+01  .33380073e+02  .20111515e-03", 
" -.31559825e+01 -.34780061e+01  .14818003e+01  .33669604e+02  .19899892e-03", 
" -.31467574e+01 -.34392117e+01  .17581783e+01  .30748181e+02  .19980265e-03", 
" -.31376021e+01 -.33954352e+01  .19394252e+01  .28557209e+02  .20166467e-03", 
" -.31284096e+01 -.33545908e+01  .22507826e+01  .24285688e+02  .20123496e-03", 
" -.31192094e+01 -.33134183e+01  .23337496e+01  .23648415e+02  .19878710e-03", 
" -.31100197e+01 -.32628920e+01  .19629231e+01  .29512682e+02  .19712462e-03", 
" -.31008104e+01 -.32248781e+01  .22436246e+01  .26471116e+02  .19844839e-03", 
" -.30915929e+01 -.31895112e+01  .28296964e+01  .17681350e+02  .19737597e-03", 
" -.30823344e+01 -.31485122e+01  .27962855e+01  .19375419e+02  .19726267e-03", 
" -.30731332e+01 -.31001705e+01  .25344785e+01  .23326701e+02  .19826418e-03", 
" -.30638383e+01 -.30666065e+01  .29683324e+01  .17012601e+02  .19739855e-03", 
" -.30545620e+01 -.30308788e+01  .33391395e+01  .12378369e+02  .19721560e-03", 
" -.30452859e+01 -.29804703e+01  .28562335e+01  .18304164e+02  .19834224e-03", 
" -.30359896e+01 -.29331482e+01  .25450045e+01  .22332614e+02  .19863617e-03", 
" -.30266424e+01 -.28989731e+01  .28990098e+01  .17068927e+02  .19762174e-03", 
" -.30172833e+01 -.28615325e+01  .30225805e+01  .15589331e+02  .19444943e-03", 
" -.30078932e+01 -.28265459e+01  .32356976e+01  .13462629e+02  .19471936e-03", 
" -.29985003e+01 -.27813230e+01  .28395954e+01  .19164510e+02  .19203816e-03", 
" -.29890447e+01 -.27505073e+01  .32278847e+01  .14324518e+02  .19275664e-03", 
" -.29795408e+01 -.27248609e+01  .37858302e+01  .80803302e+01  .19283993e-03", 
" -.29700306e+01 -.26925209e+01  .42106604e+01  .15757407e+01  .19203464e-03", 
" -.29604773e+01 -.26588076e+01  .42893894e+01  .14842209e+01  .19184532e-03", 
" -.29508921e+01 -.26335145e+01  .50739287e+01 -.97208577e+01  .19239521e-03", 
" -.29412963e+01 -.25975227e+01  .50338853e+01 -.76266023e+01  .19398339e-03", 
" -.29316643e+01 -.25649988e+01  .51723958e+01 -.86597583e+01  .19379681e-03", 
" -.29219351e+01 -.25309469e+01  .50544028e+01 -.54361805e+01  .19351838e-03", 
" -.29122534e+01 -.24908027e+01  .49633255e+01 -.44101163e+01  .19005403e-03", 
" -.29024684e+01 -.24604673e+01  .51516038e+01 -.59009067e+01  .18861997e-03", 
" -.28926178e+01 -.24407464e+01  .60541801e+01 -.18437211e+02  .18659325e-03", 
" -.28827738e+01 -.24094107e+01  .64159651e+01 -.24673586e+02  .18730758e-03", 
" -.28728489e+01 -.23740731e+01  .63347441e+01 -.22569629e+02  .19013684e-03", 
" -.28629296e+01 -.23256327e+01  .57393715e+01 -.15762017e+02  .18948552e-03", 
" -.28528734e+01 -.22995739e+01  .62132070e+01 -.21980378e+02  .19030882e-03", 
" -.28427918e+01 -.22623234e+01  .59210159e+01 -.16811638e+02  .18869778e-03", 
" -.28326227e+01 -.22175297e+01  .51508365e+01 -.55837034e+01  .19171534e-03", 
" -.28223474e+01 -.21934482e+01  .54465467e+01 -.74026124e+01  .19002871e-03", 
" -.28121105e+01 -.21477673e+01  .48726676e+01 -.13291185e-01  .19259210e-03", 
" -.28017200e+01 -.21198405e+01  .49721663e+01  .39122725e+00  .19510176e-03", 
" -.27912718e+01 -.20815900e+01  .47130322e+01  .31586424e+01  .19766116e-03", 
" -.27807470e+01 -.20457460e+01  .44570704e+01  .72206698e+01  .19950008e-03", 
" -.27700972e+01 -.20259091e+01  .50904086e+01 -.95356404e+00  .20036196e-03", 
" -.27593873e+01 -.19917485e+01  .49565385e+01  .33758198e+00  .20061616e-03", 
" -.27486048e+01 -.19566570e+01  .48261713e+01  .17165565e+01  .20394231e-03", 
" -.27377173e+01 -.19350879e+01  .54542379e+01 -.75615009e+01  .20653281e-03", 
" -.27267282e+01 -.19007831e+01  .51255020e+01 -.25653719e+01  .20728207e-03", 
" -.27156406e+01 -.18701471e+01  .50988409e+01 -.21549129e+01  .20829303e-03", 
" -.27044490e+01 -.18447790e+01  .53303808e+01 -.50830422e+01  .21131798e-03", 
" -.26931677e+01 -.18105331e+01  .49960753e+01  .29999478e+00  .21122438e-03", 
" -.26817302e+01 -.17861555e+01  .52471349e+01 -.25301806e+01  .21278367e-03", 
" -.26702158e+01 -.17607861e+01  .55314654e+01 -.66840447e+01  .20924311e-03", 
" -.26584812e+01 -.17421634e+01  .58900718e+01 -.10305967e+02  .20910525e-03", 
" -.26465752e+01 -.17218146e+01  .60730161e+01 -.11625647e+02  .20830469e-03", 
" -.26345942e+01 -.16929900e+01  .58924436e+01 -.88226998e+01  .20937192e-03", 
" -.26224798e+01 -.16621788e+01  .57737517e+01 -.76318193e+01  .20949929e-03", 
" -.26101712e+01 -.16416229e+01  .60659246e+01 -.11216035e+02  .20917283e-03", 
" -.25977530e+01 -.16135864e+01  .60584465e+01 -.11145766e+02  .21049680e-03", 
" -.25851875e+01 -.15737082e+01  .55415156e+01 -.56227365e+01  .21270543e-03", 
" -.25724561e+01 -.15357149e+01  .50868757e+01  .50123598e+00  .21476474e-03", 
" -.25594553e+01 -.15126031e+01  .51028453e+01  .17455263e+01  .21914683e-03", 
" -.25461820e+01 -.14972553e+01  .54883749e+01 -.21587651e+01  .21821430e-03", 
" -.25326344e+01 -.14933972e+01  .62295728e+01 -.89282428e+01  .22117313e-03", 
" -.25189647e+01 -.14663630e+01  .59629754e+01 -.43806509e+01  .22362350e-03", 
" -.25050615e+01 -.14391312e+01  .57115229e+01 -.95987933e+00  .22547034e-03", 
" -.24908871e+01 -.14219494e+01  .59224149e+01 -.16459583e+01  .22636564e-03", 
" -.24763658e+01 -.14135988e+01  .67124597e+01 -.13174716e+02  .23246945e-03", 
" -.24616115e+01 -.13877064e+01  .65239086e+01 -.10715217e+02  .23101432e-03", 
" -.24465982e+01 -.13463983e+01  .55913585e+01  .33104795e+00  .23291848e-03", 
" -.24311584e+01 -.13395754e+01  .61623499e+01 -.58396961e+01  .23577176e-03", 
" -.24154512e+01 -.12989584e+01  .49536892e+01  .10880242e+02  .23291462e-03", 
" -.23993348e+01 -.12896358e+01  .56459233e+01  .86106168e+00  .23447630e-03", 
" -.23827335e+01 -.12821227e+01  .60063713e+01 -.23386671e+01  .23856562e-03", 
" -.23657665e+01 -.12616339e+01  .59650406e+01 -.20845797e+01  .24092846e-03", 
" -.23481928e+01 -.12694973e+01  .69657093e+01 -.13113107e+02  .24394182e-03", 
" -.23301599e+01 -.12617326e+01  .71873376e+01 -.14372537e+02  .24798234e-03", 
" -.23115195e+01 -.12628099e+01  .75000287e+01 -.14725441e+02  .25119268e-03", 
" -.22923316e+01 -.12631629e+01  .82275205e+01 -.24112140e+02  .25564584e-03", 
" -.22724430e+01 -.12649288e+01  .87143488e+01 -.28427631e+02  .25285608e-03", 
" -.22518501e+01 -.12664035e+01  .89678939e+01 -.26095488e+02  .25990943e-03", 
" -.22303749e+01 -.12804316e+01  .10101163e+02 -.40607830e+02  .26610912e-03", 
" -.22080046e+01 -.12797067e+01  .10082247e+02 -.38571697e+02  .26615001e-03", 
" -.21846584e+01 -.12799821e+01  .10158437e+02 -.38898653e+02  .27831110e-03", 
" -.21600503e+01 -.13108088e+01  .11567419e+02 -.55931194e+02  .28800387e-03", 
" -.21341650e+01 -.13614821e+01  .13598305e+02 -.77543771e+02  .29617917e-03", 
" -.21068543e+01 -.14058240e+01  .15416110e+02 -.10003234e+03  .29594051e-03", 
" -.20780187e+01 -.14499544e+01  .17448154e+02 -.12756042e+03  .30609304e-03", 
" -.20471213e+01 -.14999363e+01  .19049985e+02 -.14657437e+03  .31087173e-03", 
" -.20139529e+01 -.15367809e+01  .19005348e+02 -.13885816e+03  .31916593e-03", 
" -.19780992e+01 -.15860911e+01  .19386128e+02 -.13574982e+03  .33507955e-03", 
" -.19388293e+01 -.16852416e+01  .21977634e+02 -.16489238e+03  .34631090e-03", 
" -.18955904e+01 -.17584681e+01  .22441570e+02 -.16559592e+03  .37089565e-03", 
" -.18472744e+01 -.18274287e+01  .21560263e+02 -.14466926e+03  .40844102e-03", 
" -.17920917e+01 -.19272718e+01  .20975241e+02 -.12578191e+03  .45204679e-03", 
" -.17271040e+01 -.21035720e+01  .21851675e+02 -.12048199e+03  .48422937e-03", 
" -.16487104e+01 -.23301406e+01  .22655667e+02 -.11355976e+03  .51242240e-03", 
" -.15487054e+01 -.26420386e+01  .23607054e+02 -.99306301e+02  .57309484e-03", 
" -.14091483e+01 -.29587449e+01  .15192428e+02  .40864100e+02  .70910542e-03", 
" -.13732975e+01 -.30185657e+01  .12274834e+02  .80429656e+02  .73716049e-03", 
" -.13328995e+01 -.32419280e+01  .16307636e+02  .33479934e+02  .77619014e-03", 
" -.12875464e+01 -.34040934e+01  .16067317e+02  .45912269e+02  .83239705e-03", 
" -.12361042e+01 -.34894905e+01  .11511855e+02  .11272187e+03  .88379658e-03", 
" -.11747702e+01 -.37745980e+01  .15213476e+02  .67301420e+02  .90558167e-03", 
" -.11007069e+01 -.39910090e+01  .11062514e+02  .14102309e+03  .10199670e-02", 
" -.10062067e+01 -.45038530e+01  .18136346e+02  .73693941e+02  .11509968e-02", 
" -.87535862e+00 -.51656591e+01  .27341327e+02 -.70598934e+02  .14138907e-02", 
" -.65880515e+00 -.62785804e+01  .35013274e+02 -.13974647e+03  .19465233e-02", 
" -.45200277e+00 -.69896464e+01  .15612045e+02  .13774144e+03  .27032968e-02", 
" -.18909904e+00 -.82598604e+01 -.29009395e+01  .51372125e+03  .41703413e-02", 
" -.58224926e-03 -.95153414e+01 -.57025642e+01  .58853246e+03  .56793814e-02", 
"coctt     3  7  2   30", " -.64017292e+01 -.53518499e+02 -.31699248e+03  .45648579e-02", 
" -.62524367e+01 -.48818238e+02 -.29292090e+03  .34095840e-02", 
" -.60466300e+01 -.43957743e+02 -.22582800e+03  .20038936e-02", 
" -.58840630e+01 -.39849624e+02 -.18760830e+03  .15263185e-02", 
" -.57113667e+01 -.36072029e+02 -.14685507e+03  .11981888e-02", 
" -.56060746e+01 -.33899437e+02 -.12477972e+03  .96665232e-03", 
" -.55283137e+01 -.32422683e+02 -.10933448e+03  .87116441e-03", 
" -.54676544e+01 -.31025730e+02 -.10379168e+03  .78354321e-03", 
" -.54160500e+01 -.30144460e+02 -.92427184e+02  .71259736e-03", 
" -.53723162e+01 -.29302649e+02 -.86523964e+02  .65455731e-03", 
" -.53339282e+01 -.28531410e+02 -.83019094e+02  .63091088e-03", 
" -.52994351e+01 -.27818194e+02 -.81521507e+02  .62144071e-03", 
" -.52678710e+01 -.27260883e+02 -.76790374e+02  .59830250e-03", 
" -.51435003e+01 -.25096724e+02 -.61178571e+02  .50402366e-03", 
" -.50519985e+01 -.23502217e+02 -.52428004e+02  .45619638e-03", 
" -.49782672e+01 -.22267035e+02 -.47036925e+02  .43909090e-03", 
" -.49163039e+01 -.21275226e+02 -.41560667e+02  .41568282e-03", 
" -.48627427e+01 -.20414802e+02 -.37416805e+02  .40094406e-03", 
" -.48151483e+01 -.19685268e+02 -.34222638e+02  .38004156e-03", 
" -.47720988e+01 -.19038673e+02 -.32051510e+02  .36322784e-03", 
" -.47327713e+01 -.18486969e+02 -.29118339e+02  .35159314e-03", 
" -.46968739e+01 -.17922647e+02 -.28082459e+02  .32842543e-03", 
" -.46632015e+01 -.17446133e+02 -.26605857e+02  .31235290e-03", 
" -.46318219e+01 -.17010054e+02 -.25055383e+02  .29718550e-03", 
" -.46021143e+01 -.16639283e+02 -.22553872e+02  .29824237e-03", 
" -.45740647e+01 -.16291789e+02 -.20446035e+02  .28962102e-03", 
" -.45476109e+01 -.15934421e+02 -.19243450e+02  .28527632e-03", 
" -.45223086e+01 -.15604523e+02 -.18452744e+02  .27807214e-03", 
" -.44981433e+01 -.15302254e+02 -.17222801e+02  .27623229e-03", 
" -.44751206e+01 -.14995334e+02 -.16540620e+02  .26767559e-03", 
" -.44529311e+01 -.14727506e+02 -.15200283e+02  .25672524e-03", 
" -.44315700e+01 -.14465078e+02 -.14189542e+02  .25641800e-03", 
" -.44109983e+01 -.14202471e+02 -.13597039e+02  .25521626e-03", 
" -.43910969e+01 -.13956414e+02 -.12918265e+02  .24824425e-03", 
" -.43717903e+01 -.13724213e+02 -.12251058e+02  .24648816e-03", 
" -.43530364e+01 -.13503622e+02 -.11625491e+02  .24842923e-03", 
" -.43347805e+01 -.13294324e+02 -.10933869e+02  .24561642e-03", 
" -.43170033e+01 -.13103360e+02 -.99768367e+01  .24441287e-03", 
" -.42997228e+01 -.12902022e+02 -.94989859e+01  .23785480e-03", 
" -.42828710e+01 -.12719350e+02 -.88036182e+01  .23461390e-03", 
" -.42664087e+01 -.12539033e+02 -.82502070e+01  .23046578e-03", 
" -.42503685e+01 -.12362919e+02 -.76801372e+01  .23074640e-03", 
" -.42346800e+01 -.12191212e+02 -.72492872e+01  .22736365e-03", 
" -.42193315e+01 -.12020496e+02 -.68968962e+01  .22445851e-03", 
" -.42043523e+01 -.11843543e+02 -.68924276e+01  .22296123e-03", 
" -.41895449e+01 -.11695101e+02 -.61432597e+01  .22462634e-03", 
" -.41750867e+01 -.11538030e+02 -.59502196e+01  .22460873e-03", 
" -.41608823e+01 -.11382102e+02 -.57990798e+01  .22013894e-03", 
" -.41469230e+01 -.11247413e+02 -.49598551e+01  .22200101e-03", 
" -.41331330e+01 -.11115370e+02 -.43774259e+01  .21855474e-03", 
" -.41196355e+01 -.10978724e+02 -.40195197e+01  .21939054e-03", 
" -.41063010e+01 -.10845088e+02 -.37219238e+01  .22022249e-03", 
" -.40932255e+01 -.10709927e+02 -.35631191e+01  .21721224e-03", 
" -.40802893e+01 -.10592540e+02 -.28792030e+01  .21614092e-03", 
" -.40675150e+01 -.10477350e+02 -.23299216e+01  .21909835e-03", 
" -.40549243e+01 -.10352868e+02 -.21934305e+01  .21964405e-03", 
" -.40425909e+01 -.10217576e+02 -.24630628e+01  .21879685e-03", 
" -.40303718e+01 -.10105570e+02 -.19359540e+01  .21521359e-03", 
" -.40182369e+01 -.99961182e+01 -.16311268e+01  .21500166e-03", 
" -.40062597e+01 -.98916237e+01 -.11036506e+01  .21519324e-03", 
" -.39944727e+01 -.97792682e+01 -.85049659e+00  .21323350e-03", 
" -.39828220e+01 -.96718192e+01 -.54753173e+00  .21215080e-03", 
" -.39712770e+01 -.95646845e+01 -.32362263e+00  .21126696e-03", 
" -.39598385e+01 -.94658144e+01  .49065139e-01  .21203245e-03", 
" -.39485492e+01 -.93602686e+01  .14243502e+00  .20972680e-03", 
" -.39373858e+01 -.92569686e+01  .24476289e+00  .20679936e-03", 
" -.39263386e+01 -.91538554e+01  .33120588e+00  .20521109e-03", 
" -.39153902e+01 -.90481126e+01  .20707704e+00  .20576864e-03", 
" -.39045043e+01 -.89534464e+01  .39607750e+00  .20257788e-03", 
" -.38937104e+01 -.88654411e+01  .74795156e+00  .20062778e-03", 
" -.38830551e+01 -.87723725e+01  .92977576e+00  .20024664e-03", 
" -.38723747e+01 -.86965783e+01  .15497607e+01  .19962829e-03", 
" -.38618809e+01 -.86078550e+01  .17458531e+01  .19901454e-03", 
" -.38514313e+01 -.85204094e+01  .19369653e+01  .19779718e-03", 
" -.38410349e+01 -.84406337e+01  .22283223e+01  .19640403e-03", 
" -.38307323e+01 -.83611972e+01  .25163839e+01  .19586655e-03", 
" -.38205103e+01 -.82768576e+01  .27219118e+01  .19406963e-03", 
" -.38103710e+01 -.81920158e+01  .28156548e+01  .19356528e-03", 
" -.38003576e+01 -.80967050e+01  .26032323e+01  .19194612e-03", 
" -.37903626e+01 -.80117124e+01  .25872831e+01  .19152116e-03", 
" -.37804563e+01 -.79219026e+01  .23962175e+01  .19058139e-03", 
" -.37705761e+01 -.78426323e+01  .25564691e+01  .19002601e-03", 
" -.37606979e+01 -.77681739e+01  .27425525e+01  .19022246e-03", 
" -.37509097e+01 -.76937639e+01  .29479355e+01  .18778273e-03", 
" -.37412023e+01 -.76127970e+01  .29262114e+01  .18996254e-03", 
" -.37315297e+01 -.75317979e+01  .28473116e+01  .18716465e-03", 
" -.37218831e+01 -.74579207e+01  .29545183e+01  .18873381e-03", 
" -.37122443e+01 -.73837270e+01  .29473742e+01  .18845792e-03", 
" -.37026744e+01 -.73156006e+01  .31564396e+01  .19116560e-03", 
" -.36930966e+01 -.72537660e+01  .35719980e+01  .18779348e-03", 
" -.36835701e+01 -.71873840e+01  .37768207e+01  .18919623e-03", 
" -.36741065e+01 -.71172635e+01  .38539922e+01  .18838048e-03", 
" -.36646978e+01 -.70457352e+01  .38461658e+01  .18976753e-03", 
" -.36553053e+01 -.69789105e+01  .39798812e+01  .18933561e-03", 
" -.36458872e+01 -.69210254e+01  .42765136e+01  .18894482e-03", 
" -.36365604e+01 -.68539834e+01  .43359698e+01  .18812743e-03", 
" -.36271953e+01 -.67957277e+01  .46974985e+01  .18746978e-03", 
" -.36179539e+01 -.67255964e+01  .45904425e+01  .18765647e-03", 
" -.36087133e+01 -.66667668e+01  .48519919e+01  .18468572e-03", 
" -.35994875e+01 -.66024014e+01  .49177359e+01  .18428780e-03", 
" -.35902724e+01 -.65422701e+01  .50045170e+01  .18460897e-03", 
" -.35811324e+01 -.64756710e+01  .50084790e+01  .18566597e-03", 
" -.35719733e+01 -.64135792e+01  .50244115e+01  .18657914e-03", 
" -.35628270e+01 -.63544839e+01  .51373536e+01  .18658224e-03", 
" -.35537176e+01 -.62919760e+01  .51483447e+01  .18661148e-03", 
" -.35445705e+01 -.62366004e+01  .53321409e+01  .18498443e-03", 
" -.35354813e+01 -.61762030e+01  .53359040e+01  .18184104e-03", 
" -.35263561e+01 -.61211235e+01  .55019555e+01  .18318338e-03", 
" -.35173023e+01 -.60598372e+01  .54980085e+01  .18271520e-03", 
" -.35081783e+01 -.60037527e+01  .55514953e+01  .18268987e-03", 
" -.34991356e+01 -.59465533e+01  .56640850e+01  .18287235e-03", 
" -.34900434e+01 -.58949749e+01  .59014684e+01  .17990987e-03", 
" -.34809510e+01 -.58392993e+01  .59431449e+01  .17978581e-03", 
" -.34718775e+01 -.57906882e+01  .63054865e+01  .18076810e-03", 
" -.34628371e+01 -.57292645e+01  .61506039e+01  .18062773e-03", 
" -.34537635e+01 -.56772743e+01  .62513039e+01  .18396770e-03", 
" -.34447068e+01 -.56191115e+01  .61852106e+01  .18263996e-03", 
" -.34356185e+01 -.55664633e+01  .62382491e+01  .18270518e-03", 
" -.34265539e+01 -.55113403e+01  .62603527e+01  .18348326e-03", 
" -.34174757e+01 -.54586120e+01  .63438040e+01  .18190757e-03", 
" -.34083835e+01 -.54052161e+01  .63841359e+01  .18172593e-03", 
" -.33992338e+01 -.53585000e+01  .65038554e+01  .18166147e-03", 
" -.33901110e+01 -.53072845e+01  .65062276e+01  .18303706e-03", 
" -.33809623e+01 -.52600109e+01  .66830190e+01  .18480252e-03", 
" -.33718146e+01 -.52109832e+01  .67464715e+01  .18567287e-03", 
" -.33626315e+01 -.51670770e+01  .69423201e+01  .18482370e-03", 
" -.33534345e+01 -.51167074e+01  .69514167e+01  .18669944e-03", 
" -.33442398e+01 -.50711479e+01  .71192205e+01  .18598352e-03", 
" -.33350285e+01 -.50190525e+01  .70469488e+01  .18890591e-03", 
" -.33257515e+01 -.49760461e+01  .72570777e+01  .18741447e-03", 
" -.33165015e+01 -.49295585e+01  .73505986e+01  .18736632e-03", 
" -.33071850e+01 -.48822642e+01  .73611496e+01  .18770323e-03", 
" -.32978604e+01 -.48437486e+01  .77225389e+01  .18692981e-03", 
" -.32885188e+01 -.48009765e+01  .79109431e+01  .18555116e-03", 
" -.32791222e+01 -.47604553e+01  .81386875e+01  .18450249e-03", 
" -.32696447e+01 -.47196365e+01  .82859522e+01  .18224481e-03", 
" -.32602304e+01 -.46730187e+01  .83465598e+01  .18264797e-03", 
" -.32507234e+01 -.46264564e+01  .82676931e+01  .18197357e-03", 
" -.32411826e+01 -.45839874e+01  .84116156e+01  .18015994e-03", 
" -.32315758e+01 -.45508441e+01  .88305668e+01  .18133043e-03", 
" -.32219456e+01 -.45121987e+01  .89912229e+01  .18355382e-03", 
" -.32122254e+01 -.44817751e+01  .94482607e+01  .18629890e-03", 
" -.32025014e+01 -.44402576e+01  .95513326e+01  .18590812e-03", 
" -.31927111e+01 -.44045212e+01  .97931785e+01  .18739684e-03", 
" -.31828745e+01 -.43608940e+01  .97023845e+01  .18991644e-03", 
" -.31729629e+01 -.43266710e+01  .10019336e+02  .19157700e-03", 
" -.31630575e+01 -.42853200e+01  .10094452e+02  .19096880e-03", 
" -.31530767e+01 -.42383048e+01  .99823977e+01  .19330980e-03", 
" -.31430189e+01 -.41975572e+01  .10013700e+02  .19408687e-03", 
" -.31328650e+01 -.41607827e+01  .10108046e+02  .19459672e-03", 
" -.31226607e+01 -.41214629e+01  .10159517e+02  .19176061e-03", 
" -.31123798e+01 -.40882260e+01  .10425502e+02  .19005486e-03", 
" -.31020524e+01 -.40425549e+01  .10298499e+02  .19288744e-03", 
" -.30916284e+01 -.40029786e+01  .10362585e+02  .19294758e-03", 
" -.30810872e+01 -.39744806e+01  .10673806e+02  .19260535e-03", 
" -.30704549e+01 -.39400350e+01  .10812613e+02  .19357213e-03", 
" -.30597529e+01 -.39082805e+01  .10971370e+02  .19392901e-03", 
" -.30489629e+01 -.38711672e+01  .11030815e+02  .19423444e-03", 
" -.30380266e+01 -.38437240e+01  .11338350e+02  .19456162e-03", 
" -.30270490e+01 -.38046917e+01  .11287632e+02  .19645013e-03", 
" -.30159180e+01 -.37717393e+01  .11357055e+02  .19775173e-03", 
" -.30046880e+01 -.37407323e+01  .11518206e+02  .20068023e-03", 
" -.29933828e+01 -.37096840e+01  .11739124e+02  .20147015e-03", 
" -.29818603e+01 -.36848453e+01  .12079987e+02  .20165501e-03", 
" -.29702532e+01 -.36549495e+01  .12309653e+02  .19983946e-03", 
" -.29584902e+01 -.36259749e+01  .12550350e+02  .20011419e-03", 
" -.29465933e+01 -.35970692e+01  .12729315e+02  .20216221e-03", 
" -.29345161e+01 -.35705645e+01  .12923914e+02  .20755062e-03", 
" -.29223078e+01 -.35428544e+01  .13133457e+02  .20679595e-03", 
" -.29099564e+01 -.35042582e+01  .12884846e+02  .21101280e-03", 
" -.28974000e+01 -.34786908e+01  .13073822e+02  .21047065e-03", 
" -.28846626e+01 -.34509466e+01  .13174569e+02  .20830967e-03", 
" -.28716909e+01 -.34263926e+01  .13351981e+02  .20774633e-03", 
" -.28585513e+01 -.33988509e+01  .13560639e+02  .20676076e-03", 
" -.28452258e+01 -.33674521e+01  .13594105e+02  .21078352e-03", 
" -.28316222e+01 -.33410136e+01  .13707699e+02  .21336685e-03", 
" -.28178068e+01 -.33124444e+01  .13759555e+02  .21247928e-03", 
" -.28036856e+01 -.32877209e+01  .13923368e+02  .21504162e-03", 
" -.27892737e+01 -.32631083e+01  .13910307e+02  .21788446e-03", 
" -.27745282e+01 -.32500978e+01  .14332539e+02  .21961221e-03", 
" -.27595392e+01 -.32188586e+01  .14161431e+02  .22269443e-03", 
" -.27442100e+01 -.31932677e+01  .14136776e+02  .22574360e-03", 
" -.27284620e+01 -.31832281e+01  .14673612e+02  .22957991e-03", 
" -.27122917e+01 -.31822268e+01  .15268165e+02  .22984393e-03", 
" -.26957201e+01 -.31711970e+01  .15583195e+02  .23251927e-03", 
" -.26786378e+01 -.31739617e+01  .16267637e+02  .23139653e-03", 
" -.26610936e+01 -.31778868e+01  .16914427e+02  .22994140e-03", 
" -.26430867e+01 -.31697551e+01  .17147692e+02  .22461490e-03", 
" -.26245097e+01 -.31592432e+01  .17280413e+02  .23345710e-03", 
" -.26052439e+01 -.31582937e+01  .17610529e+02  .23369062e-03", 
" -.25852470e+01 -.31782114e+01  .18528546e+02  .23710813e-03", 
" -.25645215e+01 -.31865723e+01  .19116540e+02  .23670869e-03", 
" -.25429902e+01 -.31966808e+01  .19639880e+02  .24553849e-03", 
" -.25205803e+01 -.32066884e+01  .20126009e+02  .25039844e-03", 
" -.24970704e+01 -.32186144e+01  .20448088e+02  .25896065e-03", 
" -.24722991e+01 -.32484567e+01  .21165532e+02  .26540415e-03", 
" -.24462208e+01 -.32683099e+01  .21348343e+02  .26721693e-03", 
" -.24186384e+01 -.32914236e+01  .21473253e+02  .27438092e-03", 
" -.23892394e+01 -.33510850e+01  .22546942e+02  .28357671e-03", 
" -.23577179e+01 -.34251188e+01  .23760475e+02  .28392479e-03", 
" -.23235374e+01 -.35071633e+01  .24739903e+02  .29811965e-03", 
" -.22865711e+01 -.36014967e+01  .25868615e+02  .30318169e-03", 
" -.22458849e+01 -.37295772e+01  .27617982e+02  .30863265e-03", 
" -.22001106e+01 -.39329280e+01  .30603027e+02  .32396476e-03", 
" -.21484249e+01 -.41812820e+01  .34395485e+02  .34218459e-03", 
" -.20885571e+01 -.44087665e+01  .35939128e+02  .38548913e-03", 
" -.20163489e+01 -.47865152e+01  .39737298e+02  .43649307e-03", 
" -.19246542e+01 -.53505625e+01  .45091445e+02  .48041861e-03", 
" -.17976961e+01 -.62612735e+01  .53848370e+02  .58167814e-03", 
" -.17650820e+01 -.65336342e+01  .56785313e+02  .60964048e-03", 
" -.17288010e+01 -.68093748e+01  .58860032e+02  .62526838e-03", 
" -.16878091e+01 -.71142227e+01  .60863598e+02  .68446264e-03", 
" -.16409551e+01 -.74799066e+01  .63258088e+02  .70153378e-03", 
" -.15857416e+01 -.79531163e+01  .67006679e+02  .77450941e-03", 
" -.15182005e+01 -.86209509e+01  .73683075e+02  .86982561e-03", 
" -.14323254e+01 -.94477897e+01  .81068164e+02  .98013687e-03", 
" -.13131855e+01 -.10561841e+02  .87665609e+02  .11763621e-02", 
" -.11138832e+01 -.12538227e+02  .97389973e+02  .16780143e-02", 
" -.91974001e+00 -.14982401e+02  .11933934e+03  .22314056e-02", 
" -.67491819e+00 -.18096211e+02  .14038309e+03  .34531170e-02", 
" -.49711256e+00 -.20579563e+02  .15653587e+03  .42951260e-02", 
"coanc     3  4  3   20", " -.60144689e+02  .10400353e+04 -.12197475e+05  .70077208e+05  .81668833e-01", 
" -.56816076e+02  .93186598e+03 -.10393043e+05  .57321251e+05  .60282104e-01", 
" -.52327597e+02  .79537859e+03 -.82315745e+04  .42325243e+05  .37628975e-01", 
" -.48894746e+02  .69825373e+03 -.67945227e+04  .33118495e+05  .27633167e-01", 
" -.45456329e+02  .61449076e+03 -.58291720e+04  .28544952e+05  .20077721e-01", 
" -.43419133e+02  .56813622e+03 -.53525288e+04  .26782047e+05  .16941710e-01", 
" -.41961964e+02  .53380049e+03 -.49141455e+04  .24204395e+05  .14812544e-01", 
" -.40806902e+02  .50583437e+03 -.45157587e+04  .21461036e+05  .13326244e-01", 
" -.39869172e+02  .48586239e+03 -.43128025e+04  .20730480e+05  .12540779e-01", 
" -.39074723e+02  .46927738e+03 -.41432206e+04  .20014354e+05  .11384809e-01", 
" -.38374288e+02  .45324404e+03 -.39229932e+04  .18563862e+05  .10846895e-01", 
" -.37754541e+02  .44023877e+03 -.37801666e+04  .17894151e+05  .10216649e-01", 
" -.37201779e+02  .42973417e+03 -.36997221e+04  .17815227e+05  .96336558e-02", 
" -.35039120e+02  .38473577e+03 -.31406065e+04  .14315891e+05  .79786196e-02", 
" -.33486165e+02  .35590775e+03 -.28728087e+04  .13290506e+05  .69551435e-02", 
" -.32256431e+02  .33179858e+03 -.25781905e+04  .11399675e+05  .64046825e-02", 
" -.31244389e+02  .31354493e+03 -.23931695e+04  .10484067e+05  .56634427e-02", 
" -.30382453e+02  .29857345e+03 -.22528424e+04  .98755387e+04  .55364007e-02", 
" -.29627741e+02  .28542729e+03 -.21192933e+04  .91969660e+04  .50080393e-02", 
" -.28958705e+02  .27444621e+03 -.20204235e+04  .87737958e+04  .48521382e-02", 
" -.28351424e+02  .26408500e+03 -.19107608e+04  .81730379e+04  .45909504e-02", 
" -.27799384e+02  .25499505e+03 -.18238116e+04  .77681888e+04  .44848711e-02", 
" -.27292641e+02  .24694938e+03 -.17488537e+04  .74095011e+04  .43141205e-02", 
" -.26821470e+02  .23924229e+03 -.16665036e+04  .69257606e+04  .41256841e-02", 
" -.26383143e+02  .23258804e+03 -.16145908e+04  .67874516e+04  .40622468e-02", 
" -.25970682e+02  .22607880e+03 -.15490858e+04  .64059565e+04  .39686133e-02", 
" -.25584772e+02  .22038995e+03 -.15041717e+04  .62626598e+04  .37962520e-02", 
" -.25218518e+02  .21473238e+03 -.14463068e+04  .59105859e+04  .36416701e-02", 
" -.24870209e+02  .20937625e+03 -.13905932e+04  .55625180e+04  .35936310e-02", 
" -.24540679e+02  .20450560e+03 -.13443412e+04  .53037507e+04  .34293594e-02", 
" -.24224640e+02  .19968813e+03 -.12940209e+04  .50026200e+04  .33097225e-02", 
" -.23923613e+02  .19545749e+03 -.12626747e+04  .49174889e+04  .32591909e-02", 
" -.23634507e+02  .19123709e+03 -.12225540e+04  .47049769e+04  .32155070e-02", 
" -.23356448e+02  .18721903e+03 -.11833249e+04  .44680381e+04  .31041613e-02", 
" -.23089154e+02  .18347033e+03 -.11517351e+04  .43396422e+04  .30853140e-02", 
" -.22831889e+02  .17998391e+03 -.11268100e+04  .42785623e+04  .30835132e-02", 
" -.22582756e+02  .17649920e+03 -.10956715e+04  .41365975e+04  .30454555e-02", 
" -.22342887e+02  .17326687e+03 -.10707571e+04  .40478294e+04  .30660504e-02", 
" -.22109227e+02  .16997999e+03 -.10385271e+04  .38533657e+04  .30308391e-02", 
" -.21883818e+02  .16691470e+03 -.10109702e+04  .37075868e+04  .30231993e-02", 
" -.21665267e+02  .16390396e+03 -.97847943e+03  .34619144e+04  .29952929e-02", 
" -.21453126e+02  .16128039e+03 -.96288088e+03  .34453070e+04  .29755311e-02", 
" -.21245740e+02  .15852231e+03 -.93826459e+03  .33181373e+04  .29278690e-02", 
" -.21045244e+02  .15621937e+03 -.93339842e+03  .34429096e+04  .28671255e-02", 
" -.20847857e+02  .15354127e+03 -.90622908e+03  .32652401e+04  .28436493e-02", 
" -.20655745e+02  .15100133e+03 -.88256044e+03  .31325769e+04  .27918144e-02", 
" -.20469223e+02  .14882299e+03 -.87303856e+03  .31711290e+04  .27839584e-02", 
" -.20285441e+02  .14644177e+03 -.85231317e+03  .30700986e+04  .27299533e-02", 
" -.20105882e+02  .14410941e+03 -.83070490e+03  .29476389e+04  .26743109e-02", 
" -.19931381e+02  .14205769e+03 -.81853676e+03  .29274214e+04  .26877793e-02", 
" -.19759750e+02  .13988993e+03 -.79876757e+03  .28061103e+04  .26792092e-02", 
" -.19591396e+02  .13778529e+03 -.78093094e+03  .27161343e+04  .26399797e-02", 
" -.19426190e+02  .13569348e+03 -.76159068e+03  .26057494e+04  .26032640e-02", 
" -.19263896e+02  .13360478e+03 -.74141639e+03  .24914836e+04  .25608247e-02", 
" -.19104929e+02  .13178714e+03 -.73257837e+03  .25080203e+04  .24903768e-02", 
" -.18948579e+02  .12988897e+03 -.71764675e+03  .24458678e+04  .24822904e-02", 
" -.18796146e+02  .12814922e+03 -.70625022e+03  .24085944e+04  .24402542e-02", 
" -.18645507e+02  .12641047e+03 -.69551589e+03  .23897125e+04  .24436697e-02", 
" -.18496839e+02  .12452777e+03 -.67602576e+03  .22590765e+04  .24443018e-02", 
" -.18351943e+02  .12298487e+03 -.66947117e+03  .22735899e+04  .24248768e-02", 
" -.18209393e+02  .12142118e+03 -.65910397e+03  .22237519e+04  .24119445e-02", 
" -.18068009e+02  .11977881e+03 -.64681449e+03  .21767286e+04  .23900391e-02", 
" -.17928908e+02  .11817514e+03 -.63428845e+03  .21143386e+04  .23403161e-02", 
" -.17791898e+02  .11660354e+03 -.62166384e+03  .20500541e+04  .23319107e-02", 
" -.17656506e+02  .11496650e+03 -.60553102e+03  .19429971e+04  .23191888e-02", 
" -.17523171e+02  .11341310e+03 -.59244832e+03  .18737943e+04  .23192593e-02", 
" -.17391225e+02  .11187169e+03 -.58028358e+03  .18300647e+04  .23201153e-02", 
" -.17261977e+02  .11054850e+03 -.57590738e+03  .18667497e+04  .23355152e-02", 
" -.17134150e+02  .10913649e+03 -.56600442e+03  .18316211e+04  .23304797e-02", 
" -.17007607e+02  .10766888e+03 -.55205981e+03  .17388991e+04  .23251231e-02", 
" -.16882574e+02  .10630542e+03 -.54277330e+03  .17063281e+04  .23337082e-02", 
" -.16759382e+02  .10498359e+03 -.53333724e+03  .16630424e+04  .22946379e-02", 
" -.16638421e+02  .10374208e+03 -.52547806e+03  .16280140e+04  .23086826e-02", 
" -.16517662e+02  .10235035e+03 -.51153223e+03  .15298661e+04  .23108534e-02", 
" -.16398369e+02  .10106089e+03 -.50285016e+03  .15052025e+04  .22985181e-02", 
" -.16280539e+02  .99772727e+02 -.49325456e+03  .14709024e+04  .22807826e-02", 
" -.16163861e+02  .98460679e+02 -.48033320e+03  .13774046e+04  .22940121e-02", 
" -.16048829e+02  .97181618e+02 -.46740596e+03  .12818700e+04  .22904284e-02", 
" -.15934834e+02  .96038528e+02 -.46158778e+03  .12756381e+04  .22756486e-02", 
" -.15822370e+02  .94951116e+02 -.45826094e+03  .13113491e+04  .22220335e-02", 
" -.15710876e+02  .93866785e+02 -.45449004e+03  .13379631e+04  .21964894e-02", 
" -.15600288e+02  .92715086e+02 -.44589347e+03  .12978896e+04  .21696559e-02", 
" -.15490819e+02  .91650939e+02 -.43964672e+03  .12727343e+04  .21861395e-02", 
" -.15382020e+02  .90530548e+02 -.43120715e+03  .12276227e+04  .21701918e-02", 
" -.15274327e+02  .89453652e+02 -.42617180e+03  .12400785e+04  .21868797e-02", 
" -.15167698e+02  .88467814e+02 -.42477662e+03  .12988395e+04  .21698562e-02", 
" -.15061495e+02  .87390184e+02 -.41786982e+03  .12768933e+04  .21671923e-02", 
" -.14957146e+02  .86414051e+02 -.41277173e+03  .12612183e+04  .21632862e-02", 
" -.14852796e+02  .85355023e+02 -.40503736e+03  .12243580e+04  .21722281e-02", 
" -.14749382e+02  .84294474e+02 -.39548793e+03  .11523722e+04  .21506073e-02", 
" -.14646995e+02  .83340312e+02 -.39155138e+03  .11616144e+04  .21566023e-02", 
" -.14545205e+02  .82305794e+02 -.38386067e+03  .11328780e+04  .21617252e-02", 
" -.14444112e+02  .81289499e+02 -.37580261e+03  .10899818e+04  .21541902e-02", 
" -.14344138e+02  .80346781e+02 -.37003855e+03  .10691216e+04  .21603509e-02", 
" -.14244479e+02  .79373991e+02 -.36349182e+03  .10437680e+04  .21551658e-02", 
" -.14145671e+02  .78376913e+02 -.35466973e+03  .98517370e+03  .21411543e-02", 
" -.14047635e+02  .77448709e+02 -.34920127e+03  .97376036e+03  .21320072e-02", 
" -.13949982e+02  .76529353e+02 -.34378128e+03  .96063071e+03  .21173583e-02", 
" -.13852872e+02  .75562249e+02 -.33491387e+03  .89549419e+03  .21104001e-02", 
" -.13756940e+02  .74740813e+02 -.33306111e+03  .92284599e+03  .21156363e-02", 
" -.13661190e+02  .73879952e+02 -.32866159e+03  .91311234e+03  .20948576e-02", 
" -.13565576e+02  .72925505e+02 -.32016965e+03  .85810156e+03  .20866564e-02", 
" -.13471038e+02  .72064246e+02 -.31516645e+03  .84286661e+03  .20794104e-02", 
" -.13376525e+02  .71126691e+02 -.30641339e+03  .78149941e+03  .20436488e-02", 
" -.13283159e+02  .70282060e+02 -.30109321e+03  .75928696e+03  .20253640e-02", 
" -.13189988e+02  .69408906e+02 -.29473316e+03  .73028761e+03  .20138900e-02", 
" -.13097331e+02  .68555885e+02 -.28946554e+03  .71738335e+03  .20346223e-02", 
" -.13005027e+02  .67716319e+02 -.28474888e+03  .70928825e+03  .20177681e-02", 
" -.12912896e+02  .66841945e+02 -.27829702e+03  .67803582e+03  .20175239e-02", 
" -.12821504e+02  .66015721e+02 -.27278275e+03  .65157797e+03  .19924247e-02", 
" -.12730394e+02  .65184798e+02 -.26637755e+03  .60762605e+03  .19755361e-02", 
" -.12640161e+02  .64414889e+02 -.26253118e+03  .59951041e+03  .19822611e-02", 
" -.12550225e+02  .63693923e+02 -.26122749e+03  .62230068e+03  .19867028e-02", 
" -.12460488e+02  .62882091e+02 -.25526042e+03  .58993980e+03  .19524492e-02", 
" -.12371449e+02  .62164850e+02 -.25303095e+03  .60185953e+03  .19167181e-02", 
" -.12282225e+02  .61273550e+02 -.24257010e+03  .51453702e+03  .19004885e-02", 
" -.12193693e+02  .60570556e+02 -.24095578e+03  .53390552e+03  .19002497e-02", 
" -.12105581e+02  .59847281e+02 -.23725169e+03  .51891770e+03  .18838155e-02", 
" -.12017193e+02  .59054427e+02 -.23114909e+03  .48238932e+03  .18829381e-02", 
" -.11929580e+02  .58338586e+02 -.22808974e+03  .48238678e+03  .18939277e-02", 
" -.11842072e+02  .57584237e+02 -.22303003e+03  .45682879e+03  .18727223e-02", 
" -.11755004e+02  .56875716e+02 -.21997876e+03  .46180683e+03  .18705717e-02", 
" -.11668116e+02  .56115370e+02 -.21369526e+03  .42079574e+03  .18569147e-02", 
" -.11580894e+02  .55315140e+02 -.20631731e+03  .36893134e+03  .18537838e-02", 
" -.11493925e+02  .54509473e+02 -.19796874e+03  .30153379e+03  .18532677e-02", 
" -.11407750e+02  .53777182e+02 -.19169369e+03  .25057415e+03  .18324852e-02", 
" -.11321421e+02  .53060740e+02 -.18780460e+03  .24223382e+03  .18190825e-02", 
" -.11235570e+02  .52398901e+02 -.18534838e+03  .24308312e+03  .17899472e-02", 
" -.11149729e+02  .51654132e+02 -.17871977e+03  .19475429e+03  .17729878e-02", 
" -.11064306e+02  .50974221e+02 -.17475865e+03  .17963570e+03  .17634667e-02", 
" -.10979073e+02  .50310004e+02 -.17076041e+03  .15573702e+03  .17569093e-02", 
" -.10893620e+02  .49688867e+02 -.17059967e+03  .18955931e+03  .17549752e-02", 
" -.10808433e+02  .49005972e+02 -.16586570e+03  .15995768e+03  .17471138e-02", 
" -.10723155e+02  .48276729e+02 -.15881175e+03  .10176676e+03  .17248431e-02", 
" -.10638583e+02  .47697480e+02 -.15842558e+03  .12222644e+03  .17233276e-02", 
" -.10553589e+02  .47052977e+02 -.15637660e+03  .13718077e+03  .17360480e-02", 
" -.10468787e+02  .46461784e+02 -.15588609e+03  .15992536e+03  .17294407e-02", 
" -.10383881e+02  .45827018e+02 -.15372374e+03  .16776459e+03  .17288384e-02", 
" -.10298805e+02  .45149865e+02 -.14901524e+03  .14257498e+03  .17150817e-02", 
" -.10213732e+02  .44456086e+02 -.14345386e+03  .10503304e+03  .17012047e-02", 
" -.10128878e+02  .43793677e+02 -.13919217e+03  .85988851e+02  .17174834e-02", 
" -.10044032e+02  .43168480e+02 -.13662571e+03  .87422971e+02  .17085190e-02", 
" -.99591331e+01  .42555791e+02 -.13464679e+03  .97245705e+02  .17184440e-02", 
" -.98741553e+01  .41888802e+02 -.12933820e+03  .62161182e+02  .17137602e-02", 
" -.97889493e+01  .41248304e+02 -.12604599e+03  .54567714e+02  .17116631e-02", 
" -.97040951e+01  .40692846e+02 -.12626910e+03  .89232770e+02  .16962677e-02", 
" -.96187704e+01  .40060228e+02 -.12231945e+03  .63391801e+02  .16651674e-02", 
" -.95337507e+01  .39481662e+02 -.12089683e+03  .73731637e+02  .16618481e-02", 
" -.94479293e+01  .38830001e+02 -.11715887e+03  .63569044e+02  .16568516e-02", 
" -.93624705e+01  .38228737e+02 -.11434150e+03  .51904204e+02  .16457227e-02", 
" -.92765123e+01  .37577786e+02 -.10984594e+03  .29325072e+02  .16373851e-02", 
" -.91909140e+01  .36966914e+02 -.10560584e+03 -.33864742e+00  .16243805e-02", 
" -.91047249e+01  .36309698e+02 -.99944486e+02 -.46750286e+02  .16257949e-02", 
" -.90186070e+01  .35722021e+02 -.97507993e+02 -.49420478e+02  .16426755e-02", 
" -.89320461e+01  .35105444e+02 -.93663458e+02 -.71597863e+02  .16388358e-02", 
" -.88445642e+01  .34402237e+02 -.86321566e+02 -.13535467e+03  .16480734e-02", 
" -.87575543e+01  .33846833e+02 -.85881777e+02 -.11251182e+03  .16300449e-02", 
" -.86701177e+01  .33193060e+02 -.80111217e+02 -.15601704e+03  .16340701e-02", 
" -.85823853e+01  .32619161e+02 -.78331177e+02 -.15088735e+03  .16345434e-02", 
" -.84944071e+01  .32076915e+02 -.78067469e+02 -.12831544e+03  .16237758e-02", 
" -.84059819e+01  .31499430e+02 -.75750448e+02 -.13503783e+03  .16422674e-02", 
" -.83172240e+01  .30903050e+02 -.73569964e+02 -.12212238e+03  .16261928e-02", 
" -.82280268e+01  .30272018e+02 -.68548562e+02 -.15734547e+03  .16161780e-02", 
" -.81381425e+01  .29645954e+02 -.63900763e+02 -.19195245e+03  .16205833e-02", 
" -.80480347e+01  .29048484e+02 -.59970812e+02 -.22119583e+03  .15942154e-02", 
" -.79567612e+01  .28398692e+02 -.55229064e+02 -.25145161e+03  .15949660e-02", 
" -.78657654e+01  .27834968e+02 -.53278938e+02 -.25017129e+03  .15779312e-02", 
" -.77741922e+01  .27302533e+02 -.53974961e+02 -.21106572e+03  .15864673e-02", 
" -.76819208e+01  .26744168e+02 -.53315148e+02 -.18378366e+03  .15776358e-02", 
" -.75884972e+01  .26120750e+02 -.49357527e+02 -.20466936e+03  .15663602e-02", 
" -.74942643e+01  .25448590e+02 -.42902689e+02 -.26084443e+03  .15613985e-02", 
" -.73996848e+01  .24874721e+02 -.41231479e+02 -.25151787e+03  .15447932e-02", 
" -.73041349e+01  .24300778e+02 -.39419827e+02 -.25173064e+03  .15514933e-02", 
" -.72073789e+01  .23624522e+02 -.33617272e+02 -.29214558e+03  .15261781e-02", 
" -.71106223e+01  .23101819e+02 -.33988168e+02 -.26638985e+03  .15227962e-02", 
" -.70123506e+01  .22484785e+02 -.30950512e+02 -.27413273e+03  .15069509e-02", 
" -.69129883e+01  .21889681e+02 -.29751861e+02 -.25019182e+03  .15201093e-02", 
" -.68130413e+01  .21337572e+02 -.28656641e+02 -.24194617e+03  .15333495e-02", 
" -.67112238e+01  .20674277e+02 -.25077616e+02 -.23841197e+03  .15498077e-02", 
" -.66082434e+01  .20061907e+02 -.22503765e+02 -.24164631e+03  .15572344e-02", 
" -.65036219e+01  .19374449e+02 -.16394511e+02 -.29441731e+03  .15492826e-02", 
" -.63981605e+01  .18780404e+02 -.13785273e+02 -.30673070e+03  .15322550e-02", 
" -.62910315e+01  .18237356e+02 -.15335231e+02 -.25875483e+03  .15275659e-02", 
" -.61825216e+01  .17655896e+02 -.14728609e+02 -.23262681e+03  .15234486e-02", 
" -.60719406e+01  .17030537e+02 -.11820587e+02 -.24071789e+03  .15278704e-02", 
" -.59594994e+01  .16404395e+02 -.86106298e+01 -.26608890e+03  .15081928e-02", 
" -.58448828e+01  .15796215e+02 -.86022175e+01 -.22668127e+03  .15116591e-02", 
" -.57283985e+01  .15218579e+02 -.82722653e+01 -.20599000e+03  .15147983e-02", 
" -.56087721e+01  .14608955e+02 -.87104045e+01 -.16825426e+03  .15316177e-02", 
" -.54866786e+01  .13997305e+02 -.84799838e+01 -.14230311e+03  .15147560e-02", 
" -.53618995e+01  .13355577e+02 -.59413715e+01 -.14823530e+03  .14846231e-02", 
" -.52333771e+01  .12638226e+02 -.75600600e+00 -.19054911e+03  .14814476e-02", 
" -.51011113e+01  .11874315e+02  .44932088e+01 -.21661724e+03  .14922733e-02", 
" -.49658194e+01  .11352440e+02 -.16465791e+01 -.10512562e+03  .15036489e-02", 
" -.48262134e+01  .10639196e+02  .36156622e+01 -.15654489e+03  .15375513e-02", 
" -.46814338e+01  .98909909e+01  .91851711e+01 -.21141280e+03  .15560840e-02", 
" -.45307337e+01  .91472643e+01  .11904735e+02 -.21428631e+03  .15847463e-02", 
" -.43745218e+01  .83956021e+01  .16571820e+02 -.25947577e+03  .15852438e-02", 
" -.42104926e+01  .76727608e+01  .18910011e+02 -.27590102e+03  .16242605e-02", 
" -.40381656e+01  .68238337e+01  .26004000e+02 -.34157029e+03  .16666011e-02", 
" -.38566752e+01  .61222285e+01  .25394859e+02 -.31211938e+03  .17104643e-02", 
" -.36630499e+01  .53357161e+01  .27030498e+02 -.32217604e+03  .17350204e-02", 
" -.34546108e+01  .46551561e+01  .17595964e+02 -.15214852e+03  .17630655e-02", 
" -.32291346e+01  .39238586e+01  .12657855e+02 -.80866325e+02  .18178099e-02", 
" -.29793519e+01  .29677219e+01  .13225633e+02 -.54265597e+02  .18902015e-02", 
" -.26995037e+01  .18776556e+01  .18630808e+02 -.94267109e+02  .19620840e-02", 
" -.23746985e+01  .82808195e+00  .15558044e+02 -.33677844e+02  .20552704e-02", 
" -.19816911e+01 -.38022806e+00  .16607205e+02 -.32128107e+02  .22583138e-02", 
" -.14627207e+01 -.21753774e+01  .28307624e+02 -.12167570e+03  .25387268e-02", 
" -.13341824e+01 -.26542850e+01  .35816179e+02 -.20693859e+03  .25926648e-02", 
" -.11937829e+01 -.27407909e+01  .26363594e+02 -.11037899e+03  .27249092e-02", 
" -.10362181e+01 -.32236995e+01  .34072434e+02 -.22132297e+03  .28758281e-02", 
" -.85870326e+00 -.34232497e+01  .26259141e+02 -.11372000e+03  .30436955e-02", 
" -.65452876e+00 -.38398751e+01  .33802382e+02 -.22980327e+03  .31708059e-02", 
" -.41061850e+00 -.39366330e+01  .24434633e+02 -.11652017e+03  .35741029e-02", 
" -.10790086e+00 -.38828215e+01  .19868373e+02 -.10899760e+03  .38986713e-02", 
"  .29681940e+00 -.29210044e+01 -.24780405e+02  .44387254e+03  .47815245e-02", 
"  .94092051e+00 -.31532769e+01  .33548162e+02 -.46680549e+03  .60529108e-02", 
"  .15210407e+01 -.81993851e+00 -.10764194e+02  .17418080e+02  .73218329e-02", 
"  .22406412e+01  .43672577e+00  .55363325e+02 -.89020116e+03  .10699757e-01", 
"  .27370421e+01  .40552300e+01 -.38030523e+01  .41826904e+00  .14865356e-01", 
"coac      3  5  3   20", " -.65187467e+02  .11770939e+04 -.14022299e+05  .80965435e+05  .84951608e-01", 
" -.61813633e+02  .10630601e+04 -.12032433e+05  .65731504e+05  .61115523e-01", 
" -.57299566e+02  .92967251e+03 -.10357895e+05  .58391777e+05  .38521395e-01", 
" -.53743243e+02  .81707235e+03 -.84438945e+04  .44051422e+05  .28759277e-01", 
" -.50211974e+02  .72214621e+03 -.72243532e+04  .37612701e+05  .19746004e-01", 
" -.48088621e+02  .66607210e+03 -.64739297e+04  .33256971e+05  .17148463e-01", 
" -.46575714e+02  .62955404e+03 -.60812777e+04  .31783275e+05  .14904983e-01", 
" -.45390908e+02  .60081868e+03 -.57370854e+04  .29986899e+05  .13554471e-01", 
" -.44426879e+02  .57833472e+03 -.54633013e+04  .28409148e+05  .12456496e-01", 
" -.43592410e+02  .55798123e+03 -.51950958e+04  .26746442e+05  .11622242e-01", 
" -.42873136e+02  .54221844e+03 -.50343156e+04  .26114224e+05  .10788150e-01", 
" -.42226654e+02  .52669864e+03 -.48212182e+04  .24660062e+05  .10330009e-01", 
" -.41650430e+02  .51390945e+03 -.46831066e+04  .24042430e+05  .10032824e-01", 
" -.39397982e+02  .46301968e+03 -.40406777e+04  .20093660e+05  .88492548e-02", 
" -.37783094e+02  .42829366e+03 -.36217064e+04  .17617178e+05  .75837689e-02", 
" -.36509350e+02  .40219326e+03 -.33249433e+04  .15939007e+05  .68086198e-02", 
" -.35452921e+02  .38094082e+03 -.30826742e+04  .14557092e+05  .62075968e-02", 
" -.34546935e+02  .36309011e+03 -.28829615e+04  .13404938e+05  .57998246e-02", 
" -.33755340e+02  .34804204e+03 -.27258387e+04  .12605056e+05  .54835613e-02", 
" -.33048515e+02  .33439501e+03 -.25695881e+04  .11688451e+05  .50596614e-02", 
" -.32414260e+02  .32257651e+03 -.24395217e+04  .10945436e+05  .47976313e-02", 
" -.31834497e+02  .31199461e+03 -.23246505e+04  .10256221e+05  .45481075e-02", 
" -.31299958e+02  .30206833e+03 -.22047160e+04  .94437427e+04  .43581924e-02", 
" -.30807592e+02  .29376240e+03 -.21342236e+04  .91811309e+04  .42184439e-02", 
" -.30346056e+02  .28573040e+03 -.20563695e+04  .88209782e+04  .39867263e-02", 
" -.29912803e+02  .27826008e+03 -.19783288e+04  .83595058e+04  .39523937e-02", 
" -.29502446e+02  .27096267e+03 -.18997935e+04  .79375529e+04  .38810788e-02", 
" -.29118780e+02  .26484609e+03 -.18520290e+04  .77974997e+04  .37093687e-02", 
" -.28752284e+02  .25878841e+03 -.17980352e+04  .75883284e+04  .36806725e-02", 
" -.28404636e+02  .25320586e+03 -.17502676e+04  .74033598e+04  .35726898e-02", 
" -.28071302e+02  .24744400e+03 -.16826170e+04  .69537252e+04  .35487139e-02", 
" -.27754392e+02  .24249129e+03 -.16393082e+04  .67578145e+04  .35293157e-02", 
" -.27448335e+02  .23750689e+03 -.15896230e+04  .64769673e+04  .34872410e-02", 
" -.27154482e+02  .23283492e+03 -.15452885e+04  .62457890e+04  .34720973e-02", 
" -.26874081e+02  .22879512e+03 -.15215861e+04  .62518058e+04  .34521559e-02", 
" -.26601281e+02  .22456719e+03 -.14836349e+04  .60730490e+04  .33736240e-02", 
" -.26338134e+02  .22049387e+03 -.14440021e+04  .58553547e+04  .33271894e-02", 
" -.26082392e+02  .21646570e+03 -.14037645e+04  .56359251e+04  .32752363e-02", 
" -.25834391e+02  .21246455e+03 -.13590640e+04  .53498794e+04  .31834279e-02", 
" -.25594805e+02  .20886863e+03 -.13252923e+04  .51552981e+04  .31500387e-02", 
" -.25362606e+02  .20541242e+03 -.12936697e+04  .49860076e+04  .30642812e-02", 
" -.25136558e+02  .20207430e+03 -.12637095e+04  .48353901e+04  .29764343e-02", 
" -.24916401e+02  .19895471e+03 -.12418974e+04  .47897926e+04  .29032079e-02", 
" -.24702612e+02  .19594178e+03 -.12199398e+04  .47316033e+04  .27990640e-02", 
" -.24493299e+02  .19304787e+03 -.12027439e+04  .47316077e+04  .27863370e-02", 
" -.24288938e+02  .19008391e+03 -.11755903e+04  .45813718e+04  .27859800e-02", 
" -.24089320e+02  .18720114e+03 -.11500250e+04  .44579789e+04  .27020430e-02", 
" -.23894389e+02  .18446492e+03 -.11264109e+04  .43330874e+04  .27201168e-02", 
" -.23703197e+02  .18172147e+03 -.11017403e+04  .42093015e+04  .27198826e-02", 
" -.23516009e+02  .17894536e+03 -.10715366e+04  .40037626e+04  .27092563e-02", 
" -.23333628e+02  .17652317e+03 -.10573503e+04  .40034799e+04  .26994950e-02", 
" -.23154477e+02  .17404273e+03 -.10364265e+04  .39157910e+04  .26985974e-02", 
" -.22977785e+02  .17153670e+03 -.10118791e+04  .37604643e+04  .26745537e-02", 
" -.22805106e+02  .16918127e+03 -.99344622e+03  .36900101e+04  .26310489e-02", 
" -.22636046e+02  .16692701e+03 -.97633442e+03  .36250773e+04  .26174125e-02", 
" -.22470084e+02  .16474689e+03 -.96023017e+03  .35664050e+04  .26139138e-02", 
" -.22306776e+02  .16259428e+03 -.94509146e+03  .35277475e+04  .25870574e-02", 
" -.22145623e+02  .16028916e+03 -.91986063e+03  .33560259e+04  .25627266e-02", 
" -.21987485e+02  .15821199e+03 -.90485324e+03  .33150451e+04  .25657402e-02", 
" -.21832343e+02  .15630147e+03 -.89477776e+03  .33166661e+04  .25315021e-02", 
" -.21679584e+02  .15432384e+03 -.87824596e+03  .32266808e+04  .25354234e-02", 
" -.21527987e+02  .15232144e+03 -.86201328e+03  .31553520e+04  .25040488e-02", 
" -.21379471e+02  .15035106e+03 -.84383963e+03  .30495237e+04  .24673377e-02", 
" -.21233299e+02  .14853680e+03 -.83245465e+03  .30322155e+04  .24429100e-02", 
" -.21088615e+02  .14663609e+03 -.81500290e+03  .29312669e+04  .24169187e-02", 
" -.20946743e+02  .14494980e+03 -.80638626e+03  .29364763e+04  .23946056e-02", 
" -.20806547e+02  .14327952e+03 -.79635893e+03  .29170283e+04  .23984544e-02", 
" -.20667108e+02  .14152876e+03 -.78394256e+03  .28805783e+04  .24420005e-02", 
" -.20529745e+02  .13976817e+03 -.76857965e+03  .27945850e+04  .24287987e-02", 
" -.20394359e+02  .13804381e+03 -.75216742e+03  .26827598e+04  .23752257e-02", 
" -.20260593e+02  .13642464e+03 -.74210144e+03  .26732689e+04  .23547329e-02", 
" -.20127964e+02  .13473977e+03 -.72771111e+03  .26040808e+04  .23415810e-02", 
" -.19996479e+02  .13304538e+03 -.71196401e+03  .25110340e+04  .23081158e-02", 
" -.19867292e+02  .13145885e+03 -.69926056e+03  .24478219e+04  .22994207e-02", 
" -.19740107e+02  .13004507e+03 -.69353534e+03  .24705611e+04  .22590748e-02", 
" -.19613724e+02  .12860227e+03 -.68471949e+03  .24395109e+04  .22676808e-02", 
" -.19488762e+02  .12710825e+03 -.67295846e+03  .23806157e+04  .22656808e-02", 
" -.19364831e+02  .12570953e+03 -.66697201e+03  .24043946e+04  .22685673e-02", 
" -.19242406e+02  .12429899e+03 -.65739401e+03  .23617694e+04  .22286468e-02", 
" -.19120248e+02  .12276724e+03 -.64220672e+03  .22556830e+04  .22245170e-02", 
" -.18999968e+02  .12139605e+03 -.63426892e+03  .22496801e+04  .22044325e-02", 
" -.18880653e+02  .11990035e+03 -.61887926e+03  .21403246e+04  .21959030e-02", 
" -.18761947e+02  .11844464e+03 -.60554278e+03  .20605607e+04  .21866960e-02", 
" -.18644772e+02  .11706369e+03 -.59498906e+03  .20121634e+04  .21743657e-02", 
" -.18528386e+02  .11572837e+03 -.58545353e+03  .19679479e+04  .21701366e-02", 
" -.18413196e+02  .11437380e+03 -.57419517e+03  .19075677e+04  .21577337e-02", 
" -.18298796e+02  .11301534e+03 -.56202634e+03  .18332842e+04  .21241959e-02", 
" -.18186189e+02  .11185105e+03 -.55792024e+03  .18541205e+04  .21166872e-02", 
" -.18074136e+02  .11065848e+03 -.55279511e+03  .18678652e+04  .20787036e-02", 
" -.17962456e+02  .10944263e+03 -.54581199e+03  .18507188e+04  .20712127e-02", 
" -.17851368e+02  .10823308e+03 -.53937460e+03  .18489704e+04  .20670285e-02", 
" -.17741357e+02  .10702847e+03 -.53140045e+03  .18220518e+04  .20556362e-02", 
" -.17631808e+02  .10579389e+03 -.52260398e+03  .17862099e+04  .20421031e-02", 
" -.17523775e+02  .10462130e+03 -.51298873e+03  .17168082e+04  .20716581e-02", 
" -.17416585e+02  .10351893e+03 -.50773426e+03  .17145032e+04  .20781662e-02", 
" -.17309257e+02  .10226290e+03 -.49541656e+03  .16330216e+04  .20647374e-02", 
" -.17203001e+02  .10110923e+03 -.48780105e+03  .16140358e+04  .20476706e-02", 
" -.17096584e+02  .99802602e+02 -.47183643e+03  .14790523e+04  .20536465e-02", 
" -.16991975e+02  .98729361e+02 -.46598878e+03  .14704524e+04  .20528479e-02", 
" -.16887643e+02  .97570670e+02 -.45503046e+03  .13922037e+04  .20492117e-02", 
" -.16783511e+02  .96455742e+02 -.44786090e+03  .13683172e+04  .20664325e-02", 
" -.16680303e+02  .95361537e+02 -.44014034e+03  .13324855e+04  .20604837e-02", 
" -.16577606e+02  .94307840e+02 -.43473565e+03  .13311255e+04  .20441835e-02", 
" -.16475173e+02  .93192246e+02 -.42596625e+03  .12868155e+04  .20438756e-02", 
" -.16373284e+02  .92106483e+02 -.41762491e+03  .12396151e+04  .20298853e-02", 
" -.16271614e+02  .90955843e+02 -.40568822e+03  .11459870e+04  .20046753e-02", 
" -.16170566e+02  .89884364e+02 -.39833438e+03  .11211363e+04  .20152208e-02", 
" -.16070005e+02  .88842685e+02 -.39219195e+03  .11142899e+04  .20026506e-02", 
" -.15969975e+02  .87877549e+02 -.38863796e+03  .11279750e+04  .20143756e-02", 
" -.15871554e+02  .86961116e+02 -.38466781e+03  .11239160e+04  .20298514e-02", 
" -.15772281e+02  .85917897e+02 -.37645092e+03  .10782898e+04  .20246162e-02", 
" -.15673701e+02  .84906832e+02 -.36866011e+03  .10339681e+04  .20231143e-02", 
" -.15575572e+02  .83927963e+02 -.36262747e+03  .10145398e+04  .20229516e-02", 
" -.15477530e+02  .82963096e+02 -.35747871e+03  .10072331e+04  .20173463e-02", 
" -.15380043e+02  .82003283e+02 -.35078291e+03  .96627467e+03  .20214049e-02", 
" -.15282653e+02  .81029249e+02 -.34436141e+03  .94209470e+03  .20060320e-02", 
" -.15186123e+02  .80149806e+02 -.34120299e+03  .94919977e+03  .19935151e-02", 
" -.15089991e+02  .79279043e+02 -.33803415e+03  .95661254e+03  .19754795e-02", 
" -.14993567e+02  .78320723e+02 -.33127311e+03  .92300929e+03  .19993794e-02", 
" -.14897644e+02  .77427092e+02 -.32742692e+03  .92368544e+03  .20128347e-02", 
" -.14801816e+02  .76519560e+02 -.32231367e+03  .90888799e+03  .19825720e-02", 
" -.14706362e+02  .75634347e+02 -.31842067e+03  .91409341e+03  .19929519e-02", 
" -.14610802e+02  .74682586e+02 -.31080739e+03  .86962219e+03  .19999751e-02", 
" -.14515898e+02  .73790143e+02 -.30549141e+03  .85272208e+03  .19869129e-02", 
" -.14421393e+02  .72921216e+02 -.30114386e+03  .84855487e+03  .19885255e-02", 
" -.14326826e+02  .72008080e+02 -.29394220e+03  .80576477e+03  .19994699e-02", 
" -.14232123e+02  .71073974e+02 -.28627545e+03  .76402339e+03  .19818443e-02", 
" -.14137805e+02  .70275264e+02 -.28509443e+03  .79813851e+03  .19832504e-02", 
" -.14043764e+02  .69443248e+02 -.28223400e+03  .81787090e+03  .19632653e-02", 
" -.13949323e+02  .68527832e+02 -.27548889e+03  .78422056e+03  .19638206e-02", 
" -.13855214e+02  .67628244e+02 -.26836559e+03  .74186181e+03  .19572910e-02", 
" -.13761270e+02  .66768958e+02 -.26320726e+03  .72189545e+03  .19540910e-02", 
" -.13667628e+02  .65994941e+02 -.26379006e+03  .78857588e+03  .19579748e-02", 
" -.13574263e+02  .65201755e+02 -.26092158e+03  .79576975e+03  .19348429e-02", 
" -.13480539e+02  .64341193e+02 -.25493344e+03  .76814321e+03  .19090689e-02", 
" -.13386563e+02  .63455588e+02 -.24789290e+03  .72420811e+03  .19013865e-02", 
" -.13292982e+02  .62628874e+02 -.24280619e+03  .70063263e+03  .18969556e-02", 
" -.13199034e+02  .61749153e+02 -.23649807e+03  .67232132e+03  .18899239e-02", 
" -.13105155e+02  .60959322e+02 -.23518288e+03  .71196126e+03  .18908944e-02", 
" -.13011603e+02  .60101138e+02 -.22870007e+03  .68186140e+03  .18780379e-02", 
" -.12917687e+02  .59265121e+02 -.22274735e+03  .64569083e+03  .18593060e-02", 
" -.12823823e+02  .58458547e+02 -.21847117e+03  .63308252e+03  .18765969e-02", 
" -.12729927e+02  .57644397e+02 -.21425487e+03  .62995346e+03  .18595390e-02", 
" -.12635767e+02  .56898363e+02 -.21378447e+03  .66639131e+03  .18443013e-02", 
" -.12541082e+02  .55960254e+02 -.20304438e+03  .57325721e+03  .18538365e-02", 
" -.12445974e+02  .55037610e+02 -.19381261e+03  .50562260e+03  .18421171e-02", 
" -.12350644e+02  .54053843e+02 -.18093838e+03  .38722088e+03  .18435322e-02", 
" -.12256196e+02  .53290476e+02 -.17807992e+03  .39476893e+03  .18513140e-02", 
" -.12161017e+02  .52372399e+02 -.16660464e+03  .28657943e+03  .18364675e-02", 
" -.12066501e+02  .51631435e+02 -.16331352e+03  .27519387e+03  .18253823e-02", 
" -.11971136e+02  .50813535e+02 -.15758121e+03  .24212764e+03  .18254328e-02", 
" -.11876021e+02  .50072280e+02 -.15513181e+03  .25034470e+03  .18248336e-02", 
" -.11780258e+02  .49293316e+02 -.15076502e+03  .22730415e+03  .18197676e-02", 
" -.11684126e+02  .48553660e+02 -.14895520e+03  .24745526e+03  .18343364e-02", 
" -.11587730e+02  .47748385e+02 -.14318880e+03  .21155732e+03  .18508080e-02", 
" -.11491249e+02  .46961242e+02 -.13748400e+03  .17460342e+03  .18613406e-02", 
" -.11394028e+02  .46185026e+02 -.13361221e+03  .16371840e+03  .18529212e-02", 
" -.11296978e+02  .45452975e+02 -.13052454e+03  .15979220e+03  .18390245e-02", 
" -.11198914e+02  .44641291e+02 -.12417143e+03  .11310671e+03  .18416788e-02", 
" -.11100408e+02  .43861164e+02 -.11997982e+03  .10148401e+03  .18442637e-02", 
" -.11001605e+02  .43105432e+02 -.11662149e+03  .94667733e+02  .18373148e-02", 
" -.10902620e+02  .42396635e+02 -.11512119e+03  .11066974e+03  .18433615e-02", 
" -.10802869e+02  .41763491e+02 -.11855096e+03  .19564002e+03  .18324779e-02", 
" -.10702457e+02  .41024713e+02 -.11588402e+03  .19926438e+03  .18179128e-02", 
" -.10601481e+02  .40327665e+02 -.11581524e+03  .23731493e+03  .18202575e-02", 
" -.10499705e+02  .39547821e+02 -.11173929e+03  .22349984e+03  .18092343e-02", 
" -.10396984e+02  .38792021e+02 -.10912166e+03  .22605099e+03  .18037591e-02", 
" -.10293703e+02  .38034211e+02 -.10610155e+03  .22627648e+03  .17860535e-02", 
" -.10190073e+02  .37287046e+02 -.10252845e+03  .21375343e+03  .17973320e-02", 
" -.10085776e+02  .36583090e+02 -.10189497e+03  .24279541e+03  .17897984e-02", 
" -.99794945e+01  .35665920e+02 -.90904675e+02  .14278229e+03  .18057909e-02", 
" -.98728429e+01  .34817497e+02 -.82966133e+02  .75980266e+02  .18107930e-02", 
" -.97654857e+01  .34146538e+02 -.83957381e+02  .11794177e+03  .18118413e-02", 
" -.96571785e+01  .33397358e+02 -.81464824e+02  .12312355e+03  .18043197e-02", 
" -.95476285e+01  .32648297e+02 -.78329704e+02  .11837273e+03  .18160911e-02", 
" -.94359249e+01  .31786016e+02 -.72070949e+02  .83143715e+02  .18154718e-02", 
" -.93240056e+01  .31033430e+02 -.68328093e+02  .58830254e+02  .17796099e-02", 
" -.92103217e+01  .30326547e+02 -.69127380e+02  .10570283e+03  .17783008e-02", 
" -.90954815e+01  .29559815e+02 -.65882595e+02  .96145213e+02  .17845908e-02", 
" -.89786840e+01  .28711817e+02 -.59921813e+02  .58265282e+02  .17689976e-02", 
" -.88603946e+01  .27850460e+02 -.53938370e+02  .28946816e+02  .17862767e-02", 
" -.87405262e+01  .27053268e+02 -.49844075e+02  .10782596e+02  .17678798e-02", 
" -.86183867e+01  .26265473e+02 -.46911587e+02  .10980615e+01  .17377872e-02", 
" -.84945904e+01  .25535811e+02 -.48480753e+02  .63429828e+02  .17092085e-02", 
" -.83684695e+01  .24767918e+02 -.48660658e+02  .11777388e+03  .17017350e-02", 
" -.82401966e+01  .23973585e+02 -.45783062e+02  .11039461e+03  .16791378e-02", 
" -.81095358e+01  .23176672e+02 -.42821748e+02  .96104077e+02  .16971926e-02", 
" -.79755331e+01  .22250381e+02 -.34731309e+02  .26906316e+02  .17329738e-02", 
" -.78379775e+01  .21204850e+02 -.22434617e+02 -.86358648e+02  .17126723e-02", 
" -.76985708e+01  .20336135e+02 -.17484722e+02 -.11659317e+03  .17140498e-02", 
" -.75549291e+01  .19461135e+02 -.13559971e+02 -.13536665e+03  .16897778e-02", 
" -.74076165e+01  .18580660e+02 -.96446490e+01 -.15605798e+03  .17090497e-02", 
" -.72553963e+01  .17576774e+02 -.15461736e+01 -.22516302e+03  .16962084e-02", 
" -.70999195e+01  .16829993e+02 -.66811270e+01 -.12117365e+03  .16906576e-02", 
" -.69384028e+01  .15984207e+02 -.77875660e+01 -.71656451e+02  .17214474e-02", 
" -.67708650e+01  .15012654e+02 -.36112302e+01 -.88048195e+02  .17305829e-02", 
" -.65965240e+01  .13986191e+02  .28356786e+01 -.13300508e+03  .17655727e-02", 
" -.64143367e+01  .12975641e+02  .69958923e+01 -.15564419e+03  .17572394e-02", 
" -.62236811e+01  .11847083e+02  .15421291e+02 -.22852782e+03  .17669069e-02", 
" -.60228782e+01  .10651274e+02  .25988582e+02 -.32309451e+03  .17923402e-02", 
" -.58103473e+01  .95928426e+01  .26538439e+02 -.28689838e+03  .18629827e-02", 
" -.55832598e+01  .84856281e+01  .25408839e+02 -.21436747e+03  .19130089e-02", 
" -.53404197e+01  .76907988e+01  .82399915e+01  .34885352e+02  .19219160e-02", 
" -.50751667e+01  .63766077e+01  .15496867e+02 -.29469055e+02  .20259614e-02", 
" -.47813026e+01  .49279560e+01  .22085403e+02 -.67640058e+02  .21340674e-02", 
" -.44526388e+01  .36933055e+01  .15654211e+02  .41435034e+02  .22944508e-02", 
" -.40693349e+01  .19702335e+01  .23058833e+02 -.40531284e+01  .24103300e-02", 
" -.36077075e+01  .45765730e+00  .15075869e+02  .81762905e+02  .26541523e-02", 
" -.30022614e+01 -.11833809e+01 -.73741619e+01  .38056958e+03  .29462382e-02", 
" -.28516909e+01 -.14621029e+01 -.22675542e+02  .62111173e+03  .30205879e-02", 
" -.26882835e+01 -.18666788e+01 -.27299069e+02  .68992121e+03  .31187044e-02", 
" -.25077046e+01 -.19987403e+01 -.46826883e+02  .95722732e+03  .32818820e-02", 
" -.23000705e+01 -.28267679e+01 -.31405815e+02  .73841311e+03  .35633052e-02", 
" -.20622200e+01 -.37183512e+01 -.10987837e+02  .44864577e+03  .39053283e-02", 
" -.17770452e+01 -.46619236e+01  .36669995e+01  .25176776e+03  .41424962e-02", 
" -.14321370e+01 -.41161999e+01 -.55189619e+02  .10089170e+04  .47136844e-02", 
" -.96375019e+00 -.57857740e+01  .97030135e+01  .26188064e+02  .53467920e-02", 
" -.23153836e+00 -.63807802e+01  .79937940e+01  .77788990e+02  .66912116e-02", 
"  .42862660e+00 -.72336521e+01  .39881753e+02 -.21903148e+03  .83685702e-02", 
"  .12145638e+01 -.67798287e+01  .10016909e+03 -.15254127e+04  .12418904e-01", 
"  .17660189e+01 -.28116861e+01 -.29459334e+02 -.36132886e+03  .17247437e-01", 
"coact     3  6  3   25", " -.72677606e+02  .14380601e+04 -.18106148e+05  .10826191e+06  .93030792e-01", 
" -.69252468e+02  .13334176e+04 -.17170028e+05  .11162214e+06  .66328533e-01", 
" -.64577222e+02  .11783454e+04 -.14853046e+05  .98479869e+05  .41728990e-01", 
" -.60874615e+02  .10503033e+04 -.12499661e+05  .78482217e+05  .30478051e-01", 
" -.57126843e+02  .92854341e+03 -.10390912e+05  .61571115e+05  .22979562e-01", 
" -.54898164e+02  .86263935e+03 -.94175973e+04  .55064917e+05  .19396507e-01", 
" -.53323440e+02  .82198492e+03 -.89930499e+04  .53796092e+05  .16933240e-01", 
" -.52062396e+02  .78663169e+03 -.84910719e+04  .50565629e+05  .15610368e-01", 
" -.51039144e+02  .75943657e+03 -.81283853e+04  .48304313e+05  .14468307e-01", 
" -.50157025e+02  .73526346e+03 -.77853824e+04  .46105197e+05  .13296454e-01", 
" -.49384330e+02  .71322582e+03 -.74111221e+04  .42876716e+05  .12693078e-01", 
" -.48708478e+02  .69548003e+03 -.71735739e+04  .41444974e+05  .12138805e-01", 
" -.48089966e+02  .67821460e+03 -.68787455e+04  .38733858e+05  .11683990e-01", 
" -.45704001e+02  .61944692e+03 -.61904673e+04  .35608150e+05  .99361263e-02", 
" -.43975001e+02  .57576500e+03 -.55550199e+04  .30956459e+05  .89203939e-02", 
" -.42610902e+02  .54399754e+03 -.51744539e+04  .28786336e+05  .76458675e-02", 
" -.41482005e+02  .51860192e+03 -.49038020e+04  .27778276e+05  .70374488e-02", 
" -.40517861e+02  .49729971e+03 -.46674822e+04  .26642707e+05  .65410977e-02", 
" -.39668703e+02  .47785442e+03 -.44072254e+04  .24844535e+05  .61772984e-02", 
" -.38912112e+02  .46088138e+03 -.41816863e+04  .23245739e+05  .59568244e-02", 
" -.38227465e+02  .44515754e+03 -.39481230e+04  .21313996e+05  .56266321e-02", 
" -.37601298e+02  .43159018e+03 -.37797098e+04  .20268032e+05  .54991960e-02", 
" -.37024393e+02  .41906099e+03 -.36108479e+04  .18992638e+05  .52109177e-02", 
" -.36492880e+02  .40816296e+03 -.34902497e+04  .18363018e+05  .51214273e-02", 
" -.35995220e+02  .39824324e+03 -.33925677e+04  .18006803e+05  .50242351e-02", 
" -.35526130e+02  .38825684e+03 -.32553193e+04  .16968820e+05  .49190789e-02", 
" -.35085411e+02  .37919101e+03 -.31436661e+04  .16233041e+05  .47481260e-02", 
" -.34666504e+02  .37058385e+03 -.30333317e+04  .15440523e+05  .46575938e-02", 
" -.34271127e+02  .36270680e+03 -.29334382e+04  .14690902e+05  .46165618e-02", 
" -.33895693e+02  .35561443e+03 -.28622241e+04  .14357933e+05  .46273487e-02", 
" -.33535360e+02  .34881858e+03 -.27983609e+04  .14141534e+05  .46329510e-02", 
" -.33191397e+02  .34253345e+03 -.27406995e+04  .13911187e+05  .44936397e-02", 
" -.32859621e+02  .33592448e+03 -.26500412e+04  .13155425e+05  .43125433e-02", 
" -.32542425e+02  .33036199e+03 -.26084208e+04  .13120507e+05  .42313389e-02", 
" -.32237186e+02  .32504614e+03 -.25677482e+04  .13065238e+05  .41628304e-02", 
" -.31941309e+02  .31938078e+03 -.24918838e+04  .12411399e+05  .40798333e-02", 
" -.31653932e+02  .31385371e+03 -.24191909e+04  .11838069e+05  .41721160e-02", 
" -.31375472e+02  .30854048e+03 -.23507372e+04  .11293301e+05  .40520768e-02", 
" -.31107278e+02  .30362378e+03 -.22970630e+04  .10985177e+05  .39328601e-02", 
" -.30846694e+02  .29896057e+03 -.22465621e+04  .10656987e+05  .38071534e-02", 
" -.30593624e+02  .29438655e+03 -.21943815e+04  .10303861e+05  .37733270e-02", 
" -.30347940e+02  .29007503e+03 -.21514456e+04  .10087507e+05  .37072518e-02", 
" -.30107940e+02  .28553686e+03 -.20856514e+04  .94886354e+04  .36576146e-02", 
" -.29873359e+02  .28138356e+03 -.20403110e+04  .92007030e+04  .36229566e-02", 
" -.29644540e+02  .27719910e+03 -.19839040e+04  .86951773e+04  .36189836e-02", 
" -.29421894e+02  .27337490e+03 -.19457473e+04  .84982243e+04  .35534676e-02", 
" -.29204113e+02  .26962976e+03 -.19095697e+04  .83451762e+04  .35547374e-02", 
" -.28991265e+02  .26610347e+03 -.18788340e+04  .82301228e+04  .35465156e-02", 
" -.28783754e+02  .26290394e+03 -.18632567e+04  .83486617e+04  .35690436e-02", 
" -.28580013e+02  .25960270e+03 -.18356965e+04  .82635531e+04  .34486512e-02", 
" -.28378726e+02  .25616759e+03 -.18034762e+04  .81506646e+04  .34793870e-02", 
" -.28182960e+02  .25291283e+03 -.17712011e+04  .79862134e+04  .34501212e-02", 
" -.27990953e+02  .24980052e+03 -.17439339e+04  .78966672e+04  .34054919e-02", 
" -.27802395e+02  .24679086e+03 -.17171953e+04  .77615032e+04  .33582431e-02", 
" -.27616495e+02  .24365583e+03 -.16811326e+04  .75078259e+04  .33457459e-02", 
" -.27434980e+02  .24091481e+03 -.16631574e+04  .75013021e+04  .33026487e-02", 
" -.27255546e+02  .23797244e+03 -.16315405e+04  .72810907e+04  .32523717e-02", 
" -.27079312e+02  .23517894e+03 -.16049495e+04  .71269894e+04  .32071207e-02", 
" -.26905804e+02  .23245803e+03 -.15827465e+04  .70636205e+04  .31676109e-02", 
" -.26734214e+02  .22969517e+03 -.15544209e+04  .68813738e+04  .31547186e-02", 
" -.26565966e+02  .22701066e+03 -.15277100e+04  .67343589e+04  .30946542e-02", 
" -.26400107e+02  .22446701e+03 -.15064410e+04  .66484658e+04  .30517520e-02", 
" -.26236052e+02  .22171490e+03 -.14716797e+04  .63795888e+04  .30630199e-02", 
" -.26075233e+02  .21931228e+03 -.14550187e+04  .63763855e+04  .30383589e-02", 
" -.25915130e+02  .21664830e+03 -.14207853e+04  .60767282e+04  .30132009e-02", 
" -.25757374e+02  .21404739e+03 -.13894491e+04  .58372201e+04  .29929604e-02", 
" -.25602292e+02  .21158681e+03 -.13631672e+04  .56678483e+04  .29980856e-02", 
" -.25449145e+02  .20928097e+03 -.13448370e+04  .56164124e+04  .29688355e-02", 
" -.25298083e+02  .20697080e+03 -.13211671e+04  .54547993e+04  .29392677e-02", 
" -.25147805e+02  .20448300e+03 -.12875152e+04  .51422268e+04  .29744138e-02", 
" -.24999615e+02  .20228485e+03 -.12708798e+04  .51039198e+04  .29725889e-02", 
" -.24853716e+02  .20013510e+03 -.12528524e+04  .50278734e+04  .29123372e-02", 
" -.24710065e+02  .19809760e+03 -.12384759e+04  .49918708e+04  .28784296e-02", 
" -.24567908e+02  .19607931e+03 -.12240230e+04  .49671156e+04  .28749093e-02", 
" -.24427159e+02  .19401736e+03 -.12045133e+04  .48468398e+04  .28447317e-02", 
" -.24287604e+02  .19202301e+03 -.11892309e+04  .47994665e+04  .28294732e-02", 
" -.24149645e+02  .19011805e+03 -.11790544e+04  .48403521e+04  .28649409e-02", 
" -.24012318e+02  .18806068e+03 -.11573056e+04  .46899367e+04  .28894520e-02", 
" -.23877015e+02  .18623639e+03 -.11474958e+04  .47094013e+04  .28749196e-02", 
" -.23741892e+02  .18407848e+03 -.11174594e+04  .44261264e+04  .28770859e-02", 
" -.23609058e+02  .18221334e+03 -.11009000e+04  .43213598e+04  .28666001e-02", 
" -.23476107e+02  .18025738e+03 -.10829084e+04  .42337264e+04  .28442815e-02", 
" -.23345570e+02  .17845263e+03 -.10676872e+04  .41470310e+04  .28292933e-02", 
" -.23215136e+02  .17660085e+03 -.10516570e+04  .40651192e+04  .28107571e-02", 
" -.23086944e+02  .17491250e+03 -.10429861e+04  .40997304e+04  .27584390e-02", 
" -.22959203e+02  .17307460e+03 -.10238599e+04  .39566932e+04  .27293296e-02", 
" -.22832545e+02  .17135309e+03 -.10108519e+04  .39157478e+04  .27178921e-02", 
" -.22707012e+02  .16970694e+03 -.10014477e+04  .39248379e+04  .26815456e-02", 
" -.22582305e+02  .16810621e+03 -.99477559e+03  .39743941e+04  .26801737e-02", 
" -.22457476e+02  .16622589e+03 -.97135162e+03  .37640686e+04  .26813829e-02", 
" -.22334487e+02  .16457560e+03 -.95828136e+03  .37025022e+04  .26682567e-02", 
" -.22212724e+02  .16292280e+03 -.94425698e+03  .36335088e+04  .26341464e-02", 
" -.22091205e+02  .16126839e+03 -.93191541e+03  .36058298e+04  .25906830e-02", 
" -.21970249e+02  .15948372e+03 -.90858685e+03  .33854275e+04  .25870550e-02", 
" -.21850178e+02  .15776309e+03 -.88932968e+03  .32356148e+04  .25790038e-02", 
" -.21731753e+02  .15639884e+03 -.88960110e+03  .33796082e+04  .25641724e-02", 
" -.21613878e+02  .15492013e+03 -.88199224e+03  .33989575e+04  .25688743e-02", 
" -.21496368e+02  .15341350e+03 -.87304715e+03  .34057691e+04  .25084063e-02", 
" -.21379527e+02  .15187843e+03 -.86097865e+03  .33547105e+04  .25103633e-02", 
" -.21263789e+02  .15054163e+03 -.85708602e+03  .33946738e+04  .25243329e-02", 
" -.21147682e+02  .14893308e+03 -.84138459e+03  .33068695e+04  .25315918e-02", 
" -.21032395e+02  .14735313e+03 -.82329556e+03  .31434223e+04  .25009432e-02", 
" -.20917428e+02  .14582703e+03 -.80982447e+03  .30644919e+04  .24840014e-02", 
" -.20803310e+02  .14439904e+03 -.80143986e+03  .30680656e+04  .24629471e-02", 
" -.20689692e+02  .14300070e+03 -.79427768e+03  .30880653e+04  .24748361e-02", 
" -.20576591e+02  .14144544e+03 -.77474763e+03  .28983518e+04  .24659352e-02", 
" -.20464274e+02  .14008536e+03 -.76710083e+03  .28949094e+04  .24418848e-02", 
" -.20352006e+02  .13862846e+03 -.75393676e+03  .28188082e+04  .24347107e-02", 
" -.20239882e+02  .13723583e+03 -.74822917e+03  .28880407e+04  .24450477e-02", 
" -.20129080e+02  .13595444e+03 -.74434318e+03  .29505776e+04  .24344780e-02", 
" -.20018312e+02  .13457778e+03 -.73348396e+03  .28909285e+04  .24218870e-02", 
" -.19907286e+02  .13307175e+03 -.71553518e+03  .27228033e+04  .24215435e-02", 
" -.19797198e+02  .13174394e+03 -.70809218e+03  .27219315e+04  .24236363e-02", 
" -.19687071e+02  .13030030e+03 -.69355169e+03  .26155533e+04  .24047826e-02", 
" -.19577471e+02  .12884196e+03 -.67642770e+03  .24599188e+04  .23759904e-02", 
" -.19468827e+02  .12760474e+03 -.67399602e+03  .25496063e+04  .23505535e-02", 
" -.19360177e+02  .12625703e+03 -.66090171e+03  .24380195e+04  .23406501e-02", 
" -.19251953e+02  .12496360e+03 -.65233905e+03  .24149906e+04  .23218163e-02", 
" -.19143763e+02  .12363086e+03 -.64154481e+03  .23647082e+04  .23081824e-02", 
" -.19035391e+02  .12224499e+03 -.62799593e+03  .22726937e+04  .22973138e-02", 
" -.18928397e+02  .12105245e+03 -.62308275e+03  .22930430e+04  .23061149e-02", 
" -.18820981e+02  .11972337e+03 -.61110394e+03  .22200433e+04  .23319579e-02", 
" -.18713586e+02  .11842451e+03 -.60140869e+03  .21827836e+04  .22949938e-02", 
" -.18606464e+02  .11719228e+03 -.59409448e+03  .21628246e+04  .22881236e-02", 
" -.18499899e+02  .11596391e+03 -.58679289e+03  .21537398e+04  .22701367e-02", 
" -.18392711e+02  .11457554e+03 -.57032400e+03  .20137102e+04  .22861870e-02", 
" -.18286077e+02  .11326061e+03 -.55738385e+03  .19227926e+04  .22656490e-02", 
" -.18180044e+02  .11191046e+03 -.53837892e+03  .17070906e+04  .22506823e-02", 
" -.18073476e+02  .11064633e+03 -.52928404e+03  .16834586e+04  .22545995e-02", 
" -.17967306e+02  .10937431e+03 -.51822546e+03  .16164454e+04  .22538086e-02", 
" -.17861256e+02  .10813572e+03 -.50850897e+03  .15694142e+04  .22698646e-02", 
" -.17755508e+02  .10697221e+03 -.50252902e+03  .15795332e+04  .22487485e-02", 
" -.17649025e+02  .10561207e+03 -.48622854e+03  .14422679e+04  .22358924e-02", 
" -.17543390e+02  .10440968e+03 -.47521859e+03  .13475827e+04  .22250847e-02", 
" -.17437458e+02  .10321532e+03 -.46725175e+03  .13241013e+04  .22145346e-02", 
" -.17331344e+02  .10195934e+03 -.45555545e+03  .12417046e+04  .21993528e-02", 
" -.17225181e+02  .10075403e+03 -.44805906e+03  .12369560e+04  .22014249e-02", 
" -.17118816e+02  .99568181e+02 -.44130530e+03  .12330250e+04  .21869235e-02", 
" -.17012089e+02  .98238642e+02 -.42698221e+03  .11361303e+04  .21766898e-02", 
" -.16905408e+02  .96927595e+02 -.40985439e+03  .94779244e+03  .21478845e-02", 
" -.16799318e+02  .95902285e+02 -.41275070e+03  .11124931e+04  .21337913e-02", 
" -.16692798e+02  .94747553e+02 -.40535526e+03  .10752261e+04  .21398525e-02", 
" -.16586341e+02  .93609451e+02 -.39963771e+03  .10927158e+04  .21110446e-02", 
" -.16479820e+02  .92395994e+02 -.38757980e+03  .99478746e+03  .20953623e-02", 
" -.16372933e+02  .91167136e+02 -.37590251e+03  .91463410e+03  .21027139e-02", 
" -.16265263e+02  .89825764e+02 -.35730397e+03  .71531300e+03  .20820102e-02", 
" -.16157589e+02  .88625289e+02 -.34868857e+03  .68996468e+03  .20912329e-02", 
" -.16050010e+02  .87415161e+02 -.33682531e+03  .58792215e+03  .20888630e-02", 
" -.15941774e+02  .86209590e+02 -.32691793e+03  .53764835e+03  .20736968e-02", 
" -.15833224e+02  .85065675e+02 -.32182530e+03  .56298181e+03  .20735935e-02", 
" -.15724981e+02  .84064579e+02 -.32424013e+03  .70451001e+03  .20762883e-02", 
" -.15616721e+02  .82985148e+02 -.32021595e+03  .74080966e+03  .20866448e-02", 
" -.15508536e+02  .81975984e+02 -.32010417e+03  .83875554e+03  .20774401e-02", 
" -.15399117e+02  .80829945e+02 -.31083175e+03  .77567086e+03  .20637154e-02", 
" -.15288819e+02  .79769102e+02 -.30816713e+03  .81425041e+03  .20703912e-02", 
" -.15178125e+02  .78558787e+02 -.29637570e+03  .72238268e+03  .20751798e-02", 
" -.15067919e+02  .77521652e+02 -.29347766e+03  .75295739e+03  .20708569e-02", 
" -.14956412e+02  .76315593e+02 -.28228092e+03  .67726707e+03  .20643790e-02", 
" -.14844236e+02  .75213657e+02 -.27727741e+03  .67931982e+03  .20721032e-02", 
" -.14732424e+02  .74229442e+02 -.27772936e+03  .76868735e+03  .20687404e-02", 
" -.14619366e+02  .73084364e+02 -.26947419e+03  .71350845e+03  .20940527e-02", 
" -.14505014e+02  .71821992e+02 -.25476983e+03  .56609292e+03  .21337974e-02", 
" -.14391686e+02  .70881993e+02 -.25627542e+03  .64393117e+03  .21248836e-02", 
" -.14276551e+02  .69725755e+02 -.24816772e+03  .61441700e+03  .21329968e-02", 
" -.14160164e+02  .68552414e+02 -.24096729e+03  .60347001e+03  .21209379e-02", 
" -.14043482e+02  .67272119e+02 -.22438583e+03  .43215966e+03  .21054640e-02", 
" -.13925744e+02  .66005872e+02 -.20773656e+03  .22691919e+03  .20918129e-02", 
" -.13807656e+02  .65000907e+02 -.20841895e+03  .31277858e+03  .20756218e-02", 
" -.13688537e+02  .63893255e+02 -.20226416e+03  .28510244e+03  .20752248e-02", 
" -.13568679e+02  .62834815e+02 -.20117695e+03  .36103607e+03  .20942316e-02", 
" -.13447508e+02  .61580976e+02 -.18400878e+03  .15088147e+03  .21215799e-02", 
" -.13325103e+02  .60484809e+02 -.17968720e+03  .15923411e+03  .21253084e-02", 
" -.13201799e+02  .59499553e+02 -.18223643e+03  .27400856e+03  .21388803e-02", 
" -.13076662e+02  .58241294e+02 -.16947288e+03  .16207776e+03  .21220418e-02", 
" -.12950376e+02  .57068965e+02 -.16165327e+03  .12168018e+03  .21103811e-02", 
" -.12822895e+02  .56009721e+02 -.16457460e+03  .28912393e+03  .21105991e-02", 
" -.12693818e+02  .54811933e+02 -.15679939e+03  .26115915e+03  .21461241e-02", 
" -.12562823e+02  .53571616e+02 -.14727081e+03  .21884614e+03  .21265538e-02", 
" -.12429621e+02  .52311098e+02 -.13805585e+03  .17122634e+03  .21410263e-02", 
" -.12295089e+02  .51069416e+02 -.12857078e+03  .10951544e+03  .21387748e-02", 
" -.12158853e+02  .49831668e+02 -.11693605e+03 -.24297388e+02  .21568834e-02", 
" -.12020358e+02  .48712470e+02 -.11607790e+03  .46750596e+02  .21433094e-02", 
" -.11879083e+02  .47348525e+02 -.10012392e+03 -.12507210e+03  .21533944e-02", 
" -.11735625e+02  .46033051e+02 -.88385253e+02 -.23905287e+03  .21223147e-02", 
" -.11589673e+02  .44713774e+02 -.77973948e+02 -.30862622e+03  .21424166e-02", 
" -.11441325e+02  .43467678e+02 -.70008653e+02 -.36718359e+03  .21341876e-02", 
" -.11289693e+02  .42272619e+02 -.68096351e+02 -.32385274e+03  .21430739e-02", 
" -.11134505e+02  .40890585e+02 -.57329484e+02 -.39426586e+03  .21436267e-02", 
" -.10975737e+02  .39522328e+02 -.51097192e+02 -.36912547e+03  .21361433e-02", 
" -.10813569e+02  .38079883e+02 -.35946692e+02 -.52501435e+03  .21707826e-02", 
" -.10646615e+02  .36628460e+02 -.25960444e+02 -.55835101e+03  .22356879e-02", 
" -.10475098e+02  .35274364e+02 -.22974648e+02 -.49942455e+03  .22361494e-02", 
" -.10299788e+02  .34017883e+02 -.23492991e+02 -.40834073e+03  .22527973e-02", 
" -.10117515e+02  .32384623e+02 -.51161407e+01 -.60244686e+03  .23008709e-02", 
" -.99300462e+01  .30909072e+02  .34609522e+01 -.64649408e+03  .22944227e-02", 
" -.97339530e+01  .29020274e+02  .31697813e+02 -.98361057e+03  .23440435e-02", 
" -.95304954e+01  .27246635e+02  .53843286e+02 -.12438050e+04  .24320905e-02", 
" -.93193779e+01  .25624727e+02  .63147755e+02 -.12993065e+04  .24627605e-02", 
" -.90968135e+01  .23728504e+02  .82349751e+02 -.14728425e+04  .24354632e-02", 
" -.88633917e+01  .21821811e+02  .99948028e+02 -.16106103e+04  .24023649e-02", 
" -.86134423e+01  .19519365e+02  .13122487e+03 -.19406191e+04  .24391458e-02", 
" -.83492051e+01  .17494705e+02  .14136173e+03 -.19385018e+04  .24011332e-02", 
" -.80655481e+01  .15557319e+02  .14208035e+03 -.18092004e+04  .24513797e-02", 
" -.77587105e+01  .13726953e+02  .12928345e+03 -.14699147e+04  .25767657e-02", 
" -.74185982e+01  .11657817e+02  .11787083e+03 -.11474309e+04  .26424762e-02", 
" -.70355718e+01  .88730188e+01  .13572477e+03 -.12710956e+04  .27532029e-02", 
" -.65933189e+01  .58815368e+01  .13828009e+03 -.10626221e+04  .29959551e-02", 
" -.60627995e+01  .31930834e+01  .10468367e+03 -.46642209e+03  .32744461e-02", 
" -.53742061e+01 -.59379350e+00  .63158743e+02  .51475176e+03  .33775293e-02", 
" -.52040085e+01 -.18060786e+01  .72768888e+02  .42076089e+03  .35409607e-02", 
" -.50194152e+01 -.31524894e+01  .88530317e+02  .23616083e+03  .37023803e-02", 
" -.48156684e+01 -.45384945e+01  .97572782e+02  .18145288e+03  .39328997e-02", 
" -.45859083e+01 -.62755925e+01  .12593932e+03 -.31110767e+03  .42504556e-02", 
" -.43237709e+01 -.71174818e+01  .69290662e+02  .76046355e+03  .43153127e-02", 
" -.40081524e+01 -.94719087e+01  .84323338e+02  .72277687e+03  .47346637e-02", 
" -.36212176e+01 -.12008092e+02  .78879446e+02  .10904404e+04  .53477885e-02", 
" -.31002922e+01 -.15997368e+02  .12272784e+03  .50172038e+03  .61654442e-02", 
" -.22811788e+01 -.21015803e+02  .14919371e+03  .55532563e+02  .80624595e-02", 
" -.15387285e+01 -.23210332e+02  .22070465e+02  .21622304e+04  .10298361e-01", 
" -.63899949e+00 -.24982841e+02 -.17813295e+03  .47572304e+04  .15220633e-01", 
" -.14871118e-01 -.25517499e+02 -.43251151e+03  .93288549e+04  .21024054e-01", 
"coactt    3  7  3   25", " -.80221503e+02  .17589037e+04 -.24637179e+05  .16504894e+06  .99531921e-01", 
" -.76485150e+02  .16072374e+04 -.21859039e+05  .14500802e+06  .70954930e-01", 
" -.71516079e+02  .14196704e+04 -.18673877e+05  .12342200e+06  .43672687e-01", 
" -.67688878e+02  .12768977e+04 -.16021357e+05  .10229803e+06  .33438702e-01", 
" -.63828848e+02  .11512928e+04 -.14173535e+05  .91217202e+05  .24563061e-01", 
" -.61487931e+02  .10701701e+04 -.12709161e+05  .79440999e+05  .20027334e-01", 
" -.59807576e+02  .10170048e+04 -.11882339e+05  .73777632e+05  .17156787e-01", 
" -.58484238e+02  .97539761e+03 -.11254974e+05  .69882361e+05  .15750043e-01", 
" -.57390732e+02  .94079276e+03 -.10688313e+05  .65644022e+05  .14585674e-01", 
" -.56471432e+02  .91212679e+03 -.10197114e+05  .61595057e+05  .13528018e-01", 
" -.55660863e+02  .88798913e+03 -.98366460e+04  .59076532e+05  .12543599e-01", 
" -.54949913e+02  .86824863e+03 -.95845922e+04  .57677114e+05  .12158245e-01", 
" -.54303941e+02  .84893516e+03 -.92729307e+04  .55208264e+05  .11740727e-01", 
" -.51785382e+02  .77759118e+03 -.82314544e+04  .48055106e+05  .95396661e-02", 
" -.49969764e+02  .72863584e+03 -.75866071e+04  .44445680e+05  .87452010e-02", 
" -.48527090e+02  .69043322e+03 -.70797618e+04  .41456027e+05  .79716161e-02", 
" -.47331699e+02  .65861592e+03 -.65983923e+04  .37683947e+05  .75508473e-02", 
" -.46310083e+02  .63306089e+03 -.62782517e+04  .35903990e+05  .74598340e-02", 
" -.45408577e+02  .61000357e+03 -.59576879e+04  .33721797e+05  .70104912e-02", 
" -.44607124e+02  .59020082e+03 -.56964731e+04  .32050281e+05  .65539278e-02", 
" -.43885077e+02  .57315969e+03 -.55017859e+04  .31129152e+05  .62629106e-02", 
" -.43219888e+02  .55715086e+03 -.52981134e+04  .29893522e+05  .59927206e-02", 
" -.42607297e+02  .54229675e+03 -.50981058e+04  .28546765e+05  .56893661e-02", 
" -.42037148e+02  .52817101e+03 -.48866943e+04  .26911733e+05  .53066050e-02", 
" -.41505540e+02  .51549286e+03 -.47130367e+04  .25669764e+05  .52203769e-02", 
" -.41010678e+02  .50448646e+03 -.45853255e+04  .24933739e+05  .51257286e-02", 
" -.40543399e+02  .49422252e+03 -.44743519e+04  .24414480e+05  .49254336e-02", 
" -.40096172e+02  .48390739e+03 -.43475100e+04  .23726379e+05  .47737492e-02", 
" -.39672147e+02  .47408461e+03 -.42125455e+04  .22741699e+05  .45978603e-02", 
" -.39270500e+02  .46498199e+03 -.40916234e+04  .21908436e+05  .44893163e-02", 
" -.38885972e+02  .45650337e+03 -.39889907e+04  .21283225e+05  .44257485e-02", 
" -.38518086e+02  .44856093e+03 -.38938280e+04  .20669604e+05  .44725138e-02", 
" -.38165019e+02  .44105182e+03 -.38112548e+04  .20275283e+05  .43517897e-02", 
" -.37823187e+02  .43339316e+03 -.37028293e+04  .19411011e+05  .42521002e-02", 
" -.37494234e+02  .42627379e+03 -.36142297e+04  .18832594e+05  .42980215e-02", 
" -.37177653e+02  .41999472e+03 -.35631956e+04  .18799119e+05  .42520517e-02", 
" -.36872068e+02  .41350441e+03 -.34822707e+04  .18244952e+05  .42097684e-02", 
" -.36574145e+02  .40698158e+03 -.33866501e+04  .17395270e+05  .40922159e-02", 
" -.36285837e+02  .40093943e+03 -.33143107e+04  .16933666e+05  .40311782e-02", 
" -.36006945e+02  .39512639e+03 -.32415714e+04  .16437830e+05  .39773160e-02", 
" -.35735316e+02  .38953934e+03 -.31781860e+04  .16090953e+05  .39533420e-02", 
" -.35470717e+02  .38392085e+03 -.31031221e+04  .15539181e+05  .39342763e-02", 
" -.35213628e+02  .37904829e+03 -.30648793e+04  .15523135e+05  .38825203e-02", 
" -.34962322e+02  .37400116e+03 -.30078627e+04  .15198880e+05  .38432900e-02", 
" -.34718306e+02  .36914220e+03 -.29513417e+04  .14843454e+05  .37491160e-02", 
" -.34477751e+02  .36414830e+03 -.28859535e+04  .14374580e+05  .37425638e-02", 
" -.34243288e+02  .35941805e+03 -.28277460e+04  .13964656e+05  .36729066e-02", 
" -.34014646e+02  .35499186e+03 -.27811376e+04  .13715534e+05  .36369435e-02", 
" -.33789776e+02  .35064207e+03 -.27358413e+04  .13479704e+05  .36279762e-02", 
" -.33570091e+02  .34626623e+03 -.26827803e+04  .13125849e+05  .36166668e-02", 
" -.33353127e+02  .34195679e+03 -.26317503e+04  .12798314e+05  .36233791e-02", 
" -.33141764e+02  .33793433e+03 -.25885313e+04  .12538547e+05  .35611238e-02", 
" -.32933743e+02  .33375271e+03 -.25309127e+04  .12048465e+05  .35087381e-02", 
" -.32731049e+02  .32995379e+03 -.24889205e+04  .11775112e+05  .35306057e-02", 
" -.32531229e+02  .32627094e+03 -.24549512e+04  .11662725e+05  .34890073e-02", 
" -.32335200e+02  .32256467e+03 -.24112937e+04  .11351268e+05  .34712892e-02", 
" -.32141036e+02  .31888131e+03 -.23711734e+04  .11126435e+05  .34872290e-02", 
" -.31950155e+02  .31516167e+03 -.23247022e+04  .10792835e+05  .34774523e-02", 
" -.31762549e+02  .31168550e+03 -.22893609e+04  .10622526e+05  .34372648e-02", 
" -.31576920e+02  .30804089e+03 -.22422075e+04  .10281933e+05  .33914231e-02", 
" -.31394872e+02  .30470362e+03 -.22098507e+04  .10150637e+05  .33884415e-02", 
" -.31215177e+02  .30144919e+03 -.21780110e+04  .10013434e+05  .33361609e-02", 
" -.31038777e+02  .29828690e+03 -.21460944e+04  .98412721e+04  .33332464e-02", 
" -.30864428e+02  .29511194e+03 -.21130387e+04  .96809700e+04  .32773215e-02", 
" -.30693276e+02  .29208850e+03 -.20827840e+04  .95211026e+04  .32588370e-02", 
" -.30523718e+02  .28906988e+03 -.20524992e+04  .93647436e+04  .32762786e-02", 
" -.30356043e+02  .28616255e+03 -.20278988e+04  .93058020e+04  .32357793e-02", 
" -.30190949e+02  .28338312e+03 -.20052218e+04  .92270978e+04  .31972554e-02", 
" -.30026734e+02  .28027281e+03 -.19627363e+04  .88641791e+04  .31775479e-02", 
" -.29865088e+02  .27747854e+03 -.19382890e+04  .87956002e+04  .31277179e-02", 
" -.29704961e+02  .27470164e+03 -.19136602e+04  .87182100e+04  .31295654e-02", 
" -.29546688e+02  .27191250e+03 -.18838301e+04  .85371263e+04  .31087557e-02", 
" -.29390733e+02  .26925621e+03 -.18594593e+04  .84318395e+04  .30889739e-02", 
" -.29235407e+02  .26645561e+03 -.18272513e+04  .82164736e+04  .31197045e-02", 
" -.29081383e+02  .26360930e+03 -.17883093e+04  .78662716e+04  .31009726e-02", 
" -.28930068e+02  .26101661e+03 -.17592479e+04  .76396840e+04  .30469664e-02", 
" -.28779119e+02  .25829788e+03 -.17269464e+04  .74206675e+04  .30390197e-02", 
" -.28630357e+02  .25584372e+03 -.17098958e+04  .74447355e+04  .30239752e-02", 
" -.28483401e+02  .25334328e+03 -.16834155e+04  .72733356e+04  .29777876e-02", 
" -.28336929e+02  .25087422e+03 -.16600841e+04  .71659826e+04  .29410015e-02", 
" -.28192504e+02  .24856748e+03 -.16424890e+04  .71240947e+04  .29071251e-02", 
" -.28048438e+02  .24613467e+03 -.16183747e+04  .69826346e+04  .28913107e-02", 
" -.27906945e+02  .24390574e+03 -.16021939e+04  .69538310e+04  .28622142e-02", 
" -.27765412e+02  .24145579e+03 -.15727306e+04  .67182876e+04  .28508712e-02", 
" -.27625037e+02  .23906288e+03 -.15476293e+04  .65772008e+04  .28312189e-02", 
" -.27486040e+02  .23677264e+03 -.15261089e+04  .64837991e+04  .28566135e-02", 
" -.27347555e+02  .23437430e+03 -.14976653e+04  .62762716e+04  .28121982e-02", 
" -.27210520e+02  .23209329e+03 -.14733816e+04  .61160365e+04  .27995856e-02", 
" -.27074116e+02  .22977936e+03 -.14461617e+04  .59159747e+04  .27684171e-02", 
" -.26938438e+02  .22751146e+03 -.14211004e+04  .57396173e+04  .27756603e-02", 
" -.26803299e+02  .22508043e+03 -.13855056e+04  .54047500e+04  .27503299e-02", 
" -.26670118e+02  .22301907e+03 -.13699390e+04  .53686692e+04  .27499237e-02", 
" -.26537618e+02  .22093361e+03 -.13538321e+04  .53426831e+04  .27541710e-02", 
" -.26405613e+02  .21885951e+03 -.13373628e+04  .53078695e+04  .27458244e-02", 
" -.26274497e+02  .21669228e+03 -.13127546e+04  .51356379e+04  .27329891e-02", 
" -.26144491e+02  .21462202e+03 -.12909079e+04  .49767927e+04  .27034340e-02", 
" -.26014974e+02  .21246665e+03 -.12646277e+04  .47627539e+04  .26968231e-02", 
" -.25886565e+02  .21052525e+03 -.12489606e+04  .46975960e+04  .26840757e-02", 
" -.25758913e+02  .20848127e+03 -.12274446e+04  .45598938e+04  .26714296e-02", 
" -.25632148e+02  .20661881e+03 -.12170755e+04  .46011255e+04  .26214103e-02", 
" -.25505746e+02  .20465058e+03 -.11972633e+04  .44653160e+04  .25886013e-02", 
" -.25380161e+02  .20280649e+03 -.11850868e+04  .44543756e+04  .26119743e-02", 
" -.25254647e+02  .20086353e+03 -.11680614e+04  .43907237e+04  .25774640e-02", 
" -.25130103e+02  .19909123e+03 -.11601695e+04  .44546057e+04  .25953114e-02", 
" -.25005529e+02  .19712038e+03 -.11378507e+04  .42717499e+04  .25896389e-02", 
" -.24881685e+02  .19517360e+03 -.11175670e+04  .41486266e+04  .25640306e-02", 
" -.24758308e+02  .19327027e+03 -.10995830e+04  .40644509e+04  .25615127e-02", 
" -.24635630e+02  .19131274e+03 -.10754443e+04  .38560706e+04  .25701297e-02", 
" -.24513581e+02  .18951114e+03 -.10615702e+04  .38166414e+04  .25492274e-02", 
" -.24391760e+02  .18760315e+03 -.10394679e+04  .36437154e+04  .25577639e-02", 
" -.24270774e+02  .18588549e+03 -.10303311e+04  .36854031e+04  .25837715e-02", 
" -.24149845e+02  .18417079e+03 -.10217654e+04  .37486733e+04  .25602344e-02", 
" -.24029629e+02  .18241182e+03 -.10061197e+04  .36662861e+04  .25490870e-02", 
" -.23909642e+02  .18065079e+03 -.99024703e+03  .35775043e+04  .25441972e-02", 
" -.23789572e+02  .17881029e+03 -.97073267e+03  .34498257e+04  .25614609e-02", 
" -.23669710e+02  .17698597e+03 -.95216582e+03  .33512074e+04  .25616467e-02", 
" -.23550946e+02  .17536527e+03 -.94385440e+03  .33848075e+04  .25367021e-02", 
" -.23432523e+02  .17360079e+03 -.92361456e+03  .31990731e+04  .25224364e-02", 
" -.23313771e+02  .17181247e+03 -.90238312e+03  .30055286e+04  .25150704e-02", 
" -.23195120e+02  .16999713e+03 -.88038417e+03  .28076487e+04  .25048711e-02", 
" -.23076473e+02  .16818596e+03 -.85945958e+03  .26376848e+04  .25027660e-02", 
" -.22958971e+02  .16661392e+03 -.85142190e+03  .26659724e+04  .25095482e-02", 
" -.22841450e+02  .16495900e+03 -.83725768e+03  .25798293e+04  .25269011e-02", 
" -.22724003e+02  .16330696e+03 -.82489891e+03  .25557358e+04  .25308358e-02", 
" -.22606856e+02  .16171983e+03 -.81449369e+03  .25369867e+04  .25349988e-02", 
" -.22489955e+02  .16014837e+03 -.80661686e+03  .25841206e+04  .25482150e-02", 
" -.22372290e+02  .15837882e+03 -.78581053e+03  .24161858e+04  .25808896e-02", 
" -.22254859e+02  .15668481e+03 -.76766142e+03  .22652980e+04  .25864040e-02", 
" -.22137654e+02  .15501717e+03 -.75371291e+03  .22158695e+04  .25681413e-02", 
" -.22021697e+02  .15355371e+03 -.74746725e+03  .22506832e+04  .25115288e-02", 
" -.21903758e+02  .15171770e+03 -.72320683e+03  .20337625e+04  .25180019e-02", 
" -.21787551e+02  .15020291e+03 -.71370272e+03  .20244015e+04  .24697721e-02", 
" -.21670854e+02  .14869176e+03 -.70501095e+03  .20135123e+04  .24540706e-02", 
" -.21553755e+02  .14704064e+03 -.68845885e+03  .18912203e+04  .24459756e-02", 
" -.21437319e+02  .14552089e+03 -.67747295e+03  .18450135e+04  .24162476e-02", 
" -.21320598e+02  .14398704e+03 -.66712729e+03  .18266361e+04  .24039875e-02", 
" -.21203319e+02  .14233461e+03 -.65157825e+03  .17398113e+04  .24195556e-02", 
" -.21085948e+02  .14066871e+03 -.63357282e+03  .15970305e+04  .24201751e-02", 
" -.20969236e+02  .13926215e+03 -.63104821e+03  .17036712e+04  .24477213e-02", 
" -.20851346e+02  .13760146e+03 -.61466958e+03  .16051293e+04  .24502183e-02", 
" -.20733928e+02  .13597491e+03 -.59865394e+03  .14991664e+04  .24626308e-02", 
" -.20616546e+02  .13439699e+03 -.58488702e+03  .14287837e+04  .24297380e-02", 
" -.20498219e+02  .13268602e+03 -.56273956e+03  .12156183e+04  .24191736e-02", 
" -.20379973e+02  .13102096e+03 -.54368988e+03  .10618966e+04  .24305421e-02", 
" -.20261808e+02  .12951375e+03 -.53442375e+03  .10609403e+04  .24404454e-02", 
" -.20143313e+02  .12796540e+03 -.52166043e+03  .99882157e+03  .24425668e-02", 
" -.20024003e+02  .12630050e+03 -.50295355e+03  .84737075e+03  .24682123e-02", 
" -.19904544e+02  .12461866e+03 -.48170205e+03  .64874404e+03  .24662204e-02", 
" -.19785555e+02  .12312160e+03 -.47195678e+03  .63163123e+03  .24717314e-02", 
" -.19666001e+02  .12158645e+03 -.45937057e+03  .55994926e+03  .24865502e-02", 
" -.19545555e+02  .12003303e+03 -.44559634e+03  .46607685e+03  .24849398e-02", 
" -.19425576e+02  .11869673e+03 -.44475787e+03  .58432565e+03  .24432014e-02", 
" -.19304799e+02  .11717547e+03 -.43415906e+03  .56533182e+03  .24042290e-02", 
" -.19182334e+02  .11545798e+03 -.41447216e+03  .41044223e+03  .24196888e-02", 
" -.19059892e+02  .11382792e+03 -.39703883e+03  .27055276e+03  .24234937e-02", 
" -.18937581e+02  .11238365e+03 -.39070840e+03  .29862990e+03  .24444126e-02", 
" -.18813676e+02  .11067791e+03 -.36992031e+03  .12251416e+03  .24091564e-02", 
" -.18690449e+02  .10930314e+03 -.36740730e+03  .20889456e+03  .24076994e-02", 
" -.18565985e+02  .10777120e+03 -.35715865e+03  .20269615e+03  .23894165e-02", 
" -.18440744e+02  .10624158e+03 -.34745430e+03  .18889643e+03  .23640916e-02", 
" -.18314943e+02  .10462600e+03 -.33014633e+03  .49564103e+02  .23324474e-02", 
" -.18188489e+02  .10304840e+03 -.31556129e+03 -.58371086e+02  .23091488e-02", 
" -.18061084e+02  .10153421e+03 -.30624006e+03 -.71027178e+02  .23296621e-02", 
" -.17931888e+02  .99728366e+02 -.28009345e+03 -.33885848e+03  .23539802e-02", 
" -.17802960e+02  .98209667e+02 -.26993345e+03 -.36608258e+03  .23584851e-02", 
" -.17672859e+02  .96680950e+02 -.25994166e+03 -.39649396e+03  .23599299e-02", 
" -.17542021e+02  .95116623e+02 -.24675012e+03 -.48462144e+03  .23446336e-02", 
" -.17410013e+02  .93475503e+02 -.22872501e+03 -.64752304e+03  .23274048e-02", 
" -.17277163e+02  .92003218e+02 -.22246458e+03 -.61607954e+03  .23262061e-02", 
" -.17142201e+02  .90245946e+02 -.20213091e+03 -.79028402e+03  .23506588e-02", 
" -.17006707e+02  .88656293e+02 -.19093038e+03 -.82770749e+03  .23850346e-02", 
" -.16870107e+02  .87153902e+02 -.18399801e+03 -.81758360e+03  .23793473e-02", 
" -.16731089e+02  .85346774e+02 -.16095817e+03 -.10376043e+04  .23771638e-02", 
" -.16591736e+02  .83781401e+02 -.15068611e+03 -.10677540e+04  .23624625e-02", 
" -.16449881e+02  .82128361e+02 -.13768430e+03 -.11334611e+04  .23730852e-02", 
" -.16306742e+02  .80582136e+02 -.13231672e+03 -.10688864e+04  .23842598e-02", 
" -.16161718e+02  .78998995e+02 -.12323266e+03 -.11046627e+04  .24053109e-02", 
" -.16015482e+02  .77473265e+02 -.12003806e+03 -.10020659e+04  .24460479e-02", 
" -.15867203e+02  .75882848e+02 -.11035742e+03 -.10448597e+04  .24499399e-02", 
" -.15715932e+02  .73998457e+02 -.84672585e+02 -.13207359e+04  .24422475e-02", 
" -.15563097e+02  .72418294e+02 -.77664551e+02 -.13178517e+04  .24362347e-02", 
" -.15408168e+02  .70886783e+02 -.74653271e+02 -.12436484e+04  .24413311e-02", 
" -.15249486e+02  .69126317e+02 -.62456084e+02 -.12944920e+04  .24386142e-02", 
" -.15089001e+02  .67433171e+02 -.52604723e+02 -.13153663e+04  .24145443e-02", 
" -.14924833e+02  .65640933e+02 -.41038987e+02 -.13372387e+04  .24670248e-02", 
" -.14756364e+02  .63593237e+02 -.19088265e+02 -.14872701e+04  .24906571e-02", 
" -.14585068e+02  .61710806e+02 -.39966384e+01 -.15884399e+04  .24853825e-02", 
" -.14410799e+02  .59851441e+02  .97992738e+01 -.16694806e+04  .24925966e-02", 
" -.14231162e+02  .57655200e+02  .39385545e+02 -.19509463e+04  .24293161e-02", 
" -.14047994e+02  .55740841e+02  .48997888e+02 -.19164800e+04  .24111124e-02", 
" -.13861408e+02  .54047552e+02  .49316553e+02 -.17915177e+04  .24530703e-02", 
" -.13667892e+02  .52002172e+02  .65755522e+02 -.19177683e+04  .25279866e-02", 
" -.13469544e+02  .49953300e+02  .79976244e+02 -.19688303e+04  .25373699e-02", 
" -.13264157e+02  .47866810e+02  .94950612e+02 -.20640227e+04  .24929877e-02", 
" -.13051323e+02  .45641802e+02  .10962011e+03 -.21034373e+04  .25234765e-02", 
" -.12830312e+02  .43272038e+02  .13029594e+03 -.22189126e+04  .24804720e-02", 
" -.12601248e+02  .41110406e+02  .13623307e+03 -.21144050e+04  .25241775e-02", 
" -.12362210e+02  .38936715e+02  .14039447e+03 -.20058946e+04  .25535124e-02", 
" -.12111261e+02  .36556627e+02  .15055587e+03 -.19743900e+04  .26357924e-02", 
" -.11846711e+02  .33885426e+02  .17560839e+03 -.22092776e+04  .26531060e-02", 
" -.11565783e+02  .31413135e+02  .17486285e+03 -.19859445e+04  .27221089e-02", 
" -.11266705e+02  .28720506e+02  .17534228e+03 -.17088708e+04  .28214954e-02", 
" -.10945192e+02  .25921235e+02  .17491034e+03 -.14446038e+04  .27102544e-02", 
" -.10593955e+02  .22444481e+02  .19850580e+03 -.14917568e+04  .27623619e-02", 
" -.10205583e+02  .18183157e+02  .25683655e+03 -.21426532e+04  .28754646e-02", 
" -.97704743e+01  .13973388e+02  .28806186e+03 -.22881199e+04  .30721367e-02", 
" -.92674997e+01  .88847478e+01  .34358121e+03 -.28382828e+04  .31671873e-02", 
" -.86611361e+01  .25097280e+01  .41996764e+03 -.35894983e+04  .35065326e-02", 
" -.78827281e+01 -.45472741e+01  .44901330e+03 -.34774538e+04  .40478919e-02", 
" -.76911750e+01 -.68066321e+01  .48447234e+03 -.38399412e+04  .42414549e-02", 
" -.74806832e+01 -.96417323e+01  .54961351e+03 -.47358362e+04  .43521763e-02", 
" -.72492318e+01 -.12254529e+02  .58920290e+03 -.52290744e+04  .44242994e-02", 
" -.69901918e+01 -.15005594e+02  .61510691e+03 -.53893642e+04  .46546005e-02", 
" -.66906548e+01 -.18830964e+02  .67063051e+03 -.58219282e+04  .47901502e-02", 
" -.63395865e+01 -.22956663e+02  .72598962e+03 -.63999216e+04  .52492838e-02", 
" -.59073935e+01 -.27634929e+02  .75695208e+03 -.64232453e+04  .55771508e-02", 
" -.53420040e+01 -.33288484e+02  .79337765e+03 -.67049315e+04  .63361311e-02", 
" -.44356771e+01 -.43257094e+02  .80965736e+03 -.58053804e+04  .86493899e-02", 
" -.35805642e+01 -.57184958e+02  .99637096e+03 -.68341330e+04  .11514110e-01", 
" -.25812553e+01 -.67919990e+02  .94913741e+03 -.49483444e+04  .16655152e-01", 
" -.18800489e+01 -.79170224e+02  .10869588e+04 -.62987727e+04  .20086113e-01"
)


.urc5 = 
c("Copyright (C) James G. MacKinnon, 1995", "conc      4  5  2   25", 
" -.57885059e+01 -.39071507e+02 -.20555679e+03  .43097423e-02", 
" -.56354879e+01 -.35902634e+02 -.16868941e+03  .29400518e-02", 
" -.54290016e+01 -.31544753e+02 -.11866603e+03  .21262474e-02", 
" -.52627579e+01 -.28283669e+02 -.89946719e+02  .15088148e-02", 
" -.50857933e+01 -.24924129e+02 -.74081098e+02  .10701356e-02", 
" -.49777684e+01 -.23071386e+02 -.63331955e+02  .91760847e-03", 
" -.48995123e+01 -.21654018e+02 -.59376552e+02  .86057909e-03", 
" -.48360275e+01 -.20747091e+02 -.52229223e+02  .77285276e-03", 
" -.47841526e+01 -.19942090e+02 -.47044507e+02  .70655500e-03", 
" -.47389996e+01 -.19270939e+02 -.43434244e+02  .66078408e-03", 
" -.46994425e+01 -.18671327e+02 -.41020363e+02  .61950540e-03", 
" -.46642684e+01 -.18183478e+02 -.37424392e+02  .58820442e-03", 
" -.46318637e+01 -.17806878e+02 -.32997069e+02  .56849907e-03", 
" -.45053348e+01 -.15987654e+02 -.26826011e+02  .49246271e-03", 
" -.44110360e+01 -.14828608e+02 -.20057599e+02  .44683881e-03", 
" -.43357380e+01 -.13845400e+02 -.17676759e+02  .39202717e-03", 
" -.42721927e+01 -.13110129e+02 -.14079172e+02  .38370330e-03", 
" -.42171363e+01 -.12439036e+02 -.12749822e+02  .35485454e-03", 
" -.41683451e+01 -.11876086e+02 -.11367457e+02  .33383424e-03", 
" -.41244707e+01 -.11371884e+02 -.10312739e+02  .32894886e-03", 
" -.40843088e+01 -.10917322e+02 -.98766185e+01  .31606395e-03", 
" -.40472074e+01 -.10533694e+02 -.88677023e+01  .29156314e-03", 
" -.40128794e+01 -.10159361e+02 -.83122768e+01  .28945046e-03", 
" -.39807420e+01 -.98478930e+01 -.69338776e+01  .28102283e-03", 
" -.39505754e+01 -.95282665e+01 -.67422462e+01  .27230467e-03", 
" -.39219412e+01 -.92564731e+01 -.59243156e+01  .26428430e-03", 
" -.38948958e+01 -.89845031e+01 -.56620567e+01  .26112476e-03", 
" -.38691800e+01 -.87231189e+01 -.54948340e+01  .26172966e-03", 
" -.38444736e+01 -.84981287e+01 -.48019345e+01  .25343436e-03", 
" -.38209561e+01 -.82763536e+01 -.45136115e+01  .24857462e-03", 
" -.37983147e+01 -.80711678e+01 -.40165306e+01  .24578654e-03", 
" -.37764352e+01 -.78842232e+01 -.34513575e+01  .24491536e-03", 
" -.37553972e+01 -.76889446e+01 -.33682353e+01  .23921778e-03", 
" -.37349671e+01 -.75087052e+01 -.32138443e+01  .23401717e-03", 
" -.37152154e+01 -.73363328e+01 -.30142454e+01  .23236287e-03", 
" -.36961092e+01 -.71684337e+01 -.28705892e+01  .22888460e-03", 
" -.36774983e+01 -.70134284e+01 -.26073129e+01  .22784321e-03", 
" -.36593244e+01 -.68749622e+01 -.20999562e+01  .22566505e-03", 
" -.36416542e+01 -.67348828e+01 -.18206084e+01  .22764052e-03", 
" -.36244584e+01 -.66002883e+01 -.13660534e+01  .22388576e-03", 
" -.36076843e+01 -.64681674e+01 -.11532211e+01  .22253246e-03", 
" -.35912860e+01 -.63440091e+01 -.81614361e+00  .21827529e-03", 
" -.35752936e+01 -.62075144e+01 -.10023819e+01  .21426358e-03", 
" -.35596113e+01 -.60887763e+01 -.76354759e+00  .21182131e-03", 
" -.35442417e+01 -.59724316e+01 -.57916681e+00  .21202140e-03", 
" -.35291855e+01 -.58570679e+01 -.50807979e+00  .21120945e-03", 
" -.35143565e+01 -.57509602e+01 -.31903889e+00  .20875622e-03", 
" -.34998517e+01 -.56472603e+01 -.89761461e-01  .20870275e-03", 
" -.34855518e+01 -.55433899e+01 -.69719232e-02  .20653826e-03", 
" -.34715484e+01 -.54413740e+01  .17035005e+00  .20627661e-03", 
" -.34577960e+01 -.53319751e+01 -.25584977e-01  .20544961e-03", 
" -.34441987e+01 -.52391752e+01  .22476683e+00  .20488057e-03", 
" -.34308289e+01 -.51472821e+01  .38275916e+00  .20389997e-03", 
" -.34176593e+01 -.50548888e+01  .44026508e+00  .20375586e-03", 
" -.34046492e+01 -.49578461e+01  .33210369e+00  .20424424e-03", 
" -.33918187e+01 -.48780109e+01  .69069956e+00  .20311650e-03", 
" -.33791885e+01 -.47893034e+01  .67329759e+00  .19858058e-03", 
" -.33666325e+01 -.47109979e+01  .81305245e+00  .19579271e-03", 
" -.33543111e+01 -.46269393e+01  .83800188e+00  .19454414e-03", 
" -.33420761e+01 -.45562637e+01  .10952086e+01  .19153468e-03", 
" -.33300241e+01 -.44754134e+01  .10877856e+01  .18940780e-03", 
" -.33181181e+01 -.44015344e+01  .12471382e+01  .18973507e-03", 
" -.33063438e+01 -.43293987e+01  .13569551e+01  .18878173e-03", 
" -.32946970e+01 -.42562361e+01  .14375336e+01  .18906792e-03", 
" -.32831328e+01 -.41859559e+01  .15624973e+01  .18877477e-03", 
" -.32717344e+01 -.41139632e+01  .15737727e+01  .18934018e-03", 
" -.32604507e+01 -.40414907e+01  .15239759e+01  .18994188e-03", 
" -.32492645e+01 -.39737746e+01  .15909188e+01  .18888827e-03", 
" -.32381225e+01 -.39073130e+01  .16261584e+01  .18943448e-03", 
" -.32270623e+01 -.38486217e+01  .18186094e+01  .18917638e-03", 
" -.32161314e+01 -.37881812e+01  .19604028e+01  .18815947e-03", 
" -.32053137e+01 -.37199575e+01  .18592892e+01  .18902382e-03", 
" -.31945576e+01 -.36542596e+01  .17471024e+01  .18859616e-03", 
" -.31839232e+01 -.35886075e+01  .16696484e+01  .18764120e-03", 
" -.31732726e+01 -.35369122e+01  .18747647e+01  .18545062e-03", 
" -.31627632e+01 -.34740133e+01  .18093654e+01  .18444462e-03", 
" -.31523373e+01 -.34161535e+01  .18517703e+01  .18575708e-03", 
" -.31419371e+01 -.33631321e+01  .19777156e+01  .18518478e-03", 
" -.31316408e+01 -.33037709e+01  .19404646e+01  .18350598e-03", 
" -.31213779e+01 -.32499686e+01  .19784694e+01  .18271622e-03", 
" -.31112093e+01 -.31968761e+01  .20467497e+01  .18143274e-03", 
" -.31010476e+01 -.31495191e+01  .22414758e+01  .17895941e-03", 
" -.30909966e+01 -.30939494e+01  .22072501e+01  .17832962e-03", 
" -.30809870e+01 -.30422062e+01  .22641901e+01  .17528546e-03", 
" -.30710122e+01 -.29923699e+01  .23132801e+01  .17498149e-03", 
" -.30610681e+01 -.29449280e+01  .23926062e+01  .17619447e-03", 
" -.30512028e+01 -.28956146e+01  .24310549e+01  .17476317e-03", 
" -.30413802e+01 -.28419900e+01  .22945216e+01  .17177140e-03", 
" -.30315783e+01 -.27974901e+01  .24311974e+01  .17131109e-03", 
" -.30217758e+01 -.27583016e+01  .25947307e+01  .16945104e-03", 
" -.30120736e+01 -.27116685e+01  .26167920e+01  .16720419e-03", 
" -.30023994e+01 -.26682236e+01  .27176970e+01  .16672460e-03", 
" -.29927760e+01 -.26237470e+01  .27696316e+01  .16821289e-03", 
" -.29831374e+01 -.25882883e+01  .30779612e+01  .16804391e-03", 
" -.29735708e+01 -.25404379e+01  .29537528e+01  .16954780e-03", 
" -.29639949e+01 -.25046762e+01  .31881644e+01  .16786798e-03", 
" -.29544319e+01 -.24659505e+01  .32978749e+01  .16702593e-03", 
" -.29449319e+01 -.24266851e+01  .34029364e+01  .16803857e-03", 
" -.29354599e+01 -.23829595e+01  .33153170e+01  .16902577e-03", 
" -.29260112e+01 -.23404338e+01  .33260620e+01  .16821311e-03", 
" -.29165638e+01 -.23014302e+01  .33406328e+01  .16812184e-03", 
" -.29071855e+01 -.22581284e+01  .32607068e+01  .16715700e-03", 
" -.28978430e+01 -.22141718e+01  .31849874e+01  .16762915e-03", 
" -.28884607e+01 -.21767584e+01  .32402027e+01  .16862802e-03", 
" -.28791091e+01 -.21359191e+01  .32489440e+01  .16972505e-03", 
" -.28697440e+01 -.21009038e+01  .33740287e+01  .17046585e-03", 
" -.28603970e+01 -.20634227e+01  .33619574e+01  .17161453e-03", 
" -.28510934e+01 -.20243334e+01  .33440571e+01  .17129830e-03", 
" -.28417502e+01 -.19927294e+01  .35163692e+01  .17140845e-03", 
" -.28324786e+01 -.19491886e+01  .33856254e+01  .17190723e-03", 
" -.28231902e+01 -.19134282e+01  .34453963e+01  .17238466e-03", 
" -.28138851e+01 -.18762077e+01  .33709464e+01  .17444119e-03", 
" -.28045797e+01 -.18445435e+01  .35078817e+01  .17710298e-03", 
" -.27952790e+01 -.18119398e+01  .35895120e+01  .17930442e-03", 
" -.27860404e+01 -.17701436e+01  .34287674e+01  .17714395e-03", 
" -.27767324e+01 -.17327752e+01  .33500711e+01  .17862977e-03", 
" -.27674466e+01 -.16989412e+01  .33743416e+01  .17854242e-03", 
" -.27581174e+01 -.16689369e+01  .34594916e+01  .17859371e-03", 
" -.27487997e+01 -.16311158e+01  .33311286e+01  .18009558e-03", 
" -.27394948e+01 -.15960490e+01  .33097944e+01  .18159746e-03", 
" -.27302001e+01 -.15630936e+01  .33316477e+01  .18176800e-03", 
" -.27209061e+01 -.15242745e+01  .31673390e+01  .18074259e-03", 
" -.27115863e+01 -.14867929e+01  .29931636e+01  .18297190e-03", 
" -.27022226e+01 -.14545945e+01  .29795705e+01  .18293831e-03", 
" -.26928836e+01 -.14202622e+01  .29162285e+01  .18404288e-03", 
" -.26834843e+01 -.13863690e+01  .27776283e+01  .18316013e-03", 
" -.26740506e+01 -.13617125e+01  .29375339e+01  .18444805e-03", 
" -.26645773e+01 -.13300346e+01  .28612201e+01  .18239536e-03", 
" -.26551272e+01 -.12993966e+01  .28505975e+01  .18293359e-03", 
" -.26456406e+01 -.12674982e+01  .27952715e+01  .18293152e-03", 
" -.26361212e+01 -.12382309e+01  .28083332e+01  .18332201e-03", 
" -.26265795e+01 -.12076143e+01  .27640307e+01  .18291345e-03", 
" -.26170377e+01 -.11749522e+01  .26707738e+01  .18364621e-03", 
" -.26074146e+01 -.11520687e+01  .28097652e+01  .18236795e-03", 
" -.25977520e+01 -.11275750e+01  .28797873e+01  .18222861e-03", 
" -.25880782e+01 -.11000500e+01  .28758564e+01  .18194951e-03", 
" -.25783611e+01 -.10724946e+01  .28656850e+01  .18136387e-03", 
" -.25686035e+01 -.10451267e+01  .28645251e+01  .18287468e-03", 
" -.25587812e+01 -.10208222e+01  .29017655e+01  .18326383e-03", 
" -.25489737e+01 -.99023262e+00  .28613712e+01  .18315802e-03", 
" -.25390478e+01 -.97068573e+00  .30219531e+01  .18081608e-03", 
" -.25291157e+01 -.94606251e+00  .30763441e+01  .18114830e-03", 
" -.25191219e+01 -.91800908e+00  .29810748e+01  .18167269e-03", 
" -.25090797e+01 -.89336296e+00  .30308138e+01  .18181714e-03", 
" -.24989575e+01 -.86890641e+00  .30015280e+01  .18273968e-03", 
" -.24887744e+01 -.84536518e+00  .30108517e+01  .18085157e-03", 
" -.24785218e+01 -.82539797e+00  .31045606e+01  .18126602e-03", 
" -.24682757e+01 -.79950252e+00  .31265695e+01  .18097617e-03", 
" -.24579018e+01 -.77661332e+00  .31318872e+01  .18152087e-03", 
" -.24474850e+01 -.75136203e+00  .30738512e+01  .18310283e-03", 
" -.24369722e+01 -.72361021e+00  .29438639e+01  .18072629e-03", 
" -.24263621e+01 -.70421852e+00  .29651677e+01  .18610281e-03", 
" -.24156874e+01 -.68586875e+00  .30930413e+01  .18823339e-03", 
" -.24049579e+01 -.66527006e+00  .31756887e+01  .18797573e-03", 
" -.23940628e+01 -.64956357e+00  .33084937e+01  .18566401e-03", 
" -.23831429e+01 -.62490550e+00  .32413836e+01  .18408134e-03", 
" -.23721065e+01 -.60235332e+00  .31802640e+01  .18490920e-03", 
" -.23610049e+01 -.57813046e+00  .31274002e+01  .18341625e-03", 
" -.23497265e+01 -.56253977e+00  .33047649e+01  .18266261e-03", 
" -.23383915e+01 -.54422333e+00  .33757586e+01  .18219696e-03", 
" -.23269405e+01 -.52228425e+00  .33096775e+01  .18625171e-03", 
" -.23153785e+01 -.49982691e+00  .32793443e+01  .18715199e-03", 
" -.23035773e+01 -.48443510e+00  .33165325e+01  .18637830e-03", 
" -.22917066e+01 -.46691834e+00  .33744143e+01  .18604082e-03", 
" -.22797617e+01 -.43965896e+00  .31598182e+01  .18714139e-03", 
" -.22675826e+01 -.42455915e+00  .32363744e+01  .19166820e-03", 
" -.22553088e+01 -.40244029e+00  .31212212e+01  .19157004e-03", 
" -.22428347e+01 -.39081347e+00  .33574830e+01  .19226658e-03", 
" -.22301947e+01 -.37351371e+00  .33622023e+01  .19782265e-03", 
" -.22173340e+01 -.36228590e+00  .35706961e+01  .19854540e-03", 
" -.22042962e+01 -.34529275e+00  .35168614e+01  .19955267e-03", 
" -.21910249e+01 -.33454122e+00  .36357230e+01  .19899756e-03", 
" -.21775881e+01 -.31846498e+00  .36348290e+01  .20016756e-03", 
" -.21638697e+01 -.31105988e+00  .38423677e+01  .20224288e-03", 
" -.21499629e+01 -.29635702e+00  .38472462e+01  .20342833e-03", 
" -.21357535e+01 -.28823448e+00  .40284205e+01  .20462593e-03", 
" -.21213743e+01 -.26937288e+00  .39584501e+01  .20470919e-03", 
" -.21066376e+01 -.26460009e+00  .42034373e+01  .20464735e-03", 
" -.20916067e+01 -.25438258e+00  .42580134e+01  .20778427e-03", 
" -.20763057e+01 -.23568861e+00  .41090614e+01  .20609343e-03", 
" -.20606979e+01 -.21372414e+00  .39219804e+01  .21049394e-03", 
" -.20446515e+01 -.19449760e+00  .37145340e+01  .20921236e-03", 
" -.20281737e+01 -.18683155e+00  .37467852e+01  .21283059e-03", 
" -.20112915e+01 -.17689911e+00  .37234108e+01  .21611727e-03", 
" -.19938998e+01 -.17102973e+00  .38268516e+01  .21751572e-03", 
" -.19760439e+01 -.16655191e+00  .39698859e+01  .21962493e-03", 
" -.19576160e+01 -.16277001e+00  .40201644e+01  .22535767e-03", 
" -.19386483e+01 -.15674608e+00  .40284148e+01  .22688018e-03", 
" -.19190140e+01 -.15814205e+00  .41845117e+01  .22969142e-03", 
" -.18985948e+01 -.16252029e+00  .43455439e+01  .23266327e-03", 
" -.18775733e+01 -.16609751e+00  .46382246e+01  .23573220e-03", 
" -.18556722e+01 -.16814927e+00  .47799896e+01  .24219129e-03", 
" -.18327977e+01 -.18059387e+00  .51449053e+01  .24399726e-03", 
" -.18088824e+01 -.18345677e+00  .51251502e+01  .25035489e-03", 
" -.17836556e+01 -.21071190e+00  .55783666e+01  .26245508e-03", 
" -.17573048e+01 -.21535943e+00  .55399465e+01  .26524974e-03", 
" -.17296092e+01 -.20596826e+00  .51997376e+01  .27820786e-03", 
" -.16998186e+01 -.23534458e+00  .56687699e+01  .28400361e-03", 
" -.16681000e+01 -.26085963e+00  .59101154e+01  .28501159e-03", 
" -.16340986e+01 -.26914785e+00  .55732750e+01  .29483785e-03", 
" -.15968761e+01 -.32322174e+00  .60514205e+01  .30526574e-03", 
" -.15564108e+01 -.37220589e+00  .65152095e+01  .31610967e-03", 
" -.15114308e+01 -.44162127e+00  .71828099e+01  .33284644e-03", 
" -.14610372e+01 -.49608380e+00  .71409113e+01  .35703859e-03", 
" -.14034914e+01 -.57657321e+00  .75825689e+01  .37764175e-03", 
" -.13356418e+01 -.70062005e+00  .86357756e+01  .40878129e-03", 
" -.12532173e+01 -.84137096e+00  .91145782e+01  .44932843e-03", 
" -.11473970e+01 -.10575930e+01  .10264868e+02  .51397228e-03", 
" -.99838515e+00 -.13730559e+01  .11424040e+02  .65564465e-03", 
" -.95966969e+00 -.14586520e+01  .11906837e+02  .66889427e-03", 
" -.91658161e+00 -.15784712e+01  .12958981e+02  .70427212e-03", 
" -.86820635e+00 -.16450504e+01  .13151457e+02  .74828819e-03", 
" -.81228209e+00 -.17078379e+01  .12368390e+02  .82848123e-03", 
" -.74573736e+00 -.18722744e+01  .13863436e+02  .89966530e-03", 
" -.66519161e+00 -.20365906e+01  .14912183e+02  .10089407e-02", 
" -.56214617e+00 -.21489167e+01  .13898576e+02  .11463820e-02", 
" -.41806588e+00 -.23756603e+01  .14475062e+02  .13954311e-02", 
" -.17757756e+00 -.23450466e+01  .59399199e+01  .19603404e-02", 
"  .58331499e-01 -.29898448e+01  .17602684e+02  .25037550e-02", 
"  .35014473e+00 -.23314784e+01  .10595491e+01  .41737744e-02", 
"  .56005862e+00 -.12904604e+01 -.25687595e+02  .54040459e-02", 
"coc       4  6  3   20", " -.60968042e+01 -.47756448e+02 -.12014599e+03 -.20430656e+04  .53240543e-02", 
" -.59512257e+01 -.42871825e+02 -.15172419e+03 -.74492638e+03  .38146292e-02", 
" -.57448023e+01 -.37435945e+02 -.14006737e+03 -.23202501e+03  .25493176e-02", 
" -.55825430e+01 -.32919164e+02 -.15792153e+03  .46565711e+03  .18281221e-02", 
" -.54078140e+01 -.29713055e+02 -.10423071e+03  .55538228e+02  .12685761e-02", 
" -.52998174e+01 -.27935590e+02 -.77369665e+02 -.84100873e+02  .10737111e-02", 
" -.52214912e+01 -.26489583e+02 -.66005129e+02 -.12736671e+03  .97204520e-03", 
" -.51595944e+01 -.25177549e+02 -.69187700e+02  .17473638e+01  .92889816e-03", 
" -.51075958e+01 -.24350458e+02 -.59811933e+02 -.44907409e+02  .85758663e-03", 
" -.50631632e+01 -.23579375e+02 -.56735145e+02 -.21777381e+02  .80112250e-03", 
" -.50237027e+01 -.23013195e+02 -.47940263e+02 -.72548970e+02  .75018044e-03", 
" -.49886071e+01 -.22383832e+02 -.49861519e+02  .22301531e+02  .72338996e-03", 
" -.49568076e+01 -.21864912e+02 -.47703031e+02  .36855267e+02  .69819530e-03", 
" -.48314035e+01 -.19812529e+02 -.41950825e+02  .92421181e+02  .55589403e-03", 
" -.47384614e+01 -.18514207e+02 -.27826481e+02 -.17090310e+02  .50143388e-03", 
" -.46637996e+01 -.17399461e+02 -.25946546e+02  .27226006e+02  .44877125e-03", 
" -.46009230e+01 -.16538357e+02 -.22213833e+02  .30772050e+02  .42128965e-03", 
" -.45462792e+01 -.15854769e+02 -.15967369e+02 -.16408269e+02  .41256840e-03", 
" -.44979106e+01 -.15226250e+02 -.13029053e+02 -.21847336e+02  .39688452e-03", 
" -.44545134e+01 -.14646496e+02 -.11469179e+02 -.15247440e+02  .37776421e-03", 
" -.44148512e+01 -.14115749e+02 -.10580178e+02 -.90368370e+01  .36209628e-03", 
" -.43780088e+01 -.13716513e+02 -.65672476e+01 -.39912644e+02  .34607772e-03", 
" -.43439588e+01 -.13279404e+02 -.67163801e+01 -.20617249e+02  .33956594e-03", 
" -.43120727e+01 -.12913259e+02 -.42941388e+01 -.44284868e+02  .33083957e-03", 
" -.42822278e+01 -.12540127e+02 -.44174075e+01 -.29485931e+02  .32118849e-03", 
" -.42538756e+01 -.12209894e+02 -.38101193e+01 -.25439519e+02  .31473944e-03", 
" -.42270627e+01 -.11898199e+02 -.29323348e+01 -.28769054e+02  .30207102e-03", 
" -.42015497e+01 -.11594832e+02 -.26635477e+01 -.27178117e+02  .29271198e-03", 
" -.41771939e+01 -.11308406e+02 -.24440094e+01 -.26472076e+02  .28959048e-03", 
" -.41538281e+01 -.11047338e+02 -.19965849e+01 -.24679223e+02  .28015150e-03", 
" -.41313475e+01 -.10785640e+02 -.21473460e+01 -.17208766e+02  .27642936e-03", 
" -.41097126e+01 -.10529192e+02 -.31575659e+01  .40139968e+01  .26872801e-03", 
" -.40887946e+01 -.10294663e+02 -.38487619e+01  .22875471e+02  .27125263e-03", 
" -.40685716e+01 -.10076897e+02 -.39051606e+01  .29933700e+02  .26541835e-03", 
" -.40490072e+01 -.98821561e+01 -.29687158e+01  .20231318e+02  .25984341e-03", 
" -.40299472e+01 -.96968821e+01 -.22634676e+01  .16556144e+02  .25354077e-03", 
" -.40114028e+01 -.95166879e+01 -.17494236e+01  .15782111e+02  .24815898e-03", 
" -.39934432e+01 -.93334591e+01 -.18331488e+01  .21657348e+02  .24563422e-03", 
" -.39759732e+01 -.91545804e+01 -.19305029e+01  .27551734e+02  .24969194e-03", 
" -.39589287e+01 -.90068372e+01 -.30748080e+00  .71255501e+01  .24747726e-03", 
" -.39422785e+01 -.88495055e+01  .30429314e+00  .32060664e+00  .24696421e-03", 
" -.39259523e+01 -.86977587e+01  .60566702e+00 -.86953657e-01  .24520617e-03", 
" -.39100092e+01 -.85563736e+01  .13692303e+01 -.80014016e+01  .24529335e-03", 
" -.38943862e+01 -.84088964e+01  .13229309e+01 -.40068772e+01  .24560718e-03", 
" -.38790991e+01 -.82807498e+01  .22215494e+01 -.12138391e+02  .24627112e-03", 
" -.38641923e+01 -.81385193e+01  .21541349e+01 -.72244488e+01  .24092273e-03", 
" -.38494893e+01 -.80078600e+01  .26732877e+01 -.14033231e+02  .23601571e-03", 
" -.38350770e+01 -.78856712e+01  .33780546e+01 -.22014041e+02  .23758457e-03", 
" -.38209102e+01 -.77605016e+01  .35385574e+01 -.21125663e+02  .23894113e-03", 
" -.38069762e+01 -.76322762e+01  .33827227e+01 -.16529081e+02  .23815378e-03", 
" -.37932518e+01 -.75172186e+01  .38058500e+01 -.21095555e+02  .23440469e-03", 
" -.37797239e+01 -.74140549e+01  .44921123e+01 -.27234158e+02  .23040480e-03", 
" -.37664668e+01 -.72767369e+01  .31826011e+01 -.59065096e+01  .22759947e-03", 
" -.37534128e+01 -.71553185e+01  .29048365e+01 -.15165330e+01  .22951545e-03", 
" -.37404758e+01 -.70498260e+01  .30348215e+01 -.80237308e+00  .22500156e-03", 
" -.37277489e+01 -.69342348e+01  .27693988e+01  .27789146e+01  .22536362e-03", 
" -.37152149e+01 -.68180813e+01  .21568180e+01  .13235903e+02  .22317808e-03", 
" -.37027374e+01 -.67312674e+01  .31087858e+01  .10095214e+01  .22233003e-03", 
" -.36904732e+01 -.66223343e+01  .25512184e+01  .10122284e+02  .22213654e-03", 
" -.36783499e+01 -.65265100e+01  .30519159e+01  .23554588e+01  .21902465e-03", 
" -.36664169e+01 -.64203568e+01  .25594706e+01  .11024473e+02  .21752870e-03", 
" -.36546018e+01 -.63238337e+01  .25201503e+01  .13361420e+02  .21541738e-03", 
" -.36428546e+01 -.62359638e+01  .27646133e+01  .11075280e+02  .21519794e-03", 
" -.36312221e+01 -.61639416e+01  .37839380e+01 -.65694697e+00  .21337634e-03", 
" -.36197147e+01 -.60748451e+01  .36594221e+01  .32049202e+01  .21461760e-03", 
" -.36083364e+01 -.59876916e+01  .36698753e+01  .44836559e+01  .21627488e-03", 
" -.35970851e+01 -.58982302e+01  .35970964e+01  .49027584e+01  .21415067e-03", 
" -.35859628e+01 -.58170922e+01  .40677756e+01 -.25827116e+01  .21189573e-03", 
" -.35748900e+01 -.57361508e+01  .41794679e+01 -.33247475e+01  .20815324e-03", 
" -.35639731e+01 -.56438604e+01  .38893412e+01 -.72457970e+00  .21085339e-03", 
" -.35531037e+01 -.55555449e+01  .33288508e+01  .83792921e+01  .21202436e-03", 
" -.35423804e+01 -.54694737e+01  .31617829e+01  .99955901e+01  .21242106e-03", 
" -.35316744e+01 -.53958772e+01  .34094683e+01  .68445745e+01  .21093134e-03", 
" -.35210512e+01 -.53225605e+01  .35315124e+01  .59943666e+01  .21212601e-03", 
" -.35104999e+01 -.52563352e+01  .40749315e+01 -.13405879e+01  .21294034e-03", 
" -.35000226e+01 -.51904097e+01  .44891628e+01 -.57423324e+01  .21187688e-03", 
" -.34896246e+01 -.51229926e+01  .48108951e+01 -.10008494e+02  .21309166e-03", 
" -.34793359e+01 -.50438942e+01  .45790735e+01 -.75423355e+01  .21256680e-03", 
" -.34690885e+01 -.49702540e+01  .44525847e+01 -.51120692e+01  .21281764e-03", 
" -.34589460e+01 -.48999508e+01  .47657877e+01 -.11890588e+02  .21509409e-03", 
" -.34487884e+01 -.48342639e+01  .50522844e+01 -.16938716e+02  .21567722e-03", 
" -.34387144e+01 -.47616054e+01  .45949621e+01 -.86219054e+01  .21645533e-03", 
" -.34286916e+01 -.46958077e+01  .44392824e+01 -.37285869e+01  .21546814e-03", 
" -.34187287e+01 -.46317993e+01  .44452241e+01 -.23256866e+01  .21518563e-03", 
" -.34087899e+01 -.45660600e+01  .44074865e+01 -.20183539e+01  .21387709e-03", 
" -.33988960e+01 -.45146155e+01  .50560193e+01 -.99760651e+01  .21208800e-03", 
" -.33890476e+01 -.44600453e+01  .55833156e+01 -.17795951e+02  .21146063e-03", 
" -.33792800e+01 -.43907466e+01  .52579797e+01 -.13296949e+02  .21037198e-03", 
" -.33695768e+01 -.43198984e+01  .47482833e+01 -.57232337e+01  .20830596e-03", 
" -.33598789e+01 -.42521886e+01  .42957429e+01  .12397629e+01  .20683652e-03", 
" -.33502106e+01 -.41947625e+01  .44560825e+01 -.12409495e+00  .20641719e-03", 
" -.33405841e+01 -.41354705e+01  .46068321e+01 -.33285409e+01  .20788522e-03", 
" -.33309780e+01 -.40736759e+01  .44142745e+01  .59576224e-01  .20587184e-03", 
" -.33214604e+01 -.40062005e+01  .38947884e+01  .71478027e+01  .20553522e-03", 
" -.33119428e+01 -.39402728e+01  .33790467e+01  .13963363e+02  .20421068e-03", 
" -.33024358e+01 -.38803325e+01  .30554800e+01  .19695058e+02  .20393722e-03", 
" -.32929286e+01 -.38332981e+01  .35075067e+01  .13601105e+02  .20547989e-03", 
" -.32834703e+01 -.37877623e+01  .41946105e+01  .31374706e+01  .20234824e-03", 
" -.32741104e+01 -.37269156e+01  .40160294e+01  .51533598e+01  .20191302e-03", 
" -.32646812e+01 -.36762000e+01  .41694899e+01  .33078137e+01  .19970985e-03", 
" -.32553152e+01 -.36183763e+01  .38819948e+01  .71070401e+01  .20141803e-03", 
" -.32459754e+01 -.35594744e+01  .35189650e+01  .12959171e+02  .20155672e-03", 
" -.32366240e+01 -.35105899e+01  .36453440e+01  .12616695e+02  .20314404e-03", 
" -.32272538e+01 -.34656969e+01  .39894962e+01  .77487651e+01  .20448615e-03", 
" -.32179005e+01 -.34302007e+01  .49569412e+01 -.54724301e+01  .20432629e-03", 
" -.32086306e+01 -.33739159e+01  .46977699e+01 -.26950992e+01  .20270426e-03", 
" -.31992965e+01 -.33301425e+01  .49796717e+01 -.57303364e+01  .20213919e-03", 
" -.31900420e+01 -.32764747e+01  .47887701e+01 -.36017491e+01  .20224080e-03", 
" -.31807409e+01 -.32420336e+01  .57642732e+01 -.18400700e+02  .20110350e-03", 
" -.31714647e+01 -.32014116e+01  .61477181e+01 -.22486896e+02  .19942006e-03", 
" -.31621873e+01 -.31612031e+01  .65817115e+01 -.27801723e+02  .20112523e-03", 
" -.31529494e+01 -.31122854e+01  .66312972e+01 -.29699815e+02  .20070235e-03", 
" -.31436463e+01 -.30663136e+01  .65669134e+01 -.28015449e+02  .20249761e-03", 
" -.31343347e+01 -.30280541e+01  .68903187e+01 -.31766876e+02  .20261991e-03", 
" -.31250538e+01 -.29865507e+01  .69675256e+01 -.31299205e+02  .20183478e-03", 
" -.31157715e+01 -.29471025e+01  .72197293e+01 -.33153822e+02  .20150039e-03", 
" -.31065932e+01 -.28878803e+01  .67664930e+01 -.29596458e+02  .19850982e-03", 
" -.30973224e+01 -.28351225e+01  .63428837e+01 -.23795592e+02  .19766036e-03", 
" -.30880325e+01 -.27964282e+01  .66196390e+01 -.26942765e+02  .19657476e-03", 
" -.30787520e+01 -.27581711e+01  .69774765e+01 -.32175059e+02  .19504304e-03", 
" -.30694263e+01 -.27153600e+01  .69925964e+01 -.32114163e+02  .19536082e-03", 
" -.30601348e+01 -.26702838e+01  .69531912e+01 -.32155555e+02  .19314975e-03", 
" -.30508587e+01 -.26166200e+01  .64018524e+01 -.25232297e+02  .19563226e-03", 
" -.30414849e+01 -.25780995e+01  .65478563e+01 -.26640934e+02  .19560952e-03", 
" -.30320823e+01 -.25431342e+01  .67799971e+01 -.28204831e+02  .19716990e-03", 
" -.30226953e+01 -.25030679e+01  .68366476e+01 -.28655337e+02  .19576734e-03", 
" -.30133029e+01 -.24634755e+01  .69556344e+01 -.31020388e+02  .19511749e-03", 
" -.30038835e+01 -.24219410e+01  .70160681e+01 -.32879596e+02  .19585949e-03", 
" -.29944398e+01 -.23813127e+01  .70597257e+01 -.33774935e+02  .19462707e-03", 
" -.29849553e+01 -.23509000e+01  .75441625e+01 -.39865416e+02  .19424076e-03", 
" -.29754853e+01 -.23127547e+01  .76432542e+01 -.41425453e+02  .19480342e-03", 
" -.29659351e+01 -.22789666e+01  .79065995e+01 -.44370574e+02  .19669206e-03", 
" -.29563760e+01 -.22454866e+01  .82631534e+01 -.49353037e+02  .19723347e-03", 
" -.29468214e+01 -.22005294e+01  .80715370e+01 -.48091546e+02  .19729022e-03", 
" -.29372288e+01 -.21598949e+01  .79363592e+01 -.46328021e+02  .19807849e-03", 
" -.29276168e+01 -.21179867e+01  .79035003e+01 -.46943071e+02  .20009015e-03", 
" -.29179265e+01 -.20826645e+01  .79894243e+01 -.47166381e+02  .19977789e-03", 
" -.29081572e+01 -.20499124e+01  .82550478e+01 -.50832508e+02  .20207845e-03", 
" -.28983767e+01 -.20142375e+01  .83093199e+01 -.51260737e+02  .20383134e-03", 
" -.28885516e+01 -.19813949e+01  .84226465e+01 -.52319915e+02  .20571413e-03", 
" -.28786401e+01 -.19630294e+01  .94284112e+01 -.65942660e+02  .20743096e-03", 
" -.28687453e+01 -.19138757e+01  .86385391e+01 -.54638370e+02  .21052771e-03", 
" -.28587753e+01 -.18839576e+01  .90091099e+01 -.58893981e+02  .20983798e-03", 
" -.28487024e+01 -.18639492e+01  .98242959e+01 -.69122102e+02  .21030398e-03", 
" -.28386103e+01 -.18351222e+01  .10225184e+02 -.74453588e+02  .20800826e-03", 
" -.28284810e+01 -.17972632e+01  .10195801e+02 -.75312665e+02  .21083870e-03", 
" -.28182900e+01 -.17556507e+01  .97807051e+01 -.69222264e+02  .21190761e-03", 
" -.28079697e+01 -.17242284e+01  .97358718e+01 -.67831442e+02  .21200248e-03", 
" -.27976099e+01 -.16861994e+01  .93767657e+01 -.61927979e+02  .21222618e-03", 
" -.27872348e+01 -.16416101e+01  .87356564e+01 -.51899002e+02  .21312131e-03", 
" -.27767435e+01 -.16168075e+01  .94018737e+01 -.61913092e+02  .21177943e-03", 
" -.27661852e+01 -.15875637e+01  .97611564e+01 -.68525356e+02  .21278685e-03", 
" -.27555495e+01 -.15453254e+01  .94536825e+01 -.66150144e+02  .21247042e-03", 
" -.27447554e+01 -.15237288e+01  .10091469e+02 -.75194404e+02  .21434059e-03", 
" -.27339710e+01 -.14855463e+01  .98223619e+01 -.71998997e+02  .21277240e-03", 
" -.27230535e+01 -.14511842e+01  .96789978e+01 -.69808191e+02  .21319229e-03", 
" -.27120416e+01 -.14168933e+01  .96439491e+01 -.70063731e+02  .21459473e-03", 
" -.27009069e+01 -.13814966e+01  .93477626e+01 -.65297770e+02  .21510434e-03", 
" -.26897004e+01 -.13544356e+01  .96320595e+01 -.68476505e+02  .21551108e-03", 
" -.26783130e+01 -.13302251e+01  .99956910e+01 -.73463583e+02  .21535806e-03", 
" -.26668825e+01 -.12899527e+01  .94482674e+01 -.66215454e+02  .21199142e-03", 
" -.26553669e+01 -.12478176e+01  .90180795e+01 -.61849337e+02  .21467100e-03", 
" -.26435961e+01 -.12133557e+01  .87110135e+01 -.58108540e+02  .21720767e-03", 
" -.26317845e+01 -.11761438e+01  .85718051e+01 -.58466237e+02  .21464683e-03", 
" -.26198014e+01 -.11451258e+01  .84835247e+01 -.57141056e+02  .21708938e-03", 
" -.26076410e+01 -.11164524e+01  .84791355e+01 -.56791893e+02  .21849872e-03", 
" -.25953874e+01 -.10858858e+01  .84010574e+01 -.55050151e+02  .21812715e-03", 
" -.25829425e+01 -.10525802e+01  .81805879e+01 -.52905666e+02  .21938529e-03", 
" -.25703388e+01 -.10162853e+01  .78874250e+01 -.50495710e+02  .22131830e-03", 
" -.25575258e+01 -.99129743e+00  .82029353e+01 -.55506635e+02  .21978727e-03", 
" -.25445234e+01 -.96174521e+00  .81480783e+01 -.55419166e+02  .21933109e-03", 
" -.25313894e+01 -.92395294e+00  .76021721e+01 -.47977463e+02  .21906946e-03", 
" -.25179359e+01 -.90141071e+00  .76748543e+01 -.48366127e+02  .21989527e-03", 
" -.25043400e+01 -.86640162e+00  .73011160e+01 -.44096092e+02  .22326365e-03", 
" -.24904245e+01 -.84074736e+00  .73896289e+01 -.46222906e+02  .22501518e-03", 
" -.24762287e+01 -.81693651e+00  .73015204e+01 -.43737155e+02  .22702142e-03", 
" -.24618433e+01 -.80840674e+00  .82559417e+01 -.56456540e+02  .22862823e-03", 
" -.24472055e+01 -.77221317e+00  .79786289e+01 -.55830190e+02  .23372538e-03", 
" -.24322389e+01 -.74634842e+00  .78055046e+01 -.52969488e+02  .23893303e-03", 
" -.24169651e+01 -.71647482e+00  .75376822e+01 -.49747675e+02  .23939696e-03", 
" -.24013255e+01 -.69206416e+00  .75852124e+01 -.51333062e+02  .24343446e-03", 
" -.23853214e+01 -.65907630e+00  .70672841e+01 -.44218745e+02  .24629799e-03", 
" -.23689534e+01 -.62572639e+00  .65126395e+01 -.37299245e+02  .24773889e-03", 
" -.23520705e+01 -.60745898e+00  .66219243e+01 -.37702302e+02  .25521495e-03", 
" -.23347228e+01 -.59659931e+00  .71265997e+01 -.43803542e+02  .25772804e-03", 
" -.23169719e+01 -.55756924e+00  .61229869e+01 -.29879323e+02  .25865097e-03", 
" -.22986202e+01 -.54738175e+00  .66520983e+01 -.37159807e+02  .26193131e-03", 
" -.22797998e+01 -.50746017e+00  .57994077e+01 -.28439117e+02  .26846413e-03", 
" -.22602227e+01 -.49075925e+00  .56092014e+01 -.24097500e+02  .27636223e-03", 
" -.22400126e+01 -.46791184e+00  .54754851e+01 -.22281143e+02  .28107165e-03", 
" -.22190042e+01 -.45396949e+00  .55475462e+01 -.22674718e+02  .28097728e-03", 
" -.21971002e+01 -.46558444e+00  .67710302e+01 -.36554334e+02  .28242296e-03", 
" -.21742629e+01 -.47000826e+00  .76490871e+01 -.48926463e+02  .28615753e-03", 
" -.21504922e+01 -.46041267e+00  .78232953e+01 -.51024496e+02  .29450399e-03", 
" -.21254514e+01 -.45213810e+00  .77152835e+01 -.47935534e+02  .30054151e-03", 
" -.20991985e+01 -.43957557e+00  .74859960e+01 -.44904971e+02  .31040731e-03", 
" -.20713200e+01 -.43967576e+00  .72882288e+01 -.38653751e+02  .32217461e-03", 
" -.20417022e+01 -.46304335e+00  .86941119e+01 -.56761811e+02  .33026094e-03", 
" -.20101392e+01 -.47530161e+00  .92358179e+01 -.63480707e+02  .33484831e-03", 
" -.19762128e+01 -.48719668e+00  .92323517e+01 -.61222666e+02  .34315970e-03", 
" -.19393316e+01 -.52909475e+00  .10654380e+02 -.78286902e+02  .34780534e-03", 
" -.18990493e+01 -.57605648e+00  .11937248e+02 -.92715448e+02  .36176000e-03", 
" -.18545528e+01 -.60679102e+00  .11628763e+02 -.84419874e+02  .39236584e-03", 
" -.18046825e+01 -.65290814e+00  .11935398e+02 -.87936271e+02  .41415776e-03", 
" -.17480077e+01 -.65254210e+00  .94139633e+01 -.58425178e+02  .44732622e-03", 
" -.16812117e+01 -.80798700e+00  .13029801e+02 -.97206177e+02  .49701814e-03", 
" -.15996793e+01 -.10250286e+01  .17836735e+02 -.15085919e+03  .55951647e-03", 
" -.14956455e+01 -.11733419e+01  .16874061e+02 -.13596275e+03  .63233067e-03", 
" -.13493609e+01 -.14756052e+01  .18177185e+02 -.13795916e+03  .77292997e-03", 
" -.13119306e+01 -.15110195e+01  .17085273e+02 -.12414195e+03  .82942699e-03", 
" -.12697640e+01 -.15631128e+01  .16324287e+02 -.11584446e+03  .87499721e-03", 
" -.12222542e+01 -.15663118e+01  .12033897e+02 -.56652975e+02  .92471059e-03", 
" -.11668176e+01 -.17932485e+01  .19639224e+02 -.16342988e+03  .10115781e-02", 
" -.11018876e+01 -.18414018e+01  .16793811e+02 -.13564435e+03  .10892039e-02", 
" -.10226604e+01 -.19900374e+01  .17686264e+02 -.14456051e+03  .11938157e-02", 
" -.92064528e+00 -.23227500e+01  .27171281e+02 -.27074857e+03  .14058543e-02", 
" -.77969153e+00 -.25619188e+01  .31022825e+02 -.33905590e+03  .17165836e-02", 
" -.54325897e+00 -.28882515e+01  .28667181e+02 -.26556239e+03  .23673326e-02", 
" -.31471801e+00 -.25454135e+01 -.20419840e+01  .90585385e+02  .32220092e-02", 
" -.27622157e-01 -.24424660e+01 -.13656018e+02  .14847540e+03  .46150187e-02", 
"  .17969241e+00 -.35465337e+01  .44836524e+02 -.57698612e+03  .60938167e-02", 
"coct      4  7  3   20", " -.63857088e+01 -.53545806e+02 -.23684242e+03 -.74377385e+03  .51585178e-02", 
" -.62368220e+01 -.49653704e+02 -.16956874e+03 -.11025920e+04  .37589567e-02", 
" -.60328578e+01 -.42804324e+02 -.20108632e+03  .78047789e+02  .24116487e-02", 
" -.58674540e+01 -.39354505e+02 -.14151426e+03 -.15397398e+03  .18156093e-02", 
" -.56962849e+01 -.35137620e+02 -.12388349e+03  .85749353e+02  .13289014e-02", 
" -.55911964e+01 -.32886748e+02 -.10530570e+03  .77881646e+02  .11449913e-02", 
" -.55134643e+01 -.31516491e+02 -.80777462e+02 -.11244379e+03  .10174993e-02", 
" -.54526299e+01 -.30120426e+02 -.80994699e+02  .21568444e+02  .92899548e-03", 
" -.54014255e+01 -.28983591e+02 -.80681040e+02  .91674706e+02  .84372905e-03", 
" -.53571856e+01 -.28165634e+02 -.76226905e+02  .13511541e+03  .79648799e-03", 
" -.53186503e+01 -.27353488e+02 -.74727900e+02  .16660054e+03  .75818631e-03", 
" -.52843962e+01 -.26598166e+02 -.76829885e+02  .25353048e+03  .72426092e-03", 
" -.52527542e+01 -.26072203e+02 -.71050330e+02  .22045442e+03  .70968910e-03", 
" -.51286343e+01 -.23748447e+02 -.64098029e+02  .28771941e+03  .60996100e-03", 
" -.50364094e+01 -.22190071e+02 -.55664421e+02  .27973100e+03  .56128423e-03", 
" -.49622167e+01 -.21060181e+02 -.44162917e+02  .18829083e+03  .53032794e-03", 
" -.48999423e+01 -.20088571e+02 -.37374393e+02  .14702853e+03  .46913131e-03", 
" -.48456866e+01 -.19305195e+02 -.31317561e+02  .11195976e+03  .45383703e-03", 
" -.47976341e+01 -.18596265e+02 -.27739577e+02  .10901776e+03  .42612805e-03", 
" -.47543525e+01 -.17989251e+02 -.23156661e+02  .78183469e+02  .39846693e-03", 
" -.47150602e+01 -.17417546e+02 -.20069495e+02  .57454535e+02  .37608115e-03", 
" -.46787709e+01 -.16901237e+02 -.18696403e+02  .66445149e+02  .35621238e-03", 
" -.46449077e+01 -.16435491e+02 -.17046487e+02  .66244402e+02  .33508862e-03", 
" -.46134243e+01 -.15986497e+02 -.16493246e+02  .76277739e+02  .33106012e-03", 
" -.45838379e+01 -.15564368e+02 -.15873750e+02  .79989604e+02  .33302407e-03", 
" -.45557649e+01 -.15178075e+02 -.15367284e+02  .88185194e+02  .32058869e-03", 
" -.45292176e+01 -.14819405e+02 -.14626916e+02  .90924431e+02  .30842664e-03", 
" -.45037869e+01 -.14530878e+02 -.11189796e+02  .55360434e+02  .29831908e-03", 
" -.44794304e+01 -.14245372e+02 -.89225510e+01  .37792093e+02  .28814000e-03", 
" -.44562707e+01 -.13930248e+02 -.89507360e+01  .45244318e+02  .28599958e-03", 
" -.44338446e+01 -.13682321e+02 -.67889490e+01  .30300543e+02  .28170184e-03", 
" -.44124459e+01 -.13381739e+02 -.79492164e+01  .53179237e+02  .28616828e-03", 
" -.43916909e+01 -.13142390e+02 -.61377009e+01  .34011357e+02  .28288687e-03", 
" -.43716499e+01 -.12896901e+02 -.58134960e+01  .39261991e+02  .27508334e-03", 
" -.43522454e+01 -.12653013e+02 -.60842095e+01  .49938164e+02  .27263417e-03", 
" -.43333918e+01 -.12421979e+02 -.64669350e+01  .64974385e+02  .26431222e-03", 
" -.43149902e+01 -.12237665e+02 -.45264453e+01  .45617045e+02  .26177091e-03", 
" -.42972123e+01 -.12011103e+02 -.52785002e+01  .60615343e+02  .26426632e-03", 
" -.42799144e+01 -.11818023e+02 -.42211349e+01  .48778677e+02  .26223793e-03", 
" -.42629992e+01 -.11623008e+02 -.42962189e+01  .57313249e+02  .26382361e-03", 
" -.42464546e+01 -.11444190e+02 -.33343436e+01  .47353809e+02  .26440635e-03", 
" -.42302463e+01 -.11290340e+02 -.17001611e+01  .31170916e+02  .25637328e-03", 
" -.42144301e+01 -.11125440e+02 -.10161876e+01  .27252843e+02  .25635217e-03", 
" -.41990328e+01 -.10940947e+02 -.13744359e+01  .34030926e+02  .25348318e-03", 
" -.41838182e+01 -.10792939e+02 -.26401412e+00  .23642703e+02  .25397037e-03", 
" -.41690570e+01 -.10614601e+02 -.88700564e+00  .35076602e+02  .25045643e-03", 
" -.41544898e+01 -.10472034e+02  .28618775e+00  .20916056e+02  .24792567e-03", 
" -.41401852e+01 -.10314871e+02  .17719417e+00  .26799918e+02  .24566025e-03", 
" -.41261776e+01 -.10163476e+02  .23503586e+00  .28569086e+02  .24274957e-03", 
" -.41122953e+01 -.10029896e+02  .79696221e+00  .23964527e+02  .23986780e-03", 
" -.40986679e+01 -.98919265e+01  .94441409e+00  .25048272e+02  .24342520e-03", 
" -.40852821e+01 -.97605875e+01  .16436257e+01  .16985533e+02  .23983697e-03", 
" -.40720632e+01 -.96413468e+01  .26844079e+01  .53406428e+01  .23489233e-03", 
" -.40590748e+01 -.95202491e+01  .33854818e+01 -.14381046e+01  .23634379e-03", 
" -.40462311e+01 -.93995848e+01  .38993256e+01 -.68996467e+01  .23556249e-03", 
" -.40336116e+01 -.92649082e+01  .33746396e+01  .28000238e+01  .23461600e-03", 
" -.40211313e+01 -.91458116e+01  .36422101e+01  .14237928e+01  .23144453e-03", 
" -.40088061e+01 -.90270070e+01  .37423214e+01  .27183517e+01  .22656520e-03", 
" -.39966182e+01 -.89111248e+01  .38963217e+01  .24771537e+01  .22481090e-03", 
" -.39845903e+01 -.88053543e+01  .45348412e+01 -.43725028e+01  .22166998e-03", 
" -.39726707e+01 -.87049487e+01  .51845359e+01 -.95593248e+01  .21520735e-03", 
" -.39609589e+01 -.85961071e+01  .52970593e+01 -.93906933e+01  .21100152e-03", 
" -.39493526e+01 -.84787316e+01  .49946850e+01 -.42174324e+01  .21373301e-03", 
" -.39378326e+01 -.83794365e+01  .52981823e+01 -.54185454e+01  .20954786e-03", 
" -.39264318e+01 -.82753479e+01  .52197582e+01 -.48169427e+00  .20871166e-03", 
" -.39151277e+01 -.81898663e+01  .62375467e+01 -.12064565e+02  .20823572e-03", 
" -.39039161e+01 -.80978010e+01  .68371935e+01 -.19419388e+02  .20963538e-03", 
" -.38929208e+01 -.79880751e+01  .63326123e+01 -.11372863e+02  .20873975e-03", 
" -.38819680e+01 -.78950951e+01  .67340832e+01 -.15371590e+02  .20869434e-03", 
" -.38712033e+01 -.77790056e+01  .58518186e+01 -.35148776e+01  .20814988e-03", 
" -.38604605e+01 -.76786400e+01  .58852367e+01 -.49340788e+01  .20830728e-03", 
" -.38497491e+01 -.75941848e+01  .64864573e+01 -.11831334e+02  .20691987e-03", 
" -.38391878e+01 -.74977011e+01  .62443523e+01 -.66445328e+01  .21007836e-03", 
" -.38286691e+01 -.74063020e+01  .63998103e+01 -.84771604e+01  .21085966e-03", 
" -.38182208e+01 -.73208333e+01  .66344011e+01 -.10987973e+02  .21027834e-03", 
" -.38078873e+01 -.72256677e+01  .63176290e+01 -.62367248e+01  .20962683e-03", 
" -.37975803e+01 -.71528341e+01  .71028952e+01 -.15214339e+02  .20842291e-03", 
" -.37873530e+01 -.70620662e+01  .66925905e+01 -.79736539e+01  .20985021e-03", 
" -.37772226e+01 -.69748769e+01  .66676666e+01 -.72232285e+01  .20951788e-03", 
" -.37671110e+01 -.68882447e+01  .64017878e+01 -.20885613e+01  .21051051e-03", 
" -.37570888e+01 -.67998021e+01  .60241915e+01  .47835803e+01  .21046295e-03", 
" -.37471417e+01 -.67054749e+01  .53576671e+01  .14295306e+02  .20774532e-03", 
" -.37372175e+01 -.66253713e+01  .53925724e+01  .15196936e+02  .20605607e-03", 
" -.37273626e+01 -.65426044e+01  .52341378e+01  .17749372e+02  .20363568e-03", 
" -.37175191e+01 -.64624744e+01  .50767205e+01  .21186937e+02  .20218094e-03", 
" -.37077393e+01 -.63725512e+01  .42462858e+01  .33864948e+02  .20034125e-03", 
" -.36979801e+01 -.62985060e+01  .43553780e+01  .33285794e+02  .20086467e-03", 
" -.36882907e+01 -.62216541e+01  .42435784e+01  .35604446e+02  .19970763e-03", 
" -.36786422e+01 -.61439984e+01  .39509317e+01  .40948212e+02  .19995227e-03", 
" -.36690332e+01 -.60786772e+01  .46759196e+01  .28730942e+02  .20020768e-03", 
" -.36594239e+01 -.60120466e+01  .47928483e+01  .29498659e+02  .20150671e-03", 
" -.36498658e+01 -.59468736e+01  .50545033e+01  .26958797e+02  .20119266e-03", 
" -.36403620e+01 -.58767364e+01  .50119536e+01  .28824333e+02  .19997954e-03", 
" -.36308854e+01 -.58030668e+01  .47394284e+01  .33592331e+02  .20229044e-03", 
" -.36214112e+01 -.57325180e+01  .45223964e+01  .38273908e+02  .20061371e-03", 
" -.36119760e+01 -.56584584e+01  .39961496e+01  .46847268e+02  .19979008e-03", 
" -.36026161e+01 -.55817306e+01  .35899773e+01  .51838620e+02  .19974773e-03", 
" -.35932124e+01 -.55242026e+01  .40957468e+01  .45682609e+02  .19813948e-03", 
" -.35839179e+01 -.54500230e+01  .37663044e+01  .49736926e+02  .19567953e-03", 
" -.35745680e+01 -.53848508e+01  .37314521e+01  .50092028e+02  .19623114e-03", 
" -.35652207e+01 -.53371183e+01  .47553489e+01  .35172593e+02  .19545599e-03", 
" -.35559605e+01 -.52747833e+01  .48861120e+01  .33839442e+02  .19588911e-03", 
" -.35467370e+01 -.52129535e+01  .51375725e+01  .29498027e+02  .19365668e-03", 
" -.35374883e+01 -.51528136e+01  .53101556e+01  .27171834e+02  .19456820e-03", 
" -.35283420e+01 -.50757799e+01  .46735440e+01  .35225679e+02  .19534348e-03", 
" -.35191504e+01 -.50037165e+01  .42107976e+01  .40883574e+02  .19664000e-03", 
" -.35099148e+01 -.49539527e+01  .48012638e+01  .33577181e+02  .19656686e-03", 
" -.35007124e+01 -.48977220e+01  .50122215e+01  .30942435e+02  .19583833e-03", 
" -.34914738e+01 -.48509181e+01  .56566622e+01  .23070907e+02  .19771045e-03", 
" -.34822719e+01 -.47969950e+01  .60547100e+01  .16958669e+02  .19794408e-03", 
" -.34730922e+01 -.47408914e+01  .62012341e+01  .15371555e+02  .19907522e-03", 
" -.34639536e+01 -.46742747e+01  .58114270e+01  .20088380e+02  .19572127e-03", 
" -.34547773e+01 -.46163224e+01  .58087073e+01  .20886872e+02  .19481358e-03", 
" -.34455673e+01 -.45751937e+01  .65361563e+01  .12965842e+02  .19314521e-03", 
" -.34364077e+01 -.45103167e+01  .60013469e+01  .21167772e+02  .19582305e-03", 
" -.34272462e+01 -.44547616e+01  .59817594e+01  .22274554e+02  .19636468e-03", 
" -.34180851e+01 -.43905867e+01  .54471560e+01  .29876683e+02  .19607559e-03", 
" -.34089107e+01 -.43348942e+01  .54762019e+01  .28451695e+02  .19459403e-03", 
" -.33997268e+01 -.42758367e+01  .52398143e+01  .31154902e+02  .19444829e-03", 
" -.33905208e+01 -.42123993e+01  .45896582e+01  .40560886e+02  .19782193e-03", 
" -.33812969e+01 -.41616914e+01  .46919812e+01  .39837653e+02  .19872996e-03", 
" -.33720837e+01 -.41088650e+01  .46813415e+01  .40661689e+02  .19893148e-03", 
" -.33628325e+01 -.40621426e+01  .50121435e+01  .36972955e+02  .19922841e-03", 
" -.33535839e+01 -.40014604e+01  .43988938e+01  .47480706e+02  .19880173e-03", 
" -.33443150e+01 -.39476995e+01  .42726027e+01  .49304047e+02  .20048020e-03", 
" -.33350458e+01 -.38945788e+01  .42473812e+01  .49115850e+02  .20077708e-03", 
" -.33257804e+01 -.38395133e+01  .42109809e+01  .48492975e+02  .20337139e-03", 
" -.33163972e+01 -.38041461e+01  .49004632e+01  .41009133e+02  .20296912e-03", 
" -.33070566e+01 -.37542156e+01  .50031406e+01  .38983073e+02  .20422079e-03", 
" -.32977096e+01 -.37106974e+01  .54678199e+01  .32236357e+02  .20128368e-03", 
" -.32883062e+01 -.36712524e+01  .60525809e+01  .24978582e+02  .19657957e-03", 
" -.32788975e+01 -.36229421e+01  .62528163e+01  .21710102e+02  .19686391e-03", 
" -.32694472e+01 -.35769329e+01  .63935774e+01  .20467697e+02  .19650830e-03", 
" -.32599811e+01 -.35228526e+01  .61164982e+01  .24222002e+02  .19696532e-03", 
" -.32504759e+01 -.34697959e+01  .57996149e+01  .28923143e+02  .19346329e-03", 
" -.32409723e+01 -.34226824e+01  .59792252e+01  .25620060e+02  .19252306e-03", 
" -.32313894e+01 -.33799175e+01  .61692466e+01  .23855619e+02  .19283777e-03", 
" -.32218112e+01 -.33248033e+01  .57993402e+01  .28398329e+02  .19424216e-03", 
" -.32121500e+01 -.32848664e+01  .62414328e+01  .21560395e+02  .19513837e-03", 
" -.32024378e+01 -.32369327e+01  .62643434e+01  .20323992e+02  .19406962e-03", 
" -.31926690e+01 -.32026113e+01  .69096302e+01  .11883176e+02  .19633313e-03", 
" -.31828028e+01 -.31719943e+01  .76229243e+01  .31393332e+01  .19562162e-03", 
" -.31729490e+01 -.31288188e+01  .78058218e+01  .19754941e+00  .19514597e-03", 
" -.31630580e+01 -.30822948e+01  .77648144e+01  .61913062e+00  .19591645e-03", 
" -.31530618e+01 -.30400240e+01  .79370743e+01 -.20814633e+01  .19601205e-03", 
" -.31430516e+01 -.30035145e+01  .84384321e+01 -.98208255e+01  .19565113e-03", 
" -.31329737e+01 -.29659951e+01  .88439339e+01 -.15359699e+02  .19776193e-03", 
" -.31228470e+01 -.29218636e+01  .88953828e+01 -.16440028e+02  .19743020e-03", 
" -.31126692e+01 -.28750416e+01  .88623802e+01 -.17172602e+02  .19792454e-03", 
" -.31023866e+01 -.28283048e+01  .85051028e+01 -.10389464e+02  .19791339e-03", 
" -.30920746e+01 -.27861603e+01  .88388763e+01 -.16465753e+02  .19700303e-03", 
" -.30816697e+01 -.27445220e+01  .89558131e+01 -.19151282e+02  .19555138e-03", 
" -.30711548e+01 -.27125317e+01  .95767579e+01 -.27961688e+02  .19640618e-03", 
" -.30605549e+01 -.26713915e+01  .96083158e+01 -.28320894e+02  .19749908e-03", 
" -.30498815e+01 -.26356694e+01  .99461338e+01 -.32751113e+02  .19399151e-03", 
" -.30390947e+01 -.25962994e+01  .99451526e+01 -.31499159e+02  .19232241e-03", 
" -.30282488e+01 -.25557393e+01  .10057076e+02 -.33312065e+02  .19507705e-03", 
" -.30173450e+01 -.25077416e+01  .97313112e+01 -.28722637e+02  .19765296e-03", 
" -.30062719e+01 -.24728209e+01  .10050323e+02 -.32260990e+02  .19902054e-03", 
" -.29951142e+01 -.24349347e+01  .10138059e+02 -.32974042e+02  .20100099e-03", 
" -.29837983e+01 -.24111815e+01  .11006542e+02 -.44521497e+02  .20255460e-03", 
" -.29723444e+01 -.23791569e+01  .11301321e+02 -.47722989e+02  .20543942e-03", 
" -.29608402e+01 -.23374408e+01  .11371677e+02 -.49898826e+02  .20760306e-03", 
" -.29492025e+01 -.22961473e+01  .11350087e+02 -.49828072e+02  .20722200e-03", 
" -.29373991e+01 -.22585690e+01  .11445445e+02 -.51286450e+02  .20915338e-03", 
" -.29254497e+01 -.22288742e+01  .11930083e+02 -.57759061e+02  .20837363e-03", 
" -.29133752e+01 -.21934147e+01  .12146273e+02 -.61072881e+02  .21074153e-03", 
" -.29011633e+01 -.21452768e+01  .11632017e+02 -.54278945e+02  .21067829e-03", 
" -.28888435e+01 -.20901614e+01  .10893799e+02 -.45424473e+02  .21260701e-03", 
" -.28763225e+01 -.20406566e+01  .10218756e+02 -.34587235e+02  .21256252e-03", 
" -.28634996e+01 -.20153011e+01  .10605339e+02 -.37059992e+02  .21407904e-03", 
" -.28505530e+01 -.19733236e+01  .10314474e+02 -.32929998e+02  .21856236e-03", 
" -.28373673e+01 -.19431697e+01  .10526203e+02 -.34512835e+02  .22185823e-03", 
" -.28239965e+01 -.19045910e+01  .10524948e+02 -.35378311e+02  .22070580e-03", 
" -.28104275e+01 -.18514246e+01  .96400667e+01 -.23981598e+02  .22115070e-03", 
" -.27964900e+01 -.18267967e+01  .98476265e+01 -.24189078e+02  .22416214e-03", 
" -.27823348e+01 -.17996319e+01  .10130992e+02 -.26135857e+02  .22320034e-03", 
" -.27679333e+01 -.17688402e+01  .10233065e+02 -.25966576e+02  .22811682e-03", 
" -.27531792e+01 -.17559183e+01  .11080384e+02 -.34971294e+02  .23080803e-03", 
" -.27383199e+01 -.16963191e+01  .99431647e+01 -.21193567e+02  .23010678e-03", 
" -.27229794e+01 -.16801791e+01  .10845581e+02 -.32106625e+02  .23068816e-03", 
" -.27073162e+01 -.16462919e+01  .10803440e+02 -.31697759e+02  .23342737e-03", 
" -.26912712e+01 -.16098156e+01  .10572919e+02 -.29849547e+02  .23350352e-03", 
" -.26748402e+01 -.15743305e+01  .10576999e+02 -.31699238e+02  .23735679e-03", 
" -.26579547e+01 -.15349917e+01  .10064569e+02 -.24672652e+02  .23737240e-03", 
" -.26405962e+01 -.15011575e+01  .98989695e+01 -.23655834e+02  .24092263e-03", 
" -.26227201e+01 -.14920881e+01  .11214601e+02 -.42011050e+02  .24589005e-03", 
" -.26043706e+01 -.14578240e+01  .10855820e+02 -.36327769e+02  .25362618e-03", 
" -.25854046e+01 -.14185570e+01  .10235779e+02 -.27757774e+02  .25857724e-03", 
" -.25658458e+01 -.13701757e+01  .91214638e+01 -.13512769e+02  .25947403e-03", 
" -.25454596e+01 -.13633197e+01  .10152724e+02 -.26928322e+02  .26129598e-03", 
" -.25242770e+01 -.13482752e+01  .10467071e+02 -.29808162e+02  .26328798e-03", 
" -.25022822e+01 -.13281578e+01  .10457093e+02 -.27503252e+02  .26621581e-03", 
" -.24791738e+01 -.13324865e+01  .11541644e+02 -.40874690e+02  .27499052e-03", 
" -.24550915e+01 -.13123083e+01  .11363732e+02 -.38009512e+02  .28338893e-03", 
" -.24297252e+01 -.13175774e+01  .12116412e+02 -.43617131e+02  .28534990e-03", 
" -.24029970e+01 -.13381889e+01  .13775787e+02 -.63179367e+02  .29298385e-03", 
" -.23747087e+01 -.13228570e+01  .12909097e+02 -.45630502e+02  .29941868e-03", 
" -.23444958e+01 -.13188357e+01  .13057704e+02 -.46765829e+02  .30681809e-03", 
" -.23122774e+01 -.13260080e+01  .13490678e+02 -.49003337e+02  .31961428e-03", 
" -.22773433e+01 -.13338617e+01  .13587561e+02 -.47567157e+02  .33503237e-03", 
" -.22392067e+01 -.13830957e+01  .15393941e+02 -.67341061e+02  .34185547e-03", 
" -.21971452e+01 -.14659368e+01  .18336041e+02 -.10141600e+03  .35134822e-03", 
" -.21503566e+01 -.14894059e+01  .16533477e+02 -.62670568e+02  .37853644e-03", 
" -.20969222e+01 -.16475053e+01  .20423848e+02 -.94506474e+02  .39816885e-03", 
" -.20344044e+01 -.18681621e+01  .25791622e+02 -.14236471e+03  .41655447e-03", 
" -.19595455e+01 -.20798956e+01  .29812715e+02 -.18473804e+03  .45841834e-03", 
" -.18648115e+01 -.22893626e+01  .29984228e+02 -.17448331e+03  .51391188e-03", 
" -.17328315e+01 -.26606444e+01  .29075287e+02 -.13028486e+03  .62755120e-03", 
" -.16990526e+01 -.27370253e+01  .27266728e+02 -.97270661e+02  .65878551e-03", 
" -.16609995e+01 -.28653330e+01  .27644458e+02 -.97627235e+02  .71366765e-03", 
" -.16176243e+01 -.30846313e+01  .30142572e+02 -.11215062e+03  .75494927e-03", 
" -.15681752e+01 -.33028781e+01  .32672931e+02 -.13520350e+03  .82878175e-03", 
" -.15099464e+01 -.35175159e+01  .33790005e+02 -.14053588e+03  .90067078e-03", 
" -.14386331e+01 -.39186037e+01  .41934975e+02 -.23985185e+03  .10211149e-02", 
" -.13461906e+01 -.43971806e+01  .44974677e+02 -.24084431e+03  .11392429e-02", 
" -.12196458e+01 -.47665103e+01  .37761346e+02 -.11897057e+03  .14549978e-02", 
" -.10073714e+01 -.55929043e+01  .33625590e+02  .17532908e+02  .19761592e-02", 
" -.79865323e+00 -.71358145e+01  .65633567e+02 -.34073945e+03  .27606638e-02", 
" -.53853688e+00 -.80922445e+01  .62692203e+02 -.38733306e+03  .40718640e-02", 
" -.34780936e+00 -.98910407e+01  .12043871e+03 -.12210692e+04  .53614161e-02", 
"coctt     4  8  3   20", " -.66583943e+01 -.58802422e+02 -.37821200e+03  .14423694e+02  .50303777e-02", 
" -.65083843e+01 -.54927416e+02 -.29874237e+03 -.18704917e+03  .39767802e-02", 
" -.63042309e+01 -.49449463e+02 -.21539271e+03 -.50570773e+03  .24928355e-02", 
" -.61418877e+01 -.44812245e+02 -.20367785e+03  .10394111e+03  .17821502e-02", 
" -.59668609e+01 -.41069246e+02 -.14288200e+03 -.16935763e+03  .13166284e-02", 
" -.58628809e+01 -.38615644e+02 -.12586402e+03 -.78045502e+01  .11685219e-02", 
" -.57864651e+01 -.36815832e+02 -.11128586e+03 -.45243195e+02  .10358554e-02", 
" -.57256298e+01 -.35275761e+02 -.11301680e+03  .13021700e+03  .95472793e-03", 
" -.56746633e+01 -.34085709e+02 -.11092375e+03  .21724942e+03  .90304511e-03", 
" -.56318117e+01 -.33111844e+02 -.10495808e+03  .21791231e+03  .84756237e-03", 
" -.55934400e+01 -.32328481e+02 -.97629841e+02  .18247911e+03  .80569022e-03", 
" -.55585573e+01 -.31737350e+02 -.86075765e+02  .91218923e+02  .75759911e-03", 
" -.55275628e+01 -.31022767e+02 -.86613470e+02  .15361916e+03  .73076829e-03", 
" -.54041746e+01 -.28568651e+02 -.73186501e+02  .19203727e+03  .60506262e-03", 
" -.53127219e+01 -.26819192e+02 -.65498746e+02  .23433185e+03  .53669880e-03", 
" -.52388243e+01 -.25495665e+02 -.58288545e+02  .24921509e+03  .50315137e-03", 
" -.51770913e+01 -.24444494e+02 -.48649445e+02  .18132842e+03  .47863386e-03", 
" -.51237214e+01 -.23523208e+02 -.42064157e+02  .14273471e+03  .43339351e-03", 
" -.50762470e+01 -.22721959e+02 -.38089529e+02  .13883152e+03  .40309395e-03", 
" -.50334676e+01 -.22024959e+02 -.34233832e+02  .13078687e+03  .39717870e-03", 
" -.49942599e+01 -.21409590e+02 -.30136783e+02  .11317726e+03  .38226094e-03", 
" -.49582794e+01 -.20795160e+02 -.29916218e+02  .13837680e+03  .37720607e-03", 
" -.49249019e+01 -.20251511e+02 -.28636687e+02  .14628912e+03  .36358364e-03", 
" -.48936352e+01 -.19767786e+02 -.25925057e+02  .12986844e+03  .35704361e-03", 
" -.48639871e+01 -.19351271e+02 -.22977150e+02  .11588823e+03  .35416544e-03", 
" -.48361816e+01 -.18925617e+02 -.22256010e+02  .12942403e+03  .35402013e-03", 
" -.48096560e+01 -.18556522e+02 -.19181104e+02  .10216023e+03  .33999217e-03", 
" -.47846101e+01 -.18182430e+02 -.17335396e+02  .84917887e+02  .33116606e-03", 
" -.47605643e+01 -.17820700e+02 -.16819784e+02  .92271375e+02  .32771532e-03", 
" -.47374971e+01 -.17496244e+02 -.15389529e+02  .86902681e+02  .32144319e-03", 
" -.47154264e+01 -.17149602e+02 -.16062492e+02  .10689570e+03  .31422229e-03", 
" -.46941466e+01 -.16833395e+02 -.16293731e+02  .12501823e+03  .30638193e-03", 
" -.46734711e+01 -.16567438e+02 -.14577342e+02  .11258044e+03  .29860055e-03", 
" -.46535381e+01 -.16304219e+02 -.13561242e+02  .11364205e+03  .28994538e-03", 
" -.46342537e+01 -.16037974e+02 -.12969205e+02  .11452968e+03  .28442405e-03", 
" -.46156392e+01 -.15783860e+02 -.12275637e+02  .11489849e+03  .27799226e-03", 
" -.45974119e+01 -.15570581e+02 -.10128336e+02  .95884877e+02  .27633116e-03", 
" -.45798406e+01 -.15306338e+02 -.11301531e+02  .12018538e+03  .27526575e-03", 
" -.45626036e+01 -.15076635e+02 -.11087771e+02  .12391048e+03  .27258153e-03", 
" -.45457921e+01 -.14871947e+02 -.97887152e+01  .11324264e+03  .26738977e-03", 
" -.45294329e+01 -.14642746e+02 -.10117684e+02  .12224219e+03  .26484333e-03", 
" -.45133862e+01 -.14451291e+02 -.90196249e+01  .11519734e+03  .26313620e-03", 
" -.44976856e+01 -.14273901e+02 -.73070484e+01  .96800744e+02  .25583793e-03", 
" -.44823738e+01 -.14080827e+02 -.69660914e+01  .99750815e+02  .24769914e-03", 
" -.44672714e+01 -.13923436e+02 -.48644108e+01  .75177744e+02  .24724465e-03", 
" -.44525099e+01 -.13741815e+02 -.46333483e+01  .77926962e+02  .24904342e-03", 
" -.44380481e+01 -.13569550e+02 -.39214138e+01  .71693223e+02  .24665744e-03", 
" -.44238721e+01 -.13401160e+02 -.32533442e+01  .65024699e+02  .24723733e-03", 
" -.44099358e+01 -.13237025e+02 -.26544640e+01  .60636960e+02  .24582672e-03", 
" -.43962561e+01 -.13067696e+02 -.27943965e+01  .68400167e+02  .24036207e-03", 
" -.43827699e+01 -.12912576e+02 -.22284108e+01  .64028154e+02  .23649723e-03", 
" -.43694387e+01 -.12767824e+02 -.15481353e+01  .60834156e+02  .23586022e-03", 
" -.43563471e+01 -.12626494e+02 -.67530607e+00  .52514976e+02  .23384951e-03", 
" -.43434197e+01 -.12485660e+02  .26859882e-01  .46299475e+02  .22939772e-03", 
" -.43307147e+01 -.12344705e+02  .52485674e+00  .43184881e+02  .22933917e-03", 
" -.43181300e+01 -.12219244e+02  .16843296e+01  .30946916e+02  .23269087e-03", 
" -.43057032e+01 -.12088361e+02  .22430759e+01  .27773956e+02  .23397567e-03", 
" -.42934628e+01 -.11960420e+02  .27072865e+01  .25957706e+02  .23271046e-03", 
" -.42814725e+01 -.11814136e+02  .23557456e+01  .32582839e+02  .23355711e-03", 
" -.42695575e+01 -.11689457e+02  .29183368e+01  .28696502e+02  .23499046e-03", 
" -.42577478e+01 -.11565970e+02  .33525394e+01  .26209395e+02  .23550351e-03", 
" -.42460334e+01 -.11456909e+02  .44214407e+01  .15631241e+02  .23700566e-03", 
" -.42345672e+01 -.11338024e+02  .50758728e+01  .97280838e+01  .23734138e-03", 
" -.42231886e+01 -.11207164e+02  .48212383e+01  .14257571e+02  .23512179e-03", 
" -.42119071e+01 -.11092364e+02  .53492761e+01  .96402059e+01  .23402518e-03", 
" -.42007695e+01 -.10970047e+02  .52485774e+01  .14322979e+02  .23336777e-03", 
" -.41897057e+01 -.10853751e+02  .52814652e+01  .17122689e+02  .23257009e-03", 
" -.41787072e+01 -.10736753e+02  .52886184e+01  .18094619e+02  .23039591e-03", 
" -.41678228e+01 -.10631296e+02  .56816026e+01  .16280546e+02  .22801378e-03", 
" -.41570034e+01 -.10539104e+02  .68639939e+01  .14605885e+01  .22608120e-03", 
" -.41464156e+01 -.10407727e+02  .60482048e+01  .12624304e+02  .22533226e-03", 
" -.41358182e+01 -.10304812e+02  .63605361e+01  .11554497e+02  .22543788e-03", 
" -.41252623e+01 -.10201477e+02  .65062702e+01  .12264193e+02  .22570146e-03", 
" -.41148294e+01 -.10098549e+02  .67701624e+01  .90028178e+01  .22581984e-03", 
" -.41044750e+01 -.99967983e+01  .70410638e+01  .58048650e+01  .22395644e-03", 
" -.40941727e+01 -.99111384e+01  .78985115e+01 -.28887599e+01  .22412528e-03", 
" -.40839624e+01 -.98116235e+01  .78731882e+01 -.20938704e+00  .22254415e-03", 
" -.40737997e+01 -.97148851e+01  .80592721e+01 -.16619504e+01  .22019898e-03", 
" -.40636815e+01 -.96234481e+01  .83295835e+01 -.38719920e+01  .22092333e-03", 
" -.40536717e+01 -.95214524e+01  .81626226e+01 -.49632977e+00  .22031186e-03", 
" -.40436968e+01 -.94351581e+01  .86876914e+01 -.65016006e+01  .21915190e-03", 
" -.40337702e+01 -.93447880e+01  .87365623e+01 -.44717152e+01  .21852191e-03", 
" -.40239245e+01 -.92583653e+01  .93163700e+01 -.12334759e+02  .21558248e-03", 
" -.40141442e+01 -.91689070e+01  .95598495e+01 -.14607419e+02  .21274630e-03", 
" -.40043862e+01 -.90818171e+01  .96748792e+01 -.13597369e+02  .21366120e-03", 
" -.39946865e+01 -.89879071e+01  .93573847e+01 -.73841749e+01  .21319140e-03", 
" -.39850739e+01 -.88973788e+01  .93245065e+01 -.60583739e+01  .21226379e-03", 
" -.39755050e+01 -.88011055e+01  .88050333e+01  .28782738e+01  .21168084e-03", 
" -.39659970e+01 -.87046413e+01  .83926407e+01  .86678858e+01  .21347638e-03", 
" -.39563504e+01 -.86412576e+01  .93999991e+01 -.27017291e+01  .21275521e-03", 
" -.39468828e+01 -.85567573e+01  .94308312e+01 -.19902548e+01  .21273343e-03", 
" -.39374291e+01 -.84713809e+01  .92355196e+01  .18169436e+01  .21175426e-03", 
" -.39280168e+01 -.83772078e+01  .85003084e+01  .13610568e+02  .21210355e-03", 
" -.39186660e+01 -.82881244e+01  .81762221e+01  .18098437e+02  .21138626e-03", 
" -.39092951e+01 -.82161331e+01  .87658548e+01  .10249591e+02  .20804597e-03", 
" -.38999597e+01 -.81392232e+01  .88560427e+01  .10714791e+02  .21020517e-03", 
" -.38906446e+01 -.80531269e+01  .83103401e+01  .20235137e+02  .20982406e-03", 
" -.38813910e+01 -.79742594e+01  .83627527e+01  .20380606e+02  .21041765e-03", 
" -.38721936e+01 -.78890007e+01  .80376660e+01  .25493670e+02  .20779094e-03", 
" -.38629443e+01 -.78067868e+01  .76985137e+01  .30916025e+02  .20612617e-03", 
" -.38537190e+01 -.77368478e+01  .79418332e+01  .29568183e+02  .20469561e-03", 
" -.38445245e+01 -.76671850e+01  .83128238e+01  .25339722e+02  .20567461e-03", 
" -.38353238e+01 -.75971784e+01  .85048060e+01  .23490771e+02  .20583366e-03", 
" -.38261594e+01 -.75354048e+01  .93263962e+01  .12529307e+02  .20615987e-03", 
" -.38170312e+01 -.74624439e+01  .92663827e+01  .15582352e+02  .20599780e-03", 
" -.38078442e+01 -.73940899e+01  .94434984e+01  .13631954e+02  .20701055e-03", 
" -.37987048e+01 -.73349653e+01  .10088916e+02  .67192523e+01  .20702677e-03", 
" -.37895564e+01 -.72681452e+01  .10305838e+02  .45241541e+01  .20628656e-03", 
" -.37804373e+01 -.72061726e+01  .10605786e+02  .32064470e+01  .20652580e-03", 
" -.37713714e+01 -.71256035e+01  .10109807e+02  .10017600e+02  .20689260e-03", 
" -.37622875e+01 -.70534740e+01  .99321811e+01  .13258279e+02  .20571751e-03", 
" -.37532430e+01 -.69765352e+01  .96645413e+01  .16663725e+02  .20580222e-03", 
" -.37441693e+01 -.69074574e+01  .96885335e+01  .16374851e+02  .20543357e-03", 
" -.37351232e+01 -.68307914e+01  .91847948e+01  .23981137e+02  .20439230e-03", 
" -.37259891e+01 -.67745988e+01  .97809697e+01  .15646471e+02  .20303015e-03", 
" -.37169222e+01 -.67024533e+01  .95137147e+01  .19763327e+02  .20337232e-03", 
" -.37078533e+01 -.66349671e+01  .95309720e+01  .19272005e+02  .20072124e-03", 
" -.36987345e+01 -.65793336e+01  .99877678e+01  .14439987e+02  .19962151e-03", 
" -.36896075e+01 -.65165344e+01  .10014352e+02  .16011088e+02  .20044987e-03", 
" -.36804844e+01 -.64491230e+01  .96896187e+01  .22002790e+02  .19883351e-03", 
" -.36713519e+01 -.63894312e+01  .99453942e+01  .18883341e+02  .19876065e-03", 
" -.36622005e+01 -.63312910e+01  .10151998e+02  .17665416e+02  .19790825e-03", 
" -.36530388e+01 -.62815070e+01  .10768710e+02  .11176079e+02  .19406065e-03", 
" -.36438384e+01 -.62323790e+01  .11389595e+02  .39310357e+01  .19355764e-03", 
" -.36346308e+01 -.61832441e+01  .11784411e+02  .16857653e+01  .19331265e-03", 
" -.36254275e+01 -.61280430e+01  .12133958e+02 -.26214435e+01  .19197984e-03", 
" -.36162334e+01 -.60644132e+01  .11918439e+02  .18681938e+01  .19246979e-03", 
" -.36070413e+01 -.59981780e+01  .11603741e+02  .73113719e+01  .19254330e-03", 
" -.35977447e+01 -.59461669e+01  .11914339e+02  .46596291e+01  .19061689e-03", 
" -.35885092e+01 -.58866897e+01  .12020823e+02  .34135252e+01  .18949924e-03", 
" -.35791920e+01 -.58311143e+01  .12166495e+02  .22099585e+01  .18854111e-03", 
" -.35698528e+01 -.57795795e+01  .12519206e+02 -.11839397e+01  .18837916e-03", 
" -.35605259e+01 -.57265356e+01  .13027878e+02 -.89323180e+01  .18802751e-03", 
" -.35512302e+01 -.56540306e+01  .12455940e+02 -.23377052e+01  .18720570e-03", 
" -.35417705e+01 -.56071464e+01  .12895063e+02 -.66744240e+01  .18934262e-03", 
" -.35323005e+01 -.55522504e+01  .13050500e+02 -.84216845e+01  .19262995e-03", 
" -.35228583e+01 -.54882459e+01  .12694375e+02 -.32304421e+01  .19352993e-03", 
" -.35133115e+01 -.54395823e+01  .13161327e+02 -.92406637e+01  .19430777e-03", 
" -.35037282e+01 -.53963664e+01  .13772339e+02 -.16380396e+02  .19564346e-03", 
" -.34941820e+01 -.53290195e+01  .13202372e+02 -.84010998e+01  .19485103e-03", 
" -.34845464e+01 -.52749834e+01  .13235031e+02 -.79217571e+01  .19594655e-03", 
" -.34748858e+01 -.52193014e+01  .13270589e+02 -.87069850e+01  .19730483e-03", 
" -.34651645e+01 -.51567520e+01  .12725091e+02  .21478686e+00  .19679665e-03", 
" -.34553865e+01 -.51041643e+01  .12769403e+02 -.16279498e+00  .19687786e-03", 
" -.34455463e+01 -.50511816e+01  .12772511e+02  .97354104e+00  .19760323e-03", 
" -.34356809e+01 -.49905207e+01  .12542497e+02  .34821509e+01  .20151630e-03", 
" -.34256634e+01 -.49465837e+01  .13040396e+02 -.30604232e+01  .20094919e-03", 
" -.34156497e+01 -.48966072e+01  .13279601e+02 -.64754561e+01  .19833147e-03", 
" -.34055444e+01 -.48538523e+01  .13913894e+02 -.15445062e+02  .19789783e-03", 
" -.33954132e+01 -.48016784e+01  .13907599e+02 -.15053858e+02  .19848930e-03", 
" -.33851472e+01 -.47555131e+01  .14251705e+02 -.20749252e+02  .19822850e-03", 
" -.33748690e+01 -.47030822e+01  .14163874e+02 -.18922243e+02  .19748062e-03", 
" -.33645481e+01 -.46494832e+01  .14181717e+02 -.19691967e+02  .20094818e-03", 
" -.33540743e+01 -.45997420e+01  .14211378e+02 -.19533658e+02  .20063790e-03", 
" -.33435099e+01 -.45549353e+01  .14502549e+02 -.22573201e+02  .20178891e-03", 
" -.33328804e+01 -.45025844e+01  .14201671e+02 -.16367545e+02  .20428601e-03", 
" -.33221652e+01 -.44456267e+01  .13862555e+02 -.12486128e+02  .20379789e-03", 
" -.33113255e+01 -.44010623e+01  .13970984e+02 -.12137461e+02  .20455836e-03", 
" -.33003717e+01 -.43701783e+01  .15128190e+02 -.27787384e+02  .20710859e-03", 
" -.32893721e+01 -.43172430e+01  .14960960e+02 -.24974652e+02  .20812024e-03", 
" -.32782426e+01 -.42680022e+01  .15034686e+02 -.26174333e+02  .20972708e-03", 
" -.32670187e+01 -.42130146e+01  .14788404e+02 -.22954790e+02  .21184947e-03", 
" -.32556364e+01 -.41734425e+01  .15039905e+02 -.23323069e+02  .21370944e-03", 
" -.32441290e+01 -.41246390e+01  .14945409e+02 -.20943166e+02  .21440701e-03", 
" -.32324309e+01 -.40970325e+01  .15751041e+02 -.28782452e+02  .21530112e-03", 
" -.32207636e+01 -.40373887e+01  .15347955e+02 -.24618376e+02  .21459355e-03", 
" -.32088319e+01 -.40021133e+01  .15904852e+02 -.31060975e+02  .21646222e-03", 
" -.31967683e+01 -.39630084e+01  .16410045e+02 -.37743645e+02  .21842890e-03", 
" -.31844794e+01 -.39304529e+01  .17056637e+02 -.45633911e+02  .22076221e-03", 
" -.31720472e+01 -.38985057e+01  .17828166e+02 -.55568837e+02  .22069502e-03", 
" -.31595268e+01 -.38471414e+01  .17485628e+02 -.50321927e+02  .21769775e-03", 
" -.31466990e+01 -.38344678e+01  .19278446e+02 -.74695818e+02  .21660463e-03", 
" -.31337807e+01 -.37870030e+01  .19236834e+02 -.73938971e+02  .21901175e-03", 
" -.31205817e+01 -.37534688e+01  .19788341e+02 -.80421408e+02  .21734714e-03", 
" -.31071128e+01 -.37249792e+01  .20464607e+02 -.88151843e+02  .21543001e-03", 
" -.30935019e+01 -.36926844e+01  .21166230e+02 -.96927038e+02  .21733442e-03", 
" -.30795905e+01 -.36580659e+01  .21435206e+02 -.98615766e+02  .22018314e-03", 
" -.30654882e+01 -.36157208e+01  .21302945e+02 -.94932182e+02  .22289122e-03", 
" -.30510685e+01 -.35816848e+01  .21744353e+02 -.99843170e+02  .22124608e-03", 
" -.30363551e+01 -.35332045e+01  .21195723e+02 -.90126375e+02  .22267621e-03", 
" -.30212561e+01 -.35063690e+01  .21670933e+02 -.93621668e+02  .22618571e-03", 
" -.30059409e+01 -.34690948e+01  .21937284e+02 -.96802952e+02  .23021239e-03", 
" -.29901691e+01 -.34473212e+01  .22970461e+02 -.10956732e+03  .23146313e-03", 
" -.29739960e+01 -.34336286e+01  .24141870e+02 -.12198732e+03  .23715715e-03", 
" -.29574551e+01 -.34046928e+01  .24462521e+02 -.12274100e+03  .24042148e-03", 
" -.29404204e+01 -.33796707e+01  .24743480e+02 -.12205238e+03  .24174193e-03", 
" -.29229035e+01 -.33604634e+01  .25502087e+02 -.12966747e+03  .24342587e-03", 
" -.29048327e+01 -.33389698e+01  .26079834e+02 -.13419065e+03  .24538464e-03", 
" -.28863055e+01 -.33073261e+01  .26199791e+02 -.13339551e+03  .25006412e-03", 
" -.28670992e+01 -.32898056e+01  .26994170e+02 -.14150745e+03  .25218017e-03", 
" -.28472183e+01 -.32529968e+01  .26501838e+02 -.13238457e+03  .25782675e-03", 
" -.28265513e+01 -.32326642e+01  .26752033e+02 -.13248183e+03  .25968386e-03", 
" -.28049801e+01 -.32279842e+01  .27477502e+02 -.13635948e+03  .26177586e-03", 
" -.27824821e+01 -.32370781e+01  .28787688e+02 -.14617096e+03  .26270941e-03", 
" -.27590095e+01 -.32414275e+01  .30068039e+02 -.15716970e+03  .27056663e-03", 
" -.27344285e+01 -.32101232e+01  .29281181e+02 -.14391822e+03  .27634687e-03", 
" -.27084314e+01 -.32151595e+01  .30004514e+02 -.14575646e+03  .27995490e-03", 
" -.26809522e+01 -.32318000e+01  .31440429e+02 -.15987172e+03  .28261617e-03", 
" -.26518293e+01 -.32345309e+01  .31507675e+02 -.15125488e+03  .29130194e-03", 
" -.26204351e+01 -.32866095e+01  .33549218e+02 -.16704030e+03  .30168443e-03", 
" -.25867006e+01 -.33263076e+01  .34813115e+02 -.17366900e+03  .30671963e-03", 
" -.25498938e+01 -.34077570e+01  .37163092e+02 -.18888110e+03  .32084615e-03", 
" -.25096392e+01 -.34896916e+01  .39443804e+02 -.20712181e+03  .33402849e-03", 
" -.24647122e+01 -.36317354e+01  .43558090e+02 -.24611342e+03  .34949188e-03", 
" -.24136585e+01 -.38162758e+01  .47868030e+02 -.27864035e+03  .36104339e-03", 
" -.23545710e+01 -.40559487e+01  .53045071e+02 -.32009593e+03  .38446714e-03", 
" -.22838603e+01 -.43170030e+01  .55536939e+02 -.31039052e+03  .44676119e-03", 
" -.21949853e+01 -.47093489e+01  .60005956e+02 -.32321356e+03  .49310996e-03", 
" -.20719987e+01 -.55274413e+01  .73499070e+02 -.41424585e+03  .60026416e-03", 
" -.20404115e+01 -.57132172e+01  .74136495e+02 -.39474631e+03  .61758591e-03", 
" -.20051496e+01 -.59808420e+01  .77918405e+02 -.41950681e+03  .65566043e-03", 
" -.19659056e+01 -.62771202e+01  .82552512e+02 -.45141072e+03  .69370606e-03", 
" -.19204316e+01 -.65997372e+01  .85024456e+02 -.44701262e+03  .74982391e-03", 
" -.18667566e+01 -.70929490e+01  .93349413e+02 -.51501745e+03  .83826853e-03", 
" -.18017529e+01 -.76977063e+01  .10341290e+03 -.59898118e+03  .90862571e-03", 
" -.17174684e+01 -.84577851e+01  .11153434e+03 -.63913971e+03  .10633553e-02", 
" -.16005558e+01 -.96592728e+01  .12938910e+03 -.78952367e+03  .12683888e-02", 
" -.14025799e+01 -.11986899e+02  .16351610e+03 -.10432371e+04  .17309947e-02", 
" -.12097982e+01 -.14255821e+02  .19251332e+03 -.12801129e+04  .23227830e-02", 
" -.96351796e+00 -.17715906e+02  .24902346e+03 -.17803594e+04  .35687746e-02", 
" -.78969290e+00 -.18895364e+02  .22928043e+03 -.14545365e+04  .48456803e-02", 
"coanc     4  5  3   25", " -.67593013e+02  .13105816e+04 -.17318118e+05  .11612515e+06  .91369179e-01", 
" -.64135785e+02  .11923404e+04 -.15370230e+05  .10195988e+06  .63325230e-01", 
" -.59556006e+02  .10340272e+04 -.12197749e+05  .72188062e+05  .44457833e-01", 
" -.56016608e+02  .92632928e+03 -.10532797e+05  .60947508e+05  .31533599e-01", 
" -.52346860e+02  .81936952e+03 -.90144356e+04  .52186371e+05  .22590829e-01", 
" -.50187013e+02  .76221138e+03 -.83726962e+04  .50459914e+05  .18961578e-01", 
" -.48639298e+02  .72050680e+03 -.77536908e+04  .45797343e+05  .17484188e-01", 
" -.47413731e+02  .68802786e+03 -.73064300e+04  .43161589e+05  .15431672e-01", 
" -.46415572e+02  .66332226e+03 -.70291403e+04  .42244907e+05  .13922078e-01", 
" -.45567885e+02  .64305883e+03 -.68093414e+04  .41472200e+05  .13299031e-01", 
" -.44812364e+02  .62351827e+03 -.65291046e+04  .39501251e+05  .12273832e-01", 
" -.44150770e+02  .60585608e+03 -.62327267e+04  .37026649e+05  .11855609e-01", 
" -.43560857e+02  .59150224e+03 -.60366439e+04  .35705676e+05  .10849711e-01", 
" -.41244230e+02  .53533341e+03 -.52385002e+04  .29885330e+05  .91462226e-02", 
" -.39567003e+02  .49554222e+03 -.46503896e+04  .25159382e+05  .83606898e-02", 
" -.38249977e+02  .46609416e+03 -.42656735e+04  .22592140e+05  .75738404e-02", 
" -.37159461e+02  .44220675e+03 -.39520289e+04  .20405746e+05  .71006805e-02", 
" -.36228641e+02  .42271990e+03 -.37220300e+04  .19084639e+05  .66232678e-02", 
" -.35414220e+02  .40612372e+03 -.35251309e+04  .17806372e+05  .61169657e-02", 
" -.34689228e+02  .39197924e+03 -.33823674e+04  .17194387e+05  .59560505e-02", 
" -.34031620e+02  .37899110e+03 -.32414726e+04  .16505073e+05  .57273584e-02", 
" -.33431927e+02  .36735139e+03 -.31105281e+04  .15723925e+05  .54498247e-02", 
" -.32880709e+02  .35666366e+03 -.29832990e+04  .14876147e+05  .52476492e-02", 
" -.32369301e+02  .34716054e+03 -.28919797e+04  .14561962e+05  .50980571e-02", 
" -.31890445e+02  .33775943e+03 -.27649187e+04  .13543407e+05  .48176276e-02", 
" -.31441194e+02  .32921413e+03 -.26671393e+04  .12965097e+05  .46674340e-02", 
" -.31018676e+02  .32142330e+03 -.25774039e+04  .12344899e+05  .45837116e-02", 
" -.30621384e+02  .31453528e+03 -.25147371e+04  .12103496e+05  .44570856e-02", 
" -.30243534e+02  .30799491e+03 -.24560616e+04  .11907329e+05  .44796735e-02", 
" -.29883307e+02  .30159400e+03 -.23860945e+04  .11486390e+05  .43013907e-02", 
" -.29538018e+02  .29521166e+03 -.23099443e+04  .11035314e+05  .41662442e-02", 
" -.29209212e+02  .28945288e+03 -.22441584e+04  .10575967e+05  .40756408e-02", 
" -.28893938e+02  .28427814e+03 -.22027436e+04  .10459079e+05  .40314340e-02", 
" -.28590354e+02  .27899595e+03 -.21417669e+04  .10029599e+05  .40051984e-02", 
" -.28298487e+02  .27437971e+03 -.21125692e+04  .10086911e+05  .38873011e-02", 
" -.28016546e+02  .26985174e+03 -.20798021e+04  .10079032e+05  .38126614e-02", 
" -.27742185e+02  .26514962e+03 -.20286172e+04  .97635562e+04  .37882307e-02", 
" -.27477249e+02  .26075934e+03 -.19882027e+04  .96174373e+04  .37382228e-02", 
" -.27221233e+02  .25655992e+03 -.19510032e+04  .95214041e+04  .36524912e-02", 
" -.26972643e+02  .25254296e+03 -.19155269e+04  .94068040e+04  .35962134e-02", 
" -.26731433e+02  .24877956e+03 -.18865451e+04  .93578025e+04  .35364651e-02", 
" -.26495396e+02  .24468727e+03 -.18335213e+04  .89217866e+04  .35202929e-02", 
" -.26267079e+02  .24101841e+03 -.17970094e+04  .87263353e+04  .34287427e-02", 
" -.26044700e+02  .23755218e+03 -.17657082e+04  .85965681e+04  .33498684e-02", 
" -.25828431e+02  .23440423e+03 -.17474833e+04  .86439209e+04  .33079245e-02", 
" -.25616751e+02  .23105662e+03 -.17151305e+04  .85008177e+04  .32580173e-02", 
" -.25411266e+02  .22818978e+03 -.17030428e+04  .86044059e+04  .32738917e-02", 
" -.25209174e+02  .22509942e+03 -.16740222e+04  .84553053e+04  .32548940e-02", 
" -.25011235e+02  .22206174e+03 -.16434618e+04  .82657998e+04  .31824136e-02", 
" -.24816610e+02  .21896178e+03 -.16073146e+04  .79916785e+04  .31106443e-02", 
" -.24625876e+02  .21601209e+03 -.15781328e+04  .78301265e+04  .31218094e-02", 
" -.24440055e+02  .21312647e+03 -.15430112e+04  .75275117e+04  .31253590e-02", 
" -.24257775e+02  .21045637e+03 -.15177561e+04  .73671670e+04  .30923785e-02", 
" -.24077963e+02  .20776247e+03 -.14934072e+04  .72622759e+04  .30605123e-02", 
" -.23901380e+02  .20495092e+03 -.14571228e+04  .69569872e+04  .30166507e-02", 
" -.23728490e+02  .20256283e+03 -.14457041e+04  .70491850e+04  .29760666e-02", 
" -.23557865e+02  .19996197e+03 -.14171656e+04  .68527713e+04  .29647617e-02", 
" -.23390359e+02  .19743389e+03 -.13873554e+04  .65964260e+04  .29346881e-02", 
" -.23225239e+02  .19485783e+03 -.13539036e+04  .63000594e+04  .29108487e-02", 
" -.23063536e+02  .19248119e+03 -.13275159e+04  .60968501e+04  .28849855e-02", 
" -.22903043e+02  .19005332e+03 -.13010880e+04  .59124955e+04  .28406643e-02", 
" -.22745109e+02  .18766260e+03 -.12719458e+04  .56706062e+04  .28245431e-02", 
" -.22589808e+02  .18539300e+03 -.12491545e+04  .55407270e+04  .27988619e-02", 
" -.22436621e+02  .18312597e+03 -.12238495e+04  .53603064e+04  .27440060e-02", 
" -.22285801e+02  .18089706e+03 -.11964005e+04  .51192606e+04  .27423930e-02", 
" -.22137297e+02  .17882010e+03 -.11777718e+04  .50157883e+04  .27393880e-02", 
" -.21990004e+02  .17668383e+03 -.11549689e+04  .48637083e+04  .26956970e-02", 
" -.21845462e+02  .17477160e+03 -.11423215e+04  .48567710e+04  .26944413e-02", 
" -.21702290e+02  .17273615e+03 -.11206800e+04  .46948768e+04  .26848254e-02", 
" -.21560966e+02  .17066963e+03 -.10947605e+04  .44636959e+04  .26730191e-02", 
" -.21421530e+02  .16871419e+03 -.10742490e+04  .43239978e+04  .26444236e-02", 
" -.21283152e+02  .16672368e+03 -.10520125e+04  .41656037e+04  .26579867e-02", 
" -.21146419e+02  .16482580e+03 -.10324389e+04  .40317599e+04  .26435365e-02", 
" -.21011331e+02  .16307924e+03 -.10215972e+04  .40362138e+04  .26080282e-02", 
" -.20877825e+02  .16133192e+03 -.10083912e+04  .39936775e+04  .26187301e-02", 
" -.20746107e+02  .15970653e+03 -.10007477e+04  .40359332e+04  .26059260e-02", 
" -.20615180e+02  .15797992e+03 -.98725387e+03  .39949797e+04  .26034507e-02", 
" -.20485824e+02  .15634823e+03 -.97803118e+03  .40131632e+04  .25762960e-02", 
" -.20357711e+02  .15469493e+03 -.96641827e+03  .40033875e+04  .25935811e-02", 
" -.20230733e+02  .15312857e+03 -.96008972e+03  .40740536e+04  .25663186e-02", 
" -.20104947e+02  .15154521e+03 -.94957886e+03  .40547306e+04  .25464712e-02", 
" -.19980848e+02  .14994837e+03 -.93471167e+03  .39449279e+04  .25341005e-02", 
" -.19856787e+02  .14828267e+03 -.91866878e+03  .38540611e+04  .24968768e-02", 
" -.19734226e+02  .14665745e+03 -.90179553e+03  .37296495e+04  .24859397e-02", 
" -.19612695e+02  .14519673e+03 -.89484134e+03  .37647393e+04  .24668145e-02", 
" -.19492571e+02  .14366857e+03 -.88251996e+03  .37202641e+04  .24446147e-02", 
" -.19372990e+02  .14214177e+03 -.86852554e+03  .36354685e+04  .24127207e-02", 
" -.19254095e+02  .14055296e+03 -.85102131e+03  .34972142e+04  .23832135e-02", 
" -.19136599e+02  .13902390e+03 -.83499760e+03  .33843637e+04  .23700136e-02", 
" -.19019813e+02  .13756311e+03 -.82344179e+03  .33413653e+04  .23609373e-02", 
" -.18903625e+02  .13610720e+03 -.81142473e+03  .32822108e+04  .23382559e-02", 
" -.18788183e+02  .13463227e+03 -.79741286e+03  .31947099e+04  .23365310e-02", 
" -.18674037e+02  .13325793e+03 -.78769932e+03  .31601009e+04  .23082804e-02", 
" -.18560517e+02  .13179211e+03 -.77033593e+03  .29920960e+04  .22700011e-02", 
" -.18447601e+02  .13032844e+03 -.75308991e+03  .28342820e+04  .22764638e-02", 
" -.18335788e+02  .12903420e+03 -.74770986e+03  .28882990e+04  .22582322e-02", 
" -.18224288e+02  .12765289e+03 -.73547144e+03  .28230215e+04  .22489166e-02", 
" -.18113343e+02  .12628332e+03 -.72297504e+03  .27409826e+04  .22678322e-02", 
" -.18003574e+02  .12497497e+03 -.71236878e+03  .26878309e+04  .22677323e-02", 
" -.17893901e+02  .12376516e+03 -.70992343e+03  .27745620e+04  .22889058e-02", 
" -.17784581e+02  .12233349e+03 -.69254709e+03  .26274592e+04  .22899634e-02", 
" -.17675827e+02  .12098591e+03 -.68009191e+03  .25602988e+04  .22773405e-02", 
" -.17568266e+02  .11977863e+03 -.67484287e+03  .25896539e+04  .22657409e-02", 
" -.17461374e+02  .11856414e+03 -.66733370e+03  .25765413e+04  .22877985e-02", 
" -.17354568e+02  .11720527e+03 -.64963250e+03  .24055157e+04  .23020198e-02", 
" -.17248151e+02  .11598757e+03 -.64250881e+03  .24026795e+04  .22550406e-02", 
" -.17142686e+02  .11483034e+03 -.63835755e+03  .24570037e+04  .22473739e-02", 
" -.17037648e+02  .11359628e+03 -.62673160e+03  .23732132e+04  .22478999e-02", 
" -.16932805e+02  .11230417e+03 -.61082355e+03  .22217892e+04  .22485134e-02", 
" -.16828343e+02  .11101249e+03 -.59595393e+03  .20965201e+04  .22343236e-02", 
" -.16724608e+02  .10991247e+03 -.59199150e+03  .21365690e+04  .22280621e-02", 
" -.16620889e+02  .10871363e+03 -.58232451e+03  .20929595e+04  .22618441e-02", 
" -.16518062e+02  .10756569e+03 -.57409379e+03  .20700408e+04  .22381624e-02", 
" -.16415686e+02  .10646997e+03 -.56704480e+03  .20403985e+04  .22492494e-02", 
" -.16313567e+02  .10529959e+03 -.55595668e+03  .19615151e+04  .22162721e-02", 
" -.16211468e+02  .10414709e+03 -.54619760e+03  .18995959e+04  .22248474e-02", 
" -.16110197e+02  .10309908e+03 -.54079976e+03  .18896214e+04  .22059120e-02", 
" -.16008501e+02  .10188048e+03 -.52649849e+03  .17640520e+04  .21940242e-02", 
" -.15907718e+02  .10085729e+03 -.52408433e+03  .18263762e+04  .22196917e-02", 
" -.15806416e+02  .99585543e+02 -.50564300e+03  .16214629e+04  .22264415e-02", 
" -.15706059e+02  .98469099e+02 -.49590617e+03  .15655593e+04  .22332897e-02", 
" -.15606195e+02  .97382732e+02 -.48613634e+03  .15005836e+04  .22480963e-02", 
" -.15506181e+02  .96386056e+02 -.48481301e+03  .15848719e+04  .22431033e-02", 
" -.15406343e+02  .95333681e+02 -.47683741e+03  .15286387e+04  .22423456e-02", 
" -.15306983e+02  .94354728e+02 -.47457342e+03  .15896936e+04  .22029350e-02", 
" -.15207829e+02  .93332167e+02 -.46742195e+03  .15515176e+04  .22220664e-02", 
" -.15108638e+02  .92217122e+02 -.45510134e+03  .14425049e+04  .22150034e-02", 
" -.15009505e+02  .91142485e+02 -.44485388e+03  .13568193e+04  .22359569e-02", 
" -.14910367e+02  .90088308e+02 -.43553531e+03  .12802299e+04  .22065004e-02", 
" -.14811573e+02  .89156878e+02 -.43578534e+03  .13788472e+04  .22034552e-02", 
" -.14713222e+02  .88219561e+02 -.43320580e+03  .14118346e+04  .21916554e-02", 
" -.14614867e+02  .87293196e+02 -.43154105e+03  .14630743e+04  .21873322e-02", 
" -.14516174e+02  .86123912e+02 -.41348746e+03  .12541479e+04  .21983733e-02", 
" -.14418139e+02  .85170822e+02 -.40648657e+03  .11854696e+04  .21993271e-02", 
" -.14319514e+02  .84088310e+02 -.39338177e+03  .10540966e+04  .21878172e-02", 
" -.14220602e+02  .83032446e+02 -.38426548e+03  .10027914e+04  .21577658e-02", 
" -.14121861e+02  .81906334e+02 -.36788115e+03  .81137415e+03  .21690776e-02", 
" -.14022972e+02  .80831923e+02 -.35597967e+03  .70578768e+03  .21639511e-02", 
" -.13924581e+02  .79970919e+02 -.35661575e+03  .79061735e+03  .21259620e-02", 
" -.13826381e+02  .79015774e+02 -.35168703e+03  .80489292e+03  .21055954e-02", 
" -.13727602e+02  .78021211e+02 -.34531196e+03  .79484454e+03  .20864697e-02", 
" -.13628923e+02  .77073257e+02 -.34019168e+03  .79412012e+03  .20693216e-02", 
" -.13529702e+02  .75980110e+02 -.32639910e+03  .65834757e+03  .20699260e-02", 
" -.13430942e+02  .75054835e+02 -.32167947e+03  .65741226e+03  .20627550e-02", 
" -.13331832e+02  .74046484e+02 -.31103901e+03  .54771870e+03  .20539078e-02", 
" -.13232824e+02  .73107720e+02 -.30466355e+03  .51268260e+03  .20539436e-02", 
" -.13133493e+02  .72151512e+02 -.29770064e+03  .47165452e+03  .20584396e-02", 
" -.13034487e+02  .71303921e+02 -.29590128e+03  .50634798e+03  .20442512e-02", 
" -.12934482e+02  .70274774e+02 -.28626891e+03  .45015420e+03  .20366573e-02", 
" -.12834528e+02  .69358392e+02 -.28101270e+03  .43058273e+03  .20328812e-02", 
" -.12734533e+02  .68405590e+02 -.27275565e+03  .36014430e+03  .20177920e-02", 
" -.12634304e+02  .67493174e+02 -.26754645e+03  .34638978e+03  .20222206e-02", 
" -.12533679e+02  .66556938e+02 -.26134296e+03  .32692797e+03  .20385273e-02", 
" -.12432582e+02  .65532820e+02 -.25038869e+03  .23816202e+03  .20494889e-02", 
" -.12331227e+02  .64606510e+02 -.24542193e+03  .24025674e+03  .20403292e-02", 
" -.12229019e+02  .63634626e+02 -.23850603e+03  .20606275e+03  .20251564e-02", 
" -.12126572e+02  .62733285e+02 -.23537118e+03  .22774048e+03  .20149341e-02", 
" -.12023935e+02  .61866872e+02 -.23541666e+03  .31498090e+03  .20320128e-02", 
" -.11920067e+02  .60839147e+02 -.22525070e+03  .23201366e+03  .20334575e-02", 
" -.11816663e+02  .59943842e+02 -.22104426e+03  .23525569e+03  .20351342e-02", 
" -.11712561e+02  .59021259e+02 -.21686416e+03  .25039532e+03  .20303689e-02", 
" -.11607702e+02  .58130170e+02 -.21508788e+03  .30990887e+03  .19998321e-02", 
" -.11502186e+02  .57205815e+02 -.21046119e+03  .30936183e+03  .20047917e-02", 
" -.11396219e+02  .56263912e+02 -.20496740e+03  .30677457e+03  .20120392e-02", 
" -.11289811e+02  .55410783e+02 -.20314051e+03  .34000251e+03  .19948061e-02", 
" -.11182748e+02  .54455076e+02 -.19541127e+03  .28599609e+03  .19934203e-02", 
" -.11074878e+02  .53482332e+02 -.18734222e+03  .23191183e+03  .19957228e-02", 
" -.10965745e+02  .52562940e+02 -.18316203e+03  .24096397e+03  .19916515e-02", 
" -.10856497e+02  .51710025e+02 -.18161702e+03  .28126466e+03  .20154816e-02", 
" -.10746437e+02  .50848573e+02 -.17896388e+03  .29485466e+03  .19989987e-02", 
" -.10635019e+02  .49895338e+02 -.17187409e+03  .24985770e+03  .19645478e-02", 
" -.10522400e+02  .48894133e+02 -.16190602e+03  .14532809e+03  .19438561e-02", 
" -.10408687e+02  .47892611e+02 -.15321910e+03  .89440270e+02  .19877842e-02", 
" -.10294634e+02  .47194443e+02 -.16270546e+03  .30561771e+03  .20177496e-02", 
" -.10179246e+02  .46251475e+02 -.15646495e+03  .28013466e+03  .20498072e-02", 
" -.10062640e+02  .45402897e+02 -.15729360e+03  .36590975e+03  .20554018e-02", 
" -.99441187e+01  .44347175e+02 -.14555653e+03  .24550341e+03  .20291110e-02", 
" -.98247175e+01  .43485053e+02 -.14537717e+03  .31157425e+03  .20397730e-02", 
" -.97035380e+01  .42567159e+02 -.14370083e+03  .36650225e+03  .20314624e-02", 
" -.95808642e+01  .41686315e+02 -.14477272e+03  .46692626e+03  .20515304e-02", 
" -.94562801e+01  .40751725e+02 -.14240040e+03  .49994007e+03  .20849318e-02", 
" -.93300194e+01  .39797352e+02 -.13804473e+03  .50234725e+03  .20733997e-02", 
" -.92008814e+01  .38620898e+02 -.12146075e+03  .30815652e+03  .20493938e-02", 
" -.90701939e+01  .37654916e+02 -.11784459e+03  .31367154e+03  .20712022e-02", 
" -.89375835e+01  .36672314e+02 -.11285783e+03  .28788366e+03  .20430502e-02", 
" -.88030242e+01  .35884040e+02 -.11977835e+03  .45670589e+03  .20339041e-02", 
" -.86656394e+01  .34951310e+02 -.11787726e+03  .47928637e+03  .20460331e-02", 
" -.85252076e+01  .34050646e+02 -.12078630e+03  .59036297e+03  .20677746e-02", 
" -.83816957e+01  .32962719e+02 -.11156923e+03  .50317374e+03  .20564898e-02", 
" -.82341408e+01  .31872102e+02 -.10680441e+03  .52321750e+03  .20217056e-02", 
" -.80825671e+01  .30568859e+02 -.86985819e+02  .28115798e+03  .20579460e-02", 
" -.79278420e+01  .29620012e+02 -.93547853e+02  .50064865e+03  .20717160e-02", 
" -.77687772e+01  .28588795e+02 -.90172304e+02  .48469160e+03  .20802025e-02", 
" -.76048041e+01  .27398175e+02 -.80197336e+02  .39256669e+03  .20864401e-02", 
" -.74342909e+01  .25985133e+02 -.57738646e+02  .91918648e+02  .20682960e-02", 
" -.72580566e+01  .24755923e+02 -.47508152e+02 -.24288779e+02  .21027130e-02", 
" -.70744173e+01  .23296750e+02 -.29396680e+02 -.19412595e+03  .20957906e-02", 
" -.68835999e+01  .21970029e+02 -.20573052e+02 -.24400953e+03  .20735203e-02", 
" -.66835971e+01  .20551638e+02 -.81059602e+01 -.34354005e+03  .21415375e-02", 
" -.64741455e+01  .19297714e+02 -.94346791e+01 -.22005281e+03  .21664335e-02", 
" -.62528968e+01  .18061850e+02 -.97903532e+01 -.14540882e+03  .21989619e-02", 
" -.60180906e+01  .16783428e+02 -.53499516e+01 -.21372968e+03  .22057478e-02", 
" -.57635315e+01  .15258802e+02 -.32513773e+01 -.12164495e+03  .22880510e-02", 
" -.54880813e+01  .13852122e+02 -.89697266e+01  .38305693e+02  .22725140e-02", 
" -.51829735e+01  .11932498e+02  .49636959e+01 -.64955096e+02  .24109776e-02", 
" -.48408533e+01  .99929403e+01  .97662958e+01  .13973962e+02  .24473140e-02", 
" -.44461125e+01  .77133725e+01  .22118179e+02 -.46310555e+02  .25646044e-02", 
" -.39705852e+01  .52826454e+01  .22261641e+02  .10935256e+03  .28493885e-02", 
" -.33566871e+01  .26303292e+01 -.15102960e+01  .70024098e+03  .31990407e-02", 
" -.32049478e+01  .16403186e+01  .23107775e+02  .26761183e+03  .32634425e-02", 
" -.30404402e+01  .11365143e+01  .11656916e+02  .44215180e+03  .33780720e-02", 
" -.28561642e+01  .62692755e+00 -.18166821e+02  .10383924e+04  .35323482e-02", 
" -.26501429e+01 -.70561806e-01 -.27247962e+02  .12061491e+04  .35450981e-02", 
" -.24141032e+01 -.10037745e+01 -.25028158e+02  .11445233e+04  .39187099e-02", 
" -.21308370e+01 -.24438093e+01 -.31750267e+01  .74555921e+03  .42963999e-02", 
" -.17761771e+01 -.53261389e+01  .88596544e+02 -.59771932e+03  .47135997e-02", 
" -.12986061e+01 -.76818384e+01  .14386548e+03 -.15535302e+04  .54817252e-02", 
" -.53755734e+00 -.79637060e+01  .85083295e+02 -.10121101e+04  .75882159e-02", 
"  .17866899e+00 -.11485103e+02  .22894346e+03 -.29732822e+04  .91987106e-02", 
"  .10158860e+01 -.11182827e+02  .32715640e+03 -.53689682e+04  .14122304e-01", 
"  .15642185e+01  .51713335e-02 -.16662033e+03  .13789844e+04  .19386099e-01", 
"coac      4  6  3   25", " -.72516309e+02  .14663277e+04 -.19697050e+05  .13147645e+06  .97271554e-01", 
" -.69063295e+02  .13433731e+04 -.17553118e+05  .11516899e+06  .65917499e-01", 
" -.64288762e+02  .11769379e+04 -.14778961e+05  .95590321e+05  .43341882e-01", 
" -.60660650e+02  .10586768e+04 -.12823745e+05  .81136916e+05  .31786881e-01", 
" -.56911325e+02  .93767078e+03 -.10660332e+05  .62402854e+05  .22881438e-01", 
" -.54670348e+02  .86769382e+03 -.94615355e+04  .52840809e+05  .19829500e-01", 
" -.53069405e+02  .82372166e+03 -.89345259e+04  .50795965e+05  .18106849e-01", 
" -.51814098e+02  .78882846e+03 -.84592960e+04  .48164128e+05  .16231397e-01", 
" -.50772604e+02  .75980438e+03 -.80689198e+04  .46099032e+05  .15093151e-01", 
" -.49893707e+02  .73748139e+03 -.78221693e+04  .45167016e+05  .14229161e-01", 
" -.49126009e+02  .71576068e+03 -.74657675e+04  .42305768e+05  .13533216e-01", 
" -.48442573e+02  .69784680e+03 -.72509915e+04  .41432408e+05  .12183440e-01", 
" -.47829262e+02  .68164552e+03 -.70273727e+04  .40032571e+05  .11652329e-01", 
" -.45441749e+02  .62188375e+03 -.62847788e+04  .36251101e+05  .95301128e-02", 
" -.43710078e+02  .57878694e+03 -.56846869e+04  .32139088e+05  .79874622e-02", 
" -.42346181e+02  .54624822e+03 -.52722903e+04  .29711034e+05  .72482770e-02", 
" -.41208187e+02  .51772642e+03 -.48168072e+04  .25896220e+05  .67163957e-02", 
" -.40240316e+02  .49523717e+03 -.45111976e+04  .23755997e+05  .65526643e-02", 
" -.39393524e+02  .47608140e+03 -.42564194e+04  .21939577e+05  .63116579e-02", 
" -.38640511e+02  .46009525e+03 -.40802452e+04  .21030661e+05  .60168924e-02", 
" -.37957656e+02  .44526361e+03 -.38932028e+04  .19797751e+05  .58767105e-02", 
" -.37336019e+02  .43234848e+03 -.37578489e+04  .19255898e+05  .56338772e-02", 
" -.36761279e+02  .42010914e+03 -.35933634e+04  .17954637e+05  .55390161e-02", 
" -.36227683e+02  .40899115e+03 -.34537320e+04  .16952279e+05  .51844770e-02", 
" -.35730425e+02  .39885620e+03 -.33341609e+04  .16179644e+05  .49680959e-02", 
" -.35262751e+02  .38917082e+03 -.32153603e+04  .15416686e+05  .48017518e-02", 
" -.34823456e+02  .38052381e+03 -.31252596e+04  .14995509e+05  .46526923e-02", 
" -.34406469e+02  .37231920e+03 -.30319300e+04  .14400061e+05  .45633504e-02", 
" -.34010797e+02  .36481917e+03 -.29628840e+04  .14179180e+05  .43634393e-02", 
" -.33632995e+02  .35737056e+03 -.28765762e+04  .13660708e+05  .42278524e-02", 
" -.33271579e+02  .35030132e+03 -.27923208e+04  .13096116e+05  .42468804e-02", 
" -.32927954e+02  .34369191e+03 -.27119203e+04  .12529588e+05  .42021056e-02", 
" -.32597852e+02  .33729449e+03 -.26326508e+04  .11983534e+05  .41679929e-02", 
" -.32279472e+02  .33136077e+03 -.25674590e+04  .11569520e+05  .39889585e-02", 
" -.31973203e+02  .32594034e+03 -.25226185e+04  .11477215e+05  .39129117e-02", 
" -.31677159e+02  .32054899e+03 -.24704494e+04  .11269008e+05  .38608921e-02", 
" -.31392571e+02  .31557163e+03 -.24271149e+04  .11119084e+05  .38515621e-02", 
" -.31115303e+02  .31031869e+03 -.23600575e+04  .10618938e+05  .38259443e-02", 
" -.30845444e+02  .30518636e+03 -.22928122e+04  .10079744e+05  .37892868e-02", 
" -.30585125e+02  .30068587e+03 -.22540296e+04  .99717596e+04  .37345133e-02", 
" -.30332056e+02  .29612546e+03 -.22026181e+04  .96542924e+04  .36295561e-02", 
" -.30085620e+02  .29183096e+03 -.21626548e+04  .94959976e+04  .36507655e-02", 
" -.29846433e+02  .28763619e+03 -.21194714e+04  .92631995e+04  .35286567e-02", 
" -.29612190e+02  .28343807e+03 -.20668835e+04  .88316133e+04  .35369612e-02", 
" -.29384404e+02  .27944626e+03 -.20236423e+04  .85762161e+04  .34975299e-02", 
" -.29161089e+02  .27552418e+03 -.19790631e+04  .82904087e+04  .34920865e-02", 
" -.28944391e+02  .27220742e+03 -.19672692e+04  .85104246e+04  .34439346e-02", 
" -.28730203e+02  .26849167e+03 -.19308086e+04  .83519887e+04  .34030288e-02", 
" -.28522204e+02  .26495236e+03 -.18912716e+04  .80764715e+04  .33663085e-02", 
" -.28317738e+02  .26156074e+03 -.18607038e+04  .79509552e+04  .33269532e-02", 
" -.28117864e+02  .25833576e+03 -.18334435e+04  .78457146e+04  .32867151e-02", 
" -.27920975e+02  .25480548e+03 -.17838465e+04  .73915546e+04  .32463421e-02", 
" -.27728791e+02  .25178123e+03 -.17617406e+04  .73560203e+04  .32641909e-02", 
" -.27540039e+02  .24892563e+03 -.17467513e+04  .74366844e+04  .31986977e-02", 
" -.27355010e+02  .24612281e+03 -.17287227e+04  .74497272e+04  .31287525e-02", 
" -.27172327e+02  .24312533e+03 -.16972831e+04  .72549015e+04  .30968662e-02", 
" -.26992955e+02  .24028234e+03 -.16738825e+04  .71998033e+04  .31179591e-02", 
" -.26816276e+02  .23753131e+03 -.16526882e+04  .71701006e+04  .30724971e-02", 
" -.26642431e+02  .23471123e+03 -.16232885e+04  .69990529e+04  .30454943e-02", 
" -.26471305e+02  .23194815e+03 -.15965478e+04  .68753125e+04  .30174152e-02", 
" -.26303817e+02  .22934667e+03 -.15717512e+04  .67420005e+04  .29559031e-02", 
" -.26138754e+02  .22692335e+03 -.15576854e+04  .67888969e+04  .29264248e-02", 
" -.25974800e+02  .22433383e+03 -.15333498e+04  .66779554e+04  .29182611e-02", 
" -.25813605e+02  .22192024e+03 -.15139086e+04  .66039110e+04  .29105212e-02", 
" -.25654458e+02  .21938923e+03 -.14864571e+04  .64054052e+04  .28847887e-02", 
" -.25497678e+02  .21704812e+03 -.14670717e+04  .63188704e+04  .28679022e-02", 
" -.25342238e+02  .21474901e+03 -.14506358e+04  .62943340e+04  .28346681e-02", 
" -.25190123e+02  .21268287e+03 -.14423929e+04  .63637481e+04  .28154294e-02", 
" -.25039317e+02  .21047474e+03 -.14259533e+04  .63338606e+04  .27978292e-02", 
" -.24889307e+02  .20808306e+03 -.13970451e+04  .60938554e+04  .27741027e-02", 
" -.24741658e+02  .20595785e+03 -.13837757e+04  .61133597e+04  .28157477e-02", 
" -.24595281e+02  .20367718e+03 -.13578245e+04  .59182362e+04  .27975822e-02", 
" -.24451167e+02  .20156752e+03 -.13383551e+04  .58014526e+04  .27901299e-02", 
" -.24308436e+02  .19952879e+03 -.13230192e+04  .57494967e+04  .27663914e-02", 
" -.24166893e+02  .19743582e+03 -.13036368e+04  .56533024e+04  .27406915e-02", 
" -.24026623e+02  .19535771e+03 -.12834963e+04  .55309505e+04  .27328935e-02", 
" -.23888044e+02  .19333266e+03 -.12632297e+04  .53850958e+04  .27490244e-02", 
" -.23750335e+02  .19120634e+03 -.12385649e+04  .51928485e+04  .27678566e-02", 
" -.23614758e+02  .18929135e+03 -.12237801e+04  .51547401e+04  .27558570e-02", 
" -.23480319e+02  .18744812e+03 -.12128458e+04  .51673520e+04  .27418796e-02", 
" -.23346181e+02  .18541222e+03 -.11901268e+04  .50016059e+04  .27561117e-02", 
" -.23213950e+02  .18349656e+03 -.11727433e+04  .49147033e+04  .27591469e-02", 
" -.23082318e+02  .18151780e+03 -.11502125e+04  .47437567e+04  .27523835e-02", 
" -.22952271e+02  .17967673e+03 -.11336814e+04  .46478766e+04  .27283792e-02", 
" -.22822727e+02  .17774312e+03 -.11117105e+04  .44826305e+04  .27065529e-02", 
" -.22695007e+02  .17606796e+03 -.11057340e+04  .45822218e+04  .27129928e-02", 
" -.22568129e+02  .17436248e+03 -.10961970e+04  .46134631e+04  .27222618e-02", 
" -.22442273e+02  .17252977e+03 -.10754871e+04  .44516914e+04  .27072800e-02", 
" -.22316247e+02  .17072011e+03 -.10599664e+04  .44103290e+04  .27090122e-02", 
" -.22191968e+02  .16900607e+03 -.10460643e+04  .43554558e+04  .26685174e-02", 
" -.22068473e+02  .16721265e+03 -.10252364e+04  .41784485e+04  .26498550e-02", 
" -.21945930e+02  .16546520e+03 -.10041339e+04  .39827576e+04  .26060687e-02", 
" -.21824259e+02  .16384299e+03 -.99240442e+03  .39566504e+04  .25964056e-02", 
" -.21703324e+02  .16211849e+03 -.97142733e+03  .37617409e+04  .26002933e-02", 
" -.21583037e+02  .16044338e+03 -.95609715e+03  .36937366e+04  .26187907e-02", 
" -.21463796e+02  .15871142e+03 -.93198878e+03  .34498681e+04  .26149461e-02", 
" -.21345194e+02  .15722101e+03 -.92560294e+03  .35036986e+04  .25953905e-02", 
" -.21227156e+02  .15565686e+03 -.91367142e+03  .34651695e+04  .25736627e-02", 
" -.21109914e+02  .15411565e+03 -.90151324e+03  .34108643e+04  .25353217e-02", 
" -.20993061e+02  .15252119e+03 -.88526431e+03  .32954392e+04  .25376056e-02", 
" -.20877396e+02  .15104069e+03 -.87197897e+03  .31920095e+04  .25028319e-02", 
" -.20762060e+02  .14956002e+03 -.86252665e+03  .32039393e+04  .25006549e-02", 
" -.20646774e+02  .14802269e+03 -.84800688e+03  .30953821e+04  .25101635e-02", 
" -.20533157e+02  .14668924e+03 -.84180748e+03  .30980988e+04  .24751321e-02", 
" -.20419299e+02  .14523947e+03 -.83231788e+03  .30894187e+04  .24592781e-02", 
" -.20306056e+02  .14385811e+03 -.82522218e+03  .31168315e+04  .25012202e-02", 
" -.20193314e+02  .14239780e+03 -.81176095e+03  .30347485e+04  .24899065e-02", 
" -.20080920e+02  .14094366e+03 -.79854203e+03  .29509159e+04  .24978698e-02", 
" -.19969249e+02  .13953664e+03 -.78869070e+03  .29344121e+04  .24848497e-02", 
" -.19857910e+02  .13817293e+03 -.78043652e+03  .29400216e+04  .24862638e-02", 
" -.19746268e+02  .13666063e+03 -.76387057e+03  .28162580e+04  .24527169e-02", 
" -.19635861e+02  .13526023e+03 -.75100932e+03  .27333800e+04  .24560479e-02", 
" -.19525804e+02  .13385473e+03 -.73749268e+03  .26457910e+04  .24385567e-02", 
" -.19415909e+02  .13245307e+03 -.72581111e+03  .26006818e+04  .24167581e-02", 
" -.19306795e+02  .13122786e+03 -.72257174e+03  .26617576e+04  .23937076e-02", 
" -.19197052e+02  .12975374e+03 -.70589722e+03  .25343209e+04  .23780808e-02", 
" -.19088186e+02  .12833090e+03 -.68824849e+03  .23612230e+04  .23880757e-02", 
" -.18979012e+02  .12693691e+03 -.67688918e+03  .23252765e+04  .23628426e-02", 
" -.18870948e+02  .12568094e+03 -.67025235e+03  .23336868e+04  .23346080e-02", 
" -.18763026e+02  .12445608e+03 -.66495681e+03  .23565454e+04  .23270395e-02", 
" -.18655014e+02  .12313202e+03 -.65426276e+03  .23105610e+04  .23253098e-02", 
" -.18547777e+02  .12190585e+03 -.64731002e+03  .23099597e+04  .23020692e-02", 
" -.18440236e+02  .12059872e+03 -.63786597e+03  .22992609e+04  .22713466e-02", 
" -.18333760e+02  .11956225e+03 -.64337944e+03  .25012585e+04  .22735734e-02", 
" -.18226323e+02  .11823726e+03 -.63169791e+03  .24362194e+04  .22665274e-02", 
" -.18119331e+02  .11693676e+03 -.61985217e+03  .23534132e+04  .22808019e-02", 
" -.18012075e+02  .11567780e+03 -.61079794e+03  .23074793e+04  .22933430e-02", 
" -.17904913e+02  .11431625e+03 -.59549907e+03  .21818945e+04  .22891441e-02", 
" -.17798509e+02  .11317165e+03 -.59245620e+03  .22454517e+04  .22754868e-02", 
" -.17691657e+02  .11180548e+03 -.57592784e+03  .20922999e+04  .23008688e-02", 
" -.17585257e+02  .11049895e+03 -.56090435e+03  .19536163e+04  .23129499e-02", 
" -.17478734e+02  .10922685e+03 -.54904158e+03  .18735360e+04  .23104276e-02", 
" -.17372241e+02  .10800501e+03 -.54186149e+03  .18792971e+04  .23271094e-02", 
" -.17266236e+02  .10676450e+03 -.52806777e+03  .17401081e+04  .23215052e-02", 
" -.17159510e+02  .10549483e+03 -.51632205e+03  .16586178e+04  .23246705e-02", 
" -.17053682e+02  .10435844e+03 -.51224926e+03  .17096419e+04  .23115679e-02", 
" -.16947601e+02  .10317281e+03 -.50498856e+03  .17121668e+04  .23034252e-02", 
" -.16841444e+02  .10205541e+03 -.50013555e+03  .17295237e+04  .22874438e-02", 
" -.16734552e+02  .10080303e+03 -.48981522e+03  .16835834e+04  .22928999e-02", 
" -.16628210e+02  .99666961e+02 -.48430291e+03  .17030609e+04  .22793846e-02", 
" -.16522017e+02  .98569424e+02 -.47974011e+03  .17312659e+04  .22812776e-02", 
" -.16415020e+02  .97313418e+02 -.46484706e+03  .15690684e+04  .22833333e-02", 
" -.16308545e+02  .96178409e+02 -.45715576e+03  .15266944e+04  .22728391e-02", 
" -.16201597e+02  .94962473e+02 -.44508962e+03  .14211979e+04  .22539413e-02", 
" -.16094988e+02  .93857385e+02 -.43709821e+03  .13673399e+04  .22430389e-02", 
" -.15987358e+02  .92573563e+02 -.42146038e+03  .12250120e+04  .22658804e-02", 
" -.15879758e+02  .91425964e+02 -.41430196e+03  .12156450e+04  .22674562e-02", 
" -.15772183e+02  .90300574e+02 -.40727080e+03  .11943938e+04  .22476006e-02", 
" -.15664105e+02  .89174869e+02 -.39949398e+03  .11501509e+04  .22275096e-02", 
" -.15555574e+02  .87980517e+02 -.38908838e+03  .10810252e+04  .22067636e-02", 
" -.15446792e+02  .86780037e+02 -.37496606e+03  .92407572e+03  .22186662e-02", 
" -.15337688e+02  .85577041e+02 -.36198334e+03  .80017144e+03  .22091247e-02", 
" -.15228057e+02  .84420494e+02 -.35362445e+03  .76612236e+03  .22102670e-02", 
" -.15118454e+02  .83306226e+02 -.34583262e+03  .72275817e+03  .22031311e-02", 
" -.15008302e+02  .82218645e+02 -.34191111e+03  .76562487e+03  .21625686e-02", 
" -.14897819e+02  .81052643e+02 -.32920344e+03  .61822832e+03  .21510938e-02", 
" -.14786657e+02  .79906827e+02 -.32032703e+03  .56362805e+03  .21852048e-02", 
" -.14674894e+02  .78793161e+02 -.31452481e+03  .56509934e+03  .21689340e-02", 
" -.14562318e+02  .77616995e+02 -.30569789e+03  .52291541e+03  .21703270e-02", 
" -.14449859e+02  .76585943e+02 -.30202510e+03  .52931743e+03  .21807416e-02", 
" -.14337093e+02  .75575376e+02 -.29948825e+03  .55423831e+03  .21756779e-02", 
" -.14223086e+02  .74392894e+02 -.28719750e+03  .42402345e+03  .21583529e-02", 
" -.14108058e+02  .73137858e+02 -.27272413e+03  .28994147e+03  .21314830e-02", 
" -.13991970e+02  .71877704e+02 -.25904189e+03  .17430318e+03  .21230666e-02", 
" -.13876210e+02  .70862307e+02 -.25718189e+03  .21138596e+03  .21236864e-02", 
" -.13759091e+02  .69718796e+02 -.24876612e+03  .15700585e+03  .21341186e-02", 
" -.13641275e+02  .68635578e+02 -.24561467e+03  .20965714e+03  .21213479e-02", 
" -.13522141e+02  .67423167e+02 -.23388718e+03  .11156332e+03  .21027842e-02", 
" -.13402312e+02  .66159883e+02 -.21758073e+03 -.64715504e+02  .20944254e-02", 
" -.13281651e+02  .65056195e+02 -.21116505e+03 -.98060335e+02  .21029113e-02", 
" -.13160002e+02  .64004281e+02 -.20822144e+03 -.77749154e+02  .20966517e-02", 
" -.13036505e+02  .62760089e+02 -.19633350e+03 -.16711515e+03  .20842016e-02", 
" -.12912364e+02  .61711949e+02 -.19675107e+03 -.72245669e+02  .20951776e-02", 
" -.12786613e+02  .60525294e+02 -.18851857e+03 -.96600629e+02  .20931524e-02", 
" -.12659459e+02  .59388956e+02 -.18700454e+03  .15123162e+02  .20829165e-02", 
" -.12531986e+02  .58317883e+02 -.18351742e+03  .30936696e+02  .21090156e-02", 
" -.12401788e+02  .57031382e+02 -.16937338e+03 -.98293330e+02  .21649905e-02", 
" -.12270443e+02  .55928463e+02 -.16600661e+03 -.69538325e+02  .21626186e-02", 
" -.12137179e+02  .54642995e+02 -.15154837e+03 -.21663823e+03  .21546133e-02", 
" -.12002536e+02  .53474853e+02 -.14390452e+03 -.26493074e+03  .21627199e-02", 
" -.11864936e+02  .52141793e+02 -.12640346e+03 -.47798733e+03  .21509519e-02", 
" -.11725806e+02  .50890912e+02 -.11616023e+03 -.56107371e+03  .21688509e-02", 
" -.11584327e+02  .49797959e+02 -.11895046e+03 -.42284796e+03  .21436448e-02", 
" -.11440531e+02  .48651371e+02 -.11772208e+03 -.34710257e+03  .21311428e-02", 
" -.11294722e+02  .47542062e+02 -.11791491e+03 -.26298006e+03  .21950466e-02", 
" -.11145685e+02  .46386721e+02 -.11465785e+03 -.24611107e+03  .22120940e-02", 
" -.10992999e+02  .45098948e+02 -.10497982e+03 -.33023483e+03  .22785786e-02", 
" -.10836863e+02  .43824344e+02 -.99360278e+02 -.31839169e+03  .22889196e-02", 
" -.10677315e+02  .42425513e+02 -.86040530e+02 -.44162688e+03  .22914530e-02", 
" -.10513429e+02  .40943601e+02 -.69091072e+02 -.62565254e+03  .22912032e-02", 
" -.10345670e+02  .39500505e+02 -.55017591e+02 -.75532175e+03  .23342137e-02", 
" -.10173764e+02  .38137742e+02 -.45094779e+02 -.83832184e+03  .22925679e-02", 
" -.99966959e+01  .36903802e+02 -.43528538e+02 -.81242163e+03  .23224344e-02", 
" -.98138708e+01  .35431268e+02 -.34320560e+02 -.83651114e+03  .23811384e-02", 
" -.96245475e+01  .33924669e+02 -.23914842e+02 -.90632118e+03  .23609114e-02", 
" -.94279822e+01  .32311516e+02 -.85135379e+01 -.10610282e+04  .23783576e-02", 
" -.92251939e+01  .30911951e+02 -.55216224e+01 -.10266225e+04  .23866922e-02", 
" -.90117131e+01  .29444135e+02 -.51774085e+01 -.93554300e+03  .24298355e-02", 
" -.87881721e+01  .27908925e+02 -.27954905e+01 -.85023132e+03  .25204476e-02", 
" -.85524546e+01  .26475059e+02 -.14884328e+02 -.51063100e+03  .25334947e-02", 
" -.83025328e+01  .24657725e+02 -.25256318e+00 -.66501027e+03  .25475066e-02", 
" -.80356996e+01  .22856502e+02  .41784337e+01 -.61005924e+03  .25652010e-02", 
" -.77489239e+01  .20859701e+02  .18938167e+02 -.76991304e+03  .26234915e-02", 
" -.74371854e+01  .18742332e+02  .31002427e+02 -.85461434e+03  .27887289e-02", 
" -.70921762e+01  .16144851e+02  .60402402e+02 -.11646755e+04  .28904189e-02", 
" -.67038870e+01  .13744770e+02  .60501076e+02 -.98838422e+03  .29307479e-02", 
" -.62559057e+01  .11282078e+02  .48257696e+02 -.61603826e+03  .31069823e-02", 
" -.57138606e+01  .79440513e+01  .63545665e+02 -.65556397e+03  .33164506e-02", 
" -.50118312e+01  .35678295e+01  .10805963e+03 -.12033446e+04  .36642782e-02", 
" -.48391019e+01  .27819040e+01  .97703658e+02 -.94454977e+03  .38326557e-02", 
" -.46516108e+01  .23102831e+01  .66478413e+02 -.39984846e+03  .39663941e-02", 
" -.44435808e+01  .14899120e+01  .58503719e+02 -.25247330e+03  .42351105e-02", 
" -.42088918e+01  .79553694e+00  .29839446e+02  .26335628e+03  .45243340e-02", 
" -.39375011e+01 -.66179961e+00  .35933532e+02  .25034235e+03  .49625718e-02", 
" -.36124443e+01 -.24521348e+01  .52250507e+02  .24063312e+02  .55044868e-02", 
" -.32103717e+01 -.47759240e+01  .72067803e+02 -.71000662e+02  .60023181e-02", 
" -.26739008e+01 -.58839363e+01  .18188264e+02  .67482666e+03  .69567554e-02", 
" -.18144771e+01 -.82773868e+01  .34614549e+02  .13977692e+03  .89097386e-02", 
" -.10103425e+01 -.12219225e+02  .19921162e+03 -.27368455e+04  .11450486e-01", 
" -.75550218e-01 -.13358735e+02  .35756408e+03 -.67105916e+04  .16025605e-01", 
"  .57669251e+00 -.19518980e+02  .79552102e+03 -.13784133e+05  .20360372e-01", 
"coact     4  7  3   25", " -.79905735e+02  .17791875e+04 -.25895752e+05  .18378721e+06  .96527528e-01", 
" -.76168272e+02  .16335448e+04 -.23411119e+05  .16715927e+06  .66877445e-01", 
" -.71158196e+02  .14378166e+04 -.19637398e+05  .13577542e+06  .43305605e-01", 
" -.67360088e+02  .12961444e+04 -.16871800e+05  .11149394e+06  .31872159e-01", 
" -.63476067e+02  .11617965e+04 -.14522349e+05  .93134451e+05  .22979236e-01", 
" -.61142907e+02  .10826631e+04 -.13116223e+05  .81830951e+05  .19923473e-01", 
" -.59466761e+02  .10291085e+04 -.12284339e+05  .76391893e+05  .17315174e-01", 
" -.58148225e+02  .98776383e+03 -.11626789e+05  .71593226e+05  .15607135e-01", 
" -.57065233e+02  .95428921e+03 -.11088240e+05  .67565458e+05  .14298566e-01", 
" -.56144946e+02  .92821543e+03 -.10763550e+05  .66087021e+05  .13783643e-01", 
" -.55330096e+02  .90299904e+03 -.10352460e+05  .63003313e+05  .12854650e-01", 
" -.54606884e+02  .87946630e+03 -.99104616e+04  .59170277e+05  .12296916e-01", 
" -.53962767e+02  .86082184e+03 -.96370670e+04  .57360454e+05  .11949394e-01", 
" -.51453270e+02  .78974702e+03 -.86065753e+04  .50605747e+05  .10204100e-01", 
" -.49628942e+02  .74116583e+03 -.79868790e+04  .47294611e+05  .93627114e-02", 
" -.48186015e+02  .70226648e+03 -.74173366e+04  .43207704e+05  .82416602e-02", 
" -.46990071e+02  .67032402e+03 -.69461643e+04  .39915117e+05  .76524116e-02", 
" -.45960706e+02  .64302084e+03 -.65361951e+04  .36928398e+05  .71925844e-02", 
" -.45062009e+02  .62034992e+03 -.62247864e+04  .34826075e+05  .69275956e-02", 
" -.44257326e+02  .59985288e+03 -.59227728e+04  .32589024e+05  .64531969e-02", 
" -.43534572e+02  .58261329e+03 -.57090477e+04  .31342185e+05  .61023824e-02", 
" -.42870284e+02  .56663752e+03 -.55108653e+04  .30282707e+05  .58174812e-02", 
" -.42257537e+02  .55142008e+03 -.52849374e+04  .28552126e+05  .56735717e-02", 
" -.41692285e+02  .53836141e+03 -.51240751e+04  .27616104e+05  .54116584e-02", 
" -.41162815e+02  .52652046e+03 -.50031977e+04  .27268692e+05  .51660999e-02", 
" -.40664682e+02  .51463238e+03 -.48323917e+04  .25981569e+05  .49957691e-02", 
" -.40194023e+02  .50367738e+03 -.46871755e+04  .25026554e+05  .48551515e-02", 
" -.39749021e+02  .49367723e+03 -.45699907e+04  .24432036e+05  .47620403e-02", 
" -.39325290e+02  .48418378e+03 -.44533176e+04  .23733303e+05  .45855698e-02", 
" -.38924248e+02  .47538830e+03 -.43470415e+04  .23095865e+05  .45367481e-02", 
" -.38539876e+02  .46714196e+03 -.42555818e+04  .22639654e+05  .44770521e-02", 
" -.38171889e+02  .45904111e+03 -.41528721e+04  .21967289e+05  .43885937e-02", 
" -.37816756e+02  .45112264e+03 -.40491512e+04  .21277565e+05  .42848274e-02", 
" -.37475883e+02  .44362206e+03 -.39492332e+04  .20565487e+05  .42557066e-02", 
" -.37145708e+02  .43627888e+03 -.38506346e+04  .19862569e+05  .41356030e-02", 
" -.36827113e+02  .42932464e+03 -.37577174e+04  .19180865e+05  .41188290e-02", 
" -.36519849e+02  .42263493e+03 -.36663650e+04  .18501328e+05  .41793302e-02", 
" -.36224866e+02  .41673746e+03 -.36066923e+04  .18224610e+05  .41780727e-02", 
" -.35936326e+02  .41086606e+03 -.35451779e+04  .17949218e+05  .41155674e-02", 
" -.35656114e+02  .40495089e+03 -.34686095e+04  .17409813e+05  .40023984e-02", 
" -.35384018e+02  .39920257e+03 -.33947032e+04  .16907936e+05  .39577075e-02", 
" -.35118509e+02  .39352244e+03 -.33178829e+04  .16349325e+05  .39391325e-02", 
" -.34861539e+02  .38840566e+03 -.32635930e+04  .16104678e+05  .38503295e-02", 
" -.34611238e+02  .38339593e+03 -.32062981e+04  .15768795e+05  .37869301e-02", 
" -.34366558e+02  .37865813e+03 -.31583813e+04  .15548451e+05  .37799403e-02", 
" -.34125577e+02  .37390557e+03 -.31117890e+04  .15406780e+05  .37499925e-02", 
" -.33891728e+02  .36913629e+03 -.30477436e+04  .14909574e+05  .36921435e-02", 
" -.33662712e+02  .36459990e+03 -.29933309e+04  .14530543e+05  .36266751e-02", 
" -.33439327e+02  .36051481e+03 -.29621688e+04  .14533377e+05  .36095688e-02", 
" -.33220110e+02  .35641789e+03 -.29219406e+04  .14350115e+05  .35273456e-02", 
" -.33003294e+02  .35202541e+03 -.28633544e+04  .13893917e+05  .35049072e-02", 
" -.32791060e+02  .34776164e+03 -.28074691e+04  .13476767e+05  .35138585e-02", 
" -.32583140e+02  .34381131e+03 -.27650277e+04  .13231233e+05  .34456896e-02", 
" -.32378448e+02  .33976253e+03 -.27149997e+04  .12905217e+05  .33799187e-02", 
" -.32177830e+02  .33575404e+03 -.26612162e+04  .12493704e+05  .33525020e-02", 
" -.31981089e+02  .33199508e+03 -.26145717e+04  .12139551e+05  .32800547e-02", 
" -.31787717e+02  .32856271e+03 -.25878262e+04  .12115069e+05  .31944923e-02", 
" -.31596966e+02  .32495356e+03 -.25471178e+04  .11869791e+05  .31442898e-02", 
" -.31408597e+02  .32124703e+03 -.24993428e+04  .11537134e+05  .30937064e-02", 
" -.31224488e+02  .31780817e+03 -.24557964e+04  .11182873e+05  .30987859e-02", 
" -.31042711e+02  .31458435e+03 -.24274609e+04  .11110724e+05  .30695065e-02", 
" -.30863068e+02  .31120690e+03 -.23852238e+04  .10795385e+05  .30721951e-02", 
" -.30685262e+02  .30777989e+03 -.23405581e+04  .10456132e+05  .30849513e-02", 
" -.30510261e+02  .30453998e+03 -.23044268e+04  .10254768e+05  .30720316e-02", 
" -.30338518e+02  .30153755e+03 -.22762961e+04  .10141469e+05  .30324659e-02", 
" -.30168595e+02  .29848488e+03 -.22435411e+04  .99452215e+04  .29707481e-02", 
" -.30000277e+02  .29545465e+03 -.22108079e+04  .97564291e+04  .29768782e-02", 
" -.29834000e+02  .29248330e+03 -.21789400e+04  .95688916e+04  .29674060e-02", 
" -.29669842e+02  .28946041e+03 -.21436593e+04  .93509740e+04  .29731108e-02", 
" -.29507230e+02  .28646448e+03 -.21061803e+04  .90681345e+04  .29424715e-02", 
" -.29347668e+02  .28379243e+03 -.20828787e+04  .89712529e+04  .29448958e-02", 
" -.29188430e+02  .28077367e+03 -.20414157e+04  .86366685e+04  .29362682e-02", 
" -.29031911e+02  .27809613e+03 -.20156113e+04  .85053903e+04  .29195606e-02", 
" -.28876715e+02  .27554331e+03 -.19984750e+04  .85266551e+04  .28909934e-02", 
" -.28723321e+02  .27278674e+03 -.19662733e+04  .83308570e+04  .28698160e-02", 
" -.28571692e+02  .27024147e+03 -.19430384e+04  .82234542e+04  .28291548e-02", 
" -.28422156e+02  .26786389e+03 -.19275814e+04  .82411649e+04  .28321434e-02", 
" -.28273486e+02  .26546917e+03 -.19098641e+04  .82105122e+04  .28187857e-02", 
" -.28126399e+02  .26305094e+03 -.18883011e+04  .81250558e+04  .28175044e-02", 
" -.27980069e+02  .26058076e+03 -.18656823e+04  .80480110e+04  .28357690e-02", 
" -.27835183e+02  .25830383e+03 -.18516231e+04  .80940076e+04  .28294189e-02", 
" -.27691890e+02  .25611838e+03 -.18416223e+04  .81907504e+04  .28111370e-02", 
" -.27549049e+02  .25379750e+03 -.18226291e+04  .81387472e+04  .27943088e-02", 
" -.27407459e+02  .25147210e+03 -.17999567e+04  .80139069e+04  .27943870e-02", 
" -.27267600e+02  .24919785e+03 -.17785526e+04  .79160620e+04  .27756922e-02", 
" -.27128336e+02  .24695509e+03 -.17591307e+04  .78445306e+04  .27677135e-02", 
" -.26989283e+02  .24453133e+03 -.17286280e+04  .76041869e+04  .27438968e-02", 
" -.26851584e+02  .24220757e+03 -.17025281e+04  .74251746e+04  .27181484e-02", 
" -.26714306e+02  .23972654e+03 -.16665981e+04  .70909605e+04  .27136042e-02", 
" -.26579347e+02  .23757486e+03 -.16486727e+04  .70453775e+04  .27203952e-02", 
" -.26444754e+02  .23533104e+03 -.16246738e+04  .69076728e+04  .27202941e-02", 
" -.26311118e+02  .23319009e+03 -.16061089e+04  .68495898e+04  .27242554e-02", 
" -.26179413e+02  .23122121e+03 -.15937594e+04  .68622705e+04  .26892339e-02", 
" -.26048290e+02  .22932254e+03 -.15858833e+04  .69557136e+04  .26736672e-02", 
" -.25917434e+02  .22739749e+03 -.15758654e+04  .70063760e+04  .26319965e-02", 
" -.25786830e+02  .22535452e+03 -.15585925e+04  .69446115e+04  .26101858e-02", 
" -.25657598e+02  .22337939e+03 -.15421979e+04  .68836821e+04  .25778543e-02", 
" -.25528376e+02  .22128889e+03 -.15205936e+04  .67642376e+04  .25236170e-02", 
" -.25400234e+02  .21921113e+03 -.14964514e+04  .65832559e+04  .25287266e-02", 
" -.25272479e+02  .21726225e+03 -.14819834e+04  .65683975e+04  .25400084e-02", 
" -.25146461e+02  .21549227e+03 -.14746285e+04  .66389641e+04  .25693530e-02", 
" -.25019754e+02  .21339697e+03 -.14460475e+04  .63567105e+04  .25584329e-02", 
" -.24894271e+02  .21157190e+03 -.14342037e+04  .63527955e+04  .25814483e-02", 
" -.24769141e+02  .20958576e+03 -.14119335e+04  .61978166e+04  .25748366e-02", 
" -.24645245e+02  .20784415e+03 -.14035416e+04  .62611305e+04  .25654601e-02", 
" -.24521646e+02  .20609730e+03 -.13946023e+04  .63159135e+04  .25453298e-02", 
" -.24397740e+02  .20408560e+03 -.13702999e+04  .61375836e+04  .25465727e-02", 
" -.24274635e+02  .20215896e+03 -.13490938e+04  .59897728e+04  .25363251e-02", 
" -.24151671e+02  .20018810e+03 -.13243904e+04  .57788314e+04  .24945589e-02", 
" -.24028915e+02  .19821906e+03 -.13000041e+04  .55698528e+04  .24755810e-02", 
" -.23907777e+02  .19652026e+03 -.12905439e+04  .56055184e+04  .24998928e-02", 
" -.23786232e+02  .19463299e+03 -.12699618e+04  .54636946e+04  .24861438e-02", 
" -.23664332e+02  .19265327e+03 -.12439512e+04  .52365776e+04  .24810522e-02", 
" -.23543685e+02  .19076943e+03 -.12211185e+04  .50527146e+04  .24448196e-02", 
" -.23423020e+02  .18904588e+03 -.12111394e+04  .50960386e+04  .24496641e-02", 
" -.23302198e+02  .18711225e+03 -.11881084e+04  .49336708e+04  .24781716e-02", 
" -.23182601e+02  .18537759e+03 -.11741461e+04  .48955344e+04  .24839848e-02", 
" -.23062795e+02  .18349428e+03 -.11514985e+04  .47268019e+04  .24761606e-02", 
" -.22944350e+02  .18189402e+03 -.11432476e+04  .47634035e+04  .24935263e-02", 
" -.22825759e+02  .18023906e+03 -.11311772e+04  .47375133e+04  .25081967e-02", 
" -.22707423e+02  .17855709e+03 -.11183608e+04  .47192889e+04  .25057519e-02", 
" -.22588616e+02  .17682735e+03 -.11022876e+04  .46318681e+04  .25360184e-02", 
" -.22470229e+02  .17515422e+03 -.10877896e+04  .45501769e+04  .25316650e-02", 
" -.22352402e+02  .17359401e+03 -.10798949e+04  .45730013e+04  .25163047e-02", 
" -.22234041e+02  .17180686e+03 -.10588213e+04  .44011042e+04  .25289104e-02", 
" -.22116094e+02  .17006261e+03 -.10398050e+04  .42633725e+04  .25188221e-02", 
" -.21998052e+02  .16831303e+03 -.10211704e+04  .41440147e+04  .25147825e-02", 
" -.21880671e+02  .16675445e+03 -.10136423e+04  .41972618e+04  .25175067e-02", 
" -.21763725e+02  .16513229e+03 -.10002234e+04  .41488952e+04  .24957973e-02", 
" -.21646676e+02  .16356109e+03 -.99025304e+03  .41668341e+04  .25110310e-02", 
" -.21529332e+02  .16182845e+03 -.96848213e+03  .39693420e+04  .25073751e-02", 
" -.21412131e+02  .16022197e+03 -.95586574e+03  .39333075e+04  .25123914e-02", 
" -.21295000e+02  .15871596e+03 -.94837088e+03  .39570031e+04  .25174042e-02", 
" -.21178010e+02  .15722636e+03 -.94011437e+03  .39655639e+04  .24766497e-02", 
" -.21060423e+02  .15561732e+03 -.92738165e+03  .39190865e+04  .24825885e-02", 
" -.20943781e+02  .15413376e+03 -.91781857e+03  .38942929e+04  .24825381e-02", 
" -.20826222e+02  .15239789e+03 -.89403500e+03  .36674650e+04  .24684590e-02", 
" -.20708478e+02  .15071440e+03 -.87396382e+03  .34924022e+04  .24512200e-02", 
" -.20591206e+02  .14919322e+03 -.86347921e+03  .34697402e+04  .24427587e-02", 
" -.20473522e+02  .14761304e+03 -.84798488e+03  .33598670e+04  .24146631e-02", 
" -.20355278e+02  .14593317e+03 -.82739176e+03  .31746361e+04  .23972466e-02", 
" -.20237148e+02  .14437322e+03 -.81529469e+03  .31399280e+04  .23975236e-02", 
" -.20119263e+02  .14293357e+03 -.80784949e+03  .31545946e+04  .23716267e-02", 
" -.20001049e+02  .14136646e+03 -.79332002e+03  .30666039e+04  .23715901e-02", 
" -.19882947e+02  .13991763e+03 -.78531767e+03  .30758917e+04  .23485219e-02", 
" -.19763870e+02  .13836189e+03 -.77260196e+03  .30183606e+04  .23348982e-02", 
" -.19644691e+02  .13674567e+03 -.75471613e+03  .28693571e+04  .23470428e-02", 
" -.19525084e+02  .13511207e+03 -.73647803e+03  .27338683e+04  .23484756e-02", 
" -.19405226e+02  .13355991e+03 -.72304909e+03  .26603697e+04  .23391698e-02", 
" -.19284829e+02  .13202281e+03 -.71170404e+03  .26333677e+04  .23266175e-02", 
" -.19164494e+02  .13053540e+03 -.70153035e+03  .26086342e+04  .23279140e-02", 
" -.19044028e+02  .12901147e+03 -.68763128e+03  .25205433e+04  .23084905e-02", 
" -.18921832e+02  .12739948e+03 -.67242053e+03  .24300985e+04  .23119616e-02", 
" -.18799726e+02  .12582842e+03 -.65854570e+03  .23641433e+04  .23167082e-02", 
" -.18677453e+02  .12427974e+03 -.64339658e+03  .22462970e+04  .22822045e-02", 
" -.18554227e+02  .12277102e+03 -.63287788e+03  .22109040e+04  .22806685e-02", 
" -.18430865e+02  .12133153e+03 -.62583575e+03  .22371178e+04  .22980816e-02", 
" -.18306235e+02  .11967334e+03 -.60666592e+03  .20726233e+04  .22838706e-02", 
" -.18181439e+02  .11813404e+03 -.59303095e+03  .19830289e+04  .22700817e-02", 
" -.18056080e+02  .11651660e+03 -.57483894e+03  .18295006e+04  .22544475e-02", 
" -.17929523e+02  .11491633e+03 -.55890994e+03  .17051064e+04  .22912243e-02", 
" -.17802455e+02  .11339992e+03 -.54788058e+03  .16671852e+04  .23272311e-02", 
" -.17674424e+02  .11187460e+03 -.53611725e+03  .16058983e+04  .22936371e-02", 
" -.17545676e+02  .11039942e+03 -.52789853e+03  .16061390e+04  .22797066e-02", 
" -.17416509e+02  .10894386e+03 -.52179947e+03  .16570626e+04  .22902879e-02", 
" -.17285237e+02  .10728889e+03 -.50315812e+03  .14935808e+04  .22710477e-02", 
" -.17153436e+02  .10564101e+03 -.48682802e+03  .13839878e+04  .22472525e-02", 
" -.17019859e+02  .10394951e+03 -.46721490e+03  .12089419e+04  .22315364e-02", 
" -.16886158e+02  .10254253e+03 -.46589353e+03  .13289275e+04  .22574818e-02", 
" -.16750791e+02  .10096820e+03 -.45436018e+03  .12812516e+04  .22761863e-02", 
" -.16614056e+02  .99343780e+02 -.44003073e+03  .11946759e+04  .23183962e-02", 
" -.16476015e+02  .97650672e+02 -.42003396e+03  .10010514e+04  .23655000e-02", 
" -.16337433e+02  .96285844e+02 -.42041586e+03  .11310769e+04  .23465060e-02", 
" -.16197174e+02  .94840596e+02 -.41933657e+03  .12645104e+04  .23252991e-02", 
" -.16054027e+02  .92997900e+02 -.39403205e+03  .10091017e+04  .23421103e-02", 
" -.15910361e+02  .91386886e+02 -.38059727e+03  .92075524e+03  .23200490e-02", 
" -.15764829e+02  .89817324e+02 -.37036017e+03  .86628972e+03  .23604399e-02", 
" -.15616374e+02  .88041185e+02 -.35302945e+03  .75458727e+03  .23378075e-02", 
" -.15466883e+02  .86472749e+02 -.34738134e+03  .82503116e+03  .23475661e-02", 
" -.15314648e+02  .84673521e+02 -.32717135e+03  .64547145e+03  .23198709e-02", 
" -.15160552e+02  .83140927e+02 -.32350275e+03  .72378077e+03  .23089537e-02", 
" -.15003674e+02  .81519884e+02 -.31477838e+03  .70178089e+03  .22742984e-02", 
" -.14844610e+02  .79865477e+02 -.30540372e+03  .68639233e+03  .22604749e-02", 
" -.14682373e+02  .78017284e+02 -.28389013e+03  .47189644e+03  .22422294e-02", 
" -.14517255e+02  .76273335e+02 -.27012417e+03  .37690054e+03  .22801035e-02", 
" -.14348948e+02  .74473850e+02 -.25420750e+03  .24953804e+03  .22948414e-02", 
" -.14176508e+02  .72656460e+02 -.24193732e+03  .20661508e+03  .23335998e-02", 
" -.14000912e+02  .70852078e+02 -.22856359e+03  .12397749e+03  .23569164e-02", 
" -.13819294e+02  .68766597e+02 -.20467667e+03 -.89009908e+02  .23941011e-02", 
" -.13633943e+02  .66741483e+02 -.18641044e+03 -.18753861e+03  .24083539e-02", 
" -.13444217e+02  .64674317e+02 -.16446134e+03 -.38735217e+03  .24482770e-02", 
" -.13249209e+02  .62633266e+02 -.14763277e+03 -.48618885e+03  .24654986e-02", 
" -.13048145e+02  .60464754e+02 -.12375842e+03 -.69771871e+03  .25150386e-02", 
" -.12841149e+02  .58435506e+02 -.10720860e+03 -.82774250e+03  .24713238e-02", 
" -.12626564e+02  .56354291e+02 -.95260792e+02 -.84603742e+03  .24683009e-02", 
" -.12405120e+02  .54413403e+02 -.94445685e+02 -.67210956e+03  .25065726e-02", 
" -.12173681e+02  .52318361e+02 -.87806740e+02 -.60941162e+03  .25063235e-02", 
" -.11932323e+02  .50004161e+02 -.65700531e+02 -.85790865e+03  .25372255e-02", 
" -.11678036e+02  .47462949e+02 -.41730036e+02 -.10368090e+04  .25941504e-02", 
" -.11410693e+02  .45105821e+02 -.35313416e+02 -.93345155e+03  .25703271e-02", 
" -.11126316e+02  .42422794e+02 -.13812270e+02 -.10973915e+04  .27294614e-02", 
" -.10824163e+02  .39944092e+02 -.11962387e+02 -.91996896e+03  .27708565e-02", 
" -.10498491e+02  .37109072e+02  .40521846e+01 -.97030677e+03  .28512046e-02", 
" -.10141590e+02  .33831229e+02  .26008873e+02 -.10562949e+04  .28535152e-02", 
" -.97484970e+01  .29729210e+02  .89950660e+02 -.18043359e+04  .29084263e-02", 
" -.93078334e+01  .25671584e+02  .13573973e+03 -.22797263e+04  .30975802e-02", 
" -.87996060e+01  .21531319e+02  .15705221e+03 -.23250531e+04  .33541040e-02", 
" -.81886878e+01  .17125253e+02  .15179253e+03 -.18780534e+04  .35210745e-02", 
" -.73901941e+01  .96793888e+01  .24947214e+03 -.29742938e+04  .40358782e-02", 
" -.71943912e+01  .78660606e+01  .26293869e+03 -.29703564e+04  .42087247e-02", 
" -.69803910e+01  .57105164e+01  .30085200e+03 -.34697176e+04  .43701397e-02", 
" -.67462792e+01  .42413939e+01  .28642850e+03 -.31265364e+04  .44190713e-02", 
" -.64827369e+01  .21319323e+01  .30094917e+03 -.32082232e+04  .46044881e-02", 
" -.61801445e+01  .44996132e-02  .28484063e+03 -.26515621e+04  .49899677e-02", 
" -.58241457e+01 -.17636265e+01  .22489122e+03 -.14877028e+04  .53746594e-02", 
" -.53875334e+01 -.47211742e+01  .22212000e+03 -.12347319e+04  .58572842e-02", 
" -.48018737e+01 -.95550446e+01  .26458904e+03 -.15956186e+04  .70639436e-02", 
" -.38766583e+01 -.13479689e+02  .92626880e+02  .14901290e+04  .90939089e-02", 
" -.30152282e+01 -.22384524e+02  .26626258e+03 -.51982282e+03  .11775223e-01", 
" -.19671160e+01 -.31275270e+02  .43508286e+03 -.33519105e+04  .17318840e-01", 
" -.12433518e+01 -.36952004e+02  .53585209e+03 -.50705332e+04  .21658256e-01", 
"coactt    4  8  3   25", " -.86980005e+02  .20953957e+04 -.32268313e+05  .23522451e+06  .96931825e-01", 
" -.83149656e+02  .19225075e+04 -.28633990e+05  .20491543e+06  .70390856e-01", 
" -.77988079e+02  .17078505e+04 -.24431385e+05  .17093166e+06  .46795197e-01", 
" -.73951939e+02  .15411892e+04 -.21062575e+05  .14193507e+06  .33436638e-01", 
" -.69886142e+02  .13918419e+04 -.18495125e+05  .12339474e+06  .23821099e-01", 
" -.67464034e+02  .13028616e+04 -.16847664e+05  .10992545e+06  .20519962e-01", 
" -.65714687e+02  .12400309e+04 -.15709313e+05  .10086981e+06  .18766458e-01", 
" -.64341855e+02  .11931729e+04 -.14931436e+05  .95285194e+05  .17207745e-01", 
" -.63213798e+02  .11560968e+04 -.14346334e+05  .91243202e+05  .15972428e-01", 
" -.62252475e+02  .11240219e+04 -.13824325e+05  .87633463e+05  .14756469e-01", 
" -.61405716e+02  .10956896e+04 -.13339722e+05  .83959889e+05  .13936767e-01", 
" -.60658517e+02  .10712972e+04 -.12925409e+05  .80642007e+05  .13365041e-01", 
" -.59988883e+02  .10498951e+04 -.12587543e+05  .78312798e+05  .12490747e-01", 
" -.57339349e+02  .96456301e+03 -.11166213e+05  .67607272e+05  .10187673e-01", 
" -.55418361e+02  .90592327e+03 -.10264388e+05  .61410937e+05  .89236800e-02", 
" -.53909085e+02  .86154599e+03 -.95999783e+04  .56921841e+05  .84074433e-02", 
" -.52653890e+02  .82490823e+03 -.90326839e+04  .52772380e+05  .75689107e-02", 
" -.51573984e+02  .79334103e+03 -.85255769e+04  .48854487e+05  .71559901e-02", 
" -.50627849e+02  .76668280e+03 -.81253753e+04  .46009457e+05  .69003314e-02", 
" -.49788409e+02  .74451587e+03 -.78479142e+04  .44625033e+05  .64220315e-02", 
" -.49026477e+02  .72440104e+03 -.75896890e+04  .43240156e+05  .62272569e-02", 
" -.48324873e+02  .70514731e+03 -.73018576e+04  .41188106e+05  .60988298e-02", 
" -.47680352e+02  .68799523e+03 -.70606866e+04  .39614315e+05  .60124581e-02", 
" -.47077940e+02  .67127261e+03 -.67876319e+04  .37397590e+05  .58489679e-02", 
" -.46517828e+02  .65692645e+03 -.66024212e+04  .36337967e+05  .56448615e-02", 
" -.45993273e+02  .64362255e+03 -.64287890e+04  .35308050e+05  .54441753e-02", 
" -.45494930e+02  .63066421e+03 -.62408809e+04  .33967934e+05  .53627583e-02", 
" -.45024983e+02  .61872342e+03 -.60753243e+04  .32846625e+05  .51686556e-02", 
" -.44577874e+02  .60751597e+03 -.59254112e+04  .31903565e+05  .50713392e-02", 
" -.44153404e+02  .59719838e+03 -.57960355e+04  .31159805e+05  .49692732e-02", 
" -.43745879e+02  .58728542e+03 -.56743613e+04  .30537403e+05  .47941510e-02", 
" -.43353966e+02  .57743463e+03 -.55350824e+04  .29565225e+05  .46914973e-02", 
" -.42979513e+02  .56837208e+03 -.54190635e+04  .28879839e+05  .46242225e-02", 
" -.42617869e+02  .55935347e+03 -.52912949e+04  .28009488e+05  .45851487e-02", 
" -.42268674e+02  .55085167e+03 -.51779856e+04  .27310977e+05  .45489053e-02", 
" -.41933996e+02  .54324771e+03 -.50966441e+04  .26992055e+05  .44042997e-02", 
" -.41609320e+02  .53575970e+03 -.50098622e+04  .26552857e+05  .43603416e-02", 
" -.41294569e+02  .52845012e+03 -.49197463e+04  .26024933e+05  .42787531e-02", 
" -.40989963e+02  .52160965e+03 -.48433046e+04  .25660303e+05  .41469504e-02", 
" -.40693820e+02  .51495739e+03 -.47676337e+04  .25270889e+05  .41041649e-02", 
" -.40405457e+02  .50827895e+03 -.46798143e+04  .24682625e+05  .40626803e-02", 
" -.40124768e+02  .50185150e+03 -.45974675e+04  .24145551e+05  .40319306e-02", 
" -.39851237e+02  .49560372e+03 -.45171862e+04  .23620260e+05  .39463109e-02", 
" -.39583378e+02  .48951269e+03 -.44434171e+04  .23219288e+05  .38191536e-02", 
" -.39320941e+02  .48340318e+03 -.43613094e+04  .22671847e+05  .37780823e-02", 
" -.39064714e+02  .47751580e+03 -.42845666e+04  .22167419e+05  .37919341e-02", 
" -.38814514e+02  .47143554e+03 -.41836991e+04  .21240654e+05  .37464804e-02", 
" -.38571151e+02  .46617874e+03 -.41256503e+04  .20967875e+05  .37098173e-02", 
" -.38331908e+02  .46085310e+03 -.40578639e+04  .20520544e+05  .37065286e-02", 
" -.38097824e+02  .45587342e+03 -.40073920e+04  .20361969e+05  .36399330e-02", 
" -.37868996e+02  .45080034e+03 -.39376661e+04  .19816604e+05  .36142140e-02", 
" -.37643616e+02  .44590689e+03 -.38804294e+04  .19504669e+05  .35961723e-02", 
" -.37422256e+02  .44097658e+03 -.38163284e+04  .19085630e+05  .35773760e-02", 
" -.37204166e+02  .43597427e+03 -.37396481e+04  .18414141e+05  .35471983e-02", 
" -.36990563e+02  .43140028e+03 -.36842159e+04  .18059776e+05  .35346530e-02", 
" -.36781349e+02  .42710817e+03 -.36412243e+04  .17903670e+05  .35471988e-02", 
" -.36574597e+02  .42273140e+03 -.35886290e+04  .17572652e+05  .35802943e-02", 
" -.36370643e+02  .41833647e+03 -.35352973e+04  .17268273e+05  .35146885e-02", 
" -.36170160e+02  .41400282e+03 -.34808606e+04  .16940459e+05  .34714093e-02", 
" -.35973028e+02  .40998391e+03 -.34395856e+04  .16768401e+05  .34819156e-02", 
" -.35777999e+02  .40594949e+03 -.33945280e+04  .16520290e+05  .34815581e-02", 
" -.35585787e+02  .40180835e+03 -.33399743e+04  .16144256e+05  .34767492e-02", 
" -.35396824e+02  .39788661e+03 -.32939791e+04  .15882833e+05  .34914965e-02", 
" -.35210432e+02  .39417195e+03 -.32583680e+04  .15766171e+05  .34098828e-02", 
" -.35025257e+02  .39021988e+03 -.32062451e+04  .15397433e+05  .34034438e-02", 
" -.34843544e+02  .38648598e+03 -.31623128e+04  .15132130e+05  .34239198e-02", 
" -.34663481e+02  .38275257e+03 -.31182935e+04  .14881067e+05  .34326302e-02", 
" -.34486399e+02  .37916647e+03 -.30783974e+04  .14674615e+05  .34273403e-02", 
" -.34311531e+02  .37571420e+03 -.30421827e+04  .14502275e+05  .34352575e-02", 
" -.34138069e+02  .37214094e+03 -.29959797e+04  .14159548e+05  .34048197e-02", 
" -.33967136e+02  .36879319e+03 -.29607200e+04  .13979273e+05  .34085688e-02", 
" -.33798309e+02  .36547388e+03 -.29247665e+04  .13799660e+05  .33789202e-02", 
" -.33630277e+02  .36214975e+03 -.28899366e+04  .13647738e+05  .33639649e-02", 
" -.33463362e+02  .35866307e+03 -.28407766e+04  .13240076e+05  .33230895e-02", 
" -.33299759e+02  .35555903e+03 -.28111118e+04  .13131326e+05  .33168186e-02", 
" -.33136522e+02  .35234134e+03 -.27735950e+04  .12885590e+05  .33212627e-02", 
" -.32974920e+02  .34910926e+03 -.27330637e+04  .12593983e+05  .32694566e-02", 
" -.32814052e+02  .34579607e+03 -.26896292e+04  .12288198e+05  .32157704e-02", 
" -.32656564e+02  .34291881e+03 -.26655226e+04  .12237288e+05  .31768172e-02", 
" -.32499058e+02  .33976750e+03 -.26244904e+04  .11921360e+05  .31696756e-02", 
" -.32343728e+02  .33677139e+03 -.25902332e+04  .11712654e+05  .31710555e-02", 
" -.32190023e+02  .33402157e+03 -.25715310e+04  .11762290e+05  .31192109e-02", 
" -.32036938e+02  .33105945e+03 -.25370317e+04  .11544596e+05  .31147357e-02", 
" -.31885187e+02  .32807488e+03 -.24989529e+04  .11265929e+05  .30588830e-02", 
" -.31735382e+02  .32541274e+03 -.24774348e+04  .11226787e+05  .30578817e-02", 
" -.31586560e+02  .32266389e+03 -.24492419e+04  .11076239e+05  .30418122e-02", 
" -.31437298e+02  .31973259e+03 -.24133918e+04  .10839020e+05  .30106504e-02", 
" -.31290056e+02  .31697863e+03 -.23839776e+04  .10677698e+05  .29938109e-02", 
" -.31143378e+02  .31429536e+03 -.23584431e+04  .10571867e+05  .30012142e-02", 
" -.30997477e+02  .31154355e+03 -.23271098e+04  .10378513e+05  .29785322e-02", 
" -.30853710e+02  .30904847e+03 -.23084118e+04  .10368648e+05  .29632212e-02", 
" -.30710111e+02  .30638759e+03 -.22786603e+04  .10183014e+05  .29542031e-02", 
" -.30567471e+02  .30375208e+03 -.22497130e+04  .10012256e+05  .29889125e-02", 
" -.30425710e+02  .30122669e+03 -.22265489e+04  .99283887e+04  .29689464e-02", 
" -.30284201e+02  .29848026e+03 -.21902026e+04  .96443911e+04  .29851084e-02", 
" -.30143017e+02  .29576218e+03 -.21553983e+04  .93981297e+04  .29834538e-02", 
" -.30002738e+02  .29307451e+03 -.21206814e+04  .91344151e+04  .29848489e-02", 
" -.29863364e+02  .29042246e+03 -.20866685e+04  .88748219e+04  .29355531e-02", 
" -.29725136e+02  .28783515e+03 -.20540394e+04  .86284425e+04  .29356816e-02", 
" -.29587414e+02  .28525316e+03 -.20223705e+04  .84190065e+04  .29092802e-02", 
" -.29450487e+02  .28275786e+03 -.19953832e+04  .82679529e+04  .29099045e-02", 
" -.29314422e+02  .28040825e+03 -.19738495e+04  .81814104e+04  .29123789e-02", 
" -.29179238e+02  .27802842e+03 -.19490927e+04  .80315245e+04  .28796145e-02", 
" -.29043885e+02  .27556839e+03 -.19219615e+04  .78774335e+04  .28705773e-02", 
" -.28909400e+02  .27315250e+03 -.18958000e+04  .77366541e+04  .28241963e-02", 
" -.28775011e+02  .27067945e+03 -.18663029e+04  .75435306e+04  .28382481e-02", 
" -.28642409e+02  .26842260e+03 -.18454991e+04  .74540784e+04  .28234052e-02", 
" -.28510034e+02  .26630025e+03 -.18347694e+04  .75431587e+04  .28133545e-02", 
" -.28377528e+02  .26398109e+03 -.18092637e+04  .73714612e+04  .28116140e-02", 
" -.28245641e+02  .26171482e+03 -.17874485e+04  .72674454e+04  .28079065e-02", 
" -.28114035e+02  .25941965e+03 -.17618700e+04  .70940135e+04  .28026292e-02", 
" -.27982456e+02  .25706469e+03 -.17341317e+04  .69067019e+04  .28022028e-02", 
" -.27851588e+02  .25476492e+03 -.17074144e+04  .67158129e+04  .27785490e-02", 
" -.27721053e+02  .25245024e+03 -.16795936e+04  .65130242e+04  .27684452e-02", 
" -.27591039e+02  .25022980e+03 -.16570605e+04  .63901573e+04  .27344605e-02", 
" -.27461664e+02  .24798167e+03 -.16307513e+04  .61929966e+04  .27251377e-02", 
" -.27332682e+02  .24583375e+03 -.16096816e+04  .60817398e+04  .26857953e-02", 
" -.27203970e+02  .24371138e+03 -.15908730e+04  .60128807e+04  .26459022e-02", 
" -.27075092e+02  .24153761e+03 -.15688137e+04  .58908717e+04  .26269506e-02", 
" -.26946614e+02  .23931550e+03 -.15420244e+04  .56772891e+04  .26208644e-02", 
" -.26818582e+02  .23717412e+03 -.15201522e+04  .55582038e+04  .25966580e-02", 
" -.26690898e+02  .23514167e+03 -.15033945e+04  .55049898e+04  .25584888e-02", 
" -.26562754e+02  .23295494e+03 -.14784914e+04  .53326057e+04  .25468205e-02", 
" -.26435650e+02  .23097726e+03 -.14639335e+04  .53086403e+04  .25110742e-02", 
" -.26307703e+02  .22890656e+03 -.14472740e+04  .52910303e+04  .25044826e-02", 
" -.26179773e+02  .22677319e+03 -.14272580e+04  .52244570e+04  .25042104e-02", 
" -.26052097e+02  .22461132e+03 -.14009575e+04  .50113874e+04  .25112877e-02", 
" -.25925163e+02  .22265694e+03 -.13873637e+04  .50047788e+04  .25147449e-02", 
" -.25798486e+02  .22068397e+03 -.13704864e+04  .49289120e+04  .25121144e-02", 
" -.25670879e+02  .21850241e+03 -.13419223e+04  .46776599e+04  .25046548e-02", 
" -.25543616e+02  .21636292e+03 -.13181119e+04  .45385723e+04  .25188388e-02", 
" -.25416038e+02  .21424465e+03 -.12935751e+04  .43612490e+04  .24946357e-02", 
" -.25288725e+02  .21218438e+03 -.12727254e+04  .42434316e+04  .24853392e-02", 
" -.25161412e+02  .21011524e+03 -.12505628e+04  .40940670e+04  .24963238e-02", 
" -.25034332e+02  .20805070e+03 -.12271048e+04  .39213389e+04  .24679312e-02", 
" -.24906489e+02  .20600416e+03 -.12072702e+04  .38265828e+04  .24537447e-02", 
" -.24779133e+02  .20403459e+03 -.11908792e+04  .37750482e+04  .24600726e-02", 
" -.24651241e+02  .20191416e+03 -.11648450e+04  .35716665e+04  .24762325e-02", 
" -.24523397e+02  .19987379e+03 -.11446644e+04  .34663430e+04  .24435458e-02", 
" -.24395258e+02  .19785477e+03 -.11244552e+04  .33442674e+04  .24796622e-02", 
" -.24266659e+02  .19579025e+03 -.11026371e+04  .32164443e+04  .24629603e-02", 
" -.24138735e+02  .19385948e+03 -.10856674e+04  .31267811e+04  .24782892e-02", 
" -.24009627e+02  .19174386e+03 -.10592115e+04  .29036123e+04  .24953560e-02", 
" -.23881343e+02  .18993660e+03 -.10508538e+04  .29662853e+04  .24907847e-02", 
" -.23752598e+02  .18801374e+03 -.10352219e+04  .29172055e+04  .24871081e-02", 
" -.23622521e+02  .18601273e+03 -.10169991e+04  .28393124e+04  .24988530e-02", 
" -.23492574e+02  .18407590e+03 -.10023865e+04  .28132765e+04  .24900699e-02", 
" -.23362760e+02  .18216898e+03 -.98661147e+03  .27445851e+04  .24580291e-02", 
" -.23231720e+02  .18017543e+03 -.96889206e+03  .26831920e+04  .24650647e-02", 
" -.23101088e+02  .17819573e+03 -.94746911e+03  .25243439e+04  .24715427e-02", 
" -.22969275e+02  .17615997e+03 -.92724383e+03  .24206374e+04  .24839832e-02", 
" -.22837198e+02  .17420561e+03 -.90911322e+03  .23183510e+04  .24689707e-02", 
" -.22704148e+02  .17218405e+03 -.89010825e+03  .22312814e+04  .24832047e-02", 
" -.22570777e+02  .17024739e+03 -.87544499e+03  .22005070e+04  .25114700e-02", 
" -.22437838e+02  .16844692e+03 -.86508488e+03  .22068661e+04  .25598770e-02", 
" -.22303075e+02  .16639482e+03 -.84312191e+03  .20685424e+04  .25799659e-02", 
" -.22167929e+02  .16443939e+03 -.82494711e+03  .19548655e+04  .26073687e-02", 
" -.22031721e+02  .16237102e+03 -.80215183e+03  .17885033e+04  .25706181e-02", 
" -.21894753e+02  .16026130e+03 -.77661562e+03  .15873953e+04  .25817371e-02", 
" -.21756646e+02  .15809076e+03 -.74761855e+03  .13172624e+04  .25719346e-02", 
" -.21618767e+02  .15624584e+03 -.73711062e+03  .13395467e+04  .25629171e-02", 
" -.21480330e+02  .15432605e+03 -.72167347e+03  .12840106e+04  .26016456e-02", 
" -.21340362e+02  .15229301e+03 -.69888859e+03  .10883472e+04  .25743104e-02", 
" -.21199795e+02  .15036790e+03 -.68544580e+03  .10771968e+04  .25997399e-02", 
" -.21057977e+02  .14842836e+03 -.67094572e+03  .10351361e+04  .25700319e-02", 
" -.20915043e+02  .14641940e+03 -.65186631e+03  .92058420e+03  .26002968e-02", 
" -.20771241e+02  .14453071e+03 -.64054688e+03  .91568587e+03  .26089126e-02", 
" -.20625316e+02  .14240760e+03 -.61747747e+03  .75415520e+03  .25984140e-02", 
" -.20479159e+02  .14056985e+03 -.61075625e+03  .83292212e+03  .26251114e-02", 
" -.20331880e+02  .13857577e+03 -.59243782e+03  .71284134e+03  .26454860e-02", 
" -.20182034e+02  .13649578e+03 -.57384633e+03  .61506181e+03  .26580170e-02", 
" -.20030903e+02  .13443890e+03 -.55505740e+03  .50000333e+03  .26411448e-02", 
" -.19877421e+02  .13240565e+03 -.54140247e+03  .47538470e+03  .26048236e-02", 
" -.19722411e+02  .13029896e+03 -.52293823e+03  .38077842e+03  .25966783e-02", 
" -.19566222e+02  .12816982e+03 -.50267184e+03  .25750839e+03  .26117767e-02", 
" -.19408208e+02  .12618364e+03 -.49290952e+03  .29883607e+03  .25905893e-02", 
" -.19248222e+02  .12402696e+03 -.47257527e+03  .16968617e+03  .25617101e-02", 
" -.19085147e+02  .12162757e+03 -.43898540e+03 -.16022716e+03  .25517129e-02", 
" -.18921144e+02  .11966136e+03 -.43372429e+03 -.30180172e+02  .25886082e-02", 
" -.18753860e+02  .11754251e+03 -.42252348e+03  .18658069e+02  .25657137e-02", 
" -.18584159e+02  .11549036e+03 -.41446612e+03  .96946178e+02  .25990258e-02", 
" -.18410174e+02  .11304910e+03 -.38647129e+03 -.12317585e+03  .25517482e-02", 
" -.18234026e+02  .11079783e+03 -.37410652e+03 -.58922082e+02  .25413193e-02", 
" -.18054360e+02  .10818719e+03 -.33589911e+03 -.45714549e+03  .26031125e-02", 
" -.17872043e+02  .10577076e+03 -.31043204e+03 -.65583560e+03  .26016429e-02", 
" -.17685369e+02  .10314881e+03 -.27533018e+03 -.98815949e+03  .26555832e-02", 
" -.17495374e+02  .10066936e+03 -.25178892e+03 -.11295599e+04  .26255780e-02", 
" -.17301275e+02  .98257672e+02 -.23257869e+03 -.12105221e+04  .26213710e-02", 
" -.17102446e+02  .95754433e+02 -.21179935e+03 -.13037429e+04  .26326960e-02", 
" -.16899257e+02  .93370538e+02 -.19765979e+03 -.13121510e+04  .26595248e-02", 
" -.16689377e+02  .90523180e+02 -.16106104e+03 -.16396334e+04  .26992708e-02", 
" -.16474341e+02  .88112762e+02 -.15275145e+03 -.15494758e+04  .27073313e-02", 
" -.16251983e+02  .85284206e+02 -.12026228e+03 -.18455676e+04  .27354865e-02", 
" -.16021407e+02  .82354205e+02 -.92833165e+02 -.19975725e+04  .28007699e-02", 
" -.15783384e+02  .79669724e+02 -.82599366e+02 -.18887070e+04  .28023223e-02", 
" -.15536461e+02  .76713043e+02 -.60578909e+02 -.19449814e+04  .28893870e-02", 
" -.15279723e+02  .73595845e+02 -.30015754e+02 -.21648882e+04  .29121054e-02", 
" -.15011760e+02  .70657450e+02 -.13682511e+02 -.21875447e+04  .30032989e-02", 
" -.14730507e+02  .67642634e+02 -.29156158e+01 -.20683131e+04  .29605105e-02", 
" -.14432951e+02  .64125554e+02  .31711430e+02 -.23212276e+04  .29150147e-02", 
" -.14117366e+02  .60476607e+02  .64032066e+02 -.25419030e+04  .29590193e-02", 
" -.13778790e+02  .56540485e+02  .98399123e+02 -.27289349e+04  .29756264e-02", 
" -.13414374e+02  .52405685e+02  .13372944e+03 -.29128910e+04  .30548168e-02", 
" -.13019535e+02  .47937527e+02  .16886068e+03 -.30204383e+04  .31131828e-02", 
" -.12582874e+02  .43395107e+02  .19045974e+03 -.29605953e+04  .32029709e-02", 
" -.12090855e+02  .37643293e+02  .25734856e+03 -.35355122e+04  .34041985e-02", 
" -.11519853e+02  .30888128e+02  .32978637e+03 -.40002211e+04  .35543792e-02", 
" -.10835752e+02  .23831669e+02  .36332033e+03 -.38403867e+04  .40045779e-02", 
" -.99507751e+01  .14929591e+02  .40416623e+03 -.36549647e+04  .46365798e-02", 
" -.97353051e+01  .12842872e+02  .41318766e+03 -.36572007e+04  .48454486e-02", 
" -.95000050e+01  .10220310e+02  .44118942e+03 -.38669286e+04  .49506264e-02", 
" -.92383063e+01  .71602448e+01  .48323661e+03 -.43113614e+04  .51318625e-02", 
" -.89434559e+01  .34480454e+01  .54238592e+03 -.49468643e+04  .54156041e-02", 
" -.86129647e+01  .60387373e+00  .53552559e+03 -.45903621e+04  .58161606e-02", 
" -.82202828e+01 -.39123420e+01  .58521397e+03 -.49282359e+04  .60267440e-02", 
" -.77337350e+01 -.97273184e+01  .65113273e+03 -.54030893e+04  .65720182e-02", 
" -.70909523e+01 -.17401828e+02  .74184847e+03 -.61004968e+04  .73447728e-02", 
" -.60653225e+01 -.32100019e+02  .10060728e+04 -.90245527e+04  .10117052e-01", 
" -.51303849e+01 -.44328603e+02  .10625373e+04 -.80383620e+04  .12294544e-01", 
" -.40075166e+01 -.59880924e+02  .12128752e+04 -.86560550e+04  .17480463e-01", 
" -.32255511e+01 -.68060887e+02  .10675064e+04 -.46505144e+04  .21476226e-01"
)


.urc6 = 
c("Copyright (C) James G. MacKinnon, 1995", "conc      5  6  2   30", 
" -.60960970e+01 -.45690516e+02 -.23000025e+03  .49156427e-02", 
" -.59475584e+01 -.42204070e+02 -.17446401e+03  .34942102e-02", 
" -.57405615e+01 -.36798018e+02 -.14656396e+03  .22918024e-02", 
" -.55748089e+01 -.33086246e+02 -.12147336e+03  .16560339e-02", 
" -.54004531e+01 -.29635341e+02 -.90745517e+02  .11475520e-02", 
" -.52933278e+01 -.27608569e+02 -.76629465e+02  .98101607e-03", 
" -.52147258e+01 -.26181509e+02 -.66177266e+02  .87177941e-03", 
" -.51530273e+01 -.24909693e+02 -.65017272e+02  .82294500e-03", 
" -.51009039e+01 -.24031030e+02 -.59271654e+02  .76598447e-03", 
" -.50566167e+01 -.23233864e+02 -.55862518e+02  .69991445e-03", 
" -.50174802e+01 -.22581200e+02 -.51860623e+02  .65075855e-03", 
" -.49823098e+01 -.22006638e+02 -.48667023e+02  .61116101e-03", 
" -.49504893e+01 -.21502971e+02 -.45385061e+02  .60373131e-03", 
" -.48243182e+01 -.19582164e+02 -.34528012e+02  .52071168e-03", 
" -.47312915e+01 -.18175296e+02 -.28116815e+02  .47328040e-03", 
" -.46568339e+01 -.17023128e+02 -.25680483e+02  .41753092e-03", 
" -.45943312e+01 -.16103217e+02 -.22997256e+02  .39705085e-03", 
" -.45397428e+01 -.15363320e+02 -.20368544e+02  .38357038e-03", 
" -.44913585e+01 -.14743598e+02 -.17488012e+02  .35773350e-03", 
" -.44476682e+01 -.14173615e+02 -.16229902e+02  .35063546e-03", 
" -.44078933e+01 -.13666664e+02 -.14785421e+02  .33719629e-03", 
" -.43713235e+01 -.13174694e+02 -.14624279e+02  .31950173e-03", 
" -.43372641e+01 -.12767589e+02 -.13257648e+02  .31027925e-03", 
" -.43054369e+01 -.12387678e+02 -.12403981e+02  .31413460e-03", 
" -.42753865e+01 -.12038907e+02 -.11688957e+02  .29802347e-03", 
" -.42470632e+01 -.11716537e+02 -.10811868e+02  .29041802e-03", 
" -.42200957e+01 -.11438292e+02 -.91273918e+01  .28637268e-03", 
" -.41945116e+01 -.11137674e+02 -.89060670e+01  .28477976e-03", 
" -.41699079e+01 -.10893409e+02 -.73965989e+01  .27783052e-03", 
" -.41464459e+01 -.10648219e+02 -.63174215e+01  .27297669e-03", 
" -.41239262e+01 -.10396791e+02 -.61129870e+01  .26556354e-03", 
" -.41021843e+01 -.10176584e+02 -.53810014e+01  .26060451e-03", 
" -.40812584e+01 -.99534309e+01 -.51096223e+01  .26127055e-03", 
" -.40609054e+01 -.97676317e+01 -.39591751e+01  .25826687e-03", 
" -.40412124e+01 -.95804881e+01 -.33631619e+01  .25722895e-03", 
" -.40221684e+01 -.94012974e+01 -.26317179e+01  .25955225e-03", 
" -.40036888e+01 -.92255818e+01 -.20099124e+01  .25128811e-03", 
" -.39857374e+01 -.90488705e+01 -.17398907e+01  .24730834e-03", 
" -.39682065e+01 -.88828627e+01 -.13266788e+01  .23948713e-03", 
" -.39511466e+01 -.87268357e+01 -.77319365e+00  .24156061e-03", 
" -.39344949e+01 -.85660601e+01 -.56269924e+00  .24303243e-03", 
" -.39182100e+01 -.84105586e+01 -.29418149e+00  .24065948e-03", 
" -.39023094e+01 -.82599216e+01 -.92939462e-01  .23465160e-03", 
" -.38867168e+01 -.81170690e+01  .22143419e+00  .22818541e-03", 
" -.38714973e+01 -.79695679e+01  .20522039e+00  .22484991e-03", 
" -.38565563e+01 -.78300327e+01  .30794816e+00  .22135807e-03", 
" -.38418802e+01 -.76953218e+01  .39453821e+00  .22218233e-03", 
" -.38274216e+01 -.75627436e+01  .37876007e+00  .22150906e-03", 
" -.38132123e+01 -.74483796e+01  .91623793e+00  .21983789e-03", 
" -.37992662e+01 -.73349062e+01  .13940455e+01  .21682016e-03", 
" -.37855144e+01 -.72200821e+01  .16235956e+01  .21652513e-03", 
" -.37719307e+01 -.71148871e+01  .19978773e+01  .21453566e-03", 
" -.37586908e+01 -.69915925e+01  .18043740e+01  .21388835e-03", 
" -.37455133e+01 -.68937132e+01  .22201900e+01  .21426540e-03", 
" -.37325907e+01 -.67846058e+01  .23311244e+01  .21359198e-03", 
" -.37198886e+01 -.66736171e+01  .22742262e+01  .21260973e-03", 
" -.37073127e+01 -.65700857e+01  .23297969e+01  .20880524e-03", 
" -.36948914e+01 -.64733482e+01  .25480341e+01  .20901935e-03", 
" -.36825811e+01 -.63816725e+01  .28016564e+01  .20773350e-03", 
" -.36704597e+01 -.62849273e+01  .29042815e+01  .20508700e-03", 
" -.36585405e+01 -.61876101e+01  .29404590e+01  .19984555e-03", 
" -.36467185e+01 -.60842482e+01  .26344579e+01  .19576272e-03", 
" -.36350008e+01 -.59905355e+01  .25911704e+01  .19393615e-03", 
" -.36234498e+01 -.58970088e+01  .25556081e+01  .19331272e-03", 
" -.36120414e+01 -.58005598e+01  .23417086e+01  .19442629e-03", 
" -.36006342e+01 -.57302187e+01  .28870578e+01  .18968300e-03", 
" -.35893631e+01 -.56559413e+01  .32892634e+01  .19151016e-03", 
" -.35781867e+01 -.55790442e+01  .35378991e+01  .19031459e-03", 
" -.35671372e+01 -.55023361e+01  .38545018e+01  .18870709e-03", 
" -.35561832e+01 -.54301881e+01  .42341131e+01  .18394835e-03", 
" -.35453386e+01 -.53497161e+01  .42640083e+01  .18469708e-03", 
" -.35345755e+01 -.52645065e+01  .40599413e+01  .18107031e-03", 
" -.35239043e+01 -.51830423e+01  .38868991e+01  .18108732e-03", 
" -.35132817e+01 -.51115746e+01  .41118655e+01  .18071891e-03", 
" -.35028046e+01 -.50289691e+01  .39323452e+01  .18029210e-03", 
" -.34923155e+01 -.49612089e+01  .41389602e+01  .17862107e-03", 
" -.34819073e+01 -.48967129e+01  .44439117e+01  .17883981e-03", 
" -.34716082e+01 -.48251370e+01  .44950831e+01  .18401606e-03", 
" -.34614228e+01 -.47489034e+01  .43661113e+01  .18317051e-03", 
" -.34512150e+01 -.46849307e+01  .44790629e+01  .18161707e-03", 
" -.34411046e+01 -.46178546e+01  .45282402e+01  .18180149e-03", 
" -.34310714e+01 -.45451071e+01  .43556479e+01  .18149700e-03", 
" -.34210655e+01 -.44835926e+01  .45389659e+01  .17978831e-03", 
" -.34111250e+01 -.44190797e+01  .45923391e+01  .18171828e-03", 
" -.34012744e+01 -.43486006e+01  .44595249e+01  .18270904e-03", 
" -.33914301e+01 -.42799780e+01  .43724742e+01  .18132451e-03", 
" -.33816591e+01 -.42140161e+01  .43404203e+01  .18173845e-03", 
" -.33718682e+01 -.41583819e+01  .45545234e+01  .18227363e-03", 
" -.33621529e+01 -.40971335e+01  .45720328e+01  .18200244e-03", 
" -.33525011e+01 -.40358365e+01  .45392418e+01  .18157863e-03", 
" -.33428766e+01 -.39759818e+01  .45346648e+01  .18416548e-03", 
" -.33332551e+01 -.39214890e+01  .47041890e+01  .18405413e-03", 
" -.33236422e+01 -.38723216e+01  .49801494e+01  .18443742e-03", 
" -.33140987e+01 -.38164697e+01  .50614915e+01  .18637343e-03", 
" -.33045842e+01 -.37633586e+01  .51903944e+01  .18434444e-03", 
" -.32950984e+01 -.37046654e+01  .50901398e+01  .18300337e-03", 
" -.32856736e+01 -.36506262e+01  .51515649e+01  .18253489e-03", 
" -.32763044e+01 -.35866782e+01  .49400219e+01  .18116360e-03", 
" -.32669094e+01 -.35319187e+01  .49145690e+01  .18019660e-03", 
" -.32575333e+01 -.34762707e+01  .48327848e+01  .17951410e-03", 
" -.32480870e+01 -.34343021e+01  .51064891e+01  .17669507e-03", 
" -.32387127e+01 -.33876074e+01  .52744609e+01  .17585781e-03", 
" -.32293710e+01 -.33407384e+01  .54831440e+01  .17701676e-03", 
" -.32200852e+01 -.32877077e+01  .54215281e+01  .18015816e-03", 
" -.32108047e+01 -.32368956e+01  .54472497e+01  .17815303e-03", 
" -.32015184e+01 -.31878976e+01  .54671887e+01  .17755866e-03", 
" -.31922807e+01 -.31321621e+01  .52630208e+01  .17623119e-03", 
" -.31830437e+01 -.30824110e+01  .52718148e+01  .17666590e-03", 
" -.31737701e+01 -.30340273e+01  .52919268e+01  .17947899e-03", 
" -.31645162e+01 -.29853775e+01  .52855387e+01  .17834285e-03", 
" -.31552718e+01 -.29369905e+01  .52632768e+01  .17866761e-03", 
" -.31460898e+01 -.28871041e+01  .52373517e+01  .17969321e-03", 
" -.31368710e+01 -.28414731e+01  .52684587e+01  .17895976e-03", 
" -.31276326e+01 -.27935752e+01  .51836727e+01  .17931475e-03", 
" -.31184115e+01 -.27492402e+01  .52693751e+01  .17657773e-03", 
" -.31091706e+01 -.27020941e+01  .52160892e+01  .17696231e-03", 
" -.30999225e+01 -.26560742e+01  .51521481e+01  .17517914e-03", 
" -.30906590e+01 -.26107985e+01  .50840356e+01  .17792286e-03", 
" -.30814469e+01 -.25591403e+01  .48469000e+01  .17810038e-03", 
" -.30721814e+01 -.25194168e+01  .49680637e+01  .17876652e-03", 
" -.30629146e+01 -.24766091e+01  .50143545e+01  .17954996e-03", 
" -.30536435e+01 -.24277728e+01  .47680544e+01  .18060324e-03", 
" -.30443422e+01 -.23870737e+01  .47861957e+01  .18048919e-03", 
" -.30350094e+01 -.23522770e+01  .50130774e+01  .18031166e-03", 
" -.30257287e+01 -.23044548e+01  .48384731e+01  .18043216e-03", 
" -.30163590e+01 -.22635836e+01  .47587737e+01  .17932190e-03", 
" -.30070139e+01 -.22236299e+01  .48168882e+01  .17953831e-03", 
" -.29976156e+01 -.21840461e+01  .48756998e+01  .18066327e-03", 
" -.29881891e+01 -.21511156e+01  .50799682e+01  .17967036e-03", 
" -.29787424e+01 -.21143978e+01  .52006869e+01  .17907204e-03", 
" -.29692873e+01 -.20734482e+01  .51627861e+01  .17670064e-03", 
" -.29598027e+01 -.20356505e+01  .51982809e+01  .17683685e-03", 
" -.29503110e+01 -.19917522e+01  .49915464e+01  .17521518e-03", 
" -.29408251e+01 -.19380519e+01  .45037940e+01  .17723275e-03", 
" -.29312014e+01 -.19043486e+01  .46179470e+01  .17854239e-03", 
" -.29215794e+01 -.18724409e+01  .48140137e+01  .18126926e-03", 
" -.29119495e+01 -.18322630e+01  .46672308e+01  .18098473e-03", 
" -.29022873e+01 -.17974269e+01  .47945517e+01  .18247518e-03", 
" -.28925619e+01 -.17577989e+01  .46803086e+01  .18002526e-03", 
" -.28827996e+01 -.17179591e+01  .46210608e+01  .17806766e-03", 
" -.28730211e+01 -.16763346e+01  .44401506e+01  .17949583e-03", 
" -.28631701e+01 -.16385641e+01  .43887446e+01  .18082730e-03", 
" -.28532238e+01 -.16096420e+01  .46482895e+01  .18028871e-03", 
" -.28432624e+01 -.15734584e+01  .46158288e+01  .17744636e-03", 
" -.28332204e+01 -.15421598e+01  .47028276e+01  .17746171e-03", 
" -.28231392e+01 -.15072296e+01  .47137401e+01  .17789433e-03", 
" -.28130107e+01 -.14701315e+01  .46578771e+01  .17623100e-03", 
" -.28027813e+01 -.14378269e+01  .46476970e+01  .17615753e-03", 
" -.27925081e+01 -.14026687e+01  .46082549e+01  .17506440e-03", 
" -.27821783e+01 -.13678918e+01  .45820236e+01  .17593880e-03", 
" -.27717807e+01 -.13338399e+01  .45798007e+01  .17672307e-03", 
" -.27612883e+01 -.12999245e+01  .45837303e+01  .17707275e-03", 
" -.27506728e+01 -.12715878e+01  .46846208e+01  .17578984e-03", 
" -.27400380e+01 -.12379593e+01  .47017064e+01  .17436653e-03", 
" -.27292723e+01 -.12124509e+01  .48719954e+01  .17165607e-03", 
" -.27184328e+01 -.11765596e+01  .47135330e+01  .17245844e-03", 
" -.27075114e+01 -.11388650e+01  .44664040e+01  .17188301e-03", 
" -.26964957e+01 -.11062143e+01  .44070503e+01  .17375845e-03", 
" -.26853493e+01 -.10742785e+01  .43783284e+01  .17422650e-03", 
" -.26741232e+01 -.10413957e+01  .43252653e+01  .17882072e-03", 
" -.26628279e+01 -.10056774e+01  .41426959e+01  .18066856e-03", 
" -.26513652e+01 -.97585519e+00  .42237640e+01  .17933664e-03", 
" -.26397675e+01 -.94253956e+00  .41016433e+01  .18505439e-03", 
" -.26280439e+01 -.90949686e+00  .39727478e+01  .18613875e-03", 
" -.26161133e+01 -.88373059e+00  .40633224e+01  .18512808e-03", 
" -.26040477e+01 -.85581551e+00  .40343860e+01  .18627457e-03", 
" -.25918583e+01 -.83165039e+00  .42224623e+01  .18681462e-03", 
" -.25795009e+01 -.81760389e+00  .47745912e+01  .18886168e-03", 
" -.25669970e+01 -.78740997e+00  .46668297e+01  .19269358e-03", 
" -.25543300e+01 -.75493718e+00  .44517084e+01  .19398059e-03", 
" -.25414683e+01 -.72920401e+00  .45497989e+01  .19434810e-03", 
" -.25284112e+01 -.70574951e+00  .47080508e+01  .19237231e-03", 
" -.25151264e+01 -.68491926e+00  .49445218e+01  .19161593e-03", 
" -.25016099e+01 -.66095996e+00  .49941669e+01  .19281565e-03", 
" -.24879633e+01 -.63112827e+00  .50060329e+01  .19681863e-03", 
" -.24740130e+01 -.60374643e+00  .49749673e+01  .19877835e-03", 
" -.24598522e+01 -.57426456e+00  .48802873e+01  .20207783e-03", 
" -.24453600e+01 -.54958896e+00  .48394106e+01  .20574277e-03", 
" -.24305919e+01 -.52677949e+00  .49393408e+01  .20493542e-03", 
" -.24155453e+01 -.49136322e+00  .45657123e+01  .20680135e-03", 
" -.24001157e+01 -.46050395e+00  .42788324e+01  .20632973e-03", 
" -.23842935e+01 -.44660871e+00  .45471409e+01  .21035516e-03", 
" -.23681493e+01 -.42156956e+00  .44269643e+01  .21334277e-03", 
" -.23516010e+01 -.39229552e+00  .41468058e+01  .21619458e-03", 
" -.23346641e+01 -.36101412e+00  .38411688e+01  .21874791e-03", 
" -.23171621e+01 -.34136932e+00  .38252108e+01  .21965153e-03", 
" -.22990985e+01 -.31817172e+00  .35634322e+01  .22052584e-03", 
" -.22806008e+01 -.28916029e+00  .33305067e+01  .23025675e-03", 
" -.22614371e+01 -.26908406e+00  .32284178e+01  .23472785e-03", 
" -.22415944e+01 -.25883015e+00  .34816880e+01  .24502391e-03", 
" -.22210224e+01 -.25060674e+00  .36831785e+01  .25210107e-03", 
" -.21996807e+01 -.23235081e+00  .35813990e+01  .25387574e-03", 
" -.21774844e+01 -.20450451e+00  .31299517e+01  .25606095e-03", 
" -.21542734e+01 -.19986040e+00  .34408489e+01  .25858991e-03", 
" -.21298915e+01 -.20073775e+00  .37748093e+01  .26643722e-03", 
" -.21042247e+01 -.20025099e+00  .38056837e+01  .26844438e-03", 
" -.20772423e+01 -.17786280e+00  .33180665e+01  .26886299e-03", 
" -.20485268e+01 -.17910944e+00  .33810710e+01  .27475451e-03", 
" -.20180155e+01 -.16826211e+00  .30217150e+01  .27991036e-03", 
" -.19851294e+01 -.19937692e+00  .36107588e+01  .29672186e-03", 
" -.19496862e+01 -.20822346e+00  .35644424e+01  .30402384e-03", 
" -.19110267e+01 -.22974126e+00  .35618498e+01  .31992225e-03", 
" -.18681825e+01 -.29072897e+00  .44912422e+01  .32366829e-03", 
" -.18204844e+01 -.32890952e+00  .45311207e+01  .33303967e-03", 
" -.17661897e+01 -.38180195e+00  .45102099e+01  .35444630e-03", 
" -.17026519e+01 -.50088172e+00  .60645433e+01  .40394153e-03", 
" -.16260281e+01 -.60858333e+00  .59279458e+01  .42036315e-03", 
" -.15284585e+01 -.74526403e+00  .51066412e+01  .52145715e-03", 
" -.13919286e+01 -.10523226e+01  .63470285e+01  .59688852e-03", 
" -.13567297e+01 -.10998860e+01  .56056794e+01  .62648384e-03", 
" -.13171458e+01 -.12474753e+01  .75453237e+01  .67605960e-03", 
" -.12724007e+01 -.13915550e+01  .90452350e+01  .71598813e-03", 
" -.12206005e+01 -.15319394e+01  .96330315e+01  .78283273e-03", 
" -.11591956e+01 -.17058442e+01  .10552787e+02  .84764173e-03", 
" -.10849383e+01 -.18826295e+01  .11170506e+02  .97229768e-03", 
" -.98872867e+00 -.21845664e+01  .14685253e+02  .11382333e-02", 
" -.85390941e+00 -.24554854e+01  .13008361e+02  .13649260e-02", 
" -.62783915e+00 -.28975095e+01  .15012066e+02  .19373637e-02", 
" -.40532080e+00 -.36662452e+01  .29014242e+02  .26969779e-02", 
" -.11609230e+00 -.42685999e+01  .26005355e+02  .39463612e-02", 
"  .89789644e-01 -.46725659e+01  .42092106e+02  .53826876e-02", 
"coc       5  7  2   30", " -.63792838e+01 -.53035830e+02 -.26012387e+03  .44666659e-02", 
" -.62305160e+01 -.48841093e+02 -.20706028e+03  .35222899e-02", 
" -.60279808e+01 -.43027315e+02 -.16509447e+03  .22650665e-02", 
" -.58638734e+01 -.38940973e+02 -.13188090e+03  .17271550e-02", 
" -.56911913e+01 -.34923826e+02 -.10313214e+03  .12084070e-02", 
" -.55839054e+01 -.32786209e+02 -.83165734e+02  .10115890e-02", 
" -.55068073e+01 -.31113180e+02 -.73185706e+02  .90083665e-03", 
" -.54456240e+01 -.29766403e+02 -.68365986e+02  .82821681e-03", 
" -.53941092e+01 -.28803888e+02 -.61460332e+02  .76474035e-03", 
" -.53500558e+01 -.27929563e+02 -.57163883e+02  .71489132e-03", 
" -.53109272e+01 -.27247552e+02 -.51381374e+02  .66283837e-03", 
" -.52764614e+01 -.26546458e+02 -.49121708e+02  .63676086e-03", 
" -.52449838e+01 -.25939650e+02 -.47126358e+02  .61094086e-03", 
" -.51201365e+01 -.23616444e+02 -.39388152e+02  .52078579e-03", 
" -.50279407e+01 -.21998689e+02 -.33853794e+02  .46534956e-03", 
" -.49534994e+01 -.20875863e+02 -.25147700e+02  .43839371e-03", 
" -.48912567e+01 -.19859466e+02 -.21559393e+02  .41757576e-03", 
" -.48373252e+01 -.18966034e+02 -.20005411e+02  .38589831e-03", 
" -.47893900e+01 -.18220934e+02 -.17879671e+02  .36946522e-03", 
" -.47462171e+01 -.17525152e+02 -.17680471e+02  .34461816e-03", 
" -.47069020e+01 -.16945782e+02 -.15585643e+02  .33278802e-03", 
" -.46705992e+01 -.16406468e+02 -.14395795e+02  .31768985e-03", 
" -.46367308e+01 -.15929865e+02 -.13217621e+02  .31308435e-03", 
" -.46051082e+01 -.15503169e+02 -.11463173e+02  .30238664e-03", 
" -.45752284e+01 -.15104919e+02 -.10147112e+02  .29251995e-03", 
" -.45470663e+01 -.14739328e+02 -.90431606e+01  .28513955e-03", 
" -.45203193e+01 -.14404199e+02 -.75434713e+01  .27635360e-03", 
" -.44949840e+01 -.14063337e+02 -.69008813e+01  .27216245e-03", 
" -.44707134e+01 -.13750993e+02 -.60176303e+01  .26942002e-03", 
" -.44473866e+01 -.13470506e+02 -.48189880e+01  .27584127e-03", 
" -.44249544e+01 -.13200205e+02 -.40676196e+01  .27223885e-03", 
" -.44035107e+01 -.12914550e+02 -.40692383e+01  .27118833e-03", 
" -.43826039e+01 -.12675939e+02 -.29995789e+01  .26990553e-03", 
" -.43624146e+01 -.12442803e+02 -.22390699e+01  .26602200e-03", 
" -.43429818e+01 -.12194955e+02 -.25166453e+01  .26294791e-03", 
" -.43240352e+01 -.11982085e+02 -.18522598e+01  .25999333e-03", 
" -.43056714e+01 -.11771495e+02 -.12939589e+01  .25540567e-03", 
" -.42877829e+01 -.11577710e+02 -.51707650e+00  .25111598e-03", 
" -.42704073e+01 -.11381950e+02 -.22347597e-01  .25194094e-03", 
" -.42534390e+01 -.11184301e+02  .18041033e+00  .24743398e-03", 
" -.42368467e+01 -.11000353e+02  .55854572e+00  .24652583e-03", 
" -.42206583e+01 -.10816682e+02  .87172834e+00  .24440159e-03", 
" -.42048531e+01 -.10644374e+02  .12314249e+01  .24432478e-03", 
" -.41893975e+01 -.10462897e+02  .10463169e+01  .24348312e-03", 
" -.41742004e+01 -.10303358e+02  .14242211e+01  .24095126e-03", 
" -.41592997e+01 -.10151711e+02  .18695692e+01  .23723095e-03", 
" -.41446681e+01 -.10000741e+02  .22045263e+01  .23311165e-03", 
" -.41303017e+01 -.98556412e+01  .25898028e+01  .23097563e-03", 
" -.41161860e+01 -.97142556e+01  .29372527e+01  .22518469e-03", 
" -.41022964e+01 -.95782869e+01  .32710974e+01  .22622050e-03", 
" -.40886934e+01 -.94390368e+01  .35142650e+01  .22718026e-03", 
" -.40752418e+01 -.93043819e+01  .37219683e+01  .22822762e-03", 
" -.40620536e+01 -.91685095e+01  .39126895e+01  .22359679e-03", 
" -.40490139e+01 -.90380660e+01  .39848553e+01  .21919506e-03", 
" -.40362064e+01 -.89129321e+01  .41799085e+01  .21480931e-03", 
" -.40235671e+01 -.87807975e+01  .40850936e+01  .21119169e-03", 
" -.40110096e+01 -.86650857e+01  .43117355e+01  .21048075e-03", 
" -.39986089e+01 -.85514701e+01  .45319442e+01  .20834086e-03", 
" -.39863904e+01 -.84403708e+01  .48335776e+01  .20475704e-03", 
" -.39743264e+01 -.83313856e+01  .50809052e+01  .20524468e-03", 
" -.39624090e+01 -.82219969e+01  .52107764e+01  .20724339e-03", 
" -.39506138e+01 -.81108679e+01  .51730358e+01  .20881456e-03", 
" -.39389279e+01 -.80043398e+01  .52063058e+01  .20427019e-03", 
" -.39274191e+01 -.78922080e+01  .51447240e+01  .20142398e-03", 
" -.39159778e+01 -.77949719e+01  .54911846e+01  .20113177e-03", 
" -.39046362e+01 -.77008421e+01  .57960704e+01  .20313512e-03", 
" -.38934338e+01 -.76015264e+01  .59241679e+01  .20242568e-03", 
" -.38824013e+01 -.74922041e+01  .57673298e+01  .20026339e-03", 
" -.38713780e+01 -.73982060e+01  .59089430e+01  .20164747e-03", 
" -.38604720e+01 -.72996432e+01  .58697451e+01  .20013631e-03", 
" -.38496585e+01 -.72079508e+01  .60636581e+01  .19599270e-03", 
" -.38389628e+01 -.71118959e+01  .60569742e+01  .19338073e-03", 
" -.38282852e+01 -.70201293e+01  .60288271e+01  .19451518e-03", 
" -.38177228e+01 -.69314521e+01  .62132484e+01  .19522338e-03", 
" -.38072483e+01 -.68463933e+01  .63967951e+01  .19389976e-03", 
" -.37967964e+01 -.67656796e+01  .66350003e+01  .19340322e-03", 
" -.37864460e+01 -.66847567e+01  .68693515e+01  .19508338e-03", 
" -.37761645e+01 -.66013715e+01  .69997419e+01  .19584881e-03", 
" -.37659619e+01 -.65162832e+01  .70306230e+01  .19304423e-03", 
" -.37558478e+01 -.64319585e+01  .70410990e+01  .19274352e-03", 
" -.37457699e+01 -.63483568e+01  .69864201e+01  .19288278e-03", 
" -.37357307e+01 -.62682732e+01  .70339488e+01  .19259289e-03", 
" -.37257723e+01 -.61867803e+01  .69998012e+01  .19262898e-03", 
" -.37158603e+01 -.61130543e+01  .71924244e+01  .18902792e-03", 
" -.37059972e+01 -.60387416e+01  .73887447e+01  .18758593e-03", 
" -.36961698e+01 -.59584762e+01  .72387695e+01  .18684125e-03", 
" -.36864139e+01 -.58771967e+01  .71230914e+01  .18539810e-03", 
" -.36767188e+01 -.57905542e+01  .67718559e+01  .18465123e-03", 
" -.36670114e+01 -.57222801e+01  .69874547e+01  .18675565e-03", 
" -.36573930e+01 -.56439173e+01  .68516590e+01  .18545493e-03", 
" -.36478100e+01 -.55697609e+01  .68180517e+01  .18516459e-03", 
" -.36382108e+01 -.55027916e+01  .69895324e+01  .18377300e-03", 
" -.36286228e+01 -.54414964e+01  .72935845e+01  .18236409e-03", 
" -.36190853e+01 -.53725162e+01  .72330319e+01  .18453306e-03", 
" -.36095954e+01 -.53043001e+01  .72853942e+01  .18444637e-03", 
" -.36001145e+01 -.52398887e+01  .73502515e+01  .18467193e-03", 
" -.35906977e+01 -.51794198e+01  .76292037e+01  .18549662e-03", 
" -.35813460e+01 -.51127389e+01  .77477396e+01  .18624783e-03", 
" -.35719848e+01 -.50488531e+01  .79142256e+01  .18846666e-03", 
" -.35626236e+01 -.49833112e+01  .79284096e+01  .19097592e-03", 
" -.35533087e+01 -.49131808e+01  .77792188e+01  .18957426e-03", 
" -.35440142e+01 -.48472712e+01  .77455839e+01  .18980382e-03", 
" -.35347735e+01 -.47751094e+01  .75108301e+01  .18854756e-03", 
" -.35254908e+01 -.47130530e+01  .75422019e+01  .19019454e-03", 
" -.35162637e+01 -.46480913e+01  .74861419e+01  .19037100e-03", 
" -.35069872e+01 -.45830172e+01  .73248260e+01  .18909483e-03", 
" -.34977407e+01 -.45220307e+01  .73659973e+01  .18867631e-03", 
" -.34885310e+01 -.44614749e+01  .74457253e+01  .19047302e-03", 
" -.34793085e+01 -.44045968e+01  .75758193e+01  .18991309e-03", 
" -.34700598e+01 -.43494762e+01  .77258017e+01  .18923865e-03", 
" -.34608725e+01 -.42876546e+01  .76666272e+01  .19040870e-03", 
" -.34517060e+01 -.42224524e+01  .75419987e+01  .19162452e-03", 
" -.34424921e+01 -.41639697e+01  .75258387e+01  .19024736e-03", 
" -.34332910e+01 -.41069526e+01  .75447552e+01  .18997497e-03", 
" -.34240414e+01 -.40506880e+01  .75086804e+01  .19145892e-03", 
" -.34147926e+01 -.39977825e+01  .76356303e+01  .19233135e-03", 
" -.34055759e+01 -.39420291e+01  .76515883e+01  .19356192e-03", 
" -.33963324e+01 -.38857100e+01  .75680953e+01  .19359105e-03", 
" -.33870663e+01 -.38403498e+01  .79140367e+01  .19305622e-03", 
" -.33778107e+01 -.37857906e+01  .78722299e+01  .19431081e-03", 
" -.33685378e+01 -.37342395e+01  .79413346e+01  .19326642e-03", 
" -.33593054e+01 -.36758291e+01  .78272855e+01  .19279788e-03", 
" -.33499716e+01 -.36294935e+01  .79857133e+01  .19344759e-03", 
" -.33406682e+01 -.35768429e+01  .80213080e+01  .19333365e-03", 
" -.33313844e+01 -.35238210e+01  .80319476e+01  .19333829e-03", 
" -.33220525e+01 -.34713703e+01  .79973968e+01  .19233801e-03", 
" -.33126922e+01 -.34261903e+01  .82304910e+01  .19113796e-03", 
" -.33032842e+01 -.33783646e+01  .83435276e+01  .19132232e-03", 
" -.32938965e+01 -.33272273e+01  .83529994e+01  .18932024e-03", 
" -.32844706e+01 -.32835995e+01  .85790874e+01  .19023515e-03", 
" -.32750482e+01 -.32312510e+01  .85246503e+01  .19110550e-03", 
" -.32655953e+01 -.31785658e+01  .84326509e+01  .18974402e-03", 
" -.32560896e+01 -.31385314e+01  .88093271e+01  .18921410e-03", 
" -.32465618e+01 -.30856908e+01  .86562792e+01  .18893073e-03", 
" -.32370059e+01 -.30378986e+01  .86980554e+01  .18767393e-03", 
" -.32273606e+01 -.29976584e+01  .88687750e+01  .18700535e-03", 
" -.32177555e+01 -.29449958e+01  .86809571e+01  .18903317e-03", 
" -.32080791e+01 -.28977589e+01  .87241858e+01  .19040982e-03", 
" -.31983860e+01 -.28503145e+01  .87620067e+01  .18943325e-03", 
" -.31886277e+01 -.28042170e+01  .88144598e+01  .19040238e-03", 
" -.31787959e+01 -.27616234e+01  .89377871e+01  .19130503e-03", 
" -.31689636e+01 -.27130875e+01  .88963940e+01  .19031558e-03", 
" -.31590816e+01 -.26582272e+01  .85910424e+01  .19026765e-03", 
" -.31491313e+01 -.26131404e+01  .86344346e+01  .19005038e-03", 
" -.31390901e+01 -.25697819e+01  .86803098e+01  .19000184e-03", 
" -.31290394e+01 -.25211287e+01  .85796192e+01  .19277821e-03", 
" -.31189399e+01 -.24734397e+01  .85809976e+01  .19274724e-03", 
" -.31087463e+01 -.24252823e+01  .84179060e+01  .19269637e-03", 
" -.30984993e+01 -.23806611e+01  .84187938e+01  .19494635e-03", 
" -.30881916e+01 -.23352303e+01  .83958155e+01  .19753866e-03", 
" -.30778357e+01 -.22815432e+01  .81028527e+01  .19689289e-03", 
" -.30673371e+01 -.22415887e+01  .82589552e+01  .19794131e-03", 
" -.30567955e+01 -.21959901e+01  .81651976e+01  .19617204e-03", 
" -.30461938e+01 -.21489114e+01  .80309093e+01  .19638821e-03", 
" -.30354979e+01 -.20985563e+01  .78185866e+01  .19934335e-03", 
" -.30246031e+01 -.20662241e+01  .81983835e+01  .19892414e-03", 
" -.30136797e+01 -.20155390e+01  .79306821e+01  .20086828e-03", 
" -.30026999e+01 -.19726756e+01  .79966284e+01  .19882305e-03", 
" -.29915965e+01 -.19241725e+01  .78000779e+01  .20005436e-03", 
" -.29803369e+01 -.18853245e+01  .78923601e+01  .19859005e-03", 
" -.29689393e+01 -.18502924e+01  .80740241e+01  .20100142e-03", 
" -.29574660e+01 -.18099723e+01  .81408178e+01  .20207585e-03", 
" -.29458920e+01 -.17647819e+01  .80259135e+01  .19975847e-03", 
" -.29341800e+01 -.17222045e+01  .79368276e+01  .19638094e-03", 
" -.29223110e+01 -.16798521e+01  .78228426e+01  .19643167e-03", 
" -.29102915e+01 -.16408633e+01  .78494154e+01  .19661595e-03", 
" -.28981710e+01 -.15888336e+01  .74996215e+01  .19911851e-03", 
" -.28858856e+01 -.15434563e+01  .73710681e+01  .19843304e-03", 
" -.28733577e+01 -.15047401e+01  .73592629e+01  .20144401e-03", 
" -.28606616e+01 -.14612331e+01  .71169533e+01  .20250168e-03", 
" -.28478126e+01 -.14191523e+01  .69989021e+01  .20148549e-03", 
" -.28348096e+01 -.13761862e+01  .69774781e+01  .20475362e-03", 
" -.28215384e+01 -.13424395e+01  .71707417e+01  .20682376e-03", 
" -.28081426e+01 -.12889578e+01  .67483081e+01  .20785294e-03", 
" -.27944380e+01 -.12487504e+01  .66980883e+01  .20945958e-03", 
" -.27805351e+01 -.12036097e+01  .64567049e+01  .21293514e-03", 
" -.27662964e+01 -.11737353e+01  .66927756e+01  .21373585e-03", 
" -.27518194e+01 -.11326645e+01  .66027660e+01  .21574268e-03", 
" -.27370084e+01 -.10979099e+01  .66081809e+01  .21427169e-03", 
" -.27218726e+01 -.10654645e+01  .66582140e+01  .21561971e-03", 
" -.27064171e+01 -.10289962e+01  .66020774e+01  .21398218e-03", 
" -.26907220e+01 -.98471326e+00  .63911806e+01  .21688891e-03", 
" -.26744354e+01 -.96883476e+00  .68119970e+01  .21431493e-03", 
" -.26578276e+01 -.93412487e+00  .67307704e+01  .21890333e-03", 
" -.26407463e+01 -.90471959e+00  .68332595e+01  .22030420e-03", 
" -.26233416e+01 -.85206921e+00  .62299747e+01  .21713758e-03", 
" -.26053283e+01 -.82269311e+00  .64237445e+01  .21967233e-03", 
" -.25867467e+01 -.79399184e+00  .65378487e+01  .22473568e-03", 
" -.25675511e+01 -.77090011e+00  .68587545e+01  .22800257e-03", 
" -.25477319e+01 -.73092412e+00  .64537181e+01  .22966827e-03", 
" -.25272899e+01 -.68137416e+00  .59593530e+01  .23011299e-03", 
" -.25059893e+01 -.64740926e+00  .59107668e+01  .22829634e-03", 
" -.24837402e+01 -.62072295e+00  .57217120e+01  .22960323e-03", 
" -.24605040e+01 -.59985123e+00  .57885536e+01  .23445384e-03", 
" -.24361318e+01 -.57463611e+00  .56827463e+01  .24099581e-03", 
" -.24105564e+01 -.54344495e+00  .55198068e+01  .24633267e-03", 
" -.23836061e+01 -.50489918e+00  .48831129e+01  .24849179e-03", 
" -.23550629e+01 -.47417838e+00  .45694431e+01  .25921099e-03", 
" -.23247554e+01 -.43292622e+00  .37050030e+01  .27100665e-03", 
" -.22921429e+01 -.40482005e+00  .33281553e+01  .27653767e-03", 
" -.22568510e+01 -.39702085e+00  .33737778e+01  .28446918e-03", 
" -.22181424e+01 -.44414032e+00  .49150215e+01  .30434055e-03", 
" -.21757985e+01 -.42526618e+00  .42572607e+01  .32098323e-03", 
" -.21285056e+01 -.41847891e+00  .35845428e+01  .34210902e-03", 
" -.20742206e+01 -.49228449e+00  .43717020e+01  .37285238e-03", 
" -.20112719e+01 -.55037601e+00  .46200231e+01  .39603363e-03", 
" -.19351467e+01 -.67439923e+00  .57261967e+01  .44428079e-03", 
" -.18381966e+01 -.85719919e+00  .70561699e+01  .51382769e-03", 
" -.17040075e+01 -.97146869e+00  .34214724e+01  .59663015e-03", 
" -.16693176e+01 -.10142643e+01  .28316045e+01  .62217969e-03", 
" -.16300704e+01 -.11414282e+01  .42565629e+01  .65721635e-03", 
" -.15865872e+01 -.11251548e+01  .13000064e+01  .72482363e-03", 
" -.15357996e+01 -.12414557e+01  .17328513e+01  .79039294e-03", 
" -.14753514e+01 -.14706851e+01  .53535433e+01  .84068523e-03", 
" -.14017831e+01 -.16325064e+01  .44603323e+01  .94254921e-03", 
" -.13078726e+01 -.18171141e+01  .38718938e+01  .10870342e-02", 
" -.11752557e+01 -.22032707e+01  .84159800e+01  .13495181e-02", 
" -.95027908e+00 -.29232075e+01  .19165729e+02  .19121254e-02", 
" -.72631404e+00 -.38881056e+01  .31748679e+02  .26843980e-02", 
" -.44352927e+00 -.43536136e+01  .31777661e+02  .40889236e-02", 
" -.23321358e+00 -.49163713e+01  .38791965e+02  .54906368e-02", 
"coct      5  8  2   30", " -.66453096e+01 -.58596311e+02 -.31493199e+03  .44722262e-02", 
" -.64983818e+01 -.53839139e+02 -.28617082e+03  .29900318e-02", 
" -.62966974e+01 -.48278723e+02 -.21669754e+03  .23189320e-02", 
" -.61324556e+01 -.44002980e+02 -.17539543e+03  .16150069e-02", 
" -.59591525e+01 -.39860671e+02 -.13539457e+03  .11884508e-02", 
" -.58541532e+01 -.37413202e+02 -.11255121e+03  .10455349e-02", 
" -.57774689e+01 -.35572323e+02 -.10367446e+03  .95861692e-03", 
" -.57160943e+01 -.34258709e+02 -.93163952e+02  .84326788e-03", 
" -.56648586e+01 -.33223551e+02 -.82327189e+02  .77236026e-03", 
" -.56209012e+01 -.32287768e+02 -.77169504e+02  .72823335e-03", 
" -.55823631e+01 -.31512182e+02 -.70527469e+02  .69450286e-03", 
" -.55474368e+01 -.30863389e+02 -.64342524e+02  .67600467e-03", 
" -.55162649e+01 -.30181764e+02 -.62251276e+02  .63324484e-03", 
" -.53927069e+01 -.27739379e+02 -.48603195e+02  .52390892e-03", 
" -.53012483e+01 -.25921154e+02 -.42539278e+02  .47572094e-03", 
" -.52279145e+01 -.24502393e+02 -.38084029e+02  .43651046e-03", 
" -.51656860e+01 -.23430660e+02 -.32687318e+02  .40202544e-03", 
" -.51117539e+01 -.22547140e+02 -.27131815e+02  .39306992e-03", 
" -.50640515e+01 -.21726754e+02 -.24773005e+02  .36417200e-03", 
" -.50210259e+01 -.21025197e+02 -.21856943e+02  .35568142e-03", 
" -.49818118e+01 -.20376592e+02 -.20033992e+02  .34004614e-03", 
" -.49458175e+01 -.19774400e+02 -.18610221e+02  .32614566e-03", 
" -.49121964e+01 -.19251019e+02 -.16560939e+02  .30537597e-03", 
" -.48807780e+01 -.18766408e+02 -.15059411e+02  .30316027e-03", 
" -.48513551e+01 -.18298284e+02 -.13940455e+02  .29841109e-03", 
" -.48233754e+01 -.17892384e+02 -.12151946e+02  .29173918e-03", 
" -.47969170e+01 -.17481179e+02 -.11599522e+02  .29410236e-03", 
" -.47717147e+01 -.17129165e+02 -.98276363e+01  .29316662e-03", 
" -.47475500e+01 -.16791620e+02 -.84234052e+01  .28575479e-03", 
" -.47245260e+01 -.16459389e+02 -.74614023e+01  .28454152e-03", 
" -.47023035e+01 -.16139819e+02 -.68122337e+01  .27873766e-03", 
" -.46808748e+01 -.15857654e+02 -.54092236e+01  .27868290e-03", 
" -.46601161e+01 -.15593347e+02 -.41703079e+01  .26624628e-03", 
" -.46401403e+01 -.15303056e+02 -.41950032e+01  .26151630e-03", 
" -.46208201e+01 -.15042270e+02 -.35023080e+01  .26571092e-03", 
" -.46020056e+01 -.14807314e+02 -.25322222e+01  .26150200e-03", 
" -.45837750e+01 -.14566122e+02 -.19585354e+01  .25766049e-03", 
" -.45660277e+01 -.14342272e+02 -.11866263e+01  .25865057e-03", 
" -.45487614e+01 -.14128172e+02 -.20576284e+00  .25483086e-03", 
" -.45318870e+01 -.13908523e+02  .14562614e+00  .25428319e-03", 
" -.45154680e+01 -.13696926e+02  .53886418e+00  .25587057e-03", 
" -.44993849e+01 -.13481702e+02  .64944910e+00  .25120276e-03", 
" -.44836527e+01 -.13290390e+02  .13135592e+01  .24994566e-03", 
" -.44683298e+01 -.13088452e+02  .14455093e+01  .24334709e-03", 
" -.44531787e+01 -.12910933e+02  .19523339e+01  .24102052e-03", 
" -.44383808e+01 -.12735164e+02  .24841577e+01  .23619548e-03", 
" -.44238749e+01 -.12553404e+02  .25883045e+01  .23362023e-03", 
" -.44096005e+01 -.12389088e+02  .30489633e+01  .22683280e-03", 
" -.43955307e+01 -.12225295e+02  .33020725e+01  .22460864e-03", 
" -.43817025e+01 -.12075961e+02  .39593063e+01  .22291714e-03", 
" -.43681033e+01 -.11923651e+02  .43398629e+01  .22122295e-03", 
" -.43547850e+01 -.11761979e+02  .44215102e+01  .22107122e-03", 
" -.43416190e+01 -.11610133e+02  .47113259e+01  .22171918e-03", 
" -.43286165e+01 -.11470799e+02  .52352984e+01  .22296675e-03", 
" -.43158203e+01 -.11321026e+02  .52561927e+01  .21992964e-03", 
" -.43032544e+01 -.11175512e+02  .53864008e+01  .21674224e-03", 
" -.42908300e+01 -.11039837e+02  .57167484e+01  .21826795e-03", 
" -.42786251e+01 -.10898766e+02  .59072511e+01  .21961620e-03", 
" -.42664551e+01 -.10776960e+02  .64549113e+01  .21810936e-03", 
" -.42544642e+01 -.10646921e+02  .66726534e+01  .21751013e-03", 
" -.42426732e+01 -.10510424e+02  .65925375e+01  .21443889e-03", 
" -.42309863e+01 -.10384823e+02  .67851578e+01  .21507694e-03", 
" -.42193577e+01 -.10268916e+02  .72015564e+01  .21455346e-03", 
" -.42078248e+01 -.10156440e+02  .75525730e+01  .21478926e-03", 
" -.41964854e+01 -.10045944e+02  .80383804e+01  .21434552e-03", 
" -.41852454e+01 -.99255438e+01  .80617405e+01  .21349162e-03", 
" -.41740434e+01 -.98185331e+01  .84020613e+01  .21262967e-03", 
" -.41630003e+01 -.97006049e+01  .83011889e+01  .21107395e-03", 
" -.41520485e+01 -.95951390e+01  .86105182e+01  .20855496e-03", 
" -.41412488e+01 -.94765110e+01  .85708143e+01  .20805167e-03", 
" -.41305193e+01 -.93651775e+01  .85909607e+01  .20721935e-03", 
" -.41198934e+01 -.92523839e+01  .85417389e+01  .20960094e-03", 
" -.41092997e+01 -.91530705e+01  .88288048e+01  .20819092e-03", 
" -.40988463e+01 -.90457975e+01  .88020669e+01  .20752961e-03", 
" -.40884091e+01 -.89466848e+01  .89536127e+01  .20764443e-03", 
" -.40780415e+01 -.88539960e+01  .92740973e+01  .20689004e-03", 
" -.40677673e+01 -.87558484e+01  .94441909e+01  .20536244e-03", 
" -.40575250e+01 -.86613455e+01  .95232407e+01  .20520043e-03", 
" -.40474013e+01 -.85621666e+01  .95136236e+01  .20505056e-03", 
" -.40373490e+01 -.84595796e+01  .93783690e+01  .20619764e-03", 
" -.40272980e+01 -.83680449e+01  .95312361e+01  .20477633e-03", 
" -.40173750e+01 -.82748760e+01  .96618840e+01  .20219851e-03", 
" -.40074611e+01 -.81807110e+01  .96117177e+01  .20357269e-03", 
" -.39975905e+01 -.80935752e+01  .97415678e+01  .20505929e-03", 
" -.39878353e+01 -.79964754e+01  .96079189e+01  .20348995e-03", 
" -.39780421e+01 -.79119566e+01  .97739460e+01  .20439301e-03", 
" -.39683481e+01 -.78156373e+01  .95249791e+01  .20306168e-03", 
" -.39586455e+01 -.77339372e+01  .97593767e+01  .20408843e-03", 
" -.39490516e+01 -.76431301e+01  .96745009e+01  .20160997e-03", 
" -.39394597e+01 -.75564973e+01  .96814158e+01  .19978558e-03", 
" -.39299427e+01 -.74686073e+01  .96042699e+01  .19800025e-03", 
" -.39204270e+01 -.73840505e+01  .95662725e+01  .19826435e-03", 
" -.39109084e+01 -.73101347e+01  .98524076e+01  .19853207e-03", 
" -.39014716e+01 -.72250054e+01  .97446065e+01  .20001675e-03", 
" -.38920545e+01 -.71412286e+01  .97011119e+01  .19866284e-03", 
" -.38826633e+01 -.70694374e+01  .10085223e+02  .19725579e-03", 
" -.38733214e+01 -.69881567e+01  .10072720e+02  .19747704e-03", 
" -.38639488e+01 -.69106344e+01  .10037123e+02  .19627594e-03", 
" -.38546584e+01 -.68297487e+01  .99947196e+01  .19432647e-03", 
" -.38454017e+01 -.67495682e+01  .99077792e+01  .19335555e-03", 
" -.38361395e+01 -.66729162e+01  .99514067e+01  .19288195e-03", 
" -.38268505e+01 -.66010169e+01  .10054810e+02  .19388317e-03", 
" -.38175667e+01 -.65341424e+01  .10303190e+02  .19544749e-03", 
" -.38083522e+01 -.64571869e+01  .10211866e+02  .19574831e-03", 
" -.37991608e+01 -.63860321e+01  .10349725e+02  .19636088e-03", 
" -.37899344e+01 -.63171505e+01  .10466380e+02  .19480643e-03", 
" -.37807433e+01 -.62500309e+01  .10583590e+02  .19500048e-03", 
" -.37715314e+01 -.61851986e+01  .10750648e+02  .19550008e-03", 
" -.37623337e+01 -.61194859e+01  .10926950e+02  .19530350e-03", 
" -.37531826e+01 -.60526796e+01  .11087234e+02  .19256261e-03", 
" -.37439787e+01 -.59956798e+01  .11502863e+02  .19330716e-03", 
" -.37348157e+01 -.59249053e+01  .11468573e+02  .19351132e-03", 
" -.37257122e+01 -.58530890e+01  .11420681e+02  .19507815e-03", 
" -.37165667e+01 -.57865003e+01  .11524692e+02  .19325369e-03", 
" -.37074591e+01 -.57153280e+01  .11412120e+02  .19149210e-03", 
" -.36983596e+01 -.56419059e+01  .11192688e+02  .18743110e-03", 
" -.36892188e+01 -.55673930e+01  .10984800e+02  .18655064e-03", 
" -.36800460e+01 -.55022334e+01  .11035735e+02  .18526185e-03", 
" -.36708957e+01 -.54328006e+01  .10942394e+02  .18746847e-03", 
" -.36617635e+01 -.53612843e+01  .10754653e+02  .18736542e-03", 
" -.36525428e+01 -.53050888e+01  .11004311e+02  .18648952e-03", 
" -.36433174e+01 -.52490983e+01  .11248976e+02  .18640865e-03", 
" -.36341094e+01 -.51862122e+01  .11294459e+02  .18766695e-03", 
" -.36248828e+01 -.51232957e+01  .11304788e+02  .18794377e-03", 
" -.36156618e+01 -.50623687e+01  .11427386e+02  .18880719e-03", 
" -.36063526e+01 -.50075278e+01  .11655943e+02  .18678755e-03", 
" -.35971082e+01 -.49425781e+01  .11576271e+02  .18615475e-03", 
" -.35878209e+01 -.48847445e+01  .11718414e+02  .18710827e-03", 
" -.35784731e+01 -.48277686e+01  .11835483e+02  .18654827e-03", 
" -.35691384e+01 -.47640269e+01  .11727369e+02  .18420314e-03", 
" -.35597686e+01 -.47065004e+01  .11855357e+02  .18622923e-03", 
" -.35503782e+01 -.46430717e+01  .11784747e+02  .18653553e-03", 
" -.35409797e+01 -.45771418e+01  .11609283e+02  .18569153e-03", 
" -.35315231e+01 -.45188129e+01  .11674675e+02  .18721868e-03", 
" -.35220112e+01 -.44601401e+01  .11655248e+02  .18502409e-03", 
" -.35125425e+01 -.43885248e+01  .11211734e+02  .18373099e-03", 
" -.35029560e+01 -.43343323e+01  .11297626e+02  .18480496e-03", 
" -.34933582e+01 -.42783492e+01  .11357294e+02  .18621399e-03", 
" -.34837619e+01 -.42147576e+01  .11188504e+02  .18801547e-03", 
" -.34741128e+01 -.41518630e+01  .11032925e+02  .18817298e-03", 
" -.34643793e+01 -.40909826e+01  .10949468e+02  .19058536e-03", 
" -.34546229e+01 -.40344151e+01  .10950979e+02  .19313414e-03", 
" -.34447709e+01 -.39806320e+01  .10994568e+02  .19426234e-03", 
" -.34348880e+01 -.39214010e+01  .10806511e+02  .19257301e-03", 
" -.34249663e+01 -.38706037e+01  .11004155e+02  .19516881e-03", 
" -.34149664e+01 -.38207190e+01  .11125651e+02  .19497451e-03", 
" -.34049441e+01 -.37579316e+01  .10904769e+02  .19276632e-03", 
" -.33947847e+01 -.37111225e+01  .11124251e+02  .19228756e-03", 
" -.33846000e+01 -.36599793e+01  .11242658e+02  .19459516e-03", 
" -.33743546e+01 -.36107374e+01  .11431004e+02  .19357837e-03", 
" -.33640867e+01 -.35538163e+01  .11376098e+02  .19464992e-03", 
" -.33536649e+01 -.35021532e+01  .11410794e+02  .19380283e-03", 
" -.33431687e+01 -.34559129e+01  .11678708e+02  .19446965e-03", 
" -.33326187e+01 -.34082301e+01  .11886377e+02  .19461961e-03", 
" -.33220231e+01 -.33482132e+01  .11708716e+02  .19209639e-03", 
" -.33112983e+01 -.32935017e+01  .11644491e+02  .18955679e-03", 
" -.33004475e+01 -.32461909e+01  .11818664e+02  .18827891e-03", 
" -.32895119e+01 -.31928770e+01  .11813422e+02  .18715736e-03", 
" -.32785460e+01 -.31327536e+01  .11573371e+02  .18469072e-03", 
" -.32674210e+01 -.30777663e+01  .11450624e+02  .18638358e-03", 
" -.32561792e+01 -.30232437e+01  .11249989e+02  .18527286e-03", 
" -.32447631e+01 -.29862649e+01  .11661939e+02  .18848173e-03", 
" -.32332871e+01 -.29314323e+01  .11523359e+02  .19082068e-03", 
" -.32216426e+01 -.28796035e+01  .11398683e+02  .18691435e-03", 
" -.32098595e+01 -.28363220e+01  .11573636e+02  .18827890e-03", 
" -.31979780e+01 -.27900323e+01  .11739098e+02  .19022971e-03", 
" -.31858843e+01 -.27450471e+01  .11852110e+02  .19187072e-03", 
" -.31737032e+01 -.26839043e+01  .11480775e+02  .19148103e-03", 
" -.31613453e+01 -.26342643e+01  .11448376e+02  .19138114e-03", 
" -.31487992e+01 -.25896712e+01  .11641118e+02  .19205536e-03", 
" -.31360773e+01 -.25425063e+01  .11757335e+02  .19359458e-03", 
" -.31231690e+01 -.24983123e+01  .11975328e+02  .20157177e-03", 
" -.31101022e+01 -.24475752e+01  .12016397e+02  .20394826e-03", 
" -.30968009e+01 -.23996122e+01  .12085337e+02  .20789305e-03", 
" -.30833137e+01 -.23448674e+01  .11900706e+02  .20641863e-03", 
" -.30694984e+01 -.22920478e+01  .11686561e+02  .20848456e-03", 
" -.30555019e+01 -.22408604e+01  .11616616e+02  .21054117e-03", 
" -.30412067e+01 -.21911246e+01  .11574394e+02  .21322429e-03", 
" -.30266294e+01 -.21465341e+01  .11669567e+02  .21494740e-03", 
" -.30117188e+01 -.21046423e+01  .11814993e+02  .21405055e-03", 
" -.29965602e+01 -.20571483e+01  .11747106e+02  .21556679e-03", 
" -.29810410e+01 -.20080173e+01  .11700924e+02  .21823319e-03", 
" -.29651979e+01 -.19505307e+01  .11481032e+02  .22160944e-03", 
" -.29488720e+01 -.18988799e+01  .11214839e+02  .22237429e-03", 
" -.29321248e+01 -.18528785e+01  .11167509e+02  .22356888e-03", 
" -.29149608e+01 -.18107078e+01  .11427990e+02  .22997957e-03", 
" -.28972961e+01 -.17693725e+01  .11570948e+02  .23427343e-03", 
" -.28790880e+01 -.17224098e+01  .11548747e+02  .23505711e-03", 
" -.28602971e+01 -.16793975e+01  .11635056e+02  .23739211e-03", 
" -.28407681e+01 -.16514123e+01  .12093494e+02  .23699378e-03", 
" -.28206224e+01 -.15984797e+01  .11753213e+02  .23670254e-03", 
" -.27997071e+01 -.15576614e+01  .11787701e+02  .24176618e-03", 
" -.27779738e+01 -.15129023e+01  .11616917e+02  .25506674e-03", 
" -.27551593e+01 -.14939688e+01  .12191421e+02  .25795098e-03", 
" -.27312931e+01 -.14686546e+01  .12536276e+02  .25904682e-03", 
" -.27062913e+01 -.14368094e+01  .12566847e+02  .26109338e-03", 
" -.26799985e+01 -.14052523e+01  .12583911e+02  .26609124e-03", 
" -.26520216e+01 -.13840200e+01  .12763297e+02  .27207002e-03", 
" -.26223726e+01 -.13563399e+01  .12929808e+02  .27543609e-03", 
" -.25905733e+01 -.13394992e+01  .13107142e+02  .28285219e-03", 
" -.25564508e+01 -.13182599e+01  .13297746e+02  .30390044e-03", 
" -.25192432e+01 -.13022653e+01  .13245512e+02  .30876371e-03", 
" -.24780281e+01 -.13427222e+01  .14502280e+02  .32313072e-03", 
" -.24323578e+01 -.13677070e+01  .15220095e+02  .33828888e-03", 
" -.23805461e+01 -.14128812e+01  .15807072e+02  .35785003e-03", 
" -.23202626e+01 -.14946548e+01  .17008448e+02  .39840897e-03", 
" -.22481799e+01 -.15967424e+01  .17545045e+02  .44294132e-03", 
" -.21570968e+01 -.17538410e+01  .18103765e+02  .50194957e-03", 
" -.20310115e+01 -.21022216e+01  .20064989e+02  .58311943e-03", 
" -.19986504e+01 -.21805875e+01  .20039656e+02  .60681499e-03", 
" -.19626699e+01 -.22921034e+01  .20927334e+02  .66114892e-03", 
" -.19214637e+01 -.24882547e+01  .22832868e+02  .68622636e-03", 
" -.18748211e+01 -.26130239e+01  .22263160e+02  .73217799e-03", 
" -.18194086e+01 -.27896362e+01  .22329724e+02  .81395545e-03", 
" -.17515169e+01 -.30728457e+01  .23648388e+02  .87830295e-03", 
" -.16653666e+01 -.33779272e+01  .23146277e+02  .10007642e-02", 
" -.15430354e+01 -.39100973e+01  .24250729e+02  .12263442e-02", 
" -.13371289e+01 -.47973786e+01  .27456257e+02  .17519060e-02", 
" -.11350583e+01 -.56846075e+01  .26666048e+02  .23655010e-02", 
" -.87373192e+00 -.71315642e+01  .36226623e+02  .35726010e-02", 
" -.68524933e+00 -.78259936e+01  .35139080e+02  .46043229e-02", 
"coctt     5  9  3   25", " -.68923904e+01 -.65474350e+02 -.39183922e+03 -.10887930e+03  .52967719e-02", 
" -.67459948e+01 -.61774275e+02 -.27845009e+03 -.78189387e+03  .39361286e-02", 
" -.65469376e+01 -.54261613e+02 -.28291420e+03  .33330933e+03  .26391162e-02", 
" -.63836773e+01 -.50275114e+02 -.19908721e+03 -.22551437e+03  .19381875e-02", 
" -.62127649e+01 -.45377467e+02 -.18363943e+03  .26275328e+03  .15264268e-02", 
" -.61076515e+01 -.42874753e+02 -.15618909e+03  .28601950e+03  .12551870e-02", 
" -.60323744e+01 -.40820424e+02 -.14764044e+03  .40425900e+03  .11044655e-02", 
" -.59716325e+01 -.39299873e+02 -.13635930e+03  .35696348e+03  .98907835e-03", 
" -.59212918e+01 -.37996752e+02 -.13419492e+03  .47038459e+03  .94410846e-03", 
" -.58778689e+01 -.36939418e+02 -.12822041e+03  .47123990e+03  .89857601e-03", 
" -.58388536e+01 -.36377253e+02 -.10235964e+03  .16876098e+03  .84517719e-03", 
" -.58046115e+01 -.35559999e+02 -.10111136e+03  .25428968e+03  .79752852e-03", 
" -.57737963e+01 -.34824094e+02 -.98522452e+02  .29412560e+03  .76207084e-03", 
" -.56504607e+01 -.32176048e+02 -.82324555e+02  .31982890e+03  .61052332e-03", 
" -.55591591e+01 -.30390202e+02 -.62103961e+02  .14697376e+03  .53267752e-03", 
" -.54860132e+01 -.28928762e+02 -.54099138e+02  .15397463e+03  .49082997e-03", 
" -.54242047e+01 -.27766563e+02 -.45470458e+02  .11380181e+03  .48255485e-03", 
" -.53707925e+01 -.26708732e+02 -.43606634e+02  .17168134e+03  .44597836e-03", 
" -.53234018e+01 -.25806168e+02 -.41666693e+02  .21152857e+03  .43102537e-03", 
" -.52806150e+01 -.25036772e+02 -.36779523e+02  .17458146e+03  .40924392e-03", 
" -.52416196e+01 -.24339688e+02 -.32658549e+02  .14564629e+03  .38820948e-03", 
" -.52057544e+01 -.23682103e+02 -.31514073e+02  .16429616e+03  .37650116e-03", 
" -.51725216e+01 -.23090191e+02 -.29796112e+02  .17755544e+03  .36566282e-03", 
" -.51411671e+01 -.22559095e+02 -.27351758e+02  .16751010e+03  .35919167e-03", 
" -.51118733e+01 -.22062047e+02 -.25130675e+02  .15435286e+03  .34699446e-03", 
" -.50840245e+01 -.21621695e+02 -.22069675e+02  .12871692e+03  .34595246e-03", 
" -.50578643e+01 -.21135225e+02 -.23151083e+02  .16554264e+03  .33314721e-03", 
" -.50327324e+01 -.20740837e+02 -.20991940e+02  .15462909e+03  .32855976e-03", 
" -.50087822e+01 -.20363596e+02 -.18737807e+02  .13758579e+03  .32334446e-03", 
" -.49859178e+01 -.19968070e+02 -.19804863e+02  .18150340e+03  .32159527e-03", 
" -.49638754e+01 -.19610002e+02 -.19786624e+02  .20394192e+03  .31986414e-03", 
" -.49425403e+01 -.19300067e+02 -.17637077e+02  .18897886e+03  .31750472e-03", 
" -.49219734e+01 -.18994015e+02 -.16229293e+02  .18014970e+03  .31194312e-03", 
" -.49021481e+01 -.18670742e+02 -.16528855e+02  .19468614e+03  .30674317e-03", 
" -.48828694e+01 -.18397433e+02 -.14605187e+02  .17768116e+03  .29913374e-03", 
" -.48640749e+01 -.18161434e+02 -.11292937e+02  .13913660e+03  .29642212e-03", 
" -.48459776e+01 -.17878014e+02 -.11650895e+02  .15923271e+03  .29001024e-03", 
" -.48283947e+01 -.17598779e+02 -.12145270e+02  .17730645e+03  .29436801e-03", 
" -.48111529e+01 -.17356429e+02 -.11292719e+02  .17527889e+03  .29632528e-03", 
" -.47945089e+01 -.17095883e+02 -.11106338e+02  .17524551e+03  .28820233e-03", 
" -.47781728e+01 -.16862583e+02 -.10835261e+02  .18574920e+03  .29216898e-03", 
" -.47621273e+01 -.16655835e+02 -.91156022e+01  .16747912e+03  .28320374e-03", 
" -.47464793e+01 -.16442232e+02 -.80389403e+01  .15798498e+03  .28220102e-03", 
" -.47311296e+01 -.16254300e+02 -.58714245e+01  .12961982e+03  .27343392e-03", 
" -.47160876e+01 -.16052040e+02 -.52029754e+01  .12653824e+03  .27716075e-03", 
" -.47013955e+01 -.15866747e+02 -.36207153e+01  .10898168e+03  .27260974e-03", 
" -.46869939e+01 -.15670751e+02 -.26670928e+01  .95485828e+02  .27050358e-03", 
" -.46727609e+01 -.15496290e+02 -.17256230e+01  .93604527e+02  .27266849e-03", 
" -.46588925e+01 -.15284909e+02 -.29400263e+01  .12012076e+03  .27101026e-03", 
" -.46452631e+01 -.15098040e+02 -.26017542e+01  .11729622e+03  .26981421e-03", 
" -.46318269e+01 -.14923296e+02 -.21269003e+01  .11898742e+03  .26800019e-03", 
" -.46185661e+01 -.14741753e+02 -.23493063e+01  .12712054e+03  .26582510e-03", 
" -.46054352e+01 -.14576968e+02 -.24152357e+01  .14073255e+03  .25881943e-03", 
" -.45925553e+01 -.14402378e+02 -.28196013e+01  .15249318e+03  .25659243e-03", 
" -.45798636e+01 -.14241733e+02 -.24810460e+01  .15469199e+03  .25972325e-03", 
" -.45673220e+01 -.14101763e+02 -.10899901e+01  .13824782e+03  .25668765e-03", 
" -.45549941e+01 -.13959340e+02  .41052048e+00  .11480688e+03  .25284593e-03", 
" -.45427627e+01 -.13823000e+02  .17360824e+01  .98010228e+02  .25197078e-03", 
" -.45307477e+01 -.13680169e+02  .29787430e+01  .76374696e+02  .24804325e-03", 
" -.45188712e+01 -.13522482e+02  .24658020e+01  .92311219e+02  .24625575e-03", 
" -.45071358e+01 -.13370113e+02  .22641698e+01  .99930761e+02  .24517462e-03", 
" -.44955637e+01 -.13218569e+02  .18590740e+01  .11176295e+03  .24338585e-03", 
" -.44840587e+01 -.13083660e+02  .23479077e+01  .10899600e+03  .24008487e-03", 
" -.44726950e+01 -.12938428e+02  .20844933e+01  .11744433e+03  .24243161e-03", 
" -.44613715e+01 -.12817060e+02  .27918760e+01  .11335158e+03  .24338430e-03", 
" -.44501694e+01 -.12698576e+02  .39304846e+01  .96125986e+02  .24364073e-03", 
" -.44390901e+01 -.12574514e+02  .46163993e+01  .86218337e+02  .23900794e-03", 
" -.44281119e+01 -.12445772e+02  .45489303e+01  .92433123e+02  .23427854e-03", 
" -.44172026e+01 -.12335056e+02  .56813888e+01  .76887561e+02  .23498340e-03", 
" -.44064075e+01 -.12213587e+02  .60104185e+01  .75010744e+02  .23333237e-03", 
" -.43957021e+01 -.12087700e+02  .58668434e+01  .80486753e+02  .23089014e-03", 
" -.43851363e+01 -.11947351e+02  .46703248e+01  .10291879e+03  .23231027e-03", 
" -.43746207e+01 -.11822565e+02  .45208226e+01  .10771730e+03  .23324997e-03", 
" -.43642725e+01 -.11675754e+02  .28622930e+01  .13681738e+03  .23296294e-03", 
" -.43540215e+01 -.11539074e+02  .19663119e+01  .15119859e+03  .23019060e-03", 
" -.43436934e+01 -.11439468e+02  .27979817e+01  .14312646e+03  .23328638e-03", 
" -.43334446e+01 -.11335330e+02  .31583659e+01  .14359073e+03  .23062066e-03", 
" -.43233455e+01 -.11217133e+02  .30570064e+01  .14671174e+03  .22807249e-03", 
" -.43132271e+01 -.11118391e+02  .36145062e+01  .14248134e+03  .22998883e-03", 
" -.43031830e+01 -.11027841e+02  .52109831e+01  .11494129e+03  .23114692e-03", 
" -.42932193e+01 -.10923624e+02  .55283579e+01  .11264334e+03  .23169761e-03", 
" -.42833226e+01 -.10819581e+02  .58892288e+01  .10595225e+03  .23099762e-03", 
" -.42734427e+01 -.10728952e+02  .69319326e+01  .90726450e+02  .23055461e-03", 
" -.42636541e+01 -.10628826e+02  .74761759e+01  .81805926e+02  .22746854e-03", 
" -.42538905e+01 -.10538674e+02  .83288806e+01  .70216412e+02  .23021575e-03", 
" -.42441162e+01 -.10462988e+02  .10025273e+02  .44648176e+02  .23220807e-03", 
" -.42344181e+01 -.10384966e+02  .11553687e+02  .21875557e+02  .22988576e-03", 
" -.42248405e+01 -.10281864e+02  .11500815e+02  .22862861e+02  .22797230e-03", 
" -.42152673e+01 -.10190431e+02  .12101839e+02  .13829991e+02  .23229601e-03", 
" -.42057207e+01 -.10097716e+02  .12012888e+02  .21805924e+02  .23113505e-03", 
" -.41962037e+01 -.10011684e+02  .12704378e+02  .11159374e+02  .22833780e-03", 
" -.41867322e+01 -.99113015e+01  .12245814e+02  .20946348e+02  .22717391e-03", 
" -.41772729e+01 -.98273352e+01  .12830696e+02  .14639169e+02  .22719521e-03", 
" -.41678858e+01 -.97253532e+01  .12229541e+02  .26652822e+02  .22526446e-03", 
" -.41585486e+01 -.96236636e+01  .11769633e+02  .34197677e+02  .22353676e-03", 
" -.41492023e+01 -.95388298e+01  .12167960e+02  .30247299e+02  .22352455e-03", 
" -.41399815e+01 -.94280562e+01  .11055913e+02  .47958352e+02  .22558845e-03", 
" -.41307061e+01 -.93336216e+01  .10741578e+02  .54424980e+02  .22490697e-03", 
" -.41214331e+01 -.92580433e+01  .11667070e+02  .39402315e+02  .22399150e-03", 
" -.41121485e+01 -.91815190e+01  .12218757e+02  .32620310e+02  .22558408e-03", 
" -.41029772e+01 -.90878579e+01  .11898664e+02  .37880208e+02  .22320258e-03", 
" -.40938182e+01 -.89928539e+01  .11599885e+02  .40171855e+02  .22262667e-03", 
" -.40846495e+01 -.88972121e+01  .10670516e+02  .60187186e+02  .22295562e-03", 
" -.40754743e+01 -.88157618e+01  .10919667e+02  .58628649e+02  .22228072e-03", 
" -.40663581e+01 -.87242055e+01  .10536112e+02  .65352710e+02  .21951500e-03", 
" -.40572032e+01 -.86420673e+01  .10635971e+02  .64082449e+02  .21963519e-03", 
" -.40481038e+01 -.85555343e+01  .10542609e+02  .64949163e+02  .21954298e-03", 
" -.40389936e+01 -.84719497e+01  .10250075e+02  .73083193e+02  .22020660e-03", 
" -.40298892e+01 -.83816202e+01  .96178281e+01  .84887359e+02  .21756570e-03", 
" -.40207944e+01 -.83099025e+01  .10163438e+02  .77623215e+02  .22128806e-03", 
" -.40116292e+01 -.82451735e+01  .10995354e+02  .65963408e+02  .22244535e-03", 
" -.40026149e+01 -.81481802e+01  .99365450e+01  .84538158e+02  .22011981e-03", 
" -.39934748e+01 -.80827406e+01  .10598607e+02  .77120709e+02  .22109881e-03", 
" -.39844024e+01 -.80052844e+01  .10534389e+02  .82096617e+02  .21846513e-03", 
" -.39753275e+01 -.79186301e+01  .98257819e+01  .96749895e+02  .21976023e-03", 
" -.39662170e+01 -.78502310e+01  .10275935e+02  .91093234e+02  .21775245e-03", 
" -.39570939e+01 -.77832481e+01  .10685813e+02  .88092276e+02  .21922398e-03", 
" -.39479952e+01 -.77081658e+01  .10825470e+02  .85808240e+02  .21939718e-03", 
" -.39388803e+01 -.76464524e+01  .11878786e+02  .67627355e+02  .21935437e-03", 
" -.39297546e+01 -.75776833e+01  .12301128e+02  .62251630e+02  .22069969e-03", 
" -.39206470e+01 -.75012888e+01  .11998370e+02  .70675915e+02  .22186635e-03", 
" -.39114789e+01 -.74431417e+01  .12863585e+02  .60404617e+02  .21732960e-03", 
" -.39023059e+01 -.73749165e+01  .13070364e+02  .58715178e+02  .21707229e-03", 
" -.38931421e+01 -.73090340e+01  .13417971e+02  .54832307e+02  .21911053e-03", 
" -.38839800e+01 -.72336725e+01  .13226699e+02  .58390696e+02  .22174666e-03", 
" -.38747915e+01 -.71633895e+01  .13309894e+02  .57971334e+02  .22133864e-03", 
" -.38655624e+01 -.70978947e+01  .13555063e+02  .57434201e+02  .21977859e-03", 
" -.38563452e+01 -.70182997e+01  .12931504e+02  .68110055e+02  .22025055e-03", 
" -.38470550e+01 -.69542176e+01  .13164413e+02  .65190490e+02  .21743545e-03", 
" -.38378089e+01 -.68671592e+01  .11891663e+02  .88013016e+02  .21906126e-03", 
" -.38284883e+01 -.68091628e+01  .12699950e+02  .75900158e+02  .21931161e-03", 
" -.38191565e+01 -.67452526e+01  .12980488e+02  .73124958e+02  .21911143e-03", 
" -.38098240e+01 -.66779527e+01  .13017078e+02  .75408114e+02  .21722541e-03", 
" -.38004122e+01 -.66149504e+01  .13186323e+02  .74421973e+02  .21753896e-03", 
" -.37909600e+01 -.65508695e+01  .13254943e+02  .75537414e+02  .21773836e-03", 
" -.37814790e+01 -.64967680e+01  .14073400e+02  .63033253e+02  .21907291e-03", 
" -.37720173e+01 -.64245038e+01  .13943029e+02  .64370182e+02  .21818958e-03", 
" -.37625054e+01 -.63444465e+01  .12910570e+02  .82849119e+02  .22050486e-03", 
" -.37529786e+01 -.62733979e+01  .12737167e+02  .85891372e+02  .22346354e-03", 
" -.37433650e+01 -.62123882e+01  .13042067e+02  .83011662e+02  .22297061e-03", 
" -.37337342e+01 -.61581876e+01  .13958051e+02  .68341434e+02  .22110709e-03", 
" -.37240278e+01 -.60843026e+01  .13295543e+02  .81191414e+02  .21850974e-03", 
" -.37142131e+01 -.60456209e+01  .14926326e+02  .55536971e+02  .22027699e-03", 
" -.37043658e+01 -.59850594e+01  .15045788e+02  .55195488e+02  .22396558e-03", 
" -.36944671e+01 -.59302500e+01  .15691832e+02  .44991345e+02  .22158710e-03", 
" -.36846396e+01 -.58528017e+01  .15001683e+02  .56553445e+02  .22178786e-03", 
" -.36746553e+01 -.57983587e+01  .15498756e+02  .50550211e+02  .22297134e-03", 
" -.36647044e+01 -.57234877e+01  .14923991e+02  .60237796e+02  .22250860e-03", 
" -.36545961e+01 -.56730920e+01  .15868624e+02  .44859775e+02  .22702684e-03", 
" -.36444120e+01 -.56278253e+01  .17054620e+02  .27695332e+02  .22268828e-03", 
" -.36341666e+01 -.55672623e+01  .16949939e+02  .32373286e+02  .22363778e-03", 
" -.36238586e+01 -.55066078e+01  .17234793e+02  .27213302e+02  .22531725e-03", 
" -.36135078e+01 -.54403392e+01  .17303847e+02  .24082060e+02  .22519686e-03", 
" -.36030820e+01 -.53710244e+01  .17183536e+02  .22590803e+02  .22730625e-03", 
" -.35925349e+01 -.53085801e+01  .17162653e+02  .22645907e+02  .22708263e-03", 
" -.35818801e+01 -.52621701e+01  .18162007e+02  .70395013e+01  .22475003e-03", 
" -.35711487e+01 -.52121259e+01  .18945816e+02 -.46063806e+01  .22315726e-03", 
" -.35603752e+01 -.51415445e+01  .18384577e+02  .56500009e+01  .22629077e-03", 
" -.35494189e+01 -.50924876e+01  .19208328e+02 -.78209402e+01  .22884218e-03", 
" -.35383576e+01 -.50426124e+01  .19817010e+02 -.17016385e+02  .22739964e-03", 
" -.35272281e+01 -.49766741e+01  .19596585e+02 -.15391209e+02  .22810674e-03", 
" -.35160476e+01 -.49021544e+01  .19018595e+02 -.93061003e+01  .22810112e-03", 
" -.35046946e+01 -.48370813e+01  .18621576e+02  .47276571e+00  .23010402e-03", 
" -.34932098e+01 -.47783515e+01  .18741153e+02 -.16470791e+01  .23144117e-03", 
" -.34815430e+01 -.47307713e+01  .19678730e+02 -.18596467e+02  .23338914e-03", 
" -.34697828e+01 -.46805365e+01  .20334333e+02 -.28674525e+02  .23290969e-03", 
" -.34578785e+01 -.46269144e+01  .21088277e+02 -.45855728e+02  .23701305e-03", 
" -.34458019e+01 -.45803638e+01  .21928396e+02 -.60591831e+02  .23897069e-03", 
" -.34335553e+01 -.45179962e+01  .21452801e+02 -.50463110e+02  .23999969e-03", 
" -.34211831e+01 -.44511267e+01  .21001273e+02 -.42950230e+02  .24017788e-03", 
" -.34086427e+01 -.43767930e+01  .19654532e+02 -.15950711e+02  .24543000e-03", 
" -.33958833e+01 -.43237289e+01  .20055547e+02 -.21592892e+02  .24719598e-03", 
" -.33829491e+01 -.42699863e+01  .20409435e+02 -.27911175e+02  .25117822e-03", 
" -.33697698e+01 -.42180584e+01  .20731993e+02 -.32026151e+02  .25333608e-03", 
" -.33563664e+01 -.41848009e+01  .22526227e+02 -.63627068e+02  .25303247e-03", 
" -.33427194e+01 -.41391807e+01  .23158997e+02 -.72577603e+02  .25884168e-03", 
" -.33287735e+01 -.41067103e+01  .24891411e+02 -.10136109e+03  .26105265e-03", 
" -.33146870e+01 -.40478900e+01  .25025402e+02 -.10467433e+03  .26226769e-03", 
" -.33003977e+01 -.39668657e+01  .23671444e+02 -.84213257e+02  .26396306e-03", 
" -.32857089e+01 -.39187791e+01  .23914324e+02 -.83467245e+02  .26480232e-03", 
" -.32706254e+01 -.38715416e+01  .23984410e+02 -.79109762e+02  .26562284e-03", 
" -.32551530e+01 -.38568571e+01  .25992501e+02 -.10596011e+03  .26724988e-03", 
" -.32394944e+01 -.38015646e+01  .26450815e+02 -.11788930e+03  .26868085e-03", 
" -.32234078e+01 -.37506649e+01  .27040798e+02 -.13064581e+03  .26649606e-03", 
" -.32068497e+01 -.36957321e+01  .26484300e+02 -.11482443e+03  .26959119e-03", 
" -.31899309e+01 -.36352884e+01  .26112574e+02 -.11022772e+03  .26998961e-03", 
" -.31725027e+01 -.35968345e+01  .27200344e+02 -.12609567e+03  .27191490e-03", 
" -.31545645e+01 -.35488120e+01  .27889011e+02 -.13928501e+03  .27536006e-03", 
" -.31361445e+01 -.34716442e+01  .26319586e+02 -.11429734e+03  .27579440e-03", 
" -.31170527e+01 -.34217253e+01  .26317505e+02 -.11034543e+03  .27869971e-03", 
" -.30972647e+01 -.33677611e+01  .25970668e+02 -.99160159e+02  .28222137e-03", 
" -.30767639e+01 -.33064448e+01  .25176747e+02 -.82605442e+02  .28420768e-03", 
" -.30553060e+01 -.32721717e+01  .25838264e+02 -.90719483e+02  .29029147e-03", 
" -.30328817e+01 -.32643322e+01  .27730395e+02 -.11568142e+03  .29311904e-03", 
" -.30095582e+01 -.32478149e+01  .29532453e+02 -.14183954e+03  .29980856e-03", 
" -.29850103e+01 -.32257303e+01  .30129190e+02 -.14256656e+03  .30240434e-03", 
" -.29591899e+01 -.32063851e+01  .30995754e+02 -.14799171e+03  .31359276e-03", 
" -.29317766e+01 -.32082720e+01  .32296622e+02 -.15458670e+03  .32494458e-03", 
" -.29027331e+01 -.31918541e+01  .33311195e+02 -.16422236e+03  .33686670e-03", 
" -.28717402e+01 -.31555790e+01  .32094621e+02 -.12845182e+03  .34266696e-03", 
" -.28381565e+01 -.31983156e+01  .35360365e+02 -.16433169e+03  .34198692e-03", 
" -.28015516e+01 -.32765663e+01  .39634383e+02 -.21086169e+03  .35000875e-03", 
" -.27617386e+01 -.32749428e+01  .37940220e+02 -.15666194e+03  .37602306e-03", 
" -.27172628e+01 -.33468014e+01  .41001674e+02 -.18690845e+03  .38700266e-03", 
" -.26669009e+01 -.35062425e+01  .45901552e+02 -.21134755e+03  .40699892e-03", 
" -.26090733e+01 -.36333869e+01  .49978551e+02 -.25086592e+03  .45211454e-03", 
" -.25393575e+01 -.38740925e+01  .56465020e+02 -.32177128e+03  .47687661e-03", 
" -.24516010e+01 -.43583981e+01  .72223139e+02 -.51874954e+03  .56425822e-03", 
" -.23320383e+01 -.49782686e+01  .83963786e+02 -.58387005e+03  .67649600e-03", 
" -.23011831e+01 -.51891068e+01  .87438003e+02 -.59637180e+03  .70255251e-03", 
" -.22675494e+01 -.53379685e+01  .88055163e+02 -.58103501e+03  .72896198e-03", 
" -.22290113e+01 -.56927774e+01  .99247160e+02 -.72948514e+03  .78474930e-03", 
" -.21850029e+01 -.59734211e+01  .10263531e+03 -.74053898e+03  .80271965e-03", 
" -.21336206e+01 -.63734347e+01  .11113894e+03 -.83922066e+03  .84437592e-03", 
" -.20701776e+01 -.70915635e+01  .13004557e+03 -.10535298e+04  .96661968e-03", 
" -.19890824e+01 -.79517180e+01  .14715640e+03 -.12017535e+04  .11332735e-02", 
" -.18765342e+01 -.88272428e+01  .15061887e+03 -.11266671e+04  .13535815e-02", 
" -.16868384e+01 -.10566615e+02  .17059064e+03 -.12432061e+04  .18517407e-02", 
" -.14991020e+01 -.12313662e+02  .17457940e+03 -.10807235e+04  .25600557e-02", 
" -.12564007e+01 -.14508603e+02  .16667736e+03 -.55676748e+03  .38621303e-02", 
" -.10755932e+01 -.17486978e+02  .24000961e+03 -.15030730e+04  .51250081e-02", 
"coanc     5  6  3   30", " -.74869894e+02  .16045714e+04 -.22914394e+05  .16464274e+06  .10750249e+00", 
" -.71344073e+02  .14789062e+04 -.20939780e+05  .15301901e+06  .79731920e-01", 
" -.66552920e+02  .13242765e+04 -.19205421e+05  .15435757e+06  .51814820e-01", 
" -.62804548e+02  .11915232e+04 -.16664028e+05  .13029029e+06  .35494530e-01", 
" -.58963169e+02  .10522394e+04 -.13543213e+05  .95854609e+05  .25471357e-01", 
" -.56688792e+02  .98100746e+03 -.12381934e+05  .87009224e+05  .21537322e-01", 
" -.55030717e+02  .92683034e+03 -.11346354e+05  .77362044e+05  .18532820e-01", 
" -.53753931e+02  .89252944e+03 -.10969399e+05  .76272526e+05  .16950814e-01", 
" -.52685601e+02  .85753516e+03 -.10189818e+05  .66960530e+05  .15720413e-01", 
" -.51780712e+02  .83110854e+03 -.97787420e+04  .63987873e+05  .15225114e-01", 
" -.50995213e+02  .80888537e+03 -.94424365e+04  .61551878e+05  .13882349e-01", 
" -.50293609e+02  .78796314e+03 -.90686658e+04  .58159369e+05  .12981974e-01", 
" -.49662847e+02  .76899049e+03 -.87073033e+04  .54559974e+05  .12438055e-01", 
" -.47210479e+02  .70430767e+03 -.78728605e+04  .50612153e+05  .10685437e-01", 
" -.45438979e+02  .65853835e+03 -.72362534e+04  .46583906e+05  .92265255e-02", 
" -.44045281e+02  .62364027e+03 -.67385707e+04  .42899375e+05  .83755170e-02", 
" -.42886718e+02  .59453886e+03 -.63235874e+04  .40155991e+05  .77767906e-02", 
" -.41894496e+02  .57062937e+03 -.59919293e+04  .37699132e+05  .72962459e-02", 
" -.41027014e+02  .55017418e+03 -.57123652e+04  .35662681e+05  .69798241e-02", 
" -.40243972e+02  .53043399e+03 -.53748751e+04  .32407554e+05  .66167762e-02", 
" -.39542333e+02  .51441957e+03 -.51625537e+04  .30891722e+05  .64130394e-02", 
" -.38902444e+02  .50014016e+03 -.49714491e+04  .29340318e+05  .61079212e-02", 
" -.38311338e+02  .48699705e+03 -.48072076e+04  .28295701e+05  .57267278e-02", 
" -.37762816e+02  .47432579e+03 -.46094031e+04  .26444298e+05  .56131734e-02", 
" -.37251264e+02  .46352241e+03 -.44969728e+04  .26072835e+05  .54898189e-02", 
" -.36770895e+02  .45295613e+03 -.43605383e+04  .25226012e+05  .53184808e-02", 
" -.36315563e+02  .44275575e+03 -.42137086e+04  .24056702e+05  .52456985e-02", 
" -.35885523e+02  .43335622e+03 -.40785213e+04  .22914019e+05  .50772723e-02", 
" -.35474833e+02  .42400990e+03 -.39241285e+04  .21390167e+05  .49742252e-02", 
" -.35087456e+02  .41624378e+03 -.38492913e+04  .21242900e+05  .48398261e-02", 
" -.34715291e+02  .40841749e+03 -.37510053e+04  .20649883e+05  .47840433e-02", 
" -.34359416e+02  .40113325e+03 -.36630117e+04  .20096509e+05  .47299984e-02", 
" -.34019716e+02  .39450357e+03 -.35994604e+04  .19946977e+05  .46313193e-02", 
" -.33690968e+02  .38752212e+03 -.34956291e+04  .19004288e+05  .46336316e-02", 
" -.33373580e+02  .38085234e+03 -.33943357e+04  .18032725e+05  .45695427e-02", 
" -.33068091e+02  .37492115e+03 -.33378428e+04  .17922100e+05  .44919194e-02", 
" -.32771312e+02  .36872635e+03 -.32500537e+04  .17227854e+05  .43997849e-02", 
" -.32487321e+02  .36313874e+03 -.31783410e+04  .16677434e+05  .43639103e-02", 
" -.32209142e+02  .35754735e+03 -.31045023e+04  .16143687e+05  .42875279e-02", 
" -.31940835e+02  .35266150e+03 -.30705164e+04  .16316315e+05  .41300847e-02", 
" -.31678605e+02  .34752735e+03 -.30102012e+04  .15978622e+05  .40830344e-02", 
" -.31423499e+02  .34243501e+03 -.29379798e+04  .15337295e+05  .40806581e-02", 
" -.31175916e+02  .33754590e+03 -.28673721e+04  .14697471e+05  .40434444e-02", 
" -.30933713e+02  .33276392e+03 -.28001835e+04  .14142430e+05  .39902859e-02", 
" -.30697422e+02  .32815003e+03 -.27301928e+04  .13394148e+05  .38779562e-02", 
" -.30467781e+02  .32393285e+03 -.26804606e+04  .13009750e+05  .38022373e-02", 
" -.30243970e+02  .32007482e+03 -.26459581e+04  .12843139e+05  .37722430e-02", 
" -.30025626e+02  .31652696e+03 -.26318721e+04  .13128190e+05  .37553371e-02", 
" -.29809012e+02  .31240220e+03 -.25789147e+04  .12736431e+05  .37786311e-02", 
" -.29598073e+02  .30858249e+03 -.25397242e+04  .12569619e+05  .37211197e-02", 
" -.29392052e+02  .30494895e+03 -.25007289e+04  .12330267e+05  .37082726e-02", 
" -.29188515e+02  .30119032e+03 -.24560013e+04  .12043892e+05  .36460702e-02", 
" -.28990128e+02  .29782735e+03 -.24306156e+04  .12068692e+05  .36328619e-02", 
" -.28793060e+02  .29401615e+03 -.23722741e+04  .11490121e+05  .36182785e-02", 
" -.28600391e+02  .29050825e+03 -.23324981e+04  .11294470e+05  .35446899e-02", 
" -.28412034e+02  .28722206e+03 -.22974432e+04  .11091532e+05  .34832647e-02", 
" -.28226330e+02  .28383484e+03 -.22528922e+04  .10723887e+05  .34910789e-02", 
" -.28044060e+02  .28070110e+03 -.22186427e+04  .10496332e+05  .34193439e-02", 
" -.27864748e+02  .27750854e+03 -.21772086e+04  .10144677e+05  .34080755e-02", 
" -.27688300e+02  .27457247e+03 -.21497158e+04  .10021639e+05  .33794648e-02", 
" -.27513990e+02  .27148824e+03 -.21077513e+04  .96041747e+04  .33077371e-02", 
" -.27342144e+02  .26880537e+03 -.20987657e+04  .98657782e+04  .32611605e-02", 
" -.27173061e+02  .26597030e+03 -.20712420e+04  .97661828e+04  .32364259e-02", 
" -.27006469e+02  .26302350e+03 -.20303035e+04  .93580262e+04  .31791440e-02", 
" -.26841249e+02  .26003929e+03 -.19893147e+04  .90286157e+04  .31768811e-02", 
" -.26678914e+02  .25742011e+03 -.19654024e+04  .89168009e+04  .31428199e-02", 
" -.26517822e+02  .25461004e+03 -.19307171e+04  .86677405e+04  .30956359e-02", 
" -.26359373e+02  .25199631e+03 -.19038550e+04  .85247524e+04  .30951586e-02", 
" -.26202592e+02  .24937435e+03 -.18769101e+04  .84196840e+04  .30927337e-02", 
" -.26047631e+02  .24682394e+03 -.18528504e+04  .83420801e+04  .30737648e-02", 
" -.25894931e+02  .24427919e+03 -.18251988e+04  .82008611e+04  .30428120e-02", 
" -.25742839e+02  .24168689e+03 -.17928271e+04  .79472632e+04  .30492660e-02", 
" -.25593688e+02  .23940044e+03 -.17770563e+04  .79730373e+04  .30038263e-02", 
" -.25446335e+02  .23705590e+03 -.17544392e+04  .78590028e+04  .29709761e-02", 
" -.25300494e+02  .23491150e+03 -.17459434e+04  .80249976e+04  .29638826e-02", 
" -.25155679e+02  .23278709e+03 -.17366482e+04  .81711989e+04  .28928876e-02", 
" -.25012158e+02  .23055904e+03 -.17171098e+04  .81076847e+04  .28593245e-02", 
" -.24869933e+02  .22836508e+03 -.16981559e+04  .80375331e+04  .28326630e-02", 
" -.24729396e+02  .22625754e+03 -.16824852e+04  .80158218e+04  .28321391e-02", 
" -.24590454e+02  .22411705e+03 -.16627845e+04  .79347969e+04  .28355902e-02", 
" -.24451994e+02  .22196362e+03 -.16426646e+04  .78439801e+04  .28371872e-02", 
" -.24314807e+02  .21978757e+03 -.16202991e+04  .77214216e+04  .28206584e-02", 
" -.24178801e+02  .21765276e+03 -.15965310e+04  .75359256e+04  .27987014e-02", 
" -.24044383e+02  .21556671e+03 -.15760382e+04  .74399431e+04  .28056603e-02", 
" -.23910718e+02  .21345958e+03 -.15521486e+04  .72762558e+04  .27957261e-02", 
" -.23778425e+02  .21140920e+03 -.15294822e+04  .71246334e+04  .27685501e-02", 
" -.23646722e+02  .20928189e+03 -.14990511e+04  .67907802e+04  .27655914e-02", 
" -.23516805e+02  .20734800e+03 -.14796928e+04  .66641724e+04  .27732291e-02", 
" -.23387319e+02  .20526699e+03 -.14480736e+04  .62802543e+04  .27388457e-02", 
" -.23258952e+02  .20339841e+03 -.14347588e+04  .62822040e+04  .27212968e-02", 
" -.23131269e+02  .20162809e+03 -.14265132e+04  .63799470e+04  .27171549e-02", 
" -.23004401e+02  .19975269e+03 -.14072209e+04  .62317571e+04  .27050777e-02", 
" -.22877774e+02  .19780621e+03 -.13858505e+04  .60833175e+04  .26891598e-02", 
" -.22752233e+02  .19580302e+03 -.13579643e+04  .58089519e+04  .27172010e-02", 
" -.22627820e+02  .19405101e+03 -.13472510e+04  .58676457e+04  .27212590e-02", 
" -.22504359e+02  .19228819e+03 -.13341646e+04  .58629185e+04  .27380907e-02", 
" -.22381756e+02  .19050833e+03 -.13195580e+04  .58469610e+04  .26868053e-02", 
" -.22258885e+02  .18861584e+03 -.12941161e+04  .55812170e+04  .27292236e-02", 
" -.22137734e+02  .18700625e+03 -.12859268e+04  .56184552e+04  .26657403e-02", 
" -.22016926e+02  .18541355e+03 -.12796750e+04  .56981782e+04  .26513708e-02", 
" -.21896610e+02  .18364662e+03 -.12622492e+04  .56099732e+04  .26333344e-02", 
" -.21776566e+02  .18198439e+03 -.12534451e+04  .56945879e+04  .26259688e-02", 
" -.21657451e+02  .18030503e+03 -.12386096e+04  .56284230e+04  .26181042e-02", 
" -.21538598e+02  .17855508e+03 -.12200501e+04  .55316077e+04  .26257400e-02", 
" -.21420570e+02  .17683519e+03 -.11992196e+04  .53337636e+04  .25839903e-02", 
" -.21303042e+02  .17515044e+03 -.11840450e+04  .52888265e+04  .25784614e-02", 
" -.21185510e+02  .17343801e+03 -.11641532e+04  .51248867e+04  .25776008e-02", 
" -.21069078e+02  .17195612e+03 -.11606219e+04  .52791674e+04  .25743315e-02", 
" -.20952574e+02  .17026929e+03 -.11434547e+04  .51978530e+04  .25534969e-02", 
" -.20837493e+02  .16894661e+03 -.11472644e+04  .54409504e+04  .25721291e-02", 
" -.20722619e+02  .16744783e+03 -.11366215e+04  .54049954e+04  .25694248e-02", 
" -.20607838e+02  .16590271e+03 -.11241182e+04  .53557118e+04  .25986821e-02", 
" -.20494073e+02  .16456578e+03 -.11275405e+04  .56369715e+04  .26006819e-02", 
" -.20380149e+02  .16295785e+03 -.11107796e+04  .55387280e+04  .26355169e-02", 
" -.20266554e+02  .16144447e+03 -.10974555e+04  .54590050e+04  .26310835e-02", 
" -.20153149e+02  .15989438e+03 -.10837160e+04  .54079917e+04  .26100330e-02", 
" -.20039944e+02  .15833747e+03 -.10656203e+04  .52246424e+04  .26093653e-02", 
" -.19927132e+02  .15673071e+03 -.10441289e+04  .49975356e+04  .26333661e-02", 
" -.19815128e+02  .15538097e+03 -.10430073e+04  .51738914e+04  .26090861e-02", 
" -.19702501e+02  .15376581e+03 -.10224678e+04  .49795196e+04  .26094064e-02", 
" -.19590703e+02  .15239848e+03 -.10162551e+04  .50117034e+04  .25840831e-02", 
" -.19478456e+02  .15085797e+03 -.10025083e+04  .49626786e+04  .25788015e-02", 
" -.19366260e+02  .14922461e+03 -.98094389e+03  .47739405e+04  .25730763e-02", 
" -.19255988e+02  .14806053e+03 -.98436678e+03  .49648223e+04  .25647231e-02", 
" -.19144501e+02  .14646696e+03 -.96109221e+03  .47036001e+04  .25069282e-02", 
" -.19033189e+02  .14499482e+03 -.94747467e+03  .46203601e+04  .24938239e-02", 
" -.18922393e+02  .14363561e+03 -.94032881e+03  .46509475e+04  .24983458e-02", 
" -.18811112e+02  .14212501e+03 -.92275326e+03  .44915801e+04  .25123566e-02", 
" -.18700821e+02  .14080398e+03 -.91576962e+03  .45044835e+04  .24814409e-02", 
" -.18590642e+02  .13947586e+03 -.90875567e+03  .45434606e+04  .24761171e-02", 
" -.18480097e+02  .13802119e+03 -.89182639e+03  .43679462e+04  .24578172e-02", 
" -.18369124e+02  .13657295e+03 -.87689994e+03  .42503070e+04  .24264264e-02", 
" -.18258787e+02  .13524532e+03 -.86797387e+03  .42155795e+04  .24395518e-02", 
" -.18148742e+02  .13396057e+03 -.86504762e+03  .43531271e+04  .24486092e-02", 
" -.18038413e+02  .13268943e+03 -.86186939e+03  .44618641e+04  .24277621e-02", 
" -.17928023e+02  .13138629e+03 -.85675846e+03  .45396781e+04  .24386592e-02", 
" -.17817307e+02  .12988345e+03 -.83578697e+03  .43097939e+04  .24613621e-02", 
" -.17706714e+02  .12858189e+03 -.83032497e+03  .43808799e+04  .24263527e-02", 
" -.17596132e+02  .12728422e+03 -.82319840e+03  .43903396e+04  .24172633e-02", 
" -.17485268e+02  .12588365e+03 -.81005537e+03  .43107833e+04  .24048501e-02", 
" -.17374468e+02  .12461654e+03 -.80671261e+03  .44312396e+04  .23818256e-02", 
" -.17263055e+02  .12313678e+03 -.78758965e+03  .42403175e+04  .23466545e-02", 
" -.17151942e+02  .12179527e+03 -.77899540e+03  .42728573e+04  .23548840e-02", 
" -.17040598e+02  .12058316e+03 -.77834546e+03  .44240975e+04  .23660702e-02", 
" -.16929091e+02  .11931131e+03 -.77228505e+03  .44625313e+04  .23617776e-02", 
" -.16817413e+02  .11793498e+03 -.75636112e+03  .42878674e+04  .23734189e-02", 
" -.16705123e+02  .11651166e+03 -.73772454e+03  .40708513e+04  .23370110e-02", 
" -.16592743e+02  .11517200e+03 -.72581860e+03  .39870116e+04  .23351838e-02", 
" -.16480718e+02  .11386460e+03 -.71223374e+03  .38296120e+04  .23120073e-02", 
" -.16368202e+02  .11261542e+03 -.70558887e+03  .38402437e+04  .22838885e-02", 
" -.16254937e+02  .11133669e+03 -.69919355e+03  .38701807e+04  .22871910e-02", 
" -.16141433e+02  .10997464e+03 -.68302136e+03  .36760348e+04  .22838030e-02", 
" -.16027429e+02  .10860880e+03 -.66779852e+03  .35221465e+04  .22670389e-02", 
" -.15912631e+02  .10715350e+03 -.65158272e+03  .34186133e+04  .22649717e-02", 
" -.15797398e+02  .10561435e+03 -.62794814e+03  .31704790e+04  .22448373e-02", 
" -.15681750e+02  .10412056e+03 -.60453167e+03  .28881679e+04  .22276188e-02", 
" -.15566191e+02  .10286585e+03 -.59683091e+03  .28618719e+04  .22228527e-02", 
" -.15449934e+02  .10164418e+03 -.59384726e+03  .29532797e+04  .22329345e-02", 
" -.15333156e+02  .10038699e+03 -.58559176e+03  .29304469e+04  .22274521e-02", 
" -.15215029e+02  .98893076e+02 -.56177358e+03  .26165986e+04  .22367628e-02", 
" -.15097038e+02  .97411576e+02 -.53613536e+03  .22459669e+04  .22014713e-02", 
" -.14978203e+02  .96039693e+02 -.52267449e+03  .21538413e+04  .22272412e-02", 
" -.14859307e+02  .94884552e+02 -.52058460e+03  .22164458e+04  .22439659e-02", 
" -.14739581e+02  .93671013e+02 -.51385584e+03  .21895106e+04  .22606702e-02", 
" -.14618844e+02  .92585446e+02 -.51968923e+03  .24259323e+04  .22537841e-02", 
" -.14496904e+02  .91279629e+02 -.50808359e+03  .22956327e+04  .22626122e-02", 
" -.14374167e+02  .90137572e+02 -.51199291e+03  .25259525e+04  .22842050e-02", 
" -.14249650e+02  .88714814e+02 -.49738842e+03  .24046690e+04  .22496591e-02", 
" -.14125521e+02  .87701277e+02 -.50977250e+03  .27805215e+04  .22649171e-02", 
" -.13999136e+02  .86202147e+02 -.48931431e+03  .25613177e+04  .22691492e-02", 
" -.13872056e+02  .84761867e+02 -.47370157e+03  .24448875e+04  .22650208e-02", 
" -.13743320e+02  .83296986e+02 -.45503567e+03  .22274919e+04  .22653348e-02", 
" -.13613571e+02  .81875734e+02 -.44027083e+03  .20994959e+04  .23066861e-02", 
" -.13482819e+02  .80523932e+02 -.43232826e+03  .21313694e+04  .22885597e-02", 
" -.13350768e+02  .79212321e+02 -.42465605e+03  .21166797e+04  .22640416e-02", 
" -.13216804e+02  .77850092e+02 -.41271616e+03  .19920583e+04  .22829753e-02", 
" -.13081443e+02  .76438511e+02 -.39834150e+03  .18414952e+04  .22858381e-02", 
" -.12944131e+02  .74929361e+02 -.37476792e+03  .14724297e+04  .22776210e-02", 
" -.12805271e+02  .73605932e+02 -.36943050e+03  .15257541e+04  .22809051e-02", 
" -.12664426e+02  .72245675e+02 -.36453656e+03  .16071786e+04  .23288035e-02", 
" -.12521152e+02  .70839206e+02 -.35579060e+03  .15706577e+04  .23284362e-02", 
" -.12375681e+02  .69286629e+02 -.33239074e+03  .12097410e+04  .23809572e-02", 
" -.12228257e+02  .67946359e+02 -.32940051e+03  .12948934e+04  .24251511e-02", 
" -.12078735e+02  .66677408e+02 -.32755336e+03  .13363329e+04  .24504512e-02", 
" -.11926012e+02  .65367895e+02 -.32764519e+03  .14710275e+04  .24696495e-02", 
" -.11770122e+02  .63772044e+02 -.30815241e+03  .12476717e+04  .24533461e-02", 
" -.11610491e+02  .62133775e+02 -.29154885e+03  .11147887e+04  .24706966e-02", 
" -.11448031e+02  .60564641e+02 -.27795056e+03  .10365501e+04  .25021423e-02", 
" -.11283177e+02  .59385740e+02 -.29175083e+03  .14258147e+04  .25108866e-02", 
" -.11113345e+02  .58012861e+02 -.29479168e+03  .16321504e+04  .25283867e-02", 
" -.10938375e+02  .56451269e+02 -.28936526e+03  .17098692e+04  .25451694e-02", 
" -.10758339e+02  .54738593e+02 -.27795674e+03  .17146420e+04  .25826321e-02", 
" -.10574621e+02  .53200951e+02 -.26787384e+03  .15933603e+04  .25818002e-02", 
" -.10383523e+02  .51376720e+02 -.24683399e+03  .13742743e+04  .26483089e-02", 
" -.10186465e+02  .49742753e+02 -.24549928e+03  .15565947e+04  .26075166e-02", 
" -.99823649e+01  .47997251e+02 -.22781509e+03  .12919027e+04  .26315334e-02", 
" -.97684989e+01  .45856913e+02 -.19426688e+03  .78878906e+03  .26508346e-02", 
" -.95472074e+01  .44264965e+02 -.20672608e+03  .12451647e+04  .26336898e-02", 
" -.93133676e+01  .41980574e+02 -.17228011e+03  .78530340e+03  .26812967e-02", 
" -.90683291e+01  .40001419e+02 -.16480306e+03  .81762517e+03  .26654414e-02", 
" -.88075660e+01  .37923275e+02 -.15798080e+03  .86535364e+03  .27058484e-02", 
" -.85303792e+01  .35703688e+02 -.14323364e+03  .75283716e+03  .27346279e-02", 
" -.82324122e+01  .33217749e+02 -.11731666e+03  .43059053e+03  .27336552e-02", 
" -.79072818e+01  .30471975e+02 -.88150380e+02  .75609070e+02  .28264315e-02", 
" -.75505706e+01  .27514343e+02 -.50656014e+02 -.47157481e+03  .28456709e-02", 
" -.71514428e+01  .25210758e+02 -.92168847e+02  .70852602e+03  .29487255e-02", 
" -.66854153e+01  .21276947e+02 -.42242200e+02  .76298540e+02  .31631750e-02", 
" -.61282267e+01  .16567174e+02  .34176985e+02 -.10582308e+04  .35318770e-02", 
" -.54079451e+01  .10949670e+02  .11080915e+03 -.21114413e+04  .40152404e-02", 
" -.52294487e+01  .92674239e+01  .14705371e+03 -.26595190e+04  .41199046e-02", 
" -.50354300e+01  .76015469e+01  .17717205e+03 -.30548652e+04  .42335991e-02", 
" -.48257565e+01  .71467886e+01  .12601443e+03 -.20425677e+04  .42447085e-02", 
" -.45875339e+01  .60145241e+01  .10826527e+03 -.17166759e+04  .44389762e-02", 
" -.43158069e+01  .49812709e+01  .66937261e+02 -.73378103e+03  .45596948e-02", 
" -.39901852e+01  .25681356e+01  .87517933e+02 -.84399789e+03  .49723594e-02", 
" -.35917618e+01  .97842303e+00  .38982183e+02  .21945959e+03  .56140647e-02", 
" -.30515791e+01 -.23732550e+01  .28072788e+02  .95347320e+03  .65166367e-02", 
" -.21935379e+01 -.53791746e+01 -.34939118e+02  .17715087e+04  .92100846e-02", 
" -.13871782e+01 -.99079224e+01  .33343177e+02  .90279005e+03  .12071969e-01", 
" -.38172198e+00 -.18662217e+02  .46799799e+03 -.75685550e+04  .17134279e-01", 
"  .28421251e+00 -.11777641e+02 -.12769021e+03  .52052034e+04  .22189997e-01", 
"coac      5  7  3   30", " -.79771753e+02  .18030864e+04 -.27697497e+05  .21799124e+06  .11193312e+00", 
" -.76042166e+02  .16330549e+04 -.23360825e+05  .16814537e+06  .78090333e-01", 
" -.71057663e+02  .14416399e+04 -.19799738e+05  .13997694e+06  .53066348e-01", 
" -.67277494e+02  .13045577e+04 -.17185815e+05  .11607182e+06  .39661027e-01", 
" -.63319085e+02  .11641440e+04 -.14666030e+05  .96447423e+05  .27124875e-01", 
" -.60991709e+02  .10877418e+04 -.13419854e+05  .87452161e+05  .21888527e-01", 
" -.59312138e+02  .10335964e+04 -.12452239e+05  .78407027e+05  .19285571e-01", 
" -.58001611e+02  .99308405e+03 -.11854745e+05  .75086673e+05  .17466297e-01", 
" -.56909136e+02  .96061750e+03 -.11465157e+05  .74332589e+05  .16300948e-01", 
" -.55980208e+02  .93201724e+03 -.11009574e+05  .71031315e+05  .15395195e-01", 
" -.55175173e+02  .90870681e+03 -.10677932e+05  .68926394e+05  .14502394e-01", 
" -.54456992e+02  .88804901e+03 -.10405072e+05  .67630533e+05  .13944613e-01", 
" -.53817511e+02  .87125213e+03 -.10249830e+05  .67790451e+05  .13042836e-01", 
" -.51300171e+02  .79973467e+03 -.91956843e+04  .60496888e+05  .10639177e-01", 
" -.49461398e+02  .74513832e+03 -.81981579e+04  .50945910e+05  .93825063e-02", 
" -.48022175e+02  .70718814e+03 -.77050673e+04  .48481950e+05  .87864853e-02", 
" -.46831692e+02  .67770506e+03 -.73888418e+04  .47797128e+05  .82049455e-02", 
" -.45807170e+02  .65186215e+03 -.70447010e+04  .45502586e+05  .77389893e-02", 
" -.44907590e+02  .62815669e+03 -.66391147e+04  .41448159e+05  .70187208e-02", 
" -.44099805e+02  .60611437e+03 -.62235728e+04  .37065982e+05  .67109701e-02", 
" -.43376348e+02  .58844208e+03 -.59817677e+04  .35342568e+05  .63109156e-02", 
" -.42716254e+02  .57319186e+03 -.58177386e+04  .34824651e+05  .61827777e-02", 
" -.42106191e+02  .55904105e+03 -.56608128e+04  .34310975e+05  .60282396e-02", 
" -.41538548e+02  .54596372e+03 -.55103647e+04  .33642427e+05  .57556867e-02", 
" -.41007636e+02  .53294981e+03 -.52935011e+04  .31418853e+05  .57098545e-02", 
" -.40509734e+02  .52158877e+03 -.51567401e+04  .30696849e+05  .55247701e-02", 
" -.40039052e+02  .51069056e+03 -.50168833e+04  .29876216e+05  .53776115e-02", 
" -.39592597e+02  .49987055e+03 -.48440347e+04  .28337794e+05  .52859540e-02", 
" -.39168525e+02  .49001383e+03 -.47061186e+04  .27318979e+05  .50956852e-02", 
" -.38765197e+02  .48085552e+03 -.45798991e+04  .26305396e+05  .50026543e-02", 
" -.38379630e+02  .47226755e+03 -.44684175e+04  .25520400e+05  .49510123e-02", 
" -.38009748e+02  .46393257e+03 -.43517077e+04  .24604235e+05  .49086714e-02", 
" -.37654826e+02  .45632087e+03 -.42619078e+04  .24041604e+05  .49293399e-02", 
" -.37316802e+02  .44910401e+03 -.41739686e+04  .23517231e+05  .48722721e-02", 
" -.36986234e+02  .44146375e+03 -.40499150e+04  .22321207e+05  .49059437e-02", 
" -.36668406e+02  .43465610e+03 -.39667464e+04  .21819413e+05  .48057236e-02", 
" -.36360517e+02  .42811707e+03 -.38871109e+04  .21326775e+05  .46776687e-02", 
" -.36063575e+02  .42184186e+03 -.38042884e+04  .20652483e+05  .45478429e-02", 
" -.35775068e+02  .41597122e+03 -.37442398e+04  .20439981e+05  .45357524e-02", 
" -.35494216e+02  .40970535e+03 -.36410234e+04  .19373263e+05  .44488434e-02", 
" -.35221323e+02  .40385566e+03 -.35609223e+04  .18751430e+05  .43888635e-02", 
" -.34956282e+02  .39842419e+03 -.34982734e+04  .18423454e+05  .43172362e-02", 
" -.34698768e+02  .39334935e+03 -.34460164e+04  .18187597e+05  .42437316e-02", 
" -.34447188e+02  .38800281e+03 -.33643355e+04  .17371862e+05  .42132007e-02", 
" -.34201289e+02  .38315830e+03 -.33208182e+04  .17363570e+05  .41270594e-02", 
" -.33961356e+02  .37821427e+03 -.32574135e+04  .16944619e+05  .40305833e-02", 
" -.33726744e+02  .37365792e+03 -.32112356e+04  .16749273e+05  .40255793e-02", 
" -.33497009e+02  .36883088e+03 -.31357729e+04  .15996192e+05  .39341483e-02", 
" -.33272220e+02  .36430634e+03 -.30800440e+04  .15647799e+05  .38846687e-02", 
" -.33053562e+02  .36039356e+03 -.30557661e+04  .15797677e+05  .38604723e-02", 
" -.32837579e+02  .35618209e+03 -.30107489e+04  .15620626e+05  .38073452e-02", 
" -.32625700e+02  .35205182e+03 -.29583487e+04  .15166930e+05  .37496251e-02", 
" -.32418926e+02  .34826206e+03 -.29278609e+04  .15194800e+05  .36992081e-02", 
" -.32214762e+02  .34437196e+03 -.28876041e+04  .15039684e+05  .36316400e-02", 
" -.32014181e+02  .34043517e+03 -.28344749e+04  .14560167e+05  .36275956e-02", 
" -.31816315e+02  .33655902e+03 -.27852055e+04  .14175583e+05  .35717110e-02", 
" -.31622565e+02  .33294132e+03 -.27453870e+04  .13936667e+05  .35610566e-02", 
" -.31431890e+02  .32944535e+03 -.27133432e+04  .13834764e+05  .35606626e-02", 
" -.31242975e+02  .32579637e+03 -.26685066e+04  .13523762e+05  .35282950e-02", 
" -.31056661e+02  .32202649e+03 -.26117488e+04  .13007706e+05  .34976568e-02", 
" -.30874149e+02  .31876926e+03 -.25840626e+04  .12957966e+05  .35017161e-02", 
" -.30693879e+02  .31508209e+03 -.25217556e+04  .12290157e+05  .34595922e-02", 
" -.30516842e+02  .31186033e+03 -.24922945e+04  .12249526e+05  .34369870e-02", 
" -.30342136e+02  .30863893e+03 -.24518506e+04  .11914486e+05  .34014829e-02", 
" -.30170011e+02  .30547753e+03 -.24146150e+04  .11649631e+05  .33663606e-02", 
" -.29999813e+02  .30243224e+03 -.23835150e+04  .11502932e+05  .33719030e-02", 
" -.29832703e+02  .29963771e+03 -.23629614e+04  .11475331e+05  .33526535e-02", 
" -.29667283e+02  .29682534e+03 -.23424276e+04  .11509490e+05  .33043792e-02", 
" -.29502716e+02  .29383375e+03 -.23079760e+04  .11280711e+05  .33135194e-02", 
" -.29340922e+02  .29103213e+03 -.22825191e+04  .11204127e+05  .33274130e-02", 
" -.29180332e+02  .28804707e+03 -.22406930e+04  .10803191e+05  .33318446e-02", 
" -.29023075e+02  .28553213e+03 -.22249025e+04  .10841271e+05  .32686494e-02", 
" -.28866219e+02  .28280372e+03 -.21996238e+04  .10771379e+05  .32408262e-02", 
" -.28711596e+02  .28026960e+03 -.21796622e+04  .10713845e+05  .32071254e-02", 
" -.28558791e+02  .27791928e+03 -.21736413e+04  .10970525e+05  .31924859e-02", 
" -.28405236e+02  .27478700e+03 -.21139271e+04  .10276039e+05  .31898259e-02", 
" -.28253701e+02  .27218592e+03 -.20901900e+04  .10221185e+05  .31335222e-02", 
" -.28104210e+02  .26957825e+03 -.20612276e+04  .10041312e+05  .30773264e-02", 
" -.27956112e+02  .26706588e+03 -.20368126e+04  .99360567e+04  .30065191e-02", 
" -.27809205e+02  .26447638e+03 -.20081882e+04  .97906481e+04  .29510867e-02", 
" -.27664813e+02  .26223093e+03 -.19978748e+04  .99366701e+04  .28943094e-02", 
" -.27520839e+02  .25973762e+03 -.19711757e+04  .98199314e+04  .29389035e-02", 
" -.27378533e+02  .25762894e+03 -.19684754e+04  .10107535e+05  .28917700e-02", 
" -.27236711e+02  .25506041e+03 -.19289014e+04  .96893916e+04  .28821196e-02", 
" -.27095847e+02  .25256517e+03 -.18944905e+04  .93728304e+04  .28919615e-02", 
" -.26956630e+02  .25033252e+03 -.18770676e+04  .93598353e+04  .28713232e-02", 
" -.26818610e+02  .24816273e+03 -.18614427e+04  .93687730e+04  .28489742e-02", 
" -.26681419e+02  .24587839e+03 -.18353904e+04  .91595442e+04  .28687760e-02", 
" -.26545143e+02  .24362022e+03 -.18107329e+04  .89828331e+04  .29113739e-02", 
" -.26409696e+02  .24132577e+03 -.17834119e+04  .87908141e+04  .28866147e-02", 
" -.26274483e+02  .23897812e+03 -.17520559e+04  .85063819e+04  .29003251e-02", 
" -.26141248e+02  .23685082e+03 -.17305221e+04  .83559629e+04  .28763726e-02", 
" -.26008065e+02  .23470064e+03 -.17070893e+04  .81627560e+04  .29084998e-02", 
" -.25875873e+02  .23255236e+03 -.16858282e+04  .80727995e+04  .28774594e-02", 
" -.25744743e+02  .23039050e+03 -.16591959e+04  .78505733e+04  .28466094e-02", 
" -.25613749e+02  .22820657e+03 -.16318211e+04  .76120277e+04  .28405641e-02", 
" -.25484653e+02  .22628359e+03 -.16193980e+04  .76386765e+04  .28556581e-02", 
" -.25355603e+02  .22422231e+03 -.15981242e+04  .75128499e+04  .28560144e-02", 
" -.25227262e+02  .22220556e+03 -.15780361e+04  .73939104e+04  .28363082e-02", 
" -.25100005e+02  .22034475e+03 -.15677611e+04  .74471861e+04  .28208012e-02", 
" -.24973456e+02  .21852744e+03 -.15609725e+04  .75876331e+04  .28140941e-02", 
" -.24846523e+02  .21641095e+03 -.15327984e+04  .73255928e+04  .27863469e-02", 
" -.24721318e+02  .21468615e+03 -.15309293e+04  .75507571e+04  .27478668e-02", 
" -.24596110e+02  .21275905e+03 -.15126823e+04  .74661018e+04  .27525352e-02", 
" -.24471692e+02  .21091765e+03 -.14990406e+04  .74559817e+04  .27574838e-02", 
" -.24347657e+02  .20896363e+03 -.14771619e+04  .73004900e+04  .27762557e-02", 
" -.24223893e+02  .20719086e+03 -.14691241e+04  .73972733e+04  .27223247e-02", 
" -.24100225e+02  .20522196e+03 -.14447010e+04  .71819025e+04  .27221671e-02", 
" -.23978020e+02  .20344293e+03 -.14298870e+04  .71217036e+04  .27371372e-02", 
" -.23856097e+02  .20173132e+03 -.14230886e+04  .72444307e+04  .27243922e-02", 
" -.23734714e+02  .20001349e+03 -.14119270e+04  .72493756e+04  .27600415e-02", 
" -.23613118e+02  .19826247e+03 -.14020453e+04  .73296190e+04  .27388372e-02", 
" -.23492742e+02  .19669566e+03 -.13995154e+04  .74838255e+04  .27315039e-02", 
" -.23372702e+02  .19511524e+03 -.13985436e+04  .77235199e+04  .27541325e-02", 
" -.23252371e+02  .19336924e+03 -.13847954e+04  .76938852e+04  .27556531e-02", 
" -.23132032e+02  .19156155e+03 -.13668153e+04  .75973984e+04  .27712368e-02", 
" -.23012393e+02  .18966623e+03 -.13383432e+04  .72768166e+04  .27515045e-02", 
" -.22892656e+02  .18783642e+03 -.13151822e+04  .70473600e+04  .27449295e-02", 
" -.22773350e+02  .18605210e+03 -.12956725e+04  .68959894e+04  .27625418e-02", 
" -.22654375e+02  .18415408e+03 -.12665365e+04  .65661262e+04  .27779416e-02", 
" -.22535315e+02  .18232457e+03 -.12432593e+04  .63545125e+04  .27402404e-02", 
" -.22417205e+02  .18082149e+03 -.12416958e+04  .65196622e+04  .27760508e-02", 
" -.22299128e+02  .17921969e+03 -.12293142e+04  .64342241e+04  .27354281e-02", 
" -.22180717e+02  .17734543e+03 -.11982469e+04  .60371818e+04  .27103035e-02", 
" -.22062725e+02  .17563769e+03 -.11797833e+04  .58807342e+04  .26922955e-02", 
" -.21944744e+02  .17391337e+03 -.11595318e+04  .56921366e+04  .27139143e-02", 
" -.21826785e+02  .17210290e+03 -.11322717e+04  .53740406e+04  .27272087e-02", 
" -.21709705e+02  .17058730e+03 -.11280106e+04  .55166837e+04  .27243321e-02", 
" -.21591854e+02  .16883852e+03 -.11073130e+04  .53563258e+04  .27218712e-02", 
" -.21474170e+02  .16716307e+03 -.10903486e+04  .52421040e+04  .27050558e-02", 
" -.21357180e+02  .16563604e+03 -.10813296e+04  .52623992e+04  .27144300e-02", 
" -.21239449e+02  .16397992e+03 -.10669588e+04  .52160595e+04  .26828705e-02", 
" -.21121710e+02  .16221127e+03 -.10431026e+04  .49771733e+04  .26479540e-02", 
" -.21004639e+02  .16075713e+03 -.10407305e+04  .51324036e+04  .26765137e-02", 
" -.20887690e+02  .15926894e+03 -.10333770e+04  .51553581e+04  .26832521e-02", 
" -.20769701e+02  .15754458e+03 -.10118836e+04  .49592512e+04  .26576142e-02", 
" -.20652604e+02  .15603206e+03 -.10022754e+04  .49530691e+04  .26822499e-02", 
" -.20534841e+02  .15439861e+03 -.98428704e+03  .47985730e+04  .27186942e-02", 
" -.20417306e+02  .15288823e+03 -.97282183e+03  .47363932e+04  .27099216e-02", 
" -.20299437e+02  .15137694e+03 -.96508788e+03  .47884333e+04  .26788458e-02", 
" -.20181541e+02  .14973704e+03 -.94647166e+03  .46374346e+04  .26918767e-02", 
" -.20063710e+02  .14833113e+03 -.94527939e+03  .48239860e+04  .27048266e-02", 
" -.19945920e+02  .14680056e+03 -.93174829e+03  .47426608e+04  .26874602e-02", 
" -.19827006e+02  .14522565e+03 -.91897895e+03  .47072767e+04  .26712203e-02", 
" -.19707670e+02  .14351110e+03 -.89568408e+03  .44658505e+04  .26765204e-02", 
" -.19588143e+02  .14195170e+03 -.88454311e+03  .44566817e+04  .26536964e-02", 
" -.19469154e+02  .14040476e+03 -.87111810e+03  .43837567e+04  .26385026e-02", 
" -.19349481e+02  .13881387e+03 -.85695188e+03  .43438265e+04  .26445782e-02", 
" -.19228892e+02  .13711284e+03 -.83450508e+03  .41217376e+04  .26543596e-02", 
" -.19109050e+02  .13567341e+03 -.82879084e+03  .41925972e+04  .26036101e-02", 
" -.18987916e+02  .13403879e+03 -.80976128e+03  .40147059e+04  .26154895e-02", 
" -.18866848e+02  .13249106e+03 -.79419075e+03  .38720691e+04  .25957307e-02", 
" -.18745441e+02  .13106556e+03 -.79186095e+03  .40449593e+04  .25969649e-02", 
" -.18622634e+02  .12944581e+03 -.77416567e+03  .38923531e+04  .26439921e-02", 
" -.18500126e+02  .12798601e+03 -.76600687e+03  .38979255e+04  .26583140e-02", 
" -.18377649e+02  .12668844e+03 -.76551327e+03  .40057798e+04  .26610162e-02", 
" -.18253863e+02  .12516916e+03 -.75262053e+03  .39050379e+04  .26192848e-02", 
" -.18129406e+02  .12358909e+03 -.73548213e+03  .37393334e+04  .26100145e-02", 
" -.18004110e+02  .12209247e+03 -.72801669e+03  .38054108e+04  .26253773e-02", 
" -.17877880e+02  .12033188e+03 -.69855793e+03  .34170926e+04  .25958175e-02", 
" -.17751229e+02  .11871431e+03 -.67898391e+03  .32129092e+04  .26200952e-02", 
" -.17623811e+02  .11702815e+03 -.65652804e+03  .29736929e+04  .26091628e-02", 
" -.17494884e+02  .11521930e+03 -.62593292e+03  .25633793e+04  .26127302e-02", 
" -.17365124e+02  .11345380e+03 -.59745833e+03  .21940913e+04  .26131763e-02", 
" -.17234893e+02  .11186241e+03 -.58218202e+03  .20751904e+04  .26011938e-02", 
" -.17104239e+02  .11044347e+03 -.57779934e+03  .21250558e+04  .26042408e-02", 
" -.16972080e+02  .10880725e+03 -.55832173e+03  .19136612e+04  .26201012e-02", 
" -.16838555e+02  .10707425e+03 -.53486294e+03  .16635866e+04  .26236400e-02", 
" -.16704974e+02  .10571298e+03 -.53682072e+03  .18886661e+04  .26016308e-02", 
" -.16570124e+02  .10433184e+03 -.53653315e+03  .20397168e+04  .25772244e-02", 
" -.16433127e+02  .10268586e+03 -.51781118e+03  .18345800e+04  .25796699e-02", 
" -.16295000e+02  .10110392e+03 -.51149221e+03  .19664741e+04  .25852514e-02", 
" -.16156010e+02  .99667172e+02 -.50690289e+03  .19957511e+04  .25736948e-02", 
" -.16015096e+02  .97945225e+02 -.48049079e+03  .16270951e+04  .25274081e-02", 
" -.15873024e+02  .96483835e+02 -.47520694e+03  .16526612e+04  .25580237e-02", 
" -.15728515e+02  .94742394e+02 -.45190218e+03  .13710921e+04  .25776121e-02", 
" -.15582619e+02  .93112918e+02 -.43934044e+03  .13255450e+04  .25483787e-02", 
" -.15434890e+02  .91460049e+02 -.42076702e+03  .10955301e+04  .25185703e-02", 
" -.15285086e+02  .89827765e+02 -.40840157e+03  .10438779e+04  .25450071e-02", 
" -.15133092e+02  .88128420e+02 -.39268450e+03  .91608775e+03  .25519107e-02", 
" -.14978939e+02  .86651939e+02 -.39263117e+03  .10627722e+04  .25753284e-02", 
" -.14821970e+02  .85006858e+02 -.38578570e+03  .11479415e+04  .25712721e-02", 
" -.14661678e+02  .83108622e+02 -.35981023e+03  .82630398e+03  .25539352e-02", 
" -.14498349e+02  .81246163e+02 -.33871610e+03  .61405933e+03  .25588566e-02", 
" -.14332809e+02  .79564500e+02 -.32992566e+03  .62236799e+03  .26023953e-02", 
" -.14163475e+02  .77658284e+02 -.31022744e+03  .46047453e+03  .25797545e-02", 
" -.13990969e+02  .75982476e+02 -.30581736e+03  .53900584e+03  .25636642e-02", 
" -.13814675e+02  .74099800e+02 -.28662460e+03  .34082460e+03  .25274295e-02", 
" -.13633759e+02  .72090405e+02 -.26004643e+03  .28206500e+02  .25641256e-02", 
" -.13449771e+02  .70170357e+02 -.23346633e+03 -.37730302e+03  .25376661e-02", 
" -.13259247e+02  .68124795e+02 -.20856380e+03 -.69593423e+03  .25234284e-02", 
" -.13065133e+02  .66439705e+02 -.21243274e+03 -.42036310e+03  .25020418e-02", 
" -.12863883e+02  .64305889e+02 -.18824547e+03 -.65970300e+03  .25442051e-02", 
" -.12656268e+02  .62363475e+02 -.17746492e+03 -.71802709e+03  .25437524e-02", 
" -.12441485e+02  .60566005e+02 -.18709740e+03 -.31609518e+03  .24781670e-02", 
" -.12218505e+02  .58725247e+02 -.19120481e+03 -.87948095e+02  .25803188e-02", 
" -.11986803e+02  .56841435e+02 -.19976904e+03  .25968664e+03  .26263006e-02", 
" -.11743787e+02  .54474804e+02 -.17775226e+03  .84148319e+02  .26258506e-02", 
" -.11489381e+02  .52024257e+02 -.15596209e+03 -.88653566e+02  .27238547e-02", 
" -.11220851e+02  .49589777e+02 -.13990295e+03 -.21054221e+03  .27111781e-02", 
" -.10937741e+02  .47628610e+02 -.16134752e+03  .35064733e+03  .28184540e-02", 
" -.10633645e+02  .44851113e+02 -.13162437e+03 -.58610497e+02  .29608916e-02", 
" -.10306878e+02  .42349515e+02 -.13868285e+03  .35588746e+03  .30829264e-02", 
" -.99502214e+01  .39086367e+02 -.10895703e+03  .16361780e+03  .31796941e-02", 
" -.95567243e+01  .35853181e+02 -.88191870e+02 -.33974650e+02  .31613582e-02", 
" -.91125373e+01  .31675201e+02 -.25468173e+02 -.97860475e+03  .32792429e-02", 
" -.86009936e+01  .28070022e+02 -.39508158e+02 -.36786644e+03  .34980592e-02", 
" -.79839323e+01  .24429305e+02 -.10148429e+03  .11916919e+04  .36548334e-02", 
" -.71767148e+01  .17344235e+02 -.80349323e+01 -.47443049e+02  .41068871e-02", 
" -.69808464e+01  .16294188e+02 -.11457627e+02 -.15783080e+03  .42783818e-02", 
" -.67629397e+01  .14306431e+02  .11425007e+02 -.25191480e+03  .44666989e-02", 
" -.65276972e+01  .12937253e+02  .12723932e+01  .13514253e+03  .47257411e-02", 
" -.62584798e+01  .10321882e+02  .64047468e+02 -.86949458e+03  .51422147e-02", 
" -.59548957e+01  .87465939e+01  .21887708e+02  .35597620e+03  .54661451e-02", 
" -.55933084e+01  .55221661e+01  .93298964e+02 -.76112408e+03  .58720890e-02", 
" -.51458679e+01  .20162247e+01  .16570137e+03 -.21917339e+04  .64650685e-02", 
" -.45473184e+01 -.27313005e+01  .26810116e+03 -.42066187e+04  .72462418e-02", 
" -.35842873e+01 -.88376849e+01  .28832442e+03 -.37654417e+04  .96205415e-02", 
" -.26728491e+01 -.16992877e+02  .47460067e+03 -.60431093e+04  .12895245e-01", 
" -.15691819e+01 -.22890745e+02  .62606439e+03 -.90968877e+04  .19334752e-01", 
" -.80054550e+00 -.28004616e+02  .93958719e+03 -.15815684e+05  .24602777e-01", 
"coact     5  8  3   35", 
" -.86735293e+02  .21694034e+04 -.38669094e+05  .36941808e+06  .11078032e+00", 
" -.82946499e+02  .19934567e+04 -.34315959e+05  .32013285e+06  .81300096e-01", 
" -.77831606e+02  .17628246e+04 -.27899818e+05  .22970410e+06  .55238439e-01", 
" -.73823689e+02  .15941723e+04 -.24354351e+05  .19979362e+06  .39544657e-01", 
" -.69718647e+02  .14331611e+04 -.21153731e+05  .17273059e+06  .29430014e-01", 
" -.67268757e+02  .13364061e+04 -.18956185e+05  .14896448e+06  .25496217e-01", 
" -.65515655e+02  .12738131e+04 -.17755363e+05  .13724780e+06  .22290680e-01", 
" -.64133840e+02  .12257494e+04 -.16976787e+05  .13318055e+06  .19843820e-01", 
" -.63003012e+02  .11883337e+04 -.16423424e+05  .13056958e+06  .17556188e-01", 
" -.62026710e+02  .11544494e+04 -.15781194e+05  .12458653e+06  .17004906e-01", 
" -.61192196e+02  .11288770e+04 -.15494595e+05  .12492304e+06  .16361689e-01", 
" -.60439694e+02  .11044923e+04 -.15117975e+05  .12268377e+06  .15616802e-01", 
" -.59767840e+02  .10846472e+04 -.14913893e+05  .12324903e+06  .14552413e-01", 
" -.57121543e+02  .99552999e+03 -.13120795e+05  .10445863e+06  .12100328e-01", 
" -.55209568e+02  .93649576e+03 -.12090877e+05  .94614566e+05  .10939277e-01", 
" -.53683961e+02  .88788356e+03 -.11155757e+05  .85316773e+05  .96310563e-02", 
" -.52425930e+02  .85086174e+03 -.10530769e+05  .79485288e+05  .89462174e-02", 
" -.51346478e+02  .81860921e+03 -.99359361e+04  .73333482e+05  .84106670e-02", 
" -.50393591e+02  .78940537e+03 -.93450560e+04  .66567631e+05  .79927800e-02", 
" -.49546869e+02  .76507415e+03 -.89210199e+04  .62387159e+05  .78765297e-02", 
" -.48777878e+02  .74350553e+03 -.85785632e+04  .59505284e+05  .75300600e-02", 
" -.48079437e+02  .72500398e+03 -.83256745e+04  .57923320e+05  .70358151e-02", 
" -.47437809e+02  .70857063e+03 -.81284963e+04  .57103726e+05  .68843320e-02", 
" -.46840296e+02  .69313588e+03 -.79272214e+04  .56071246e+05  .66229484e-02", 
" -.46281031e+02  .67896709e+03 -.77691657e+04  .55878454e+05  .65755352e-02", 
" -.45755800e+02  .66557088e+03 -.75992403e+04  .55140191e+05  .64261318e-02", 
" -.45259677e+02  .65264538e+03 -.74056212e+04  .53616210e+05  .62728630e-02", 
" -.44787792e+02  .64023222e+03 -.72174418e+04  .52282258e+05  .62080772e-02", 
" -.44339413e+02  .62811268e+03 -.69907351e+04  .49746587e+05  .60901509e-02", 
" -.43912798e+02  .61691206e+03 -.68062943e+04  .48045087e+05  .60071922e-02", 
" -.43503668e+02  .60624633e+03 -.66273284e+04  .46314596e+05  .57746419e-02", 
" -.43114615e+02  .59664660e+03 -.64886436e+04  .45209572e+05  .55818390e-02", 
" -.42739081e+02  .58706803e+03 -.63300315e+04  .43605644e+05  .54798247e-02", 
" -.42376011e+02  .57746331e+03 -.61424910e+04  .41281870e+05  .54598268e-02", 
" -.42025632e+02  .56844156e+03 -.59880647e+04  .39749231e+05  .54749160e-02", 
" -.41689518e+02  .56060640e+03 -.58943315e+04  .39283486e+05  .54378490e-02", 
" -.41365036e+02  .55339240e+03 -.58475003e+04  .40010228e+05  .54451312e-02", 
" -.41048115e+02  .54548699e+03 -.57222507e+04  .38929321e+05  .53790144e-02", 
" -.40741283e+02  .53814251e+03 -.56185783e+04  .38129188e+05  .53321321e-02", 
" -.40443812e+02  .53122109e+03 -.55331448e+04  .37642449e+05  .53157774e-02", 
" -.40156012e+02  .52465672e+03 -.54500968e+04  .37119002e+05  .51895749e-02", 
" -.39874426e+02  .51806310e+03 -.53590652e+04  .36429151e+05  .51100428e-02", 
" -.39599571e+02  .51153394e+03 -.52557896e+04  .35398059e+05  .50466163e-02", 
" -.39332295e+02  .50527664e+03 -.51587715e+04  .34439442e+05  .48687015e-02", 
" -.39070852e+02  .49911594e+03 -.50644504e+04  .33581533e+05  .48060204e-02", 
" -.38815167e+02  .49312263e+03 -.49721496e+04  .32748387e+05  .46742917e-02", 
" -.38565425e+02  .48755082e+03 -.49045283e+04  .32367315e+05  .46164701e-02", 
" -.38320022e+02  .48168923e+03 -.48069490e+04  .31382088e+05  .45202317e-02", 
" -.38080130e+02  .47630359e+03 -.47410848e+04  .31080660e+05  .43967190e-02", 
" -.37844723e+02  .47072496e+03 -.46450926e+04  .30030609e+05  .43931480e-02", 
" -.37614332e+02  .46543102e+03 -.45680269e+04  .29431370e+05  .44174356e-02", 
" -.37388352e+02  .46014277e+03 -.44761411e+04  .28393524e+05  .43484257e-02", 
" -.37167370e+02  .45536480e+03 -.44112717e+04  .27805223e+05  .43569471e-02", 
" -.36949815e+02  .45058920e+03 -.43497130e+04  .27392379e+05  .43235284e-02", 
" -.36736757e+02  .44629286e+03 -.43136287e+04  .27429547e+05  .43465710e-02", 
" -.36526601e+02  .44198088e+03 -.42781042e+04  .27544456e+05  .43418428e-02", 
" -.36319444e+02  .43758135e+03 -.42300907e+04  .27426955e+05  .43408406e-02", 
" -.36116410e+02  .43334567e+03 -.41884792e+04  .27395641e+05  .42784493e-02", 
" -.35916299e+02  .42918560e+03 -.41382782e+04  .27011565e+05  .42354327e-02", 
" -.35717971e+02  .42473334e+03 -.40735335e+04  .26541937e+05  .42116518e-02", 
" -.35521895e+02  .42012202e+03 -.39852522e+04  .25443065e+05  .42193495e-02", 
" -.35329601e+02  .41598833e+03 -.39293063e+04  .25007630e+05  .41745292e-02", 
" -.35139819e+02  .41202285e+03 -.38845936e+04  .24851410e+05  .41575676e-02", 
" -.34952627e+02  .40795335e+03 -.38199452e+04  .24111217e+05  .41278330e-02", 
" -.34768385e+02  .40443540e+03 -.38014473e+04  .24435198e+05  .40990609e-02", 
" -.34587208e+02  .40093225e+03 -.37708902e+04  .24392559e+05  .39993523e-02", 
" -.34405677e+02  .39668587e+03 -.36850129e+04  .23271697e+05  .40152228e-02", 
" -.34227890e+02  .39294305e+03 -.36288057e+04  .22661529e+05  .39365924e-02", 
" -.34052300e+02  .38926197e+03 -.35749026e+04  .22135321e+05  .39076693e-02", 
" -.33878338e+02  .38576665e+03 -.35379281e+04  .22016750e+05  .39167188e-02", 
" -.33706578e+02  .38238145e+03 -.35067583e+04  .21988843e+05  .38831714e-02", 
" -.33536978e+02  .37914031e+03 -.34774381e+04  .21941561e+05  .38252523e-02", 
" -.33369026e+02  .37563705e+03 -.34242754e+04  .21379278e+05  .37877542e-02", 
" -.33201871e+02  .37226914e+03 -.33915139e+04  .21396500e+05  .38156087e-02", 
" -.33037198e+02  .36914574e+03 -.33576928e+04  .21113360e+05  .38152768e-02", 
" -.32873420e+02  .36575266e+03 -.33078678e+04  .20633746e+05  .38194031e-02", 
" -.32711417e+02  .36236587e+03 -.32559575e+04  .20105144e+05  .38032602e-02", 
" -.32551722e+02  .35939912e+03 -.32336482e+04  .20192191e+05  .38043064e-02", 
" -.32392925e+02  .35612046e+03 -.31862145e+04  .19795385e+05  .37872833e-02", 
" -.32234920e+02  .35267083e+03 -.31187371e+04  .18879848e+05  .37564309e-02", 
" -.32078746e+02  .34969703e+03 -.30921340e+04  .18856402e+05  .37856530e-02", 
" -.31924753e+02  .34687117e+03 -.30705691e+04  .18919354e+05  .37721989e-02", 
" -.31771152e+02  .34393246e+03 -.30391529e+04  .18761961e+05  .37848482e-02", 
" -.31618145e+02  .34083607e+03 -.29914715e+04  .18226721e+05  .37486798e-02", 
" -.31466932e+02  .33808204e+03 -.29779356e+04  .18574841e+05  .37526648e-02", 
" -.31317837e+02  .33546368e+03 -.29613139e+04  .18679108e+05  .37450077e-02", 
" -.31169192e+02  .33253580e+03 -.29166538e+04  .18161069e+05  .37536810e-02", 
" -.31021944e+02  .32994396e+03 -.28980622e+04  .18183240e+05  .37327224e-02", 
" -.30875171e+02  .32709457e+03 -.28604463e+04  .17888971e+05  .36845647e-02", 
" -.30729757e+02  .32442143e+03 -.28352720e+04  .17821508e+05  .36646524e-02", 
" -.30585224e+02  .32187313e+03 -.28141245e+04  .17760170e+05  .36153909e-02", 
" -.30441775e+02  .31939485e+03 -.27987803e+04  .17870635e+05  .36035825e-02", 
" -.30299491e+02  .31704536e+03 -.27929768e+04  .18188908e+05  .35519744e-02", 
" -.30156618e+02  .31429839e+03 -.27603367e+04  .18012847e+05  .35507708e-02", 
" -.30015748e+02  .31183886e+03 -.27446956e+04  .18161892e+05  .35481256e-02", 
" -.29875073e+02  .30914909e+03 -.27049296e+04  .17699598e+05  .35325811e-02", 
" -.29735179e+02  .30652107e+03 -.26717057e+04  .17455623e+05  .35721213e-02", 
" -.29595810e+02  .30385006e+03 -.26354366e+04  .17151843e+05  .35604210e-02", 
" -.29458033e+02  .30162173e+03 -.26331335e+04  .17561514e+05  .35095725e-02", 
" -.29320480e+02  .29907083e+03 -.26027307e+04  .17365450e+05  .34736952e-02", 
" -.29184081e+02  .29675053e+03 -.25892659e+04  .17538604e+05  .34331719e-02", 
" -.29047072e+02  .29406474e+03 -.25441936e+04  .16999661e+05  .33835215e-02", 
" -.28910754e+02  .29152586e+03 -.25137099e+04  .16811437e+05  .33420123e-02", 
" -.28775080e+02  .28904577e+03 -.24892838e+04  .16788653e+05  .33480268e-02", 
" -.28639147e+02  .28627810e+03 -.24391861e+04  .16189269e+05  .33215541e-02", 
" -.28504499e+02  .28366096e+03 -.23968825e+04  .15734553e+05  .33257645e-02", 
" -.28371460e+02  .28146682e+03 -.23832836e+04  .15832260e+05  .33399402e-02", 
" -.28238014e+02  .27887645e+03 -.23403140e+04  .15357711e+05  .33510700e-02", 
" -.28105172e+02  .27641202e+03 -.23044906e+04  .14992979e+05  .33383318e-02", 
" -.27973857e+02  .27424949e+03 -.22885782e+04  .15027287e+05  .33408716e-02", 
" -.27843145e+02  .27212729e+03 -.22759841e+04  .15142936e+05  .33314916e-02", 
" -.27711468e+02  .26959407e+03 -.22274262e+04  .14427200e+05  .33395744e-02", 
" -.27580916e+02  .26720536e+03 -.21910030e+04  .14018698e+05  .33453641e-02", 
" -.27450729e+02  .26478538e+03 -.21475595e+04  .13404385e+05  .33428999e-02", 
" -.27320731e+02  .26253420e+03 -.21214533e+04  .13221775e+05  .33641207e-02", 
" -.27190715e+02  .26027004e+03 -.20985049e+04  .13165446e+05  .33513072e-02", 
" -.27060232e+02  .25784426e+03 -.20587159e+04  .12676441e+05  .33003530e-02", 
" -.26931311e+02  .25583165e+03 -.20489842e+04  .12797098e+05  .32698551e-02", 
" -.26802634e+02  .25373339e+03 -.20298484e+04  .12716392e+05  .32878298e-02", 
" -.26673795e+02  .25146431e+03 -.19986294e+04  .12415136e+05  .32707550e-02", 
" -.26545392e+02  .24942479e+03 -.19825037e+04  .12370295e+05  .32404847e-02", 
" -.26416543e+02  .24703372e+03 -.19391760e+04  .11795852e+05  .32415901e-02", 
" -.26287943e+02  .24459255e+03 -.18907939e+04  .11088225e+05  .32053944e-02", 
" -.26159697e+02  .24236656e+03 -.18616506e+04  .10836543e+05  .32403202e-02", 
" -.26031783e+02  .24032637e+03 -.18464627e+04  .10860537e+05  .32188671e-02", 
" -.25903979e+02  .23829845e+03 -.18297178e+04  .10808678e+05  .31992981e-02", 
" -.25776052e+02  .23605147e+03 -.17915518e+04  .10259782e+05  .32164799e-02", 
" -.25648314e+02  .23386628e+03 -.17583127e+04  .98228804e+04  .32304654e-02", 
" -.25521184e+02  .23199584e+03 -.17554440e+04  .10115449e+05  .32119076e-02", 
" -.25393939e+02  .23013330e+03 -.17516903e+04  .10374970e+05  .31908612e-02", 
" -.25267031e+02  .22822802e+03 -.17428364e+04  .10503360e+05  .31659222e-02", 
" -.25139361e+02  .22602185e+03 -.17115166e+04  .10206059e+05  .31459433e-02", 
" -.25011199e+02  .22377952e+03 -.16763895e+04  .98167310e+04  .30944676e-02", 
" -.24883962e+02  .22189503e+03 -.16664569e+04  .98950655e+04  .30980429e-02", 
" -.24756395e+02  .21997202e+03 -.16584343e+04  .10083831e+05  .30810338e-02", 
" -.24628785e+02  .21797107e+03 -.16409884e+04  .10027565e+05  .30319369e-02", 
" -.24501193e+02  .21622178e+03 -.16418085e+04  .10329094e+05  .29787525e-02", 
" -.24373343e+02  .21425873e+03 -.16241296e+04  .10229097e+05  .30173143e-02", 
" -.24245663e+02  .21242395e+03 -.16190475e+04  .10456315e+05  .29954622e-02", 
" -.24117833e+02  .21036714e+03 -.15973830e+04  .10377582e+05  .30328067e-02", 
" -.23989002e+02  .20837820e+03 -.15787824e+04  .10274752e+05  .30585560e-02", 
" -.23859663e+02  .20635653e+03 -.15586071e+04  .10154681e+05  .30631783e-02", 
" -.23730996e+02  .20452114e+03 -.15526351e+04  .10338520e+05  .30707639e-02", 
" -.23602463e+02  .20277434e+03 -.15464922e+04  .10446220e+05  .30425682e-02", 
" -.23473174e+02  .20068563e+03 -.15148786e+04  .10034996e+05  .30445941e-02", 
" -.23342671e+02  .19835264e+03 -.14647595e+04  .92208871e+04  .30412538e-02", 
" -.23212918e+02  .19651040e+03 -.14552256e+04  .92791030e+04  .30714440e-02", 
" -.23082898e+02  .19468981e+03 -.14465424e+04  .93855630e+04  .30548953e-02", 
" -.22951961e+02  .19280342e+03 -.14370775e+04  .95433390e+04  .30469889e-02", 
" -.22820044e+02  .19053989e+03 -.13952658e+04  .89786895e+04  .30464624e-02", 
" -.22688025e+02  .18853123e+03 -.13724389e+04  .87709488e+04  .30095104e-02", 
" -.22555660e+02  .18672809e+03 -.13667101e+04  .89147219e+04  .30052234e-02", 
" -.22422702e+02  .18473498e+03 -.13454386e+04  .87422777e+04  .29842792e-02", 
" -.22289603e+02  .18290045e+03 -.13367768e+04  .88417313e+04  .30264456e-02", 
" -.22155927e+02  .18106703e+03 -.13249262e+04  .88135655e+04  .30111791e-02", 
" -.22020323e+02  .17872414e+03 -.12804492e+04  .82258154e+04  .29965649e-02", 
" -.21884706e+02  .17672622e+03 -.12562621e+04  .79327069e+04  .29716595e-02", 
" -.21748098e+02  .17458560e+03 -.12233852e+04  .75276025e+04  .29884593e-02", 
" -.21611087e+02  .17250177e+03 -.11983790e+04  .73231558e+04  .29692873e-02", 
" -.21473300e+02  .17044139e+03 -.11727094e+04  .70461884e+04  .29695060e-02", 
" -.21334277e+02  .16834289e+03 -.11461642e+04  .67720019e+04  .28980424e-02", 
" -.21195396e+02  .16674488e+03 -.11606364e+04  .73784566e+04  .29231464e-02", 
" -.21054876e+02  .16456588e+03 -.11229959e+04  .68048162e+04  .29565838e-02", 
" -.20913137e+02  .16235715e+03 -.10864208e+04  .63056585e+04  .29541121e-02", 
" -.20770894e+02  .16042245e+03 -.10759440e+04  .64354321e+04  .29956111e-02", 
" -.20627096e+02  .15816514e+03 -.10358742e+04  .58787865e+04  .30037857e-02", 
" -.20482669e+02  .15632805e+03 -.10356984e+04  .62151107e+04  .30612788e-02", 
" -.20335099e+02  .15380934e+03 -.97896108e+03  .52973495e+04  .30374749e-02", 
" -.20187914e+02  .15176220e+03 -.95977571e+03  .52019161e+04  .30269560e-02", 
" -.20039182e+02  .14965415e+03 -.93466718e+03  .49436456e+04  .30276369e-02", 
" -.19887942e+02  .14746775e+03 -.90671557e+03  .45977949e+04  .30125792e-02", 
" -.19736504e+02  .14563306e+03 -.90801797e+03  .49300330e+04  .31072822e-02", 
" -.19584059e+02  .14376910e+03 -.91032874e+03  .53858096e+04  .31027813e-02", 
" -.19429647e+02  .14192839e+03 -.90972210e+03  .56483838e+04  .30863831e-02", 
" -.19272511e+02  .13974808e+03 -.88784726e+03  .55381495e+04  .31157054e-02", 
" -.19113867e+02  .13791673e+03 -.89773151e+03  .61326418e+04  .31048916e-02", 
" -.18952993e+02  .13562261e+03 -.86051297e+03  .55527387e+04  .31422308e-02", 
" -.18789418e+02  .13325488e+03 -.82233487e+03  .49957101e+04  .31319148e-02", 
" -.18623531e+02  .13084351e+03 -.78274988e+03  .44296032e+04  .31256781e-02", 
" -.18455712e+02  .12882019e+03 -.78469002e+03  .48783386e+04  .31328576e-02", 
" -.18285349e+02  .12674399e+03 -.77548731e+03  .49761683e+04  .31567723e-02", 
" -.18111379e+02  .12435362e+03 -.74908778e+03  .48174417e+04  .31360618e-02", 
" -.17934855e+02  .12193164e+03 -.71537592e+03  .44014381e+04  .30751444e-02", 
" -.17756004e+02  .11959626e+03 -.68150214e+03  .38903896e+04  .31182975e-02", 
" -.17572886e+02  .11704175e+03 -.64099000e+03  .33665681e+04  .32076124e-02", 
" -.17385597e+02  .11434791e+03 -.59095852e+03  .25757050e+04  .32479933e-02", 
" -.17194631e+02  .11201289e+03 -.57360785e+03  .25224365e+04  .32350548e-02", 
" -.17000317e+02  .10980580e+03 -.56926555e+03  .27900982e+04  .32425037e-02", 
" -.16799886e+02  .10715435e+03 -.53603348e+03  .24600483e+04  .31800970e-02", 
" -.16594593e+02  .10447257e+03 -.50210819e+03  .21007250e+04  .31973261e-02", 
" -.16386947e+02  .10293041e+03 -.55116542e+03  .33324493e+04  .32153296e-02", 
" -.16170867e+02  .10054997e+03 -.54309803e+03  .34897446e+04  .32282185e-02", 
" -.15947284e+02  .97745098e+02 -.51757546e+03  .34647829e+04  .32456587e-02", 
" -.15715428e+02  .94820712e+02 -.48502885e+03  .31969708e+04  .32563025e-02", 
" -.15476901e+02  .92338631e+02 -.49857240e+03  .40220966e+04  .32555921e-02", 
" -.15227040e+02  .89132180e+02 -.45669224e+03  .35791170e+04  .32672878e-02", 
" -.14968542e+02  .86208317e+02 -.43727078e+03  .35438226e+04  .33280482e-02", 
" -.14698695e+02  .83129129e+02 -.42264156e+03  .38211447e+04  .33595669e-02", 
" -.14415307e+02  .80063920e+02 -.40954024e+03  .39423039e+04  .34198502e-02", 
" -.14118154e+02  .77175999e+02 -.41063853e+03  .43441145e+04  .35245305e-02", 
" -.13799092e+02  .73249181e+02 -.35387024e+03  .36721269e+04  .35376236e-02", 
" -.13459651e+02  .69613660e+02 -.32131493e+03  .33260380e+04  .37078635e-02", 
" -.13094231e+02  .65610421e+02 -.28106370e+03  .29197721e+04  .37472328e-02", 
" -.12693861e+02  .60644316e+02 -.18781408e+03  .14088703e+04  .38981070e-02", 
" -.12251393e+02  .55565388e+02 -.12862072e+03  .89463215e+03  .41679027e-02", 
" -.11754623e+02  .49829636e+02 -.27963256e+02 -.85734349e+03  .42645130e-02", 
" -.11183551e+02  .44978003e+02 -.61917721e+02  .61765458e+03  .45536405e-02", 
" -.10497709e+02  .40142501e+02 -.13696455e+03  .25093528e+04  .49339439e-02", 
" -.96006654e+01  .31721163e+02 -.73108273e+02  .16956212e+04  .52656292e-02", 
" -.93804671e+01  .29511763e+02 -.39641796e+02  .11064790e+04  .54296720e-02", 
" -.91415747e+01  .27361231e+02 -.37892464e+02  .14083650e+04  .56788945e-02", 
" -.88777715e+01  .25124332e+02 -.45099568e+02  .19012220e+04  .57621034e-02", 
" -.85828269e+01  .23173493e+02 -.91741482e+02  .33969871e+04  .58294905e-02", 
" -.82426105e+01  .20083447e+02 -.63635140e+02  .28270448e+04  .61883499e-02", 
" -.78451051e+01  .17374417e+02 -.96631529e+02  .35947207e+04  .67595776e-02", 
" -.73526409e+01  .12877042e+02 -.34550521e+02  .22051511e+04  .73267660e-02", 
" -.66919547e+01  .56159587e+01  .62422364e+02  .15160712e+04  .87452364e-02", 
" -.56507719e+01 -.34690820e+01  .78350211e+02  .27186275e+04  .11203360e-01", 
" -.46981448e+01 -.89335292e+01 -.21043127e+03  .10765451e+05  .14734811e-01", 
" -.35253899e+01 -.20270409e+02 -.72551329e+02  .82937760e+04  .21336618e-01", 
" -.26899268e+01 -.31245938e+02  .11195922e+03  .93105197e+04  .27914642e-01", 
"coactt    5  9  3   35", " -.93433815e+02  .24472750e+04 -.42738294e+05  .38503743e+06  .11956632e+00", 
" -.89616632e+02  .22911012e+04 -.40193348e+05  .36885022e+06  .84532092e-01", 
" -.84321735e+02  .20425602e+04 -.33919496e+05  .29455280e+06  .51609028e-01", 
" -.80189661e+02  .18528704e+04 -.29070775e+05  .23430730e+06  .38435753e-01", 
" -.75956151e+02  .16852505e+04 -.26280552e+05  .22113056e+06  .30307828e-01", 
" -.73418617e+02  .15780238e+04 -.23812194e+05  .19476008e+06  .24177117e-01", 
" -.71610314e+02  .15108323e+04 -.22585708e+05  .18405334e+06  .21751329e-01", 
" -.70184384e+02  .14560102e+04 -.21479704e+05  .17369865e+06  .20590528e-01", 
" -.69005105e+02  .14117708e+04 -.20640719e+05  .16654670e+06  .18656224e-01", 
" -.67987998e+02  .13723249e+04 -.19855441e+05  .16039939e+06  .16828050e-01", 
" -.67112888e+02  .13401019e+04 -.19198521e+05  .15344882e+06  .16553957e-01", 
" -.66323576e+02  .13105197e+04 -.18599720e+05  .14790316e+06  .15416234e-01", 
" -.65615890e+02  .12843747e+04 -.18045208e+05  .14177297e+06  .14584445e-01", 
" -.62866278e+02  .11886869e+04 -.16309776e+05  .12677364e+06  .12055651e-01", 
" -.60858038e+02  .11196142e+04 -.15065090e+05  .11679615e+06  .10700786e-01", 
" -.59273786e+02  .10669538e+04 -.14115687e+05  .10837581e+06  .96263739e-02", 
" -.57956388e+02  .10231161e+04 -.13287027e+05  .10042085e+06  .91336600e-02", 
" -.56832549e+02  .98770960e+03 -.12669056e+05  .94763354e+05  .84412865e-02", 
" -.55836305e+02  .95538137e+03 -.12030323e+05  .87739332e+05  .81473277e-02", 
" -.54949257e+02  .92777004e+03 -.11555440e+05  .83586811e+05  .76272633e-02", 
" -.54145333e+02  .90234206e+03 -.11057708e+05  .78135790e+05  .72699891e-02", 
" -.53406470e+02  .87886829e+03 -.10585629e+05  .72797147e+05  .71188649e-02", 
" -.52729153e+02  .85868665e+03 -.10262376e+05  .70510323e+05  .69700197e-02", 
" -.52100805e+02  .83988817e+03 -.99400582e+04  .67742185e+05  .66160404e-02", 
" -.51512806e+02  .82298988e+03 -.96668059e+04  .65256424e+05  .65953202e-02", 
" -.50962146e+02  .80730063e+03 -.94231829e+04  .63376937e+05  .64810556e-02", 
" -.50443765e+02  .79269969e+03 -.92075031e+04  .61966518e+05  .63928517e-02", 
" -.49947315e+02  .77817763e+03 -.89570622e+04  .59715092e+05  .62559227e-02", 
" -.49477085e+02  .76437375e+03 -.87024799e+04  .57094342e+05  .61317770e-02", 
" -.49030123e+02  .75215593e+03 -.85340103e+04  .56217624e+05  .59638674e-02", 
" -.48603894e+02  .74095734e+03 -.83977740e+04  .55752840e+05  .57455767e-02", 
" -.48192516e+02  .72966481e+03 -.82322085e+04  .54705250e+05  .55686280e-02", 
" -.47797927e+02  .71938318e+03 -.81270915e+04  .55001818e+05  .55479289e-02", 
" -.47416614e+02  .70883800e+03 -.79602122e+04  .53710512e+05  .56031989e-02", 
" -.47049704e+02  .69907968e+03 -.78287138e+04  .53041100e+05  .54941400e-02", 
" -.46695844e+02  .69010931e+03 -.77245583e+04  .52659993e+05  .54589284e-02", 
" -.46351684e+02  .68096771e+03 -.75968453e+04  .51892537e+05  .53509047e-02", 
" -.46018782e+02  .67260523e+03 -.75164886e+04  .52133723e+05  .52900047e-02", 
" -.45694996e+02  .66394100e+03 -.73874860e+04  .51246868e+05  .52035815e-02", 
" -.45380557e+02  .65535703e+03 -.72377648e+04  .49756219e+05  .52615782e-02", 
" -.45076077e+02  .64746852e+03 -.71139793e+04  .48517803e+05  .51688006e-02", 
" -.44780731e+02  .64013039e+03 -.70183896e+04  .47918353e+05  .50389732e-02", 
" -.44490362e+02  .63201089e+03 -.68576584e+04  .46047550e+05  .49288615e-02", 
" -.44208813e+02  .62503530e+03 -.67720432e+04  .45665402e+05  .49005092e-02", 
" -.43933264e+02  .61784551e+03 -.66568021e+04  .44700110e+05  .48235643e-02", 
" -.43664312e+02  .61134353e+03 -.65793346e+04  .44323904e+05  .47313469e-02", 
" -.43400193e+02  .60408546e+03 -.64336355e+04  .42557491e+05  .46106080e-02", 
" -.43143112e+02  .59783045e+03 -.63615400e+04  .42372794e+05  .45690155e-02", 
" -.42887453e+02  .59093935e+03 -.62386955e+04  .41163993e+05  .45183120e-02", 
" -.42640344e+02  .58491506e+03 -.61539172e+04  .40419342e+05  .44815306e-02", 
" -.42396487e+02  .57869168e+03 -.60485612e+04  .39301771e+05  .44265171e-02", 
" -.42158187e+02  .57294747e+03 -.59778003e+04  .38975108e+05  .43869337e-02", 
" -.41922348e+02  .56682495e+03 -.58717045e+04  .37880604e+05  .43900949e-02", 
" -.41692745e+02  .56158628e+03 -.58312558e+04  .38232057e+05  .43646204e-02", 
" -.41465532e+02  .55583115e+03 -.57414822e+04  .37504737e+05  .42985106e-02", 
" -.41244046e+02  .55055321e+03 -.56685459e+04  .36975434e+05  .42748390e-02", 
" -.41024471e+02  .54509008e+03 -.55830844e+04  .36220105e+05  .42248234e-02", 
" -.40809070e+02  .53983013e+03 -.54996474e+04  .35419560e+05  .41923365e-02", 
" -.40598702e+02  .53531131e+03 -.54648869e+04  .35543088e+05  .41852856e-02", 
" -.40390264e+02  .53039864e+03 -.54006063e+04  .35186926e+05  .41786670e-02", 
" -.40184104e+02  .52556728e+03 -.53392918e+04  .34815309e+05  .41411991e-02", 
" -.39980487e+02  .52055763e+03 -.52500921e+04  .33740956e+05  .41182044e-02", 
" -.39779998e+02  .51577408e+03 -.51806038e+04  .33182583e+05  .40828101e-02", 
" -.39581649e+02  .51135083e+03 -.51390686e+04  .33227975e+05  .40528817e-02", 
" -.39386717e+02  .50687810e+03 -.50802665e+04  .32825975e+05  .40243347e-02", 
" -.39193738e+02  .50234639e+03 -.50128540e+04  .32222528e+05  .39998424e-02", 
" -.39002887e+02  .49809657e+03 -.49684810e+04  .32115269e+05  .39324901e-02", 
" -.38813738e+02  .49356817e+03 -.48952445e+04  .31368000e+05  .39102269e-02", 
" -.38627033e+02  .48926397e+03 -.48370358e+04  .30955411e+05  .39382816e-02", 
" -.38442286e+02  .48508118e+03 -.47859202e+04  .30688334e+05  .39464604e-02", 
" -.38260693e+02  .48122471e+03 -.47549856e+04  .30835288e+05  .39884947e-02", 
" -.38080570e+02  .47708194e+03 -.46945827e+04  .30331904e+05  .40078497e-02", 
" -.37902066e+02  .47301563e+03 -.46375027e+04  .29888658e+05  .39784237e-02", 
" -.37724888e+02  .46901007e+03 -.45800589e+04  .29361352e+05  .39368391e-02", 
" -.37549618e+02  .46493399e+03 -.45138770e+04  .28660736e+05  .39388698e-02", 
" -.37375917e+02  .46093125e+03 -.44544949e+04  .28162558e+05  .38674255e-02", 
" -.37205123e+02  .45721603e+03 -.44100595e+04  .27914068e+05  .38506565e-02", 
" -.37034576e+02  .45329582e+03 -.43533010e+04  .27500543e+05  .38368890e-02", 
" -.36865129e+02  .44934797e+03 -.42924038e+04  .26980833e+05  .38268011e-02", 
" -.36697965e+02  .44570638e+03 -.42496030e+04  .26795115e+05  .38325730e-02", 
" -.36532245e+02  .44192604e+03 -.41881643e+04  .26175453e+05  .38022918e-02", 
" -.36369083e+02  .43871776e+03 -.41683511e+04  .26390475e+05  .38312241e-02", 
" -.36204716e+02  .43496723e+03 -.41114815e+04  .25886512e+05  .38583957e-02", 
" -.36042854e+02  .43147273e+03 -.40707272e+04  .25733656e+05  .38392099e-02", 
" -.35882196e+02  .42815856e+03 -.40410411e+04  .25776793e+05  .38317806e-02", 
" -.35721916e+02  .42456646e+03 -.39904245e+04  .25420316e+05  .38078083e-02", 
" -.35564215e+02  .42128504e+03 -.39566794e+04  .25376578e+05  .37507395e-02", 
" -.35407465e+02  .41806650e+03 -.39242071e+04  .25266094e+05  .37755845e-02", 
" -.35250465e+02  .41444312e+03 -.38557919e+04  .24353607e+05  .37540931e-02", 
" -.35095492e+02  .41120172e+03 -.38179151e+04  .24137833e+05  .37215743e-02", 
" -.34941585e+02  .40802344e+03 -.37798257e+04  .23870471e+05  .37403042e-02", 
" -.34787595e+02  .40485062e+03 -.37571362e+04  .24151182e+05  .36913261e-02", 
" -.34635266e+02  .40164436e+03 -.37136330e+04  .23753409e+05  .36662550e-02", 
" -.34482992e+02  .39836712e+03 -.36711976e+04  .23479706e+05  .36655487e-02", 
" -.34331953e+02  .39499487e+03 -.36139773e+04  .22847235e+05  .36504245e-02", 
" -.34181813e+02  .39190448e+03 -.35798698e+04  .22695375e+05  .36321959e-02", 
" -.34032503e+02  .38854606e+03 -.35170999e+04  .21878940e+05  .36135716e-02", 
" -.33884865e+02  .38566599e+03 -.34944605e+04  .21968122e+05  .35933421e-02", 
" -.33736930e+02  .38238608e+03 -.34353214e+04  .21232858e+05  .36140127e-02", 
" -.33590554e+02  .37954964e+03 -.34099283e+04  .21193167e+05  .36139537e-02", 
" -.33444597e+02  .37674675e+03 -.33903307e+04  .21330280e+05  .36389664e-02", 
" -.33298246e+02  .37363436e+03 -.33480383e+04  .21023465e+05  .36209501e-02", 
" -.33152655e+02  .37044286e+03 -.32943857e+04  .20434639e+05  .35789632e-02", 
" -.33007938e+02  .36741784e+03 -.32536135e+04  .20110123e+05  .35578021e-02", 
" -.32864001e+02  .36448399e+03 -.32163973e+04  .19820185e+05  .35629563e-02", 
" -.32720847e+02  .36163639e+03 -.31839824e+04  .19627670e+05  .35723343e-02", 
" -.32577693e+02  .35859845e+03 -.31356180e+04  .19125462e+05  .35672671e-02", 
" -.32435636e+02  .35574288e+03 -.31019852e+04  .18932850e+05  .35247152e-02", 
" -.32293857e+02  .35284306e+03 -.30602623e+04  .18509236e+05  .35773681e-02", 
" -.32153102e+02  .35013475e+03 -.30333057e+04  .18446630e+05  .35350463e-02", 
" -.32012196e+02  .34739533e+03 -.30091397e+04  .18508802e+05  .35196747e-02", 
" -.31871925e+02  .34479631e+03 -.29953130e+04  .18794923e+05  .35365358e-02", 
" -.31732846e+02  .34223117e+03 -.29711126e+04  .18706699e+05  .35358424e-02", 
" -.31592828e+02  .33942021e+03 -.29322361e+04  .18329449e+05  .35103643e-02", 
" -.31452589e+02  .33653215e+03 -.28876157e+04  .17821005e+05  .34952487e-02", 
" -.31313035e+02  .33363689e+03 -.28443747e+04  .17428632e+05  .34633274e-02", 
" -.31173638e+02  .33073036e+03 -.27959247e+04  .16888439e+05  .34755184e-02", 
" -.31034920e+02  .32804896e+03 -.27657579e+04  .16713947e+05  .34797036e-02", 
" -.30897184e+02  .32559698e+03 -.27523268e+04  .16912573e+05  .35065276e-02", 
" -.30758962e+02  .32286445e+03 -.27180720e+04  .16697323e+05  .34855760e-02", 
" -.30620729e+02  .32000543e+03 -.26729673e+04  .16251889e+05  .34898357e-02", 
" -.30482172e+02  .31718829e+03 -.26274247e+04  .15695192e+05  .34588158e-02", 
" -.30345526e+02  .31466655e+03 -.26010907e+04  .15570171e+05  .34105261e-02", 
" -.30208607e+02  .31221993e+03 -.25842638e+04  .15677222e+05  .33534368e-02", 
" -.30071863e+02  .30976283e+03 -.25662145e+04  .15759618e+05  .33319186e-02", 
" -.29934822e+02  .30721062e+03 -.25412616e+04  .15706598e+05  .33333288e-02", 
" -.29797890e+02  .30465475e+03 -.25091968e+04  .15392740e+05  .33264912e-02", 
" -.29661716e+02  .30233929e+03 -.24974916e+04  .15567562e+05  .33470275e-02", 
" -.29524482e+02  .29975680e+03 -.24692300e+04  .15431724e+05  .33715503e-02", 
" -.29388017e+02  .29724172e+03 -.24420147e+04  .15296172e+05  .34124837e-02", 
" -.29251440e+02  .29450482e+03 -.23896066e+04  .14523405e+05  .34328263e-02", 
" -.29113893e+02  .29175185e+03 -.23421967e+04  .13903093e+05  .34410447e-02", 
" -.28977244e+02  .28922828e+03 -.23135140e+04  .13761619e+05  .34061822e-02", 
" -.28839774e+02  .28672652e+03 -.22958028e+04  .13962476e+05  .33699143e-02", 
" -.28702218e+02  .28402711e+03 -.22511073e+04  .13417009e+05  .33840729e-02", 
" -.28565236e+02  .28149528e+03 -.22162241e+04  .13057936e+05  .33782998e-02", 
" -.28427367e+02  .27888211e+03 -.21813705e+04  .12781931e+05  .33908913e-02", 
" -.28290029e+02  .27638163e+03 -.21548008e+04  .12691442e+05  .33551210e-02", 
" -.28152150e+02  .27378099e+03 -.21178645e+04  .12329995e+05  .33526411e-02", 
" -.28013722e+02  .27104774e+03 -.20683645e+04  .11676450e+05  .33810523e-02", 
" -.27875960e+02  .26854805e+03 -.20325710e+04  .11258652e+05  .33691280e-02", 
" -.27737682e+02  .26608412e+03 -.20052165e+04  .11103107e+05  .33522827e-02", 
" -.27599310e+02  .26367581e+03 -.19777111e+04  .10861461e+05  .33161803e-02", 
" -.27460136e+02  .26132093e+03 -.19623047e+04  .10971583e+05  .33391499e-02", 
" -.27320781e+02  .25898314e+03 -.19408029e+04  .10844886e+05  .33467974e-02", 
" -.27181713e+02  .25670664e+03 -.19252245e+04  .10903843e+05  .33395162e-02", 
" -.27041105e+02  .25406612e+03 -.18823198e+04  .10393453e+05  .33252951e-02", 
" -.26900684e+02  .25164306e+03 -.18580470e+04  .10322595e+05  .33391590e-02", 
" -.26759064e+02  .24899574e+03 -.18151917e+04  .98281497e+04  .33757971e-02", 
" -.26618092e+02  .24672881e+03 -.18036551e+04  .10044924e+05  .33732556e-02", 
" -.26475586e+02  .24403177e+03 -.17555813e+04  .94260146e+04  .32964356e-02", 
" -.26333341e+02  .24176483e+03 -.17446987e+04  .96270214e+04  .33011132e-02", 
" -.26191137e+02  .23960601e+03 -.17365807e+04  .97827586e+04  .32765130e-02", 
" -.26047092e+02  .23698846e+03 -.16922959e+04  .91762939e+04  .32398226e-02", 
" -.25901986e+02  .23441059e+03 -.16546254e+04  .87448617e+04  .32367958e-02", 
" -.25756879e+02  .23198528e+03 -.16344435e+04  .88106577e+04  .32177129e-02", 
" -.25611279e+02  .22979271e+03 -.16275598e+04  .90356785e+04  .31995118e-02", 
" -.25464029e+02  .22728885e+03 -.16012665e+04  .89051836e+04  .31738816e-02", 
" -.25315849e+02  .22481603e+03 -.15779216e+04  .88313287e+04  .32089626e-02", 
" -.25166881e+02  .22215806e+03 -.15298497e+04  .80750316e+04  .31882085e-02", 
" -.25017244e+02  .21968489e+03 -.14995146e+04  .77536419e+04  .32139231e-02", 
" -.24867468e+02  .21755098e+03 -.15007419e+04  .81635716e+04  .32262909e-02", 
" -.24714870e+02  .21489236e+03 -.14603865e+04  .76660469e+04  .32785482e-02", 
" -.24561250e+02  .21225812e+03 -.14275532e+04  .74205128e+04  .32542143e-02", 
" -.24407563e+02  .20980072e+03 -.14040034e+04  .73262450e+04  .32565343e-02", 
" -.24252301e+02  .20744037e+03 -.13949152e+04  .75943036e+04  .32308794e-02", 
" -.24095299e+02  .20484101e+03 -.13678710e+04  .74904647e+04  .32913675e-02", 
" -.23937815e+02  .20254924e+03 -.13567983e+04  .75725240e+04  .32656312e-02", 
" -.23777998e+02  .19983216e+03 -.13196680e+04  .72515213e+04  .32820980e-02", 
" -.23616162e+02  .19687267e+03 -.12566520e+04  .62325219e+04  .33007789e-02", 
" -.23453263e+02  .19399156e+03 -.11972697e+04  .52568282e+04  .33533384e-02", 
" -.23289607e+02  .19143941e+03 -.11686546e+04  .50136516e+04  .33795409e-02", 
" -.23123975e+02  .18888076e+03 -.11416724e+04  .47651737e+04  .33950030e-02", 
" -.22956371e+02  .18637067e+03 -.11235344e+04  .47970725e+04  .34274882e-02", 
" -.22785247e+02  .18321398e+03 -.10589700e+04  .39329720e+04  .34333069e-02", 
" -.22613570e+02  .18059864e+03 -.10283811e+04  .35748296e+04  .34615617e-02", 
" -.22439664e+02  .17827951e+03 -.10348499e+04  .41683665e+04  .34871743e-02", 
" -.22262771e+02  .17552262e+03 -.10031866e+04  .39194370e+04  .34752376e-02", 
" -.22083291e+02  .17300572e+03 -.99698928e+03  .42335499e+04  .33995700e-02", 
" -.21901212e+02  .17023685e+03 -.96343701e+03  .38476244e+04  .34478241e-02", 
" -.21715075e+02  .16696469e+03 -.89130462e+03  .26727456e+04  .34881070e-02", 
" -.21527444e+02  .16436058e+03 -.88301369e+03  .29702563e+04  .35173429e-02", 
" -.21335568e+02  .16141951e+03 -.84750372e+03  .26260784e+04  .34969332e-02", 
" -.21139859e+02  .15854590e+03 -.82524559e+03  .26421161e+04  .34799755e-02", 
" -.20940606e+02  .15549504e+03 -.78474563e+03  .21632517e+04  .34499232e-02", 
" -.20737728e+02  .15267528e+03 -.76947085e+03  .23137995e+04  .34736594e-02", 
" -.20531138e+02  .14998615e+03 -.76406176e+03  .26149704e+04  .34483286e-02", 
" -.20318530e+02  .14673146e+03 -.71564899e+03  .19538819e+04  .34039480e-02", 
" -.20100044e+02  .14317657e+03 -.65203998e+03  .10784582e+04  .34206248e-02", 
" -.19877635e+02  .14006663e+03 -.62111661e+03  .84102899e+03  .34208171e-02", 
" -.19647544e+02  .13647766e+03 -.56850211e+03  .29663234e+03  .34639413e-02", 
" -.19413425e+02  .13351507e+03 -.55606730e+03  .43527775e+03  .34880196e-02", 
" -.19169144e+02  .13028311e+03 -.54598221e+03  .76064335e+03  .34758910e-02", 
" -.18917874e+02  .12665996e+03 -.49554766e+03  .16547699e+03  .35454272e-02", 
" -.18655135e+02  .12238041e+03 -.40728390e+03 -.12229576e+04  .37222691e-02", 
" -.18384654e+02  .11893026e+03 -.38850341e+03 -.11740165e+04  .38047691e-02", 
" -.18102716e+02  .11530633e+03 -.37564398e+03 -.81170666e+03  .38755159e-02", 
" -.17806245e+02  .11103036e+03 -.31465069e+03 -.15919699e+04  .39007854e-02", 
" -.17494669e+02  .10646813e+03 -.25159092e+03 -.22378323e+04  .39058736e-02", 
" -.17166771e+02  .10213570e+03 -.21343919e+03 -.24451238e+04  .38708927e-02", 
" -.16818714e+02  .98159478e+02 -.23803745e+03 -.10889606e+04  .39736528e-02", 
" -.16445414e+02  .93126995e+02 -.18021144e+03 -.15904375e+04  .40811765e-02", 
" -.16042008e+02  .87744209e+02 -.13459813e+03 -.16918803e+04  .42290044e-02", 
" -.15605279e+02  .82509996e+02 -.11463737e+03 -.12837296e+04  .43274880e-02", 
" -.15119925e+02  .76010433e+02 -.46843583e+02 -.14825409e+04  .44980206e-02", 
" -.14572994e+02  .68417154e+02  .70069314e+02 -.29563003e+04  .47221811e-02", 
" -.13940050e+02  .59469777e+02  .23170243e+03 -.50865083e+04  .49505517e-02", 
" -.13176500e+02  .49527648e+02  .36684541e+03 -.65033475e+04  .54746186e-02", 
" -.12189953e+02  .37214095e+02  .50002784e+03 -.74197829e+04  .59402040e-02", 
" -.11947933e+02  .34263978e+02  .52104721e+03 -.72993179e+04  .61311291e-02", 
" -.11686337e+02  .32147488e+02  .46254319e+03 -.55173836e+04  .61813597e-02", 
" -.11397518e+02  .29245418e+02  .45599989e+03 -.49317926e+04  .64724988e-02", 
" -.11074404e+02  .25530143e+02  .50083658e+03 -.55274030e+04  .68436541e-02", 
" -.10703768e+02  .21151436e+02  .53400668e+03 -.54177312e+04  .73482426e-02", 
" -.10265895e+02  .16057947e+02  .57636735e+03 -.56104413e+04  .78196334e-02", 
" -.97236123e+01  .94764567e+01  .61084402e+03 -.49743180e+04  .85329589e-02", 
" -.90147044e+01  .33622487e+01  .45435308e+03  .49017181e+03  .96596665e-02", 
" -.78902067e+01 -.12613483e+02  .78052831e+03 -.44157102e+04  .11863977e-01", 
" -.68562057e+01 -.26823893e+02  .95948513e+03 -.63904587e+04  .16025902e-01", 
" -.56139362e+01 -.44134320e+02  .12646368e+04 -.12313814e+05  .22680316e-01", 
" -.47093837e+01 -.74056508e+02  .25898406e+04 -.36440266e+05  .29589659e-01"
)


.urc7 = 
c("Copyright (C) James G. MacKinnon, 1995", "conc      6  7  3   25", 
" -.63751631e+01 -.52812913e+02 -.15265281e+03 -.23009777e+04  .54753342e-02", 
" -.62232720e+01 -.50035651e+02 -.20661991e+02 -.38849330e+04  .41180662e-02", 
" -.60219664e+01 -.42552513e+02 -.13358402e+03 -.94263146e+03  .27329557e-02", 
" -.58559390e+01 -.38862513e+02 -.88920400e+02 -.10153884e+04  .20925470e-02", 
" -.56852880e+01 -.34054986e+02 -.12079769e+03  .31260794e+02  .13820476e-02", 
" -.55790730e+01 -.32044856e+02 -.90141633e+02 -.13819141e+03  .12156083e-02", 
" -.55015023e+01 -.30467682e+02 -.84159385e+02  .14806961e+02  .10914839e-02", 
" -.54400721e+01 -.29291228e+02 -.73299927e+02 -.15941673e+02  .10047156e-02", 
" -.53887081e+01 -.28327244e+02 -.63459911e+02 -.67368903e+02  .92417845e-03", 
" -.53447655e+01 -.27459026e+02 -.59682060e+02 -.54526053e+02  .85578897e-03", 
" -.53062952e+01 -.26598811e+02 -.62287057e+02  .49439679e+02  .80043380e-03", 
" -.52712482e+01 -.26086475e+02 -.50069288e+02 -.99699365e+02  .77567667e-03", 
" -.52398568e+01 -.25492175e+02 -.49058920e+02 -.55280472e+02  .75017901e-03", 
" -.51150870e+01 -.23198631e+02 -.47755864e+02  .16428068e+03  .68027920e-03", 
" -.50226843e+01 -.21676157e+02 -.38648005e+02  .15110965e+03  .60673255e-03", 
" -.49487634e+01 -.20465509e+02 -.30632048e+02  .76402530e+02  .54428742e-03", 
" -.48862001e+01 -.19492633e+02 -.25085028e+02  .44697806e+02  .50049779e-03", 
" -.48319916e+01 -.18632663e+02 -.23463525e+02  .69063553e+02  .48116758e-03", 
" -.47840114e+01 -.17937434e+02 -.16980627e+02 -.24440796e+02  .44590285e-03", 
" -.47407892e+01 -.17313189e+02 -.12574962e+02 -.69101107e+02  .42051998e-03", 
" -.47012244e+01 -.16755980e+02 -.91350174e+01 -.10387655e+03  .40668687e-03", 
" -.46647252e+01 -.16257664e+02 -.68989944e+01 -.11618185e+03  .38264207e-03", 
" -.46307157e+01 -.15838134e+02 -.25269172e+01 -.16069831e+03  .37484653e-03", 
" -.45992257e+01 -.15344881e+02 -.54474643e+01 -.87647929e+02  .35868683e-03", 
" -.45693496e+01 -.14983959e+02 -.19501964e+01 -.12784042e+03  .34223559e-03", 
" -.45410325e+01 -.14655660e+02  .11440535e+01 -.15167840e+03  .33305293e-03", 
" -.45143114e+01 -.14297853e+02  .12217426e+01 -.13546083e+03  .32492201e-03", 
" -.44889569e+01 -.13968978e+02  .26907244e+01 -.15292520e+03  .32583069e-03", 
" -.44647559e+01 -.13630236e+02  .18366933e+01 -.12870069e+03  .31752915e-03", 
" -.44415793e+01 -.13346394e+02  .36653401e+01 -.15384061e+03  .30941224e-03", 
" -.44192312e+01 -.13059495e+02  .37102711e+01 -.14495297e+03  .30646441e-03", 
" -.43976743e+01 -.12784346e+02  .34141867e+01 -.13031461e+03  .31345628e-03", 
" -.43770095e+01 -.12504387e+02  .21212974e+01 -.98937847e+02  .30911064e-03", 
" -.43568569e+01 -.12280029e+02  .38412095e+01 -.12006331e+03  .30496375e-03", 
" -.43373539e+01 -.12075193e+02  .61883762e+01 -.15481969e+03  .30034409e-03", 
" -.43184175e+01 -.11853659e+02  .66969072e+01 -.15643087e+03  .30178188e-03", 
" -.42999654e+01 -.11669788e+02  .87447736e+01 -.17976123e+03  .29924122e-03", 
" -.42821077e+01 -.11470999e+02  .96213288e+01 -.18924862e+03  .29734478e-03", 
" -.42647008e+01 -.11273252e+02  .10130531e+02 -.19623379e+03  .29383539e-03", 
" -.42478248e+01 -.11045880e+02  .78561966e+01 -.15128789e+03  .29214898e-03", 
" -.42313085e+01 -.10841343e+02  .67336053e+01 -.12600245e+03  .28696002e-03", 
" -.42151335e+01 -.10653638e+02  .62043681e+01 -.11077037e+03  .27990376e-03", 
" -.41994057e+01 -.10471021e+02  .61832887e+01 -.10883729e+03  .27362685e-03", 
" -.41839087e+01 -.10291626e+02  .58193373e+01 -.99741015e+02  .27507344e-03", 
" -.41686989e+01 -.10144813e+02  .69592332e+01 -.11447790e+03  .26725892e-03", 
" -.41538088e+01 -.99971242e+01  .80272702e+01 -.13019427e+03  .26777869e-03", 
" -.41391681e+01 -.98583688e+01  .90298472e+01 -.14015503e+03  .26926780e-03", 
" -.41248169e+01 -.96946770e+01  .82745009e+01 -.12554402e+03  .26862029e-03", 
" -.41106749e+01 -.95662125e+01  .98639303e+01 -.14983594e+03  .26401459e-03", 
" -.40967595e+01 -.94330620e+01  .10513712e+02 -.15549520e+03  .25692700e-03", 
" -.40831246e+01 -.93106189e+01  .11754051e+02 -.17178457e+03  .26304458e-03", 
" -.40696422e+01 -.91894196e+01  .12617260e+02 -.18036870e+03  .25877470e-03", 
" -.40564249e+01 -.90472003e+01  .11909673e+02 -.16098392e+03  .25870687e-03", 
" -.40433196e+01 -.89457688e+01  .13620541e+02 -.18522490e+03  .25720040e-03", 
" -.40304798e+01 -.88214990e+01  .13989429e+02 -.18931677e+03  .25645905e-03", 
" -.40178302e+01 -.86974201e+01  .14071907e+02 -.18640832e+03  .25410585e-03", 
" -.40053725e+01 -.85500893e+01  .12404171e+02 -.15403815e+03  .25369699e-03", 
" -.39931007e+01 -.84086824e+01  .11092049e+02 -.13187560e+03  .25209905e-03", 
" -.39808218e+01 -.83102571e+01  .12001058e+02 -.13973747e+03  .24682768e-03", 
" -.39687009e+01 -.82151487e+01  .12964490e+02 -.15033416e+03  .24382941e-03", 
" -.39567437e+01 -.81083016e+01  .13059829e+02 -.14747997e+03  .24104759e-03", 
" -.39449622e+01 -.79932171e+01  .12855072e+02 -.14290555e+03  .24050663e-03", 
" -.39333148e+01 -.78916111e+01  .13539491e+02 -.15558861e+03  .23757679e-03", 
" -.39218171e+01 -.77821323e+01  .13365527e+02 -.14927312e+03  .23894340e-03", 
" -.39103625e+01 -.76936195e+01  .14249232e+02 -.16014010e+03  .23959026e-03", 
" -.38991370e+01 -.75805185e+01  .13754037e+02 -.15091763e+03  .23948501e-03", 
" -.38879612e+01 -.74718971e+01  .13061913e+02 -.13595608e+03  .23948034e-03", 
" -.38768235e+01 -.73770301e+01  .13184352e+02 -.13590543e+03  .23915438e-03", 
" -.38658326e+01 -.72806952e+01  .13181593e+02 -.13261745e+03  .23607920e-03", 
" -.38549427e+01 -.71838491e+01  .13138176e+02 -.13073830e+03  .23723217e-03", 
" -.38441083e+01 -.70898273e+01  .13146847e+02 -.12939823e+03  .23297555e-03", 
" -.38334101e+01 -.69882324e+01  .12773629e+02 -.12433714e+03  .22956699e-03", 
" -.38227935e+01 -.68893795e+01  .12089844e+02 -.10941339e+03  .22789234e-03", 
" -.38122927e+01 -.67941878e+01  .11794162e+02 -.10339718e+03  .22764247e-03", 
" -.38018651e+01 -.66973314e+01  .11259572e+02 -.92623888e+02  .22553936e-03", 
" -.37914699e+01 -.66162415e+01  .11831487e+02 -.10215143e+03  .22613036e-03", 
" -.37810288e+01 -.65518281e+01  .12788350e+02 -.11377529e+03  .22778369e-03", 
" -.37706576e+01 -.64955690e+01  .14326125e+02 -.13466551e+03  .22990328e-03", 
" -.37605120e+01 -.63974528e+01  .13415914e+02 -.11726061e+03  .22604472e-03", 
" -.37504021e+01 -.63053333e+01  .12751617e+02 -.10436928e+03  .22328616e-03", 
" -.37404067e+01 -.61952605e+01  .11245684e+02 -.83946959e+02  .22275561e-03", 
" -.37303140e+01 -.61327376e+01  .12135022e+02 -.96669872e+02  .22643008e-03", 
" -.37203505e+01 -.60561473e+01  .12385318e+02 -.10110851e+03  .22514425e-03", 
" -.37104568e+01 -.59705490e+01  .11919109e+02 -.93591604e+02  .22584658e-03", 
" -.37005641e+01 -.59074730e+01  .12745387e+02 -.10573831e+03  .22669844e-03", 
" -.36907997e+01 -.58081611e+01  .11395312e+02 -.84767556e+02  .22631828e-03", 
" -.36810459e+01 -.57397189e+01  .12027853e+02 -.96391806e+02  .22435836e-03", 
" -.36713079e+01 -.56659141e+01  .11979194e+02 -.93330299e+02  .22344857e-03", 
" -.36616499e+01 -.55875396e+01  .11874862e+02 -.93417312e+02  .22439630e-03", 
" -.36520228e+01 -.55044221e+01  .11029819e+02 -.77381571e+02  .22238407e-03", 
" -.36423908e+01 -.54454315e+01  .11944193e+02 -.92413583e+02  .21831174e-03", 
" -.36328611e+01 -.53696987e+01  .11511479e+02 -.82168404e+02  .21618978e-03", 
" -.36233579e+01 -.52930620e+01  .11196375e+02 -.76681725e+02  .21695020e-03", 
" -.36138658e+01 -.52305882e+01  .11872020e+02 -.89225677e+02  .21883262e-03", 
" -.36043825e+01 -.51626641e+01  .11854223e+02 -.88673041e+02  .21762540e-03", 
" -.35948837e+01 -.51064102e+01  .12268422e+02 -.91969634e+02  .21586492e-03", 
" -.35855121e+01 -.50272786e+01  .11518602e+02 -.78531805e+02  .21389990e-03", 
" -.35761864e+01 -.49472413e+01  .10787312e+02 -.67443709e+02  .21500762e-03", 
" -.35668420e+01 -.48775183e+01  .10641859e+02 -.66250320e+02  .21511607e-03", 
" -.35575315e+01 -.48080165e+01  .10265818e+02 -.59826589e+02  .21452324e-03", 
" -.35482258e+01 -.47267276e+01  .89127230e+01 -.35232300e+02  .21267545e-03", 
" -.35389015e+01 -.46770388e+01  .98534808e+01 -.50440558e+02  .21127725e-03", 
" -.35296661e+01 -.45947814e+01  .88199687e+01 -.35042225e+02  .21339255e-03", 
" -.35204202e+01 -.45342786e+01  .90269927e+01 -.37510976e+02  .21320109e-03", 
" -.35111560e+01 -.44700818e+01  .86061649e+01 -.26130196e+02  .21168652e-03", 
" -.35019355e+01 -.44026773e+01  .81577114e+01 -.17910462e+02  .20863909e-03", 
" -.34927533e+01 -.43277132e+01  .70802114e+01  .16597278e+01  .20580556e-03", 
" -.34834812e+01 -.42742022e+01  .75617456e+01 -.80378219e+01  .20571747e-03", 
" -.34742285e+01 -.42292341e+01  .83156353e+01 -.18068297e+02  .20669507e-03", 
" -.34650049e+01 -.41777051e+01  .87331832e+01 -.23900670e+02  .20799733e-03", 
" -.34558275e+01 -.41130769e+01  .83168370e+01 -.17410941e+02  .20796865e-03", 
" -.34466249e+01 -.40549307e+01  .81499957e+01 -.12756544e+02  .20944948e-03", 
" -.34374380e+01 -.40016302e+01  .85487141e+01 -.20196934e+02  .20872116e-03", 
" -.34282405e+01 -.39430545e+01  .85107624e+01 -.20841087e+02  .20990010e-03", 
" -.34190702e+01 -.38783869e+01  .80303828e+01 -.13584996e+02  .21035025e-03", 
" -.34098334e+01 -.38340293e+01  .85934668e+01 -.21532938e+02  .21080691e-03", 
" -.34006569e+01 -.37712917e+01  .81085629e+01 -.14815107e+02  .21136702e-03", 
" -.33914161e+01 -.37343278e+01  .92592559e+01 -.34466700e+02  .21131230e-03", 
" -.33821902e+01 -.36879844e+01  .98197867e+01 -.43773570e+02  .21159865e-03", 
" -.33730027e+01 -.36347940e+01  .99219265e+01 -.44603862e+02  .21210031e-03", 
" -.33638357e+01 -.35710962e+01  .95032844e+01 -.38655518e+02  .21182115e-03", 
" -.33546407e+01 -.35059465e+01  .86343805e+01 -.22258549e+02  .21253182e-03", 
" -.33453825e+01 -.34528595e+01  .86228322e+01 -.22114281e+02  .21342099e-03", 
" -.33361060e+01 -.34044018e+01  .88479013e+01 -.25036358e+02  .21612390e-03", 
" -.33268768e+01 -.33407544e+01  .83106877e+01 -.18399300e+02  .21624637e-03", 
" -.33175846e+01 -.32938126e+01  .87325496e+01 -.26680218e+02  .21532497e-03", 
" -.33082298e+01 -.32578305e+01  .96164492e+01 -.41171800e+02  .21670185e-03", 
" -.32989352e+01 -.32008803e+01  .92711901e+01 -.35532355e+02  .21713037e-03", 
" -.32895449e+01 -.31544080e+01  .94157613e+01 -.36558183e+02  .21700407e-03", 
" -.32801924e+01 -.30913104e+01  .87207720e+01 -.27695466e+02  .21725056e-03", 
" -.32707417e+01 -.30420491e+01  .86292636e+01 -.26280727e+02  .21420765e-03", 
" -.32613017e+01 -.29890243e+01  .82430135e+01 -.19254895e+02  .21296998e-03", 
" -.32518556e+01 -.29411969e+01  .85252855e+01 -.25496215e+02  .21437297e-03", 
" -.32423395e+01 -.28919855e+01  .84642892e+01 -.25057381e+02  .21427558e-03", 
" -.32328497e+01 -.28306944e+01  .76862310e+01 -.13491236e+02  .21236008e-03", 
" -.32233030e+01 -.27784616e+01  .74817521e+01 -.10728337e+02  .21238003e-03", 
" -.32136853e+01 -.27301864e+01  .74704491e+01 -.11446637e+02  .21145797e-03", 
" -.32039838e+01 -.26949625e+01  .80167937e+01 -.18317333e+02  .21396490e-03", 
" -.31943072e+01 -.26444775e+01  .80475627e+01 -.23336760e+02  .21172653e-03", 
" -.31845443e+01 -.26051902e+01  .83005703e+01 -.24889276e+02  .20806421e-03", 
" -.31747498e+01 -.25691152e+01  .88364032e+01 -.33489619e+02  .20833911e-03", 
" -.31649470e+01 -.25220384e+01  .86449702e+01 -.27957369e+02  .20876709e-03", 
" -.31550161e+01 -.24965821e+01  .97662608e+01 -.43133655e+02  .21133500e-03", 
" -.31450933e+01 -.24490101e+01  .95546389e+01 -.38367004e+02  .21204398e-03", 
" -.31351404e+01 -.23999626e+01  .91551801e+01 -.29964830e+02  .21234417e-03", 
" -.31251542e+01 -.23546388e+01  .93339565e+01 -.34591274e+02  .21088262e-03", 
" -.31150207e+01 -.23312354e+01  .10760170e+02 -.57945692e+02  .21165601e-03", 
" -.31048721e+01 -.22860665e+01  .10735288e+02 -.57858609e+02  .21426010e-03", 
" -.30946757e+01 -.22393357e+01  .10579222e+02 -.54784008e+02  .21376596e-03", 
" -.30843924e+01 -.22044570e+01  .11270842e+02 -.67210570e+02  .21346665e-03", 
" -.30739996e+01 -.21678982e+01  .11563043e+02 -.70249082e+02  .21448515e-03", 
" -.30635321e+01 -.21383775e+01  .12483960e+02 -.85141013e+02  .21599499e-03", 
" -.30530558e+01 -.20861793e+01  .12110153e+02 -.81811342e+02  .21624491e-03", 
" -.30425206e+01 -.20298152e+01  .11366142e+02 -.69758514e+02  .21583610e-03", 
" -.30318780e+01 -.19789385e+01  .10992088e+02 -.64911691e+02  .21505123e-03", 
" -.30210700e+01 -.19486860e+01  .11574741e+02 -.72323400e+02  .21690670e-03", 
" -.30102314e+01 -.18917748e+01  .10495440e+02 -.55003519e+02  .22207235e-03", 
" -.29992637e+01 -.18591958e+01  .11253902e+02 -.69081457e+02  .22429611e-03", 
" -.29881928e+01 -.18129557e+01  .11035921e+02 -.67451173e+02  .22553359e-03", 
" -.29770061e+01 -.17814210e+01  .11844518e+02 -.83036137e+02  .22452402e-03", 
" -.29657297e+01 -.17459699e+01  .12403336e+02 -.93616729e+02  .22210057e-03", 
" -.29543027e+01 -.17041988e+01  .12320584e+02 -.91819387e+02  .22535186e-03", 
" -.29427391e+01 -.16621596e+01  .11769662e+02 -.77591534e+02  .22686367e-03", 
" -.29310929e+01 -.16212814e+01  .11845451e+02 -.79610791e+02  .22494008e-03", 
" -.29192691e+01 -.15883442e+01  .12464246e+02 -.92965128e+02  .22768622e-03", 
" -.29073429e+01 -.15444510e+01  .12096937e+02 -.85861203e+02  .22943544e-03", 
" -.28952409e+01 -.15120493e+01  .12840520e+02 -.10077579e+03  .23278577e-03", 
" -.28829742e+01 -.14904764e+01  .13967841e+02 -.11828893e+03  .22896594e-03", 
" -.28705990e+01 -.14348444e+01  .12856231e+02 -.10015686e+03  .22748991e-03", 
" -.28579949e+01 -.14059627e+01  .13330001e+02 -.10626173e+03  .22976374e-03", 
" -.28452002e+01 -.13648656e+01  .13068804e+02 -.10072625e+03  .23725856e-03", 
" -.28322313e+01 -.13342121e+01  .13401739e+02 -.10401173e+03  .24108772e-03", 
" -.28191026e+01 -.12985951e+01  .13550397e+02 -.10542949e+03  .24163235e-03", 
" -.28057437e+01 -.12581311e+01  .13430547e+02 -.10512633e+03  .24379754e-03", 
" -.27921683e+01 -.12177088e+01  .13199375e+02 -.10026931e+03  .24057212e-03", 
" -.27783240e+01 -.11850773e+01  .13225673e+02 -.97342320e+02  .23803256e-03", 
" -.27642594e+01 -.11426483e+01  .13158308e+02 -.98655019e+02  .24279850e-03", 
" -.27499170e+01 -.11188190e+01  .14168008e+02 -.11597683e+03  .24842655e-03", 
" -.27352835e+01 -.10942060e+01  .14995497e+02 -.12957212e+03  .24887044e-03", 
" -.27202076e+01 -.10884091e+01  .16248940e+02 -.14297531e+03  .25341380e-03", 
" -.27048901e+01 -.10670067e+01  .17119807e+02 -.15536077e+03  .25656394e-03", 
" -.26892913e+01 -.10227375e+01  .16727478e+02 -.15050839e+03  .25922844e-03", 
" -.26733154e+01 -.96866086e+00  .15401962e+02 -.12821988e+03  .26214149e-03", 
" -.26569856e+01 -.90795282e+00  .13884720e+02 -.10629365e+03  .26821081e-03", 
" -.26401490e+01 -.88557325e+00  .14614479e+02 -.11724151e+03  .27309524e-03", 
" -.26228442e+01 -.87263990e+00  .16131701e+02 -.14357072e+03  .27554827e-03", 
" -.26050307e+01 -.84953007e+00  .16826089e+02 -.15688351e+03  .27885845e-03", 
" -.25866325e+01 -.83161124e+00  .17180107e+02 -.15766455e+03  .28522758e-03", 
" -.25676035e+01 -.83419100e+00  .19251350e+02 -.18745174e+03  .28590315e-03", 
" -.25479645e+01 -.83030654e+00  .20834022e+02 -.21209990e+03  .28394151e-03", 
" -.25276483e+01 -.81337591e+00  .21325387e+02 -.21699440e+03  .28911459e-03", 
" -.25067189e+01 -.76258193e+00  .20278654e+02 -.20259351e+03  .28859206e-03", 
" -.24849290e+01 -.70308301e+00  .18415229e+02 -.17172437e+03  .29202238e-03", 
" -.24620394e+01 -.67309373e+00  .17810512e+02 -.15704559e+03  .29805613e-03", 
" -.24380289e+01 -.68312577e+00  .19773189e+02 -.18554934e+03  .29856734e-03", 
" -.24129441e+01 -.65272143e+00  .19353603e+02 -.17368789e+03  .30922651e-03", 
" -.23863951e+01 -.65262430e+00  .20802188e+02 -.19902666e+03  .32188491e-03", 
" -.23582938e+01 -.61849581e+00  .19286802e+02 -.16843466e+03  .33030109e-03", 
" -.23283832e+01 -.62904467e+00  .20912715e+02 -.19186045e+03  .33657433e-03", 
" -.22964266e+01 -.60349883e+00  .19494980e+02 -.16134291e+03  .34161532e-03", 
" -.22620260e+01 -.60527898e+00  .20772541e+02 -.18254188e+03  .35353461e-03", 
" -.22242647e+01 -.66637703e+00  .23850334e+02 -.21768935e+03  .36762308e-03", 
" -.21826201e+01 -.70850734e+00  .25036828e+02 -.22437878e+03  .37061696e-03", 
" -.21362254e+01 -.78695338e+00  .28520424e+02 -.27523145e+03  .39909244e-03", 
" -.20836404e+01 -.90448382e+00  .32828952e+02 -.32506843e+03  .42553880e-03", 
" -.20229106e+01 -.92504035e+00  .32550746e+02 -.33200674e+03  .47558966e-03", 
" -.19497806e+01 -.10325139e+01  .33642131e+02 -.33970012e+03  .50938627e-03", 
" -.18573812e+01 -.11177858e+01  .31061942e+02 -.29113329e+03  .58210075e-03", 
" -.17297841e+01 -.13253459e+01  .32099379e+02 -.31064074e+03  .69683703e-03", 
" -.16962094e+01 -.15770740e+01  .42549519e+02 -.46199157e+03  .72777276e-03", 
" -.16589967e+01 -.17337051e+01  .48084912e+02 -.55000564e+03  .77165619e-03", 
" -.16177234e+01 -.17531229e+01  .44696025e+02 -.49648012e+03  .84744793e-03", 
" -.15694820e+01 -.18678268e+01  .42098581e+02 -.42822181e+03  .89011636e-03", 
" -.15130114e+01 -.19487385e+01  .38270953e+02 -.35064230e+03  .10048085e-02", 
" -.14437648e+01 -.21424401e+01  .38728842e+02 -.34124278e+03  .11319614e-02", 
" -.13543323e+01 -.23805706e+01  .37621039e+02 -.30062579e+03  .12925991e-02", 
" -.12293410e+01 -.24598603e+01  .18434757e+02  .66373681e+02  .15732910e-02", 
" -.10134959e+01 -.27714656e+01 -.51296205e+01  .48301186e+03  .20900774e-02", 
" -.79896906e+00 -.36734513e+01  .16846898e+02  .11478714e+03  .27859957e-02", 
" -.52397686e+00 -.57854458e+01  .12476777e+03 -.16965997e+04  .46209001e-02", 
" -.32798414e+00 -.52282654e+01  .70751512e+02 -.10778113e+04  .64137154e-02", 
"coc       6  8  3   25", " -.66314707e+01 -.62109242e+02 -.17767675e+02 -.52555403e+04  .57117375e-02", 
" -.64861468e+01 -.56681985e+02 -.55976853e+02 -.37100794e+04  .39441964e-02", 
" -.62886779e+01 -.48972185e+02 -.11650832e+03 -.16465087e+04  .25212760e-02", 
" -.61238515e+01 -.44308807e+02 -.13016037e+03 -.49428521e+03  .18866415e-02", 
" -.59535696e+01 -.39479922e+02 -.12416006e+03 -.82623609e+02  .13985443e-02", 
" -.58489728e+01 -.36893924e+02 -.11410889e+03  .98187509e+02  .12262928e-02", 
" -.57724556e+01 -.35090334e+02 -.10534608e+03  .18816725e+03  .11330362e-02", 
" -.57119715e+01 -.33634473e+02 -.10062122e+03  .26182692e+03  .10113664e-02", 
" -.56609554e+01 -.32599787e+02 -.88837558e+02  .16828828e+03  .96678155e-03", 
" -.56170826e+01 -.31584020e+02 -.88890265e+02  .26033022e+03  .90305512e-03", 
" -.55781504e+01 -.30815742e+02 -.82185108e+02  .22730545e+03  .83941493e-03", 
" -.55435398e+01 -.30221175e+02 -.71100850e+02  .11762334e+03  .80272012e-03", 
" -.55123191e+01 -.29575989e+02 -.67596706e+02  .11216159e+03  .75982223e-03", 
" -.53879048e+01 -.27258416e+02 -.48093771e+02  .21126682e+02  .66436599e-03", 
" -.52964827e+01 -.25432934e+02 -.46933177e+02  .17048616e+03  .59960080e-03", 
" -.52224061e+01 -.24206889e+02 -.35087182e+02  .10125364e+03  .53655002e-03", 
" -.51601891e+01 -.23191898e+02 -.22995395e+02 -.38000429e+02  .51961334e-03", 
" -.51068022e+01 -.22138531e+02 -.26274626e+02  .71648363e+02  .48422525e-03", 
" -.50593080e+01 -.21330500e+02 -.22335952e+02  .42276378e+02  .44593301e-03", 
" -.50159805e+01 -.20714348e+02 -.13439851e+02 -.64710447e+02  .41270622e-03", 
" -.49768351e+01 -.20050168e+02 -.12119208e+02 -.49384760e+02  .40161720e-03", 
" -.49406409e+01 -.19435960e+02 -.12839125e+02 -.70226968e+01  .39153900e-03", 
" -.49069422e+01 -.18905058e+02 -.11248197e+02 -.10098583e+02  .38837690e-03", 
" -.48754256e+01 -.18428657e+02 -.88881627e+01 -.28225734e+02  .37600914e-03", 
" -.48456615e+01 -.18020709e+02 -.47741419e+01 -.79411293e+02  .35765101e-03", 
" -.48176681e+01 -.17623835e+02 -.14543550e+01 -.12176889e+03  .35114443e-03", 
" -.47910867e+01 -.17263841e+02  .22221376e+01 -.16940378e+03  .35791811e-03", 
" -.47657909e+01 -.16885395e+02  .22010191e+01 -.14764744e+03  .34750801e-03", 
" -.47415830e+01 -.16551738e+02  .38867017e+01 -.16043057e+03  .33960851e-03", 
" -.47184014e+01 -.16214776e+02  .43520036e+01 -.15430807e+03  .32352320e-03", 
" -.46961057e+01 -.15890408e+02  .42834206e+01 -.14386032e+03  .32083994e-03", 
" -.46746315e+01 -.15621213e+02  .69955267e+01 -.17434129e+03  .31094570e-03", 
" -.46540572e+01 -.15302627e+02  .57773762e+01 -.14823950e+03  .30259630e-03", 
" -.46340541e+01 -.15035282e+02  .64544498e+01 -.14619432e+03  .29482301e-03", 
" -.46146378e+01 -.14782911e+02  .74820076e+01 -.15445748e+03  .29259078e-03", 
" -.45958055e+01 -.14537518e+02  .79245993e+01 -.14660083e+03  .29256394e-03", 
" -.45774484e+01 -.14321541e+02  .99702277e+01 -.17026742e+03  .28793699e-03", 
" -.45597172e+01 -.14070853e+02  .91426295e+01 -.14858031e+03  .27818715e-03", 
" -.45424215e+01 -.13845043e+02  .96306273e+01 -.15076161e+03  .27478364e-03", 
" -.45255710e+01 -.13621189e+02  .96648388e+01 -.14567132e+03  .27271842e-03", 
" -.45090551e+01 -.13422478e+02  .10850826e+02 -.15916182e+03  .27237617e-03", 
" -.44928477e+01 -.13236364e+02  .12056315e+02 -.16879307e+03  .26848934e-03", 
" -.44771256e+01 -.13047025e+02  .12934642e+02 -.17642979e+03  .26389587e-03", 
" -.44615995e+01 -.12877745e+02  .14271958e+02 -.18947514e+03  .26839138e-03", 
" -.44464767e+01 -.12695854e+02  .15121611e+02 -.19892239e+03  .26587268e-03", 
" -.44315768e+01 -.12537055e+02  .16632668e+02 -.21669144e+03  .26669560e-03", 
" -.44170574e+01 -.12356344e+02  .16747398e+02 -.21584301e+03  .26838841e-03", 
" -.44027887e+01 -.12174786e+02  .16343468e+02 -.20573694e+03  .27262735e-03", 
" -.43887333e+01 -.12011239e+02  .16663398e+02 -.20476884e+03  .27439517e-03", 
" -.43749491e+01 -.11841017e+02  .16348605e+02 -.19571830e+03  .27324666e-03", 
" -.43614367e+01 -.11663048e+02  .15525203e+02 -.17931469e+03  .27157070e-03", 
" -.43480293e+01 -.11522995e+02  .16753681e+02 -.19770388e+03  .26449130e-03", 
" -.43347579e+01 -.11394914e+02  .18132951e+02 -.21376422e+03  .26022166e-03", 
" -.43216767e+01 -.11278685e+02  .20066340e+02 -.23963123e+03  .25868664e-03", 
" -.43089020e+01 -.11129642e+02  .19737480e+02 -.22838346e+03  .25404010e-03", 
" -.42963262e+01 -.10985228e+02  .19834015e+02 -.22636686e+03  .25475473e-03", 
" -.42839326e+01 -.10817464e+02  .18096982e+02 -.19457832e+03  .25172680e-03", 
" -.42716985e+01 -.10670357e+02  .17969121e+02 -.19382299e+03  .24989400e-03", 
" -.42595213e+01 -.10536438e+02  .18065907e+02 -.19422804e+03  .24843916e-03", 
" -.42475516e+01 -.10385590e+02  .16894989e+02 -.17236228e+03  .24803666e-03", 
" -.42356430e+01 -.10267490e+02  .17603174e+02 -.18124328e+03  .25453811e-03", 
" -.42238302e+01 -.10144857e+02  .17354100e+02 -.16863171e+03  .25255594e-03", 
" -.42122640e+01 -.10011806e+02  .16952521e+02 -.15888910e+03  .24992909e-03", 
" -.42008511e+01 -.98713141e+01  .15886967e+02 -.13782518e+03  .24764643e-03", 
" -.41895065e+01 -.97444627e+01  .15378563e+02 -.12513472e+03  .24797544e-03", 
" -.41781991e+01 -.96485000e+01  .16777360e+02 -.14503858e+03  .24543253e-03", 
" -.41670553e+01 -.95367211e+01  .17187559e+02 -.14917573e+03  .24288371e-03", 
" -.41560227e+01 -.94148545e+01  .16615245e+02 -.13559703e+03  .24398428e-03", 
" -.41450854e+01 -.93139863e+01  .17811102e+02 -.15695310e+03  .24580695e-03", 
" -.41342591e+01 -.91919404e+01  .17236714e+02 -.14721083e+03  .24270273e-03", 
" -.41235475e+01 -.90713505e+01  .16727554e+02 -.13864236e+03  .24133799e-03", 
" -.41128456e+01 -.89590071e+01  .16333239e+02 -.13004260e+03  .24052886e-03", 
" -.41022311e+01 -.88463642e+01  .15642522e+02 -.11291165e+03  .24049285e-03", 
" -.40917274e+01 -.87416262e+01  .15720037e+02 -.11302165e+03  .23987192e-03", 
" -.40812865e+01 -.86371362e+01  .15167102e+02 -.97155871e+02  .23830942e-03", 
" -.40709580e+01 -.85268830e+01  .14894982e+02 -.94163732e+02  .23804262e-03", 
" -.40606747e+01 -.84053692e+01  .13274314e+02 -.63607579e+02  .23774329e-03", 
" -.40503858e+01 -.83215680e+01  .14028669e+02 -.72031536e+02  .23565814e-03", 
" -.40401863e+01 -.82346043e+01  .14529229e+02 -.77705023e+02  .23253557e-03", 
" -.40301350e+01 -.81274947e+01  .14107424e+02 -.71819423e+02  .23216930e-03", 
" -.40201014e+01 -.80417323e+01  .14890217e+02 -.84030544e+02  .23282298e-03", 
" -.40101361e+01 -.79487034e+01  .15135076e+02 -.88109939e+02  .23353325e-03", 
" -.40001390e+01 -.78593505e+01  .15114389e+02 -.84573650e+02  .23238229e-03", 
" -.39902603e+01 -.77659950e+01  .15156817e+02 -.86921289e+02  .22857601e-03", 
" -.39804534e+01 -.76675681e+01  .14667507e+02 -.77315025e+02  .22769651e-03", 
" -.39706821e+01 -.75750338e+01  .14316706e+02 -.69806651e+02  .22766369e-03", 
" -.39609176e+01 -.74947363e+01  .14789423e+02 -.76062233e+02  .22947394e-03", 
" -.39512683e+01 -.73980603e+01  .14443794e+02 -.72997014e+02  .22856188e-03", 
" -.39415362e+01 -.73362277e+01  .15935247e+02 -.95744625e+02  .22760917e-03", 
" -.39319106e+01 -.72535363e+01  .16147540e+02 -.98432619e+02  .22862345e-03", 
" -.39222757e+01 -.71829276e+01  .16882443e+02 -.10907185e+03  .22598683e-03", 
" -.39127328e+01 -.70902537e+01  .16282440e+02 -.98328474e+02  .22719848e-03", 
" -.39031834e+01 -.70198552e+01  .16965206e+02 -.10708178e+03  .22752920e-03", 
" -.38936826e+01 -.69406079e+01  .17000405e+02 -.10592700e+03  .22733779e-03", 
" -.38842944e+01 -.68574728e+01  .16923580e+02 -.10379733e+03  .23003516e-03", 
" -.38748708e+01 -.67697353e+01  .16189332e+02 -.89488809e+02  .22962273e-03", 
" -.38654507e+01 -.67070857e+01  .17235457e+02 -.10592075e+03  .22947215e-03", 
" -.38561508e+01 -.66102915e+01  .16185256e+02 -.89589848e+02  .23225749e-03", 
" -.38468293e+01 -.65339165e+01  .16305902e+02 -.90270202e+02  .23246254e-03", 
" -.38374754e+01 -.64669814e+01  .16866510e+02 -.97700450e+02  .23264614e-03", 
" -.38281805e+01 -.63872169e+01  .16681198e+02 -.94753400e+02  .23109935e-03", 
" -.38189177e+01 -.63101562e+01  .16509775e+02 -.90145586e+02  .23094584e-03", 
" -.38096729e+01 -.62245126e+01  .15722136e+02 -.76421148e+02  .22836755e-03", 
" -.38004636e+01 -.61416827e+01  .15296843e+02 -.70081342e+02  .22777035e-03", 
" -.37912488e+01 -.60675488e+01  .15256720e+02 -.69035831e+02  .22999284e-03", 
" -.37820392e+01 -.59938526e+01  .15292468e+02 -.70489572e+02  .23062100e-03", 
" -.37728083e+01 -.59200906e+01  .15090923e+02 -.66727979e+02  .23085569e-03", 
" -.37635935e+01 -.58561771e+01  .15885400e+02 -.82333729e+02  .23091209e-03", 
" -.37544466e+01 -.57769506e+01  .15643239e+02 -.81796096e+02  .23191535e-03", 
" -.37452539e+01 -.57046640e+01  .15368218e+02 -.74519735e+02  .23070069e-03", 
" -.37360449e+01 -.56450649e+01  .16051941e+02 -.86354400e+02  .22655182e-03", 
" -.37268454e+01 -.55751065e+01  .15761616e+02 -.79876720e+02  .22626080e-03", 
" -.37177125e+01 -.54825623e+01  .14319367e+02 -.56555000e+02  .22728219e-03", 
" -.37084933e+01 -.54284994e+01  .15036164e+02 -.67077023e+02  .22753133e-03", 
" -.36993772e+01 -.53384163e+01  .13718300e+02 -.46187200e+02  .22760874e-03", 
" -.36901650e+01 -.52881254e+01  .14689734e+02 -.60104178e+02  .22802331e-03", 
" -.36810267e+01 -.52117507e+01  .14149085e+02 -.52587084e+02  .22350748e-03", 
" -.36718264e+01 -.51448009e+01  .14088212e+02 -.51972139e+02  .22220302e-03", 
" -.36626067e+01 -.50856037e+01  .14360538e+02 -.54824491e+02  .22335303e-03", 
" -.36534074e+01 -.50193572e+01  .14294904e+02 -.53693674e+02  .22315183e-03", 
" -.36442007e+01 -.49527729e+01  .14435440e+02 -.59197140e+02  .21820692e-03", 
" -.36349445e+01 -.48935975e+01  .14858987e+02 -.67376364e+02  .21643740e-03", 
" -.36256864e+01 -.48392077e+01  .15518067e+02 -.78459063e+02  .21609321e-03", 
" -.36164670e+01 -.47652399e+01  .14777459e+02 -.66153456e+02  .21559582e-03", 
" -.36071890e+01 -.47110075e+01  .15163259e+02 -.69331234e+02  .21487273e-03", 
" -.35979135e+01 -.46637645e+01  .16353850e+02 -.91172508e+02  .21833880e-03", 
" -.35886129e+01 -.46065790e+01  .16780023e+02 -.10054602e+03  .21664339e-03", 
" -.35793127e+01 -.45425065e+01  .16420874e+02 -.92843363e+02  .21384642e-03", 
" -.35699885e+01 -.44811291e+01  .16708650e+02 -.99849862e+02  .21423493e-03", 
" -.35605977e+01 -.44300631e+01  .17403473e+02 -.11347854e+03  .21429508e-03", 
" -.35511949e+01 -.43679298e+01  .17382810e+02 -.11436885e+03  .21621449e-03", 
" -.35418436e+01 -.43004782e+01  .17207495e+02 -.11353877e+03  .21758521e-03", 
" -.35324344e+01 -.42311308e+01  .16802930e+02 -.10834482e+03  .21469516e-03", 
" -.35229268e+01 -.41802490e+01  .17216129e+02 -.11378056e+03  .21613993e-03", 
" -.35133991e+01 -.41246098e+01  .17534459e+02 -.12019095e+03  .21556530e-03", 
" -.35038852e+01 -.40614949e+01  .17302546e+02 -.11656476e+03  .21371231e-03", 
" -.34943218e+01 -.39933832e+01  .16738734e+02 -.10907707e+03  .21511959e-03", 
" -.34846759e+01 -.39447957e+01  .17339168e+02 -.11905019e+03  .21418118e-03", 
" -.34749904e+01 -.38948212e+01  .17774502e+02 -.12541732e+03  .21641277e-03", 
" -.34652268e+01 -.38463905e+01  .18193830e+02 -.12992603e+03  .21788626e-03", 
" -.34554868e+01 -.37885384e+01  .18205703e+02 -.13156104e+03  .21952596e-03", 
" -.34456630e+01 -.37449912e+01  .19045996e+02 -.14414126e+03  .22009697e-03", 
" -.34357905e+01 -.37076124e+01  .20432286e+02 -.16779948e+03  .21968046e-03", 
" -.34259372e+01 -.36324106e+01  .19571892e+02 -.15878157e+03  .22106271e-03", 
" -.34159720e+01 -.35771909e+01  .19607601e+02 -.15970954e+03  .22125452e-03", 
" -.34059757e+01 -.35048511e+01  .18325256e+02 -.13705348e+03  .22029975e-03", 
" -.33959032e+01 -.34433125e+01  .18006829e+02 -.13348678e+03  .22221798e-03", 
" -.33857058e+01 -.33990516e+01  .18583078e+02 -.14303879e+03  .22148820e-03", 
" -.33754325e+01 -.33710584e+01  .20497235e+02 -.17692367e+03  .22356031e-03", 
" -.33651837e+01 -.33177056e+01  .20866993e+02 -.18627924e+03  .22367798e-03", 
" -.33548009e+01 -.32633181e+01  .20820146e+02 -.18595445e+03  .22519980e-03", 
" -.33444514e+01 -.31964108e+01  .20359119e+02 -.18204946e+03  .22614563e-03", 
" -.33339291e+01 -.31442637e+01  .20297406e+02 -.17985879e+03  .22539825e-03", 
" -.33233793e+01 -.30966680e+01  .21059265e+02 -.19595848e+03  .22249876e-03", 
" -.33126990e+01 -.30447346e+01  .21107388e+02 -.19570611e+03  .22605744e-03", 
" -.33019400e+01 -.29859166e+01  .20661004e+02 -.18818585e+03  .22474122e-03", 
" -.32910933e+01 -.29277583e+01  .20139087e+02 -.17686970e+03  .22828898e-03", 
" -.32801505e+01 -.28604016e+01  .19207532e+02 -.16209986e+03  .22568869e-03", 
" -.32690928e+01 -.28138426e+01  .19670371e+02 -.17161977e+03  .22550316e-03", 
" -.32578460e+01 -.27730708e+01  .20193018e+02 -.17907785e+03  .22707238e-03", 
" -.32465427e+01 -.27175487e+01  .19884601e+02 -.17421806e+03  .22758373e-03", 
" -.32351777e+01 -.26516732e+01  .18999727e+02 -.15944316e+03  .22928071e-03", 
" -.32236588e+01 -.26079451e+01  .19662995e+02 -.17212981e+03  .23109526e-03", 
" -.32118899e+01 -.25883202e+01  .21470936e+02 -.19999146e+03  .23233151e-03", 
" -.32000409e+01 -.25432271e+01  .21759464e+02 -.20372230e+03  .23184795e-03", 
" -.31881302e+01 -.24803848e+01  .21022642e+02 -.19164406e+03  .23325137e-03", 
" -.31760560e+01 -.24243294e+01  .20656519e+02 -.18543453e+03  .23450310e-03", 
" -.31637595e+01 -.24017340e+01  .22530732e+02 -.21674713e+03  .23384935e-03", 
" -.31513977e+01 -.23435053e+01  .22222953e+02 -.21409564e+03  .23608627e-03", 
" -.31388194e+01 -.22939410e+01  .22433823e+02 -.21927688e+03  .23933637e-03", 
" -.31260839e+01 -.22388234e+01  .22234363e+02 -.21710495e+03  .24018311e-03", 
" -.31131167e+01 -.22040339e+01  .23336478e+02 -.23640475e+03  .23948341e-03", 
" -.30999951e+01 -.21279132e+01  .21503413e+02 -.20399473e+03  .24118263e-03", 
" -.30865691e+01 -.20970706e+01  .22594872e+02 -.22024358e+03  .24119068e-03", 
" -.30729394e+01 -.20512649e+01  .22521821e+02 -.21538787e+03  .24132711e-03", 
" -.30592010e+01 -.19663584e+01  .20430094e+02 -.18245406e+03  .24085260e-03", 
" -.30451547e+01 -.19121222e+01  .20337221e+02 -.18222239e+03  .24184790e-03", 
" -.30307502e+01 -.18765154e+01  .21207611e+02 -.19840554e+03  .24225665e-03", 
" -.30160738e+01 -.18290124e+01  .21298540e+02 -.20022629e+03  .24439412e-03", 
" -.30011559e+01 -.17601142e+01  .20171534e+02 -.18531657e+03  .24928805e-03", 
" -.29858359e+01 -.17171997e+01  .20390735e+02 -.18860562e+03  .25454411e-03", 
" -.29702433e+01 -.16472657e+01  .18817764e+02 -.16287996e+03  .25720930e-03", 
" -.29542686e+01 -.16019349e+01  .19121229e+02 -.16731730e+03  .25613571e-03", 
" -.29378825e+01 -.15675915e+01  .20163385e+02 -.18340757e+03  .25742811e-03", 
" -.29210667e+01 -.15213489e+01  .20375005e+02 -.18682384e+03  .26124780e-03", 
" -.29037837e+01 -.14748492e+01  .20475789e+02 -.18927267e+03  .26608956e-03", 
" -.28859185e+01 -.14427695e+01  .21131217e+02 -.19830999e+03  .27252016e-03", 
" -.28676321e+01 -.13676264e+01  .19124093e+02 -.16279403e+03  .27499098e-03", 
" -.28486209e+01 -.13259681e+01  .19002669e+02 -.15861209e+03  .27564554e-03", 
" -.28290330e+01 -.12648377e+01  .17750453e+02 -.13762443e+03  .28083940e-03", 
" -.28086862e+01 -.12613212e+01  .20442039e+02 -.18116752e+03  .28568537e-03", 
" -.27876968e+01 -.12093055e+01  .19829538e+02 -.16878405e+03  .28834461e-03", 
" -.27657030e+01 -.11749676e+01  .20037638e+02 -.16951984e+03  .29499653e-03", 
" -.27427608e+01 -.11381223e+01  .20259295e+02 -.17420578e+03  .29961542e-03", 
" -.27186749e+01 -.11392897e+01  .22647436e+02 -.20844090e+03  .30569424e-03", 
" -.26936077e+01 -.10735497e+01  .20854877e+02 -.17558259e+03  .31933944e-03", 
" -.26671112e+01 -.10236092e+01  .20187350e+02 -.16474356e+03  .33238128e-03", 
" -.26391218e+01 -.99961568e+00  .21178139e+02 -.18053203e+03  .33544627e-03", 
" -.26092095e+01 -.10209106e+01  .25101889e+02 -.24473914e+03  .34808561e-03", 
" -.25771370e+01 -.99362720e+00  .24915586e+02 -.23808247e+03  .36077255e-03", 
" -.25426761e+01 -.96947118e+00  .25182015e+02 -.23798375e+03  .36303861e-03", 
" -.25050800e+01 -.97098479e+00  .26303063e+02 -.24809330e+03  .37580002e-03", 
" -.24638436e+01 -.96397688e+00  .26753278e+02 -.24983504e+03  .38941106e-03", 
" -.24176548e+01 -.99673861e+00  .28702935e+02 -.26645509e+03  .40409592e-03", 
" -.23653285e+01 -.11010047e+01  .35456997e+02 -.37057685e+03  .44369671e-03", 
" -.23043962e+01 -.11917785e+01  .38161314e+02 -.38954040e+03  .49806513e-03", 
" -.22311709e+01 -.13660664e+01  .44501028e+02 -.47172905e+03  .51788732e-03", 
" -.21392296e+01 -.15020526e+01  .46180904e+02 -.47512406e+03  .58230688e-03", 
" -.20115436e+01 -.18274319e+01  .49779252e+02 -.46241784e+03  .69865390e-03", 
" -.19786039e+01 -.19242193e+01  .51244180e+02 -.48096708e+03  .73788493e-03", 
" -.19423694e+01 -.19394622e+01  .50047622e+02 -.48111719e+03  .78758408e-03", 
" -.19012872e+01 -.19031427e+01  .41745006e+02 -.33082120e+03  .83851217e-03", 
" -.18536173e+01 -.21038822e+01  .46930879e+02 -.39444596e+03  .91407267e-03", 
" -.17978750e+01 -.21102156e+01  .40720684e+02 -.30921121e+03  .10434667e-02", 
" -.17294515e+01 -.23789713e+01  .48191893e+02 -.43902499e+03  .11278189e-02", 
" -.16413989e+01 -.25387601e+01  .41301446e+02 -.27820390e+03  .12822290e-02", 
" -.15160010e+01 -.30533300e+01  .45663640e+02 -.25041196e+03  .15529240e-02", 
" -.13049520e+01 -.38293583e+01  .63688357e+02 -.55286393e+03  .22418065e-02", 
" -.10950786e+01 -.38727019e+01  .35972532e+02 -.18748638e+03  .31298112e-02", 
" -.82154582e+00 -.42049916e+01  .14183401e+02  .11466939e+03  .48015395e-02", 
" -.61937627e+00 -.49482013e+01  .64404849e+02 -.98428010e+03  .65568113e-02", 
"coct      6  9  3   30", " -.68876626e+01 -.66068979e+02 -.22438053e+03 -.28019574e+04  .61332602e-02", 
" -.67376254e+01 -.60866322e+02 -.19982073e+03 -.22768548e+04  .43094743e-02", 
" -.65298600e+01 -.56324914e+02 -.37509598e+02 -.40298280e+04  .28175127e-02", 
" -.63738642e+01 -.49913704e+02 -.13273894e+03 -.11305680e+04  .21601015e-02", 
" -.62046068e+01 -.44927820e+02 -.12116989e+03 -.56639611e+03  .16139011e-02", 
" -.60987524e+01 -.42469276e+02 -.84113117e+02 -.89842689e+03  .13571974e-02", 
" -.60219823e+01 -.40524382e+02 -.73152354e+02 -.88490697e+03  .11727803e-02", 
" -.59616168e+01 -.38867085e+02 -.77283867e+02 -.60116193e+03  .10338355e-02", 
" -.59110024e+01 -.37657344e+02 -.69989260e+02 -.59332493e+03  .93855647e-03", 
" -.58680002e+01 -.36600135e+02 -.58515613e+02 -.78736259e+03  .91430399e-03", 
" -.58299620e+01 -.35607073e+02 -.63030016e+02 -.58401740e+03  .83291950e-03", 
" -.57959353e+01 -.34809889e+02 -.58497758e+02 -.55443388e+03  .83462688e-03", 
" -.57650650e+01 -.34098633e+02 -.57405851e+02 -.47135640e+03  .80962121e-03", 
" -.56414398e+01 -.31605624e+02 -.35845221e+02 -.49354940e+03  .70395154e-03", 
" -.55502676e+01 -.29795168e+02 -.19242958e+02 -.62919592e+03  .64458260e-03", 
" -.54770375e+01 -.28390667e+02 -.48283900e+01 -.80262530e+03  .57895984e-03", 
" -.54152674e+01 -.27216440e+02  .34211010e+01 -.85620608e+03  .53437138e-03", 
" -.53614437e+01 -.26274428e+02  .12049870e+02 -.91313894e+03  .49102039e-03", 
" -.53139463e+01 -.25368435e+02  .12497056e+02 -.82680919e+03  .47428004e-03", 
" -.52715506e+01 -.24406615e+02  .14648644e+01 -.54179779e+03  .45543260e-03", 
" -.52325480e+01 -.23654490e+02 -.13171071e+01 -.40904403e+03  .44583225e-03", 
" -.51964147e+01 -.23034645e+02  .15394550e+01 -.41140908e+03  .44249095e-03", 
" -.51631187e+01 -.22400847e+02 -.74295658e+00 -.31212340e+03  .42328878e-03", 
" -.51319226e+01 -.21815007e+02 -.39915471e+01 -.20196442e+03  .41321999e-03", 
" -.51025841e+01 -.21262902e+02 -.76181774e+01 -.82660840e+02  .40675929e-03", 
" -.50749322e+01 -.20766391e+02 -.67329717e+01 -.94365482e+02  .39273142e-03", 
" -.50486637e+01 -.20321958e+02 -.47823708e+01 -.12137652e+03  .39185187e-03", 
" -.50234466e+01 -.19915901e+02 -.41229974e+01 -.96595512e+02  .37825678e-03", 
" -.49992687e+01 -.19567877e+02 -.74706562e+00 -.12974593e+03  .36306261e-03", 
" -.49762532e+01 -.19180272e+02 -.14055178e+01 -.10219130e+03  .34954041e-03", 
" -.49541014e+01 -.18810406e+02 -.21229284e+01 -.66514149e+02  .34525076e-03", 
" -.49326466e+01 -.18522500e+02  .18267923e+01 -.12483367e+03  .33757272e-03", 
" -.49121298e+01 -.18176821e+02 -.17852399e+00 -.61378927e+02  .34339692e-03", 
" -.48921658e+01 -.17896393e+02  .17522819e+01 -.82408051e+02  .34305088e-03", 
" -.48728679e+01 -.17607332e+02  .27124739e+01 -.87339245e+02  .33786083e-03", 
" -.48541328e+01 -.17344613e+02  .48008441e+01 -.11389799e+03  .32934433e-03", 
" -.48359981e+01 -.17078064e+02  .57622439e+01 -.11558886e+03  .32620142e-03", 
" -.48182389e+01 -.16826976e+02  .63614163e+01 -.10588829e+03  .32790199e-03", 
" -.48009687e+01 -.16582338e+02  .73957172e+01 -.11401828e+03  .32769589e-03", 
" -.47841480e+01 -.16351046e+02  .89688697e+01 -.13644794e+03  .31599029e-03", 
" -.47676855e+01 -.16132088e+02  .10211416e+02 -.14712880e+03  .31653035e-03", 
" -.47516258e+01 -.15920473e+02  .11689888e+02 -.16535610e+03  .31163409e-03", 
" -.47358299e+01 -.15748897e+02  .15635127e+02 -.23534956e+03  .30303364e-03", 
" -.47205575e+01 -.15528820e+02  .15812198e+02 -.22899829e+03  .29779464e-03", 
" -.47055149e+01 -.15314516e+02  .15751708e+02 -.22170785e+03  .29994111e-03", 
" -.46908039e+01 -.15097030e+02  .15027596e+02 -.19990012e+03  .30028123e-03", 
" -.46761803e+01 -.14940321e+02  .18318670e+02 -.26019608e+03  .29784775e-03", 
" -.46619803e+01 -.14740633e+02  .17936451e+02 -.24285978e+03  .29261914e-03", 
" -.46481135e+01 -.14540437e+02  .17844826e+02 -.24110357e+03  .29089103e-03", 
" -.46343260e+01 -.14361653e+02  .18329767e+02 -.24431668e+03  .29014957e-03", 
" -.46207867e+01 -.14197332e+02  .19445064e+02 -.26033417e+03  .28539462e-03", 
" -.46075391e+01 -.13995004e+02  .17232777e+02 -.20909675e+03  .28550289e-03", 
" -.45944211e+01 -.13814190e+02  .16461464e+02 -.18857244e+03  .28009292e-03", 
" -.45815380e+01 -.13644637e+02  .16456123e+02 -.18326828e+03  .27808862e-03", 
" -.45688661e+01 -.13459791e+02  .15225448e+02 -.15727016e+03  .28019649e-03", 
" -.45563006e+01 -.13305605e+02  .15558447e+02 -.15281675e+03  .27652987e-03", 
" -.45439734e+01 -.13126521e+02  .14056011e+02 -.11967839e+03  .27456490e-03", 
" -.45316672e+01 -.12987987e+02  .15147962e+02 -.13312530e+03  .27708431e-03", 
" -.45195945e+01 -.12811987e+02  .12846937e+02 -.78722144e+02  .27504101e-03", 
" -.45076478e+01 -.12673548e+02  .13889265e+02 -.97989141e+02  .27337919e-03", 
" -.44957509e+01 -.12571776e+02  .17250131e+02 -.16179382e+03  .27353535e-03", 
" -.44840625e+01 -.12444677e+02  .18662862e+02 -.18594505e+03  .27468901e-03", 
" -.44724866e+01 -.12309573e+02  .19097211e+02 -.18961118e+03  .27321183e-03", 
" -.44609912e+01 -.12193198e+02  .20525609e+02 -.21088746e+03  .27646729e-03", 
" -.44496710e+01 -.12057835e+02  .20548488e+02 -.20784855e+03  .27564796e-03", 
" -.44383830e+01 -.11961229e+02  .22872230e+02 -.24232339e+03  .27480023e-03", 
" -.44272994e+01 -.11820117e+02  .22086148e+02 -.22112185e+03  .27306724e-03", 
" -.44164098e+01 -.11656088e+02  .19955511e+02 -.18393251e+03  .27222405e-03", 
" -.44055252e+01 -.11531056e+02  .20328719e+02 -.18954907e+03  .27405630e-03", 
" -.43946931e+01 -.11419035e+02  .21150395e+02 -.19849846e+03  .27098480e-03", 
" -.43839995e+01 -.11289936e+02  .20596443e+02 -.18228259e+03  .26889088e-03", 
" -.43734273e+01 -.11144126e+02  .19381517e+02 -.16309216e+03  .26538046e-03", 
" -.43629523e+01 -.11004613e+02  .18568998e+02 -.15015412e+03  .26562263e-03", 
" -.43524597e+01 -.10889333e+02  .18826699e+02 -.15377975e+03  .26686136e-03", 
" -.43420290e+01 -.10784777e+02  .19679135e+02 -.16814406e+03  .26629530e-03", 
" -.43317112e+01 -.10668897e+02  .19693643e+02 -.16642993e+03  .26432277e-03", 
" -.43215177e+01 -.10533459e+02  .18133412e+02 -.13197348e+03  .26221086e-03", 
" -.43113408e+01 -.10426080e+02  .18404827e+02 -.13261449e+03  .26186643e-03", 
" -.43012339e+01 -.10309952e+02  .17963438e+02 -.12171496e+03  .25808200e-03", 
" -.42911109e+01 -.10218528e+02  .18918629e+02 -.13122459e+03  .25954118e-03", 
" -.42811120e+01 -.10091637e+02  .17160078e+02 -.92024750e+02  .25918847e-03", 
" -.42711286e+01 -.10002972e+02  .18591588e+02 -.11862690e+03  .25818550e-03", 
" -.42611967e+01 -.99038334e+01  .18848858e+02 -.12161289e+03  .25841442e-03", 
" -.42513088e+01 -.98181351e+01  .20070131e+02 -.14251601e+03  .25781729e-03", 
" -.42414955e+01 -.97293861e+01  .21126957e+02 -.16120494e+03  .25704927e-03", 
" -.42318063e+01 -.96136179e+01  .20244136e+02 -.14561316e+03  .25709837e-03", 
" -.42221442e+01 -.94995012e+01  .19172757e+02 -.12256414e+03  .25509222e-03", 
" -.42125053e+01 -.93985136e+01  .19190557e+02 -.12402433e+03  .25296562e-03", 
" -.42028958e+01 -.93059740e+01  .19707448e+02 -.13316321e+03  .24853189e-03", 
" -.41933545e+01 -.92036843e+01  .19415752e+02 -.12348301e+03  .24504269e-03", 
" -.41837567e+01 -.91264702e+01  .20755728e+02 -.14751297e+03  .24546759e-03", 
" -.41742788e+01 -.90202296e+01  .19929291e+02 -.13133361e+03  .24353263e-03", 
" -.41648589e+01 -.89140351e+01  .19323305e+02 -.12193072e+03  .24259948e-03", 
" -.41554180e+01 -.88215722e+01  .19516512e+02 -.12728562e+03  .24195617e-03", 
" -.41460704e+01 -.87209955e+01  .19100713e+02 -.12124494e+03  .24199635e-03", 
" -.41366635e+01 -.86428804e+01  .20252766e+02 -.14466308e+03  .24067800e-03", 
" -.41272927e+01 -.85515962e+01  .19978307e+02 -.13688390e+03  .24232789e-03", 
" -.41179681e+01 -.84648474e+01  .20226590e+02 -.13912534e+03  .24236458e-03", 
" -.41086586e+01 -.83832849e+01  .20732676e+02 -.14793976e+03  .24298982e-03", 
" -.40993662e+01 -.83013921e+01  .21481038e+02 -.16552082e+03  .24453454e-03", 
" -.40901306e+01 -.81969016e+01  .20045341e+02 -.13663163e+03  .24412420e-03", 
" -.40809164e+01 -.81120576e+01  .20311070e+02 -.14006271e+03  .24397031e-03", 
" -.40717316e+01 -.80093937e+01  .18853840e+02 -.11015081e+03  .24172733e-03", 
" -.40625046e+01 -.79330917e+01  .19791733e+02 -.12989791e+03  .24171309e-03", 
" -.40532775e+01 -.78722233e+01  .21669668e+02 -.16760590e+03  .23946971e-03", 
" -.40440995e+01 -.77914287e+01  .22076750e+02 -.17804486e+03  .23666212e-03", 
" -.40349433e+01 -.77018410e+01  .21651495e+02 -.16847086e+03  .23755362e-03", 
" -.40257785e+01 -.76151943e+01  .21320009e+02 -.16318402e+03  .23683058e-03", 
" -.40165458e+01 -.75529751e+01  .22496460e+02 -.18136792e+03  .23751108e-03", 
" -.40074352e+01 -.74544978e+01  .21519294e+02 -.16814837e+03  .23700857e-03", 
" -.39982759e+01 -.73754963e+01  .21313672e+02 -.15935921e+03  .23744494e-03", 
" -.39890896e+01 -.73123691e+01  .22461722e+02 -.18005706e+03  .23751119e-03", 
" -.39799183e+01 -.72500128e+01  .23775524e+02 -.20558487e+03  .23641435e-03", 
" -.39708191e+01 -.71592851e+01  .23096192e+02 -.19451644e+03  .23634769e-03", 
" -.39617166e+01 -.70685652e+01  .22113961e+02 -.17453824e+03  .23455769e-03", 
" -.39525376e+01 -.70079607e+01  .23061444e+02 -.18660801e+03  .23150149e-03", 
" -.39433765e+01 -.69256948e+01  .22568134e+02 -.17633024e+03  .23055253e-03", 
" -.39342575e+01 -.68464297e+01  .22546885e+02 -.17806322e+03  .23257299e-03", 
" -.39251582e+01 -.67547743e+01  .21383213e+02 -.15453198e+03  .23510082e-03", 
" -.39160351e+01 -.66667805e+01  .20231600e+02 -.12652505e+03  .23647100e-03", 
" -.39068241e+01 -.66060470e+01  .21116319e+02 -.14200606e+03  .23676488e-03", 
" -.38976591e+01 -.65266956e+01  .20671209e+02 -.13206607e+03  .23803934e-03", 
" -.38884732e+01 -.64519999e+01  .20877475e+02 -.13808360e+03  .23733500e-03", 
" -.38793063e+01 -.63713762e+01  .20659350e+02 -.13870240e+03  .23690699e-03", 
" -.38700539e+01 -.63230479e+01  .22478316e+02 -.17267458e+03  .23655309e-03", 
" -.38608204e+01 -.62481216e+01  .22626602e+02 -.18044681e+03  .23835983e-03", 
" -.38515611e+01 -.61832157e+01  .23096200e+02 -.18839153e+03  .23606567e-03", 
" -.38422285e+01 -.61327783e+01  .24678231e+02 -.21857452e+03  .23650794e-03", 
" -.38329404e+01 -.60521852e+01  .23710864e+02 -.19743908e+03  .23684422e-03", 
" -.38235955e+01 -.59962040e+01  .24612861e+02 -.21006984e+03  .23468399e-03", 
" -.38142767e+01 -.59053625e+01  .22970997e+02 -.17610936e+03  .23601976e-03", 
" -.38048974e+01 -.58412013e+01  .23609382e+02 -.19113194e+03  .23545196e-03", 
" -.37954416e+01 -.57988503e+01  .26013114e+02 -.24179275e+03  .23686422e-03", 
" -.37859626e+01 -.57436361e+01  .26817196e+02 -.25520083e+03  .23595328e-03", 
" -.37765232e+01 -.56689048e+01  .26455331e+02 -.24846388e+03  .24108750e-03", 
" -.37670520e+01 -.55910181e+01  .25929112e+02 -.23933399e+03  .23967924e-03", 
" -.37575264e+01 -.55266844e+01  .26596509e+02 -.25662235e+03  .23970627e-03", 
" -.37479865e+01 -.54564552e+01  .26808863e+02 -.26385648e+03  .24349458e-03", 
" -.37383450e+01 -.53898901e+01  .26600189e+02 -.25760784e+03  .24351234e-03", 
" -.37286914e+01 -.53155692e+01  .25723010e+02 -.23840614e+03  .24344604e-03", 
" -.37189566e+01 -.52540237e+01  .26186846e+02 -.24753459e+03  .24228970e-03", 
" -.37092138e+01 -.51857790e+01  .26200478e+02 -.24812025e+03  .24170860e-03", 
" -.36994225e+01 -.51119149e+01  .25807343e+02 -.24394141e+03  .24295177e-03", 
" -.36896238e+01 -.50414280e+01  .25398596e+02 -.23239433e+03  .24292421e-03", 
" -.36797443e+01 -.49700103e+01  .25028785e+02 -.22471650e+03  .24256187e-03", 
" -.36697587e+01 -.49137345e+01  .25703881e+02 -.23756897e+03  .24219605e-03", 
" -.36597433e+01 -.48537920e+01  .26096977e+02 -.24462861e+03  .24285180e-03", 
" -.36497159e+01 -.47726152e+01  .25026469e+02 -.22315887e+03  .24640222e-03", 
" -.36394541e+01 -.47356117e+01  .26928264e+02 -.25886307e+03  .24580489e-03", 
" -.36292771e+01 -.46701036e+01  .27176942e+02 -.26632551e+03  .24725699e-03", 
" -.36190590e+01 -.45980617e+01  .26898406e+02 -.26218334e+03  .24685620e-03", 
" -.36086972e+01 -.45501468e+01  .28371023e+02 -.29328300e+03  .24697025e-03", 
" -.35982649e+01 -.44887655e+01  .28335054e+02 -.28892028e+03  .24673114e-03", 
" -.35877444e+01 -.44185053e+01  .27887031e+02 -.28057829e+03  .24926243e-03", 
" -.35771527e+01 -.43490151e+01  .27479893e+02 -.27239701e+03  .24863340e-03", 
" -.35664965e+01 -.42634604e+01  .26047005e+02 -.24672708e+03  .25156704e-03", 
" -.35557272e+01 -.41982835e+01  .26099641e+02 -.24844417e+03  .25396771e-03", 
" -.35448633e+01 -.41376077e+01  .26557459e+02 -.25947851e+03  .25192531e-03", 
" -.35338628e+01 -.40777230e+01  .26744897e+02 -.25952673e+03  .25147797e-03", 
" -.35228195e+01 -.40009752e+01  .25532964e+02 -.23401275e+03  .24758769e-03", 
" -.35115823e+01 -.39478457e+01  .26087099e+02 -.24241811e+03  .24787207e-03", 
" -.35002613e+01 -.38958699e+01  .26938134e+02 -.25591903e+03  .24330785e-03", 
" -.34888328e+01 -.38276489e+01  .26569820e+02 -.25167761e+03  .24674774e-03", 
" -.34772098e+01 -.37845715e+01  .27986064e+02 -.27723119e+03  .25040924e-03", 
" -.34654999e+01 -.37290156e+01  .28701095e+02 -.29258114e+03  .25180273e-03", 
" -.34536942e+01 -.36531396e+01  .27697417e+02 -.27045672e+03  .25782954e-03", 
" -.34416888e+01 -.35998878e+01  .28242617e+02 -.27948391e+03  .25490032e-03", 
" -.34294815e+01 -.35593777e+01  .29819616e+02 -.30804768e+03  .25162060e-03", 
" -.34171782e+01 -.34828413e+01  .28394888e+02 -.27920292e+03  .25544804e-03", 
" -.34047147e+01 -.34166709e+01  .28323827e+02 -.28214493e+03  .25563365e-03", 
" -.33921293e+01 -.33220058e+01  .26024495e+02 -.24015053e+03  .25688413e-03", 
" -.33792192e+01 -.32796237e+01  .27163858e+02 -.25987136e+03  .25540840e-03", 
" -.33660404e+01 -.32598245e+01  .30080908e+02 -.31574946e+03  .25586555e-03", 
" -.33527517e+01 -.32180830e+01  .31395063e+02 -.33826384e+03  .26191365e-03", 
" -.33393839e+01 -.31350745e+01  .30097775e+02 -.31572639e+03  .26367775e-03", 
" -.33256372e+01 -.30803211e+01  .30321192e+02 -.31833279e+03  .26716080e-03", 
" -.33116850e+01 -.30110246e+01  .29310943e+02 -.29376197e+03  .27255806e-03", 
" -.32974794e+01 -.29504820e+01  .29069925e+02 -.28592387e+03  .27519781e-03", 
" -.32831453e+01 -.28418592e+01  .25824233e+02 -.22628216e+03  .27541012e-03", 
" -.32683386e+01 -.27904952e+01  .26730037e+02 -.24797317e+03  .27637676e-03", 
" -.32532578e+01 -.27259687e+01  .26815578e+02 -.25414910e+03  .27798422e-03", 
" -.32377457e+01 -.26726697e+01  .26290703e+02 -.22981247e+03  .27604027e-03", 
" -.32219428e+01 -.26004291e+01  .25299469e+02 -.20806880e+03  .28066018e-03", 
" -.32056905e+01 -.25337760e+01  .24850286e+02 -.20102188e+03  .28341349e-03", 
" -.31889981e+01 -.24805937e+01  .25295930e+02 -.20846253e+03  .28639588e-03", 
" -.31719451e+01 -.24162711e+01  .25705709e+02 -.22296023e+03  .29177809e-03", 
" -.31542970e+01 -.23847440e+01  .27652667e+02 -.25729734e+03  .29589302e-03", 
" -.31363344e+01 -.22802559e+01  .24792363e+02 -.20859185e+03  .30090057e-03", 
" -.31176479e+01 -.22025095e+01  .23430471e+02 -.18665985e+03  .30300518e-03", 
" -.30982181e+01 -.21850835e+01  .26216975e+02 -.23558004e+03  .31153662e-03", 
" -.30782070e+01 -.21341163e+01  .26872192e+02 -.25093736e+03  .31521088e-03", 
" -.30574928e+01 -.20682832e+01  .26163127e+02 -.23761217e+03  .32751198e-03", 
" -.30359028e+01 -.20247945e+01  .26579847e+02 -.23303612e+03  .33632046e-03", 
" -.30134567e+01 -.19628464e+01  .26145898e+02 -.22046012e+03  .34054083e-03", 
" -.29897160e+01 -.19653684e+01  .29992858e+02 -.28748385e+03  .34742255e-03", 
" -.29650432e+01 -.19355065e+01  .31843525e+02 -.31928476e+03  .35457685e-03", 
" -.29389429e+01 -.18923929e+01  .32154649e+02 -.31669447e+03  .35592294e-03", 
" -.29113647e+01 -.18654861e+01  .33843921e+02 -.34275342e+03  .37612165e-03", 
" -.28819848e+01 -.18219733e+01  .33790943e+02 -.33408341e+03  .37865184e-03", 
" -.28505823e+01 -.17768439e+01  .33508857e+02 -.32023314e+03  .37707028e-03", 
" -.28168118e+01 -.17322950e+01  .33249314e+02 -.31153382e+03  .39041380e-03", 
" -.27798927e+01 -.17287585e+01  .35807710e+02 -.36267196e+03  .40650358e-03", 
" -.27393958e+01 -.17217415e+01  .36261509e+02 -.35056216e+03  .42521328e-03", 
" -.26945782e+01 -.16649324e+01  .33708018e+02 -.29587910e+03  .43786926e-03", 
" -.26439181e+01 -.16717929e+01  .34250564e+02 -.28576079e+03  .47018117e-03", 
" -.25851463e+01 -.17202940e+01  .36172898e+02 -.28699158e+03  .50426813e-03", 
" -.25147493e+01 -.18083295e+01  .36589765e+02 -.23892499e+03  .55081268e-03", 
" -.24261730e+01 -.19956065e+01  .42932530e+02 -.32152822e+03  .60352662e-03", 
" -.23046269e+01 -.21913631e+01  .36934204e+02 -.88718330e+02  .74401693e-03", 
" -.22732294e+01 -.23721154e+01  .43495241e+02 -.17459633e+03  .78177472e-03", 
" -.22382087e+01 -.26145421e+01  .54879853e+02 -.38757505e+03  .81605477e-03", 
" -.21989689e+01 -.28045463e+01  .59556176e+02 -.44453660e+03  .88461268e-03", 
" -.21543261e+01 -.29283569e+01  .58891572e+02 -.41159024e+03  .92165316e-03", 
" -.21017738e+01 -.30284468e+01  .54253750e+02 -.30426178e+03  .98842813e-03", 
" -.20380073e+01 -.30125700e+01  .33151615e+02  .15613337e+03  .10888427e-02", 
" -.19549317e+01 -.36187734e+01  .52554438e+02 -.12245642e+03  .12509829e-02", 
" -.18392208e+01 -.42919186e+01  .66264672e+02 -.28785551e+03  .15068903e-02", 
" -.16432877e+01 -.47436873e+01  .34583060e+02  .38929969e+03  .21096898e-02", 
" -.14488036e+01 -.56999386e+01  .40366655e+02  .42157890e+03  .29863042e-02", 
" -.11949044e+01 -.72295917e+01  .71597333e+02 -.17131842e+03  .48906027e-02", 
" -.10075767e+01 -.94956886e+01  .17155744e+03 -.17971787e+04  .66034331e-02", 
"coctt     6 10  3   30", " -.71156668e+01 -.74190664e+02 -.13021974e+03 -.71016174e+04  .58804751e-02", 
" -.69718625e+01 -.68042360e+02 -.19897318e+03 -.38509669e+04  .42364023e-02", 
" -.67705079e+01 -.61280006e+02 -.17054585e+03 -.24165285e+04  .29576859e-02", 
" -.66106108e+01 -.54997128e+02 -.25402286e+03  .52965278e+03  .21862643e-02", 
" -.64408043e+01 -.49703472e+02 -.24360021e+03  .13486654e+04  .16497930e-02", 
" -.63385722e+01 -.47019594e+02 -.19696539e+03  .83227285e+03  .13249874e-02", 
" -.62625156e+01 -.45362671e+02 -.13743653e+03 -.18699551e+03  .12119462e-02", 
" -.62022154e+01 -.43710577e+02 -.13334090e+03  .94213234e+01  .11309058e-02", 
" -.61517136e+01 -.42459584e+02 -.12270186e+03  .30381737e+02  .10473262e-02", 
" -.61083256e+01 -.41432542e+02 -.11385040e+03  .74312467e+02  .98923923e-03", 
" -.60695285e+01 -.40843333e+02 -.81238671e+02 -.40460900e+03  .94377057e-03", 
" -.60360382e+01 -.39772589e+02 -.91877076e+02 -.10929956e+03  .91603529e-03", 
" -.60056481e+01 -.38785637e+02 -.10504089e+03  .25015436e+03  .86814263e-03", 
" -.58832095e+01 -.35878392e+02 -.88787759e+02  .28097241e+03  .73040960e-03", 
" -.57926658e+01 -.33773469e+02 -.81795246e+02  .40711428e+03  .64916242e-03", 
" -.57197833e+01 -.32331128e+02 -.59415978e+02  .14803829e+03  .59664896e-03", 
" -.56580475e+01 -.31158056e+02 -.43717613e+02 -.13112239e+02  .53997045e-03", 
" -.56045709e+01 -.30159525e+02 -.29156485e+02 -.19723052e+03  .51486645e-03", 
" -.55573868e+01 -.29229158e+02 -.20994896e+02 -.28810720e+03  .49110696e-03", 
" -.55145758e+01 -.28454556e+02 -.13347381e+02 -.35157587e+03  .45708428e-03", 
" -.54758300e+01 -.27594864e+02 -.17870244e+02 -.18875102e+03  .42789638e-03", 
" -.54401248e+01 -.26858469e+02 -.19287263e+02 -.84318269e+02  .41858341e-03", 
" -.54069001e+01 -.26197420e+02 -.19510004e+02 -.19125608e+02  .41342694e-03", 
" -.53757903e+01 -.25618099e+02 -.15886369e+02 -.57364299e+02  .40264109e-03", 
" -.53464831e+01 -.25054111e+02 -.17383603e+02  .32636999e+02  .38946875e-03", 
" -.53185659e+01 -.24605267e+02 -.12261534e+02 -.97239498e+01  .37964537e-03", 
" -.52922887e+01 -.24145916e+02 -.88736459e+01 -.48793981e+02  .37302145e-03", 
" -.52672764e+01 -.23695987e+02 -.77962196e+01 -.38984174e+02  .36656761e-03", 
" -.52435017e+01 -.23259772e+02 -.69853553e+01 -.30681637e+02  .36136314e-03", 
" -.52206723e+01 -.22807777e+02 -.93100378e+01  .31237233e+02  .35517497e-03", 
" -.51985546e+01 -.22481817e+02 -.37517956e+01 -.54787652e+02  .34188217e-03", 
" -.51774217e+01 -.22079994e+02 -.58676396e+01  .19213442e+02  .33572093e-03", 
" -.51569918e+01 -.21702242e+02 -.70403299e+01  .62584770e+02  .33323184e-03", 
" -.51371774e+01 -.21400432e+02 -.39939240e+01  .27493316e+02  .32767676e-03", 
" -.51180582e+01 -.21086349e+02 -.14212872e+01 -.22033956e+02  .32146701e-03", 
" -.50995028e+01 -.20761316e+02 -.17168705e+01 -.11685896e+01  .31742054e-03", 
" -.50814173e+01 -.20455497e+02 -.22486581e+01  .27107067e+02  .31654106e-03", 
" -.50639667e+01 -.20129076e+02 -.42238031e+01  .81047647e+02  .31293300e-03", 
" -.50469776e+01 -.19816476e+02 -.50026266e+01  .92366150e+02  .31058770e-03", 
" -.50302394e+01 -.19545712e+02 -.50303041e+01  .11342583e+03  .30865655e-03", 
" -.50139337e+01 -.19288007e+02 -.49365278e+01  .13079582e+03  .31297285e-03", 
" -.49980504e+01 -.19015449e+02 -.57842964e+01  .15662446e+03  .31671072e-03", 
" -.49823462e+01 -.18807600e+02 -.29575451e+01  .11800656e+03  .31292078e-03", 
" -.49670321e+01 -.18576873e+02 -.19403405e+01  .10703837e+03  .31094304e-03", 
" -.49520592e+01 -.18351242e+02 -.11737455e+01  .10169121e+03  .30558793e-03", 
" -.49373464e+01 -.18152164e+02  .70586728e+00  .82482706e+02  .30204543e-03", 
" -.49228965e+01 -.17948096e+02  .18685650e+01  .76680419e+02  .29784934e-03", 
" -.49087615e+01 -.17756888e+02  .41389941e+01  .41080028e+02  .29711462e-03", 
" -.48948453e+01 -.17570007e+02  .62474515e+01  .59124645e+01  .29486144e-03", 
" -.48811815e+01 -.17368636e+02  .62672621e+01  .20145836e+02  .28937378e-03", 
" -.48677899e+01 -.17163190e+02  .64717623e+01  .19297067e+02  .29183588e-03", 
" -.48545842e+01 -.16969640e+02  .66201671e+01  .29594463e+02  .29039881e-03", 
" -.48415031e+01 -.16786662e+02  .76051412e+01  .15221188e+02  .28846585e-03", 
" -.48286382e+01 -.16598948e+02  .79541905e+01  .11800850e+02  .28812105e-03", 
" -.48159257e+01 -.16439976e+02  .96713873e+01 -.12884938e+02  .28695571e-03", 
" -.48034094e+01 -.16284515e+02  .11541062e+02 -.41110770e+02  .28586734e-03", 
" -.47911211e+01 -.16104830e+02  .11292768e+02 -.26693643e+02  .28653122e-03", 
" -.47788679e+01 -.15940003e+02  .11461168e+02 -.19111222e+02  .28290713e-03", 
" -.47668581e+01 -.15772686e+02  .11798371e+02 -.20050090e+02  .27825952e-03", 
" -.47549666e+01 -.15618401e+02  .13196380e+02 -.47027017e+02  .27869135e-03", 
" -.47431188e+01 -.15473771e+02  .14007186e+02 -.52057120e+02  .27744139e-03", 
" -.47315000e+01 -.15322715e+02  .14731913e+02 -.60817083e+02  .27460648e-03", 
" -.47199988e+01 -.15178730e+02  .15669062e+02 -.71133122e+02  .27229567e-03", 
" -.47086268e+01 -.15034151e+02  .16853829e+02 -.92341378e+02  .27483364e-03", 
" -.46973941e+01 -.14886203e+02  .17372309e+02 -.97988467e+02  .27243281e-03", 
" -.46862239e+01 -.14735760e+02  .17158501e+02 -.87565962e+02  .26834666e-03", 
" -.46752205e+01 -.14567022e+02  .15161094e+02 -.35224280e+02  .26745604e-03", 
" -.46642562e+01 -.14443820e+02  .16875727e+02 -.64363104e+02  .26611695e-03", 
" -.46534522e+01 -.14281718e+02  .15483658e+02 -.34048663e+02  .26488385e-03", 
" -.46427357e+01 -.14137826e+02  .15194249e+02 -.24544432e+02  .26099526e-03", 
" -.46320343e+01 -.14018641e+02  .16781313e+02 -.53952936e+02  .26050841e-03", 
" -.46214284e+01 -.13875166e+02  .15762473e+02 -.23292724e+02  .26144480e-03", 
" -.46109369e+01 -.13744252e+02  .16093068e+02 -.27294843e+02  .25775322e-03", 
" -.46005222e+01 -.13611657e+02  .15883161e+02 -.17260008e+02  .25908728e-03", 
" -.45901467e+01 -.13495063e+02  .17054187e+02 -.38167388e+02  .26099608e-03", 
" -.45798297e+01 -.13384392e+02  .18548447e+02 -.65612826e+02  .25994870e-03", 
" -.45696133e+01 -.13268447e+02  .19560138e+02 -.82914288e+02  .25777998e-03", 
" -.45594159e+01 -.13143744e+02  .19138512e+02 -.65055126e+02  .25779466e-03", 
" -.45493401e+01 -.13026561e+02  .19867466e+02 -.78462713e+02  .25830183e-03", 
" -.45393635e+01 -.12892684e+02  .19546648e+02 -.77542642e+02  .25800270e-03", 
" -.45293996e+01 -.12785866e+02  .20768526e+02 -.10038844e+03  .25706426e-03", 
" -.45194756e+01 -.12663962e+02  .20767622e+02 -.10212801e+03  .25866942e-03", 
" -.45096467e+01 -.12551736e+02  .21083317e+02 -.10318551e+03  .25561611e-03", 
" -.44999235e+01 -.12414781e+02  .19672557e+02 -.73861593e+02  .25496243e-03", 
" -.44902364e+01 -.12295414e+02  .19644946e+02 -.71904989e+02  .25237962e-03", 
" -.44804966e+01 -.12194840e+02  .20520170e+02 -.84093398e+02  .24992820e-03", 
" -.44709036e+01 -.12060677e+02  .19269243e+02 -.62404211e+02  .24733587e-03", 
" -.44613676e+01 -.11938111e+02  .18778213e+02 -.53823343e+02  .24398206e-03", 
" -.44518097e+01 -.11829665e+02  .19266807e+02 -.65024389e+02  .24531986e-03", 
" -.44423073e+01 -.11711725e+02  .18697276e+02 -.54426891e+02  .24632597e-03", 
" -.44328121e+01 -.11602325e+02  .18460842e+02 -.44697325e+02  .24706766e-03", 
" -.44233238e+01 -.11514714e+02  .19828620e+02 -.66344851e+02  .24753803e-03", 
" -.44139161e+01 -.11404399e+02  .19508529e+02 -.58040932e+02  .24946757e-03", 
" -.44045296e+01 -.11294449e+02  .19291407e+02 -.54379548e+02  .24780570e-03", 
" -.43951634e+01 -.11191124e+02  .19111580e+02 -.46358163e+02  .24823238e-03", 
" -.43858575e+01 -.11076731e+02  .18265933e+02 -.28242410e+02  .24770484e-03", 
" -.43764944e+01 -.10994937e+02  .19380974e+02 -.43921046e+02  .24366094e-03", 
" -.43671985e+01 -.10905092e+02  .20264177e+02 -.60901337e+02  .24591841e-03", 
" -.43579459e+01 -.10808930e+02  .20588121e+02 -.66814613e+02  .24648239e-03", 
" -.43487741e+01 -.10688310e+02  .19320405e+02 -.44978710e+02  .24601367e-03", 
" -.43395523e+01 -.10597938e+02  .20005106e+02 -.55601989e+02  .24602292e-03", 
" -.43303145e+01 -.10507191e+02  .20404229e+02 -.61178089e+02  .24576983e-03", 
" -.43211539e+01 -.10409752e+02  .20443797e+02 -.61089699e+02  .24395362e-03", 
" -.43119890e+01 -.10324716e+02  .21061915e+02 -.67086588e+02  .24476164e-03", 
" -.43028985e+01 -.10225538e+02  .21183766e+02 -.71218894e+02  .24191094e-03", 
" -.42937255e+01 -.10154623e+02  .23144138e+02 -.11102046e+03  .24096130e-03", 
" -.42845954e+01 -.10085922e+02  .25588584e+02 -.16395498e+03  .23921482e-03", 
" -.42754784e+01 -.99900400e+01  .25030358e+02 -.14973290e+03  .23800994e-03", 
" -.42663681e+01 -.99130444e+01  .26157527e+02 -.16981364e+03  .24033686e-03", 
" -.42572910e+01 -.98216390e+01  .26290651e+02 -.17073255e+03  .24089381e-03", 
" -.42481776e+01 -.97386955e+01  .26806118e+02 -.17820965e+03  .23741896e-03", 
" -.42390905e+01 -.96420055e+01  .26343853e+02 -.16833689e+03  .23665247e-03", 
" -.42300421e+01 -.95278985e+01  .24284069e+02 -.12379897e+03  .23939837e-03", 
" -.42209545e+01 -.94444026e+01  .24698286e+02 -.13105975e+03  .23693205e-03", 
" -.42119204e+01 -.93496603e+01  .24529011e+02 -.13013440e+03  .23506196e-03", 
" -.42028165e+01 -.92703838e+01  .25485635e+02 -.15100826e+03  .23440633e-03", 
" -.41937654e+01 -.91717797e+01  .24825712e+02 -.14090354e+03  .23340949e-03", 
" -.41846471e+01 -.90886641e+01  .25125118e+02 -.14600101e+03  .23296161e-03", 
" -.41754952e+01 -.90222914e+01  .26704418e+02 -.17853042e+03  .23162030e-03", 
" -.41663981e+01 -.89306705e+01  .26181636e+02 -.16604221e+03  .23317751e-03", 
" -.41573058e+01 -.88328622e+01  .25398058e+02 -.15118892e+03  .23554065e-03", 
" -.41481687e+01 -.87551994e+01  .25829530e+02 -.15639248e+03  .23456309e-03", 
" -.41390682e+01 -.86533001e+01  .24530463e+02 -.13040961e+03  .23213981e-03", 
" -.41299223e+01 -.85672874e+01  .24442068e+02 -.12899919e+03  .23355010e-03", 
" -.41207685e+01 -.84817072e+01  .24268103e+02 -.12343361e+03  .23296879e-03", 
" -.41115820e+01 -.83992215e+01  .24493057e+02 -.13237722e+03  .23202267e-03", 
" -.41024117e+01 -.83051411e+01  .23727806e+02 -.11606742e+03  .23197834e-03", 
" -.40931143e+01 -.82442551e+01  .25183154e+02 -.14421969e+03  .23171196e-03", 
" -.40838335e+01 -.81732432e+01  .26200035e+02 -.16618430e+03  .23113409e-03", 
" -.40745491e+01 -.81022688e+01  .26962534e+02 -.18027043e+03  .23282600e-03", 
" -.40652361e+01 -.80244798e+01  .26980860e+02 -.17715424e+03  .23274889e-03", 
" -.40559171e+01 -.79367897e+01  .26336931e+02 -.16324996e+03  .23107166e-03", 
" -.40466150e+01 -.78376844e+01  .24976470e+02 -.13728734e+03  .22966336e-03", 
" -.40372563e+01 -.77438651e+01  .23984326e+02 -.11887538e+03  .23189164e-03", 
" -.40277660e+01 -.76926223e+01  .25852068e+02 -.15178125e+03  .23190509e-03", 
" -.40183254e+01 -.75937932e+01  .24191573e+02 -.11959022e+03  .23395966e-03", 
" -.40088639e+01 -.75027042e+01  .23331412e+02 -.10388678e+03  .23375371e-03", 
" -.39992699e+01 -.74457826e+01  .24453560e+02 -.11779322e+03  .23470973e-03", 
" -.39896570e+01 -.73876684e+01  .25739073e+02 -.13973523e+03  .23358098e-03", 
" -.39799999e+01 -.73230392e+01  .26580230e+02 -.15798357e+03  .23376922e-03", 
" -.39703360e+01 -.72572054e+01  .27480805e+02 -.17552749e+03  .23322850e-03", 
" -.39606315e+01 -.71937373e+01  .28367152e+02 -.18968219e+03  .23534395e-03", 
" -.39509221e+01 -.71122697e+01  .27913093e+02 -.17822904e+03  .23531474e-03", 
" -.39411392e+01 -.70278332e+01  .26976208e+02 -.15506007e+03  .23495977e-03", 
" -.39312803e+01 -.69582965e+01  .27345204e+02 -.15927511e+03  .23576089e-03", 
" -.39214031e+01 -.68738987e+01  .26644356e+02 -.14499634e+03  .23806507e-03", 
" -.39114066e+01 -.68140219e+01  .27395118e+02 -.15382190e+03  .24090322e-03", 
" -.39014215e+01 -.67396462e+01  .27380418e+02 -.15093600e+03  .24185973e-03", 
" -.38913399e+01 -.66622761e+01  .26912188e+02 -.13889152e+03  .24140705e-03", 
" -.38811846e+01 -.66018919e+01  .27798912e+02 -.15224712e+03  .24109815e-03", 
" -.38709670e+01 -.65273626e+01  .27734379e+02 -.15023940e+03  .24351959e-03", 
" -.38605923e+01 -.64686502e+01  .28526041e+02 -.16141765e+03  .24480090e-03", 
" -.38502166e+01 -.63983146e+01  .28562473e+02 -.15898191e+03  .24376501e-03", 
" -.38397845e+01 -.63376420e+01  .29861301e+02 -.18469013e+03  .24322376e-03", 
" -.38292877e+01 -.62538987e+01  .29015833e+02 -.16698998e+03  .24427130e-03", 
" -.38186191e+01 -.62026759e+01  .30763140e+02 -.20301053e+03  .24237051e-03", 
" -.38079214e+01 -.61253390e+01  .30443145e+02 -.19518732e+03  .24160424e-03", 
" -.37971331e+01 -.60430488e+01  .29756894e+02 -.18157083e+03  .24528821e-03", 
" -.37862506e+01 -.59681291e+01  .29681067e+02 -.18134950e+03  .25059773e-03", 
" -.37752957e+01 -.58874036e+01  .29340185e+02 -.17577669e+03  .25304409e-03", 
" -.37641420e+01 -.58256697e+01  .30295721e+02 -.19734497e+03  .25267395e-03", 
" -.37529454e+01 -.57357442e+01  .28804468e+02 -.16367040e+03  .25449834e-03", 
" -.37416473e+01 -.56634110e+01  .29346687e+02 -.17915061e+03  .25638478e-03", 
" -.37301579e+01 -.55834820e+01  .28519473e+02 -.16100767e+03  .26032253e-03", 
" -.37185517e+01 -.55175562e+01  .28704758e+02 -.15946800e+03  .25496344e-03", 
" -.37067274e+01 -.54795796e+01  .31342353e+02 -.21078865e+03  .25525059e-03", 
" -.36947896e+01 -.54253807e+01  .32646490e+02 -.23521358e+03  .25720807e-03", 
" -.36827347e+01 -.53638080e+01  .33054046e+02 -.23671454e+03  .25896948e-03", 
" -.36706079e+01 -.52749170e+01  .32030688e+02 -.21889226e+03  .25957647e-03", 
" -.36582673e+01 -.51992785e+01  .31798102e+02 -.21289083e+03  .26056923e-03", 
" -.36457132e+01 -.51394618e+01  .32466325e+02 -.22095345e+03  .26077953e-03", 
" -.36330345e+01 -.50352826e+01  .29404367e+02 -.15296960e+03  .26393948e-03", 
" -.36200904e+01 -.49633998e+01  .29315234e+02 -.15097427e+03  .26652017e-03", 
" -.36068777e+01 -.49186180e+01  .31760151e+02 -.20481095e+03  .26568464e-03", 
" -.35934630e+01 -.48773402e+01  .34533673e+02 -.26639718e+03  .26989229e-03", 
" -.35798700e+01 -.48139740e+01  .35221096e+02 -.28011437e+03  .27108091e-03", 
" -.35660360e+01 -.47426761e+01  .35683609e+02 -.29450066e+03  .26841464e-03", 
" -.35519826e+01 -.46774026e+01  .36351804e+02 -.30430016e+03  .26979127e-03", 
" -.35377057e+01 -.45733839e+01  .33958221e+02 -.25764916e+03  .27452988e-03", 
" -.35230349e+01 -.44988588e+01  .33350477e+02 -.24192021e+03  .27625508e-03", 
" -.35080117e+01 -.44316727e+01  .33309360e+02 -.23791889e+03  .28028405e-03", 
" -.34926113e+01 -.44001470e+01  .35836672e+02 -.27950485e+03  .27969597e-03", 
" -.34768612e+01 -.43523509e+01  .37440032e+02 -.31048434e+03  .28301588e-03", 
" -.34608070e+01 -.42704573e+01  .36384342e+02 -.28720754e+03  .28777871e-03", 
" -.34444256e+01 -.41911920e+01  .35983167e+02 -.27842129e+03  .28912245e-03", 
" -.34275891e+01 -.40824775e+01  .32791661e+02 -.21175355e+03  .29514062e-03", 
" -.34102195e+01 -.40067513e+01  .32345438e+02 -.20011288e+03  .29607026e-03", 
" -.33921317e+01 -.39890513e+01  .35775379e+02 -.26423107e+03  .29820769e-03", 
" -.33736975e+01 -.39182934e+01  .35072305e+02 -.24031091e+03  .29964477e-03", 
" -.33545978e+01 -.38774245e+01  .37533284e+02 -.28800174e+03  .30420512e-03", 
" -.33347772e+01 -.38115355e+01  .37288140e+02 -.28145688e+03  .31197960e-03", 
" -.33141595e+01 -.37688234e+01  .38366731e+02 -.29840374e+03  .31623740e-03", 
" -.32928659e+01 -.37058902e+01  .38347342e+02 -.29161961e+03  .32033746e-03", 
" -.32706782e+01 -.36588612e+01  .39718454e+02 -.31359890e+03  .32520169e-03", 
" -.32475092e+01 -.36164213e+01  .40929429e+02 -.32197877e+03  .33536755e-03", 
" -.32231115e+01 -.35814162e+01  .43270396e+02 -.36476631e+03  .34558611e-03", 
" -.31974560e+01 -.35437407e+01  .45499993e+02 -.40841357e+03  .35296159e-03", 
" -.31700616e+01 -.35859565e+01  .52076403e+02 -.51837624e+03  .35732972e-03", 
" -.31410648e+01 -.35581657e+01  .52943833e+02 -.51429613e+03  .36636568e-03", 
" -.31102291e+01 -.34968730e+01  .52012297e+02 -.48543905e+03  .37296757e-03", 
" -.30769928e+01 -.34841042e+01  .54387651e+02 -.51133801e+03  .38866200e-03", 
" -.30409730e+01 -.34480819e+01  .54655597e+02 -.49962721e+03  .40507384e-03", 
" -.30012072e+01 -.35260923e+01  .61786925e+02 -.60477979e+03  .41182363e-03", 
" -.29571721e+01 -.35961181e+01  .68161250e+02 -.70820628e+03  .43549679e-03", 
" -.29075876e+01 -.35921527e+01  .66822972e+02 -.63409566e+03  .47231920e-03", 
" -.28499703e+01 -.37433967e+01  .73061346e+02 -.67092012e+03  .49127005e-03", 
" -.27812376e+01 -.40055224e+01  .87429121e+02 -.90140323e+03  .54185787e-03", 
" -.26948871e+01 -.43508131e+01  .99974330e+02 -.10503276e+04  .59540315e-03", 
" -.25776490e+01 -.47699069e+01  .10592548e+03 -.10244099e+04  .67049960e-03", 
" -.25475976e+01 -.49175748e+01  .10795600e+03 -.10098054e+04  .70441174e-03", 
" -.25142580e+01 -.51794807e+01  .11909774e+03 -.11992552e+04  .73612279e-03", 
" -.24769612e+01 -.53571239e+01  .11949432e+03 -.11458707e+04  .76044840e-03", 
" -.24342452e+01 -.55662301e+01  .12023680e+03 -.11067893e+04  .82655039e-03", 
" -.23839730e+01 -.60429380e+01  .13789396e+03 -.13728587e+04  .90642426e-03", 
" -.23229726e+01 -.64486792e+01  .14512573e+03 -.14583794e+04  .98740210e-03", 
" -.22446373e+01 -.73618992e+01  .17700504e+03 -.19038819e+04  .11813587e-02", 
" -.21354628e+01 -.84350722e+01  .19554700e+03 -.20143043e+04  .14779119e-02", 
" -.19494558e+01 -.10577066e+02  .23663302e+03 -.23869363e+04  .19115828e-02", 
" -.17644164e+01 -.13384642e+02  .31908290e+03 -.35759818e+04  .27728413e-02", 
" -.15188862e+01 -.17757674e+02  .44822769e+03 -.53254318e+04  .40572094e-02", 
" -.13437432e+01 -.20037569e+02  .52084160e+03 -.67279514e+04  .55889380e-02", 
"coanc     6  7  3   35", " -.81897261e+02  .19470433e+04 -.31846775e+05  .27578164e+06  .11229577e+00", 
" -.78162731e+02  .17823918e+04 -.27601203e+05  .22014679e+06  .85304231e-01", 
" -.73150500e+02  .15984331e+04 -.24970260e+05  .21305318e+06  .54420554e-01", 
" -.69224259e+02  .14372114e+04 -.21130910e+05  .16753077e+06  .40544839e-01", 
" -.65283624e+02  .13008736e+04 -.18806361e+05  .14913797e+06  .27933468e-01", 
" -.62908712e+02  .12175738e+04 -.17371813e+05  .14012290e+06  .24667404e-01", 
" -.61192081e+02  .11566366e+04 -.16095150e+05  .12577781e+06  .21522836e-01", 
" -.59835718e+02  .11039124e+04 -.14685274e+05  .10678262e+06  .19055082e-01", 
" -.58733125e+02  .10698613e+04 -.14264666e+05  .10679430e+06  .17591648e-01", 
" -.57784086e+02  .10381098e+04 -.13615854e+05  .99525731e+05  .16667548e-01", 
" -.56959702e+02  .10105524e+04 -.13017474e+05  .92216957e+05  .15497621e-01", 
" -.56234303e+02  .98949450e+03 -.12783583e+05  .92385956e+05  .14952011e-01", 
" -.55573100e+02  .96724766e+03 -.12290994e+05  .86437883e+05  .14562595e-01", 
" -.52993229e+02  .88687939e+03 -.10854813e+05  .73569411e+05  .12099268e-01", 
" -.51124679e+02  .83103251e+03 -.99067915e+04  .65774556e+05  .11226604e-01", 
" -.49658389e+02  .79133281e+03 -.94266222e+04  .64705680e+05  .10126366e-01", 
" -.48430535e+02  .75666487e+03 -.88509003e+04  .59604856e+05  .90609415e-02", 
" -.47379156e+02  .72760467e+03 -.83924908e+04  .56058861e+05  .85435106e-02", 
" -.46460799e+02  .70277793e+03 -.79884271e+04  .52505690e+05  .79723165e-02", 
" -.45636782e+02  .68016707e+03 -.75711049e+04  .47713966e+05  .76233265e-02", 
" -.44893557e+02  .66062228e+03 -.72583873e+04  .44837580e+05  .71198410e-02", 
" -.44216562e+02  .64382461e+03 -.70373557e+04  .43373654e+05  .67470156e-02", 
" -.43588772e+02  .62824641e+03 -.68640719e+04  .43299435e+05  .66022572e-02", 
" -.43005647e+02  .61322158e+03 -.66378574e+04  .41716208e+05  .63750664e-02", 
" -.42463392e+02  .59984294e+03 -.64490271e+04  .40365418e+05  .60448015e-02", 
" -.41951597e+02  .58707986e+03 -.62554420e+04  .38649032e+05  .59052097e-02", 
" -.41472149e+02  .57547353e+03 -.60902429e+04  .37492927e+05  .56783314e-02", 
" -.41016072e+02  .56447416e+03 -.59353902e+04  .36372261e+05  .55351680e-02", 
" -.40582019e+02  .55385963e+03 -.57678820e+04  .34713225e+05  .55016461e-02", 
" -.40169356e+02  .54394683e+03 -.56336263e+04  .33994946e+05  .54156735e-02", 
" -.39776150e+02  .53482015e+03 -.55125507e+04  .33225958e+05  .52480601e-02", 
" -.39397243e+02  .52561753e+03 -.53578544e+04  .31605682e+05  .51642185e-02", 
" -.39035399e+02  .51771909e+03 -.52919914e+04  .31948898e+05  .52039575e-02", 
" -.38685549e+02  .50953896e+03 -.51684975e+04  .30819809e+05  .51576598e-02", 
" -.38347564e+02  .50143844e+03 -.50345037e+04  .29507831e+05  .51004449e-02", 
" -.38023145e+02  .49403606e+03 -.49238035e+04  .28461531e+05  .49950400e-02", 
" -.37708137e+02  .48696442e+03 -.48268390e+04  .27627361e+05  .49989564e-02", 
" -.37401109e+02  .47976542e+03 -.47153341e+04  .26616081e+05  .49328448e-02", 
" -.37105723e+02  .47304375e+03 -.46171090e+04  .25859386e+05  .48103209e-02", 
" -.36820971e+02  .46729923e+03 -.45711118e+04  .25951188e+05  .47453937e-02", 
" -.36541754e+02  .46135988e+03 -.45048702e+04  .25651453e+05  .47015627e-02", 
" -.36269692e+02  .45510018e+03 -.44010140e+04  .24566584e+05  .45965711e-02", 
" -.36004836e+02  .44929502e+03 -.43160556e+04  .23743080e+05  .45022198e-02", 
" -.35747386e+02  .44411187e+03 -.42797233e+04  .24095087e+05  .44543901e-02", 
" -.35494133e+02  .43842335e+03 -.41976547e+04  .23513651e+05  .44087483e-02", 
" -.35247886e+02  .43329186e+03 -.41424624e+04  .23331207e+05  .43629781e-02", 
" -.35006648e+02  .42823243e+03 -.40893578e+04  .23289521e+05  .43490065e-02", 
" -.34771260e+02  .42324512e+03 -.40279068e+04  .22956816e+05  .43423148e-02", 
" -.34540270e+02  .41828260e+03 -.39575834e+04  .22378076e+05  .42685181e-02", 
" -.34314834e+02  .41405907e+03 -.39330855e+04  .22645999e+05  .42450522e-02", 
" -.34092802e+02  .40939563e+03 -.38745592e+04  .22303056e+05  .42912685e-02", 
" -.33875004e+02  .40474886e+03 -.38099325e+04  .21797912e+05  .42632174e-02", 
" -.33661623e+02  .40025850e+03 -.37493341e+04  .21369225e+05  .42324771e-02", 
" -.33452178e+02  .39607763e+03 -.37128998e+04  .21491961e+05  .41869555e-02", 
" -.33245770e+02  .39190250e+03 -.36689894e+04  .21428095e+05  .41258006e-02", 
" -.33042420e+02  .38765566e+03 -.36178132e+04  .21256908e+05  .40337160e-02", 
" -.32843309e+02  .38357114e+03 -.35591132e+04  .20721333e+05  .40892648e-02", 
" -.32647280e+02  .37973420e+03 -.35190381e+04  .20556414e+05  .40068303e-02", 
" -.32455790e+02  .37626701e+03 -.34900251e+04  .20467873e+05  .39830505e-02", 
" -.32265714e+02  .37276846e+03 -.34706699e+04  .20774946e+05  .39508250e-02", 
" -.32078652e+02  .36930443e+03 -.34476687e+04  .20958818e+05  .38731851e-02", 
" -.31893365e+02  .36554525e+03 -.34001193e+04  .20709221e+05  .38654415e-02", 
" -.31709777e+02  .36156051e+03 -.33282660e+04  .19844107e+05  .38644142e-02", 
" -.31529699e+02  .35813138e+03 -.32971379e+04  .19861498e+05  .38394576e-02", 
" -.31352162e+02  .35474761e+03 -.32603726e+04  .19629479e+05  .37729543e-02", 
" -.31177161e+02  .35154423e+03 -.32406931e+04  .19891538e+05  .37343737e-02", 
" -.31003182e+02  .34776229e+03 -.31744599e+04  .19218950e+05  .37250918e-02", 
" -.30832397e+02  .34448881e+03 -.31357772e+04  .18970403e+05  .37640756e-02", 
" -.30664822e+02  .34173233e+03 -.31306835e+04  .19382742e+05  .37868195e-02", 
" -.30498051e+02  .33841860e+03 -.30788234e+04  .18816700e+05  .37782683e-02", 
" -.30333138e+02  .33523788e+03 -.30380109e+04  .18492797e+05  .37072328e-02", 
" -.30169090e+02  .33189194e+03 -.29732231e+04  .17539108e+05  .36642625e-02", 
" -.30006966e+02  .32868689e+03 -.29261203e+04  .17104519e+05  .36162615e-02", 
" -.29847048e+02  .32575702e+03 -.29007885e+04  .17151077e+05  .35465155e-02", 
" -.29688182e+02  .32271215e+03 -.28625437e+04  .16880359e+05  .35054729e-02", 
" -.29531226e+02  .31978932e+03 -.28303391e+04  .16746160e+05  .34714703e-02", 
" -.29376360e+02  .31683145e+03 -.27853521e+04  .16214339e+05  .34803622e-02", 
" -.29222484e+02  .31384840e+03 -.27325754e+04  .15443363e+05  .34278632e-02", 
" -.29070145e+02  .31119661e+03 -.27177588e+04  .15702116e+05  .34250877e-02", 
" -.28919859e+02  .30873042e+03 -.27125839e+04  .16115471e+05  .34810626e-02", 
" -.28769883e+02  .30604734e+03 -.26885113e+04  .16114996e+05  .34786676e-02", 
" -.28620585e+02  .30318106e+03 -.26498848e+04  .15816261e+05  .34615404e-02", 
" -.28473255e+02  .30038472e+03 -.26122131e+04  .15529034e+05  .34278756e-02", 
" -.28327280e+02  .29758802e+03 -.25691155e+04  .15104148e+05  .34310749e-02", 
" -.28182461e+02  .29500089e+03 -.25382537e+04  .14848832e+05  .34410853e-02", 
" -.28038381e+02  .29239063e+03 -.25101330e+04  .14748584e+05  .34150222e-02", 
" -.27895704e+02  .28988554e+03 -.24825328e+04  .14556116e+05  .34115207e-02", 
" -.27754419e+02  .28737091e+03 -.24515645e+04  .14289680e+05  .33603975e-02", 
" -.27613927e+02  .28483112e+03 -.24178550e+04  .13991669e+05  .33567932e-02", 
" -.27474511e+02  .28242910e+03 -.23895968e+04  .13743104e+05  .33434484e-02", 
" -.27336584e+02  .28025818e+03 -.23816564e+04  .13965466e+05  .33441491e-02", 
" -.27199336e+02  .27804620e+03 -.23678386e+04  .14039558e+05  .33331908e-02", 
" -.27062726e+02  .27589092e+03 -.23613641e+04  .14337788e+05  .32964260e-02", 
" -.26927147e+02  .27362025e+03 -.23402618e+04  .14281621e+05  .32985126e-02", 
" -.26791807e+02  .27116559e+03 -.23084951e+04  .14071538e+05  .32748892e-02", 
" -.26656760e+02  .26859098e+03 -.22600298e+04  .13383806e+05  .32719315e-02", 
" -.26523552e+02  .26641292e+03 -.22432110e+04  .13414475e+05  .32285292e-02", 
" -.26390443e+02  .26422649e+03 -.22281060e+04  .13491333e+05  .32310624e-02", 
" -.26257668e+02  .26174018e+03 -.21847084e+04  .12966594e+05  .31861529e-02", 
" -.26126285e+02  .25940268e+03 -.21496096e+04  .12608706e+05  .31294738e-02", 
" -.25996124e+02  .25766402e+03 -.21672147e+04  .13395564e+05  .31445131e-02", 
" -.25865964e+02  .25561245e+03 -.21543051e+04  .13465185e+05  .31294088e-02", 
" -.25736576e+02  .25355540e+03 -.21410055e+04  .13553658e+05  .31206298e-02", 
" -.25607016e+02  .25115630e+03 -.20983833e+04  .13002756e+05  .31052173e-02", 
" -.25478696e+02  .24899509e+03 -.20669432e+04  .12596156e+05  .31178417e-02", 
" -.25350309e+02  .24674654e+03 -.20353644e+04  .12279407e+05  .31164493e-02", 
" -.25222923e+02  .24483897e+03 -.20287671e+04  .12470447e+05  .30976771e-02", 
" -.25095120e+02  .24257196e+03 -.19952419e+04  .12106487e+05  .30769298e-02", 
" -.24968501e+02  .24040904e+03 -.19655882e+04  .11806625e+05  .30334470e-02", 
" -.24842124e+02  .23838262e+03 -.19444245e+04  .11649750e+05  .30376451e-02", 
" -.24716726e+02  .23646944e+03 -.19331686e+04  .11726215e+05  .30560469e-02", 
" -.24591396e+02  .23462292e+03 -.19308190e+04  .12051396e+05  .30645762e-02", 
" -.24466459e+02  .23260294e+03 -.19068314e+04  .11827191e+05  .30421523e-02", 
" -.24340519e+02  .23025987e+03 -.18639728e+04  .11291861e+05  .30243263e-02", 
" -.24216118e+02  .22829846e+03 -.18472800e+04  .11277080e+05  .30324474e-02", 
" -.24091624e+02  .22621636e+03 -.18255268e+04  .11225627e+05  .30279753e-02", 
" -.23967805e+02  .22397765e+03 -.17791957e+04  .10500493e+05  .29718379e-02", 
" -.23844961e+02  .22219173e+03 -.17704702e+04  .10601303e+05  .29574168e-02", 
" -.23722220e+02  .22032845e+03 -.17548976e+04  .10552775e+05  .29330735e-02", 
" -.23599523e+02  .21845952e+03 -.17404455e+04  .10558964e+05  .29575943e-02", 
" -.23476503e+02  .21638314e+03 -.17041760e+04  .10031962e+05  .29742468e-02", 
" -.23354292e+02  .21437892e+03 -.16752907e+04  .97110989e+04  .29583465e-02", 
" -.23232538e+02  .21251544e+03 -.16576975e+04  .96477803e+04  .29757108e-02", 
" -.23110467e+02  .21043466e+03 -.16259112e+04  .93471863e+04  .29847508e-02", 
" -.22988857e+02  .20858225e+03 -.16083516e+04  .92393911e+04  .29791003e-02", 
" -.22867009e+02  .20659712e+03 -.15801504e+04  .89495942e+04  .29365805e-02", 
" -.22746165e+02  .20497447e+03 -.15773025e+04  .91386190e+04  .29496339e-02", 
" -.22624858e+02  .20311886e+03 -.15562700e+04  .89689920e+04  .29490962e-02", 
" -.22503569e+02  .20134256e+03 -.15425947e+04  .89415248e+04  .29392286e-02", 
" -.22382208e+02  .19958167e+03 -.15257327e+04  .87851736e+04  .29197357e-02", 
" -.22260541e+02  .19767062e+03 -.15011585e+04  .85274979e+04  .29202923e-02", 
" -.22139627e+02  .19599067e+03 -.14958502e+04  .87157671e+04  .28881306e-02", 
" -.22017861e+02  .19405500e+03 -.14706734e+04  .84987896e+04  .28552449e-02", 
" -.21896443e+02  .19223065e+03 -.14540566e+04  .84459556e+04  .28349032e-02", 
" -.21775333e+02  .19059444e+03 -.14527931e+04  .87438649e+04  .28649090e-02", 
" -.21653546e+02  .18861814e+03 -.14209916e+04  .83583537e+04  .28822712e-02", 
" -.21532343e+02  .18692026e+03 -.14086711e+04  .83324088e+04  .28633396e-02", 
" -.21410317e+02  .18498158e+03 -.13800569e+04  .80363430e+04  .28415689e-02", 
" -.21288535e+02  .18314162e+03 -.13544677e+04  .77238446e+04  .28388731e-02", 
" -.21166519e+02  .18134618e+03 -.13338498e+04  .75109128e+04  .28359054e-02", 
" -.21045243e+02  .17988569e+03 -.13439496e+04  .80462683e+04  .27785536e-02", 
" -.20923364e+02  .17826113e+03 -.13344578e+04  .80544216e+04  .27370018e-02", 
" -.20800349e+02  .17642754e+03 -.13158055e+04  .79639531e+04  .27435447e-02", 
" -.20677863e+02  .17475420e+03 -.13091036e+04  .81358606e+04  .27252426e-02", 
" -.20555208e+02  .17294087e+03 -.12834981e+04  .78036659e+04  .27477340e-02", 
" -.20432677e+02  .17139344e+03 -.12828871e+04  .80621863e+04  .27602772e-02", 
" -.20309263e+02  .16959914e+03 -.12601203e+04  .78016186e+04  .27532159e-02", 
" -.20185433e+02  .16786561e+03 -.12431964e+04  .76594335e+04  .27673392e-02", 
" -.20061002e+02  .16607455e+03 -.12211016e+04  .74225401e+04  .27676691e-02", 
" -.19935820e+02  .16406489e+03 -.11867040e+04  .69983281e+04  .27755321e-02", 
" -.19810472e+02  .16225747e+03 -.11678799e+04  .68929967e+04  .28239040e-02", 
" -.19685534e+02  .16053335e+03 -.11532294e+04  .68911316e+04  .28315874e-02", 
" -.19559797e+02  .15888912e+03 -.11460988e+04  .70302157e+04  .28260602e-02", 
" -.19433141e+02  .15719484e+03 -.11397057e+04  .72501951e+04  .27959118e-02", 
" -.19306103e+02  .15548189e+03 -.11280318e+04  .72912791e+04  .28253214e-02", 
" -.19178883e+02  .15376743e+03 -.11096536e+04  .70939719e+04  .28161469e-02", 
" -.19050738e+02  .15205472e+03 -.10985645e+04  .71742342e+04  .28429802e-02", 
" -.18921982e+02  .15021274e+03 -.10697947e+04  .67426649e+04  .28638482e-02", 
" -.18792522e+02  .14841427e+03 -.10428304e+04  .63400879e+04  .28723641e-02", 
" -.18662060e+02  .14651236e+03 -.10128301e+04  .59424603e+04  .28565823e-02", 
" -.18530082e+02  .14440611e+03 -.96874719e+03  .52620007e+04  .28488386e-02", 
" -.18398141e+02  .14249413e+03 -.94053491e+03  .49245154e+04  .28727366e-02", 
" -.18266116e+02  .14084820e+03 -.93163157e+03  .49886593e+04  .28828017e-02", 
" -.18131853e+02  .13882493e+03 -.89323132e+03  .43983631e+04  .28599394e-02", 
" -.17997862e+02  .13715688e+03 -.87589500e+03  .41513106e+04  .28391207e-02", 
" -.17861142e+02  .13497716e+03 -.83035105e+03  .34910247e+04  .28339776e-02", 
" -.17724491e+02  .13341266e+03 -.83459268e+03  .38677446e+04  .28575231e-02", 
" -.17585658e+02  .13110460e+03 -.77643527e+03  .29128107e+04  .28629599e-02", 
" -.17447458e+02  .12958583e+03 -.78197074e+03  .33088220e+04  .28394242e-02", 
" -.17306822e+02  .12751645e+03 -.74286452e+03  .27471752e+04  .28826280e-02", 
" -.17165032e+02  .12588905e+03 -.74653861e+03  .31673975e+04  .28347782e-02", 
" -.17021827e+02  .12403034e+03 -.72853570e+03  .30761421e+04  .28573961e-02", 
" -.16877619e+02  .12252668e+03 -.73955732e+03  .35741557e+04  .28836722e-02", 
" -.16732379e+02  .12088118e+03 -.73421735e+03  .36692370e+04  .29116975e-02", 
" -.16585356e+02  .11922831e+03 -.73115362e+03  .38505187e+04  .29409711e-02", 
" -.16435790e+02  .11725606e+03 -.70774907e+03  .36488901e+04  .29774518e-02", 
" -.16283404e+02  .11496933e+03 -.65695621e+03  .28057020e+04  .30252433e-02", 
" -.16130587e+02  .11321361e+03 -.65027094e+03  .29329346e+04  .30742939e-02", 
" -.15974235e+02  .11097262e+03 -.61127450e+03  .24703761e+04  .30840160e-02", 
" -.15816390e+02  .10899113e+03 -.60014508e+03  .26957143e+04  .30614400e-02", 
" -.15656309e+02  .10726206e+03 -.60198158e+03  .30097954e+04  .30524862e-02", 
" -.15493878e+02  .10527738e+03 -.58004650e+03  .28043122e+04  .30251250e-02", 
" -.15329768e+02  .10383128e+03 -.60186907e+03  .35081828e+04  .30210611e-02", 
" -.15161543e+02  .10188444e+03 -.58485063e+03  .33796622e+04  .30189122e-02", 
" -.14990338e+02  .10018505e+03 -.59444045e+03  .39008766e+04  .30726216e-02", 
" -.14814985e+02  .97867221e+02 -.55337553e+03  .33044269e+04  .30995401e-02", 
" -.14636229e+02  .95711314e+02 -.52549080e+03  .29780830e+04  .31195132e-02", 
" -.14452485e+02  .93378342e+02 -.49188826e+03  .25694216e+04  .31042318e-02", 
" -.14265846e+02  .91262664e+02 -.47672067e+03  .25809021e+04  .30990217e-02", 
" -.14073663e+02  .88816215e+02 -.43246318e+03  .18761986e+04  .31044551e-02", 
" -.13877000e+02  .86660142e+02 -.41689096e+03  .17830478e+04  .30963607e-02", 
" -.13674912e+02  .84606108e+02 -.42463118e+03  .24054245e+04  .30878387e-02", 
" -.13466672e+02  .82330754e+02 -.40707729e+03  .23413937e+04  .30906540e-02", 
" -.13251875e+02  .80269992e+02 -.40813111e+03  .26253900e+04  .31184788e-02", 
" -.13027342e+02  .77206511e+02 -.33695779e+03  .13970932e+04  .31475734e-02", 
" -.12796036e+02  .74569163e+02 -.29665133e+03  .74863985e+03  .32128406e-02", 
" -.12555894e+02  .72444481e+02 -.31613939e+03  .16139595e+04  .32445827e-02", 
" -.12304368e+02  .69917768e+02 -.30975312e+03  .18854549e+04  .32845215e-02", 
" -.12040814e+02  .67358732e+02 -.30782065e+03  .22458997e+04  .33706896e-02", 
" -.11763122e+02  .64807637e+02 -.31042095e+03  .26488350e+04  .33853448e-02", 
" -.11467363e+02  .61691960e+02 -.27929112e+03  .23482342e+04  .33966128e-02", 
" -.11152143e+02  .58696979e+02 -.27275428e+03  .26523810e+04  .33722504e-02", 
" -.10813967e+02  .55423399e+02 -.25156977e+03  .26139390e+04  .33692420e-02", 
" -.10445136e+02  .51831440e+02 -.21160671e+03  .19561633e+04  .35232275e-02", 
" -.10037018e+02  .47087906e+02 -.11602239e+03  .51120881e+03  .35622402e-02", 
" -.95771313e+01  .42673369e+02 -.95766341e+02  .77638672e+03  .38479738e-02", 
" -.90497498e+01  .37650744e+02 -.14422022e+02 -.78877522e+03  .41856860e-02", 
" -.84140964e+01  .32098798e+02  .29031109e+02 -.13366392e+04  .44941295e-02", 
" -.75942457e+01  .25001796e+02  .13165290e+03 -.35455560e+04  .50509951e-02", 
" -.73915808e+01  .22461890e+02  .21572847e+03 -.52892850e+04  .52602280e-02", 
" -.71739940e+01  .20482835e+02  .25080923e+03 -.58709705e+04  .54021649e-02", 
" -.69309936e+01  .18383542e+02  .28871005e+03 -.66679983e+04  .57364622e-02", 
" -.66583482e+01  .16149944e+02  .29976593e+03 -.66568022e+04  .58777015e-02", 
" -.63451336e+01  .12664921e+02  .38111940e+03 -.78557850e+04  .63618043e-02", 
" -.59789237e+01  .10342097e+02  .35637308e+03 -.71062238e+04  .70486213e-02", 
" -.55315304e+01  .85109745e+01  .20286445e+03 -.26281621e+04  .74293928e-02", 
" -.49236495e+01  .32628738e+01  .23750769e+03 -.26632972e+04  .83407460e-02", 
" -.39598166e+01 -.32653222e+01  .25861370e+03 -.33664666e+04  .10585460e-01", 
" -.30490516e+01 -.14722762e+02  .78006785e+03 -.15546329e+05  .13406182e-01", 
" -.19578613e+01 -.20174151e+02  .61481950e+03 -.97150295e+04  .21720331e-01", 
" -.11856324e+01 -.28026020e+02  .95294381e+03 -.15631526e+05  .28886889e-01", 
"coac      6  8  3   35", " -.86345986e+02  .20666737e+04 -.30619644e+05  .20399284e+06  .12268812e+00", 
" -.82737142e+02  .19489252e+04 -.30553912e+05  .24048804e+06  .83352454e-01", 
" -.77622076e+02  .17381059e+04 -.26178435e+05  .19908911e+06  .54450925e-01", 
" -.73646624e+02  .15859245e+04 -.23759831e+05  .18873082e+06  .39003294e-01", 
" -.69577284e+02  .14355987e+04 -.21272471e+05  .17364846e+06  .31163950e-01", 
" -.67136247e+02  .13398474e+04 -.19052288e+05  .14803784e+06  .25722568e-01", 
" -.65409237e+02  .12797574e+04 -.17965875e+05  .13900245e+06  .22232719e-01", 
" -.64036028e+02  .12327028e+04 -.17178544e+05  .13304451e+06  .20560254e-01", 
" -.62899546e+02  .11930506e+04 -.16413340e+05  .12539023e+06  .18464955e-01", 
" -.61922653e+02  .11576233e+04 -.15646964e+05  .11696533e+06  .16655164e-01", 
" -.61074440e+02  .11288394e+04 -.15124821e+05  .11261910e+06  .15701116e-01", 
" -.60319333e+02  .11021975e+04 -.14560662e+05  .10655258e+06  .15574455e-01", 
" -.59635425e+02  .10763653e+04 -.13916440e+05  .98396939e+05  .14635155e-01", 
" -.56994174e+02  .99220965e+03 -.12583692e+05  .89958414e+05  .12507050e-01", 
" -.55081778e+02  .93283868e+03 -.11586579e+05  .81645125e+05  .11188042e-01", 
" -.53563416e+02  .88610125e+03 -.10774183e+05  .74680962e+05  .10346465e-01", 
" -.52308025e+02  .84806806e+03 -.10081451e+05  .67830321e+05  .93685206e-02", 
" -.51229893e+02  .81688950e+03 -.95831579e+04  .63938571e+05  .87104822e-02", 
" -.50283184e+02  .79030551e+03 -.91916892e+04  .61269254e+05  .83500203e-02", 
" -.49437787e+02  .76647752e+03 -.87884722e+04  .57317084e+05  .78039912e-02", 
" -.48676950e+02  .74628190e+03 -.85129031e+04  .55570009e+05  .72769606e-02", 
" -.47973539e+02  .72599026e+03 -.81197316e+04  .51066713e+05  .67956938e-02", 
" -.47329627e+02  .70986242e+03 -.79796923e+04  .51913722e+05  .65803718e-02", 
" -.46725677e+02  .69289062e+03 -.76905322e+04  .49439409e+05  .67888703e-02", 
" -.46163407e+02  .67727792e+03 -.73983826e+04  .46081321e+05  .65521695e-02", 
" -.45635950e+02  .66301406e+03 -.71391886e+04  .43121164e+05  .62634638e-02", 
" -.45138501e+02  .65059045e+03 -.70046970e+04  .43114723e+05  .61869866e-02", 
" -.44666410e+02  .63813941e+03 -.68128342e+04  .41660764e+05  .59378591e-02", 
" -.44218547e+02  .62693086e+03 -.66750817e+04  .41158363e+05  .57038114e-02", 
" -.43791597e+02  .61619509e+03 -.65307931e+04  .40299147e+05  .54859416e-02", 
" -.43384600e+02  .60601259e+03 -.63844388e+04  .39176568e+05  .54110680e-02", 
" -.42993874e+02  .59609163e+03 -.62369381e+04  .38136209e+05  .53642614e-02", 
" -.42617784e+02  .58683501e+03 -.61155241e+04  .37490747e+05  .51269774e-02", 
" -.42256236e+02  .57806910e+03 -.59999828e+04  .36775501e+05  .50667910e-02", 
" -.41907856e+02  .57008037e+03 -.59294292e+04  .37047589e+05  .49911180e-02", 
" -.41571404e+02  .56185795e+03 -.58063051e+04  .35986111e+05  .48668178e-02", 
" -.41245789e+02  .55368289e+03 -.56607863e+04  .34307004e+05  .46423027e-02", 
" -.40931039e+02  .54613580e+03 -.55534447e+04  .33529371e+05  .45616796e-02", 
" -.40624665e+02  .53884185e+03 -.54550290e+04  .32892496e+05  .44889577e-02", 
" -.40327373e+02  .53163681e+03 -.53480327e+04  .32089318e+05  .44135728e-02", 
" -.40037944e+02  .52493713e+03 -.52649583e+04  .31648730e+05  .43940735e-02", 
" -.39755819e+02  .51849155e+03 -.51970348e+04  .31621332e+05  .43652877e-02", 
" -.39482395e+02  .51264718e+03 -.51502567e+04  .31800951e+05  .44316317e-02", 
" -.39214368e+02  .50599618e+03 -.50219874e+04  .30183726e+05  .43603423e-02", 
" -.38951536e+02  .49971435e+03 -.49294581e+04  .29512626e+05  .43508728e-02", 
" -.38695592e+02  .49392209e+03 -.48555823e+04  .29054647e+05  .43463589e-02", 
" -.38446758e+02  .48867755e+03 -.48154349e+04  .29305365e+05  .43631752e-02", 
" -.38202253e+02  .48292368e+03 -.47199155e+04  .28300412e+05  .43202941e-02", 
" -.37962779e+02  .47748571e+03 -.46429307e+04  .27667275e+05  .43258585e-02", 
" -.37727932e+02  .47218280e+03 -.45699481e+04  .27137709e+05  .42587776e-02", 
" -.37498736e+02  .46749852e+03 -.45350664e+04  .27308691e+05  .42244940e-02", 
" -.37272970e+02  .46255584e+03 -.44717531e+04  .26843646e+05  .42618691e-02", 
" -.37051411e+02  .45781645e+03 -.44234717e+04  .26770239e+05  .41830652e-02", 
" -.36833022e+02  .45290193e+03 -.43540419e+04  .26239925e+05  .41154716e-02", 
" -.36618556e+02  .44823328e+03 -.42997850e+04  .26009338e+05  .41035202e-02", 
" -.36406436e+02  .44330810e+03 -.42175677e+04  .25140173e+05  .41065484e-02", 
" -.36199898e+02  .43900591e+03 -.41738535e+04  .25040560e+05  .40682375e-02", 
" -.35996464e+02  .43453319e+03 -.41070448e+04  .24395411e+05  .40180297e-02", 
" -.35797001e+02  .43027697e+03 -.40472410e+04  .23845124e+05  .40680001e-02", 
" -.35598065e+02  .42575790e+03 -.39774189e+04  .23234451e+05  .41510245e-02", 
" -.35403883e+02  .42192605e+03 -.39531085e+04  .23537357e+05  .40858319e-02", 
" -.35211092e+02  .41754573e+03 -.38756575e+04  .22681015e+05  .40757151e-02", 
" -.35021089e+02  .41354158e+03 -.38278062e+04  .22430014e+05  .40784708e-02", 
" -.34834579e+02  .40967681e+03 -.37814131e+04  .22162375e+05  .41013535e-02", 
" -.34651794e+02  .40628590e+03 -.37658063e+04  .22516913e+05  .40250780e-02", 
" -.34469591e+02  .40260072e+03 -.37295768e+04  .22467623e+05  .39909048e-02", 
" -.34290238e+02  .39915707e+03 -.37047839e+04  .22592309e+05  .39656477e-02", 
" -.34112061e+02  .39525849e+03 -.36385596e+04  .21821289e+05  .39458535e-02", 
" -.33936340e+02  .39188983e+03 -.36116838e+04  .21860161e+05  .39404474e-02", 
" -.33761889e+02  .38819210e+03 -.35614697e+04  .21507255e+05  .38710385e-02", 
" -.33589397e+02  .38457917e+03 -.35133385e+04  .21138563e+05  .38749516e-02", 
" -.33418961e+02  .38096653e+03 -.34545274e+04  .20433066e+05  .37679453e-02", 
" -.33249731e+02  .37722882e+03 -.33836315e+04  .19482166e+05  .37948835e-02", 
" -.33082912e+02  .37368323e+03 -.33268977e+04  .18872145e+05  .37027003e-02", 
" -.32917967e+02  .37073747e+03 -.33216191e+04  .19428860e+05  .37080194e-02", 
" -.32754895e+02  .36737966e+03 -.32709265e+04  .18905660e+05  .37324851e-02", 
" -.32593856e+02  .36440015e+03 -.32478751e+04  .18941742e+05  .36754313e-02", 
" -.32433508e+02  .36122000e+03 -.32119354e+04  .18799028e+05  .36642877e-02", 
" -.32274690e+02  .35815838e+03 -.31787063e+04  .18603912e+05  .35996792e-02", 
" -.32117902e+02  .35526992e+03 -.31549886e+04  .18601258e+05  .35832237e-02", 
" -.31962143e+02  .35242773e+03 -.31359967e+04  .18730437e+05  .35702776e-02", 
" -.31807515e+02  .34946682e+03 -.31033094e+04  .18558415e+05  .35635003e-02", 
" -.31654238e+02  .34648841e+03 -.30683888e+04  .18353178e+05  .35463189e-02", 
" -.31501703e+02  .34372049e+03 -.30569709e+04  .18747275e+05  .35021556e-02", 
" -.31351833e+02  .34123520e+03 -.30540583e+04  .19168265e+05  .35409554e-02", 
" -.31201871e+02  .33826638e+03 -.30118040e+04  .18796629e+05  .35041837e-02", 
" -.31052387e+02  .33519784e+03 -.29645060e+04  .18350879e+05  .34721665e-02", 
" -.30904122e+02  .33219136e+03 -.29172686e+04  .17842171e+05  .34651057e-02", 
" -.30757204e+02  .32948192e+03 -.28914942e+04  .17750716e+05  .34393526e-02", 
" -.30610722e+02  .32663688e+03 -.28519666e+04  .17356566e+05  .34965445e-02", 
" -.30464352e+02  .32365270e+03 -.28046699e+04  .16873288e+05  .34702559e-02", 
" -.30319878e+02  .32076645e+03 -.27587514e+04  .16354156e+05  .35048523e-02", 
" -.30176859e+02  .31809325e+03 -.27293648e+04  .16235903e+05  .35033339e-02", 
" -.30034433e+02  .31532979e+03 -.26953605e+04  .16077685e+05  .34854705e-02", 
" -.29893524e+02  .31297255e+03 -.26837586e+04  .16236962e+05  .34831966e-02", 
" -.29752732e+02  .31027708e+03 -.26483666e+04  .15976534e+05  .34222703e-02", 
" -.29613452e+02  .30789583e+03 -.26351723e+04  .16168415e+05  .34276895e-02", 
" -.29474184e+02  .30536570e+03 -.26084664e+04  .16028407e+05  .34390697e-02", 
" -.29335148e+02  .30243830e+03 -.25458877e+04  .15096842e+05  .34355004e-02", 
" -.29197102e+02  .29993804e+03 -.25175123e+04  .14879712e+05  .34140039e-02", 
" -.29060663e+02  .29768082e+03 -.25037646e+04  .14951150e+05  .34108251e-02", 
" -.28923805e+02  .29502556e+03 -.24616118e+04  .14473115e+05  .33997235e-02", 
" -.28787658e+02  .29239641e+03 -.24201598e+04  .14005132e+05  .33660101e-02", 
" -.28652959e+02  .28998309e+03 -.23848069e+04  .13538045e+05  .33366034e-02", 
" -.28517895e+02  .28743472e+03 -.23471285e+04  .13137861e+05  .33649666e-02", 
" -.28384231e+02  .28524575e+03 -.23384244e+04  .13396040e+05  .32962680e-02", 
" -.28251327e+02  .28310168e+03 -.23301701e+04  .13634088e+05  .33008248e-02", 
" -.28118199e+02  .28067186e+03 -.22920841e+04  .13131877e+05  .33148916e-02", 
" -.27985406e+02  .27835386e+03 -.22676381e+04  .12986533e+05  .32770705e-02", 
" -.27853479e+02  .27625758e+03 -.22607884e+04  .13232756e+05  .32372632e-02", 
" -.27720960e+02  .27380098e+03 -.22253359e+04  .12863004e+05  .32244315e-02", 
" -.27589802e+02  .27158079e+03 -.22036697e+04  .12724074e+05  .32352066e-02", 
" -.27459075e+02  .26943688e+03 -.21906041e+04  .12838950e+05  .32129087e-02", 
" -.27327238e+02  .26684477e+03 -.21461409e+04  .12358185e+05  .32158826e-02", 
" -.27197181e+02  .26447628e+03 -.21059744e+04  .11798540e+05  .31858689e-02", 
" -.27067577e+02  .26229645e+03 -.20811483e+04  .11530205e+05  .31649438e-02", 
" -.26938005e+02  .26013994e+03 -.20593581e+04  .11396651e+05  .31362928e-02", 
" -.26808260e+02  .25760109e+03 -.20065651e+04  .10623064e+05  .31009024e-02", 
" -.26679035e+02  .25540854e+03 -.19787057e+04  .10329337e+05  .30809519e-02", 
" -.26550621e+02  .25326490e+03 -.19586616e+04  .10296055e+05  .30249580e-02", 
" -.26421986e+02  .25115005e+03 -.19381028e+04  .10178701e+05  .30135814e-02", 
" -.26293309e+02  .24878576e+03 -.18998198e+04  .97391035e+04  .30014281e-02", 
" -.26165240e+02  .24683033e+03 -.18903392e+04  .98388833e+04  .29922699e-02", 
" -.26036976e+02  .24472864e+03 -.18740687e+04  .98706407e+04  .29942242e-02", 
" -.25907822e+02  .24211828e+03 -.18165961e+04  .90654224e+04  .29637750e-02", 
" -.25780469e+02  .24019378e+03 -.18082113e+04  .92097008e+04  .29245983e-02", 
" -.25652351e+02  .23784801e+03 -.17680720e+04  .87178721e+04  .29067379e-02", 
" -.25525072e+02  .23584043e+03 -.17514116e+04  .86800167e+04  .28737620e-02", 
" -.25398483e+02  .23396905e+03 -.17439699e+04  .88345078e+04  .28570077e-02", 
" -.25270554e+02  .23190573e+03 -.17248057e+04  .87595470e+04  .28678011e-02", 
" -.25143902e+02  .23009643e+03 -.17194055e+04  .89146390e+04  .28618101e-02", 
" -.25015409e+02  .22765864e+03 -.16642565e+04  .79769704e+04  .28688263e-02", 
" -.24887138e+02  .22531798e+03 -.16219975e+04  .74511568e+04  .28939239e-02", 
" -.24758855e+02  .22314063e+03 -.15934827e+04  .72082900e+04  .29502435e-02", 
" -.24630842e+02  .22109459e+03 -.15745702e+04  .71455327e+04  .29558790e-02", 
" -.24502728e+02  .21884170e+03 -.15359486e+04  .66563189e+04  .29753200e-02", 
" -.24375336e+02  .21692526e+03 -.15243563e+04  .67541373e+04  .29516084e-02", 
" -.24246593e+02  .21479694e+03 -.14956570e+04  .64323167e+04  .29721617e-02", 
" -.24117686e+02  .21253718e+03 -.14604239e+04  .60738121e+04  .29725410e-02", 
" -.23989451e+02  .21054809e+03 -.14412847e+04  .59660091e+04  .29396990e-02", 
" -.23861233e+02  .20851922e+03 -.14142496e+04  .56344070e+04  .29050651e-02", 
" -.23732788e+02  .20652164e+03 -.13927871e+04  .54985600e+04  .29193587e-02", 
" -.23604313e+02  .20469884e+03 -.13859035e+04  .56678437e+04  .29654600e-02", 
" -.23474692e+02  .20267212e+03 -.13661572e+04  .55964312e+04  .30104366e-02", 
" -.23344706e+02  .20053143e+03 -.13386190e+04  .53982359e+04  .30086498e-02", 
" -.23214744e+02  .19847886e+03 -.13065607e+04  .48889430e+04  .30354026e-02", 
" -.23084597e+02  .19659962e+03 -.12946216e+04  .49203568e+04  .30386727e-02", 
" -.22953966e+02  .19451527e+03 -.12656076e+04  .45900835e+04  .30416267e-02", 
" -.22823427e+02  .19275452e+03 -.12642888e+04  .48829383e+04  .30312356e-02", 
" -.22692031e+02  .19089996e+03 -.12552251e+04  .49987556e+04  .30229596e-02", 
" -.22560105e+02  .18881705e+03 -.12236126e+04  .45832334e+04  .30294897e-02", 
" -.22427172e+02  .18663368e+03 -.11865582e+04  .40515952e+04  .30240305e-02", 
" -.22294573e+02  .18486753e+03 -.11883080e+04  .44629826e+04  .30158163e-02", 
" -.22160289e+02  .18265309e+03 -.11505108e+04  .39521254e+04  .29347090e-02", 
" -.22025606e+02  .18051671e+03 -.11188391e+04  .35952231e+04  .29273100e-02", 
" -.21890655e+02  .17856979e+03 -.11039136e+04  .35694265e+04  .29152243e-02", 
" -.21755696e+02  .17672462e+03 -.10941402e+04  .36749333e+04  .28831085e-02", 
" -.21619216e+02  .17452556e+03 -.10559216e+04  .31297753e+04  .29081300e-02", 
" -.21482063e+02  .17232765e+03 -.10182411e+04  .26100096e+04  .29141403e-02", 
" -.21344642e+02  .17068725e+03 -.10311153e+04  .32011779e+04  .28821292e-02", 
" -.21205838e+02  .16876716e+03 -.10205012e+04  .33108709e+04  .28686419e-02", 
" -.21066433e+02  .16699783e+03 -.10251017e+04  .37718065e+04  .28903203e-02", 
" -.20925513e+02  .16495179e+03 -.10091215e+04  .38391881e+04  .29239140e-02", 
" -.20783814e+02  .16279839e+03 -.97579487e+03  .33721561e+04  .29551884e-02", 
" -.20641028e+02  .16084815e+03 -.96823431e+03  .35726331e+04  .29109445e-02", 
" -.20497550e+02  .15885556e+03 -.95192109e+03  .35545794e+04  .29104364e-02", 
" -.20352184e+02  .15673181e+03 -.92732692e+03  .33589062e+04  .29063140e-02", 
" -.20205378e+02  .15457374e+03 -.90187931e+03  .31592090e+04  .29080932e-02", 
" -.20057507e+02  .15245727e+03 -.87754982e+03  .29148366e+04  .29267299e-02", 
" -.19908813e+02  .15047617e+03 -.86899348e+03  .31408096e+04  .29819350e-02", 
" -.19758222e+02  .14831619e+03 -.84560730e+03  .29908957e+04  .29760332e-02", 
" -.19606724e+02  .14641051e+03 -.83994243e+03  .32025150e+04  .29700213e-02", 
" -.19453396e+02  .14435195e+03 -.82624870e+03  .32847804e+04  .29542748e-02", 
" -.19299020e+02  .14260431e+03 -.83047926e+03  .35895101e+04  .29787881e-02", 
" -.19143318e+02  .14099612e+03 -.85277482e+03  .44191151e+04  .29960093e-02", 
" -.18984396e+02  .13879692e+03 -.82419456e+03  .40823690e+04  .30553098e-02", 
" -.18823751e+02  .13675966e+03 -.81190758e+03  .41542325e+04  .31041994e-02", 
" -.18659877e+02  .13461914e+03 -.79401247e+03  .40562145e+04  .30587490e-02", 
" -.18494781e+02  .13253145e+03 -.77429185e+03  .38269449e+04  .30799541e-02", 
" -.18326974e+02  .13047780e+03 -.76602449e+03  .39689030e+04  .30665901e-02", 
" -.18156281e+02  .12821617e+03 -.74168020e+03  .37471492e+04  .31018352e-02", 
" -.17982553e+02  .12582282e+03 -.70595058e+03  .32834730e+04  .31200251e-02", 
" -.17806135e+02  .12366483e+03 -.69432755e+03  .33447412e+04  .31563219e-02", 
" -.17626021e+02  .12132647e+03 -.67032457e+03  .31350015e+04  .31121033e-02", 
" -.17442905e+02  .11891280e+03 -.63510484e+03  .26255464e+04  .31204608e-02", 
" -.17255371e+02  .11647142e+03 -.60834249e+03  .24012463e+04  .31810889e-02", 
" -.17064849e+02  .11428073e+03 -.60414398e+03  .26508903e+04  .31903004e-02", 
" -.16870112e+02  .11194956e+03 -.58620162e+03  .25707294e+04  .31441989e-02", 
" -.16669967e+02  .10956864e+03 -.57336412e+03  .26529871e+04  .31972328e-02", 
" -.16465027e+02  .10733756e+03 -.58174624e+03  .32609994e+04  .31823047e-02", 
" -.16253054e+02  .10431994e+03 -.52425113e+03  .24224078e+04  .32352900e-02", 
" -.16034258e+02  .10111710e+03 -.45526910e+03  .12767464e+04  .33053533e-02", 
" -.15810752e+02  .98673211e+02 -.46202028e+03  .19692017e+04  .33324987e-02", 
" -.15578886e+02  .95425222e+02 -.39592128e+03  .92930241e+03  .34364027e-02", 
" -.15340357e+02  .92879916e+02 -.39583157e+03  .13910345e+04  .34269632e-02", 
" -.15091439e+02  .89939480e+02 -.37056441e+03  .12713901e+04  .34769084e-02", 
" -.14832052e+02  .86673555e+02 -.32353679e+03  .66801843e+03  .34684473e-02", 
" -.14560922e+02  .83624986e+02 -.30352188e+03  .65098325e+03  .35430860e-02", 
" -.14277513e+02  .80752713e+02 -.31198473e+03  .13936262e+04  .36377586e-02", 
" -.13977181e+02  .77058785e+02 -.24476933e+03  .24864271e+03  .36353583e-02", 
" -.13661282e+02  .74151856e+02 -.25135539e+03  .69572697e+03  .37795541e-02", 
" -.13321706e+02  .70906730e+02 -.26903048e+03  .17185621e+04  .37725147e-02", 
" -.12955456e+02  .67114280e+02 -.25317829e+03  .19470053e+04  .38081594e-02", 
" -.12554582e+02  .62309097e+02 -.16979245e+03  .68136156e+03  .40157277e-02", 
" -.12112208e+02  .57337044e+02 -.11009263e+03  .71256163e+02  .40738910e-02", 
" -.11615526e+02  .52512473e+02 -.99759985e+02  .74020582e+03  .43456004e-02", 
" -.11040307e+02  .46046557e+02  .87512249e+01 -.88087787e+03  .46048429e-02", 
" -.10345637e+02  .38769986e+02  .11729823e+03 -.27490941e+04  .49281212e-02", 
" -.94447649e+01  .29741568e+02  .24565468e+03 -.49841054e+04  .54646055e-02", 
" -.92233335e+01  .27727146e+02  .24805871e+03 -.46174966e+04  .56734261e-02", 
" -.89822302e+01  .25155189e+02  .29478163e+03 -.54523584e+04  .59461033e-02", 
" -.87181249e+01  .23445600e+02  .24294616e+03 -.37354576e+04  .62130696e-02", 
" -.84206189e+01  .20937730e+02  .29227043e+03 -.51917212e+04  .66138810e-02", 
" -.80724556e+01  .16932304e+02  .39599982e+03 -.74806404e+04  .71486269e-02", 
" -.76677190e+01  .12176769e+02  .52640691e+03 -.99034979e+04  .76071393e-02", 
" -.71723810e+01  .77188657e+01  .59664140e+03 -.11255262e+05  .84198703e-02", 
" -.65070877e+01  .20893082e+01  .63845754e+03 -.11558424e+05  .95277250e-02", 
" -.54464869e+01 -.65534264e+01  .60589401e+03 -.84281631e+04  .13069864e-01", 
" -.44777830e+01 -.11415910e+02  .52979416e+03 -.73668895e+04  .16345998e-01", 
" -.32523128e+01 -.24931711e+02  .99790331e+03 -.16415268e+05  .23198522e-01", 
" -.24246620e+01 -.22343057e+02  .61549914e+03 -.10904881e+05  .32599656e-01", 
"coact     6  9  3   35", " -.93357883e+02  .24748962e+04 -.42760510e+05  .35801719e+06  .11734055e+00", 
" -.89286775e+02  .22623863e+04 -.37171701e+05  .29354716e+06  .84262127e-01", 
" -.83981265e+02  .20326906e+04 -.33156884e+05  .27580112e+06  .54023167e-01", 
" -.79965933e+02  .18679482e+04 -.29983281e+05  .25063160e+06  .40060526e-01", 
" -.75735851e+02  .16908067e+04 -.26111015e+05  .21033646e+06  .30010868e-01", 
" -.73197962e+02  .15859444e+04 -.23791093e+05  .18659286e+06  .24751366e-01", 
" -.71374693e+02  .15159101e+04 -.22437148e+05  .17446779e+06  .22447034e-01", 
" -.69955174e+02  .14636300e+04 -.21532191e+05  .16791672e+06  .20358110e-01", 
" -.68777515e+02  .14216753e+04 -.20850104e+05  .16405462e+06  .18557625e-01", 
" -.67770537e+02  .13836876e+04 -.20062926e+05  .15595542e+06  .17305828e-01", 
" -.66886179e+02  .13490605e+04 -.19312226e+05  .14857400e+06  .16059834e-01", 
" -.66110280e+02  .13202321e+04 -.18724135e+05  .14295327e+06  .15166481e-01", 
" -.65399567e+02  .12928928e+04 -.18121249e+05  .13647781e+06  .14367887e-01", 
" -.62650922e+02  .11961297e+04 -.16323185e+05  .12107067e+06  .12399612e-01", 
" -.60658194e+02  .11279291e+04 -.15053124e+05  .10986841e+06  .11109527e-01", 
" -.59075092e+02  .10747274e+04 -.14033416e+05  .99728789e+05  .10273823e-01", 
" -.57761856e+02  .10325939e+04 -.13293630e+05  .92966585e+05  .93339076e-02", 
" -.56624322e+02  .99429249e+03 -.12533153e+05  .85462448e+05  .85831884e-02", 
" -.55638037e+02  .96479275e+03 -.12124558e+05  .83528858e+05  .85197275e-02", 
" -.54750425e+02  .93696408e+03 -.11649180e+05  .79726487e+05  .81635840e-02", 
" -.53946019e+02  .91145536e+03 -.11190163e+05  .75895010e+05  .76383763e-02", 
" -.53214456e+02  .89070202e+03 -.10941903e+05  .75509644e+05  .76385790e-02", 
" -.52535637e+02  .87042633e+03 -.10625105e+05  .73416641e+05  .73210463e-02", 
" -.51910449e+02  .85260297e+03 -.10364253e+05  .71734537e+05  .69875845e-02", 
" -.51326426e+02  .83655734e+03 -.10156174e+05  .70793181e+05  .66450990e-02", 
" -.50778320e+02  .82172823e+03 -.99743259e+04  .70140868e+05  .65330794e-02", 
" -.50256628e+02  .80683680e+03 -.97526875e+04  .68801783e+05  .62695398e-02", 
" -.49760679e+02  .79258424e+03 -.95226735e+04  .66944427e+05  .61933629e-02", 
" -.49289658e+02  .77874417e+03 -.92543929e+04  .63825782e+05  .59863004e-02", 
" -.48839810e+02  .76566389e+03 -.90156590e+04  .61377939e+05  .59149682e-02", 
" -.48412627e+02  .75400719e+03 -.88392324e+04  .60013807e+05  .57731476e-02", 
" -.47997459e+02  .74220076e+03 -.86518728e+04  .58673830e+05  .57556588e-02", 
" -.47601196e+02  .73105344e+03 -.84690597e+04  .57141372e+05  .56871541e-02", 
" -.47222165e+02  .72129946e+03 -.83632431e+04  .57161231e+05  .56659501e-02", 
" -.46855271e+02  .71179619e+03 -.82520766e+04  .56901422e+05  .55151698e-02", 
" -.46499384e+02  .70213532e+03 -.80953832e+04  .55508979e+05  .53720076e-02", 
" -.46154169e+02  .69236909e+03 -.79104526e+04  .53454232e+05  .52990770e-02", 
" -.45820644e+02  .68301301e+03 -.77326606e+04  .51456836e+05  .52062095e-02", 
" -.45496162e+02  .67390602e+03 -.75656872e+04  .49782317e+05  .52257547e-02", 
" -.45182179e+02  .66514076e+03 -.73947425e+04  .47785141e+05  .50748048e-02", 
" -.44875937e+02  .65684213e+03 -.72630926e+04  .46871409e+05  .49243352e-02", 
" -.44579596e+02  .64929042e+03 -.71660431e+04  .46440687e+05  .48983132e-02", 
" -.44289423e+02  .64164738e+03 -.70550633e+04  .45820997e+05  .48635285e-02", 
" -.44005083e+02  .63395357e+03 -.69187739e+04  .44443127e+05  .48185181e-02", 
" -.43729007e+02  .62721116e+03 -.68437973e+04  .44286213e+05  .48341722e-02", 
" -.43459823e+02  .62069686e+03 -.67751250e+04  .44278478e+05  .47106505e-02", 
" -.43196694e+02  .61445712e+03 -.67174918e+04  .44463514e+05  .47679221e-02", 
" -.42938162e+02  .60798295e+03 -.66310895e+04  .43975678e+05  .47527195e-02", 
" -.42684726e+02  .60169755e+03 -.65491567e+04  .43549044e+05  .47103605e-02", 
" -.42436138e+02  .59518829e+03 -.64326725e+04  .42266965e+05  .46881569e-02", 
" -.42192482e+02  .58882944e+03 -.63202216e+04  .41060921e+05  .47128779e-02", 
" -.41954926e+02  .58322197e+03 -.62511433e+04  .40659509e+05  .46562784e-02", 
" -.41720183e+02  .57745004e+03 -.61688738e+04  .40047428e+05  .45671340e-02", 
" -.41489098e+02  .57160017e+03 -.60756229e+04  .39244107e+05  .45412254e-02", 
" -.41262701e+02  .56598265e+03 -.59856118e+04  .38350612e+05  .44231653e-02", 
" -.41041168e+02  .56084067e+03 -.59276065e+04  .38203740e+05  .44174460e-02", 
" -.40822553e+02  .55572626e+03 -.58710110e+04  .38086916e+05  .44532507e-02", 
" -.40606031e+02  .55032833e+03 -.57823868e+04  .37205778e+05  .44199883e-02", 
" -.40392250e+02  .54487532e+03 -.56889285e+04  .36317255e+05  .43348392e-02", 
" -.40182408e+02  .53973281e+03 -.56032899e+04  .35386692e+05  .43263095e-02", 
" -.39975266e+02  .53446330e+03 -.55018771e+04  .34116957e+05  .42682239e-02", 
" -.39771734e+02  .52980042e+03 -.54533169e+04  .34119170e+05  .42357835e-02", 
" -.39570632e+02  .52499305e+03 -.53810295e+04  .33474343e+05  .42529694e-02", 
" -.39372310e+02  .52020088e+03 -.52997394e+04  .32549160e+05  .42666312e-02", 
" -.39176987e+02  .51553888e+03 -.52288138e+04  .31928060e+05  .42209008e-02", 
" -.38984056e+02  .51116233e+03 -.51780286e+04  .31761015e+05  .41886071e-02", 
" -.38792099e+02  .50648715e+03 -.50996030e+04  .30951981e+05  .41894909e-02", 
" -.38603341e+02  .50204776e+03 -.50327911e+04  .30365469e+05  .42180933e-02", 
" -.38416810e+02  .49764083e+03 -.49669275e+04  .29834600e+05  .42245955e-02", 
" -.38233550e+02  .49374482e+03 -.49353694e+04  .29963168e+05  .42121622e-02", 
" -.38051496e+02  .48957013e+03 -.48729418e+04  .29357090e+05  .41699920e-02", 
" -.37869996e+02  .48535048e+03 -.48150497e+04  .28980528e+05  .40757579e-02", 
" -.37691265e+02  .48147667e+03 -.47810259e+04  .29118765e+05  .40675349e-02", 
" -.37514392e+02  .47729083e+03 -.47075833e+04  .28244256e+05  .40431975e-02", 
" -.37339088e+02  .47345824e+03 -.46664392e+04  .28126617e+05  .40311512e-02", 
" -.37166015e+02  .46964691e+03 -.46174936e+04  .27773319e+05  .40067223e-02", 
" -.36994633e+02  .46572596e+03 -.45555756e+04  .27156559e+05  .39957113e-02", 
" -.36824490e+02  .46195172e+03 -.45072434e+04  .26878137e+05  .39209424e-02", 
" -.36655461e+02  .45823795e+03 -.44659598e+04  .26798274e+05  .39292026e-02", 
" -.36487153e+02  .45424135e+03 -.44012224e+04  .26235940e+05  .39772715e-02", 
" -.36320078e+02  .45042188e+03 -.43453063e+04  .25788481e+05  .39587999e-02", 
" -.36156221e+02  .44710034e+03 -.43187701e+04  .25873183e+05  .39577575e-02", 
" -.35993283e+02  .44357636e+03 -.42753467e+04  .25642102e+05  .39317858e-02", 
" -.35831646e+02  .44008593e+03 -.42341130e+04  .25458119e+05  .39170477e-02", 
" -.35670549e+02  .43660296e+03 -.41923498e+04  .25251747e+05  .38816107e-02", 
" -.35511595e+02  .43329798e+03 -.41552046e+04  .25048147e+05  .38559101e-02", 
" -.35352731e+02  .42993008e+03 -.41131211e+04  .24743447e+05  .38320207e-02", 
" -.35195458e+02  .42657115e+03 -.40694950e+04  .24415972e+05  .37875446e-02", 
" -.35037759e+02  .42283265e+03 -.40021604e+04  .23680526e+05  .37757795e-02", 
" -.34882290e+02  .41961261e+03 -.39676931e+04  .23549303e+05  .37312437e-02", 
" -.34728027e+02  .41633071e+03 -.39198976e+04  .23050660e+05  .37340205e-02", 
" -.34575103e+02  .41321569e+03 -.38889141e+04  .22985656e+05  .36999524e-02", 
" -.34422519e+02  .41005244e+03 -.38507401e+04  .22747629e+05  .36706264e-02", 
" -.34270730e+02  .40687998e+03 -.38155131e+04  .22643307e+05  .36500446e-02", 
" -.34119840e+02  .40378111e+03 -.37802249e+04  .22452541e+05  .36072497e-02", 
" -.33970383e+02  .40081577e+03 -.37514688e+04  .22370596e+05  .36278509e-02", 
" -.33821119e+02  .39770506e+03 -.37150270e+04  .22222632e+05  .35427711e-02", 
" -.33673447e+02  .39479009e+03 -.36824219e+04  .21983545e+05  .35505948e-02", 
" -.33525449e+02  .39160889e+03 -.36294049e+04  .21332523e+05  .35117065e-02", 
" -.33378691e+02  .38854498e+03 -.35854124e+04  .20930692e+05  .34556179e-02", 
" -.33232956e+02  .38575008e+03 -.35668070e+04  .21140473e+05  .34318839e-02", 
" -.33086637e+02  .38269268e+03 -.35279752e+04  .20889755e+05  .33942661e-02", 
" -.32941590e+02  .37971320e+03 -.34901363e+04  .20662555e+05  .33955943e-02", 
" -.32796503e+02  .37652106e+03 -.34350258e+04  .20050264e+05  .33983063e-02", 
" -.32652576e+02  .37357355e+03 -.33978231e+04  .19808800e+05  .34034697e-02", 
" -.32508448e+02  .37043136e+03 -.33450745e+04  .19241649e+05  .33914107e-02", 
" -.32365648e+02  .36755910e+03 -.33084181e+04  .18966211e+05  .34051780e-02", 
" -.32223368e+02  .36478581e+03 -.32794735e+04  .18852992e+05  .33532826e-02", 
" -.32080413e+02  .36157205e+03 -.32147036e+04  .17994983e+05  .33623866e-02", 
" -.31938847e+02  .35872855e+03 -.31786056e+04  .17721478e+05  .33082939e-02", 
" -.31797503e+02  .35593232e+03 -.31447620e+04  .17487147e+05  .33235962e-02", 
" -.31656353e+02  .35312873e+03 -.31130226e+04  .17356093e+05  .32994754e-02", 
" -.31516499e+02  .35052283e+03 -.30915463e+04  .17360935e+05  .33005176e-02", 
" -.31376416e+02  .34777728e+03 -.30632582e+04  .17315665e+05  .32982687e-02", 
" -.31236138e+02  .34492427e+03 -.30213697e+04  .16902568e+05  .32830448e-02", 
" -.31097222e+02  .34231688e+03 -.29935925e+04  .16727536e+05  .33365552e-02", 
" -.30957864e+02  .33945919e+03 -.29546753e+04  .16462847e+05  .33839858e-02", 
" -.30819531e+02  .33683878e+03 -.29286007e+04  .16431028e+05  .34243675e-02", 
" -.30681390e+02  .33428947e+03 -.29056920e+04  .16392322e+05  .33861804e-02", 
" -.30544290e+02  .33196572e+03 -.28961007e+04  .16590500e+05  .33646910e-02", 
" -.30406588e+02  .32937083e+03 -.28681878e+04  .16463789e+05  .33708854e-02", 
" -.30268913e+02  .32687727e+03 -.28504755e+04  .16574567e+05  .33390400e-02", 
" -.30131194e+02  .32410850e+03 -.28062596e+04  .16104558e+05  .33358070e-02", 
" -.29993343e+02  .32131506e+03 -.27582143e+04  .15481400e+05  .33522262e-02", 
" -.29856016e+02  .31868262e+03 -.27238942e+04  .15205556e+05  .33475235e-02", 
" -.29718357e+02  .31587179e+03 -.26740261e+04  .14557983e+05  .33310744e-02", 
" -.29580419e+02  .31307044e+03 -.26272901e+04  .14016317e+05  .33461401e-02", 
" -.29443744e+02  .31063247e+03 -.26088428e+04  .14079092e+05  .33690333e-02", 
" -.29307599e+02  .30839446e+03 -.26044296e+04  .14443849e+05  .32947163e-02", 
" -.29170903e+02  .30584466e+03 -.25763241e+04  .14325272e+05  .32990607e-02", 
" -.29033265e+02  .30303782e+03 -.25288540e+04  .13808381e+05  .32799420e-02", 
" -.28895950e+02  .30035721e+03 -.24858966e+04  .13276552e+05  .32649930e-02", 
" -.28759230e+02  .29791736e+03 -.24636648e+04  .13260263e+05  .32523445e-02", 
" -.28622036e+02  .29539257e+03 -.24307010e+04  .12918522e+05  .32076888e-02", 
" -.28484085e+02  .29276516e+03 -.23968460e+04  .12689934e+05  .32012735e-02", 
" -.28347515e+02  .29040061e+03 -.23727574e+04  .12526009e+05  .32324987e-02", 
" -.28209574e+02  .28765119e+03 -.23243953e+04  .11943495e+05  .32615302e-02", 
" -.28072365e+02  .28533921e+03 -.23113509e+04  .12104158e+05  .32791817e-02", 
" -.27934099e+02  .28255805e+03 -.22580335e+04  .11378871e+05  .32871221e-02", 
" -.27795426e+02  .27995122e+03 -.22232828e+04  .11106664e+05  .32542527e-02", 
" -.27657252e+02  .27763537e+03 -.22071969e+04  .11151428e+05  .32472289e-02", 
" -.27517469e+02  .27474286e+03 -.21449990e+04  .10224798e+05  .32225683e-02", 
" -.27378286e+02  .27217608e+03 -.21096815e+04  .98854028e+04  .32336623e-02", 
" -.27238956e+02  .26973216e+03 -.20856377e+04  .98229166e+04  .32495894e-02", 
" -.27099186e+02  .26711396e+03 -.20451427e+04  .93916286e+04  .32404199e-02", 
" -.26959695e+02  .26485035e+03 -.20288378e+04  .93587700e+04  .32237441e-02", 
" -.26819415e+02  .26255488e+03 -.20187602e+04  .96255022e+04  .32387723e-02", 
" -.26678343e+02  .25992492e+03 -.19763945e+04  .91320206e+04  .32272978e-02", 
" -.26537563e+02  .25758617e+03 -.19540193e+04  .90042385e+04  .32398441e-02", 
" -.26394662e+02  .25476972e+03 -.18985267e+04  .82276780e+04  .32415121e-02", 
" -.26252229e+02  .25226859e+03 -.18671668e+04  .79601905e+04  .32379915e-02", 
" -.26109238e+02  .24992075e+03 -.18483984e+04  .79525293e+04  .32215433e-02", 
" -.25965241e+02  .24712064e+03 -.17896979e+04  .70602214e+04  .32013699e-02", 
" -.25821317e+02  .24486911e+03 -.17792665e+04  .72461464e+04  .31977720e-02", 
" -.25676769e+02  .24233941e+03 -.17460587e+04  .69651858e+04  .31970405e-02", 
" -.25531069e+02  .23984693e+03 -.17187810e+04  .68272320e+04  .31443258e-02", 
" -.25384857e+02  .23724966e+03 -.16792718e+04  .63692435e+04  .31623332e-02", 
" -.25237392e+02  .23464946e+03 -.16376409e+04  .58086709e+04  .31497256e-02", 
" -.25089355e+02  .23231349e+03 -.16234337e+04  .59031821e+04  .31226373e-02", 
" -.24940024e+02  .22965655e+03 -.15821239e+04  .54462754e+04  .31146770e-02", 
" -.24790790e+02  .22721615e+03 -.15512767e+04  .51194695e+04  .31285096e-02", 
" -.24639907e+02  .22485192e+03 -.15339575e+04  .51350383e+04  .31334801e-02", 
" -.24486885e+02  .22231577e+03 -.15146041e+04  .52844399e+04  .31623330e-02", 
" -.24334039e+02  .21991107e+03 -.14952250e+04  .52826053e+04  .31939162e-02", 
" -.24179441e+02  .21736006e+03 -.14663898e+04  .50889765e+04  .31965658e-02", 
" -.24023407e+02  .21457408e+03 -.14141304e+04  .43365788e+04  .32224394e-02", 
" -.23866817e+02  .21207396e+03 -.13890715e+04  .41974933e+04  .31932722e-02", 
" -.23707971e+02  .20925207e+03 -.13388169e+04  .35184928e+04  .32248680e-02", 
" -.23549807e+02  .20725961e+03 -.13532082e+04  .42006386e+04  .32195162e-02", 
" -.23389051e+02  .20488870e+03 -.13408261e+04  .43435398e+04  .32274050e-02", 
" -.23226613e+02  .20251086e+03 -.13332481e+04  .46536887e+04  .32266962e-02", 
" -.23061848e+02  .19973172e+03 -.12912630e+04  .41899970e+04  .32397566e-02", 
" -.22895645e+02  .19709676e+03 -.12638010e+04  .40680851e+04  .31967354e-02", 
" -.22727416e+02  .19449888e+03 -.12405310e+04  .40235423e+04  .32244800e-02", 
" -.22556437e+02  .19158498e+03 -.11924330e+04  .34124296e+04  .32120235e-02", 
" -.22383890e+02  .18908572e+03 -.11785966e+04  .35609405e+04  .32480155e-02", 
" -.22208704e+02  .18638967e+03 -.11511924e+04  .33935382e+04  .32229870e-02", 
" -.22031310e+02  .18362171e+03 -.11177595e+04  .30854015e+04  .32292190e-02", 
" -.21851663e+02  .18084565e+03 -.10844493e+04  .28079873e+04  .33074569e-02", 
" -.21669648e+02  .17814562e+03 -.10622481e+04  .27968266e+04  .33118605e-02", 
" -.21484386e+02  .17537877e+03 -.10295506e+04  .24169919e+04  .33663742e-02", 
" -.21295327e+02  .17254276e+03 -.10053163e+04  .24043619e+04  .33723302e-02", 
" -.21104613e+02  .17024185e+03 -.10261754e+04  .33957652e+04  .33595234e-02", 
" -.20909423e+02  .16738458e+03 -.99500811e+03  .30903049e+04  .33818887e-02", 
" -.20710109e+02  .16434517e+03 -.95525907e+03  .26912513e+04  .34514624e-02", 
" -.20507221e+02  .16170805e+03 -.95331619e+03  .30957261e+04  .35092389e-02", 
" -.20299282e+02  .15878449e+03 -.93228953e+03  .31518955e+04  .35749300e-02", 
" -.20086017e+02  .15542222e+03 -.87151137e+03  .22032263e+04  .35923577e-02", 
" -.19867148e+02  .15225514e+03 -.84807714e+03  .23156842e+04  .35789459e-02", 
" -.19642982e+02  .14899173e+03 -.81283404e+03  .20691318e+04  .35833315e-02", 
" -.19411850e+02  .14546103e+03 -.76436738e+03  .15702229e+04  .35811059e-02", 
" -.19176783e+02  .14260139e+03 -.76945768e+03  .22223689e+04  .35760405e-02", 
" -.18930709e+02  .13847087e+03 -.67151246e+03  .56523599e+03  .36256675e-02", 
" -.18676958e+02  .13486552e+03 -.63812192e+03  .48658359e+03  .37546894e-02", 
" -.18417257e+02  .13162102e+03 -.62798313e+03  .82868468e+03  .37870719e-02", 
" -.18144861e+02  .12784309e+03 -.59100372e+03  .61849447e+03  .39024641e-02", 
" -.17861177e+02  .12371291e+03 -.52045778e+03 -.45477652e+03  .39779685e-02", 
" -.17566335e+02  .12028451e+03 -.51607408e+03 -.12879219e+03  .39886083e-02", 
" -.17254119e+02  .11624756e+03 -.49190194e+03  .78297226e+02  .40097825e-02", 
" -.16925826e+02  .11236673e+03 -.48544860e+03  .58499974e+03  .40917920e-02", 
" -.16576485e+02  .10789218e+03 -.44122643e+03  .17297453e+03  .41985654e-02", 
" -.16201983e+02  .10297743e+03 -.39790276e+03  .75893953e+02  .41806636e-02", 
" -.15797873e+02  .97481555e+02 -.31407678e+03 -.10727410e+04  .42300609e-02", 
" -.15356147e+02  .91113705e+02 -.20409857e+03 -.24901061e+04  .43437037e-02", 
" -.14867665e+02  .85206009e+02 -.17947089e+03 -.18406935e+04  .44147401e-02", 
" -.14322663e+02  .79558005e+02 -.20147277e+03 -.33027173e+03  .45686081e-02", 
" -.13690532e+02  .72637552e+02 -.18302678e+03  .46766141e+03  .47965588e-02", 
" -.12924845e+02  .64360793e+02 -.16135274e+03  .11264280e+04  .53800561e-02", 
" -.11929434e+02  .52193685e+02 -.55722275e+01 -.56770185e+03  .58165970e-02", 
" -.11688115e+02  .50316268e+02 -.56837264e+02  .10973159e+04  .60149829e-02", 
" -.11423105e+02  .47845372e+02 -.84066381e+02  .22296495e+04  .63805353e-02", 
" -.11124784e+02  .41938320e+02  .16169132e+03 -.26457826e+04  .68114471e-02", 
" -.10793356e+02  .37988827e+02  .22915923e+03 -.37797262e+04  .71391141e-02", 
" -.10408529e+02  .31076322e+02  .47528897e+03 -.83784156e+04  .77030949e-02", 
" -.99698535e+01  .27758442e+02  .40126289e+03 -.59175956e+04  .82496278e-02", 
" -.94277848e+01  .22457941e+02  .43505615e+03 -.61754951e+04  .88817938e-02", 
" -.87036630e+01  .15405453e+02  .50395102e+03 -.75064714e+04  .95749417e-02", 
" -.75527630e+01  .13496536e+01  .73439917e+03 -.10311275e+05  .12617093e-01", 
" -.65159095e+01 -.68391561e+01  .61173349e+03 -.63675433e+04  .16532393e-01", 
" -.52016556e+01 -.33792587e+02  .16356583e+04 -.24556289e+05  .23737775e-01", 
" -.42993218e+01 -.52807259e+02  .28031860e+04 -.52370759e+05  .33099174e-01", 
"coactt    6 10  3   35", " -.99812121e+02  .28277094e+04 -.53194820e+05  .50319861e+06  .11120948e+00", 
" -.95890197e+02  .26256945e+04 -.47688225e+05  .43227086e+06  .84355256e-01", 
" -.90354417e+02  .23432836e+04 -.40362438e+05  .34698408e+06  .57025117e-01", 
" -.86141760e+02  .21591972e+04 -.37115652e+05  .33127313e+06  .40733766e-01", 
" -.81810092e+02  .19710790e+04 -.33183383e+05  .29613312e+06  .30212493e-01", 
" -.79201549e+02  .18556677e+04 -.30458897e+05  .26603343e+06  .24956867e-01", 
" -.77326797e+02  .17775690e+04 -.28708182e+05  .24614440e+06  .22782819e-01", 
" -.75845404e+02  .17120078e+04 -.27050831e+05  .22636422e+06  .21082025e-01", 
" -.74618240e+02  .16596306e+04 -.25846132e+05  .21367642e+06  .19374162e-01", 
" -.73575339e+02  .16154301e+04 -.24783065e+05  .20161462e+06  .18271730e-01", 
" -.72668487e+02  .15807232e+04 -.24166158e+05  .19736863e+06  .17563452e-01", 
" -.71862697e+02  .15514726e+04 -.23713476e+05  .19533101e+06  .16509509e-01", 
" -.71136334e+02  .15233644e+04 -.23129599e+05  .18930678e+06  .15804580e-01", 
" -.68266215e+02  .14096609e+04 -.20606571e+05  .16212127e+06  .13511076e-01", 
" -.66192586e+02  .13338316e+04 -.19142151e+05  .14858585e+06  .12098590e-01", 
" -.64547461e+02  .12751160e+04 -.18048142e+05  .13907068e+06  .10897401e-01", 
" -.63174116e+02  .12247127e+04 -.17017134e+05  .12903277e+06  .99813871e-02", 
" -.61993883e+02  .11817033e+04 -.16118115e+05  .11999177e+06  .94922575e-02", 
" -.60956501e+02  .11446246e+04 -.15321617e+05  .11106095e+06  .87513974e-02", 
" -.60034358e+02  .11135821e+04 -.14778477e+05  .10697407e+06  .84380700e-02", 
" -.59202798e+02  .10876203e+04 -.14434426e+05  .10598917e+06  .79983792e-02", 
" -.58438386e+02  .10622235e+04 -.13961635e+05  .10152495e+06  .75042125e-02", 
" -.57732311e+02  .10400801e+04 -.13645812e+05  .10029203e+06  .73933934e-02", 
" -.57073963e+02  .10185362e+04 -.13258571e+05  .96992448e+05  .72467119e-02", 
" -.56458037e+02  .99825851e+03 -.12869030e+05  .93214892e+05  .70794717e-02", 
" -.55878040e+02  .97946252e+03 -.12520594e+05  .89961739e+05  .68420637e-02", 
" -.55332929e+02  .96230596e+03 -.12215097e+05  .87138113e+05  .67705983e-02", 
" -.54817737e+02  .94647499e+03 -.11954322e+05  .85076181e+05  .65524265e-02", 
" -.54326249e+02  .93116777e+03 -.11682460e+05  .82520759e+05  .64212855e-02", 
" -.53858306e+02  .91696812e+03 -.11449498e+05  .80722058e+05  .62266338e-02", 
" -.53410937e+02  .90344766e+03 -.11231348e+05  .79097257e+05  .59762430e-02", 
" -.52979809e+02  .88994990e+03 -.10970512e+05  .76271307e+05  .59417572e-02", 
" -.52565160e+02  .87703954e+03 -.10733473e+05  .74089308e+05  .58970964e-02", 
" -.52167519e+02  .86537655e+03 -.10554521e+05  .72869253e+05  .57083781e-02", 
" -.51783850e+02  .85423094e+03 -.10386229e+05  .71724066e+05  .55211284e-02", 
" -.51412224e+02  .84307049e+03 -.10193751e+05  .70005633e+05  .53816173e-02", 
" -.51054848e+02  .83302984e+03 -.10048165e+05  .69004971e+05  .53003264e-02", 
" -.50708914e+02  .82355860e+03 -.99226993e+04  .68338416e+05  .52616742e-02", 
" -.50370841e+02  .81410379e+03 -.97864507e+04  .67402835e+05  .53188598e-02", 
" -.50041570e+02  .80437307e+03 -.96155173e+04  .65824541e+05  .52024049e-02", 
" -.49719863e+02  .79442867e+03 -.94122066e+04  .63514269e+05  .52177207e-02", 
" -.49406775e+02  .78508523e+03 -.92403628e+04  .61842730e+05  .51589538e-02", 
" -.49102066e+02  .77601452e+03 -.90676300e+04  .59995716e+05  .51621049e-02", 
" -.48806732e+02  .76761958e+03 -.89247357e+04  .58692866e+05  .51254075e-02", 
" -.48517869e+02  .75967530e+03 -.88144646e+04  .58123498e+05  .50560203e-02", 
" -.48234897e+02  .75167873e+03 -.86862753e+04  .57157453e+05  .49622051e-02", 
" -.47959057e+02  .74466007e+03 -.86238701e+04  .57496232e+05  .49043266e-02", 
" -.47688180e+02  .73726821e+03 -.85181349e+04  .56899727e+05  .49423784e-02", 
" -.47421647e+02  .72953355e+03 -.83700612e+04  .55248342e+05  .48854459e-02", 
" -.47160825e+02  .72235215e+03 -.82598427e+04  .54476967e+05  .48207323e-02", 
" -.46905072e+02  .71539311e+03 -.81573446e+04  .53797210e+05  .48200378e-02", 
" -.46653814e+02  .70836391e+03 -.80325369e+04  .52544617e+05  .48013195e-02", 
" -.46407712e+02  .70168356e+03 -.79337274e+04  .51882035e+05  .47579572e-02", 
" -.46164713e+02  .69479410e+03 -.78016936e+04  .50368363e+05  .47551203e-02", 
" -.45927047e+02  .68844210e+03 -.77069227e+04  .49691995e+05  .46213093e-02", 
" -.45693382e+02  .68231832e+03 -.76309717e+04  .49496850e+05  .45988824e-02", 
" -.45463995e+02  .67619850e+03 -.75345996e+04  .48681246e+05  .45258244e-02", 
" -.45236272e+02  .66986263e+03 -.74245425e+04  .47696619e+05  .45086724e-02", 
" -.45012476e+02  .66399469e+03 -.73429305e+04  .47249339e+05  .44995863e-02", 
" -.44791175e+02  .65817561e+03 -.72603293e+04  .46714122e+05  .44575707e-02", 
" -.44573000e+02  .65207853e+03 -.71423647e+04  .45360144e+05  .43658980e-02", 
" -.44359324e+02  .64655936e+03 -.70627889e+04  .44786020e+05  .42910852e-02", 
" -.44148334e+02  .64113929e+03 -.69887050e+04  .44365564e+05  .42680465e-02", 
" -.43939349e+02  .63554724e+03 -.68879618e+04  .43246023e+05  .42177372e-02", 
" -.43732625e+02  .63014393e+03 -.68059865e+04  .42601672e+05  .42145189e-02", 
" -.43529439e+02  .62500788e+03 -.67434588e+04  .42445471e+05  .41792154e-02", 
" -.43329425e+02  .62037772e+03 -.67112493e+04  .42806382e+05  .41318931e-02", 
" -.43128957e+02  .61481850e+03 -.66040602e+04  .41607429e+05  .41237883e-02", 
" -.42932932e+02  .61012747e+03 -.65538769e+04  .41495056e+05  .41348347e-02", 
" -.42739472e+02  .60538811e+03 -.64963275e+04  .41287115e+05  .41437656e-02", 
" -.42546188e+02  .60023615e+03 -.64085087e+04  .40474482e+05  .41458687e-02", 
" -.42355509e+02  .59548049e+03 -.63460932e+04  .40156695e+05  .41356492e-02", 
" -.42166826e+02  .59071277e+03 -.62777973e+04  .39717886e+05  .41506444e-02", 
" -.41979694e+02  .58596788e+03 -.62049131e+04  .39094967e+05  .41315816e-02", 
" -.41794890e+02  .58136673e+03 -.61406827e+04  .38694968e+05  .41311974e-02", 
" -.41611421e+02  .57664165e+03 -.60623404e+04  .37953212e+05  .40737936e-02", 
" -.41429857e+02  .57234998e+03 -.60170032e+04  .37916928e+05  .40129090e-02", 
" -.41250537e+02  .56799119e+03 -.59589477e+04  .37594230e+05  .40256574e-02", 
" -.41072473e+02  .56360842e+03 -.58952634e+04  .37110639e+05  .40233935e-02", 
" -.40897236e+02  .55979068e+03 -.58672483e+04  .37289042e+05  .40603571e-02", 
" -.40722009e+02  .55540786e+03 -.57980669e+04  .36678129e+05  .40748335e-02", 
" -.40547998e+02  .55121924e+03 -.57373895e+04  .36130312e+05  .40512824e-02", 
" -.40374774e+02  .54686115e+03 -.56677283e+04  .35568055e+05  .40487341e-02", 
" -.40203074e+02  .54258319e+03 -.55969434e+04  .34867671e+05  .40007459e-02", 
" -.40033133e+02  .53842546e+03 -.55295629e+04  .34153256e+05  .39551440e-02", 
" -.39865748e+02  .53471687e+03 -.54915107e+04  .34045728e+05  .39312810e-02", 
" -.39698045e+02  .53071716e+03 -.54408351e+04  .33841725e+05  .39026825e-02", 
" -.39532048e+02  .52691475e+03 -.53945413e+04  .33580458e+05  .39131808e-02", 
" -.39366489e+02  .52290941e+03 -.53335776e+04  .33049097e+05  .38790955e-02", 
" -.39201854e+02  .51899031e+03 -.52787138e+04  .32665689e+05  .38938979e-02", 
" -.39038990e+02  .51528454e+03 -.52390052e+04  .32616524e+05  .38416716e-02", 
" -.38876312e+02  .51134868e+03 -.51757894e+04  .32007345e+05  .38453825e-02", 
" -.38715004e+02  .50753396e+03 -.51238257e+04  .31683123e+05  .38215455e-02", 
" -.38554509e+02  .50381434e+03 -.50747663e+04  .31367723e+05  .37963938e-02", 
" -.38394469e+02  .50006773e+03 -.50247832e+04  .31047681e+05  .37694098e-02", 
" -.38235702e+02  .49640012e+03 -.49819962e+04  .30958857e+05  .37911655e-02", 
" -.38076905e+02  .49229309e+03 -.48974823e+04  .29898190e+05  .37794828e-02", 
" -.37920542e+02  .48871556e+03 -.48471518e+04  .29475410e+05  .37827242e-02", 
" -.37764195e+02  .48496249e+03 -.47798762e+04  .28653330e+05  .37612126e-02", 
" -.37608602e+02  .48137988e+03 -.47294049e+04  .28266082e+05  .37415649e-02", 
" -.37452767e+02  .47759796e+03 -.46647017e+04  .27583642e+05  .37381158e-02", 
" -.37297696e+02  .47387642e+03 -.46040775e+04  .26997633e+05  .37440451e-02", 
" -.37143741e+02  .47040954e+03 -.45570396e+04  .26637845e+05  .37222170e-02", 
" -.36990636e+02  .46683490e+03 -.44997384e+04  .26066676e+05  .37048306e-02", 
" -.36837443e+02  .46316109e+03 -.44436783e+04  .25653543e+05  .36867598e-02", 
" -.36686591e+02  .45985643e+03 -.44004962e+04  .25338785e+05  .36666704e-02", 
" -.36535338e+02  .45660905e+03 -.43686787e+04  .25340391e+05  .36439557e-02", 
" -.36384322e+02  .45315637e+03 -.43152303e+04  .24813739e+05  .36324265e-02", 
" -.36233720e+02  .44968116e+03 -.42602778e+04  .24259181e+05  .35317795e-02", 
" -.36084285e+02  .44663962e+03 -.42371429e+04  .24364877e+05  .35855674e-02", 
" -.35934658e+02  .44336810e+03 -.42020379e+04  .24295820e+05  .35541805e-02", 
" -.35785365e+02  .44016550e+03 -.41695864e+04  .24254952e+05  .35937398e-02", 
" -.35636544e+02  .43679869e+03 -.41178866e+04  .23771287e+05  .35955250e-02", 
" -.35488735e+02  .43357940e+03 -.40715048e+04  .23316431e+05  .35845108e-02", 
" -.35340016e+02  .43017674e+03 -.40162739e+04  .22743225e+05  .35400838e-02", 
" -.35192971e+02  .42715169e+03 -.39827534e+04  .22520658e+05  .35382304e-02", 
" -.35045199e+02  .42379633e+03 -.39299409e+04  .22007716e+05  .35254708e-02", 
" -.34897978e+02  .42059626e+03 -.38887616e+04  .21749205e+05  .34905321e-02", 
" -.34750576e+02  .41749495e+03 -.38569012e+04  .21681461e+05  .34774203e-02", 
" -.34604047e+02  .41429116e+03 -.38156827e+04  .21447660e+05  .34467632e-02", 
" -.34457648e+02  .41122117e+03 -.37805770e+04  .21300430e+05  .34347852e-02", 
" -.34311393e+02  .40813184e+03 -.37436115e+04  .21084025e+05  .34229770e-02", 
" -.34164983e+02  .40482375e+03 -.36862459e+04  .20439108e+05  .34621177e-02", 
" -.34018475e+02  .40169083e+03 -.36480025e+04  .20235284e+05  .34884211e-02", 
" -.33872188e+02  .39844576e+03 -.35932895e+04  .19608521e+05  .34506670e-02", 
" -.33726569e+02  .39546562e+03 -.35609752e+04  .19488068e+05  .34137161e-02", 
" -.33580814e+02  .39249844e+03 -.35274799e+04  .19308070e+05  .33976175e-02", 
" -.33434953e+02  .38930700e+03 -.34752865e+04  .18738463e+05  .34130798e-02", 
" -.33288833e+02  .38614172e+03 -.34278116e+04  .18303510e+05  .34145638e-02", 
" -.33142857e+02  .38309461e+03 -.33876457e+04  .17993823e+05  .33598309e-02", 
" -.32997250e+02  .38025024e+03 -.33631724e+04  .18005887e+05  .33534719e-02", 
" -.32851187e+02  .37717858e+03 -.33200535e+04  .17627365e+05  .33432262e-02", 
" -.32705323e+02  .37422916e+03 -.32890143e+04  .17558546e+05  .32747320e-02", 
" -.32559731e+02  .37144677e+03 -.32711404e+04  .17753741e+05  .33700214e-02", 
" -.32413766e+02  .36864745e+03 -.32537588e+04  .17999353e+05  .33748266e-02", 
" -.32267325e+02  .36576521e+03 -.32281363e+04  .18027589e+05  .33945205e-02", 
" -.32120693e+02  .36273240e+03 -.31900433e+04  .17805710e+05  .33202395e-02", 
" -.31973571e+02  .35975955e+03 -.31594137e+04  .17749405e+05  .33377645e-02", 
" -.31825967e+02  .35663031e+03 -.31138486e+04  .17377134e+05  .33430234e-02", 
" -.31678914e+02  .35371031e+03 -.30833281e+04  .17322181e+05  .33305018e-02", 
" -.31531142e+02  .35069291e+03 -.30458206e+04  .17084323e+05  .33309564e-02", 
" -.31382663e+02  .34760508e+03 -.30013817e+04  .16695333e+05  .33245702e-02", 
" -.31235513e+02  .34468988e+03 -.29648152e+04  .16451115e+05  .33114618e-02", 
" -.31086595e+02  .34153345e+03 -.29162583e+04  .16015593e+05  .33280244e-02", 
" -.30936378e+02  .33809355e+03 -.28405534e+04  .14911101e+05  .32739147e-02", 
" -.30787358e+02  .33520266e+03 -.28064129e+04  .14682226e+05  .32956793e-02", 
" -.30637452e+02  .33230822e+03 -.27805144e+04  .14758906e+05  .33006192e-02", 
" -.30486962e+02  .32907117e+03 -.27214713e+04  .14061399e+05  .33067645e-02", 
" -.30336489e+02  .32629673e+03 -.26973555e+04  .14018352e+05  .32962361e-02", 
" -.30184091e+02  .32307797e+03 -.26418119e+04  .13393450e+05  .33220610e-02", 
" -.30031707e+02  .32008255e+03 -.26072940e+04  .13264937e+05  .33447726e-02", 
" -.29879245e+02  .31716503e+03 -.25732617e+04  .13054342e+05  .33000358e-02", 
" -.29726849e+02  .31448974e+03 -.25611115e+04  .13344280e+05  .32753302e-02", 
" -.29572648e+02  .31152380e+03 -.25247794e+04  .13057651e+05  .32820034e-02", 
" -.29417861e+02  .30842128e+03 -.24786467e+04  .12624956e+05  .32827816e-02", 
" -.29263459e+02  .30569266e+03 -.24540972e+04  .12493067e+05  .32313593e-02", 
" -.29107991e+02  .30290081e+03 -.24271909e+04  .12370774e+05  .32832954e-02", 
" -.28950141e+02  .29962281e+03 -.23696788e+04  .11705193e+05  .33013050e-02", 
" -.28791748e+02  .29647714e+03 -.23162457e+04  .11029404e+05  .33342810e-02", 
" -.28631947e+02  .29315679e+03 -.22482003e+04  .10020720e+05  .34154628e-02", 
" -.28471469e+02  .29028428e+03 -.22246768e+04  .10032646e+05  .33690641e-02", 
" -.28309995e+02  .28730341e+03 -.21893361e+04  .97808748e+04  .33757163e-02", 
" -.28147758e+02  .28431475e+03 -.21546128e+04  .95831083e+04  .33922749e-02", 
" -.27983594e+02  .28108635e+03 -.21022836e+04  .90026199e+04  .33750413e-02", 
" -.27818361e+02  .27794215e+03 -.20600867e+04  .86580567e+04  .33637662e-02", 
" -.27651187e+02  .27479424e+03 -.20195319e+04  .83462327e+04  .34126313e-02", 
" -.27483368e+02  .27164198e+03 -.19732556e+04  .78558238e+04  .34298328e-02", 
" -.27313345e+02  .26832386e+03 -.19161621e+04  .71088968e+04  .34495155e-02", 
" -.27141593e+02  .26492758e+03 -.18573559e+04  .64094876e+04  .34671375e-02", 
" -.26968912e+02  .26167060e+03 -.18061610e+04  .58106309e+04  .34138482e-02", 
" -.26793643e+02  .25834058e+03 -.17568773e+04  .53316530e+04  .33834495e-02", 
" -.26617538e+02  .25538517e+03 -.17376464e+04  .54609388e+04  .34019824e-02", 
" -.26439161e+02  .25227349e+03 -.17071907e+04  .53754830e+04  .34585722e-02", 
" -.26257610e+02  .24883921e+03 -.16523758e+04  .47462279e+04  .34761032e-02", 
" -.26075048e+02  .24561315e+03 -.16179872e+04  .46084440e+04  .34961833e-02", 
" -.25889174e+02  .24224593e+03 -.15743677e+04  .42442070e+04  .35241513e-02", 
" -.25702555e+02  .23908142e+03 -.15330866e+04  .37400808e+04  .35328984e-02", 
" -.25511840e+02  .23575167e+03 -.14974516e+04  .36231019e+04  .35441639e-02", 
" -.25318449e+02  .23225916e+03 -.14478683e+04  .31566222e+04  .35526849e-02", 
" -.25121820e+02  .22896618e+03 -.14152015e+04  .29729037e+04  .36096311e-02", 
" -.24922839e+02  .22567191e+03 -.13835620e+04  .28331443e+04  .36271203e-02", 
" -.24719679e+02  .22223807e+03 -.13513740e+04  .27803767e+04  .36768953e-02", 
" -.24512281e+02  .21867770e+03 -.13087819e+04  .24221597e+04  .37170360e-02", 
" -.24301808e+02  .21509204e+03 -.12647224e+04  .20740727e+04  .37289971e-02", 
" -.24087142e+02  .21121984e+03 -.11911294e+04  .96483827e+03  .37456834e-02", 
" -.23868477e+02  .20774447e+03 -.11648299e+04  .10065354e+04  .37875734e-02", 
" -.23645585e+02  .20412672e+03 -.11203798e+04  .49231675e+03  .37528371e-02", 
" -.23415599e+02  .20053711e+03 -.11064800e+04  .97977449e+03  .37782075e-02", 
" -.23182232e+02  .19691839e+03 -.10752465e+04  .85544732e+03  .37525187e-02", 
" -.22941764e+02  .19363128e+03 -.10889916e+04  .18280261e+04  .37910805e-02", 
" -.22695511e+02  .19004688e+03 -.10602310e+04  .15668264e+04  .38721002e-02", 
" -.22439727e+02  .18588443e+03 -.10194776e+04  .14808385e+04  .38654521e-02", 
" -.22175363e+02  .18153759e+03 -.96787901e+03  .10751630e+04  .39260650e-02", 
" -.21901678e+02  .17715509e+03 -.93515733e+03  .13671850e+04  .39861878e-02", 
" -.21618797e+02  .17220320e+03 -.84126531e+03  .11641921e+02  .40072745e-02", 
" -.21323539e+02  .16736053e+03 -.77946714e+03 -.48074133e+03  .40609187e-02", 
" -.21014924e+02  .16171938e+03 -.66482593e+03 -.20795797e+04  .41014306e-02", 
" -.20695319e+02  .15748933e+03 -.67662134e+03 -.91601913e+03  .42249163e-02", 
" -.20357291e+02  .15203531e+03 -.60920823e+03 -.13832595e+04  .42677468e-02", 
" -.19999693e+02  .14604252e+03 -.50184111e+03 -.28472655e+04  .43640696e-02", 
" -.19619604e+02  .14009436e+03 -.42678477e+03 -.34837084e+04  .44287770e-02", 
" -.19214501e+02  .13404009e+03 -.35839258e+03 -.39879849e+04  .45622826e-02", 
" -.18777276e+02  .12785533e+03 -.30799931e+03 -.41264237e+04  .47241057e-02", 
" -.18299792e+02  .12052549e+03 -.19634187e+03 -.55664900e+04  .46570747e-02", 
" -.17769373e+02  .11257808e+03 -.99183412e+02 -.63825743e+04  .48461432e-02", 
" -.17171782e+02  .10343986e+03  .28535863e+02 -.75752535e+04  .51760828e-02", 
" -.16481749e+02  .94281582e+02  .82133122e+02 -.71049050e+04  .55617359e-02", 
" -.15645713e+02  .83113870e+02  .11519406e+03 -.51188482e+04  .58345104e-02", 
" -.14565473e+02  .69000731e+02  .20338069e+03 -.42840525e+04  .64181757e-02", 
" -.14299777e+02  .64896559e+02  .29457911e+03 -.58510421e+04  .66261615e-02", 
" -.14010831e+02  .60953393e+02  .35930734e+03 -.68428182e+04  .65509886e-02", 
" -.13691998e+02  .57028307e+02  .37907519e+03 -.66321165e+04  .70896019e-02", 
" -.13329475e+02  .50284850e+02  .57620873e+03 -.98667665e+04  .70989175e-02", 
" -.12921653e+02  .45537753e+02  .56765912e+03 -.85802420e+04  .77907522e-02", 
" -.12440442e+02  .39570264e+02  .58212242e+03 -.75800268e+04  .81767673e-02", 
" -.11842963e+02  .31091255e+02  .68322486e+03 -.79802906e+04  .91925538e-02", 
" -.11045291e+02  .17822836e+02  .99593376e+03 -.12675964e+05  .10690104e-01", 
" -.97924533e+01 -.64704072e+01  .18143245e+04 -.27454072e+05  .13441049e-01", 
" -.86557703e+01 -.26559722e+02  .23785790e+04 -.37300798e+05  .17418392e-01", 
" -.73004295e+01 -.47837511e+02  .26163965e+04 -.35985293e+05  .24820887e-01", 
" -.63342732e+01 -.64282905e+02  .28230049e+04 -.37098692e+05  .31452948e-01"
)


.urc8 = 
c("Copyright (C) James G. MacKinnon, 1995", "conc      7  8  2   40", 
" -.66344587e+01 -.58864523e+02 -.26090541e+03  .47812271e-02", 
" -.64892968e+01 -.53081704e+02 -.26920647e+03  .35318056e-02", 
" -.62856620e+01 -.47199259e+02 -.22250886e+03  .23328336e-02", 
" -.61227128e+01 -.42885567e+02 -.18661212e+03  .18200803e-02", 
" -.59504281e+01 -.39034624e+02 -.13034730e+03  .13004707e-02", 
" -.58450279e+01 -.36639585e+02 -.10655578e+03  .10756317e-02", 
" -.57679690e+01 -.34903067e+02 -.93548483e+02  .10048296e-02", 
" -.57064969e+01 -.33517501e+02 -.86310603e+02  .92358582e-03", 
" -.56553005e+01 -.32453833e+02 -.78442922e+02  .84298881e-03", 
" -.56117398e+01 -.31430134e+02 -.76579676e+02  .80108145e-03", 
" -.55730194e+01 -.30680663e+02 -.70440634e+02  .74947534e-03", 
" -.55387150e+01 -.29953968e+02 -.66061712e+02  .70666055e-03", 
" -.55074769e+01 -.29361692e+02 -.60848006e+02  .66267416e-03", 
" -.53839801e+01 -.26847700e+02 -.50525070e+02  .56789758e-03", 
" -.52918390e+01 -.25197748e+02 -.38896143e+02  .51694405e-03", 
" -.52183840e+01 -.23843979e+02 -.33353557e+02  .46468251e-03", 
" -.51562775e+01 -.22772140e+02 -.27785366e+02  .43501442e-03", 
" -.51024418e+01 -.21853566e+02 -.23125572e+02  .39743481e-03", 
" -.50547635e+01 -.21006965e+02 -.22318152e+02  .39242490e-03", 
" -.50115163e+01 -.20343285e+02 -.18275360e+02  .36625677e-03", 
" -.49722046e+01 -.19740246e+02 -.14934853e+02  .36092373e-03", 
" -.49359390e+01 -.19179415e+02 -.12591574e+02  .34311273e-03", 
" -.49023651e+01 -.18621045e+02 -.12430049e+02  .33559944e-03", 
" -.48710043e+01 -.18140672e+02 -.10580802e+02  .33148297e-03", 
" -.48413365e+01 -.17725328e+02 -.79102948e+01  .32417790e-03", 
" -.48134605e+01 -.17307477e+02 -.63839269e+01  .31228679e-03", 
" -.47870387e+01 -.16896842e+02 -.60316981e+01  .30042207e-03", 
" -.47615997e+01 -.16568347e+02 -.34533398e+01  .30001109e-03", 
" -.47375057e+01 -.16200714e+02 -.34409292e+01  .29767259e-03", 
" -.47142911e+01 -.15886145e+02 -.22894276e+01  .28899490e-03", 
" -.46919240e+01 -.15587208e+02 -.11285973e+01  .28170450e-03", 
" -.46704564e+01 -.15288796e+02 -.48975150e+00  .26888698e-03", 
" -.46498091e+01 -.15007469e+02  .19395287e+00  .25604806e-03", 
" -.46296403e+01 -.14747924e+02  .97349694e+00  .25515164e-03", 
" -.46102253e+01 -.14496817e+02  .17046340e+01  .24630517e-03", 
" -.45914346e+01 -.14244074e+02  .20186306e+01  .23600658e-03", 
" -.45731518e+01 -.14003411e+02  .26201494e+01  .23699371e-03", 
" -.45552929e+01 -.13799625e+02  .41559608e+01  .23504596e-03", 
" -.45379691e+01 -.13583312e+02  .48799668e+01  .23044660e-03", 
" -.45211238e+01 -.13348205e+02  .43640859e+01  .23165718e-03", 
" -.45047319e+01 -.13122977e+02  .40849380e+01  .23249327e-03", 
" -.44886149e+01 -.12929332e+02  .47796756e+01  .23384742e-03", 
" -.44729772e+01 -.12713565e+02  .47020066e+01  .23476320e-03", 
" -.44575680e+01 -.12528924e+02  .51592852e+01  .23223600e-03", 
" -.44424046e+01 -.12346093e+02  .54665467e+01  .22982010e-03", 
" -.44276227e+01 -.12161400e+02  .55322791e+01  .22703320e-03", 
" -.44130085e+01 -.12008171e+02  .65406324e+01  .23213638e-03", 
" -.43987356e+01 -.11835201e+02  .66353795e+01  .23064175e-03", 
" -.43847683e+01 -.11663039e+02  .66973237e+01  .23164511e-03", 
" -.43709854e+01 -.11503246e+02  .69199684e+01  .23279132e-03", 
" -.43574206e+01 -.11347740e+02  .73311900e+01  .23421940e-03", 
" -.43441355e+01 -.11183851e+02  .73361240e+01  .23504259e-03", 
" -.43310167e+01 -.11015845e+02  .68069763e+01  .23426110e-03", 
" -.43179183e+01 -.10903711e+02  .84974450e+01  .23286339e-03", 
" -.43051582e+01 -.10758085e+02  .86368010e+01  .23069530e-03", 
" -.42925328e+01 -.10612559e+02  .85362792e+01  .23080458e-03", 
" -.42801224e+01 -.10466710e+02  .84061190e+01  .22782508e-03", 
" -.42678665e+01 -.10326864e+02  .83579525e+01  .22745071e-03", 
" -.42557473e+01 -.10183727e+02  .80035174e+01  .22201071e-03", 
" -.42437371e+01 -.10054121e+02  .81661615e+01  .21907062e-03", 
" -.42319034e+01 -.99187431e+01  .81359687e+01  .21626981e-03", 
" -.42202076e+01 -.97854818e+01  .80055001e+01  .21435128e-03", 
" -.42085781e+01 -.96758623e+01  .87063541e+01  .21873174e-03", 
" -.41971065e+01 -.95510258e+01  .86441125e+01  .21600615e-03", 
" -.41857533e+01 -.94371646e+01  .88957384e+01  .21425383e-03", 
" -.41744241e+01 -.93366769e+01  .96194770e+01  .20978528e-03", 
" -.41632637e+01 -.92273150e+01  .98524298e+01  .21090342e-03", 
" -.41522584e+01 -.91127499e+01  .99447918e+01  .21406045e-03", 
" -.41413312e+01 -.89926551e+01  .96762556e+01  .20857305e-03", 
" -.41304873e+01 -.88783902e+01  .96166569e+01  .20368915e-03", 
" -.41197233e+01 -.87683938e+01  .95929439e+01  .20298215e-03", 
" -.41090682e+01 -.86666549e+01  .98583620e+01  .20186446e-03", 
" -.40984881e+01 -.85466245e+01  .91939424e+01  .20126517e-03", 
" -.40879239e+01 -.84561488e+01  .97674111e+01  .20073356e-03", 
" -.40774911e+01 -.83507502e+01  .97289358e+01  .20224677e-03", 
" -.40671151e+01 -.82453152e+01  .94515857e+01  .20343181e-03", 
" -.40568255e+01 -.81437295e+01  .94115346e+01  .20372609e-03", 
" -.40465622e+01 -.80570125e+01  .99166459e+01  .20327490e-03", 
" -.40364238e+01 -.79515842e+01  .95900161e+01  .20254436e-03", 
" -.40263396e+01 -.78560113e+01  .97324966e+01  .20062550e-03", 
" -.40162827e+01 -.77606688e+01  .96880603e+01  .20186782e-03", 
" -.40062687e+01 -.76827198e+01  .10311318e+02  .19693723e-03", 
" -.39963137e+01 -.76053016e+01  .10961819e+02  .19588131e-03", 
" -.39864672e+01 -.75096843e+01  .10862568e+02  .19466004e-03", 
" -.39766766e+01 -.74186554e+01  .10948105e+02  .19877272e-03", 
" -.39668919e+01 -.73287923e+01  .10857660e+02  .19894843e-03", 
" -.39571636e+01 -.72462425e+01  .11172764e+02  .19964178e-03", 
" -.39474950e+01 -.71631218e+01  .11453060e+02  .19831374e-03", 
" -.39378539e+01 -.70748997e+01  .11394629e+02  .20039818e-03", 
" -.39282317e+01 -.69913477e+01  .11508677e+02  .20176004e-03", 
" -.39186838e+01 -.69038482e+01  .11539684e+02  .20329480e-03", 
" -.39091533e+01 -.68172161e+01  .11354373e+02  .20217115e-03", 
" -.38996577e+01 -.67293832e+01  .11075637e+02  .20125398e-03", 
" -.38901942e+01 -.66524654e+01  .11162986e+02  .19875607e-03", 
" -.38807476e+01 -.65678554e+01  .10990423e+02  .19719976e-03", 
" -.38713118e+01 -.64996588e+01  .11415198e+02  .19659518e-03", 
" -.38618983e+01 -.64279150e+01  .11606624e+02  .19899388e-03", 
" -.38525565e+01 -.63439128e+01  .11438210e+02  .19851132e-03", 
" -.38432493e+01 -.62632985e+01  .11434598e+02  .19885697e-03", 
" -.38339122e+01 -.61950204e+01  .11838962e+02  .19794962e-03", 
" -.38246344e+01 -.61179331e+01  .11826515e+02  .19761106e-03", 
" -.38153457e+01 -.60427442e+01  .11781430e+02  .19620295e-03", 
" -.38061200e+01 -.59668863e+01  .11837998e+02  .19567427e-03", 
" -.37969014e+01 -.58964671e+01  .11961449e+02  .19415857e-03", 
" -.37876890e+01 -.58226694e+01  .11937564e+02  .19192585e-03", 
" -.37785355e+01 -.57464878e+01  .11931818e+02  .19311398e-03", 
" -.37693313e+01 -.56757217e+01  .11934102e+02  .19460606e-03", 
" -.37601938e+01 -.55993805e+01  .11776373e+02  .19357308e-03", 
" -.37510190e+01 -.55279693e+01  .11826975e+02  .19656799e-03", 
" -.37418729e+01 -.54519276e+01  .11514508e+02  .19582442e-03", 
" -.37326993e+01 -.53877427e+01  .11680536e+02  .19400815e-03", 
" -.37235244e+01 -.53161139e+01  .11492810e+02  .19311787e-03", 
" -.37144080e+01 -.52436098e+01  .11387709e+02  .19185152e-03", 
" -.37052412e+01 -.51680809e+01  .10983182e+02  .19374818e-03", 
" -.36960976e+01 -.50915070e+01  .10577425e+02  .19517253e-03", 
" -.36868826e+01 -.50297515e+01  .10718349e+02  .19621294e-03", 
" -.36777180e+01 -.49650913e+01  .10822620e+02  .19785647e-03", 
" -.36685492e+01 -.48997118e+01  .10798004e+02  .19797865e-03", 
" -.36593729e+01 -.48264288e+01  .10424042e+02  .20017366e-03", 
" -.36502345e+01 -.47545912e+01  .10097492e+02  .19909795e-03", 
" -.36410192e+01 -.46941425e+01  .10269070e+02  .19720330e-03", 
" -.36318356e+01 -.46266515e+01  .10066312e+02  .19358948e-03", 
" -.36226236e+01 -.45605991e+01  .10011033e+02  .19467967e-03", 
" -.36133934e+01 -.45016291e+01  .10195196e+02  .19700677e-03", 
" -.36041778e+01 -.44280528e+01  .97007540e+01  .19594619e-03", 
" -.35949017e+01 -.43679700e+01  .97010772e+01  .19437335e-03", 
" -.35856418e+01 -.43001070e+01  .94035362e+01  .19279516e-03", 
" -.35763609e+01 -.42396680e+01  .93564732e+01  .19132194e-03", 
" -.35670465e+01 -.41789064e+01  .94315030e+01  .19117508e-03", 
" -.35576526e+01 -.41332565e+01  .99369146e+01  .19354173e-03", 
" -.35483520e+01 -.40611929e+01  .95023753e+01  .19081762e-03", 
" -.35389668e+01 -.39926870e+01  .90884368e+01  .18769512e-03", 
" -.35295555e+01 -.39379252e+01  .93993581e+01  .18738430e-03", 
" -.35201580e+01 -.38718900e+01  .92179448e+01  .18624898e-03", 
" -.35105829e+01 -.38249677e+01  .96075564e+01  .18691801e-03", 
" -.35011097e+01 -.37570290e+01  .92978239e+01  .18643923e-03", 
" -.34915543e+01 -.37005896e+01  .94308355e+01  .18785842e-03", 
" -.34819384e+01 -.36441140e+01  .95265163e+01  .19130104e-03", 
" -.34723093e+01 -.35835115e+01  .93561549e+01  .19311905e-03", 
" -.34625965e+01 -.35354158e+01  .97934452e+01  .19238485e-03", 
" -.34528314e+01 -.34786137e+01  .96010826e+01  .19071069e-03", 
" -.34430374e+01 -.34293912e+01  .99426198e+01  .18973926e-03", 
" -.34331966e+01 -.33748272e+01  .99173200e+01  .19262324e-03", 
" -.34233192e+01 -.33165324e+01  .98519150e+01  .19578632e-03", 
" -.34133688e+01 -.32623198e+01  .98924447e+01  .19326670e-03", 
" -.34033744e+01 -.32115498e+01  .10087459e+02  .19373051e-03", 
" -.33933090e+01 -.31621628e+01  .10303275e+02  .19554630e-03", 
" -.33832034e+01 -.31035857e+01  .10156629e+02  .19656316e-03", 
" -.33729967e+01 -.30535878e+01  .10289979e+02  .20138016e-03", 
" -.33626482e+01 -.30215505e+01  .11084102e+02  .20055204e-03", 
" -.33523137e+01 -.29770682e+01  .11443575e+02  .20356178e-03", 
" -.33419673e+01 -.29199131e+01  .11381181e+02  .20510267e-03", 
" -.33314875e+01 -.28704429e+01  .11511863e+02  .20748176e-03", 
" -.33208617e+01 -.28332464e+01  .12072260e+02  .20657207e-03", 
" -.33102495e+01 -.27810380e+01  .12126093e+02  .20715099e-03", 
" -.32995481e+01 -.27227307e+01  .11912899e+02  .20501881e-03", 
" -.32887499e+01 -.26705656e+01  .11938954e+02  .20772794e-03", 
" -.32778724e+01 -.26135409e+01  .11853515e+02  .20563237e-03", 
" -.32668594e+01 -.25592651e+01  .11707506e+02  .20820702e-03", 
" -.32556758e+01 -.25199756e+01  .12131821e+02  .20765257e-03", 
" -.32445177e+01 -.24584786e+01  .11778856e+02  .20866485e-03", 
" -.32331735e+01 -.24065485e+01  .11783885e+02  .21009468e-03", 
" -.32217007e+01 -.23581880e+01  .11909463e+02  .21303761e-03", 
" -.32100398e+01 -.23150393e+01  .11954897e+02  .21194308e-03", 
" -.31982723e+01 -.22649243e+01  .11940798e+02  .21300904e-03", 
" -.31863536e+01 -.22272034e+01  .12348661e+02  .21836642e-03", 
" -.31743694e+01 -.21719334e+01  .12292944e+02  .22024946e-03", 
" -.31622146e+01 -.21149911e+01  .12019919e+02  .22380882e-03", 
" -.31498166e+01 -.20853398e+01  .12914298e+02  .22365829e-03", 
" -.31373536e+01 -.20246623e+01  .12476706e+02  .22217975e-03", 
" -.31247011e+01 -.19758391e+01  .12599638e+02  .22174290e-03", 
" -.31119258e+01 -.19058363e+01  .11949916e+02  .22430295e-03", 
" -.30988206e+01 -.18746183e+01  .12861847e+02  .22390799e-03", 
" -.30855441e+01 -.18288566e+01  .13092296e+02  .22553717e-03", 
" -.30719871e+01 -.17892406e+01  .13439713e+02  .22536869e-03", 
" -.30582417e+01 -.17354385e+01  .13218128e+02  .23098369e-03", 
" -.30442434e+01 -.16893466e+01  .13443243e+02  .23125992e-03", 
" -.30299901e+01 -.16326549e+01  .13152978e+02  .23273236e-03", 
" -.30154255e+01 -.15836397e+01  .12998954e+02  .23881994e-03", 
" -.30006395e+01 -.15159243e+01  .12231418e+02  .23330448e-03", 
" -.29854878e+01 -.14786572e+01  .12785701e+02  .23562044e-03", 
" -.29699816e+01 -.14169435e+01  .12169306e+02  .23841321e-03", 
" -.29540919e+01 -.13686440e+01  .12070063e+02  .23848961e-03", 
" -.29377889e+01 -.13199612e+01  .12031172e+02  .24688618e-03", 
" -.29210919e+01 -.12668297e+01  .11672908e+02  .24897776e-03", 
" -.29040004e+01 -.12019074e+01  .10905407e+02  .25364307e-03", 
" -.28864376e+01 -.11227991e+01  .95282712e+01  .25608526e-03", 
" -.28682525e+01 -.10913018e+01  .10140687e+02  .26312082e-03", 
" -.28494916e+01 -.10648578e+01  .11021484e+02  .27150187e-03", 
" -.28302037e+01 -.99518951e+00  .10118167e+02  .27304709e-03", 
" -.28101344e+01 -.95977170e+00  .10542888e+02  .28289985e-03", 
" -.27893551e+01 -.91383326e+00  .10690261e+02  .29018089e-03", 
" -.27676854e+01 -.88213071e+00  .11336943e+02  .29784570e-03", 
" -.27450045e+01 -.88050673e+00  .13263739e+02  .29340756e-03", 
" -.27212711e+01 -.84006390e+00  .13296122e+02  .28669789e-03", 
" -.26964306e+01 -.81928262e+00  .14177412e+02  .29372763e-03", 
" -.26702866e+01 -.78825468e+00  .14381124e+02  .29393090e-03", 
" -.26426006e+01 -.76473823e+00  .14902386e+02  .30213035e-03", 
" -.26131299e+01 -.72707690e+00  .14763202e+02  .30554476e-03", 
" -.25817345e+01 -.71568017e+00  .16178743e+02  .31920404e-03", 
" -.25477757e+01 -.68240441e+00  .16218072e+02  .32400707e-03", 
" -.25109055e+01 -.67352127e+00  .17092483e+02  .33485883e-03", 
" -.24704495e+01 -.65634475e+00  .17118263e+02  .34275594e-03", 
" -.24253182e+01 -.66207219e+00  .17507141e+02  .36583350e-03", 
" -.23746268e+01 -.64224599e+00  .17650553e+02  .38706471e-03", 
" -.23151493e+01 -.75187705e+00  .21059685e+02  .41928244e-03", 
" -.22444614e+01 -.77819869e+00  .20431142e+02  .46608227e-03", 
" -.21546129e+01 -.10412160e+01  .26930705e+02  .52949648e-03", 
" -.20315911e+01 -.13383844e+01  .31904318e+02  .61627387e-03", 
" -.19998500e+01 -.14702180e+01  .34087478e+02  .63635172e-03", 
" -.19645867e+01 -.15497227e+01  .34316939e+02  .66605892e-03", 
" -.19250627e+01 -.16681536e+01  .36673448e+02  .71006712e-03", 
" -.18796654e+01 -.18129025e+01  .38546317e+02  .77803220e-03", 
" -.18259678e+01 -.18865798e+01  .36113763e+02  .86929216e-03", 
" -.17598420e+01 -.22305471e+01  .41526455e+02  .96934879e-03", 
" -.16755213e+01 -.25970330e+01  .47200649e+02  .11237223e-02", 
" -.15577412e+01 -.29803215e+01  .50675079e+02  .13508491e-02", 
" -.13541407e+01 -.38743288e+01  .60300520e+02  .19514660e-02", 
" -.11538861e+01 -.44850351e+01  .61886915e+02  .27723463e-02", 
" -.89636583e+00 -.46854282e+01  .43875646e+02  .39036280e-02", 
" -.71271330e+00 -.36097545e+01 -.51089616e+01  .52661677e-02", 
"coc       7  9  2   35", " -.68741212e+01 -.66560360e+02 -.27314971e+03  .48756226e-02", 
" -.67352373e+01 -.60495559e+02 -.25931853e+03  .34954054e-02", 
" -.65359254e+01 -.53148859e+02 -.23362275e+03  .22936590e-02", 
" -.63745641e+01 -.48372712e+02 -.19878515e+03  .16687295e-02", 
" -.62021465e+01 -.44071547e+02 -.15142613e+03  .13154595e-02", 
" -.60969701e+01 -.41558310e+02 -.12475008e+03  .11190439e-02", 
" -.60199948e+01 -.39761149e+02 -.10857476e+03  .99167050e-03", 
" -.59588655e+01 -.38337998e+02 -.96180828e+02  .88651443e-03", 
" -.59088548e+01 -.37047028e+02 -.91232778e+02  .80989144e-03", 
" -.58654535e+01 -.36001568e+02 -.85367610e+02  .72643968e-03", 
" -.58270401e+01 -.35134133e+02 -.79884536e+02  .67974568e-03", 
" -.57927583e+01 -.34321405e+02 -.76453752e+02  .65024907e-03", 
" -.57618513e+01 -.33578857e+02 -.74044245e+02  .63388836e-03", 
" -.56391622e+01 -.30838082e+02 -.61570406e+02  .57011637e-03", 
" -.55478274e+01 -.28933086e+02 -.52384053e+02  .50591335e-03", 
" -.54741721e+01 -.27535507e+02 -.42360925e+02  .47740046e-03", 
" -.54122327e+01 -.26359051e+02 -.35961349e+02  .44734178e-03", 
" -.53586904e+01 -.25348582e+02 -.30504406e+02  .42801484e-03", 
" -.53110835e+01 -.24475781e+02 -.26352063e+02  .39550121e-03", 
" -.52682393e+01 -.23708556e+02 -.22930557e+02  .37715189e-03", 
" -.52290046e+01 -.23023512e+02 -.20063498e+02  .35624250e-03", 
" -.51931023e+01 -.22373058e+02 -.18609547e+02  .34629438e-03", 
" -.51596071e+01 -.21805819e+02 -.16119862e+02  .33025650e-03", 
" -.51283679e+01 -.21275175e+02 -.14166900e+02  .32253830e-03", 
" -.50989382e+01 -.20769532e+02 -.13054430e+02  .32280155e-03", 
" -.50710779e+01 -.20303157e+02 -.11645960e+02  .31372442e-03", 
" -.50445951e+01 -.19879595e+02 -.99087461e+01  .30340597e-03", 
" -.50194236e+01 -.19484269e+02 -.83569467e+01  .29528439e-03", 
" -.49953922e+01 -.19087019e+02 -.77579799e+01  .28955469e-03", 
" -.49722260e+01 -.18748467e+02 -.60113566e+01  .28584978e-03", 
" -.49500320e+01 -.18419730e+02 -.43889231e+01  .27154473e-03", 
" -.49286378e+01 -.18091981e+02 -.37369168e+01  .26479865e-03", 
" -.49080057e+01 -.17794588e+02 -.22520175e+01  .26532044e-03", 
" -.48881927e+01 -.17481866e+02 -.18147847e+01  .26679637e-03", 
" -.48687843e+01 -.17203846e+02 -.66247401e+00  .26686013e-03", 
" -.48500787e+01 -.16922867e+02 -.93952489e-01  .26352826e-03", 
" -.48319372e+01 -.16656062e+02  .66685441e+00  .25821085e-03", 
" -.48142206e+01 -.16400941e+02  .13920682e+01  .25210883e-03", 
" -.47969523e+01 -.16147473e+02  .17065726e+01  .25184368e-03", 
" -.47802147e+01 -.15897764e+02  .20957854e+01  .24380082e-03", 
" -.47637772e+01 -.15670006e+02  .27996017e+01  .24224616e-03", 
" -.47477443e+01 -.15440281e+02  .32076812e+01  .24119851e-03", 
" -.47319357e+01 -.15241666e+02  .41506137e+01  .23682598e-03", 
" -.47165720e+01 -.15015370e+02  .39941212e+01  .23384367e-03", 
" -.47014559e+01 -.14817568e+02  .46238288e+01  .22911253e-03", 
" -.46866580e+01 -.14623039e+02  .51916413e+01  .22961987e-03", 
" -.46721877e+01 -.14424988e+02  .54440046e+01  .22866885e-03", 
" -.46580231e+01 -.14219740e+02  .53993063e+01  .23195468e-03", 
" -.46440450e+01 -.14041615e+02  .60416703e+01  .23044751e-03", 
" -.46302520e+01 -.13874117e+02  .69252962e+01  .23369656e-03", 
" -.46167378e+01 -.13698727e+02  .73228541e+01  .22969078e-03", 
" -.46034021e+01 -.13528116e+02  .76559896e+01  .22708196e-03", 
" -.45902373e+01 -.13351992e+02  .76489997e+01  .22302836e-03", 
" -.45773322e+01 -.13185665e+02  .79454117e+01  .22272674e-03", 
" -.45645521e+01 -.13026287e+02  .83312538e+01  .22072565e-03", 
" -.45520363e+01 -.12853788e+02  .80464264e+01  .22020213e-03", 
" -.45396650e+01 -.12687994e+02  .79175791e+01  .21749525e-03", 
" -.45273658e+01 -.12549645e+02  .86327238e+01  .21892896e-03", 
" -.45153553e+01 -.12385674e+02  .85460755e+01  .21856027e-03", 
" -.45033992e+01 -.12238298e+02  .86900206e+01  .21849215e-03", 
" -.44915505e+01 -.12105803e+02  .93818540e+01  .21838152e-03", 
" -.44798549e+01 -.11961844e+02  .94187096e+01  .21911693e-03", 
" -.44682408e+01 -.11831014e+02  .98075154e+01  .21442270e-03", 
" -.44567877e+01 -.11702252e+02  .10282116e+02  .21298706e-03", 
" -.44454554e+01 -.11560080e+02  .10221878e+02  .20976202e-03", 
" -.44341560e+01 -.11436932e+02  .10540359e+02  .20747732e-03", 
" -.44229879e+01 -.11306168e+02  .10471697e+02  .20660046e-03", 
" -.44119881e+01 -.11175716e+02  .10586864e+02  .20493152e-03", 
" -.44010719e+01 -.11048443e+02  .10612381e+02  .20254109e-03", 
" -.43903032e+01 -.10918369e+02  .10607956e+02  .20147292e-03", 
" -.43795575e+01 -.10796384e+02  .10766452e+02  .20071681e-03", 
" -.43689191e+01 -.10678662e+02  .11045611e+02  .20037622e-03", 
" -.43583556e+01 -.10566098e+02  .11451835e+02  .20177928e-03", 
" -.43479137e+01 -.10435289e+02  .11003901e+02  .19657314e-03", 
" -.43374812e+01 -.10316883e+02  .10963854e+02  .19566292e-03", 
" -.43271430e+01 -.10204649e+02  .11161309e+02  .19409184e-03", 
" -.43168790e+01 -.10091929e+02  .11347604e+02  .19135123e-03", 
" -.43066789e+01 -.99797506e+01  .11348727e+02  .19152334e-03", 
" -.42965758e+01 -.98610945e+01  .11144084e+02  .19096232e-03", 
" -.42865198e+01 -.97609236e+01  .11624464e+02  .18801359e-03", 
" -.42765302e+01 -.96502432e+01  .11610092e+02  .18800963e-03", 
" -.42666033e+01 -.95386417e+01  .11449730e+02  .18719378e-03", 
" -.42566993e+01 -.94404364e+01  .11794368e+02  .18668139e-03", 
" -.42468013e+01 -.93444783e+01  .12159847e+02  .19042923e-03", 
" -.42370308e+01 -.92360591e+01  .12029564e+02  .19037142e-03", 
" -.42272824e+01 -.91404059e+01  .12364225e+02  .19359611e-03", 
" -.42176299e+01 -.90317648e+01  .12186608e+02  .19192703e-03", 
" -.42079343e+01 -.89335511e+01  .12224860e+02  .19150807e-03", 
" -.41983255e+01 -.88304234e+01  .11988463e+02  .19178503e-03", 
" -.41886875e+01 -.87448586e+01  .12476211e+02  .19076490e-03", 
" -.41791187e+01 -.86509316e+01  .12578370e+02  .18785674e-03", 
" -.41695794e+01 -.85590804e+01  .12583078e+02  .19080325e-03", 
" -.41600766e+01 -.84723076e+01  .12808758e+02  .18962820e-03", 
" -.41506414e+01 -.83761967e+01  .12719179e+02  .19079957e-03", 
" -.41412198e+01 -.82762977e+01  .12365913e+02  .18924847e-03", 
" -.41318380e+01 -.81842674e+01  .12347990e+02  .19290519e-03", 
" -.41225445e+01 -.80869330e+01  .12245120e+02  .19370761e-03", 
" -.41132060e+01 -.80029888e+01  .12538094e+02  .19677112e-03", 
" -.41039062e+01 -.79107259e+01  .12416126e+02  .19663010e-03", 
" -.40946076e+01 -.78202635e+01  .12378625e+02  .19837425e-03", 
" -.40853318e+01 -.77344183e+01  .12420306e+02  .19750760e-03", 
" -.40761045e+01 -.76422182e+01  .12247305e+02  .19679344e-03", 
" -.40668765e+01 -.75551897e+01  .12218595e+02  .19419046e-03", 
" -.40576232e+01 -.74709850e+01  .12153363e+02  .19302957e-03", 
" -.40484141e+01 -.73887230e+01  .12181840e+02  .19307312e-03", 
" -.40391902e+01 -.73146954e+01  .12537121e+02  .19043838e-03", 
" -.40300275e+01 -.72293558e+01  .12532599e+02  .18951441e-03", 
" -.40208442e+01 -.71516545e+01  .12740300e+02  .18619112e-03", 
" -.40116505e+01 -.70725888e+01  .12847600e+02  .18535824e-03", 
" -.40025641e+01 -.69820277e+01  .12633267e+02  .18405601e-03", 
" -.39934034e+01 -.69034337e+01  .12664502e+02  .18517974e-03", 
" -.39842373e+01 -.68292934e+01  .12870714e+02  .18451082e-03", 
" -.39750647e+01 -.67577946e+01  .13116227e+02  .18498186e-03", 
" -.39659429e+01 -.66798727e+01  .13203160e+02  .18646004e-03", 
" -.39567949e+01 -.66034275e+01  .13322724e+02  .18768628e-03", 
" -.39476389e+01 -.65323509e+01  .13559423e+02  .18752596e-03", 
" -.39385075e+01 -.64511189e+01  .13448974e+02  .18856765e-03", 
" -.39293565e+01 -.63737799e+01  .13425303e+02  .18873280e-03", 
" -.39202450e+01 -.62901680e+01  .13319271e+02  .19070485e-03", 
" -.39110863e+01 -.62127393e+01  .13256456e+02  .19122729e-03", 
" -.39018529e+01 -.61421597e+01  .13374106e+02  .19186381e-03", 
" -.38926380e+01 -.60747880e+01  .13649380e+02  .19146199e-03", 
" -.38834822e+01 -.59929474e+01  .13474683e+02  .18904911e-03", 
" -.38742700e+01 -.59176442e+01  .13426501e+02  .18745020e-03", 
" -.38650543e+01 -.58417322e+01  .13286389e+02  .18819842e-03", 
" -.38558262e+01 -.57673063e+01  .13235442e+02  .18813034e-03", 
" -.38465076e+01 -.57061309e+01  .13633970e+02  .18728388e-03", 
" -.38371739e+01 -.56394701e+01  .13795858e+02  .18557821e-03", 
" -.38278776e+01 -.55672561e+01  .13870221e+02  .18821077e-03", 
" -.38185704e+01 -.54877506e+01  .13605314e+02  .18754529e-03", 
" -.38091858e+01 -.54189819e+01  .13662159e+02  .18612264e-03", 
" -.37997881e+01 -.53546213e+01  .13884241e+02  .18688630e-03", 
" -.37903995e+01 -.52783601e+01  .13686999e+02  .18699875e-03", 
" -.37809365e+01 -.52117089e+01  .13824497e+02  .18763098e-03", 
" -.37714157e+01 -.51484059e+01  .14011647e+02  .18879131e-03", 
" -.37619292e+01 -.50755945e+01  .13977006e+02  .18949087e-03", 
" -.37523625e+01 -.50080240e+01  .13943462e+02  .19075773e-03", 
" -.37427834e+01 -.49341964e+01  .13700318e+02  .18961639e-03", 
" -.37331534e+01 -.48648185e+01  .13695491e+02  .18853290e-03", 
" -.37234584e+01 -.47929611e+01  .13492945e+02  .19074423e-03", 
" -.37137481e+01 -.47208448e+01  .13402146e+02  .18973097e-03", 
" -.37039654e+01 -.46559142e+01  .13413699e+02  .19066497e-03", 
" -.36941735e+01 -.45826879e+01  .13225401e+02  .19249285e-03", 
" -.36843061e+01 -.45211262e+01  .13461943e+02  .19115788e-03", 
" -.36743895e+01 -.44514218e+01  .13345085e+02  .19367437e-03", 
" -.36643614e+01 -.43893518e+01  .13482656e+02  .19328205e-03", 
" -.36543738e+01 -.43103168e+01  .13102728e+02  .19451644e-03", 
" -.36443139e+01 -.42368336e+01  .12904169e+02  .19613469e-03", 
" -.36341471e+01 -.41696991e+01  .12915739e+02  .19493104e-03", 
" -.36238994e+01 -.41092423e+01  .13139642e+02  .19363979e-03", 
" -.36135840e+01 -.40479564e+01  .13348224e+02  .19647139e-03", 
" -.36032287e+01 -.39820151e+01  .13353621e+02  .19789839e-03", 
" -.35927745e+01 -.39155359e+01  .13203482e+02  .19514874e-03", 
" -.35823006e+01 -.38358734e+01  .12677288e+02  .19598655e-03", 
" -.35716486e+01 -.37864694e+01  .13269771e+02  .19621513e-03", 
" -.35609218e+01 -.37212825e+01  .13266453e+02  .19717204e-03", 
" -.35501554e+01 -.36513097e+01  .13185367e+02  .19788297e-03", 
" -.35392249e+01 -.35874834e+01  .13191529e+02  .19424767e-03", 
" -.35282170e+01 -.35276348e+01  .13359698e+02  .19405892e-03", 
" -.35171186e+01 -.34583221e+01  .13160392e+02  .19750679e-03", 
" -.35059389e+01 -.33870262e+01  .12995754e+02  .19663904e-03", 
" -.34945970e+01 -.33196750e+01  .12787569e+02  .19905108e-03", 
" -.34830603e+01 -.32707011e+01  .13173883e+02  .20054748e-03", 
" -.34714836e+01 -.32135371e+01  .13483184e+02  .20597152e-03", 
" -.34596850e+01 -.31604070e+01  .13776029e+02  .20618888e-03", 
" -.34478615e+01 -.30910893e+01  .13535396e+02  .20556847e-03", 
" -.34358662e+01 -.30330150e+01  .13777602e+02  .20616606e-03", 
" -.34236855e+01 -.29754236e+01  .13892606e+02  .21089307e-03", 
" -.34113437e+01 -.29134928e+01  .13906986e+02  .20887404e-03", 
" -.33989044e+01 -.28402462e+01  .13618924e+02  .21060489e-03", 
" -.33861886e+01 -.27729592e+01  .13434778e+02  .21309541e-03", 
" -.33733138e+01 -.26988969e+01  .12942187e+02  .21612542e-03", 
" -.33602200e+01 -.26423675e+01  .13112837e+02  .21875226e-03", 
" -.33469602e+01 -.25750564e+01  .12905546e+02  .22320845e-03", 
" -.33334653e+01 -.25136449e+01  .12888255e+02  .22652504e-03", 
" -.33197583e+01 -.24454634e+01  .12705137e+02  .22479472e-03", 
" -.33057821e+01 -.23801395e+01  .12714875e+02  .22731075e-03", 
" -.32914642e+01 -.23241301e+01  .12893788e+02  .22580627e-03", 
" -.32768625e+01 -.22732600e+01  .13264417e+02  .22591899e-03", 
" -.32620254e+01 -.22188681e+01  .13533184e+02  .23216983e-03", 
" -.32469176e+01 -.21494665e+01  .13295801e+02  .23284146e-03", 
" -.32313849e+01 -.20858282e+01  .13140606e+02  .23355710e-03", 
" -.32154939e+01 -.20333750e+01  .13562000e+02  .23743596e-03", 
" -.31991988e+01 -.19728911e+01  .13532520e+02  .24236210e-03", 
" -.31825601e+01 -.19001570e+01  .13180244e+02  .24405824e-03", 
" -.31654310e+01 -.18213229e+01  .12553471e+02  .24629574e-03", 
" -.31477223e+01 -.17578707e+01  .12374029e+02  .24915879e-03", 
" -.31295468e+01 -.17062391e+01  .12746164e+02  .25169748e-03", 
" -.31108193e+01 -.16583681e+01  .13320781e+02  .24965383e-03", 
" -.30914618e+01 -.15994725e+01  .13354087e+02  .24988367e-03", 
" -.30713635e+01 -.15397209e+01  .13219724e+02  .25236966e-03", 
" -.30505270e+01 -.14696006e+01  .12958615e+02  .25504233e-03", 
" -.30289338e+01 -.13918686e+01  .12494122e+02  .25940016e-03", 
" -.30063740e+01 -.13306748e+01  .12368180e+02  .27108542e-03", 
" -.29827649e+01 -.12813090e+01  .12764892e+02  .27695837e-03", 
" -.29579431e+01 -.12363939e+01  .13395971e+02  .28061895e-03", 
" -.29317401e+01 -.11876340e+01  .13741218e+02  .28921667e-03", 
" -.29038986e+01 -.11666118e+01  .14745781e+02  .29481882e-03", 
" -.28744842e+01 -.11232982e+01  .15312338e+02  .30146323e-03", 
" -.28430318e+01 -.10617694e+01  .15176215e+02  .30745512e-03", 
" -.28093520e+01 -.95876379e+00  .13560180e+02  .31209894e-03", 
" -.27724191e+01 -.92903729e+00  .14442989e+02  .32335229e-03", 
" -.27318731e+01 -.89461706e+00  .14630150e+02  .33423694e-03", 
" -.26868406e+01 -.87321885e+00  .15354113e+02  .34510039e-03", 
" -.26356407e+01 -.87996592e+00  .16631007e+02  .36907054e-03", 
" -.25766037e+01 -.86240585e+00  .16592618e+02  .39125292e-03", 
" -.25059117e+01 -.88899173e+00  .17474260e+02  .43560532e-03", 
" -.24168595e+01 -.10174902e+01  .21053215e+02  .48410893e-03", 
" -.22939275e+01 -.13234665e+01  .26504285e+02  .58961728e-03", 
" -.22625873e+01 -.13522349e+01  .26039372e+02  .60474631e-03", 
" -.22274502e+01 -.15125354e+01  .29498292e+02  .64363326e-03", 
" -.21881276e+01 -.16426702e+01  .32039593e+02  .68953478e-03", 
" -.21431044e+01 -.17396735e+01  .32666156e+02  .79949782e-03", 
" -.20896376e+01 -.19234464e+01  .35152367e+02  .81905033e-03", 
" -.20244342e+01 -.21302510e+01  .36391989e+02  .87807387e-03", 
" -.19420738e+01 -.22122451e+01  .32979930e+02  .99982834e-03", 
" -.18241405e+01 -.26074038e+01  .33262872e+02  .12315609e-02", 
" -.16248123e+01 -.35250204e+01  .45341418e+02  .17523241e-02", 
" -.14267291e+01 -.40742408e+01  .45130160e+02  .24215773e-02", 
" -.11650039e+01 -.50278652e+01  .48853386e+02  .39211015e-02", 
" -.96862524e+00 -.53665644e+01  .38809398e+02  .53741439e-02", 
"coct      7 10  2   40", " -.71176284e+01 -.71415522e+02 -.34377777e+03  .47038775e-02", 
" -.69681068e+01 -.66813620e+02 -.27988025e+03  .35876255e-02", 
" -.67703789e+01 -.58642308e+02 -.29137077e+03  .24463477e-02", 
" -.66055213e+01 -.54477190e+02 -.20445358e+03  .17670974e-02", 
" -.64368804e+01 -.49402604e+02 -.16799433e+03  .13626322e-02", 
" -.63332740e+01 -.46293454e+02 -.15629986e+03  .11660120e-02", 
" -.62578773e+01 -.44152280e+02 -.14312387e+03  .10261371e-02", 
" -.61978148e+01 -.42523348e+02 -.13318600e+03  .95118089e-03", 
" -.61469293e+01 -.41394405e+02 -.11718605e+03  .89693739e-03", 
" -.61034476e+01 -.40333922e+02 -.10844558e+03  .84409010e-03", 
" -.60652158e+01 -.39441241e+02 -.99602902e+02  .79744087e-03", 
" -.60311700e+01 -.38637231e+02 -.92995440e+02  .75209719e-03", 
" -.60002852e+01 -.37863555e+02 -.88595647e+02  .71010730e-03", 
" -.58774786e+01 -.35091136e+02 -.67841031e+02  .56264535e-03", 
" -.57871921e+01 -.32897722e+02 -.62285710e+02  .50873472e-03", 
" -.57139260e+01 -.31361320e+02 -.52350836e+02  .47081986e-03", 
" -.56524155e+01 -.30060716e+02 -.45540700e+02  .45545522e-03", 
" -.55988748e+01 -.29044175e+02 -.36082840e+02  .44199348e-03", 
" -.55514693e+01 -.28138717e+02 -.29240071e+02  .42471089e-03", 
" -.55087764e+01 -.27319162e+02 -.24538444e+02  .41371731e-03", 
" -.54698951e+01 -.26474966e+02 -.25137700e+02  .38851539e-03", 
" -.54340277e+01 -.25787058e+02 -.21573125e+02  .36602659e-03", 
" -.54007378e+01 -.25151043e+02 -.19427444e+02  .35515353e-03", 
" -.53693747e+01 -.24623003e+02 -.15258366e+02  .35164116e-03", 
" -.53400786e+01 -.24087585e+02 -.13025192e+02  .34501603e-03", 
" -.53123790e+01 -.23566254e+02 -.12397895e+02  .33697659e-03", 
" -.52859896e+01 -.23129232e+02 -.95683608e+01  .33242301e-03", 
" -.52608714e+01 -.22687012e+02 -.81302547e+01  .33084240e-03", 
" -.52369818e+01 -.22266109e+02 -.66638639e+01  .32259200e-03", 
" -.52139525e+01 -.21897906e+02 -.41817019e+01  .30371438e-03", 
" -.51919333e+01 -.21508882e+02 -.32807486e+01  .28874086e-03", 
" -.51706768e+01 -.21149894e+02 -.20120647e+01  .28748565e-03", 
" -.51501680e+01 -.20799130e+02 -.10260508e+01  .28032031e-03", 
" -.51303785e+01 -.20461472e+02 -.36416127e+00  .27994322e-03", 
" -.51111809e+01 -.20133137e+02  .13886858e+00  .27525058e-03", 
" -.50924819e+01 -.19840100e+02  .15708271e+01  .26880747e-03", 
" -.50743717e+01 -.19545946e+02  .23667114e+01  .27037695e-03", 
" -.50567567e+01 -.19243570e+02  .21401649e+01  .27269697e-03", 
" -.50395994e+01 -.18962908e+02  .23627411e+01  .27690075e-03", 
" -.50228471e+01 -.18702933e+02  .32897321e+01  .27279089e-03", 
" -.50064922e+01 -.18433848e+02  .33027799e+01  .26826692e-03", 
" -.49904595e+01 -.18200504e+02  .45572175e+01  .26270031e-03", 
" -.49747608e+01 -.17972019e+02  .55794578e+01  .26210871e-03", 
" -.49594224e+01 -.17734153e+02  .58285092e+01  .25863050e-03", 
" -.49444750e+01 -.17502269e+02  .63246323e+01  .25443041e-03", 
" -.49298190e+01 -.17265768e+02  .62580529e+01  .25263069e-03", 
" -.49154369e+01 -.17040754e+02  .63784214e+01  .25576281e-03", 
" -.49011629e+01 -.16862593e+02  .80340652e+01  .25309591e-03", 
" -.48871884e+01 -.16663259e+02  .85479047e+01  .25041689e-03", 
" -.48734724e+01 -.16471020e+02  .91779244e+01  .25179703e-03", 
" -.48599617e+01 -.16272116e+02  .92039752e+01  .25107180e-03", 
" -.48467042e+01 -.16090870e+02  .10012301e+02  .25021849e-03", 
" -.48336614e+01 -.15896380e+02  .10199119e+02  .24699607e-03", 
" -.48207932e+01 -.15715392e+02  .10702608e+02  .24458577e-03", 
" -.48081023e+01 -.15528313e+02  .10784214e+02  .24405423e-03", 
" -.47955640e+01 -.15353442e+02  .11223198e+02  .24610944e-03", 
" -.47831912e+01 -.15189397e+02  .11879162e+02  .23819054e-03", 
" -.47709710e+01 -.15028681e+02  .12585082e+02  .24052062e-03", 
" -.47588950e+01 -.14865144e+02  .13000206e+02  .23504257e-03", 
" -.47469361e+01 -.14716695e+02  .13990661e+02  .23528250e-03", 
" -.47351870e+01 -.14553799e+02  .14096925e+02  .23437959e-03", 
" -.47235580e+01 -.14391708e+02  .14265269e+02  .23194645e-03", 
" -.47120929e+01 -.14221118e+02  .14002428e+02  .23102888e-03", 
" -.47006896e+01 -.14069671e+02  .14268165e+02  .23171813e-03", 
" -.46893946e+01 -.13912001e+02  .14239625e+02  .22953378e-03", 
" -.46781726e+01 -.13775103e+02  .14654683e+02  .23009415e-03", 
" -.46670908e+01 -.13633679e+02  .15012015e+02  .22905689e-03", 
" -.46560489e+01 -.13500612e+02  .15514676e+02  .22600834e-03", 
" -.46451596e+01 -.13360195e+02  .15712385e+02  .22579157e-03", 
" -.46343210e+01 -.13235268e+02  .16367874e+02  .22623127e-03", 
" -.46235943e+01 -.13101836e+02  .16755169e+02  .22376659e-03", 
" -.46129802e+01 -.12971856e+02  .17001683e+02  .22334353e-03", 
" -.46024505e+01 -.12840177e+02  .17271543e+02  .22113364e-03", 
" -.45919594e+01 -.12719222e+02  .17622767e+02  .21748583e-03", 
" -.45815738e+01 -.12575530e+02  .17000068e+02  .21718735e-03", 
" -.45713112e+01 -.12441771e+02  .16799788e+02  .21538277e-03", 
" -.45610689e+01 -.12324657e+02  .17217886e+02  .21841541e-03", 
" -.45509585e+01 -.12188229e+02  .16885577e+02  .21629745e-03", 
" -.45408671e+01 -.12076089e+02  .17571623e+02  .21600601e-03", 
" -.45308287e+01 -.11955337e+02  .17634660e+02  .21603571e-03", 
" -.45208894e+01 -.11835296e+02  .17846490e+02  .21365612e-03", 
" -.45109952e+01 -.11704645e+02  .17383450e+02  .21264438e-03", 
" -.45010652e+01 -.11595325e+02  .17688700e+02  .21193144e-03", 
" -.44913057e+01 -.11461224e+02  .17026216e+02  .21108383e-03", 
" -.44815982e+01 -.11331085e+02  .16453476e+02  .21161482e-03", 
" -.44718821e+01 -.11221435e+02  .16643992e+02  .21133225e-03", 
" -.44622464e+01 -.11103729e+02  .16658516e+02  .21171746e-03", 
" -.44526400e+01 -.10997042e+02  .16960824e+02  .21262325e-03", 
" -.44430582e+01 -.10893849e+02  .17292756e+02  .21279278e-03", 
" -.44334729e+0