#' missingEndDateTime
#
#' @description missingEndDateTime registers all ESM questionnaires with a missing time stamp of when they were finished.
#
#' @param esDf a data.frame. A single ESM dataset. It must contain the 2 columns that hold the date-time object for when an ESM questionnaire was started and finished, respectively.
#
#' @param RELEVANTVN_ES a list. This list is generated by function \code{\link{setES}} and it is extended once either by function \code{\link{genDateTime}} or by function \code{\link{splitDateTime}}.
#
#' @details Both new columns are dummy variables: 1 = true = end date/end time is missing, 0 = false = end date/end time is present.
#
#' @return \code{esDf} with the additional columns NOENDDATE and NOENDTIME. See \strong{Details} for more information
#
#' @examples
#' # o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o
#' # Prerequisites in order to execute missingEndDateTime. Start -------
#' # Use example list delivered with the package
#' RELEVANTVN_ES <- RELEVANTVN_ESext
#' # esAssigned is a list of datasets, delivered with the package. It is
#' # the result of the assignment of the ESM questionnaires to ALL 8
#' # participants in the reference dataset.
#' # Prerequisites in order to execute missingEndDateTime. End ---------
#' # -------------------------------------------------------
#' # Run function 17 of 28; see esmprep functions' hierarchy.
#' # -------------------------------------------------------
#' # esAssigned[["ES"]] is one of the results of function 'esAssign'.
#' noEndDf <- missingEndDateTime(esAssigned[["ES"]], RELEVANTVN_ES)
#' # o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o
#
#' @seealso Exemplary code (fully executable) in the documentation of \code{\link{esmprep}} (function 17 of 28).
#
#' @export
#
missingEndDateTime <- function(esDf, RELEVANTVN_ES = NULL) {
	
	
	# Possible errors when passing arguments to the function -----------------------------
    if(!is.data.frame(esDf)) {
        stop("Argument 'esDf' must be of type data.frame.")
    }
	
	
	# Error handling function for all set-up lists generated by setES and setREF.
    # Both lists RELEVANTVN_ES and RELEVANTVN_REF get extended either by function
    # genDateTime or by function splitDateTime!
    SETUPLISTCheck(RELEVANTINFO_ES=NULL,
    			   RELEVANTVN_ES=RELEVANTVN_ES,
    			   RELEVANTVN_REF=NULL)
	
	
    noEndDate <- noEndTime <- rep(0, times = nrow(esDf))

    # remove any empty spaces from end_date and end_time
    endDateCollapsed <- gsub(" ", "", as.character(esDf[,RELEVANTVN_ES[["ES_END_DATE"]]]))
    endTimeCollapsed <- gsub(" ", "", as.character(esDf[,RELEVANTVN_ES[["ES_END_TIME"]]]))
    # Check for missing end_date and missing end_time
    idxNoEndDate <- nchar(endDateCollapsed)==0 | is.na(endDateCollapsed)
    idxNoEndTime <- nchar(endTimeCollapsed)==0 | is.na(endTimeCollapsed)

    noEndDate[idxNoEndDate] <- 1
    noEndTime[idxNoEndTime] <- 1

    esDf[,"NOENDDATE"] <- noEndDate
    esDf[,"NOENDTIME"] <- noEndTime

    # return es dataset with 2 new columns
    return(esDf)
}
