% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sql.R
\docType{class}
\name{EplusSql}
\alias{EplusSql}
\title{Retrieve Simulation Outputs Using EnergyPlus SQLite Output File}
\description{
\code{EplusSql} class wraps SQL queries that can retrieve simulation outputs using
EnergyPlus SQLite output file.
}
\details{
SQLite output is an optional output format for EnergyPlus. It will be created
if there is an object in class \code{Output:SQLite}. If the value of field
\code{Option} in class \code{Output:SQLite} is set to \code{"SimpleAndTabular"}, then
database tables related to the tabular reports will be also included.

There are more than 30 tables in the SQLite output file which contains all of
the data found in EnergyPlus's tabular output files, standard variable and
meter output files, plus a number of reports that are found in the
eplusout.eio output file. The full description for SQLite outputs can be
found in the EnergyPlus \emph{"Output Details and Examples"} documentation.  Note
that all column names of tables returned have been tidied, i.e. \code{"KeyValue"}
becomes \code{"key_value"}, \code{"IsMeter"} becomes \code{"is_meter"} and etc.

\code{EplusSql} class makes it possible to directly retrieve simulation results
without creating an \link{EplusJob} object. \link{EplusJob} can only get simulation
outputs after the job was successfully run before.

However, it should be noted that, unlike \link{EplusJob}, there is no checking on
whether the simulation is terminated or completed unsuccessfully or, the
parent Idf has been changed since last simulation. This means that you may
encounter some problems when retrieve data from an unsuccessful simulation.
It is suggested to carefully go through the \code{.err} file using \code{\link[=read_err]{read_err()}} to
make sure the output data in the SQLite is correct and reliable.
}
\examples{

## ------------------------------------------------
## Method `EplusSql$new`
## ------------------------------------------------

\dontrun{
if (is_avail_eplus(8.8)) {
    idf_name <- "1ZoneUncontrolled.idf"
    epw_name <-  "USA_CA_San.Francisco.Intl.AP.724940_TMY3.epw"

    idf_path <- file.path(eplus_config(8.8)$dir, "ExampleFiles", idf_name)
    epw_path <- file.path(eplus_config(8.8)$dir, "WeatherData", epw_name)

    # copy to tempdir and run the model
    idf <- read_idf(idf_path)
    idf$run(epw_path, tempdir(), echo = FALSE)

    # create from local file
    sql <- eplus_sql(file.path(tempdir(), "1ZoneUncontrolled.sql"))
}
}


## ------------------------------------------------
## Method `EplusSql$path`
## ------------------------------------------------

\dontrun{
# get path
sql$path()
}


## ------------------------------------------------
## Method `EplusSql$path_idf`
## ------------------------------------------------

\dontrun{
# get path
sql$path_idf()
}


## ------------------------------------------------
## Method `EplusSql$list_table`
## ------------------------------------------------

\dontrun{
sql$list_table()
}


## ------------------------------------------------
## Method `EplusSql$read_table`
## ------------------------------------------------

\dontrun{
# read a specific table
sql$read_table("Zones")
}


## ------------------------------------------------
## Method `EplusSql$report_data_dict`
## ------------------------------------------------

\dontrun{
sql$report_data_dict()
}


## ------------------------------------------------
## Method `EplusSql$report_data`
## ------------------------------------------------

\dontrun{
# read all report data
sql$report_data()

# specify output variables using report data dictionary
dict <- sql$report_data_dict()
sql$report_data(dict[units == "C"])

# specify output variables using 'key_value' and 'name'
sql$report_data("environment", "site outdoor air drybulb temperature")

# explicitly specify year value and time zone
sql$report_data(dict[1], year = 2020, tz = "Etc/GMT+8")

# explicitly specify case name
sql$report_data(dict[1], case = "example")

# get all possible columns
sql$report_data(dict[1], all = TRUE)

# return in a format that is similar as EnergyPlus CSV output
sql$report_data(dict[1], wide = TRUE)

# return in a format that is similar as EnergyPlus CSV output with
# extra columns
sql$report_data(dict[1], wide = TRUE, all = TRUE)

# only get data at the working hour on the first Monday
sql$report_data(dict[1], hour = 8:18, day_type = "monday", simulation_days = 1:7)

# only get specified run period data
sql$read_table("EnvironmentPeriods") # possible environment name
sql$report_data(dict[1], environment_name = "San Francisco Intl Ap CA USA TMY3 WMO#=724940")
# can also be done using 'environment_period_index' column
sql$report_data(dict[1], all = TRUE)[environment_period_index == 3L]
}


## ------------------------------------------------
## Method `EplusSql$tabular_data`
## ------------------------------------------------

\dontrun{
# read all tabular data
sql$tabular_data()

# explicitly specify data you want
str(sql$tabular_data(
    report_name = "AnnualBuildingUtilityPerformanceSummary",
    table_name = "Site and Source Energy",
    column_name = "Total Energy",
    row_name = "Total Site Energy"
))

# get tabular data in wide format and coerce numeric values
str(sql$tabular_data(
    report_name = "AnnualBuildingUtilityPerformanceSummary",
    table_name = "Site and Source Energy",
    column_name = "Total Energy",
    row_name = "Total Site Energy",
    wide = TRUE, string_value = FALSE
))
}


## ------------------------------------------------
## Method `EplusSql$print`
## ------------------------------------------------

\dontrun{
sql$print()
}

}
\author{
Hongyuan Jia
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-new}{\code{EplusSql$new()}}
\item \href{#method-path}{\code{EplusSql$path()}}
\item \href{#method-path_idf}{\code{EplusSql$path_idf()}}
\item \href{#method-list_table}{\code{EplusSql$list_table()}}
\item \href{#method-read_table}{\code{EplusSql$read_table()}}
\item \href{#method-report_data_dict}{\code{EplusSql$report_data_dict()}}
\item \href{#method-report_data}{\code{EplusSql$report_data()}}
\item \href{#method-tabular_data}{\code{EplusSql$tabular_data()}}
\item \href{#method-print}{\code{EplusSql$print()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-new"></a>}}
\if{latex}{\out{\hypertarget{method-new}{}}}
\subsection{Method \code{new()}}{
Create an \code{EplusSql} object
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusSql$new(sql)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{sql}}{A path to an local EnergyPlus SQLite output file.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An \code{EplusSql} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
if (is_avail_eplus(8.8)) {
    idf_name <- "1ZoneUncontrolled.idf"
    epw_name <-  "USA_CA_San.Francisco.Intl.AP.724940_TMY3.epw"

    idf_path <- file.path(eplus_config(8.8)$dir, "ExampleFiles", idf_name)
    epw_path <- file.path(eplus_config(8.8)$dir, "WeatherData", epw_name)

    # copy to tempdir and run the model
    idf <- read_idf(idf_path)
    idf$run(epw_path, tempdir(), echo = FALSE)

    # create from local file
    sql <- eplus_sql(file.path(tempdir(), "1ZoneUncontrolled.sql"))
}
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-path"></a>}}
\if{latex}{\out{\hypertarget{method-path}{}}}
\subsection{Method \code{path()}}{
Get the file path of current \code{EpwSql} object
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusSql$path()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$path()} returns the path of EnergyPlus SQLite file.
}

\subsection{Returns}{
A single string.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get path
sql$path()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-path_idf"></a>}}
\if{latex}{\out{\hypertarget{method-path_idf}{}}}
\subsection{Method \code{path_idf()}}{
Get the path of corresponding IDF file
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusSql$path_idf()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$path_idf()} returns the IDF file path with same name as the SQLite
file in the same folder. \code{NULL} is returned if no corresponding IDF
is found.
}

\subsection{Returns}{
NULL or a single string.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get path
sql$path_idf()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-list_table"></a>}}
\if{latex}{\out{\hypertarget{method-list_table}{}}}
\subsection{Method \code{list_table()}}{
List all table names in current EnergyPlus SQL output
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusSql$list_table()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$list_table()} returns all available table and view names in the
EnergyPlus SQLite file.
}

\subsection{Returns}{
A character vector
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
sql$list_table()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-read_table"></a>}}
\if{latex}{\out{\hypertarget{method-read_table}{}}}
\subsection{Method \code{read_table()}}{
Read a single table from current EnergyPlus SQL output
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusSql$read_table(name)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{name}}{A single string specifying the name of table to read.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$read_table()} takes a valid table \code{name} of those from
\href{../../eplusr/html/EplusSql.html#method-list_table}{\code{$list_table()}}
and returns that table data in a \code{\link[data.table:data.table]{data.table::data.table()}} format.
}

\subsection{Returns}{
A \code{\link[data.table:data.table]{data.table::data.table()}}.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# read a specific table
sql$read_table("Zones")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-report_data_dict"></a>}}
\if{latex}{\out{\hypertarget{method-report_data_dict}{}}}
\subsection{Method \code{report_data_dict()}}{
Read report data dictionary from current EnergyPlus SQL output
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusSql$report_data_dict()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$report_data_dict()} returns a \code{\link[data.table:data.table]{data.table::data.table()}} which
contains all information about report data.

For details on the meaning of each columns, please see "2.20.2.1
ReportDataDictionary Table" in EnergyPlus "Output Details and
Examples" documentation.
}

\subsection{Returns}{
A \code{\link[data.table:data.table]{data.table::data.table()}} of 10 columns:
\itemize{
\item \code{report_data_dictionary_index}: The integer used to link the
dictionary data to the variable data. Mainly useful when joining
different tables
\item \code{is_meter}: Whether report data is a meter data. Possible values:
\code{0} and \code{1}
\item \code{timestep_type}: Type of data timestep. Possible values: \code{Zone} and
\verb{HVAC System}
\item \code{key_value}: Key name of the data
\item \code{name}: Actual report data name
\item \code{reporting_frequency}:
\item \code{schedule_name}: Name of the the schedule that controls reporting
frequency.
\item \code{units}: The data units
}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
sql$report_data_dict()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-report_data"></a>}}
\if{latex}{\out{\hypertarget{method-report_data}{}}}
\subsection{Method \code{report_data()}}{
Read report data
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusSql$report_data(
  key_value = NULL,
  name = NULL,
  year = NULL,
  tz = "UTC",
  case = "auto",
  all = FALSE,
  wide = FALSE,
  period = NULL,
  month = NULL,
  day = NULL,
  hour = NULL,
  minute = NULL,
  interval = NULL,
  simulation_days = NULL,
  day_type = NULL,
  environment_name = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{key_value}}{A character vector to identify key values of the
data. If \code{NULL}, all keys of that variable will be returned.
\code{key_value} can also be data.frame that contains \code{key_value}
and \code{name} columns. In this case, \code{name} argument in
\verb{$report_data()} is ignored. All available \code{key_value} for
current simulation output can be obtained using
\href{../../eplusr/html/EplusSql.html#method-report_data_dict}{\code{$report_data_dict()}}.
Default: \code{NULL}.}

\item{\code{name}}{A character vector to identify names of the data. If
\code{NULL}, all names of that variable will be returned. If
\code{key_value} is a data.frame, \code{name} is ignored. All available
\code{name} for current simulation output can be obtained using
\href{../../eplusr/html/EplusSql.html#method-report_data_dict}{\code{$report_data_dict()}}.
Default: \code{NULL}.}

\item{\code{year}}{Year of the date time in column \code{datetime}. If \code{NULL}, it
will calculate a year value that meets the start day of week
restriction for each environment. Default: \code{NULL}.}

\item{\code{tz}}{Time zone of date time in column \code{datetime}. Default:
\code{"UTC"}.}

\item{\code{case}}{A single string used to add a character column \code{case} in
the returned results to indicate the case of this simulation.
If \code{NULL}, no column is added. If \code{"auto"}, the name of the
IDF file without extension is used. Default: \code{"auto"}.}

\item{\code{all}}{If \code{TRUE}, extra columns are also included in the returned
\code{\link[data.table:data.table]{data.table::data.table()}}.}

\item{\code{wide}}{If \code{TRUE}, the output is formatted in the same way as
standard EnergyPlus csv output file.}

\item{\code{period}}{A Date or POSIXt vector used to specify which time
period to return. The year value does not matter and only
month, day, hour and minute value will be used when
subsetting. If \code{NULL}, all time period of data is returned.
Default: \code{NULL}.}

\item{\code{month, day, hour, minute}}{Each is an integer vector for month,
day, hour, minute subsetting of \code{datetime} column when
querying on the SQL database. If \code{NULL}, no subsetting is
performed on those components. All possible \code{month}, \code{day},
\code{hour} and \code{minute} can be obtained using
\href{../../eplusr/html/EplusSql.html#method-read_table}{\code{$read_table("Time")}}.
Default: \code{NULL}.}

\item{\code{interval}}{An integer vector used to specify which interval
length of report to extract. If \code{NULL}, all interval will be
used. Default: \code{NULL}.}

\item{\code{simulation_days}}{An integer vector to specify which simulation
day data to extract. Note that this number resets after warmup
and at the beginning of an environment period. All possible
\code{simulation_days} can be obtained using
\href{../../eplusr/html/EplusSql.html#method-read_table}{\code{$read_table("Time")}}.
If \code{NULL}, all simulation days will be used. Default: \code{NULL}.}

\item{\code{day_type}}{A character vector to specify which day type of data
to extract. All possible day types are: \code{Sunday}, \code{Monday},
\code{Tuesday}, \code{Wednesday}, \code{Thursday}, \code{Friday}, \code{Saturday},
\code{Holiday}, \code{SummerDesignDay}, \code{WinterDesignDay}, \code{CustomDay1},
and \code{CustomDay2}. All possible values for current simulation
output can be obtained using
\href{../../eplusr/html/EplusSql.html#method-read_table}{\code{$read_table("Time")}}.
A few grouped options are also provided:
\itemize{
\item \code{"Weekday"}: All working days, i.e. from Monday to Friday
\item \code{"Weekend"}: Saturday and Sunday
\item \code{"DesignDay"}: Equivalent to \code{"SummerDesignDay"} plus \code{"WinterDesignDay"}
\item \code{"CustomDay"}: CustomDay1 and CustomDay2
\item \code{"SpecialDay"}: Equivalent to \code{"DesignDay"} plus \code{"CustomDay"}
\item \code{"NormalDay"}: Equivalent to \code{"Weekday"} and \code{"Weekend"} plus \code{"Holiday"}
}}

\item{\code{environment_name}}{A character vector to specify which
environment data to extract. If \code{NULL}, all environment data
are returned. Default: \code{NULL}. All possible
\code{environment_name} for current simulation output can be
obtained using:\preformatted{$read_table("EnvironmentPeriods")
}}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$report_data()} extracts the report data in a
\code{\link[data.table:data.table]{data.table::data.table()}} using key values, variable names and other
specifications.

\verb{$report_data()} can also directly take all or subset output from
\verb{$report_data_dict()} as input, and extract all data specified.

The returned column numbers varies depending on \code{all} argument.
\itemize{
\item \code{all} is \code{FALSE}, the returned \code{\link[data.table:data.table]{data.table::data.table()}} has 6 columns:
\itemize{
\item \code{case}: Simulation case specified using \code{case} argument
\item \code{datetime}: The date time of simulation result
\item \code{key_value}: Key name of the data
\item \code{name}: Actual report data name
\item \code{units}: The data units
\item \code{value}: The data value
}
\item \code{all} is \code{TRUE}, besides columns described above, extra columns are also
included:
\itemize{
\item \code{month}: The month of reported date time
\item \code{day}: The day of month of reported date time
\item \code{hour}: The hour of reported date time
\item \code{minute}: The minute of reported date time
\item \code{dst}: Daylight saving time indicator. Possible values: \code{0} and \code{1}
\item \code{interval}: Length of reporting interval
\item \code{simulation_days}: Day of simulation
\item \code{day_type}: The type of day, e.g. \code{Monday}, \code{Tuesday} and etc.
\item \code{environment_period_index}: The indices of environment.
\item \code{environment_name}: A text string identifying the environment.
\item \code{is_meter}: Whether report data is a meter data. Possible values: \code{0} and
\code{1}
\item \code{type}: Nature of data type with respect to state. Possible values: \code{Sum}
and \code{Avg}
\item \code{index_group}: The report group, e.g. \code{Zone}, \code{System}
\item \code{timestep_type}: Type of data timestep. Possible values: \code{Zone} and \verb{HVAC System}
\item \code{reporting_frequency}: The reporting frequency of the variable, e.g.
\verb{HVAC System Timestep}, \verb{Zone Timestep}.
\item \code{schedule_name}: Name of the the schedule that controls reporting
frequency.
}
}

With the \code{datetime} column, it is quite straightforward to apply time-series
analysis on the simulation output. However, another painful thing is that
every simulation run period has its own \verb{Day of Week for Start Day}. Randomly
setting the \code{year} may result in a date time series that does not have
the same start day of week as specified in the RunPeriod objects.

eplusr provides a simple solution for this. By setting \code{year} to \code{NULL},
which is the default behavior, eplusr will calculate a year value (from
current year backwards) for each run period that compliances with the start
day of week restriction.

It is worth noting that EnergyPlus uses 24-hour clock system where 24 is only
used to denote midnight at the end of a calendar day. In EnergyPlus output,
"00:24:00" with a time interval being 15 mins represents a time period from
"00:23:45" to "00:24:00", and similarly "00:15:00" represents a time period
from "00:24:00" to "00:15:00" of the next day. This means that if current day
is Friday, day of week rule applied in schedule time period "00:23:45" to
"00:24:00" (presented as "00:24:00" in the output) is also Friday, but not
Saturday. However, if you try to get the day of week of time "00:24:00" in R,
you will get Saturday, but not Friday. This introduces inconsistency and may
cause problems when doing data analysis considering day of week value.

With \code{wide} equals \code{TRUE}, \verb{$report_data()} will format the simulation output
in the same way as standard EnergyPlus csv output file. Sometimes this can be
useful as there may be existing tools/workflows that depend on this format.
When both \code{wide} and \code{all} are \code{TRUE}, columns of runperiod environment names
and date time components are also returned, including:
\verb{environment_period_index", "environment_name}, \code{simulation_days},
\code{datetime}, \code{month}, \code{day}, \code{hour}, \code{minute}, \code{day_type}.

For convenience, input character arguments matching in
\verb{$report_data()} are \strong{case-insensitive}.
}

\subsection{Returns}{
A \code{\link[data.table:data.table]{data.table::data.table()}}.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# read all report data
sql$report_data()

# specify output variables using report data dictionary
dict <- sql$report_data_dict()
sql$report_data(dict[units == "C"])

# specify output variables using 'key_value' and 'name'
sql$report_data("environment", "site outdoor air drybulb temperature")

# explicitly specify year value and time zone
sql$report_data(dict[1], year = 2020, tz = "Etc/GMT+8")

# explicitly specify case name
sql$report_data(dict[1], case = "example")

# get all possible columns
sql$report_data(dict[1], all = TRUE)

# return in a format that is similar as EnergyPlus CSV output
sql$report_data(dict[1], wide = TRUE)

# return in a format that is similar as EnergyPlus CSV output with
# extra columns
sql$report_data(dict[1], wide = TRUE, all = TRUE)

# only get data at the working hour on the first Monday
sql$report_data(dict[1], hour = 8:18, day_type = "monday", simulation_days = 1:7)

# only get specified run period data
sql$read_table("EnvironmentPeriods") # possible environment name
sql$report_data(dict[1], environment_name = "San Francisco Intl Ap CA USA TMY3 WMO#=724940")
# can also be done using 'environment_period_index' column
sql$report_data(dict[1], all = TRUE)[environment_period_index == 3L]
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-tabular_data"></a>}}
\if{latex}{\out{\hypertarget{method-tabular_data}{}}}
\subsection{Method \code{tabular_data()}}{
Read tabular data
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusSql$tabular_data(
  report_name = NULL,
  report_for = NULL,
  table_name = NULL,
  column_name = NULL,
  row_name = NULL,
  case = "auto",
  wide = FALSE,
  string_value = !wide
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{report_name, report_for, table_name, column_name, row_name}}{Each is
a character vector for subsetting when querying the SQL
database.  For the meaning of each argument, please see the
description above.}

\item{\code{case}}{A single string used to add a character column \code{case} in
the returned results to indicate the case of this simulation.
If \code{NULL}, no column is added. If \code{"auto"}, the name of the
IDF file without extension is used. Default: \code{"auto"}.}

\item{\code{wide}}{If \code{TRUE}, each table will be converted into the similar
format as it is shown in EnergyPlus HTML output file. Default:
\code{FALSE}.}

\item{\code{string_value}}{Only applicable when \code{wide} is \code{TRUE}. If
\code{string_value} is \code{FALSE}, instead of keeping all values as
characters, values in possible numeric columns are converted
into numbers. Default: the opposite of \code{wide}. Possible
numeric columns indicate column that:
\itemize{
\item columns that have associated units
\item columns that contents numbers
}}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$tabular_data()} extracts the tabular data in a
\code{\link[data.table:data.table]{data.table::data.table()}} using report, table, column and row name
specifications. The returned \code{\link[data.table:data.table]{data.table::data.table()}} has
9 columns:
\itemize{
\item \code{case}: Simulation case specified using \code{case} argument
\item \code{index}: Tabular data index
\item \code{report_name}: The name of the report that the record belongs to
\item \code{report_for}: The \code{For} text that is associated with the record
\item \code{table_name}: The name of the table that the record belongs to
\item \code{column_name}: The name of the column that the record belongs to
\item \code{row_name}: The name of the row that the record belongs to
\item \code{units}: The units of the record
\item \code{value}: The value of the record \strong{in string format} by default.
}

For convenience, input character arguments matching in
\verb{$tabular_data()} are \strong{case-insensitive}.
}

\subsection{Returns}{
A \code{\link[data.table:data.table]{data.table::data.table()}} with 9 columns (when \code{wide} is
\code{FALSE}) or a named list of \code{\link[data.table:data.table]{data.table::data.table()}}s where the
names are the combination of \code{report_name}, \code{report_for} and
\code{table_name}.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# read all tabular data
sql$tabular_data()

# explicitly specify data you want
str(sql$tabular_data(
    report_name = "AnnualBuildingUtilityPerformanceSummary",
    table_name = "Site and Source Energy",
    column_name = "Total Energy",
    row_name = "Total Site Energy"
))

# get tabular data in wide format and coerce numeric values
str(sql$tabular_data(
    report_name = "AnnualBuildingUtilityPerformanceSummary",
    table_name = "Site and Source Energy",
    column_name = "Total Energy",
    row_name = "Total Site Energy",
    wide = TRUE, string_value = FALSE
))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-print"></a>}}
\if{latex}{\out{\hypertarget{method-print}{}}}
\subsection{Method \code{print()}}{
Print \code{EplusSql} object
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusSql$print()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$print()} shows the core information of this \code{EplusSql} object,
including the path of the EnergyPlus SQLite file, last modified
time of the SQLite file and the path of the IDF file with the
same name in the same folder.
}

\subsection{Returns}{
The \code{EplusSql} object itself, invisibly.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
sql$print()
}

}
\if{html}{\out{</div>}}

}

}
}
