% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aim.R
\name{imd}
\alias{imd}
\title{Integral Matching Design}
\usage{
imd(o, op, x = NULL, params = NULL)
}
\arguments{
\item{o}{An object of class \code{\link{ode}} (created via \code{\link{mak}}, \code{\link{plk}}, etc.).}

\item{op}{An object of class \code{\link{opt}} (created via \code{\link{opt}}). Compatibility check with \code{o} is conducted.}

\item{x}{Matrix of dimension m-x-(d+1) containing custom time and smoothed values of the processes used for the integral matching, see details. If \code{NULL} (default) a linear-interpolation smoother is employed.}

\item{params}{List of vectors of initial parameter values. Default (\code{NULL}) translates to the zero-vector. If the ODE system is linear in the parameter vector and the boundary constraints on are not unusual, then \code{params} is ignored.}
}
\value{
A list with:
\item{X}{List of design matrices, one for each parameter argument.}
\item{Y}{A vector of observations.}
\item{W}{A vector of weights.}
\item{X0}{A matrix of initial states, one for each context. Interpolated from \code{x}.}
\item{xout}{A matrix containing the values of the process at the time points in \code{y}, interpolated from \code{x}. Use it to check that \code{x} works as intended.}
}
\description{
Get the design matrix, formated data and weights used in integral matching.
}
\details{
The design matrix is as follows:
\deqn{X = (\int_{t_{i-1}}^{t_{i}}{\frac{df}{dparam}(x(s), context_l * param) * context_l ds})_{i=1}^n}
Here f is the vector field of the ODE-system, x is the smoothed process and params is (internally) scaled with \code{scales} in \code{\link{reg}} objects in \code{o}.

Similiarly the observations and weights are concatinateed as follows:
\deqn{Y = (x_{t_i} - x_{t_{i-1}})_{i=1}^n}
\deqn{W = ((w_{t_i} + w_{t_{i-1}}) / 2)_{i=1}^n}

The number of decreases in time in \code{x} must match the number of decreases in time in \code{y} in \code{op}. The process x is simply a linear interpolation of \code{x}, hence finer discretisations give more refined integral matching. Each context is handled seperately. Moreover, the compatibility checks in \code{rodeo} are also conducted here.
}
\examples{
set.seed(123)
# Michaelis-Menten system with two 0-rate reactions
A <- matrix(c(1, 1, 0, 0,
              0, 0, 1, 0,
              0, 0, 1, 0,
              0, 1, 0, 0,
              0, 0, 0, 1), ncol = 4, byrow = TRUE)
B <- matrix(c(0, 0, 1, 0,
              1, 1, 0, 0,
              1, 0, 0, 1,
              0, 0, 0, 1,
              0, 0, 1, 0), ncol = 4, byrow = TRUE)
k <- c(2.1, 2.25, 1.5, 0, 0); x0 <- c(E = 8, S = 10, ES = 1.5, P = 1.5)
Time <- seq(0, 10, by = 1)

# Simulate data, in second context the catalytic rate has been doubled
contexts <- cbind(1, c(1, 1, 2, 1, 1))
m <- mak(A, B, r = reg(contexts = contexts))
y <- numsolve(m, c(Time, Time), cbind(x0, x0 + c(2, -2, 0, 0)), contexts * k)
y[, -1] <- y[, -1] + matrix(rnorm(prod(dim(y[, -1])), sd = .5), nrow = nrow(y))

# Get the design matrix used in integral matching
d <- imd(m, opt(y))
head(d$X[[1]])

# Compare with glmnet
lasso <- glmnet::glmnet(x = d$X[[1]], y = d$Y, intercept = FALSE, lower.limits = 0)
a <- aim(m, opt(y, nlambda = 100), adapts = "scales")
all.equal(lasso$beta, a$params$rate)

}
\seealso{
aim, rodeo.aim, numint
}
