% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aim.R
\name{aim}
\alias{aim}
\title{Adaptive Integral Matching (AIM)}
\usage{
aim(o, op, x = NULL, adapts = c("scales", "penalty_factor"), xout = FALSE,
  params = NULL, ...)
}
\arguments{
\item{o}{An object of class \code{\link{ode}} (created via \code{\link{mak}}, \code{\link{plk}}, etc.).}

\item{op}{An object of class \code{\link{opt}} (created via \code{\link{opt}}). Compatibility check with \code{o} is conducted.}

\item{x}{Matrix of dimension mx(d+1) containing custom time and smoothed values of the processes used for the integral matching, see details. If \code{NULL} (default) a linear-interpolation smoother is employed.}

\item{adapts}{Character vector holding names of what quantities to adapt during algorithm. Possible quantities are: \code{"scales"}, \code{"weights"} and \code{"penalty_factors"}, see details. Default is \code{"scales"} and \code{"penalty_factor"}.}

\item{xout}{Logical indicating if a matrix containing the process values at the time points in \code{y} in \code{op}, linearly interpolated from \code{x} should be returned. Default is \code{FALSE}.}

\item{params}{List of vectors of initial parameter values. Default (\code{NULL}) translates to the zero-vector. If the ODE system is linear in the parameter vector and the boundary constraints on are not unusual, then \code{params} is ignored.}

\item{...}{Additional arguments passed to \code{aim}.}
}
\value{
An object with S3 class "aim":
\item{o}{Original \code{\link{ode}}-object with adapted quantities.}
\item{op}{Original \code{\link{opt}}-object with adapted quantities and default values for \code{lambda_min_ratio}, \code{lambda} and (if needed) \code{a} inserted, if these were originally \code{NULL}.}
\item{params}{A list of matrices (of type dgCMatrix) with the parameter estimates (scaled by \code{scales} in \code{reg}), one column for each lambda value. One element in the list for each parameter (other than initial state parameter).}
\item{x0s}{A matrix with the initial state estimates.}
\item{rss}{A vector of residual sum of squares.}
\item{x}{Original \code{x}, or \code{y} in \code{op} if \code{x} was \code{NULL}.}
\item{xout}{(If \code{xout} = \code{TRUE}) A matrix containing the values of the process at the time points in \code{y} in \code{op}, interpolated from \code{x}. Use it to check that \code{x} works as intended.}
}
\description{
Gives approximate parameter estimates using integral matching and optionally adapts weights and scales to these. Feed this to \code{\link{rodeo}} for initialising exact estimation.
}
\details{
\subsection{Loss function}{
Integral matching requires a smoothed process in order to approximate the parameter estimates. More precisely, the loss function
\deqn{RSS / (2 * (n - s)) + lambda*penalty}
is minimised, where RSS is the sum of the squared 2-norms of
\deqn{x_i - x_{i-1} - \int_{t_{i-1}}^{t_{i}}{f(x(s), context_l * param) ds}}
Here f is the vector field of the ODE-system, x is the smoothed process and param is (internally) scaled with \code{scales} in \code{reg}.
}
\subsection{Custom smoother}{
The supplied \code{x} is a way of customising how x in the loss function is made. Firstly, \code{x} must have similiar layout as \code{y} in \code{op}, i.e., first column is time and the remaining columns contain smoothed values of the process at those time points, see \code{\link{opt}}-documentation for details.

The number of decreases in time in \code{x} must match the number of decreases in time in \code{y} in \code{op}. The process x in the loss function is simply a linear interpolation of \code{x}, hence finer discretisations give more refined integral matching. Each context is handled seperately. Moreover, the compatibility checks in \code{rodeo} are also conducted here.
}
\subsection{Adaptations}{
The \code{adapts} are ways to adapt quantities in the \code{\link{opt}}-object and \code{\link{reg}}-objects in \code{o} to the data:
\describe{
\item{\code{"scales"}}{A standardisation of the columns in the linearisation takes place and carry over to \code{scales} in \code{\link{reg}}-objects in \code{o} (generally recommended and default).}
\item{\code{"weights"}}{The observational weights (\code{weights} in \code{op}) are adjusted coordinate-for-coordinate (column-wise) by reciprocal average residual sum of squares across penalty parameters.}
\item{\code{"penalty_factor"}}{The penalty factors in \code{\link{reg}} are adjusted by the reciprocal average magnitude of the estimates (parameters whose average magnitude is 0 join \code{exclude}). For extremely large systems, this option can heavily reduce further computations if the returned \code{\link{aim}}-object is passed to \code{\link{rodeo}}.}
}
If either \code{"penalty_factor"} or \code{"weights"} are in \code{adapts} a refitting takes place.
}
Finally, note that \code{lambda} and \code{nlambda} in returned \code{\link{opt}}-object may have changed.
}
\examples{
set.seed(123)
# Michaelis-Menten system with two 0-rate reactions
A <- matrix(c(1, 1, 0, 0,
  0, 0, 1, 0,
  0, 0, 1, 0,
  0, 0, 1, 1,
  0, 0, 1, 1), ncol = 4, byrow = TRUE)
B <- matrix(c(0, 0, 1, 0,
  1, 1, 0, 0,
  1, 0, 0, 1,
  0, 1, 0, 0,
  1, 0, 0, 1), ncol = 4, byrow = TRUE)
k <- c(0.1, 1.25, 0.5, 0, 0); x0 <- c(E = 5, S = 5, ES = 1.5, P = 1.5)
Time <- seq(0, 10, by = 1)

# Simulate data, in second context the catalytic rate has been inhibited
contexts <- cbind(1, c(1, 1, 0, 1, 1))
m <- mak(A, B, r = reg(contexts = contexts))
y <- numsolve(m, c(Time, Time), cbind(x0, x0 + c(2, -2, 0, 0)), contexts * k)
y[, -1] <- y[, -1] + matrix(rnorm(prod(dim(y[, -1])), sd = .25), nrow = nrow(y))

# Create optimisation object via data
o <- opt(y, nlambda = 10)

# Fit data using Adaptive Integral Matching on mak-object
a <- aim(m, o)
a$params$rate

# Compare with true parameter on column vector form
matrix(k, ncol = 1)


# Example: Power Law Kinetics
A <- matrix(c(1, 0, 1,
  1, 1, 0), byrow = TRUE, nrow = 2)
p <- plk(A)
x0 <- c(10, 4, 1)
theta <- matrix(c(0, -0.25,
  0.75, 0,
  0, -0.1), byrow = TRUE, nrow = 3)
Time <- seq(0, 1, by = .025)

# Simulate data
y <- numsolve(p, Time, x0, theta)
y[, -1] <- y[, -1] + matrix(rnorm(prod(dim(y[, -1])), sd = .25), nrow = nrow(y))

# Estimation
a <- aim(p, opt(y, nlambda = 10))
a$params$theta

# Compare with true parameter on column vector form
matrix(theta, ncol = 1)


# Example: use custom lowess smoother on data
# Smooth each coordinate of data to get curve
# on extended time grid
ext_time <- seq(0, 1, by = 0.001)
x_smooth <- apply(y[, -1], 2, function(z) {
  # Create loess object
  data <- data.frame(Time = y[, -1], obs = z)
  lo <- loess(obs ~ Time, data)

  # Get smoothed curve on extended time vector
  predict(lo, newdata = data.frame(Time = ext_time))
})

# Bind the extended time
x_smooth <- cbind(ext_time, x_smooth)

# Run aim on the custom smoothed curve
a_custom <- aim(p, opt(y), x = x_smooth)


}
\seealso{
rodeo, rodeo.ode, rodeo.aim, imd
}
