#' Joint DVI search
#'
#' Victims are given as a list of singletons, and references as a list of
#' pedigrees. All possible assignments are evaluated and solutions ranked
#' according to the likelihood.
#'
#' @param pm A list of singletons.
#' @param am A list of pedigrees.
#' @param missing Character vector with names of missing persons.
#' @param pairings A list of possible pairings for each victim. If NULL, all
#'   sex-consistent pairings are used.
#' @param ignoreSex A logical.
#' @param assignments A data frame containing the assignments to be considered
#'   in the joint analysis. By default, this is automatically generated by
#'   taking all combinations from `pairings`.
#' @param limit A positive number, by default 0. Only pairwise LR values above
#'   this are considered.
#' @param markers A vector indicating which markers should be included in the
#'   analysis. By default all markers are included.
#' @param disableMutations A logical, or NA (default). The default action is to
#'   disable mutations in all reference families without Mendelian errors.
#' @param undisputed A logical, by default TRUE.
#' @param threshold A positive number, passed onto [findUndisputed()]. Default:
#'   1e4.
#' @param relax A logical, passed onto [findUndisputed()]. Default: FALSE.
#' @param numCores Integer. The number of cores used in parallelisation.
#'   Default: 1.
#' @param check A logical, indicating if the input data should be checked for
#'   consistency.
#' @param verbose A logical.
#'
#' @return A data frame. Each row describes an assignment of victims to missing
#'   persons, accompanied with its log likelihood, the LR compared to the null
#'   (i.e., no identifications), and the posterior corresponding to a flat
#'   prior.
#'
#' @seealso [pairwiseLR()], [findUndisputed()]
#'
#' @examples
#' pm = example2$pm
#' am = example2$am
#' missing = example2$missing
#'
#' jointDVI(pm, am, missing)
#'
#' @importFrom parallel makeCluster stopCluster detectCores parLapply
#'   clusterEvalQ clusterExport
#'
#' @export
jointDVI = function(pm, am, missing, pairings = NULL, ignoreSex = FALSE, assignments = NULL, limit = 0, undisputed = TRUE, markers = NULL,
                    threshold = 1e4, relax = FALSE, disableMutations = NA, numCores = 1, check = TRUE, verbose = TRUE) {
  
  st = Sys.time()
  
  if(length(pm) == 0)
    undisputed = FALSE
  
  if(is.singleton(pm)) 
    pm = list(pm)
  if(is.ped(am)) 
    am = list(am)
  
  names(pm) = origVics = vics = unlist(labels(pm)) 
  
  if(!is.null(markers)) {
    pm = selectMarkers(pm, markers)
    am = selectMarkers(am, markers)
  }
  
  if(verbose)
    summariseDVI(pm, am, missing, method = "Joint identification", printMax = 10)
  
  if(check)
    checkDVI(pm, am, missing, pairings = pairings, ignoreSex = ignoreSex)
  
  ### Mutation disabling
  if(any(allowsMutations(am))) {
    
    if(verbose) 
      message("\nMutation modelling:")
    
    if(isTRUE(disableMutations)) {
      if(verbose) message(" Disabling mutations in all families")
      disableFams = seq_along(am)
    }
    else if(identical(disableMutations, NA)) {
      am.nomut = pedprobr::setMutationModel(am, model = NULL)
      badFams = vapply(am.nomut, loglikTotal, FUN.VALUE = 1) == -Inf
      if(verbose) {
        if(any(badFams)) 
          message(" ", sum(badFams), " inconsistent families: ", trunc(which(badFams)))
        message(" ", sum(!badFams), " consistent families. Disabling mutations in these")
      }
      disableFams = which(!badFams)
    }
    else disableFams = NULL
  
    if(length(disableFams)) {
      am[disableFams] = setMutationModel(am[disableFams], model = NULL)
    }
  }
  
  ### Identify and fixate "undisputed" matches
  undisp = list()
  
  if(undisputed && is.null(assignments)) {
    
    if(verbose) {
      message("\nUndisputed matches:")
      message(" Pairwise LR threshold = ", threshold)
    }
    
    r = findUndisputed(pm, am, missing, pairings = pairings, ignoreSex = ignoreSex, threshold = threshold, 
                       relax = relax, limit = limit, check = FALSE, verbose = verbose)
    
    # List of undisputed, and their LR's
    undisp = r$undisp 
    
    # If all are undisputed, return early
    if(length(undisp) == length(pm)) {
      solution = as.data.frame(lapply(undisp, function(v) v$match))
      
      # Run through jointDVI() with the solution as the only assignment 
      res = jointDVI(pm, am, missing, ignoreSex = ignoreSex, assignments = solution, undisputed = FALSE,
                     markers = markers, threshold = NULL, check = FALSE, verbose = FALSE)
      return(res)
    }
    
    # Reduced DVI problem to be used in the joint analysis
    pm = r$pmReduced
    am = r$amReduced
    missing = r$missingReduced
    vics = names(pm)
    
    # pairings: These exclude those with LR = 0!
    pairings = r$pairings
  }
    
  if(is.null(pairings) && is.null(assignments)) {
    pairings = pairwiseLR(pm, am, missing = missing, pairings = pairings, ignoreSex = ignoreSex, limit = limit)$pairings
  }
 
  if(is.null(assignments)) {
    # Expand pairings to assignment data frame
    assignments = expand.grid.nodup(pairings)
  }
  
  nAss = nrow(assignments)
  if(nAss == 0)
    stop("No possible solutions!")
  if(verbose)
    message("\nAssignments to consider in the joint analysis: ", nAss, "\n")
  
  # Convert to list; more handy below
  assignmentList = lapply(1:nAss, function(i) as.character(assignments[i, ]))
  
  # Initial loglikelihoods
  logliks.PM = vapply(pm, loglikTotal, FUN.VALUE = 1)
  logliks.AM = vapply(am, loglikTotal, FUN.VALUE = 1)
  
  loglik0 = sum(logliks.PM) + sum(logliks.AM)
  if(loglik0 == -Inf)
    stop("Impossible initial data: AM component ", toString(which(logliks.AM == -Inf)))
  
  
  # Parallelise
  if(numCores > 1) {
    cl = makeCluster(numCores)
    on.exit(stopCluster(cl))
    clusterEvalQ(cl, library(dvir))
    clusterExport(cl, "loglikAssign", envir = environment())
    
    if(verbose) message("Using ", length(cl), " cores")
    
    # Loop through assignments
    loglik = parLapply(cl, assignmentList, function(a) 
      loglikAssign(pm, am, vics, a, loglik0, logliks.PM, logliks.AM))
  }
  else {
    loglik = lapply(assignmentList, function(a) 
      loglikAssign(pm, am, vics, a, loglik0, logliks.PM, logliks.AM))
  }
  
  loglik = unlist(loglik)
  
  LR = exp(loglik - loglik0)
  posterior = LR/sum(LR) # assumes a flat prior
  
  # Add undisputed matches
  if(length(undisp)) {
    # Add ID columns
    for(v in names(undisp)) assignments[[v]] = undisp[[v]]$match
    
    # Fix ordering
    assignments = assignments[origVics]
    
    # Fix LR: Multiply with that of the undisputed
    LR = LR * prod(sapply(undisp, `[[`, "LR"))
  }
    
  # Collect results
  tab = cbind(assignments, loglik = loglik, LR = LR, posterior = posterior)
  
  # Sort in decreasing likelihood, break ties with grid
  g = assignments
  g[g == "*"] = NA
  tab = tab[do.call(order, g), , drop = FALSE] # first sort assignments alphabetically
  tab = tab[order(round(tab$loglik, 10), decreasing = TRUE), , drop = FALSE]
  
  rownames(tab) = NULL
  
  if(verbose)
    message("Time used: ", format(Sys.time() - st, digits = 3))
  
  tab
}



# Function for computing the total log-likelihood of a single assignment
loglikAssign = function(pm, am, vics, assignment, loglik0, logliks.PM, logliks.AM) {
  
  # Victims which actually move
  vicMove = vics[assignment != "*"]
  mpsMove = assignment[assignment != "*"]
  
  if(length(vicMove) == 0)
    return(loglik0)
  
  # The relevant AM components
  compNo = unique.default(getComponent(am, mpsMove, checkUnique = TRUE))
  
  # Move victim data
  changedComps = transferMarkers(pm[vicMove], am[compNo], idsFrom = vicMove, 
                                 idsTo = mpsMove, erase = FALSE)

  # Update likelihood of modified AM comps
  logliks.AM.new = logliks.AM
  logliks.AM.new[compNo] = vapply(changedComps, function(a) loglikTotal(a), FUN.VALUE = 1)
  
  # Likelihood of remaining PMs
  logliks.PM.new = logliks.PM[setdiff(vics, vicMove)]
  
  # Return total loglik of assignments
  sum(logliks.PM.new) + sum(logliks.AM.new)
}



# @rdname jointDVI
# @export
checkDVI = function(pm, am, missing, pairings, errorIfEmpty = FALSE, ignoreSex = FALSE){
  
  # MDV: added to avoid crash in certain cases.
  if(length(pm) == 0 || length(missing) == 0) {
    if(errorIfEmpty) stop("Empty DVI problem") 
    else return()
  }
  
  if(!all(missing %in% unlist(labels(am))))
    stop("Missing person not part of the AM pedigree(s): ", toString(setdiff(missing, unlist(labels(am)))))
  
  if(is.null(pairings))
    return()
  
  vics = unlist(labels(pm))
  vicSex = getSex(pm, vics, named = TRUE)
  
  candidMP = setdiff(unlist(pairings), "*")
  candidSex = getSex(am, candidMP, named = TRUE)
              
  if(!all(candidMP %in% missing))
    stop("Indicated pairing candidate is not a missing person: ", toString(setdiff(candidMP, missing)))
  
  for(v in vics) {
    candid = pairings[[v]]
    if(length(candid) == 0)
      stop("No available candidate for victim ", v)
    
    if(any(duplicated(candid)))
      stop("Duplicated candidate for victim ", v)
    
    cand = setdiff(candid, "*")
    if(length(cand) == 0)
      next
    
    if(!ignoreSex) {
      correctSex = candidSex[cand] == vicSex[v]
      if(!all(correctSex)) 
        stop("Candidate for victim ", v, " has wrong sex: ", toString(cand[correctSex]))
    }
  }
}


#' Summarise a DVI problem
#'
#' Prints a summary of a given DVI problem, including the number of victims,
#' missing persons, reference families and typed reference individuals. This
#' function primarily exists for being called from `jointDVI()` and other
#' high-level methods, but can also be used on its own.
#'
#' @param pm A list of singletons.
#' @param am A list of pedigrees.
#' @param missing Character vector with names of missing persons.
#' @param method A character, used by other methods.
#' @param printMax A positive integer. Vectors longer than this are truncated.
#'
#' @return No return value, called for side effects.
#'
#' @examples
#' pm = planecrash$pm
#' am = planecrash$am
#' missing = planecrash$missing
#'
#' summariseDVI(pm, am, missing)
#' summariseDVI(pm, am, missing, printMax = 5)
#'
#' @export
summariseDVI = function(pm, am, missing, method = NULL, printMax = 10) {
  vics = unlist(labels(pm))
  refs = typedMembers(am)
  nam = if(is.ped(am)) 1 else length(am)

  message("DVI problem:")
  message(sprintf(" %d victims: %s", length(pm), trunc(vics, printMax)))
  message(sprintf(" %d missing: %s", length(missing), trunc(missing, printMax)))
  message(sprintf(" %d typed refs: %s", length(refs), trunc(refs, printMax)))
  message(sprintf(" %d reference famil%s", nam, ifelse(nam == 1, "y", "ies")))
  if(!is.null(method))
    message("\n", method)
}


trunc = function(x, printMax = 10) {
  if(length(x) <= printMax)
    return(toString(x))
  y = c(x[1:5], "...", x[length(x)])
  toString(y)
}
