#' @title Function Makefile_recipe
#' @export
#' @seealso \code{\link{default_recipe_command}},
#' \code{\link{r_recipe_wildcard}}, \code{\link{make}}
#' @description See what your \code{Makefile} recipes
#' will look like in advance.
#'
#' @param recipe_command The Makefile recipe command.
#' See \code{\link{default_recipe_command}()}.
#'
#' @param target character scalar, name of your target
#'
#' @param cache_path path to the drake cache.
#' In practice, this defaults to the hidden \code{.drake/} folder,
#' but this can be customized. In the Makefile, the drake cache
#' is coded with the Unix variable `DRAKE_CACHE` and then dereferenced
#' with `$(DRAKE_CACHE)`. To simplify things for users who may
#' be unfamiliar with Unix variables, the \code{recipe()} function
#' just shows the literal path to the cache.
#'
#' @details Makefile recipes to build targets are customizable.
#' Use the \code{Makefile_recipe()} function to show and tweak
#' Makefile recipes in advance, and see
#' \code{\link{default_recipe_command}()} and
#' \code{\link{r_recipe_wildcard}()} for more clues.
#' The default recipe is \code{Rscript -e 'R_RECIPE'}, where
#' \code{R_RECIPE} is the wildcard for the recipe in R for making the target.
#' In writing the Makefile, \code{R_RECIPE} is replaced with something like
#' \code{drake::mk("name_of_target", "path_to_cache")}.
#' So when you call
#' \code{make(..., parallelism = "Makefile", recipe_command = "R -e 'R_RECIPE' -q")}, # nolint
#' from within R, the \code{Makefile} builds each target
#' with the \code{Makefile} recipe,
#' \code{R -e 'drake::mk("this_target", "path_to_cache")' -q}.
#' But since \code{R -q -e} fails on Windows,
#' so the default \code{recipe_command} argument is
#' \code{"Rscript -e 'R_RECIPE'"}
#' (equivalently just \code{"Rscript -e"}),
#' so the default \code{Makefile} recipe for each target is
#' \code{Rscript -e 'drake::mk("this_target", "path_to_cache")'}.
#'
#' @examples
#' Makefile_recipe()
#' Makefile_recipe(
#'   target = "this_target",
#'   recipe_command = "R -e 'R_RECIPE' -q",
#'   cache_path = "custom_cache"
#' )
#' default_recipe_command()
#' r_recipe_wildcard()
#' \dontrun{
#' load_basic_example()
#' # Look at the Makefile generated by the following.
#' make(my_plan, paralleliem = "Makefile")
#' # Generates a Makefile with "R -q -e" rather than
#' # "Rscript -e".
#' # Be aware the R -q -e fails on Windows.
#' make(my_plan, parallelism = "Makefile", jobs = 2
#'   recipe_command = "R -q -e")
#' # Same thing:
#' clean()
#' make(my_plan, parallelism = "Makefile", jobs = 2,
#'   recipe_command = "R -q -e 'R_RECIPE'")
#' clean()
#' make(my_plan, parallelism = "Makefile", jobs = 2,
#'   recipe_command = "R -e 'R_RECIPE' -q")
#' }
Makefile_recipe <- function( # nolint
  recipe_command = drake::default_recipe_command(),
  target = "your_target",
  cache_path = drake::default_cache_path()){
  build_recipe(
    target = target,
    recipe_command = recipe_command,
    cache_path = cache_path
  ) %>%
    cat("\n")
}

#' @title default_recipe_command
#' @export
#' @seealso \code{\link{Makefile_recipe}}
#' @description Function to give the default
#' recipe command for Makefile parallelism.
#' @details See the help file of \code{\link{Makefile_recipe}}
#' for details and examples.
default_recipe_command <- function(){
  paste0("Rscript -e '", r_recipe_wildcard(), "'")
}

#' @title r_recipe_wildcard
#' @export
#' @seealso \code{\link{default_recipe_command}}
#' @description Function to give the R recipe wildcard
#' for Makefiles.
#' @details See the help file of \code{\link{Makefile_recipe}}
#' for details and examples.
r_recipe_wildcard <- function(){
  "R_RECIPE"
}
