#' Compute hybrid index from genotypes, file, or numeric vector
#'
#' Provides a unified way to obtain the hybrid index
#' regardless of the data source. It accepts:
#' * a genotype matrix or data frame or the I4 matrix and computes the hybrid indices using
#'   `sStateCount()` and `pHetErrOnStateCount()`;
#' * a file with hybrid index values, typically generated by `diem()`;
#' * or a numeric vector of values.
#'
#' The function returns a numeric vector of hybrid indices and can
#' optionally subset individuals or rescale the values to the range 0-1.
#'
#' @param x Either a genotype matrix/data.frame, a path to a text file
#'   produced by `diem`, or a numeric vector of hybrid indices.
#' @inheritParams diem
#' @param rescale Logical, whether to linearly rescale the resulting hybrid
#'   indices to the interval 0–1. Defaults to `FALSE`.
#'
#' @details
#' The function automatically detects the input type:
#' * **File path** - the file is expected to have hybrid indices in the last
#'   column, that can optionally be named `"HybridIndex"`.
#' * **Numeric vector** - values are passed through unchanged.
#' * **Genotype matrix** - typically polarised genotypes resulting from
#'     `importPolarized`, with rows representing individuals. Each row is
#'   processed by `pHetErrOnStateCount(sStateCount(row))`.
#'
#' If `rescale = TRUE`, values are mapped to \deqn{[0, 1]} unless all are equal
#' or non-finite, in which case the original values are returned with a warning.
#'
#' @return
#' A numeric vector of hybrid index values. Names are not preserved.
#' @seealso \link{pHetErrOnStateCount}, \link{sStateCount}
#' @examples
#' hybridIndex(c(0.3, 0.5, 0.7))
#'
#' hybridIndex(1:10, ChosenInds = 1:5, rescale = TRUE)
#'
#' @export

hybridIndex <- function(x, ChosenInds = "all", rescale = FALSE) {
  # File path
  if (is.character(x) && length(x) == 1L) {
    if (!file.exists(x)) stop("File not found: ", x)
    hasHeader <- ifelse(grepl("HybridIndex", readLines(x, n = 1)[[1]]), TRUE, FALSE)
    HI <- read.table(x, header = hasHeader)
    HI <- unname(unlist(HI[, ncol(HI)]))
  } else {
    # Numeric vector
    if (is.numeric(x) && is.null(dim(x))) {
      HI <- unname(as.numeric(x))
    } else {
      # Imported genotypes
      if (is.matrix(x) || is.data.frame(x)) {
        gen <- if (is.data.frame(x)) as.matrix(x) else x
        # I4 matrix
        if (ncol(gen) == 4 && is.numeric(gen[1, 1])) {
          HI <- unname(unlist(apply(gen, 1L, function(row) pHetErrOnStateCount(row))[1, ]))
        } else {
          HI <- unname(unlist(apply(gen, 1L, function(row) pHetErrOnStateCount(sStateCount(row)))[1, ]))
        }
      } else {
        stop("Unsupported input. Provide a genotype table/matrix, a file path, or a numeric vector.")
      }
    }
  }

  # subset ChosenInds
  if (length(ChosenInds) > 0L && ChosenInds[1] != "all") {
    if (any(ChosenInds < 1L | ChosenInds > length(HI))) {
      stop("ChosenInds out of bounds for length ", length(HI), ".")
    }
    HI <- HI[ChosenInds]
  }

  # rescale
  if (isTRUE(rescale)) {
    r <- range(HI, na.rm = TRUE)
    if (is.finite(r[1]) && is.finite(r[2]) && r[1] != r[2]) {
      HI <- (HI - r[1]) / (r[2] - r[1])
    } else {
      warning("Rescale requested but all values equal or non-finite; returning unscaled hybrid indices.")
    }
  }
  return(HI)
}
