% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fitting_functions.R
\name{deseats}
\alias{deseats}
\title{Locally Weighted Regression for Trend and Seasonality in Equidistant Time 
Series under Short Memory}
\usage{
deseats(
  y,
  smoothing_options = set_options(),
  bwidth_start = NULL,
  inflation_rate = c("optimal", "naive"),
  correction_factor = TRUE,
  autocor = TRUE,
  drop = NULL,
  error_model = c("free", "ARMA")
)
}
\arguments{
\item{y}{a numerical vector or a time series object of class \code{ts} or 
that can be transformed with \code{\link[stats]{as.ts}} to an object of class 
\code{ts}; for these observations, trend and seasonality will be obtained.}

\item{smoothing_options}{an S4 object of class \code{smoothing_options}, which 
is returned by the function \code{\link{set_options}}; it 
includes details about the
options to consider in the locally weighted regression such as the order of
polynomial and the bandwidth for smoothing among others.}

\item{bwidth_start}{a single numeric value that is only relevant if the slot
\code{bwidth} in \code{smoothing_options} is set to \code{NA}; 
as the bandwidth will then
be selected automatically, \code{bwidth_start} sets the initial bandwidth for 
the algorithm; the default, \code{bwidth_start = NULL}, corresponds to 
\code{bwidth_start = 0.1} for a local linear trend and to 
\code{bwidth_start = 0.2} for a local cubic trend.}

\item{inflation_rate}{a character vector of length one that indicates, which inflation rate 
to use in the bandwidth selection; for a local linear trend, we have 
\code{inflation_rate = "optimal"} as the default, for a local cubic trend
it is \code{inflation_rate = "naive"}, which correspond to inflation rates
of 5/7 and 9/13, respectively.}

\item{correction_factor}{A logical vector of length one; theoretically, a 
larger bandwidth to estimate the sum of autocovariances from residuals of 
pilot trend and seasonality estimates is advisable than for estimating trend
and seasonality; for \code{correction_factor = TRUE}, this is implemented;
for \code{error_model = "ARMA"}, \code{correction_factor = FALSE} is 
enforced.}

\item{autocor}{a logical vector of length one; indicates whether to consider
autocorrelated errors (\code{TRUE}) or independent but identically 
distributed errors (\code{FALSE}); the default is \code{autocor = TRUE}.}

\item{drop}{a numeric vector of length one that indicates the proportion of 
the observations to not include at each boundary in the bandwidth estimation 
process, if a bandwidth is selected automatically; the default is 
\code{drop = NULL}, which corresponds to \code{drop = 0.05} for a 
local linear trend and to \code{drop = 0.1} for a local cubic trend.}

\item{error_model}{a character vector of length one that indicates whether
for \code{autocor = TRUE} the sum of autocovariances of the errors is 
obtained purely nonparametrically (\code{"free"}) or whether an
autoregressive moving-average (ARMA) model is assumed \code{"ARMA"}; the
default is \code{error_model = "free"}.}
}
\value{
The function returns and S4 object with the following elements (access them 
via \code{@}):

\describe{
\item{\code{boundary_method}}{the applied boundary method.}
\item{\code{bwidth}}{the globally applied bandwidth in the smoothing process; if not 
if no input is given in the function call, this is the automatically selected 
optimal bandwidth.}
\item{\code{decomp}}{An object of class \code{"mts"} that consists of the
decomposed time series data.}
\item{\code{frequency}}{the frequency of the time series.}
\item{\code{kernel_fun}}{the second-order kernel function considered for weighting.}
\item{\code{order_poly}}{the order of polynomial considered locally for the trend.}
\item{\code{order_poly}}{the order of polynomial considered locally for the trend.}
\item{\code{sum_autocov}}{the estimated sum of autocovariances.}
\item{\code{ts_name}}{the object name of the initially provided time series object.}
\item{\code{weights_wfun}}{a matrix that gives the weights of the weighting 
function \eqn{K} at each estimation time point; ; if 
\eqn{n} is the length of the given time series and \eqn{b} is the applied 
(relative) bandwidth, then the first row of the weighting system gives the 
weighting function weights when estimating at \eqn{t=1}, the second row gives 
the weights when estimating at \eqn{t=2} and so on for all left-hand side 
boundary points 
until the middle row, which contains the 
weights used at all interior points; the rows following the middle row contain
the weights for right-hand side boundary points (the rows are ordered
chronologically)}
\item{\code{weights}}{an array with many slices that represent the weighting 
systems for various filters; each slice is a matrix, which gives the weighting 
system to estimate a component, for example trend + seasonality, as a weighted 
average from the given time series; if 
\eqn{n} is the length of the given time series and \eqn{b} is the applied 
(relative) bandwidth, then the first row of the weighting system gives the 
weights to obtain estimates at \eqn{t=1}, the second row gives the weights to 
obtain estimates at \eqn{t=2} and so on for all left-hand side boundary points 
until the middle row, which contains the 
weights used at all interior points; the rows following the middle row contain
the weights for right-hand side boundary points (the rows are ordered
chronologically); 
the slice names are \code{"Trend"}, \code{"Season"} and \code{"Combined"},
where \code{"Combined"} are the weights to estimate trend + seasonality 
combined.}
}
}
\description{
Simultaneously estimate the trend and the 
seasonality via locally weighted regression in an equidistant time series 
under short memory. The default setting uses an iterative plug-in algorithm 
for identifying the asymptotically globally optimal bandwidth for smoothing.
}
\details{
Trend and seasonality are estimated based on the additive
nonparametric regression model for an equidistant time series
\deqn{y_t = m(x_t) + s(x_t) + \epsilon_t,}
where \eqn{y_t} is the observed time series with \eqn{t=1,...n}, \eqn{x_t = t / n} is the rescaled time
on the interval \eqn{[0, 1]}, \eqn{m(x_t)} is a smooth and deterministic
trend function, \eqn{s(x_t)} is a (slowly changing) seasonal component with 
seasonal period \eqn{p_s} and \eqn{\epsilon_t} are stationary errors 
with \eqn{E(\epsilon_t) = 0} and short-range dependence (see for example also 
Feng, 2013, for a similar model, where the stochastic term is however i.i.d.).

It is assumed that \eqn{m} and \eqn{s} can be approximated locally by a polynomial of
small order and by a trigonometric polynomial, respectively. Through locally
weighted regression, \eqn{m} and \eqn{s} can therefore be estimated given 
a suitable bandwidth.

The iterative-plug-in (IPI) algorithm, which numerically minimizes the
asymptotic mean squared error (AMISE) to select a bandwidth is 
an extension of Feng (2013) to the case with short-range dependence in the 
errors. To achieve this goal, the error variance in the AMISE in Feng (2013) 
is replaced by the sum of autocovariances of the error process and this 
quantity is being estimated using a slightly adjusted version of the 
Bühlmann (1996) algorithm. This procedure is similar to the method described 
in Feng, Gries and Fritz (2020), 
where data-driven local polynomial regression with an automatically selected 
bandwidth is used to estimate the trend according to a model without 
seasonality and where the same adjusted Bühlmann (1996) algorithm is 
considered to estimate the sum of autocovariances in the error process.

Define \eqn{I[m^{(k)}] = \int_{c_b}^{d_b} [m^{(k)}(x)]^2 dx}{I[m^(k)] =
int_[c_b]^[d_b] \{m^(k)\}^2 dx}, \eqn{\beta_{(k)} = \int_{-1}^{1} u^k
K(u) du}{\beta_(k) = int_[-1]^[1] u^k K(u) du}
and \eqn{R(K) = \int_{-1}^{1} K^{2}(u) du}{R(K) = int_[-1]^[1]
\{K\}^2 du}, where \eqn{p} is the order of the (local) polynomial 
considered for \eqn{m},
\eqn{k = p + 1} is the order of the asymptotically equivalent kernel \eqn{K} 
for estimating \eqn{m}, \eqn{0 \leq c_{b}< d_{b} \leq 1}{0 \le c_b < d_b \le 1}, and
\eqn{c_f} is the variance factor, i.e. the sum of autocovariances divided by 
\eqn{2\pi}.

Furthermore, we define
\deqn{C_{1} = \frac{I[m^{(k)}] \beta_{(k)}^2}{(k!)^2}}{C_1 =
[I[m^(k)]\{\beta_(k)\}^2] / (k!)^2}
and
\deqn{C_{2} = \frac{2 \pi c_{f} (d_b - c_b)[R(K) + (p_s - 1) R(W)]}{nh}}{C_2 =
2\pi(d_b - c_b)R(K)c_f / (nh)}
with \eqn{h} being the bandwidth, \eqn{n} being the number of
observations and \eqn{W} being the weighting function considered in the 
weighted least squares approach, for example a second-order kernel function 
with support on \eqn{[-1,1]}. The AMISE is then
\deqn{AMISE(h) = h^{2k}C_{1} + C_{2}.}{AMISE(h) = h^[2k] C_1 +
C_2.}

The function calculates suitable estimates for \eqn{c_f}, the variance
factor, and \eqn{I[m^{(k)}]}{I[m^(k)]} over different iterations. In each
iteration, a bandwidth is obtained in accordance with the AMISE that once
more serves as an input for the following iteration. The process repeats
until either convergence or the 40th iteration is reached. For further
details on the asymptotic theory or the algorithm, please consult Feng,
Gries and Fritz (2020) or Feng et al. (2019).

To apply the function, only few arguments are needed: a data input \code{y},
an object with smoothing options \code{smoothing_options} returned by 
\code{\link{set_options}} and 
a starting value for the relative bandwidth
\code{bwidth_start}. Aside from \code{y}, each argument has a default setting.
By default, a local linear trend is considered. In some cases, a local cubic 
trend may, however, be more suitable. For more specific information on the input arguments
consult the section \emph{Arguments}.

When applying the function, an optimal bandwidth is obtained based on the
IPI algorithm proposed by Feng, Gries and Fritz (2020). In a second step,
the nonparametric trend of the series and the seasonality are calculated with 
respect to the chosen bandwidth.

Note that with this function \eqn{m(x_t)} and \eqn{s(x_t)} can be 
estimated without a parametric
model assumption for the error series. Thus, after estimating and removing
the trend and the seasonality, any suitable parametric model, e.g. an 
ARMA(\eqn{p,q}) model for \code{errors = "autocor"}, can be fitted to the 
residuals (see \code{\link[stats]{arima}}).

Usually, a local cubic trend (\code{smoothing_options = set_options(order_poly = 3)})
gives more suitable results. Moreover, if the resulting bandwidth is too large, 
adjustments to the arguments \code{inflation_rate}, \code{drop} and 
\code{correction_factor} should be tried first in that order before any other changes 
to the input arguments.

The default print method for this function delivers key numbers such as the 
bandwidth considered for smoothing.

NOTE:

This function implements C++ code by means
of the \code{\link[Rcpp:Rcpp-package]{Rcpp}} and
\code{\link[RcppArmadillo:RcppArmadillo-package]{RcppArmadillo}} packages for
better performance.
}
\examples{
\donttest{
Xt <- log(EXPENDITURES)
smoothing_options <- set_options(order_poly = 3)
est <- deseats(Xt, smoothing_options = smoothing_options)
est
plot(est, which = 1)
}

}
\references{
\itemize{
\item{Bühlmann, P. (1996). Locally Adaptive Lag-Window Spectral Estimation.
Journal of Time Series Analysis, 17(3): 247-270. 
DOI: 10.1111/j.1467-9892.1996.tb00275.x.}
\item{Feng, Y. (2013). An iterative plug-in algorithm for decomposing seasonal 
time series using the Berlin Method. Journal of Applied Statistics, 40(2): 
266-281. DOI: 10.1080/02664763.2012.740626.}
\item{Feng, Y., Gries. T, and Fritz, M. (2020). Data-driven local polynomial 
for the trend and its derivatives in economic time series. Journal of 
Nonparametric Statistics, 32(2): 510-533. DOI: 10.1080/10485252.2020.1759598.}
}
}
\author{
\itemize{
\item Dominik Schulz (Research Assistant) (Department of Economics, Paderborn
University), \cr
Author and Package Creator
\item Yuanhua Feng (Department of Economics, Paderborn
University), \cr
Author
}
}
