\docType{methods}
\name{DEMIPathway}
\alias{DEMIPathway}
\title{Functional annotation of DEMI results}
\usage{
  DEMIPathway(object = "DEMIDiff")
}
\arguments{
  \item{object}{A \code{DEMIDiff} object. The
  \code{DEMIDiff} object contains the results to
  differential expression analysis that will be used for
  functional annotation analysis.}
}
\value{
  Returns the results of the functional annotation analysis
  in a \code{data.frame}.
}
\description{
  The function \code{DEMIPathway} performs functional
  annotation analysis on DEMI differential expression
  results stored in the \code{DEMIDiff} object. It takes
  into account the number of up- and down-regulated targets
  as well as the total number of targets for each
  functional category to calculate the statistical
  significance of the functional annotation. PS! This
  function can only be used if in the underlying
  \code{DEMIExperiment} object the \code{analysis}
  paramater was set as 'gene' or 'transcript' for it will
  before functional annotation only on genes.
}
\examples{
\dontrun{

# To use the example we need to download a subset of CEL files from
# http://www.ncbi.nlm.nih.gov/geo/query/acc.cgi?acc=GSE9819 published
# by Pradervand et al. 2008.

# Set the destination folder where the downloaded files fill be located.
# It can be any folder of your choosing.
destfolder <- "demitest/testdata/"

# Download packed CEL files and change the names according to the feature
# they represent (for example to include UHR or BRAIN in them to denote the
# features).
# It is good practice to name the files according to their features which
# allows easier identification of the files later.

ftpaddress <- "ftp://ftp.ncbi.nlm.nih.gov/geo/samples/GSM247nnn"
download.file( paste( ftpaddress, "GSM247694/suppl/GSM247694.CEL.gz", sep = "/" ),
		destfile = paste( destfolder, "UHR01_GSM247694.CEL.gz", sep = "" ) )
download.file( paste( ftpaddress, "GSM247695/suppl/GSM247695.CEL.gz", sep = "/" ),
		destfile = paste( destfolder, "UHR02_GSM247695.CEL.gz", sep = "" ) )
download.file( paste( ftpaddress, "GSM247698/suppl/GSM247698.CEL.gz", sep = "/" ),
		destfile = paste( destfolder, "UHR03_GSM247698.CEL.gz", sep = "" ) )
download.file( paste( ftpaddress, "GSM247699/suppl/GSM247699.CEL.gz", sep = "/" ),
		destfile = paste( destfolder, "UHR04_GSM247699.CEL.gz", sep = "" ) )
download.file( paste( ftpaddress, "GSM247696/suppl/GSM247696.CEL.gz", sep = "/" ),
		destfile = paste( destfolder, "BRAIN01_GSM247696.CEL.gz", sep = "" ) )
download.file( paste( ftpaddress, "GSM247697/suppl/GSM247697.CEL.gz", sep = "/" ),
		destfile = paste( destfolder, "BRAIN02_GSM247697.CEL.gz", sep = "" ) )
download.file( paste( ftpaddress, "GSM247700/suppl/GSM247700.CEL.gz", sep = "/" ),
		destfile = paste( destfolder, "BRAIN03_GSM247700.CEL.gz", sep = "" ) )
download.file( paste( ftpaddress, "GSM247701/suppl/GSM247701.CEL.gz", sep = "/" ),
		destfile = paste( destfolder, "BRAIN04_GSM247701.CEL.gz", sep = "" ) )

# We need the gunzip function (located in the R.utils package) to unpack the gz files.
# Also we will remove the original unpacked files for we won't need them.
library( R.utils )
for( i in list.files( destfolder ) ) {
	gunzip( paste( destfolder, i, sep = "" ), remove = TRUE )
}

# Now we can continue the example of the function demi

# Set up an experiment
demiexp <- DEMIExperiment(analysis = 'gene', celpath = destfolder,
			experiment = 'myexperiment', organism = 'homo_sapiens')

# Create clusters with an optimized wilcoxon's rank sum test incorporated within demi that
# precalculates the probabilities.
demiclust <- DEMIClust( demiexp, group = c( "BRAIN", "UHR" ), clust.method = demi.wilcox.test.fast )

# Calcuate differential expression
demidiff <- DEMIDiff( demiclust )

# Perform functiona annotation analysis on the DEMI analysis results
demipath <- DEMIPathway( demidiff )

}
}
\author{
  Sten Ilmjarv
}

