% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/coxmeg_m.R
\name{coxmeg_m}
\alias{coxmeg_m}
\title{Fit a Cox mixed-effects model for estimating HRs for a set of predictors}
\usage{
coxmeg_m(
  X,
  outcome,
  corr,
  type,
  cov = NULL,
  tau = NULL,
  min_tau = 1e-04,
  max_tau = 5,
  eps = 1e-06,
  order = NULL,
  detap = NULL,
  opt = "bobyqa",
  score = FALSE,
  threshold = 0,
  solver = NULL,
  spd = TRUE,
  verbose = TRUE,
  mc = 100
)
}
\arguments{
\item{X}{A matrix of the preidctors. Can be quantitative or binary values. Categorical variables need to be converted to dummy variables. Each row is a sample, and the predictors are columns.}

\item{outcome}{A matrix contains time (first column) and status (second column). The status is a binary variable (1 for failure / 0 for censored).}

\item{corr}{A relatedness matrix or a List object of matrices if there are multiple relatedness matrices. They can be a matrix or a 'dgCMatrix' class in the Matrix package. The matrix (or the sum if there are multiple) must be symmetric positive definite or symmetric positive semidefinite. The order of subjects must be consistent with that in outcome.}

\item{type}{A string indicating the sparsity structure of the relatedness matrix. Should be 'bd' (block diagonal), 'sparse', or 'dense'. See details.}

\item{cov}{An optional matrix of the covariates included in the null model for estimating the variance component. Can be quantitative or binary values. Categorical variables need to be converted to dummy variables. Each row is a sample, and the covariates are columns.}

\item{tau}{An optional positive value or vector for the variance component(s). If \code{tau} is given, the function will skip estimating the variance component, and use the given \code{tau} to analyze the predictors.}

\item{min_tau}{An optional positive value indicating the lower bound in the optimization algorithm for the variance component \code{tau}. Default is 1e-4.}

\item{max_tau}{An optional positive value indicating the upper bound in the optimization algorithm for the variance component \code{tau}. Default is 5.}

\item{eps}{An optional positive value indicating the relative convergence tolerance in the optimization algorithm. Default is 1e-6. A smaller value (e.g., 1e-8) can be used for better precision of the p-values in the situation where most SNPs under investigation have a very low minor allele count (<5).}

\item{order}{An optional integer value starting from 0. Only valid when \code{dense=FALSE}. It specifies the order of approximation used in the inexact newton method. Default is NULL, which lets coxmeg choose an optimal order.}

\item{detap}{An optional string indicating whether to use an approximation for log-determinant. Can be 'exact', 'diagonal', 'gkb', or 'slq'. Default is NULL, which lets the function select a method based on 'type' and other information. See details.}

\item{opt}{An optional logical scalar for the Optimization algorithm for estimating the variance component(s). Can be one of the following values: 'bobyqa', 'Brent', 'NM', or 'L-BFGS-B' (only for >1 variance components). Default is 'bobyqa'.}

\item{score}{An optional logical value indicating whether to perform a score test. Default is FALSE.}

\item{threshold}{An optional non-negative value. If threshold>0, coxmeg_m will reestimate HRs for those SNPs with a p-value<threshold by first estimating a variant-specific variance component. Default is 0.}

\item{solver}{An optional bianry value that can be either 1 (Cholesky Decomposition using RcppEigen), 2 (PCG) or 3 (Cholesky Decomposition using Matrix). Default is NULL, which lets the function select a solver. See details.}

\item{spd}{An optional logical value indicating whether the relatedness matrix is symmetric positive definite. Default is TRUE. See details.}

\item{verbose}{An optional logical value indicating whether to print additional messages. Default is TRUE.}

\item{mc}{An optional integer scalar specifying the number of Monte Carlo samples used for approximating the log-determinant when \code{detap='gkb'} or \code{detap='slq'}. Default is 100.}
}
\value{
beta: The estimated coefficient for each predictor in X.

HR: The estimated HR for each predictor in X.

sd_beta: The estimated standard error of beta.

p: The p-value of each SNP.

tau: The estimated variance component.

iter: The number of iterations until convergence.
}
\description{
\code{coxmeg_m} first estimates the variance component under a null model with only cov, and then analyzing each predictor in X one by one.
}
\section{About \code{type}}{

Specifying the type of the relatedness matrix (whether it is block-diagonal, general sparse, or dense). In the case of multiple relatedness matrices, it refers to the type of the sum of these matrices.
\itemize{ 
\item{"bd"}{ - used for a block-diagonal relatedness matrix, or a sparse matrix the inverse of which is also sparse. }
\item{"sparse"}{ - used for a general sparse relatedness matrix the inverse of which is not sparse.}
\item{"dense"}{ - used for a dense relatedness matrix.}
}
}

\section{About \code{spd}}{

When \code{spd=TRUE}, the relatedness matrix is treated as SPD. If the matrix is SPSD or not sure, use \code{spd=FALSE}.
}

\section{About \code{solver}}{

Specifying which method is used to solve the linear system in the optimization algorithm.  
\itemize{ 
\item{"1"}{ - Cholesky decompositon (RcppEigen:LDLT) is used to solve the linear system.}
\item{"2"}{ - PCG is used to solve the linear system. When \code{type='dense'}, it is recommended to set \code{solver=2} to have better computational performance.}
}
}

\section{About \code{detap}}{

Specifying the method to compute the log-determinant for estimating the variance component(s).
\itemize{ 
\item{"exact"}{ - the exact log-determinant is computed for estimating the variance component.}
\item{"diagonal"}{ - uses diagonal approximation and is only effective for a sparse relatedness matrix.}
\item{"slq"}{ - uses stochastic lanczos quadrature approximation. It uses the Lanczos algorithm to compute the weights and nodes. When type is 'bd' or 'sparse', it is often faster than 'gkb' and has the same accuracy. When type='dense', it is fater than 'gkb' by around half, but can be inaccurate if the relatedness matrix is (almost) singular.}
\item{"gkb"}{ - uses stochastic lanczos quadrature approximation. It uses the Golub-Kahan bidiagonalization algorithm to compute the weights and nodes. It is robust against an (almost) singular relatedness matrix when type='dense', but it is generally slower than 'slq'.}  
}
}

\examples{
library(Matrix)
library(MASS)
library(coxmeg)

## simulate a block-diagonal relatedness matrix
tau_var <- 0.2
n_f <- 100
mat_list <- list()
size <- rep(10,n_f)
offd <- 0.5
for(i in 1:n_f)
{
  mat_list[[i]] <- matrix(offd,size[i],size[i])
  diag(mat_list[[i]]) <- 1
}
sigma <- as.matrix(bdiag(mat_list))
n <- nrow(sigma)

## simulate random effects and outcomes
x <- mvrnorm(1, rep(0,n), tau_var*sigma)
myrates <- exp(x-1)
y <- rexp(n, rate = myrates)
cen <- rexp(n, rate = 0.02 )
ycen <- pmin(y, cen)
outcome <- cbind(ycen,as.numeric(y <= cen))

## simulate genotypes
g = matrix(rbinom(n*5,2,0.5),n,5)

## The following command will first estimate the variance component without g, 
## and then use it to estimate the HR for each preditor in g.
re = coxmeg_m(g,outcome,sigma,type='bd',tau=0.5,detap='diagonal',order=1)
re
}
\keyword{Cox}
\keyword{mixed-effects}
\keyword{model}
