\name{redundant}
\alias{redundant}

\title{
Identify structurally redundant asf in a csf
}
\description{
\code{redundant} takes a character vector \code{cond} containing complex solution formulas (csf) as input and tests for each element of \code{cond} whether the atomic solution formulas (asf) it consists of are structurally redundant. %determines the redundant \code{asf}s parts of one or several \code{csf}s.
}
\usage{
redundant(cond, x = NULL, simplify = TRUE)
}

\arguments{
  \item{cond}{
Character vector specifying complex solution formulas (csf); only strings of type csf are allowed, meaning conjunctions of one or more asf. 
}
  \item{x}{
An optional argument providing a \code{configTable}, a data frame, or a list specifying the factors' value ranges if \code{cond} contains multi-value factors; if \code{x} is not \code{NULL}, \code{cond} is tested for redundancy-freeness relative to \code{full.ct(x)}, otherwise relative to \code{full.ct(cond)}.
}
  \item{simplify}{
Logical; if \code{TRUE}, the result for csfs with the same number of component asfs is presented as a matrix, otherwise all results are presented as a list of logical vectors.  
}
}

\details{
According to the regularity theory of causation underlying CNA, a Boolean dependency structure is causally interpretable only if it does not contain any redundant elements. Boolean dependency structures may feature various types of redundancies, one of which are so-called \emph{structural redundancies}. A csf \eqn{\Phi} has a structural redundancy if, and only if, reducing \eqn{\Phi} by one or more of the asf it is composed of results in a csf \eqn{\Phi'} that is logically equivalent to \eqn{\Phi}. To illustrate, suppose that \eqn{\Phi} is composed of three asf: asf1 * asf2 * asf3; and suppose that \eqn{\Phi} is logically equivalent to \eqn{\Phi'}: asf1 * asf2. In that case, asf3 makes no difference to the behavior of the factors in \eqn{\Phi} and \eqn{\Phi'}; it is structurally redundant and, accordingly, must not be causally interpreted. For more details see the package vignette (\code{vignette("cna")}) or Baumgartner and Falk (2023).



The function \code{redundant} takes a character vector \code{cond} composed of csf as input an tests for each element of \code{cond} whether it is structurally redundant or not. As a test for structural redundancies amounts to a test of logical equivalencies, it must be conducted relative to all logically possible configurations of the factors in \code{cond}. That space of logical possibilities is generated by \code{full.ct(cond)} in case of \code{x =  NULL}, and by \code{full.ct(x)} otherwise. If all factors in \code{cond} are binary, \code{x} is optional and without influence on the output of \code{redundant}. If some factors in \code{cond} are multi-value, \code{redundant} needs to be given the range of these values. \code{x} can be a data frame or \code{configTable} listing all possible value configurations or a list of the possible values for each factor in \code{cond}.

If \code{redundant} returns \code{TRUE} for a csf, that csf must not be causally interpreted but further processed by \code{\link{minimalizeCsf}}. As of version 3.0 of the \pkg{cna} package, standard calls of the \code{cna} and \code{csf} functions automatically eliminate all structurally redundant asf.

}

\value{
A list of logical vectors or a logical matrix.

If all \code{csf} in \code{cond} have the same number of \code{asf} and \code{simplify = TRUE},  the result is a logical matrix with \code{length(cond)} rows and the number of columns corresponds to the number of \code{asf} in each \code{csf}. In all other cases, a list of logical vectors of the same length as \code{cond} is returned.
}

\section{Contributors}{
Falk, Christoph: identification and solution of the problem of structural redundancies
}

\references{
Baumgartner, Michael and Christoph Falk. 2023. \dQuote{Boolean Difference-Making: A Modern Regularity Theory of Causation}. \emph{The British Journal for the Philosophy of Science}, 74(1), 171-197. 

}

\seealso{
\code{\link[cna]{condition}}, \code{\link{full.ct}}, \code{\link{is.inus}}, \code{\link{csf}}, \code{\link{minimalizeCsf}}.
}

\examples{
# Binary factors.
cond1 <- c("(f + a*D <-> C)*(C + A*B <-> D)*(c + a*E <-> F)", "f + a*D <-> C")
redundant(cond1)

edu.sol <- csf(cna(d.educate), inus.only = FALSE)$condition
redundant(edu.sol, d.educate)

redundant(edu.sol, d.educate, simplify = FALSE)


# Default application of csf() with automatic elimination of structural redundancies.
ct.pban <- configTable(d.pban)
cna.pban <- cna(ct.pban, con = .75, cov = .75)
csf.pban <- csf(cna.pban)
# check for structural redundancies in the csf:
redund.pban <- redundant(csf.pban$condition, ct.pban) 
# show result for the first few:
head(redund.pban)  
# verify that no solutions with structural redundancies are returned
any(unlist(redund.pban))  # FALSE - no redundancies

# Non-default application of csf() without automatic elimination of structural redundancies.
csf.pban <- csf(cna.pban, inus.only = FALSE)
redund.pban <- redundant(csf.pban$condition, ct.pban)
head(redund.pban)
# various solutions with structural redundancies are returned:
table(apply(redund.pban, 1, any)) # each TRUE corresponds to a csf with struct. redundancies


# If no x is specified defining the factors' value ranges, the space of
# logically possible configurations is limited to the factor values contained in
# cond, resulting in structural redundancies that disappear as soon as x is specified.
cond2 <- "(C=0*F=0 + G=1<-> T=2)*(T=2 + G=2 <-> P=1)"
redundant(cond2)
redundant(cond2, list(C=0:2, F=0:2, G=0:3, T=0:2, P=0:2))

}
