#' Visualise climate cross correlation or autocorrelation.
#' 
#' Create a colour plot to visualise the results of \code{\link{autowin}} or
#' \code{\link{crosswin}}. Displays correlation across all desired climate
#' windows.
#' @param CorWindowOutput Output of \code{\link{autowin}} or
#'   \code{\link{crosswin}}
#' @param TYPE Should be either "A" for data generated by \code{\link{autowin}}
#'   or "C" for data generated by \code{\link{crosswin}}.
#' @return Will generate a colour plot to visualise the correlation data.
#' @author Liam D. Bailey and Martijn van de Pol
#' @import ggplot2
#' @examples
#' \dontrun{
#' # Visualise climate autocorrelation
#' 
#' data(Mass)
#' data(MassClimate)
#' 
#' # Fit a single climate window using the datasets Mass and MassClimate.
#' 
#' single <- singlewin(Xvar = MassClimate$Temp, CDate = MassClimate$Date, BDate = Mass$Date,
#'                     baseline = lm(Mass ~ 1, data = Mass), furthest = 72, closest = 15,
#'                     STAT = "mean", FUNC = "L",
#'                     FIXED = TRUE, cutoff.day = 20, cutoff.month = 5,
#'                     CMISSING = FALSE, CINTERVAL = "D")            
#' 
#' # Test the autocorrelation between the climate in this single window and other climate windows.
#' 
#' auto <- autowin(reference = single[[2]]$temporary,
#'                 Xvar  = MassClimate$Temp, CDate = MassClimate$Date, BDate = Mass$Date,
#'                 furthest = 365, closest = 0, STAT = "mean",
#'                 FIXED = TRUE, cutoff.day = 20, cutoff.month = 5,
#'                 CMISSING = FALSE, CINTERVAL = "D")
#'                 
#' # Plot the auto-correlation data
#' 
#' plotcor(auto, TYPE = "A")
#' }
#' @export


plotcor <- function(CorWindowOutput, TYPE=TYPE){
  ifelse (TYPE == "A", title <- "Correlation between single window and all other windows", 
          title <- "Correlation between 2 climate variables in all windows")
  with(CorWindowOutput, {
    ggplot(CorWindowOutput, aes(x = WindowClose, y = WindowOpen, z = cor))+
      geom_tile(aes(fill = cor))+
      scale_fill_gradient2(low = "red", mid = "yellow", high = "blue", 
                           midpoint = mean(CorWindowOutput$cor), name = "")+
      theme_classic()+
      theme(panel.grid.major = element_blank(),
            panel.grid.minor = element_blank(),
            axis.line = element_line(size = 0.25, colour = "black"))+
      ggtitle(title)+
      ylab("Window open")+
      xlab("Window close")
  }
  )
}