% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/operators.R
\name{hessian}
\alias{hessian}
\alias{\%hessian\%}
\title{Numerical and Symbolic Hessian}
\usage{
hessian(f, var, params = list(), accuracy = 4, stepsize = NULL, drop = TRUE)

f \%hessian\% var
}
\arguments{
\item{f}{array of \code{characters} or a \code{function} returning a \code{numeric} array.}

\item{var}{vector giving the variable names with respect to which the derivatives are to be computed and/or the point where the derivatives are to be evaluated. See \code{\link{derivative}}.}

\item{params}{\code{list} of additional parameters passed to \code{f}.}

\item{accuracy}{degree of accuracy for numerical derivatives.}

\item{stepsize}{finite differences stepsize for numerical derivatives. It is based on the precision of the machine by default.}

\item{drop}{if \code{TRUE}, return the Hessian as a matrix and not as an \code{array} for scalar-valued functions.}
}
\value{
Hessian matrix for scalar-valued functions when \code{drop=TRUE}, \code{array} otherwise.
}
\description{
Computes the numerical Hessian of \code{functions} or the symbolic Hessian of \code{characters}.
}
\details{
In Cartesian coordinates, the Hessian of a scalar-valued function \eqn{F} is the 
square matrix of second-order partial derivatives:

\deqn{(H(F))_{ij} = \partial_{ij}F}

When the function \eqn{F} is a tensor-valued function \eqn{F_{d_1,\dots,d_n}}, 
the \code{hessian} is computed for each scalar component.

\deqn{(H(F))_{d_1\dots d_n,ij} = \partial_{ij}F_{d_1\dots d_n}}

It might be tempting to apply the definition of the Hessian as the Jacobian of the 
gradient to write it in arbitrary orthogonal coordinate systems. However, this results in a 
Hessian matrix that is not symmetric and ignores the distinction between vector 
and covectors in tensor analysis. The generalization to arbitrary coordinate system 
is not currently supported.
}
\section{Functions}{
\itemize{
\item \code{f \%hessian\% var}: binary operator with default parameters.

}}
\examples{
### symbolic Hessian 
hessian("x*y*z", var = c("x", "y", "z"))

### numerical Hessian in (x=1, y=2, z=3)
f <- function(x, y, z) x*y*z
hessian(f = f, var = c(x=1, y=2, z=3))

### vectorized interface
f <- function(x) x[1]*x[2]*x[3]
hessian(f = f, var = c(1, 2, 3))

### symbolic vector-valued functions
f <- c("y*sin(x)", "x*cos(y)")
hessian(f = f, var = c("x","y"))

### numerical vector-valued functions
f <- function(x) c(sum(x), prod(x))
hessian(f = f, var = c(0,0,0))

### binary operator
"x*y^2" \%hessian\% c(x=1, y=3)

}
\references{
Guidotti, E. (2020). "calculus: High dimensional numerical and symbolic calculus in R". \url{https://arxiv.org/abs/2101.00086}
}
\seealso{
Other differential operators: 
\code{\link{curl}()},
\code{\link{derivative}()},
\code{\link{divergence}()},
\code{\link{gradient}()},
\code{\link{jacobian}()},
\code{\link{laplacian}()}
}
\concept{differential operators}
