#' Tidying methods for multinomial logistic regression models
#' 
#' These methods tidy the coefficients of multinomial logistic regression 
#' models generated by \code{multinom} of the \code{nnet} package.
#' 
#' @param x A model object of class \code{multinom}
#' 
#' @return All tidying methods return a \code{data.frame} without rownames.
#' The structure depends on the method chosen.
#' 
#' @name multinom_tidiers
#' 
#' @examples
#' 
#' if (require(nnet) & require(MASS)){
#'   example(birthwt)
#'   bwt.mu <- multinom(low ~ ., bwt)
#'   tidy(bwt.mu)
#'   glance(bwt.mu)
#'   
#'   #* This model is a truly terrible model
#'   #* but it should show you what the output looks
#'   #* like in a multinomial logistic regression
#'   
#'   fit.gear <- multinom(gear ~ mpg + factor(am), data=mtcars)
#'   tidy(fit.gear)
#'   glance(fit.gear)
#' }
#'   

NULL

#' @rdname multinom_tidiers
#' 
#' @param conf.int whether to include a confidence interval
#' @param conf.level confidence level of the interval, used only if
#' \code{conf.int=TRUE}
#' @param exponentiate whether to exponentiate the coefficient estimates
#' and confidence intervals (typical for multinomial logistic regression)
#' @param ... extra arguments, not used
#' 
#' @details If \code{conf.int=TRUE}, the confidence interval is computed with
#' the \code{\link{confint}} function.
#' 
#' While \code{tidy} and \code{glance} are supported for "multinom" objects, 
#' \code{augment} is not.
#' 
#' @return \code{tidy.multinom} returns one row for each coefficient at each 
#' level of the response variable, with six columns:
#'   \item{y.value}{The response level}
#'   \item{term}{The term in the model being estimated and tested}
#'   \item{estimate}{The estimated coefficient}
#'   \item{std.error}{The standard error from the linear model}
#'   \item{statistic}{Wald z-statistic}
#'   \item{p.value}{two-sided p-value}
#' 
#' If \code{conf.int=TRUE}, it also includes columns for \code{conf.low} and
#' \code{conf.high}, computed with \code{\link{confint}}.
#' 
#' \code{glance.multinom} returns a 
#' 
#' @export
tidy.multinom <- function(x, 
                          conf.int=FALSE, 
                          conf.level=.95, 
                          exponentiate=TRUE, ...) {
  col_names <- if (length(x$lev) > 2) colnames(coef(x)) else names(coef(x))
  s <- summary(x)
  
  coef <- matrix(coef(s), 
                 byrow=FALSE, 
                 nrow=length(x$lev)-1,
                 dimnames=list(x$lev[-1], 
                               col_names))
  se <- matrix(s$standard.errors, 
               byrow=FALSE, 
               nrow=length(x$lev)-1,
               dimnames=list(x$lev[-1], 
                             col_names))
  
  #* Quick utility to convert each row of coef to a data frame
  multinomRowToDf <- function(r, coef, se, col_names){
    unrowname(data.frame(y.level = rep(r, length(col_names)),
                         term = colnames(coef),
                         estimate = coef[r, ],
                         std.error = se[r, ],
                         stringsAsFactors=FALSE))
  }
    
  #* Convert to coefficients data frame
  ret <- lapply(rownames(coef), multinomRowToDf, coef, se, col_names)
  ret <- do.call("rbind", ret)
  
  #* Calculate Wald-type Z and p-value
  ret$statistic <- ret$estimate / ret$std.error
  ret$p.value <- stats::pnorm(abs(ret$statistic), 0, 1, lower.tail=FALSE) * 2
  
  #* Confidence Interval
  if (conf.int){
    ci <- apply(stats::confint(x), 2, function(a) unlist(as.data.frame(a)))
    ci <- as.data.frame(ci)
    names(ci) <- c("conf.low", "conf.high")
    ret <- cbind(ret, ci)
  }
  
  #* Exponentiate (for Odds Ratios)
  if (exponentiate){
    exp.col <- if(conf.int) c("estimate", "conf.low", "conf.high") else "estimate"
    ret[, exp.col] <- lapply(ret[, exp.col, drop=FALSE], exp)
  }
  
  unrowname(ret)
}

#' @rdname multinom_tidiers
#' @return \code{glance.multinom} returns a one-row data.frame with the columns
#'   \item{edf}{The effective degrees of freedom}
#'   \item{deviance}{deviance}
#'   \item{AIC}{the Akaike Information Criterion}
#'
#' @export
glance.multinom <- function(x, ...) {
  ret <- with(x,
              data.frame(edf=edf, 
                         deviance = deviance,
                         AIC = AIC))
  ret
}
