#' Attach a full model to the tibble of model terms
#'
#' To facilitate the use of broom helpers with pipe, it is recommended to
#' attach the original model as an attribute to the tibble of model terms
#' generated by `broom::tidy()`.
#'
#' `tidy_attach_model()` attach the model to a tibble already generated while
#' `tidy_and_attach()` will apply `broom::tidy()` and attach the model.
#'
#' Use `tidy_get_model()` to get the model attached to the tibble and
#' `tidy_detach_model()` to remove the attribute containing the model.
#' @param model a model to be attached/tidied
#' @param x a tibble of model terms
#' @param tidy_fun option to specify a custom tidier function
#' @param ... other arguments passed to `tidy_fun()`
#' @family tidy_helpers
#' @examples
#' mod <- lm(Sepal.Length ~ Sepal.Width + Species, data = iris)
#' tt <- mod %>%
#'   tidy_and_attach(conf.int = TRUE)
#' tt
#' tidy_get_model(tt)
#' @export
tidy_attach_model <- function(x, model) {
  attr(x, "model") <- model
  x
}

#' @rdname tidy_attach_model
#' @export
tidy_and_attach <- function(model, tidy_fun = broom::tidy, ...) {
  tidy_fun(model, ...) %>%
    tidy_attach_model(model)
}

#' @rdname tidy_attach_model
#' @export
tidy_get_model <- function(x) {
  attr(x, "model")
}

#' @rdname tidy_attach_model
#' @export
tidy_detach_model <- function(x) {
  attr(x, "model") <- NULL
  x
}
