stan_llh <- function(family, ...) {
  # Stan code for the model likelihood 
  UseMethod("stan_llh")
}

#' @export
stan_llh.default <- function(family, bterms, data, mix = "", 
                             ptheta = FALSE, ...) {
  # Likelihood in Stan language
  # Args:
  #   family: the model family
  #   bterms: object of class brmsterms
  #   data: data passed by the user
  #   autocor: object of classe cor_brms
  #   mix: optional mixture component ID
  #   ptheta: are mixing proportions predicted?
  stopifnot(is.family(family))
  stopifnot(is.brmsterms(bterms))
  stopifnot(length(mix) == 1L)
  bterms$family <- family
  resp <- usc(combine_prefix(bterms))
  # prepare family part of the likelihood
  llh_args <- nlist(bterms, resp, mix)
  llh_fun <- paste0("stan_llh_", prepare_family(bterms)$fun)
  llh <- do.call(llh_fun, llh_args)
  # incorporate other parts into the likelihood
  args <- nlist(llh, bterms, data, resp, mix, ptheta)
  if (nzchar(mix)) {
    out <- do.call(stan_llh_mix, args)
  } else if (is.formula(bterms$adforms$cens)) {
    out <- do.call(stan_llh_cens, args)
  } else if (is.formula(bterms$adforms$weights)) {
    out <- do.call(stan_llh_weights, args)
  } else {
    out <- do.call(stan_llh_general, args)
  }
  if (grepl("\\[n\\]", out) && !nzchar(mix)) {
    # loop over likelihood if it cannot be vectorized
    out <- paste0("  for (n in 1:N) { \n    ", out, "    } \n")
  }
  out
}

#' @export
stan_llh.mixfamily <- function(family, bterms, ...) {
  dp_ids <- dpar_id(names(bterms$dpars))
  fdp_ids <- dpar_id(names(bterms$fdpars))
  ptheta <- any(dpar_class(names(bterms$dpars)) %in% "theta")
  llh <- rep(NA, length(family$mix))
  for (i in seq_along(family$mix)) {
    sbterms <- bterms
    sbterms$dpars <- sbterms$dpars[dp_ids == i]
    sbterms$fdpars <- sbterms$fdpars[fdp_ids == i]
    llh[i] <- stan_llh(
      family$mix[[i]], sbterms, mix = i, ptheta = ptheta, ...
    )
  }
  resp <- usc(combine_prefix(bterms))
  has_weights <- is.formula(bterms$adforms$weights)  
  weights <- if (has_weights) paste0("weights", resp, "[n] * ")
  paste0(
    "  for (n in 1:N) { \n",
    "      real ps[", length(llh), "]; \n",
    collapse("    ", llh),
    "    ", tp(), weights, "log_sum_exp(ps); \n",
    "    } \n"
  )
}

#' @export
stan_llh.brmsterms <- function(family, ...) {
  paste0("  ", stan_llh(family$family, bterms = family, ...))
}

#' @export
stan_llh.mvbrmsterms <- function(family, ...) {
  if (family$rescor) {
    out <- stan_llh(as.brmsterms(family), ...)
  } else {
    out <- collapse(ulapply(family$terms, stan_llh, ...)) 
  }
  out
}

stan_llh_general <- function(llh, bterms, data, resp = "", ...) {
  # default likelihood in Stan language
  stopifnot(is.sdist(llh))
  n <- if (grepl("\\[n\\]", llh$args)) "[n]"
  lpdf <- stan_llh_lpdf_name(bterms)
  Y <- stan_llh_Y_name(bterms)
  tr <- stan_llh_trunc(llh, bterms, data, resp = resp)
  paste0(
    tp(), llh$dist, "_", lpdf, "(", Y, resp, n, llh$shift,
    " | ", llh$args, ")", tr, "; \n"
  )
}

stan_llh_cens <- function(llh, bterms, data, resp = "", ...) {
  # censored likelihood in Stan language
  stopifnot(is.sdist(llh))
  s <- collapse(rep(" ", 6))
  cens <- has_cens(bterms, data = data)
  interval <- isTRUE(attr(cens, "interval"))
  lpdf <- stan_llh_lpdf_name(bterms)
  has_weights <- is.formula(bterms$adforms$weights)
  Y <- stan_llh_Y_name(bterms)
  w <- if (has_weights) paste0("weights", resp, "[n] * ")
  tr <- stan_llh_trunc(llh, bterms, data, resp = resp)
  tp <- tp()
  if (interval) {
    int_cens <- paste0(
      s, "} else if (cens", resp, "[n] == 2) { \n",
      s, tp, w, "log_diff_exp(", 
      llh$dist, "_lcdf(rcens", resp, "[n]", llh$shift, " | ", llh$args, "), ",
      llh$dist, "_lcdf(", Y, resp, "[n]", llh$shift, " | ", llh$args, "))", 
      tr, "; \n"
    )
  } else {
    int_cens <- ""
  }
  paste0(
    "  // special treatment of censored data \n",
    s, "if (cens", resp, "[n] == 0) {\n", 
    s, tp, w, llh$dist, "_", lpdf, "(", Y, resp, "[n]", llh$shift,
    " | ", llh$args, ")", tr, ";\n",
    s, "} else if (cens", resp, "[n] == 1) {\n",         
    s, tp, w, llh$dist, "_lccdf(", Y, resp, "[n]", llh$shift, 
    " | ", llh$args, ")", tr, ";\n",
    s, "} else if (cens", resp, "[n] == -1) {\n",
    s, tp, w, llh$dist, "_lcdf(", Y, resp, "[n]", llh$shift, 
    " | ", llh$args, ")", tr, ";\n",
    int_cens, s, "} \n"
  )
}

stan_llh_weights <- function(llh, bterms, data, resp = "", ...) {
  # weighted likelihood in Stan language
  stopifnot(is.sdist(llh))
  tr <- stan_llh_trunc(llh, bterms, data, resp = resp)
  lpdf <- stan_llh_lpdf_name(bterms)
  Y <- stan_llh_Y_name(bterms)
  paste0(
    tp(), "weights", resp, "[n] * ", llh$dist, "_", lpdf, 
    "(", Y, resp, "[n]", llh$shift, " | ", llh$args,")", tr, "; \n"
  )
}

stan_llh_mix <- function(llh, bterms, data, mix, 
                         ptheta, resp = "", ...) {
  # likelihood of a single mixture component
  stopifnot(is.sdist(llh))
  theta <- ifelse(ptheta,
    paste0("theta", mix, resp, "[n]"), 
    paste0("log(theta", mix, resp, ")")
  )
  tr <- stan_llh_trunc(llh, bterms, data, resp = resp)
  lpdf <- stan_llh_lpdf_name(bterms)
  Y <- stan_llh_Y_name(bterms)
  paste0(
    "  ps[", mix, "] = ", theta, " + ", llh$dist, "_", 
    lpdf, "(", Y, resp, "[n] | ", llh$args, ")", tr, "; \n"
  )
}

stan_llh_trunc <- function(llh, bterms, data, resp = "", short = FALSE) {
  # truncated part of the likelihood
  # Args:
  #   short: use the T[, ] syntax?
  stopifnot(is.sdist(llh))
  bounds <- get_bounds(bterms, data = data)
  if (!any(bounds$lb > -Inf | bounds$ub < Inf)) {
    return("")
  }
  lb <- ifelse(any(bounds$lb > -Inf), paste0("lb", resp, "[n]"), "")
  ub <- ifelse(any(bounds$ub < Inf), paste0("ub", resp, "[n]"), "")
  if (short) {
    # truncation using T[, ] syntax
    out <- paste0(" T[", lb, ", ", ub, "]")
  } else {
    # truncation making use of _lcdf functions
    ms <- paste0(" - \n", collapse(rep(" ", 8)))
    if (any(bounds$lb > -Inf) && !any(bounds$ub < Inf)) {
      out <- paste0(ms, llh$dist, "_lccdf(", lb, " | ", llh$args, ")")
    } else if (!any(bounds$lb > -Inf) && any(bounds$ub < Inf)) {
      out <- paste0(ms, llh$dist, "_lcdf(", ub, " | ", llh$args, ")")
    } else if (any(bounds$lb > -Inf) && any(bounds$ub < Inf)) {
      trr <- paste0(llh$dist, "_lcdf(", ub, " | ", llh$args, ")")
      trl <- paste0(llh$dist, "_lcdf(", lb, " | ", llh$args, ")")
      out <- paste0(ms, "log_diff_exp(", trr, ", ", trl, ")")
    }
  }
  out
}

stan_llh_lpdf_name <- function(bterms) {
  ifelse(use_int(bterms$family), "lpmf", "lpdf")
}

stan_llh_Y_name <- function(bterms) {
  ifelse(is.formula(bterms$adforms$mi), "Yl", "Y")
}

stan_llh_dpars <- function(bterms, reqn, resp = "", mix = "", dpars = NULL) {
  # prepare names of distributional parameters
  # Args:
  #   reqn: will the likelihood be wrapped in a loop over n?
  #   dpars: optional names of distributional parameters to be prepared
  if (is.null(dpars)) {
    dpars <- paste0(valid_dpars(bterms), mix)
  }
  is_pred <- dpars %in% c("mu", names(bterms$dpars))
  out <- paste0(dpars, resp, ifelse(reqn & is_pred, "[n]", ""))
  named_list(dpars, out)
}

stan_llh_simple_lpdf <- function(lpdf, link, bterms, sep = "_") {
  # adjust lpdf name if a more efficient version is available
  # for a specific link. For instance poisson_log
  stopifnot(is.brmsterms(bterms))
  cens_or_trunc <- stan_llh_adj(bterms, c("cens", "trunc"))
  if (bterms$family$link == link && !cens_or_trunc) {
    lpdf <- paste0(lpdf, sep, link)
  }
  lpdf
}

stan_llh_dpar_usc_logit <- function(dpar, bterms) {
  # prepare _logit suffix for distributional parameters
  # currently only used in zero-inflated and hurdle models
  stopifnot(dpar %in% c("zi", "hu"))
  stopifnot(is.brmsterms(bterms))
  cens_or_trunc <- stan_llh_adj(bterms, c("cens", "trunc"))
  usc_logit <- isTRUE(bterms$dpars[[dpar]]$family$link == "logit")
  ifelse(usc_logit && !cens_or_trunc, "_logit", "")
}

stan_llh_add_se <- function(sigma, bterms, reqn, resp = "") {
  # prepare the code for 'sigma' in the likelihood statement
  if (is.formula(bterms$adforms$se)) {
    nse <- if (reqn) "[n]"
    if (no_sigma(bterms)) {
      sigma <- paste0("se", resp, nse) 
    } else {
      sigma <- paste0("sqrt(square(", sigma, ") + se2", resp, nse, ")")
    }
  }
  sigma
}

stan_llh_adj <- function(x, adds = c("weights", "cens", "trunc")) {
  # checks if certain 'adds' are present so that the LL has to be adjusted
  # Args:
  #   x: named list of formulas or brmsterms object
  #   adds: vector of addition argument names
  stopifnot(all(adds %in% c("weights", "cens", "trunc")))
  if (is.brmsterms(x)) x <- x$adforms
  any(ulapply(x[adds], is.formula))
}

# one function per family
stan_llh_gaussian <- function(bterms, resp = "", mix = "") {
  reqn <- stan_llh_adj(bterms) || nzchar(mix)
  p <- stan_llh_dpars(bterms, reqn, resp, mix)
  p$sigma <- stan_llh_add_se(p$sigma, bterms, reqn, resp)
  sdist("normal", p$mu, p$sigma)
}

stan_llh_gaussian_mv <- function(bterms, resp = "", mix = "") {
  reqn <- stan_llh_adj(bterms) || nzchar(mix) || bterms$sigma_pred
  p <- list(Mu = paste0("Mu", if (reqn) "[n]"))
  p$LSigma <- paste0("LSigma", if (bterms$sigma_pred) "[n]")
  sdist("multi_normal_cholesky", p$Mu, p$LSigma)
}

stan_llh_gaussian_cov <- function(bterms, resp = "", mix = "") {
  if (stan_llh_adj(bterms)) {
    stop2("Invalid addition arguments for this model.")
  }
  p <- stan_llh_dpars(bterms, FALSE, resp, mix)
  v <- c("se2", "N_tg", "begin_tg", "end_tg", "nobs_tg", "res_cov_matrix")
  p[v] <- as.list(paste0(v, resp))
  sdist("normal_cov", 
    p$mu, p$se2, p$N_tg, p$begin_tg, 
    p$end_tg, p$nobs_tg, p$res_cov_matrix
  )
}

stan_llh_gaussian_fixed <- function(bterms, resp = "", mix = "") {
  has_se <- is.formula(bterms$adforms$se)
  if (stan_llh_adj(bterms) || has_se) {
    stop2("Invalid addition arguments for this model.")
  }
  p <- stan_llh_dpars(bterms, FALSE, resp, mix)
  p$LV <- paste0("LV", resp)
  sdist("multi_normal_cholesky", p$mu, p$LV)
}

stan_llh_gaussian_lagsar <- function(bterms, resp = "", mix = "") {
  p <- stan_llh_dpars(bterms, FALSE, resp, mix)
  p$sigma <- stan_llh_add_se(p$sigma, bterms, FALSE, resp)
  v <- c("lagsar", "W")
  p[v] <- as.list(paste0(v, resp))
  sdist("normal_lagsar", p$mu, p$sigma, p$lagsar, p$W)
}

stan_llh_gaussian_errorsar <- function(bterms, resp = "", mix = "") {
  p <- stan_llh_dpars(bterms, FALSE, resp, mix)
  p$sigma <- stan_llh_add_se(p$sigma, bterms, FALSE, resp)
  v <- c("errorsar", "W")
  p[v] <- as.list(paste0(v, resp))
  sdist("normal_errorsar", p$mu, p$sigma, p$errorsar, p$W)
}

stan_llh_student <- function(bterms, resp = "", mix = "") {
  reqn <- stan_llh_adj(bterms) || nzchar(mix)
  p <- stan_llh_dpars(bterms, reqn, resp, mix)
  p$sigma <- stan_llh_add_se(p$sigma, bterms, reqn, resp)
  sdist("student_t", p$nu, p$mu, p$sigma)
}

stan_llh_student_mv <- function(bterms, resp = "", mix = "") {
  reqn <- stan_llh_adj(bterms) || nzchar(mix) || bterms$sigma_pred
  p <- stan_llh_dpars(bterms, reqn, resp, mix, dpars = "nu")
  p$Mu <- paste0("Mu", if (reqn) "[n]")
  p$Sigma <- paste0("Sigma", if (bterms$sigma_pred) "[n]")
  sdist("multi_student_t", p$nu, p$Mu, p$Sigma)
}

stan_llh_student_cov <- function(bterms, resp = "", mix = "") {
  if (stan_llh_adj(bterms)) {
    stop2("Invalid addition arguments for this model.")
  }
  p <- stan_llh_dpars(bterms, FALSE, resp, mix)
  v <- c("se2", "N_tg", "begin_tg", "end_tg", "nobs_tg", "res_cov_matrix")
  p[v] <- as.list(paste0(v, resp))
  sdist("student_t_cov", 
    p$nu, p$mu, p$se2, p$N_tg, p$begin_tg,
    p$end_tg, p$nobs_tg, p$res_cov_matrix
  )
}

stan_llh_student_fixed <- function(bterms, resp = "", mix = "") {
  has_se <- is.formula(bterms$adforms$se)
  if (stan_llh_adj(bterms) || has_se) {
    stop2("Invalid addition arguments for this model.")
  }
  p <- stan_llh_dpars(bterms, FALSE, resp, mix)
  p$V <- paste0("V", resp)
  sdist("multi_student_t", p$nu, p$mu, p$V)
}

stan_llh_student_lagsar <- function(bterms, resp = "", mix = "") {
  p <- stan_llh_dpars(bterms, FALSE, resp, mix)
  p$sigma <- stan_llh_add_se(p$sigma, bterms, FALSE, resp)
  v <- c("lagsar", "W")
  p[v] <- as.list(paste0(v, resp))
  sdist("student_t_lagsar", p$nu, p$mu, p$sigma, p$lagsar, p$W)
}

stan_llh_student_errorsar <- function(bterms, resp = "", mix = "") {
  p <- stan_llh_dpars(bterms, FALSE, resp, mix)
  p$sigma <- stan_llh_add_se(p$sigma, bterms, FALSE, resp)
  v <- c("errorsar", "W")
  p[v] <- as.list(paste0(v, resp))
  sdist("student_t_errorsar", p$nu, p$mu, p$sigma, p$errorsar, p$W)
}

stan_llh_lognormal <- function(bterms, resp = "", mix = "") {
  reqn <- stan_llh_adj(bterms) || nzchar(mix)
  p <- stan_llh_dpars(bterms, reqn, resp, mix)
  sdist("lognormal", p$mu, p$sigma)
}

stan_llh_shifted_lognormal <- function(bterms, resp = "", mix = "") {
  reqn <- stan_llh_adj(bterms) || nzchar(mix)
  p <- stan_llh_dpars(bterms, reqn, resp, mix)
  sdist("lognormal", p$mu, p$sigma, shift = paste0(" - ", p$ndt))
}

stan_llh_asym_laplace <- function(bterms, resp = "", mix = "") {
  p <- stan_llh_dpars(bterms, TRUE, resp, mix)
  sdist("asym_laplace", p$mu, p$sigma, p$quantile)
}

stan_llh_skew_normal <- function(bterms, resp = "", mix = "", ...) {
  reqn <- stan_llh_adj(bterms) || nzchar(mix)
  p <- stan_llh_dpars(bterms, reqn, resp, mix)
  p$sigma <- stan_llh_add_se(p$sigma, bterms, reqn, resp)
  # required because of CP parameterization of mu and sigma
  nomega <- any(grepl("\\[n\\]", c(p$sigma, p$alpha)))
  nomega <- if (reqn && nomega) "[n]"
  p$omega <- paste0("omega", mix, resp, nomega)
  sdist("skew_normal", p$mu, p$omega, p$alpha)
}

stan_llh_poisson <- function(bterms, resp = "", mix = "") {
  reqn <- stan_llh_adj(bterms) || nzchar(mix)
  p <- stan_llh_dpars(bterms, reqn, resp, mix)
  lpdf <- stan_llh_simple_lpdf("poisson", "log", bterms)
  sdist(lpdf, p$mu)
}

stan_llh_negbinomial <- function(bterms, resp = "", mix = "") {
  reqn <- stan_llh_adj(bterms) || nzchar(mix)
  p <- stan_llh_dpars(bterms, reqn, resp, mix)
  lpdf <- stan_llh_simple_lpdf("neg_binomial_2", "log", bterms)
  sdist(lpdf, p$mu, p$shape)
}

stan_llh_geometric <- function(bterms, resp = "", mix = "") {
  reqn <- stan_llh_adj(bterms) || nzchar(mix)
  p <- stan_llh_dpars(bterms, reqn, resp, mix)
  lpdf <- stan_llh_simple_lpdf("neg_binomial_2", "log", bterms)
  sdist(lpdf, p$mu, "1")
}

stan_llh_binomial <- function(bterms, resp = "", mix = "") {
  reqn <- stan_llh_adj(bterms) || nzchar(mix)
  p <- stan_llh_dpars(bterms, reqn, resp, mix)
  p$trials <- paste0("trials", resp, if (reqn) "[n]")
  lpdf <- stan_llh_simple_lpdf("binomial", "logit", bterms)
  sdist(lpdf, p$trials, p$mu)
}

stan_llh_bernoulli <- function(bterms, resp = "", mix = "") {
  reqn <- stan_llh_adj(bterms) || nzchar(mix)
  p <- stan_llh_dpars(bterms, reqn, resp, mix)
  lpdf <- stan_llh_simple_lpdf("bernoulli", "logit", bterms)
  sdist(lpdf, p$mu)
}

stan_llh_gamma <- function(bterms, resp = "", mix = "") {
  reqn <- stan_llh_adj(bterms) || nzchar(mix)
  p <- stan_llh_dpars(bterms, reqn, resp, mix)
  sdist("gamma", p$shape, p$mu)
}

stan_llh_exponential <- function(bterms, resp = "", mix = "") {
  reqn <- stan_llh_adj(bterms) || nzchar(mix)
  p <- stan_llh_dpars(bterms, reqn, resp, mix)
  sdist("exponential", p$mu)
}

stan_llh_weibull <- function(bterms, resp = "", mix = "") {
  reqn <- stan_llh_adj(bterms) || nzchar(mix)
  p <- stan_llh_dpars(bterms, reqn, resp, mix)
  sdist("weibull", p$shape, p$mu)
}

stan_llh_frechet <- function(bterms, resp = "", mix = "") {
  reqn <- stan_llh_adj(bterms) || nzchar(mix)
  p <- stan_llh_dpars(bterms, reqn, resp, mix)
  sdist("frechet", p$nu, p$mu)
}

stan_llh_gen_extreme_value <- function(bterms, resp = "", mix = "") {
  p <- stan_llh_dpars(bterms, TRUE, resp, mix)
  sdist("gen_extreme_value", p$mu, p$sigma, p$xi)
}

stan_llh_exgaussian <- function(bterms, resp = "", mix = "") {
  reqn <- stan_llh_adj(bterms) || nzchar(mix)
  p <- stan_llh_dpars(bterms, reqn, resp, mix)
  sdist(
    "exp_mod_normal", paste0(p$mu, " - ", p$beta), 
    p$sigma, paste0("inv(", p$beta, ")")
  )
}

stan_llh_inverse.gaussian <- function(bterms, resp = "", mix = "") {
  reqn <- stan_llh_adj(bterms) || nzchar(mix)
  p <- stan_llh_dpars(bterms, reqn, resp, mix)
  lpdf <- paste0("inv_gaussian", if (!reqn) "_vector")
  n <- if (reqn) "[n]"
  sdist(lpdf, p$mu, p$shape)
}

stan_llh_wiener <- function(bterms, resp = "", mix = "") {
  p <- stan_llh_dpars(bterms, TRUE, resp, mix)
  p$dec <- paste0("dec", resp, "[n]")
  sdist("wiener_diffusion", p$dec, p$bs, p$ndt, p$bias, p$mu)
}

stan_llh_beta <- function(bterms, resp = "", mix = "") {
  reqn <- stan_llh_adj(bterms) || nzchar(mix) ||
    paste0("phi", mix) %in% names(bterms$dpars)
  p <- stan_llh_dpars(bterms, reqn, resp, mix)
  sdist("beta",
    paste0(p$mu, " * ", p$phi), 
    paste0("(1 - ", p$mu, ") * ", p$phi)
  )
}

stan_llh_von_mises <- function(bterms, resp = "", mix = "") {
  reqn <- stan_llh_adj(bterms) || nzchar(mix) ||
    "kappa" %in% names(bterms$dpars)
  p <- stan_llh_dpars(bterms, reqn, resp, mix)
  lpdf <- paste0("von_mises_", ifelse(reqn, "real", "vector"))
  sdist(lpdf, p$mu, p$kappa)
}

stan_llh_cumulative <- function(bterms, resp = "", mix = "") {
  simplify <- bterms$family$link == "logit" && 
    !"disc" %in% names(bterms$dpars) && !has_cs(bterms)
  if (simplify) {
    prefix <- paste0(resp, if (nzchar(mix)) paste0("_mu", mix))
    p <- stan_llh_dpars(bterms, TRUE, resp, mix)
    p$ord_intercept <- paste0("temp", prefix, "_Intercept")
    out <- sdist("ordered_logistic", p$mu, p$ord_intercept)
  } else {
    out <- stan_llh_ordinal(bterms, resp, mix)
  }
  out
}

stan_llh_sratio <- function(bterms, resp = "", mix = "") {
  stan_llh_ordinal(bterms, resp, mix)
}

stan_llh_cratio <- function(bterms, resp = "", mix = "") {
  stan_llh_ordinal(bterms, resp, mix)
}

stan_llh_acat <- function(bterms, resp = "", mix = "") {
  stan_llh_ordinal(bterms, resp, mix)
}

stan_llh_categorical <- function(bterms, resp = "", mix = "") {
  stopifnot(bterms$family$link == "logit")
  p <- stan_llh_dpars(bterms, TRUE, resp, mix, dpars = "mu")
  sdist("categorical_logit", p$mu)
}

stan_llh_ordinal <- function(bterms, resp = "", mix = "") {
  # helper function for ordinal families
  has_cs <- has_cs(bterms)
  prefix <- paste0(resp, if (nzchar(mix)) paste0("_mu", mix))
  p <- stan_llh_dpars(bterms, TRUE, resp, mix)
  p$ord_intercept <- paste0("temp", prefix, "_Intercept")
  p$cs <- if (has_cs) paste0("mucs", prefix, "[n]")
  lpdf <- bterms$family$family
  lpdf <- paste0(lpdf, "_", bterms$family$link, if (has_cs) "_cs")
  sdist(lpdf, p$mu, p$cs, p$ord_intercept, p$disc)
}

stan_llh_hurdle_poisson <- function(bterms, resp = "", mix = "") {
  p <- stan_llh_dpars(bterms, TRUE, resp, mix)
  lpdf <- stan_llh_simple_lpdf("hurdle_poisson", "log", bterms)
  lpdf <- paste0(lpdf, stan_llh_dpar_usc_logit("hu", bterms))
  sdist(lpdf, p$mu, p$hu)
}

stan_llh_hurdle_negbinomial <- function(bterms, resp = "", mix = "") {
  p <- stan_llh_dpars(bterms, TRUE, resp, mix)
  lpdf <- stan_llh_simple_lpdf("hurdle_neg_binomial", "log", bterms)
  lpdf <- paste0(lpdf, stan_llh_dpar_usc_logit("hu", bterms))
  sdist(lpdf, p$mu, p$shape, p$hu)
}

stan_llh_hurdle_gamma <- function(bterms, resp = "", mix = "") {
  p <- stan_llh_dpars(bterms, TRUE, resp, mix)
  usc_logit <- stan_llh_dpar_usc_logit("hu", bterms)
  lpdf <- paste0("hurdle_gamma", usc_logit)
  sdist(lpdf, p$shape, p$mu, p$hu)
}

stan_llh_hurdle_lognormal <- function(bterms, resp = "", mix = "") {
  p <- stan_llh_dpars(bterms, TRUE, resp, mix)
  usc_logit <- stan_llh_dpar_usc_logit("hu", bterms)
  lpdf <- paste0("hurdle_lognormal", usc_logit)
  sdist(lpdf, p$mu, p$sigma, p$hu)
}

stan_llh_zero_inflated_poisson <- function(bterms, resp = "", mix = "") {
  p <- stan_llh_dpars(bterms, TRUE, resp, mix)
  lpdf <- stan_llh_simple_lpdf("zero_inflated_poisson", "log", bterms)
  lpdf <- paste0(lpdf, stan_llh_dpar_usc_logit("zi", bterms))
  sdist(lpdf, p$mu, p$zi)
}

stan_llh_zero_inflated_negbinomial <- function(bterms, resp = "", mix = "") {
  p <- stan_llh_dpars(bterms, TRUE, resp, mix)
  lpdf <- stan_llh_simple_lpdf("zero_inflated_neg_binomial", "log", bterms)
  lpdf <- paste0(lpdf, stan_llh_dpar_usc_logit("zi", bterms))
  sdist(lpdf, p$mu, p$shape, p$zi)
}

stan_llh_zero_inflated_binomial <- function(bterms, resp = "", mix = "") {
  p <- stan_llh_dpars(bterms, TRUE, resp, mix)
  p$trials <- "trials[n]"
  lpdf <- "zero_inflated_binomial"
  lpdf <- stan_llh_simple_lpdf(lpdf, "logit", bterms, sep = "_b")
  lpdf <- paste0(lpdf, stan_llh_dpar_usc_logit("zi", bterms))
  sdist(lpdf, p$trials, p$mu, p$zi)
}

stan_llh_zero_inflated_beta <- function(bterms, resp = "", mix = "") {
  p <- stan_llh_dpars(bterms, TRUE, resp, mix)
  usc_logit <- stan_llh_dpar_usc_logit("zi", bterms)
  lpdf <- paste0("zero_inflated_beta", usc_logit)
  sdist(lpdf, p$mu, p$phi, p$zi)
}

stan_llh_zero_one_inflated_beta <- function(bterms, resp = "", mix = "") {
  p <- stan_llh_dpars(bterms, TRUE, resp, mix)
  sdist("zero_one_inflated_beta", p$mu, p$phi, p$zoi, p$coi)
}

stan_llh_custom <- function(bterms, resp = "", mix = "") {
  p <- stan_llh_dpars(bterms, TRUE, resp, mix)
  family <- bterms$family
  sdist(family$name, p[family$dpars], family$vars)
}

sdist <- function(dist, ..., shift = NULL) {
  # prepare distribution and arguments for use in Stan
  args <- sargs(...)
  structure(nlist(dist, args, shift), class = "sdist")
}

sargs <- function(...) {
  # prepare arguments for Stan likelihood statements
  paste0(c(...), collapse = ", ")
}

is.sdist <- function(x) {
  inherits(x, "sdist")
}

tp <- function(wsp = 2) {
  wsp <- collapse(rep(" ", wsp))
  paste0(wsp, "target += ")
}
