% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/design.r
\name{design}
\alias{design}
\title{General block and treatment designs.}
\usage{
design(treatments, blocks = NULL, treatments_model = NULL,
  weighting = 0.5, searches = NULL, seed = NULL, jumps = 1)
}
\arguments{
\item{treatments}{a single treatment factor or a data frame containing multiple treatment factors. 
Treatment factors can be qualitative or quantitative (numeric) level factors.}

\item{blocks}{a single blocks factor or a data frame containing multiple block factors in the
required order of fitting. Block factors must be qualitative level factors.}

\item{treatments_model}{a model formula for the required treatments design. The default 
is a fully crossed factorial model.}

\item{weighting}{a weighting factor between 0 and 1 for general crossed block designs. 
The default weighting is 0.5.}

\item{searches}{the maximum number of local optima searched at each stage of a treatment 
and block design optimization. The default depends on the design size.}

\item{seed}{an integer initializing the random number generator. The default is a positive 
integer seed chosen randomly.}

\item{jumps}{the number of pairwise random treatment swaps used to escape a local maxima. 
The default is a single swap.}
}
\value{
\item{design}{The design layout showing the allocation of treatment and block design 
factors to individual plots.}
\item{treatments_model}{The fractional size of the treatment design together with the 
D-efficiency of that fraction.}
\item{blocks_model}{The blocks sub-model design together with the D-efficiency factor 
of each successively fitted blocks sub-model.}
\item{seed}{Numerical seed for random number generator.}
\item{searches}{Maximum number of searches in each stratum.}
\item{jumps}{Number of random treatment swaps to escape a local maxima.}
}
\description{
Constructs D-optimal block and treatment designs for feasible combinations of nested or crossed block 
factors and feasible linear treatment models.
}
\details{
\code{treatments} is a treatments factor or a data frame for two or more columns of 
treatment factors where the factors can be qualitative or quantitative. 

\code{blocks} is a blocks factor or a data frame for two or more columns of qualitative block factors
where the default is a single level factor of length equal to the length of the \code{treatments} factors. 

\code{treatments_model} is a design formula for the \code{treatments} factors based on the
\code{models} formula of the \code{\link[stats]{model.matrix}} package. The default assumes
 a complete factorial design.

The total number of design plots is defined by the length of the \code{blocks} factors, if present, 
otherwise by the length of the \code{treatments} factors. 

The treatments are replicated in the ratio of the total number of design plots to the length
of the \code{treatments} factors. The integer part of the ratio, possibly zero, defines the number of
complete replications of the \code{treatments} factors while the fractional part of the ratio, if any, 
defines a sample fraction of that size drawn from the set of \code{treatments} factor combinations. 
Samples are drawn without replacement and are chosen to optimize the D-optimality of the \code{treatments_model}. 

The \code{design} algorithm builds the blocks design by sequentially adding \code{blocks} factors
in the column order of the \code{blocks} data frame. Each block factor is optimized  
conditional on all preceding block factors but ignoring all succeeding block factors.
This method of optimization allows the blocking factors to be fitted in order of importance with the 
largest and most important blocks fitted first and the smaller and less important blocks fitted subsequently. 

For fully nested block designs, the block effects are optimized recursively assuming a simple additive block effects model.
For general crossed block designs, the crossed blocks intersections may contain blocks of two or more plots and 
then both the main effects and the intersection blocks must be optimized simultaneously.
For this situation, the \code{design} algorithm 
provides an option for optimizing a weighted combination of the
additive crossed blocks information matrix and the multiplicative crossed blocks information matrix. If the \code{weighting} 
factor is zero, the design is a fully additive crossed blocks model, if the \code{weighting} factor is one the design is a 
fully multiplicative crossed blocks model while for any intermediate \code{weighting}, the design is a
compromise between a fully additive and a fully multiplicative crossed blocks model.
 
The \code{blocks_model} output shows the overall achieved D-efficiency for each sequentially fitted blocks models. 
For nested blocks, the \code{blocks_model} output shows the efficiency factors for each successively nested
blocks design whereas for crossed blocks the \code{blocks_model} output shows the efficiency 
factors for both the additive and, where available, for the multiplicative effects of each sequentially fitted crossed 
blocks design. Comparison of the efficiency factors of the weighted crossed block designs using different weightings 
will provide guidance on the best choice of weighting for an efficient design. 

The efficiency factor used here is the ratio of the generalized variance of the full treatment design
relative to the generalized variance of the optimized block and 
treatment design. Using this definition, it is possible for quantitative level treatment designs to have efficiency 
factors greater than one therefore the main use of efficiency factors is to compare different optimizations
of the same design.  

  
Outputs:

The principle design outputs comprise:
\itemize{
 \item  A data frame showing a randomized allocation of treatments to blocks. \cr
 \item  A table showing the fractional size of the treatment design and the 
 D-efficiency factor of that fraction. \cr
 \item  A table showing the blocks sub-model design and the D-efficiency factor 
 of each successively fitted blocks sub-model. \cr
}
}
\examples{

## For optimum results, the number of searches may need to be increased in practice.
## Designs should be rebuilt repeatedly to check that a near-optimum design has been found.  

## 4 replicates of 12 treatments with 16 nested blocks of size 3
treatments = factor(1:12)
Blocks = factor(rep(1:4,each=12))
subBlocks = factor(rep(1:16,each=3))
blocks = data.frame(Blocks,subBlocks)
design(treatments,blocks)$blocks_model

## 4 x 12 design for 4 replicates of 12 treatments with 16 nested blocks of size 3
## only the intermediate weighting will give an optimal Trojan design 
treatments = factor(1:12)
MainCols = factor(rep(rep(1:4,each=3),4))
MainRows = factor(rep(1:4,each=12))
Columns = factor(rep(1:12,4))
blocks = data.frame(MainCols,MainRows,Columns)
\donttest{design(treatments,blocks,searches=100,weighting=0)$blocks_model
design(treatments,blocks,searches=100,weighting=0.5)$blocks_model
design(treatments,blocks,searches=100,weighting=1)$blocks_model}
 
## 4 x 13 Row-and-column design for 4 replicates of 13 treatments 
## Youden design Plan 13.5 Cochran and Cox (1957).
treatments=factor(1:13)
Rows =factor(rep(1:4,each=13))
Cols =factor(rep(1:13,4))
blocks =data.frame(Rows,Cols)
\donttest{design(treatments,blocks,searches=500)}

## Two replicates of 272 treatments in a 16 x 34 design with nested row blocks
treatments=factor(1:272)
Reps = factor(rep(1:2,each=272))
Rows = factor(rep(1:16,each=34))
MainCols = factor(rep(rep(1:4,c(9,8,8,9)),16))
SubCols = factor(rep(1:34,16))
blocks = data.frame(Reps,Rows,MainCols,SubCols)
\donttest{design(treatments,blocks,searches=1)$blocks_model}

## differential replication including single replicate treatments
treatments=factor(c(rep(1:12,2), rep(13:24,1)))
Main=factor(rep(1:2,each=18))
Sub =factor(rep(1:6,each=6))
blocks =data.frame(Main,Sub)
design(treatments,blocks,searches=5)


## Factorial treatment designs defined by a treatments data frame and a factorial model equation.

## Main effects of five 2-level factors in a half-fraction of a 2/2/2 nested blocks design
treatments = expand.grid(F1=factor(1:2),F2=factor(1:2),F3=factor(1:2),F4=factor(1:2),F5=factor(1:2))
blocks=data.frame(b1=factor(rep(1:2,each=8)),b2=factor(rep(1:4,each=4)),b3=factor(rep(1:8,each=2)))
treatments_model="F1 + F2 + F3 + F4 + F5"
design(treatments,blocks,treatments_model,searches=5)

# Second-order model for five qualitative 2-level factors in 4 randomized blocks
treatments=expand.grid(F1=factor(1:2),F2=factor(1:2),F3=factor(1:2),F4=factor(1:2),F5=factor(1:2))
blocks=factor(rep(1:4,each=8))
treatments_model="(F1+F2+F3+F4+F5)^2"
design(treatments,blocks,treatments_model,searches=5)

# Main effects of five 2-level factors in a half-fraction of a 4 x 4 row-and column design.
treatments = expand.grid(F1=factor(1:2),F2=factor(1:2),F3=factor(1:2),F4=factor(1:2),
F5=factor(1:2))
blocks=data.frame( rows=factor(rep(1:4,each=4)), cols=factor(rep(1:4,4)))
treatments_model="~ F1+F2+F3+F4+F5"
design(treatments,blocks,treatments_model,searches=20)

# Quadratic regression for one 6-level numeric factor in 2 randomized
#  blocks assuming 10/6 fraction
treatments=expand.grid(X=1:6)
blocks=factor(rep(1:2,each=5))
treatments_model=" ~ poly(X,2)"
design(treatments,blocks,treatments_model,searches=5) 

# First-order model for 1/3rd fraction of four qualitative 3-level factors in 3  blocks
treatments=expand.grid(F1=factor(1:3),F2=factor(1:3),F3=factor(1:3),F4=factor(1:3))
blocks=factor(rep(1:3,each=9))
treatments_model=" ~ F1+F2+F3+F4"
design(treatments,blocks,treatments_model,searches=5)

# Second-order model for a 1/3rd fraction of five qualitative 3-level factors in 3 blocks
treatments=expand.grid( F1=factor(1:3), F2=factor(1:3), F3=factor(1:3), F4=factor(1:3), 
F5=factor(1:3))
blocks=factor(rep(1:3,each=27))
treatments_model=" ~ (F1+F2+F3+F4+F5)^2"
 \donttest{design(treatments,blocks,treatments_model,searches=100)}

# Second-order model for two qualitative and two quantitative level factors in 4 randomized blocks
treatments=expand.grid(F1=factor(1:2),F2=factor(1:3),V1=1:3,V2=1:4)
blocks=factor(rep(1:4,each=18))
treatments_model = " ~ F1 + F2 + poly(V1,2) +  poly(V2,2) + (poly(V1,1)+F1+F2):(poly(V2,1)+F1+F2) "
 \donttest{design(treatments,blocks,treatments_model,searches=5)}
 
# Plackett and Burman design for eleven 2-level factors in 12 runs (needs large number of searches)
GF=expand.grid(F1=factor(1:2),F2=factor(1:2),F3=factor(1:2),F4=factor(1:2),F5=factor(1:2),
F6=factor(1:2),F7=factor(1:2),F8=factor(1:2),F9=factor(1:2),F10=factor(1:2),F11=factor(1:2))
blocks=factor(rep(1,12))
model=model="~ F1+F2+F3+F4+F5+F6+F7+F8+F9+F10+F11"
\donttest{design(GF,blocks,model)}

}
\references{
Cochran W.G & Cox G.M (1957) Experimental Designs 2nd Edition John Wiley & sons
 
Edmondson R. N. (1998). Trojan square and incomplete Trojan square designs for crop research. 
Journal of Agricultural Science, Cambridge, 131, pp.135-142
}
