% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BIOMOD_Tuning.R
\name{BIOMOD_Tuning}
\alias{BIOMOD_Tuning}
\title{Tune models parameters}
\usage{
BIOMOD_Tuning(
  bm.format,
  bm.options = BIOMOD_ModelingOptions(),
  models = c("GLM", "GBM", "GAM", "CTA", "ANN", "SRE", "FDA", "MARS", "RF",
    "MAXENT.Phillips"),
  metric.eval = "ROC",
  ctrl.train = NULL,
  ctrl.train.tuneLength = 30,
  ctrl.ANN = NULL,
  ctrl.CTA = NULL,
  ctrl.FDA = NULL,
  ctrl.GAM = NULL,
  ctrl.GBM = NULL,
  ctrl.GLM = NULL,
  ctrl.MARS = NULL,
  ctrl.RF = NULL,
  ANN.method = "avNNet",
  ANN.decay.tune = c(0.001, 0.01, 0.05, 0.1),
  ANN.size.tune = c(2, 4, 6, 8),
  ANN.maxit = 500,
  ANN.MaxNWts = 10 * (ncol(bm.format@data.env.var) + 1) + 10 + 1,
  MARS.method = "earth",
  GAM.method = "gam",
  GLM.method = "glmStepAIC",
  GLM.type = c("simple", "quadratic", "polynomial", "s_smoother"),
  GLM.interaction = c(0, 1),
  ME.cvmethod = "randomkfold",
  ME.overlap = FALSE,
  ME.kfolds = 10,
  ME.n.bg = 10000,
  ME.env = NULL,
  ME.metric = "ROC",
  ME.clamp = TRUE,
  ME.parallel = FALSE,
  ME.numCores = NULL,
  RF.method = "rf",
  weights = NULL
)
}
\arguments{
\item{bm.format}{a \code{\link{BIOMOD.formated.data}} or \code{\link{BIOMOD.formated.data.PA}} 
object returned by the \code{\link{BIOMOD_FormatingData}} function}

\item{bm.options}{a \code{\link{BIOMOD.models.options}} object returned by the
\code{\link{BIOMOD_ModelingOptions}} function}

\item{models}{a \code{vector} containing model names to be tuned, \cr 
must be among \code{GLM}, \code{GBM}, \code{GAM}, \code{CTA}, \code{ANN}, \code{SRE}, 
\code{FDA}, \code{MARS}, \code{RF}, \code{MAXENT.Phillips}}

\item{metric.eval}{a \code{character} corresponding to the evaluation metric used to select 
optimal models and tune parameters, must be either \code{ROC} or \code{TSS} 
(\emph{maximizing Sensitivity and Specificity})}

\item{ctrl.train}{global control parameters that can be obtained from the 
\code{\link[caret]{trainControl}} function}

\item{ctrl.train.tuneLength}{(see \code{tuneLength} parameter in \code{\link[caret]{train}})}

\item{ctrl.ANN}{control parameters for \code{ANN}}

\item{ctrl.CTA}{control parameters for \code{CTA}}

\item{ctrl.FDA}{control parameters for \code{FDA}}

\item{ctrl.GAM}{control parameters for \code{GAM}}

\item{ctrl.GBM}{control parameters for \code{GBM}}

\item{ctrl.GLM}{control parameters for \code{GLM}}

\item{ctrl.MARS}{control parameters for \code{MARS}}

\item{ctrl.RF}{control parameters for \code{RF}}

\item{ANN.method}{a \code{character} corresponding to the classification or regression model 
to use for \code{ANN}, \cr 
must be \code{avNNet} (see \url{http://topepo.github.io/caret/train-models-by-tag.html#Neural_Network})}

\item{ANN.decay.tune}{a \code{vector} of weight decay parameters for \code{ANN}}

\item{ANN.size.tune}{a \code{vector} of size parameters (number of units in the hidden layer) 
for \code{ANN}}

\item{ANN.maxit}{an \code{integer} corresponding to the maximum number of iterations for 
\code{ANN}}

\item{ANN.MaxNWts}{an \code{integer} corresponding to the maximum allowable number of weights 
for \code{ANN}}

\item{MARS.method}{a \code{character} corresponding to the classification or regression model 
to use for \code{MARS}, \cr 
must be \code{earth} (see 
\url{http://topepo.github.io/caret/train-models-by-tag.html#Multivariate_Adaptive_Regression_Splines})}

\item{GAM.method}{a \code{character} corresponding to the classification or regression model 
to use for \code{GAM}, \cr 
must be \code{gam} (see \url{http://topepo.github.io/caret/train-models-by-tag.html#generalized-additive-model})}

\item{GLM.method}{a \code{character} corresponding to the classification or regression model 
to use for \code{GLM}, \cr 
must be \code{glmStepAIC} (see 
\url{http://topepo.github.io/caret/train-models-by-tag.html#Generalized_Linear_Model})}

\item{GLM.type}{a \code{vector} of \code{character} corresponding to modeling types for 
\code{GLM}, \cr must be among \code{simple}, \code{quadratic}, \code{polynomial}, 
\code{s_smoother}}

\item{GLM.interaction}{a \code{vector} of interaction types, must be among \code{0}, \code{1}}

\item{ME.cvmethod}{a \code{character} corresponding to the method used to partition data for 
\code{MAXENT.Phillips}, \cr must be \code{randomkfold}}

\item{ME.overlap}{(\emph{optional, default} \code{FALSE}) \cr 
A \code{logical} value defining whether to calculate pairwise metric of niche overlap or not 
(see \code{\link[ENMeval]{calc.niche.overlap}})}

\item{ME.kfolds}{an \code{integer} corresponding to the number of bins for k-fold 
cross-validation for \code{MAXENT.Phillips}}

\item{ME.n.bg}{an \code{integer} corresponding to the number of background points used to run 
\code{MAXENT.Phillips}}

\item{ME.env}{a \code{RasterStack} object containing model predictor variables}

\item{ME.metric}{a \code{character} corresponding to the evaluation metric used to select 
optimal model and tune parameters for \code{MAXENT.Phillips}, must be either 
\code{auc.val.avg}, \code{auc.diff.avg}, \code{or.mtp.avg}, \code{or.10p.avg} or \code{AICc}}

\item{ME.clamp}{(\emph{optional, default} \code{TRUE}) \cr 
A \code{logical} value defining whether \emph{Features are constrained to remain within the 
range of values in the training data} (Elith et al. 2011) or not}

\item{ME.parallel}{(\emph{optional, default} \code{TRUE}) \cr 
A \code{logical} value defining whether to enable parallel computing for 
\code{MAXENT.Phillips} or not}

\item{ME.numCores}{an \code{integer} corresponding to the number of cores to be used to 
train \code{MAXENT.Phillips}}

\item{RF.method}{a \code{character} corresponding to the classification or regression model 
to use for \code{RF}, \cr 
must be \code{rf} (see \url{http://topepo.github.io/caret/train-models-by-tag.html#random-forest})}

\item{weights}{a \code{vector} of \code{numeric} values corresponding to observation weights}
}
\value{
A \code{\link{BIOMOD.models.options}} object (see \code{\link{BIOMOD_ModelingOptions}}) with 
optimized parameters
}
\description{
Function to tune \pkg{biomod2} single models parameters
}
\details{
\itemize{
  \item \code{ctrl.train} parameter is set by default to : \cr
  \code{caret::trainControl(method = 'cv', summaryFunction = caret::twoClassSummary,} \cr
  \code{classProbs = TRUE, returnData = FALSE)}.
  \item All control parameters for other models are set to \code{ctrl.train} if unspecified.
  \item For more details on \code{MAXENT.Phillips} tuning, please refer to 
  \code{\link[ENMeval]{ENMevaluate}}.
  \item For more details on other models tuning, please refer to \code{\link[caret]{train}}.
}
}
\examples{

# Load species occurrences (6 species available)
myFile <- system.file('external/species/mammals_table.csv', package = 'biomod2')
DataSpecies <- read.csv(myFile, row.names = 1)
head(DataSpecies)

# Select the name of the studied species
myRespName <- 'GuloGulo'

# Get corresponding presence/absence data
myResp <- as.numeric(DataSpecies[, myRespName])

# Get corresponding XY coordinates
myRespXY <- DataSpecies[, c('X_WGS84', 'Y_WGS84')]

# Load environmental variables extracted from BIOCLIM (bio_3, bio_4, bio_7, bio_11 & bio_12)
myFiles <- paste0('external/bioclim/current/bio', c(3, 4, 7, 11, 12), '.grd')
myExpl <- raster::stack(system.file(myFiles, package = 'biomod2'))

\dontshow{
myExtent <- raster::extent(0,30,45,70)
myExpl <- raster::stack(raster::crop(myExpl, myExtent))
}

# ---------------------------------------------------------------
# Format Data with true absences
myBiomodData <- BIOMOD_FormatingData(resp.var = myResp,
                                     expl.var = myExpl,
                                     resp.xy = myRespXY,
                                     resp.name = myRespName)


# ---------------------------------------------------------------
### Duration for turing all models sequential with default settings 
### on 3.4 GHz processor: approx. 45 min tuning all models in parallel
### (on 8 cores) using foreach loops runs much faster: approx. 14 min

\dontrun{
# library(doParallel)
# cl <- makeCluster(8)
# doParallel::registerDoParallel(cl) 

time.seq <- system.time(
  bm.tuning <- BIOMOD_Tuning(bm.format = myBiomodData, ME.env = myExpl, ME.n.bg = ncell(myExpl))
)

# stopCluster(cl)

plot(bm.tuning$tune.CTA.rpart)
plot(bm.tuning$tune.CTA.rpart2)
plot(bm.tuning$tune.RF)
plot(bm.tuning$tune.ANN)
plot(bm.tuning$tune.MARS)
plot(bm.tuning$tune.FDA)
plot(bm.tuning$tune.GBM)
plot(bm.tuning$tune.GAM)

# Get tuned modeling options
myBiomodOptions <- bm.tuning$models.options
}


}
\references{
\itemize{
  \item Kuhn, Max. 2008. Building predictive models in R using the caret package. 
  \emph{Journal of Statistical Software} \bold{28}, 1-26.
  \item Kuhn, Max, and Kjell Johnson. 2013. Applied predictive modeling. New York: Springer.
  \item Muscarella, Robert, et al. 2014. ENMeval: An R package for conducting spatially 
  independent evaluations and estimating optimal model complexity for Maxent ecological 
  niche models. \emph{Methods in Ecology and Evolution}, \bold{5}, 1198-1205.
}
}
\seealso{
\code{\link[caret]{trainControl}}, \code{\link[caret]{train}}, 
\code{\link[ENMeval]{calc.niche.overlap}}, \code{\link[ENMeval]{ENMevaluate}}, 
\code{\link{BIOMOD_ModelingOptions}}, \code{\link{BIOMOD_Modeling}}

Other Main functions: 
\code{\link{BIOMOD_CrossValidation}()},
\code{\link{BIOMOD_EnsembleForecasting}()},
\code{\link{BIOMOD_EnsembleModeling}()},
\code{\link{BIOMOD_FormatingData}()},
\code{\link{BIOMOD_LoadModels}()},
\code{\link{BIOMOD_ModelingOptions}()},
\code{\link{BIOMOD_Modeling}()},
\code{\link{BIOMOD_PresenceOnly}()},
\code{\link{BIOMOD_Projection}()},
\code{\link{BIOMOD_RangeSize}()}
}
\author{
Frank Breiner
}
\concept{Main functions}
