% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BIOMOD_PresenceOnly.R
\name{BIOMOD_PresenceOnly}
\alias{BIOMOD_PresenceOnly}
\title{Evaluate models with presence-only metrics}
\usage{
BIOMOD_PresenceOnly(
  bm.mod = NULL,
  bm.em = NULL,
  bg.env = NULL,
  perc = 0.9,
  save.output = TRUE
)
}
\arguments{
\item{bm.mod}{a \code{\link{BIOMOD.models.out}} object returned by the 
\code{\link{BIOMOD_Modeling}} function}

\item{bm.em}{a \code{\link{BIOMOD.ensemble.models.out}} object returned by the 
\code{\link{BIOMOD_EnsembleModeling}} function}

\item{bg.env}{(\emph{optional, default} \code{NULL}) \cr 
A \code{matrix}, \code{data.frame} or \code{\link[raster:stack]{RasterStack}} object 
containing values of environmental variables extracted from the background (\emph{if 
presences are to be compared to background instead of absences or pseudo-absences selected 
for modeling})}

\item{perc}{a \code{numeric} between \code{0} and \code{1} corresponding to the percentage of 
correctly classified presences for Minimal Predicted Area (see 
\code{ecospat.mpa()} in \pkg{ecospat})}

\item{save.output}{(\emph{optional, default} \code{TRUE}) \cr 
A \code{logical} value defining whether the output is to be saved within the 
\code{.BIOMOD_DATA} folder or not}
}
\value{
A \code{data.frame} containing evaluation scores both for the evaluation metrics used in the 
\code{\link{BIOMOD_Modeling}} function and additional Boyce index and Minimal Predicted Area.
}
\description{
This function computes presence-only evaluation metrics (Boyce index and Minimal 
Predicted Area) for \code{\link{BIOMOD.models.out}} or \code{\link{BIOMOD.ensemble.models.out}} 
objects that can be obtained with the \code{\link{BIOMOD_Modeling}} or 
\code{\link{BIOMOD_EnsembleModeling}} functions.
}
\details{
\code{em.by} parameter of \code{\link{BIOMOD_EnsembleModeling}} must have been set to 
\code{PA_dataset+repet} in order to have an ensemble for each \emph{RUN} of the 
\code{NbRunEval} parameter of the \code{\link{BIOMOD_Modeling}} function for evaluation. 
\cr \cr

The Boyce index returns \code{NA} values for \code{SRE} models because it can not be 
calculated with binary predictions. \cr This is also the reason why some \code{NA} values 
might appear for \code{GLM} models if they do not converge.
}
\note{
In order to break dependency loop between packages \pkg{biomod2} and \pkg{ecospat}, 
code of \code{ecospat.boyce()} and \code{ecospat.mpa()} in \pkg{ecospat})
functions have been copied within this file from version 3.2.2 (august 2022).
}
\examples{

# Load species occurrences (6 species available)
myFile <- system.file('external/species/mammals_table.csv', package = 'biomod2')
DataSpecies <- read.csv(myFile, row.names = 1)
head(DataSpecies)

# Select the name of the studied species
myRespName <- 'GuloGulo'

# Get corresponding presence/absence data
myResp <- as.numeric(DataSpecies[, myRespName])

# Get corresponding XY coordinates
myRespXY <- DataSpecies[, c('X_WGS84', 'Y_WGS84')]

# Load environmental variables extracted from BIOCLIM (bio_3, bio_4, bio_7, bio_11 & bio_12)
myFiles <- paste0('external/bioclim/current/bio', c(3, 4, 7, 11, 12), '.grd')
myExpl <- raster::stack(system.file(myFiles, package = 'biomod2'))

\dontshow{
myExtent <- raster::extent(0,30,45,70)
myExpl <- raster::stack(raster::crop(myExpl, myExtent))
}

# ---------------------------------------------------------------
file.out <- paste0(myRespName, "/", myRespName, ".AllModels.models.out")
if (file.exists(file.out)) {
  myBiomodModelOut <- get(load(file.out))
} else {

  # Format Data with true absences
  myBiomodData <- BIOMOD_FormatingData(resp.var = myResp,
                                       expl.var = myExpl,
                                       resp.xy = myRespXY,
                                       resp.name = myRespName)

  # Create default modeling options
  myBiomodOptions <- BIOMOD_ModelingOptions()

  # Model single models
  myBiomodModelOut <- BIOMOD_Modeling(bm.format = myBiomodData,
                                      modeling.id = 'AllModels',
                                      models = c('RF', 'GLM'),
                                      bm.options = myBiomodOptions,
                                      nb.rep = 2,
                                      data.split.perc = 80,
                                      metric.eval = c('TSS','ROC'),
                                      var.import = 3,
                                      do.full.models = FALSE,
                                      seed.val = 42)
}


file.EM <- paste0(myRespName, "/", myRespName, ".AllModels.ensemble.models.out")
if (file.exists(file.EM)) {
  myBiomodEM <- get(load(file.EM))
} else {

  # Model ensemble models
  myBiomodEM <- BIOMOD_EnsembleModeling(bm.mod = myBiomodModelOut,
                                        models.chosen = 'all',
                                        em.by = 'all',
                                        metric.select = c('TSS'),
                                        metric.select.thresh = c(0.7),
                                        metric.eval = c('TSS', 'ROC'),
                                        var.import = 3,
                                        prob.mean = TRUE,
                                        prob.median = FALSE,
                                        prob.cv = FALSE,
                                        prob.ci = FALSE,
                                        prob.ci.alpha = 0.05,
                                        committee.averaging = TRUE,
                                        prob.mean.weight = FALSE,
                                        prob.mean.weight.decay = 'proportional',
                                        seed.val = 42)
}


# ---------------------------------------------------------------
# Evaluate models with Boyce index and MPA
myBiomodPO <- BIOMOD_PresenceOnly(bm.mod = myBiomodModelOut,
                                  bm.em = myBiomodEM)
myBiomodPO

# Evaluate models with Boyce index and MPA (using background data)
myBiomodPO <- BIOMOD_PresenceOnly(bm.mod = myBiomodModelOut,
                                  bm.em = myBiomodEM, 
                                  bg.env = myExpl)
myBiomodPO


}
\references{
\itemize{
  \item Engler, R., Guisan, A., and Rechsteiner L. 2004. An improved approach for predicting 
  the distribution of rare and endangered species from occurrence and pseudo-absence data. 
  \emph{Journal of Applied Ecology}, \bold{41(2)}, 263-274.
  \item Hirzel, A. H., Le Lay, G., Helfer, V., Randin, C., and Guisan, A. 2006. Evaluating 
  the ability of habitat suitability models to predict species presences. \emph{Ecological 
  Modelling}, \bold{199(2)}, 142-152.
}
}
\seealso{
\code{ecospat.boyce()} and \code{ecospat.mpa()} in \pkg{ecospat}, 
\code{\link{BIOMOD.models.out}}, \code{\link{BIOMOD_Modeling}}, 
\code{\link{BIOMOD.ensemble.models.out}}, \code{\link{BIOMOD_EnsembleModeling}}

Other Main functions: 
\code{\link{BIOMOD_CrossValidation}()},
\code{\link{BIOMOD_EnsembleForecasting}()},
\code{\link{BIOMOD_EnsembleModeling}()},
\code{\link{BIOMOD_FormatingData}()},
\code{\link{BIOMOD_LoadModels}()},
\code{\link{BIOMOD_ModelingOptions}()},
\code{\link{BIOMOD_Modeling}()},
\code{\link{BIOMOD_Projection}()},
\code{\link{BIOMOD_RangeSize}()},
\code{\link{BIOMOD_Tuning}()}
}
\author{
Frank Breiner, Maya Gueguen
}
\concept{Main functions}
