% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/boundary_layer_conductance.r
\name{Gb.Choudhury}
\alias{Gb.Choudhury}
\title{Boundary Layer Conductance according to Choudhury & Monteith 1988}
\usage{
Gb.Choudhury(data, Tair = "Tair", pressure = "pressure", wind = "wind",
  ustar = "ustar", H = "H", leafwidth, LAI, zh, zr, d,
  stab_formulation = c("Dyer_1970", "Businger_1971"),
  constants = bigleaf.constants())
}
\arguments{
\item{data}{Data.frame or matrix containing all required variables}

\item{Tair}{Air temperature (degC)}

\item{pressure}{Atmospheric pressure (kPa)}

\item{wind}{Wind speed at sensor height (m s-1)}

\item{ustar}{Friction velocity (m s-1)}

\item{H}{Sensible heat flux (W m-2)}

\item{leafwidth}{Leaf width (m)}

\item{LAI}{One-sided leaf area index}

\item{zh}{Canopy height (m)}

\item{zr}{Instrument (reference) height (m)}

\item{d}{Zero-plane displacement height (-), can be calculated using \code{roughness.parameters}}

\item{stab_formulation}{Stability correction function used (If \code{stab_correction = TRUE}).
Either \code{"Dyer_1970"} or \code{"Businger_1971"}.}

\item{constants}{k - von-Karman constant (-) \cr
Rbwc - Ratio of the transfer efficiency through the boundary layer for water vapor and CO2 (-)}
}
\value{
A data frame with the following columns:
    \item{Rb}{Boundary layer resistance for heat and water (s m-1)}
    \item{Rb_CO2}{Boundary layer resistance for CO2 (s m-1)}
    \item{Gb}{Boundary layer conductance (m s-1)}
    \item{kB}{kB-1 parameter (-)}
}
\description{
A formulation for the canopy boundary layer conductance 
             to heat/water vapor according to Choudhury & Monteith 1988.
}
\details{
Boundary layer conductance according to Choudhury & Monteith 1988 is
         given by:
         
           \deqn{Gb = LAI((2a/\alpha)*sqrt(u(h)/w)*(1-exp(-\alpha/2)))}
         
         where u(zh) is the wind speed at the canopy surface, approximated from
         measured wind speed at sensor height zr and a wind extinction coefficient \eqn{\alpha}:
         
           \deqn{u(zh) = u(zr) / (exp(\alpha(zr/zh -1)))}.
         
         \eqn{\alpha} is modeled as an empirical relation to LAI (McNaughton & van den Hurk 1995):
         
           \deqn{\alpha = 4.39 - 3.97*exp(-0.258*LAI)}
         
         Rb for water vapor and heat is assumed to be equal. Rb for CO2 (Rb_CO2) is given as:
 
           \deqn{Rb_CO2 = 1.37 * Rb}
 
         The factor 1.37 arises due the lower molecular diffusivity of CO2 compared to water.
         It is lower than the ratio of the molecular diffusivities (Dw/DCO2 = 1.6), as movement
         across the boundary layer is assumed to be partly by diffusion and partly by turbulent
         mixing (Nobel 2005).
}
\examples{
## bulk canopy boundary layer resistance for a closed canopy (LAI=5) 
## with large leaves (leafwdith=0.1)            
df <- data.frame(Tair=25,pressure=100,wind=c(3,4,5),ustar=c(0.5,0.6,0.65),H=c(200,230,250))    
Gb.Choudhury(data=df,leafwidth=0.1,LAI=5,zh=25,d=17.5,zr=40)

## same conditions, but smaller leaves (leafwidth=0.01)
Gb.Choudhury(data=df,leafwidth=0.01,LAI=5,zh=25,d=17.5,zr=40) 

}
\references{
Choudhury, B. J., Monteith J.L., 1988: A four-layer model for the heat
            budget of homogeneous land surfaces. Q. J. R. Meteorol. Soc. 114, 373-398.
            
            McNaughton, K. G., Van den Hurk, B.J.J.M., 1995: A 'Lagrangian' revision of
            the resistors in the two-layer model for calculating the energy budget of a
            plant canopy. Boundary-Layer Meteorology 74, 261-288.
            
            Nobel, P. S., 2005: Physicochemical and Environmental Plant Physiology. Third 
            Edition. Elsevier Academic Press, Burlington, USA.
}
\seealso{
\code{\link{Gb.Thom}}, \code{\link{Gb.Su}}, \code{\link{aerodynamic.conductance}}
}
