#' Create an image plot for a factor variable.
#'
#' \code{legend.scale} plots a color gradient with an associated quantitive scale.  This is used to add the color scale in the \code{\link{autoimage}} function.
#' 
#' The length of the \code{col} vector indicates the number of partitions for the quantitative range.
#'
#' @param zlim A two-dimensional vector containing the minimum and maximum quantitative limits, respectively, or the color scale.
#' @param col A vector of colors used for the color scale.  Typically, this is a gradient of colors. The default is the 12 colors generated by \code{heat.colors(12)}.
#' @param horizontal A logical value indicating whether the legend should extend horizontally (\code{TRUE}) or vertically (\code{FALSE}).  The default is \code{TRUE}.
#' @param breaks The sequence of values defining the partition of \code{zlim}.  The length should be one more than then number of colors.  If not specified, then equidistant breaks are automatically used.
#' @param axis.args A list of named elements corresponding to the arguments of the \code{\link[graphics]{axis}} function.  This is used to modify the appearance of the scale of the legend.  See Examples.
#' @references The code for this function is derived from the internals of the \code{\link[fields]{image.plot}} function written by Doug Nychka and from the \code{image.scale} function written by Marc Taylor and discussed at \url{http://menugget.blogspot.com/2013/12/new-version-of-imagescale-function.html}.
#' @seealso \code{\link[graphics]{image}}, \code{\link[fields]{image.plot}}, \code{\link[graphics]{axis}}
#' @return NULL
#' @importFrom grDevices heat.colors
#' @importFrom utils head tail
#' @importFrom graphics axis image
#' @export
#' @examples
#' # default horizontal scale
#' legend.scale(c(0, 1))
#'
#' # default vertical scale
#' legend.scale(c(0, 1), horizontal = FALSE)
#'
#' # different color scheme with 24 colors
#' legend.scale(c(0, 1), col = cm.colors(24))
#'
#' # irregular color breaks
#' legend.scale(c(0, 1), col = heat.colors(4), breaks = c(0, 0.5, 0.75, 0.875, 1))
#'
#' # irregular color breaks with modified ticks and vertical orientation of labels
#' legend.scale(c(0, 1), col = heat.colors(4),
#'              breaks = c(0, 0.5, 0.75, 0.875, 1),
#'              axis.args = list(at = c(0, 0.5, 0.75, 0.875, 1), las = 2))
#'
#' # change size of axis labels
#' legend.scale(c(0, 1), axis.args = list(cex.axis = 2))
#'
#' # change color of axis labels and ticks
#' legend.scale(c(0, 1), axis.args = list(col.axis = "blue", col.ticks = "blue"))
#'
#' # log base 10 values with colors labeled on original scale
#' options(scipen = 2)
#' legend.scale(c(0, 6), col = heat.colors(6), axis.args = list(at = 0:6, labels = 10^(0:6), las = 2))
#'
legend.scale <- function(zlim, col = heat.colors(12), horizontal = TRUE, breaks, axis.args){

  if(missing(breaks)){
    breaks <- seq(zlim[1], zlim[2], length.out=(length(col)+1))
  }else{
    if(length(breaks) != length(col) + 1) stop("the length of breaks must be one more than length of col")
  }
  if(missing(axis.args)){
    axis.args = list(side = ifelse(horizontal, 1, 4),
                     las = ifelse(horizontal, 0, 2))
  }
  if(is.null(axis.args$side)) axis.args$side = ifelse(horizontal, 1, 4)
  if(is.null(axis.args$las)) axis.args$las = ifelse(horizontal, 0, 2)

  ncol <- length(col)

  # middle of breaks
  z = matrix((head(breaks, ncol) + tail(breaks, ncol))/2)

  if(horizontal){
    x = breaks
    y = 0:1
  }else{
    x = 0:1
    y = breaks
    z = t(z)
  }

  image(x, y, z, xaxt = "n", yaxt = "n", xlab = "",
        ylab = "", col = col, breaks = breaks)
  box()
  do.call("axis", axis.args)
}
