
#' Fit models over each of a set of independent variables with a response variable
#'
#' Fit and summarize models for each independent (x) variable with a response variable (y), with options to adjust by variables for each model.
#'
#' @param formula an object of class \code{\link{formula}}; a symbolic description of the variables to be modeled.  See "Details" for more information.
#' @param adjust an object of class \code{\link{formula}}, listing variables to adjust by in all models. Specify as a one-sided formula,
#'   like: \code{~Age+ Sex}.
#' @param family similar mechanism to \code{\link[stats]{glm}}, where the model to be fit is driven by the family.
#'   Options include: binomial, gaussian, survival, poisson, negbin, and ordinal. These can be passed as a string, as a function,
#'   or as a list resulting from a call to one of the functions. See \code{\link{modelsum.family}} for details on
#'   survival and ordinal families.
#' @param data an optional data.frame, list or environment (or object coercible by \code{\link[base]{as.data.frame}} to a data frame) containing the
#'   variables in the model. If not found in \code{data}, the variables are taken from \code{environment(formula)}, typically
#'   the environment from which \code{modelsum} is called.
#' @param subset an optional vector specifying a subset of observations (rows of \code{data}) to be used in the results.
#'   Works as vector of logicals or an index.
#' @param weights an optional vector specifying the weights to apply to each data observation (rows of \code{data})
#' @param na.action a function which indicates what should happen when the data contain \code{NA}s.
#'   The default (\code{NULL}) is to use the defaults of \code{\link[stats]{lm}}, \code{\link[stats]{glm}}, or \code{\link[survival]{coxph}},
#'   depending on the \code{family} specifications.
#' @param control control parameters to handle optional settings within \code{modelsum}.  Arguments for \code{modelsum.control}
#'   can be passed to \code{modelsum} via the \code{...} argument, but if a control object and \code{...} arguments are both supplied,
#'   the latter are used. See \code{\link{modelsum.control}} for other details.
#' @param ... additional arguments to be passed to internal \code{modelsum} functions.
#' @param x An object of class \code{'modelsum'}, or a list of such objects.
#' @return
#' An object with class \code{'modelsum'}, which is effectively a list with the variables from the right-side in x and the group variable in y.
#'   Then, each item in x has these:
#'   \item{fits}{a list with an item in X for each x in y ~ X + adjust variables}
#'   \item{family}{family used in glm}
#'   \item{Call}{Original call to modelsum}
#'   \item{control}{list of control parameters used in \code{modelsum}, and to be used in \code{\link{summary.modelsum}},
#'     the result of \code{\link{modelsum.control}}}
#' @author Jason Sinnwell, Patrick Votruba, Beth Atkinson, Gregory Dougherty, and Ethan Heinzen, adapted from SAS Macro of the same name
#' @seealso \code{\link{modelsum.control}}, \code{\link{summary.modelsum}}, \code{\link{formulize}}
#' @examples
#'
#' data(mockstudy)
#'
#' tab1 <- modelsum(bmi ~ sex + age, data = mockstudy)
#' summary(tab1, text = TRUE)
#'
#' tab2 <- modelsum(alk.phos ~ arm + ps + hgb, adjust = ~ age + sex,
#'                  family = "gaussian", data = mockstudy)
#' summary(tab2, text = TRUE)
#'
#' summary(tab2, show.intercept = FALSE, text = TRUE)
#'
#' tab2.df <- as.data.frame(tab2)
#'
#' tab2.df[1:5,]
#' @name modelsum
NULL
#> NULL

#' @rdname modelsum
#' @export

modelsum <- function(formula,  family="gaussian", data, adjust=NULL, na.action = NULL,
                     subset=NULL, weights=NULL, control = NULL, ...) {
  Call <- match.call()

  ## Allow family parameter to passed with or without quotes
  ## Here, we force quotes to simplify in for loop below
  if(is.function(family) || is.character(family))
  {
    family.list <- match.fun(family)()
    family <- family.list$family
  } else
  {
    family.list <- family
    family <- family$family
  }

  if(family %nin% c("survival", "gaussian", "binomial", "poisson", "quasibinomial", "quasipoisson", "ordinal", "negbin"))
    stop("Family ", family, " not supported.\n")

  if(family != "survival" && any(grepl("Surv\\(", formula))) {
    warning("Found Surv in formula, assuming family='survival'\n")
    family <- "survival"
  }
  ## pick up extra control arguments from command via ...
  control <- c(list(...), control)
  control <- do.call("modelsum.control", control[!duplicated(names(control))])

  ## Tell user if they passed an argument that was not expected, either here or in control
  expectArgs <- c("formula", "family", "adjust", "data", "na.action", "subset", "weights", "control", names(control))
  match.idx <- match(names(Call)[-1], expectArgs)
  if(anyNA(match.idx)) warning("Unused arguments: ", paste(names(Call)[c(FALSE, is.na(match.idx))], collapse=", "), "\n")

  #### Set up "main effects" dataset ####
  indx.main <- match(c("formula", "data", "subset"), names(Call), 0L)
  if(indx.main[1] == 0) stop("A formula argument is required")
  if(length(formula) == 2) stop("'formula' should have a response variable!")
  if(!is.null(adjust) && length(adjust) != 2) stop("'adjust' shouldn't have a response variable!")
  main.call <- Call[c(1, indx.main)]
  main.call[[1]] <- quote(stats::model.frame)
  main.call$na.action <- quote(stats::na.pass) # for now, keep all rows, except for what is subset out
  if(!missing(data))
  {
    # instead of call("keep.labels", ...), which breaks when arsenal isn't loaded (Can't find "keep.labels")
    main.call$data <- as.call(list(keep.labels, main.call$data))
  }
  maindf <- eval(main.call, parent.frame())
  if(nrow(maindf) == 0) stop("No (non-missing) observations")
  Terms <- stats::terms(maindf)

  #### Set up "adjustment" dataset ####
  if(missing(adjust))
  {
    adjustdf <- NULL
  } else
  {
    adj.call <- main.call
    adj.call$formula <- adjust
    adjustdf <- eval(adj.call, parent.frame())
  }

  effCols <- seq_along(maindf)[-1]
  yTerm <- colnames(maindf)[1]
  yLabel <- attributes(maindf[[1]])$label
  if(is.null(yLabel)) yLabel <- yTerm
  fitList <- list()

  is.numericish <- function(x) is.numeric(x) || is.Date(x)

  for(eff in effCols) {

    currCol <- maindf[[eff]]
    adj.formula <- join_formula(stats::drop.terms(Terms, if(length(effCols) > 1) setdiff(effCols, eff) - 1L else NULL, keep.response = TRUE), adjust)
    adj.Terms <- stats::terms(adj.formula)
    adjVars <- colnames(adjustdf)
    adjVars2 <- attr(adj.Terms, "term.labels")[-1]

    xname <- colnames(maindf)[eff]
    xname2 <- attr(adj.Terms, "term.labels")[1] # this one will have backticks for non-syntactic names
    labelEff <-  attributes(currCol)$label
    if(is.null(labelEff)) labelEff <- xname

    temp.call <- Call[c(1, match(c("data", "subset", "na.action", "weights"), names(Call), 0L))]
    temp.call$formula <- adj.formula

    ## placeholder for ordered, don't do any fitting
    ## y is ordered factor
    if (family == "ordinal") {
      temp.call[[1]] <- quote(MASS::polr)
      temp.call$Hess <- TRUE
      temp.call$method <- family.list$method
      fit <- eval(temp.call, parent.frame())
      coeffORTidy <- broom::tidy(fit, exponentiate=TRUE, conf.int=TRUE, conf.level=control$conf.level)
      coeffORTidy[coeffORTidy$coefficient_type == "zeta", names(coeffORTidy) %nin% c("term", "coefficient_type")] <- NA
      coeffTidy <- broom::tidy(fit, exponentiate=FALSE, conf.int=TRUE, conf.level=control$conf.level)
      coeffTidy$p.value <- 2*stats::pnorm(abs(coeffTidy$statistic), lower.tail = FALSE)
      coeffTidy <- cbind(coeffTidy, OR=coeffORTidy$estimate, CI.lower.OR=coeffORTidy$conf.low, CI.upper.OR=coeffORTidy$conf.high)
      # sort so that zeta comes first, but hold all else fixed
      coeffTidy <- coeffTidy[order(coeffTidy$coefficient_type == "coefficient", seq_len(nrow(coeffTidy))), ]
      modelGlance <- broom::glance(fit)

    } else if (family == "gaussian") {
      # ## issue warning if appears categorical
      if(length(unique(maindf[[1]])) <= 5) {
        warning("Input family=gaussian, but dependent variable has 5 or fewer categories\n")
      }
      temp.call[[1]] <- quote(stats::lm)
      temp.call$x <- TRUE
      fit <- eval(temp.call, parent.frame())
      coeffTidy <- broom::tidy(fit, conf.int=TRUE, conf.level=control$conf.level)

      if("(weights)" %in% colnames(fit$model)) fit$model <- fit$model[, colnames(fit$model) != "(weights)"]

      coeffTidy$standard.estimate <- lm.beta(fit)
      ## Continuous variable (numeric) ###############
      ## Note: Using tidy changes colname from 't value' to 'statistic'
      modelGlance <- broom::glance(fit)
      names(modelGlance)[names(modelGlance) == "p.value"] <- "p.value.F"


    } else if (family == "binomial" || family == "quasibinomial") {
      ## These families are used in glm

      temp.call[[1]] <- quote(stats::glm)
      temp.call$x <- TRUE
      temp.call$family <- family.list
      fit <- eval(temp.call, parent.frame())

      rocOut <- pROC::roc(fit$y ~ predict(fit, type='response'))
      #coeffbeta <- summary(fit)$coef
      ## find out that broom:::tidy.lm allows conf.int and exp
      coeffORTidy <- broom::tidy(fit, exponentiate=TRUE, conf.int=TRUE, conf.level=control$conf.level)
      coeffORTidy[coeffORTidy$term == "Intercept", -1] <- NA
      coeffTidy <- broom::tidy(fit, exponentiate=FALSE, conf.int=TRUE, conf.level=control$conf.level)
      coeffTidy <- cbind(coeffTidy, OR=coeffORTidy$estimate, CI.lower.OR=coeffORTidy$conf.low, CI.upper.OR=coeffORTidy$conf.high)
      modelGlance <- c(broom::glance(fit), concordance = pROC::auc(rocOut))

    } else if (family == "quasipoisson" || family == "poisson") {
      ## These families use glm

      temp.call[[1]] <- quote(stats::glm)
      temp.call$x <- TRUE
      temp.call$family <- family.list
      fit <- eval(temp.call, parent.frame())

      coeffRRTidy <- broom::tidy(fit, exponentiate=TRUE, conf.int=TRUE, conf.level=control$conf.level)
      coeffRRTidy[coeffRRTidy$term == "Intercept", -1] <- NA
      coeffTidy <- broom::tidy(fit, exponentiate=FALSE, conf.int=TRUE, conf.level=control$conf.level)
      coeffTidy <- cbind(coeffTidy, RR=coeffRRTidy$estimate, CI.lower.RR=coeffRRTidy$conf.low, CI.upper.RR=coeffRRTidy$conf.high)
      modelGlance <- broom::glance(fit)

    } else if (family == "negbin") {
      ## Also uses glm
      temp.call[[1]] <- quote(MASS::glm.nb)
      temp.call$x <- TRUE
      temp.call$link <- family.list$link
      fit <- eval(temp.call, parent.frame())

      coeffRRTidy <- broom::tidy(fit, exponentiate=TRUE, conf.int=TRUE, conf.level=control$conf.level)
      coeffRRTidy[coeffRRTidy$term == "Intercept", -1] <- NA
      coeffTidy <- broom::tidy(fit, exponentiate=FALSE, conf.int=TRUE, conf.level=control$conf.level)
      coeffTidy <- cbind(coeffTidy, RR=coeffRRTidy$estimate, CI.lower.RR=coeffRRTidy$conf.low, CI.upper.RR=coeffRRTidy$conf.high)
      modelGlance <- broom::glance(fit)
      modelGlance$theta <- fit$theta
      modelGlance$SE.theta <- fit$SE.theta

    } else if(family=="survival") {

      temp.call[[1]] <- quote(survival::coxph)
      fit <- eval(temp.call, parent.frame())

      ## use tidy to get both CIs, merge
      coeffHRTidy <- broom::tidy(fit, exponentiate=TRUE, conf.int=.95)
      coeffTidy <- broom::tidy(fit, exponentiate=FALSE, conf.int=.95)
      coeffTidy <- cbind(coeffTidy, HR=coeffHRTidy$estimate, CI.lower.HR=coeffHRTidy$conf.low, CI.upper.HR=coeffHRTidy$conf.high)
      modelGlance <-  broom::glance(fit)
    }

    names(coeffTidy)[names(coeffTidy) == "conf.low"] <- "CI.lower.estimate"
    names(coeffTidy)[names(coeffTidy) == "conf.high"] <- "CI.upper.estimate"

    if(!is.numericish(currCol))
    {
      lvls <- unique(currCol)
      findlvls <- if(identical(fit$contrasts[[xname]], "contr.treatment"))
      {
        paste0(xname2, lvls)
      } else if(identical(fit$contrasts[[xname]], "contr.poly"))
      {
        paste0(xname2, c(".L", ".Q", ".C", paste0("^", seq_along(lvls))))
      } else paste0(xname2, seq_along(lvls))

      xterms <- coeffTidy$term[coeffTidy$term %in% findlvls]
      labelEff <- sub(xname2, paste0(labelEff, " "), xterms, fixed = TRUE)
    } else xterms <- xname2


    adjterms <- adjlabels <- NULL
    for(adj in seq_along(adjustdf)) { ## manage adj terms and labels
      adjcol <- adjustdf[[adj]]
      alabel <- attributes(adjcol)$label
      if(is.null(alabel)) alabel <- adjVars[adj]

      if(!is.numericish(adjcol))
      {
        lvls.a <- unique(adjcol)
        findlvls.a <- if(identical(fit$contrasts[[adjVars[adj]]], "contr.treatment"))
        {
          paste0(adjVars2[adj], lvls.a)
        } else if(identical(fit$contrasts[[adjVars[adj]]], "contr.poly"))
        {
          paste0(adjVars2[adj], c(".L", ".Q", ".C", paste0("^", seq_along(lvls.a))))
        } else paste0(adjVars2[adj], seq_along(lvls.a))

        aterm <- coeffTidy$term[coeffTidy$term %in% findlvls.a]
        alabel <- sub(adjVars2[adj], paste0(alabel, " "), aterm, fixed = TRUE)
      } else aterm <- adjVars2[adj]

      adjterms <- c(adjterms, aterm)
      adjlabels <- c(adjlabels, alabel)
    }

    fitList[[xname]] <- list(coeff=coeffTidy, family=family,
                             xterms=xterms, label=labelEff,
                             adjterms=adjterms, adjlabels=adjlabels)

    ## put xname and endpoint in glance, summary and as.data.frame to pull from there
    fitList[[xname]]$glance <- c(modelGlance,
                                 N = sum(!is.na(currCol)),
                                 Nmiss = sum(is.na(currCol)),
                                 Nmiss2 = sum(is.na(currCol)),
                                 endpoint=yTerm,
                                 endlabel=yLabel,
                                 x=xname,
                                 contrasts=list(fit$contrasts))

  } # end for: eff

  msList <- list(fits=fitList, control = control, Call = Call, family=family)
                 ##, adjust=adjVars)
  class(msList) <- "modelsum"
  return(msList)
}


#' @rdname modelsum
#' @export
print.modelsum <- function(x, ...) {
  cat("Modelsum S3 Object\n\n")
  cat("Function Call: \n")
  print(x$Call)
  cat("\n")
  cat("y variable:\n")
  print(x$fits[[1]]$glance$endpoint)
  cat("x variables:\n")
  print(unname(vapply(x$fits, function(tmp) tmp$glance$x, "")))
  invisible(x)
}

#' @rdname modelsum
#' @export
print.modelsumList <- function(x, ...) {
  lapply(x, print.modelsum, ...)
  invisible(x)
}
