% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/setSimulationPath.R
\name{setSimulationPath}
\alias{setSimulationPath}
\title{Set Path to an Antares simulation}
\usage{
setSimulationPath(path, simulation = NULL)
}
\arguments{
\item{path}{(optional) 
Path to the simulation. It can either be the path to a directory containing
an antares project or directly to the directory containing the output of a
simulation.  If missing, a window opens and lets the user choose the
directory of the simulation interactively. Can also choose .h5 file, if \code{rhdf5} is installed.}

\item{simulation}{(optional) Only used if "path" represents the path of a study and not of the
output of a simulation. It can be either the name of the simulation or a
number indicating which simulation to use. It is possible to use negative 
values to select a simulation from the last one: for instance -1 will
select the most recent simulation, -2 will the penultimate one, etc. There 
are two special values 0 and "input" that tells the function that the user
is not interested by the results of any simulation, but only by the inputs.
In such a case, the function \code{\link{readAntares}} is unavailable.}
}
\value{
A list containing various information about the simulation, in particular:
  \item{studyPath}{path of the Antares study}
  \item{simPath}{path of the simulation}
  \item{inputPath}{path of the input folder of the study}
  \item{studyName}{Name of the study}
  \item{simDataPath}{path of the folder containing the data of the simulation}
  \item{name}{name of the simulation}
  \item{mode}{type of simulation: economy, adequacy, draft or input}
  \item{synthesis}{Are synthetic results available ?}
  \item{yearByYear}{Are the results for each Monte Carlo simulation available ?}
  \item{scenarios}{Are the Monte-Carlo scenarii stored in output ? This is 
    important to reconstruct some input time series that have been used in 
    each Monte-Carlo simulation.}
  \item{mcYears}{Vector containing the number of the exported Monte-Carlo scenarios}
  \item{antaresVersion}{Version of Antares used to run the simulation.}
  \item{areaList}{Vector of the available areas.}
  \item{districtList}{Vector of the available districts.}
  \item{linkList}{Vector of the available links.}
  \item{areasWithClusters}{Vector of areas containing clusters.}
  \item{variables}{Available variables for areas, districts and links.}
  \item{parameters}{Other parameters of the simulation.}
  \item{timeIdMin}{
    Minimum time id of the simulation. It is generally equal to one but can
    be higher if working on a subperiod.
  }
  \item{timeIdMax}{maximum time id of the simulation.}
  \item{start}{
    Date of the first day of the year in the simulation. This date corresponds
    to timeId = 1.
  }
  \item{firstWeekday}{First day of the week.}
  \item{districtsDef}{data.table containing the specification of the districts.}
  \item{energyCosts}{list containing the cost of spilled and unsupplied energy.}
}
\description{
This function has to be used before the \code{read} functions. It sets the path to
the Antares simulation to work on and other useful options (list of areas,
links, areas with clusters, variables, etc.).
}
\details{
The simulation chosen with \code{setSimulationPath} becomes the default
simulation for all functions of the package. This behavior is fine when
working on only one simulation, but it may become problematic when working
on multiple simulations at same time.

In such case, you can store the object returned by the function in a variable
and pass this variable to the functions of the package (see examples).
}
\examples{

\dontrun{
# Select interactively a study. It only works on windows.

setSimulationPath()

# Specify path of the study. Note: if there are more than one simulation
# output in the study, the function will asks the user to interactively choose
# one simulation.

setSimulationPath("path_of_the_folder_of_the_study")

# Select the first simulation of a study

setSimulationPath("path_of_the_folder_of_the_study", 1)

# Select the last simulation of a study

setSimulationPath("path_of_the_folder_of_the_study", -1)

# Select a simulation by name

setSimulationPath("path_of_the_folder_of_the_study", "name of the simulation")

# Just need to read input data

setSimulationPath("path_of_the_folder_of_the_study", "input")
# or
setSimulationPath("path_of_the_folder_of_the_study", 0)



# WORKING WITH MULTIPLE SIMULATIONS
#----------------------------------
# Let us assume ten simulations have been run and we want to collect the
# variable "LOAD" for each area. We can create a list containing options
# for each simulation and iterate through this list.

opts <- lapply(1:10, function(i) {
   setSimulationPath("path_of_the_folder_of_the_study", i)
})

output <- lapply(opts, function(o) {
  res <- readAntares(areas = "all", select = "LOAD", timeStep = "monthly", opts = o)
  # Add a column "simulation" containing the name of the simulation
  res$simulation <- o$name
  res
})

# Concatenate all the tables in one super table
output <- rbindlist(output)

# Reshape output for easier comparisons: one line per timeId and one column
# per simulation
output <- dcast(output, timeId + areaId ~ simulation, value.var = "LOAD")

output

# Quick visualization
matplot(output[area == area[1], !c("area", "timeId"), with = FALSE], 
        type = "l")
}

}
\seealso{
\code{\link{simOptions}}, \code{\link{readAntares}}, \code{\link{readLayout}}, 
  \code{\link{readClusterDesc}}, \code{\link{readBindingConstraints}}
}
