% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/time_decompose.R
\name{time_decompose}
\alias{time_decompose}
\title{Decompose a time series in preparation for anomaly detection}
\usage{
time_decompose(data, target, method = c("stl", "twitter"),
  frequency = "auto", trend = "auto", ..., merge = FALSE,
  message = TRUE)
}
\arguments{
\item{data}{A \code{tibble} or \code{tbl_time} object.}

\item{target}{A column to apply the function to}

\item{method}{The time series decomposition method. One of \code{"stl"} or \code{"twitter"}.
The STL method uses seasonal decomposition (see \code{\link[=decompose_stl]{decompose_stl()}}).
The Twitter method uses \code{trend} to remove the trend (see \code{\link[=decompose_twitter]{decompose_twitter()}}).}

\item{frequency}{Controls the seasonal adjustment (removal of seasonality).
Input can be either "auto", a time-based definition (e.g. "2 weeks"),
or a numeric number of observations per frequency (e.g. 10).
Refer to \code{\link[=time_frequency]{time_frequency()}}.}

\item{trend}{Controls the trend component
For stl, the trend controls the sensitivity of the lowess smoother, which is used to remove the remainder.
For twitter, the trend controls the period width of the median, which are used to remove the trend and center the remainder.}

\item{...}{Additional parameters passed to the underlying method functions.}

\item{merge}{A boolean. \code{FALSE} by default. If \code{TRUE}, will append results to the original data.}

\item{message}{A boolean. If \code{TRUE}, will output information related to \code{tbl_time} conversions, frequencies,
and trend / median spans (if applicable).}
}
\value{
Returns a \code{tbl_time} object.
}
\description{
Decompose a time series in preparation for anomaly detection
}
\details{
The \code{time_decompose()} function generates a time series decomposition on
\code{tbl_time} objects. The function is "tidy" in the sense that it works
on data frames. It is designed to work with time-based data, and as such
must have a column that contains date or datetime information. The function
also works with grouped data. The function implements several methods
of time series decomposition, each with benefits.

\strong{STL}:

The STL method (\code{method = "stl"}) implements time series decomposition using
the underlying \code{\link[=decompose_stl]{decompose_stl()}} function. If you are familiar with \code{\link[stats:stl]{stats::stl()}},
the function is a "tidy" version that is designed to work with \code{tbl_time} objects.
The decomposition separates the "season" and "trend" components from
the "observed" values leaving the "remainder" for anomaly detection.
The user can control two parameters: \code{frequency} and \code{trend}.
The \code{frequency} parameter adjusts the "season" component that is removed
from the "observed" values. The \code{trend} parameter adjusts the
trend window (\code{t.window} parameter from \code{stl()}) that is used.
The user may supply both \code{frequency}
and \code{trend} as time-based durations (e.g. "6 weeks") or numeric values
(e.g. 180) or "auto", which predetermines the frequency and/or trend
based on the scale of the time series.

\strong{Twitter}:

The Twitter method (\code{method = "twitter"}) implements time series decomposition using
the methodology from the Twitter \href{https://github.com/twitter/AnomalyDetection}{AnomalyDetection} package.
The decomposition separates the "seasonal" component and then removes
the median data, which is a different approach than the STL method for removing
the trend. This approach works very well for low-growth + high seasonality data.
STL may be a better approach when trend is a large factor.
The user can control two parameters: \code{frequency} and \code{trend}.
The \code{frequency} parameter adjusts the "season" component that is removed
from the "observed" values. The \code{trend} parameter adjusts the
period width of the median spans that are used. The user may supply both \code{frequency}
and \code{trend} as time-based durations (e.g. "6 weeks") or numeric values
(e.g. 180) or "auto", which predetermines the frequency and/or median spans
based on the scale of the time series.
}
\examples{

library(dplyr)

data(tidyverse_cran_downloads)

# Basic Usage
tidyverse_cran_downloads \%>\%
    time_decompose(count, method = "stl")

# twitter
tidyverse_cran_downloads \%>\%
    time_decompose(count,
                   method       = "twitter",
                   frequency    = "1 week",
                   trend        = "2 months",
                   merge        = TRUE,
                   message      = FALSE)

}
\references{
\enumerate{
\item CLEVELAND, R. B., CLEVELAND, W. S., MCRAE, J. E., AND TERPENNING, I.
STL: A Seasonal-Trend Decomposition Procedure Based on Loess. Journal of Official Statistics, Vol. 6, No. 1 (1990), pp. 3-73.
\item \href{https://www.usenix.org/system/files/conference/hotcloud14/hotcloud14-vallis.pdf}{Owen S. Vallis, Jordan Hochenbaum and Arun Kejariwal (2014). A Novel Technique for Long-Term Anomaly Detection in the Cloud. Twitter Inc.}
\item \href{https://github.com/twitter/AnomalyDetection}{Owen S. Vallis, Jordan Hochenbaum and Arun Kejariwal (2014). AnomalyDetection: Anomaly Detection Using Seasonal Hybrid Extreme Studentized Deviate Test. R package version 1.0.}
}
}
\seealso{
Decomposition Methods (Powers \code{time_decompose})
\itemize{
\item \code{\link[=decompose_stl]{decompose_stl()}}
\item \code{\link[=decompose_twitter]{decompose_twitter()}}
}

Time Series Anomaly Detection Functions (anomaly detection workflow):
\itemize{
\item \code{\link[=anomalize]{anomalize()}}
\item \code{\link[=time_recompose]{time_recompose()}}
}
}
