% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/derive_vars_dtm.R
\name{derive_vars_dtm}
\alias{derive_vars_dtm}
\title{Derive/Impute a Datetime from a Date Character Vector}
\usage{
derive_vars_dtm(
  dataset,
  new_vars_prefix,
  dtc,
  highest_imputation = "h",
  date_imputation = "first",
  time_imputation = "first",
  flag_imputation = "auto",
  min_dates = NULL,
  max_dates = NULL,
  preserve = FALSE,
  ignore_seconds_flag = FALSE
)
}
\arguments{
\item{dataset}{Input dataset

The variables specified by the \code{dtc} argument are expected to be in the dataset.}

\item{new_vars_prefix}{Prefix used for the output variable(s).

A character scalar is expected. For the date variable "DT" is appended to
the specified prefix, for the date imputation flag "DTF", and for the time
imputation flag "TMF". I.e., for \code{new_vars_prefix = "AST"} the variables
\code{ASTDT}, \code{ASTDTF}, and \code{ASTTMF} are created.}

\item{dtc}{The \code{'--DTC'} date to impute

A character date is expected in a format like \code{yyyy-mm-dd} or
\code{yyyy-mm-ddThh:mm:ss}. Trailing components can be omitted and \code{-} is a
valid "missing" value for any component.}

\item{highest_imputation}{Highest imputation level

The \code{highest_imputation} argument controls which components of the DTC
value are imputed if they are missing. All components up to the specified
level are imputed.

If a component at a higher level than the highest imputation level is
missing, \code{NA_character_} is returned. For example, for \code{highest_imputation = "D"} \code{"2020"} results in \code{NA_character_} because the month is missing.

If \code{"n"} is specified, no imputation is performed, i.e., if any component is
missing, \code{NA_character_} is returned.

If \code{"Y"} is specified, \code{date_imputation} should be \code{"first"} or \code{"last"}
and \code{min_dates} or \code{max_dates} should be specified respectively. Otherwise,
\code{NA_character_} is returned if the year component is missing.

\emph{Permitted Values}: \code{"Y"} (year, highest level), \code{"M"} (month), \code{"D"}
(day), \code{"h"} (hour), \code{"m"} (minute), \code{"s"} (second), \code{"n"} (none, lowest
level)}

\item{date_imputation}{The value to impute the day/month when a datepart is
missing.

A character value is expected, either as a
\itemize{
\item format with month and day specified as \code{"mm-dd"}: e.g. \code{"06-15"} for the
15th of June (The year can not be specified; for imputing the year
\code{"first"} or \code{"last"} together with \code{min_dates} or \code{max_dates} argument can
be used (see examples).),
\item or as a keyword: \code{"first"}, \code{"mid"}, \code{"last"} to impute to the first/mid/last
day/month. If \code{"mid"} is specified, missing components are imputed as the
middle of the possible range:
\itemize{
\item If both month and day are missing, they are imputed as \code{"06-30"}
(middle of the year).
\item If only day is missing, it is imputed as \code{"15"} (middle of the month).
}
}

The argument is ignored if \code{highest_imputation} is less then \code{"D"}.}

\item{time_imputation}{The value to impute the time when a timepart is
missing.

A character value is expected, either as a
\itemize{
\item format with hour, min and sec specified as \code{"hh:mm:ss"}: e.g. \code{"00:00:00"}
for the start of the day,
\item or as a keyword: \code{"first"},\code{"last"} to impute to the start/end of a day.
}

The argument is ignored if \code{highest_imputation = "n"}.}

\item{flag_imputation}{Whether the date/time imputation flag(s) must also be derived.

If \code{"both"} or \code{"date"} is specified, then date imputation flag is derived.
If \code{"auto"} is specified and \code{highest_imputation} argument is greater than
\code{"h"}, then date imputation flag is derived.

If \code{"both"} or \code{"time"} is specified, then time imputation flag is derived.
If \code{"auto"} is specified and \code{highest_imputation} argument is not \code{"n"},
then time imputation flag is derived.

If \code{"none"} is specified, then no date or time imputation flag is derived.

\emph{Permitted Values}: \code{"auto"}, \code{"date"}, \code{"time"}, \code{"both"}, or \code{"none"}}

\item{min_dates}{Minimum dates

A list of dates is expected. It is ensured that the imputed date is not
before any of the specified dates, e.g., that the imputed adverse event start
date is not before the first treatment date. Only dates which are in the
range of possible dates of the \code{dtc} value are considered. The possible dates
are defined by the missing parts of the \code{dtc} date (see example below). This
ensures that the non-missing parts of the \code{dtc} date are not changed.
A date or date-time object is expected.
For example

\if{html}{\out{<div class="sourceCode r">}}\preformatted{impute_dtc_dtm(
  "2020-11",
  min_dates = list(
   ymd_hms("2020-12-06T12:12:12"),
   ymd_hms("2020-11-11T11:11:11")
  ),
  highest_imputation = "M"
)
}\if{html}{\out{</div>}}

returns \code{"2020-11-11T11:11:11"} because the possible dates for \code{"2020-11"}
range from \code{"2020-11-01T00:00:00"} to \code{"2020-11-30T23:59:59"}. Therefore
\code{"2020-12-06T12:12:12"} is ignored. Returning \code{"2020-12-06T12:12:12"} would
have changed the month although it is not missing (in the \code{dtc} date).

For date variables (not datetime) in the list the time is imputed to
\code{"00:00:00"}. Specifying date variables makes sense only if the date is
imputed. If only time is imputed, date variables do not affect the result.}

\item{max_dates}{Maximum dates

A list of dates is expected. It is ensured that the imputed date is not after
any of the specified dates, e.g., that the imputed date is not after the data
cut off date. Only dates which are in the range of possible dates are
considered. A date or date-time object is expected.

For date variables (not datetime) in the list the time is imputed to
\code{"23:59:59"}. Specifying date variables makes sense only if the date is
imputed. If only time is imputed, date variables do not affect the result.}

\item{preserve}{Preserve lower level date/time part when higher order part
is missing, e.g. preserve day if month is missing or
preserve minute when hour is missing.

For example \code{"2019---07"} would return \verb{"2019-06-07} if \code{preserve = TRUE}
(and \code{date_imputation = "mid"}).

Permitted Values: \code{TRUE}, \code{FALSE}}

\item{ignore_seconds_flag}{ADaM IG states that given SDTM (\code{'--DTC'}) variable,
if only hours and minutes are ever collected, and seconds are imputed in
(\code{'--DTM'}) as 00, then it is not necessary to set (\code{'--TMF'}) to \code{'S'}. A user can set this
to \code{TRUE} so the \code{'S'} Flag is dropped from (\code{'--TMF'}).

\emph{Permitted Values}: A logical value}
}
\value{
The input dataset with the datetime \code{'--DTM'} (and the date/time imputation
flag \code{'--DTF'}, \code{'--TMF'}) added.
}
\description{
Derive a datetime object (\code{'--DTM'}) from a date character vector (\code{'--DTC'}).
The date and time can be imputed (see \code{date_imputation}/\code{time_imputation} arguments)
and the date/time imputation flag (\code{'--DTF'}, \code{'--TMF'}) can be added.
}
\details{
In \code{{admiral}} we don't allow users to pick any single part of the date/time to
impute, we only enable to impute up to a highest level, i.e. you couldn't
choose to say impute months, but not days.

The presence of a \code{'--DTF'} variable is checked and the variable is not derived
if it already exists in the input dataset. However, if \code{'--TMF'} already exists
in the input dataset, a warning is issued and \code{'--TMF'} will be overwritten.
}
\examples{
library(tibble)
library(lubridate)

mhdt <- tribble(
  ~MHSTDTC,
  "2019-07-18T15:25:40",
  "2019-07-18T15:25",
  "2019-07-18",
  "2019-02",
  "2019",
  "2019---07",
  ""
)

derive_vars_dtm(
  mhdt,
  new_vars_prefix = "AST",
  dtc = MHSTDTC,
  highest_imputation = "M"
)

# Impute AE end date to the last date and ensure that the imputed date is not
# after the death or data cut off date
adae <- tribble(
  ~AEENDTC, ~DTHDT, ~DCUTDT,
  "2020-12", ymd("2020-12-06"), ymd("2020-12-24"),
  "2020-11", ymd("2020-12-06"), ymd("2020-12-24")
)

derive_vars_dtm(
  adae,
  dtc = AEENDTC,
  new_vars_prefix = "AEN",
  highest_imputation = "M",
  date_imputation = "last",
  time_imputation = "last",
  max_dates = exprs(DTHDT, DCUTDT)
)

# Seconds has been removed from the input dataset.  Function now uses
# ignore_seconds_flag to remove the 'S' from the --TMF variable.
mhdt <- tribble(
  ~MHSTDTC,
  "2019-07-18T15:25",
  "2019-07-18T15:25",
  "2019-07-18",
  "2019-02",
  "2019",
  "2019---07",
  ""
)

derive_vars_dtm(
  mhdt,
  new_vars_prefix = "AST",
  dtc = MHSTDTC,
  highest_imputation = "M",
  ignore_seconds_flag = TRUE
)

# A user imputing dates as middle month/day, i.e. date_imputation = "MID" can
# use preserve argument to "preserve" partial dates.  For example, "2019---07",
# will be displayed as "2019-06-07" rather than 2019-06-15 with preserve = TRUE

derive_vars_dtm(
  mhdt,
  new_vars_prefix = "AST",
  dtc = MHSTDTC,
  highest_imputation = "M",
  date_imputation = "mid",
  preserve = TRUE
)
}
\seealso{
Date/Time Derivation Functions that returns variable appended to dataset: 
\code{\link{derive_var_trtdurd}()},
\code{\link{derive_vars_dt}()},
\code{\link{derive_vars_dtm_to_dt}()},
\code{\link{derive_vars_dtm_to_tm}()},
\code{\link{derive_vars_duration}()},
\code{\link{derive_vars_dy}()}
}
\concept{der_date_time}
\keyword{der_date_time}
\keyword{der_gen}
