% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/derive_date_vars.R
\name{derive_vars_dt}
\alias{derive_vars_dt}
\title{Derive/Impute a Date from a Date Character Vector}
\usage{
derive_vars_dt(
  dataset,
  new_vars_prefix,
  dtc,
  highest_imputation = "n",
  date_imputation = "first",
  flag_imputation = "auto",
  min_dates = NULL,
  max_dates = NULL,
  preserve = FALSE
)
}
\arguments{
\item{dataset}{Input dataset

The variables specified by the \code{dtc} argument are expected to be in the dataset.}

\item{new_vars_prefix}{Prefix used for the output variable(s).

A character scalar is expected. For the date variable "DT" is appended to
the specified prefix and for the date imputation flag "DTF". I.e., for
\code{new_vars_prefix = "AST"} the variables \code{ASTDT} and \code{ASTDTF} are created.}

\item{dtc}{The \code{'--DTC'} date to impute

A character date is expected in a format like \code{yyyy-mm-dd} or
\code{yyyy-mm-ddThh:mm:ss}. Trailing components can be omitted and \code{-} is a
valid "missing" value for any component.}

\item{highest_imputation}{Highest imputation level

The \code{highest_imputation} argument controls which components of the DTC
value are imputed if they are missing. All components up to the specified
level are imputed.

If a component at a higher level than the highest imputation level is
missing, \code{NA_character_} is returned. For example, for \code{highest_imputation = "D"} \code{"2020"} results in \code{NA_character_} because the month is missing.

If \code{"n"} is specified no imputation is performed, i.e., if any component is
missing, \code{NA_character_} is returned.

If \code{"Y"} is specified, \code{date_imputation} should be \code{"first"} or \code{"last"}
and \code{min_dates} or \code{max_dates} should be specified respectively. Otherwise,
\code{NA_character_} is returned if the year component is missing.

\emph{Permitted Values}: \code{"Y"} (year, highest level), \code{"M"} (month), \code{"D"}
(day), \code{"n"} (none, lowest level)}

\item{date_imputation}{The value to impute the day/month when a datepart is
missing.

A character value is expected, either as a
\itemize{
\item format with month and day specified as \code{"mm-dd"}: e.g. \code{"06-15"} for the
15th of June (The year can not be specified; for imputing the year
\code{"first"} or \code{"last"} together with \code{min_dates} or \code{max_dates} argument can
be used (see examples).),
\item or as a keyword: \code{"first"}, \code{"mid"}, \code{"last"} to impute to the first/mid/last
day/month. If \code{"mid"} is specified, missing components are imputed as the
middle of the possible range:
\itemize{
\item If both month and day are missing, they are imputed as \code{"06-30"}
(middle of the year).
\item If only day is missing, it is imputed as \code{"15"} (middle of the month).
}
}

The argument is ignored if \code{highest_imputation} is less then \code{"D"}.}

\item{flag_imputation}{Whether the date imputation flag must also be derived.

If \code{"auto"} is specified and \code{highest_imputation} argument is not \code{"n"},
then date imputation flag is derived.

If \code{"date"} is specified, then date imputation flag is derived.

If \code{"none"} is specified, then no date imputation flag is derived.

\emph{Permitted Values}: \code{"auto"}, \code{"date"} or \code{"none"}}

\item{min_dates}{Minimum dates

A list of dates is expected. It is ensured that the imputed date is not
before any of the specified dates, e.g., that the imputed adverse event start
date is not before the first treatment date. Only dates which are in the
range of possible dates of the \code{dtc} value are considered. The possible dates
are defined by the missing parts of the \code{dtc} date (see example below). This
ensures that the non-missing parts of the \code{dtc} date are not changed.
A date or date-time object is expected.
For example

\if{html}{\out{<div class="sourceCode r">}}\preformatted{impute_dtc_dtm(
  "2020-11",
  min_dates = list(
   ymd_hms("2020-12-06T12:12:12"),
   ymd_hms("2020-11-11T11:11:11")
  ),
  highest_imputation = "M"
)
}\if{html}{\out{</div>}}

returns \code{"2020-11-11T11:11:11"} because the possible dates for \code{"2020-11"}
range from \code{"2020-11-01T00:00:00"} to \code{"2020-11-30T23:59:59"}. Therefore
\code{"2020-12-06T12:12:12"} is ignored. Returning \code{"2020-12-06T12:12:12"} would
have changed the month although it is not missing (in the \code{dtc} date).}

\item{max_dates}{Maximum dates

A list of dates is expected. It is ensured that the imputed date is not after
any of the specified dates, e.g., that the imputed date is not after the data
cut off date. Only dates which are in the range of possible dates are
considered. A date or date-time object is expected.}

\item{preserve}{Preserve day if month is missing and day is present

For example \code{"2019---07"} would return \verb{"2019-06-07} if \code{preserve = TRUE}
(and \code{date_imputation = "MID"}).

Permitted Values: \code{TRUE}, \code{FALSE}}
}
\value{
The input dataset with the date \code{'--DT'} (and the date imputation flag \code{'--DTF'}
if requested) added.
}
\description{
Derive a date (\code{'--DT'}) from a date character vector (\verb{'--DTC}').
The date can be imputed (see \code{date_imputation} argument)
and the date imputation flag ('\verb{--DTF'}) can be added.
}
\details{
In \code{{admiral}} we don't allow users to pick any single part of the date/time to
impute, we only enable to impute up to a highest level, i.e. you couldn't
choose to say impute months, but not days.

The presence of a \code{'--DTF'} variable is checked and if it already exists in the input dataset,
a warning is issued and \code{'--DTF'} will be overwritten.
}
\examples{
library(tibble)
library(lubridate)

mhdt <- tribble(
  ~MHSTDTC,
  "2019-07-18T15:25:40",
  "2019-07-18T15:25",
  "2019-07-18",
  "2019-02",
  "2019",
  "2019---07",
  ""
)

# Create ASTDT and ASTDTF
# No imputation for partial date
derive_vars_dt(
  mhdt,
  new_vars_prefix = "AST",
  dtc = MHSTDTC
)

# Create ASTDT and ASTDTF
# Impute partial dates to first day/month
derive_vars_dt(
  mhdt,
  new_vars_prefix = "AST",
  dtc = MHSTDTC,
  highest_imputation = "M"
)

# Impute partial dates to 6th of April
derive_vars_dt(
  mhdt,
  new_vars_prefix = "AST",
  dtc = MHSTDTC,
  highest_imputation = "M",
  date_imputation = "04-06"
)

# Create AENDT and AENDTF
# Impute partial dates to last day/month
derive_vars_dt(
  mhdt,
  new_vars_prefix = "AEN",
  dtc = MHSTDTC,
  highest_imputation = "M",
  date_imputation = "last"
)

# Create BIRTHDT
# Impute partial dates to 15th of June. No Date Imputation Flag
derive_vars_dt(
  mhdt,
  new_vars_prefix = "BIRTH",
  dtc = MHSTDTC,
  highest_imputation = "M",
  date_imputation = "mid",
  flag_imputation = "none"
)

# Impute AE start date to the first date and ensure that the imputed date
# is not before the treatment start date
adae <- tribble(
  ~AESTDTC, ~TRTSDTM,
  "2020-12", ymd_hms("2020-12-06T12:12:12"),
  "2020-11", ymd_hms("2020-12-06T12:12:12")
)

derive_vars_dt(
  adae,
  dtc = AESTDTC,
  new_vars_prefix = "AST",
  highest_imputation = "M",
  min_dates = exprs(TRTSDTM)
)

# A user imputing dates as middle month/day, i.e. date_imputation = "mid" can
# use preserve argument to "preserve" partial dates.  For example, "2019---07",
# will be displayed as "2019-06-07" rather than 2019-06-15 with preserve = TRUE

derive_vars_dt(
  mhdt,
  new_vars_prefix = "AST",
  dtc = MHSTDTC,
  highest_imputation = "M",
  date_imputation = "mid",
  preserve = TRUE
)
}
\seealso{
Date/Time Derivation Functions that returns variable appended to dataset: 
\code{\link{derive_var_trtdurd}()},
\code{\link{derive_vars_dtm_to_dt}()},
\code{\link{derive_vars_dtm_to_tm}()},
\code{\link{derive_vars_dtm}()},
\code{\link{derive_vars_duration}()},
\code{\link{derive_vars_dy}()}
}
\concept{der_date_time}
\keyword{der_date_time}
\keyword{der_gen}
