% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/derive_var_confirmation_flag.R
\name{derive_var_confirmation_flag}
\alias{derive_var_confirmation_flag}
\title{Derive Confirmation Flag}
\usage{
derive_var_confirmation_flag(
  dataset,
  by_vars,
  order,
  new_var,
  join_vars,
  join_type,
  first_cond = NULL,
  filter,
  true_value = "Y",
  false_value = NA_character_,
  check_type = "warning"
)
}
\arguments{
\item{dataset}{Input dataset

The variables specified by the \code{by_vars} and \code{join_vars} parameter are
expected.}

\item{by_vars}{By variables

The specified variables are used as by variables for joining the input
dataset with itself.}

\item{order}{Order

The observations are ordered by the specified order.}

\item{new_var}{New variable

The specified variable is added to the input dataset.}

\item{join_vars}{Variables to keep from joined dataset

The variables needed from the other observations should be specified
for this parameter. The specified variables are added to the joined dataset
with suffix ".join". For example to flag all observations with \code{AVALC == "Y"} and \code{AVALC == "Y"} for at least one subsequent visit \code{join_vars = vars(AVALC, AVISITN)} and \code{filter = AVALC == "Y" & AVALC.join == "Y" & AVISITN < AVISITN.join} could be specified.

The \verb{*.join} variables are not included in the output dataset.}

\item{join_type}{Observations to keep after joining

The argument determines which of the joined observations are kept with
respect to the original observation. For example, if \code{join_type = "after"}
is specified all observations after the original observations are kept.

For example for confirmed response or BOR in the oncology setting or
confirmed deterioration in questionnaires the confirmatory assessment must
be after the assessment to be flagged. Thus \code{join_type = "after"} could be
used.

Whereas, sometimes you might allow for confirmatory observations to occur
prior to the observation to be flagged. For example, to flag AEs occurring
on or after seven days before a COVID AE. Thus \code{join_type = "all"} could be
used.

\emph{Permitted Values:} \code{"before"}, \code{"after"}, \code{"all"}}

\item{first_cond}{Condition for selecting range of data

If this argument is specified, the other observations are restricted up to
the first observation where the specified condition is fulfilled. If the
condition is not fulfilled for any of the other observations, no
observations are considered, i.e., the observation is not flagged.

This parameter should be specified if \code{filter} contains summary functions
which should not apply to all observations but only up to the confirmation
assessment. For an example see the third example below.}

\item{filter}{Condition for selecting observations

The filter is applied to the joined dataset for flagging the confirmed
observations. The condition can include summary functions. The joined
dataset is grouped by the original observations. I.e., the summary function
are applied to all observations up to the confirmation observation. For
example, \code{filter = AVALC == "CR" & all(AVALC.join \%in\% c("CR", "NE")) & count_vals(var = AVALC.join, val = "NE") <= 1} selects observations with
response "CR" and for all observations up to the confirmation observation
the response is "CR" or "NE" and there is at most one "NE".}

\item{true_value}{Value of \code{new_var} for flagged observations

\emph{Default}: \code{"Y"}}

\item{false_value}{Value of \code{new_var} for observations not flagged

\emph{Default}: \code{NA_character_}}

\item{check_type}{Check uniqueness?

If \code{"warning"} or \code{"error"} is specified, the specified message is issued
if the observations of the input dataset are not unique with respect to the
by variables and the order.

\emph{Default:} \code{"warning"}

\emph{Permitted Values:} \code{"none"}, \code{"warning"}, \code{"error"}}
}
\value{
The input dataset with the variable specified by \code{new_var} added.
}
\description{
Derive a flag which depends on other observations of the dataset. For
example, flagging events which need to be confirmed by a second event.
}
\details{
An example usage might be flagging if a patient received two required
medications within a certain timeframe of each other.

In the oncology setting, for example, the function could be used to flag if a
response value can be confirmed by an other assessment. This is commonly
used in endpoints such as best overall response.

The following steps are performed to produce the output dataset.
\subsection{Step 1}{

The input dataset is joined with itself by the variables specified for
\code{by_vars}. From the right hand side of the join only the variables
specified for \code{join_vars} are kept. The suffix ".join" is added to these
variables.

For example, for \code{by_vars = USUBJID}, \code{join_vars = vars(AVISITN, AVALC)} and input dataset

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# A tibble: 2 x 4
USUBJID AVISITN AVALC  AVAL
<chr>     <dbl> <chr> <dbl>
1             1 Y         1
1             2 N         0
}\if{html}{\out{</div>}}

the joined dataset is

\if{html}{\out{<div class="sourceCode r">}}\preformatted{A tibble: 4 x 6
USUBJID AVISITN AVALC  AVAL AVISITN.join AVALC.join
<chr>     <dbl> <chr> <dbl>        <dbl> <chr>
1             1 Y         1            1 Y
1             1 Y         1            2 N
1             2 N         0            1 Y
1             2 N         0            2 N
}\if{html}{\out{</div>}}
}

\subsection{Step 2}{

The joined dataset is restricted to observations with respect to
\code{join_type} and \code{order}.

The dataset from the example in the previous step with \code{join_type = "after"} and \code{order = vars(AVISITN)} is restricted to

\if{html}{\out{<div class="sourceCode r">}}\preformatted{A tibble: 4 x 6
USUBJID AVISITN AVALC  AVAL AVISITN.join AVALC.join
<chr>     <dbl> <chr> <dbl>        <dbl> <chr>
1             1 Y         1            2 N
}\if{html}{\out{</div>}}
}

\subsection{Step 3}{

If \code{first_cond} is specified, for each observation of the input dataset the
joined dataset is restricted to observations up to the first observation
where \code{first_cond} is fulfilled (the observation fulfilling the condition
is included). If for an observation of the input dataset the condition is
not fulfilled, the observation is removed.
}

\subsection{Step 4}{

The joined dataset is grouped by the observations from the input dataset
and restricted to the observations fulfilling the condition specified by
\code{filter}.
}

\subsection{Step 5}{

The first observation of each group is selected
}

\subsection{Step 6}{

The variable specified by \code{new_var} is added to the input dataset. It is
set to \code{true_value} for all observations which were selected in the
previous step. For the other observations it is set to \code{false_value}.
}
}
\examples{
library(tibble)
library(admiral)

# flag observations with a duration longer than 30 and
# at, after, or up to 7 days before a COVID AE (ACOVFL == "Y")
adae <- tribble(
  ~USUBJID, ~ADY, ~ACOVFL, ~ADURN,
  "1",        10, "N",          1,
  "1",        21, "N",         50,
  "1",        23, "Y",         14,
  "1",        32, "N",         31,
  "1",        42, "N",         20,
  "2",        11, "Y",         13,
  "2",        23, "N",          2,
  "3",        13, "Y",         12,
  "4",        14, "N",         32,
  "4",        21, "N",         41
)

derive_var_confirmation_flag(
  adae,
  new_var = ALCOVFL,
  by_vars = vars(USUBJID),
  join_vars = vars(ACOVFL, ADY),
  join_type = "all",
  order = vars(ADY),
  filter = ADURN > 30 & ACOVFL.join == "Y" & ADY >= ADY.join - 7
)

# flag observations with AVALC == "Y" and AVALC == "Y" at one subsequent visit
data <- tribble(
  ~USUBJID, ~AVISITN, ~AVALC,
  "1",      1,        "Y",
  "1",      2,        "N",
  "1",      3,        "Y",
  "1",      4,        "N",
  "2",      1,        "Y",
  "2",      2,        "N",
  "3",      1,        "Y",
  "4",      1,        "N",
  "4",      2,        "N",
)

derive_var_confirmation_flag(
  data,
  by_vars = vars(USUBJID),
  new_var = CONFFL,
  join_vars = vars(AVALC, AVISITN),
  join_type = "after",
  order = vars(AVISITN),
  filter = AVALC == "Y" & AVALC.join == "Y" & AVISITN < AVISITN.join
)

# select observations with AVALC == "CR", AVALC == "CR" at a subsequent visit,
# only "CR" or "NE" in between, and at most one "NE" in between
data <- tribble(
  ~USUBJID, ~AVISITN, ~AVALC,
  "1",      1,        "PR",
  "1",      2,        "CR",
  "1",      3,        "NE",
  "1",      4,        "CR",
  "1",      5,        "NE",
  "2",      1,        "CR",
  "2",      2,        "PR",
  "2",      3,        "CR",
  "3",      1,        "CR",
  "4",      1,        "CR",
  "4",      2,        "NE",
  "4",      3,        "NE",
  "4",      4,        "CR",
  "4",      5,        "PR"
)

derive_var_confirmation_flag(
  data,
  by_vars = vars(USUBJID),
  join_vars = vars(AVALC),
  join_type = "after",
  order = vars(AVISITN),
  new_var = CONFFL,
  first_cond = AVALC.join == "CR",
  filter = AVALC == "CR" & all(AVALC.join \%in\% c("CR", "NE")) &
    count_vals(var = AVALC.join, val = "NE") <= 1
)

# flag observations with AVALC == "PR", AVALC == "CR" or AVALC == "PR"
# at a subsequent visit at least 20 days later, only "CR", "PR", or "NE"
# in between, at most one "NE" in between, and "CR" is not followed by "PR"
data <- tribble(
  ~USUBJID, ~ADY, ~AVALC,
  "1",         6, "PR",
  "1",        12, "CR",
  "1",        24, "NE",
  "1",        32, "CR",
  "1",        48, "PR",
  "2",         3, "PR",
  "2",        21, "CR",
  "2",        33, "PR",
  "3",        11, "PR",
  "4",         7, "PR",
  "4",        12, "NE",
  "4",        24, "NE",
  "4",        32, "PR",
  "4",        55, "PR"
)

derive_var_confirmation_flag(
  data,
  by_vars = vars(USUBJID),
  join_vars = vars(AVALC, ADY),
  join_type = "after",
  order = vars(ADY),
  new_var = CONFFL,
  first_cond = AVALC.join \%in\% c("CR", "PR") & ADY.join - ADY >= 20,
  filter = AVALC == "PR" &
    all(AVALC.join \%in\% c("CR", "PR", "NE")) &
    count_vals(var = AVALC.join, val = "NE") <= 1 &
    (
      min_cond(var = ADY.join, cond = AVALC.join == "CR") >
        max_cond(var = ADY.join, cond = AVALC.join == "PR") |
        count_vals(var = AVALC.join, val = "CR") == 0
    )
)
}
\seealso{
\code{\link[=filter_confirmation]{filter_confirmation()}}

General Derivation Functions for all ADaMs that returns variable appended to dataset:
\code{\link{derive_var_extreme_flag}()},
\code{\link{derive_var_last_dose_amt}()},
\code{\link{derive_var_last_dose_date}()},
\code{\link{derive_var_last_dose_grp}()},
\code{\link{derive_var_merged_cat}()},
\code{\link{derive_var_merged_character}()},
\code{\link{derive_var_merged_exist_flag}()},
\code{\link{derive_var_obs_number}()},
\code{\link{derive_var_worst_flag}()},
\code{\link{derive_vars_last_dose}()},
\code{\link{derive_vars_merged_lookup}()},
\code{\link{derive_vars_merged}()},
\code{\link{derive_vars_transposed}()},
\code{\link{get_summary_records}()}
}
\author{
Stefan Bundfuss
}
\concept{der_gen}
\keyword{der_gen}
