% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/derive_var_extreme_flag.R
\name{derive_var_extreme_flag}
\alias{derive_var_extreme_flag}
\title{Add a Variable Flagging the First or Last Observation Within Each By Group}
\usage{
derive_var_extreme_flag(
  dataset,
  by_vars,
  order,
  new_var,
  mode,
  check_type = "warning"
)
}
\arguments{
\item{dataset}{Input dataset

The variables specified by the \code{by_vars} parameter are expected.}

\item{by_vars}{Grouping variables

Permitted Values: list of variables}

\item{order}{Sort order

The first or last observation is determined with respect to the specified
order.

Permitted Values: list of variables or functions of variables}

\item{new_var}{Variable to add

The specified variable is added to the output dataset. It is set to \code{"Y"}
for the first or last observation (depending on the mode) of each by group.

Permitted Values: list of name-value pairs}

\item{mode}{Flag mode

Determines of the first or last observation is flagged.

Permitted Values: \code{"first"}, \code{"last"}}

\item{check_type}{Check uniqueness?

If \code{"warning"} or \code{"error"} is specified, the specified message is issued
if the observations of the input dataset are not unique with respect to the
by variables and the order.

Default: \code{"warning"}

Permitted Values: \code{"none"}, \code{"warning"}, \code{"error"}}
}
\value{
The input dataset with the new flag variable added
}
\description{
Add a variable flagging the first or last observation within each by group
}
\details{
For each group (with respect to the variables specified for the
\code{by_vars} parameter), \code{new_var} is set to "Y" for the first or last observation
(with respect to the order specified for the \code{order} parameter and the flag mode
specified for the \code{mode} parameter). Only observations included by the \code{filter} parameter
are considered for flagging.
Otherwise, \code{new_var} is set to \code{NA}. Thus, the direction of "worst" is considered fixed for
all parameters in the dataset depending on the \code{order} and the \code{mode}, i.e. for every
parameter the first or last record will be flagged across the whole dataset.
}
\examples{
library(tibble)
library(dplyr, warn.conflicts = FALSE)
library(admiral.test)
data("admiral_vs")

# Flag last value for each patient, test, and visit, baseline observations are ignored
admiral_vs \%>\%
  restrict_derivation(
    derivation = derive_var_extreme_flag,
    args = params(
      by_vars = exprs(USUBJID, VSTESTCD, VISIT),
      order = exprs(VSTPTNUM),
      new_var = LASTFL,
      mode = "last"
    ),
    filter = VISIT != "BASELINE"
  ) \%>\%
  arrange(USUBJID, VSTESTCD, VISITNUM, VSTPTNUM) \%>\%
  select(USUBJID, VSTESTCD, VISIT, VSTPTNUM, VSSTRESN, LASTFL)

# Baseline (ABLFL) examples:

input <- tribble(
  ~STUDYID, ~USUBJID,  ~PARAMCD,     ~AVISIT,                  ~ADT, ~AVAL,    ~DTYPE,
  "TEST01",  "PAT01", "PARAM01",  "BASELINE", as.Date("2021-04-27"),  15.0,        NA,
  "TEST01",  "PAT01", "PARAM01",  "BASELINE", as.Date("2021-04-25"),  14.0,        NA,
  "TEST01",  "PAT01", "PARAM01",  "BASELINE", as.Date("2021-04-23"),  15.0, "AVERAGE",
  "TEST01",  "PAT01", "PARAM01",    "WEEK 1", as.Date("2021-04-27"),  10.0, "AVERAGE",
  "TEST01",  "PAT01", "PARAM01",    "WEEK 2", as.Date("2021-04-30"),  12.0,        NA,
  "TEST01",  "PAT02", "PARAM01", "SCREENING", as.Date("2021-04-27"),  15.0, "AVERAGE",
  "TEST01",  "PAT02", "PARAM01",  "BASELINE", as.Date("2021-04-25"),  14.0, "AVERAGE",
  "TEST01",  "PAT02", "PARAM01",  "BASELINE", as.Date("2021-04-23"),  15.0, "AVERAGE",
  "TEST01",  "PAT02", "PARAM01",    "WEEK 1", as.Date("2021-04-27"),  10.0, "AVERAGE",
  "TEST01",  "PAT02", "PARAM01",    "WEEK 2", as.Date("2021-04-30"),  12.0, "AVERAGE",
  "TEST01",  "PAT01", "PARAM02", "SCREENING", as.Date("2021-04-27"),  15.0, "AVERAGE",
  "TEST01",  "PAT01", "PARAM02", "SCREENING", as.Date("2021-04-25"),  14.0, "AVERAGE",
  "TEST01",  "PAT01", "PARAM02", "SCREENING", as.Date("2021-04-23"),  15.0,        NA,
  "TEST01",  "PAT01", "PARAM02",  "BASELINE", as.Date("2021-04-27"),  10.0, "AVERAGE",
  "TEST01",  "PAT01", "PARAM02",    "WEEK 2", as.Date("2021-04-30"),  12.0,        NA,
  "TEST01",  "PAT02", "PARAM02", "SCREENING", as.Date("2021-04-27"),  15.0,        NA,
  "TEST01",  "PAT02", "PARAM02",  "BASELINE", as.Date("2021-04-25"),  14.0,        NA,
  "TEST01",  "PAT02", "PARAM02",    "WEEK 1", as.Date("2021-04-23"),  15.0,        NA,
  "TEST01",  "PAT02", "PARAM02",    "WEEK 1", as.Date("2021-04-27"),  10.0,        NA,
  "TEST01",  "PAT02", "PARAM02",  "BASELINE", as.Date("2021-04-30"),  12.0,        NA
)

# Last observation
restrict_derivation(
  input,
  derivation = derive_var_extreme_flag,
  args = params(
    by_vars = exprs(USUBJID, PARAMCD),
    order = exprs(ADT),
    new_var = ABLFL,
    mode = "last"
  ),
  filter = AVISIT == "BASELINE"
)

# Worst observation - Direction = High
restrict_derivation(
  input,
  derivation = derive_var_extreme_flag,
  args = params(
    by_vars = exprs(USUBJID, PARAMCD),
    order = exprs(AVAL, ADT),
    new_var = ABLFL,
    mode = "last"
  ),
  filter = AVISIT == "BASELINE"
)

# Worst observation - Direction = Lo
restrict_derivation(
  input,
  derivation = derive_var_extreme_flag,
  args = params(
    by_vars = exprs(USUBJID, PARAMCD),
    order = exprs(desc(AVAL), ADT),
    new_var = ABLFL,
    mode = "last"
  ),
  filter = AVISIT == "BASELINE"
)

# Average observation
restrict_derivation(
  input,
  derivation = derive_var_extreme_flag,
  args = params(
    by_vars = exprs(USUBJID, PARAMCD),
    order = exprs(ADT, desc(AVAL)),
    new_var = ABLFL,
    mode = "last"
  ),
  filter = AVISIT == "BASELINE" & DTYPE == "AVERAGE"
)

# OCCURDS Examples
data("admiral_ae")

# Most severe AE first occurrence per patient
admiral_ae \%>\%
  mutate(
    TEMP_AESEVN =
      as.integer(factor(AESEV, levels = c("SEVERE", "MODERATE", "MILD")))
  ) \%>\%
  derive_var_extreme_flag(
    new_var = AOCCIFL,
    by_vars = exprs(USUBJID),
    order = exprs(TEMP_AESEVN, AESTDY, AESEQ),
    mode = "first"
  ) \%>\%
  arrange(USUBJID, AESTDY, AESEQ) \%>\%
  select(USUBJID, AEDECOD, AESEV, AESTDY, AESEQ, AOCCIFL)

# Most severe AE first occurrence per patient per body system
admiral_ae \%>\%
  mutate(
    TEMP_AESEVN =
      as.integer(factor(AESEV, levels = c("SEVERE", "MODERATE", "MILD")))
  ) \%>\%
  derive_var_extreme_flag(
    new_var = AOCCSIFL,
    by_vars = exprs(USUBJID, AEBODSYS),
    order = exprs(TEMP_AESEVN, AESTDY, AESEQ),
    mode = "first"
  ) \%>\%
  arrange(USUBJID, AESTDY, AESEQ) \%>\%
  select(USUBJID, AEBODSYS, AESEV, AESTDY, AOCCSIFL)
}
\seealso{
\code{\link[=derive_var_worst_flag]{derive_var_worst_flag()}}

General Derivation Functions for all ADaMs that returns variable appended to dataset:
\code{\link{derive_var_joined_exist_flag}()},
\code{\link{derive_var_last_dose_amt}()},
\code{\link{derive_var_last_dose_date}()},
\code{\link{derive_var_last_dose_grp}()},
\code{\link{derive_var_merged_cat}()},
\code{\link{derive_var_merged_character}()},
\code{\link{derive_var_merged_exist_flag}()},
\code{\link{derive_var_merged_summary}()},
\code{\link{derive_var_obs_number}()},
\code{\link{derive_var_relative_flag}()},
\code{\link{derive_vars_joined}()},
\code{\link{derive_vars_last_dose}()},
\code{\link{derive_vars_merged_lookup}()},
\code{\link{derive_vars_merged}()},
\code{\link{derive_vars_transposed}()},
\code{\link{get_summary_records}()}
}
\concept{der_gen}
\keyword{der_gen}
