% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/riskdiff.R
\name{add_risk_diff}
\alias{add_risk_diff}
\title{Add risk difference to a count layer}
\usage{
add_risk_diff(layer, ..., args = list(), distinct = TRUE)
}
\arguments{
\item{layer}{Layer upon which the risk difference will be attached}

\item{...}{Comparison groups, provided as character vectors where the first group is the comparison,
and the second is the reference}

\item{args}{Arguments passed directly into \code{\link{prop.test}}}

\item{distinct}{Logical - Use distinct counts (if available).}
}
\description{
A very common requirement for summary tables is to calculate the risk difference between treatment
groups. \code{add_risk_diff} allows you to do this. The underlying risk difference calculations
are performed using the Base R function \code{\link{prop.test}} - so prior to using this function,
be sure to familiarize yourself with its functionality.
}
\details{
\code{add_risk_diff} can only be attached to a count layer, so the count layer must be constructed
first. \code{add_risk_diff} allows you to compare the difference between treatment group, so all
comparisons should be based upon the values within the specified \code{treat_var} in your
\code{tplyr_table} object.

Comparisons are specified by providing two-element character vectors. You can provide as many of
these groups as you want. You can also use groups that have been constructed using
\code{\link{add_treat_grps}} or \code{\link{add_total_group}}. The first element provided will be considered
the 'reference' group (i.e. the left side of the comparison), and the second group will be considered
the 'comparison'. So if you'd like to see the risk difference of 'T1 - Placebo', you would specify
this as \code{c('T1', 'Placebo')}.

Tplyr forms your two-way table in the background, and then runs \code{\link{prop.test}} appropriately.
Similar to way that the display of layers are specified, the exact values and format of how you'd like
the risk difference display are set using \code{\link{set_format_strings}}. This controls both the values
and the format of how the risk difference is displayed. Risk difference formats are set within
\code{\link{set_format_strings}} by using the name 'riskdiff'.

You have 5 variables to choose from in your data presentation:
\describe{
  \item{\strong{comp}}{Probability of the left hand side group (i.e. comparison)}
  \item{\strong{ref}}{Probability of the right hand side group (i.e. reference)}
  \item{\strong{dif}}{Difference of comparison - reference}
  \item{\strong{low}}{Lower end of the confidence interval (default is 95\%, override with the \code{args} paramter)}
  \item{\strong{high}}{Upper end of the confidence interval (default is 95\%, override with the \code{args} paramter)}
}

Use these variable names when forming your \code{\link{f_str}} objects. The default presentation, if no
string format is specified, will be:

  \code{f_str('xx.xxx (xx.xxx, xx.xxx)', dif, low, high)}

Note - within Tplyr, you can account for negatives by allowing an extra space within your integer
side settings. This will help with your alignment.

If columns are specified on a Tplyr table, risk difference comparisons still only take place between
groups within the \code{treat_var} variable - but they are instead calculated treating the \code{cols}
variables as by variables. Just like the tplyr layers themselves, the risk difference will then be transposed
and display each risk difference as separate variables by each of the \code{cols} variables.

If \code{distinct} is TRUE (the default), all calculations will take place on the distinct counts, if
they are available. Otherwise, non-distinct counts will be used.

One final note - \code{\link{prop.test}} may throw quite a few warnings. This is natural, because it
alerts you when there's not enough data for the approximations to be correct. This may be unnerving
coming from a SAS programming world, but this is R is trying to alert you that the values provided
don't have enough data to truly be statistically accurate.
}
\examples{
library(magrittr)

## Two group comparisons with default options applied
t <- tplyr_table(mtcars, gear)

# Basic risk diff for two groups, using defaults
l1 <- group_count(t, carb) \%>\%
  # Compare 3 vs. 4, 3 vs. 5
  add_risk_diff(
    c('3', '4'),
    c('3', '5')
  )

# Build and show output
add_layers(t, l1) \%>\% build()

## Specify custom formats and display variables
t <- tplyr_table(mtcars, gear)

# Create the layer with custom formatting
l2 <- group_count(t, carb) \%>\%
  # Compare 3 vs. 4, 3 vs. 5
  add_risk_diff(
    c('3', '4'),
    c('3', '5')
  ) \%>\%
  set_format_strings(
    'n_counts' = f_str('xx (xx.x)', n, pct),
    'riskdiff' = f_str('xx.xxx, xx.xxx, xx.xxx, xx.xxx, xx.xxx', comp, ref, dif, low, high)
  )

# Build and show output
add_layers(t, l2) \%>\% build()

## Passing arguments to prop.test
t <- tplyr_table(mtcars, gear)

# Create the layer with args option
l3 <- group_count(t, carb) \%>\%
  # Compare 3 vs. 4, 4 vs. 5
  add_risk_diff(
    c('3', '4'),
    c('3', '5'),
    args = list(conf.level = 0.9, correct=FALSE, alternative='less')
  )

# Build and show output
add_layers(t, l3) \%>\% build()
}
