% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/run_model.R
\name{run_model}
\alias{run_model}
\title{Estimate mobility flows based on different trip distribution models}
\usage{
run_model(
  proba,
  model = "UM",
  nb_trips = 1000,
  out_trips = NULL,
  in_trips = out_trips,
  average = FALSE,
  nbrep = 3,
  maxiter = 50,
  mindiff = 0.01,
  check_names = FALSE
)
}
\arguments{
\item{proba}{a squared matrix of probability. The sum of the matrix element
must be equal to 1. It will be normalized automatically if it is not the
case.}

\item{model}{a character indicating which model to use.}

\item{nb_trips}{a numeric value indicating the total number of trips. Must
be an integer if \code{average = FALSE} (see Details).}

\item{out_trips}{a numeric vector representing the number of outgoing
trips per location. Must be a vector of integers
if \code{average = FALSE} (see Details).}

\item{in_trips}{a numeric vector representing the number of incoming
trips per location. Must be a vector of integers
if \code{average = FALSE} (see Details).}

\item{average}{a boolean indicating if the average mobility flow matrix
should be generated instead of the \code{nbrep} matrices based on
random draws (see Details).}

\item{nbrep}{an integer indicating the number of replications
associated to the model run. Note that \code{nbrep = 1} if \code{average = TRUE}
(see Details).}

\item{maxiter}{an integer indicating the maximal number of iterations for
adjusting the Doubly Constrained Model (see Details).}

\item{mindiff}{a numeric strictly positive value indicating the
stopping criterion for adjusting the Doubly Constrained Model (see Details).}

\item{check_names}{a boolean indicating if the ID location are used as
vector names, matrix rownames and colnames and if they should be checked
(see Note).}
}
\value{
An object of class \code{TDLM}. A list of matrices containing the
\code{nbrep} simulated matrices.
}
\description{
This function estimates mobility flows using different distribution models.
As described in \insertCite{Lenormand2016;textual}{TDLM}, we
propose a two-step approach to generate mobility flows by separating the trip
distribution law, gravity or intervening opportunities, from the modeling
approach used to generate the flows from this law. This function only uses
the second step to generate mobility flow based on a matrix of probabilities
using different models.
}
\details{
\loadmathjax

We propose four constrained models to generate the flow from the matrix
of probabilities. These models respect different level of
constraints. These constraints can preserve the total number of trips
(argument \code{nb_trips}) OR the number of out-going trips
\mjeqn{O_{i}}{O_{i}} (argument \code{out_trips}) AND/OR the number of in-coming
\mjeqn{D_{j}}{D_{j}} (argument \code{in_trips}) according to the model. The sum of
out-going trips \mjeqn{\sum_{i} O_{i}}{\sum_{i} O_{i}} should be equal to the
sum of in-coming trips \mjeqn{\sum_{j} D_{j}}{\sum_{j} D_{j}}.
\enumerate{
\item Unconstrained model (\code{model = "UM"}). Only \code{nb_trips} will be preserved
(arguments \code{out_trips} and \code{in_trips} will not be used).
\item Production constrained model (\code{model = "PCM"}). Only \code{out_trips} will be
preserved (arguments \code{nb_trips} and \code{in_trips} will not be used).
\item Attraction constrained model (\code{model = "ACM"}). Only \code{in_trips} will be
preserved (arguments \code{nb_trips} and \code{out_trips} will not be used).
\item Doubly constrained model (\code{model = "DCM"}). Both \code{out_trips} and
\code{in_trips} will be preserved (arguments \code{nb_trips}will not be used). The
doubly constrained model is based on an Iterative Proportional Fitting
process \insertCite{Deming1940}{TDLM}. The arguments \code{maxiter} (50 by
default) and \code{mindiff} (0.01 by default) can be used to tune the model.
\code{mindiff} is the minimal tolerated relative error between the
simulated and observed marginals. \code{maxiter}
ensures that the algorithm stops even if it has not converged toward the
\code{mindiff} wanted value.
}

By default, when \code{average = FALSE}, \code{nbrep} matrices are generated from
\code{proba} with multinomial random draws that will take different forms
according to the model used. In this case, the models will deal with positive
integers as inputs and outputs. Nevertheless, it is also possible to generate
an average matrix based on a multinomial distribution (based on an infinite
number of drawings). In this case, the models' inputs can be either positive
integer or real numbers and the output (\code{nbrep = 1} in this case) will be a
matrix of positive real numbers.
}
\note{
All the inputs should be based on the same number of
locations sorted in the same order. It is recommended to use the location ID
as vector names, matrix rownames and matrix colnames and to set
\code{check_names = TRUE} to verify that everything is in order before running
this function (\code{check_names = FALSE} by default). Note that the function
\code{\link[=check_format_names]{check_format_names()}} can be used to control the validity of all the inputs
before running the main package's functions.
}
\examples{
data(mass)
data(od)

proba <- od / sum(od)

Oi <- as.numeric(mass[, 2])
Dj <- as.numeric(mass[, 3])

res <- run_model(
  proba = proba,
  model = "DCM", nb_trips = NULL, out_trips = Oi, in_trips = Dj,
  average = FALSE, nbrep = 3, maxiter = 50, mindiff = 0.01,
  check_names = FALSE
)

# print(res)

}
\references{
\insertRef{Lenormand2016}{TDLM}

\insertRef{Deming1940}{TDLM}
}
\seealso{
\code{\link[=gof]{gof()}} \code{\link[=run_law_model]{run_law_model()}} \code{\link[=run_law]{run_law()}} \code{\link[=check_format_names]{check_format_names()}}
}
\author{
Maxime Lenormand (\email{maxime.lenormand@inrae.fr})
}
