% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rips_complexes.R
\name{plot_vr_graph}
\alias{plot_vr_graph}
\title{Plot a VR graph using the igraph package.}
\usage{
plot_vr_graph(
  graphs,
  eps,
  cols = NULL,
  layout = NULL,
  title = NULL,
  component_of = NULL,
  plot_isolated_vertices = FALSE,
  return_layout = FALSE,
  vertex_labels = TRUE
)
}
\arguments{
\item{graphs}{the output of a `vr_graphs` function call.}

\item{eps}{the numeric radius of the graph in `graphs` to plot.}

\item{cols}{an optional character vector of vertex colors, default `NULL`.}

\item{layout}{an optional 2D matrix of vertex coordinates, default `NULL`. If row names are supplied they can be used to subset a graph by those vertex names.}

\item{title}{an optional str title for the plot, default `NULL`.}

\item{component_of}{a vertex name (integer or character), only the component of the graph containing that vertex will be plotted (useful for identifying representative (co)cycles in graphs). Default `NULL` (plot the whole graph).}

\item{plot_isolated_vertices}{a boolean representing whether or not to plot isolated vertices, default `FALSE`.}

\item{return_layout}{a boolean representing whether or not to return the plotting layout (x-y coordinates of each vertex) and the vertex labels, default `FALSE`.}

\item{vertex_labels}{a boolean representing whether or not to plot vertex labels, default `TRUE`.}
}
\value{
if `return_layout` is `TRUE` then a list with elements "layout" (the numeric matrix of vertex x-y coordinates) and "vertices" (character vertex labels), otherwise the function does not return anything.
}
\description{
This function will throw an error if the igraph package is not installed.
}
\examples{

if(require("TDAstats") & require("igraph"))
{
  # simulate data from the unit circle and calculate 
  # its diagram
  df <- TDAstats::circle2d[sample(1:100,25),]
  diag <- TDAstats::calculate_homology(df,
                                       dim = 1,
                                       threshold = 2)
  
  # get minimum death radius of any data cluster
  min_death_H0 <- 
  min(diag[which(diag[,1] == 0),3L])
  
  # get birth and death radius of the loop
  loop_birth <- as.numeric(diag[nrow(diag),2L])
  loop_death <- as.numeric(diag[nrow(diag),3L])

  # compute VR graphs at radii half of 
  # min_death_H0 and the mean of loop_birth and 
  # loop_death, returning clusters
  graphs <- vr_graphs(X = df,eps = 
  c(0.5*min_death_H0,(loop_birth + loop_death)/2))
  
  # plot graph of smaller (first) radius
  plot_vr_graph(graphs = graphs,eps = 0.5*min_death_H0,
                  plot_isolated_vertices = TRUE)
  
  # plot graph of larger (second) radius
  plot_vr_graph(graphs = graphs,eps = (loop_birth + loop_death)/2)
  
  # repeat but with rownames for df, each vertex
  # will be plotted with its rownames
  rownames(df) <- paste0("V",1:25)
  graphs <- vr_graphs(X = df,eps = 
  c(0.5*min_death_H0,(loop_birth + loop_death)/2))
  plot_vr_graph(graphs = graphs,eps = 0.5*min_death_H0,
                  plot_isolated_vertices = TRUE)
  
  # plot without vertex labels
  plot_vr_graph(graphs = graphs,eps = (loop_birth + loop_death)/2,
                  vertex_labels = FALSE)
                 
  # plot only the graph component containing vertex "1"
  plot_vr_graph(graphs = graphs,eps = 0.5*min_death_H0,
                  component_of = "V1",plot_isolated_vertices = TRUE)
 
  # save the layout of the graph for adding features to 
  # the same graph layout, like color
  layout <- plot_vr_graph(graphs = graphs,eps = (loop_birth + loop_death)/2,
                            return_layout = TRUE,vertex_labels = TRUE)
  cols <- rep("blue",25)
  cols[1:5] <- "red"
  plot_vr_graph(graphs = graphs,eps = (loop_birth + loop_death)/2,cols = cols,
                  layout = layout)
  
}
}
\author{
Shael Brown - \email{shaelebrown@gmail.com}
}
