% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/emmeans_power.R
\name{emmeans_power}
\alias{emmeans_power}
\alias{emmeans_power.emmGrid}
\alias{emmeans_power.summary_em}
\alias{emmeans_power.data.frame}
\title{Compute power for \pkg{emmeans} contrasts}
\usage{
emmeans_power(x, ...)

\method{emmeans_power}{emmGrid}(x, ...)

\method{emmeans_power}{summary_em}(x, ...)

\method{emmeans_power}{data.frame}(
  x,
  alpha_level = Superpower_options("alpha_level"),
  liberal_lambda = Superpower_options("liberal_lambda"),
  ...
)
}
\arguments{
\item{x}{\code{emmGrid}. Grid of contrasts to estimate power from.}

\item{...}{Other arguments passed to the function if object is not already a \code{emmGrid} object.}

\item{alpha_level}{Alpha level used to determine statistical significance}

\item{liberal_lambda}{Logical indictor of whether to use the liberal (cohen_f^2\*(num_df+den_df)) or conservative (cohen_f^2\*den_df) calculation of the noncentrality (lambda) parameter estimate. Default is FALSE.}
}
\value{
Returns dataframe with simulation data (power and effect sizes!), anova results and simple effect results, plot of exact data, and alpha_level. Note: Cohen's f = sqrt(pes/1-pes) and the noncentrality parameter is = f^2*df(error)

\describe{
  \item{\code{"dataframe"}}{A dataframe of the simulation result.}
  \item{\code{"aov_result"}}{\code{aov} object returned from \code{\link{aov_car}}.}
  \item{\code{"aov_result"}}{\code{emmeans} object returned from \code{\link{emmeans}}.}
  \item{\code{"main_result"}}{The power analysis results for ANOVA level effects.}
  \item{\code{"pc_results"}}{The power analysis results for the pairwise (t-test) comparisons.}
  \item{\code{"emm_results"}}{The power analysis results of the pairwise comparison results.}
  \item{\code{"manova_results"}}{Default is "NULL". If a within-subjects factor is included, then the power of the multivariate (i.e. MANOVA) analyses will be provided.}
  \item{\code{"alpha_level"}}{The alpha level, significance cut-off, used for the power analysis.}
  \item{\code{"method"}}{Record of the function used to produce the simulation}
  \item{\code{"plot"}}{A plot of the dataframe from the simulation; should closely match the meansplot in \code{\link{ANOVA_design}}}

}
}
\description{
Computes power based on t value and degrees of freedom for contrasts.
\emph{Do not use to calculate "observed power" for empirical datasets
(Hoenig & Heisey, 2001).}
}
\details{
Note that calculation of power is based on the F- and t-ratio assuming
  two-sided testing. Thus, the function does not honor adjustments of the
  testing procedure due to either one-sided testing (or two-one sideded tests)
  or corrections for multiple comparisons via the \code{p.adjust} option in 
  \code{emmeans}.
  
  Power for one-sided tests can be calculated, if the means of the simulated
  dataset are consistent with the directional hypothesis, by doubeling
  \code{alpha_level}. Similarly, power for Bonferroni-corrected contrasts can be
  calculated by adjusting \code{alpha_level} accordingly (see examples).
  \code{...} Other arugments passed onto the function
}
\examples{
\dontrun{
# Set up a within design with 2 factors, each with 2 levels
design_result <- ANOVA_design(design = "2w*2w",
n = 40, mu = c(1, 0, 1, 0),
sd = 2, r = 0.8,
labelnames = c("condition", "cheerful",
 "sad", "voice", "human", "robot"))
 
exact_result <- ANOVA_exact(design_result, 
alpha_level = 0.05, verbose = FALSE, 
emm = TRUE, contrast_type = "pairwise")

# Power for pairwise contrasts
exact_result$emm_results

# Corresponding emmeans contrasts
exact_result$emmeans$contrasts

# Manually recalculate power
emmeans_power(exact_result$emmeans$contrasts, 
alpha_level = 0.05)

# Calculate power for Bonferroni-adjusted pairwise comparisons
n_contrasts <- nrow(as.data.frame(exact_result$emmeans$contrasts))
emmeans_power(exact_result$emmeans$contrasts, 
alpha_level = 0.05 / n_contrasts)

# Calculate power for one-sided custom contrasts
exact_result$emmeans$emmeans
custom_contrast <- contrast(exact_result$emmeans$emmeans, 
list(robot_vs_sad_human = c(0, 1, -0.5, -0.5)))
emmeans_power(custom_contrast,
 alpha_level = 0.05 * 2)

# Calculate power for follow-up ANOVA
follow_up <- joint_tests(exact_result$emmeans$emmeans, 
by = "condition")
emmeans_power(follow_up, 
alpha_level = 0.05 / 2)
emmeans_power(emmeans(exact_result$emmeans$emmeans, 
pairwise ~ voice | condition)$contrasts, 
alpha_level = 0.05 / 2)
}
}
\references{
Hoenig, J. M., & Heisey, D. M. (2001). The Abuse of Power. The American Statistician, 55(1), 19–24. https://doi.org/10.1198/000313001300339897
}
\author{
Frederik Aust
}
