% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simMixedSpots.R
\name{simMixedProfiles}
\alias{simMixedProfiles}
\title{Simulate training and test mixed spot profiles}
\usage{
simMixedProfiles(
  object,
  type.data = "both",
  mixing.function = "AddRawCount",
  file.backend = NULL,
  compression.level = NULL,
  block.processing = FALSE,
  block.size = 1000,
  chunk.dims = NULL,
  threads = 1,
  verbose = TRUE
)
}
\arguments{
\item{object}{\code{\linkS4class{SpatialDDLS}} object with
\code{single.cell.real}/\code{single.cell.simul}, and
\code{prob.cell.types} slots.}

\item{type.data}{Type of data to generate: \code{'train'}, \code{'test'} or
\code{'both'} (the last by default).}

\item{mixing.function}{Function used to build mixed transcriptional profiles.
It may be: \itemize{  \item \code{"AddRawCount"}: single-cell profiles (raw
counts) are added up across cells. Then, log-CPMs are calculated (by
default). \item \code{"MeanCPM"}: single-cell profiles (raw counts) are
transformed into CPMs and cross-cell averages are calculated. Then,
\code{log2(CPM + 1)} is calculated. \item \code{"AddCPM"}: single-cell
profiles (raw counts) are transformed into CPMs and are added up across
cells. Then, log-CPMs are calculated.}}

\item{file.backend}{Valid file path to store simulated mixed expression
profiles as an HDF5 file (\code{NULL} by default). If provided, data are
stored in HDF5 files used as back-end by using the \pkg{DelayedArray},
\pkg{HDF5Array} and \pkg{rhdf5} packages instead of loading all data into
RAM. Note that operations on this matrix will be performed in blocks (i.e
subsets of determined size) which may result in longer execution times.}

\item{compression.level}{The compression level used if \code{file.backend} is
provided. It is an integer value between 0 (no compression) and 9 (highest
and slowest compression). See
\code{?\link[HDF5Array]{getHDF5DumpCompressionLevel}} from the
\pkg{HDF5Array} package for more information.}

\item{block.processing}{Boolean indicating whether data should be simulated
in blocks (only if \code{file.backend} is used, \code{FALSE} by default).
This functionality is suitable for cases where it is not possible to load
all data into memory, and it leads to longer execution times.}

\item{block.size}{Only if \code{block.processing = TRUE}. Number of mixed
expression profiles that will be simulated in each iteration. Larger
numbers result in higher memory usage but shorter execution times. Set
accordingly to available computational resources (1000 by default).}

\item{chunk.dims}{Specifies the dimensions that HDF5 chunk will have. If
\code{NULL}, the default value is a vector of two items: the number of
genes considered by \code{\linkS4class{SpatialDDLS}} object during the
simulation, and a single sample to reduce read times in the following
steps. A larger number of columns written in each chunk can lead to longer
read times.}

\item{threads}{Number of threads used during simulation (1 by default).}

\item{verbose}{Show informative messages during the execution (\code{TRUE} by
default).}
}
\value{
A \code{\linkS4class{SpatialDDLS}} object with \code{mixed.profiles}
slot containing a list with one or two entries (depending on selected
\code{type.data} argument): \code{'train'} and \code{'test'}. Each entry
consists of a \code{\linkS4class{SummarizedExperiment}} object with the
simulated mixed slot profiles.
}
\description{
Simulate training and test mixed spot transcriptional profles using cell
composition matrices generated by the \code{\link{genMixedCellProp}}
function.
}
\details{
Mixed profiles are generated under the assumption that the expression level
of a particular gene in a given spot is the sum of the expression levels of
the cell types that make it up weighted by their proportions. In practice, as
described in Torroja and Sanchez-Cabo, 2019, these profiles are generated by
summing gene expression levels of a determined number of cells specified by a
known cell composition matrix. The number of simulated spots and cells used
to simulate each spot are determined by the \code{\link{genMixedCellProp}}
function. This step can be avoided by using the \code{on.the.fly} argument in
the \code{\link{trainDeconvModel}} function.

\pkg{SpatialDDLS} allows to use HDF5 files as back-end to store simulated
data using the \pkg{DelayedArray} and \pkg{HDF5Array} packages. This
functionality allows to work without keeping the data loaded into RAM, which
could be useful during some computationally heavy steps such as neural
network training on RAM-limited machines. You must provide a valid file path
in the \code{file.backend} argument to store the resulting file with the
'.h5' extension. This option slows down execution times, as subsequent
transformations of the data will be done in blocks. Note that if you use the
\code{file.backend} argument with \code{block.processing = FALSE}, all mixed
profiles will be simulated in one step and, thus, loaded into RAM. Then, the
matrix will be written to an HDF5 file. To avoid the RAM collapse, these
profiles can be simulated and written to HDF5 files in blocks of
\code{block.size} size by setting \code{block.processing = TRUE}. We
recommend this option accordingly to the computational resources available
and the number of simulated spots to be generated, but, in most of the cases,
it is not necessary.
}
\examples{
set.seed(123)
sce <- SingleCellExperiment::SingleCellExperiment(
  assays = list(
    counts = matrix(
      rpois(100, lambda = 5), nrow = 40, ncol = 30,
      dimnames = list(paste0("Gene", seq(40)), paste0("RHC", seq(30)))
    )
  ),
  colData = data.frame(
    Cell_ID = paste0("RHC", seq(30)),
    Cell_Type = sample(x = paste0("CellType", seq(4)), size = 30,
                       replace = TRUE)
  ),
  rowData = data.frame(
    Gene_ID = paste0("Gene", seq(40))
  )
)

SDDLS <- createSpatialDDLSobject(
  sc.data = sce,
  sc.cell.ID.column = "Cell_ID",
  sc.gene.ID.column = "Gene_ID",
  project = "Simul_example"
)
SDDLS <- genMixedCellProp(
  object = SDDLS,
  cell.ID.column = "Cell_ID",
  cell.type.column = "Cell_Type",
  num.sim.spots = 10,
  verbose = TRUE
)
SDDLS <- simMixedProfiles(SDDLS, verbose = TRUE)

}
\references{
Fischer B, Smith M and Pau, G (2020). rhdf5: R Interface to HDF5.
R package version 2.34.0.

Pagès H, Hickey P and Lun A (2020). DelayedArray: A unified framework for
working transparently with on-disk and in-memory array-like datasets. R
package version 0.16.0.

Pagès H (2020). HDF5Array: HDF5 backend for DelayedArray objects. R package
version 1.18.0.
}
\seealso{
\code{\link{genMixedCellProp}} \code{\linkS4class{PropCellTypes}}
\code{\link{trainDeconvModel}}
}
