% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulateIDM.R
\name{idmModel}
\alias{idmModel}
\title{Generate illness-death model objects}
\usage{
idmModel(scale.illtime = 1/100, shape.illtime = 1, scale.lifetime = 1/100,
  shape.lifetime = 1, scale.waittime = 1/100, shape.waittime = 1,
  scale.censtime = 1/100, shape.censtime = 1, n.inspections = 5,
  schedule = 10, punctuality = 5)
}
\arguments{
\item{scale.illtime}{Weilbull scale for latent illness time}

\item{shape.illtime}{Weilbull shape for latent illness time}

\item{scale.lifetime}{Weilbull scale for latent life time}

\item{shape.lifetime}{Weilbull shape for latent life time}

\item{scale.waittime}{Weilbull scale for latent life time}

\item{shape.waittime}{Weilbull shape for latent life time}

\item{scale.censtime}{Weilbull scale for censoring time}

\item{shape.censtime}{Weilbull shape for censoring time}

\item{n.inspections}{Number of inspection times}

\item{schedule}{Mean of the waiting time between adjacent
inspections.}

\item{punctuality}{Standard deviation of waiting time between
inspections.}
}
\value{
A latent variable model object \code{lvm}
}
\description{
Function to generate an illness-death model for simulation.
}
\details{
Based on the functionality of the lava PACKAGE the function generates
a latent variable model (latent illtime, waittime and lifetime)
and censoring mechanism (censtime, inspection1,inspection2,...,inspectionK).

The function \code{\link{sim.idmModel}} then simulates
right censored lifetimes and interval censored illness times.
}
\examples{
\dontrun{
library(lava)
library(prodlim)
# generate illness-death model based on exponentially
# distributed times
m <- idmModel(scale.illtime=1/70,
              shape.illtime=1.8,
              scale.lifetime=1/50,
              shape.lifetime=0.7,
              scale.waittime=1/30,
              shape.waittime=0.7)
round(sim(m,6),1)

# Estimate the parameters of the Weibull models
# based on the uncensored exact event times
# and the uncensored illstatus.
set.seed(18)
d <- sim(m,100,latent=FALSE)
d$uncensored.status <- 1
f <- idm(formula01=Hist(time=illtime,event=illstatus)~1,
         formula02=Hist(time=lifetime,event=uncensored.status)~1,
         data=d,
         conf.int=FALSE)
print(f)

# Change the rate of the 0->2 and 0->1 transitions
# also the rate of the 1->2 transition
# and also lower the censoring rate
m <- idmModel(scale.lifetime=1/2000,
              scale.waittime=1/30,
              scale.illtime=1/1000,
              scale.censtime=1/1000)
set.seed(18)
d <- sim(m,50,latent=TRUE)
d$uncensored.status <- 1

f <- idm(formula01=Hist(time=observed.illtime,event=illstatus)~1,
         formula02=Hist(time=observed.lifetime,event=uncensored.status)~1,
         data=d,
         conf.int=FALSE)
print(f)

# Estimate based on the right censored observations
fc <- idm(formula01=Hist(time=illtime,event=seen.ill)~1,
          formula02=Hist(time=observed.lifetime,event=seen.exit)~1,
          data=d,
          conf.int=FALSE)
print(fc)

# Estimate based on interval censored and right censored observations
fi <- idm(formula01=Hist(time=list(L,R),event=seen.ill)~1,
          formula02=Hist(time=observed.lifetime,event=seen.exit)~1,
          data=d,
          conf.int=FALSE)
print(fi)

# Estimation of covariate effects:
# X1, X2, X3
m <- idmModel(shape.waittime=2,
              scale.lifetime=1/2000,
              scale.waittime=1/300,
              scale.illtime=1/10000,
              scale.censtime=1/10000)
distribution(m,"X1") <- binomial.lvm(p=0.3)
distribution(m,"X2") <- normal.lvm(mean=120,sd=20)
distribution(m,"X3") <- normal.lvm(mean=50,sd=20)
regression(m,to="latent.illtime",from="X1") <- 1.7
regression(m,to="latent.illtime",from="X2") <- 0.07
regression(m,to="latent.illtime",from="X3") <- -0.1
regression(m,to="latent.waittime",from="X1") <- 1.8
regression(m,to="latent.lifetime",from="X1") <- 0.7
set.seed(28)
d <- sim(m,100,latent=TRUE)
head(d)
table(ill=d$seen.ill,death=d$seen.exit)

# Estimation based on uncensored data
d$uncensored.status <- 1
# uncensored data
F1 <- idm(formula01=Hist(time=illtime,event=illstatus)~X1+X2+X3,
          formula02=Hist(time=lifetime,event=uncensored.status)~X1+X2+X3,
          data=d,conf.int=FALSE)
print(F1)

# Estimation based on right censored data
F2 <- idm(formula01=Hist(time=illtime,event=seen.ill)~X1+X2+X3,
          formula02=Hist(time=observed.lifetime,event=seen.exit)~X1+X2+X3,
          data=d,conf.int=FALSE)
print(F2)

# Estimation based on interval censored and right censored data
F3 <- idm(formula01=Hist(time=list(L,R),event=seen.ill)~X1+X2+X3,
          formula02=Hist(time=observed.lifetime,event=seen.exit)~X1+X2+X3,
          data=d,conf.int=FALSE)
print(F3)
cbind(uncensored=F1$coef,right.censored=F2$coef,interval.censored=F3$coef)
}
}
\author{
Thomas Alexander Gerds
}

