% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/descript.R
\name{descript}
\alias{descript}
\alias{get_descriptFuns}
\title{Compute univariate descriptive statistics}
\usage{
descript(df, funs = get_descriptFuns(), discrete = FALSE)

get_descriptFuns()
}
\arguments{
\item{df}{a \code{data.frame} or \code{tibble}-like structure
containing the variables of interest.
Note that \code{factor} and \code{character} vectors will be treated as
discrete observations, and by default are omitted from the computation
of the descriptive statistics specified in \code{funs}}

\item{funs}{functions to apply when \code{discrete = FALSE}. Can be modified
 by the user to include or exclude further functions, however each supplied
 function must return a scalar. Use \code{get_discreteFuns()} to return
 the full list of functions, which may then be augmented or subsetted
 based on the user's requirements. Default descriptive statistic returned are:

 \describe{
  \item{\code{n}}{number of non-missing observations}
  \item{\code{miss}}{number of missing observations}
  \item{\code{mean}}{mean}
  \item{\code{trimmed}}{trimmed mean (10\%)}
  \item{\code{sd}}{standard deviation}
  \item{\code{mad}}{mean absolute deviation}
  \item{\code{skewness}}{skewness (from \code{e1701})}
  \item{\code{kurtosis}}{kurtosis (from \code{e1071})}
  \item{\code{min}}{minimum}
  \item{\code{Q_25}}{25\% quantile}
  \item{\code{Q_50}}{50\% quantile (a.k.a., the median)}
  \item{\code{Q_75}}{75\% quantile}
  \item{\code{max}}{maximum}
 }}

\item{discrete}{logical; include summary statistics for \code{discrete}
variables only? If \code{TRUE} then only count and proportion
information will be returned}
}
\description{
Function returns univariate data summaries for each variable supplied, however
discrete and continuous variables are treated separately. Structure provides
a more pipe-friendly API for selecting and subsetting variables using the
\code{dplyr} syntax, however conditional statistics are evaluated internally using the
\code{\link{by}} function. Quantitative/continuous variable
information is kept distinct in the output, while discrete variables (e.g.,
\code{factors} and \code{character} vectors)
can be returned by using the \code{discrete} argument.
}
\details{
\emph{Conditioning}: As the function is intended to support
pipe-friendly code specifications, conditioning/group subset
specifications are declared using \code{\link[dplyr]{group_by}}
and subsequently passed to \code{descript}. This is true
of all the verbs available in \code{dplyr}.
}
\examples{

library(dplyr)

data(mtcars)

if(FALSE){
  # run the following to see behavior with NA values in dataset
  mtcars[sample(1:nrow(mtcars), 3), 'cyl'] <- NA
  mtcars[sample(1:nrow(mtcars), 5), 'mpg'] <- NA
}

fmtcars <- within(mtcars, {
	cyl <- factor(cyl)
	am <- factor(am, labels=c('automatic', 'manual'))
	vs <- factor(vs)
})

# with and without factor variables
mtcars |> descript()
fmtcars |> descript()               # factors/discrete vars omitted
fmtcars |> descript(discrete=TRUE)  # discrete variables only

# usual pipe chaining
fmtcars |> select(mpg, wt) |> descript()
fmtcars |> filter(mpg > 20) |> select(mpg, wt) |> descript()

# conditioning with group_by()
fmtcars |> group_by(cyl) |> descript()
fmtcars |> group_by(cyl, am) |> descript()

# conditioning also works with group_by()
fmtcars |> group_by(cyl) |> descript(discrete=TRUE)
fmtcars |> group_by(am) |> descript(discrete=TRUE)
fmtcars |> group_by(cyl, am) |> descript(discrete=TRUE)

# only return a subset of summary statistics
funs <- get_descriptFuns()
sfuns <- funs[c('mean', 'sd')] # subset
fmtcars |> descript(funs=sfuns) # only mean/sd

# add a new functions
funs2 <- c(sfuns,
           Q_5 = \(x) quantile(x, .05, na.rm=TRUE),
           median= \(x) median(x, na.rm=TRUE),
           Q_95 = \(x) quantile(x, .95, na.rm=TRUE))
fmtcars |> descript(funs=funs2)

}
\seealso{
\code{\link[dplyr]{summarise}}, \code{\link[dplyr]{group_by}}
}
