% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ssn_assemble.R
\name{ssn_assemble}
\alias{ssn_assemble}
\title{Assemble an \code{SSN} object from an LSN}
\usage{
ssn_assemble(
  edges,
  lsn_path = NULL,
  obs_sites = NULL,
  preds_list = NULL,
  ssn_path,
  import = TRUE,
  check = TRUE,
  afv_col = NULL,
  overwrite = FALSE,
  verbose = TRUE
)
}
\arguments{
\item{edges}{An \code{sf} object with LINESTING geometry created
using \code{\link{lines_to_lsn}} (see Details).}

\item{lsn_path}{Local pathname to a directory in character format
specifying where relationships.csv resides, which is created
using \code{link{lines_to_lsn}}.}

\item{obs_sites}{Optional. A single \code{sf} object with POINT
geometry created using \code{link{sites_to_lsn}} that represents
the observation locations (i.e. where data were
collected). Default = NULL (see Details).}

\item{preds_list}{Optional. A list of one or more \code{sf} objects
representing prediction sites.}

\item{ssn_path}{Pathname to an output directory where output files
will be stored. A .ssn extension will be added if it is not
included.}

\item{import}{Logical indicating whether the output files should be
returned as an \code{SSN} object. Defaults to \code{TRUE}.}

\item{check}{Logical indicating whether the validity of the
\code{SSN} should be checked using \code{[ssn_check]} when both
\code{import = TRUE} and \code{verbose = TRUE}. Default = \code{TRUE}.}

\item{afv_col}{Character vector containing the names of the
additive function value columns that will be checked when
\code{check = TRUE}. Columns must be present in \code{edges},
\code{obs_sites} and \code{preds_list}, if all are
included. Default is \code{NULL}.}

\item{overwrite}{Logical. If \code{TRUE} and \code{ssn_path}
already exists, the contents of \code{ssn_path} will be
overwritten. Defaults to \code{FALSE}.}

\item{verbose}{Logical. Indicates whether messages about the
function progress and object validity check (when \code{check = TRUE} should be printed to the console. Defaults to
\code{TRUE}.}
}
\value{
The components of an \code{SSN} object are written to
\code{ssn_path} (see Details). When \code{import = TRUE}, the
function also returns an object of class \code{SSN}. If
\code{check = TRUE} and \code{verbose = TRUE}, the validity of the returned \code{SSN}
object is checked using \code{[ssn_check]} and results are
printed to the console.
}
\description{
Create an \code{SSN} (spatial stream network) object from a Landscape Network (LSN).
}
\details{
The \code{SSNbler} package is used to generate the
spatial, topological, and attribute information needed to fit
spatial stream-network models using the 'SSN2' package. The
\code{ssn_assemble} function will often be the final step in the
'SSNbler' data-processing workflow and it is important that
the previous processing steps have been followed. Prior to
running \code{ssn_assemble}, the \code{edges} must be processed
using \code{link{lines_to_lsn}}, \code{link{updist_edges}}, and
\code{link{afv_edges}}. The \code{obs_sites} and prediction site
datasets in \code{preds_list} must be processed with
\code{link{sites_to_lsn}}, \code{link{updist_sites}}, and
\code{link{afv_sites}}. In addition, the \code{edges},
\code{obs_sites}, and all of the \code{sf} objects in
\code{preds_list} must be part of the same LSN.

The \code{obs_sites} and \code{preds_list} are optional arguments,
with the Default = NULL. If \code{obs_sites = NULL}, an
\code{SSN} object will be returned with NA stored in
\code{ssn.object$obs} and a warning returned that
\code{ssn.object$obs} is required for fitting spatial statistical
models in 'SSN2'.

\code{ssn_assemble} stores the output locally in \code{ssn_path}. If
\code{ssn_path} does not include the .ssn extension, it is added
before the new directory is created. This directory contains:
\itemize{
\item edges.gpkg: edges in GeoPackage format. A new network identifier, netID, is added that is unique to each subnetwork.
\item sites.gpkg: observed sites in GeoPackage format (if present). Three new ID columns are added that are unique to the measurement (pid), the location (locID), and the network (netID).
\item prediction datasets in GeoPackage format (if present). The prediction sites also contain pid, locID, and netID. The naming convention is taken from the names provided in \code{preds_list}.
\item netID.dat files for each distinct network, which store the binaryID values for line segments in edges.
}
A more detailed description of the .ssn directory and its contents is provided in Peterson and Ver Hoef (2014).
}
\examples{
# Get temporary directory, where the example LSN will be stored
# locally.
temp_dir <- tempdir()
# Build the LSN. When working with your own data, lsn_path will be
# a local folder of your choice rather than a temporary directory.
edges<- lines_to_lsn(
   streams = MF_streams,
   lsn_path = temp_dir,
   snap_tolerance = 1,
   check_topology = FALSE,
   overwrite = TRUE,
   verbose = FALSE
)

# Incorporate observed sites, MF_obs, into LSN
obs<- sites_to_lsn(
   sites = MF_obs,
   edges = edges,
   save_local = FALSE,
   snap_tolerance = 100,
   overwrite = TRUE,
   verbose = FALSE
)

# Incorporate prediction dataset, MF_preds, into LSN
preds<- sites_to_lsn(sites = MF_preds,
   edges = edges,
   save_local = FALSE,
   snap_tolerance = 1,
   overwrite = TRUE,
   verbose = FALSE
)

# Calculate the AFV for the edges using
# a column representing watershed area (h2oAreaKm2).
edges<- afv_edges(
   edges=edges,
   infl_col = "h2oAreaKm2",
   segpi_col = "areaPI",
   lsn_path = temp_dir,
   afv_col = "afvArea",
   overwrite = TRUE,
   save_local = FALSE
)

# Calculate the AFV for observed sites (obs) and prediction
# dataset, preds.
site.list<- afv_sites(
   sites = list(obs = obs,
                preds = preds),
   edges=edges,
   afv_col = "afvArea",
   save_local = FALSE,
   overwrite = TRUE
)

# Calculate upstream distance for edges
edges<- updist_edges(
   edges = edges,
   lsn_path = temp_dir,
   calc_length = TRUE,
   length_col = "Length",
   overwrite = TRUE,
   save_local = FALSE,
   verbose = FALSE
)

# Calculate upstream distance for observed sites (obs) and one
# prediction dataset (preds)
site.list<- updist_sites(
   sites = site.list,
   edges = edges,
   length_col= "Length",
   lsn_path = temp_dir,
   save_local = FALSE,
   overwrite = TRUE
)

# Assemble SSN object
ssn.obj<- ssn_assemble(
   edges = edges,
   lsn_path = temp_dir,
   obs_sites = site.list[["obs"]],
   preds_list = site.list[c("preds")],
   ssn_path = paste0(temp_dir, "/example.ssn"),
   import = TRUE,
   overwrite = TRUE
)

# Summarise SSN object
summary(ssn.obj)

}
