% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CDL.R
\name{CDL}
\alias{CDL}
\title{Compressive Dictionary Learning}
\usage{
CDL(
  Data,
  K,
  SK_Data = NULL,
  Frequencies = NULL,
  D = NULL,
  pos.dic = TRUE,
  learn_rate = 0.1,
  alpha = 0.9,
  gamma = 0,
  maxEpoch = 5,
  batch_size,
  lambda = 0,
  ncores = 1,
  typeOptim = "Nesterov",
  DIR_tmp = tempdir(),
  grad_t_1 = NULL,
  verbose = 0,
  m = nrow(Frequencies),
  ...
)
}
\arguments{
\item{Data}{is a Filebacked Big Matrix \eqn{s \times N} with data vectors stored in the matrix columns.}

\item{K}{is a dictionary size.}

\item{SK_Data}{is a data sketch. It is a \eqn{2m}-dimensional complex vector. The first \eqn{m} coordinates
correspond to the real parts and the last \eqn{m} coordinates to the imaginary parts.
If it is NULL, the sketch is computed using \code{Sketch} function of
\href{https://CRAN.R-project.org/package=chickn}{chickn} package.}

\item{Frequencies}{is a frequency matrix \eqn{m \times s} with frequency vectors in the matrix rows.
If NULL, the frequencies are generated using \code{GenerateFrequencies}
function of \href{https://CRAN.R-project.org/package=chickn}{chickn} package.}

\item{D}{is an initial dictionary. If it is NULL, the dictionary is initialized by
random selection of \code{K} signals from \code{Data} and it is saved in the \code{DIR_tmp} directory.}

\item{pos.dic}{indicates whether the dictionary is positive (default) or not.}

\item{learn_rate}{is a learning rate value. The default value is 0.1.}

\item{alpha}{is a momentum weight.}

\item{gamma}{is a decay parameter. The default value is 0, which corresponds to the constant learning rate.}

\item{maxEpoch}{is a number of epochs.}

\item{batch_size}{is a batch size.}

\item{lambda}{is a regularization parameter.}

\item{ncores}{is a number of cores. The default value is 1.}

\item{typeOptim}{is a type of the optimization scheme used in the dictionary update.
Possible values are c('Nesterov', 'Momentum'). It is suggested to use 'Nesterov' scheme.}

\item{DIR_tmp}{is a directory to save the initial dictionary and intermediate results.}

\item{grad_t_1}{is an initial momentum matrix. By default it is NULL, and it is initialized as a zero matrix.}

\item{verbose}{controls how much output is shown and saved during the optimization process. Possible values:
\itemize{
\item 0 no output (default value)
\item 1 show iteration number and value of objective function
\item 2 1 + save a dictionary and a momentum matrix at the end of each epoch.
}}

\item{m}{is a number of the frequency vectors.}

\item{...}{are additional parameters passed to \link[chickn]{GenerateFrequencies} function.}
}
\value{
a list
\itemize{
\item \code{D} is the obtained dictionary,
\item  \code{objFunProcess} is objective function values computed at the end of each iteration,
\item \code{learning_rate} is learning rate values.
}
}
\description{
Implementation of the Sketched Stochastic Dictionary Learning (SSDL) method, which
learns a dictionary from a large-scale matrix-like dataset by operating on a compressed version of data (a.k.a. data sketch).
}
\details{
\code{CDL} builds a dictionary by alternating two steps:
calculating the code matrix that contains the weights of the dictionary elements,
and updating the dictionary. For computational efficiency, the code matrix is
computed only for a randomly selected subset of data vectors \eqn{x_1, \dots, x_n}
(a.k.a. batch). The code matrix is obtained as a solution of the following optimization problem:
\eqn{\min\limits_{A \in R^{+}_{K\times n}} \sum_{i=1}^{n} \|x_i - D\cdot \alpha_i\|^2 + 
\lambda \cdot\|\alpha_i\|_1}, where \eqn{n} denotes a batch size,
\eqn{A = \{\alpha_1,\dots, \alpha_{n}\}} is a code matrix and \eqn{\lambda} is a regularization parameter which
defines the data sparsity level.

The dictionary is updated by taking one step along the gradient of the objective function
\eqn{F(D, A) = \|SK(Data) - SK(A\cdot D)\|^2}.
Two gradient descent update rules are available: Nesterov accelerated and momentum.

\eqn{SK(\cdot)} is a sketch operator, which compresses a matrix
into a fixed size complex vector referred to as a data sketch.  It has been introduced in \insertRef{DBLP:journals/corr/KerivenBGP16}{SSDL} and
it is defined as \eqn{SK(Data) = \frac{1}{N}\sum_{i=1}^N \exp{\left(-1i \cdot W\cdot x_i\right)}},
where \eqn{W} is a frequency matrix and \eqn{x_1, \dots, x_N} are data vectors.
The data compression is performed using routines from \href{https://CRAN.R-project.org/package=chickn}{chickn} package.

\code{CDL} allows also to use the decaying learning rate, \emph{i.e.}
\eqn{\code{learn\_rate}^t = \frac{\code{learn\_rate}}{1+ (t-1)\cdot \code{gamma}}}, where \eqn{t} is the iteration number.
}
\examples{
\donttest{
X = matrix(abs(rnorm(n = 1000)), ncol = 100, nrow = 10)
X_fbm = bigstatsr::as_FBM(X)$save()
W = chickn::GenerateFrequencies(Data = X_fbm, m = 64, N0 = ncol(X_fbm),
                                ncores = 1, niter= 3, nblocks = 2, sigma_start = 0.001)$W
SK= chickn::Sketch(X_fbm, W)
D = CDL(Data = X_fbm, K = 10, SK_Data = SK, Frequencies = W,
        D = NULL, pos.dic = TRUE, maxEpoch = 3, batch_size = 50, 
        lambda = 0, learn_rate = 0.1, alpha = 0.9, 
        gamma = 0, ncores = 2, DIR_tmp = tempdir(), 
        verbose=0, typeOptim = "Nesterov")$D
}                   
}
\references{
\itemize{
\item \insertRef{permiakova2021SSDL}{SSDL}
\item \insertRef{permiakova2021chickn}{SSDL}}
}
\seealso{
\code{\link{Gradient_D_cpp_parallel}}, \href{https://CRAN.R-project.org/package=chickn}{chickn},
\code{chickn::Sketch}, \code{chickn::GenerateFrequencies}
}
