% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/features_10_minor_revisions.R
\name{claim_minRev}
\alias{claim_minRev}
\alias{claim_minRev_freq}
\alias{claim_minRev_time}
\alias{claim_minRev_size}
\title{Minor Revisions of Outstanding Claim Payments}
\usage{
claim_minRev_freq(
  claims,
  prob_atP = 0.5,
  rfun_notatP,
  paramfun_notatP,
  frequency_vector = claims$frequency_vector,
  settlement_list = claims$settlement_list,
  no_payments_list = claims$no_payments_list,
  ...
)

claim_minRev_time(
  claims,
  minRev_list,
  rfun_notatP,
  paramfun_notatP,
  settlement_list = claims$settlement_list,
  payment_delay_list = claims$payment_delay_list,
  ...
)

claim_minRev_size(
  claims,
  majRev_list,
  minRev_list,
  rfun,
  paramfun_atP,
  paramfun_notatP,
  settlement_list = claims$settlement_list,
  ...
)
}
\arguments{
\item{claims}{an \code{claims} object containing all the simulated quantities
(other than those related to incurred loss), see
\code{\link[SynthETIC]{claims}}.}

\item{prob_atP}{(optional) probability that a minor revision will occur at
the time of a partial payment; default value 0.5.}

\item{rfun_notatP}{optional alternative random sampling function for:
\itemize{
\item \code{claim_minRev_freq}: the number of minor revisions that occur at an epoch
other than those of partial payments;
\item \code{claim_minRev_time}: the epochs of such minor revisions measured from claim
notification;
\item \code{claim_minRev_size}: the sizes of the minor revision multipliers (common for
\verb{_atP} and \verb{_notatP}, hence simply termed \code{rfun} in this case).
}

See Details for default.}

\item{paramfun_notatP}{parameters for the above random sampling function,
as a function of other claim characteristics (e.g. \code{lambda} as a function of
\code{claim_size} for an \code{rpois} simulation); see Examples.}

\item{frequency_vector}{a vector of claim frequencies for all the periods
(not required if the \code{claims} argument is provided); see
\code{\link[SynthETIC]{claim_frequency}}.}

\item{settlement_list}{list of settlement delays (not required if the
\code{claims} argument is provided); see \code{\link[SynthETIC]{claim_closure}}.}

\item{no_payments_list}{list of number of partial payments (not required if
the \code{claims} argument is provided); see
\code{\link[SynthETIC]{claim_payment_no}}.}

\item{...}{other arguments/parameters to be passed onto \code{paramfun}.}

\item{minRev_list}{nested list of minor revision histories (with non-empty
revision frequencies).}

\item{payment_delay_list}{(compound) list of inter partial delays (not
required if the \code{claims} argument is provided); see
\code{\link[SynthETIC]{claim_payment_delay}}.}

\item{majRev_list}{nested list of major revision histories (with non-empty
revision frequencies).}

\item{rfun}{optional alternative random sampling function for the sizes of
the minor revision multipliers (common for \verb{_atP} and \verb{_notatP}, hence simply
termed \code{rfun} in this case).}

\item{paramfun_atP}{parameters for \code{rfun} in \code{claim_minRev_size()} for minor
revisions that occur at the time of a partial payment.}
}
\value{
A nested list structure such that the \emph{j}th component of the \emph{i}th
sub-list is a list of information on minor revisions of the \emph{j}th claim of
occurrence period \emph{i}. The "unit list" (i.e. the smallest, innermost
sub-list) contains the following components:
\tabular{ll}{
\code{minRev_atP} \tab A vector of indicators showing whether there is a minor
revision at each partial payment [\code{claim_minRev_freq()}]. \cr
\code{minRev_freq_atP} \tab Number of minor revisions that occur simultaneously with
a partial payment, numerically equals to the sum of \code{minRev_atP}
[\code{claim_minRev_freq()}]. \cr
\code{minRev_freq_notatP} \tab Number of minor revisions that do not occur with a
partial payment [\code{claim_minRev_freq()}]. \cr
\code{minRev_time_atP} \tab Time of minor revisions that occur simultaneously with
a partial payment (time measured from claim notification)
[\code{claim_minRev_time()}]. \cr
\code{minRev_time_notatP} \tab Time of minor revisions that do \emph{not} occur
simultaneously with a partial payment (time measured from claim notification)
[\code{claim_minRev_time()}]. \cr
\code{minRev_factor_atP} \tab Minor revision multipliers of \strong{outstanding claim
payments} for revisions at partial payments [\code{claim_minRev_size()}]. \cr
\code{minRev_factor_notatP} \tab Minor revision multipliers of \strong{outstanding claim
payments} for revisions at any other times [\code{claim_minRev_size()}]. \cr
}
}
\description{
A suite of functions that works together to simulate, in order,
the (1) frequency, (2) time, and (3) size of minor revisions of outstanding
claim payments, for each of the claims occurring in each of the periods.

We separate the case of minor revisions that occur simultaneously with a
partial payment (denoted \verb{_atP}), and the ones that do not coincide with a
payment (denoted \verb{_notatP}).
}
\section{Details - \code{claim_minRev_freq} (Frequency)}{
 Minor revisions may occur
simultaneously with a partial payment, or at any other time.

For the former case, we sample the occurrence of minor revisions as Bernoulli
random variables with default probability parameter \code{prob_atP} \eqn{= 1/2}.

For the latter case, by default we sample the number of (non payment
simultaneous) minor revisions from a geometric distribution with mean =
\eqn{min(3, setldel / 4)}.

One can modify the above sampling distributions by plugging in their own
\code{prob_atP} parameter and \code{rfun_notatP} function, where the former dictates
the probability of incurring a minor revision at the time of a payment, and
the latter simulates and returns the number of minor revisions at any other
points in time, with possible dependence on the settlement delay of the claim
and/or other claim characteristics.
}

\section{Details - \code{claim_minRev_time} (Time)}{
 For minor revisions that occur
simultaneously with a partial payment, the revision times simply coincide
with the epochs of the relevant partial payments.

For minor revisions that occur at a different time, by default the revision
times are sampled from a uniform distribution with parameters \code{min} \eqn{=
settlement_delay / 6} and \code{max} \eqn{= settlement_delay}.

One can modify the above sampling distribution by plugging in their own
\code{rfun_notatP} and \code{paramfun_notatP} in \code{claim_minRev_time()}, which together
simulate the epochs of minor revisions that do not coincide with a payment,
with possible dependence on the settlement delay of the claim and/or other
claim characteristics (see Examples).
}

\section{Details - \code{claim_minRev_size} (Revision Multiplier)}{
 The sampling
distribution for minor revision multipliers is the same for both revisions
that occur with and without a partial payment. In the default setting, we
incorporate sampling dependence on the delay from notification to settlement
(\code{setldel}), the delay from notification to the subject minor revisions
(\code{minRev_time}), and the history of major revisions (in particular, the time
of the second major revision).

Let \eqn{\tau} denote the delay from notification to the epoch of the minor
revision, and \eqn{w} the settlement delay. Then
\itemize{
\item For \eqn{\tau \le w / 3}, the revision multiplier is sampled from a
lognormal distribution with parameters \code{meanlog} \eqn{= 0.15} and \code{sdlog}
\eqn{= 0.05} if preceded by a 2nd major revision, \code{sdlog} \eqn{= 0.1}
otherwise;
\item For \eqn{w / 3 < \tau \le 2w / 3}, the revision multiplier is sampled
from a lognormal distribution with parameters \code{meanlog} \eqn{= 0} and \code{sdlog}
\eqn{= 0.05} if preceded by a 2nd major revision, \code{sdlog} \eqn{= 0.1}
otherwise;
\item For \eqn{\tau > 2w / 3}, the revision multiplier is sampled from a
lognormal distribution with parameters \code{meanlog} \eqn{= -0.1} and \code{sdlog}
\eqn{= 0.05} if preceded by a 2nd major revision, \code{sdlog} \eqn{= 0.1}
otherwise.
}

Note that minor revisions tend to be upward in the early part of a claim’s
life, and downward in the latter part.

The revision multipliers are subject to further constraints to ensure that
the revised incurred estimate never falls below what has already been paid.
This is dicussed in \code{\link{claim_history}}.

\strong{Important note:} Unlike the major revision multipliers which apply to the
\strong{incurred loss estimates}, the minor revision multipliers apply to the case
estimate of \strong{outstanding claim payments} i.e. a revision multiplier of 2.54
means that at the time of the minor revision the outstanding claims payment
increases by a factor of 2.54.
}

\examples{
set.seed(1)
test_claims <- SynthETIC::test_claims_object

# generate major revisions (required for the simulation of minor revisions)
major <- claim_majRev_freq(test_claims)
major <- claim_majRev_time(test_claims, major)
major <- claim_majRev_size(major)

# generate frequency of minor revisions
minor <- claim_minRev_freq(test_claims)
minor[[1]][[1]] # the "unit list" for the first claim

# update the timing information
minor <- claim_minRev_time(test_claims, minor)
# observe how this has changed
minor[[1]][[1]]
# with an alternative sampling distribution e.g. triangular
minRev_time_notatP <- function(n, setldel) {
  sort(rtri(n, min = setldel/6, max = setldel, mode = setldel))
}
minor_2 <- claim_minRev_time(test_claims, minor, minRev_time_notatP)

# update the revision multipliers (need to generate "major" first)
minor <- claim_minRev_size(test_claims, major, minor)
}
\seealso{
\code{\link[SynthETIC]{claims}}
}
