% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulate_one_trial.R
\name{simulate_one_trial}
\alias{simulate_one_trial}
\title{Simulate One Trial Using an Obtained Optimal Adaptive Allocation Rule}
\usage{
simulate_one_trial(
  allocation_rule,
  models,
  true_response,
  N_total,
  N_ini,
  N_block,
  Delta,
  outcome_type = c("continuous", "binary"),
  sd_normal = NULL,
  alpha = 0.025,
  selModel = c("AIC", "maxT", "aveAIC"),
  seed = NULL,
  eval_type = c("all", "pVal")
)
}
\arguments{
\item{allocation_rule}{An object of class \link[RLoptimal]{AllocationRule}
specifying an obtained optimal adaptive allocation rule.}

\item{models}{An object of class \link[DoseFinding]{Mods} specifying assumed
dose-response models. When \code{outcome_type} is "binary", \code{models} should
be specified on the logit scale. This is used in the MCPMod method
at the end of this trial.}

\item{true_response}{A numeric vector specifying the true response values of
the true model. When \code{outcome_type} is "binary", \code{true_response} should
be specified on the logit scale.}

\item{N_total}{A positive integer value. The total number of subjects.}

\item{N_ini}{A positive integer vector in which each element is greater than
or equal to 2. The number of subjects initially assigned to each dose.}

\item{N_block}{A positive integer value. The number of subjects allocated
adaptively in each round.}

\item{Delta}{A positive numeric value. The clinically relevant target effect.
When \code{outcome_type} is "binary", \code{Delta} should be specified
on the logit scale. See \link[DoseFinding]{TD} for details.}

\item{outcome_type}{A character value specifying the outcome type.
Possible values are "continuous" (default), and "binary".}

\item{sd_normal}{A positive numeric value. The standard deviation of the
observation noise. When \code{outcome_type} is "continuous",
\code{sd_normal} must be specified.}

\item{alpha}{A positive numeric value. The significance level. Default is 0.025.}

\item{selModel}{A character value specifying the model selection criterion
for dose estimation. Possible values are "AIC" (default), "maxT", or
"aveAIC". See \link[DoseFinding]{MCPMod} for details.}

\item{seed}{An integer value. Random seed for data generation in this trial.}

\item{eval_type}{A character value specifying the evaluation type. Possible
values are "all" (default) and "pVal". "all" returns all metrics,
which contain the minimum p value, the selected model name,
the estimated target dose, and the MAE. "pVal" returns only the
minimum p value without fitting models.}
}
\value{
A list which contains the minimum p value, the selected model name,
the estimated target dose, and the MAE.
}
\description{
Simulate One Trial Using an Obtained Optimal Adaptive Allocation Rule
}
\examples{
library(RLoptimal)

doses <- c(0, 2, 4, 6, 8)

models <- DoseFinding::Mods(
  doses = doses, maxEff = 1.65,
  linear = NULL, emax = 0.79, sigEmax = c(4, 5)
)

\dontrun{
allocation_rule <- learn_allocation_rule(
  models,
  N_total = 150, N_ini = rep(10, 5), N_block = 10, Delta = 1.3,
  outcome_type = "continuous", sd_normal = sqrt(4.5), 
  seed = 123, rl_config = rl_config_set(iter = 1000),
  alpha = 0.025
)

# Simulation-based adjustment of the significance level using `allocation_rule`
adjusted_alpha <- adjust_significance_level(
  allocation_rule, models,
  N_total = 150, N_ini = rep(10, 5), N_block = 10,
  outcome_type = "continuous", sd_normal = sqrt(4.5),
  alpha = 0.025, n_sim = 10000, seed = 123
)}

eval_models <- DoseFinding::Mods(
  doses = doses, maxEff = 1.65,
  linear = NULL, emax = 0.79, sigEmax = c(4, 5), exponential = 1, quadratic = - 1/12
)
true_response_matrix <- DoseFinding::getResp(eval_models, doses = doses)
true_response_list <- as.list(data.frame(true_response_matrix, check.names = FALSE))

true_model_name <- "emax"

# Simulate one trial using the obtained `allocation_rule` When the true model is "emax"
\dontrun{
res_one <- simulate_one_trial(
  allocation_rule, models, 
  true_response = true_response_list[[true_model_name]],
  N_total = 150, N_ini = rep(10, 5), N_block = 10, 
  Delta = 1.3, outcome_type = "continuous", sd_normal = sqrt(4.5),
  alpha = adjusted_alpha, seed = simID, eval_type = "all"
)}

}
