% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/resi.R
\name{resi}
\alias{resi}
\alias{resi.default}
\alias{resi.glm}
\alias{resi.lm}
\alias{resi.nls}
\alias{resi.survreg}
\alias{resi.coxph}
\alias{resi.hurdle}
\alias{resi.zeroinfl}
\alias{resi.geeglm}
\alias{resi.gee}
\alias{resi.lme}
\alias{resi.lmerMod}
\title{Robust Effect Size Index (RESI) point and interval estimation for models}
\usage{
resi(model.full, ...)

\method{resi}{default}(
  model.full,
  model.reduced = NULL,
  data,
  anova = TRUE,
  coefficients = TRUE,
  overall = TRUE,
  nboot = 1000,
  boot.method = "nonparam",
  vcovfunc = sandwich::vcovHC,
  alpha = 0.05,
  store.boot = FALSE,
  Anova.args = list(),
  vcov.args = list(),
  unbiased = TRUE,
  parallel = c("no", "multicore", "snow"),
  ncpus = getOption("boot.ncpus", 1L),
  long = FALSE,
  clvar = NULL,
  ...
)

\method{resi}{glm}(
  model.full,
  model.reduced = NULL,
  data,
  anova = TRUE,
  coefficients = TRUE,
  overall = TRUE,
  nboot = 1000,
  vcovfunc = sandwich::vcovHC,
  alpha = 0.05,
  store.boot = FALSE,
  Anova.args = list(),
  vcov.args = list(),
  unbiased = TRUE,
  parallel = c("no", "multicore", "snow"),
  ncpus = getOption("boot.ncpus", 1L),
  ...
)

\method{resi}{lm}(
  model.full,
  model.reduced = NULL,
  data,
  anova = TRUE,
  coefficients = TRUE,
  overall = TRUE,
  nboot = 1000,
  boot.method = "nonparam",
  vcovfunc = sandwich::vcovHC,
  alpha = 0.05,
  store.boot = FALSE,
  Anova.args = list(),
  vcov.args = list(),
  unbiased = TRUE,
  parallel = c("no", "multicore", "snow"),
  ncpus = getOption("boot.ncpus", 1L),
  ...
)

\method{resi}{nls}(
  model.full,
  model.reduced = NULL,
  data,
  coefficients = TRUE,
  overall = TRUE,
  nboot = 1000,
  boot.method = "nonparam",
  anova = FALSE,
  vcovfunc = r_nlshc,
  alpha = 0.05,
  store.boot = FALSE,
  vcov.args = list(),
  unbiased = TRUE,
  parallel = c("no", "multicore", "snow"),
  ncpus = getOption("boot.ncpus", 1L),
  ...
)

\method{resi}{survreg}(
  model.full,
  model.reduced = NULL,
  data,
  anova = TRUE,
  coefficients = TRUE,
  overall = TRUE,
  nboot = 1000,
  vcovfunc = vcov,
  alpha = 0.05,
  store.boot = FALSE,
  Anova.args = list(),
  unbiased = TRUE,
  parallel = c("no", "multicore", "snow"),
  ncpus = getOption("boot.ncpus", 1L),
  ...
)

\method{resi}{coxph}(
  model.full,
  model.reduced = NULL,
  data,
  anova = TRUE,
  coefficients = TRUE,
  overall = TRUE,
  nboot = 1000,
  vcovfunc = vcov,
  alpha = 0.05,
  store.boot = FALSE,
  Anova.args = list(),
  unbiased = TRUE,
  parallel = c("no", "multicore", "snow"),
  ncpus = getOption("boot.ncpus", 1L),
  ...
)

\method{resi}{hurdle}(
  model.full,
  model.reduced = NULL,
  data,
  coefficients = TRUE,
  overall = TRUE,
  nboot = 1000,
  vcovfunc = sandwich::sandwich,
  anova = FALSE,
  alpha = 0.05,
  store.boot = FALSE,
  vcov.args = list(),
  unbiased = TRUE,
  parallel = c("no", "multicore", "snow"),
  ncpus = getOption("boot.ncpus", 1L),
  ...
)

\method{resi}{zeroinfl}(
  model.full,
  model.reduced = NULL,
  data,
  coefficients = TRUE,
  overall = TRUE,
  nboot = 1000,
  vcovfunc = sandwich::sandwich,
  anova = FALSE,
  alpha = 0.05,
  store.boot = FALSE,
  vcov.args = list(),
  unbiased = TRUE,
  parallel = c("no", "multicore", "snow"),
  ncpus = getOption("boot.ncpus", 1L),
  ...
)

\method{resi}{geeglm}(
  model.full,
  model.reduced = NULL,
  data,
  anova = TRUE,
  coefficients = TRUE,
  overall = TRUE,
  nboot = 1000,
  alpha = 0.05,
  store.boot = FALSE,
  unbiased = TRUE,
  parallel = c("no", "multicore", "snow"),
  ncpus = getOption("boot.ncpus", 1L),
  ...
)

\method{resi}{gee}(
  model.full,
  data,
  nboot = 1000,
  alpha = 0.05,
  store.boot = FALSE,
  unbiased = TRUE,
  parallel = c("no", "multicore", "snow"),
  ncpus = getOption("boot.ncpus", 1L),
  ...
)

\method{resi}{lme}(
  model.full,
  alpha = 0.05,
  nboot = 1000,
  anova = TRUE,
  vcovfunc = clubSandwich::vcovCR,
  vcov.args = list(),
  ...
)

\method{resi}{lmerMod}(
  model.full,
  alpha = 0.05,
  nboot = 1000,
  anova = TRUE,
  vcovfunc = clubSandwich::vcovCR,
  vcov.args = list(),
  ...
)
}
\arguments{
\item{model.full}{\code{lm, glm, nls, survreg, coxph, hurdle, zeroinfl, gee, geeglm} or \code{lme} model object.}

\item{...}{Ignored.}

\item{model.reduced}{Fitted model object of same type as model.full. By default `NULL`; the same model as the full model but only having intercept.}

\item{data}{Data.frame or object coercible to data.frame of model.full data (required for some model types).}

\item{anova}{Logical, whether to produce an Anova table with the RESI columns added. By default = `TRUE`.}

\item{coefficients}{Logical, whether to produce a coefficients (summary) table with the RESI columns added. By default = `TRUE`.}

\item{overall}{Logical, whether to produce an overall Wald test comparing full to reduced model with RESI columns added. By default = `TRUE`.}

\item{nboot}{Numeric, the number of bootstrap replicates. By default, 1000.}

\item{boot.method}{String, which type of bootstrap to use: `nonparam` = non-parametric bootstrap (default); `bayes` = Bayesian bootstrap.}

\item{vcovfunc}{The variance estimator function for constructing the Wald test statistic. By default, sandwich::vcovHC (the robust (sandwich) variance estimator).}

\item{alpha}{Numeric, significance level of the constructed CIs. By default, 0.05.}

\item{store.boot}{Logical, whether to store all the bootstrapped estimates. By default, `FALSE`.}

\item{Anova.args}{List, additional arguments to be passed to \link[car]{Anova} function.}

\item{vcov.args}{List, additional arguments to be passed to vcovfunc.}

\item{unbiased}{Logical, whether to use the unbiased or alternative T/Z statistic to RESI conversion. By default, `TRUE`. See details.}

\item{parallel}{See documentation for \code{\link{boot}}.}

\item{ncpus}{See documentation for \code{\link{boot}}.}

\item{long}{Logical, whether the data is longitudinal/clustered. By default, `FALSE`.}

\item{clvar}{Character, the name of the cluster/id variable if data is clustered. By default, `NULL`.}
}
\value{
Returns a list that includes function arguments, RESI point estimates,
and confidence intervals in coefficients/anova-style tables
}
\description{
This function will estimate the robust effect size (RESI) from Vandekar, Tao, & Blume (2020) and its confidence interval in various ways for a fitted model object. The overall RESI is estimated via a Wald test. RESI is (optionally) estimated for each factor in coefficients-style table. RESI is (optionally) estimated for each variable/interaction in an Anova-style table for models with existing Anova methods. CIs can be calculated using either non-parametric or Bayesian bootstrapping.
}
\details{
The RESI, denoted as S, is applicable across many model types. It is a unitless
index and can be easily be compared across models. The RESI can also be
converted to Cohen's \emph{d} (\code{\link{S2d}}) under  model homoskedasticity.

This function computes the RESI point estimates and bootstrapped confidence
intervals based on Chi-square, F, T, or Z statistics. The robust (sandwich)
variance is used by default, allowing for consistency under
model-misspecification. The RESI is related to the non-centrality parameter
of the test statistic. The RESI estimate is consistent for all four
(Chi-square, F, T, and Z) types of statistics used. The Chi-square and F-based
calculations rely on asymptotic theory, so they may be biased in small samples.
When possible, the T and Z statistics are used. There are two formulas for both
the T and Z statistic conversion. The first (default, unbiased = TRUE)
are based on solving the expected value of the T or Z statistic for the RESI.
The alternative is based on squaring the T or Z statistic and using the
F or Chi-square statistic conversion. Both of these methods are consistent, but
the alternative exhibits a notable amount of finite sample bias. The alternative
may be appealing because its absolute value will be equal to the RESI based on
the F or Chi-square statistic. The RESI based on the Chi-Square and F statistics
is always greater than or equal to 0. The type of statistic
used is listed with the output. See \code{\link{f2S}}, \code{\link{chisq2S}},
\code{\link{t2S}}, and \code{\link{z2S}} for more details on the formulas.

For GEE (\code{geeglm}) models, a longitudinal RESI (L-RESI) and a cross-sectional,
per-measurement RESI (CS-RESI) is estimated. The longitudinal RESI takes the
specified clustering into account, while the cross-sectional RESI is estimated
using a model where each measurement is its own cluster.

For most \code{lm} and \code{nls} model types, there is a Bayesian bootstrap
option available as an alternative to the default, standard non-parametric
bootstrap. The interpretation of a Bayesian bootstrapped interval is similar to
that of a credible interval.

Certain model types require the data used for the model be entered as an argument.
These are: \code{nls, survreg,} and \code{coxph}. Additionally, if a model
includes certain functions (splines, factor, I), the data needs to be provided.

If running into convergence issues with nls models, it is advised to refit the
nls model with starting values equal to the estimates provided by the model
and then try rerunning \code{resi}.
}
\section{Methods (by class)}{
\itemize{
\item \code{resi(default)}: RESI point and interval estimation for models

\item \code{resi(glm)}: RESI point and interval estimation for models

\item \code{resi(lm)}: RESI point and interval estimation for lm models

\item \code{resi(nls)}: RESI point and interval estimation for nls models

\item \code{resi(survreg)}: RESI point and interval estimation for survreg models

\item \code{resi(coxph)}: RESI point and interval estimation for coxph models

\item \code{resi(hurdle)}: RESI point and interval estimation for hurdle models

\item \code{resi(zeroinfl)}: RESI point and interval estimation for zeroinfl models

\item \code{resi(geeglm)}: RESI point and interval estimation for GEE models

\item \code{resi(gee)}: RESI point and interval estimation for GEE models

\item \code{resi(lme)}: RESI point and interval estimation for LME (nlme) models

\item \code{resi(lmerMod)}: RESI point and interval estimation for lmerMod models

}}
\examples{
## for timing purposes, a small number of bootstrap replicates is used in the
## examples. Run them with a higher or default `nboot` argument for better performance

## RESI on a linear model
# fit linear model
mod = lm(charges ~ region * age + bmi + sex, data = RESI::insurance)

# run resi on fitted model with desired number of bootstrap replicates
# store bootstrap results for calculating different CIs later
resi_obj = resi(mod, nboot = 50, store.boot = TRUE)
# print output
resi_obj

# fit a reduced model for comparison
mod_red = lm(charges ~ bmi, data = RESI::insurance)

# running resi and including the reduced model will provide almost the exact same
# output as not including a reduced model. The difference is that the "overall"
# portion of the output will compare the full model to the reduced model.
# The "summary" and "anova" RESI estimates will be the same. (The bootstrapped
# confidence intervals may differ.)
resi(model.full = mod, model.reduced = mod_red, nboot = 10)

# used stored bootstrap results to get a different alpha-level confidence interval
summary(resi_obj, alpha = c(0.01, 0.1))
car::Anova(resi_obj, alpha = c(0.01, 0.1))

# the result of resi, as well as the summary or Anova of a `resi` object can be plotted
# if the resi object was created with the store.boot = `TRUE` option, any alpha
# can be specified
plot(resi_obj, alpha = 0.01)
# if the variable names on the y-axis are too long, you can reduce their size with
# the ycex.axis argument (or use regular common solutions like changing the margins)
plot(resi_obj, alpha = 0.01, ycex.axis = 0.5)

# for some model types and formula structures, data argument is required
if(requireNamespace("splines")){
  # fit logistic regression model with splines
  mod = glm(smoker ~ splines::ns(age, df = 3) + region, data = RESI::insurance,
    family = "binomial")

  # specify additional arguments to the variance-covariance function via vcov.args
  resi_obj = resi(mod, data = RESI::insurance, alpha = 0.01,
    vcov.args = list(type = "HC0"), nboot = 25)
  summary(resi_obj)
  car::Anova(resi_obj)}


## RESI on a survival model with alternate Z2S
if(requireNamespace("survival")){
  # fit coxph model on example data from survival package
  # Note: for survival models, you need to specify robust variance in the model
  # creation. resi will ignore the vcovfunc argument for this reason.
  mod.coxph =  survival::coxph(survival::Surv(time, status) ~ age + sex + wt.loss,
   data=survival::lung, robust = TRUE)

  # run resi on the model
  # to use the alternative Z to RESI formula (which is equal in absolute value to the
  # chi-square to RESI (S) formula), specify unbiased = FALSE.
  resi(mod.coxph, data = survival::lung, unbiased = FALSE, nboot = 10)}

}
\references{
Vandekar S, Tao R, Blume J. A Robust Effect Size Index. \emph{Psychometrika}. 2020 Mar;85(1):232-246. doi: 10.1007/s11336-020-09698-2.

Kang, K., Armstrong, K., Avery, S., McHugo, M., Heckers, S., & Vandekar, S. (2021). Accurate confidence interval estimation for non-centrality parameters and effect size indices. \emph{arXiv preprint arXiv:2111.05966}.
}
\seealso{
\code{\link{resi_pe}}, \code{\link{vcovHC}},
\code{\link{f2S}}, \code{\link{chisq2S}}, \code{\link{z2S}}, \code{\link{t2S}}
}
\concept{RESI functions}
