% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/QBAD.R
\name{QBAD}
\alias{QBAD}
\alias{dQBAD}
\alias{pQBAD}
\alias{qQBAD}
\alias{rQBAD}
\title{Quantile-based asymmetric family of distributions}
\usage{
dQBAD(y, mu, phi, alpha, f)

pQBAD(q, mu, phi, alpha, F)

qQBAD(beta, mu, phi, alpha, F, QF = NULL)

rQBAD(n, mu, phi, alpha, F, QF = NULL)
}
\arguments{
\item{y, q}{These are each a vector of quantiles.}

\item{mu}{This is the location parameter \eqn{\mu}.}

\item{phi}{This is the scale parameter  \eqn{\phi}.}

\item{alpha}{This is the index parameter  \eqn{\alpha}.}

\item{f}{This is the reference density function \eqn{f} which is a standard version of a unimodal and symmetric around 0 density.}

\item{F}{This is the cumulative distribution function \eqn{F} of a unimodal and symmetric around 0 reference density function \eqn{f}.}

\item{beta}{This is a vector of probabilities.}

\item{QF}{This is the quantile function of the reference density \eqn{f}.}

\item{n}{This is the number of observations, which must be a positive integer that has length 1.}
}
\value{
\code{\link{dQBAD}} provides the density, \code{\link{pQBAD}} provides the cumulative distribution function, \code{\link{qQBAD}} provides the quantile function, and \code{\link{rQBAD}} generates a random sample from the quantile-based asymmetric family of distributions.
The length of the result is determined by \eqn{n} for \code{\link{rQBAD}}, and is the maximum of the lengths of the numerical arguments for the other functions.
}
\description{
Density, cumulative distribution function, quantile function and random sample generation
from the quantile-based asymmetric family of densities defined in Gijbels et al. (2019a).
}
\examples{
# Example 1: Let F be a standard normal cumulative distribution function then
f_N<-function(s){dnorm(s, mean = 0,sd = 1)} # density function of N(0,1)
F_N<-function(s){pnorm(s, mean = 0,sd = 1)} # distribution function of N(0,1)
QF_N<-function(beta){qnorm(beta, mean = 0, sd = 1, lower.tail = TRUE, log.p = FALSE)}
rnum<-rnorm(100)
beta=c(0.25,0.50,0.75)

# Density
dQBAD(y=rnum,mu=0,phi=1,alpha=.5,f=f_N)

# Distribution function
pQBAD(q=rnum,mu=0,phi=1,alpha=.5,F=F_N)

# Quantile function
qQBAD(beta=beta,mu=0,phi=1,alpha=.5,F=F_N,QF=QF_N)
qQBAD(beta=beta,mu=0,phi=1,alpha=.5,F=F_N)

# random sample generation
rQBAD(n=100,mu=0,phi=1,alpha=.5,QF=QF_N)
rQBAD(n=100,mu=0,phi=1,alpha=.5,F=F_N)



# Example 2: Let F be a standard Laplace cumulative distribution function then
f_La<-function(s){0.5*exp(-abs(s))} # density function of Laplace(0,1)
F_La<-function(s){0.5+0.5*sign(s)*(1-exp(-abs(s)))} # distribution function of Laplace(0,1)
QF_La<-function(beta){-sign(beta-0.5)*log(1-2*abs(beta-0.5))}
rnum<-rnorm(100)
beta=c(0.25,0.50,0.75)

# Density
dQBAD(y=rnum,mu=0,phi=1,alpha=.5,f=f_La)

# Distribution function
pQBAD(q=rnum,mu=0,phi=1,alpha=.5,F=F_La)

# Quantile function
qQBAD(beta=c(0.25,0.50,0.75),mu=0,phi=1,alpha=.5,F=F_La,QF=QF_La)
qQBAD(beta=c(0.25,0.50,0.75),mu=0,phi=1,alpha=.5,F=F_La)

# random sample generation
rQBAD(n=100,mu=0,phi=1,alpha=.5,QF=QF_La)
rQBAD(n=100,mu=0,phi=1,alpha=.5,F=F_La)

}
\references{
{
 Gijbels, I., Karim, R. and Verhasselt, A. (2019a). On quantile-based asymmetric family of distributions: properties and inference. \emph{International Statistical Review}, \url{https://doi.org/10.1111/insr.12324}.
}
}
