% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/05_STEP_MIV.R
\name{stepMIV}
\alias{stepMIV}
\title{Stepwise logistic regression based on marginal information value (MIV)}
\usage{
stepMIV(start.model, miv.threshold, m.ch.p.val, coding, db)
}
\arguments{
\item{start.model}{Formula class that represent starting model. If can include some risk factors, but it can be
defined only with intercept (\code{y ~ 1} where \code{y} is target variable).}

\item{miv.threshold}{MIV entrance threshold. Only the risk factors with MIV higher than thresholds are candidate
for the new model. Additional criteria is that MIV value should significantly separate
good from bad cases measured by marginal chi-square test.}

\item{m.ch.p.val}{Significance level of p-value for marginal chi-square test. This test additionally supports MIV value of
of candidate risk factor for final decision.}

\item{coding}{Type of risk factor coding within the model. Available options are: \code{"WoE"} and
\code{"dummy"}. If \code{"WoE"} is selected, then modalities of the risk factors are replaced
with WoE values, while for \code{"dummy"} option dummies (0/1) will be created for \code{n-1}
modalities when \code{n} is total number of modalities of analyzed risk factor.}

\item{db}{Modeling data with risk factors and target variable. All risk factors should be categorized and as of
character type.}
}
\value{
The command \code{stepMIV} returns a list of five objects.\cr
The first object (\code{model}), is the final model, an object of class inheriting from \code{"glm"}.\cr
The second object (\code{steps}), is the data frame with risk factors selected at each iteration.\cr
The third object (\code{miv.iter}), is the data frame with iteration details.\cr
The fourth object (\code{warnings}), is the data frame with warnings if any observed.
The warnings refer to the following checks: if risk factor has more than 10 modalities,
if any of the bins (groups) has less then 5\% of observations and
if there are problems with WoE calculations.\cr
The final, fifth, object \code{dev.db} is returned only is \code{coding} is selected as \code{"WoE"}.
In that case data frame with replaced WoE values for risk factors that are selected in the
final model with be returned.
}
\description{
\code{stepMIV} performs stepwise logistic regression based on MIV.
}
\examples{
suppressMessages(library(PDtoolkit))
data(loans)
#identify numeric risk factors
num.rf <- sapply(loans, is.numeric)
num.rf <- names(num.rf)[!names(num.rf)\%in\%"Creditability" & num.rf]
#discretized numeric risk factors using ndr.bin from monobin package
loans[, num.rf] <- sapply(num.rf, function(x) 
ndr.bin(x = loans[, x], y = loans[, "Creditability"])[[2]])
str(loans)
#run stepMIV
res <- stepMIV(start.model = Creditability ~ 1, 
	   miv.threshold = 0.02, 
	   m.ch.p.val = 0.05,
	   coding = "WoE",
	   db = loans)
#check output elements
names(res)
#extract the final model
final.model <- res$model
#print coefficients
summary(final.model)$coefficients
#print steps of stepwise
res$steps
#print head of all iteration details
head(res$miv.iter)
#print warnings
res$warnings
#print head of coded development data
head(res$dev.db)
#calculate AUC
auc.model(predictions = predict(final.model, type = "response", newdata = res$dev.db),
    observed = res$dev.db$Creditability)
}
\references{
Scallan, G. (2011). Class(ic) Scorecards: Selecting Characteristics and Attributes in Logistic Regression,
Edinburgh Credit Scoring Conference, downloaded from
\href{https://www.scoreplus.com/papers/papers}{here}.
}
