#' Project lat-lon points
#'
#' Project long lat points to e.g. UTM projection.
#' Basics copied from \code{OpenStreetMap::\link[OpenStreetMap]{projectMercator}}
#'
#' @return data.frame (or matrix, if \code{dfout=FALSE})  with points in new projection
#' @author Berry Boessenkool, \email{berry-b@@gmx.de}, Jun 2016
#' @seealso \code{\link{scaleBar}}, \code{OpenStreetMap::\link[OpenStreetMap]{projectMercator}},
#'          \url{http://gis.stackexchange.com/a/74723}, \url{http://spatialreference.org} on proj4strings
#' @keywords spatial
#' @importFrom OpenStreetMap osm projectMercator
#' @importFrom sp coordinates coordinates<- CRS proj4string proj4string<- spTransform
#' @export
#' @examples
#' library("OpenStreetMap")
#' lat <- runif(100, 6, 12)
#' lon <- runif(100, 48, 58)
#' plot(lat,lon)
#' plot(projectMercator(lat,lon), main="Mercator")
#' plot(projectPoints(lat,lon), main="UTM32")
#' stopifnot(all( projectPoints(lat,lon, to=posm()) == projectMercator(lat,lon) ))
#'
#' projectPoints(c(52.4,NA),      c(13.6,12.9))
#' projectPoints(c(52.4,NA),      c(13.6,12.9), quiet=TRUE)
#' projectPoints(c(52.4,52.3,NA), c(13.6,12.9,13.1))
#' projectPoints(c(52.4,52.3,NA), c(13.6,NA  ,13.1))
#' projectPoints(c(52.4,52.3,NA), c(NA  ,12.9,13.1))
#'
#' # Reference system ETRS89 with GRS80-Ellipsoid (common in Germany)
#' set.seed(42)
#' d <- data.frame(N=runif(50,5734000,6115000), E=runif(50, 33189000,33458000))
#' d$VALUES <- berryFunctions::rescale(d$N, 20,40) + rnorm(50, sd=5)
#' head(d)
#' c1 <- projectPoints(lat=d$N, long=d$E-33e6, to=pll(),
#'           from=sp::CRS("+proj=utm +zone=33 +ellps=GRS80 +units=m +no_defs") )
#' c2 <- projectPoints(c1$y, c1$x, to=posm() )
#' head(c1)
#' head(c2)
#'
#' \donttest{
#' map <- pointsMap(c1, "x", "y", plot=FALSE)
#' pdf("ETRS89.pdf")
#' par(mar=c(0,0,0,0))
#' plot(map)
#' rect(par("usr")[1], par("usr")[3], par("usr")[2], par("usr")[4],
#'      col=berryFunctions::addAlpha("white", 0.7))
#' scaleBar(map, y=0.2)
#' points(c2)
#' berryFunctions::colPoints(c2$x, c2$y, d$VALUE )
#' dev.off()
#' system2("open", "ETRS89.pdf")
#' #unlink("ETRS89.pdf")
#' }
#'
#' @param lat A vector of latitudes
#' @param long A vector of longitudes
#' @param from Original Projection CRS (do not change for latlong-coordinates).
#'             DEFAULT: pll() = sp::CRS("+proj=longlat +datum=WGS84")
#' @param to target projection CRS (Coordinate Reference System) Object.
#'           Other projections can be specified as sp::CRS("your_proj4_character_string").
#'           DEFAULT: putm(long=long)
#' @param drop Drop to lowest dimension? DEFAULT: FALSE (unlike \code{\link[OpenStreetMap]{projectMercator}})
#' @param dfout Convert output to data.frame to allow easier indexing? DEFAULT: TRUE
#' @param quiet Suppress warning about NA coordinates? DEFAULT: FALSE
#'
projectPoints <- function (
lat,
long,
from=pll(),
to=putm(long=long),
drop=FALSE,
dfout=TRUE,
quiet=FALSE
)
{
# NA management
nas <- is.na(lat)|is.na(long)
if(any(nas) & !quiet) warning("there are ", sum(nas), " NAs in coordinates.")
lat <- lat[!nas] ; long <- long[!nas]
# Original points into object of class "SpatialPoints":
df <- data.frame(long=long, lat=lat)
coordinates(df) <- ~long + lat
proj4string(df) <- from
# Actual transformation:
df1 <- spTransform(df, to)
# Use only coordinates of result:
coords <- coordinates(df1)
# Post processing, NA management:
out <- matrix(NA, nrow=length(nas), ncol=2)
colnames(out) <- c("x", "y")
out[!nas,] <- coords
# formatting
if(dfout) out <- as.data.frame(out)
if(drop) out <- drop(out)
out
}
